/* SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use, copy,
 * modify, merge, publish, distribute, sublicense, and/or sell copies
 * of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 * Copyright:
 *   2021      Kunwar Maheep Singh <kunwar.maheep@students.iiit.ac.in>
 */

#define SIMDE_TEST_X86_AVX512_INSN 4dpwssd

#include <test/x86/avx512/test-avx512.h>
#include <simde/x86/avx512/4dpwssd.h>

static int
test_simde_mm512_4dpwssd_epi32 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const int32_t src[16];
    const int32_t a0[16];
    const int32_t a1[16];
    const int32_t a2[16];
    const int32_t a3[16];
    const int32_t b[4];
    const int32_t r[16];
  } test_vec[] = {
    { { -INT32_C(  1100148463),  INT32_C(  1853901119),  INT32_C(  2044082609), -INT32_C(   477201994),  INT32_C(   681824215), -INT32_C(   691225362), -INT32_C(  1484834652), -INT32_C(   221921568),
         INT32_C(   263205839),  INT32_C(   763179388),  INT32_C(   547837034), -INT32_C(  1543293491), -INT32_C(   221469180),  INT32_C(   130586723), -INT32_C(  1313978160), -INT32_C(   710708218) },
      {  INT32_C(   602166183), -INT32_C(   296721788), -INT32_C(  2096171082),  INT32_C(   824644141),  INT32_C(   455406520),  INT32_C(  1529015435),  INT32_C(   973918516), -INT32_C(   334450875),
        -INT32_C(  2028997629),  INT32_C(   209084246), -INT32_C(  2070969257),  INT32_C(  1337308822),  INT32_C(   879417769), -INT32_C(    91189818), -INT32_C(  1556767650),  INT32_C(  1334789452) },
      { -INT32_C(  1864982983),  INT32_C(  1419529469),  INT32_C(  1742220497), -INT32_C(  1934192926),  INT32_C(   767566183),  INT32_C(   203968686),  INT32_C(   967794157), -INT32_C(   611828062),
        -INT32_C(   647274788),  INT32_C(  2083391659),  INT32_C(   383977012), -INT32_C(    56452460),  INT32_C(  1764320187), -INT32_C(  1602924109),  INT32_C(  1356407982),  INT32_C(  1043095906) },
      {  INT32_C(  1796708288), -INT32_C(   739818081), -INT32_C(   521483700),  INT32_C(   534547556), -INT32_C(  1551366417),  INT32_C(   105119063), -INT32_C(  2091508447),  INT32_C(  1052869246),
        -INT32_C(  1180051174),  INT32_C(  1787662366), -INT32_C(  1085573285), -INT32_C(   203479293), -INT32_C(  2053740755), -INT32_C(  2037655195),  INT32_C(  1963647478),  INT32_C(  2125712228) },
      { -INT32_C(  1019781724),  INT32_C(  1210959085),  INT32_C(  1057519675), -INT32_C(   852301920), -INT32_C(  1286420402), -INT32_C(  1740972639),  INT32_C(   588072127), -INT32_C(  1264467953),
         INT32_C(   192403485), -INT32_C(   665607011), -INT32_C(  1122542819), -INT32_C(  1853208254), -INT32_C(  1287332591),  INT32_C(  2034990778), -INT32_C(   761505598),  INT32_C(   914767128) },
      { -INT32_C(  1287521002), -INT32_C(  1098148703),  INT32_C(   846963696), -INT32_C(    20773140) },
      {  INT32_C(  1893457071),  INT32_C(  1687491397), -INT32_C(   273821735), -INT32_C(    72978533),  INT32_C(  1795046685),  INT32_C(   485147853),  INT32_C(   899372899), -INT32_C(   114315126),
         INT32_C(  1144379574), -INT32_C(   927530077),  INT32_C(  1806572544), -INT32_C(  1770269694),  INT32_C(   962800920),  INT32_C(   285230668), -INT32_C(   467742172),  INT32_C(   846655336) } },
    { { -INT32_C(  1649342237),  INT32_C(  1226308742),  INT32_C(  1813754708),  INT32_C(   128098801),  INT32_C(  1069212574),  INT32_C(  1761494392), -INT32_C(   711296536),  INT32_C(  1658019455),
        -INT32_C(   335510426), -INT32_C(   734718080), -INT32_C(  1153412918), -INT32_C(  1883053071),  INT32_C(  1053785286), -INT32_C(  1431908926), -INT32_C(   964738746),  INT32_C(    86528671) },
      {  INT32_C(  1458645206),  INT32_C(   170534719),  INT32_C(  1774545783),  INT32_C(   351832142), -INT32_C(   967588092), -INT32_C(   613287532), -INT32_C(   643698630),  INT32_C(   434030915),
         INT32_C(   829477106),  INT32_C(  1849400055),  INT32_C(  1406599429), -INT32_C(  1922510711),  INT32_C(   743750551), -INT32_C(   284703308), -INT32_C(   121067339),  INT32_C(  1662166897) },
      {  INT32_C(  1855292023),  INT32_C(   584896540),  INT32_C(  1517663441),  INT32_C(   468245892),  INT32_C(  1296514200), -INT32_C(  1237561855),  INT32_C(  1756300534),  INT32_C(   583778731),
         INT32_C(  1603297347),  INT32_C(    42036529), -INT32_C(  1520568543),  INT32_C(  1841317332), -INT32_C(  2101737343),  INT32_C(  1295578710), -INT32_C(  1498028038), -INT32_C(   339181400) },
      {  INT32_C(   306928097), -INT32_C(   418067258), -INT32_C(  1735626301),  INT32_C(   923094198), -INT32_C(  1413890220), -INT32_C(  1342639435), -INT32_C(  2108314151),  INT32_C(   242032173),
         INT32_C(  1025554551),  INT32_C(  1210332549),  INT32_C(  1575006374),  INT32_C(  1368712700),  INT32_C(  1509707428),  INT32_C(   420082752), -INT32_C(   828678495), -INT32_C(   203617924) },
      {  INT32_C(  1177615809), -INT32_C(   644918222),  INT32_C(    20344580), -INT32_C(   112014763),  INT32_C(  1481854488), -INT32_C(   479044542),  INT32_C(   934350266), -INT32_C(   668299754),
        -INT32_C(  1105306740), -INT32_C(  1265128017),  INT32_C(  1924517149), -INT32_C(  1335162984), -INT32_C(  1760969131), -INT32_C(   713393382), -INT32_C(  1626592376),  INT32_C(  1182217914) },
      {  INT32_C(  1090819473),  INT32_C(  1626709059),  INT32_C(    30583401),  INT32_C(   112344497) },
      { -INT32_C(  1173836316),  INT32_C(  1522264557), -INT32_C(   768712038),  INT32_C(  1195733743),  INT32_C(   935772999),  INT32_C(  2086930362), -INT32_C(  1740154046), -INT32_C(  1482893640),
        -INT32_C(    75402118),  INT32_C(  1175210797), -INT32_C(   185606726),  INT32_C(  1292916603), -INT32_C(  1007298633),  INT32_C(     3414690),  INT32_C(   200556787), -INT32_C(  2007577437) } },
    { {  INT32_C(   379436028), -INT32_C(  1091889354), -INT32_C(    60950718), -INT32_C(  1086139346), -INT32_C(  1392490646),  INT32_C(  1275983331),  INT32_C(  1347346335),  INT32_C(   408289308),
        -INT32_C(   248515653),  INT32_C(  1303321098),  INT32_C(  1078529298),  INT32_C(  1275038945), -INT32_C(  1225260845), -INT32_C(  1795029515),  INT32_C(    31805668),  INT32_C(   186203216) },
      {  INT32_C(   989612079),  INT32_C(  1971825763), -INT32_C(  1699360583),  INT32_C(   803648604), -INT32_C(  1427775820), -INT32_C(   935335708), -INT32_C(  1983306696), -INT32_C(  1869290656),
        -INT32_C(  1899327189), -INT32_C(   167554755),  INT32_C(  2123413537),  INT32_C(   565016173),  INT32_C(   969642837), -INT32_C(  1291777413), -INT32_C(  1891841233), -INT32_C(   652226130) },
      { -INT32_C(  1620514462),  INT32_C(  1553296186), -INT32_C(  1847974620), -INT32_C(   239958116), -INT32_C(  1792377318), -INT32_C(  1219941496), -INT32_C(  1538816522), -INT32_C(  1199675818),
        -INT32_C(  1973950896),  INT32_C(  1978068049), -INT32_C(  1375289326),  INT32_C(  1654634824), -INT32_C(  1074214601), -INT32_C(   344505868),  INT32_C(   495959750),  INT32_C(  1960119588) },
      {  INT32_C(  1174351347),  INT32_C(   733668633), -INT32_C(   287719002), -INT32_C(  1320126342),  INT32_C(   913328193),  INT32_C(  1344399241), -INT32_C(   898781019), -INT32_C(  1321319746),
        -INT32_C(  1980351121), -INT32_C(   910905053), -INT32_C(   323514766),  INT32_C(  1218250502), -INT32_C(   646050480), -INT32_C(  1708548107),  INT32_C(   224695887),  INT32_C(  1220518873) },
      {  INT32_C(    64075232), -INT32_C(   657684890),  INT32_C(   449086228), -INT32_C(   614309493),  INT32_C(  1689575535), -INT32_C(   805380481),  INT32_C(  1306289012), -INT32_C(   426337530),
        -INT32_C(  1209374895),  INT32_C(    26261229), -INT32_C(   987999174),  INT32_C(   631275190), -INT32_C(   561425313), -INT32_C(  1481799886), -INT32_C(   252409110),  INT32_C(  1993837093) },
      { -INT32_C(   550583822), -INT32_C(  1310605705), -INT32_C(   931726062), -INT32_C(   621996421) },
      {  INT32_C(  1306916441), -INT32_C(  1157758729),  INT32_C(   396079240),  INT32_C(  1241257201), -INT32_C(  1454350047),  INT32_C(   988515018), -INT32_C(  1932247157),  INT32_C(  2102636848),
         INT32_C(  1158098244),  INT32_C(   191666118),  INT32_C(  1092655398), -INT32_C(   120428489),  INT32_C(   617886679),  INT32_C(  1780417690),  INT32_C(   548988535), -INT32_C(   347241123) } },
    { { -INT32_C(  1648789909), -INT32_C(   398170370),  INT32_C(   383334896), -INT32_C(  1249070909), -INT32_C(   392840591), -INT32_C(  1952811400), -INT32_C(   296546189), -INT32_C(  1832304345),
        -INT32_C(  1255177545), -INT32_C(   643926808),  INT32_C(  1894741933), -INT32_C(  1742308569), -INT32_C(  1367295178), -INT32_C(  1539761615),  INT32_C(  1385336107), -INT32_C(  2048631858) },
      { -INT32_C(   969206819),  INT32_C(   882891143),  INT32_C(  2007338576),  INT32_C(  1057999625), -INT32_C(  1209102458), -INT32_C(   715446358), -INT32_C(  2111312460),  INT32_C(   638061385),
        -INT32_C(  1494465762),  INT32_C(  1826261788),  INT32_C(   602111769), -INT32_C(   798821814),  INT32_C(   730288257),  INT32_C(   738255480),  INT32_C(   430909392),  INT32_C(  1363130162) },
      {  INT32_C(   385297658), -INT32_C(   779955785), -INT32_C(  1678482095), -INT32_C(   663988649),  INT32_C(   520418215), -INT32_C(  1505033002),  INT32_C(  1589639724), -INT32_C(  1414529359),
        -INT32_C(   490625494), -INT32_C(   910998664), -INT32_C(    10180696), -INT32_C(  1529294595), -INT32_C(  1698439996),  INT32_C(   205524960), -INT32_C(  1150550263),  INT32_C(   677780221) },
      {  INT32_C(   956967105),  INT32_C(   335723883),  INT32_C(  1628661348), -INT32_C(    83432649), -INT32_C(  1466578489), -INT32_C(   491465256), -INT32_C(   778231852), -INT32_C(    67566790),
        -INT32_C(  1758198997),  INT32_C(   615200448), -INT32_C(   746144100),  INT32_C(  1909361834),  INT32_C(   773415765),  INT32_C(   202427960),  INT32_C(   668904941), -INT32_C(   601696336) },
      { -INT32_C(  1686939941),  INT32_C(   700456588), -INT32_C(  2030287140),  INT32_C(   670616530),  INT32_C(  1733628206), -INT32_C(   848075297), -INT32_C(  1007398638),  INT32_C(    77535017),
        -INT32_C(    90238355),  INT32_C(   203644720),  INT32_C(  2006130597),  INT32_C(   429886442),  INT32_C(  2088826012),  INT32_C(  1816785754),  INT32_C(  1848589637), -INT32_C(  1032663468) },
      {  INT32_C(   280760800),  INT32_C(   371056497), -INT32_C(   376524546), -INT32_C(   687723205) },
      {  INT32_C(  2124663838), -INT32_C(  1254947481),  INT32_C(  1032389311), -INT32_C(  1082895409), -INT32_C(  1528194733),  INT32_C(  1337207260), -INT32_C(   890836357),  INT32_C(  1241700779),
        -INT32_C(   388687182),  INT32_C(  1391176999), -INT32_C(   458522355),  INT32_C(  1537477269), -INT32_C(  2054069089), -INT32_C(  1954919772),  INT32_C(  1475788759),  INT32_C(   679800115) } },
    { {  INT32_C(  2069070369), -INT32_C(  1159160715),  INT32_C(   774379482), -INT32_C(   957310234),  INT32_C(   500673708), -INT32_C(  1976306549), -INT32_C(   546061916),  INT32_C(   280393198),
         INT32_C(  1821051639), -INT32_C(  2144898138),  INT32_C(  1907314571), -INT32_C(  1774739478), -INT32_C(   676131252), -INT32_C(  1503533566), -INT32_C(  1769614168),  INT32_C(  1084636233) },
      { -INT32_C(   324193722),  INT32_C(   812504229),  INT32_C(   228662307),  INT32_C(   128178619), -INT32_C(   354461977), -INT32_C(   443531203),  INT32_C(  1585124885), -INT32_C(  1734401454),
        -INT32_C(   108770220),  INT32_C(  1126887712), -INT32_C(   917386483), -INT32_C(  1932462940), -INT32_C(  2005487797),  INT32_C(    91031280),  INT32_C(  1852041244),  INT32_C(  1577452042) },
      {  INT32_C(  1851296590), -INT32_C(  1968012676), -INT32_C(   229440691),  INT32_C(  1115562999), -INT32_C(  1010109229),  INT32_C(   398997498),  INT32_C(   713370400),  INT32_C(  2089389102),
        -INT32_C(  1813257961), -INT32_C(  1340236445), -INT32_C(  1750962016),  INT32_C(  1742348436),  INT32_C(   254452756), -INT32_C(    64556324),  INT32_C(  1260825373),  INT32_C(  1321774903) },
      { -INT32_C(   203246704), -INT32_C(   257622192),  INT32_C(    75974256),  INT32_C(  2070634855), -INT32_C(   494234362), -INT32_C(  1528909689), -INT32_C(  1812986788),  INT32_C(  1155709108),
        -INT32_C(  1153907605),  INT32_C(   866901187), -INT32_C(  1992871390), -INT32_C(  1710906732), -INT32_C(  1099133129), -INT32_C(  1671210432),  INT32_C(   305091422),  INT32_C(  1985417483) },
      { -INT32_C(  1724805163), -INT32_C(  1915954069), -INT32_C(  1575549938), -INT32_C(   566485850), -INT32_C(   342050645),  INT32_C(  1904738066),  INT32_C(  1568913234), -INT32_C(  1630283064),
        -INT32_C(   734591895), -INT32_C(   278789152), -INT32_C(  1366198009),  INT32_C(  1082969749), -INT32_C(  1725159290),  INT32_C(  2047521576),  INT32_C(   869829994), -INT32_C(   774788248) },
      { -INT32_C(  1868166992),  INT32_C(   327092235),  INT32_C(   381751681),  INT32_C(  1700154847) },
      { -INT32_C(  1425490233), -INT32_C(  1899508923), -INT32_C(  1812047491),  INT32_C(   709857869),  INT32_C(  1457439950),  INT32_C(  1496245983),  INT32_C(  1604848601),  INT32_C(  1179223891),
        -INT32_C(  1808532518), -INT32_C(   696267086), -INT32_C(   708541624), -INT32_C(  1902783242),  INT32_C(  1684587179), -INT32_C(  1393355237),  INT32_C(   625875422), -INT32_C(  1125730195) } },
    { { -INT32_C(  1644264843), -INT32_C(  1609037770), -INT32_C(    19664746),  INT32_C(  1271899291), -INT32_C(  1193511508), -INT32_C(    20227203),  INT32_C(  1259637868),  INT32_C(  1320250329),
         INT32_C(   602714093),  INT32_C(  1321469112), -INT32_C(  1890805772), -INT32_C(   388293828),  INT32_C(   245413777),  INT32_C(  2131585810), -INT32_C(   792059401),  INT32_C(  2048883596) },
      { -INT32_C(   476247253),  INT32_C(    53567759),  INT32_C(   898792953),  INT32_C(   689794456),  INT32_C(   926465572),  INT32_C(   565593385), -INT32_C(   202276762),  INT32_C(   661459196),
         INT32_C(   705301019),  INT32_C(  1697463148),  INT32_C(  1352319160),  INT32_C(  1383773997), -INT32_C(  1618365835),  INT32_C(  1572880375), -INT32_C(  1152339521), -INT32_C(   572342846) },
      {  INT32_C(   873000136), -INT32_C(   543607513),  INT32_C(   590361589),  INT32_C(  1618324202),  INT32_C(  1409285724), -INT32_C(    55525571),  INT32_C(   850919792), -INT32_C(  2045732162),
        -INT32_C(  1363535737),  INT32_C(  1133400909),  INT32_C(  1902558854), -INT32_C(   992879768),  INT32_C(   370659545), -INT32_C(    15546481), -INT32_C(  2026714424), -INT32_C(   334675355) },
      { -INT32_C(  1483028390), -INT32_C(  1561712613),  INT32_C(  1309888742),  INT32_C(    68346923),  INT32_C(  1125788084), -INT32_C(  1186845199),  INT32_C(  1547728119),  INT32_C(   273174198),
         INT32_C(   834200342), -INT32_C(   221011445),  INT32_C(   507569907),  INT32_C(  2116244426),  INT32_C(  1841380732),  INT32_C(  1646724202),  INT32_C(   801007480), -INT32_C(   885061707) },
      { -INT32_C(   167905302), -INT32_C(  1914187622), -INT32_C(  2119489353), -INT32_C(   150942085),  INT32_C(  1986380044),  INT32_C(  1054379205), -INT32_C(  1452435469), -INT32_C(  2005619554),
         INT32_C(  1048408484), -INT32_C(   104110782),  INT32_C(   142243469),  INT32_C(  1375697477),  INT32_C(    29844540), -INT32_C(   465592080), -INT32_C(   712135625), -INT32_C(    44236455) },
      { -INT32_C(  1254368653), -INT32_C(   861010369), -INT32_C(  1043060612), -INT32_C(   552414045) },
      { -INT32_C(  2096224540), -INT32_C(  1398322017), -INT32_C(   694792845),  INT32_C(    31944588), -INT32_C(  1110118864),  INT32_C(   170313719),  INT32_C(  1363683764),  INT32_C(  1757163491),
         INT32_C(  1726283270), -INT32_C(   314170218), -INT32_C(   165627053),  INT32_C(  1884876188),  INT32_C(   978823045), -INT32_C(   758526260), -INT32_C(  1844853580),  INT32_C(   267322666) } },
    { {  INT32_C(   702601784), -INT32_C(  1324539782),  INT32_C(   629578444),  INT32_C(   237167515), -INT32_C(    37528131), -INT32_C(   523669149),  INT32_C(  1034002074), -INT32_C(  1440959374),
         INT32_C(   148110478), -INT32_C(   373694436),  INT32_C(   370032506), -INT32_C(   551276254), -INT32_C(   220403570), -INT32_C(   204298663), -INT32_C(  1238338748), -INT32_C(  1251980249) },
      {  INT32_C(  1706898505), -INT32_C(  1890683372), -INT32_C(   677028427),  INT32_C(   481806734),  INT32_C(   185504689),  INT32_C(  2113855545),  INT32_C(  2066952020), -INT32_C(  1003383685),
        -INT32_C(   601166136),  INT32_C(   443250789),  INT32_C(  1676808405), -INT32_C(  1954502182),  INT32_C(  1989578300), -INT32_C(  1007446673),  INT32_C(  1061038020), -INT32_C(  2096861253) },
      { -INT32_C(  1017172386),  INT32_C(  2094910374), -INT32_C(  1243623461), -INT32_C(  1254006920),  INT32_C(  1563154414),  INT32_C(   807411308),  INT32_C(     7364165),  INT32_C(   746812622),
         INT32_C(  1223680930), -INT32_C(  1983591250),  INT32_C(   339715228), -INT32_C(   238452733), -INT32_C(  1001458601),  INT32_C(  1492413971), -INT32_C(  1688640307),  INT32_C(  2059918552) },
      {  INT32_C(  1841542847),  INT32_C(   519538562),  INT32_C(   791885355),  INT32_C(   220265654),  INT32_C(    80834545), -INT32_C(  1436760355),  INT32_C(    54900010),  INT32_C(  1367149714),
         INT32_C(  1170161858), -INT32_C(   211568952), -INT32_C(  1574791444), -INT32_C(  2068888685), -INT32_C(  1886879310),  INT32_C(  1899684935),  INT32_C(   745832346),  INT32_C(  1316876940) },
      { -INT32_C(    91014094), -INT32_C(   554764302),  INT32_C(   545263757),  INT32_C(    94646355), -INT32_C(   124441423), -INT32_C(  1419129071), -INT32_C(   623387058),  INT32_C(    36263120),
        -INT32_C(  2080523119),  INT32_C(  1097001907),  INT32_C(  1331815163), -INT32_C(  1017903598),  INT32_C(  1136388402),  INT32_C(   133047736), -INT32_C(   740178685), -INT32_C(  1412101606) },
      {  INT32_C(  2049888967), -INT32_C(  1178890051), -INT32_C(  2063065741),  INT32_C(  1430871075) },
      { -INT32_C(  1576554490),  INT32_C(  1130601863), -INT32_C(  1679317811),  INT32_C(  1199250850),  INT32_C(   528150953), -INT32_C(  1623480570), -INT32_C(   584083810),  INT32_C(  1500924057),
        -INT32_C(  1309908607), -INT32_C(   518161622), -INT32_C(   669186506), -INT32_C(  1062160141),  INT32_C(  1668489060),  INT32_C(  1406593621), -INT32_C(  1256096436),  INT32_C(   873108893) } },
    { { -INT32_C(    23526330),  INT32_C(   738559785),  INT32_C(  1744824141), -INT32_C(  1206725135),  INT32_C(  1697857703),  INT32_C(  1142877905),  INT32_C(   784934411), -INT32_C(   930868606),
         INT32_C(  1086790935), -INT32_C(   244462428), -INT32_C(  1520931661), -INT32_C(   379753919),  INT32_C(  2085523627), -INT32_C(  1967100801),  INT32_C(   347703698), -INT32_C(  1277346404) },
      { -INT32_C(    17587110),  INT32_C(   619733104),  INT32_C(   248072397),  INT32_C(  1593255603),  INT32_C(   920274359),  INT32_C(  1136695985), -INT32_C(  1084720861),  INT32_C(   275920310),
         INT32_C(  1225746137), -INT32_C(  1821507642), -INT32_C(    90032569),  INT32_C(   341350749), -INT32_C(  1874185505), -INT32_C(   254539060),  INT32_C(   984558724),  INT32_C(   978002529) },
      {  INT32_C(  1317231240), -INT32_C(  1595739815), -INT32_C(  2070182873), -INT32_C(    57085155), -INT32_C(   225582554),  INT32_C(  1910661613), -INT32_C(   290745715),  INT32_C(  1009317556),
        -INT32_C(  1450529968), -INT32_C(  1001755492),  INT32_C(   239658480), -INT32_C(    32841256), -INT32_C(  1326344253), -INT32_C(  2061380616),  INT32_C(   427019621),  INT32_C(   341154755) },
      { -INT32_C(   474095545),  INT32_C(  1034356556), -INT32_C(   984878868), -INT32_C(  1799137839), -INT32_C(   448416531), -INT32_C(   311728504), -INT32_C(   150479309), -INT32_C(  1056220038),
        -INT32_C(  1985689540), -INT32_C(  1127854896),  INT32_C(   226562364),  INT32_C(  1403077990), -INT32_C(  2126911751), -INT32_C(  2140167091), -INT32_C(    59279742),  INT32_C(   264078034) },
      {  INT32_C(   462971467), -INT32_C(   354984274), -INT32_C(   705210001), -INT32_C(  1758881634), -INT32_C(   870751617), -INT32_C(  2008250362), -INT32_C(   796539650), -INT32_C(  1847639482),
         INT32_C(  1387034532),  INT32_C(  1144816853),  INT32_C(  2065314781),  INT32_C(  1259488204), -INT32_C(  1424544859), -INT32_C(  1322032205),  INT32_C(  1854060583), -INT32_C(  1610653189) },
      { -INT32_C(  1359827751),  INT32_C(   234040880),  INT32_C(   780668258), -INT32_C(   176580016) },
      { -INT32_C(    31367985),  INT32_C(  1679230073),  INT32_C(   962128907), -INT32_C(   779858416),  INT32_C(   954582441),  INT32_C(  1784652151), -INT32_C(  1450057917), -INT32_C(   258265955),
         INT32_C(  1799037981),  INT32_C(   510401450),  INT32_C(  1798839633), -INT32_C(  1348553251), -INT32_C(   960277757), -INT32_C(   581802848),  INT32_C(  1202308689),  INT32_C(  2050199441) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512i src = simde_mm512_loadu_epi32(test_vec[i].src);
    simde__m512i a0 = simde_mm512_loadu_epi32(test_vec[i].a0);
    simde__m512i a1 = simde_mm512_loadu_epi32(test_vec[i].a1);
    simde__m512i a2 = simde_mm512_loadu_epi32(test_vec[i].a2);
    simde__m512i a3 = simde_mm512_loadu_epi32(test_vec[i].a3);
    simde__m128i b = simde_x_mm_loadu_epi32(test_vec[i].b);
    simde__m512i r = simde_mm512_4dpwssd_epi32(src, a0, a1, a2, a3, &b);
    simde_test_x86_assert_equal_i32x16(r, simde_mm512_loadu_epi32(test_vec[i].r));
  }

  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde__m512i src = simde_test_x86_random_i32x16();
    simde__m512i a0 = simde_test_x86_random_i32x16();
    simde__m512i a1 = simde_test_x86_random_i32x16();
    simde__m512i a2 = simde_test_x86_random_i32x16();
    simde__m512i a3 = simde_test_x86_random_i32x16();
    simde__m128i b = simde_test_x86_random_i32x4();
    simde__m512i r = simde_mm512_4dpwssd_epi32(src, a0, a1, a2, a3, &b);

    simde_test_x86_write_i32x16(2, src, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_x86_write_i32x16(2, a0, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i32x16(2, a1, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i32x16(2, a2, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i32x16(2, a3, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i32x4(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i32x16(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_mm512_mask_4dpwssd_epi32 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const int32_t src[16];
    const simde__mmask16 k;
    const int32_t a0[16];
    const int32_t a1[16];
    const int32_t a2[16];
    const int32_t a3[16];
    const int32_t b[4];
    const int32_t r[16];
  } test_vec[] = {
    { {  INT32_C(  2040565958),  INT32_C(   455857139), -INT32_C(  2021020276), -INT32_C(   416905202),  INT32_C(  1687558196), -INT32_C(  1452177081), -INT32_C(   422053482),  INT32_C(  1524322452),
        -INT32_C(   741114912), -INT32_C(   621871538), -INT32_C(  1168017493),  INT32_C(   883001344), -INT32_C(   409389152),  INT32_C(  1452280512), -INT32_C(  1740871932), -INT32_C(  1745741897) },
      UINT16_C(50578),
      {  INT32_C(  1506074987),  INT32_C(   500264891), -INT32_C(   878325463), -INT32_C(  1644018427), -INT32_C(  1112947923),  INT32_C(  1445309721),  INT32_C(   929946693), -INT32_C(   537132940),
        -INT32_C(  1674002207),  INT32_C(  1505298736), -INT32_C(   534487334), -INT32_C(   780261468),  INT32_C(    59647978),  INT32_C(   425309140),  INT32_C(    72402832), -INT32_C(  1461498425) },
      {  INT32_C(  1027873549), -INT32_C(     6816219),  INT32_C(    14662492), -INT32_C(   858693918),  INT32_C(  1523605637), -INT32_C(  1552733933), -INT32_C(  1213741840),  INT32_C(   509577745),
        -INT32_C(   883120986), -INT32_C(    20253790), -INT32_C(  1845515602), -INT32_C(  1923231736),  INT32_C(  1156001072),  INT32_C(  1206344535),  INT32_C(   822054687), -INT32_C(  1085383143) },
      { -INT32_C(  1534416382),  INT32_C(  1285772701),  INT32_C(   131965695), -INT32_C(  1550566798), -INT32_C(  1075348376), -INT32_C(   167325993),  INT32_C(  1998980445),  INT32_C(  1714844772),
        -INT32_C(  1140080353),  INT32_C(   352890390), -INT32_C(  1038293680), -INT32_C(  2006601696),  INT32_C(    55004204),  INT32_C(  2029604123), -INT32_C(  1209065645), -INT32_C(  1289935212) },
      { -INT32_C(    43046681),  INT32_C(   638744790),  INT32_C(  2112433757),  INT32_C(   184897246), -INT32_C(  1240576869), -INT32_C(   315750246), -INT32_C(  1146872281),  INT32_C(   728678724),
        -INT32_C(  1071063318), -INT32_C(  1276691882),  INT32_C(  1177669736), -INT32_C(  1185860066),  INT32_C(   493838467), -INT32_C(  1895129496),  INT32_C(     4959932),  INT32_C(  1496037743) },
      { -INT32_C(   317041769), -INT32_C(   173997683), -INT32_C(   281292079), -INT32_C(  1951822584) },
      {  INT32_C(  2040565958),  INT32_C(   860212987), -INT32_C(  2021020276), -INT32_C(   416905202),  INT32_C(  1840970705), -INT32_C(  1452177081), -INT32_C(   422053482),  INT32_C(  1496398148),
         INT32_C(  2131739329), -INT32_C(   621871538),  INT32_C(   218535548),  INT32_C(   883001344), -INT32_C(   409389152),  INT32_C(  1452280512),  INT32_C(  1863674573),  INT32_C(  2016566140) } },
    { {  INT32_C(  1437079789),  INT32_C(  1927590582), -INT32_C(   797823136), -INT32_C(  2144756247),  INT32_C(  2104378352),  INT32_C(   359796548), -INT32_C(   385503519),  INT32_C(   678800955),
         INT32_C(  2105351622),  INT32_C(   820994512),  INT32_C(  2046845585), -INT32_C(   252040449), -INT32_C(  1301452691),  INT32_C(  1489494135), -INT32_C(   918369138),  INT32_C(  1106360186) },
      UINT16_C(28372),
      { -INT32_C(  1378900802), -INT32_C(   720412459), -INT32_C(   704704550),  INT32_C(  1815965183), -INT32_C(   431180513),  INT32_C(  1337186829), -INT32_C(  1811534429),  INT32_C(   738384750),
         INT32_C(  1423626879),  INT32_C(   220916018), -INT32_C(   136173064), -INT32_C(  1251729514), -INT32_C(   493047596),  INT32_C(   774983563), -INT32_C(   339527556), -INT32_C(  1843935981) },
      { -INT32_C(   890769001), -INT32_C(   740879909), -INT32_C(   775177926), -INT32_C(  1383714856),  INT32_C(  1787765471), -INT32_C(   291913615),  INT32_C(   198794488), -INT32_C(  1180766175),
        -INT32_C(  1115453982), -INT32_C(   795845994), -INT32_C(   341746925),  INT32_C(  1788356746), -INT32_C(  1126946998), -INT32_C(   525701656), -INT32_C(   336821303),  INT32_C(  1436846707) },
      { -INT32_C(  1525537009), -INT32_C(  1804229759), -INT32_C(  1988159490), -INT32_C(  1980557249),  INT32_C(   658884415), -INT32_C(    32968907), -INT32_C(   420875150), -INT32_C(  1925476994),
         INT32_C(   892489396), -INT32_C(   271996943), -INT32_C(    42448450), -INT32_C(  1601737887),  INT32_C(  1757989939),  INT32_C(   778490044),  INT32_C(  1108627652), -INT32_C(  1831907107) },
      { -INT32_C(  1882783330),  INT32_C(  1736348072),  INT32_C(   996472794), -INT32_C(  1780749470),  INT32_C(  1962779576),  INT32_C(   933389427), -INT32_C(  1854294092), -INT32_C(  1524414457),
        -INT32_C(   231412919),  INT32_C(  1448718972),  INT32_C(   210877865),  INT32_C(  1637969065), -INT32_C(  2083152112), -INT32_C(  1212451069),  INT32_C(   893924398), -INT32_C(   958763907) },
      { -INT32_C(   742912425),  INT32_C(  1781076417),  INT32_C(  2021046991),  INT32_C(   920197158) },
      {  INT32_C(  1437079789),  INT32_C(  1927590582), -INT32_C(     7097305), -INT32_C(  2144756247),  INT32_C(  1455571827),  INT32_C(   359796548), -INT32_C(  1652561201),  INT32_C(  1445552822),
         INT32_C(  2105351622), -INT32_C(   836265486), -INT32_C(   789504247), -INT32_C(   690919312), -INT32_C(  1301452691),  INT32_C(  2022773360),  INT32_C(   878609001),  INT32_C(  1106360186) } },
    { { -INT32_C(  1162170697),  INT32_C(  1416721701),  INT32_C(   646560169),  INT32_C(  2095932453),  INT32_C(   860857714), -INT32_C(  2036434761),  INT32_C(  1509823538), -INT32_C(   477112532),
        -INT32_C(  1432532603),  INT32_C(  1761480382), -INT32_C(   309425976),  INT32_C(  1584036844), -INT32_C(   661472991),  INT32_C(  1683894322),  INT32_C(  1908235332), -INT32_C(  1202434765) },
      UINT16_C(62102),
      {  INT32_C(  1627411810),  INT32_C(  1273612477),  INT32_C(   549967286), -INT32_C(   975574989),  INT32_C(   519375808),  INT32_C(   762985072), -INT32_C(     8737365), -INT32_C(   940502683),
         INT32_C(   589886054),  INT32_C(  1886261690),  INT32_C(   445658598), -INT32_C(   572560867), -INT32_C(   436480651), -INT32_C(  1173129969), -INT32_C(  2017882846),  INT32_C(    72330142) },
      {  INT32_C(  1462204317),  INT32_C(  1875416713), -INT32_C(   376874805),  INT32_C(   952527298),  INT32_C(  1293795902),  INT32_C(  1510420535),  INT32_C(  1558298814),  INT32_C(   140587115),
         INT32_C(   828410024), -INT32_C(   358602722),  INT32_C(  1137912448), -INT32_C(   780428909), -INT32_C(  1826711461), -INT32_C(  2014501431),  INT32_C(  1373884133), -INT32_C(  1487321857) },
      { -INT32_C(   338118195),  INT32_C(  1658157281),  INT32_C(   900049058), -INT32_C(  1660542910), -INT32_C(  2127551304),  INT32_C(   789060937), -INT32_C(   343872276), -INT32_C(    40707792),
         INT32_C(  1961454227), -INT32_C(  2049523998), -INT32_C(  1447396505),  INT32_C(  1413923227),  INT32_C(   802519013), -INT32_C(  2141266284), -INT32_C(    93593910),  INT32_C(  1257831863) },
      {  INT32_C(  1254089063),  INT32_C(   114267551), -INT32_C(  1397782255),  INT32_C(   805369418),  INT32_C(    23058029),  INT32_C(  2122497460),  INT32_C(  1383656859),  INT32_C(  1386049771),
        -INT32_C(   241410991),  INT32_C(    49769457),  INT32_C(  1068476405),  INT32_C(   175091613),  INT32_C(   957075077),  INT32_C(   649563531),  INT32_C(  1719218299), -INT32_C(   222816608) },
      {  INT32_C(  1675842930), -INT32_C(  1251550528),  INT32_C(   519378305),  INT32_C(  1244161220) },
      { -INT32_C(  1162170697),  INT32_C(   971792663),  INT32_C(  1252457442),  INT32_C(  2095932453), -INT32_C(  1282752375), -INT32_C(  2036434761),  INT32_C(  1509823538), -INT32_C(  1249627999),
        -INT32_C(  1432532603), -INT32_C(   314579870), -INT32_C(   309425976),  INT32_C(  1584036844), -INT32_C(  2052621779), -INT32_C(  2035534325), -INT32_C(  1908017994), -INT32_C(   460353350) } },
    { { -INT32_C(  1098697677),  INT32_C(  1021656001),  INT32_C(   195190379), -INT32_C(   419603596),  INT32_C(  1883889840),  INT32_C(  1009168315), -INT32_C(  1990517820), -INT32_C(  1294761089),
         INT32_C(  2037470903), -INT32_C(    55224687),  INT32_C(   671635636),  INT32_C(  1661863347), -INT32_C(  1579984922), -INT32_C(   874645241), -INT32_C(  1806419948),  INT32_C(  1933977532) },
      UINT16_C(46974),
      { -INT32_C(  1576202260),  INT32_C(   351977740), -INT32_C(   149312023), -INT32_C(   481296624),  INT32_C(  2128434848),  INT32_C(  1991700769), -INT32_C(   862096763), -INT32_C(   763159578),
         INT32_C(   930386219),  INT32_C(  1011576402),  INT32_C(   724788251),  INT32_C(    68125284), -INT32_C(    92082984),  INT32_C(  1668299229), -INT32_C(  1859187285),  INT32_C(  1415885353) },
      { -INT32_C(  1769220029),  INT32_C(  1657984583), -INT32_C(  1634859718),  INT32_C(  1621335432),  INT32_C(  1717183881),  INT32_C(   180996702),  INT32_C(    26999000), -INT32_C(   296353877),
         INT32_C(   528802008), -INT32_C(   259959114), -INT32_C(   460386468),  INT32_C(   893661868), -INT32_C(  1231249577),  INT32_C(  1103127913),  INT32_C(   155343710),  INT32_C(   871864155) },
      {  INT32_C(   760380535),  INT32_C(   773772242), -INT32_C(  1894601245),  INT32_C(   918902751),  INT32_C(  1609327094),  INT32_C(   614575302),  INT32_C(  1663951624), -INT32_C(   225041029),
         INT32_C(  1931536545), -INT32_C(  1633534277), -INT32_C(   902908693),  INT32_C(    16839435),  INT32_C(   442625364), -INT32_C(  1589706087),  INT32_C(  1610902757),  INT32_C(   844339857) },
      {  INT32_C(  1051095938), -INT32_C(  1663285071),  INT32_C(   140905212),  INT32_C(  1359570685), -INT32_C(   311661997),  INT32_C(  1385081708), -INT32_C(  1447914729), -INT32_C(  1327823571),
         INT32_C(   703496568), -INT32_C(   960116023), -INT32_C(   758240299), -INT32_C(   450570350), -INT32_C(  1361932222),  INT32_C(  1375756603),  INT32_C(   570143732),  INT32_C(   835835832) },
      {  INT32_C(   576372568),  INT32_C(  1609048202), -INT32_C(   583944629), -INT32_C(   809282163) },
      { -INT32_C(  1098697677),  INT32_C(  1466954739),  INT32_C(  1196529139), -INT32_C(  1569783972), -INT32_C(   449635307), -INT32_C(   391333491), -INT32_C(   402338280), -INT32_C(  1294761089),
        -INT32_C(   705639127), -INT32_C(   663447195), -INT32_C(  1200988392),  INT32_C(  1661863347),  INT32_C(  1331547471),  INT32_C(  1894089839), -INT32_C(  1806419948), -INT32_C(   238903297) } },
    { {  INT32_C(   545166821), -INT32_C(   361529610), -INT32_C(   368284111), -INT32_C(  1642341051),  INT32_C(   666924445), -INT32_C(   511268715), -INT32_C(   339822754), -INT32_C(   222592755),
         INT32_C(   219363607), -INT32_C(   369588553),  INT32_C(   986907892),  INT32_C(  2128146145), -INT32_C(   106588061), -INT32_C(  1629869248), -INT32_C(   259417885),  INT32_C(   836912154) },
      UINT16_C(62845),
      {  INT32_C(   914044222), -INT32_C(   247828450), -INT32_C(  2099307350),  INT32_C(  1075462810),  INT32_C(   359356987), -INT32_C(  2119282952),  INT32_C(   566675518),  INT32_C(   924206073),
        -INT32_C(  1771138440), -INT32_C(  1400395774),  INT32_C(  1596876484), -INT32_C(   459323223), -INT32_C(  1694889310), -INT32_C(  1759795111),  INT32_C(  1773724016), -INT32_C(  1650405595) },
      {  INT32_C(  1664355937),  INT32_C(  2064628663), -INT32_C(   874889950),  INT32_C(   665811333), -INT32_C(   574510716), -INT32_C(  1049306031), -INT32_C(   483775042),  INT32_C(  1568721916),
        -INT32_C(  1866418983), -INT32_C(  1844654224), -INT32_C(  1856117236), -INT32_C(   457700256),  INT32_C(   130120118),  INT32_C(   331953493),  INT32_C(  1593242466), -INT32_C(  1732544578) },
      { -INT32_C(  1691845845),  INT32_C(  1445803082),  INT32_C(  2078771739),  INT32_C(  1298112407),  INT32_C(  1851007000), -INT32_C(  1199497899), -INT32_C(   820611055),  INT32_C(   443011823),
        -INT32_C(  1732865971), -INT32_C(   537992252),  INT32_C(    89839214), -INT32_C(  1907181195),  INT32_C(   805086937), -INT32_C(   706249276), -INT32_C(   458949131),  INT32_C(   503254223) },
      {  INT32_C(  1622521244),  INT32_C(   104899480), -INT32_C(   284452231),  INT32_C(   763190612), -INT32_C(   933463804), -INT32_C(   325237770),  INT32_C(   282083649), -INT32_C(   382873779),
         INT32_C(   491446916), -INT32_C(    14447994),  INT32_C(  2028875556), -INT32_C(  1868141684), -INT32_C(   614989084), -INT32_C(  2016938682), -INT32_C(  2070440137), -INT32_C(   345062041) },
      {  INT32_C(   755546279),  INT32_C(  1714236482), -INT32_C(   404808869),  INT32_C(  1803060615) },
      {  INT32_C(   161996097), -INT32_C(   361529610), -INT32_C(  1860142463),  INT32_C(  1792623033),  INT32_C(   348714542),  INT32_C(   469150906), -INT32_C(  1172062604), -INT32_C(   222592755),
         INT32_C(   521138939), -INT32_C(   369588553), -INT32_C(   972721255),  INT32_C(  2128146145), -INT32_C(   886346490),  INT32_C(  1989314877),  INT32_C(    27967130), -INT32_C(  1052424451) } },
    { { -INT32_C(   850997113), -INT32_C(    44823098),  INT32_C(   209841317),  INT32_C(  1492643761), -INT32_C(   377093977), -INT32_C(  2024754388),  INT32_C(  1433284558),  INT32_C(  1002563252),
         INT32_C(  2097743799), -INT32_C(  1166385899), -INT32_C(    87622839), -INT32_C(  1840005653), -INT32_C(   377693763),  INT32_C(  1517341836), -INT32_C(  1347362821),  INT32_C(  2095739333) },
      UINT16_C(62328),
      {  INT32_C(  1934724601),  INT32_C(   225417799), -INT32_C(   389326187),  INT32_C(  1757513964), -INT32_C(   499888782),  INT32_C(  1472278439), -INT32_C(   909604898), -INT32_C(    37994237),
         INT32_C(   376442318),  INT32_C(  1025761192),  INT32_C(   640020025), -INT32_C(   376510857), -INT32_C(   624180429), -INT32_C(   785216014),  INT32_C(   396032532),  INT32_C(   169104955) },
      {  INT32_C(   203457892), -INT32_C(  1639365788), -INT32_C(  1446744526), -INT32_C(  2020453804),  INT32_C(   140598549), -INT32_C(    19295254), -INT32_C(   921275506),  INT32_C(   768813769),
         INT32_C(   339342255),  INT32_C(  1773306423),  INT32_C(  1142060784), -INT32_C(   557079352), -INT32_C(   320459519),  INT32_C(  1324007360), -INT32_C(    82313166), -INT32_C(   634787029) },
      {  INT32_C(   384721631), -INT32_C(   713056027), -INT32_C(   551907050),  INT32_C(   918414645), -INT32_C(   752704750), -INT32_C(  1809773214),  INT32_C(   948910349),  INT32_C(    68401189),
         INT32_C(     1704219), -INT32_C(  1193961054),  INT32_C(  1603792682), -INT32_C(   409643819),  INT32_C(  1505409271), -INT32_C(   756163387),  INT32_C(   973831445),  INT32_C(  1346248245) },
      { -INT32_C(  1051699169),  INT32_C(   460990193), -INT32_C(   361098987),  INT32_C(  1574047846), -INT32_C(  1917350712),  INT32_C(  2120197224),  INT32_C(  1471703585), -INT32_C(  1465387128),
         INT32_C(  1097463887),  INT32_C(   861725470),  INT32_C(  1528748021), -INT32_C(  1346834201), -INT32_C(   449024132),  INT32_C(   895720212), -INT32_C(  1903420666),  INT32_C(  1647719442) },
      {  INT32_C(  1252237356),  INT32_C(  2021523331), -INT32_C(  1110205482),  INT32_C(   141331595) },
      { -INT32_C(   850997113), -INT32_C(    44823098),  INT32_C(   209841317),  INT32_C(  1848434572), -INT32_C(   463968950),  INT32_C(  1758026917),  INT32_C(   115696998),  INT32_C(  1002563252),
         INT32_C(    70776615), -INT32_C(  2097231967), -INT32_C(    87622839), -INT32_C(  1840005653),  INT32_C(  2050601782), -INT32_C(   188187195),  INT32_C(  1974467753), -INT32_C(  1883434256) } },
    { {  INT32_C(   267233531),  INT32_C(  1246056516),  INT32_C(  2128138603),  INT32_C(   836767493),  INT32_C(   846955439),  INT32_C(  1504442754),  INT32_C(   538345108),  INT32_C(   103318282),
         INT32_C(  1863652651), -INT32_C(   793159067),  INT32_C(   827232812),  INT32_C(  1348677281),  INT32_C(   880991921),  INT32_C(  1821191639), -INT32_C(  1232297044),  INT32_C(  1388098598) },
      UINT16_C(53961),
      {  INT32_C(  2066493121),  INT32_C(  1292720382), -INT32_C(   310661494), -INT32_C(  2134168066), -INT32_C(   290544799), -INT32_C(  1684973041), -INT32_C(   850413552), -INT32_C(   878766070),
         INT32_C(  1145490246), -INT32_C(  1366207708), -INT32_C(     6615807), -INT32_C(  1686084038),  INT32_C(   411643401), -INT32_C(  1733092728), -INT32_C(   597359918),  INT32_C(  1621623834) },
      { -INT32_C(   190452016),  INT32_C(  1134704194),  INT32_C(  2101558595), -INT32_C(  1390885981),  INT32_C(  2042995185), -INT32_C(  1911457348), -INT32_C(  1771341957),  INT32_C(  1274418043),
         INT32_C(  1128241921),  INT32_C(   361226706), -INT32_C(  1030501858),  INT32_C(  2121247629),  INT32_C(   167195725),  INT32_C(   697764013), -INT32_C(    71368065),  INT32_C(   390509845) },
      {  INT32_C(   593135185), -INT32_C(  2043092633),  INT32_C(   944294827), -INT32_C(   994658441), -INT32_C(  1714573845),  INT32_C(   901932214),  INT32_C(  2083553639), -INT32_C(  2020378826),
         INT32_C(  1688923901),  INT32_C(  2078991311),  INT32_C(   632500910), -INT32_C(   706123031), -INT32_C(   848382441), -INT32_C(  2113720293), -INT32_C(   402771023), -INT32_C(  1485860182) },
      {  INT32_C(  1326193024), -INT32_C(  1412761860),  INT32_C(   315653673), -INT32_C(     1590552), -INT32_C(  1949477520),  INT32_C(   906875013), -INT32_C(  1373762557),  INT32_C(   508923294),
        -INT32_C(  1553112666), -INT32_C(  2125580200), -INT32_C(  1634525514),  INT32_C(  1235057624),  INT32_C(  1456761552),  INT32_C(  1049420346), -INT32_C(  1930646802), -INT32_C(   559267529) },
      { -INT32_C(    75425885),  INT32_C(    92065615), -INT32_C(   962392083),  INT32_C(  1527726474) },
      {  INT32_C(  1477176086),  INT32_C(  1246056516),  INT32_C(  2128138603), -INT32_C(   826516308),  INT32_C(   846955439),  INT32_C(  1504442754),  INT32_C(   601616227),  INT32_C(   697946240),
         INT32_C(  1863652651),  INT32_C(   369833062),  INT32_C(   827232812),  INT32_C(  1348677281),  INT32_C(   219675344),  INT32_C(  1821191639), -INT32_C(  1536783177),  INT32_C(  1782741980) } },
    { { -INT32_C(   424549461), -INT32_C(  1289470523),  INT32_C(   524226792), -INT32_C(   184686255),  INT32_C(  1357872641),  INT32_C(   995519309),  INT32_C(    84015483), -INT32_C(   429912006),
        -INT32_C(  1177808397),  INT32_C(   929886287),  INT32_C(  1364634624), -INT32_C(  1773775723),  INT32_C(   552023506),  INT32_C(   475741601),  INT32_C(  1898011702),  INT32_C(  1599570540) },
      UINT16_C( 9107),
      { -INT32_C(  2062294504),  INT32_C(  1882264345), -INT32_C(  1429944732),  INT32_C(  1155569245),  INT32_C(   293700022), -INT32_C(  1100105827),  INT32_C(  2134956328),  INT32_C(  1369625657),
        -INT32_C(   791235146),  INT32_C(   759171016),  INT32_C(   735511758),  INT32_C(  1366275994), -INT32_C(   714936264), -INT32_C(   795553625), -INT32_C(   498084695),  INT32_C(  1597305512) },
      {  INT32_C(  1882131111), -INT32_C(   526553326),  INT32_C(   218854515),  INT32_C(  1683913260),  INT32_C(   289063274),  INT32_C(   987877009),  INT32_C(  1260204450), -INT32_C(   894807773),
         INT32_C(  1832573275), -INT32_C(  1152526520),  INT32_C(  2026395724),  INT32_C(  1021061074),  INT32_C(  2035160808), -INT32_C(  2018299932), -INT32_C(  2083336096),  INT32_C(  2102295585) },
      { -INT32_C(  1645574059), -INT32_C(  1403504544),  INT32_C(  1663312016),  INT32_C(   798949447), -INT32_C(    72815337),  INT32_C(  2105695261),  INT32_C(  1308709932),  INT32_C(   634081232),
         INT32_C(   935507415),  INT32_C(  2128812781), -INT32_C(  2099181765),  INT32_C(   531791880), -INT32_C(  1961207186), -INT32_C(   485974602), -INT32_C(  1036973583),  INT32_C(   820509784) },
      { -INT32_C(  1620596047), -INT32_C(    14857276),  INT32_C(  1518534226), -INT32_C(   327601026),  INT32_C(  1165464718),  INT32_C(   573079601), -INT32_C(   488351351),  INT32_C(   101895253),
         INT32_C(   983923062),  INT32_C(   389726916),  INT32_C(  1064418496),  INT32_C(  2116807664), -INT32_C(  1329355905), -INT32_C(  1395463389), -INT32_C(  1718700220), -INT32_C(   106913661) },
      { -INT32_C(   567065318), -INT32_C(   923439864),  INT32_C(   419915561), -INT32_C(   778554542) },
      {  INT32_C(  1442012917),  INT32_C(   383363578),  INT32_C(   524226792), -INT32_C(   184686255), -INT32_C(   691122320),  INT32_C(   995519309),  INT32_C(    84015483), -INT32_C(   917233015),
        -INT32_C(  1498127163), -INT32_C(   569990241),  INT32_C(  1364634624), -INT32_C(  1773775723),  INT32_C(   552023506),  INT32_C(  2040540073),  INT32_C(  1898011702),  INT32_C(  1599570540) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512i src = simde_mm512_loadu_epi32(test_vec[i].src);
    simde__m512i a0 = simde_mm512_loadu_epi32(test_vec[i].a0);
    simde__m512i a1 = simde_mm512_loadu_epi32(test_vec[i].a1);
    simde__m512i a2 = simde_mm512_loadu_epi32(test_vec[i].a2);
    simde__m512i a3 = simde_mm512_loadu_epi32(test_vec[i].a3);
    simde__m128i b = simde_x_mm_loadu_epi32(test_vec[i].b);
    simde__m512i r = simde_mm512_mask_4dpwssd_epi32(src, test_vec[i].k, a0, a1, a2, a3, &b);
    simde_test_x86_assert_equal_i32x16(r, simde_mm512_loadu_epi32(test_vec[i].r));
  }

  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde__m512i src = simde_test_x86_random_i32x16();
    simde__mmask16 k = simde_test_x86_random_mmask16();
    simde__m512i a0 = simde_test_x86_random_i32x16();
    simde__m512i a1 = simde_test_x86_random_i32x16();
    simde__m512i a2 = simde_test_x86_random_i32x16();
    simde__m512i a3 = simde_test_x86_random_i32x16();
    simde__m128i b = simde_test_x86_random_i32x4();
    simde__m512i r = simde_mm512_mask_4dpwssd_epi32(src, k, a0, a1, a2, a3, &b);

    simde_test_x86_write_i32x16(2, src, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_x86_write_mmask16(2, k, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i32x16(2, a0, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i32x16(2, a1, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i32x16(2, a2, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i32x16(2, a3, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i32x4(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i32x16(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_mm512_maskz_4dpwssd_epi32 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const simde__mmask16 k;
    const int32_t src[16];
    const int32_t a0[16];
    const int32_t a1[16];
    const int32_t a2[16];
    const int32_t a3[16];
    const int32_t b[4];
    const int32_t r[16];
  } test_vec[] = {
    { UINT16_C(23510),
      {  INT32_C(  1397225857),  INT32_C(   873106341), -INT32_C(   975925979), -INT32_C(  1173688698), -INT32_C(  1037626675), -INT32_C(   500608549),  INT32_C(    34962282), -INT32_C(   849417396),
        -INT32_C(  1977506332),  INT32_C(  1438526256),  INT32_C(  1058706105),  INT32_C(  1324950913),  INT32_C(   303046967), -INT32_C(   587908494),  INT32_C(    48171701), -INT32_C(   623952394) },
      {  INT32_C(   308605154), -INT32_C(   714595812),  INT32_C(   907313845), -INT32_C(   544993625), -INT32_C(  1578003409), -INT32_C(  2072123698), -INT32_C(   410624783),  INT32_C(  2076267929),
         INT32_C(  1653483078), -INT32_C(    46664120),  INT32_C(   540232568), -INT32_C(  1996507303),  INT32_C(   438956108), -INT32_C(   929126441), -INT32_C(  1649466365), -INT32_C(  1072140166) },
      { -INT32_C(   551377258),  INT32_C(   366762396), -INT32_C(    30076764),  INT32_C(   327562439), -INT32_C(    80826332),  INT32_C(  1522781271),  INT32_C(  1811378929),  INT32_C(  2032865507),
         INT32_C(  1398295990),  INT32_C(  1248343462),  INT32_C(   206085445), -INT32_C(   182399023), -INT32_C(   688828801),  INT32_C(   187741210),  INT32_C(   175515687), -INT32_C(   293363400) },
      { -INT32_C(  1807622930),  INT32_C(  1457498385),  INT32_C(   392308550),  INT32_C(  1963819766), -INT32_C(   347341104), -INT32_C(   638157646), -INT32_C(   589075036), -INT32_C(    37066994),
         INT32_C(  1418792003), -INT32_C(    55938891), -INT32_C(  1911354216),  INT32_C(  1594105999), -INT32_C(   783659234),  INT32_C(  1873428939), -INT32_C(  1135898962),  INT32_C(   951654133) },
      { -INT32_C(   678671582),  INT32_C(  1406351035), -INT32_C(   773658814),  INT32_C(   640804359),  INT32_C(    32996149),  INT32_C(  1785766332),  INT32_C(   606583855), -INT32_C(   195239726),
        -INT32_C(   406067157),  INT32_C(  1631231774), -INT32_C(  1926095738),  INT32_C(   951280386), -INT32_C(  1690719521),  INT32_C(  2064034124),  INT32_C(   933244261),  INT32_C(   959183885) },
      {  INT32_C(    52492261),  INT32_C(   476338838),  INT32_C(  2041157495), -INT32_C(   642687494) },
      {  INT32_C(           0),  INT32_C(  1158094231),  INT32_C(   445124539),  INT32_C(           0),  INT32_C(   296320555),  INT32_C(           0), -INT32_C(   570813021),  INT32_C(   316426631),
         INT32_C(  1800425769),  INT32_C(   281146543),  INT32_C(           0),  INT32_C(  1714718027),  INT32_C(   429761665),  INT32_C(           0),  INT32_C(   787682556),  INT32_C(           0) } },
    { UINT16_C(59911),
      {  INT32_C(  2073318261),  INT32_C(  1873344975),  INT32_C(  1550562609),  INT32_C(   240341230), -INT32_C(  1201018540), -INT32_C(  1353719803),  INT32_C(   185354841), -INT32_C(  1728769245),
         INT32_C(   907250023), -INT32_C(  1264206973),  INT32_C(  1594888305), -INT32_C(  1251056543),  INT32_C(  1399707469),  INT32_C(   268615095),  INT32_C(   706416135), -INT32_C(  2000482015) },
      {  INT32_C(   499046042),  INT32_C(    64054162), -INT32_C(   714939788), -INT32_C(  1819619258),  INT32_C(  1592194983), -INT32_C(  1150293836),  INT32_C(   417762038),  INT32_C(   916498843),
         INT32_C(   290676607),  INT32_C(   907290050),  INT32_C(  1292596999), -INT32_C(   270494393),  INT32_C(  1095616397), -INT32_C(  1493320529), -INT32_C(   490806457),  INT32_C(   186146444) },
      { -INT32_C(  2145555267), -INT32_C(  1732890223), -INT32_C(   253377880), -INT32_C(   455097001),  INT32_C(  1009134732),  INT32_C(   820126697), -INT32_C(  1844273146), -INT32_C(  1130550274),
         INT32_C(   675068567), -INT32_C(  1799294228),  INT32_C(   210019764), -INT32_C(   135240854),  INT32_C(  2033391248),  INT32_C(  1068045625), -INT32_C(  1278100555),  INT32_C(  2104455142) },
      {  INT32_C(   363178793),  INT32_C(  1386898846),  INT32_C(  1969106442),  INT32_C(   577523602), -INT32_C(  1617191067),  INT32_C(  1776174260), -INT32_C(   434327297),  INT32_C(  1214483487),
        -INT32_C(   715257545),  INT32_C(  2032666734), -INT32_C(   923892170),  INT32_C(  1005214421), -INT32_C(  1378187783), -INT32_C(   921257783), -INT32_C(  2018561432), -INT32_C(   154135874) },
      { -INT32_C(  1966395877),  INT32_C(  1812198198),  INT32_C(  1328935290),  INT32_C(  1149902667),  INT32_C(  1861313701), -INT32_C(  2059991267), -INT32_C(   133306823),  INT32_C(   351198713),
         INT32_C(  1100921099),  INT32_C(   665690541), -INT32_C(   579411310), -INT32_C(  1491009278), -INT32_C(  2112548251),  INT32_C(  1393052953),  INT32_C(   743118131), -INT32_C(    46056974) },
      { -INT32_C(  1623269390),  INT32_C(   331803521), -INT32_C(   789496370), -INT32_C(  1552477634) },
      { -INT32_C(   354560687), -INT32_C(  2044013734),  INT32_C(  1629227175),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),
         INT32_C(           0), -INT32_C(  1998392527),  INT32_C(           0),  INT32_C(   196489071),  INT32_C(           0),  INT32_C(  1303841081), -INT32_C(   898466369),  INT32_C(   521458675) } },
    { UINT16_C(35877),
      {  INT32_C(   785989158), -INT32_C(   599585391),  INT32_C(  2048210233),  INT32_C(  1884948530),  INT32_C(  1851513767), -INT32_C(   542430738),  INT32_C(  1894967801), -INT32_C(  1292036212),
        -INT32_C(   404695467),  INT32_C(   499327971), -INT32_C(  1969759912), -INT32_C(  1996819999), -INT32_C(  1141483828),  INT32_C(  2040177023),  INT32_C(   401181834), -INT32_C(   104208733) },
      { -INT32_C(  1595888963),  INT32_C(   649962445),  INT32_C(  1571837308),  INT32_C(   333884230), -INT32_C(  2133926911),  INT32_C(   150562942), -INT32_C(  1742740491), -INT32_C(  2037257783),
         INT32_C(  1613197715), -INT32_C(  1853430763), -INT32_C(  2131806407), -INT32_C(   476850718),  INT32_C(   795042225), -INT32_C(  1103602231),  INT32_C(   156718912), -INT32_C(   745478080) },
      {  INT32_C(  1865725786), -INT32_C(   721372517), -INT32_C(   749342735),  INT32_C(  1991698628),  INT32_C(   312810057), -INT32_C(  1210983049),  INT32_C(  1975527477),  INT32_C(  1783189520),
        -INT32_C(  1546027769),  INT32_C(   695786039), -INT32_C(  1896034871), -INT32_C(    33246283),  INT32_C(  1141942733), -INT32_C(  1124343417),  INT32_C(   439466761),  INT32_C(   327449100) },
      {  INT32_C(   800481015),  INT32_C(    22556216), -INT32_C(  1332783877), -INT32_C(   726756601), -INT32_C(  1005011139), -INT32_C(  1434446944), -INT32_C(   607866417),  INT32_C(   602818604),
        -INT32_C(   565008986), -INT32_C(   807359789),  INT32_C(   109015039),  INT32_C(  1088105987), -INT32_C(  1929055251), -INT32_C(   701004793),  INT32_C(  1639054133), -INT32_C(   360341692) },
      {  INT32_C(   415881028), -INT32_C(  2115524222),  INT32_C(   461858328), -INT32_C(  2124717420),  INT32_C(  1561288534),  INT32_C(   422790883), -INT32_C(  2038766526), -INT32_C(   932118653),
         INT32_C(  1507867095), -INT32_C(    69548062), -INT32_C(  1038720723),  INT32_C(   440627908), -INT32_C(  1250471215), -INT32_C(   607212903),  INT32_C(   291588238),  INT32_C(   534368584) },
      { -INT32_C(   310855413), -INT32_C(  1360506240),  INT32_C(  2003894195),  INT32_C(  1116844913) },
      { -INT32_C(   938695927),  INT32_C(           0), -INT32_C(  1773481120),  INT32_C(           0),  INT32_C(           0),  INT32_C(   909969848),  INT32_C(           0),  INT32_C(           0),
         INT32_C(           0),  INT32_C(           0),  INT32_C(   228520683),  INT32_C(   476577442),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(   510841049) } },
    { UINT16_C( 2054),
      { -INT32_C(   978149385), -INT32_C(   619822982),  INT32_C(   732714578), -INT32_C(   303712395), -INT32_C(  1925225051), -INT32_C(  2087914989), -INT32_C(    80216726),  INT32_C(   923024704),
         INT32_C(  1459467996),  INT32_C(  1227951095), -INT32_C(   696984223),  INT32_C(   985880981), -INT32_C(   758709569),  INT32_C(  1582716148), -INT32_C(  1856336559), -INT32_C(  1496752438) },
      {  INT32_C(   201115156),  INT32_C(   844377553), -INT32_C(  1626814198), -INT32_C(   505820382), -INT32_C(  1045126707),  INT32_C(  1176505077),  INT32_C(  1641577111), -INT32_C(   335044136),
         INT32_C(   955777895),  INT32_C(   980044848),  INT32_C(   953840149),  INT32_C(   169456445),  INT32_C(  1238093140),  INT32_C(  1854991319),  INT32_C(  1037002853),  INT32_C(  1881855497) },
      {  INT32_C(   162013913), -INT32_C(  2075913618), -INT32_C(  1044636028),  INT32_C(   650892753),  INT32_C(  2054133411), -INT32_C(   404095102),  INT32_C(  1881520231),  INT32_C(  1759530895),
        -INT32_C(   546142095),  INT32_C(   509851290), -INT32_C(  1512103980), -INT32_C(  1731482891), -INT32_C(  1022215359), -INT32_C(  1565852870),  INT32_C(  1125306292), -INT32_C(  1884556514) },
      {  INT32_C(   376380795), -INT32_C(  1489710381), -INT32_C(   414378766), -INT32_C(     8447810), -INT32_C(  1899851437),  INT32_C(  1093692813),  INT32_C(  1535394364), -INT32_C(  1310052555),
         INT32_C(   549935437),  INT32_C(   516487980), -INT32_C(   838527729), -INT32_C(  2117237715), -INT32_C(  1576038379),  INT32_C(   971194365), -INT32_C(  1215010943), -INT32_C(   462913641) },
      {  INT32_C(    67383256),  INT32_C(   975359018),  INT32_C(   252192737), -INT32_C(  1047472725),  INT32_C(  1650696037),  INT32_C(  1604077534),  INT32_C(  1159082158), -INT32_C(  2010546513),
        -INT32_C(   661901651), -INT32_C(   602755078), -INT32_C(  2098521386),  INT32_C(  1430485999), -INT32_C(   122182118), -INT32_C(  1671998483),  INT32_C(   870411908), -INT32_C(  1699017748) },
      {  INT32_C(   863127609), -INT32_C(   854620937), -INT32_C(  1924138338), -INT32_C(  1880976779) },
      {  INT32_C(           0), -INT32_C(  1199910029),  INT32_C(  2087129462),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),
         INT32_C(           0),  INT32_C(           0),  INT32_C(           0), -INT32_C(   174400579),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0) } },
    { UINT16_C(39481),
      { -INT32_C(   538106233), -INT32_C(  1538428478),  INT32_C(  1622096293),  INT32_C(  1168697555),  INT32_C(   734633755), -INT32_C(  1138399379), -INT32_C(   682648587), -INT32_C(  1301182422),
         INT32_C(  1905352622),  INT32_C(  1964367568), -INT32_C(   338246633), -INT32_C(   953123156), -INT32_C(  1947010530),  INT32_C(  1447499873), -INT32_C(   584149325), -INT32_C(   846225634) },
      { -INT32_C(   818011906),  INT32_C(   373576702), -INT32_C(  1023337961), -INT32_C(  1215680103), -INT32_C(  1925022421),  INT32_C(  1222871445),  INT32_C(  1042616608), -INT32_C(  1358187087),
        -INT32_C(   729921067), -INT32_C(  1276460132),  INT32_C(  1987505117),  INT32_C(  1210908956),  INT32_C(   332754814),  INT32_C(   425441529),  INT32_C(  2069332170),  INT32_C(   187327285) },
      {  INT32_C(  1239394732),  INT32_C(  1241303404), -INT32_C(   792693836), -INT32_C(   233247372),  INT32_C(  1443229021),  INT32_C(  1886347430),  INT32_C(   384550624), -INT32_C(   702474967),
         INT32_C(   706674878),  INT32_C(  2104761289),  INT32_C(    38614158),  INT32_C(  2129946145), -INT32_C(   103483053),  INT32_C(   979977049),  INT32_C(   860902410),  INT32_C(   671707498) },
      {  INT32_C(   978528369), -INT32_C(   759707836),  INT32_C(   483722491), -INT32_C(  1097086614),  INT32_C(   481783747), -INT32_C(  1118428749), -INT32_C(   537876875), -INT32_C(  2012677609),
         INT32_C(  1724013346),  INT32_C(   490305826), -INT32_C(   398913922), -INT32_C(  1700342569), -INT32_C(   138977724), -INT32_C(   189526657), -INT32_C(   875256652), -INT32_C(  1051403106) },
      {  INT32_C(  1495733815),  INT32_C(   242638992),  INT32_C(  1190571886), -INT32_C(   924803708),  INT32_C(  2059376635),  INT32_C(  1500410789), -INT32_C(  1239137769),  INT32_C(  1433892894),
         INT32_C(   531537551),  INT32_C(  1831675135),  INT32_C(  1488135124), -INT32_C(  1155492672), -INT32_C(   801775829),  INT32_C(  1764336722),  INT32_C(    85937894),  INT32_C(  1432000198) },
      {  INT32_C(   880019765),  INT32_C(    27369773), -INT32_C(  2057742907),  INT32_C(   339835369) },
      { -INT32_C(  1701437429),  INT32_C(           0),  INT32_C(           0), -INT32_C(   366754954), -INT32_C(  1821080743),  INT32_C(  1099787046),  INT32_C(           0),  INT32_C(           0),
         INT32_C(           0), -INT32_C(   303243172),  INT32_C(           0), -INT32_C(   266084822),  INT32_C(  1248880520),  INT32_C(           0),  INT32_C(           0), -INT32_C(   114760546) } },
    { UINT16_C(30296),
      {  INT32_C(   236628709),  INT32_C(   861667604),  INT32_C(  1623859974), -INT32_C(   311820424), -INT32_C(   728852685), -INT32_C(   232172648),  INT32_C(   443224793),  INT32_C(   210813991),
        -INT32_C(  2112181394), -INT32_C(  1296664660),  INT32_C(   303202458), -INT32_C(  1291879297), -INT32_C(  1417245165), -INT32_C(  1164070687), -INT32_C(   371979838),  INT32_C(  1005938125) },
      { -INT32_C(  1128394736),  INT32_C(   560886919),  INT32_C(  1932755188),  INT32_C(   254096380), -INT32_C(  1564759103),  INT32_C(   509433947),  INT32_C(   772223329), -INT32_C(  1502937706),
        -INT32_C(  1805506547), -INT32_C(  1867132772),  INT32_C(  1292101713), -INT32_C(   597874661),  INT32_C(   796858579), -INT32_C(   766649232), -INT32_C(  1543482355),  INT32_C(  1581935185) },
      {  INT32_C(   787656082), -INT32_C(   826366083), -INT32_C(  1424178800), -INT32_C(  1115129367),  INT32_C(    32245649), -INT32_C(   254592541), -INT32_C(   543894387), -INT32_C(   784474306),
         INT32_C(   167719052),  INT32_C(  1742257879),  INT32_C(  1762915455), -INT32_C(    31024275), -INT32_C(  2046881118), -INT32_C(   663366837), -INT32_C(   440989017),  INT32_C(  1974924777) },
      { -INT32_C(    58804699), -INT32_C(   211528076), -INT32_C(  1218676918), -INT32_C(  1263173102), -INT32_C(   549866092),  INT32_C(   800632456), -INT32_C(  1592496200), -INT32_C(  1994994844),
        -INT32_C(   175729279),  INT32_C(   921234155),  INT32_C(  1944929633),  INT32_C(  1546101703), -INT32_C(   532979624), -INT32_C(   955256050), -INT32_C(   949476509),  INT32_C(  1867612142) },
      { -INT32_C(    10168556),  INT32_C(   573918657),  INT32_C(  1502946194),  INT32_C(   515226822),  INT32_C(   721350940),  INT32_C(  1206980068),  INT32_C(   521099824), -INT32_C(   309436199),
        -INT32_C(   118688969), -INT32_C(   770039232),  INT32_C(   187412293), -INT32_C(  2027298453), -INT32_C(  1229903918),  INT32_C(  1694409524), -INT32_C(   696054275), -INT32_C(  1530719635) },
      {  INT32_C(  1167896325),  INT32_C(   370652881), -INT32_C(   803126683), -INT32_C(   178828765) },
      {  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(  1500144077),  INT32_C(  1044287997),  INT32_C(           0), -INT32_C(  1212303157),  INT32_C(           0),
         INT32_C(           0), -INT32_C(   750556884),  INT32_C(  1435262483),  INT32_C(           0), -INT32_C(   791503062), -INT32_C(   622987711), -INT32_C(  2014352988),  INT32_C(           0) } },
    { UINT16_C( 2161),
      { -INT32_C(  1431591508), -INT32_C(  1917343734),  INT32_C(  1100948606),  INT32_C(  1693557960),  INT32_C(     1753833), -INT32_C(    96239655), -INT32_C(  1505401265),  INT32_C(   145667676),
         INT32_C(  1722964572), -INT32_C(  2114754302),  INT32_C(  1438815117),  INT32_C(   565818167),  INT32_C(  1327616886), -INT32_C(  1589025710),  INT32_C(   659001035), -INT32_C(  1590692539) },
      {  INT32_C(  1376248143), -INT32_C(   673973686), -INT32_C(   986933875), -INT32_C(  1075387063),  INT32_C(   168691640),  INT32_C(   933975916),  INT32_C(   727642854),  INT32_C(   936152551),
        -INT32_C(  1198926994),  INT32_C(  1536122061),  INT32_C(   975223793),  INT32_C(  1492715168),  INT32_C(  2036467469),  INT32_C(  1169165663), -INT32_C(   428863745),  INT32_C(   152911003) },
      { -INT32_C(   591288817), -INT32_C(   197701630), -INT32_C(  1423026421),  INT32_C(  1795369053), -INT32_C(  1897634513),  INT32_C(  1909691762),  INT32_C(  1062749091), -INT32_C(  1907853953),
         INT32_C(   510331420),  INT32_C(  1712497242),  INT32_C(  1460748794), -INT32_C(  1732110999), -INT32_C(   316168326), -INT32_C(   547423684), -INT32_C(  1122060738),  INT32_C(  1212966700) },
      { -INT32_C(   882460815),  INT32_C(  1395751257),  INT32_C(   598426554), -INT32_C(   759468712),  INT32_C(  1354752532),  INT32_C(   456138461),  INT32_C(    14175956),  INT32_C(   642262197),
         INT32_C(   905097179), -INT32_C(   494394584), -INT32_C(  1106955418), -INT32_C(  1248739167), -INT32_C(  2147069789),  INT32_C(  1134245486),  INT32_C(   977499012),  INT32_C(  1935707288) },
      {  INT32_C(  1671975483), -INT32_C(   599445130),  INT32_C(    94063204), -INT32_C(  1380242422), -INT32_C(   349322884),  INT32_C(  2066663671), -INT32_C(   726306500),  INT32_C(   944183037),
        -INT32_C(   560205720), -INT32_C(  2051284959),  INT32_C(   898324010), -INT32_C(    18725246), -INT32_C(    18280441),  INT32_C(   343480280), -INT32_C(  2031603831), -INT32_C(  1379979195) },
      {  INT32_C(  1099717152),  INT32_C(  1707493179),  INT32_C(   530207133), -INT32_C(  1642234473) },
      {  INT32_C(  1920187037),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0), -INT32_C(   560048160), -INT32_C(   389506256),  INT32_C(   753591262),  INT32_C(           0),
         INT32_C(           0),  INT32_C(           0),  INT32_C(           0), -INT32_C(  2104637306),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0) } },
    { UINT16_C( 1933),
      {  INT32_C(   371090844),  INT32_C(  1648732026), -INT32_C(   309163474), -INT32_C(  1018711497), -INT32_C(  1173716237), -INT32_C(  2113165336), -INT32_C(   453008442), -INT32_C(   554988479),
         INT32_C(  1827932658), -INT32_C(   540132943), -INT32_C(    87268925),  INT32_C(   129897235),  INT32_C(  2126629014),  INT32_C(   906022000), -INT32_C(  1323696017),  INT32_C(  2123302284) },
      { -INT32_C(  1058372849),  INT32_C(  2141174204),  INT32_C(   762931994),  INT32_C(   338966398),  INT32_C(  1888679168),  INT32_C(   833000386),  INT32_C(   534954387), -INT32_C(   711036474),
        -INT32_C(  1332377356),  INT32_C(  1529820225),  INT32_C(   512338335), -INT32_C(   533545504),  INT32_C(  1968227763), -INT32_C(   341379240),  INT32_C(  2114619832), -INT32_C(   279664390) },
      {  INT32_C(  1923082545), -INT32_C(  1110520034),  INT32_C(  1490769784), -INT32_C(   952562156),  INT32_C(   725387731),  INT32_C(   941024128),  INT32_C(  1723277420), -INT32_C(    95089719),
         INT32_C(   309130740),  INT32_C(  1020279492), -INT32_C(  1500206191), -INT32_C(  1938960967), -INT32_C(   675829161), -INT32_C(   116339315), -INT32_C(  1218459666), -INT32_C(   961432110) },
      {  INT32_C(  1859722666), -INT32_C(   374691496),  INT32_C(   244268628),  INT32_C(  1671101708),  INT32_C(   876237479),  INT32_C(   221071903), -INT32_C(   473658351), -INT32_C(   341150399),
        -INT32_C(   346455149), -INT32_C(  2133589204),  INT32_C(  1317954626),  INT32_C(   145828193), -INT32_C(  1707283589),  INT32_C(  1202219318),  INT32_C(   925592821),  INT32_C(  1965217250) },
      { -INT32_C(  2074051496), -INT32_C(  1056689025), -INT32_C(   116419688),  INT32_C(   922861756), -INT32_C(   489603668), -INT32_C(  1675003482), -INT32_C(   925674266), -INT32_C(  2126645975),
        -INT32_C(   251290255),  INT32_C(  1790053073),  INT32_C(  1499710109),  INT32_C(   781215106),  INT32_C(  1225810338), -INT32_C(  1041941797), -INT32_C(  1198933874),  INT32_C(   523880109) },
      {  INT32_C(   873480035), -INT32_C(   425803191),  INT32_C(   121569924),  INT32_C(   154521447) },
      {  INT32_C(   238588751),  INT32_C(           0), -INT32_C(   779908484),  INT32_C(   838578599),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0), -INT32_C(  1962461513),
        -INT32_C(  1759180061), -INT32_C(  2001127192),  INT32_C(   189807726),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512i src = simde_mm512_loadu_epi32(test_vec[i].src);
    simde__m512i a0 = simde_mm512_loadu_epi32(test_vec[i].a0);
    simde__m512i a1 = simde_mm512_loadu_epi32(test_vec[i].a1);
    simde__m512i a2 = simde_mm512_loadu_epi32(test_vec[i].a2);
    simde__m512i a3 = simde_mm512_loadu_epi32(test_vec[i].a3);
    simde__m128i b = simde_x_mm_loadu_epi32(test_vec[i].b);
    simde__m512i r = simde_mm512_maskz_4dpwssd_epi32(test_vec[i].k, src, a0, a1, a2, a3, &b);
    simde_test_x86_assert_equal_i32x16(r, simde_mm512_loadu_epi32(test_vec[i].r));
  }

  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde__mmask16 k = simde_test_x86_random_mmask16();
    simde__m512i src = simde_test_x86_random_i32x16();
    simde__m512i a0 = simde_test_x86_random_i32x16();
    simde__m512i a1 = simde_test_x86_random_i32x16();
    simde__m512i a2 = simde_test_x86_random_i32x16();
    simde__m512i a3 = simde_test_x86_random_i32x16();
    simde__m128i b = simde_test_x86_random_i32x4();
    simde__m512i r = simde_mm512_maskz_4dpwssd_epi32(k, src, a0, a1, a2, a3, &b);

    simde_test_x86_write_mmask16(2, k, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_x86_write_i32x16(2, src, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i32x16(2, a0, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i32x16(2, a1, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i32x16(2, a2, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i32x16(2, a3, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i32x4(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i32x16(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

SIMDE_TEST_FUNC_LIST_BEGIN
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_4dpwssd_epi32)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_mask_4dpwssd_epi32)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_maskz_4dpwssd_epi32)
SIMDE_TEST_FUNC_LIST_END

#include <test/x86/avx512/test-avx512-footer.h>
