use super::Element;

impl Element {
    /// Returns the location (country, in most cases) where the element was discovered, if known.
    /// There can be multiple locations if it was an international effort or if multiple teams
    /// discovered or isolated the element independently.
    ///
    /// ```
    /// use mendeleev::Element;
    ///
    /// assert_eq!(Element::H.discovery_location(), Some(["England"].as_slice()));
    /// assert_eq!(Element::He.discovery_location(), Some(["Scotland", "Sweden"].as_slice()));
    /// assert_eq!(Element::Og.discovery_location(), Some(["Russia"].as_slice()));
    /// assert_eq!(Element::Au.discovery_location(), None);
    /// ```
    pub const fn discovery_location(&self) -> Option<&'static [&'static str]> {
        use Element as E;
        match self {
            E::H => Some(&["England"]),
            E::He => Some(&["Scotland", "Sweden"]),
            E::Li => Some(&["Sweden"]),
            E::Be => Some(&["Germany", "France"]),
            E::B => Some(&["England", "France"]),
            E::C => None,
            E::N => Some(&["Scotland"]),
            E::O => Some(&["England", "Sweden"]),
            E::F => Some(&["France"]),
            E::Ne => Some(&["England"]),
            E::Na => Some(&["England"]),
            E::Mg => Some(&["England"]),
            E::Al => Some(&["Denmark"]),
            E::Si => Some(&["Sweden"]),
            E::P => Some(&["Germany"]),
            E::S => None,
            E::Cl => Some(&["Sweden"]),
            E::Ar => Some(&["Scotland"]),
            E::K => Some(&["England"]),
            E::Ca => Some(&["England"]),
            E::Sc => Some(&["Sweden"]),
            E::Ti => Some(&["England"]),
            E::V => Some(&["Sweden"]),
            E::Cr => Some(&["France"]),
            E::Mn => Some(&["Sweden"]),
            E::Fe => None,
            E::Co => Some(&["Sweden"]),
            E::Ni => Some(&["Sweden"]),
            E::Cu => None,
            E::Zn => None,
            E::Ga => Some(&["France"]),
            E::Ge => Some(&["Germany"]),
            E::As => None,
            E::Se => Some(&["Sweden"]),
            E::Br => Some(&["France"]),
            E::Kr => Some(&["Great Britain"]),
            E::Rb => Some(&["Germany"]),
            E::Sr => Some(&["Scotland"]),
            E::Y => Some(&["Finland"]),
            E::Zr => Some(&["Germany"]),
            E::Nb => Some(&["England"]),
            E::Mo => Some(&["Sweden"]),
            E::Tc => Some(&["Italy"]),
            E::Ru => Some(&["Russia"]),
            E::Rh => Some(&["England"]),
            E::Pd => Some(&["England"]),
            E::Ag => None,
            E::Cd => Some(&["Germany"]),
            E::In => Some(&["Germany"]),
            E::Sn => None,
            E::Sb => None,
            E::Te => Some(&["Romania"]),
            E::I => Some(&["France"]),
            E::Xe => Some(&["England"]),
            E::Cs => Some(&["Germany"]),
            E::Ba => Some(&["England"]),
            E::La => Some(&["Sweden"]),
            E::Ce => Some(&["Sweden", "Germany"]),
            E::Pr => Some(&["Austria"]),
            E::Nd => Some(&["Austria"]),
            E::Pm => Some(&["United States"]),
            E::Sm => Some(&["France"]),
            E::Eu => Some(&["France"]),
            E::Gd => Some(&["Switzerland"]),
            E::Tb => Some(&["Sweden"]),
            E::Dy => Some(&["France"]),
            E::Ho => Some(&["Switzerland"]),
            E::Er => Some(&["Sweden"]),
            E::Tm => Some(&["Sweden"]),
            E::Yb => Some(&["Switzerland"]),
            E::Lu => Some(&["France"]),
            E::Hf => Some(&["Denmark"]),
            E::Ta => Some(&["Sweden"]),
            E::W => Some(&["Spain"]),
            E::Re => Some(&["Germany"]),
            E::Os => Some(&["England"]),
            E::Ir => Some(&["England", "France"]),
            E::Pt => Some(&["Italy"]),
            E::Au => None,
            E::Hg => None,
            E::Tl => Some(&["England"]),
            E::Pb => None,
            E::Bi => None,
            E::Po => Some(&["France"]),
            E::At => Some(&["United States"]),
            E::Rn => Some(&["Germany"]),
            E::Fr => Some(&["France"]),
            E::Ra => Some(&["France"]),
            E::Ac => Some(&["France"]),
            E::Th => Some(&["Sweden"]),
            E::Pa => Some(&["England", "France"]),
            E::U => Some(&["Germany"]),
            E::Np => Some(&["United States"]),
            E::Pu => Some(&["United States"]),
            E::Am => Some(&["United States"]),
            E::Cm => Some(&["United States"]),
            E::Bk => Some(&["United States"]),
            E::Cf => Some(&["United States"]),
            E::Es => Some(&["United States"]),
            E::Fm => Some(&["United States"]),
            E::Md => Some(&["United States"]),
            E::No => Some(&["Sweden"]),
            E::Lr => Some(&["United States"]),
            E::Rf => Some(&["United States"]),
            E::Db => Some(&["United States"]),
            E::Sg => Some(&["USSR", "United States"]),
            E::Bh => Some(&["Germany"]),
            E::Hs => Some(&["Germany"]),
            E::Mt => Some(&["Germany"]),
            E::Ds => Some(&["Germany"]),
            E::Rg => Some(&["Germany"]),
            E::Cn => Some(&["Germany"]),
            E::Nh => Some(&["Japan"]),
            E::Fl => Some(&["Russia"]),
            E::Mc => Some(&["Russia"]),
            E::Lv => Some(&["United States"]),
            E::Ts => Some(&["Russia", "United States"]),
            E::Og => Some(&["Russia"]),
        }
    }
}
