// Copyright (C) 2016 The Qt Company Ltd.
// SPDX-License-Identifier: LicenseRef-Qt-Commercial OR GPL-3.0-only WITH Qt-GPL-exception-1.0
#ifndef DOCPARSER_H
#define DOCPARSER_H

#include "abstractmetalang_typedefs.h"
#include "modifications_typedefs.h"
#include "documentation.h"

#include <QtCore/QString>

#include <memory>

class AbstractMetaClass;
class DocModification;

class XQuery;

struct FunctionDocumentation;

struct ModuleDocumentation
{
    Documentation documentation;
    QString qmlTypesUrl;
};

class DocParser
{
public:
    Q_DISABLE_COPY_MOVE(DocParser)

    using XQueryPtr = std::shared_ptr<XQuery>;

    DocParser();
    virtual ~DocParser();
    virtual QString fillDocumentation(const AbstractMetaClassPtr &metaClass) = 0;
    virtual void fillGlobalFunctionDocumentation(const AbstractMetaFunctionPtr &f);
    virtual void fillGlobalEnumDocumentation(AbstractMetaEnum &e);

    void setDocumentationDataDirectory(const QString& dir)
    {
        m_docDataDir = dir;
    }

    /**
     *   Informs the location of the XML data generated by the tool
     *   (e.g.: DoxyGen, qdoc) used to extract the library's documentation
     *   comment.
     *   \return the path for the directory containing the XML data created
     *   from the library's documentation beign parsed.
     */
    QString documentationDataDirectory() const
    {
        return m_docDataDir;
    }

    /**
    *   Process and retrieves documentation concerning the entire
    *   module or library.
    *   \param name module name
    *   \return object containing module/library documentation information
    *   \todo Merge with retrieveModuleDocumentation() on next ABI change.
    */
    virtual ModuleDocumentation retrieveModuleDocumentation(const QString &name) = 0;

    static bool skipForQuery(const AbstractMetaFunctionCPtr &func);

    /// Helper to return the documentation modifications for a class
    /// or a member function.
    static DocModificationList getDocModifications(const AbstractMetaClassCPtr &cppClass);
    static DocModificationList getDocModifications(const AbstractMetaFunctionCPtr &func,
                                                   const AbstractMetaClassCPtr &cppClass = {});
    static DocModificationList getXpathDocModifications(const AbstractMetaClassCPtr &cppClass);
    static DocModificationList getXpathDocModifications(const AbstractMetaFunctionCPtr &func,
                                                        const AbstractMetaClassCPtr &cppClass = {});

    static QString enumBaseClass(const AbstractMetaEnum &e);

protected:
    static QString getDocumentation(const XQueryPtr &xquery,
                                    const QString &query,
                                    const DocModificationList &mods);

    static AbstractMetaFunctionCList documentableFunctions(const AbstractMetaClassCPtr &metaClass);

    static QString applyDocModifications(const DocModificationList &xpathMods, const QString &xml);

private:
    QString m_docDataDir;

    static QString execXQuery(const XQueryPtr &xquery, const QString &query) ;
};

#endif // DOCPARSER_H
