/* This file deca_double_functions.h defines the arithmetical operations 
   for deca double numbers.  A deca double is defined by ten doubles.

The ten doubles of a deca double are named after the ten finger count,
which begins at the fingers on the right hand, starting with the thumb 
for the most significant double, then to the index, middle, ring, and pink,
which is the fifth significant part of a deca double number.
The next five significant parts are then the thumb, index, middle, ring
and pink on the left hand.
Abbreviations for thumb, index, middle, ring, and pink on the right hand
are respectively rtb, rix, rmi, rrg, and rpk, and on the left hand they
are respectively ltb, lix, lmi, lrg, and lpk.

The algorithms are from the CAMPARY and QD software libraries
with the modification that a deca double is not stored as an array
of ten doubles, but plainly by ten double numbers.
All functions have the prefix daf_ to avoid name clashes. */

#ifndef __deca_double_functions_h__
#define __deca_double_functions_h__

/************************** renormalizations **************************/

void daf_renorm10
 ( double f0, double f1, double f2, double f3, double f4, double f5,
   double f6, double f7, double f8, double f9, double f10, double *pr,
   double *r0, double *r1, double *r2, double *r3, double *r4, double *r5,
   double *r6, double *r7, double *r8, double *r9 );
/*
 * DESCRIPTION :
 *   Definitions common to fast_renorm2L<11,10> and renorm2L_4Add1<10,10>
 *   from code of the specRenorm.h, generated by the CAMPARY library,
 *   in an unrolled form (Valentina Popescu, Mioara Joldes), with
 *   double double basics of QD-2.3.9 (Y. Hida, X.S. Li, and D.H. Bailey).
 *
 * ON ENTRY :
 *   f0       most significant word;
 *   f1       second most significant word;
 *   f2       third most significant word;
 *   f3       fourth most significant word;
 *   f4       fifth most significant word;
 *   f5       sixth most significant word;
 *   f6       seventh most significant word;
 *   f7       eighth most significant word;
 *   f8       nineth most significant word;
 *   f9       tenth most significant word;
 *   f10      least significant word;
 *   pr       computed by the start of the renormalization.
 *
 * ON RETURN :
 *   pr       updated value by renormalization;
 *   r0       highest part of a deca double number;
 *   r1       second highest part of a deca double number;
 *   r2       third highest part of a deca double number.
 *   r3       fourth highest part of a deca double number;
 *   r4       fifth highest part of a deca double number;
 *   r5       fifth lowest part of a deca double number;
 *   r6       fourth lowest part of a deca double number;
 *   r7       third lowest part of a deca double number;
 *   r8       second lowest part of a deca double number;
 *   r9       lowest part of a deca double number. */

void daf_fast_renorm
 ( double x0, double x1, double x2, double x3, double x4, double x5,
   double x6, double x7, double x8, double x9, double x10,
   double *r0, double *r1, double *r2, double *r3, double *r4, double *r5,
   double *r6, double *r7, double *r8, double *r9 );
/*
 * DESCRIPTION :
 *   The definition is based on the fast_renorm2L<11,10>,
 *   from code of the specRenorm.h, generated by the CAMPARY library,
 *   in an unrolled form (Valentina Popescu, Mioara Joldes), with
 *   double double basics of QD-2.3.9 (Y. Hida, X.S. Li, and D.H. Bailey).
 *
 * ON ENTRY :
 *   x0       most significant word;
 *   x1       second most significant word;
 *   x2       third most significant word;
 *   x3       fourth most significant word;
 *   x4       fifth most significant word;
 *   x5       sixth most significant word;
 *   x6       seventh most significant word;
 *   x7       eighth most significant word;
 *   x8       nineth most significant word;
 *   x9       tenth most significant word;
 *   x10      least significant word.
 *
 * ON RETURN :
 *   r0       highest part of an deca double number;
 *   r1       second highest part of an deca double number;
 *   r2       third highest part of an deca double number.
 *   r3       fourth highest part of an deca double number;
 *   r4       fifth highest part of an deca double number;
 *   r5       fifth lowest part of an deca double number;
 *   r6       fourth lowest part of an deca double number;
 *   r7       third lowest part of an deca double number;
 *   r8       second lowest part of an deca double number;
 *   r9       lowest part of an deca double number. */

void daf_renorm_add1
 ( double x0, double x1, double x2, double x3, double x4, double x5,
   double x6, double x7, double x8, double x9, double y,
   double *r0, double *r1, double *r2, double *r3, double *r4, double *r5,
   double *r6, double *r7, double *r8, double *r9 );
/*
 * DESCRIPTION :
 *   The definition is based on the renorm2L_4Add1<10,10>,
 *   from code of the specRenorm.h, generated by the CAMPARY library,
 *   in an unrolled form (Valentina Popescu, Mioara Joldes), with
 *   double double basics of QD-2.3.9 (Y. Hida, X.S. Li, and D.H. Bailey).
 *
 * ON ENTRY :
 *   x0       most significant word of an deca double x;
 *   x1       second most significant word of an deca double x;
 *   x2       third most significant word of an deca double x;
 *   x3       fourth most significant word of an deca double x;
 *   x4       fifth most significant word of an deca double x;
 *   x5       fifth lowest part of an deca double number x;
 *   x6       fourth lowest part of an deca double number x;
 *   x7       third lowest part of an deca double number x;
 *   x8       second lowest part of an deca double number x;
 *   x9       least significant word of an deca double x;
 *   y        double to be added to x.
 *
 * ON RETURN :
 *   r0       highest part of x + y;
 *   r1       second highest part of x + y;
 *   r2       third highest part of x + y;
 *   r3       fourth highest part of x + y;
 *   r4       fifth highest part of x + y;
 *   r5       fifth lowest part of x + y;
 *   r6       fourth lowest part of x + y;
 *   r7       third lowest part of x + y;
 *   r8       second lowest part of x + y;
 *   r9       lowest part of x + y. */

/************************ copy and abs *******************************/

void daf_copy
 ( double a_rtb, double a_rix, double a_rmi, double a_rrg, double a_rpk,
   double a_ltb, double a_lix, double a_lmi, double a_lrg, double a_lpk,
   double *b_rtb, double *b_rix, double *b_rmi, double *b_rrg, double *b_rpk,
   double *b_ltb, double *b_lix, double *b_lmi, double *b_lrg, double *b_lpk );
/*
 * DESCRIPTION :
 *   Copies the content of the deca double a (a_rtb, a_rix, a_rmi, a_rrg,
 *   a_rpk, a_ltb, a_lix, a_lmi, a_lrg, a_lpk) to the deca double b (b_rtb,
 *   b_rix, b_rmi, b_rrg, b_rpk, b_ltb, b_lix, b_lmi, b_lrg, b_lpk). */

void daf_abs
 ( double a_rtb, double a_rix, double a_rmi, double a_rrg, double a_rpk,
   double a_ltb, double a_lix, double a_lmi, double a_lrg, double a_lpk,
   double *b_rtb, double *b_rix, double *b_rmi, double *b_rrg, double *b_rpk,
   double *b_ltb, double *b_lix, double *b_lmi, double *b_lrg, double *b_lpk );
/*
 * DESCRIPTION :
 *   Returns in b the absolute value of a. */

/****************** additions and substractions ************************/

void daf_add
 ( double a_rtb, double a_rix, double a_rmi, double a_rrg, double a_rpk,
   double a_ltb, double a_lix, double a_lmi, double a_lrg, double a_lpk,
   double b_rtb, double b_rix, double b_rmi, double b_rrg, double b_rpk,
   double b_ltb, double b_lix, double b_lmi, double b_lrg, double b_lpk,
   double *c_rtb, double *c_rix, double *c_rmi, double *c_rrg, double *c_rpk,
   double *c_ltb, double *c_lix, double *c_lmi, double *c_lrg, double *c_lpk );
/*
 * DESCRIPTION : c = a + b.
 *   Adds the deca doubles a (a_rtb, a_rix, a_rmi, a_rrg, a_rpk, a_ltb,
 *   a_lix, a_lmi, a_lrg, a_lpk) and b (b_rtb, b_rix, b_rmi, b_rrg, b_rpk,
 *   b_ltb, b_lix, b_lmi, b_lrg, b_lpk) to make the deca double c (c_rtb,
 *   c_rix, c_rmi, c_rrg, c_rpk, c_ltb, c_lix, c_lmi, c_lrg, c_lpk).
 *
 * ON ENTRY :
 *   a_rtb    highest part of the deca double a;
 *   a_rix    second highest part of the deca double a;
 *   a_rmi    third highest part of the deca double a;
 *   a_rrg    fourth highest part of the deca double a;
 *   a_rpk    fifth highest part of the deca double a;
 *   a_ltb    fifth lowest part of the deca double a;
 *   a_lix    fourth lowest part of the deca double a;
 *   a_lmi    third lowest part of the deca double a;
 *   a_lrg    second lowest part of the deca double a;
 *   a_lpk    lowest part of the deca double a;
 *   b_rtb    highest part of the deca double b;
 *   b_rix    second highest part of the deca double b;
 *   b_rmi    third highest part of the deca double b;
 *   b_rrg    fourth highest part of the deca double b;
 *   b_rpk    fifth highest part of the deca double b;
 *   b_ltb    fifth lowest part of the deca double b;
 *   b_lix    fourth lowest part of the deca double b;
 *   b_lmi    third lowest part of the deca double b;
 *   b_lrg    second lowest part of the deca double b;
 *   b_lpk    lowest part of the deca double b.
 *
 * ON RETURN :
 *   c_rtb    highest part of the deca double c = a + b;
 *   c_rix    second highest part of the deca double c = a + b;
 *   c_rmi    third highest part of the deca double c = a + b;
 *   c_rrg    fourth highest part of the deca double c = a + b;
 *   c_rpk    fifth highest part of the deca double c = a + b;
 *   c_ltb    fifth lowest part of the deca double c = a + b;
 *   c_lix    fourth lowest part of the deca double c = a + b;
 *   c_lmi    third lowest part of the deca double c = a + b;
 *   c_lrg    second lowest part of the deca double c = a + b;
 *   c_lpk    lowest part of the deca double c = a + b. */

void daf_inc
 ( double *a_rtb, double *a_rix, double *a_rmi, double *a_rrg, double *a_rpk,
   double *a_ltb, double *a_lix, double *a_lmi, double *a_lrg, double *a_lpk,
   double b_rtb, double b_rix, double b_rmi, double b_rrg, double b_rpk,
   double b_ltb, double b_lix, double b_lmi, double b_lrg, double b_lpk );
/*
 * DESCRIPTION : a = a + b.
 *   Adds to the deca double a (a_rtb, a_rix, a_rmi, a_rrg, a_rpk, a_ltb,
 *   a_lix, a_lmi, a_lrg, a_lpk) the deca double b (b_rtb, b_rix, b_rmi,
 *   b_rrg, b_rpk, b_ltb, b_lix, b_lmi, b_lrg, b_lpk).
 *
 * ON ENTRY :
 *   a_rtb    highest part of the deca double a;
 *   a_rix    second highest part of the deca double a;
 *   a_rmi    third highest part of the deca double a;
 *   a_rrg    fourth highest part of the deca double a;
 *   a_rpk    fifth highest part of the deca double a;
 *   a_ltb    fifth lowest part of the deca double a;
 *   a_lix    fourth lowest part of the deca double a;
 *   a_lmi    third lowest part of the deca double a;
 *   a_lrg    second lowest part of the deca double a;
 *   a_lpk    lowest part of the deca double a;
 *   b_rtb    highest part of the deca double b;
 *   b_rix    second highest part of the deca double b;
 *   b_rmi    third highest part of the deca double b;
 *   b_rrg    fourth highest part of the deca double b;
 *   b_rpk    fifth highest part of the deca double b;
 *   b_ltb    fifth lowest part of the deca double b;
 *   b_lix    fourth lowest part of the deca double b;
 *   b_lmi    third lowest part of the deca double b;
 *   b_lrg    second lowest part of the deca double b;
 *   b_lpk    lowest part of the deca double b.
 *
 * ON RETURN :
 *   a_rtb    highest part of the deca double a + b;
 *   a_rix    second highest part of the deca double a + b;
 *   a_rmi    third highest part of the deca double a + b;
 *   a_rrg    fourth highest part of the deca double a + b;
 *   a_rpk    fifth highest part of the deca double a + b;
 *   a_ltb    fifth lowest part of the deca double a + b;
 *   a_lix    fourth lowest part of the deca double a + b;
 *   a_lmi    third lowest part of the deca double a + b;
 *   a_lrg    second lowest part of the deca double a + b;
 *   a_lpk    lowest part of the deca double a + b. */

void daf_inc_d
 ( double *a_rtb, double *a_rix, double *a_rmi, double *a_rrg, double *a_rpk,
   double *a_ltb, double *a_lix, double *a_lmi, double *a_lrg, double *a_lpk,
   double b );
/*
 * DESCRIPTION : a = a + b.
 *   Adds b to the deca double a (a_rtb, a_rix, a_rmi, a_rrg, a_rpk, a_ltb,
 *   a_lix, a_lmi, a_lrg, a_lpk).
 *
 * ON ENTRY :
 *   a_rtb    highest part of the deca double a;
 *   a_rix    second highest part of the deca double a;
 *   a_rmi    third highest part of the deca double a;
 *   a_rrg    fourth highest part of the deca double a;
 *   a_rpk    fifth highest part of the deca double a;
 *   a_ltb    fifth lowest part of the deca double a;
 *   a_lix    fourth lowest part of the deca double a;
 *   a_lmi    third lowest part of the deca double a;
 *   a_lrg    second lowest part of the deca double a;
 *   a_lpk    lowest part of the deca double a;
 *   b        some double.
 *
 * ON RETURN :
 *   a_rtb    highest part of the deca double a * b;
 *   a_rix    second highest part of the deca double a * b;
 *   a_rmi    third highest part of the deca double a * b;
 *   a_rrg    fourth highest part of the deca double a * b;
 *   a_rpk    fifth highest part of the deca double a * b;
 *   a_ltb    fifth lowest part of the deca double a * b;
 *   a_lix    fourth lowest part of the deca double a * b;
 *   a_lmi    third lowest part of the deca double a * b;
 *   a_lrg    second lowest part of the deca double a * b;
 *   a_lpk    lowest part of the deca double a * b. */

void daf_minus
 ( double *a_rtb, double *a_rix, double *a_rmi, double *a_rrg, double *a_rpk,
   double *a_ltb, double *a_lix, double *a_lmi, double *a_lrg, double *a_lpk );
/*
 * DESCRIPTION :
 *   Flips the sign of a (a_rtb, a_rix, a_rmi, a_rrg, a_rpk, a_ltb, a_lix,
 *   a_lmi, a_lrg, a_lpk). */

void daf_sub
 ( double a_rtb, double a_rix, double a_rmi, double a_rrg, double a_rpk,
   double a_ltb, double a_lix, double a_lmi, double a_lrg, double a_lpk,
   double b_rtb, double b_rix, double b_rmi, double b_rrg, double b_rpk,
   double b_ltb, double b_lix, double b_lmi, double b_lrg, double b_lpk,
   double *c_rtb, double *c_rix, double *c_rmi, double *c_rrg, double *c_rpk,
   double *c_ltb, double *c_lix, double *c_lmi, double *c_lrg, double *c_lpk );
/*
 * DESCRIPTION : c = a - b.
 *   Subtracts the deca double in b (b_rtb, b_rix, b_rmi, b_rrg, b_rpk,
 *   b_ltb, b_lix, b_lmi, b_lrg, b_lpk) from the deca double in a (a_rtb,
 *   a_rix, a_rmi, a_rrg, a_rpk, a_ltb, a_lix, a_lmi, a_lrg, a_lpk) and
 *   places the result in the deca double c (c_rtb, c_rix, c_rmi, c_rrg,
 *   c_rpk, c_ltb, c_lix, c_lmi, c_lrg, c_lpk).
 *
 * ON ENTRY :
 *   a_rtb    highest part of the deca double a;
 *   a_rix    second highest part of the deca double a;
 *   a_rmi    third highest part of the deca double a;
 *   a_rrg    fourth highest part of the deca double a;
 *   a_rpk    fifth highest part of the deca double a;
 *   a_ltb    fifth lowest part of the deca double a;
 *   a_lix    fourth lowest part of the deca double a;
 *   a_lmi    third lowest part of the deca double a;
 *   a_lrg    second lowest part of the deca double a;
 *   a_lpk    lowest part of the deca double a;
 *   b_rtb    highest part of the deca double b;
 *   b_rix    second highest part of the deca double b;
 *   b_rmi    third highest part of the deca double b;
 *   b_rrg    fourth highest part of the deca double b;
 *   b_rpk    fifth highest part of the deca double b;
 *   b_ltb    fifth lowest part of the deca double b;
 *   b_lix    fourth lowest part of the deca double b;
 *   b_lmi    third lowest part of the deca double b;
 *   b_lrg    second lowest part of the deca double b;
 *   b_lpk    lowest part of the deca double b.
 *
 * ON RETURN :
 *   c_rtb    highest part of the deca double c = a - b;
 *   c_rix    second highest part of the deca double c = a - b;
 *   c_rmi    third highest part of the deca double c = a - b;
 *   c_rrg    fourth highest part of the deca double c = a - b;
 *   c_rpk    fifth highest part of the deca double c = a - b;
 *   c_ltb    fifth lowest part of the deca double c = a - b;
 *   c_lix    fourth lowest part of the deca double c = a - b;
 *   c_lmi    third lowest part of the deca double c = a - b;
 *   c_lrg    second lowest part of the deca double c = a - b;
 *   c_lpk    lowest part of the deca double c = a - b. */

/***************** multiplications and division ********************/

void daf_mul_pwr2
 ( double a_rtb, double a_rix, double a_rmi, double a_rrg, double a_rpk,
   double a_ltb, double a_lix, double a_lmi, double a_lrg, double a_lpk,
   double b,
   double *c_rtb, double *c_rix, double *c_rmi, double *c_rrg, double *c_rpk,
   double *c_ltb, double *c_lix, double *c_lmi, double *c_lrg, double *c_lpk );
/*
 * DESCRIPTION : c = a * b, where b is a power of two.
 *   As b is a power of two, all components of the octo double
 *   are plainly multiplied by b.
 *
 * ON ENTRY :
 *   a_rtb    highest part of the octo double a;
 *   a_rix    second highest part of the octo double a;
 *   a_rmi    third highest part of the octo double a;
 *   a_rrg    fourth highest part of the octo double a;
 *   a_rpk    fifth highest part of the octo double a;
 *   a_ltb    fifth lowest part of the octo double a;
 *   a_lix    fourth lowest part of the octo double a;
 *   a_lmi    third lowest part of the octo double a;
 *   a_lrg    second lowest part of the octo double a;
 *   a_lpk    lowest part of the octo double a;
 *   b        a power of two.
 *
 * ON RETURN :
 *   c_rtb    highest part of the octo double c = a * b;
 *   c_rix    second highest part of the octo double c = a * b;
 *   c_rmi    third highest part of the octo double c = a * b;
 *   c_rrg    fourth highest part of the octo double c = a * b;
 *   c_rpk    fifth highest part of the octo double c = a * b.
 *   c_ltb    fifth lowest part of the octo double c = a * b;
 *   c_lix    fourth lowest part of the octo double c = a * b;
 *   c_lmi    third lowest part of the octo double c = a * b;
 *   c_lrg    second lowest part of the octo double c = a * b;
 *   c_lpk    lowest part of the octo double c = a * b. */

void daf_mul
 ( double a_rtb, double a_rix, double a_rmi, double a_rrg, double a_rpk,
   double a_ltb, double a_lix, double a_lmi, double a_lrg, double a_lpk,
   double b_rtb, double b_rix, double b_rmi, double b_rrg, double b_rpk,
   double b_ltb, double b_lix, double b_lmi, double b_lrg, double b_lpk,
   double *c_rtb, double *c_rix, double *c_rmi, double *c_rrg, double *c_rpk,
   double *c_ltb, double *c_lix, double *c_lmi, double *c_lrg, double *c_lpk );
/*
 * DESCRIPTION : c = a * b.
 *   Multiplies two deca doubles a (a_rtb, a_rix, a_rmi, a_rrg, a_rpk, a_ltb,
 *   a_lix, a_lmi, a_lrg, a_lpk) and b (b_rtb, b_rix, b_rmi, b_rrg, b_rpk,
 *   b_ltb, b_lix, b_lmi, b_lrg, b_lpk) to make the deca double c (c_rtb,
 *   c_rix, c_rmi, c_rrg, c_rpk, c_ltb, c_lix, c_lmi, c_lrg, c_lpk).
 *
 * ON ENTRY :
 *   a_rtb    highest part of the deca double a;
 *   a_rix    second highest part of the deca double a;
 *   a_rmi    third highest part of the deca double a;
 *   a_rrg    fourth highest part of the deca double a;
 *   a_rpk    fifth highest part of the deca double a;
 *   a_ltb    fifth lowest part of the deca double a;
 *   a_lix    fourth lowest part of the deca double a;
 *   a_lmi    third lowest part of the deca double a;
 *   a_lrg    second lowest part of the deca double a;
 *   a_lpk    lowest part of the deca double a;
 *   b_rtb    highest part of the deca double b;
 *   b_rix    second highest part of the deca double b;
 *   b_rmi    third highest part of the deca double b;
 *   b_rrg    fourth highest part of the deca double b;
 *   b_rpk    fifth highest part of the deca double b;
 *   b_ltb    fifth lowest part of the deca double b;
 *   b_lix    fourth lowest part of the deca double b;
 *   b_lmi    third lowest part of the deca double b;
 *   b_lrg    second lowest part of the deca double b;
 *   b_lpk    lowest part of the deca double b.
 *
 * ON RETURN :
 *   c_rtb    highest part of the deca double c = a * b;
 *   c_rix    second highest part of the deca double c = a * b;
 *   c_rmi    third highest part of the deca double c = a * b;
 *   c_rrg    fourth highest part of the deca double c = a * b;
 *   c_rpk    fifth highest part of the deca double c = a * b;
 *   c_ltb    fifth lowest part of the deca double c = a * b;
 *   c_lix    fourth lowest part of the deca double c = a * b;
 *   c_lmi    third lowest part of the deca double c = a * b;
 *   c_lrg    second lowest part of the deca double c = a * b;
 *   c_lpk    lowest part of the deca double c = a * b. */

void daf_sqr
 ( double a_rtb, double a_rix, double a_rmi, double a_rrg, double a_rpk,
   double a_ltb, double a_lix, double a_lmi, double a_lrg, double a_lpk,
   double *c_rtb, double *c_rix, double *c_rmi, double *c_rrg, double *c_rpk,
   double *c_ltb, double *c_lix, double *c_lmi, double *c_lrg, double *c_lpk );
/*
 * DESCRIPTION : c = a * a.
 *   Multiplies the deca double a (a_rtb, a_rix, a_rmi, a_rrg, a_rpk, a_ltb,
 *   a_lix, a_lmi, a_lrg, a_lpk) with a to make the deca double c (c_rtb,
 *   c_rix, c_rmi, c_rrg, c_rpk, c_ltb, c_lix, c_lmi, c_lrg, c_lpk).
 *
 * ON ENTRY :
 *   a_rtb    highest part of the deca double a;
 *   a_rix    second highest part of the deca double a;
 *   a_rmi    third highest part of the deca double a;
 *   a_rrg    fourth highest part of the deca double a;
 *   a_rpk    fifth highest part of the deca double a;
 *   a_ltb    fifth lowest part of the deca double a;
 *   a_lix    fourth lowest part of the deca double a;
 *   a_lmi    third lowest part of the deca double a;
 *   a_lrg    second lowest part of the deca double a;
 *   a_lpk    lowest part of the deca double a.
 *
 * ON RETURN :
 *   c_rtb    highest part of the deca double c = a * a;
 *   c_rix    second highest part of the deca double c = a * a;
 *   c_rmi    third highest part of the deca double c = a * a;
 *   c_rrg    fourth highest part of the deca double c = a * a;
 *   c_rpk    fifth highest part of the deca double c = a * a;
 *   c_ltb    fifth lowest part of the deca double c = a * a;
 *   c_lix    fourth lowest part of the deca double c = a * a;
 *   c_lmi    third lowest part of the deca double c = a * a;
 *   c_lrg    second lowest part of the deca double c = a * a;
 *   c_lpk    lowest part of the deca double c = a * a. */

void daf_mul_da_d
 ( double a_rtb, double a_rix, double a_rmi, double a_rrg, double a_rpk,
   double a_ltb, double a_lix, double a_lmi, double a_lrg, double a_lpk,
   double b,
   double *c_rtb, double *c_rix, double *c_rmi, double *c_rrg, double *c_rpk,
   double *c_ltb, double *c_lix, double *c_lmi, double *c_lrg, double *c_lpk );
/*
 * DESCRIPTION : c = a * b.
 *   Multiplies the deca double a (a_rtb, a_rix, a_rmi, a_rrg, a_rpk, a_ltb,
 *   a_lix, a_lmi, a_lrg, a_lpk) by b to make the deca double c (c_rtb,
 *   c_rix, c_rmi, c_rrg, c_rpk, c_ltb, c_lix, c_lmi, c_lrg, c_lpk).
 *
 * ON ENTRY :
 *   a_rtb    highest part of the deca double a;
 *   a_rix    second highest part of the deca double a;
 *   a_rmi    third highest part of the deca double a;
 *   a_rrg    fourth highest part of the deca double a;
 *   a_rpk    fifth highest part of the deca double a;
 *   a_ltb    fifth lowest part of the deca double a;
 *   a_lix    fourth lowest part of the deca double a;
 *   a_lmi    third lowest part of the deca double a;
 *   a_lrg    second lowest part of the deca double a;
 *   a_lpk    lowest part of the deca double a;
 *   b        some double.
 *
 * ON RETURN :
 *   c_rtb    highest part of the deca double c = a * b;
 *   c_rix    second highest part of the deca double c = a * b;
 *   c_rmi    third highest part of the deca double c = a * b;
 *   c_rrg    fourth highest part of the deca double c = a * b;
 *   c_rpk    fifth highest part of the deca double c = a * b;
 *   c_ltb    fifth lowest part of the deca double c = a * b;
 *   c_lix    fourth lowest part of the deca double c = a * b;
 *   c_lmi    third lowest part of the deca double c = a * b;
 *   c_lrg    second lowest part of the deca double c = a * b;
 *   c_lpk    lowest part of the deca double c = a * b. */

void daf_div
 ( double a_rtb, double a_rix, double a_rmi, double a_rrg, double a_rpk,
   double a_ltb, double a_lix, double a_lmi, double a_lrg, double a_lpk,
   double b_rtb, double b_rix, double b_rmi, double b_rrg, double b_rpk,
   double b_ltb, double b_lix, double b_lmi, double b_lrg, double b_lpk,
   double *c_rtb, double *c_rix, double *c_rmi, double *c_rrg, double *c_rpk,
   double *c_ltb, double *c_lix, double *c_lmi, double *c_lrg, double *c_lpk );
/*
 * DESCRIPTION : c = a / b.
 *   Divides the deca double a (a_rtb, a_rix, a_rmi, a_rrg, a_rpk, a_ltb,
 *   a_lix, a_lmi, a_lrg, a_lpk) by b (b_rtb, b_rix, b_rmi, b_rrg, b_rpk,
 *   b_ltb, b_lix, b_lmi, b_lrg, b_lpk) to make the deca double c (c_rtb,
 *   c_rix, c_rmi, c_rrg, c_rpk, c_ltb, c_lix, c_lmi, c_lrg, c_lpk).
 *
 * ON ENTRY :
 *   a_rtb    highest part of the deca double a;
 *   a_rix    second highest part of the deca double a;
 *   a_rmi    third highest part of the deca double a;
 *   a_rrg    fourth highest part of the deca double a;
 *   a_rpk    fifth highest part of the deca double a;
 *   a_ltb    fifth lowest part of the deca double a;
 *   a_lix    fourth lowest part of the deca double a;
 *   a_lmi    third lowest part of the deca double a;
 *   a_lrg    second lowest part of the deca double a;
 *   a_lpk    lowest part of the deca double a;
 *   b_rtb    highest part of the deca double a;
 *   b_rix    second highest part of the deca double b;
 *   b_rmi    third highest part of the deca double b;
 *   b_rrg    fourth highest part of the deca double b;
 *   b_rpk    fifth highest part of the deca double b;
 *   b_ltb    fifth lowest part of the deca double b;
 *   b_lix    fourth lowest part of the deca double b;
 *   b_lmi    third lowest part of the deca double b;
 *   b_lrg    second lowest part of the deca double b;
 *   b_lpk    lowest part of the deca double b.
 *
 * ON RETURN :
 *   c_rtb    highest part of the deca double c = a / b;
 *   c_rix    second highest part of the deca double c = a / b;
 *   c_rmi    third highest part of the deca double c = a / b;
 *   c_rrg    fourth highest part of the deca double c = a / b;
 *   c_rpk    fifth highest part of the deca double c = a / b;
 *   c_ltb    fifth lowest part of the deca double c = a / b;
 *   c_lix    fourth lowest part of the deca double c = a / b;
 *   c_lmi    third lowest part of the deca double c = a / b;
 *   c_lrg    second lowest part of the deca double c = a / b;
 *   c_lpk    lowest part of the deca double c = a / b. */

/***************************** square root *****************************/

void daf_sqrt
 ( double a_rtb, double a_rix, double a_rmi, double a_rrg, double a_rpk,
   double a_ltb, double a_lix, double a_lmi, double a_lrg, double a_lpk,
   double *b_rtb, double *b_rix, double *b_rmi, double *b_rrg, double *b_rpk,
   double *b_ltb, double *b_lix, double *b_lmi, double *b_lrg, double *b_lpk );
/*
 * DESCRIPTION :
 *   Returns in the deca double b (b_rtb, b_rix, b_rmi, b_rrg, b_rpk, b_ltb,
 *   b_lix, b_lmi, b_lrg, b_lpk)  the square root of the deca double a
 *   (a_rtb, a_rix, a_rmi, a_rrg, a_rpk, a_ltb, a_lix, a_lmi, a_lrg, a_lpk).
 *
 * ON ENTRY :
 *   a_rtb    highest part of the deca double a;
 *   a_rix    second highest part of the deca double a;
 *   a_rmi    third highest part of the deca double a;
 *   a_rrg    fourth highest part of the deca double a;
 *   a_rpk    fifth highest part of the deca double a;
 *   a_ltb    fifth lowest part of the deca double a;
 *   a_lix    fourth lowest part of the deca double a;
 *   a_lmi    third lowest part of the deca double a;
 *   a_lrg    second lowest part of the deca double a;
 *   a_lpk    lowest part of the deca double a;
 *
 * ON RETURN :
 *   b_rtb    highest part of the deca double a;
 *   b_rix    second highest part of the deca double b;
 *   b_rmi    third highest part of the deca double b;
 *   b_rrg    fourth highest part of the deca double b;
 *   b_rpk    fifth highest part of the deca double b;
 *   b_ltb    fifth lowest part of the deca double b;
 *   b_lix    fourth lowest part of the deca double b;
 *   b_lmi    third lowest part of the deca double b;
 *   b_lrg    second lowest part of the deca double b. */

/*************************** basic output ***************************/

void daf_write_doubles
 ( double a_rtb, double a_rix, double a_rmi, double a_rrg, double a_rpk, 
   double a_ltb, double a_lix, double a_lmi, double a_lrg, double a_lpk );
/*
 * DESCRIPTION :
 *   Writes the ten doubles (a_rtb, a_rix, a_rmi, a_rrg, a_rlo, a_ltb,
 *   a_lix, a_lmi, a_lrg, a_llo) of the deca double a in scientific format
 *   with 16 decimal places of precision. */

#endif
