package transformers

import (
	"fmt"
	"os"
	"strings"

	"github.com/johnkerl/miller/v6/pkg/cli"
	"github.com/johnkerl/miller/v6/pkg/lib"
	"github.com/johnkerl/miller/v6/pkg/mlrval"
	"github.com/johnkerl/miller/v6/pkg/types"
)

const verbNameLatin1ToUTF8 = "latin1-to-utf8"

var Latin1ToUTF8Setup = TransformerSetup{
	Verb:         verbNameLatin1ToUTF8,
	UsageFunc:    transformerLatin1ToUTF8Usage,
	ParseCLIFunc: transformerLatin1ToUTF8ParseCLI,
	IgnoresInput: false,
}

func transformerLatin1ToUTF8Usage(
	o *os.File,
) {
	fmt.Fprintf(o, "Usage: %s %s, with no options.\n", "mlr", verbNameLatin1ToUTF8)
	fmt.Fprintf(o, "Recursively converts record strings from Latin-1 to UTF-8.\n")
	fmt.Fprintf(o, "For field-level control, please see the latin1_to_utf8 DSL function.\n")
	fmt.Fprintf(o, "Options:\n")
	fmt.Fprintf(o, "-h|--help Show this message.\n")
}

func transformerLatin1ToUTF8ParseCLI(
	pargi *int,
	argc int,
	args []string,
	_ *cli.TOptions,
	doConstruct bool, // false for first pass of CLI-parse, true for second pass
) (RecordTransformer, error) {

	// Skip the verb name from the current spot in the mlr command line
	argi := *pargi
	argi++

	for argi < argc /* variable increment: 1 or 2 depending on flag */ {
		opt := args[argi]
		if !strings.HasPrefix(opt, "-") {
			break // No more flag options to process
		}
		if args[argi] == "--" {
			break // All transformers must do this so main-flags can follow verb-flags
		}
		argi++

		if opt == "-h" || opt == "--help" {
			transformerLatin1ToUTF8Usage(os.Stdout)
			return nil, cli.ErrHelpRequested

		}
		return nil, cli.VerbErrorf(verbNameLatin1ToUTF8, "option \"%s\" not recognized", opt)
	}

	*pargi = argi
	if !doConstruct { // All transformers must do this for main command-line parsing
		return nil, nil
	}

	transformer, err := NewTransformerLatin1ToUTF8()
	if err != nil {
		return nil, err
	}

	return transformer, nil
}

type TransformerLatin1ToUTF8 struct {
}

func NewTransformerLatin1ToUTF8() (*TransformerLatin1ToUTF8, error) {
	tr := &TransformerLatin1ToUTF8{}
	return tr, nil
}

func (tr *TransformerLatin1ToUTF8) Transform(
	inrecAndContext *types.RecordAndContext,
	outputRecordsAndContexts *[]*types.RecordAndContext, // list of *types.RecordAndContext
	inputDownstreamDoneChannel <-chan bool,
	outputDownstreamDoneChannel chan<- bool,
) {
	HandleDefaultDownstreamDone(inputDownstreamDoneChannel, outputDownstreamDoneChannel)
	if !inrecAndContext.EndOfStream {
		inrec := inrecAndContext.Record

		for pe := inrec.Head; pe != nil; pe = pe.Next {
			inval := pe.Value
			if inval.IsString() {
				output, err := lib.TryLatin1ToUTF8(pe.Value.String())
				if err == nil {
					pe.Value = mlrval.FromString(output)
				} else {
					pe.Value = mlrval.FromError(err)
				}
			}
		}

		*outputRecordsAndContexts = append(*outputRecordsAndContexts, types.NewRecordAndContext(inrec, &inrecAndContext.Context))

	} else { // end of record stream
		*outputRecordsAndContexts = append(*outputRecordsAndContexts, inrecAndContext)
	}
}
