/*

  Author:  Bob Dean
  Copyright (c) 1999, 2000


   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public Licensse as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
 
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
 
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

 */

#ifdef __cplusplus
extern "C"
{
#endif

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <sys/types.h>
#include <limits.h>
#include <fcntl.h>
#include <signal.h>
#include <stdarg.h>
#include <sys/shm.h>
#include <sys/ipc.h>
#include <sys/msg.h>
#include <glib.h>
#include <math.h>

#include <dbsoundcard.h>

#include <dbchannel.h>
#include <dbdebug.h>
#include <dbaudiolib.h>

#include "prototypes.h"

	/* variables to access DBMix channels shared memory */
	int shmid, sysshmid;
	dbfsd_data * sysdata = NULL;
	local_channel * local_channels = NULL;
	/* static local_channel * cue_channels; */
	local_channel * ch = NULL;

	/* variables to describe the input audio stream */
	int format;
	int sample_rate;
	int num_channels;
	int paused;

	/* globally static audio write variables */
	float local_pitch;
	float sample1, sample2;
	float float_index;
	int   sampleindex;
	int   outlen;

	signed short output_buf[OUTPUT_BUFSIZE];  /* buffer used to hold main output to dbfsd */
	signed short cue_buf[OUTPUT_BUFSIZE];     /* buffer used to hold cue output to dbfsd */
	signed short conv_buf[OUTPUT_BUFSIZE];    /* buffer used to hold format converted input */

	extern int errno;
	extern int debug_level; /* declared in debug module */

	/*
	  DBAudio_Init - initializes this instance.  If an error occurs, returns FAILURE
	  otherwise, returns SUCCESS.

	  Parameters: 
	  fmt   - format of input data (see OSS Programmer's Manual 
	  http://www.opensound.com/pguide/index.html)
	  rte   - input data rate expressed as a whole integer. i.e. 44.1 kHz = 44100
	  numch - number of channels in the input data
	  type  - DBMix channel type, see enum channel_type_e in channel.h
	  chindex - if 0 returns the first available channel, otherwise tries to use the
	  channel associated with (chindex -1). If this channel is not free,
	  FAILURE is returned.

	  Pre Condition:
	  - None

	  Post Condition
	  - Shared memory channel data created by dbfsd is attached to this process..
	  - From this data, determine the next free/unused channel.  The pointer
	  to this channel is stored in the static global ch variable.
	  - Open communication and signal pipes for communicating with dbfsd

	  Note: this function makes use of goto statements.  More specifically,
            a label called cleanup exists at the end of the function allowing
            a single cleanup point in the function.  This is a result of the 
			coding standards my group at work enforces. Despite what you were
			taught about the evils of goto, this method of their use makes the
			code cleaner and more elegant, otherwise the exact same shmdt() 
			statements would be scattered half a dozen times through out the code.
			The single "goto cleanup" line replaces 4-10 lines of repetitive
			cleanup code each time it is used.
	*/
	int DBAudio_Init(char * name, int fmt, int rte, int numch,
					 enum channel_type_e type, int chindex)
	{
		char init_buffer[PIPE_BUF];
		int templen;

#ifdef DBMIX_DEBUG
		debug_level = 1;
#else
		debug_level = 0;
#endif

		/* init errno to SUCCESS to maintain error state */
		errno = SUCCESS;
		
		if (ch != NULL)
		{
			goto cleanup;
		}

		if (local_channels != NULL)
		{
			goto cleanup;
		}
		
		if (sysdata != NULL)
		{
			goto cleanup;
		}

		/* init variables used by DBAudio_Write() */
		local_pitch = 1.0;
		sample1 = sample2 = 0.0;
		float_index = 0.0;
		sampleindex = 0;
		outlen = 0;
		
		/* verify format */
		if (fmt != 0)
		{
			format = fmt;
		}
		else
		{
			format = AFMT_S16_NE;
		}

		/* verify sample rate */
		if (rte == 0)
		{
			sample_rate = DB_SAMPLE_RATE;
		} 
		else 
		{
			if ((rte >= 8000) && (rte < 49000))
			{
				sample_rate = rte;
			}
			else
			{
				errno = ERROR_BAD_SAMPLERATE;
				goto cleanup;
			}
		}

		/* verify number of channels */
		switch(numch)
		{
			case 0: 
				num_channels = 2; 
				break;  /* default to stereo */

			case 1: 
				num_channels = 1; 
				break;  /* mono */

			case 2: 
				num_channels = 2; 
				break;  /* stereo */

			default: 
				errno = ERROR_BAD_NUMCH; 
				goto cleanup;
		}


		/* Get system data */
		{
			sysshmid = shmget((key_t) DB_SYSTEM_SM_KEY, sizeof(dbfsd_data), 
							  0666 | O_RDWR );
		
			if (sysshmid == -1) 
			{
				Error("DBAudioLib ERROR: could not create shared memory for system data.\n Is dbfsd running?");
				errno = ERROR_INIT_FAILURE;
				goto cleanup; 
			}
		
			sysdata = shmat(sysshmid,(void *)0, 0);
		
			if ((int)sysdata == -1)
			{
				Error("DBAudioLib ERROR: error attaching system data shared memory.");	
				errno = ERROR_INIT_FAILURE;
				goto cleanup;
			}
		
			/* verify that there is a free channel into dbfsd */
			if (sysdata->free_channel_index == -1)
			{
				ch = NULL;
				
				Error("DBAudio_Init: no free channels.");

				if (shmdt(sysdata) == -1)
				{
					Error("DBAudio_Init: could not detach system data memory segment.");
					errno = ERROR_INIT_FAILURE;
					goto cleanup;
				}
			
				errno = ERROR_NO_FREE_CHANNELS;
				goto cleanup;
			}
		}
		/* End get system data */

		/* retrieve the channel memory id */
		{
			shmid = shmget((key_t) DB_CHANNELS_SM_KEY, 
						   (sysdata->num_channels * sizeof(local_channel)), 0666);
		
			if (shmid == -1) 
			{
				Error("DBAudioLib ERROR: error creating channel shared memory.");
				errno = ERROR_INIT_FAILURE;
				goto cleanup;
			}
			else
			{
				Debug("DBAudioLib: shmid is: %d ",shmid);
			}
		
			/* attach the channel memory segment*/
			local_channels = (local_channel *) shmat(shmid,(void *)0, 0);
		
			if ((int)local_channels == -1)
			{
				Error("DBAudioLib ERROR: error attaching channel shared memory.");

				errno = ERROR_INIT_FAILURE;
				goto cleanup;
			}

			/*  get channel id */		
			if (chindex == 0)
			{
				ch = &(local_channels[sysdata->free_channel_index]);
			}
			else
			{
				/* chindex is begins at 1, so decrement to directly reference the 
                   channel array */
				chindex--;

				if ((chindex < 0) || (chindex > sysdata->num_channels))
				{
					errno = ERROR_BAD_CHANNEL_ID;
					goto cleanup;
				}
			
				if (local_channels[chindex].free_channel)
				{
					ch = &(local_channels[chindex]);
				}
			}

			Debug("DBAudioLib: opening comm pipe:  %s",ch->comm_filename);
		
			/* open comm pipe to server */
			if ((ch->client_comm_fd = open(ch->comm_filename, O_WRONLY)) == -1)
			{
				Debug("%s",ch->comm_filename);

				DBAudio_perror("DBAudioLib: Error opening comm pipe.");
				errno = ERROR_INIT_FAILURE;
				goto cleanup;
			}

			Debug("comm fd is: %d\n",ch->client_comm_fd);

			if (CUE_ENABLED)
			{
				printf("flags is %d\n",ch->channel_flags);

				/* open cue_pipe to be NON BLOCKING -- very important since the server does
				   not check the cue channel with its select() call*/
				Debug("DBAudioLib: opening cue pipe: %s",ch->cue_filename);

				if ((ch->client_cue_fd = open(ch->cue_filename, O_WRONLY | O_NONBLOCK)) == -1)
				{
					Debug("%s",ch->cue_filename);
					DBAudio_perror("DBAudioLib: Error opening cue pipe.");
					errno = ERROR_INIT_FAILURE;
					goto cleanup;
				}

				Debug("Cue fd is: %d\n",ch->client_cue_fd);
			}
		
			ch->left_gain = DEFAULT_GAIN;
			ch->right_gain = DEFAULT_GAIN;
			ch->cue = FALSE;
			ch->channel_type = type;
			ch->base_pitch = DEFAULT_PITCH;
		
			switch(ch->channel_type)
			{
				case PIPE_CHANNEL:
					Debug("DBAudioLib: Channel type: PIPE CHANNEL");
					break;
				default:
					break;
			}

			/* update channel name  check against 48 to allow for channel index */
			DBAudio_Set_Channel_Name(name);
		} /* end retrieve the channel memory id */

		DBAudio_Set_Rate(rte);

		/* clear msg queue */
		if (ch->msg_q_id != -1)
		{
			dbfsd_msg msg;

			while (msgrcv(ch->msg_q_id, &msg,sizeof(dbfsd_msg) - sizeof(long int),0,IPC_NOWAIT) != -1)
			{
				Debug("DBAudio_Init: cleared message %d",msg.msg_type);
			}
		}

		/* init message handler to NULL */
		ch->message_handler = NULL;

		/* init sample buffer */
		{
			ch->sampler_state = SAMPLER_OFF;
			ch->sampler_bufsize = (ch->sampler_time   /* number of seconds */
								  * (DB_SAMPLE_SIZE * DB_CHANNEL_NUM_CHANNELS) /* size of 1 stereo sample */
								  * DB_SAMPLE_RATE); /* num samples per second */
			ch->sampler_size = 0;
			ch->sampler_readoffset = 0;
			ch->sampler_startoffset = 0;
			ch->sampler_buf = (char *) malloc(ch->sampler_bufsize);

			Debug("sampler bufsize is: %d",ch->sampler_bufsize);
		}
		
		/* write an empty buffer to the mixer to init this channel */
		{
			memset(init_buffer,0,PIPE_BUF);
			ch->cue = TRUE;
			Debug("writing init buffer... \n");
			templen = PIPE_BUF;
			
			if (DBAudio_Write(init_buffer,templen) == FAILURE)
			{
				DBAudio_perror("Error writing init data to channel:: ");
			}
			
			Debug("done\n");
			ch->cue = FALSE;
			
			Debug("DBAudioLib: dbaudio_init complete.");
		}

	cleanup:

		/* ENOMSG is a side effect of opening/clearing the message queue, so remap it */
		if (errno == ENOMSG) errno = SUCCESS;

		/* if error, retur to initial state */
		if (errno != SUCCESS)
		{
			ch = NULL;

			if (local_channels != NULL)
			{
				shmdt(local_channels);
			}

			if (sysdata != NULL)
			{
				shmdt(sysdata);
			}

			return FAILURE;
		}

		return SUCCESS;
	}


#ifdef __cplusplus
}
#endif


