/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2016 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */


/**
   \file cdw_safe_input_line.c

   Widget that allows entering text into single line input field.

   The widget doesn't accept string with insecure characters, as
   defined in cdw_string.h -> CDW_STRING_UNSAFE_CHARS_STRING.

   Insecure characters are reported through error message displayed in
   dynamic label. The dynamic label widget must be associated by
   client code with the safe input line widget using
   cdw_safe_input_line_bind_message_area().

   Text entered into the input line is not stored in this widget. It
   is stored in ncurses form field associated with this widget (the
   'field' argument to _new() function).

   The widget provides _driver() function that should be used to
   control user's keyboard actions when keyboard focus is on the
   widget.
*/



#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <ctype.h> /* isprint() */

#include "cdw_safe_input_line.h"
#include "cdw_widgets.h"
#include "gettext.h"
#include "cdw_string.h"
#include "cdw_debug.h"



static void     cdw_safe_input_line_print_message(CDW_SAFE_INPUT_LINE *line, char *insecure);
static void     cdw_safe_input_line_prepare_message(char *message, int len_max, char *insecure);
static int      cdw_safe_input_line_form_driver(CDW_SAFE_INPUT_LINE *line);
static cdw_rv_t cdw_safe_input_line_test_content_security(CDW_SAFE_INPUT_LINE *line, char *insecure);
static void     cdw_safe_input_line_add_return_key(cdw_widget_t *widget, int key);





/**
   \brief Create new safe input line widget

   \date Function's top-level comment reviewed on 2014-03-30
   \date Function's body reviewed on 2014-03-30

   Create widget that consists of a single line input field.

   The input field has width \p n_cols, and is placed in \p
   parent. The \p parent is a subwindow used in
   set_form_sub(form, subwindow) call.

   Exact position of the widget is specified by ncurses FIELD \p
   field. The \p field must be created by caller before calling the
   function. The \p field is owned by the caller. This widget may
   alter text string stored by the \p field, but it won't attempt to
   destroy/deallocate the \p field.

   In addition to \p field, the widget also uses ncurses FORM \p form.
   \p form needs to be a pointer to pointer because in some cases a
   ncurses form is not created yet at time of calling this function.

   The field may be pre-filled with \p initial_content.  You may put a
   restriction on number of characters that can be entered into the
   field by setting \p chars_max to value > 0.

   Input behavior may be tuned by \p input_type:
   none/hidden/integer/numeric, see cdw_ncurses.h for specifics.

   \param parent - window in which to display the widget
   \param form - ncurses form, to which \p fields will belong to
   \param field - ncurses field
   \param n_cols - width of the input line
   \param initial_content - initial data displayed in input line
   \param input_type - type of input
   \param chars_max - limit on number of chars that can be entered by user

   \return NULL pointer on errors
   \return pointer to new widget on success
*/
CDW_SAFE_INPUT_LINE *cdw_safe_input_line_new(WINDOW *parent, FORM **form, FIELD *field, int n_cols, const char *initial_content, int input_type, int chars_max)
{
	CDW_SAFE_INPUT_LINE *line = (CDW_SAFE_INPUT_LINE *) malloc(sizeof(CDW_SAFE_INPUT_LINE));
	if (!line) {
		cdw_vdm ("ERROR: failed to allocate memory for safe input line widget\n");
		return (CDW_SAFE_INPUT_LINE *) NULL;
	}

	cdw_widget_init(&line->widget);
	cdw_widget_add_return_keys(&line->widget, CDW_KEY_ESCAPE, CDW_KEY_TAB, CDW_KEY_BTAB, KEY_UP, KEY_DOWN, 0);
	line->widget.add_return_key = cdw_safe_input_line_add_return_key; /* Widget's custom function for adding return keys by parent cdw_form. */
	line->widget.type_id = CDW_WIDGET_ID_SAFE_INPUT_LINE;

	line->parent = parent;
	line->form = form;
	line->field = field;
	line->n_cols = n_cols;

	if (initial_content) {
		int rv = set_field_buffer(line->field, 0, initial_content);
		if (rv != E_OK) {
			cdw_vdm ("ERROR: failed to set field buffer with string \"%s\": %s\n",
				 initial_content, cdw_ncurses_error_string(rv));

			cdw_safe_input_line_delete(&line);
			return (CDW_SAFE_INPUT_LINE *) NULL;
		}
	} else {
		cdw_vdm ("WARNING: NULL initial content\n");
		/* TODO: shouldn't we call set_field_buffer(line->field, 0, "") ? */
	}

	line->chars_max = chars_max;
	line->input_type = input_type;
	line->cursor_pos = 0;

	line->attempts_max = CDW_SAFE_INPUT_LINE_ATTEMPTS_MAX;

	line->message_area = (CDW_DYNAMIC_LABEL *) NULL;

	wrefresh(line->parent);
	return line;
}





/**
   \brief Delete a safe input line widget

   \date Function's top-level comment reviewed on 2014-03-30
   \date Function's body reviewed on 2014-03-30

   Deallocate all memory associated with \p line, free the \p line
   itself, set the widget to NULL.

   Notice that the function does not touch ncurses field that has been
   passed to cdw_safe_input_line_new(), nor the string that is stored
   in the field. The field is owned by the caller.

   \param line - pointer to safe input line widget
*/
void cdw_safe_input_line_delete(CDW_SAFE_INPUT_LINE **line)
{
	cdw_assert (line, "ERROR: passing NULL pointer to the function\n");

	if (!(*line)) {
		cdw_vdm ("WARNING: passed NULL widget to the function\n");
		return;
	}

	free(*line);
	*line = (CDW_SAFE_INPUT_LINE *) NULL;

	return;
}





/**
   \brief Assign dynamic label widget to given safe input line widget

   \date Function's top-level comment reviewed on 2014-03-30
   \date Function's body reviewed on 2014-03-30

   Safe input line widget needs a dynamic label widget to display
   error message. There needs to be a way to bind the two widgets
   together. This function does the binding.

   \param line - safe input line widget
   \param label - dynamic label widget
*/
void cdw_safe_input_line_bind_message_area(CDW_SAFE_INPUT_LINE *line, CDW_DYNAMIC_LABEL *label)
{
	cdw_assert (label, "ERROR: dynamic label for message area is NULL\n");

	line->message_area = label;

	return;
}





/**
   \brief Top-level driver for input line widget

   \date Function's top-level comment reviewed on 2016-02-07
   \date Function's body reviewed on 2016-02-07

   Driver for safe input line widget - it handles keyboard input when
   keyboard focus is on the widget.

   Unless configured as return key, Enter key is not treated as a
   special key - it is *not* treated as "I'm telling you to accept
   this input" command from user.

   Upon detecting TAB, BackTAB, Down or Up key the function checks the
   string entered by user for insecure characters, and if there are
   any, displays an error message. Up to
   CDW_SAFE_INPUT_LINE_ATTEMPTS_MAX attempts to enter a valid string
   are allowed.

   \param line - widget to be controlled
   \param dummy - unused parameter

   \return CDW_KEY_TAB, KEY_BTAB, KEY_DOWN or KEY_UP if user pressed any of those keys
   \return CDW_KEY_ESCAPE if user pressed Escape key
   \return CDW_KEY_ENTER if user pressed Enter key, and the key is configured as return key
   \return KEY_EXIT on errors
*/
int cdw_safe_input_line_driver(CDW_SAFE_INPUT_LINE *line, __attribute__((unused)) void *dummy)
{
	cdw_assert (line, "ERROR: NULL pointer to safe input line\n");
	cdw_assert (line->widget.type_id == CDW_WIDGET_ID_SAFE_INPUT_LINE, "ERROR: this is not a safe input line widget\n");
	cdw_assert (line->parent, "ERROR: can't control safe input line without parent window\n");

	int key = 'a'; /* Safe initial value. */

	for (int i = 0; i < line->attempts_max; i++) {

		curs_set(1);
		key = cdw_safe_input_line_form_driver(line);
		curs_set(0);

		if (key == CDW_KEY_ESCAPE) {
			/* This is also return key, but deserves
			   special treatment (i.e. no action at all).
			   Handle it before any other return key. */
			break;
		} else if (cdw_widget_is_return_key(&line->widget, key)) {
			/* Don't allow leaving the field when there is
			   unsafe input in it. */

			char insecure[2];
			cdw_rv_t crv = cdw_safe_input_line_test_content_security(line, insecure);
			if (crv == CDW_OK) {
				/* String is secure, keyboard focus is
				   allowed to leave the widget. */
				break;
			} else if (crv == CDW_ERROR) {
				/* Internal error. */
				key = KEY_EXIT;
				break;
			} else { /* crv == CDW_NO, string is not secure. */
				cdw_safe_input_line_print_message(line, insecure);
				if (i == line->attempts_max - 1) {
					/* Too many attempts to enter
					   correct string. */
					key = CDW_KEY_ESCAPE;
					break;
				} else {
					/* Allow user to fix the
					   insecure string. */
				}
			}
		} else if (key == KEY_EXIT) {
			/* Invalid form content. */
			cdw_vdm ("ERROR: failed to get field buffer\n");
			key = KEY_EXIT;
			break;
		} else {
			/* May be a control key - to be investigated
			   by parent. */
			break;
		}
	}

	return key;
}





/**
   \brief Print (in line's message field) message about insecure character

   \date Function's top-level comment reviewed on 2014-03-30
   \date Function's body reviewed on 2014-03-30

   If user enters insecure character into safe input line, the widget
   won't accept entered string. The widget has to inform user why
   string hasn't been accepted, and it does it by printing "insecure
   character X" message in a dynamic label (message area) associated
   with the \p line.

   The message should contain information about the insecure character
   - pass it using a regular string \p insecure of length=1,
   terminated with NUL.

   \param line - safe input line widget
   \param insecure - string representing insecure character
*/
void cdw_safe_input_line_print_message(CDW_SAFE_INPUT_LINE *line, char *insecure)
{
	cdw_assert (line->message_area, "ERROR: message area is NULL\n");

	/* Error message can't be longer than width of associated
	   message area. */
	int len_max = line->message_area->n_cols;
	char *message = (char *) malloc((size_t) len_max + 1);
	if (!message) {
		cdw_vdm ("ERROR: failed to malloc() message buffer\n");
		return;
	}

	/* Produce error message that is as verbose as possible, but
	   no longer than len_max characters. */
	cdw_safe_input_line_prepare_message(message, len_max, insecure);

	/* At this point we have in "message" one of three possible
	   message string without truncation, or - as worst case
	   scenario - last of the message strings (the shortest one)
	   with truncation. */

	cdw_dynamic_label_set_new_temporary(line->message_area, message,
					    CDW_SAFE_INPUT_LINE_ERROR_MESSAGE_TIME,
					    CDW_COLORS_WARNING);

	return;
}





/**
   \brief Prepare error message about insecure character

   \date Function's top-level comment reviewed on 2014-03-30
   \date Function's body reviewed on 2014-03-30

   Since the length of message area may be less than sufficient for
   fully verbose error message, let's have N variants of error message
   with the same meaning, but decreasing lengths.  Let's also hope
   that at least one of them will be no longer than width of message
   area.

   This function tries up to 3 times to produce an error message with
   decreasing length.

   Message is returned through \p message. It will contain an insecure
   character \p insecure (unless the message will be truncated so much
   that the character won't fit into the message).

   \param message - buffer to fill with error message
   \param len_max - maximal length of message to be produced
   \param insecure - string with insecure character to be included in the message
*/
void cdw_safe_input_line_prepare_message(char *message, int len_max, char *insecure)
{
	/* +1 for ending '\0'.*/
	/* 2TRANS: this is error message in dialog window, "%s" is
	   string with single insecure character; keep short */
	int n = snprintf(message, (size_t) len_max + 1, _("ERROR: insecure char %s "), insecure);
	if (n < 0) {
		cdw_vdm ("ERROR: snprintf() returns negative value for string #1\n");
	} else if (n >= 0 && n <= len_max) {
		/* snprintf() printed at most len_max chars (excluding
		   ending '\0'); message #1 was short enough, accept
		   it */
		return;
	} else {
		; /* Pass. Try creating shorter message. */
	}


	/* Second attempt, with shorter message. */


	/* +1 for ending '\0'.*/
	/* 2TRANS: this is error message in dialog window, "%s" is
	   string with single insecure character; keep short */
	n = snprintf(message, (size_t) len_max + 1, _("Insecure char %s "), insecure);
	if (n < 0) {
		cdw_vdm ("ERROR: snprintf() returns negative value for string #2\n");
	} else if (n >= 0 && n <= len_max) {
		/* snprintf() printed at most len_max chars (excluding
		   ending '\0'); message #2 was short enough, accept
		   it */
		return;
	} else {
		; /* Pass. Try creating shorter message. */
	}


	/* Third attempt with even shorter message. */


	/* +1 for ending '\0'. */
	/* 2TRANS: this is error message in dialog window,
	   "%s" is string with single insecure character; keep
	   short */
	n = snprintf(message, (size_t) len_max + 1, _("Insec. char %s"), insecure);
	if (n < 0) {
		cdw_vdm ("ERROR: snprintf() returns negative value for string #3\n");
	} else if (n >= 0 && n <= len_max) {
		/* snprintf() printed at most len_max chars
		   (excluding ending '\0'); message #3 was
		   short enough, accept it */
		return;
	} else {
		/* Unfortunately the error message will be truncated. */
		cdw_vdm ("ERROR: snprintf() truncates message; limit of printable chars = %d, will print %d chars\n",
			 len_max, n);
		cdw_vdm ("ERROR: printed string:            \"%s\"\n", message);
	}

	return;
}





/**
   \brief Put a new string into safe input line

   \date Function's top-level comment reviewed on 2014-03-30
   \date Function's body reviewed on 2014-03-30

   You can pass initial string to widget's constructor, but it is sometimes
   also necessary to update the widget with a string after the widget has
   been created. The function does just that - enters new content into
   given safe input line.

   As said elsewhere, the string is not stored directly in the widget,
   but in ncursed FIELD field. Both string and field are owned by
   client code.

   \param line - widget to update
   \param content - string to put into input line

   \return CDW_OK on success
   \return CDW_ERROR on failure
*/
cdw_rv_t cdw_safe_input_line_set_content(CDW_SAFE_INPUT_LINE *line, const char *content)
{
	cdw_assert (content, "ERROR: string passed to function is NULL\n");

	int rv = set_field_buffer(line->field, 0, content);

#ifndef NDEBUG
	cdw_assert (rv == E_OK, "ERROR: failed to set buffer of input line, error = %s\n", cdw_ncurses_error_string(rv));
	char *buffer_string = cdw_string_rtrim(field_buffer(line->field, 0));
	cdw_assert (!strcmp(buffer_string, content),
		    "ERROR: buffer string and input string are different:\nbuffer string: \"%s\"\ninput string: \"%s\"\n",
		    buffer_string, content);
#endif

	wrefresh(line->parent);

	if (rv == E_OK) {
		return CDW_OK;
	} else {
		return CDW_ERROR;
	}
}





/**
   \brief Low-level input line driver

   \date Function's top-level comment reviewed on 2016-02-07
   \date Function's body reviewed on 2016-02-07

   Low level driver used by cdw_safe_input_line_driver(). It deals
   directly with ncurses form/field that is the heart of the safe
   input line widget.

   Function returns on configured return keys (TAB, BackTAB, Up, Down,
   Escape. Also on Enter (if configured by client code)).

   Returned key should be interpreted by caller, and content of the
   widget should be extracted from widget and used as caller wants it.
   The widget does not try to actively pass its content to the caller.

   \param line - widget to be controlled

   \return CDW_KEY_ESCAPE if user pressed ESCAPE key.
   \return CDW_KEY_TAB, KEY_BTAB, KEY_DOWN or KEY_UP if user pressed any of those keys
   \return CDW_KEY_ENTER if Enter is configured as return key
   \return KEY_EXIT on errors
*/
int cdw_safe_input_line_form_driver(CDW_SAFE_INPUT_LINE *line)
{
	int key = 'a';

	cdw_ncurses_set_field_cursor_pos(*(line->form), line->cursor_pos);

	while (!cdw_widget_is_return_key(&line->widget, key)) {
		int rv = E_OK;
		key = wgetch(line->parent);
		switch (key) {
			case KEY_LEFT:
				rv = form_driver(*(line->form), REQ_PREV_CHAR);
				if (rv == E_OK) {
					line->cursor_pos--;
				}
				break;
			case KEY_RIGHT:
				rv = form_driver(*(line->form), REQ_NEXT_CHAR);
				if (rv == E_OK) {
					line->cursor_pos++;
				}
				break;
			case KEY_BACKSPACE:
				/* If cursor is at the beginning of
				   input field, this can be
				   interpreted as "move to previous
				   field in a form". I don't want to
				   move to previous field. */

				/* Test current cursor position. */
				rv = form_driver(*(line->form), REQ_PREV_CHAR);
				if (rv == E_REQUEST_DENIED) {
					/* We are at the beginning of input field. Ignore the key. */
					continue;
				} else {
					/* Restore original position of cursor. */
					rv = form_driver(*(line->form), REQ_NEXT_CHAR);
					cdw_assert (rv == E_OK, "ERROR: can't move back the cursor.\n");

					/* Delete the char as requested. */
					rv = form_driver(*(line->form), REQ_DEL_PREV);
					if (rv == E_OK) {
						line->cursor_pos--;
					}
					break;
				}

			case KEY_DC:
				rv = form_driver(*(line->form), REQ_DEL_CHAR);
				break;
		        case KEY_HOME:
				rv = form_driver(*(line->form), REQ_BEG_LINE);
				if (rv == E_OK) {
					line->cursor_pos = 0;
				}
				break;
		        case KEY_END:
				rv = form_driver(*(line->form), REQ_END_LINE);
				if (rv == E_OK) {
					line->cursor_pos = cdw_ncurses_get_field_cursor_pos(*(line->form));
				}
				break;
			default:
				if (isprint(key)) {
					/* Push entered text to form. */
					rv = form_driver(*(line->form), key);
					if (rv == E_OK) {
						line->cursor_pos++;
					}
					wrefresh(line->parent);
				}
				break;
		} /* switch () */

		if (rv != E_OK) {
			cdw_vdm ("WARNING: form_driver(..., key) returns %s for key %s\n",
				 cdw_ncurses_error_string(rv), cdw_ncurses_key_label(key));
		}
	}

	if (key == CDW_KEY_ESCAPE) { /* This is the only "quit" key for this widget. 'Q'/'q' doesn't work. */

		/* No need to validate field content since Escape will
		   discard the contents. Notice that not doing
		   validation/flushing may result in not-up-to-data
		   contents of field buffer. Remember about it when
		   you decide to print the buffer on Escape. */
		return key;
	}

	/* Flush form buffer */
	int rv = form_driver(*(line->form), REQ_VALIDATION);
	if (rv != E_OK) {
		cdw_vdm ("WARNING: form_driver(..., REQ_VALIDATION) returns %s for key %s\n",
			 cdw_ncurses_error_string(rv), cdw_ncurses_key_label(key));
		return KEY_EXIT;
	} else {
		/* Escape, Tab, BTab, Down or Up (or perhaps Enter). */
		return key;
	}
}





/**
   \brief Get string from input line

   \date Function's top-level comment reviewed on 2014-03-30
   \date Function's body reviewed on 2014-03-30

   Function returns freshly allocated string with content of \p
   line. The string is owned by caller.  Function returns NULL on
   errors.

   The function does not perform validity or security checks of the
   string. String is trimmed from right side.

   \return copy of string from \p line on success
   \return NULL on errors
*/
char *cdw_safe_input_line_get_content(CDW_SAFE_INPUT_LINE *line)
{
	char *s = cdw_ncurses_get_field_string(line->field);
	if (s) {
		return strdup(s);
	} else {
		return (char *) NULL;
	}
}





/**
   \brief Check if a widget contains insecure characters

   \date Function's top-level comment reviewed on 2014-03-30
   \date Function's body reviewed on 2014-03-30

   The function only checks if there are any insecure characters
   stored in \p line. It does not print any error message. First
   insecure character encountered in \p line is stored in \p insecure
   string.

   The \p insecure string should be a two-char buffer (the second char
   in the buffer is a space for terminating NUL).

   \param line - widget to be checked
   \param insecure - output buffer for insecure character

   \return CDW_OK if content of string is secure
   \return CDW_NO if content of string is insecure
   \return CDW_ERROR on errors
*/
cdw_rv_t cdw_safe_input_line_test_content_security(CDW_SAFE_INPUT_LINE *line, char *insecure)
{
	char *local_buffer = (char *) NULL;
	/* all that can go wrong at this point is malloc()
	   failure - return value will be CDW_ERROR */
	cdw_rv_t crv = cdw_ncurses_get_field_buffer(line->field, &local_buffer, (size_t) line->chars_max);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to get field buffer with cdw_ncurses_get_field_buffer()\n");
		return CDW_ERROR;
	}

	/* TODO: Placing security check here makes input line less
	   versatile, as I can imagine places where security checking
	   may be unwanted, or may need different implementation. I
	   have to update implementation of input line so that it is
	   more configurable and universal. */

	cdw_rv_t sec = cdw_string_security_parser(local_buffer, insecure);
	if (sec == CDW_OK) {
		cdw_vdm ("INFO: \"%s\" is secure\n", local_buffer);
	} else if (sec == CDW_NO) {
		cdw_vdm ("ERROR: insecure char in \"%s\"\n", local_buffer);
 	} else {
		cdw_vdm ("ERROR: security parser returns CDW_ERROR for string \"%s\"\n", local_buffer);
		sec = CDW_ERROR; /* Just in case if we haven't caught some other return value. */
	}

	free(local_buffer);
	local_buffer = (char *) NULL;
	return sec;
}




void cdw_safe_input_line_add_return_key(cdw_widget_t *widget, int key)
{
	cdw_assert (widget, "ERROR: \"widget\" argument is NULL\n");
	cdw_assert (widget->n_return_keys < CDW_WIDGET_N_RETURN_KEYS_MAX,
		    "ERROR: there are already %d / %d return keys in the widget, can't add another one\n",
		    widget->n_return_keys, CDW_WIDGET_N_RETURN_KEYS_MAX);
	cdw_assert (key != 0, "ERROR: trying to add key == 0, but 0 is an initializer value\n");

	if (isprint(key)) {
		/* Keys satisfying 'isprint()' should be entered into
		   input line, so they can't be return keys. */
		return;
	}

	cdw_widget_add_return_key(widget, key);

	return;
}
