/*
 * Copyright (c) 2021, 2023-2024 Arm Limited.
 *
 * SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
#ifdef __aarch64__

#include "arm_gemm.hpp"
#include "../../utils.hpp"

#include <cassert>
#include <cstdint>

namespace arm_gemm {

void a64_hybrid_s8qa_mmla_4x16 (
    unsigned int num_strings, const unsigned int *string_lengths, IndirectInputArg<int8_t> A_arg,
    size_t M, size_t N, const int8_t *B_ptr, IndirectOutputArg<int8_t> output_arg,
    const Requantize32 *qp, const int32_t *col_bias, unsigned int
)
{
    struct KernelArgs {
        unsigned int num_strings = {};
        const unsigned int *string_lengths = {};
        size_t N = {};
        const int8_t *B_ptr = {};
        size_t output_offset = {};
        size_t input_initial_col = {};
        size_t input_offset = {};
        void *output_ptr = {};
    } ka;

    unsigned long flags=0;
    void *input_ptr;

    if (output_arg.is_indirect) {
        ka.output_ptr=(void *)(output_arg.indirect.ptr);
        ka.output_offset=output_arg.indirect.offset;
        flags |= 0x4;
    } else {
        ka.output_ptr=(void *)(output_arg.direct.base);
        ka.output_offset=output_arg.direct.stride;
    }

    if (A_arg.is_indirect) {
        input_ptr=(void *)(A_arg.indirect.ptr);
        ka.input_offset=A_arg.indirect.start_row;
        ka.input_initial_col=A_arg.indirect.start_col;
        flags |= 0x8;
    } else {
        assert(num_strings==1);
        input_ptr=(void *)(A_arg.direct.base);
        ka.input_offset=A_arg.direct.stride;
    }
    ka.num_strings = num_strings;
    ka.string_lengths = string_lengths;
    ka.N = N;
    ka.B_ptr = B_ptr;
    if (qp->c_offset > qp->minval) {
        flags |= 0x20;
    }
    __asm__ __volatile__(
      "1:"  // Row loop
      "cmp %x[M], #0x4\n"
      "bge 97f\n"
      "cmp %x[M], #0x2\n"
      "bgt 65f\n"
      "beq 33f\n"
      "mov x10, %x[col_bias]\n"
      "movi v11.4s, #0x0\n"
      "movi v15.16b, #0x1\n"
      "bic %x[flags], %x[flags], #0x80000000\n"
      "ldr x9, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x28, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x27, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "2:"  // Height 1: Column loop
      "movi v16.4s, #0x0\n"
      "movi v17.4s, #0x0\n"
      "movi v18.4s, #0x0\n"
      "movi v19.4s, #0x0\n"
      "movi v20.4s, #0x0\n"
      "movi v21.4s, #0x0\n"
      "movi v22.4s, #0x0\n"
      "movi v23.4s, #0x0\n"
      "3:"  // Height 1: setup done
      "mov x26, #0x0\n"
      "4:"  // Height 1: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w25, [x20, x26, LSL #0x2]\n"
      "tbz %x[flags], #3, 5f\n"
      "ldr x20, [%x[input_ptr], x26, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x24, [x20, #0x0]\n"
      "cbnz x26, 6f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x24, x24, x20\n"
      "b 6f\n"
      "5:"  // Height 1: setup direct input
      "mov x24, %x[input_ptr]\n"
      "6:"  // Height 1: input setup done
      "cmp x25, #0x10\n"
      "blt 11f\n"
      "ldr q1, [x24, #0x0]\n"
      "ldr q5, [x28, #0x0]\n"
      "cmp x25, #0x20\n"
      "ldr q6, [x28, #0x10]\n"
      "ldr q7, [x28, #0x20]\n"
      "ldr q8, [x28, #0x30]\n"
      "ldr q9, [x28, #0x40]\n"
      "ldr q10, [x28, #0x50]\n"
      "ldr q4, [x28, #0x60]\n"
      "blt 9f\n"
      "7:"  // Height 1: Multiply loop: Main loop head
      "add x24, x24, #0x10\n"
      "trn1 v0.2d, v1.2d, v27.2d\n"
      ".inst 0x4e85a410  // smmla v16.4s, v0.16b, v5.16b\n"
      "ldr q25, [x28, #0x70]\n"
      "trn2 v1.2d, v1.2d, v27.2d\n"
      ".inst 0x4e86a414  // smmla v20.4s, v0.16b, v6.16b\n"
      "ldr q24, [x28, #0x80]\n"
      ".inst 0x4e87a411  // smmla v17.4s, v0.16b, v7.16b\n"
      "ldr q30, [x28, #0x90]\n"
      ".inst 0x4e88a415  // smmla v21.4s, v0.16b, v8.16b\n"
      "ldr q29, [x28, #0xa0]\n"
      ".inst 0x4e89a412  // smmla v18.4s, v0.16b, v9.16b\n"
      "ldr q28, [x28, #0xb0]\n"
      ".inst 0x4e8aa416  // smmla v22.4s, v0.16b, v10.16b\n"
      "ldr q27, [x28, #0xc0]\n"
      ".inst 0x4e84a413  // smmla v19.4s, v0.16b, v4.16b\n"
      "ldr q26, [x28, #0xd0]\n"
      ".inst 0x4e99a417  // smmla v23.4s, v0.16b, v25.16b\n"
      "ldr q25, [x28, #0xe0]\n"
      ".inst 0x4e98a430  // smmla v16.4s, v1.16b, v24.16b\n"
      "ldr q24, [x28, #0xf0]\n"
      ".inst 0x4e9ea434  // smmla v20.4s, v1.16b, v30.16b\n"
      "add x28, x28, #0x100\n"
      ".inst 0x4e9da431  // smmla v17.4s, v1.16b, v29.16b\n"
      ".inst 0x4e9ca435  // smmla v21.4s, v1.16b, v28.16b\n"
      ".inst 0x4e9ba432  // smmla v18.4s, v1.16b, v27.16b\n"
      ".inst 0x4e9aa436  // smmla v22.4s, v1.16b, v26.16b\n"
      ".inst 0x4e99a433  // smmla v19.4s, v1.16b, v25.16b\n"
      ".inst 0x4e98a437  // smmla v23.4s, v1.16b, v24.16b\n"
      "tbnz %x[flags], #31, 8f\n"
      ".inst 0x4e8f940b  // sdot v11.4s, v0.16b, v15.16b\n"
      ".inst 0x4e8f942b  // sdot v11.4s, v1.16b, v15.16b\n"
      "8:"  // Height 1: Multiply loop: unique 1: skip row sum
      "ldr q1, [x24, #0x0]\n"
      "ldr q5, [x28, #0x0]\n"
      "sub x25, x25, #0x10\n"
      "ldr q6, [x28, #0x10]\n"
      "ldr q7, [x28, #0x20]\n"
      "cmp x25, #0x20\n"
      "ldr q8, [x28, #0x30]\n"
      "ldr q9, [x28, #0x40]\n"
      "ldr q10, [x28, #0x50]\n"
      "ldr q4, [x28, #0x60]\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      "bge 7b\n"
      "9:"  // Height 1: Multiply loop: Single iteration only
      "sub x25, x25, #0x10\n"
      "add x24, x24, #0x10\n"
      "trn1 v0.2d, v1.2d, v24.2d\n"
      "trn2 v1.2d, v1.2d, v24.2d\n"
      ".inst 0x4e85a410  // smmla v16.4s, v0.16b, v5.16b\n"
      "ldr q25, [x28, #0x70]\n"
      ".inst 0x4e86a414  // smmla v20.4s, v0.16b, v6.16b\n"
      "ldr q24, [x28, #0x80]\n"
      ".inst 0x4e87a411  // smmla v17.4s, v0.16b, v7.16b\n"
      "ldr q30, [x28, #0x90]\n"
      ".inst 0x4e88a415  // smmla v21.4s, v0.16b, v8.16b\n"
      "ldr q29, [x28, #0xa0]\n"
      ".inst 0x4e89a412  // smmla v18.4s, v0.16b, v9.16b\n"
      "ldr q28, [x28, #0xb0]\n"
      ".inst 0x4e8aa416  // smmla v22.4s, v0.16b, v10.16b\n"
      "ldr q27, [x28, #0xc0]\n"
      ".inst 0x4e84a413  // smmla v19.4s, v0.16b, v4.16b\n"
      "ldr q26, [x28, #0xd0]\n"
      ".inst 0x4e99a417  // smmla v23.4s, v0.16b, v25.16b\n"
      "ldr q25, [x28, #0xe0]\n"
      ".inst 0x4e98a430  // smmla v16.4s, v1.16b, v24.16b\n"
      "ldr q24, [x28, #0xf0]\n"
      ".inst 0x4e9ea434  // smmla v20.4s, v1.16b, v30.16b\n"
      "add x28, x28, #0x100\n"
      ".inst 0x4e9da431  // smmla v17.4s, v1.16b, v29.16b\n"
      ".inst 0x4e9ca435  // smmla v21.4s, v1.16b, v28.16b\n"
      ".inst 0x4e9ba432  // smmla v18.4s, v1.16b, v27.16b\n"
      ".inst 0x4e9aa436  // smmla v22.4s, v1.16b, v26.16b\n"
      ".inst 0x4e99a433  // smmla v19.4s, v1.16b, v25.16b\n"
      ".inst 0x4e98a437  // smmla v23.4s, v1.16b, v24.16b\n"
      "tbnz %x[flags], #31, 10f\n"
      ".inst 0x4e8f940b  // sdot v11.4s, v0.16b, v15.16b\n"
      ".inst 0x4e8f942b  // sdot v11.4s, v1.16b, v15.16b\n"
      "10:"  // Height 1: Multiply loop: unique 2: skip row sum
      "prfm pldl1keep, [x24, #0x80]\n"
      "11:"  // Height 1: Multiply loop: Main loop skip
      "cbz x25, 20f\n"
      "cmp x25, #0x8\n"
      "blt 14f\n"
      "12:"  // Height 1: Multiply loop: Odd block loop
      "ldr d25, [x24], #0x8\n"
      "trn1 v0.2d, v25.2d, v24.2d\n"
      "tbnz %x[flags], #31, 13f\n"
      ".inst 0x4e8f940b  // sdot v11.4s, v0.16b, v15.16b\n"
      "13:"  // Height 1: Multiply loop: unique 3: skip row sum
      "ldr q24, [x28, #0x0]\n"
      "ldr q30, [x28, #0x10]\n"
      "sub x25, x25, #0x8\n"
      "ldr q29, [x28, #0x20]\n"
      "ldr q28, [x28, #0x30]\n"
      "cmp x25, #0x8\n"
      "ldr q27, [x28, #0x40]\n"
      "ldr q26, [x28, #0x50]\n"
      "ldr q25, [x28, #0x60]\n"
      ".inst 0x4e98a410  // smmla v16.4s, v0.16b, v24.16b\n"
      "ldr q24, [x28, #0x70]\n"
      ".inst 0x4e9ea414  // smmla v20.4s, v0.16b, v30.16b\n"
      ".inst 0x4e9da411  // smmla v17.4s, v0.16b, v29.16b\n"
      ".inst 0x4e9ca415  // smmla v21.4s, v0.16b, v28.16b\n"
      "add x28, x28, #0x80\n"
      ".inst 0x4e9ba412  // smmla v18.4s, v0.16b, v27.16b\n"
      ".inst 0x4e9aa416  // smmla v22.4s, v0.16b, v26.16b\n"
      ".inst 0x4e99a413  // smmla v19.4s, v0.16b, v25.16b\n"
      ".inst 0x4e98a417  // smmla v23.4s, v0.16b, v24.16b\n"
      "bge 12b\n"
      "14:"  // Height 1: Multiply loop: Skip odd blocks
      "cbz x25, 20f\n"
      "tbz x25, #2, 16f\n"
      "ldr s1, [x24], #0x4\n"
      "tbz x25, #1, 15f\n"
      "ld1 { v1.h }[2], [x24], #0x2\n"
      "tbz x25, #0, 18f\n"
      "ld1 { v1.b }[6], [x24]\n"
      "b 18f\n"
      "15:"  // Height 1: Multiply loop: Ragged operand read: partial_1_4
      "tbz x25, #0, 18f\n"
      "ld1 { v1.b }[4], [x24]\n"
      "b 18f\n"
      "16:"  // Height 1: Multiply loop: Ragged operand read: partial_2_0
      "tbz x25, #1, 17f\n"
      "ldr h1, [x24], #0x2\n"
      "tbz x25, #0, 18f\n"
      "ld1 { v1.b }[2], [x24]\n"
      "b 18f\n"
      "17:"  // Height 1: Multiply loop: Ragged operand read: partial_1_0
      "ldr b1, [x24, #0x0]\n"
      "18:"  // Height 1: Multiply loop: Ragged operand read: Done
      "trn1 v0.2d, v1.2d, v24.2d\n"
      "tbnz %x[flags], #31, 19f\n"
      ".inst 0x4e8f940b  // sdot v11.4s, v0.16b, v15.16b\n"
      "19:"  // Height 1: Multiply loop: unique 4: skip row sum
      "ldr q24, [x28, #0x0]\n"
      "ldr q30, [x28, #0x10]\n"
      "ldr q29, [x28, #0x20]\n"
      "ldr q28, [x28, #0x30]\n"
      "ldr q27, [x28, #0x40]\n"
      "ldr q26, [x28, #0x50]\n"
      "ldr q25, [x28, #0x60]\n"
      ".inst 0x4e98a410  // smmla v16.4s, v0.16b, v24.16b\n"
      "ldr q24, [x28, #0x70]\n"
      ".inst 0x4e9ea414  // smmla v20.4s, v0.16b, v30.16b\n"
      ".inst 0x4e9da411  // smmla v17.4s, v0.16b, v29.16b\n"
      ".inst 0x4e9ca415  // smmla v21.4s, v0.16b, v28.16b\n"
      "add x28, x28, #0x80\n"
      ".inst 0x4e9ba412  // smmla v18.4s, v0.16b, v27.16b\n"
      ".inst 0x4e9aa416  // smmla v22.4s, v0.16b, v26.16b\n"
      ".inst 0x4e99a413  // smmla v19.4s, v0.16b, v25.16b\n"
      ".inst 0x4e98a417  // smmla v23.4s, v0.16b, v24.16b\n"
      "20:"  // Height 1: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x26, x26, #0x1\n"
      "cmp x26, x20\n"
      "bne 4b\n"
      "uzp1 v16.2d, v16.2d, v20.2d\n"
      "uzp1 v17.2d, v17.2d, v21.2d\n"
      "prfm pstl1keep, [x27, #0x0]\n"
      "uzp1 v18.2d, v18.2d, v22.2d\n"
      "uzp1 v19.2d, v19.2d, v23.2d\n"
      "mov v23.16b, v16.16b\n"
      "tbnz %x[flags], #31, 21f\n"
      "add x20, %x[qp], %[b_offset]\n"
      "addp v11.4s, v11.4s, v11.4s\n"
      "ld1r { v16.4s }, [x20]\n"
      "neg v16.4s, v16.4s\n"
      "dup v11.4s, v11.s[0]\n"
      "mul v11.4s, v11.4s, v16.4s\n"
      "21:"  // Height 1: skip row sum fixup
      "ldr q24, [x10, #0x0]\n"
      "ldr q22, [x10, #0x10]\n"
      "add v23.4s, v23.4s, v11.4s\n"
      "add v17.4s, v17.4s, v11.4s\n"
      "ldr q21, [x10, #0x20]\n"
      "ldr q20, [x10, #0x30]\n"
      "add v18.4s, v18.4s, v11.4s\n"
      "add v19.4s, v19.4s, v11.4s\n"
      "add x20, %x[qp], %[per_layer_mul]\n"
      "orr %x[flags], %x[flags], #0x80000000\n"
      "ld1r { v16.4s }, [x20]\n"
      "add v23.4s, v23.4s, v24.4s\n"
      "add v17.4s, v17.4s, v22.4s\n"
      "add x20, %x[qp], %[per_layer_right_shift]\n"
      "add x10, x10, #0x40\n"
      "ld1r { v0.4s }, [x20]\n"
      "add v18.4s, v18.4s, v21.4s\n"
      "add v19.4s, v19.4s, v20.4s\n"
      "sqrdmulh v23.4s, v23.4s, v16.4s\n"
      "sqrdmulh v17.4s, v17.4s, v16.4s\n"
      "sqrdmulh v18.4s, v18.4s, v16.4s\n"
      "sqrdmulh v19.4s, v19.4s, v16.4s\n"
      "tbz %x[flags], #5, 22f\n"
      "and v22.16b, v23.16b, v0.16b\n"
      "and v21.16b, v17.16b, v0.16b\n"
      "and v20.16b, v18.16b, v0.16b\n"
      "and v16.16b, v19.16b, v0.16b\n"
      "sshr v22.4s, v22.4s, #0x1f\n"
      "sshr v21.4s, v21.4s, #0x1f\n"
      "sshr v20.4s, v20.4s, #0x1f\n"
      "sshr v16.4s, v16.4s, #0x1f\n"
      "sqadd v23.4s, v23.4s, v22.4s\n"
      "sqadd v17.4s, v17.4s, v21.4s\n"
      "sqadd v18.4s, v18.4s, v20.4s\n"
      "sqadd v19.4s, v19.4s, v16.4s\n"
      "22:"  // Height 1: no shift correction
      "add x21, %x[qp], %[c_offset]\n"
      "srshl v23.4s, v23.4s, v0.4s\n"
      "srshl v17.4s, v17.4s, v0.4s\n"
      "add x20, %x[qp], %[maxval]\n"
      "ld1r { v21.4s }, [x21]\n"
      "ld1r { v20.4s }, [x20]\n"
      "srshl v18.4s, v18.4s, v0.4s\n"
      "srshl v19.4s, v19.4s, v0.4s\n"
      "add x20, %x[qp], %[minval]\n"
      "cmp x9, #0x10\n"
      "ld1r { v16.4s }, [x20]\n"
      "add v23.4s, v23.4s, v21.4s\n"
      "add v17.4s, v17.4s, v21.4s\n"
      "add v18.4s, v18.4s, v21.4s\n"
      "add v19.4s, v19.4s, v21.4s\n"
      "smin v23.4s, v23.4s, v20.4s\n"
      "smin v17.4s, v17.4s, v20.4s\n"
      "smin v18.4s, v18.4s, v20.4s\n"
      "smin v19.4s, v19.4s, v20.4s\n"
      "smax v23.4s, v23.4s, v16.4s\n"
      "smax v17.4s, v17.4s, v16.4s\n"
      "smax v18.4s, v18.4s, v16.4s\n"
      "smax v19.4s, v19.4s, v16.4s\n"
      "uzp1 v23.8h, v23.8h, v17.8h\n"
      "uzp1 v16.8h, v18.8h, v19.8h\n"
      "uzp1 v23.16b, v23.16b, v16.16b\n"
      "bge 31f\n"
      "tbz x9, #3, 26f\n"
      "str d23, [x27], #0x8\n"
      "tbz x9, #2, 24f\n"
      "st1 { v23.s }[2], [x27], #0x4\n"
      "tbz x9, #1, 23f\n"
      "st1 { v23.h }[6], [x27], #0x2\n"
      "tbz x9, #0, 30f\n"
      "st1 { v23.b }[14], [x27]\n"
      "b 30f\n"
      "23:"  // Height 1: Partial direct writeback: partial_1_12
      "tbz x9, #0, 30f\n"
      "st1 { v23.b }[12], [x27]\n"
      "b 30f\n"
      "24:"  // Height 1: Partial direct writeback: partial_2_8
      "tbz x9, #1, 25f\n"
      "st1 { v23.h }[4], [x27], #0x2\n"
      "tbz x9, #0, 30f\n"
      "st1 { v23.b }[10], [x27]\n"
      "b 30f\n"
      "25:"  // Height 1: Partial direct writeback: partial_1_8
      "tbz x9, #0, 30f\n"
      "st1 { v23.b }[8], [x27]\n"
      "b 30f\n"
      "26:"  // Height 1: Partial direct writeback: partial_4_0
      "tbz x9, #2, 28f\n"
      "str s23, [x27], #0x4\n"
      "tbz x9, #1, 27f\n"
      "st1 { v23.h }[2], [x27], #0x2\n"
      "tbz x9, #0, 30f\n"
      "st1 { v23.b }[6], [x27]\n"
      "b 30f\n"
      "27:"  // Height 1: Partial direct writeback: partial_1_4
      "tbz x9, #0, 30f\n"
      "st1 { v23.b }[4], [x27]\n"
      "b 30f\n"
      "28:"  // Height 1: Partial direct writeback: partial_2_0
      "tbz x9, #1, 29f\n"
      "str h23, [x27], #0x2\n"
      "tbz x9, #0, 30f\n"
      "st1 { v23.b }[2], [x27]\n"
      "b 30f\n"
      "29:"  // Height 1: Partial direct writeback: partial_1_0
      "str b23, [x27, #0x0]\n"
      "30:"  // Height 1: Partial direct writeback: Done
      "b 32f\n"
      "31:"  // Height 1: Full writeback
      "str q23, [x27, #0x0]\n"
      "add x27, x27, #0x10\n"
      "32:"  // Height 1: Writeback done
      "subs x9, x9, #0x10\n"
      "bgt 2b\n"
      "b 130f\n"
      "33:"  // Height 2
      "mov x10, %x[col_bias]\n"
      "movi v11.4s, #0x0\n"
      "movi v12.4s, #0x0\n"
      "bic %x[flags], %x[flags], #0x80000000\n"
      "movi v15.16b, #0x1\n"
      "ldr x9, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x28, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x27, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "34:"  // Height 2: Column loop
      "movi v16.4s, #0x0\n"
      "movi v17.4s, #0x0\n"
      "movi v18.4s, #0x0\n"
      "movi v19.4s, #0x0\n"
      "movi v20.4s, #0x0\n"
      "movi v21.4s, #0x0\n"
      "movi v22.4s, #0x0\n"
      "movi v23.4s, #0x0\n"
      "35:"  // Height 2: setup done
      "mov x26, #0x0\n"
      "36:"  // Height 2: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w25, [x20, x26, LSL #0x2]\n"
      "tbz %x[flags], #3, 37f\n"
      "ldr x20, [%x[input_ptr], x26, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x24, [x20, #0x0]\n"
      "ldr x23, [x20, #0x8]\n"
      "cbnz x26, 38f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x24, x24, x20\n"
      "add x23, x23, x20\n"
      "b 38f\n"
      "37:"  // Height 2: setup direct input
      "mov x24, %x[input_ptr]\n"
      "add x23, x24, x21\n"
      "38:"  // Height 2: input setup done
      "cmp x25, #0x10\n"
      "blt 43f\n"
      "ldr q1, [x24, #0x0]\n"
      "ldr q2, [x23, #0x0]\n"
      "cmp x25, #0x20\n"
      "ldr q5, [x28, #0x0]\n"
      "ldr q6, [x28, #0x10]\n"
      "ldr q7, [x28, #0x20]\n"
      "ldr q8, [x28, #0x30]\n"
      "ldr q9, [x28, #0x40]\n"
      "ldr q10, [x28, #0x50]\n"
      "ldr q4, [x28, #0x60]\n"
      "blt 41f\n"
      "39:"  // Height 2: Multiply loop: Main loop head
      "trn1 v0.2d, v1.2d, v2.2d\n"
      "trn2 v1.2d, v1.2d, v2.2d\n"
      "add x24, x24, #0x10\n"
      "add x23, x23, #0x10\n"
      ".inst 0x4e85a410  // smmla v16.4s, v0.16b, v5.16b\n"
      "ldr q25, [x28, #0x70]\n"
      ".inst 0x4e86a414  // smmla v20.4s, v0.16b, v6.16b\n"
      "ldr q24, [x28, #0x80]\n"
      ".inst 0x4e87a411  // smmla v17.4s, v0.16b, v7.16b\n"
      "ldr q30, [x28, #0x90]\n"
      ".inst 0x4e88a415  // smmla v21.4s, v0.16b, v8.16b\n"
      "ldr q29, [x28, #0xa0]\n"
      ".inst 0x4e89a412  // smmla v18.4s, v0.16b, v9.16b\n"
      "ldr q28, [x28, #0xb0]\n"
      ".inst 0x4e8aa416  // smmla v22.4s, v0.16b, v10.16b\n"
      "ldr q27, [x28, #0xc0]\n"
      ".inst 0x4e84a413  // smmla v19.4s, v0.16b, v4.16b\n"
      "ldr q26, [x28, #0xd0]\n"
      ".inst 0x4e99a417  // smmla v23.4s, v0.16b, v25.16b\n"
      "ldr q25, [x28, #0xe0]\n"
      ".inst 0x4e98a430  // smmla v16.4s, v1.16b, v24.16b\n"
      "ldr q24, [x28, #0xf0]\n"
      ".inst 0x4e9ea434  // smmla v20.4s, v1.16b, v30.16b\n"
      "add x28, x28, #0x100\n"
      ".inst 0x4e9da431  // smmla v17.4s, v1.16b, v29.16b\n"
      ".inst 0x4e9ca435  // smmla v21.4s, v1.16b, v28.16b\n"
      ".inst 0x4e9ba432  // smmla v18.4s, v1.16b, v27.16b\n"
      ".inst 0x4e9aa436  // smmla v22.4s, v1.16b, v26.16b\n"
      ".inst 0x4e99a433  // smmla v19.4s, v1.16b, v25.16b\n"
      ".inst 0x4e98a437  // smmla v23.4s, v1.16b, v24.16b\n"
      "tbnz %x[flags], #31, 40f\n"
      ".inst 0x4e8f940b  // sdot v11.4s, v0.16b, v15.16b\n"
      ".inst 0x4e8f942b  // sdot v11.4s, v1.16b, v15.16b\n"
      "40:"  // Height 2: Multiply loop: unique 5: skip row sum
      "ldr q1, [x24, #0x0]\n"
      "ldr q2, [x23, #0x0]\n"
      "sub x25, x25, #0x10\n"
      "ldr q5, [x28, #0x0]\n"
      "ldr q6, [x28, #0x10]\n"
      "cmp x25, #0x20\n"
      "ldr q7, [x28, #0x20]\n"
      "ldr q8, [x28, #0x30]\n"
      "ldr q9, [x28, #0x40]\n"
      "ldr q10, [x28, #0x50]\n"
      "ldr q4, [x28, #0x60]\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      "prfm pldl1keep, [x23, #0x80]\n"
      "bge 39b\n"
      "41:"  // Height 2: Multiply loop: Single iteration only
      "trn1 v0.2d, v1.2d, v2.2d\n"
      "trn2 v1.2d, v1.2d, v2.2d\n"
      "sub x25, x25, #0x10\n"
      "add x24, x24, #0x10\n"
      "add x23, x23, #0x10\n"
      ".inst 0x4e85a410  // smmla v16.4s, v0.16b, v5.16b\n"
      "ldr q25, [x28, #0x70]\n"
      ".inst 0x4e86a414  // smmla v20.4s, v0.16b, v6.16b\n"
      "ldr q24, [x28, #0x80]\n"
      ".inst 0x4e87a411  // smmla v17.4s, v0.16b, v7.16b\n"
      "ldr q30, [x28, #0x90]\n"
      ".inst 0x4e88a415  // smmla v21.4s, v0.16b, v8.16b\n"
      "ldr q29, [x28, #0xa0]\n"
      ".inst 0x4e89a412  // smmla v18.4s, v0.16b, v9.16b\n"
      "ldr q28, [x28, #0xb0]\n"
      ".inst 0x4e8aa416  // smmla v22.4s, v0.16b, v10.16b\n"
      "ldr q27, [x28, #0xc0]\n"
      ".inst 0x4e84a413  // smmla v19.4s, v0.16b, v4.16b\n"
      "ldr q26, [x28, #0xd0]\n"
      ".inst 0x4e99a417  // smmla v23.4s, v0.16b, v25.16b\n"
      "ldr q25, [x28, #0xe0]\n"
      ".inst 0x4e98a430  // smmla v16.4s, v1.16b, v24.16b\n"
      "ldr q24, [x28, #0xf0]\n"
      ".inst 0x4e9ea434  // smmla v20.4s, v1.16b, v30.16b\n"
      "add x28, x28, #0x100\n"
      ".inst 0x4e9da431  // smmla v17.4s, v1.16b, v29.16b\n"
      ".inst 0x4e9ca435  // smmla v21.4s, v1.16b, v28.16b\n"
      ".inst 0x4e9ba432  // smmla v18.4s, v1.16b, v27.16b\n"
      ".inst 0x4e9aa436  // smmla v22.4s, v1.16b, v26.16b\n"
      ".inst 0x4e99a433  // smmla v19.4s, v1.16b, v25.16b\n"
      ".inst 0x4e98a437  // smmla v23.4s, v1.16b, v24.16b\n"
      "tbnz %x[flags], #31, 42f\n"
      ".inst 0x4e8f940b  // sdot v11.4s, v0.16b, v15.16b\n"
      ".inst 0x4e8f942b  // sdot v11.4s, v1.16b, v15.16b\n"
      "42:"  // Height 2: Multiply loop: unique 6: skip row sum
      "prfm pldl1keep, [x24, #0x80]\n"
      "prfm pldl1keep, [x23, #0x80]\n"
      "43:"  // Height 2: Multiply loop: Main loop skip
      "cbz x25, 52f\n"
      "cmp x25, #0x8\n"
      "blt 46f\n"
      "44:"  // Height 2: Multiply loop: Odd block loop
      "ldr d25, [x24], #0x8\n"
      "ldr d24, [x23], #0x8\n"
      "trn1 v0.2d, v25.2d, v24.2d\n"
      "tbnz %x[flags], #31, 45f\n"
      ".inst 0x4e8f940b  // sdot v11.4s, v0.16b, v15.16b\n"
      "45:"  // Height 2: Multiply loop: unique 7: skip row sum
      "ldr q24, [x28, #0x0]\n"
      "ldr q30, [x28, #0x10]\n"
      "sub x25, x25, #0x8\n"
      "ldr q29, [x28, #0x20]\n"
      "ldr q28, [x28, #0x30]\n"
      "cmp x25, #0x8\n"
      "ldr q27, [x28, #0x40]\n"
      "ldr q26, [x28, #0x50]\n"
      "ldr q25, [x28, #0x60]\n"
      ".inst 0x4e98a410  // smmla v16.4s, v0.16b, v24.16b\n"
      "ldr q24, [x28, #0x70]\n"
      ".inst 0x4e9ea414  // smmla v20.4s, v0.16b, v30.16b\n"
      ".inst 0x4e9da411  // smmla v17.4s, v0.16b, v29.16b\n"
      ".inst 0x4e9ca415  // smmla v21.4s, v0.16b, v28.16b\n"
      "add x28, x28, #0x80\n"
      ".inst 0x4e9ba412  // smmla v18.4s, v0.16b, v27.16b\n"
      ".inst 0x4e9aa416  // smmla v22.4s, v0.16b, v26.16b\n"
      ".inst 0x4e99a413  // smmla v19.4s, v0.16b, v25.16b\n"
      ".inst 0x4e98a417  // smmla v23.4s, v0.16b, v24.16b\n"
      "bge 44b\n"
      "46:"  // Height 2: Multiply loop: Skip odd blocks
      "cbz x25, 52f\n"
      "tbz x25, #2, 48f\n"
      "ldr s1, [x24], #0x4\n"
      "ldr s2, [x23], #0x4\n"
      "tbz x25, #1, 47f\n"
      "ld1 { v1.h }[2], [x24], #0x2\n"
      "ld1 { v2.h }[2], [x23], #0x2\n"
      "tbz x25, #0, 50f\n"
      "ld1 { v1.b }[6], [x24]\n"
      "ld1 { v2.b }[6], [x23]\n"
      "b 50f\n"
      "47:"  // Height 2: Multiply loop: Ragged operand read: partial_1_4
      "tbz x25, #0, 50f\n"
      "ld1 { v1.b }[4], [x24]\n"
      "ld1 { v2.b }[4], [x23]\n"
      "b 50f\n"
      "48:"  // Height 2: Multiply loop: Ragged operand read: partial_2_0
      "tbz x25, #1, 49f\n"
      "ldr h1, [x24], #0x2\n"
      "ldr h2, [x23], #0x2\n"
      "tbz x25, #0, 50f\n"
      "ld1 { v1.b }[2], [x24]\n"
      "ld1 { v2.b }[2], [x23]\n"
      "b 50f\n"
      "49:"  // Height 2: Multiply loop: Ragged operand read: partial_1_0
      "ldr b1, [x24, #0x0]\n"
      "ldr b2, [x23, #0x0]\n"
      "50:"  // Height 2: Multiply loop: Ragged operand read: Done
      "trn1 v0.2d, v1.2d, v2.2d\n"
      "tbnz %x[flags], #31, 51f\n"
      ".inst 0x4e8f940b  // sdot v11.4s, v0.16b, v15.16b\n"
      "51:"  // Height 2: Multiply loop: unique 8: skip row sum
      "ldr q24, [x28, #0x0]\n"
      "ldr q30, [x28, #0x10]\n"
      "ldr q29, [x28, #0x20]\n"
      "ldr q28, [x28, #0x30]\n"
      "ldr q27, [x28, #0x40]\n"
      "ldr q26, [x28, #0x50]\n"
      "ldr q25, [x28, #0x60]\n"
      ".inst 0x4e98a410  // smmla v16.4s, v0.16b, v24.16b\n"
      "ldr q24, [x28, #0x70]\n"
      ".inst 0x4e9ea414  // smmla v20.4s, v0.16b, v30.16b\n"
      ".inst 0x4e9da411  // smmla v17.4s, v0.16b, v29.16b\n"
      ".inst 0x4e9ca415  // smmla v21.4s, v0.16b, v28.16b\n"
      "add x28, x28, #0x80\n"
      ".inst 0x4e9ba412  // smmla v18.4s, v0.16b, v27.16b\n"
      ".inst 0x4e9aa416  // smmla v22.4s, v0.16b, v26.16b\n"
      ".inst 0x4e99a413  // smmla v19.4s, v0.16b, v25.16b\n"
      ".inst 0x4e98a417  // smmla v23.4s, v0.16b, v24.16b\n"
      "52:"  // Height 2: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x26, x26, #0x1\n"
      "cmp x26, x20\n"
      "bne 36b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "uzp1 v24.2d, v16.2d, v20.2d\n"
      "uzp2 v16.2d, v16.2d, v20.2d\n"
      "prfm pstl1keep, [x27, #0x0]\n"
      "uzp1 v20.2d, v17.2d, v21.2d\n"
      "uzp2 v17.2d, v17.2d, v21.2d\n"
      "uzp1 v21.2d, v18.2d, v22.2d\n"
      "uzp2 v18.2d, v18.2d, v22.2d\n"
      "add x24, x27, x20\n"
      "uzp1 v22.2d, v19.2d, v23.2d\n"
      "uzp2 v19.2d, v19.2d, v23.2d\n"
      "prfm pstl1keep, [x24, #0x0]\n"
      "mov v23.16b, v24.16b\n"
      "tbnz %x[flags], #31, 53f\n"
      "add x20, %x[qp], %[b_offset]\n"
      "addp v11.4s, v11.4s, v11.4s\n"
      "ld1r { v24.4s }, [x20]\n"
      "neg v24.4s, v24.4s\n"
      "dup v12.4s, v11.s[3]\n"
      "dup v11.4s, v11.s[0]\n"
      "mul v11.4s, v11.4s, v24.4s\n"
      "mul v12.4s, v12.4s, v24.4s\n"
      "53:"  // Height 2: skip row sum fixup
      "ldr q28, [x10, #0x0]\n"
      "ldr q27, [x10, #0x10]\n"
      "add v23.4s, v23.4s, v11.4s\n"
      "add v20.4s, v20.4s, v11.4s\n"
      "ldr q26, [x10, #0x20]\n"
      "ldr q25, [x10, #0x30]\n"
      "add v21.4s, v21.4s, v11.4s\n"
      "add v22.4s, v22.4s, v11.4s\n"
      "add v16.4s, v16.4s, v12.4s\n"
      "add v17.4s, v17.4s, v12.4s\n"
      "add x20, %x[qp], %[per_layer_mul]\n"
      "orr %x[flags], %x[flags], #0x80000000\n"
      "ld1r { v24.4s }, [x20]\n"
      "add v18.4s, v18.4s, v12.4s\n"
      "add v19.4s, v19.4s, v12.4s\n"
      "add x20, %x[qp], %[per_layer_right_shift]\n"
      "add v23.4s, v23.4s, v28.4s\n"
      "add v20.4s, v20.4s, v27.4s\n"
      "add x10, x10, #0x40\n"
      "add v21.4s, v21.4s, v26.4s\n"
      "add v22.4s, v22.4s, v25.4s\n"
      "add v16.4s, v16.4s, v28.4s\n"
      "ld1r { v0.4s }, [x20]\n"
      "add v17.4s, v17.4s, v27.4s\n"
      "add v18.4s, v18.4s, v26.4s\n"
      "add v19.4s, v19.4s, v25.4s\n"
      "sqrdmulh v23.4s, v23.4s, v24.4s\n"
      "sqrdmulh v20.4s, v20.4s, v24.4s\n"
      "sqrdmulh v21.4s, v21.4s, v24.4s\n"
      "sqrdmulh v22.4s, v22.4s, v24.4s\n"
      "sqrdmulh v16.4s, v16.4s, v24.4s\n"
      "sqrdmulh v17.4s, v17.4s, v24.4s\n"
      "sqrdmulh v18.4s, v18.4s, v24.4s\n"
      "sqrdmulh v19.4s, v19.4s, v24.4s\n"
      "tbz %x[flags], #5, 54f\n"
      "and v24.16b, v23.16b, v0.16b\n"
      "and v30.16b, v20.16b, v0.16b\n"
      "and v29.16b, v21.16b, v0.16b\n"
      "and v28.16b, v22.16b, v0.16b\n"
      "and v27.16b, v16.16b, v0.16b\n"
      "and v26.16b, v17.16b, v0.16b\n"
      "sshr v24.4s, v24.4s, #0x1f\n"
      "and v25.16b, v18.16b, v0.16b\n"
      "sshr v30.4s, v30.4s, #0x1f\n"
      "sshr v29.4s, v29.4s, #0x1f\n"
      "sshr v28.4s, v28.4s, #0x1f\n"
      "sshr v27.4s, v27.4s, #0x1f\n"
      "sqadd v23.4s, v23.4s, v24.4s\n"
      "and v24.16b, v19.16b, v0.16b\n"
      "sshr v26.4s, v26.4s, #0x1f\n"
      "sshr v25.4s, v25.4s, #0x1f\n"
      "sqadd v20.4s, v20.4s, v30.4s\n"
      "sqadd v21.4s, v21.4s, v29.4s\n"
      "sshr v24.4s, v24.4s, #0x1f\n"
      "sqadd v22.4s, v22.4s, v28.4s\n"
      "sqadd v16.4s, v16.4s, v27.4s\n"
      "sqadd v17.4s, v17.4s, v26.4s\n"
      "sqadd v18.4s, v18.4s, v25.4s\n"
      "sqadd v19.4s, v19.4s, v24.4s\n"
      "54:"  // Height 2: no shift correction
      "add x21, %x[qp], %[c_offset]\n"
      "srshl v23.4s, v23.4s, v0.4s\n"
      "srshl v20.4s, v20.4s, v0.4s\n"
      "add x20, %x[qp], %[maxval]\n"
      "ld1r { v26.4s }, [x21]\n"
      "ld1r { v25.4s }, [x20]\n"
      "srshl v21.4s, v21.4s, v0.4s\n"
      "srshl v22.4s, v22.4s, v0.4s\n"
      "srshl v16.4s, v16.4s, v0.4s\n"
      "srshl v17.4s, v17.4s, v0.4s\n"
      "add x20, %x[qp], %[minval]\n"
      "cmp x9, #0x10\n"
      "ld1r { v24.4s }, [x20]\n"
      "srshl v18.4s, v18.4s, v0.4s\n"
      "srshl v19.4s, v19.4s, v0.4s\n"
      "add v23.4s, v23.4s, v26.4s\n"
      "add v20.4s, v20.4s, v26.4s\n"
      "add v21.4s, v21.4s, v26.4s\n"
      "add v22.4s, v22.4s, v26.4s\n"
      "add v16.4s, v16.4s, v26.4s\n"
      "add v17.4s, v17.4s, v26.4s\n"
      "add v18.4s, v18.4s, v26.4s\n"
      "add v19.4s, v19.4s, v26.4s\n"
      "smin v23.4s, v23.4s, v25.4s\n"
      "smin v20.4s, v20.4s, v25.4s\n"
      "smin v21.4s, v21.4s, v25.4s\n"
      "smin v22.4s, v22.4s, v25.4s\n"
      "smin v16.4s, v16.4s, v25.4s\n"
      "smin v17.4s, v17.4s, v25.4s\n"
      "smin v18.4s, v18.4s, v25.4s\n"
      "smin v19.4s, v19.4s, v25.4s\n"
      "smax v23.4s, v23.4s, v24.4s\n"
      "smax v20.4s, v20.4s, v24.4s\n"
      "smax v21.4s, v21.4s, v24.4s\n"
      "smax v22.4s, v22.4s, v24.4s\n"
      "smax v16.4s, v16.4s, v24.4s\n"
      "smax v17.4s, v17.4s, v24.4s\n"
      "smax v18.4s, v18.4s, v24.4s\n"
      "smax v19.4s, v19.4s, v24.4s\n"
      "uzp1 v23.8h, v23.8h, v20.8h\n"
      "uzp1 v20.8h, v21.8h, v22.8h\n"
      "uzp1 v16.8h, v16.8h, v17.8h\n"
      "uzp1 v17.8h, v18.8h, v19.8h\n"
      "uzp1 v23.16b, v23.16b, v20.16b\n"
      "uzp1 v16.16b, v16.16b, v17.16b\n"
      "bge 63f\n"
      "tbz x9, #3, 58f\n"
      "str d23, [x27], #0x8\n"
      "str d16, [x24], #0x8\n"
      "tbz x9, #2, 56f\n"
      "st1 { v23.s }[2], [x27], #0x4\n"
      "st1 { v16.s }[2], [x24], #0x4\n"
      "tbz x9, #1, 55f\n"
      "st1 { v23.h }[6], [x27], #0x2\n"
      "st1 { v16.h }[6], [x24], #0x2\n"
      "tbz x9, #0, 62f\n"
      "st1 { v23.b }[14], [x27]\n"
      "st1 { v16.b }[14], [x24]\n"
      "b 62f\n"
      "55:"  // Height 2: Partial direct writeback: partial_1_12
      "tbz x9, #0, 62f\n"
      "st1 { v23.b }[12], [x27]\n"
      "st1 { v16.b }[12], [x24]\n"
      "b 62f\n"
      "56:"  // Height 2: Partial direct writeback: partial_2_8
      "tbz x9, #1, 57f\n"
      "st1 { v23.h }[4], [x27], #0x2\n"
      "st1 { v16.h }[4], [x24], #0x2\n"
      "tbz x9, #0, 62f\n"
      "st1 { v23.b }[10], [x27]\n"
      "st1 { v16.b }[10], [x24]\n"
      "b 62f\n"
      "57:"  // Height 2: Partial direct writeback: partial_1_8
      "tbz x9, #0, 62f\n"
      "st1 { v23.b }[8], [x27]\n"
      "st1 { v16.b }[8], [x24]\n"
      "b 62f\n"
      "58:"  // Height 2: Partial direct writeback: partial_4_0
      "tbz x9, #2, 60f\n"
      "str s23, [x27], #0x4\n"
      "str s16, [x24], #0x4\n"
      "tbz x9, #1, 59f\n"
      "st1 { v23.h }[2], [x27], #0x2\n"
      "st1 { v16.h }[2], [x24], #0x2\n"
      "tbz x9, #0, 62f\n"
      "st1 { v23.b }[6], [x27]\n"
      "st1 { v16.b }[6], [x24]\n"
      "b 62f\n"
      "59:"  // Height 2: Partial direct writeback: partial_1_4
      "tbz x9, #0, 62f\n"
      "st1 { v23.b }[4], [x27]\n"
      "st1 { v16.b }[4], [x24]\n"
      "b 62f\n"
      "60:"  // Height 2: Partial direct writeback: partial_2_0
      "tbz x9, #1, 61f\n"
      "str h23, [x27], #0x2\n"
      "str h16, [x24], #0x2\n"
      "tbz x9, #0, 62f\n"
      "st1 { v23.b }[2], [x27]\n"
      "st1 { v16.b }[2], [x24]\n"
      "b 62f\n"
      "61:"  // Height 2: Partial direct writeback: partial_1_0
      "str b23, [x27, #0x0]\n"
      "str b16, [x24, #0x0]\n"
      "62:"  // Height 2: Partial direct writeback: Done
      "b 64f\n"
      "63:"  // Height 2: Full writeback
      "str q23, [x27, #0x0]\n"
      "add x27, x27, #0x10\n"
      "str q16, [x24, #0x0]\n"
      "64:"  // Height 2: Writeback done
      "subs x9, x9, #0x10\n"
      "bgt 34b\n"
      "b 130f\n"
      "65:"  // Height 3
      "mov x10, %x[col_bias]\n"
      "movi v11.4s, #0x0\n"
      "movi v12.4s, #0x0\n"
      "bic %x[flags], %x[flags], #0x80000000\n"
      "movi v13.4s, #0x0\n"
      "movi v15.16b, #0x1\n"
      "ldr x9, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x28, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x27, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "66:"  // Height 3: Column loop
      "movi v16.4s, #0x0\n"
      "movi v17.4s, #0x0\n"
      "movi v18.4s, #0x0\n"
      "movi v19.4s, #0x0\n"
      "movi v20.4s, #0x0\n"
      "movi v21.4s, #0x0\n"
      "movi v22.4s, #0x0\n"
      "movi v23.4s, #0x0\n"
      "movi v24.4s, #0x0\n"
      "movi v25.4s, #0x0\n"
      "movi v26.4s, #0x0\n"
      "movi v27.4s, #0x0\n"
      "movi v28.4s, #0x0\n"
      "movi v29.4s, #0x0\n"
      "movi v30.4s, #0x0\n"
      "movi v31.4s, #0x0\n"
      "67:"  // Height 3: setup done
      "mov x26, #0x0\n"
      "68:"  // Height 3: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w25, [x20, x26, LSL #0x2]\n"
      "tbz %x[flags], #3, 69f\n"
      "ldr x20, [%x[input_ptr], x26, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x24, [x20, #0x0]\n"
      "ldr x23, [x20, #0x8]\n"
      "ldr x22, [x20, #0x10]\n"
      "cbnz x26, 70f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x24, x24, x20\n"
      "add x23, x23, x20\n"
      "add x22, x22, x20\n"
      "b 70f\n"
      "69:"  // Height 3: setup direct input
      "mov x24, %x[input_ptr]\n"
      "add x23, x24, x21\n"
      "add x22, x23, x21\n"
      "70:"  // Height 3: input setup done
      "cmp x25, #0x10\n"
      "blt 75f\n"
      "ldr q1, [x24, #0x0]\n"
      "ldr q2, [x23, #0x0]\n"
      "cmp x25, #0x20\n"
      "ldr q3, [x22, #0x0]\n"
      "ldr q5, [x28, #0x0]\n"
      "ldr q6, [x28, #0x10]\n"
      "ldr q7, [x28, #0x20]\n"
      "ldr q8, [x28, #0x30]\n"
      "ldr q9, [x28, #0x40]\n"
      "ldr q10, [x28, #0x50]\n"
      "blt 73f\n"
      "71:"  // Height 3: Multiply loop: Main loop head
      "trn1 v0.2d, v1.2d, v2.2d\n"
      "trn2 v1.2d, v1.2d, v2.2d\n"
      "add x24, x24, #0x10\n"
      "add x23, x23, #0x10\n"
      "add x22, x22, #0x10\n"
      "trn1 v2.2d, v3.2d, v4.2d\n"
      "trn2 v3.2d, v3.2d, v4.2d\n"
      "ldr q14, [x28, #0x60]\n"
      ".inst 0x4e85a410  // smmla v16.4s, v0.16b, v5.16b\n"
      ".inst 0x4e86a414  // smmla v20.4s, v0.16b, v6.16b\n"
      ".inst 0x4e87a411  // smmla v17.4s, v0.16b, v7.16b\n"
      ".inst 0x4e88a415  // smmla v21.4s, v0.16b, v8.16b\n"
      ".inst 0x4e85a458  // smmla v24.4s, v2.16b, v5.16b\n"
      "ldr q5, [x28, #0x70]\n"
      ".inst 0x4e86a45c  // smmla v28.4s, v2.16b, v6.16b\n"
      "ldr q4, [x28, #0x80]\n"
      ".inst 0x4e89a412  // smmla v18.4s, v0.16b, v9.16b\n"
      ".inst 0x4e87a459  // smmla v25.4s, v2.16b, v7.16b\n"
      "ldr q7, [x28, #0x90]\n"
      ".inst 0x4e88a45d  // smmla v29.4s, v2.16b, v8.16b\n"
      "ldr q8, [x28, #0xa0]\n"
      ".inst 0x4e89a45a  // smmla v26.4s, v2.16b, v9.16b\n"
      "ldr q9, [x28, #0xb0]\n"
      ".inst 0x4e8aa416  // smmla v22.4s, v0.16b, v10.16b\n"
      ".inst 0x4e8aa45e  // smmla v30.4s, v2.16b, v10.16b\n"
      "ldr q10, [x28, #0xc0]\n"
      ".inst 0x4e8ea413  // smmla v19.4s, v0.16b, v14.16b\n"
      ".inst 0x4e8ea45b  // smmla v27.4s, v2.16b, v14.16b\n"
      "ldr q6, [x28, #0xd0]\n"
      ".inst 0x4e85a417  // smmla v23.4s, v0.16b, v5.16b\n"
      ".inst 0x4e85a45f  // smmla v31.4s, v2.16b, v5.16b\n"
      "ldr q5, [x28, #0xe0]\n"
      ".inst 0x4e84a430  // smmla v16.4s, v1.16b, v4.16b\n"
      ".inst 0x4e84a478  // smmla v24.4s, v3.16b, v4.16b\n"
      "ldr q4, [x28, #0xf0]\n"
      "add x28, x28, #0x100\n"
      ".inst 0x4e87a434  // smmla v20.4s, v1.16b, v7.16b\n"
      ".inst 0x4e87a47c  // smmla v28.4s, v3.16b, v7.16b\n"
      ".inst 0x4e88a431  // smmla v17.4s, v1.16b, v8.16b\n"
      ".inst 0x4e88a479  // smmla v25.4s, v3.16b, v8.16b\n"
      ".inst 0x4e89a435  // smmla v21.4s, v1.16b, v9.16b\n"
      ".inst 0x4e89a47d  // smmla v29.4s, v3.16b, v9.16b\n"
      ".inst 0x4e8aa432  // smmla v18.4s, v1.16b, v10.16b\n"
      ".inst 0x4e8aa47a  // smmla v26.4s, v3.16b, v10.16b\n"
      ".inst 0x4e86a436  // smmla v22.4s, v1.16b, v6.16b\n"
      ".inst 0x4e86a47e  // smmla v30.4s, v3.16b, v6.16b\n"
      ".inst 0x4e85a433  // smmla v19.4s, v1.16b, v5.16b\n"
      ".inst 0x4e85a47b  // smmla v27.4s, v3.16b, v5.16b\n"
      ".inst 0x4e84a437  // smmla v23.4s, v1.16b, v4.16b\n"
      ".inst 0x4e84a47f  // smmla v31.4s, v3.16b, v4.16b\n"
      "tbnz %x[flags], #31, 72f\n"
      ".inst 0x4e8f940b  // sdot v11.4s, v0.16b, v15.16b\n"
      ".inst 0x4e8f944d  // sdot v13.4s, v2.16b, v15.16b\n"
      ".inst 0x4e8f942b  // sdot v11.4s, v1.16b, v15.16b\n"
      ".inst 0x4e8f946d  // sdot v13.4s, v3.16b, v15.16b\n"
      "72:"  // Height 3: Multiply loop: unique 9: skip row sum
      "ldr q1, [x24, #0x0]\n"
      "ldr q2, [x23, #0x0]\n"
      "sub x25, x25, #0x10\n"
      "ldr q3, [x22, #0x0]\n"
      "ldr q5, [x28, #0x0]\n"
      "cmp x25, #0x20\n"
      "ldr q6, [x28, #0x10]\n"
      "ldr q7, [x28, #0x20]\n"
      "ldr q8, [x28, #0x30]\n"
      "ldr q9, [x28, #0x40]\n"
      "ldr q10, [x28, #0x50]\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      "prfm pldl1keep, [x23, #0x80]\n"
      "prfm pldl1keep, [x22, #0x80]\n"
      "bge 71b\n"
      "73:"  // Height 3: Multiply loop: Single iteration only
      "trn1 v0.2d, v1.2d, v2.2d\n"
      "trn2 v1.2d, v1.2d, v2.2d\n"
      "sub x25, x25, #0x10\n"
      "add x24, x24, #0x10\n"
      "add x23, x23, #0x10\n"
      "add x22, x22, #0x10\n"
      "trn1 v2.2d, v3.2d, v4.2d\n"
      "trn2 v3.2d, v3.2d, v4.2d\n"
      "ldr q14, [x28, #0x60]\n"
      ".inst 0x4e85a410  // smmla v16.4s, v0.16b, v5.16b\n"
      ".inst 0x4e86a414  // smmla v20.4s, v0.16b, v6.16b\n"
      ".inst 0x4e87a411  // smmla v17.4s, v0.16b, v7.16b\n"
      ".inst 0x4e88a415  // smmla v21.4s, v0.16b, v8.16b\n"
      ".inst 0x4e85a458  // smmla v24.4s, v2.16b, v5.16b\n"
      "ldr q5, [x28, #0x70]\n"
      ".inst 0x4e86a45c  // smmla v28.4s, v2.16b, v6.16b\n"
      "ldr q4, [x28, #0x80]\n"
      ".inst 0x4e89a412  // smmla v18.4s, v0.16b, v9.16b\n"
      ".inst 0x4e87a459  // smmla v25.4s, v2.16b, v7.16b\n"
      "ldr q7, [x28, #0x90]\n"
      ".inst 0x4e88a45d  // smmla v29.4s, v2.16b, v8.16b\n"
      "ldr q8, [x28, #0xa0]\n"
      ".inst 0x4e89a45a  // smmla v26.4s, v2.16b, v9.16b\n"
      "ldr q9, [x28, #0xb0]\n"
      ".inst 0x4e8aa416  // smmla v22.4s, v0.16b, v10.16b\n"
      ".inst 0x4e8aa45e  // smmla v30.4s, v2.16b, v10.16b\n"
      "ldr q10, [x28, #0xc0]\n"
      ".inst 0x4e8ea413  // smmla v19.4s, v0.16b, v14.16b\n"
      ".inst 0x4e8ea45b  // smmla v27.4s, v2.16b, v14.16b\n"
      "ldr q6, [x28, #0xd0]\n"
      ".inst 0x4e85a417  // smmla v23.4s, v0.16b, v5.16b\n"
      ".inst 0x4e85a45f  // smmla v31.4s, v2.16b, v5.16b\n"
      "ldr q5, [x28, #0xe0]\n"
      ".inst 0x4e84a430  // smmla v16.4s, v1.16b, v4.16b\n"
      ".inst 0x4e84a478  // smmla v24.4s, v3.16b, v4.16b\n"
      "ldr q4, [x28, #0xf0]\n"
      "add x28, x28, #0x100\n"
      ".inst 0x4e87a434  // smmla v20.4s, v1.16b, v7.16b\n"
      ".inst 0x4e87a47c  // smmla v28.4s, v3.16b, v7.16b\n"
      ".inst 0x4e88a431  // smmla v17.4s, v1.16b, v8.16b\n"
      ".inst 0x4e88a479  // smmla v25.4s, v3.16b, v8.16b\n"
      ".inst 0x4e89a435  // smmla v21.4s, v1.16b, v9.16b\n"
      ".inst 0x4e89a47d  // smmla v29.4s, v3.16b, v9.16b\n"
      ".inst 0x4e8aa432  // smmla v18.4s, v1.16b, v10.16b\n"
      ".inst 0x4e8aa47a  // smmla v26.4s, v3.16b, v10.16b\n"
      ".inst 0x4e86a436  // smmla v22.4s, v1.16b, v6.16b\n"
      ".inst 0x4e86a47e  // smmla v30.4s, v3.16b, v6.16b\n"
      ".inst 0x4e85a433  // smmla v19.4s, v1.16b, v5.16b\n"
      ".inst 0x4e85a47b  // smmla v27.4s, v3.16b, v5.16b\n"
      ".inst 0x4e84a437  // smmla v23.4s, v1.16b, v4.16b\n"
      ".inst 0x4e84a47f  // smmla v31.4s, v3.16b, v4.16b\n"
      "tbnz %x[flags], #31, 74f\n"
      ".inst 0x4e8f940b  // sdot v11.4s, v0.16b, v15.16b\n"
      ".inst 0x4e8f944d  // sdot v13.4s, v2.16b, v15.16b\n"
      ".inst 0x4e8f942b  // sdot v11.4s, v1.16b, v15.16b\n"
      ".inst 0x4e8f946d  // sdot v13.4s, v3.16b, v15.16b\n"
      "74:"  // Height 3: Multiply loop: unique 10: skip row sum
      "prfm pldl1keep, [x24, #0x80]\n"
      "prfm pldl1keep, [x23, #0x80]\n"
      "prfm pldl1keep, [x22, #0x80]\n"
      "75:"  // Height 3: Multiply loop: Main loop skip
      "cbz x25, 84f\n"
      "cmp x25, #0x8\n"
      "blt 78f\n"
      "76:"  // Height 3: Multiply loop: Odd block loop
      "ldr d3, [x24], #0x8\n"
      "ldr d0, [x23], #0x8\n"
      "ldr d1, [x22], #0x8\n"
      "trn1 v0.2d, v3.2d, v0.2d\n"
      "trn1 v2.2d, v1.2d, v2.2d\n"
      "tbnz %x[flags], #31, 77f\n"
      ".inst 0x4e8f940b  // sdot v11.4s, v0.16b, v15.16b\n"
      ".inst 0x4e8f944d  // sdot v13.4s, v2.16b, v15.16b\n"
      "77:"  // Height 3: Multiply loop: unique 11: skip row sum
      "ldr q1, [x28, #0x0]\n"
      "ldr q8, [x28, #0x10]\n"
      "sub x25, x25, #0x8\n"
      "ldr q7, [x28, #0x20]\n"
      "ldr q6, [x28, #0x30]\n"
      "cmp x25, #0x8\n"
      "ldr q5, [x28, #0x40]\n"
      "ldr q4, [x28, #0x50]\n"
      "ldr q3, [x28, #0x60]\n"
      ".inst 0x4e81a410  // smmla v16.4s, v0.16b, v1.16b\n"
      ".inst 0x4e81a458  // smmla v24.4s, v2.16b, v1.16b\n"
      "ldr q1, [x28, #0x70]\n"
      ".inst 0x4e88a414  // smmla v20.4s, v0.16b, v8.16b\n"
      ".inst 0x4e88a45c  // smmla v28.4s, v2.16b, v8.16b\n"
      "add x28, x28, #0x80\n"
      ".inst 0x4e87a411  // smmla v17.4s, v0.16b, v7.16b\n"
      ".inst 0x4e87a459  // smmla v25.4s, v2.16b, v7.16b\n"
      ".inst 0x4e86a415  // smmla v21.4s, v0.16b, v6.16b\n"
      ".inst 0x4e86a45d  // smmla v29.4s, v2.16b, v6.16b\n"
      ".inst 0x4e85a412  // smmla v18.4s, v0.16b, v5.16b\n"
      ".inst 0x4e85a45a  // smmla v26.4s, v2.16b, v5.16b\n"
      ".inst 0x4e84a416  // smmla v22.4s, v0.16b, v4.16b\n"
      ".inst 0x4e84a45e  // smmla v30.4s, v2.16b, v4.16b\n"
      ".inst 0x4e83a413  // smmla v19.4s, v0.16b, v3.16b\n"
      ".inst 0x4e83a45b  // smmla v27.4s, v2.16b, v3.16b\n"
      ".inst 0x4e81a417  // smmla v23.4s, v0.16b, v1.16b\n"
      ".inst 0x4e81a45f  // smmla v31.4s, v2.16b, v1.16b\n"
      "bge 76b\n"
      "78:"  // Height 3: Multiply loop: Skip odd blocks
      "cbz x25, 84f\n"
      "tbz x25, #2, 80f\n"
      "ldr s1, [x24], #0x4\n"
      "ldr s2, [x23], #0x4\n"
      "ldr s3, [x22], #0x4\n"
      "tbz x25, #1, 79f\n"
      "ld1 { v1.h }[2], [x24], #0x2\n"
      "ld1 { v2.h }[2], [x23], #0x2\n"
      "ld1 { v3.h }[2], [x22], #0x2\n"
      "tbz x25, #0, 82f\n"
      "ld1 { v1.b }[6], [x24]\n"
      "ld1 { v2.b }[6], [x23]\n"
      "ld1 { v3.b }[6], [x22]\n"
      "b 82f\n"
      "79:"  // Height 3: Multiply loop: Ragged operand read: partial_1_4
      "tbz x25, #0, 82f\n"
      "ld1 { v1.b }[4], [x24]\n"
      "ld1 { v2.b }[4], [x23]\n"
      "ld1 { v3.b }[4], [x22]\n"
      "b 82f\n"
      "80:"  // Height 3: Multiply loop: Ragged operand read: partial_2_0
      "tbz x25, #1, 81f\n"
      "ldr h1, [x24], #0x2\n"
      "ldr h2, [x23], #0x2\n"
      "ldr h3, [x22], #0x2\n"
      "tbz x25, #0, 82f\n"
      "ld1 { v1.b }[2], [x24]\n"
      "ld1 { v2.b }[2], [x23]\n"
      "ld1 { v3.b }[2], [x22]\n"
      "b 82f\n"
      "81:"  // Height 3: Multiply loop: Ragged operand read: partial_1_0
      "ldr b1, [x24, #0x0]\n"
      "ldr b2, [x23, #0x0]\n"
      "ldr b3, [x22, #0x0]\n"
      "82:"  // Height 3: Multiply loop: Ragged operand read: Done
      "trn1 v0.2d, v1.2d, v2.2d\n"
      "trn1 v2.2d, v3.2d, v4.2d\n"
      "tbnz %x[flags], #31, 83f\n"
      ".inst 0x4e8f940b  // sdot v11.4s, v0.16b, v15.16b\n"
      ".inst 0x4e8f944d  // sdot v13.4s, v2.16b, v15.16b\n"
      "83:"  // Height 3: Multiply loop: unique 12: skip row sum
      "ldr q1, [x28, #0x0]\n"
      "ldr q8, [x28, #0x10]\n"
      "ldr q7, [x28, #0x20]\n"
      "ldr q6, [x28, #0x30]\n"
      "ldr q5, [x28, #0x40]\n"
      "ldr q4, [x28, #0x50]\n"
      "ldr q3, [x28, #0x60]\n"
      ".inst 0x4e81a410  // smmla v16.4s, v0.16b, v1.16b\n"
      ".inst 0x4e81a458  // smmla v24.4s, v2.16b, v1.16b\n"
      "ldr q1, [x28, #0x70]\n"
      ".inst 0x4e88a414  // smmla v20.4s, v0.16b, v8.16b\n"
      ".inst 0x4e88a45c  // smmla v28.4s, v2.16b, v8.16b\n"
      "add x28, x28, #0x80\n"
      ".inst 0x4e87a411  // smmla v17.4s, v0.16b, v7.16b\n"
      ".inst 0x4e87a459  // smmla v25.4s, v2.16b, v7.16b\n"
      ".inst 0x4e86a415  // smmla v21.4s, v0.16b, v6.16b\n"
      ".inst 0x4e86a45d  // smmla v29.4s, v2.16b, v6.16b\n"
      ".inst 0x4e85a412  // smmla v18.4s, v0.16b, v5.16b\n"
      ".inst 0x4e85a45a  // smmla v26.4s, v2.16b, v5.16b\n"
      ".inst 0x4e84a416  // smmla v22.4s, v0.16b, v4.16b\n"
      ".inst 0x4e84a45e  // smmla v30.4s, v2.16b, v4.16b\n"
      ".inst 0x4e83a413  // smmla v19.4s, v0.16b, v3.16b\n"
      ".inst 0x4e83a45b  // smmla v27.4s, v2.16b, v3.16b\n"
      ".inst 0x4e81a417  // smmla v23.4s, v0.16b, v1.16b\n"
      ".inst 0x4e81a45f  // smmla v31.4s, v2.16b, v1.16b\n"
      "84:"  // Height 3: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x26, x26, #0x1\n"
      "cmp x26, x20\n"
      "bne 68b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "uzp1 v0.2d, v16.2d, v20.2d\n"
      "uzp2 v16.2d, v16.2d, v20.2d\n"
      "prfm pstl1keep, [x27, #0x0]\n"
      "uzp1 v20.2d, v17.2d, v21.2d\n"
      "uzp2 v17.2d, v17.2d, v21.2d\n"
      "uzp1 v21.2d, v18.2d, v22.2d\n"
      "uzp2 v18.2d, v18.2d, v22.2d\n"
      "add x24, x27, x20\n"
      "add x23, x24, x20\n"
      "uzp1 v22.2d, v19.2d, v23.2d\n"
      "uzp2 v19.2d, v19.2d, v23.2d\n"
      "prfm pstl1keep, [x24, #0x0]\n"
      "prfm pstl1keep, [x23, #0x0]\n"
      "uzp1 v24.2d, v24.2d, v28.2d\n"
      "uzp1 v25.2d, v25.2d, v29.2d\n"
      "uzp1 v26.2d, v26.2d, v30.2d\n"
      "uzp1 v27.2d, v27.2d, v31.2d\n"
      "mov v31.16b, v0.16b\n"
      "tbnz %x[flags], #31, 85f\n"
      "add x20, %x[qp], %[b_offset]\n"
      "addp v11.4s, v11.4s, v11.4s\n"
      "addp v13.4s, v13.4s, v13.4s\n"
      "ld1r { v23.4s }, [x20]\n"
      "neg v23.4s, v23.4s\n"
      "dup v12.4s, v11.s[3]\n"
      "dup v11.4s, v11.s[0]\n"
      "dup v13.4s, v13.s[0]\n"
      "mul v11.4s, v11.4s, v23.4s\n"
      "mul v12.4s, v12.4s, v23.4s\n"
      "mul v13.4s, v13.4s, v23.4s\n"
      "85:"  // Height 3: skip row sum fixup
      "ldr q0, [x10, #0x0]\n"
      "ldr q30, [x10, #0x10]\n"
      "add v31.4s, v31.4s, v11.4s\n"
      "add v20.4s, v20.4s, v11.4s\n"
      "ldr q29, [x10, #0x20]\n"
      "ldr q28, [x10, #0x30]\n"
      "add v21.4s, v21.4s, v11.4s\n"
      "add v22.4s, v22.4s, v11.4s\n"
      "add v16.4s, v16.4s, v12.4s\n"
      "add v17.4s, v17.4s, v12.4s\n"
      "add x20, %x[qp], %[per_layer_mul]\n"
      "orr %x[flags], %x[flags], #0x80000000\n"
      "ld1r { v23.4s }, [x20]\n"
      "add v18.4s, v18.4s, v12.4s\n"
      "add v19.4s, v19.4s, v12.4s\n"
      "add x20, %x[qp], %[per_layer_right_shift]\n"
      "add v24.4s, v24.4s, v13.4s\n"
      "add v25.4s, v25.4s, v13.4s\n"
      "add x10, x10, #0x40\n"
      "add v26.4s, v26.4s, v13.4s\n"
      "add v27.4s, v27.4s, v13.4s\n"
      "add v31.4s, v31.4s, v0.4s\n"
      "add v20.4s, v20.4s, v30.4s\n"
      "add v21.4s, v21.4s, v29.4s\n"
      "add v22.4s, v22.4s, v28.4s\n"
      "add v16.4s, v16.4s, v0.4s\n"
      "add v17.4s, v17.4s, v30.4s\n"
      "add v18.4s, v18.4s, v29.4s\n"
      "add v19.4s, v19.4s, v28.4s\n"
      "add v24.4s, v24.4s, v0.4s\n"
      "ld1r { v0.4s }, [x20]\n"
      "add v25.4s, v25.4s, v30.4s\n"
      "add v26.4s, v26.4s, v29.4s\n"
      "add v27.4s, v27.4s, v28.4s\n"
      "sqrdmulh v31.4s, v31.4s, v23.4s\n"
      "sqrdmulh v20.4s, v20.4s, v23.4s\n"
      "sqrdmulh v21.4s, v21.4s, v23.4s\n"
      "sqrdmulh v22.4s, v22.4s, v23.4s\n"
      "sqrdmulh v16.4s, v16.4s, v23.4s\n"
      "sqrdmulh v17.4s, v17.4s, v23.4s\n"
      "sqrdmulh v18.4s, v18.4s, v23.4s\n"
      "sqrdmulh v19.4s, v19.4s, v23.4s\n"
      "sqrdmulh v24.4s, v24.4s, v23.4s\n"
      "sqrdmulh v25.4s, v25.4s, v23.4s\n"
      "sqrdmulh v26.4s, v26.4s, v23.4s\n"
      "sqrdmulh v27.4s, v27.4s, v23.4s\n"
      "tbz %x[flags], #5, 86f\n"
      "and v1.16b, v31.16b, v0.16b\n"
      "and v30.16b, v20.16b, v0.16b\n"
      "and v29.16b, v21.16b, v0.16b\n"
      "and v28.16b, v22.16b, v0.16b\n"
      "and v23.16b, v16.16b, v0.16b\n"
      "and v3.16b, v17.16b, v0.16b\n"
      "sshr v1.4s, v1.4s, #0x1f\n"
      "sshr v30.4s, v30.4s, #0x1f\n"
      "sshr v29.4s, v29.4s, #0x1f\n"
      "sshr v28.4s, v28.4s, #0x1f\n"
      "sshr v23.4s, v23.4s, #0x1f\n"
      "and v2.16b, v18.16b, v0.16b\n"
      "sqadd v31.4s, v31.4s, v1.4s\n"
      "sqadd v20.4s, v20.4s, v30.4s\n"
      "sqadd v21.4s, v21.4s, v29.4s\n"
      "sqadd v22.4s, v22.4s, v28.4s\n"
      "sqadd v16.4s, v16.4s, v23.4s\n"
      "and v1.16b, v19.16b, v0.16b\n"
      "and v30.16b, v24.16b, v0.16b\n"
      "and v29.16b, v25.16b, v0.16b\n"
      "and v28.16b, v26.16b, v0.16b\n"
      "and v23.16b, v27.16b, v0.16b\n"
      "sshr v3.4s, v3.4s, #0x1f\n"
      "sshr v2.4s, v2.4s, #0x1f\n"
      "sshr v1.4s, v1.4s, #0x1f\n"
      "sshr v30.4s, v30.4s, #0x1f\n"
      "sshr v29.4s, v29.4s, #0x1f\n"
      "sshr v28.4s, v28.4s, #0x1f\n"
      "sshr v23.4s, v23.4s, #0x1f\n"
      "sqadd v17.4s, v17.4s, v3.4s\n"
      "sqadd v18.4s, v18.4s, v2.4s\n"
      "sqadd v19.4s, v19.4s, v1.4s\n"
      "sqadd v24.4s, v24.4s, v30.4s\n"
      "sqadd v25.4s, v25.4s, v29.4s\n"
      "sqadd v26.4s, v26.4s, v28.4s\n"
      "sqadd v27.4s, v27.4s, v23.4s\n"
      "86:"  // Height 3: no shift correction
      "add x21, %x[qp], %[c_offset]\n"
      "srshl v31.4s, v31.4s, v0.4s\n"
      "srshl v20.4s, v20.4s, v0.4s\n"
      "add x20, %x[qp], %[maxval]\n"
      "ld1r { v29.4s }, [x21]\n"
      "ld1r { v28.4s }, [x20]\n"
      "srshl v21.4s, v21.4s, v0.4s\n"
      "srshl v22.4s, v22.4s, v0.4s\n"
      "srshl v16.4s, v16.4s, v0.4s\n"
      "srshl v17.4s, v17.4s, v0.4s\n"
      "add x20, %x[qp], %[minval]\n"
      "cmp x9, #0x10\n"
      "ld1r { v23.4s }, [x20]\n"
      "srshl v18.4s, v18.4s, v0.4s\n"
      "srshl v19.4s, v19.4s, v0.4s\n"
      "srshl v24.4s, v24.4s, v0.4s\n"
      "srshl v25.4s, v25.4s, v0.4s\n"
      "srshl v26.4s, v26.4s, v0.4s\n"
      "srshl v27.4s, v27.4s, v0.4s\n"
      "add v31.4s, v31.4s, v29.4s\n"
      "add v20.4s, v20.4s, v29.4s\n"
      "add v21.4s, v21.4s, v29.4s\n"
      "add v22.4s, v22.4s, v29.4s\n"
      "add v16.4s, v16.4s, v29.4s\n"
      "add v17.4s, v17.4s, v29.4s\n"
      "add v18.4s, v18.4s, v29.4s\n"
      "add v19.4s, v19.4s, v29.4s\n"
      "add v24.4s, v24.4s, v29.4s\n"
      "add v25.4s, v25.4s, v29.4s\n"
      "add v26.4s, v26.4s, v29.4s\n"
      "add v27.4s, v27.4s, v29.4s\n"
      "smin v31.4s, v31.4s, v28.4s\n"
      "smin v20.4s, v20.4s, v28.4s\n"
      "smin v21.4s, v21.4s, v28.4s\n"
      "smin v22.4s, v22.4s, v28.4s\n"
      "smin v16.4s, v16.4s, v28.4s\n"
      "smin v17.4s, v17.4s, v28.4s\n"
      "smin v18.4s, v18.4s, v28.4s\n"
      "smin v19.4s, v19.4s, v28.4s\n"
      "smin v24.4s, v24.4s, v28.4s\n"
      "smin v25.4s, v25.4s, v28.4s\n"
      "smin v26.4s, v26.4s, v28.4s\n"
      "smin v27.4s, v27.4s, v28.4s\n"
      "smax v31.4s, v31.4s, v23.4s\n"
      "smax v20.4s, v20.4s, v23.4s\n"
      "smax v21.4s, v21.4s, v23.4s\n"
      "smax v22.4s, v22.4s, v23.4s\n"
      "smax v16.4s, v16.4s, v23.4s\n"
      "smax v17.4s, v17.4s, v23.4s\n"
      "smax v18.4s, v18.4s, v23.4s\n"
      "smax v19.4s, v19.4s, v23.4s\n"
      "smax v24.4s, v24.4s, v23.4s\n"
      "smax v25.4s, v25.4s, v23.4s\n"
      "smax v26.4s, v26.4s, v23.4s\n"
      "smax v27.4s, v27.4s, v23.4s\n"
      "uzp1 v31.8h, v31.8h, v20.8h\n"
      "uzp1 v20.8h, v21.8h, v22.8h\n"
      "uzp1 v16.8h, v16.8h, v17.8h\n"
      "uzp1 v18.8h, v18.8h, v19.8h\n"
      "uzp1 v24.8h, v24.8h, v25.8h\n"
      "uzp1 v17.8h, v26.8h, v27.8h\n"
      "uzp1 v31.16b, v31.16b, v20.16b\n"
      "uzp1 v16.16b, v16.16b, v18.16b\n"
      "uzp1 v24.16b, v24.16b, v17.16b\n"
      "bge 95f\n"
      "tbz x9, #3, 90f\n"
      "str d31, [x27], #0x8\n"
      "str d16, [x24], #0x8\n"
      "str d24, [x23], #0x8\n"
      "tbz x9, #2, 88f\n"
      "st1 { v31.s }[2], [x27], #0x4\n"
      "st1 { v16.s }[2], [x24], #0x4\n"
      "st1 { v24.s }[2], [x23], #0x4\n"
      "tbz x9, #1, 87f\n"
      "st1 { v31.h }[6], [x27], #0x2\n"
      "st1 { v16.h }[6], [x24], #0x2\n"
      "st1 { v24.h }[6], [x23], #0x2\n"
      "tbz x9, #0, 94f\n"
      "st1 { v31.b }[14], [x27]\n"
      "st1 { v16.b }[14], [x24]\n"
      "st1 { v24.b }[14], [x23]\n"
      "b 94f\n"
      "87:"  // Height 3: Partial direct writeback: partial_1_12
      "tbz x9, #0, 94f\n"
      "st1 { v31.b }[12], [x27]\n"
      "st1 { v16.b }[12], [x24]\n"
      "st1 { v24.b }[12], [x23]\n"
      "b 94f\n"
      "88:"  // Height 3: Partial direct writeback: partial_2_8
      "tbz x9, #1, 89f\n"
      "st1 { v31.h }[4], [x27], #0x2\n"
      "st1 { v16.h }[4], [x24], #0x2\n"
      "st1 { v24.h }[4], [x23], #0x2\n"
      "tbz x9, #0, 94f\n"
      "st1 { v31.b }[10], [x27]\n"
      "st1 { v16.b }[10], [x24]\n"
      "st1 { v24.b }[10], [x23]\n"
      "b 94f\n"
      "89:"  // Height 3: Partial direct writeback: partial_1_8
      "tbz x9, #0, 94f\n"
      "st1 { v31.b }[8], [x27]\n"
      "st1 { v16.b }[8], [x24]\n"
      "st1 { v24.b }[8], [x23]\n"
      "b 94f\n"
      "90:"  // Height 3: Partial direct writeback: partial_4_0
      "tbz x9, #2, 92f\n"
      "str s31, [x27], #0x4\n"
      "str s16, [x24], #0x4\n"
      "str s24, [x23], #0x4\n"
      "tbz x9, #1, 91f\n"
      "st1 { v31.h }[2], [x27], #0x2\n"
      "st1 { v16.h }[2], [x24], #0x2\n"
      "st1 { v24.h }[2], [x23], #0x2\n"
      "tbz x9, #0, 94f\n"
      "st1 { v31.b }[6], [x27]\n"
      "st1 { v16.b }[6], [x24]\n"
      "st1 { v24.b }[6], [x23]\n"
      "b 94f\n"
      "91:"  // Height 3: Partial direct writeback: partial_1_4
      "tbz x9, #0, 94f\n"
      "st1 { v31.b }[4], [x27]\n"
      "st1 { v16.b }[4], [x24]\n"
      "st1 { v24.b }[4], [x23]\n"
      "b 94f\n"
      "92:"  // Height 3: Partial direct writeback: partial_2_0
      "tbz x9, #1, 93f\n"
      "str h31, [x27], #0x2\n"
      "str h16, [x24], #0x2\n"
      "str h24, [x23], #0x2\n"
      "tbz x9, #0, 94f\n"
      "st1 { v31.b }[2], [x27]\n"
      "st1 { v16.b }[2], [x24]\n"
      "st1 { v24.b }[2], [x23]\n"
      "b 94f\n"
      "93:"  // Height 3: Partial direct writeback: partial_1_0
      "str b31, [x27, #0x0]\n"
      "str b16, [x24, #0x0]\n"
      "str b24, [x23, #0x0]\n"
      "94:"  // Height 3: Partial direct writeback: Done
      "b 96f\n"
      "95:"  // Height 3: Full writeback
      "str q31, [x27, #0x0]\n"
      "add x27, x27, #0x10\n"
      "str q16, [x24, #0x0]\n"
      "str q24, [x23, #0x0]\n"
      "96:"  // Height 3: Writeback done
      "subs x9, x9, #0x10\n"
      "bgt 66b\n"
      "b 130f\n"
      "97:"  // Height 4
      "ldr x21, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "ldr x27, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "mov x20, #0x4\n"
      "mov x10, %x[col_bias]\n"
      "movi v11.4s, #0x0\n"
      "movi v12.4s, #0x0\n"
      "bic %x[flags], %x[flags], #0x80000000\n"
      "ldr x9, [%x[args_ptr], %[offsetof_N]]\n"
      "movi v13.4s, #0x0\n"
      "movi v14.4s, #0x0\n"
      "ldr x28, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "madd x20, x21, x20, x27\n"
      "movi v15.16b, #0x1\n"
      "str x20, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "98:"  // Height 4: Column loop
      "movi v16.4s, #0x0\n"
      "movi v17.4s, #0x0\n"
      "movi v18.4s, #0x0\n"
      "movi v19.4s, #0x0\n"
      "movi v20.4s, #0x0\n"
      "movi v21.4s, #0x0\n"
      "movi v22.4s, #0x0\n"
      "movi v23.4s, #0x0\n"
      "movi v24.4s, #0x0\n"
      "movi v25.4s, #0x0\n"
      "movi v26.4s, #0x0\n"
      "movi v27.4s, #0x0\n"
      "movi v28.4s, #0x0\n"
      "movi v29.4s, #0x0\n"
      "movi v30.4s, #0x0\n"
      "movi v31.4s, #0x0\n"
      "99:"  // Height 4: setup done
      "mov x26, #0x0\n"
      "100:"  // Height 4: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w25, [x20, x26, LSL #0x2]\n"
      "tbz %x[flags], #3, 101f\n"
      "ldr x20, [%x[input_ptr], x26, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x24, [x20, #0x0]\n"
      "ldr x23, [x20, #0x8]\n"
      "ldr x22, [x20, #0x10]\n"
      "ldr x21, [x20, #0x18]\n"
      "cbnz x26, 102f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x24, x24, x20\n"
      "add x23, x23, x20\n"
      "add x22, x22, x20\n"
      "add x21, x21, x20\n"
      "b 102f\n"
      "101:"  // Height 4: setup direct input
      "mov x24, %x[input_ptr]\n"
      "add x23, x24, x21\n"
      "add x22, x23, x21\n"
      "add x21, x22, x21\n"
      "102:"  // Height 4: input setup done
      "cmp x25, #0x10\n"
      "blt 107f\n"
      "ldr q1, [x24, #0x0]\n"
      "ldr q2, [x23, #0x0]\n"
      "cmp x25, #0x20\n"
      "ldr q3, [x22, #0x0]\n"
      "ldr q4, [x21, #0x0]\n"
      "ldr q5, [x28, #0x0]\n"
      "ldr q6, [x28, #0x10]\n"
      "ldr q7, [x28, #0x20]\n"
      "ldr q8, [x28, #0x30]\n"
      "ldr q9, [x28, #0x40]\n"
      "ldr q10, [x28, #0x50]\n"
      "blt 105f\n"
      "103:"  // Height 4: Multiply loop: Main loop head
      "trn1 v0.2d, v1.2d, v2.2d\n"
      "trn2 v1.2d, v1.2d, v2.2d\n"
      "add x24, x24, #0x10\n"
      "add x23, x23, #0x10\n"
      "trn1 v2.2d, v3.2d, v4.2d\n"
      "trn2 v3.2d, v3.2d, v4.2d\n"
      "ldr q4, [x28, #0x60]\n"
      "add x22, x22, #0x10\n"
      "add x21, x21, #0x10\n"
      ".inst 0x4e85a410  // smmla v16.4s, v0.16b, v5.16b\n"
      ".inst 0x4e86a414  // smmla v20.4s, v0.16b, v6.16b\n"
      ".inst 0x4e87a411  // smmla v17.4s, v0.16b, v7.16b\n"
      ".inst 0x4e88a415  // smmla v21.4s, v0.16b, v8.16b\n"
      ".inst 0x4e85a458  // smmla v24.4s, v2.16b, v5.16b\n"
      "ldr q5, [x28, #0x70]\n"
      ".inst 0x4e86a45c  // smmla v28.4s, v2.16b, v6.16b\n"
      "ldr q6, [x28, #0x80]\n"
      ".inst 0x4e87a459  // smmla v25.4s, v2.16b, v7.16b\n"
      "ldr q7, [x28, #0x90]\n"
      ".inst 0x4e88a45d  // smmla v29.4s, v2.16b, v8.16b\n"
      "ldr q8, [x28, #0xa0]\n"
      ".inst 0x4e89a412  // smmla v18.4s, v0.16b, v9.16b\n"
      ".inst 0x4e89a45a  // smmla v26.4s, v2.16b, v9.16b\n"
      "ldr q9, [x28, #0xb0]\n"
      ".inst 0x4e8aa416  // smmla v22.4s, v0.16b, v10.16b\n"
      ".inst 0x4e8aa45e  // smmla v30.4s, v2.16b, v10.16b\n"
      "ldr q10, [x28, #0xc0]\n"
      ".inst 0x4e84a413  // smmla v19.4s, v0.16b, v4.16b\n"
      ".inst 0x4e84a45b  // smmla v27.4s, v2.16b, v4.16b\n"
      "ldr q4, [x28, #0xd0]\n"
      ".inst 0x4e85a417  // smmla v23.4s, v0.16b, v5.16b\n"
      ".inst 0x4e85a45f  // smmla v31.4s, v2.16b, v5.16b\n"
      "ldr q5, [x28, #0xe0]\n"
      ".inst 0x4e86a430  // smmla v16.4s, v1.16b, v6.16b\n"
      ".inst 0x4e86a478  // smmla v24.4s, v3.16b, v6.16b\n"
      "ldr q6, [x28, #0xf0]\n"
      "add x28, x28, #0x100\n"
      ".inst 0x4e87a434  // smmla v20.4s, v1.16b, v7.16b\n"
      ".inst 0x4e87a47c  // smmla v28.4s, v3.16b, v7.16b\n"
      ".inst 0x4e88a431  // smmla v17.4s, v1.16b, v8.16b\n"
      ".inst 0x4e88a479  // smmla v25.4s, v3.16b, v8.16b\n"
      ".inst 0x4e89a435  // smmla v21.4s, v1.16b, v9.16b\n"
      ".inst 0x4e89a47d  // smmla v29.4s, v3.16b, v9.16b\n"
      ".inst 0x4e8aa432  // smmla v18.4s, v1.16b, v10.16b\n"
      ".inst 0x4e8aa47a  // smmla v26.4s, v3.16b, v10.16b\n"
      ".inst 0x4e84a436  // smmla v22.4s, v1.16b, v4.16b\n"
      ".inst 0x4e84a47e  // smmla v30.4s, v3.16b, v4.16b\n"
      ".inst 0x4e85a433  // smmla v19.4s, v1.16b, v5.16b\n"
      ".inst 0x4e85a47b  // smmla v27.4s, v3.16b, v5.16b\n"
      ".inst 0x4e86a437  // smmla v23.4s, v1.16b, v6.16b\n"
      ".inst 0x4e86a47f  // smmla v31.4s, v3.16b, v6.16b\n"
      "tbnz %x[flags], #31, 104f\n"
      ".inst 0x4e8f940b  // sdot v11.4s, v0.16b, v15.16b\n"
      ".inst 0x4e8f944d  // sdot v13.4s, v2.16b, v15.16b\n"
      ".inst 0x4e8f942b  // sdot v11.4s, v1.16b, v15.16b\n"
      ".inst 0x4e8f946d  // sdot v13.4s, v3.16b, v15.16b\n"
      "104:"  // Height 4: Multiply loop: unique 13: skip row sum
      "ldr q1, [x24, #0x0]\n"
      "ldr q2, [x23, #0x0]\n"
      "sub x25, x25, #0x10\n"
      "ldr q3, [x22, #0x0]\n"
      "ldr q4, [x21, #0x0]\n"
      "cmp x25, #0x20\n"
      "ldr q5, [x28, #0x0]\n"
      "ldr q6, [x28, #0x10]\n"
      "ldr q7, [x28, #0x20]\n"
      "ldr q8, [x28, #0x30]\n"
      "ldr q9, [x28, #0x40]\n"
      "ldr q10, [x28, #0x50]\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      "prfm pldl1keep, [x23, #0x80]\n"
      "prfm pldl1keep, [x22, #0x80]\n"
      "prfm pldl1keep, [x21, #0x80]\n"
      "bge 103b\n"
      "105:"  // Height 4: Multiply loop: Single iteration only
      "trn1 v0.2d, v1.2d, v2.2d\n"
      "trn2 v1.2d, v1.2d, v2.2d\n"
      "sub x25, x25, #0x10\n"
      "add x24, x24, #0x10\n"
      "trn1 v2.2d, v3.2d, v4.2d\n"
      "trn2 v3.2d, v3.2d, v4.2d\n"
      "ldr q4, [x28, #0x60]\n"
      "add x23, x23, #0x10\n"
      "add x22, x22, #0x10\n"
      "add x21, x21, #0x10\n"
      ".inst 0x4e85a410  // smmla v16.4s, v0.16b, v5.16b\n"
      ".inst 0x4e86a414  // smmla v20.4s, v0.16b, v6.16b\n"
      ".inst 0x4e87a411  // smmla v17.4s, v0.16b, v7.16b\n"
      ".inst 0x4e88a415  // smmla v21.4s, v0.16b, v8.16b\n"
      ".inst 0x4e85a458  // smmla v24.4s, v2.16b, v5.16b\n"
      "ldr q5, [x28, #0x70]\n"
      ".inst 0x4e86a45c  // smmla v28.4s, v2.16b, v6.16b\n"
      "ldr q6, [x28, #0x80]\n"
      ".inst 0x4e87a459  // smmla v25.4s, v2.16b, v7.16b\n"
      "ldr q7, [x28, #0x90]\n"
      ".inst 0x4e88a45d  // smmla v29.4s, v2.16b, v8.16b\n"
      "ldr q8, [x28, #0xa0]\n"
      ".inst 0x4e89a412  // smmla v18.4s, v0.16b, v9.16b\n"
      ".inst 0x4e89a45a  // smmla v26.4s, v2.16b, v9.16b\n"
      "ldr q9, [x28, #0xb0]\n"
      ".inst 0x4e8aa416  // smmla v22.4s, v0.16b, v10.16b\n"
      ".inst 0x4e8aa45e  // smmla v30.4s, v2.16b, v10.16b\n"
      "ldr q10, [x28, #0xc0]\n"
      ".inst 0x4e84a413  // smmla v19.4s, v0.16b, v4.16b\n"
      ".inst 0x4e84a45b  // smmla v27.4s, v2.16b, v4.16b\n"
      "ldr q4, [x28, #0xd0]\n"
      ".inst 0x4e85a417  // smmla v23.4s, v0.16b, v5.16b\n"
      ".inst 0x4e85a45f  // smmla v31.4s, v2.16b, v5.16b\n"
      "ldr q5, [x28, #0xe0]\n"
      ".inst 0x4e86a430  // smmla v16.4s, v1.16b, v6.16b\n"
      ".inst 0x4e86a478  // smmla v24.4s, v3.16b, v6.16b\n"
      "ldr q6, [x28, #0xf0]\n"
      "add x28, x28, #0x100\n"
      ".inst 0x4e87a434  // smmla v20.4s, v1.16b, v7.16b\n"
      ".inst 0x4e87a47c  // smmla v28.4s, v3.16b, v7.16b\n"
      ".inst 0x4e88a431  // smmla v17.4s, v1.16b, v8.16b\n"
      ".inst 0x4e88a479  // smmla v25.4s, v3.16b, v8.16b\n"
      ".inst 0x4e89a435  // smmla v21.4s, v1.16b, v9.16b\n"
      ".inst 0x4e89a47d  // smmla v29.4s, v3.16b, v9.16b\n"
      ".inst 0x4e8aa432  // smmla v18.4s, v1.16b, v10.16b\n"
      ".inst 0x4e8aa47a  // smmla v26.4s, v3.16b, v10.16b\n"
      ".inst 0x4e84a436  // smmla v22.4s, v1.16b, v4.16b\n"
      ".inst 0x4e84a47e  // smmla v30.4s, v3.16b, v4.16b\n"
      ".inst 0x4e85a433  // smmla v19.4s, v1.16b, v5.16b\n"
      ".inst 0x4e85a47b  // smmla v27.4s, v3.16b, v5.16b\n"
      ".inst 0x4e86a437  // smmla v23.4s, v1.16b, v6.16b\n"
      ".inst 0x4e86a47f  // smmla v31.4s, v3.16b, v6.16b\n"
      "tbnz %x[flags], #31, 106f\n"
      ".inst 0x4e8f940b  // sdot v11.4s, v0.16b, v15.16b\n"
      ".inst 0x4e8f944d  // sdot v13.4s, v2.16b, v15.16b\n"
      ".inst 0x4e8f942b  // sdot v11.4s, v1.16b, v15.16b\n"
      ".inst 0x4e8f946d  // sdot v13.4s, v3.16b, v15.16b\n"
      "106:"  // Height 4: Multiply loop: unique 14: skip row sum
      "prfm pldl1keep, [x24, #0x80]\n"
      "prfm pldl1keep, [x23, #0x80]\n"
      "prfm pldl1keep, [x22, #0x80]\n"
      "prfm pldl1keep, [x21, #0x80]\n"
      "107:"  // Height 4: Multiply loop: Main loop skip
      "cbz x25, 116f\n"
      "cmp x25, #0x8\n"
      "blt 110f\n"
      "108:"  // Height 4: Multiply loop: Odd block loop
      "ldr d3, [x24], #0x8\n"
      "ldr d0, [x23], #0x8\n"
      "ldr d2, [x22], #0x8\n"
      "ldr d1, [x21], #0x8\n"
      "trn1 v0.2d, v3.2d, v0.2d\n"
      "trn1 v2.2d, v2.2d, v1.2d\n"
      "tbnz %x[flags], #31, 109f\n"
      ".inst 0x4e8f940b  // sdot v11.4s, v0.16b, v15.16b\n"
      ".inst 0x4e8f944d  // sdot v13.4s, v2.16b, v15.16b\n"
      "109:"  // Height 4: Multiply loop: unique 15: skip row sum
      "ldr q1, [x28, #0x0]\n"
      "ldr q8, [x28, #0x10]\n"
      "sub x25, x25, #0x8\n"
      "ldr q7, [x28, #0x20]\n"
      "ldr q6, [x28, #0x30]\n"
      "cmp x25, #0x8\n"
      "ldr q5, [x28, #0x40]\n"
      "ldr q4, [x28, #0x50]\n"
      "ldr q3, [x28, #0x60]\n"
      ".inst 0x4e81a410  // smmla v16.4s, v0.16b, v1.16b\n"
      ".inst 0x4e81a458  // smmla v24.4s, v2.16b, v1.16b\n"
      "ldr q1, [x28, #0x70]\n"
      ".inst 0x4e88a414  // smmla v20.4s, v0.16b, v8.16b\n"
      ".inst 0x4e88a45c  // smmla v28.4s, v2.16b, v8.16b\n"
      "add x28, x28, #0x80\n"
      ".inst 0x4e87a411  // smmla v17.4s, v0.16b, v7.16b\n"
      ".inst 0x4e87a459  // smmla v25.4s, v2.16b, v7.16b\n"
      ".inst 0x4e86a415  // smmla v21.4s, v0.16b, v6.16b\n"
      ".inst 0x4e86a45d  // smmla v29.4s, v2.16b, v6.16b\n"
      ".inst 0x4e85a412  // smmla v18.4s, v0.16b, v5.16b\n"
      ".inst 0x4e85a45a  // smmla v26.4s, v2.16b, v5.16b\n"
      ".inst 0x4e84a416  // smmla v22.4s, v0.16b, v4.16b\n"
      ".inst 0x4e84a45e  // smmla v30.4s, v2.16b, v4.16b\n"
      ".inst 0x4e83a413  // smmla v19.4s, v0.16b, v3.16b\n"
      ".inst 0x4e83a45b  // smmla v27.4s, v2.16b, v3.16b\n"
      ".inst 0x4e81a417  // smmla v23.4s, v0.16b, v1.16b\n"
      ".inst 0x4e81a45f  // smmla v31.4s, v2.16b, v1.16b\n"
      "bge 108b\n"
      "110:"  // Height 4: Multiply loop: Skip odd blocks
      "cbz x25, 116f\n"
      "tbz x25, #2, 112f\n"
      "ldr s1, [x24], #0x4\n"
      "ldr s2, [x23], #0x4\n"
      "ldr s3, [x22], #0x4\n"
      "ldr s9, [x21], #0x4\n"
      "tbz x25, #1, 111f\n"
      "ld1 { v1.h }[2], [x24], #0x2\n"
      "ld1 { v2.h }[2], [x23], #0x2\n"
      "ld1 { v3.h }[2], [x22], #0x2\n"
      "ld1 { v9.h }[2], [x21], #0x2\n"
      "tbz x25, #0, 114f\n"
      "ld1 { v1.b }[6], [x24]\n"
      "ld1 { v2.b }[6], [x23]\n"
      "ld1 { v3.b }[6], [x22]\n"
      "ld1 { v9.b }[6], [x21]\n"
      "b 114f\n"
      "111:"  // Height 4: Multiply loop: Ragged operand read: partial_1_4
      "tbz x25, #0, 114f\n"
      "ld1 { v1.b }[4], [x24]\n"
      "ld1 { v2.b }[4], [x23]\n"
      "ld1 { v3.b }[4], [x22]\n"
      "ld1 { v9.b }[4], [x21]\n"
      "b 114f\n"
      "112:"  // Height 4: Multiply loop: Ragged operand read: partial_2_0
      "tbz x25, #1, 113f\n"
      "ldr h1, [x24], #0x2\n"
      "ldr h2, [x23], #0x2\n"
      "ldr h3, [x22], #0x2\n"
      "ldr h9, [x21], #0x2\n"
      "tbz x25, #0, 114f\n"
      "ld1 { v1.b }[2], [x24]\n"
      "ld1 { v2.b }[2], [x23]\n"
      "ld1 { v3.b }[2], [x22]\n"
      "ld1 { v9.b }[2], [x21]\n"
      "b 114f\n"
      "113:"  // Height 4: Multiply loop: Ragged operand read: partial_1_0
      "ldr b1, [x24, #0x0]\n"
      "ldr b2, [x23, #0x0]\n"
      "ldr b3, [x22, #0x0]\n"
      "ldr b9, [x21, #0x0]\n"
      "114:"  // Height 4: Multiply loop: Ragged operand read: Done
      "trn1 v0.2d, v1.2d, v2.2d\n"
      "trn1 v2.2d, v3.2d, v9.2d\n"
      "tbnz %x[flags], #31, 115f\n"
      ".inst 0x4e8f940b  // sdot v11.4s, v0.16b, v15.16b\n"
      ".inst 0x4e8f944d  // sdot v13.4s, v2.16b, v15.16b\n"
      "115:"  // Height 4: Multiply loop: unique 16: skip row sum
      "ldr q1, [x28, #0x0]\n"
      "ldr q8, [x28, #0x10]\n"
      "ldr q7, [x28, #0x20]\n"
      "ldr q6, [x28, #0x30]\n"
      "ldr q5, [x28, #0x40]\n"
      "ldr q4, [x28, #0x50]\n"
      "ldr q3, [x28, #0x60]\n"
      ".inst 0x4e81a410  // smmla v16.4s, v0.16b, v1.16b\n"
      ".inst 0x4e81a458  // smmla v24.4s, v2.16b, v1.16b\n"
      "ldr q1, [x28, #0x70]\n"
      ".inst 0x4e88a414  // smmla v20.4s, v0.16b, v8.16b\n"
      ".inst 0x4e88a45c  // smmla v28.4s, v2.16b, v8.16b\n"
      "add x28, x28, #0x80\n"
      ".inst 0x4e87a411  // smmla v17.4s, v0.16b, v7.16b\n"
      ".inst 0x4e87a459  // smmla v25.4s, v2.16b, v7.16b\n"
      ".inst 0x4e86a415  // smmla v21.4s, v0.16b, v6.16b\n"
      ".inst 0x4e86a45d  // smmla v29.4s, v2.16b, v6.16b\n"
      ".inst 0x4e85a412  // smmla v18.4s, v0.16b, v5.16b\n"
      ".inst 0x4e85a45a  // smmla v26.4s, v2.16b, v5.16b\n"
      ".inst 0x4e84a416  // smmla v22.4s, v0.16b, v4.16b\n"
      ".inst 0x4e84a45e  // smmla v30.4s, v2.16b, v4.16b\n"
      ".inst 0x4e83a413  // smmla v19.4s, v0.16b, v3.16b\n"
      ".inst 0x4e83a45b  // smmla v27.4s, v2.16b, v3.16b\n"
      ".inst 0x4e81a417  // smmla v23.4s, v0.16b, v1.16b\n"
      ".inst 0x4e81a45f  // smmla v31.4s, v2.16b, v1.16b\n"
      "116:"  // Height 4: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x26, x26, #0x1\n"
      "cmp x26, x20\n"
      "bne 100b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "uzp1 v0.2d, v16.2d, v20.2d\n"
      "uzp2 v16.2d, v16.2d, v20.2d\n"
      "prfm pstl1keep, [x27, #0x0]\n"
      "uzp1 v20.2d, v17.2d, v21.2d\n"
      "uzp2 v17.2d, v17.2d, v21.2d\n"
      "uzp1 v21.2d, v18.2d, v22.2d\n"
      "uzp2 v18.2d, v18.2d, v22.2d\n"
      "add x24, x27, x20\n"
      "add x23, x24, x20\n"
      "add x22, x23, x20\n"
      "uzp1 v22.2d, v19.2d, v23.2d\n"
      "uzp2 v19.2d, v19.2d, v23.2d\n"
      "prfm pstl1keep, [x24, #0x0]\n"
      "uzp1 v23.2d, v24.2d, v28.2d\n"
      "uzp2 v24.2d, v24.2d, v28.2d\n"
      "prfm pstl1keep, [x23, #0x0]\n"
      "prfm pstl1keep, [x22, #0x0]\n"
      "uzp1 v28.2d, v25.2d, v29.2d\n"
      "uzp2 v25.2d, v25.2d, v29.2d\n"
      "uzp1 v29.2d, v26.2d, v30.2d\n"
      "uzp2 v26.2d, v26.2d, v30.2d\n"
      "uzp1 v30.2d, v27.2d, v31.2d\n"
      "uzp2 v27.2d, v27.2d, v31.2d\n"
      "mov v31.16b, v0.16b\n"
      "tbnz %x[flags], #31, 117f\n"
      "add x20, %x[qp], %[b_offset]\n"
      "addp v11.4s, v11.4s, v11.4s\n"
      "addp v13.4s, v13.4s, v13.4s\n"
      "ld1r { v0.4s }, [x20]\n"
      "neg v0.4s, v0.4s\n"
      "dup v12.4s, v11.s[3]\n"
      "dup v11.4s, v11.s[0]\n"
      "dup v14.4s, v13.s[3]\n"
      "dup v13.4s, v13.s[0]\n"
      "mul v11.4s, v11.4s, v0.4s\n"
      "mul v12.4s, v12.4s, v0.4s\n"
      "mul v14.4s, v14.4s, v0.4s\n"
      "mul v13.4s, v13.4s, v0.4s\n"
      "117:"  // Height 4: skip row sum fixup
      "ldr q0, [x10, #0x0]\n"
      "ldr q4, [x10, #0x10]\n"
      "add v31.4s, v31.4s, v11.4s\n"
      "add v20.4s, v20.4s, v11.4s\n"
      "ldr q3, [x10, #0x20]\n"
      "ldr q2, [x10, #0x30]\n"
      "add v21.4s, v21.4s, v11.4s\n"
      "add v22.4s, v22.4s, v11.4s\n"
      "add v16.4s, v16.4s, v12.4s\n"
      "add v17.4s, v17.4s, v12.4s\n"
      "add x20, %x[qp], %[per_layer_mul]\n"
      "orr %x[flags], %x[flags], #0x80000000\n"
      "ld1r { v1.4s }, [x20]\n"
      "add v18.4s, v18.4s, v12.4s\n"
      "add v19.4s, v19.4s, v12.4s\n"
      "add x20, %x[qp], %[per_layer_right_shift]\n"
      "add v23.4s, v23.4s, v13.4s\n"
      "add v28.4s, v28.4s, v13.4s\n"
      "add x10, x10, #0x40\n"
      "add v29.4s, v29.4s, v13.4s\n"
      "add v30.4s, v30.4s, v13.4s\n"
      "add v24.4s, v24.4s, v14.4s\n"
      "add v25.4s, v25.4s, v14.4s\n"
      "add v26.4s, v26.4s, v14.4s\n"
      "add v27.4s, v27.4s, v14.4s\n"
      "add v31.4s, v31.4s, v0.4s\n"
      "add v20.4s, v20.4s, v4.4s\n"
      "add v21.4s, v21.4s, v3.4s\n"
      "add v22.4s, v22.4s, v2.4s\n"
      "add v16.4s, v16.4s, v0.4s\n"
      "add v17.4s, v17.4s, v4.4s\n"
      "add v18.4s, v18.4s, v3.4s\n"
      "add v19.4s, v19.4s, v2.4s\n"
      "add v23.4s, v23.4s, v0.4s\n"
      "add v28.4s, v28.4s, v4.4s\n"
      "add v29.4s, v29.4s, v3.4s\n"
      "add v30.4s, v30.4s, v2.4s\n"
      "add v24.4s, v24.4s, v0.4s\n"
      "ld1r { v0.4s }, [x20]\n"
      "add v25.4s, v25.4s, v4.4s\n"
      "add v26.4s, v26.4s, v3.4s\n"
      "add v27.4s, v27.4s, v2.4s\n"
      "sqrdmulh v31.4s, v31.4s, v1.4s\n"
      "sqrdmulh v20.4s, v20.4s, v1.4s\n"
      "sqrdmulh v21.4s, v21.4s, v1.4s\n"
      "sqrdmulh v22.4s, v22.4s, v1.4s\n"
      "sqrdmulh v16.4s, v16.4s, v1.4s\n"
      "sqrdmulh v17.4s, v17.4s, v1.4s\n"
      "sqrdmulh v18.4s, v18.4s, v1.4s\n"
      "sqrdmulh v19.4s, v19.4s, v1.4s\n"
      "sqrdmulh v23.4s, v23.4s, v1.4s\n"
      "sqrdmulh v28.4s, v28.4s, v1.4s\n"
      "sqrdmulh v29.4s, v29.4s, v1.4s\n"
      "sqrdmulh v30.4s, v30.4s, v1.4s\n"
      "sqrdmulh v24.4s, v24.4s, v1.4s\n"
      "sqrdmulh v25.4s, v25.4s, v1.4s\n"
      "sqrdmulh v26.4s, v26.4s, v1.4s\n"
      "sqrdmulh v27.4s, v27.4s, v1.4s\n"
      "tbz %x[flags], #5, 118f\n"
      "and v2.16b, v31.16b, v0.16b\n"
      "and v1.16b, v20.16b, v0.16b\n"
      "and v7.16b, v21.16b, v0.16b\n"
      "and v6.16b, v22.16b, v0.16b\n"
      "and v5.16b, v16.16b, v0.16b\n"
      "and v4.16b, v17.16b, v0.16b\n"
      "sshr v2.4s, v2.4s, #0x1f\n"
      "sshr v1.4s, v1.4s, #0x1f\n"
      "and v3.16b, v18.16b, v0.16b\n"
      "sshr v7.4s, v7.4s, #0x1f\n"
      "sshr v6.4s, v6.4s, #0x1f\n"
      "sshr v5.4s, v5.4s, #0x1f\n"
      "sqadd v31.4s, v31.4s, v2.4s\n"
      "sqadd v20.4s, v20.4s, v1.4s\n"
      "and v2.16b, v19.16b, v0.16b\n"
      "and v1.16b, v23.16b, v0.16b\n"
      "sshr v4.4s, v4.4s, #0x1f\n"
      "sshr v3.4s, v3.4s, #0x1f\n"
      "sqadd v21.4s, v21.4s, v7.4s\n"
      "sqadd v22.4s, v22.4s, v6.4s\n"
      "sshr v2.4s, v2.4s, #0x1f\n"
      "sshr v1.4s, v1.4s, #0x1f\n"
      "sqadd v16.4s, v16.4s, v5.4s\n"
      "sqadd v17.4s, v17.4s, v4.4s\n"
      "sqadd v18.4s, v18.4s, v3.4s\n"
      "and v7.16b, v28.16b, v0.16b\n"
      "sqadd v19.4s, v19.4s, v2.4s\n"
      "sqadd v23.4s, v23.4s, v1.4s\n"
      "and v6.16b, v29.16b, v0.16b\n"
      "and v5.16b, v30.16b, v0.16b\n"
      "and v4.16b, v24.16b, v0.16b\n"
      "and v3.16b, v25.16b, v0.16b\n"
      "and v2.16b, v26.16b, v0.16b\n"
      "and v1.16b, v27.16b, v0.16b\n"
      "sshr v7.4s, v7.4s, #0x1f\n"
      "sshr v6.4s, v6.4s, #0x1f\n"
      "sshr v5.4s, v5.4s, #0x1f\n"
      "sshr v4.4s, v4.4s, #0x1f\n"
      "sshr v3.4s, v3.4s, #0x1f\n"
      "sshr v2.4s, v2.4s, #0x1f\n"
      "sshr v1.4s, v1.4s, #0x1f\n"
      "sqadd v28.4s, v28.4s, v7.4s\n"
      "sqadd v29.4s, v29.4s, v6.4s\n"
      "sqadd v30.4s, v30.4s, v5.4s\n"
      "sqadd v24.4s, v24.4s, v4.4s\n"
      "sqadd v25.4s, v25.4s, v3.4s\n"
      "sqadd v26.4s, v26.4s, v2.4s\n"
      "sqadd v27.4s, v27.4s, v1.4s\n"
      "118:"  // Height 4: no shift correction
      "add x21, %x[qp], %[c_offset]\n"
      "srshl v31.4s, v31.4s, v0.4s\n"
      "srshl v20.4s, v20.4s, v0.4s\n"
      "add x20, %x[qp], %[maxval]\n"
      "ld1r { v3.4s }, [x21]\n"
      "ld1r { v2.4s }, [x20]\n"
      "srshl v21.4s, v21.4s, v0.4s\n"
      "srshl v22.4s, v22.4s, v0.4s\n"
      "srshl v16.4s, v16.4s, v0.4s\n"
      "srshl v17.4s, v17.4s, v0.4s\n"
      "add x20, %x[qp], %[minval]\n"
      "cmp x9, #0x10\n"
      "ld1r { v1.4s }, [x20]\n"
      "srshl v18.4s, v18.4s, v0.4s\n"
      "srshl v19.4s, v19.4s, v0.4s\n"
      "srshl v23.4s, v23.4s, v0.4s\n"
      "srshl v28.4s, v28.4s, v0.4s\n"
      "srshl v29.4s, v29.4s, v0.4s\n"
      "srshl v30.4s, v30.4s, v0.4s\n"
      "srshl v24.4s, v24.4s, v0.4s\n"
      "srshl v25.4s, v25.4s, v0.4s\n"
      "srshl v26.4s, v26.4s, v0.4s\n"
      "srshl v27.4s, v27.4s, v0.4s\n"
      "add v31.4s, v31.4s, v3.4s\n"
      "add v20.4s, v20.4s, v3.4s\n"
      "add v21.4s, v21.4s, v3.4s\n"
      "add v22.4s, v22.4s, v3.4s\n"
      "add v16.4s, v16.4s, v3.4s\n"
      "add v17.4s, v17.4s, v3.4s\n"
      "add v18.4s, v18.4s, v3.4s\n"
      "add v19.4s, v19.4s, v3.4s\n"
      "add v23.4s, v23.4s, v3.4s\n"
      "add v28.4s, v28.4s, v3.4s\n"
      "add v29.4s, v29.4s, v3.4s\n"
      "add v30.4s, v30.4s, v3.4s\n"
      "add v24.4s, v24.4s, v3.4s\n"
      "add v25.4s, v25.4s, v3.4s\n"
      "add v26.4s, v26.4s, v3.4s\n"
      "add v27.4s, v27.4s, v3.4s\n"
      "smin v31.4s, v31.4s, v2.4s\n"
      "smin v20.4s, v20.4s, v2.4s\n"
      "smin v21.4s, v21.4s, v2.4s\n"
      "smin v22.4s, v22.4s, v2.4s\n"
      "smin v16.4s, v16.4s, v2.4s\n"
      "smin v17.4s, v17.4s, v2.4s\n"
      "smin v18.4s, v18.4s, v2.4s\n"
      "smin v19.4s, v19.4s, v2.4s\n"
      "smin v23.4s, v23.4s, v2.4s\n"
      "smin v28.4s, v28.4s, v2.4s\n"
      "smin v29.4s, v29.4s, v2.4s\n"
      "smin v30.4s, v30.4s, v2.4s\n"
      "smin v24.4s, v24.4s, v2.4s\n"
      "smin v25.4s, v25.4s, v2.4s\n"
      "smin v26.4s, v26.4s, v2.4s\n"
      "smin v27.4s, v27.4s, v2.4s\n"
      "smax v31.4s, v31.4s, v1.4s\n"
      "smax v20.4s, v20.4s, v1.4s\n"
      "smax v21.4s, v21.4s, v1.4s\n"
      "smax v22.4s, v22.4s, v1.4s\n"
      "smax v16.4s, v16.4s, v1.4s\n"
      "smax v17.4s, v17.4s, v1.4s\n"
      "smax v18.4s, v18.4s, v1.4s\n"
      "smax v19.4s, v19.4s, v1.4s\n"
      "smax v23.4s, v23.4s, v1.4s\n"
      "smax v28.4s, v28.4s, v1.4s\n"
      "smax v29.4s, v29.4s, v1.4s\n"
      "smax v30.4s, v30.4s, v1.4s\n"
      "smax v24.4s, v24.4s, v1.4s\n"
      "smax v25.4s, v25.4s, v1.4s\n"
      "smax v26.4s, v26.4s, v1.4s\n"
      "smax v27.4s, v27.4s, v1.4s\n"
      "uzp1 v31.8h, v31.8h, v20.8h\n"
      "uzp1 v20.8h, v21.8h, v22.8h\n"
      "uzp1 v16.8h, v16.8h, v17.8h\n"
      "uzp1 v19.8h, v18.8h, v19.8h\n"
      "uzp1 v23.8h, v23.8h, v28.8h\n"
      "uzp1 v18.8h, v29.8h, v30.8h\n"
      "uzp1 v24.8h, v24.8h, v25.8h\n"
      "uzp1 v17.8h, v26.8h, v27.8h\n"
      "uzp1 v31.16b, v31.16b, v20.16b\n"
      "uzp1 v16.16b, v16.16b, v19.16b\n"
      "uzp1 v23.16b, v23.16b, v18.16b\n"
      "uzp1 v24.16b, v24.16b, v17.16b\n"
      "bge 127f\n"
      "tbz x9, #3, 122f\n"
      "str d31, [x27], #0x8\n"
      "str d16, [x24], #0x8\n"
      "str d23, [x23], #0x8\n"
      "str d24, [x22], #0x8\n"
      "tbz x9, #2, 120f\n"
      "st1 { v31.s }[2], [x27], #0x4\n"
      "st1 { v16.s }[2], [x24], #0x4\n"
      "st1 { v23.s }[2], [x23], #0x4\n"
      "st1 { v24.s }[2], [x22], #0x4\n"
      "tbz x9, #1, 119f\n"
      "st1 { v31.h }[6], [x27], #0x2\n"
      "st1 { v16.h }[6], [x24], #0x2\n"
      "st1 { v23.h }[6], [x23], #0x2\n"
      "st1 { v24.h }[6], [x22], #0x2\n"
      "tbz x9, #0, 126f\n"
      "st1 { v31.b }[14], [x27]\n"
      "st1 { v16.b }[14], [x24]\n"
      "st1 { v23.b }[14], [x23]\n"
      "st1 { v24.b }[14], [x22]\n"
      "b 126f\n"
      "119:"  // Height 4: Partial direct writeback: partial_1_12
      "tbz x9, #0, 126f\n"
      "st1 { v31.b }[12], [x27]\n"
      "st1 { v16.b }[12], [x24]\n"
      "st1 { v23.b }[12], [x23]\n"
      "st1 { v24.b }[12], [x22]\n"
      "b 126f\n"
      "120:"  // Height 4: Partial direct writeback: partial_2_8
      "tbz x9, #1, 121f\n"
      "st1 { v31.h }[4], [x27], #0x2\n"
      "st1 { v16.h }[4], [x24], #0x2\n"
      "st1 { v23.h }[4], [x23], #0x2\n"
      "st1 { v24.h }[4], [x22], #0x2\n"
      "tbz x9, #0, 126f\n"
      "st1 { v31.b }[10], [x27]\n"
      "st1 { v16.b }[10], [x24]\n"
      "st1 { v23.b }[10], [x23]\n"
      "st1 { v24.b }[10], [x22]\n"
      "b 126f\n"
      "121:"  // Height 4: Partial direct writeback: partial_1_8
      "tbz x9, #0, 126f\n"
      "st1 { v31.b }[8], [x27]\n"
      "st1 { v16.b }[8], [x24]\n"
      "st1 { v23.b }[8], [x23]\n"
      "st1 { v24.b }[8], [x22]\n"
      "b 126f\n"
      "122:"  // Height 4: Partial direct writeback: partial_4_0
      "tbz x9, #2, 124f\n"
      "str s31, [x27], #0x4\n"
      "str s16, [x24], #0x4\n"
      "str s23, [x23], #0x4\n"
      "str s24, [x22], #0x4\n"
      "tbz x9, #1, 123f\n"
      "st1 { v31.h }[2], [x27], #0x2\n"
      "st1 { v16.h }[2], [x24], #0x2\n"
      "st1 { v23.h }[2], [x23], #0x2\n"
      "st1 { v24.h }[2], [x22], #0x2\n"
      "tbz x9, #0, 126f\n"
      "st1 { v31.b }[6], [x27]\n"
      "st1 { v16.b }[6], [x24]\n"
      "st1 { v23.b }[6], [x23]\n"
      "st1 { v24.b }[6], [x22]\n"
      "b 126f\n"
      "123:"  // Height 4: Partial direct writeback: partial_1_4
      "tbz x9, #0, 126f\n"
      "st1 { v31.b }[4], [x27]\n"
      "st1 { v16.b }[4], [x24]\n"
      "st1 { v23.b }[4], [x23]\n"
      "st1 { v24.b }[4], [x22]\n"
      "b 126f\n"
      "124:"  // Height 4: Partial direct writeback: partial_2_0
      "tbz x9, #1, 125f\n"
      "str h31, [x27], #0x2\n"
      "str h16, [x24], #0x2\n"
      "str h23, [x23], #0x2\n"
      "str h24, [x22], #0x2\n"
      "tbz x9, #0, 126f\n"
      "st1 { v31.b }[2], [x27]\n"
      "st1 { v16.b }[2], [x24]\n"
      "st1 { v23.b }[2], [x23]\n"
      "st1 { v24.b }[2], [x22]\n"
      "b 126f\n"
      "125:"  // Height 4: Partial direct writeback: partial_1_0
      "str b31, [x27, #0x0]\n"
      "str b16, [x24, #0x0]\n"
      "str b23, [x23, #0x0]\n"
      "str b24, [x22, #0x0]\n"
      "126:"  // Height 4: Partial direct writeback: Done
      "b 128f\n"
      "127:"  // Height 4: Full writeback
      "str q31, [x27, #0x0]\n"
      "add x27, x27, #0x10\n"
      "str q16, [x24, #0x0]\n"
      "str q23, [x23, #0x0]\n"
      "str q24, [x22, #0x0]\n"
      "128:"  // Height 4: Writeback done
      "subs x9, x9, #0x10\n"
      "bgt 98b\n"
      "subs %x[M], %x[M], #0x4\n"
      "beq 130f\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "tbz %x[flags], #3, 129f\n"
      "add x21, x21, #0x4\n"
      "str x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "b 1b\n"
      "129:"  // Update direct input
      "mov x20, #0x4\n"
      "madd %x[input_ptr], x20, x21, %x[input_ptr]\n"
      "b 1b\n"
      "130:"  // Exit
      : [M] "+&r" (M), [flags] "+&r" (flags), [input_ptr] "+&r" (input_ptr)
      : [args_ptr] "r" (&ka), [b_offset] "I" (offsetof(Requantize32, b_offset)), [c_offset] "I" (offsetof(Requantize32, c_offset)), [col_bias] "r" (col_bias), [maxval] "I" (offsetof(Requantize32, maxval)), [minval] "I" (offsetof(Requantize32, minval)), [offsetof_B_ptr] "I" (offsetof(KernelArgs, B_ptr)), [offsetof_N] "I" (offsetof(KernelArgs, N)), [offsetof_input_initial_col] "I" (offsetof(KernelArgs, input_initial_col)), [offsetof_input_offset] "I" (offsetof(KernelArgs, input_offset)), [offsetof_num_strings] "I" (offsetof(KernelArgs, num_strings)), [offsetof_output_offset] "I" (offsetof(KernelArgs, output_offset)), [offsetof_output_ptr] "I" (offsetof(KernelArgs, output_ptr)), [offsetof_string_lengths] "I" (offsetof(KernelArgs, string_lengths)), [per_layer_mul] "I" (offsetof(Requantize32, per_layer_mul)), [per_layer_right_shift] "I" (offsetof(Requantize32, per_layer_right_shift)), [qp] "r" (qp)
      : "cc", "memory", "v0", "v1", "v2", "v3", "v4", "v5", "v6", "v7", "v8", "v9", "v10", "v11", "v12", "v13", "v14", "v15", "v16", "v17", "v18", "v19", "v20", "v21", "v22", "v23", "v24", "v25", "v26", "v27", "v28", "v29", "v30", "v31", "x9", "x10", "x20", "x21", "x22", "x23", "x24", "x25", "x26", "x27", "x28"
    );
}

} // namespace arm_gemm
#endif // __aarch64__
