// Licensed to the .NET Foundation under one or more agreements.
// The .NET Foundation licenses this file to you under the MIT license.

using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;
using Xunit;

namespace System.Security.Cryptography.Tests
{
    public class Kmac256Tests : KmacTestDriver<Kmac256Tests.Traits, Kmac256>
    {
        public class Traits : IKmacTrait<Kmac256>
        {
            public static Kmac256 Create(byte[] key, byte[] customizationString) =>
                new Kmac256(key, customizationString);

            public static Kmac256 Create(ReadOnlySpan<byte> key, ReadOnlySpan<byte> customizationString) =>
                new Kmac256(key, customizationString);

            public static bool IsSupported => Kmac256.IsSupported;

            public static void AppendData(Kmac256 kmac, byte[] data) => kmac.AppendData(data);
            public static void AppendData(Kmac256 kmac, ReadOnlySpan<byte> data) => kmac.AppendData(data);
            public static byte[] GetHashAndReset(Kmac256 kmac, int outputLength) => kmac.GetHashAndReset(outputLength);
            public static void GetHashAndReset(Kmac256 kmac, Span<byte> destination) => kmac.GetHashAndReset(destination);
            public static byte[] GetCurrentHash(Kmac256 kmac, int outputLength) => kmac.GetCurrentHash(outputLength);
            public static void GetCurrentHash(Kmac256 kmac, Span<byte> destination) => kmac.GetCurrentHash(destination);
            public static Kmac256 Clone(Kmac256 kmac) => kmac.Clone();

            public static byte[] HashData(byte[] key, byte[] source, int outputLength, byte[] customizationString) =>
                Kmac256.HashData(key, source, outputLength, customizationString);

            public static byte[] HashData(ReadOnlySpan<byte> key, ReadOnlySpan<byte> source, int outputLength, ReadOnlySpan<byte> customizationString) =>
                Kmac256.HashData(key, source, outputLength, customizationString);

            public static void HashData(ReadOnlySpan<byte> key, ReadOnlySpan<byte> source, Span<byte> destination, ReadOnlySpan<byte> customizationString) =>
                Kmac256.HashData(key, source, destination, customizationString);

            public static byte[] HashData(byte[] key, Stream source, int outputLength, byte[] customizationString) =>
                Kmac256.HashData(key, source, outputLength, customizationString);

            public static byte[] HashData(ReadOnlySpan<byte> key, Stream source, int outputLength, ReadOnlySpan<byte> customizationString) =>
                Kmac256.HashData(key, source, outputLength, customizationString);

            public static void HashData(ReadOnlySpan<byte> key, Stream source, Span<byte> destination, ReadOnlySpan<byte> customizationString) =>
                Kmac256.HashData(key, source, destination, customizationString);

            public static ValueTask HashDataAsync(
                ReadOnlyMemory<byte> key,
                Stream source,
                Memory<byte> destination,
                ReadOnlyMemory<byte> customizationString,
                CancellationToken cancellationToken)
            {
                return Kmac256.HashDataAsync(key, source, destination, customizationString, cancellationToken);
            }

            public static ValueTask<byte[]> HashDataAsync(
                ReadOnlyMemory<byte> key,
                Stream source,
                int outputLength,
                ReadOnlyMemory<byte> customizationString,
                CancellationToken cancellationToken)
            {
                return Kmac256.HashDataAsync(key, source, outputLength, customizationString, cancellationToken);
            }

            public static ValueTask<byte[]> HashDataAsync(
                byte[] key,
                Stream source,
                int outputLength,
                byte[] customizationString,
                CancellationToken cancellationToken)
            {
                return Kmac256.HashDataAsync(key, source, outputLength, customizationString, cancellationToken);
            }
        }

        protected override IEnumerable<KmacTestVector> TestVectors
        {
            get
            {
                // Generated test vectors.
                // See KmacTestDriver for how they were generated.
                yield return new(Key: "01020304", Msg: "", Custom: "", Mac: "86");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "", Mac: "8f");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "", Mac: "cf");
                yield return new(Key: "01020304", Msg: "", Custom: "0102030405060708", Mac: "8a");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "13");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "31");
                yield return new(Key: "01020304", Msg: "", Custom: "", Mac: "2eb0901e25dd7ffe");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "", Mac: "90323e294af4c67f");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "", Mac: "ade8cea33dcfa4eb");
                yield return new(Key: "01020304", Msg: "", Custom: "0102030405060708", Mac: "01cf774f8d5c2327");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "3d4510d3114a9823");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "d22bfd35e476f437");
                yield return new(Key: "01020304", Msg: "", Custom: "", Mac: "7fab91e9a2d1cad7cf47ea162b687684b48117e0a7bf63d8");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "", Mac: "9e4608e152b614eacde68e7b200e666ecfc05bd9eed12306");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "", Mac: "bcd5b6e528301b92fbb0c1e610213e4c4b28401e27c3f03e");
                yield return new(Key: "01020304", Msg: "", Custom: "0102030405060708", Mac: "e6cde78682ea1496b0666310c9dd5217e7eec21d52294137");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "d623fc7f4d7c999a5f4e904ecbce4731269a99565af79ef9");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "9fc7ceae372e03d3c168dea155cba48a2dce86f2d5ce02fc");
                yield return new(Key: "01020304", Msg: "", Custom: "", Mac: "606c942e5a09a7348da921061cd3b9c8f82603775df55326abf093be344da68a3716e754e3458710");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "", Mac: "a74e756f9344f490713382d697454453b07e6a2c98da4e5d8d4c2205c736ceee3b5ece74a4c4c850");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "", Mac: "b46acaae73f8ce695ebab2e188a828db580cf0c15c9e7aaca20432e8f619c2af19c89f6d881e4e01");
                yield return new(Key: "01020304", Msg: "", Custom: "0102030405060708", Mac: "a95bd7f5274aa7063a3c3ff99fbf5cf5898827c4185268f8d6ed8f98669c21575014bde475b87786");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "1eafd5dfb9736e61f641fd49285861866aaa6cf228357dce42da9dc3b24532b0a4b0db79eff2f7b4");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "56c7f0aad7fa3ee890e66719c99c6a4060b254ca4869193edbf065dbe83e43ec6b41aad8867597b0");
                yield return new(Key: "01020304", Msg: "", Custom: "", Mac: "9f6bf85ce1939320178efbd1d5b5b3bcc09734d22a5c83ecc5a95f2ef7532764540e20b3a4f2a77097b9997acb426835b318d08cfb0a3436");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "", Mac: "9c6ac3f8a57240abc95ea7e1cb1f5d68aa91dff9ce1aef80b53616d0def7faa5690a1c6158e32b2b44ce79a445d42641208d751e3f37c667");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "", Mac: "85e9d110c3ebfd7a20c288bc52fb1031cd4cd1851fff7e83503cccec09e97839cbc3b0b9ce1f118ab67db0a5b2b3c3b9ea25c66beb8d30be");
                yield return new(Key: "01020304", Msg: "", Custom: "0102030405060708", Mac: "5585da74e158ea3fafe70d2ad28a439a184086abfb3a0064e6c5e95a307a896127afa8f2a4094e04032967418926f3fdd75e19705d07cabd");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "415bb3308ae1daad3487a6dd97c61f6ae0f86677531bec7550dd720a97a1e299c2d2b153a3c2769aa70d4c7d59bc74d30c0e420f2881db31");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "686a5b5fd13666da7ba09da1092ca054640eee9359a785d6e611c242b064be398cf69802ed98a90fea18b04738b0ae4b8edb42e06fa8f627");
                yield return new(Key: "01020304", Msg: "", Custom: "", Mac: "2f8bf73e738d664182a7b964fa2f7536573f6ad54efe4168a48cf26f6e92f0dcde9af5fc38653ead5982bde4c937f8f583fb7ee02244e77a1f883cfacb42dd12ea8d852e6c8b71f5");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "", Mac: "fbbaf0305ece36a261bb5f2e93aeb988b491a38c802c310a88a61c41833c30ca171c742d38842677c9865d8569ae2496d7e00b8a1a0dbc95f994d9dc304d038b8a41339a8c298072");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "", Mac: "81881146c53044c39431678a1798f589181d1a3940f0a3ad3a8819ea35379e341cb93974f3ab43b1c6fb30a6344035f28accae25420ddde2e9a9ae987edd1311713ac60cd9f99da2");
                yield return new(Key: "01020304", Msg: "", Custom: "0102030405060708", Mac: "4fd1de00a5400825adf588a48559671bdf1a2040fa0a6dc1442639e30322832c46f093b6f32e6d5e4750febc0c63a8dbdb136e1641b54281eb02e886c424f622985fe36e0a0b055a");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "e502e301169b051a80680b74047dd4c8403389276777123d616f880bcb3b00d91408f7d31b55847e024baf4cd348bfa177b7dffed9db0e50df69ac13573d25e72fcaaaab0e89f6da");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "49f936135e85509e4888a17be6055140919fb08ff73e9624f3e999b023ea11b669fcb5f1db0a118110488dba371b0bc9c57e5a3bdfd4f05ac2a48c1c9d96f74d60081233b1875952");
                yield return new(Key: "01020304", Msg: "", Custom: "", Mac: "205ac59f48faedf8210bbf44924e6e0d01ab82284d0de1ffbc00a63e97c3b6a9754b6b22ec2de0b8d04be8db92aef68f534a429aa6863b395aebce61ac5ebcac6e4b9ac7f9a4b20d45a62cc2c84da4f1e655f12b48132c13");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "", Mac: "e99fae2b89cc04ba4974f753266a664dd0b95796d63aee3dfab7d56cc647bce95e68d87e0d3bcb19ce2cb93fb4c51388ae9b953f90fda872365b11a17ddd40a3c6a46623967ff200cdb6c0d5a43458ccf7db0213df3d206d");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "", Mac: "96232ae8de407d7da623f8e905edef6736f42923c35d79eb67d3bd48f2e45cc1e086c9a296dec58e264a11ac156077b12860ba9128dbd9161ac597738fb3a577787e3245b1e17aab70e69044f3d29980ebf4373a470c14ae");
                yield return new(Key: "01020304", Msg: "", Custom: "0102030405060708", Mac: "b3f1ec17e7914ef3aaeec0d0d8245dea430f880f84b33d86f42c6f48ad90a84c4e6950e37ebcbc23536961011adb37aa1c5ab6d313c394cda09a49e196f349368a380fa2043029506369b625214524112d3c9010c6d482a4");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "2393d1a88443045f65cbd6945ec38a20d4bcbc4f38aa062e0b3e979e6932918951ded15479d59bc8c3b6075dfb3ca8d6c772867d687f4ef659e02b96a035ce9107c885bcc79f47e76546f595738a4b2f58a13cedc6f8bccc");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "7db09616f526e83b90ddb7c43c92c4f554c7a0012f992c3f712690a2ddb1daa07f53673edd7025175eda72fe73ea1539a89bc8114307d719ad3ea0ed802b639f61f4a3fd00be1b0c0f4ade04c919372ea1f4aa5f36cab4ae");
                yield return new(Key: "01020304", Msg: "", Custom: "", Mac: "d955e7476cf75f6791e2ebd45fbe125aa391eb5ce1cec04e3dab8fc0a79ef6de9864d316c6901b1d30428cf21210ed1fec45c565258b1d82159bf3a93e2eea2d0ba514e708cd90d13a9ea0aa129dbe1ad557c2873a819cf4b2bfda22fa7fd84aa4d364b8191a964a");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "", Mac: "d8b500020cee5ad56b9d6a01d68434cb02061ec0fa5818fae54aac81c0af2e75241371411ed60b975101380134d53a70c8168e9954d0174cfcea75c5b08042b31d3c378b7654d63ca33c18092480432141f9b8be6d3a2b7d66a9c89d6c66a159ed449c097f49eca5");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "", Mac: "757347e7aec063c33fbb1272abb8f5c42ad7995d8395dba073fcac86987a28d3d98a9b1a0f8711836cb43cab74e2e75ce6de7e4af5f08db469e8ff3751ddc373aa7b4de553f9ee997314bfcb29f25773a3a79d8ab0f8e897f37a891ce5f7568dfc7058d15e3317ac");
                yield return new(Key: "01020304", Msg: "", Custom: "0102030405060708", Mac: "1fc908b4f6cb1f0b5373a4b886ae8e53033155f775b0c9a8c39a295862bc01356ac2bde34b606d4d1f3187b5cb3fe3767e4fbb09d8a5ecc04df3af36a2178ee56a4bdf6364780f6c3192b36cd0c50851ad2c6d9be53c8b161f462b7656f6bdba01c94fd78ff4c521");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "afeb41357d44182a1442872085fefb44857fab7cb26f450e9dc6359de4817c3466c0b1984e6983987f4e8665b6c362f676a57266eb98e87796f6f908189ef6cffd8e68d35508b9dc5d282e9adf31fdc226b76a32bc90d49af2b2f0e5e12b78dd52b31248985b5730");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "008d8069bdd6d6fb596082d0f2372d47280d334ed02cbbcc88454d3e8b76fe3567161dcf7899edcd4ed6fb0d07f66382d39480efe92b26e9573cee1b299557d470f3811daaacb2ecf828391f5eaf32652ef5f7d1f34d62f03942d194f772e4b525b9636e52fd1b63");
                yield return new(Key: "01020304", Msg: "", Custom: "", Mac: "9d613a9cf32b76eadb34887c65337d133c48eebd3ad7678587b21357c3f7d3350bb58f0e964d552887385521b8c0b37b43b13174e3427324a88b70001c435f13b526954a969e9bd419bfa5139c458e951bfd14cdb9c7b385b02dd15dd5e9ab70e777f219308b3f6f10dab5c1c54129cc6a39368ba6766d05");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "", Mac: "cda5ad1be13c077fec386eb18dd14e448ea4b31a9ae29c442233712dbfca4d64e89eacea9509b3e1760c9dcd05fdb28751163e9f3714ec193ef473d847cda60ffd0d169391fe321b915713e0272deab6ff2d43cd7eb88bc08df038d1b03c9051f8b5eb2bbc0ef7f5dd31df3a174519039d75327a014770b4");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "", Mac: "9b46e291d136f808457e4218a80c618c4b19b9b78e0ccb6948a377d6075797db94a129a15e249dfe98f0796aebe8f4e68e4d957388b039c5d362a3d664457be15567069db9ec8d2116a256ee59c4cc1611ef20118a71dbaf14bf3f020d5151e7ff218c45f18df3a472601c4c0b90210541f618da8cdf3603");
                yield return new(Key: "01020304", Msg: "", Custom: "0102030405060708", Mac: "a1ae85991716c4ca52a7c821a3e0e723038c81ecf0859da08fcfb56c376aebe39772ddb35171a0cfcfdaec3be1ce08eae496f81212e9a63414ff79feb80ef10091e9de23db73dc19239e81225a33802589352649cb06388048acb00004bfa93b0ca40ed51cc268f7380ea9e2f8685b8e0b8b8a23fb59190d");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "1a442a1c92736506e24787202a7eaa4f08f6e9134d1d280b087ee02158755be6b78ab971a59d3e7ff865cc0360238f6c0eacf2c0ac0ecf839849def7ca4b799c5f12141f7c876e40dcc874bf835528b633885188730a3231d6bffd841a6fe2b1a3348fcae7200e1db8dba90bdd3b77a7e2db1bf2928a8299");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "74a2896601a41fb8b7649335d2ec2e76d66cf8012885ca77b0677a1b93d43d85570a5d2b16f5ef65d2a6b31377d3326849ceb03578b71b02a9547de5a5bd7d24413ed96846a5fba5406f34dfffbb3abf36dac8943c118db795c6b8a02d7593d1883dae5c665443764d089026fc6003fbdf057e296a866d11");
                yield return new(Key: "01020304", Msg: "", Custom: "", Mac: "23186f257986f28f0361afefced8568fb98da99f7ceeb6ea769649564ae2dc2d05186fadc7e6e23d1593ffb843d103b803f73b13611f4dcbf0ce6c6146b356dea9c07290990d69b086a100376885d5a0cb43f5e738db53709bbda60a7aaa7a633a4939c926bc22a031ad84da3963b4218396630689cfbcff1f40a3017270122abcd446df450fb788");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "", Mac: "0800e48e4d7e11ef776977cf84bdaaebcad700b75878ffdb81c9724186083bdcf4119cd1dd44689a79bc50573f01074af20ea02ab2f8d21a1c76b3b788736349befbb912ab1b801818c9ba00a038d648a715fac2a216e62e0144f7f7520cbaac90c88c3e1e57fc3966a9e0b8045e96102fa71f079f6241b47a6e106a4588eef9f9097fa76a335eab");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "", Mac: "a2d2853d9c42b2cceafebc9d5e7ed201c9e813c5cf1d65b4d3fb47ba061fce903d6cd25f4b247416d58edad796ee419de0257d43e382622a82ab2d3388f2d16e6d42a66e5d27535e8818ebdff4173172c830ebc67ddf4e1ca87e3698dbab64210bd50c74ab2eacb7ea5c9b29862adc74dfd034c206ca5cbc993a6eb6d3d97c3a1fc3eae64147d3d7");
                yield return new(Key: "01020304", Msg: "", Custom: "0102030405060708", Mac: "ee0615c87d4df70b5177f36da5f52dab0f71a39c386468574be98b9871c7ddba8b68cf6fb80f2a94a01eb382430e583fdee63c2ca0e7241040546a991e83b3ce74e3c8db9d4aaa6a307c782a28723beb1147cacb659fcae1a0f29cf9428746d33c2afd1d2fcaf1b9b69cb20790a56eb421e582fad353dbff1d6a479d1cb6290072832d634f9e6fa0");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "0380a66e291d32bc40d3bcac75c8da5fee869c8991ddee41d7a205d196c80e57b9f1364ca84e2792786faa71edf27ede9712eb9c0f8a9252ba0ac7e79ccef1de52ad7b0dccb43e91b1413d2b4ad990748d563f6c4dba25f014c95ef31c1a0db8ab558f053512ca2236caa86a16c022b560ea94c6aa3e59b576aaf4dca81810f547a36a6dacfa7519");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "14359fad0d72e5142ce044d4a1ac1f144c69f45cda4f83140953f77407ba3d9f928ed6d69493ee61e690d80330da423277c1d40749058689864c043b23301eae7a820d9c12dae1d999cf533cf7cee3c82b108791a7ce2a714343bab1fc7acb015bc91b1b73a06c1a63b81a62327f25faec32ddfdbdeeaa0c4cc414a733c36569a5d9acf88e8072cb");
                yield return new(Key: "01020304", Msg: "", Custom: "", Mac: "69a990d56ca18bac9a215f5355428e9c70ce4c4305489f7e111c435776dba6117e3e2316375afcefafeb2097b79f84ea658e8cf8d93a818af0f43d874740951b1ddd5ccf0e1395ced5b06eb0685b5eeab50570a70307aa4911eee36ef96b6bf8baaf6bfc07e0357fb17d87851cf66a774250b0714582fa72a96ec1a8b066ce237beb6af933d086ca423d01ac3fe7cfd01a9abde71f095715");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "", Mac: "a9ab22bdd6a317c1607b556cf13082143a549fcc3d34a9bc688ed6b3e4a6a8a155ba2a92ab63fa950864430011399ee30fadb8c12aa5cbecc48d710bcba32336016db850e82b5de72709069f882ed08b224b00e8348db79f09877f6422d80a88a4d79420696d94a53f9c76dc0d8c5930fa8fc6b9208276d5776f1fd9f948f8f532f52cb877793cbd52f1df9eb28d155b853504f6ce9ae7ff");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "", Mac: "f8d7944d3d3f25b7373fac85131323817983d92dd98c7798b6faf12e4e8cbf98d44da78726816c31f1dac0f5c062f6949f16dade145ae596e54b31f63e6da1d6ac7100231cc06670c3b34f5205ac7551c908dddbc70f6c1c463b5ce39491e6984b8846e6df48e30dbff064f4b3d1dc71da1b2261e813617518e5f5966bedb00df403878a1f19ecd15499ca4e77ff4fe84e3ede973b64805a");
                yield return new(Key: "01020304", Msg: "", Custom: "0102030405060708", Mac: "257ed4714dd3b5a87f1997abc6d627538b7ed893d60a771d607ef7bd6e3198d9a92b53687ac926a0e50b7b96c681a33ec0b70afa96e3b7da872f7dc7e05be32cdaba0ab1b80640096fdd5963b830d75c2b5485851f5452d392886e843098ee513c4d7bfa88ec85104c1913caff3821dcac0bd425ecee2373803a48cad066a61d5a339307b0d99c2d6272ae4e45ec3cf11659bac60b87311c");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "338e15ca0faaea8ad3651b98ea0aea0a3ca896c71726b9b467e82fb64a35b79b2b40fddf77c334c95cd256f2a7c5e571dcde198efdbccb8c9e07082e3dd110c03bb2bacd87dc4fe1851c9ad4dba34ed14d3d45a95ee00185968c6a21c8a00fc6e66226f85db512937cf17a3ecd2214878c71dc6f0822b35dcc651929421b8a919cacc110ddc639a663ef34e8b3b98171458004e68b9b6837");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "d38f02156df4a718c4b608eb1a3935ebef61c1a158d6cc891ad245c787c848e83be24187a8e7845d16a387aee40a7795b1302eaad532afe4d493be3c22169c2829b9f3b5e68133b7a0fe7fe1a26600edd0985a11a3df33daa4a5094b097b4654217f5ac98af9b09db4d0390e9ffd67bf009b24ca48ae0d1a4d7296bb6f2f7d7951f8545ae782b8913846c301fc32e5ebd52e6c59daf17f38");
                yield return new(Key: "01020304", Msg: "", Custom: "", Mac: "153388473e35c92800feeef2797a1221a8fafb92fb23ea608e1a20c13ac124ae9cf5e8133b79b8a880f972f74532225dd3002c7d714dc53e5307bd583884eacc78ba98b2b53064febd7b7e1e9991849d50742243e522862b0010361e350369561ed69a93fb4e9110d01ac35c9f41e237ae759250e8eeb8226fef004ae0dfe035c9b0e809d2420af7ccf5606d3719ad4e5dc7583767a5e1d19b1cbb4d698d09a22df2bb6bac8d60ec");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "", Mac: "2f75e336da43f4b2680cbf7f61ee025475cae0d4f8e29d1ffabfd9371e44a2fa1d753bd0d3262e75c8e3d0c077bc57f8c616be9b02376d40b13b0d1f799557a5d99e4bb5aa90ccb1ab5e7c7e0dececa70ec5430aac0e7ec84479e01f31f8b5e237dc4d063f3285bbd895b22caa1b6e603138cc7a0304a6decabe1e2dd84be33e5d43ca66b1abeb41e961962025810d3c97a4b06672a6660dd9094e2a9844b064a3318112a82757a6");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "", Mac: "12ee2332b224f99ca6b4b093eb664eb9f838a780159701e6bcf860352c03831fdf0bf6dd06e62ac4a2d2f7a8a1a7b9807d57a2aa701caaf233d0b17333e5525898f127f53822c09765ad656acfbb0905dbb3a75152254be35c1d9e5e3e3f1c2f459f0e9f5c017f2b5e98fc27ef619340fe60f4dd33b8e6b6e5cc9c3d89d97bf2940e491e1ff306f8ac7fc5ec0f169f05f510a544f23bb0f3726d2c3175f6d7583eaf5a994a153a03");
                yield return new(Key: "01020304", Msg: "", Custom: "0102030405060708", Mac: "ab4d1b2760c8f8d7e95cb17b8588411134b9348a8cc59c44a430b448791d254f668e6140e17036149ec1474257fcbdb7a93ed0f7147a400be65b737ce2b8d1276edb516a56d51126c45e1c5785bdff35ad82df5500140e74e58a9d5c1d98765b61cbcab5c5d708d0043472082cb65cb36eb67c113165dc32da357fb75cb8bc1d6d31a0b837fd777f733a5b9afb743171ecdbf0bc72b953d618733fe68d803547ca2d951bdacded5f");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "eb932cd58d38ebd27dead60ba699da31eddbe13fbd27a01d565c153e84cd493077c1a95be24f507141f12ebce847ead30ffbd9b33f1b76353319514937f1c60d6b19f7d8be00fc446f79ec5b8691bd3312a6a71d008154855633270a1ba283f8b40bb11062657be8b3c3165f243a9557637040393397832482465a0e5ebc748ac680ceb16936461b721d03fe220a956e8958d12c81e886307b37b3c71ff260647525a6a795c312f2");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "61a9861fb9f166c5ac18ddca52567e33a18900fe644a60bfc47a7f7dc7af9c546e7bbdf92524bb6095a3a29c05cb00f011b0a6ff315a057594eafc26809cf6c9e15e1c26560ce01161ffe69f45e34d8d6e97998d36c34c38927a6877db74186af23b48d598ba37806784f6897ac69c763f3911bb0cb168ca46fb6b3ad979aeb2393fe745ab978279ea6b469e2d33fbcc81a89c35187c1114b001b380a0b59b67862553a1635c2b16");
                yield return new(Key: "01020304", Msg: "", Custom: "", Mac: "604c6b9c911e6e3bc9176b6c5823207e53327b6f65dbc06faafd2fd2d661e2dae99f1d4e6df7ad3771576348f4146ad35d44c535f028400dc0ea050b989aae5ff4b9865ebc264e1adcf81d11368c33dee6f4076527bf7ab6ae95d2543207dcb47656e43e39e2f30476cc68cf1786a303028f24fae0197ff4499a1c46c09499546e664f2c7817144b1cbae0c890fdb3e1d13dd26027d7413cd7561337d5d65aed2bcf301a65d4922eb21569e5b0a4838a1dc8408dd01eb8e3");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "", Mac: "5bf6e274a995497f532448cb555f5a814528decf88b666ab188eb3de33e376ad7901e3534408b32448c9dc1cdd147e5fecf6dd9814b584d33a3425c57bf64ce0a422bf6f4104f3abcdb1a2ff18faf1b792f12d5827b1dfe30e26dbb056603ea029e6f2e00f4de2f3061f8851f7d7177004917f88be507df8ee61277c92a8e4ae77efef03538f3e021edce8b1ba6dc2ccca075de3654545e9a49fc2749825544ca2c33bd9fda364843a54fe36c72186ea7760f7766b040f9b");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "", Mac: "7c8bf7a2ce63932f2d32c5429e9d42a1a1740b193de39614c744b2ef9847ad3ca9ef7cce11034cdd95951f3885e5647e4f044158f700df77360ca6ebec352611b4ae3c47190702c0aa5d25781dd5f8a67db3ba5edf9c92dc3c7f44b55468805d32dc791de3eda35b2d40f409d487b1fa7075bffedb65f38b7d4ce2315cefa2d4620040456d7790977cb5c6329299422486ebc8d1e50411f7a3a6f1c9830596cfa8e5b8b4acc35afdb527df48aa566293aa9c9a498c8cf803");
                yield return new(Key: "01020304", Msg: "", Custom: "0102030405060708", Mac: "127e17b1c6f6936651df0ccecc248ff6875bdd95f8d46e37f26cb6983dd4c436720387d0221f12acb27903005ab62f67c05418ca0d4a30ab6830e5e9cecf2533b374e507c6c57484bd807f003fda6daa7e31a1bfaa23679f041f8eefd3f3ea3cb58f52d2dfb53a327368ab4eb3f13334a75c38deb27c343b9fa7c76885b410cb9e44d067bc7dfeb376a20f57ad305185a58675826cb11fed8460ba83b3c0566707aeaf28f36a23f3e8c473b8594bd2a74f5647ad4e0fb1e3");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "fd5ad07d9badba2fbc5d78ee3831327afb742a195c48e2f4249b33c88670b3f00d37a9d6833d66e2c2410f5c222648b948eb55a6e3abd06e18cc3ee162e44d7525f5f6cd4e9b0174691ec926ce1460f334088e02fd51b30c3f269e7188c1f18f5eba3f1deb9b62633681f823c91365602dd7c998e68cda4a2674438ad20aa7b6ab7e618f32b42f5837041919eb1a616aa70a1d327b66a324a7d17ea7bd16a3d7d95a2e3fbf21ac158b56edd7195fd5df27d51a83693e9adc");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "e5ffd753f0a70872efe916504e7e1c612148690b1dfc2eb37c7441a310b65035a531a38c9bcbfdaeebb3a0a65e48c06fd80f55ba310d414c18191fcf6ce080d75977f076e2fc4a23d42e3909d80d8889a50800b5b9b2eef703f8b9f12dfba24be960725855c00e65eddd29f1ce0c69adcfe9e9855847f4c936e898f194d02bf2245e4507572eaf81485eaa9fb84191a0a2ab9eb7fde1aedc849992d66d04d25f1416b540dfce2b1a8d6be391092f0c7f32eda246bd952519");
                yield return new(Key: "01020304", Msg: "", Custom: "", Mac: "23830063eadf9ea24dcd879e546333244017dfba5eeb7af77d7b8f51530319f632d1fc0274dcf0a786c14947bd3b15a1f2a21af4adf31d6fba056d070d6bea965649d13e4a0eefd6c95b9b4ad28a82845fa7148691c67457b7fb18b8a06cbab270e1b9dcb4d889a57d022abe1cfa9947aa140b35c1c429e83ab86039c0e241d2b4fff7be4c7957a9ac9e4dcb152a08d6a9a71007aa6c9078d9d6e9a9af05c7497f3d503f63a26ba49291390bf1e5c3c261ca8e462370c3479d0b61e49a26d041df66f75fb6e92d1d");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "", Mac: "1a7cba7fc3c1e81493449c652c76f42022386e998bb3ab74b4987563f9a36d9423f2a7b6b9233c9c655fb78f76ae7879eaf16d4f6815c182427e74c3ab8e980d3795df585921479b31fe172bcc889b4d7133f7e5439278ca0b3d54b02dabd7efa34a6a495cd602be23349f91115190f15809eeb10baf6ed92849385ceb8f3f2fdde8f70ba348a2b95f435ae8a75f97a80b8e4242e178a3d86a91dfe347dfbe0d974ef0962ffa44ac1d51004321447a6b412b1eb89560e2c8b7d5a2b2f59741f2d61023bcd5d693b9");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "", Mac: "372e62910ff1c6a0e0eed2fa05fde84ca9638182270398dec252e6970f58afe8c21ee0e79c0a82381e9c82e1e5af482577cd5f346a8c74aa152962173cb91533732400dbaffdc1bb6f118ec8ac9f35e8a9e601c011478f60756a7827aefa88211f311ad554dc9a8205d3860c7f886ae7fe42ce7fb865c16b097280835d80d4f91dea41ae5d034381ac879f34bd45e1789226dc2d6566c40ac47ad36910423841ebbe75b9bdcb7917e4941f7a88687deab9e1d82711f7c3010a0267488acea15c07b44af8baa83917");
                yield return new(Key: "01020304", Msg: "", Custom: "0102030405060708", Mac: "e8c03dd991077566dab63959fe2f8d2873c15fbf3746d78e9449741104db5fcb56ac9964f4f670d9cd2d5ecc24998b53273e80c9fd1cd35a99705a10917ca4edbee325b19d31ae7ca36cd8cb142b02b6b51ce9cf2ae6a59202bd88a50422ffa230370be28721577aacb3fcd9613ec058d2e1255930e5735709fc0abc2e937a9c35b2e5ec31f3ec6089b488a760431d9d355ff2bb7098ad3594c8573ac567cf62f842196fce0653e73bafea2c3e6d989f271329e6c7ebeac7d3e34a221c08b325f574d0810b211cdc");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "d8e814f7480dca8b19f11802391877a7f5e5a3dfe0a8c4da52076a81bf60f12cbc89bc8ba25ea3d83667e2fef70b8370dab536a3303ea9a9216fb0bcc66741eb5888aae2e59ed94c5a85927bc0483f3315b87bed9686deb7f1bf11dddbf5009aedba4c81a357ba51914ec6ccff01c352663b33e02d837099937758465830bbd493a114845e37e61868501eb18716965186e76d813c8f9fba485b4cdc2184344326614caba410a65bed1de432e3a3da0acc562be6dd2b5dd13de37971634bfef590f814778e6f2166");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "2322051a9ba76064bf6a9a7481cf5f5fa470ea3efc09b1a8da25febbc6af6a8beb6a5491eb2c9dd10ff6c7c91ec97ae1d9fe4c3efcb83262b9814ce5bf229b32a5f35563ae7cfca14934f4050c408e1e01fc718b27ceaf8b533c2e87e077c3f92f53e3a2239cfd271db54f5bb862d6c6ab4e1d5b87c4c82478b96387a6e9b9cf07ec08c65f5b3fb43d1809a0f2ebca3b0f85a5f3ac45cda39ec348c61f49e6d97f0d4a940952edffcfc36313c1e7bc6857578e1e652db2a8ce2f7544fc24e3a97b69e40a8d8e5f51");
                yield return new(Key: "01020304", Msg: "", Custom: "", Mac: "bfe26a7405f6ff082ac80aceee4cdece7a50ac5608783a52ca14397bbb4f5b7167999f4228f9882c5a24ea0f04655a835a34e2d898101dd864bae2e24b761b26ca903078dea1c89b45ce6b157edfcc4632697f16c7cfa2667739d6a464467395aa217f8b7c7818f5ae69edcc8a82c419c7595bcce85d86a150d5b1b57be2835baf602ac7f8f44476063642758596516f49c8ccb2ffb80a5a754532050777d36dd2a3cb2889da08e87da24963459aeda02b10d70863de8c02b085e36c8173eb4a0caa6cdbfd5d0fff92f5ca8089eb3537faa7b3fd64490273");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "", Mac: "399d73c8fac7ee1b6e849a68486a80051fae64e962ddaebbed9a688fb1e3b7629b757852fdf13d360dd6866443277c5acf6713ffc9e9cccd4df3c21bd043a048aceb8115ef814f0002000ddfbe7948177b8c3f67fc34458357fedfc43ecaccf795309de19cdf454dbb2c1268d255b8a8cc8d1ca0421202a5bc503be1eb6806dad07080a8ff30ce2b6c1f7f6565154b3c91749ac5aeba915a5a20eeab5738f815c36c80e18a1da6139c56d1df7963cf40744e4a74626992d69d18e2d2bc7657d951e519576d4d3ae012a4757e5a29d93100af8a3a960a5a52");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "", Mac: "ec58c5e2f7d5187880167fa95b3558f23961a49c28dc560759335d7fbe13845e88f0b6691c4e12e81ad61f311eb3c151c3460be24aa0a0a8a9847662d0b551f5c2bd9ba9ab82b6be0211d68ebee1033d8e4aabacd133113785a911f977c2836fdeb81655a2a24364a08bf5d1102c3b44d8522fe8955a04126788056e50b357ef0d795c7a790e8856dd6c5f9b33a0df88d68ff61495d908f8c62c5f2db2c492c0716ca46f8f726e965c88dd0e9b6d62b8317973d02e0f4c075c75595519d9212e8e4614cb36b6f75ca3720a56d674e81189218ef0f4bbbf51");
                yield return new(Key: "01020304", Msg: "", Custom: "0102030405060708", Mac: "ec0c26224cde85fc81793e1726cfcd6bfe7dded30d33318c71244632fdc767d41655e8ad471dc8699ad00c41f6106daf623bf57c00130d59e7ceea036c1e7ab97aea6eac64691a1f96f84eb8e96eb40a568c8eb8deaa3cec1285f0a5fbea70b6290ac3f904a7f2040d771483c66714d86409538209aee6ad2d4264d3517700a7f8edfb9b6ed218a313ff5af01a1ed17f709177d80d6175586de534a057af285e1223efaf8c30a721b898ba2bd0c4d203230a33ffcfc5039dc15186c9af9ccb977b1fe66053823fa1a53a7f61f37471e23bcd8385d7d8cede");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "2ee20981b987daaa1ca31ec635df02808f177115334c2d68fa83651f76e718577f400fc170a8e622079e29612f405f63495ace8c7e993e38529ca65b04d682876b7de35a16fc7a430787b8a09a051aa9190bb0da9064e0b46a278e517a7896abb72b7f0bdd1ce78246cae2f89a5aa04fd59c1a97674dffbc5651e8dcbf79579c07e245e08a5b2b56720197db81e4ac2a3a1665fbd4fe1536f5d7c2555f4b5c0ab1b4065616f67ba17cada4e18e066bbd9f7c5851537363d02395d0bea787b0cf0955a4f030284ee9d4af98f3e7a4e6bc1f6b9cf1d03effe4");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "bea2772de5afbbdddcde258d6fd562b21b65a366d348b3a2d3dc19a4627216daa4b626722506d68298ede7198204dcab72cf123ba0ae3c5c96add95dbc2d1ef739e46e26d8417f97d5ee57ead699763c1aecb2649ef64bd5f857e9151ae6e596de8f9fcd556a4409f78c85ba000c3e423c1b1be2f8b61d85a397bc1f688c9c4df032c930a0ee6ad2c535e0894f7d0ebcd70ef6d120644a30e8be8f4c75b936dfb6cea3bf2672e9eb2c69ec360d8d35807538488754f5a0041fa1d90dc0d7967d9b21bc09635bc4871f3eb59f72d2dad7205972e41d6d6cf0");
                yield return new(Key: "01020304", Msg: "", Custom: "", Mac: "86bc6fe1be3d40769101ba6bfebe1b9d2fac03f7fa0896e08ff9d74231db62886ee3eabc333eac34c631c5d63ea0dc3652e0516464248d299f216a226d58f711a444b2869646d26b4040bd6ffc2caba0368779c168f2affebdc081e789f5c080e175501aead19aa46bd179a7856e95f25c4905ce564d6dfe3eb26778431102bfa99e51622a6550b9a3c2a79603ca4120e9b7978d908e21170583977ba5afdab8c954287861a0c9cab12dd3a096ac31047e4f173ef68f2fd330f74066d3160737bebae9a2b97b63c4c736531e0498d7ee164ce081c08731d305e7cca5351ccd8a33a2acc8c7a9dc49");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "", Mac: "e4158effb70757dc2c308daeb1fed1cf4ce52587d1decfdec07434f247a26dfb8bd1926a7d08d7b2cf0e42597ca728e59c09687e80869e23820440e55dfc0ee7caa6d23d25d55ced23988b8b360802f6f132852db4fc5e2c71cd1c8f5ac5c21a8ffcf5acdcb4589b0e1c5e473f1514b4ff2c4b84ba72cc7d114410633e4678873bcde3739257adceec95850f0c1f129a6a9a8c26e820176b6ed8e63aa118ae73d88a8b45aa1df6325b80956a483c76893451d3a3eaf746cc1f903304765d5777e3650bffb1755b80eda06b03734af4e836e658de1df00ddfd2dee8b688ee2c245ceeef62af1dbcd4");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "", Mac: "6495f23bb676ee6073bc34cc900da1bf3ef4a2b0902bfe0bf0e6976ab27015223160bc67de95eb24f32968eb8f5fddd5b1e5c912979e9966285002b33c50621ca8c9ddd09321c3c91e6436dfcbec2cbf7710fb94064cef1d09dc8ab6d924418b9c3a41dd053a5f5622e57053e3610cf9a718f5891c8f80ec8847222b916ec232bcdbd2c2bf7c2f337f848e73051916023bbc2f9d85d1ae504ad4b172bbccacd6596558d6f411634170cdefca6e0d9ab72e7ef1ad2a2e0db6e90030a08be0d98f35244c5c6eb010a010efeb8ffacc3df0738202cf95c475f8985ea8fe3a7fa3809a3b65d71647fa18");
                yield return new(Key: "01020304", Msg: "", Custom: "0102030405060708", Mac: "db129af5634d0558d76278a04bdc71e364a5b125cd870ec858dce61b592e215c54842701a7b44237aa744c6379840dd56e9a49858e05467142340550a0824b5b7799e2083886b5ccf0a0132e2c41031c2051b1ad3178d8279d3a8b8127665598d9626b45439fe619122f372642a7ee02b054f9c3be108423a0d14f4b2d9acd931a39c3eb4f364b8b391deebf6047c2e357526ffb7d430237a2ea01dcd4079ab2e6a51925b3d04a6cfdf007c1bb6502e8471c7a6ee2576abe58cf05a9463373b11b76dcdf9d00801c64c596d573d9d4d08b228945db97a9462fc0e80eee2b2d7beaa071df8ca54a57");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "5dfa581a9e5636d6cc582fa81be9336a104dfaa66e015ffee8df78f9c32fec7d95b7340f1f2e02eab85d5857eecdbfa81758d607800437f5c246ec55a8b1737b831a6235b0160699c831164350afb331f7204c9ca3c0df03a0e60459596e150438df2fa253a5e33cb84f39e6df7f005efb1609720163a22878cca22edb68e3f9f49dbef8a9f5260bd155fabd4790fa29638adf78ce6b15139f218ac69ac20eb366409960ca43fd40846c1bd799ba14ed50deced8209dfa474f37adb24d921e78711c88ab65c34c5dd51958b41a8b49377ff6bbf2310569be31cf6a07a46bd9f35bd20e14f04851fb");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "ac99a024f3a250ad7fa6fbc94f5cfa31965cd10c0132af48ec5651f73e6ba24010667acdea4c62c38c50da90016e387f88fd69491e463be916374d8e54188b916ca1fca54cf0506c3f48ecccb1f2d3fd80d414f0972b6e607879363cfea48ce44e71fe3559e697cc5ec9e32beede4de00112e4ddbec2a8e6dc92d59f2b52719303e79a3b95f38650b3114a9093c4aef4aa387d3c49b6b2e2e63c13bfe72791ee72c484bb33daff31a7b8b88006f3633cfc18b8c0d8f26b96d8a3528cbe73ce36243449e367289dbdcb9e55588eaca10c07b108c1a0ad392771caa01c0e8e6ebd68171348debc36fb");
                yield return new(Key: "01020304", Msg: "", Custom: "", Mac: "2adb4e7e1ada8f538a8734af88066361bc2dc94e3113b1acd3fb37ff3bddb2eb77f09a192196f16a60173be47af79ff02d3a4069e003d181c274a23fbdd056f01fc59f95a94e8130dd610f43df3e3766d6dbf2bcd2ce493bd03eda76a08d6f9f047cbf540f571bd8c7bd72945a37830e842b397677bfcdba060bc31320e2f7c0f9e46297c382c6c7cabc1a8baedf40597e1023f5a7259cf5b5bb587d2d9277002ee77d93108cee75913cb23d021cf9c624c754fca8a82a201f0a3eea72d41c8723c6a16d4098cb4839aa02825bec3b6eef31bec497346d66b21f20a22a423aaaa3df9c238cbc1226ffd6a7e9e9b8a349655f82e83bd0494e");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "", Mac: "f9de6d0349bf4691eab471432035185e5afff507abb187685494ada4d1ff6bbcbf52011f59fee5717f5dd2803fd30be6c5b99501f7b3185b98830c7134b031c1750f60df856c5258f5499b7cd3b8a37083ca45069aefda5623697eb385a6176a2c279eb52c26594b83bd47e4c91c6880e5f144e59edf04a32b7c5fade8a944a944c0d3182f2da123236a9e710d1023441c8006b15f0ae3d989320274047f628bb715dc0bdfc7718056e4278bad9bb269e5e103ee457d5950822e84c3163659a02e61e52a0383e71d65d2ab3816f60418f086dc61684a165f1ced411bbdd6c45fa163f25fb0cb4f25f52e4344a6bf5fc9d053ca53f24ddeba");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "", Mac: "e5a3363a0fdc8bd9c8bdf3d5d5813f830948099c87f8ec476f8e8806a1a93a578158a49c3185c239c7f71447c0b54fdfb536963137804e6a05730d8c89c194ed8199ddaf33660c749082dadac7a5a26418d21fb31a8db89812143884d3c7aee22ed257973a1b407853eecb7ef30f23dfdd0daf40098c1b6ad9cf7aba0766bb52efd9fea9f1f01205ffef8acd797009088183070553a515015384f33c35e45d4e070fd17544ca641dd44f3d68efba8c6772fb3f8cf28cb7b7739f38d97a4be0c6f3b386ff0976f9386168d3a306548ddedca9433014c6d39ecc25dbe54bf3a8b4e502200fffa67105d66adce875533a5f7a66452f151ba08f");
                yield return new(Key: "01020304", Msg: "", Custom: "0102030405060708", Mac: "545e1ffdd5f608ef7748f5137ea0c9cf3726aed55db90d42edf357836f2066e49b79f9c8d946ac7c26e995362ef7719bffb1b3e3ca4fafc4c6bfefb1c27077f5ffceecc15b34c2eb5251be81e036c1bea2400172234e2bdf72a83c5e87883e7fbb8be42b912f02748c075c80fcf669dc17ce5298356a2b14bd362bd9d4e0b642dbaa6e97b53df0d020b782218cec3ae2190aea79e4685f416f9701f3d15a397dfa77ad6cef96b7794ed3c81a348c46f0a9485ea655a3931c10661c0548703fe0310834a051bc9ce1e59fc27c6cc17c4ae24c8043cff8b6c7eb981fe3d733ae17e8d2e165e09d3ad2d775b1302316063b8aeaddc49c5d3c91");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "fd736667e30bb9c99c6ed0e8072407ea12fb5031894ed3d98014f6331bca19fbf13bc9ec46dc4053aa4a2fc602b2cbc221883e4aed377dd06f1160ce571f013e2add94e591ecab53b1a8d360f6560dcb85b36bd98afc7804e244c7c2a5e6863068c87f86855be0083f5f33222daebd029f2c5fdf444ace1e7abee28726187339dcc0b16cbe8c6fa26f6eeb5352fb397b3bc1887d543dcd867241d67a26e589fbd02e1ea53091059be8f28f9437044dd7f68c43f02b9c9e52c20d85936831a5d93f23a30e6d91058354e36c1851c07dfdb8ec8103d7f82d7b001044ccc5d5cfea66c47b06e679d7759661c75605031011d8958163a4dbbc4c");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "b2fd1f0c8b9f358393190a28834c441f21dfa7d14685a79f25f6ad26c962f76ebfacff6df7392377c3560663fbe4bc56a4a642b1d80ffec12c251685c8c701ce657253df08f076f7617a52da61ee6ac210bddd5fc2838654f3c6718e47f1fdf96d057d171654dec971bfdab506e1e674d4ee01989ede548d84c81683713053aa50ff793b5ecbfc0c63753c5270171c8acc5cc08b8320eabd2b645b14640ee535658c955e8678d0a38e805e476e7d0726c6f0c9feb13858c9eca94b9161c5c597b8473b1e49f4d2686d612b03ecfcf6d339ae99f365c56e92f12acd751eb2086bef29c54d2b6138c7b5816be5ce88221d8d7e46a18b434bbe");
                yield return new(Key: "01020304", Msg: "", Custom: "", Mac: "9b278fbfd374bab6f8d526d31180937e676e64530e849ff8650cd2c8074c983dcae589f667a0a8abc333a95bf3911cb3ff68d9c5da88b182b63321f726a1f00ae49f5f5d34763bc7244b684431e1b66d713e7341f9ca0660fef114bb11655c6526bed9b776d33857609c68197e94669f5dada81df8bbfda028f014d8d68dc6d36fa8b0c8af2e57b8808f72e01a3bb6f28ce37d819fa665bae1bc8376ec9f1747263946436ce5bcb04fa612eb09bcdb46132ba70b67294d15ee8250a157327d3d2144ebc3163706d45fbc436fcb793458609085fc9b1dadb481b4a8f91e2b1b63d4cc77260cadf125057c1a13cab109e678a69eb3e1acbde723ef65ddd118041b95978e9ed7a454d0");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "", Mac: "e742d5df46dedd01d055425867319399e8363d858d0943101d13199e68612e1d2b9ff3453baa231f7d8cb4071475082d514c8fad747953f48b2d9cd131cb65859cf7cdfb9cc3f750baa1f8495d4e064954cefda37b8465e04a0f111828b90b67e5a8e616f15d37895a6e64a03cf4dee03fa14d1749e350bba31771aae0204fca839dc072b75354daabf1387d5983f0c48dd419aa72b64d3aa4e977f6d87324f6dee5868cda6016dc25f4d79874fa36978c683bc058047d1132e32a1047d6b79373bbd3595a2cb33994cc2b710166f868b5cfc7b2b0d0c5b819d857d3ac127bff1dd7b73398c9357b72d7be0d8469d0e7c0fdd69117c2ed1fd3aae4a8552040a276fe47a6b0d4f226");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "", Mac: "55baf66d361ab2fbeca56afa17192b276e39c60ee784411c57a31da07c640bd27741af196a783d7e890087cc87a96f3f46608d3270741d32ba5fcc37cbaf2a82cf641fe38c8b5b37a8af3fb245fc5437a59f05d9020822d8f9fb23e087ffbf55ef2c3c2ecedc7d98ad559078ea3a2701f18f0bda3ffd2fac17a7d426a640eaf5fddb9f92be03bad9b536d536d00ad54f9a03fe6a9e662f34839c2edd4a10bfa95f63f1df33ee73e01e3dde79e66a97a128c7b2faa9a722508b5145c9aa40d2710e1ca16958cf62e0e0c37888d1e0581dcbb0c207e2aeb1628f525c39b46cf610ce3f22c8b7e5466145979068edd521e1a47d92ad46334696d8df2317e97313b856b3bfe76345e526");
                yield return new(Key: "01020304", Msg: "", Custom: "0102030405060708", Mac: "9848510b81ee3297e3bec36a3e88657deda7105faef5e2293ad9b245ca68c53a94ffa35738df2e3b71f928b5963becc57c69546f5b60b041c946f2e6897caf4a02635a982662836b4fbeb46ebb43fde02f81a5442ea47f7322aef733b14915d8492d9f69279f33ee5290d0f5bdcf72144198d67aaa2deba592c511bd0cf30b8a35bf9ebd0d203380f436d40e92eea097c35098b767abab0d9a9aeca428b53acaeb568b4d714aa121fd963bfafd4a6d93ad8849ecd32af005d44cc5f8f80b9bf0d5c3436ef65bc206db9ccb75ae7efe509496b055be44e1a378751127d75aecd802d99cccd18bf36cffcb2ba7e2768eee3bb0268734cbc6fb477651151024569694b962712ec387b5");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "b59d218fc19c836fb466ef73e8d07d8e4f1ae0de7a42d8b2501f08d7b798e59937df977c0d1301c4d5a6b755098308e1ab8ae83312f73176259a39ac332ad12840450bcc75278c76a9b09bb26fa5b64b6ecac03dfc1858d81496e4e26e04844ee349175416ad4c03ad9ce030704c052bd05bebaecde5b21087ca1be7cd58f2de309a62bab1d30611621533b671eb6f9fd411b7b5f8e3569abf240857dddcdd6732cd6617059bb501d23e99bd7c182d6b3d51e5283187ee902abd852382839153738129b4329d9dc02aa0563f0e6e797c36bbaed22ed2961af043fe12072da2497aebb1bb0ededfd1dcfa569bca47cad56fa06edcfa5bff0ec253dbcaf8cdadf9c672ec0b06c53d45");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "c318b8493040729d09550b102d0f736a4dc8be57bb22d3ff860e65c5ebaf6dac15faa07487ff9a653a140ae67926dc538027bc6e5c956a41d666dd9c60a5cb901e1506fd8f55976bbae567721540acdc8fa8976a9abcff57e27221c0ffdb25f2c7370c1a0afffa31940d88340f5001ebf71ae0ddd98469c377ceb222d2f25877b06491ba2e96bd9347f36f53676ddb30830b85cd836af43e5c2952aa90671058de63fb746e18fb4e6deb90acb9bdf5157b264865fa4f6881f27e085575b59cd9c2f399e2bc5b18f56ec40e19b8d6b43d8b54370c8020e9d37ea761f76447f711d630c94751759ace2334333ae8a7e4ca2e74243085cfba2fa0698ba5a30f33ac5ff8e1018aa07e14");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "", Mac: "7c");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "", Mac: "4b");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "", Mac: "7a");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "0102030405060708", Mac: "d0");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "41");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "e1");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "", Mac: "49b075d8262a0eb0");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "", Mac: "cb382c1dffd054c6");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "", Mac: "7a45d3545743bfb6");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "0102030405060708", Mac: "2022be9bf489fc30");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "0a420726becf8c8e");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "416a69715288b58b");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "", Mac: "7d0b8896f867e5d658c8a13717272df5b763b8a49428fae5");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "", Mac: "e3a0e00fb689274f2a66888f2d85c15fb4c98d5bb3207123");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "", Mac: "3bdc8ac74c8c56faecdb55dab401b752d369e4a631d48949");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "0102030405060708", Mac: "dc0fc25110ff3b275f9b4176d42b93a2614a74fea94edc50");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "3f6faf2e8f1010d6825219304a25e9d79571530a2d0607a7");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "b270fe031da1f75c4eb2108b57ff2ad9e53791fd884a5218");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "", Mac: "e8d5255f96bffef22a43b08ef160735e666c2fc3b066a2617eaab0795da0ffb4043229822842c629");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "", Mac: "848efb371e83c15255deb95cd71c537645d54ddb90445825b772b66a93752f0cd79ce8eb8d20ae1e");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "", Mac: "c5b844129fe31552d1aa0ba702a56d24e706052f93e5ed01b0671fb5e818aeb0f399c125a85024f4");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "0102030405060708", Mac: "8a8d271c5f195c7dc82aa6926668523f94454519a0ddad86ed1441527aec297f2ad068f20312a650");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "76e917d0b1ce20b216d7996809cc8d4b9a4f7e032eab5b6fddae8e67bfaf040e2bfc5ac05da0c772");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "6b6c2aed209f743c1d4b84b66bb660b4bea9892e55c6d31d73efd4516e0be7c29cd88853f29698a1");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "", Mac: "f8891b3069f1f16b03563ff036ea703d88648499dbd58fc816cc23b670bd816920f55854417b72e199c29d9b27b97f7b4f432658e58c41f4");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "", Mac: "43ae326086f261827308b3cad2ea1cba09095cb041b08e28517dfbccb3e4c5d06668e0aa4fd428eac9b6f22e7ea7dac0132f5722d5956ca5");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "", Mac: "8b283e9dcd29384cef42095db498fbf3f7093a1222276f36c156ec207ea86c823fbe0beeed448c10c06321e1e3a10467fead2b57d2111eb9");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "0102030405060708", Mac: "be2337271eeaaee0238753786ab541042b8c9e71938ad8f71d7f9cd78fd7f1b2acff363a6e1dd13f73cb25b3d3f7820fa125dd6ae46099f3");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "ebe3dab1aa98f70f9fc529db075bde3f5b63e2aa6ac4d28dc222144ec6264536bad199dbddfce328fadd251529ebaac0defc7711f1b562b1");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "73725041b05d204196f66b87498001bbb8eb1f331ceef297937018c0261c9529e5dbf9ba2cc0177caf078f5d5e918beccc60228f6b519747");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "", Mac: "f3229f73d2e8928f8d682911fbfad775c7092d35cbb886e73fbfbd5384ad070fc1c01fb4afcb5c2690d9ff19b4283e83f1b477d734f80636155668767bdc06e667a613d83c89f766");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "", Mac: "7b4d3ca452e19d2e135d8865eaababe062fc64869cdc3a54f7827af5734c6f944c41e41bed63bcc3e9125e2d7507cea8c90168af27ba24039cd1964e781b1cf539bd2a5bb145e2d5");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "", Mac: "f50f72cc748bc121148bd5a93007a80c653159eeb6e6900231fba1e31b32f42d6cb2992db46ebb26f6fdb4bee5babadb183d98d36c869e866982dcd719074ce6960ed85810d6e84c");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "0102030405060708", Mac: "0dc1d2d9ea80052812860293f4121c9850c0c543dd4c496d2260ad815665fb5d60c81020f8b67454a6434a40ac2df43366f612cdc678fdd9d4cbf06ce53a44625ab997ee8dc71386");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "87ef88d96ccd49b25025ebac7b732c297059db4d1a45d2197e752fed025ee61089b5d7c0a1618a86db8c8414942ef42107add7e25a158ef26ec9607de017cb496100915be074b4ac");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "bdd482a987ffe96b5b57f362be6f34c4fbe3de533f818b07c12b648003664e0425ab7de6a26f70e7ee727c53cb2d286f3fd14cfcc845b60cb83d3796909f7b65c75e991794f7e84c");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "", Mac: "519fcb478e679e1776f3103eb850efb12381d1620705235cdcf3cae601bb651a456ca01eb9bc24d10ffbaef59a9fb0367c8376ad3112780f75139e2b0064bf04c9a1fe51f0052537d7f4c230057cbba29a3df562b5caa34c");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "", Mac: "c03a04a88e3583ba7bcecac348f2d9abd04b36cd13c166617ebcfc90126df72b0ac09567f8aa52e5419c0c610f9e318075716652cd602c2b6d4fd913a3e5aa613fe1516ba1960dd7231b103402d46d9e133c61de55113c51");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "", Mac: "e2866c708a8893270c0a8a82a7fe9ee8bfd7ac03e34a8e237130d39a6cea257520102f89c219a02fd3847f4e485bb007b76f82fd129ab7297f9462146f53e3d2a8465cf680fba39e5be014a0a8a0d9d4d593f5dafa363cf7");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "0102030405060708", Mac: "4ccf8bfd6690ff62f0fb0461ae0da4c5794cdf97361956c3d7d477a2907c6fdcffc7c4f10e4b4fb6db1c64d3945157750e58a8710ee3aafcbb55ff062fa257aa7c3b8fa26058bce6ae5effe4d479676c191b691501b441ba");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "763ab220c309fe1e260f1a873df345cc92c7703fa86465fe6e81e27c66327571cf8e3502f5717b568f6069970a7315492a71400d1ec7d33a65d6b41ab0a247c61bfc527b86a64743261a06ce7bad58ef58b3bb08b7db85db");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "64506d4bcb1b47ce5e5fa272c3f78968c8c93006b862729cfb12f16fe72ac500af3bc5f4de101572a5aff4b95bbc53122b9352edcd93ddee79ade1798b518468e36238e1f5ff23e845c395d0ce583b8cd9ade46c3dd45c55");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "", Mac: "f9ee06f6730352ef0a979f4cdfea241fb1dead91908f28f6d3c83d44b5c998717de73609430fd2559444ce353ba755e54d53daa8f9400997ee4734fa6d9267b9617f8cffea6adc92241112b0a22416a990d07e90a24e530448026778c54ebe7ee1800e0a5d23d7ad");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "", Mac: "b932544e2ecee155815af8b8042472e522eb19148740e12b40975305c15c971d91b9d30d39ca3d95ca40b29b79b77729fbbfcf4a86f4750fcc1e7919d6c25b1a92a20fac934755243f628c52e64e692bf38106f065eef92a7cd447df8c88bbc9b2f219ba907ada18");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "", Mac: "334d9f0290c8ebef8f6f62ed4e323fb5cab2200689a6fe2b8f427fd15eeaa2842bc80c0443b3a09a562209faecb406e07892181312dfdd764e697278c53560578bec82e8ba69a6a121f90746e1b07bd67dc03495c9a458f0d289608f8a261c4f3da9e61028c8e56d");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "0102030405060708", Mac: "fafd3030cd6361d99c854dacfe5afa3aba0a2994e8cc59b25147519c4fa41f1b1683fbc4143f499a37e99bfab07c975dc065d34974c88181d845576709f4f2d13373c2c6350208a3bab383ed1b33b512585c575cab025e4dc4e18be060137c9d880ae7e827a74f3e");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "90856f33dc04abc21e94a8b3bfa5aa66d063031fe4cdb80cdcd65396795b69f6a1c0fc575442b44e3468b63a21d161d381103bd399d2bbadf10b2710f3edf2e4323bcd98b479020b9c046d5956c1f43d0229311076d9f3ca838352de078f8062081b9910469ae564");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "86729a441884b52619e9ca3cd9177304379b545b7a8efef2d1e87f3497d26ce597b8e824e5d984dbffd8876766434e8dc8aac863ab336d30b9b09c2f44147cdd1b685b63b508ae34530b539fd447e28ef8f668d34645581d697ba643547b5971e118094bd0071494");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "", Mac: "18b428cbc4075d5fb172f5e1f374585aeda3d6b6ce7d2cf60b17e92afc40d3f8cea13677667e4cf91ad3670ee4725355b2f2e3b6f5ac80a48e5dc96787768e6b30b25739a6eaababad28f35b71f37eba49c28d9a793febace3c384d7ed5893be3365deaf784b8033165341b68b70fa23f27cc2702e518ebf");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "", Mac: "5ae1b397e84524d305781cc8334d2fc07f92e8812dfae5eae413d2e20d95b8faf3b8bec12e237ae11dd9bc1df3ade1ff198e89450e1daa71051ed76e33b28f45b284f4257baf5f1ed0c7b511439877f83ad5969d2f8278692962b94f6f7e0b2ad8dc5bf6e75adc912d4a11235fb020dde10c9e7dd787163b");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "", Mac: "51b57b9ac7ef25c3c3b89f7fb96d492810ad8a1686833d2d02fccc260a8f109aaabfe7e8c7d90218a1236e77c3349cbb1c801332f5db6358ab47e626a6172478a392d197c8ce74f434ab4e230d432bf4ebce524aef7ffc2b8777598a7f4fa424929932c791488b19aa33b241d4789f7b601df81daf761432");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "0102030405060708", Mac: "0ca553d6c3edee6592d483d36cc9a9e6dab518cd81ee89687769a491a286560c654332604fe39d1cb44515d3393fc661e2065878f54d055ee3171db7a0d613c0fedeb2bdcc12b1c510f0a7897858bd97345b9a78b624d78bcbee424ddb4b6d29c91d1118232a6fd4f7f0eec88ccc4ed3aa0141e27119692b");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "5a8d4748fc2a45449c781ea563612c1c4c9de7977e9ad18d5982e898465fc0de1c13bdc2a9cddf418b9d35859e7d68ad2a37e8abe4612cb9e7daa59621c1dc9e8ba5daef33e79a0d737a94af72cab18a8e62a229c4c6730482b9ce2404a3b7a0db98fa4a18eaa16ac729202b189c9d535c05d8e94439344a");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "5287986c9d32fa4f38ff64cda7ff2777516a34b49f85eda755f77eb3a1203fe282c1cd2fcc79c0fd0a20a837246ef2f8f1f9b94db3a291af9e6dce142e1779f91b0259f1c6404e647a06e864715b8aabed5960a662e3cfc9ef862fa76900358715f45ac2c07d5097c7b6e7c14c028e98b8ef5bd18a06e705");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "", Mac: "e12bb5c3d3fdca478d1c936fb4a10c2987e9f06400de0a3c53ffa704295a5cbde71d45c5f2a19c082d2fc557fa991ac5045f5fea68d050a3fc9227f25c2da75e53996f62976a956143a412322520feab5c8ceaa03fcae11847810639cfc8df43f7ed5825c2679d27ef0717765e140dda627073ecb7f5895e29c34a7b55d3d7a4dd02337354e37581");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "", Mac: "1d4a7d98b7235d8719f432172e7615468d64b358d6924998c17a04a8912fe56230ed9d6642ee3d36a047e765882735acab68823479e41c8f3b1b3534f0273130709dbab8f477a281f9399ca71bf6b3e539fc9b8241df24b0c17ed27ae695882f22e17f0d21d5c5e034730e9064f976b8ffeb9e35d672b90fe1412204bcbee7930ee166e905a0cfd0");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "", Mac: "8b9a52c47724a074a67f36b71249c12c1203b5d78f5a8dbd0c4ff95d3f35a5523e947e1eae86c1a72a056e9928153bd0736fdfc1a116f390be3bc9926f8141d914325a1059a293893edd6d8d29469919466dd7586767b48b461d33017ba5a690fe2ee835c39d722768aef05b8ed9708913655eb85494cb3efc055ae582957c4c46cd6f3d2b69c07a");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "0102030405060708", Mac: "76b209a9acbeaec79cd26016011e0dc60b8ea2879f3f8a0781728ecaf2c5ba76aa854a3d3e9c934ca212467f1c562d03563db5923fe018120aa990e0389490a4694821e0fb4a769a5a56d107f9a3174e8864de73253b88203bbcb4b253b294d6e99a442f01ee5e99d28dc1d2752429f4cc693a7f22cfaf597e0a8263b45cc2d95059f20342ab6638");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "db23f87e181d31e8f496f5de1f44de62dd90642b94678c883ae1670e66e3df3afc9845e90cf8c18c72674d68b1451f4d6c3cffecaf2e535b145030068d2c00c13dca9ff74002f260a900571af2020416c9176bf2eb113b0130eff7f3bd9e33daad6a8d86c9f7deaf2629679cff2723d288e9b4a0b2e5c5fa8ced0d46d9529e7a7a4d0f8298930347");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "026d4a0fa90e320e1550000a8d077da757739a04eda8016e5101d22b1a4aaf6faa811b1cc3a51591d9ee589a7b3d0abd540510b3c645525cd245c980540c134b32acd75828f38bbc665fbc101e67d9d0b3a60184289c25218f924347e2e9c047542be993decd4949a2ab652670dbd9f9ff69205b0665725d52aa20c1b2c7c3c34735f63f52df447f");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "", Mac: "73ef392994b78cd43ea8060c555736757471fee9b0745398cfd477d1c4959f1f39fee80c0f1cc19147bbb0fe0a547ce05b58a8dd1fd0a93806023e16405dfe61ae81dad50d9d9b04bd381cb00d87107aa1becb4d938661e8576ba8e5e19fb5f5ab638bc3668b5f02e938f06f7de59f8bf546aeae38583943c654be4c8c6235ee995603d7531d71ba654ef7818c0e63fa74aa76108ee75074");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "", Mac: "ba4f8b8f700aa88103552aa1fab53fd7fefe8ad138f7b69d5e3635e3e054a2af9398ec4910ae289047f8839d6c53f82eada48110fdb195bf10c76bff98dd1b0025e63382832e5c71098a23e4ebe2f8261825109223ed7b19515697d2702d2b0fdb6aab581ebd4bd8722be94005f4787c2ead227c107f2cf877ef1d205ec6f0f94e91edac998858ab3363e68e5f09b006e6158ac74077e6ae");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "", Mac: "88e61ba039525fb6dbbc9b5eee42d8b9d7e17a82fca5e091d9bc69cd6ed63c064c41609902ad4f116f706329bc3a1855162ed75058bd4129f01132345f87cc3c7c2cfd74ccf43d31059964ebe74e0340e1e90d342661fd6bf998ef789ebae6e7abafe8aee406cfd46b4693f4fec4ac73ae3f08c721a2f9a30387e88df20c361b6f584db7d4619c72314d3218273da80572382a8bbaca5e2c");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "0102030405060708", Mac: "6c9c08585b9c57317adf410b6ba80cc582c37737ca87557bcc94144091b4f27f25664a78c8ba2700b3235a51af6ea43fa5f32b0250dccef6f5e1cd42d62f7d950067108fc7420f8fe06a01f32b40f286b9aa94551721d69ec999c0e65797ace2b222c2c751bb6e78c736a17dfcee1a51dbc9ade02cb711f0b5dc15b7afcbd2f5bd3152d9d5fc252705b1c9f85213bd6f36dd2d2b778de962");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "8fb76df9822e05d950e5c53e23e1f5ddc901498b6b3923c24b12398bc4e01c67c1198e8cae8a5cae629043b4afd0d3feb69cc4d34680cf39b87d474ac4863c7a9ebaf03716a9531835cc65882c86570de3cf64f5c3b0918863ed19fc9319ccf2bb7d210c76dcd422a46611040203dee0d5901d0789ab452dcc1f9d3407d316dff0c7f06999bd30745c6582205127cd59948ca29f9daed277");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "44a0b79ccf6e61a65010fcc5fd78993c7b6b4ef60a6c652d8e99aa39959f4ac698df9d939cc2edfe034cf96be816ab8e6b32cbeb07e5f9a1e55ac3c9f081535ecf033f3c34aac5483262af10d8c930d4437105dfb13fd806321eef1aed69f243f55f3820e8828e1c07360207e738c0263c00204420be87f7ae77ccede26b529042ebdd8cf742a0bcffd2a015dea2a5ade8701c0463a72eaa");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "", Mac: "ddf97c45f3322c79d9073d27c58f3d2488e54d26dc684837b8e9b55be7e5f1ac06af0db70ab9932deeeb661a73effc27712ee36540121596b7df466f76cbfe690e228ef7a44f349be323ae6719bec373cc62155df1f5387f95ffc430413dc7862fc9a384a9e84fff4b2843733c13f143dedcf8fb8a52f4d6e4fe6e9f70668cd4a485f784550e7472b840b9b957add9ae721e3e6789c8a42181fa730c983ce865d1379bea9395a226");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "", Mac: "822876f4773ed11248097a3e3bd3359e223e595785dbac2b47d94a176f15c46b1d2b0a8913384e42456d45ebab9475033cb989ece6e298161e8ade0da012ca5a09bcdca177b81f87d52a5f9975a0384541a07e17a15d93c139bfcb54581b0c40abc093ab82b77ecb06e57f673e8104f09fbb4824d39215b209ce33fa450b768e1a501f75109ed89f74ed73f755a1538d600108cfb8d3be11d2957510c91826a34a0fabcf0939ae3e");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "", Mac: "b631e07d358ff988c7a07b90cc0f1cdc086a38b77927064f07ae4a43ef53a77f5d6908108452cccc920d370064a3fcfed55f02ac1cd8d1b38791b69764a7a0350f6437ebc07d85d962d50d01cbb97d11f31a5dc2ce176d80d8c71fa1c1b5bb8b1c2344c09d78d11bdb6eb7eb71e9c4666d56f4db14f62ca60f4c855e212e840f5d937e01b179f750cee8355bc0c24e5ffc052fde39f0c079b487789bb88c6c82a6ad13e18913d593");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "0102030405060708", Mac: "14e2e6db85ef032a928e9081ea2df8ec65a555fed4aa49bc80e9cca67175123165a6e54a8c9af52f13c477ccf9bd0da5e216ca9b9fd545886b834c07be8db995f5e1152265adaa84750b6dfc73bbc9ad68c68866534b2a8eed1a289f6337b8996913c4c1e5d2ad9faa82cec0ab470a513a32bf2b27bb8206dd5ed2237515f7b22e15c2239f2250101bc6e261fc0f74d55f49b8b59c19872368c66722ff4174cd33b771cf1c7cfa96");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "0d125df9734bdc2394ff98706bfd7948ccdf137ac6d9440131b47b7c9c52d0c34f42813475a678a955e942863c1b4e6f45fe8df77efdcccf480ce0cfc9d2b98d16aa3c8406548e1c27c963096691b6c64add1b7385686bd95a4daf05de0d7182b4033b940e889787a11a25ed122312974bccb7ab2f97fab800b1a7789c34c636ad0b43b0b6a1bc784190992c7c98bd17980e9dc153349190cd5163483de526a0ce26ce76166071d5");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "0a33c8cb894fc914684f6c18518eba45e0053c298ae332a277aee5f3020579d728f86881429e2d724ee45e87869c2e68ae36819a49c7f614a6212ef0f78c3c819b222846b6c9ad98f59810c658060d491cb0f6e31d311751b070739f330c9c9adf11ed4cb280e2b3b491864ba2607374becf7fcacde5f71b4b25b84daafda0e0a3b717832db5bb6933d33e215411985783e3e7274c764d38c1402ba7c00e6e522bc9d9f0d0270df2");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "", Mac: "6d5016e8d5d2ab22c88df93e92f60fd39943b9359541415f08f4d1e38935a2dfd60d428c7337a445a678471a24e07f999c483470c90fda52d110a6095ee5fd2d6bbc6d8d08da55741d6f891db79faf322e504f6a41d926d7e3f7a425a114cef69cd8631146c80270a551ff690b082ec48f3b1f8952848f94e2e5db9e7822931f3b767d969aec788b8bc098f30f785b1afde0c8cdc62906a15c73c69d89da90b085dbe014e4885513b1e4fcbe5e14263a946a3b37a94a666f");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "", Mac: "0c8559418a5726e9a6cca0a792de95b58ec5c75552e74e6aaf7fa875b235b3c0e6b82643b2b9fbfb7bf938e84f2aebdc678ad7380be63899180313f17b18b3fcfa5a3be1265f2963a94610f2cbeb7efe8b62eeb1429ca0e37e319912fef5d4e191ffb8638e41f1f7b92169ced9e749468deeb74812e13d964e2400186418268eb0fb97460ab4ae8c9cc70816a739b8c3159256a87063e824c89a9bfbd95659871aa3763b42f3865f222c89122fadbc5c784dc0e60b82f536");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "", Mac: "0467c76685caff9cc980eb0cb59d9b0ae7433cafdad1ac4e51a50ca9d5055459323af34960a9a2d7bb619f6e2105994eaffa4dfec2e3a4d2ec60d9480ca89c9460dc5329e40f807f1e3d7b8e27eb909aa3fc0d11d6f71bb869f27ce18d1867a52aaff2f4e7204268fc27e0dc3a9fc0310b4715e942da8f6749ead696c663958e9243d7d1084226e7cf16786aca0bcda990735b0c60e887a88727dd41b663fef0e3c7e9d57f1dca34176efaf231a988b3302b43788dd0818d");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "0102030405060708", Mac: "d76f7855e18eefed57f08da5d9bfd26f0e0a08f8e69ef9e3f760d6d38e8b7f9c218d95c0a1bba2640aa5199efd7f2a7fc5eb70a92af0fe2552d3226cd2540f3bd8a52b7e417b892733bbb2d5004bfc4d7c3310a656030bb77c2523d4d9774cb1a425b42bf8b70e16d66f5a11021890572002d9086b029b33df5bc35e08ef27d84069aeb0a10da28555efa69753a1549a179584079a506a5e2fed95f5f4016fa4376f9e81b06894f2f887e90c8abccd7df737ecc80a09fae5");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "c5a117fbd628435bfeebddae81de7ab0b9f303ba6563030212ad40026fc6a2ccc38e99d9fb9be3fceadd69a1b7d9963f97f66bc1cf0df8b79497b6ef1afed2b07ce80a4f8e07ea99ba75757e6700cc1e9f91754485aec7dbe3cb2db181e5ab77fbc10bf5981dc2a06bebe6d0c6c595c9c00ab8ddfc86dff65b648961e7a495ede0c1163832ec077c1d5500274b32c4856f09e5dcc3a10d48f44acd8d194d391f183b73d247653b1f4b210a21f212db92718868ab6b4c304f");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "af67d62b36bbf96ada759260f303aa0347bbb1b2a9e23968a390938af53f5391fa0d98fc9d7fbea8ede691265e6e9d6d668e0ffbd615653e94841c5d24eab8f6e0c28b518ad755fb19cfa990fbf82aa2a50a80abf232afd57dc0e1c0cafb7bc4fcc06986acb8a20044465cf5bdf71f5bc6e38973d8b3f4f1fa54c0b7b017a7294ed3c6187883861f58668edca00b014a5f0e648f44380e0e63eeeffcd26d2a087ee6872215879945dae0dc2ff724c142fdc5031c3e586a1c");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "", Mac: "414f7cb04af3ac3afe554093310e530d7183f838c15c30c6e04b7a33e5e59e4199c9d71caa0601b98c10c82baace784e305c3db3308c175a75bc7867ffb285f95cac1667e09a2bfdde96bf3a0330e2dcde3200dfd40adc3bee712d06602daa5eaedd2578971b2667b13e8da49d0564db276b8aa13dda1ddfcc83f209bda595c87c60856ede5a32f246aa6d2bdd5290c915f2031255ca442bfa7393d90c2b7f74e8aeef77b10ffa5fdf91d5522448c78e4ef5e5480543091c3ce6b87106ce6322ffed25784903628f");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "", Mac: "c871f6778309c27e617ba362542519569bc01262a53a4435ae60d70911dfa5ec1f8835b20c7b4b100f1c068752cc23c2fdb3e201b3d7e72b6cb16f4b742fd771d95bb249b43bfb30da1cb6be0b67b99432a35ac037cda1458ebb89791e267fff722f6b86bebd58b827e43edfff1fa809beeff850725ebec30291ef6811de6672734e9c712fd838e6d73be05fdbbdac0f1e8caa246f3cdcbc27cb3c2d0cebaf9242a7b0e671b0f6b190254acc3740c2a6feef29faad94ac6dc16c0a0af68fbf32f21e308c6cd64477");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "", Mac: "af45c49bbf735bdab9ade7f3b0554fe20a9e143cc9393be090b25f70793542265744922e9c22f2786f846cb45a0e436270a18f2785a8fe5ddb25fabdcc34f3a2bdfd7a8bf94b12c76268a703edd78aeccad96e4ecaf793ab0184b495c1821b00583c998a7c1f3d4d655b34ed66acf96d7cb92f43fe905641c1a0515e8f45e57bd27b1f7641dc0c2a21434dae2b1cceba23fdf52d367b931b5d43020ee7e5fbada3646c678d1e58a9d644142d39551cafc32bb0e015cc9d76146616e167bde97e57450f2550327321");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "0102030405060708", Mac: "66b08d7a325eb99ad756871567b23fd37a19811536911d585183759822cb23a50922a78ac2da173d751125648e1383488bafe282b820b395b9c1ac1097168759ce1520ec01d7635c449f056e2880c648388ca55ae6a50b274dc7c5edfe2c9d812ddad7ea09f3f92284542e61af3df5bbf7a24ecd19c3ae04e72e8f6c1a45a49d91f1dfb58380ae39869da0f43ef93dd4b52e78767009a07e1d7ccabc0f385d539ad2c25185467ad25b6b2b888c059fbd2479be4d594e192577122d2a34bc6d9f9fe1ee96da48b4d6");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "383192984d8bfb37f68999c36b7fadaf97d7c6c635060f53a97d929804a27e73c61da5c9d256c151e2426388574a2de19984af61b7fb71aab5ceb1cd8996745e57ae2cd59f4058f41203bdeaa102169bcbfa7ed3aa73f4c6c5d3e36f5e4a4f7bc99be5fe4e00868aa90c97128d3f16a2611b1114e6e5041a06bbbba6b8122efca0c9c9adf7be9cb3e90979c5586f2793b08b0011c4f891d29726db3933fa842fa681e074905c3900236db635ead0e332a280438cb59a5ee81818be37014a2478b5754f031e2a2162");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "da42d62cdfd5d206d8301f46ab7c52108b6b93eae7268791ea0adbab71f98e62c85906153e8da19568d0e9f3238f9c05be4ed58b0fade88d22b1469a86e38c92594ec652d53ee7016be540a3b4ef64739c22af2bb23cb28e1e4b755b58af5a1810fbb9fb8f66d9c5e0fab6a781a546fd6a3ec654679047bb8baecac20ab1ecf946f859d410d7b643676ef8a1e385dc8b14bee61c0c94de231579ca13fe68b3eca32fd1135865782b274d992473ec3fb79b0ff2c3360f8d88621c0f71c1da7a6f343122acee1f7caa");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "", Mac: "477ebae550ea7f9af3a6ad198d9316f2e78bde82d2dde5e82252c5fa6824d4176d613eae1ab03db85576c6c7b42ec9c854cd629aeb065a905f167350810faf23e145e712678f6cc6a9cb99460da3e0fb0528fd25f665ef4476b6d7cfede2549bd3ba15abb81edc5e4e9c85f580505ca61439dfd4e724d8a85e268e75c884a7ccbb6cc46949cec26b9ae172162d582ef62fe42430b8a2ebe3e6e84a7bf5598e8140da61f59cb2bb6effc647b47894967568d97bb25d81c6a6fada02afa84f11aeebd8ba08931c1624ff6281faf00ba644847a107399d15e61");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "", Mac: "a1615f782da02c987075edd5576d16c989171c71d5b18abd5fe7563f9a876871c03fd96e9b2bfaab2943d7984222ad355c2a65a7126ed95e6dc31286c41a15f1401b8cd55cdefd10bd9201cb476694f1fe90fa6575da4d729a066e01b7e35ccbc937a1f13e86df3ea63e5f419335906a15e6cffdf3a41b90edf29ffdcd8c181752e1727b667e3ed4f0a937ced06c508206ac862052477ebea1d042c999d3a6025cbdf4c0e956ebebd0f330916e3f9c08a9b5d22b3f19b6b304b82728b5238e84112d5acc7c5cda97d1a84ea8886906890e66830607d3f69f");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "", Mac: "b82fb15269b994efbc31acb1926ee721b4a9c08ef6ab78c9d4aebbacd9f54f0ade9197833276087443171f113e43d98d1390e53a4bd1cf792477d78ab4c79797b816d7674f178ebf2e28b9c5fa87cf2b5acdb7034431f61f60d82d29f76b53cf58823bcca3c290ade15b25dcb2d82c1561c142efcb71b2e8af4e7657bd7988261bcd53693499bf12a25743109edf11887d7d600669062d842988c61a0f2dec990814386acaadb61e0b24ec2f85971067bdaaac1850c6ac8f332d4f88aaa7321698d21df139a235447c08905f1b7816b269bee4adda2be7a1");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "0102030405060708", Mac: "58550145229e7b3571fa459355f9cda2dd2036967ff86b2798f1730e634bbe179de8e5a56306ec284f8b58da7182dec0384328747a1f56b2a5f73baa567341ec0eb2d99fde19c87d4ca4b9e5a90af360f70117d09d88b64d95dded39ad3a880ac6a9d8dee2843c142f0496d95674daa1eb204ac5f6d02d55cff3dace31e4a6b0f4561aa1a74ae4552b40bcf27917f1111ca2225dae97753681d8a4d7fe80f6a07683a7f6c6b3d8cb0c2eb5443f45b40f521c49d60c4a3d43d9c0403a0cddeecf821f28fc90c6a37bea806b9cef7d34965a4d6e7ac807d8a4");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "25d6d6b2ef0aeff99c3437eff02d22e770159e7b07febe2a45ad8e212f1e4fc7a65a23b749c9e0789fc0f5eefd83397baa1b7141056ac2131655ed0b21542ee0b6aa355bf46b62a295a4517ceea2572852fda165a157ec903c1ab9f8a083be23b38f9e048b017e09301b541b76a5c2ce90b9144b216bf9251cbb3ca388d0fb055717ac7e7befb054a7c78035e758d939e8f318b3befa40088c46466fce8f2159b469ab6131af93fd0c2f8f966763c55445f40a1ed382b4c06d6acf298c56228b11915f20b77377315871e68015c740da556e5e3103e02739");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "04878eba99dba952856e7e7775a5a9ae9881162f858d358f3ebcf72e118b48881b2a3f509ec0f48e12ccc81ee290f27cd7d6c95e078179eb800a7bda9ce234ac22c30ff36f9fb7493af88a6229fd8cdf0ff25fc0dd72c7f223a9c9bdc89b52a9ce37289eb623aa240d44c7b7b1c2ad7135bf8ec0e1494f41159afae3ec8e7982ec2671d1072c4b6619db3a64ec194dfde36b4f3746c3a56c0125f47cb1b752a45e3155de792e05d1ed84dfa3d894ef4280465ec90255302f69160b9e577ea38e925cf85e1952e79802ee208d382f8c6b099612b7253a9772");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "", Mac: "4a9fe0e25c14b36fa0eb52114fbc9facdb97a4467f02a6eb137ba124fcda7d1946b00272452dd9be49959f06dc546fa867756451f8f1fcb62163fdb9c988e60ea38dca7b730684fd7ec91aba403b6f681b718cfd89e0cdbc21a9c8a2ffda7636c772467c3849d071dc05f1d7b963a63ff7619dee5d1c354bf544b31687bf3bb14c959b32327b686292400e7532c18a7843f473bcf52accf6dc8fc60aff6164c25eeabfba1ec1bf1a51766458695ce69a913b2a1c6f99e443a499b9ad9e4bafdeabd7b9f345b02672a3f8f1af0710862a356d2214b5143dae96bdfedf4db87b38d63bb443af74a279");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "", Mac: "84058fab67811f3978a92362c7fb1211d0d3a577ecd60db5760310b8d62e24ecf6fb89b17403b03aaa49f783851a9f1b83255988c465abeaf6815a0d057c1278d7286e22c5c8fad56d54ff417ae48c10864e9ec3136de669a3aa5d394a2dd5cb0ea3980d3246279705a13ec7b5a845226c56b2fa89ddf5ecb945756f8919f5393d9de856de7f446760b6fe9f6a07107c391d1a635d4f7ddeb94409cfde540accb546ee045dae6b0eb2f2490a4bc8214cd30ed89053a023b909247c9caae221892a4328a670711d1e57b0009677050ac26df1aa2ab4b425f334e5fb1cc6a6459dee9bd537d13992ca");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "", Mac: "9f9a1886d31ab569b44524eb6f7f570cc7d2f7317f5abb7256a26aa81c9c7d4678176d57f2864caeeeb4c05b6d462797b8be669e519068f0d1c879b306ff6bbd44484e635a5f1fdf76aec9df3e992d8f2d1ec2ef5813465115ecc305b6509c1a5c69db5f5ca6f722060050ac4a9bcfd48ffc57c38e75f434cf1e4f610822dba8e2563e04d3b6f6081a1ade56e3ef8a6719e9d28810c24a369f2f9b9c5b495ae7230331cb2ad9f8a099d3fadf49decf73248a236a270885894a39b0dc437a56146655f07857788a69a420d2bb4b919c4d88f4515a7b3b9d5b82eb8c28414ef706489cb981a5ecef60");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "0102030405060708", Mac: "f838c6ec30b7b6c67a2991ddb1c3d9963f323097a20d996e76bd4ece0345a76225a83463c6f3e9f6db4944445edd2cf77217180133f4051dedd781dd5f79a35b8c03abb00ebeabd5e76c1a81ad846ad6bd8e7ee468fb82eb0adbc91f7e97a0b99a8d2ce60e534eab1284b6ef9e68d329255a221f0b3798c1c7bb61bb35ab02297e2813c87e0eb3a29d4583609c00e41fb9de7a9fc46a7efed811a1603333f01db55ae6f183d30e116762c4d96e6bbc80adebca989bbb77dabd370fc05c05de4c04e349930838e73c9a2b6ae0259f8410ccb2da5f214ade5361b5606b26498d844048eb7f16d1f1ba");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "7f81ce4078eff550dba0c7c74285b7f8bdaf7e103be099853f467456925c49920654741a4bffe57d72612503d24e7039d977e502388df05bd2d1a977349fc1cc645e7298e81946464a9f0e293f8b4555716979ddba672ff20c95d4805f84b363ddfb5f0863e49dcd4714eeb469eb96152d00da9e1abd449d7d0a8bdada0d6ffd43946dd0cd2b3f61c45f7b15986771fcf9de57a80e804e3ed6b5d1f84a0e83a4b7a8dcd84addc4f5ef8f9d82590a9108a61eb13b122d0d38f2fe9afed78e37fa0b677cd9e480b4a9f265d6ba2c02b200a465e4830792b016301d5b553107d727be80efe9c8feb4ea");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "c18fe8ee73ff8969b4457a5b544c0bcf29c2f1f33968f7880fe400bd68e397b1a485e44e68752d0affa23a3e2fcb9b18c0b802be2b90ed479f76ef53d51b399c2c9b9e51c56b9a4b613e76f83e542e8eb003d1a749cd90e4f071dcecee9f33027f159381d0624c21d24355fcd4286656e127be28d4bd5408f6fef86e40c6e2d859d58815d83832b9689233e870f0f3fd9667b508085bce4e9693b1a1a6978cd80d97b55056f4727d40ea2ef51b6a547f079711808cebb151d711e0d902e3cab4fd757b48c8ed3c7502d9efe496a589cddceb5dc07a88071dd4dcc9e3dace65003e45a74b4ce1e054");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "", Mac: "08b95bcb9453a6cbf5169a90b957090f87ff8018328d2fe92ebe31b74f9b87dc5a67f3176976130ace1f21f74a2089e86949087c2b6254f0f376509a32f2fd0491634fa170b05a6ed8e0cb879303a1cd48d47536ce4d24c773cb30de0705bd5c1287b7d76da717d69c401a46d9b3f9384ed1371c831750eeb8066aafedb078701d8ef871e9f166a585784a45a9fa948c40df9f1e85282cf3fa390846b458f51d15965c261dad9928743020a3b3c261042c61bb8ffa7e5d9b5383e52aecc0bc53d9a2755adcdad2b31dccf6ad8ba97e2ca7214b4ee0ecd58fee7a80c799eba13fa2e2f8ec0ebd08d0384c6b376177b9a0f8696f0c9c4e1283");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "", Mac: "9598f820257cd9ff942a1d37ee94fd7afb2c4d5de5608eec45e0efce5e082837d50a75bf1c019d04c30fc1374283e24371e6c24f01eba3b70a7acd467e2b2f6df3aede6d3441117f39756b941ede47b430e95e55725310054489c1effefc5198717ec0857a54f2aaf8b44f48d50ea7b6b27c8a9f4330de2a45f8b8665e3244f8dd60f4c44c090e4e6da7d372d1e42883ba46fdbe0ccb3928cc90fb0433143bd4705603a4ae58cb1cb27c85fd235c73e324b1ab1a2d035afa923286ea344846305a6c9cec186125033acef8566f2d8b2b5c52fb59d4da6d767ca6cc27230e48d8e36168f48abae573ab698cb8d700a6807692615777a95a03");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "", Mac: "15a6e5af61973527d30f5a2cb3da2a163eccffe30f2b2d5c89eac5f21c2248fa269d2ab6a9dd2f0350899dbb16aed578c83b66b7193ecd7ca494468d56298d2976c68048e2d7466bb993af429ff308b9c8357cd8face63791a997b82172402de6ac195b2e2a48cfab9ef0e5b9862828e59ac6952473569f5897c92fae004054943bf022d874c29a5c042920d45b077e5f5570451b0b84e7aa2e1966ea786c2d25e2ef892b9aad844cfdf68be1b4f257e205184d876f5f84cf80578a01dfedabd543a85193fbaeda9fbd1e955484dbdd418843f444e70665fe4343439bcbb4aed53e954ba9704841a1d066a744a1d0c616745c7f3933dc0ae");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "0102030405060708", Mac: "aa60d18d734b43207fec672900232c0f43567bf13fb97eb76ef26cc87397fbccf4b0aca15fc430f077e75ae99f822d758cd9d4e531ea88e3a1721e2fb4cd743e0fbe325c8f4fdfdd55a812d42d48604ceeeb122cea8f1ada1c342509683fbba56be48d4e22d6d5778e05f26549f7bf64363ead1aef01dbc689dfaf4fede1f46b3ca5b51ec64134b5e5451a2ccba578f1bbe7e7013b64cbdf56941512d54908a9c2c180fc3c49bb746dd29cef7094f968107e2ef44e92c318b3411b3ed7aa9db7f904a9d34e6a9be877c5176291a77b59d76433778360d8c3d5cdaa43080c8e178fe6ee26faa68e82e49d70bd24edd0672c10c22ae89e4890");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "7dc01bf63fb2f24102da7442ba2cd521d366e4021b0158a61f4d83cd6434908f4e7a62d59541dbea4b9b73ddd0c105f5ae3883bb6813fdb1225056747354a25cb56a5d3ef0a2dafbc3a06d886efddb03e3f1f8bb86c0a87d8f9c2ea3998aa4060117997870fc7222e4c8b331b0f0ada6f1bf306826ba735f79e8be6a2eae8b3afd164c7ff7f05a6d1e0248f2e66b05f7959164112bfd7610bbbb856d6b9cd02649f6b6f882d1b7931dfd97cf34f146ec81d9fe50dd1c17f21af7a8a3755f4706236fdf9e2d819f3df464757e2feac1829b5cd1f7f11fcb7b5b5fe15c94a0e69efd8e3171036af947cd469974b491ea089f266d6c2d363109");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "0b6bfb499d761bde15c570f311c642c4f7229b92cb8c1a039f3441e85f232d73119a49af221aed8463a4b2bbde797e851499abc354edb6fe805461c57fe2fee9531f72335f26b9362dc17b6c42636b7d86e13efb208c3947de30e3f8fd3cc5bdc54381ce5718833454c50538aa9cef0c999db62314a98d08c90f7467124c8b9c2487e4caf97761c74c9c2f51800c9565d03db7e3a72e6fc24e4a5d792207e22edda68bfe2f5eca2143fd3f2a17d4f16a8b8f1987055db1bf07165a44f54028cd1eafe3d4854fe3f101e4e8a5ee8253360044c4260209e3e2b8fac76a2fcf6de6920c3ca40dd469939b59341352b8a05f4efb01db0cb01004");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "", Mac: "c3c13e4e58483f3b3219dcb6c6ddff114e175bef00791307bbae1e370cc56802473deb972cada97d668ff4c0f391b9865513bfe3dc55d72d9be74d7b57fc4fb42f56f80e72f3dae3fb09ac73d520968fd4e093ae2498d2c3f39defc895a65e5dfb94aa2769a8d256abefbe20aba825f95d7989361ee0e3435ecdb004cb38860cc7fb7c94b656addb1e0f955bed5c93d379a39ede6d1539e965f2895005f1a07099100d6a3b65361392c5527e210bf58f1732aea6fa04ea4e62435b81dba07c8da7036958d8b4111d5432bd686841de7e2e5c14a80820ce1ced8482b518b788e91d89153eaec80f518ba4658b0949d60064d4f38ba64bf2bbc8b2ffc500ccd3073e0e57f53c2fea42");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "", Mac: "02851780edd268aff4f01d3aaaa34a3653d81d101c6cfb0d41c01938e6b958db9c1920c0c2095326297c1797471c710bf632b0ee5e2a6531f94faa73d1adc6a595d54c9ff4f4a6b310c9d6c0f99d851e15bc35a74f02cfb88c9686df7a453c9893a9b2cdc0d6325564dbb7dc38249753742ef569944d409db6b35b0c071d6367914707d59b4e2c95e4ff9eb726d02901360f90780d4c92eb3daabad77c9ae74bddb5b2b67430d8ca209c33c905b9b5e47f73dbff6e9ff869e9d42fab6fcbe1d774446aeefedffe3387aaa8daaf653782a13ee2e5f3a7d7e7eef9a90a5445329149da2b5deb4715836b334a6a6c839487af53e4fa73d20aa49cb23fa9f15ea88c46cb7b4daf6a82a5");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "", Mac: "66b08822cd2b3df24d622b97fe17776b32eb95287d5fd5426dc455cecbbc7c7df94e36d4debe7b944b714ad6f45e21c9e9d6beb0f756a0d81feeb77013ed93899bbe64f4a3af23a8e021e03e0c1dee835656ef834778c38ed69dc51e88054c8a2441392b86e57b92a794c9296290a81d0dd28ac055b18c6e41579ada5e49fac74c08d0fca59fb345f07eb57c79f6b83812359c3d431fc574f8a6efab73a69db0f1988937a58cd90a5235bf5b9c7aa59d6a96788bb7edf538d8846b09ab796a996083ee1bde2c5efa0b8937a56c3a23987fdc937ed6c8808ddc4a02c509ae8fb4d64718b0b7cf7a6b023aac468266999f888c33f240ef264506e8e35a5ad093c6d0ed5672c178921c");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "0102030405060708", Mac: "1ab8f06896bdaa4eebe7648a04fc7fcff7087898a44d6dccdb789ba3d6868393895caef126e084bec811308d84c44c6a36a08c8aa2308e28ce896b1dcba555a69253d63c49455d12d8031cefd1992eb7d226a37f29e2a54f9f3f03ed3ad687e8140ad70ee85d0cb2128404ec2c73b8628bb09b84ef5da7934ba7c1ad4dca7c0bb348e283c698d97a7e9e81bd6a44885c250f70acec93b41f70b6bfaf4d1f506163fb8494a048ee8802f5a13e0f5d2c7583f3c349a93323fa5ba3d267b43068f990786bfb5a5ccc8b1610ae1e3efcdbc48de8adb685ec8651319ad1f05cf06824b1de59a1765cba1c7d7ed3a84a8395aebb0ad9251677f17682588d2228d19317265764902aed3ce6");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "ed3674ed63a5d16f0efc9ac8aa27d7ed528e47793fbda1d61b621ee2430d3dcf7d634597173f414ecfa6d16befc802c6b65c8eb3d901f692cc46fe6a766aee0c79144a58c0bfcf624d89aafe69b67f6006f3e7bab5584cf048afe773ff00efbdc3fcf70ee1d8ca8cac6f202c0ac21294152479289fb28c2b3873058c0210b7db28f044c787806548b50644c249cc7b6472d4076c2d35c66e4c60c3a9ed571cfcad299196da8cdc9c8cd572f92f279a36743d2da29121def49e20652f1d8190361f30e77d618aaaff3f7e1bb930447ad3a3ab58dd899d432dbd1be717d267cc6cf25bc00e33aa41f415b0d6aa2dfda084be7376d0b07bd541dcfbdd4b647bf5f8175c8799de88a21e");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "9a634921ffc0a7c99da521a368bbce490ba2a043a495405cd66db5ab1ec04d089833edc5223a10f43f15239a153fe603d1c8d826a366268d4c3b9220e718c1c048e87d9b644558537ebb28691c2ad76e479e57bd6e2f95101fd8a8f12437ae72adc220cb4b30bb3e3359c7d1d0c797d4a0744b05ad64d4e7f56fd51d5620e5e2c9fb145e7658485b563b6b671c4566d8de582834075a6c1f36345bbe38009f38b47998c4972b2c2ac6c7561359de149822a4d6b8329e7029bc4aa8845271aaddea7f716d087ae54d0f0950798c65603aa204e28131bf1301eb37983080316c3f736e1ce509ee13b15b235fe2da2cfd367d4cf73d29a324f717ae48bb9097be9e3919dbc9b8f25d7a");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "", Mac: "b9");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "", Mac: "8c");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "", Mac: "5d");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "0102030405060708", Mac: "45");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "b0");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "b6");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "", Mac: "f81cc98039f88a28");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "", Mac: "efd4638540f35deb");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "", Mac: "26b749fd09104520");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "0102030405060708", Mac: "317e9ee273964950");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "11d5930931ed3bce");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "718e48706d8e4fc3");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "", Mac: "fe1f2d7549c871b954952e4bfd0ba91e400e061534844500");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "", Mac: "ca091dd027b21f8dcef0b85b5eaced581357281a2db28ada");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "", Mac: "9f017bda32c78a2082ca7ecf901a938529aae11d028a381c");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "0102030405060708", Mac: "a6ea99ee0caf4e62b65e7b8a7baa7fc1d537dfd833c4b1ba");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "f1c9f20359c70a391d462894e678b2a9a4b94c6a426ef2f4");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "3994b1f4fa152670aab83a1538e2da0a726e85d0e67144de");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "", Mac: "3c4870e4c2a27acb898a8319527c8e4588fe2faed51f8ccc1f4172a2470544dda545fc5cce1fd781");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "", Mac: "289c9321985cc12ea426b48b62b3dd730a273ad0e47a8d6990954583cef8308211c2607fae5ce2e3");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "", Mac: "cd4d17e32fe62bdde2732925aeeab4333e48a43d62ff0bbccc03f323b9dc9c7f8e403a3386483027");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "0102030405060708", Mac: "51e74e00777f387e76f2af213dad4bc8bcab8a5ff942f6e8cc9d6d443760dde3421fb459be683587");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "600c3e1b0c98590fc0f3f82479a9aeaa15a2770dfac7275b5e4afec17282e59eaa9c37c0abd7bda4");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "8d94f055604e0218b52adc34f375884a0dd334c87c2adb51c9b564f1bccb8808a33dd8bb32909b72");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "", Mac: "5c3435116c10d871b1815040e73176f244cecee139913a0e670dfabf07e630ac476f6465368cf6958b280312dfad0724e79bc7ae28d8b2f1");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "", Mac: "2cb56234d8ce423c7bba0fc3a4708b0534e7201908631f92957880646c020679cb90195e9bab93aebf7cdad28bf36f57a0ac663e9eb33507");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "", Mac: "003d20f1a3873110fa98389c21cacbc853a4c83d643a28cbc138003dbffe1a4a29e938d56ebdab5f8ab25e3de92afbf418c93986280686ba");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "0102030405060708", Mac: "6f405bd04f85485f3421077186d6383562ba3e3e116348927fd7939eeea346d60381f4284eb987683f7deb346c6cc390005a835c8c68b1b0");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "ce29cea54c291fb93e68c0ad28a314c34fa819c7db5328fc55fbed9437ded5ff8d1c0269b7579ae33fb019ae5dbfc232c9234cdd54b83b82");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "b2d45a92a09b4695550adeeee46eb201f260038bd11b18e506b35cb8c9587bccb6adb262b7d80f48e6327e13987200a3f65cbdd9fe2ae77b");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "", Mac: "b171bc5765e57c1a8ab0fbb8e6fd5940867d69515714482fe8a918fb32050786fd3fd32630c01325946356a43d9731a4493a6c4eca17f1a7afa81d96a649038615626aa3690ea93a");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "", Mac: "0eb9d56ff68e98b3eb43f94c19fb8eefb5c820acdda0dd27b1007c27ff471819f00eddc39e5db629642955df582efb6ac94e943ba43d07f719426910be53f3189bc8869b2ecbcd7b");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "", Mac: "83d0bd2e10fedf56e09af4957053268564a57dec59e2718495cd50ee19063636eaeed858b86704d43de01272fc1d9f1a9c03b03fe33c0f22ee629455892f4e6bce49b64d322bb776");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "0102030405060708", Mac: "58d6ead0df389ab67a89f134e0aa20830fea653f72959ad2471914ba795df9c78d3820f0b496a618c23264168f7d2877e2ac1038d0f324e6224b3a84231f95265cd1891a66d71f0a");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "87dadc97e0e067d2bdacdc9e26a467c8a3de362b6d062c67f0c12500478ac0053fb94d0933a7707d9cd8098c7a78e2ae585bed6e9d80256e1e39e9f9b9ff17274e2e5c1ad892ad1b");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "1fa562b056b2a7699b35219bc8d0a2fb07495ebccee925d66cb1853539854045c299485c94b94f3ef604e3ddeb22e4e645aeb1681340f76b090c263430a11c72b6257fcd4b36c9c4");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "", Mac: "204ad5b7995bb67d0376c3eedf8faaf17eaff6f4e644424abb08065a7197f69e337e85764fec43d066a0be6ebaecb5e18ecd7bf2f8c769d8681d71beb172882ea0872dfc1dc310e8d86d8f3fcd7bd8eec4913d46ae55553c");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "", Mac: "d8762f7e7d4281c7a5f68e9db6300e3023388366cf94d32f2d9a7f90b740fe0384675376a2bbfac67de6c9f496b15ea1899f9859b0a233d971a8620217ba2f94441e6d0a1392ade3c0ff8672da1087bc8ef06c979c82b5f3");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "", Mac: "e07bfbfe6f85e670ad3232561b7e6912b616e35c94474f4b7f6d6e8d7ea762af610aec2e6966f4d1452297ce66c85a9b20f9719413b7cec8d6881394eed63257a8528f525c90eaee19b2defb35a6315a681d8313b5dffebe");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "0102030405060708", Mac: "3f39cd5e0a6dbf38576088ea097f95fc39cc33d66882d6306f4e7c7e1cecf68e096f45618dbef515bb0c0e5efecd58228ecd23689e4fd2820e34a3509dd62ae1be85e3bbf6d1534231842072a142719aaf41d4b40715787d");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "319119b140c6d7d5d5310c28117c54fb203667d452eb2e33a2a3622b9b9cde54e96cb5092e2f1078569130b9d25574b6be4bf1b71366cc0de880a4c53bbf2ec01eb9e843780ba1d7abd5d185fd93b3198f1b24db77f5bf7d");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "bb59fb957f9a3cd3530adc6faea837bea77bb2405e825cf2a9323c34e68cd4928fc1c909734cb489a602a6edbf3202425aea87d3d6e84e95c04f89fe51f0463d226d77b189e5376f3fbdfd20f72c5d68be21c3bb62ce4794");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "", Mac: "d72608e5a3521ba0ca4cb5fe925f88ff0495fe3c2f75ec53c0e71ad8facb22c1ef0be667d203ba015f2c913411c6157a3d93ec90c236436a8ed65ca1b90e4f79f43ac1c9005fcb59d583ede0cf3da150049b222b8b5b53ff52305306c36a6f3ef72205cc8be18ce6");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "", Mac: "eed5dc76f8a949c470b010b945ae0789bbb5db610d57ec458cbb694bc61ad4eda7ebb4003070a9f029d21929adad3d12ddd3482a4039fd2e3c472174d7ee056a88e58006504548b7bef4dee4dac21abd7365b21a46bc357a06513123c8a758229708924fd55e467a");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "", Mac: "5c4799861d0066574908423a93e88af22d45e0dcf802b6254b2778e4955d88f7de915aeb3c21bf6d143069d3d7787c8d63c94c49ccbe0a7d5ed2055f4064cb7a0e3dc5ba3ded58af720968d641429fcebb958c6aab9d1f29cee91a4dc4c38dcb5a05514648559eeb");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "0102030405060708", Mac: "72de0b8c18ee00e1cd64ebb254afb9320db764108dd18c1f10d7a35247bf0d2db3a79d38dc2acabcc979075984cd44620e2058e945a691c18870839ca267d2a99db54a522f2e99962014f98bd78974c0e762725bee117de72232f3d337d7efe6802c652edfbe6f7e");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "ee9e6f45e54b27a0263f7d9a53a0eee08941fd0b58b4c942b47d8adcdac8a37a26ce774150df6c9ff2192ff0b11b297f1fe5e26a49bd6737c2bf32927d73217cfeda112167405e78ef98adab53e7311f6ec3bcde770ed593e4cac3154c4800ceda69486d5f67b517");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "1d20f0c4821257edc2b8dfde3cbd327519541b12460278430ed73f2addc264d67367a42d42c843b8cb01560963329655517be154d8ab15b9d9bf5ce0508d969d3786bb8f5539d54679a5c5af22645d7f673a57fb76ab5f3b3892d2667335026c68154d45af2e4778");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "", Mac: "3ea417bc006c0f0213a0e039951b0ffdf357b85cd68b1c89e351dbebcf233bdf90cf2986afa616db2fd2a148d1e1ec8c89bd59bbec1af49f5a136aeee8a4935e0c92428ccf8f30faf6363feeac931c93fbfd3aaab84cfd776a81a6d6dcce5f7fa5926251b44b22ea69b86c54fcb01f4721df6b63eef5bf9d");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "", Mac: "6d063e2f67e8a0a6c0040d47b6e38be695af64cf8204ce6638d9088dbae2b25b9b8bb28c787dd6edac4d606fa90d290cc9541b0c19f0310f9b0ff288a4146177671edc9d030ba17987594706c29341f245e8965b312f87d22a0ba4a078ec138c0a7599a2fdd034d584994e5f7e0bda05cce5e6d0232f444a");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "", Mac: "68d48cf3e2b739826ae5cc8290efb1b4d6da9db0eecccfb639e64891d31bf995195035193386892b8672b5cdd5e066b472d23effd7904082be191c28247f8604a59d727695e868353cd8b0fd1943e6ee2f5b9f4c7083c7e8ab603b03f16a3d9b7003d09c831447b79044677342ee64f61bdf135e0ce29c6a");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "0102030405060708", Mac: "203283ff5fe75333099071113663b0fff4b5ee686e96c3d0d87cbd6ef0a9d12053f94d114b4536fde29e2befe6c83bb3ed9688d7e8e9e543ef26bd279a212b597f9c38829a33b8831c0f9889347aea347bbe4e2cdf4af606313ccbcbaa527c75426328fa619572183ca56eb034e206b2eec10d30b19650bb");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "8b2afc7151d1b6bfc6407cd668b12a42999584ad92a21b81e1053a91cd7003e39545d3c61896d31d98e3cfb68b76826a75911bd8194601aa6c04d09728130a368dfc0f2a1c39efb0c1ff2628f95da4c2c064a6c34ccbe09cb7860c4f6e0523048aef878a2d0c8b70aa59e348370f76a44d4433ffeb4a8ba8");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "e68dc56d70cdf4546f0a519fc3578a8180f220c57f27cd82eb47ac70eb14178a74f6ca3c3cde2ce7adff86c07c134fadf2c6e3ad3ef0bc87dd4bf102777f0aa686c53629f6c04d9068ff08489355eb5e57b520c8b6d7ce7f364eb828d0a4a260ea805452605b0017b236fd0a2a1bf900ed92abb57f02d6ec");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "", Mac: "789b9f0553d8b9f4b815f31daae2f16c064b8e1c8e3205a2cf3f014882a8ffd2de611014eb2d1d3a475135f1e3c16a07df8f179eb7dcd46415dd013a541e5429c8f9b079ecadeb3294a559666aec9bb082e49fb379a465758381bb6cf0f21834136eddb3687b0c0706bbf127fb1f658c1c5d836999977f821cedfbe32a340a1945913e4ecfcd0135");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "", Mac: "a56afb22d261b8d8a5f41ea407cad80222b8323d8290f25c4f876382793db02161176df9323fe167950dbf1cac9e89102385a5ebbe752b5f042f94236435c7288263fd5b7eb29f57462c02a1a4ac38e17716be073610275922d5e0883be6e57af775d0da54ee8ba08aec4a980ad80b28ee44df1cc6de5fd3cb68faca1634f5002236294802d9e9b0");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "", Mac: "cd13542ffad9f55cd95159f12b9bdeed4f35d9e99d13d43a436879635c6053420a171818c7ec2c148650ffba0c443c83cf7fc461671a2a206e1d8655847cbcfae9c9d5d3ba5b3ad0c6e3983626c58ef74bb5097dc7abd41707058a49753b04ea55056d6ee2d42364c0231376b36b67e4c8656859762afebd5d7d204c575721c949f9feede1996b6d");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "0102030405060708", Mac: "186042a9385f5525e94fef92f2ce64fd963a8798ff53cddb5a51014f4715d565e3dcbc95b3dbd86e69f7a2b4e7086d53af6e74e426834996a901f5bbfdcaddc9b59308e2fb884c9ff8fb23f3a544522998a0fd6d595432d4dff92b3614b5ef74a94dcf0d620cc2cc9e481c022953fc7ae8ba4b318431e658a40f9e92b0d5bd89c3022f915cded01e");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "6f34d838546be944180b28bcead5c142f436fc53ec7289da4332c8bc2b6a7887892eb9c2bb2009e7ffb8afced0ed4512b862abf01d1e23bf9ded80ff50c2aa0d3247aa2cef1c146f0feebc5ae9587d65f1594fbad9e9fb8c8aa39aeadf73e7af7f199ee3f2c8a1cec02d11300d9c3d4e13a33c034a3a969331bd6f09a735620d3896913fe783e514");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "3157c6d14ae60c6f972d4df130a2c435203229a54586b4c9d964f37aeea0091af4f7cb9c5ca8bf766fc353e96128fd374387ec0e6000dc7f2d3cd87902d609406d458e4bcaa59fb2a5f703a84d30a10eed896e4955b25fd939576ec6438fcc936e0948f321bdb58ecfc31b05784fc85cb61ca08011b6aa4478e2c81669daed8f6021b98c0c2b8044");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "", Mac: "04f654b0d3827982b0bce94135b38a9ec66cce612023ecc4c6d5b45d345f326372ef3998da210900a913d3040eb8037b954c06725992f20272cf7e112e9be6af3624556a1359b5b4e4aa74f0e379f99d1c57bf2b4b9f3f3044a2f41af5e020c432b50483138801c86fbe4a957b08b4b7fa9a38cfbf3e813abb1b9a0394015d224c2e08335382da0cc928638ee923e500f58d6801be859f51");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "", Mac: "d313dcd83b919067014ec4ed2fda3e1387dc14a0fdc322b850734aaa305d6177801b646d89fb11c6d179ca59f9107aa81348dc67802043cc8491186d74626c03b1d7b68244622e39e7f0cd6c0b4029fe5489894473e2c2beb9386f1057397e7d796891d7ab24b74bd924f55f5286cc5c3b38d3d0eb8a7afe7dd11949c313cadc8ee56452960719ce7428d40e13a659ce8c784d41f79b08a5");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "", Mac: "bd8503b14eef7a7bab7b20f77a0729f30c6dcc5ed63f04f30cdb8dbd0f4facb0545c6da29455f18cd8153ba41dc4d17133c81d62074caf2ac1ea9c66b8304322730d67dcc83a0db2df212256a278575f4a3431c470a63673e9f166d697e1c08a1c811db8bbecc717f84ab35d5d56ee0070a6aeab580c34d3978a50f69b12988b1c05c75d19882bac50428565950257c7419c5bbbd9d124ce");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "0102030405060708", Mac: "2a82e124cc504d94c89ebe9ab231484a625fd9abb4632da6776632685384f45612bebd86209aca18c312a49e131008cc225a4774650e0483b95a408c7938c694e780f6c52df04543b3095c733c07bf33fd0622c599b9cb8650530fb610a7f192b8fe0d1b00a4f8673ee2c23f4fb7a5e4ef4cd13a3e2db851ac98c74a1fbee07102335b076755006050163ecc1511a0b611d23e634a1a3e2f");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "c277b6e7ec97855338edb91dd7b2cec7b25fa87fdf6c06e94b475748d178e0cebe5fc676858e6ae99540134bf452d6e8f86749d23e059964aa9d020ef10d2d9d9ce241e6be8e16111adce97ee44745c69c23ffa5a5c5ded4a83c1290fd560f16871a458c9139d62b949ccbd4c4866a2901af79e176a6f46d043de5156587fa89566a6d09afa147d43bcc6adb9cb0cea652d0cf36e1827756");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "2122d62c532a8bf0c5bfa87322163b58dd54f5ef654147e175cc118a568c6552e156e3ca3d43fb33b144a5c59efbfa876ba6314bd5d4f6470c5efcfd1dda1e120f5bcdcfd23f383ffd02643df0115861849acc00aa4f9f434be66c4a0e27df2daf98b33b0f05b6cad212daf4404917f0903f0ab8e01399bd9e04b6842713a91e89163abea0489b5d784f3c16169e9bc02c39400360751bb5");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "", Mac: "78a4ac487c4a4b1265e812a8cd176771b203e8884e2c25d98cc101506d5a8e25ae59c824613be5dbc7f95c087f0cf48c826658d67ef5b896c53bf99e90ee703e51d8e031ced46dfaa4fbc659abd95452e469264021f0a3ab564a6796249dd56cac651df4945f076dce5fa18d7fbb185de8cb37bc100bc4009ecd038b81716e7340f7080db14de1fdf6b9a622ae713fb86e9ed9b87fb042c3fe5903b95764a64b51e9a41094ac3167");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "", Mac: "04a534a986ad99dbb2b452af3396062e436c44c2002a45bdb8a9a6d28da8a49acea7f4dbf216306a8cfa565a89b5298d7e89725c4cc7beadc9494a4eb33f78801699ac0cac67e0b28267fb55b0dda51e694d661430b15e4feb96e89433168dc4e24039fc0f90db2fdfeaf4080f6c6ae7dd76b672cc6b6cce39da545229f3d8675ee985cd6f7b9e0fe1a2e23a43df18df15afd6782fbb55b108c1f470da01e7939e1baab5fc19dd85");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "", Mac: "4f487f4212eb33eb6a6d2840fd506339a5a5c8905303b1e5168b67de1f27ea11869acc9d9be791e5d46295c0eabc88f71231b70d2d266ab1089286dc06c346908792ae845c2666d3ca5e3385df3e888626d10ea3feaf0e2402c1cedb231737561ca4d6cf7d9f6168300816d8ed921e84030692b03f9d05ae732c5a5957e71f47593f3ba012e2039af61922237f3f46318fa1784bc7a43eb1702b0130a0b16d6c0f31698c050631ef");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "0102030405060708", Mac: "d3ff7e00e3c2b45a350aa27f23966876647f06a9f7cd482474b8d700e5f15827278abdde20353f05b92d37c3e9445b071ef7605c2cd695b4b309fccb3a124e0e0af79bce3352156f80d3223ea2d3cc44a43e15f416eec11f6385023c24890f3ceca443691daba9857451bcb37d9462cffb2c4382be9860cb28281883ef053ea58618e2c939a3c7c3a8aad4b4d2bb8bd7a4a6568b7ca6c10380f6bbe867416ec166f60fe6e75e4d14");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "110f66c1b325a9262c4473ef6f7c2dda25c5658b83f0c95fcf02a5e58e14246298f2c2b5285e0d817d7a0e79d127bd240b783815cfa7f62e9ac994c1c6c76be976b250942efbdca29331fa295092a75bed0311dd2827e0eba20d73c7d25a2a7a5ac8307fbdbc790248199b31f1c9ce686e87cb2cba9aa4fd25fe0bced9ddeef6fd4a6ad4b636e800044964012f65a5b918d230f3ee580e937a384472cec722e349c75f805ca03863");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "7e0fc904536050a13d6b17f92f65e40e578c72b9ec4eb46925d369831158d3cf4b0f30a8d28faa536be73f2ca52fdfee43453f5398051ba51946e54b0f79a185fb9ddab968c08182dd39da5f21e13f8e3b5893be1925c5de3c5dd93f7a0f209ef02bb89647fafc34ea0df29d14834409c1d9f9c3b5780c1838e9099ee3c3f63dcbad95831832935f9d7c8e720e96dcd1c5d7f11b84d36487345c49ebceee3634c745f57e9a4d8125");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "", Mac: "9d64e44337f8cf27cb32d7bbb1cc44956ccc16f5344899854766b09bcad92d9fd35ec74a7408798e26b5eadf39dfec3cbcff8a9b3a6fe485eb7ad24d9a57d488bb865ae9865b0e78dbd087008f50f24d8231dc541b9234b79a7760e39726db3209c3cd9368d68c478efd617c839cf121f1707218dcfd94fc1a22ec8558ea06edc244da860e28a559fa56a2eefa189c9c6337da07fedc8f349bfdc89628bdf010b80217bf5f5c4a9eaa6ed436f53239567a58d2046bde569d");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "", Mac: "fd0df263eacc893a9a3f10e6668b187c8d0e81112ab17841e91850badef65ed6089f80dca61e0787e7fefa2e5270a0bd6713275a3378968b08e9d61f2a3b4e9c9ecc902a8b5f0311a3d27d8c1a6397c5e866f8f6eef75dd6f44dbc8120062c1c5e12b60dd1d4517797db8168af9d6fb1f6146b2896a0e919bff516035a7a784312d1764cde4cc66c95330e948bd43ef41e9659ad2278bcc5c3446a734717587a4b969c36d9477fd45523df76862f258320c6b5f81d86afeb");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "", Mac: "ee281cf2cd4bf6abfda232007e5a9d49805d42fb9bdeed7326b5390290cb04e4840c9ce37c6f5caf84d947c8a5195073f84738216b1f7d9863f271df42931ffd1ebf8777e6f063ed82566198697f2905eeb389f487c0ce48d296b86461be22223a4e481774accbf940451f46ed7fc06e19eb07e2d7e40933fc2937b78f0aa8b3ffac67ddea8e60665c625f70c95c6015ca1781c71e7c49836b7b2f25b8ef41f756baad0074e8ad5e5bce3a9009756fb52f285053610c8190");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "0102030405060708", Mac: "9589873a52e89cf8f1a11f4638bc7f285fd1d518f59ad8cba947a6f629ef7fc7f9738802cf56a1a6191488156633c515efccca55a7442f06e0d10ee31e759fee772fef8864e3e5648a029ceb59db31769303136326a7dcca46191bdbd98bfc6c1c401cb15ba3421eeb0c1718b16dc162a8985b6de3eab92682ecc7e78a162683bfcc80caffb8cdc335e11d29e997c5afd5c7eae827b6dc0fc82734a72e3fbce6f04d7f92614ff39a22eb76b17d329322b8baaa15c09b3647");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "d4852bb0eda4320e4015133aafa140069f52f24aba9d845ff577c18cda13cf043d1a07b2eb9f0232ffd06ff729b713b8a6a9b3c608d845f971ff58f8f6015e2d7cb13a4012ec5778c03ec302074317a8d5ca3e8bf1ac331c93bb103ff59eeece6324af9b799d664959fbf62274e5e948933e9e57b70418ec4f6c39e390aac6c0a7daae47a989e7933e9ee214b6e4b3c0026d58a8f1254abe51c420443110eebd958e2a996928f2efa330e03613f6ffd0b8d8680d5d7e5d85");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "f1625d14fa4cdcdb51d73b8ebe370843e367fa776df12b98ca496c17276eafe3b74756c685d1907c959bc39b640161715e2641052d974cf0ab9efd780b228462c75b031d854de3419ecdd8cacf47f6b4f0805d9e97b41d38637b3784665166f786b3b1581816c6ba326a6a8c7e1ee8252b451aff85846e949756f063aded38c9293442313b9684883c30727b826ebcb0b15fb6e13b289fdde4ab2c40186478187652368471e7d7651e9b20358aed0a2854a7a969996cca3c");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "", Mac: "9e9b860f449b0278d43e3a5891355d4a6028641709ca22b1477b6bc77f77ef17ec78c811c4cdb982d4de2dbc73b70528005ba15378bc163927c845603000f620a016706abd286128dcef80c3e13c954dfabc85ca6c4ecd99cc711efdd2b411d4ae2b85d268ec065fb6276fbcf07a86e345538d5428c7123d309f65b7c599f3e60f90a2ea7cb30caee4fb041c562de8b7a684942c2c3f7c148fdb0bd197d8fb05831f1baf8aa566c946631f72646b1e4140eaf7fe0f00ba13b6bf129879627a6ee337df4c784048f3");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "", Mac: "16c3e3a30f798863cf3342f85202edd0391cb375ff22cc72c53b458cbf927d17f35fb78a769ccba7090d82a4db904c39c63c4ca159beaf9653ea8a4f2aa29f9dddbbca0f7c857c5350ad0a7a4732ff42df74488fffe53f7d12d4d69b65e3422a054d99fa981f64f93bad0b5f635857456568b6055b001798a6cfc4a6c8cb4934892bd2cc337df6cae217237e6592bb045727e72dad090c70578206ae1794ff23281d840492ac2c000f759e05df47c261f4ad845fe4e2bbcfab3dfd5ce322b95c9639a392463317fd");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "", Mac: "d1956b3e4859afa135b7f1731b02e20fac56e30ab2c284584fe0ca9806549aab991446a1617b0a9c2bcc4ff879feba39e5c047ed86c522e2c7e109c2f15c3f20847dc462a1d66e96ed1a530f3d774a62edc8b4449420508252077f7f2487b3a19298de3440b98a9d92aa4a0b28ce2f70ffa28285ca5d1f3ef9ce99b20b8c953893ec5c22caa62e2ce943f99b89fb7a869b4da8b94b0412b82cfa4ce1c0ed4d13212986f208d4e48ce50c9019f7cd319eab1e046d0529d7a885f7c89f0835e0f70d5ed307788e34ae");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "0102030405060708", Mac: "f029af18eebecf39a043a918a5bb9e4cb8dc3e4e7bb701fb05c8d0dd5d90a68727152c9d3e2164ee4054d0685c3d22f98d5d5a9e6fc6ca2dfaa0324ec8d4914f4a4c2c53cff4e01f6dcd8326dd3f9e5df3cce5dd0235f3c85d1492e954def7e52e7c2bfdd8319e30732d851c85b9490a052a978eaef2478efeb447e27b23038fe8ec83b3753f9b65f183eea788acfd4b1e7d6ea209650c9bf16599457a6004c161facadd9f950115d1b892c278c96cd5543a1dd1566b3c92d4a6da3697a15a05ffe330c157e2d66a");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "9fe8449df5d2f0948c2ea04f185fb3455b4ab1a8a4de7e6804e0e318c4dd92cfd8a87046e233bc701842970fe32aa7750736eb9548a29e68e84781bda407eeb19668d4f882b5bef1464803321f664bb8a879b2d049e6f8e063bb00fa8e5674128426a87a094f4a9373f6859dd0e3b2ef855bca98bf013d28f725762ab4a61f6dcb18e4a51ff30b286ab3eda1ef6b5b98c09c2a3302e74c0603d4153456b931b719ee5ab304af20a00b2bf88fb5dff9f1208adf95a4f5f1f39d3cebab9e45ed7e69a30520fc68a383");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "217c5229005559eb143d6d4a183f9a2c049bb404c9931bbb7b8dadd1ac83b954ff9408c413b225f7f71ee80d66ea16ad370527763c76fe1bc45c3ae03c442b4f276cefc50c5425183c3ec700d05d955e9d19c1a29dba017445fc58ff628cc77ac60a607ba4c356255b107eb12c8d622a2d50c205a4c9f50b3492a075b18663eae686d2f590d8070d056299624bc36fc74252e27b55f5bdd78b040270b2a8f80c50fa7ac6e5336593a9838256fbae6932226eab5b8f48eca7fa2fd315e1c124f1f09c812600b0bcb5");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "", Mac: "70584c22bc9855620ee30f39f7b2f097a45f8100f7140c48b4e73640068f6815594354a499d5fc850e47b7ccf03730aafbdb2b7c4abe27237b1b7166db7781a9bddbea57e8144f0f6f4932c8ff93916780689c90004c1f9f276f6e99b25f6bc50a6a4ef530ba2d1daff31d813d242a1f38a0bcc2f51becdb723fdc8511d2a63ae4bbf19f2816516e87b850a9480dcc6fddc42e46af0f670722ccdcc60a278cf123090f2b20fc6cf1311d795e640c17669ac3e26f4ebe591ae8a4116fa40bd5c4bb63d1976a796405d6e7969952090c5655e3e4dc4458703c");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "", Mac: "3703007ef6ca08006717dc8f923aa0c93517984c0eb2c9948726d6403737a625044205922de399627a4444b7823de0c75ed4e44abdabf2d1990e561244bb0940308563e0fbb97526082d21b0b3f3443e4dafd5668197f918892b82eb31f6c3de50b239ed4dde71d5b143206758955af23d0219ab526c4a0840f03393739fbc3b211f2de0fef60c07449cb5446df43ef29a002f6f76b1dcb455ce27df2d76d8692129115aa0df8064ff0688f00008d23db892e3f7b438418be0a22cfac0829846a2034793c54215ae4288877a175d5e5b4d1d3209d03de449");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "", Mac: "9d5ca5ab86d4c832cff24642657da7ebbb6ed4bcec59f48bd799b07afee9ff8aeaa270e82e48512dcd11aaac3111c04e1c10a3992d24042945635b05faff7aab340f90e17ccd3b3f4b10e45eee894fd87e939e39ae25392b97e241184ac4797b3266d336cf0b253990fedd1555952690342212a475ea5a3e4be0d40d913b21a6ee18a9020bf4cb95e45dc695d71dd54d38dc0a511ef15093f2027d79c4f31b1bcdda090a6158d7551c24bc0a71763653b676a4fe0b29b885a0fa83c8c9cea5041c1ed083b718b0aed4b56ad1208d7d7401b9b47c6003a647");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "0102030405060708", Mac: "3478d9b36da3ba003c2a7e5868bb90af90a7ae36c15dfb0a0ae698f60f0368b74a657a22a66e093b5c144e02a179203c3b1544274c5afd3ba72546c11762599777a1e0cbc1504c57155df440a8027707be31c0b2a433f3af81705e8ae808aa1874174e9d86a5baf6a17fcf8376550137f697f44d6d208ae487cd3035bb61b756a7dbee846fbd89da061486290dc329e2738892846b9f59bcdcf1ad9fc4802ed88956db3048ae0447f3269091c299edaaf0cc267c13ec79aa7e6446fdfe24d749d60248e436a233b46822740e96625d5d3d0005b73e3b8ff7");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "a43c0790800ab6fe726110d3f4d45cf8fde67d71cb00b24fd52bac50aa802ec48c3c5993a03573b8f3c4949a780fd6ef10af4e3d8a95361ced0cdfccce76d1b04aff42b72dd4db0656f866a6ecbcfac49cd4535bee3c591286e46c821434acf0eb4de59026c846153a4045845996d7d1b14da1d4ff7faa7e7bb8715bb4943c1246a6b81c53b6cae9ffeafbd0ebd056728465ebde4faf61c5fd2e2f712940dfccb0888158b7c91f86dbd30809a63f1a3f2a8892efb2d211db4d34f7df8bb09c7757b52e3c0307a82309954c203299123f0b3f77100438333c");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "e0e6dba8b037fcf2a9d863aef33da21beda324d10320c3ebc06a384c318ad14aa93cf8eef11651221d514fdb1f1156d5968d821c36fece33d06c4164f8ddab16487713d08277b68477f216dbd149a389ae9230638fc14beb4aa78e9a14555dfc04de24f34422824c35da0cd47f9d98e2b288f08c1df4a1463ec9347cbd69e5a05f0304e1833dbead9a37e3b86f2a1261629abb99dc8e4cb4d02517baa76aec003f2866c46e76fd4d3ff4ea2020e0804de81c9b1a26d80abd7bdb5bb38c8a412549ac34723fb9491b13b3e0e8c1be2d1580951a260523fc56");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "", Mac: "99375e488e66b01a1671f9176d0fd3de0e3d185cfbe28b13ac0d1a992e0357924a983c64551d9f6e8bff7b7790595e5ee367a01f1e5acf76fcd115963201b2cf7ec26a5395050bd53328c879dffdc1f465702fb465083ac660e8b2f2edcb2a9367463d20f7756b96e47e5789a792478e37a971d588565f8b4bed2d06a27a8bc103452cd322a35cd3edab34c76761f782621cdecf8dfa60ebc61aa3a3b0cfd9246c4acc2b5beb2efbefefc30b1bf1f3a6d9b6a9bf49e0411da248cd9978a6e744c10d342cd6e8ad6db21c3260c047df31cd3a29f0152c7f7e7452f05ca3cb7ce701d43e58afe99e8e");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "", Mac: "e515bdb292a2ff0fd1a6996d5f00eb19bad0e1fd4552f2d52124d4c7a157f561e3807ed406039a6594ba3f1f96dc111323ebc34ecfb6a50ef01dcb55fb05b024584af73bb5b2392903d3cf0e9be5322792c53df3aea34528f812286bba7c689c59eb72724c2c16cc2eb407bd7563cbb4be2f1829c0c1b68b5c7647d2ffa4f1c187ef78ed10f0f30246f1d49257d71e1b0fcd3e0d36dd967b1a1cf105697f267137adbe645849e5574be303abcd530175caf4ada107ae67d62a467bc2b0f9e8453049f7d6e8e9a41e1a135757545c2a72f3650f301a2c54c841ffdfb4cfdc9dc7cc009e1a726e3848");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "", Mac: "bf52205841f752c929eb629794f87cc2d4af2bdb805541919ac3e409ff912797e2b712b97f00c9684b516b8d294e61c14a6d7cc776680f019a95b291bfb578288cd64401901f47a1aed9a42182430b2303e9392dc75a95aa3831eb8f956c0897ff2fdb8d4e9af7dd0d90dc48238791b2369550128ba70493b8c8e0402f692d155562c088379e66f7b57518321cef323c9b8ce3a461c9680e1e541defab49fbdc356e26e572739a1539615fbb1b5584d1bdecfddb5ee23afbcf9c44dcb5a92443d3a6768df6b80f40177b73cf339cc52876ff21ca60e0ab6f0ee7d39c25ff3ad7f356472885b1a4f5");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "0102030405060708", Mac: "11416f707ec4051902515632d5c95de78bd08236560d8567addb170197b1bb99b176494c980aa4694d1adfe30229cba1553333d3a378e4a7f10f9f3157d4d460aa72fd948c2abdded27e2d9a232bf0268ab4fd6f785f490c56bd0c33c60e096f4813b847c355dee56f788324a1e7e3308b99453cf8533e93aaf0481bdfae89acba8acac97623d6b902e93d933d300acc094c488c3c77eeaf3c1b95b151359db0c2668c0a399575e3e4d7c7929485f4a882138118d4c3c5c303900e14133380b610b08d58517a5f85f51f8478af9e6e224ae454c0592fe6fad1aa962a9b6cec3f9786869b14178844");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "ff90b4349bce32d3d6f7d4381e8e34e0e99e31c2bb33f4ca3007014339da868f2a0118a55de08d826702c8b2321e40316b828e56a31a7bafccc9dff5334eb46168364bad2efbc729d9f6b85c9c96da6572a1cc6ecdaa35997d36cf1131480a29bac9338d00936e849016c3730d5e94a4756a4d7cbc426f6b2d42116f6ac39acee1429ae24f893e7609ee7e57fb78e7d17d87e5f8dad01742be238e60969a5214baadbe4f8c6ff8209b89bbdfce3a49f6fb7a30f62e9ca79e64f87413d0a79c68b3245d3808181a9e7c7d34eb8ca5233b1205d115482fc04a3ff6f174617babea4c954c951316f209");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "16353d2d8cc29d16d53eafddd189616c0f3b9673311cfa53d293b44b592614bbf49d4de2a426aeff5c5354cb55ecf157d9418d5f242d1d785033929aaddf2c678577c60dcdd93ff058ed8542b522e48eab6f53e5becf9d3db650809382e0dc1d10700e826928ab23d172e907079ae8b1237971ab643bedbf53968cf1571667148e493956204f25a2a8c9bdddc6ed8c802477896a44835aaa0b4cf7008794b063cfe26b8f2ab031ba699bc29deec68d6967d7eec7564fa9b2bd57d567392ad72c8e847dbfb51192fec9dd8cc66fde0d6332c5fa6268107dca4354435eb775aba279ec841ebc22ca9a");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "", Mac: "41fea40cd7dad31de7913f7803ba7d6b91f54341541b38d75d762a655b3d39d37606fe8ff28a796bba7db7c97b9634ba5d9d43149a2c97cc740e8eae25305f31b50503d2e4efbcb7f2b61808b97ff3c572c779ef607c698a2e2f58a4ea48721e26f49b3f9e488c89596235231c3211dad294f8f35b5a803fc0c931d1ca8694c2264b009e18a454c853712724031c0741d86506b82814477ea399d24d4816040d9a7475a94b7ca46b1bcb07cc80693f7a46b3be9d3dc999216e79170275b4847d51b25ede110602ee03e6d81435c4f12e516a964aeda49198b44730e0b7be01d8d5b8a8c9df326bd2b610f78c1ddac92ba1bc99e2df3ecc52");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "", Mac: "53804051a48cc96b3685315e5518c5e16ffe92a0777f1ce1d6e32733428dba6fbee5a682814e128d031bdbb0ab4d62b0c2000baf5565c084556ffb42f6f5de0e868d565eedc4597189ea0c1e5a9ad1c7fe1f88b5d5f00a5694f186a514014e69202b4acd670f160c342197da874932d02c7cdacd81d8618ef5e82a363642d29ab552e5ef77a3ef5e91420c5ee1c81388fdf4bfc6d6539cb8097ebae9fa3495df019c61f18d9207e446fea122dae64315b1b442d9ee314f8abbf4f188d836d32a9e4adaf2d666b52d22248a69f94db38cb467f8492edcc9e054d94f6968b186c8ae7c78f7c882e909021ea443f784aa5635741cbefdaa54bf");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "", Mac: "80e09eda04f45579395772f6c755e84b84688599dab06dc49025670788a2f17dd34462b6334bb078453311e21a1f6d7d8475d14efee3437ef75509bafd0dcbbc15083d6c5b598c18a1804487eb11924fd706fa9197431b26e018a65b52a8ca91a08b1ee322549dc3189eadd0c03b0ae33ad92b1d20a481d93d5133648cf280a98a36afd60ad620ebf7e6e713b00d97c8f61354299c685aaaf6a64272a975939281900fa9366a33c24796a81a3c2baa5f5d887a8f5c8384777bfa9bd3a711e2adb532196be6c2eea136463d0b46e7eefa53e09ef3d3dfa440eb71aefa7ad94d0cf1df15e7d23e9678dde6be4cd11d70233da4771322e5e54c");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "0102030405060708", Mac: "7217d07e4768411d6606ece09599dbf3418bdd63e737a3afde449da2fb993dc817e27abc1f0d6e179e02540d662883a86e2508365fd3a4493cb55f3a917bcc652cb445244e527dc0cb788ebf97e329101968a6e481f3f28dc4445ac57a2091433bae5e4f36559b2a8cf2775a341ac33134fcb4ff3614cbe8793ed042dd36e010bd0895e560d2b5d9f097f696cc2e0b9e849d6ccb3a44121005f00ea0c7a820de553b542ae38449cac51ba2df652ab90fb66eb7ab436a5949955bf0818f07747ce1df407a600e74bf66e4271439e311f43cca4c499552d1db7bc4eb7482828e17398027a4434a59a0c497310cfa5a01b7cf02c47ad9271250");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "3ad8403c74586c88c449ead188495b09f938496bab2d997e9bfd97a07a7de0afe3d98291b6d6df6d49b5e4b374910180e1356c177d91f69cdec69b91c0add5760dfb8a5cd6eb218256bae60bd88b994a521dd268870e417b597a57d61ada05cfd3e64daac84c6b21f844225c22c3db288db719f5c1aa4fae6e2a1d269ef08e3f5ec98d523fe55b0bacfd0a5d9241612a17761355c74a5404ac8f426db0b1b0ef738c2ac7d80bba6ee6053aaf88cb9f399234f7f85631c9c66226d079029e7a636dde388c63587ded7d515011b8d21b72b5b0c219c6efa4f5007cd132d99ef9f681c11eb4986944640de0bdd2bed32ee1cf0ef9427cb23bec");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "bc0478a8f3da87a227acb6f051f631fbadf848563e1b69ad479651d32e9dcdf5c8bdb0f74b171fa42d35900fc403b3df972c64691156c33dbbd4b6dfb08d4f6ecef62b7b15b8491e1d5209c7aca049a039909786ba273f2993ddd26d75ecec8cf7335e0c32e1166fc16dad68c87d9687fd2ea9058ced0ecf153c04be0533f3eeb7a694a11225ba42b8f469fb7dcb893c0044ef00907ea241f622eabca2291561d414dba2389d08f998cd6c52905a9e240e133a96bdc93e3a10e8458190d5a8aea3c06ad8f78e7cf2c7ca16ae2ae4875b29f910815e27742845ba24eba5c61b65f81e4b20434eb156b83d8ea371529153c5e11385eb5ebdd5");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "", Mac: "c1cdc71d29b726f1b39151ad1c240a1f047f4cd7b32c93c9e6fdfa00b963f21a8718c488359c204fefe446cab56cfb69da35c5cbcbae94e2b017de57fc234d8681c46d7e1ad33e8fe31c211fe555595657d470a48223502bc14526a57f55484a2aedc1f55f821b1b2be4874ab25f0307e486ea2f303442d6366791843aad05afde68025d4842192cbd2f3696d8daffc73cc8cb702b5c58f4b41f352057601ba2a968102e21c826928de825ccf3d3fabee6eb2493d668f048dba26477fdaffb86a1f7b38b99520fa2daafd32d122a2747d487b59168e3324d3c293d469095c0ebe4c96321bcf7ba2f89e2701456bc7a16b2fdd785131b44e4ed0628399c0753a2d67c1a0a0c7b1758");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "", Mac: "6c4da8bf8e36b3f3732b151c73f7bbe9207103f3683d244362dd3462ec4973f89c7582ccdd609aba1b92b55244ad756717ce509520c8654c2aa398179e1430ab887eb24f8a2bf1c3990fc2fffb02e330304965ca1c601d9f5544eaddfa74e7effda5a6900ca096d19e2c3ad1b73b13e809f07a368289a4d6ef4dd60d978e99373d224d16c9af6e2f96b8391a55ea8ad0b8c92802f48d40a9debb352b52fff24295d45e53206af89f281f6f9df5b3bd45a23692121d179b02b31722ecbc1235222d82a45a7f38a347064e48e219328ba5d7b945aa8d46d95fda9af08ad331525ca659cd0420ef51e022f9c48317f8475fbfa8027e32fcbcae443199587d356a6e3bb570df2c066955");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "", Mac: "ccb1f3b91ad1550d1e020a01bed2aa072939fdc46f8d26fd510ff79598f984bad35c3108362a8b57e53b36eab3bc68278563fd4a37d801a45457def371f535ce10d1d89a7ed89826d78906e5f37d92b81ec0b615239058950d5f778112cc1b7488933432df11b2a046d9500b5a8b177c62054bd82dc5735e1c5aa48546c233a121415fa9cc3f3a197488201a6c2bcb8d9b16eefcca3aef3ba65357a603ecaefe3de07acd4a249b85b42defebd50829081f6d1571ffa3e21fc8d6117c38bf2c560b64eb9e3a2764b841450fa3ca39cb5d600e61c8ca2838d004b041c3d524db32f8a9713f9c5bb46a14eb2757335179d5f893d050c64caa47cc15c37275821ff7772dc1783ddd7e22");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "0102030405060708", Mac: "676e21ba8a6821497354fd5752120e9274c92686f6cb3b291cbf5153827a3ff044cfcf933a0876afac57478d5b3833a2c22bd3373d7336757a74ed2ee929d2ca977267a70a62ac78a1b02d964bc20acf9ddd92575917da78d95fb269955d052c45236563722d3812d86efb11ac9f2b547454bcc9aa0de25d396da33b738c9dc522cee7d2b000ddd67e0b74639040a4340dd6a372680c0e7ee2472bec381c7261e2e89d602358e2f24549db905ddba113abe2b6e1567619bf0e7ac4a1f5239c4e64ade5cc2faeff19d81c0c350814711626df0246667fec8f3025ee6bb1908d9d384e7805b9536c46afab24e5f78083573cff8c3098a1524145d2080fd04e4ca97b0f126e973b5df2");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "a1c4867fa39ae9dff348668efde26e9927c6fc9eb15f12a035493eb860539d10c10174fbe04d77a299f9fe671b8cac33073353af2c770d61dcd0e9e3dea1b3d4104d686817893c713fdf0a9242337729936f656feeebbd4937a11136dba75aa729e1613cc466d52825adbacb6fc0f824eddfe858364d9fd019f4df248507d7a9c1fc5d5bbe716940f06a1dbe7b3aef2550cb13d106e128446753d669fdcf859074943bb32b6b1bd826aebbd0a55b72030250c89cbfbb779dc937a9658e9761f196abf8a46c09d81a82a178df5db25d8c290dac275a91410416319d1efaa5003cd600cc6af1ebbbd1b03c8918a849f1f413c19115ac0ab5db5235cde85a97e6887b5f427bae10aec8");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "7e57aa93294bae5a0f1b13665813ef1f1e2e09f2c22347dfc6f15e34562f7a7c557f2726b1d08708e2e1ea7fa30d6e73754c52d4de0e8920ae3dbb9dfdb665743311fcc0103107baf720e68329b62275be66ec0db71acc583390c3c4a44eaffabd39bf73bfce44182c1bf4d58e6ce90bcedcb24c69713092fdb6a28b64c8e5bb1bfb1d52d372650b12559ea9c5ad495f38595110fcc4091fb19012ffc5ef12c701964614293523820a7074d01de18cbdbb721da9738adfb2e2211ba1a86d84873299fd5095f252d367620748e2ae010def80fdc4e9d78d0d481846aca96d0cbb9cfb015e9e44523434715e7ff0a0b85673a357a5d3242274d3808348a4e01ab1a0e4b9082a0d6ea4");

                // NIST test vectors
                // From https://csrc.nist.gov/CSRC/media/Projects/Cryptographic-Standards-and-Guidelines/documents/examples/KMAC_samples.pdf
                yield return new NistKmacTestVector(Name: "Sample #4", Key:"404142434445464748494A4B4C4D4E4F505152535455565758595A5B5C5D5E5F", Msg: "00010203", CustomText: "My Tagged Application", Mac: "20C570C31346F703C9AC36C61C03CB64C3970D0CFC787E9B79599D273A68D2F7F69D4CC3DE9D104A351689F27CF6F5951F0103F33F4F24871024D9C27773A8DD");
                yield return new NistKmacTestVector(Name: "Sample #5", Key:"404142434445464748494A4B4C4D4E4F505152535455565758595A5B5C5D5E5F", Msg: "000102030405060708090A0B0C0D0E0F101112131415161718191A1B1C1D1E1F202122232425262728292A2B2C2D2E2F303132333435363738393A3B3C3D3E3F404142434445464748494A4B4C4D4E4F505152535455565758595A5B5C5D5E5F606162636465666768696A6B6C6D6E6F707172737475767778797A7B7C7D7E7F808182838485868788898A8B8C8D8E8F909192939495969798999A9B9C9D9E9FA0A1A2A3A4A5A6A7A8A9AAABACADAEAFB0B1B2B3B4B5B6B7B8B9BABBBCBDBEBFC0C1C2C3C4C5C6C7", CustomText: "", Mac: "75358CF39E41494E949707927CEE0AF20A3FF553904C86B08F21CC414BCFD691589D27CF5E15369CBBFF8B9A4C2EB17800855D0235FF635DA82533EC6B759B69");
                yield return new NistKmacTestVector(Name: "Sample #6", Key:"404142434445464748494A4B4C4D4E4F505152535455565758595A5B5C5D5E5F", Msg: "000102030405060708090A0B0C0D0E0F101112131415161718191A1B1C1D1E1F202122232425262728292A2B2C2D2E2F303132333435363738393A3B3C3D3E3F404142434445464748494A4B4C4D4E4F505152535455565758595A5B5C5D5E5F606162636465666768696A6B6C6D6E6F707172737475767778797A7B7C7D7E7F808182838485868788898A8B8C8D8E8F909192939495969798999A9B9C9D9E9FA0A1A2A3A4A5A6A7A8A9AAABACADAEAFB0B1B2B3B4B5B6B7B8B9BABBBCBDBEBFC0C1C2C3C4C5C6C7", CustomText: "My Tagged Application", Mac: "B58618F71F92E1D56C1B8C55DDD7CD188B97B4CA4D99831EB2699A837DA2E4D970FBACFDE50033AEA585F1A2708510C32D07880801BD182898FE476876FC8965");
            }
        }
    }
}
