;------------------------------------------------------------------------------
; returns free XMS
; In:	AH=8
; Out:	AX=size of largest free XMS block in kbytes
;	DX=total amount of free XMS in kbytes
;	BL=0 if ok
;	BL=080h -> function not implemented
;	BL=081h -> VDISK is detected
;	BL=0a0h -> all XMS is allocated

proc	xms_query_free_xms
	push	bx cx si ds

	mov	ax,cs
	mov	ds,ax

	mov	cx, [ xms_num_handles ]
	mov	bx, offset driver_end
	
	xor	ax, ax				; Contains largest free block.
	xor	dx, dx				; Contains total free XMS.

@@check_next:
	cmp	[ bx+xms_handle.used], 0
	jne	@@in_use
	
	mov	si, [ bx+xms_handle.xsize ]	; Get size.
	add	dx, si				; Update total amount.
	cmp	si, ax				; Is this block bigger than what we've found so far?
	jbe	@@not_larger
	
	mov	ax, si				; Larger, update.
	
@@in_use:
@@not_larger:
	add	bx, size xms_handle
	loop	@@check_next

	pop	ds si cx bx
	
	test	ax, ax		; Is there any free memory?
	jz	@@no_free_xms
	
	xor	bl, bl
	popf
	retf

@@no_free_xms:
	mov	bl, XMS_ALREADY_ALLOCATED
	popf
	retf
endp	xms_query_free_xms

;------------------------------------------------------------------------------
; allocates an XMS block
; In:	AH=9
;	DX=amount of XMS being requested in kbytes
; Out:	AX=1 if successful
;	  DX=handle
;	AX=0 if not successful
;	  BL=080h -> function not implemented
;	  BL=081h -> VDISK is detected
;	  BL=0a0h -> all XMS is allocated
;	  BL=0a1h -> no free handles left

proc	xms_alloc_xms
	push	bx cx si ds

	mov	ax,cs
	mov	ds,ax
	
	call	xms_find_free_block	; See if there's a free block.
	jc	@@no_free_memory	; If there isn't, fail.
	jmp	@@check_size
	
@@get_next_block:
	call	xms_find_next_free_block
	jc	@@no_free_memory
@@check_size:
	cmp	dx,[bx+xms_handle.xsize]	; check if it's large enough
	ja	@@get_next_block			; no, get next block

	mov	si,bx			; save handle address
	inc	[bx+xms_handle.used]	; this block is used from now on

@@find_handle:	
	call	xms_find_free_handle	; see if there's a blank handle
	jc	@@perfect_fit		; no, there isn't, alloc all mem left
	mov	ax,[si+xms_handle.xsize]	; get size of old block
	sub	ax,dx				; calculate resting memory
	jz	@@perfect_fit			; if it fits perfectly, go on
	mov	[bx+xms_handle.xsize],ax	; store sizes of new blocks
	mov	[si+xms_handle.xsize],dx
	mov	ax,[si+xms_handle.xbase]	; get base address of old block
	add	ax,dx				; calculate new base address
	mov	[bx+xms_handle.xbase],ax	; store it in new handle

@@perfect_fit:
	mov	dx, si				; Return handle in DX.

	;; Success.
	pop	ds si cx bx
	mov	ax,1
	xor	bl,bl
	popf
	retf
	
@@no_free_memory:
	;; If no memory was asked for, just allocate a handle.
	test	dx, dx
	jz	@@zero_size_allocation
	
	mov	al, XMS_ALREADY_ALLOCATED
	jmp	@@failure_epilogue

@@zero_size_allocation:
	call	xms_find_free_handle	; see if there's a blank handle
	jc	@@no_handles_left	; No, there isn't, fail.

	mov	si, bx			; Save handle address.
	
	;; We have the handle. Mark it as used.
	inc	[ bx + xms_handle.used ]
	jmp	@@perfect_fit

@@no_handles_left:
	mov	al, XMS_NO_HANDLE_LEFT
	
@@failure_epilogue:	
	pop	ds si cx bx
	mov	bl, al
	xor	ax, ax
	popf
	retf

endp	xms_alloc_xms

;------------------------------------------------------------------------------
; returns XMS handle information
; In:	AH=0eh
;	DX=XMS block handle
; Out:	AX=1 if successful
;	  BH=block's lock count
;	  BL=number of free XMS handles
;	  DX=block's length in kbytes
;	AX=0 if not successful
;	  BL=080h -> function not implemented
;	  BL=081h -> VDISK is detected
;	  BL=0a2h -> handle is invalid

proc	xms_get_handle_info
	call 	xms_handle_valid
	jc	@@not_valid

	push 	ds cx		; Save ds and cx
	mov	ax, cs		; and setup ds to this segment.
	mov	ds, ax

	push	dx		; Save handle for later.

	xor	ax, ax		; ax is number of free handles.

	;; Setup for loop.
	mov	bx, offset driver_end
	mov	cx, [ xms_num_handles ]
	
@@look_again:
	cmp	[ bx + xms_handle.used ], 0	; In use?
	jne	@@add_some			; Yes, go to next.
	inc	ax				; No, one more free handle.

@@add_some:
	add	bx, size xms_handle
	loop	@@look_again

	;;  Now ax contains number of free handles.
	
	pop 	bx 				; Get handle saved earlier.
	mov	dx, [ bx+xms_handle.xsize ]	; Store block size.
	mov	bh, [ bx+xms_handle.used ]	; Store lock count.
	dec	bh
	
	cmp	ax, 100h	; Make sure that we don't overflow bl.
	jb	@@less_than_256
	mov	al, 0ffh
@@less_than_256:	
	mov	bl, al				; Store number of free handles.

	pop	cx ds		; Restore.
	mov	ax, 1		; Success.

@@not_valid:	
	popf
	retf

endp	xms_get_handle_info

;------------------------------------------------------------------------------
; reallocates an XMS block. only supports shrinking.
; In:	AH=0fh
;	BX=new size for the XMS block in kbytes
;	DX=unlocked XMS handle
; Out:	AX=1 if successful
;	AX=0 if not successful
;	  BL=080h -> function not implemented
;	  BL=081h -> VDISK is detected
;	  BL=0a0h -> all XMS is allocated
;	  BL=0a1h -> all handles are in use
;	  BL=0a2h -> invalid handle
;	  BL=0abh -> block is locked

proc	xms_realloc_xms
	call	xms_handle_valid
	jc	@@xms_realloc_not_valid
	
	push	bx dx si ds
	mov	ax,cs
	mov	ds,ax

	xchg	bx,dx
	cmp	dx,[bx+xms_handle.xsize]
	jbe	@@shrink_it

@@no_xms_handles_left:
	pop	ds si dx bx
	xor	ax,ax
	mov	bl,XMS_NO_HANDLE_LEFT		; simulate a "no handle" error
	popf
	retf

@@shrink_it:
	mov	si,bx
	call	xms_find_free_handle		; get blank handle
	jc	@@no_xms_handles_left		; return if there's an error
	mov	ax,[si+xms_handle.xsize]	; get old size
	mov	[si+xms_handle.xsize],dx
	sub	ax,dx				; calculate what's left over
	jz	@@dont_need_handle		; skip if we don't need it
	add	dx,[si+xms_handle.xbase]	; calculate new base address
	mov	[bx+xms_handle.xbase],dx	; store it
	mov	[bx+xms_handle.xsize],ax	; store size
	mov	[ bx+xms_handle.used ], 0	; Block is not locked nor used.
@@dont_need_handle:
	pop	ds si dx bx
	mov	ax,1
	xor	bl,bl
	
@@xms_realloc_not_valid:	
	popf
	retf
endp	xms_realloc_xms

