/* *******************************************************************
FC: File compare

This program implements Paul Heckel's algorithm from the
Communications of the Association for Computing Machinery,
April 1978 for detecting the differences between two files.
This algorithm has the advantage over more commonly used compare
algorithms that it is fast and can detect differences of an
arbitrary number of lines.
It reads both files twice if there are differences.
**********************************************************************
A design limitation of this program is that only the first
MAX_LINES lines are compared; the remaining lines are ignored.
********************************************************************* */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <io.h>
#include <dir.h>
#include <dos.h>
#include <sys\stat.h>

#define MAX_LINES	 16300
#define HASH_BUCKETS	 (2*MAX_LINES)	/* Bigger is better */
#define BINARY_MAX_DIFF	 20  /* Bytes */
#define STANDARD_CONTEXT 1   /* Lines */
#define NO_CONTEXT	 0   /* Lines */

/* Values for ERRORLEVEL */
#define EL_NO_DIFF		0
#define EL_DIFFERENT		1
#define EL_PARAM_ERROR		2
#define EL_NO_FILE_ERROR	3
#define EL_OPEN_ERROR		4

typedef enum { FALSE, TRUE } bool;

#define TAB_SIZE	8
#define END_OF_STRING	0

typedef signed long hashing_values;
#define BitMask(Bit)	(1 << Bit)

/* ------------------------------------------------------------------- */
/* Option flags */
#define BRIEF		BitMask(7)
#define NO_TAB_EXPAND	BitMask(6)
#define PACK_SPACE	BitMask(5)
#define IGNORE_CASE	BitMask(4)
#define LINE_NUMBER	BitMask(3)
unsigned char OptFlags = 0;

/* ------------------------------------------------------------------- */
FILE *File1, *File2;
char FileNames[2][MAXPATH];	/* Full pathnames */
char* Name[2];			/* Pointers to the name part of the files */
int MaxBinDifferences = BINARY_MAX_DIFF;
int LineForResync = 2;
int ContextLines = STANDARD_CONTEXT; /* 0 = do not show the context lines */
enum { DEFAULT, ASCII, BINARY } CompareMode = DEFAULT;

const char NoDifferences[] = "No differences\n\n";
const char AsciiDiffStart[] = "***** %s\n";
const char AsciiDiffEnd[] = "*****\n\n";
const char Omissis[] = "...\n";
const char AllFiles[] = "*.*";

/* ******************************************************************* */
void HelpMessage(void)
{
  printf("FreeDOS FC version 2.00 [Nov 08 1999]");
  printf("\nCompares two files or sets of files and displays the differences between them");
  printf("\n");
  printf("\nFC [switches] [drive1:][path1]filename1 [drive2][path2]filename2 [switches]");
  printf("\n");
  printf("\n /A    Display only first and last lines for each set of differences");
  printf("\n /B    Perform a binary comparison");
  printf("\n /C    Disregard the case of letters");
  printf("\n /L    Compare files as ASCII text");
  printf("\n /Mn   Set the maximum differences in binary comparison to n bytes");
  printf("\n       (default = %d, 0 = unlimited)", BINARY_MAX_DIFF);
  printf("\n /N    Display the line numbers on a text comparison");
  printf("\n /T    Do not expand tabs to spaces");
  printf("\n /V    Version info");
  printf("\n /W    Pack white space (tabs and spaces) for text comparison");
  printf("\n /X    Do not show context lines in text comparison");
  printf("\n");
}
/* ------------------------------------------------------------------- */
void VersionInfo(void)
{
  printf("FreeDOS FC Copyright (C) 1999 Maurizio Spagni");
  printf("\n");
  printf("\nThe FreeDOS ASCII File Compare engine has been extracted from COMPARE.C,");
  printf("\na program similar to unix diff whose source code was donated to the");
  printf("\nPublic Domain by the author Nick Ramsay the 5 August 1996.");
  printf("\n");
  printf("\nThe FreeDOS File Compare comes with ABSOLUTELY NO WARRANTY.");
  printf("\nThis program is free software; you can redistribute it and/or modify");
  printf("\nit under the terms of the GNU General Public License as published by");
  printf("\nthe Free Software Foundation; either version 2 of the License, or (at");
  printf("\nyour option) any later version.");
  printf("\n");
  printf("\nThis program is distributed in the hope that it will be useful, but");
  printf("\nWITHOUT ANY WARRANTY; without even the implied warranty of");
  printf("\nMERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU");
  printf("\nGeneral Public License for more details.");
  printf("\n");
  printf("\nSend bug reports to flurmy@comune.re.it.");
  printf("\n");
}

/* ******************************************************************* */
bool IsADirectory(char Filename[])
{
  struct stat statbuf;

  if (stat(Filename, &statbuf) != -1)
    if (statbuf.st_mode & S_IFDIR)
      return TRUE;
  return FALSE;
}
/* ------------------------------------------------------------------- */
bool HasWildcards(char* Filename)
{
  while (*Filename != END_OF_STRING)
  {
    if ((*Filename == '*') || (*Filename == '?')) return TRUE;
    Filename++;
  }
  return FALSE;
}
/* ------------------------------------------------------------------- */
/* Try to find a match between the filename pattern in Pattern and the
   filename in Name. If the match is found, the result is in Target and
   the function returns TRUE */
bool MatchNames(char Pattern[], char Name[], char Target[])
{
  /* Parse the name part */
  while ((*Pattern != END_OF_STRING) && (*Name != END_OF_STRING) &&
	 (*Name != '.') && ((*Pattern == *Name) || (*Pattern == '?')))
  {
    *Target = *Name;
    Pattern++; Name++; Target++;
  }
  if (*Pattern == '*')
  {
    while ((*Pattern != END_OF_STRING) && (*Pattern != '.')) Pattern++;
    while ((*Name != END_OF_STRING) && (*Name != '.'))
    {
      *Target = *Name;
      Name++; Target++;
    }
  }
  *Target = END_OF_STRING;
  /* Here must be a dot or the end of string */
  if (*Pattern == '.')
    Pattern++;
  else
    if (*Pattern != END_OF_STRING) return FALSE;
  if (*Name == '.')
    Name++;
  else
    if (*Name != END_OF_STRING) return FALSE;
  *Target = '.'; Target++;
  /* Parse the ext part */
  while ((*Pattern != END_OF_STRING) && (*Name != END_OF_STRING) &&
	 ((*Pattern == *Name) || (*Pattern == '?')))
  {
    *Target = *Name;
    Pattern++; Name++; Target++;
  }
  if (*Pattern == '*')
  {
    while (*Pattern != END_OF_STRING) Pattern++;
    while (*Name != END_OF_STRING)
    {
      *Target = *Name;
      Name++; Target++;
    }
  }
  *Target = END_OF_STRING;
  return ((*Name == END_OF_STRING) && (*Pattern == END_OF_STRING));
}
/* ------------------------------------------------------------------- */
/* Check if the file extension indicates a binary file */
bool BinaryFile(const char Filename[])
{
  const char* BinExt[] = { ".EXE", ".COM", ".SYS", ".OBJ", ".LIB", ".BIN" };
  char Ext[MAXEXT];
  int i;

  if (fnsplit(Filename, NULL, NULL, NULL, Ext) & EXTENSION)
    for (i = 0; i < sizeof (BinExt)/sizeof (BinExt[0]); i++)
      if (strcmpi(BinExt[i], Ext) == 0) return TRUE;
  return FALSE;
}
/* ------------------------------------------------------------------- */
/* Pointer to the start of the file name, with path stripped off */
char* FileNameStart(char Filename[])
{
  char* CurrPos = Filename;
  while (*CurrPos != END_OF_STRING)
  {
    if ((*CurrPos == ':') || (*CurrPos == '\\')) Filename = CurrPos + 1;
    CurrPos++;
  }
  return Filename;
}

/* ******************************************************************* */
// Scan arguments for switches and files
bool ScanArguments(int argc, char* argv[])
{
  int FileCounter = 0;
  int i;

  for (i = 1; i < argc; i++)
  {
    int j = 0;
    while ((argv[i][j] == '/') || (argv[i][j] == '-')) // Switch chars
      switch (toupper(argv[i][j + 1]))
      {
	case 'A':       /* The following two switches are incompatibles */
	  OptFlags |= BRIEF;
	  ContextLines = STANDARD_CONTEXT;
	  j += 2;
	  break;
	case 'X':
	  OptFlags &= ~BRIEF;
	  ContextLines = NO_CONTEXT;
	  j += 2;
	  break;
	case 'B':
	  CompareMode = BINARY;
	  j += 2;
	  break;
	case 'L':
	  CompareMode = ASCII;
	  j += 2;
	  break;
	case 'C':
	  OptFlags |= IGNORE_CASE;
	  j += 2;
	  break;
	case 'N':
	  OptFlags |= LINE_NUMBER;
	  j += 2;
	  break;
	case 'T':
	  OptFlags |= NO_TAB_EXPAND;
	  j += 2;
	  break;
	case 'W':
	  OptFlags |= PACK_SPACE;
	  j += 2;
	  break;
	case 'M':
	  MaxBinDifferences = atoi(&argv[i][j + 2]);
	  j += strlen(&argv[i][j]);
	  break;
	case '?':
	case 'H':
	  HelpMessage();
	  return FALSE;
	case 'V':
	  VersionInfo();
	  return FALSE;
	default:
	  printf("Invalid switch: %s\n", argv[i][j]);
	  return FALSE;
      }

    if (j == 0) // Not a switch: it's a filename
    {
      if (FileCounter >= 2)
      {
	printf("Too many filespecs\n");
	return FALSE;
      }
      strcpy(FileNames[FileCounter], argv[i]);
      strupr(FileNames[FileCounter]);
      FileCounter++;
    }
  }

  switch (FileCounter)
  {
    case 0:
      printf("No file specified\n");
      return FALSE;
    case 1:
      strcpy(FileNames[1], AllFiles);  /* Default: the current directory */
  }

  for (i = 0; i < 2; i++)
  {
    int LastChar = strlen(FileNames[i]) - 1;
    if ((FileNames[i][LastChar] == '\\') ||   /* Path only */
	(FileNames[i][LastChar] == ':'))      /* Disk drive only */
    {
      Name[i] = &FileNames[i][LastChar + 1];
      strcpy(Name[i], AllFiles);	      /* Assume "All files" */
    }
    else if (!HasWildcards(FileNames[i]) &&
	     IsADirectory(FileNames[i]))      /* Directory */
    {
      Name[i] = &FileNames[i][LastChar + 2];
      sprintf(Name[i] - 1, "\\%s", AllFiles); /* Assume "All files" */
    }
    else	/* Regular file (possibly with wildcards) */
      Name[i] = FileNameStart(FileNames[i]);
  }
  return TRUE;
}

/* ******************************************************************* */
bool BinaryCompare(void)
{
  unsigned char a, b;
  unsigned long Offset, FileSize1, FileSize2;
  int Differences = 0;

  FileSize1 = filelength(fileno(File1));
  FileSize2 = filelength(fileno(File2));
  if (FileSize1 != FileSize2)
  {
    printf("Warning: the files are of different size!\n");
    if (FileSize1 < FileSize2) FileSize1 = FileSize2;
  }
  for (Offset = 0; Offset < FileSize1; Offset++)
  {
    a = fgetc(File1); b = fgetc(File2);
    if (a != b)
    {
      Differences++;
      if ((MaxBinDifferences > 0) && (Differences > MaxBinDifferences))
      {
	printf("Comparison stopped after %d mismatches\n\n", MaxBinDifferences);
	return FALSE;
      }
      printf("%08lX:     %02X  %c      %02X  %c\n", Offset,
	     a, (iscntrl(a) ? ' ' : a), b, (iscntrl(b) ? ' ' : b));
    }
  }
  if (Differences == 0) printf(NoDifferences);
  return (Differences == 0);
}

/* ******************************************************************* */
/* These are flags that indicate the kind of occurrence for a line.
   Of course SINGLE_OCCURR and MULTIPLE_OCCURR are mutually exclusive */
#define NO_OCCURR	0
#define SINGLE_OCCURR	BitMask(0)
#define MULTIPLE_OCCURR BitMask(1)
#define DATA_BITS	2
typedef unsigned char packed_occurr;
#define BIT_PER_BYTE	  8
#define DATA_PER_ELEMENT  ((BIT_PER_BYTE*sizeof(packed_occurr))/DATA_BITS)
#define DATA_BIT_MASK	  ((1 << DATA_BITS) - 1)

/* ------------------------------------------------------------------- */
/* Extract bits from the packed occurrence array */
char GetOccurrences(const packed_occurr Array[], unsigned long Index)
{
  Index %= HASH_BUCKETS;
  return ((Array[(unsigned)(Index/DATA_PER_ELEMENT)] >>
	   (DATA_BITS*(Index % DATA_PER_ELEMENT))) & DATA_BIT_MASK);
}
/* ------------------------------------------------------------------- */
/* Store bits in the packed occurrence array */
void SetOccurrences(packed_occurr Array[], unsigned long Index, char Occurrence)
{
  unsigned Shifts;

  Index %= HASH_BUCKETS;
  Shifts = (unsigned)(DATA_BITS*(Index % DATA_PER_ELEMENT));
  Index /= DATA_PER_ELEMENT;
  Array[Index] &= ~(DATA_BIT_MASK << Shifts);
  Array[Index] |= (Occurrence << Shifts);
}

/* ******************************************************************* */
hashing_values HashValue = 0;

/* ------------------------------------------------------------------- */
/* Hash a character */
void Hash(char c)
{
  /* Rotate left 1 bit circularily */
  hashing_values Temp = HashValue;
  HashValue = HashValue << 1;
  if (Temp < 0) HashValue |= 1;

  HashValue ^= c;
}
/* ------------------------------------------------------------------- */
/* Read file, build hash & occurrence tables.
   Return the number of lines in the file. */
int HashFile(FILE* file, hashing_values HashVect[], packed_occurr Occurrences[])
{
  int c;
  bool Skip = FALSE;
  int LineCounter = 0;
  unsigned int LineLength = 0;

  HashValue = 0;
  do
  {
    c = fgetc(file);
    /* Beware of order! There are fallthroughs */
    switch (c)
    {
      case EOF:
	if (LineLength == 0) break;	/* End of file */
	/* Unterminated line: assume line feed */
      case '\n':
	/* This is needed to avoid problems with trailing
	   blanks when blanks are packed */
	if (!Skip) Hash(' ');

	/* HashVect[] must be < 0 */
	if (HashValue < 0)
	  HashValue = -HashValue;
	else
	  if (HashValue == 0) HashValue = 1;
	HashVect[LineCounter] = -HashValue;
	switch (GetOccurrences(Occurrences, HashValue))
	{
	  case NO_OCCURR:
	    SetOccurrences(Occurrences, HashValue, SINGLE_OCCURR);
	    break;
	  case SINGLE_OCCURR:
	    SetOccurrences(Occurrences, HashValue, MULTIPLE_OCCURR);
	}
	HashValue = 0; LineCounter++; LineLength = 0;
	Skip = FALSE;
	break;

      case '\t':
	/* Tabs must be expanded and blanks are not packed */
	if (!(OptFlags & NO_TAB_EXPAND) && !(OptFlags & PACK_SPACE))
	{
	  do
	  {
	    Hash(' '); LineLength++;
	  }
	  while ((LineLength % TAB_SIZE) != 0);
	  break;
	}
      case ' ':
      case '\r':
	/* Blank and tab are packed */
	if (OptFlags & PACK_SPACE)
	{
	  if (!Skip)
	  {
	    Hash(' '); LineLength++;
	    Skip = TRUE;
	  }
	  break;
	}
      default:
	if (OptFlags & IGNORE_CASE) c = toupper(c);
	Hash(c); LineLength++;
	Skip = FALSE;
    }
  }
  while ((c != EOF) && (LineCounter < MAX_LINES));

  if (LineCounter >= MAX_LINES)
    printf("Warning: comparison interrupted after %d lines\n", LineCounter);
  return LineCounter;
}
/* ------------------------------------------------------------------- */
/* Skip to the next line in the file File */
void SkipLine(FILE* File)
{
  int c;

  do
    c = fgetc(File);
  while ((c != '\n') && (c != EOF));
}
/* ------------------------------------------------------------------- */
/* Display a line from the file File */
void OutputLine(FILE* File, int LineNumber)
{
  unsigned int LineLength = 0;
  int c;

  if (OptFlags & LINE_NUMBER) printf("%5d: ", LineNumber);

  while (TRUE)
  {
    c = fgetc(File);
    if ((c == '\n') || (c == EOF)) /* End of line */
    {
      putchar('\n'); break;
    }
    if ((c == '\t') && !(OptFlags & NO_TAB_EXPAND)) /* Tab */
      do
      {
	putchar(' '); LineLength++;
      }
      while ((LineLength % TAB_SIZE) != (TAB_SIZE - 1));
    else
      if (c != '\r')    /* Ignore the carriage return */
      {
	putchar(c); LineLength++;
      }
  }
}
/* ------------------------------------------------------------------- */
/* Shows a block of differences */
bool ShowBlockOfDiff(int* Line, FILE* File, int* LinesRead, int LinesInFile,
		     char FileName[], hashing_values Hash[])
{
  int i;
  int ScanLine = *Line;
		    /* Go to the starting line (of the context) */
  while (*LinesRead < ScanLine - ContextLines)
  {
    (*LinesRead)++; SkipLine(File);
  };

  i = 0;
  do		    /* Find the end of the mismatching block */
  {
    ScanLine += i;
		    /* Skip mismatching lines */
    while ((ScanLine < LinesInFile - 1) &&
	   ((Hash[ScanLine] < 0) ||
	    (Hash[ScanLine] + 1 != Hash[ScanLine + 1]))) ScanLine++;

    i = 1;	    /* Count matching lines */
    while ((i < LineForResync - 2) && (ScanLine < LinesInFile - 1) &&
	   (Hash[ScanLine] >= 0) &&
	   (Hash[ScanLine + i] + 1 == Hash[ScanLine + i + 1])) i++;
  }
  while ((i < LineForResync - 2) && (ScanLine < LinesInFile - 1));
		    /* End of file or enough matching to resync */

  /* If there are really differences or the context is required */
  if ((ScanLine > (*Line)) || (ContextLines > NO_CONTEXT))
  {
    printf(AsciiDiffStart, FileName);
		    /* Show the starting context */
    for (i = 0; i < ContextLines; i++)
    {
      (*LinesRead)++; OutputLine(File, *LinesRead);
    }

    if (OptFlags & BRIEF)
    {
      printf(Omissis);
      while ((*LinesRead) < ScanLine)
      {
	(*LinesRead)++; SkipLine(File);
      }
    }
    else
      while ((*LinesRead) < ScanLine)
      {
	(*LinesRead)++; OutputLine(File, *LinesRead);
      }
		    /* Show the final context */
    while ((*LinesRead) < ScanLine + ContextLines)
    {
      (*LinesRead)++; OutputLine(File, *LinesRead);
    };
    (*Line) = ScanLine;
    if ((*LinesRead) >= LinesInFile) (*Line)++;
    return TRUE;
  }
  return FALSE;
}
/* ------------------------------------------------------------------- */
bool AsciiCompare(void)
{
  int i;
  bool Linked;
  bool Different = FALSE;
  int LinesInFile1, LinesInFile2, LinesRead1, LinesRead2, Line1, Line2;
  hashing_values* Hash1, *Hash2;
  /* Arrays of packed occurrences */
  packed_occurr* Occurr1, *Occurr2;

  /* Allocate the big structures on the heap */
#define PACKED_ARRAY_ITEMS  (HASH_BUCKETS + DATA_PER_ELEMENT - 1)/DATA_PER_ELEMENT
#define OCCURR_ARRAY_SIZE   (sizeof(packed_occurr)*PACKED_ARRAY_ITEMS)
  Occurr1 = (packed_occurr*)malloc(OCCURR_ARRAY_SIZE);
  Occurr2 = (packed_occurr*)malloc(OCCURR_ARRAY_SIZE);
#define HASH_ARRAY_SIZE	    (sizeof(hashing_values)*MAX_LINES)
  Hash1 = (hashing_values*)malloc(HASH_ARRAY_SIZE);
  Hash2 = (hashing_values*)malloc(HASH_ARRAY_SIZE);
  if ((Hash1 == NULL) || (Hash2 == NULL) ||
      (Occurr1 == NULL) || (Occurr2 == NULL))
  {
    printf("Insufficient memory\n");
    return FALSE;
  }
  memset(Hash1, 0, HASH_ARRAY_SIZE);
  memset(Hash2, 0, HASH_ARRAY_SIZE);
  memset(Occurr1, 0, OCCURR_ARRAY_SIZE);
  memset(Occurr2, 0, OCCURR_ARRAY_SIZE);

  /* step 1: read files and hash them */
  LinesInFile1 = HashFile(File1, Hash1, Occurr1);
  LinesInFile2 = HashFile(File2, Hash2, Occurr2);

  /* step 2: identify lines that are unique in both files */
  for (i = 0; i < PACKED_ARRAY_ITEMS; i++)
    Occurr1[i] &= Occurr2[i];

  /* step 3: link together the nearest matching unique lines
     N.B. Hashx set to matching line number (>= 0) if match found */
  for (i = 0; i < LinesInFile1; i++)
    if (GetOccurrences(Occurr1, -Hash1[i]) == SINGLE_OCCURR)
    {
      long Forward = i, Backward = i;
      while ((Forward < LinesInFile2) || (Backward >= 0))
      {
	if (Forward < LinesInFile2)
	  if (Hash2[Forward] == Hash1[i])
	  {
	    Hash1[i] = Forward; Hash2[Forward] = i;
	    break;
	  }
	if (Backward >= 0)
	  if (Hash2[Backward] == Hash1[i])
	  {
	    Hash1[i] = Backward; Hash2[Backward] = i;
	    break;
	  }
	Forward++; Backward--;
      }
    }
  free (Occurr1); free (Occurr2);	/* No more needed */

  /* step 4: link the first and last lines, if possible */
  if ((Hash1[0] < 0) && (Hash1[0] == Hash2[0])) Hash1[0] = Hash2[0] = 0;
  if ((Hash1[LinesInFile1 - 1] < 0) &&
      (Hash1[LinesInFile1 - 1] == Hash2[LinesInFile2 - 1]))
  {
    Hash1[LinesInFile1 - 1] = LinesInFile2 - 1;
    Hash2[LinesInFile2 - 1] = LinesInFile1 - 1;
  }

  /* step 5: starting from linked lines, link following lines that match
     N.B. Hashx set to matching line number (>= 0) if match found */
  Linked = FALSE;
  for (i = 0; i < LinesInFile1; i++)
    if (Hash1[i] >= 0)
      Linked = TRUE;
    else
      if (Linked)
	if (Hash1[i] == Hash2[Hash1[i - 1] + 1])
	{
	  Hash1[i] = Hash1[i - 1] + 1;
	  Hash2[Hash1[i]] = i;
	}
	else
	  Linked = FALSE;

  /* step 6: link matching lines that precede linked lines */
  Linked = FALSE;
  for (i = LinesInFile1 - 1; i >= 0; i--)
    if (Hash1[i] >= 0)
      Linked = TRUE;
    else
      if (Linked)
	if (Hash1[i] == Hash2[Hash1[i + 1] - 1])
	{
	  Hash1[i] = Hash1[i + 1] - 1;
	  Hash2[Hash1[i]] = i;
	}
	else
	  Linked = FALSE;

  /* step 7: display the results */
  rewind(File1); rewind(File2);
  LinesRead1 = LinesRead2 = 0;
  Line1 = Line2 = 0;
  while (TRUE)
  {
			      /* Skip matching lines */
    while ((Line1 < LinesInFile1) && (Line2 < LinesInFile2) &&
	   (Hash1[Line1] == Line2) /* &&
	   (Hash2[Line2] == Line1) (pleonastic) */)
    {
      Line1++; Line2++;
    }
			      /* File terminated */
    if ((Line1 >= LinesInFile1) || (Line2 >= LinesInFile2)) break;

    Linked = ShowBlockOfDiff(&Line1, File1, &LinesRead1, LinesInFile1,
			     FileNames[0], Hash1);
    if (ShowBlockOfDiff(&Line2, File2, &LinesRead2, LinesInFile2,
			FileNames[1],  Hash2))
      Linked = TRUE;
    if (Linked)
    {
      printf(AsciiDiffEnd);
      Different = TRUE;
    }
  }
  free (Hash1); free (Hash2);
  if (!Different) printf(NoDifferences);
  return (!Different);
}

/* ******************************************************************* */
int main(int argc, char* argv[])
{
  unsigned FileCount;
  struct ffblk FindData1, FindData2;
  bool Done1, Done2, Different;
  enum { MatchingNames, CommonSource, CommonTarget } NameMatching;
  char TargetNamePattern[MAXFILE + MAXEXT];

  if (!ScanArguments(argc, argv)) return EL_PARAM_ERROR;

  NameMatching = MatchingNames;
  if (!HasWildcards(Name[1]))
    NameMatching = CommonTarget;
  if (!HasWildcards(Name[0]) && (strcmp(Name[1], AllFiles) != 0))
    NameMatching = CommonSource;

  if (NameMatching == MatchingNames)
  {
    strncpy(TargetNamePattern, Name[1], sizeof(TargetNamePattern) - 1);
    TargetNamePattern[sizeof(TargetNamePattern) - 1] = END_OF_STRING;
  }

  FileCount = 0; Different = FALSE;
  Done1 = (findfirst(FileNames[0], &FindData1, FA_RDONLY) != 0);
  while (!Done1)
  {
    strcpy(Name[0], FindData1.ff_name);
    Done2 = FALSE;
    switch (NameMatching)
    {
      case MatchingNames:
	Done2 = !MatchNames(TargetNamePattern, Name[0], Name[1]);
	break;
      case CommonSource:
	Done2 = (findfirst(FileNames[1], &FindData2, FA_RDONLY) != 0);
	strcpy(Name[1], FindData2.ff_name);
    }
    while (!Done2)
    {
      File2 = fopen(FileNames[1], "rb");
      if (File2 != NULL)
      {
	File1 = fopen(FileNames[0], "rb");
	if (File1 == NULL)
	{
	  fcloseall();
	  printf("Error opening file %s\n", FileNames[0]);
	  return EL_OPEN_ERROR;
	}
	FileCount++;
	printf("Comparing %s and %s\n", FileNames[0], FileNames[1]);
	switch (CompareMode)
	{
	  case ASCII:
	    if (!AsciiCompare()) Different = TRUE;
	    break;
	  case BINARY:
	    if (!BinaryCompare()) Different = TRUE;
	    break;
	  default:
	    if (BinaryFile(FileNames[0]) || BinaryFile(FileNames[1]))
	    {
	      if (!BinaryCompare()) Different = TRUE;
	    }
	    else
	      if (!AsciiCompare()) Different = TRUE;
	}
	fcloseall();
      }
      Done2 = TRUE;
      if (NameMatching == CommonSource)
      {
	Done2 = (findnext(&FindData2) != 0);
	strcpy(Name[1], FindData2.ff_name);
      }
    }
    Done1 = (findnext(&FindData1) != 0);
  }
  if (FileCount == 0)
  {
    printf("No such file or directory\n");
    return EL_NO_FILE_ERROR;
  }
  if (Different)
    return EL_DIFFERENT;
  else
    return EL_NO_DIFF;
}
