/*
 * Micro-C/PC in the spirit of simplicity, performs only minimal parsing of
 * the command line arguments. In particular, Quotes are not supported to
 * allow spaces/tabs etc. to be included in the arguments. This is adaquate
 * and even desired for most programs and utilities, however it may be
 * desirable for certain programs to allow the use of arguments with embedded
 * spaces. To accomplish this, we can use unix style quoted arguments.
 *
 * This program demonstrates how to re-parse the command tail to allow the
 * use of quoted arguments. Arguments contained in double quotes are placed
 * into the argv[] array as a single argument, including any spaces or tabs
 * which occur between the quotes. The quotes themselves are removed. To
 * include a '"' character in a quoted argument, simply include two of them
 * in the argument string: eg: "one""two" results in the argument: one"two
 *
 * The method to access the command tail differs in the TINY and SMALL memory
 * models. In TINY model, the command tail is directly accessable, while in
 * SMALL model, it resides in a far segment. This program uses pre-processor
 * macros to implement the different types of memory access. You must specify
 * the memory model by predefining the symbol "TINY" or "SMALL".
 *
 * Compile command: cc quotearg -fop TINY=
 *              or: cc quotearg -fop m=s SMALL=
 */
#include <stdio.h>

/* Provide global access to argc and argv (via Micro-C/PC runtime library) */
extern int ARGC;
extern char *ARGV[];

#ifdef SMALL
	extern unsigned PSP;
	#define GET_TAIL peek(PSP, tptr)
	#define GET_TAIL_INC peek(PSP, tptr++)
#endif

#ifdef TINY
	#define GET_TAIL *tptr
	#define GET_TAIL_INC *tptr++
#endif

#ifndef GET_TAIL
	#error Must define TINY or SMALL
#endif

/*
 * Re-parse the command line arguments to obtain values in quotations
 *
 * New arguments are parsed into the same holding buffer as the original
 * ones, which means that the value of ARGV will not change. The global
 * ARGC is updated by this function, and the new argc value is returned
 * so that local copies may be easily updated as well.
 */
parse_quoted_args()
{
	unsigned char *tptr, *aptr;
	unsigned l;
	char c, flag;

	tptr = 0x80;					/* Command tail (TINY model only) */
	aptr = ARGV[ARGC=1];
	l = GET_TAIL_INC;
	flag = 0;
	while(l--) {
		c = GET_TAIL_INC;
		if(!flag) {					/* Looking for next argument */
			if(isspace(c))
				continue;
			ARGV[ARGC++] = aptr;
			if(c == '"') {			/* Quoted argument */
				flag = 2;
				continue; }
			flag = 1; }
		switch(c) {
			case '"' :				/* Quote is special case */
				if(flag != 2)		/* Not processing within quotes */
					goto norchar;
				if(GET_TAIL == '"') {	/* Double quotes, insert a single one */
					++tptr;
					--l;
					goto norchar; }
			/* Not a double quote, fall through & end this argument */
			case 0x0D :				/* Secondary end of line indicator */
				flag = 1;
			case ' ' :				/* Space between normal arguments */
			case '\t' :
				if(flag == 1) {		/* Not quoted, end this argument */
					*aptr++ = flag = 0;
					continue; }
			default:				/* Normal character, add to arg */
			norchar:
				*aptr++ = c; } }
	*aptr = 0;
	return ARGC;
}

/*
 * Test program, demo the parse_quoted_args() function.
 * Note that since argc is our local variable, it has not been
 * updated, hence we re-assign it from the function return value.
 * argv will not change, since arguments are parsed into the same
 * holding buffer.
 */
main(int argc, char *argv[])
{
	int i;
	printf("Before parse_quoted_args(), argc=%d\n", argc);
	for(i=1; i < argc; ++i)
		printf("\targv[%d]='%s'\n", i, argv[i]);
	argc = parse_quoted_args();
	printf("After parse_quoted_args(), argc=%d\n", argc);
	for(i=1; i < argc; ++i)
		printf("\targv[%d]='%s'\n", i, argv[i]);
}
