/*
 * vortex.c
 *
 * Aureal Vortex AU88xx audio driver module.
 *
 * Copyright (c) 1999 Aureal Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or 
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 * 
 * Contact: http://linux.aureal.com
 *
 * Some of this code is based on the es1371 driver, which is
 * Copyright (C) 1998  Thomas Sailer (sailer@ife.ee.ethz.ch)
 *
 * This file must be linked with the appropriate Vortex core object file
 * (asp30.o, asp10.o, or asp20.o).
 *
 * Devices:
 * /dev/dsp	full duplex, multiple streams
 * /dev/mixer
 * /dev/midi
 * /dev/audio
 * /dev/sndstat
 *
 * The joystick interface is enabled at I/O port 0x201.
 * Joystick support can be provided by the standard joystick driver.
 *
 * Revision History
 *   11-01-99  	Michael Minnick		Initial release
 *   02-10-00	Michael Minnick		Fill entire dma buf before sending
 */

#include <linux/autoconf.h>
#if defined(CONFIG_MODVERSIONS) && !defined(MODVERSIONS)
#define MODVERSIONS
#endif
#if defined(MODVERSIONS)
#include <linux/modversions.h>
#endif
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/version.h>
#include <linux/wrapper.h>
#include <linux/config.h>
#include <linux/string.h>
#include <linux/ioport.h>
#include <linux/sched.h>
#include <linux/delay.h>
#include <linux/sound.h>
#include <linux/malloc.h>
#include <linux/soundcard.h>
#include <linux/pci.h>
#include <asm/io.h>
#include <linux/poll.h>
#include <asm/spinlock.h>
#include <asm/uaccess.h>
#include <linux/utsname.h>

#define DRIVER_NAME 	"au88xx"

#define VENDOR_ID	0x12eb
/* pci device ids */
#define DEV_AU8810	3
#define DEV_AU8830	2
#define DEV_AU8820	1

#define PWARN(fmt, args...) printk(KERN_WARNING DRIVER_NAME ": " fmt, ## args)
#ifdef DEBUG_AU
#define PDEBUG(fmt, args...) printk(KERN_DEBUG DRIVER_NAME ": " fmt, ## args)
#else
#define PDEBUG(fmt, args...)
#endif

#ifndef TRUE
#define TRUE 1
#endif
#ifndef FALSE
#define FALSE 0
#endif
#define SUCCESS 0

#ifndef MIN
#define MIN(a,b)  (((a) < (b)) ? (a) : (b))
#endif

#define DEF_FRAG_SIZE	PAGE_SIZE
#define MAX_FRAGS	4
#define DEF_NR_FRAGS	MAX_FRAGS

/* MIDI buffer sizes */
#define MIDIINBUF  256
#define MIDIOUTBUF 256

/* vortex gameport midi status register */
#define MPU401_CMDOK	0x40        /* ok to write if 0 */
#define MPU401_MIDIVAL	0x80        /* data avail if 0 */

/*
 * Memory transfer buffer.
 * Each stream contains one or more of these buffers,
 * which are used to transfer application buffers
 * to the vortex hardware.
 */
typedef struct mem_buf {
	unsigned long	count;
	unsigned long	xfer;
	void		*page;
	unsigned	in_use: 1;
} mem_buf_t;

/*
 * Wave stream.
 * Either a playback or a record stream.
 */
typedef struct wave_stream {
	struct wait_queue	*io_wait;
	void			*wave;		/* core wave object pointer */
	void			*dmabuf;
	mem_buf_t		buf[MAX_FRAGS];
	unsigned long		total_played;
	unsigned long		total_sent;
	unsigned long		frag_size;
	unsigned long		num_frags;
	int			buf_order;
	int			fill_index;
	int			drain_index;
	int			data_format;
	int			sample_rate;
	int			channel_count;
	unsigned		active: 1;
	unsigned		pre_roll: 1;
	unsigned		mapped: 1;
	unsigned		mapped_setup: 1;
} wave_stream_t;

/*
 * Wave device.
 * Contains a playback stream and a record stream (full duplex).
 */
typedef struct wave_device {
	struct wave_device	*next;
	struct wave_device	*prev;
	struct file		*fp;
	wave_stream_t		play_stream;
	wave_stream_t		record_stream;
} wave_device_t;

/*
 * Midi device.
 * Contains an input buffer and an output buffer.
 */
typedef struct midi_device {
	unsigned		ird, iwr, icnt;
	unsigned		ord, owr, ocnt;
	struct wait_queue	*iwait;
	struct wait_queue	*owait;
	unsigned char		ibuf[MIDIINBUF];
	unsigned char		obuf[MIDIOUTBUF];
} midi_device_t;

/*
 * Each device has multiple "functions" - audio, mixer, midi, synth.
 * Functions are assigned a minor number by the soundcore module.
 * The lower nibble is the function type.
 * For example /dev/dsp minor number is 3 and would be
 * assigned to the audio (wave) function on the first vortex card.
 * Minor numbers are "chained" using the high nibble, so the
 * second device would get /dev/dsp1, whose minor number is 19.
 * That's 0x13, low nibble = 3 (dsp), high nibble = 1 (chain).
 */
#define DEVICE_TYPE(m)	(m & 0x0f)
#define DEV_MIXER	0
#define DEV_SEQUENCER	1
#define DEV_MIDI	2
#define DEV_AUDIO	3
#define DEV_SUNDSP	4

/*
 * Per-device state.
 * Each device represents a vortex chip, typically on a pci card.
 * Each device has its own vortex core instance.
 */
typedef struct au_device {
	struct au_device	*next;
	struct pci_dev		*pcidev;
	void			*core;		/* core object pointer */
	int			dev_id;		/* vortex chip id */
	int			dev_audio;	/* audio (dsp) minor num from soundcore */
	int			dev_mixer;
	int			dev_midi;
	int			dev_sun;
	int			mix_modcnt;
	unsigned int		mix_recsrc;
	short			mix_vol[SOUND_MIXER_NRDEVICES];
	spinlock_t		irq_lock;
	wave_device_t		*wave_head;
	wave_device_t		*wave_tail;
	midi_device_t		midi;
	unsigned		record_in_use: 1;
	unsigned		midiin_in_use: 1;
	unsigned		midiout_in_use: 1;
} au_device_t;

/* wrapper around kernel timer that remembers last queued time */
typedef struct au_timer {
	struct timer_list	*timer;
	unsigned long		last_time;
} au_timer_t;

static au_device_t *dev_head;	/* list of device found on pci bus */
static au_device_t *dev_tail;
static int dev_sndstat = -1;

/* local functions */
static int device_scan(void);
static void remove_devices(void);
static au_device_t *find_device(int minor_num);
static au_device_t *find_device_core(void *core_obj);
static void au_isr(int irq, void *dev_id, struct pt_regs *regs);
static int add_device(struct pci_dev *pcidev, int dev_id);
static int scan_id(int dev_id);
static void *alloc_core(struct pci_dev *pcidev, au_device_t *dev);
static int add_wave_device(au_device_t *dev, struct file *file, int dev_type);
static int get_frags(wave_stream_t *stream, unsigned long count);
static void free_wave_device(wave_device_t *wdev);
static wave_device_t *find_wave_device(au_device_t *dev, struct file *file);
static void remove_wave_device(au_device_t *dev, wave_device_t *wdev);
static unsigned long receive_mem(wave_stream_t *stream, char *mem, unsigned long max_size);
static unsigned long send_mem(wave_stream_t *stream, const char *mem, unsigned long size);
static void set_data_format(wave_stream_t *stream, int val);
static void set_asp_format(wave_stream_t *stream);
static void stop_wave(wave_stream_t *stream);
static loff_t au_llseek(struct file *file, loff_t offset, int origin);
static ssize_t au_audio_read(struct file *file, char *buffer, size_t count, loff_t *ppos);
static ssize_t au_audio_write(struct file *file, const char *buffer, size_t count, loff_t *ppos);
static unsigned int au_audio_poll(struct file *file, struct poll_table_struct *wait);
static int au_audio_ioctl(struct inode *inode, struct file *file, unsigned int cmd, unsigned long arg);
static int au_audio_mmap(struct file *file, struct vm_area_struct *vma);
static int au_audio_open(struct inode *inode, struct file *file);
static int au_audio_release(struct inode *inode, struct file *file);
static int au_mixer_ioctl(struct inode *inode, struct file *file, unsigned int cmd, unsigned long arg);
static int au_mixer_open(struct inode *inode, struct file *file);
static int au_mixer_release(struct inode *inode, struct file *file);
static void write_mixer(au_device_t *dev, unsigned int chan, int val);
static void init_mixer(au_device_t *dev);
static int au_midi_open(struct inode *inode, struct file *file);
static int au_midi_release(struct inode *inode, struct file *file);
static int au_midi_ioctl(struct inode *inode, struct file *file, unsigned int cmd, unsigned long arg);
static ssize_t au_midi_read(struct file *file, char *buffer, size_t count, loff_t *ppos);
static ssize_t au_midi_write(struct file *file, const char *buffer, size_t count, loff_t *ppos);
static unsigned int au_midi_poll(struct file *file, struct poll_table_struct *wait);
static int au_sndstat_open(struct inode *inode, struct file *file);
static int au_sndstat_release(struct inode *inode, struct file *file);
static ssize_t au_sndstat_read(struct file *file, char *buffer, size_t count, loff_t *ppos);
static int sndstat_file_read(struct file * file, char * buf, size_t nbytes, loff_t *ppos);
static int sound_proc_get_info(char *buffer, char **start, off_t offset, int length, int inout);
static void drain_playback(wave_device_t *wdev, int nonblock);
static void set_rec_src(au_device_t *dev, unsigned int val);
static void au_handle_midi(au_device_t *dev);
static void au_handle_timer(au_device_t *dev);

/* file operations */

static struct file_operations audio_fops = {
	au_llseek,
	au_audio_read,
	au_audio_write,
	NULL,			/* readdir */
	au_audio_poll,
	au_audio_ioctl,
	au_audio_mmap,
	au_audio_open,
	NULL,			/* flush */
	au_audio_release
};
static struct file_operations mixer_fops = {
	au_llseek,
	NULL,			/* read */
	NULL,			/* write */
	NULL,			/* readdir */
	NULL,			/* poll */
	au_mixer_ioctl,
	NULL,			/* mmap */
	au_mixer_open,
	NULL,			/* flush */
	au_mixer_release
};
static struct file_operations midi_fops = {
	au_llseek,
	au_midi_read,
	au_midi_write,
	NULL,			/* readdir */
	au_midi_poll,
	au_midi_ioctl,
	NULL,			/* mmap */
	au_midi_open,
	NULL,			/* flush */
	au_midi_release
};
static struct file_operations sndstat_fops = {
	NULL,			/* llseek */
	au_sndstat_read,
	NULL,			/* write */
	NULL,			/* readdir */
	NULL,			/* poll */
	NULL,			/* ioctl */
	NULL,			/* mmap */
	au_sndstat_open,
	NULL,			/* flush */
	au_sndstat_release
};

/********************************/
/* Vortex Core interface	*/
/********************************/

/* We're not including core header files, so we must count on
   these interfaces not changing. */

typedef enum {
	ASPFMTLINEAR16 = 0,
	ASPFMTLINEAR8,
	ASPFMTULAW, 
	ASPFMTALAW, 
	ASPFMTSPORT,
	ASPFMTSPDIF,
} ASPENCODING;

typedef enum {
	ASPDIRPLAY,
	ASPDIRRECORD
} ASPDIRECTION;

typedef struct _ASPWAVEFORMAT {
	unsigned short	wBitWidth;
	unsigned short	wChannels;
	unsigned long	dwSampleRate;
	ASPENCODING	eEncoding;
} ASPWAVEFORMAT, *PASPWAVEFORMAT;

typedef struct _ASPVOLUME {
	long	lVolume;
	long	lPanPosition;
} ASPVOLUME, *PASPVOLUME;

#define	MAX_MEM_REGISTERS	9
#define	MAX_IO_PORTS		20
#define	MAX_IRQS		7
#define	MAX_DMA_CHANNELS	7
typedef struct Config_Buff_s {
	unsigned short	wNumMemWindows;
	unsigned long	dMemBase[MAX_MEM_REGISTERS];
	unsigned long	dMemLength[MAX_MEM_REGISTERS];
	unsigned short	wMemAttrib[MAX_MEM_REGISTERS];
	unsigned short	wNumIOPorts;
	unsigned short	wIOPortBase[MAX_IO_PORTS];
	unsigned short	wIOPortLength[MAX_IO_PORTS];
	unsigned short	wNumIRQs;
	unsigned char	bIRQRegisters[MAX_IRQS];
	unsigned char	bIRQAttrib[MAX_IRQS];
	unsigned short	wNumDMAs;
	unsigned char	bDMALst[MAX_DMA_CHANNELS];
	unsigned short	wDMAAttrib[MAX_DMA_CHANNELS];
	unsigned char	bReserved1[3];
	unsigned long	BusNumber;
} CMCONFIG;

/* AC97 codec registers */
#define AC97REG_MASTER			0x02
#define AC97REG_MASTER_MONO		0x06
#define AC97REG_PC_BEEP			0x0a
#define AC97REG_PHONE			0x0c
#define AC97REG_MIC			0x0e
#define AC97REG_LINE_IN			0x10
#define AC97REG_CD			0x12
#define AC97REG_VIDEO			0x14
#define AC97REG_PCM			0x18
#define AC97REG_AUX			0x16
#define AC97REG_REC_GAIN		0x1c
#define AC97REG_REC_SELECT		0x1a

/* mixer channels */
#define ASP_MIX_INPUT_WAVE_AT_PCM	12
#define ASP_MIX_INPUT_WAVE_AT_WAVEIN	14

typedef void  ClassAsp4Core;
typedef void  ClassWave;

extern void *allocCore(CMCONFIG *pConfig);
extern void deallocCore(ClassAsp4Core *pCore);
extern unsigned long coreISR(ClassAsp4Core *pCore);
extern void coreEnableInts(ClassAsp4Core *pCore);
extern void coreDisableInts(ClassAsp4Core *pCore);
extern void EnableHardCodedJoystickPort(ClassAsp4Core *pCore, int bEnable, unsigned long dwPort);
extern int CreateWaveBuffer(ClassAsp4Core *pCore, PASPWAVEFORMAT pWaveFormat, ASPDIRECTION eDirection, ClassWave **ppWaveObj);
extern void StopWave(ClassWave *pWave);
extern void ReleaseWave(ClassWave *pWave);
extern void StartWave(ClassWave *pWave);
extern unsigned long GetWaveBytesPlayed(ClassWave *pWave);
extern int AddWaveBuffer(ClassWave *pWave, void *buf, unsigned long size);
extern void SetWaveFormat(ClassWave *pWave, PASPWAVEFORMAT pWaveFormat);
extern void WriteCodec(ClassAsp4Core *pCore, short reg, short val);
extern void ReadCodec(ClassAsp4Core *pCore, short reg, unsigned short *val);
extern void WriteMixer(ClassAsp4Core *pCore, unsigned long id, unsigned long left, unsigned long right);
extern void EnableHardCodedMPU401Port(ClassAsp4Core *pCore, int bEnable, unsigned long dwPort);
extern void EnableMPU401Interrupt(ClassAsp4Core *pCore, int bEnable);
extern void Mpu401UartWriteData(ClassAsp4Core *pCore, unsigned char bData);
extern unsigned char Mpu401UartReadData(ClassAsp4Core *pCore);
extern unsigned char Mpu401UartReadStatus(ClassAsp4Core *pCore);
extern void Mpu401UartWriteCommand(ClassAsp4Core *pCore, unsigned char bData);
extern int Mpu401UartInit(ClassAsp4Core *pCore, unsigned char bClockDivider, unsigned char bMidiMode, int bMpuMode);

/********************************/
/* Module support		*/
/********************************/

int init_module()
{
	EXPORT_NO_SYMBOLS;

	PDEBUG("loading\n");

	/* scan pci bus and create device structures */
	return device_scan();
}

void cleanup_module()
{
	remove_devices();
	PDEBUG("unloaded\n");
}

/********************************/
/* Device support		*/
/********************************/

static int device_scan()
{
	int device_count = 0;

	if (!pci_present())
		return -ENODEV;
	/*
	 * This driver binary supports multiple instances of vortex
	 * cards of one particular chip type.
	 */
#ifdef AU8810
	device_count += scan_id(DEV_AU8810);
#endif
#ifdef AU8820
	device_count += scan_id(DEV_AU8820);
#endif
#ifdef AU8830
	device_count += scan_id(DEV_AU8830);
#endif
	return device_count ? SUCCESS : -ENODEV;
}

static int scan_id(int dev_id)
{
	struct pci_dev *pcidev = NULL;
	int device_count = 0;

	PDEBUG("scanning for device VEND=0x%x, ID=0x%x\n", VENDOR_ID, dev_id);

	while ((pcidev = pci_find_device(VENDOR_ID, dev_id, pcidev)) != 0) {

		PDEBUG("Found vortex PCI device:\n");
		PDEBUG("id=%d\n", dev_id);
		PDEBUG("bar0=0x%08lx\n", pcidev->base_address[0]);
		PDEBUG("irq=%d\n", pcidev->irq);

		if ((pcidev->base_address[0] == 0) || 
		    ((pcidev->base_address[0] & PCI_BASE_ADDRESS_SPACE) != PCI_BASE_ADDRESS_SPACE_MEMORY) ||
		    (pcidev->irq == 0)) {
			PWARN("found uninitialized vortex card\n");
			continue;
		}
		if (add_device(pcidev, dev_id) == 0)
			device_count++;
	}
	return device_count;
}

static int add_device(struct pci_dev *pcidev, int dev_id)
{
	au_device_t *dev;
	int status;
	unsigned char clock_divider = 0x61;
	unsigned char midi_mode = 0;
	int mpu_mode = 0x1;
	unsigned long midi_port = 3; // 330-331

	dev = (au_device_t *)kmalloc(sizeof(au_device_t), GFP_KERNEL);
	if (!dev)
		return -ENOMEM;
	memset(dev, 0, sizeof(au_device_t));

	dev->dev_id = dev_id;
	dev->pcidev = pcidev;
	init_waitqueue(&dev->midi.iwait);
	init_waitqueue(&dev->midi.owait);
  
	/* allocate the vortex core */
	if ((dev->core = alloc_core(pcidev, dev)) == NULL) {
		PWARN("could not allocate core\n");
		kfree(dev);
		return -ENOMEM;
	}

	/* fixme alloc synth - HwAllocSynth */

	/* register devices with soundcore */
	status = dev->dev_audio = register_sound_dsp(&audio_fops, -1);
	if (status < 0) {
		PWARN("could not register audio device\n");
		deallocCore(dev->core);
		kfree(dev);
		return status;
	}

	status = dev->dev_mixer = register_sound_mixer(&mixer_fops, -1);
	if (status < 0) {
		PWARN("could not register mixer device\n");
		unregister_sound_dsp(dev->dev_audio);
		deallocCore(dev->core);
		kfree(dev);
		return status;
	}

	status = dev->dev_midi = register_sound_midi(&midi_fops, -1);
	if (status < 0) {
		PWARN("could not register midi device\n");
		unregister_sound_dsp(dev->dev_audio);
		unregister_sound_mixer(dev->dev_mixer);
		deallocCore(dev->core);
		kfree(dev);
		return status;
	}

	/* this is /dev/audio */
	status = dev->dev_sun = register_sound_special(&audio_fops, 4);
	if (status < 0) {
		PWARN("could not register sun audio device\n");
		unregister_sound_dsp(dev->dev_audio);
		unregister_sound_mixer(dev->dev_mixer);
		unregister_sound_midi(dev->dev_midi);
		deallocCore(dev->core);
		kfree(dev);
		return status;
	}

	/*
	 * There is only one sndstat device. Register it if we can.
	 */
	if (dev_sndstat < 0) {
		status = dev_sndstat = register_sound_special(&sndstat_fops, 6);
		if (status < 0)
			PWARN("could not register sndstat device\n");
	}

	if (request_irq(dev->pcidev->irq, au_isr, SA_SHIRQ, "au88xx", dev) != 0) {
		PWARN("cannot alloc irq\n");
		unregister_sound_dsp(dev->dev_audio);
		unregister_sound_mixer(dev->dev_mixer);
		unregister_sound_midi(dev->dev_midi);
		unregister_sound_special(dev->dev_sun);
		if (dev_sndstat >= 0)
			unregister_sound_special(6);
		deallocCore(dev->core);
		kfree(dev);
		return -ENOMEM;
	}

	/* fixme - pass port as driver arg for each card, allow disable */
	EnableHardCodedJoystickPort(dev->core, TRUE, 0x201);

	coreEnableInts(dev->core);

	/* fixme - make legacy mpu port address selectable? */
	EnableHardCodedMPU401Port(dev->core, TRUE, midi_port);
	if (!Mpu401UartInit(dev->core, clock_divider, midi_mode, mpu_mode)) {
		PWARN("no ACK from midi port\n");
	}

	init_mixer(dev);

	PDEBUG("add device, audio=%d, mixer=%d, midi=%d\n", dev->dev_audio, dev->dev_mixer, dev->dev_midi);

	/* link to end of device list */
	if (!dev_head) {
		dev_head = dev_tail = dev;
	} else {
		dev_tail->next = dev;
		dev_tail = dev;
	}
	return SUCCESS;
}

static void remove_devices()
{
	au_device_t *next_dev = dev_head;
	au_device_t *dev;

	while (next_dev) {
		dev = next_dev;
		next_dev = dev->next;

		coreDisableInts(dev->core);
		free_irq(dev->pcidev->irq, dev);
		/* HwFreeSynth() */
		deallocCore(dev->core);
		unregister_sound_dsp(dev->dev_audio);
		unregister_sound_mixer(dev->dev_mixer);
		unregister_sound_midi(dev->dev_midi);
		unregister_sound_special(dev->dev_sun);
		kfree(dev);
	}
	dev_head = dev_tail = (au_device_t *)NULL;

	if (dev_sndstat >= 0)
		unregister_sound_special(6);
}

/*
 * Find the device that registered the specified minor number
 */
static au_device_t *find_device(int minor_num)
{
	au_device_t *dev = dev_head;
	au_device_t *found_dev = (au_device_t *)NULL;
	int dev_type = DEVICE_TYPE(minor_num);

	while (dev) {
		switch (dev_type) {

		case DEV_AUDIO:
			if (dev->dev_audio == minor_num)
				found_dev = dev;
			break;
		case DEV_MIXER:
			if (dev->dev_mixer == minor_num)
				found_dev = dev;
			break;
		case DEV_MIDI:
			if (dev->dev_midi == minor_num)
				found_dev = dev;
			break;
		case DEV_SUNDSP:
			if (dev->dev_sun == minor_num)
				found_dev = dev;
			break;
		default:
			break;
		}
		if (found_dev)
			break;

		dev = dev->next;
	}
	return found_dev;
}

/*
 * Find the device that supports the specified asp core
 */
static au_device_t *find_device_core(void *core_obj)
{
	au_device_t *dev = dev_head;

	while (dev) {
		if (dev->core == core_obj)
			break;
		dev = dev->next;
	}
	return dev;
}

static void *alloc_core(struct pci_dev *pcidev, au_device_t *dev)
{
	CMCONFIG config;
	int bar_len = dev->dev_id == DEV_AU8820 ? 0x20000 : 0x40000;

	config.dMemBase[0] = pcidev->base_address[0];
	config.dMemLength[0] = bar_len;
	config.wIOPortBase[0] = pcidev->base_address[1];
	config.wIOPortLength[0] = 2;
	config.bIRQRegisters[0] = pcidev->irq;
	config.bIRQAttrib[0] = 0;
	config.BusNumber = pcidev->bus->number;

	return allocCore(&config);
}

static void au_isr(int irq, void *dev_id, struct pt_regs *regs)
{
	au_device_t *dev = (au_device_t *)dev_id;
	unsigned long irq_status;

	//PDEBUG("AU_ISR - dev = 0x%x\n", (unsigned int)dev);

	/* this lock is exported to the core */
	spin_lock(&dev->irq_lock);

	/* the core ISR acknowledges the irq and returns the mask */
	irq_status = coreISR(dev->core);
	//PDEBUG("irq_s = 0x%x\n", (unsigned int)irq_status);

	/* shared interrupt handler - return if not ours */
	if (!irq_status) {
		spin_unlock(&dev->irq_lock);
		return;
	}

	if (irq_status & 0x2000) {
		au_handle_midi(dev);
	}

	if (irq_status & 0x1000) {
		au_handle_timer(dev);
	}

	spin_unlock(&dev->irq_lock);
}

/********************************/
/* Wave audio support		*/
/********************************/

static loff_t au_llseek(struct file *file, loff_t offset, int origin)
{
	return -ESPIPE;
}

static ssize_t au_audio_read(struct file *file, char *buffer, size_t count, loff_t *ppos)
{
	au_device_t *dev = file->private_data;
	wave_device_t *wdev;
	wave_stream_t *stream;
	size_t received = 0;
	size_t xfer;

	//PDEBUG("au_audio_read, count=%d\n", count);

	wdev = find_wave_device(dev, file);
	if (!wdev) {
		PWARN("could not find wave device for file = 0x%p\n", file);
		return -ENXIO;
	}
	stream = &wdev->record_stream;
	/* assert(stream->wave) */

	if (ppos != &file->f_pos)
		return -ESPIPE;
	if (stream->mapped)
		return -ENXIO;

	if (!access_ok(VERIFY_WRITE, buffer, count))
		return -EFAULT;

	if (!stream->active)
		set_asp_format(stream);

	while (received < count) {
		/*
		 * If inactive, receive_mem will add the first buffers.
		 * Otherwise, it will copy completed buffers to user memory
		 * and then queue up another one.
		 */
		xfer = receive_mem(stream, buffer + received, count - received);

		received += xfer;

		if (!stream->active) {
			StartWave(stream->wave);
			stream->active = TRUE;
		}

		if (received < count) {
			if (file->f_flags & O_NONBLOCK)
				return received ? (ssize_t)received : -EAGAIN;
			interruptible_sleep_on(&stream->io_wait);
			if (signal_pending(current)) {
				return received ? (ssize_t)received : -ERESTARTSYS;
			}
		}
	}
	return received ? (ssize_t)received : -EFAULT;
}

static ssize_t au_audio_write(struct file *file, const char *buffer, size_t count, loff_t *ppos)
{
	au_device_t *dev = file->private_data;
	wave_device_t *wdev;
	wave_stream_t *stream;
	size_t sent = 0;

	//PDEBUG("au_audio_write, count=%d\n", count);

	wdev = find_wave_device(dev, file);
	if (!wdev) {
		PWARN("could not find wave device for file = 0x%p\n", file);
		return -ENXIO;
	}
	stream = &wdev->play_stream;
	/* assert(stream->wave) */

	if (ppos != &file->f_pos)
		return -ESPIPE;
	if (stream->mapped)
		return -ENXIO;

	if (!access_ok(VERIFY_READ, buffer, count))
		return -EFAULT;

	/* asp core format must be set before first add buffer */
	if (!stream->active)
		set_asp_format(stream);

	while (sent < count) {
		/*
		 * Copy as much as we can into the dma buffers.
		 */
		size_t xfer = send_mem(stream, buffer + sent, count - sent);
		sent += xfer;
		stream->total_sent += sent;
		/*
		 * If pre-rolling, we can't allow blocking since app must send trigger.
		 */
		if (stream->pre_roll) {
			if (sent < count) {
				return sent ? (ssize_t)sent : -EAGAIN;
			} else {
				break;
			}
		}

		if (stream->active) {
			/*
			 * DMA is running, so we must block if not enough room for user data.
			 */
			if (sent < count) {	
				if (file->f_flags & O_NONBLOCK)
					return sent ? (ssize_t)sent : -EAGAIN;
				interruptible_sleep_on(&stream->io_wait);
				if (signal_pending(current)) {	/* app was killed */
					return sent ? (ssize_t)sent : -ERESTARTSYS;
				}
			}
		} else {
			/*
			 * DMA is not running, so start it if we have enough data. If not,
			 * app will have to send DSP_SYNC (or a close) to start.
			 */
			if (stream->total_sent >= stream->frag_size) {
				stream->active = TRUE;
				StartWave(stream->wave);
			} else {
				if (sent < count) {
					/*
					 * Inactive, but couldn't copy even a frag size.
					 * Shouldn't happen.
					 */
					return sent ? (ssize_t)sent : -EAGAIN;
				}
			}
		} /* if (stream->active) */
	}
	return sent ? (ssize_t)sent : -EFAULT;
}

static unsigned int au_audio_poll(struct file *file, struct poll_table_struct *wait)
{
	/* fixme */
	return SUCCESS;
}

static int au_audio_ioctl(struct inode *inode, struct file *file, unsigned int cmd, unsigned long arg)
{
	au_device_t *dev = file->private_data;
	wave_device_t *wdev;
	wave_stream_t *stream;
	audio_buf_info abinfo;
	count_info cinfo;
	int val, i;

	//PDEBUG("au_audio_ioctl, minor %d cmd 0x%x\n", dev->dev_audio, cmd);

	wdev = find_wave_device(dev, file);
	if (!wdev) {
		PWARN("could not find wave device for file = 0x%p\n", file);
		return -ENXIO;
	}

	/* fixme: dynamic update speed, fmt, chans, etc.? */

	switch(cmd) {

	case OSS_GETVERSION:
		return put_user(SOUND_VERSION, (int *)arg);

	case SNDCTL_DSP_POST:
		if (!wdev->play_stream.wave)
			return -EINVAL;
		drain_playback(wdev, (file->f_flags & O_NONBLOCK));
		stop_wave(&(wdev->play_stream));
		return SUCCESS;

	case SNDCTL_DSP_SYNC:
		if (wdev->play_stream.wave) {
			drain_playback(wdev, (file->f_flags & O_NONBLOCK));
			stop_wave(&(wdev->play_stream));
		}
		if (wdev->record_stream.wave) {
			stop_wave(&(wdev->record_stream));
		}
		return SUCCESS;

	case SNDCTL_DSP_SETDUPLEX:
		return SUCCESS;

	case SNDCTL_DSP_GETCAPS:
		return put_user(DSP_CAP_DUPLEX | DSP_CAP_REALTIME | DSP_CAP_TRIGGER | DSP_CAP_MMAP, (int *)arg);

	case SNDCTL_DSP_RESET:
		stop_wave(&(wdev->play_stream));
		stop_wave(&(wdev->record_stream));
		return SUCCESS;

	case SNDCTL_DSP_SPEED:
		get_user_ret(val, (int *)arg, -EFAULT);
		wdev->play_stream.sample_rate =  wdev->record_stream.sample_rate = val;
		return put_user(val, (int *)arg);

	case SNDCTL_DSP_STEREO:
		get_user_ret(val, (int *)arg, -EFAULT);
		wdev->play_stream.channel_count = wdev->record_stream.channel_count = (val ? 2 : 1);
		return put_user(val, (int *)arg);
      
	case SNDCTL_DSP_CHANNELS:
		get_user_ret(val, (int *)arg, -EFAULT);
		/* fixme: quad support */
		wdev->play_stream.channel_count = wdev->record_stream.channel_count = (val == 1 ? 1 : 2);
		return put_user(val, (int *)arg);
    
	case SNDCTL_DSP_GETFMTS:
		return put_user(AFMT_MU_LAW | AFMT_A_LAW | AFMT_S16_LE | AFMT_U8, (int *)arg);

	case SNDCTL_DSP_SETFMT:
		get_user_ret(val, (int *)arg, -EFAULT);
		if (val != AFMT_QUERY) {
			set_data_format(&wdev->play_stream, val);
			set_data_format(&wdev->record_stream, val);
		}
		return put_user(wdev->play_stream.data_format, (int *)arg);

	case SNDCTL_DSP_GETTRIGGER:
		val = 0;
		if (wdev->record_stream.wave && !wdev->record_stream.pre_roll)
			val |= PCM_ENABLE_INPUT;
		if (wdev->play_stream.wave && !wdev->play_stream.pre_roll)
			val |= PCM_ENABLE_OUTPUT;
		return put_user(val, (int *)arg);
		
	case SNDCTL_DSP_SETTRIGGER:
		get_user_ret(val, (int *)arg, -EFAULT);
		if (val & PCM_ENABLE_INPUT) {
			stream = &wdev->record_stream;
			if (stream->wave && (stream->pre_roll || stream->mapped)) {
				if (stream->mapped && !stream->mapped_setup) {
					set_asp_format(stream);
					for (i = 0; i < stream->num_frags; i++) {
						stream->buf[i].count = stream->frag_size;
						stream->buf[i].in_use = TRUE;
						AddWaveBuffer(stream->wave, stream->buf[i].page, stream->frag_size);
					}
					stream->mapped_setup = TRUE;
				}
				StartWave(stream->wave);
				stream->pre_roll = FALSE;
			}
		} else if (wdev->record_stream.wave) {
			if (wdev->record_stream.mapped && wdev->record_stream.mapped_setup)
				StopWave(wdev->record_stream.wave);
			wdev->record_stream.pre_roll = TRUE;
		}
		if (val & PCM_ENABLE_OUTPUT) {
			stream = &wdev->play_stream;
			if (stream->wave && (stream->pre_roll || stream->mapped)) {
				if (stream->mapped && !stream->mapped_setup) {
					set_asp_format(stream);
					for (i = 0; i < stream->num_frags; i++) {
						stream->buf[i].count = stream->frag_size;
						stream->buf[i].in_use = TRUE;
						memset(stream->buf[i].page, (stream->data_format == AFMT_S16_LE) ? 0 : 0x80, stream->frag_size);
						AddWaveBuffer(stream->wave, stream->buf[i].page, stream->frag_size);
					}
					stream->mapped_setup = TRUE;
				}
				StartWave(stream->wave);
				stream->pre_roll = FALSE;
			}
		} else if (wdev->play_stream.wave) {
			if (wdev->play_stream.mapped && wdev->play_stream.mapped_setup)
				StopWave(wdev->play_stream.wave);
			wdev->play_stream.pre_roll = TRUE;
		}
		return SUCCESS;

	case SNDCTL_DSP_GETOSPACE:
		if (!wdev->play_stream.wave)
			return -EINVAL;
		abinfo.fragsize = wdev->play_stream.frag_size;
		abinfo.fragstotal = wdev->play_stream.num_frags;
		abinfo.bytes = 0;
		for (i = 0; i < wdev->play_stream.num_frags; i++) {
			if (!wdev->play_stream.buf[i].in_use)
				abinfo.bytes += (wdev->play_stream.frag_size - wdev->play_stream.buf[i].count);
		}
		abinfo.fragments = abinfo.bytes / wdev->play_stream.frag_size ;
		return copy_to_user((void *)arg, &abinfo, sizeof(abinfo)) ? -EFAULT : 0;

	case SNDCTL_DSP_GETISPACE:
		if (!wdev->record_stream.wave)
			return -EINVAL;
		abinfo.fragsize = wdev->record_stream.frag_size;
		abinfo.fragstotal = wdev->record_stream.num_frags;
		abinfo.bytes = 0;
		/* fixme: this is questionable... */
		for (i = 0; i < wdev->record_stream.num_frags; i++) {
			if (!wdev->record_stream.buf[i].in_use)
				abinfo.bytes += (wdev->record_stream.frag_size - wdev->record_stream.buf[i].count);
		}
		abinfo.fragments = abinfo.bytes / wdev->record_stream.frag_size ;
		return copy_to_user((void *)arg, &abinfo, sizeof(abinfo)) ? -EFAULT : 0;

	case SNDCTL_DSP_NONBLOCK:
		file->f_flags |= O_NONBLOCK;
		return SUCCESS;

	case SNDCTL_DSP_GETODELAY:
		val = 0;
		for (i = 0; i < wdev->play_stream.num_frags; i++) {
			val += (wdev->play_stream.buf[i].count - wdev->play_stream.buf[i].xfer);
		}
		return put_user(val, (int *)arg);

	case SNDCTL_DSP_GETIPTR:
		stream = &wdev->record_stream;
		cinfo.bytes = stream->total_played;
		cinfo.blocks = stream->total_played / stream->frag_size;
		cinfo.ptr = (stream->frag_size * stream->fill_index) + stream->buf[stream->fill_index].xfer;
		return copy_to_user((void *)arg, &cinfo, sizeof(cinfo));

	case SNDCTL_DSP_GETOPTR:
		stream = &wdev->play_stream;
		cinfo.bytes = stream->total_played;
		cinfo.blocks = stream->total_played / stream->frag_size;
		cinfo.ptr = (stream->frag_size * stream->drain_index) + stream->buf[stream->drain_index].xfer;
		return copy_to_user((void *)arg, &cinfo, sizeof(cinfo));

	case SNDCTL_DSP_GETBLKSIZE:
		return put_user(wdev->play_stream.frag_size, (int *)arg);

	case SNDCTL_DSP_SETFRAGMENT:
		get_user_ret(val, (int *)arg, -EFAULT);
		PDEBUG("DSP_SETFRAMENT: 0x%x\n", val);
		/* fixme - because of S/G hardware we only support a fixed (4096) frag size */
		return SUCCESS;

	case SNDCTL_DSP_SUBDIVIDE:
		get_user_ret(val, (int *)arg, -EFAULT);
		PDEBUG("DSP_SUBDIVIDE: 0x%x\n", val);
		/* fixme */
		return SUCCESS;

	case SOUND_PCM_WRITE_FILTER:
	case SNDCTL_DSP_SETSYNCRO:
	case SOUND_PCM_READ_RATE:
	case SOUND_PCM_READ_CHANNELS:
	case SOUND_PCM_READ_BITS:
	case SOUND_PCM_READ_FILTER:
		return -EINVAL;
	}
	return au_mixer_ioctl(inode, file, cmd, arg);
}

static int au_audio_mmap(struct file *file, struct vm_area_struct *vma)
{
	au_device_t *dev = file->private_data;
	wave_device_t *wdev;
	unsigned long size;

	//PDEBUG("au_audio_mmap, minor = %d\n", dev->dev_audio);

	wdev = find_wave_device(dev, file);
	if (!wdev) {
		PWARN("could not find wave device for file = 0x%p\n", file);
		return -ENODEV;
	}
  
	if (vma->vm_offset != 0)
		return -EINVAL;
	size = vma->vm_end - vma->vm_start;

	if (vma->vm_flags & VM_WRITE) {
		wave_stream_t *stream = &wdev->play_stream;
		if (!stream->wave || stream->active || stream->mapped)
			return -EINVAL;
		if (size > (PAGE_SIZE << stream->buf_order))
			return -EINVAL;
		if (remap_page_range(vma->vm_start, virt_to_phys(stream->dmabuf), size, vma->vm_page_prot))
			return -EAGAIN;
		stream->mapped = TRUE;
	}
	if (vma->vm_flags & VM_READ) {
		wave_stream_t *stream = &wdev->record_stream;
		if (!stream->wave || stream->active || stream->mapped)
			return -EINVAL;
		if (size > (PAGE_SIZE << stream->buf_order))
			return -EINVAL;
		if (remap_page_range(vma->vm_start, virt_to_phys(stream->dmabuf), size, vma->vm_page_prot))
			return -EAGAIN;
		stream->mapped = TRUE;
	}
	return SUCCESS;
}

/*
 * Open a wave audio device.
 * If FMODE_WRITE is set, open a playback stream.
 * If FMODE_READ is set, open a recording stream.
 * If both are set, full duplex is possible.
 * Multiple opens are allowed, up to the number of
 * available vortex dma channels.
 *
 * FIXME: if BLOCKING open and no dma channel is
 * free, implement a wait.
 */
static int au_audio_open(struct inode *inode, struct file *file)
{
	au_device_t *dev;
	int minor_num = MINOR(inode->i_rdev);
	int dev_type = DEVICE_TYPE(minor_num);
	int status;

	//PDEBUG("au_audio_open, minor = %d\n", minor_num);

	dev = find_device(minor_num);
	if (!dev) {
		PWARN("could not find device for minor = %d\n", minor_num);
		return -ENODEV;
	}

	/* add new wave device and associate it with this file pointer */
	status = add_wave_device(dev, file, dev_type);
	if (status != 0) {
		PWARN("could not allocate wave device, status = %d\n", status);
		return -ENOMEM;
	}

	file->private_data = (void *)dev;

#ifndef DEBUG_AU
	MOD_INC_USE_COUNT;
#endif
	return SUCCESS;
}

static int au_audio_release(struct inode *inode, struct file *file)
{
	au_device_t *dev = file->private_data;
	wave_device_t *wdev;

	//PDEBUG("au_audio_release, minor = %d\n", dev->dev_audio);

	wdev = find_wave_device(dev, file);
	if (!wdev) {
		PWARN("could not find wave device for file = 0x%p\n", file);
		return -ENODEV;
	}
  
	if (wdev->play_stream.wave)
		drain_playback(wdev, (file->f_flags & O_NONBLOCK));
	if (wdev->record_stream.wave)
		dev->record_in_use = FALSE;
  
	/* kill streams */
	stop_wave(&(wdev->play_stream));
	stop_wave(&(wdev->record_stream));

	remove_wave_device(dev, wdev);

#ifndef DEBUG_AU
	MOD_DEC_USE_COUNT;
#endif
	return SUCCESS;
}

static void drain_playback(wave_device_t *wdev, int nonblock)
{
	if (wdev->play_stream.mapped)
		return;

	while (wdev->play_stream.total_played < wdev->play_stream.total_sent) {
		if (!wdev->play_stream.active) {
			wdev->play_stream.active = TRUE;
			StartWave(&(wdev->play_stream.wave));
		}
		if (nonblock)
			break;
		interruptible_sleep_on(&wdev->play_stream.io_wait);
		if (signal_pending(current))	/* app was killed */
			break;
	}
}

static int add_wave_device(au_device_t *dev, struct file *file, int dev_type)
{
	wave_device_t *wdev;
	int status;
	unsigned long flags = 0;
	int enable_playback = file->f_mode & FMODE_WRITE;
	int enable_record = file->f_mode & FMODE_READ;
	ASPWAVEFORMAT fmt = { 8, 1, 8000, ASPFMTLINEAR8 }; /* arbitrary format, reset by read/write */

	/* only allow one record stream */
	if (enable_record && dev->record_in_use)
		return -EBUSY;

	wdev = (wave_device_t *)kmalloc(sizeof(wave_device_t), GFP_KERNEL);
	if (!wdev)
		return -ENOMEM;
	memset(wdev, 0, sizeof(wave_device_t));

	/* set default data format based on device type */
	wdev->play_stream.sample_rate = 8000;
	wdev->play_stream.channel_count = 1;
	wdev->record_stream.sample_rate = 8000;
	wdev->record_stream.channel_count = 1;
	switch (dev_type) {
	case DEV_AUDIO:
	default:
		wdev->play_stream.data_format = AFMT_U8;
		wdev->record_stream.data_format = AFMT_U8;
		break;
	case DEV_SUNDSP:
		wdev->play_stream.data_format = AFMT_MU_LAW;
		wdev->record_stream.data_format = AFMT_MU_LAW;
		break;
	}

	wdev->fp = file;
	init_waitqueue(&wdev->play_stream.io_wait);
	init_waitqueue(&wdev->record_stream.io_wait);

	/* fixme: frag size should be dynamic and calculated based on stream format */
	wdev->play_stream.frag_size = DEF_FRAG_SIZE;
	wdev->play_stream.num_frags = DEF_NR_FRAGS;
	wdev->record_stream.frag_size = DEF_FRAG_SIZE;
	wdev->record_stream.num_frags = DEF_NR_FRAGS;

	if (enable_playback) {
		if (get_frags(&(wdev->play_stream), wdev->play_stream.num_frags) != SUCCESS) {
			PWARN("could not get playback stream pages\n");
			free_wave_device(wdev);
			return -ENOMEM;
		}
		status = CreateWaveBuffer(dev->core, &fmt, ASPDIRPLAY, &wdev->play_stream.wave);
		if (status != 0) {
			PWARN("could not create play stream wave buffer, status=%d\n", status);
			free_wave_device(wdev);
			return -EBUSY;	// ran out of dma channels
		}
	}

	if (enable_record) {
		if (get_frags(&(wdev->record_stream), wdev->record_stream.num_frags) != SUCCESS) {
			PWARN("could not get record stream pages\n");
			free_wave_device(wdev);
			return -ENOMEM;
		}
		status = CreateWaveBuffer(dev->core, &fmt, ASPDIRRECORD, &wdev->record_stream.wave);
		if (status != 0) {
			PWARN("could not create record stream wave buffer, status=%d\n", status);
			free_wave_device(wdev);
			return -ENOMEM;
		}
		dev->record_in_use = TRUE;
	}

	spin_lock_irqsave(&dev->irq_lock, flags);

	/* link to end of wave device list */
	if (!dev->wave_head) {
		dev->wave_head = dev->wave_tail = wdev;
	} else {
		dev->wave_tail->next = wdev;
		wdev->prev = dev->wave_tail;
		dev->wave_tail = wdev;
	}
	spin_unlock_irqrestore(&dev->irq_lock, flags);
	return SUCCESS;  
}

static int get_frags(wave_stream_t *stream, unsigned long count)
{
	int order = 2; // fixme: base on count
	unsigned long page_addr;
	unsigned long map, mapend;
	int i;

	stream->dmabuf = (void *)__get_free_pages(GFP_KERNEL, order);
	if (!stream->dmabuf)
		return -ENOMEM;

	stream->buf_order = order;

	/* now mark the pages as reserved; otherwise remap_page_range doesn't do what we want */
	/* this is from the es1371 driver */
	mapend = MAP_NR(stream->dmabuf + (PAGE_SIZE << order) - 1);
	for (map = MAP_NR(stream->dmabuf); map <= mapend; map++)
		set_bit(PG_reserved, &mem_map[map].flags);

	page_addr = (unsigned long)stream->dmabuf;
	for (i = 0; i < count; i++) {
		stream->buf[i].page = (void *)page_addr;
		page_addr += PAGE_SIZE;
	}
	return SUCCESS;
}

static void free_wave_device(wave_device_t *wdev)
{
	unsigned long map, mapend;

	if (wdev->play_stream.dmabuf) {
		/* undo marking the pages as reserved */
		mapend = MAP_NR(wdev->play_stream.dmabuf + (PAGE_SIZE << wdev->play_stream.buf_order) - 1);
		for (map = MAP_NR(wdev->play_stream.dmabuf); map <= mapend; map++)
			clear_bit(PG_reserved, &mem_map[map].flags);	
		free_pages((unsigned long)wdev->play_stream.dmabuf, wdev->play_stream.buf_order);
	}
	if (wdev->record_stream.dmabuf) {
		mapend = MAP_NR(wdev->record_stream.dmabuf + (PAGE_SIZE << wdev->record_stream.buf_order) - 1);
		for (map = MAP_NR(wdev->record_stream.dmabuf); map <= mapend; map++)
			clear_bit(PG_reserved, &mem_map[map].flags);	
		free_pages((unsigned long)wdev->record_stream.dmabuf, wdev->record_stream.buf_order);
	}

	if (wdev->play_stream.wave)
		ReleaseWave(wdev->play_stream.wave);
	if (wdev->record_stream.wave)
		ReleaseWave(wdev->record_stream.wave);
	kfree(wdev);
}

static wave_device_t *find_wave_device(au_device_t *dev, struct file *file)
{
	wave_device_t *wdev = dev->wave_head;

	while (wdev) {
		if (wdev->fp == file)
			break;
		wdev = wdev->next;
	}
	return wdev;
}

static void remove_wave_device(au_device_t *dev, wave_device_t *wdev)
{
	unsigned long flags = 0;

	spin_lock_irqsave(&dev->irq_lock, flags);

	/* unlink the wave device */
	if (wdev == dev->wave_head) {
		dev->wave_head = wdev->next;
		if (dev->wave_head)
			dev->wave_head->prev = NULL;
	} else {
		wdev->prev->next = wdev->next;
		if (wdev->next)
			wdev->next->prev = wdev->prev;
		else
			dev->wave_tail = wdev->prev;
	}
	spin_unlock_irqrestore(&dev->irq_lock, flags);

	free_wave_device(wdev);
}

/*
 * If the passed format is not valid, the stream's format
 * is not changed.
 */
static void set_data_format(wave_stream_t *stream, int val)
{
	switch (val) {
	case AFMT_MU_LAW:
	case AFMT_A_LAW:
	case AFMT_S16_LE:
	case AFMT_U8:
		stream->data_format = val;
		break;
	case AFMT_S16_BE:
	case AFMT_U16_LE:
	case AFMT_U16_BE:
	case AFMT_IMA_ADPCM:
	case AFMT_S8:
		/* fixme: do the conversion */
	default:
		PWARN("unsupported data format: %d\n", val);
	}
}

static void set_asp_format(wave_stream_t *stream)
{
	ASPWAVEFORMAT fmt;

	switch (stream->data_format) {
	case AFMT_S16_LE:
	default:
		fmt.eEncoding = ASPFMTLINEAR16;
		fmt.wBitWidth = 16;
		break;
	case AFMT_U8:
		fmt.eEncoding = ASPFMTLINEAR8;
		fmt.wBitWidth = 8;
		break;
	case AFMT_A_LAW:
		fmt.eEncoding = ASPFMTALAW;
		fmt.wBitWidth = 8;
		break;
	case AFMT_MU_LAW:
		fmt.eEncoding = ASPFMTULAW;
		fmt.wBitWidth = 8;
		break;
	}

	fmt.wChannels = stream->channel_count ? stream->channel_count : 1;
	fmt.dwSampleRate = stream->sample_rate ? stream->sample_rate : 8000;

	//PDEBUG("bw %d ch %d sr %d en %d\n", fmt.wBitWidth, fmt.wChannels, (int)fmt.dwSampleRate, fmt.eEncoding);
	SetWaveFormat(stream->wave, &fmt);
}

static void stop_wave(wave_stream_t *stream)
{
	int i;

	if (stream->wave)
		StopWave(stream->wave);
	stream->active = stream->pre_roll = FALSE;
	stream->total_played = stream->total_sent = 0;
	stream->fill_index = stream->drain_index = 0;
	for (i = 0; i < MAX_FRAGS; i++) {
		stream->buf[i].count = stream->buf[i].xfer = 0;
		stream->buf[i].in_use = FALSE;
	}
}

static unsigned long receive_mem(wave_stream_t *stream, char *mem, unsigned long max_size)
{
	unsigned long received = 0;
	unsigned long count;
	int i;

	/*
	 * Add the first buffers to an inactive stream.
	 */
	if (!stream->active) {
		for (i = 0; i < stream->num_frags; i++) {
			stream->buf[i].count = stream->frag_size;
			stream->buf[i].in_use = TRUE;
			AddWaveBuffer(stream->wave, stream->buf[i].page, stream->frag_size);
		}
		return 0; /* no memory received yet */
	}

	while ((received < max_size) && !stream->buf[stream->drain_index].in_use) {
		count = max_size < stream->frag_size ? max_size : stream->frag_size;
		copy_to_user(mem, stream->buf[stream->drain_index].page, count);
		/*
		 * Add this buffer back.
		 */
		stream->buf[stream->drain_index].count = stream->frag_size;
		stream->buf[stream->drain_index].xfer = 0;
		stream->buf[stream->drain_index].in_use = TRUE;
		AddWaveBuffer(stream->wave, stream->buf[stream->drain_index].page, stream->frag_size);

		stream->drain_index++;
		if (stream->drain_index == stream->num_frags)
			stream->drain_index = 0;
		received += count;
		mem += count;
	}
	return received;
}

static unsigned long send_mem(wave_stream_t *stream, const char *mem, unsigned long size)
{
	unsigned long sent = 0;
	unsigned long count;
	mem_buf_t *mb = &stream->buf[stream->fill_index];

	/*
	 * Put as much as we can into each unused buffer.
	 */
	while (size && !mb->in_use) {
		count = size < stream->frag_size - mb->count ? size : stream->frag_size - mb->count;
		copy_from_user(mb->page + mb->count, mem, count);
		mb->count += count;

		/*
		 * Send buffer to the dma engine if full.
		 */
		if (mb->count >= stream->frag_size) {
			mb->in_use = TRUE;
			mb->xfer = 0;
			AddWaveBuffer(stream->wave, mb->page, mb->count);

			/* update the fill buffer pointer, wrap if needed */
			stream->fill_index++;
			if (stream->fill_index == stream->num_frags)
				stream->fill_index = 0;
			mb = &stream->buf[stream->fill_index];
		}

		size -= count;
		sent += count;
		mem += count;
	}
	return sent;
}

static void au_handle_timer(au_device_t *dev)
{
	wave_device_t *wdev = dev->wave_head;
	wave_stream_t *stream;
	mem_buf_t *cur;
	unsigned long count, remaining;

	while (wdev) {

		if (wdev->play_stream.active || wdev->play_stream.mapped) {
			stream = &wdev->play_stream;
			count = GetWaveBytesPlayed(stream->wave);
			while (count) {
				cur = &(stream->buf[stream->drain_index]);
				/* assert cur->in_use */
				stream->total_played += count;
				remaining = cur->count - cur->xfer;
				if (count < remaining) {
					cur->xfer += count;
					count = 0;
				} else {
					stream->drain_index++;
					if (stream->drain_index == stream->num_frags) {
						stream->drain_index = 0;
					}
					if (stream->mapped) {
						cur->xfer = 0;
						AddWaveBuffer(stream->wave, cur->page, cur->count);
					} else {
						cur->xfer = cur->count;
						cur->count = 0;
						cur->in_use = FALSE;
						wake_up_interruptible(&stream->io_wait);
					}
					count -= remaining;
				}
			}
		}

		if (wdev->record_stream.active || wdev->record_stream.mapped) {
			stream = &wdev->record_stream;
			count = GetWaveBytesPlayed(stream->wave);
			while (count) {
				cur = &(stream->buf[stream->fill_index]);
				/* assert cur->in_use */
				stream->total_played += count;
				remaining = cur->count - cur->xfer;
				if (count < remaining) {
					cur->xfer += count;
					count = 0;
				} else {
					stream->fill_index++;
					if (stream->fill_index == stream->num_frags) {
						stream->fill_index = 0;
					}
					if (stream->mapped) {
						cur->xfer = 0;
						AddWaveBuffer(stream->wave, cur->page, cur->count);
					} else {
						cur->xfer = cur->count;
						cur->count = 0;
						cur->in_use = FALSE;
						wake_up_interruptible(&stream->io_wait);
					}
					count -= remaining;
				}
			}
		}
		wdev = wdev->next;
	}
}

/********************************/
/* Mixer support		*/
/********************************/

static int au_mixer_ioctl(struct inode *inode, struct file *file, unsigned int cmd, unsigned long arg)
{
	au_device_t *dev = file->private_data;
	int val, i;

	//PDEBUG("au_mixer_ioctl, minor %d cmd 0x%x\n", dev->dev_mixer, cmd);

	if (cmd == OSS_GETVERSION) {
		return put_user(SOUND_VERSION, (int *)arg);
	}
	if (cmd == SOUND_MIXER_INFO) {
		mixer_info info;
		strncpy(info.id, "AU88xx", sizeof(info.id));
		strncpy(info.name, "Aureal AU88xx", sizeof(info.name));
		info.modify_counter = dev->mix_modcnt;
		if (copy_to_user((void *)arg, &info, sizeof(info)))
			return -EFAULT;
		return SUCCESS;
	}
	if (cmd == SOUND_OLD_MIXER_INFO) {
		_old_mixer_info info;
		strncpy(info.id, "AU88xx", sizeof(info.id));
		strncpy(info.name, "Aureal AU88xx", sizeof(info.name));
		if (copy_to_user((void *)arg, &info, sizeof(info)))
			return -EFAULT;
		return SUCCESS;
	}

	if (_IOC_TYPE(cmd) != 'M' || _IOC_SIZE(cmd) != sizeof(int)) {
		//fixme I get a 0x5401 on RedHat 6.1
		//PWARN("invalid mixer read ioctl 0x%x\n", cmd);
		return -EINVAL;
	}

	if (_IOC_DIR(cmd) == _IOC_READ) {
		switch (_IOC_NR(cmd)) {
    
		case SOUND_MIXER_RECSRC:
			return put_user(dev->mix_recsrc, (int *)arg);

		case SOUND_MIXER_DEVMASK:
			/* fixme: get codec caps and add tone, headphone... */
			return put_user(SOUND_MASK_LINE | SOUND_MASK_CD | SOUND_MASK_VIDEO |
					SOUND_MASK_LINE1 | SOUND_MASK_PCM | SOUND_MASK_VOLUME |
					SOUND_MASK_OGAIN | SOUND_MASK_PHONEIN | SOUND_MASK_SPEAKER |
					SOUND_MASK_MIC | SOUND_MASK_RECLEV, (int *)arg);

		case SOUND_MIXER_RECMASK:
			return put_user(SOUND_MASK_MIC | SOUND_MASK_CD | SOUND_MASK_VIDEO | SOUND_MASK_LINE1 |
					SOUND_MASK_LINE | SOUND_MASK_VOLUME | SOUND_MASK_PHONEOUT |
					SOUND_MASK_PHONEIN, (int *)arg);

		case SOUND_MIXER_STEREODEVS:
			return put_user(SOUND_MASK_LINE | SOUND_MASK_CD | SOUND_MASK_VIDEO |
					SOUND_MASK_LINE1 | SOUND_MASK_PCM | SOUND_MASK_VOLUME |
					SOUND_MASK_PHONEOUT | SOUND_MASK_RECLEV, (int *)arg);

		case SOUND_MIXER_CAPS:
			return put_user(SOUND_CAP_EXCL_INPUT, (int *)arg);

		default:
			i = _IOC_NR(cmd);
			if (i >= SOUND_MIXER_NRDEVICES)
				return -EINVAL;
			return put_user(dev->mix_vol[i], (int *)arg);
		}
	}

	if (_IOC_DIR(cmd) != (_IOC_READ|_IOC_WRITE)) {
		PWARN("invalid mixer write ioctl 0x%x\n", cmd);
		return -EINVAL;
	}

	dev->mix_modcnt++;
	switch (_IOC_NR(cmd)) {

	case SOUND_MIXER_RECSRC:
		get_user_ret(dev->mix_recsrc, (int *)arg, -EFAULT);
		set_rec_src(dev, dev->mix_recsrc);
		return SUCCESS;

	default:
		i = _IOC_NR(cmd);
		if (i >= SOUND_MIXER_NRDEVICES)
			return -EINVAL;
		get_user_ret(val, (int *)arg, -EFAULT);
		write_mixer(dev, (unsigned)i, val);
		dev->mix_vol[i] = val;
		return put_user(val, (int *)arg);
	}
}

static void write_mixer(au_device_t *dev, unsigned int chan, int val)
{
	unsigned int left, right;
	unsigned int boost = 0;
	short reg;

	left = val & 0xff;
	right = (val >> 8) & 0xff;
	if (left > 100)
		left = 100;
	if (right > 100)
		right = 100;

	switch (chan) {
	case SOUND_MIXER_VOLUME:
		if (left < 7 && right < 7) {
			WriteCodec(dev->core, AC97REG_MASTER, 0x8000);
			break;
		}
		if (left < 7)
			left = 7;
		if (right < 7)
			right = 7;
		/* codec scales 0x1f (-46db) to 0 (loudest) */
		WriteCodec(dev->core, AC97REG_MASTER, (((100 - left) / 3) << 8) | ((100 - right) / 3));
		break;
	case SOUND_MIXER_PCM:
		if (dev->dev_id == DEV_AU8820) {
			// Must update codec pcm register on 8820
			if (left < 7 && right < 7) {
				WriteCodec(dev->core, AC97REG_PCM, 0x8000);
				break;
			} else {
				WriteCodec(dev->core, AC97REG_PCM, 0x0606);
			}
		}
		/* pcm scales 0 to 65535 */
		WriteMixer(dev->core, ASP_MIX_INPUT_WAVE_AT_PCM, (left * 655), (right * 655));
		break;
	case SOUND_MIXER_LINE:
	case SOUND_MIXER_VIDEO:
	case SOUND_MIXER_LINE1:
	case SOUND_MIXER_CD:
		switch (chan) {
		case SOUND_MIXER_LINE:
			reg = AC97REG_LINE_IN;
			break;
		case SOUND_MIXER_LINE1:
			reg = AC97REG_AUX;
			break;
		case SOUND_MIXER_VIDEO:
			reg = AC97REG_VIDEO;
			break;
		case SOUND_MIXER_CD:
		default:
			reg = AC97REG_CD;
			break;
		}
		if (left < 7 && right < 7) {
			WriteCodec(dev->core, reg, 0x8000);
			break;
		}
		if (left < 7)
			left = 7;
		if (right < 7)
			right = 7;
		WriteCodec(dev->core, reg, (((100 - left) / 3) << 8) | ((100 - right) / 3));
		break;
	case SOUND_MIXER_TREBLE:
	case SOUND_MIXER_BASS:
		/* fixme: check codec caps, use EQ block */
		break;
	case SOUND_MIXER_PHONEOUT:
	case SOUND_MIXER_IGAIN:
		/* fixme: check codec caps */
		break;
	case SOUND_MIXER_MIC:
		if (left < 11) {
			WriteCodec(dev->core, AC97REG_MIC, 0x8000);
			break;
		}
		boost = 0;
		if (left >= 27) {
			left -= 27;
			boost = 0x40;
		}
		if (left < 11)
			left = 11;
		WriteCodec(dev->core, AC97REG_MIC, ((73 - left) / 2) | boost);
		break;
	case SOUND_MIXER_SPEAKER:
		WriteCodec(dev->core, AC97REG_PC_BEEP, (left < 10) ? 0x8000 : ((100 - left) / 6) << 1);
		break;
	case SOUND_MIXER_OGAIN:
		WriteCodec(dev->core, AC97REG_MASTER_MONO, (left < 7) ? 0x8000 : (100 - left) / 3);
		break;
	case SOUND_MIXER_PHONEIN:
		WriteCodec(dev->core, AC97REG_PHONE, (left < 7) ? 0x8000 : (100 - left) / 3);
		break;
	case SOUND_MIXER_RECLEV:
		if (left < 10 || right < 10) {
			WriteCodec(dev->core, AC97REG_REC_GAIN, 0x8000);
			WriteMixer(dev->core, ASP_MIX_INPUT_WAVE_AT_WAVEIN, 0, 0);
			break;
		}
		if (left < 10)
			left = 10;
		if (right < 10)
			right = 10;
		WriteCodec(dev->core, AC97REG_REC_GAIN, (((left - 10) / 6) << 8) | ((right - 10) / 6));
		WriteMixer(dev->core, ASP_MIX_INPUT_WAVE_AT_WAVEIN, (left * 655), (right * 655));
		break;
	default:
		//PWARN("write mixer: unknown mixer chan %d\n", (int)chan);
		break;
	}
}

static int au_mixer_open(struct inode *inode, struct file *file)
{
	au_device_t *dev;
	int minor_num = MINOR(inode->i_rdev);

	//PDEBUG("au_mixer_open, minor = %d\n", minor_num);

	dev = find_device(minor_num);
	if (!dev) {
		PWARN("could not find device for minor = %d\n", minor_num);
		return -ENODEV;
	}
	file->private_data = (void *)dev;

#ifndef DEBUG_AU
	MOD_INC_USE_COUNT;
#endif
	return SUCCESS;
}

static int au_mixer_release(struct inode *inode, struct file *file)
{
#ifndef DEBUG_AU
	MOD_DEC_USE_COUNT;
#endif
	return SUCCESS;
}

static void init_mixer(au_device_t *dev)
{
	int i;
	for (i = 0; i < SOUND_MIXER_NRDEVICES; i++) {
		dev->mix_vol[i] = 0x5050;
		write_mixer(dev, i, dev->mix_vol[i]);
	}

	dev->mix_recsrc = SOUND_MASK_MIC;
	set_rec_src(dev, dev->mix_recsrc);		// default to mic source
}

static void set_rec_src(au_device_t *dev, unsigned int recsrc)
{
	short val;

	switch (recsrc) {
	case SOUND_MASK_CD:
		val = 0x1;
		break;
	case SOUND_MASK_VIDEO:
		val = 0x2;
		break;
	case SOUND_MASK_LINE1:	//aux?
		val = 0x3;
		break;
	case SOUND_MASK_LINE:
		val = 0x4;
		break;
	case SOUND_MASK_VOLUME:
		val = 0x5;
		break;
	case SOUND_MASK_PHONEOUT:	//mono mix?
		val = 0x6;
		break;
	case SOUND_MASK_PHONEIN:
		val = 0x7;
		break;
	case SOUND_MASK_MIC:
	default:
		val = 0;
		break;
	}
	WriteCodec(dev->core, AC97REG_REC_SELECT, (val << 8 | val));
}

/********************************/
/* Midi support			*/
/********************************/

static int au_midi_open(struct inode *inode, struct file *file)
{
	au_device_t *dev;
	int minor_num = MINOR(inode->i_rdev);
	int enable_output = file->f_mode & FMODE_WRITE;
	int enable_input = file->f_mode & FMODE_READ;
	unsigned long flags;

	//PDEBUG("au_midi_open, minor = %d\n", minor_num);

	dev = find_device(minor_num);
	if (!dev) {
		PWARN("could not find device for minor = %d\n", minor_num);
		return -ENODEV;
	}

	/* FIXME: if BLOCKING open and in use, implement a wait. */
	if (enable_output) {
		if (dev->midiout_in_use)
			return -EBUSY;
		dev->midiout_in_use = TRUE;
	}
	if (enable_input) {
		if (dev->midiin_in_use)
			return -EBUSY;
		dev->midiin_in_use = TRUE;
	}

	file->private_data = (void *)dev;

  	spin_lock_irqsave(&dev->irq_lock, flags);
	if (enable_output)
		dev->midi.ord = dev->midi.owr = dev->midi.ocnt = 0;
	if (enable_input)
		dev->midi.ird = dev->midi.iwr = dev->midi.icnt = 0;
	au_handle_midi(dev);
  	EnableMPU401Interrupt(dev->core, TRUE);
  	spin_unlock_irqrestore(&dev->irq_lock, flags);

#ifndef DEBUG_AU
	MOD_INC_USE_COUNT;
#endif
	return SUCCESS;
}

static int au_midi_release(struct inode *inode, struct file *file)
{
	au_device_t *dev = file->private_data;

	//PDEBUG("au_midi_release, minor = %d\n", dev->dev_midi);

	EnableMPU401Interrupt(dev->core, FALSE);
	dev->midiin_in_use = dev->midiout_in_use = FALSE;

#ifndef DEBUG_AU
	MOD_DEC_USE_COUNT;
#endif
	return SUCCESS;
}

static ssize_t au_midi_read(struct file *file, char *buffer, size_t count, loff_t *ppos)
{
	au_device_t *dev = file->private_data;
	ssize_t ret = 0;
	unsigned long flags;
	unsigned ptr;
	int cnt;

	//PDEBUG("au_midi_read, count=%d\n", count);

	if (ppos != &file->f_pos)
		return -ESPIPE;

	if (!access_ok(VERIFY_WRITE, buffer, count))
		return -EFAULT;

	while (count > 0) {
		spin_lock_irqsave(&dev->irq_lock, flags);
		ptr = dev->midi.ird;
		cnt = MIDIINBUF - ptr;
		if (dev->midi.icnt < cnt)
			cnt = dev->midi.icnt;
		spin_unlock_irqrestore(&dev->irq_lock, flags);
		if (cnt > count)
			cnt = count;
		if (cnt <= 0) {
			if (file->f_flags & O_NONBLOCK)
				return ret ? ret : -EAGAIN;
			interruptible_sleep_on(&dev->midi.iwait);
			if (signal_pending(current))
				return ret ? ret : -ERESTARTSYS;
			continue;
		}
		if (copy_to_user(buffer, dev->midi.ibuf + ptr, cnt))
			return ret ? ret : -EFAULT;
		ptr = (ptr + cnt) % MIDIINBUF;
		spin_lock_irqsave(&dev->irq_lock, flags);
		dev->midi.ird = ptr;
		dev->midi.icnt -= cnt;
		spin_unlock_irqrestore(&dev->irq_lock, flags);
		count -= cnt;
		buffer += cnt;
		ret += cnt;
	}
	return ret;
}

static ssize_t au_midi_write(struct file *file, const char *buffer, size_t count, loff_t *ppos)
{
	au_device_t *dev = file->private_data;
	ssize_t ret = 0;
	unsigned long flags;
	unsigned ptr;
	int cnt;

	//PDEBUG("au_midi_write, count=%d\n", count);

	if (ppos != &file->f_pos)
		return -ESPIPE;

	if (!access_ok(VERIFY_READ, buffer, count))
		return -EFAULT;

	while (count > 0) {
		spin_lock_irqsave(&dev->irq_lock, flags);
		ptr = dev->midi.owr;
		cnt = MIDIOUTBUF - ptr;
		if (dev->midi.ocnt + cnt > MIDIOUTBUF)
			cnt = MIDIOUTBUF - dev->midi.ocnt;
		if (cnt <= 0)
			au_handle_midi(dev);
		spin_unlock_irqrestore(&dev->irq_lock, flags);
		if (cnt > count)
			cnt = count;
		if (cnt <= 0) {
			if (file->f_flags & O_NONBLOCK)
				return ret ? ret : -EAGAIN;
			interruptible_sleep_on(&dev->midi.owait);
			if (signal_pending(current))
				return ret ? ret : -ERESTARTSYS;
			continue;
		}
		if (copy_from_user(dev->midi.obuf + ptr, buffer, cnt))
			return ret ? ret : -EFAULT;
		ptr = (ptr + cnt) % MIDIOUTBUF;
		spin_lock_irqsave(&dev->irq_lock, flags);
		dev->midi.owr = ptr;
		dev->midi.ocnt += cnt;
		spin_unlock_irqrestore(&dev->irq_lock, flags);
		count -= cnt;
		buffer += cnt;
		ret += cnt;
		spin_lock_irqsave(&dev->irq_lock, flags);
		au_handle_midi(dev);
		spin_unlock_irqrestore(&dev->irq_lock, flags);
	}
	return ret;
}

static unsigned int au_midi_poll(struct file *file, struct poll_table_struct *wait)
{
	au_device_t *dev = file->private_data;
	unsigned long flags;
	unsigned int mask = 0;

	if (file->f_mode & FMODE_WRITE)
		poll_wait(file, &dev->midi.owait, wait);
	if (file->f_mode & FMODE_READ)
		poll_wait(file, &dev->midi.iwait, wait);
	spin_lock_irqsave(&dev->irq_lock, flags);
	if (file->f_mode & FMODE_READ) {
		if (dev->midi.icnt > 0)
			mask |= POLLIN | POLLRDNORM;
	}
	if (file->f_mode & FMODE_WRITE) {
		if (dev->midi.ocnt < MIDIOUTBUF)
			mask |= POLLOUT | POLLWRNORM;
	}
	spin_unlock_irqrestore(&dev->irq_lock, flags);
	return mask;
}

static int au_midi_ioctl(struct inode *inode, struct file *file, unsigned int cmd, unsigned long arg)
{
	/* fixme */
	return SUCCESS;
}

static void au_handle_midi(au_device_t *dev)
{
	unsigned char ch;
	int wake;

	//PDEBUG("au_handle_midi\n");

	if (!(dev->midiin_in_use || dev->midiout_in_use))
		return;
	wake = FALSE;
	while ((Mpu401UartReadStatus(dev->core) & MPU401_MIDIVAL) == 0) {
		ch = Mpu401UartReadData(dev->core);
		if (dev->midi.icnt < MIDIINBUF) {
			dev->midi.ibuf[dev->midi.iwr] = ch;
			dev->midi.iwr = (dev->midi.iwr + 1) % MIDIINBUF;
			dev->midi.icnt++;
		}
		wake = TRUE;
	}
  	if (wake)
  		wake_up_interruptible(&dev->midi.iwait);

	wake = FALSE;
	while (dev->midi.ocnt > 0 && ((Mpu401UartReadStatus(dev->core) & MPU401_CMDOK) == 0)) {
		Mpu401UartWriteData(dev->core, dev->midi.obuf[dev->midi.ord]);
		dev->midi.ord = (dev->midi.ord + 1) % MIDIOUTBUF;
		dev->midi.ocnt--;
		if (dev->midi.ocnt < MIDIOUTBUF-16)
			wake = TRUE;
	}
  	if (wake)
  		wake_up_interruptible(&dev->midi.owait);
}

/********************************/
/* sndstat  support		*/
/********************************/

/*
 * Some of this sndstat code is from soundcard.c, which is
 * Copyright 1993-1997 Hannu Savolainen
 */

static int au_sndstat_open(struct inode *inode, struct file *file)
{
	PDEBUG("au_sndstat_open\n");

#ifndef DEBUG_AU
	MOD_INC_USE_COUNT;
#endif
	return SUCCESS;
}

static ssize_t au_sndstat_read(struct file *file, char *buffer, size_t count, loff_t *ppos)
{
	PDEBUG("au_sndstat_read, count=%d\n", count);
	return sndstat_file_read(file, buffer, count, &file->f_pos);
}

/* 4K page size but our output routines use some slack for overruns */
#define PROC_BLOCK_SIZE (3*1024)

/*
 * basically copied from fs/proc/generic.c:proc_file_read 
 * should be removed sometime in the future together with /dev/sndstat
 * (a symlink /dev/sndstat -> /proc/sound will do as well)
 */
static int sndstat_file_read(struct file * file, char * buf, size_t nbytes, loff_t *ppos)
{
	char	*page;
	int	retval=0;
	int	eof=0;
	int	n, count;
	char	*start;

	if (!(page = (char*) __get_free_page(GFP_KERNEL)))
		return -ENOMEM;

	while ((nbytes > 0) && !eof) 
	{
		count = MIN(PROC_BLOCK_SIZE, nbytes);

		start = NULL;
		n = sound_proc_get_info(page, &start, *ppos, count, 0);
		if (n < count)
			eof = 1;
                        
		if (!start) 
		{
			/* For proc files that are less than 4k */
			start = page + *ppos;
			n -= *ppos;
			if (n <= 0)
				break;
			if (n > count)
				n = count;
		}
		if (n == 0)
			break;  /* End of file */
		if (n < 0) 
		{
			if (retval == 0)
				retval = n;
			break;
		}
                
		n -= copy_to_user(buf, start, n);       /* BUG ??? */
		if (n == 0) 
		{
			if (retval == 0)
				retval = -EFAULT;
			break;
		}
                
		*ppos += n;     /* Move down the file */
		nbytes -= n;
		buf += n;
		retval += n;
	}
	free_page((unsigned long) page);
	return retval;
}

static int sound_proc_get_info(char *buffer, char **start, off_t offset, int length, int inout)
{
	int len, count;
	off_t begin = 0;
	au_device_t *dev;
	char *dev_str;

#ifdef MODULE
#define MODULEPROCSTRING "Driver loaded as a module"
#else
#define MODULEPROCSTRING "Driver compiled into kernel"
#endif

	if (!dev_head)
		return 0;

	len = sprintf(buffer, "OSS/Free: 0x%x\n"
		      "Load type: " MODULEPROCSTRING "\n"
		      "Kernel: %s %s %s %s %s\n"
		      "Config options: %x\n\nInstalled drivers: \n", SOUND_VERSION,
		      system_utsname.sysname, system_utsname.nodename, system_utsname.release,
		      system_utsname.version, system_utsname.machine, 0);
	len += sprintf(buffer + len, "Type 0: %s\n", "Aureal Vortex");

	len += sprintf(buffer + len, "\nCard config: \n");
	dev = dev_head;
	while (dev) {
		switch (dev->dev_id) {
		case DEV_AU8820:
			dev_str = "Au8820";
			break;
		case DEV_AU8810:
			dev_str = "Au8810";
			break;
		case DEV_AU8830:
			dev_str = "Au8830";
			break;
		default:
			dev_str = "Au88xx";
			break;
		}
		len += sprintf(buffer + len, "(");
		len += sprintf(buffer + len, "%s", dev_str);
		len += sprintf(buffer + len, " at 0x%lx", dev->pcidev->base_address[0]);
		len += sprintf(buffer + len, " irq %d", dev->pcidev->irq);
		len += sprintf(buffer + len, ")");
		len += sprintf(buffer + len, "\n");
		dev = dev->next;
	}

	len += sprintf(buffer + len, "\nAudio devices:\n");
	dev = dev_head;
	count = 0;
	while (dev) 
	{
		len += sprintf(buffer + len, "%d: %s%s\n", count++, "Au88xx", " (DUPLEX)");
		dev = dev->next;
	}

	len += sprintf(buffer + len, "\nSynth devices: \nNot supported by current driver\n");

	len += sprintf(buffer + len, "\nMidi devices: \n");
	dev = dev_head;
	count = 0;
	while (dev) 
	{
		len += sprintf(buffer + len, "%d: %s\n", count++, "Au88xx");
		dev = dev->next;
	}

	len += sprintf(buffer + len, "\nTimers:\n");
	len += sprintf(buffer + len, "0: %s\n", "System Clock");

	len += sprintf(buffer + len, "\nMixers:\n");
	dev = dev_head;
	count = 0;
	while (dev) 
	{
		len += sprintf(buffer + len, "%d: %s\n", count++, "Au88xx");
		dev = dev->next;
	}

	*start = buffer + (offset - begin);
	len -= (offset - begin);
	if (len > length) 
		len = length;
	return len;
}

static int au_sndstat_release(struct inode *inode, struct file *file)
{
	PDEBUG("au_sndstat_release\n");

#ifndef DEBUG_AU
	MOD_DEC_USE_COUNT;
#endif
	return SUCCESS;
}

/********************************/
/* Vortex core support		*/
/********************************/

void *linux_kmalloc(unsigned int size)
{
	return kmalloc(size, GFP_KERNEL);
}

void linux_kfree(void *p)
{
	if (p) {
		kfree(p);
	}
}

void *linux_ioremap(unsigned long offset, unsigned long size)
{
	return ioremap(offset, size);
}

void linux_iounmap(void *addr)
{
	iounmap(addr);
}

void linux_udelay(unsigned long usecs)
{
	udelay(usecs);
}

void linux_dprintf(const char *fmt, ...)
{
#ifdef DEBUG_AU
	char buf[256];
	va_list ap;
	//strcpy(buf, KERN_DEBUG DRIVER_NAME ": ");
	va_start(ap, fmt);
	vsprintf(buf, fmt, ap);
	va_end(ap);
	printk(buf);
	printk("\n");
#endif
}

void *linux_memset(void *s, int c, unsigned long n)
{
	return memset(s, c, (size_t)n);
}

void *linux_memcpy(void *s, const void *ct, unsigned long n)
{
	return memcpy(s, ct, (size_t)n);
}

unsigned long linux_virt_to_phys(volatile void *addr)
{
	return virt_to_bus(addr);
}

void *linux_phys_to_virt(unsigned long addr)
{
	return bus_to_virt(addr);
}

void linux_acquire_device_spinlock(unsigned long core_obj)
{
	unsigned long flags = 0;
	au_device_t *dev;
  
	if ((dev = find_device_core((void *)core_obj)) != NULL) {
		spin_lock_irqsave(&dev->irq_lock, flags);
	} else {
		/* not an error - core calls back during alloc, before
		   we have a dev object */
		//PWARN("could not find device for spin lock\n");
	}
}

void linux_release_device_spinlock(unsigned long core_obj)
{
	unsigned long flags = 0;
	au_device_t *dev;

	if ((dev = find_device_core((void *)core_obj)) != NULL) {
		spin_unlock_irqrestore(&dev->irq_lock, flags);
	} else {
		/* not an error - core calls back during alloc, before
		   we have a dev object */
		//PWARN("could not find device for spin lock\n");
	}
}

void linux_init_timer(unsigned long *timer, unsigned long func, unsigned long context)
{
	struct timer_list *new_timer = (struct timer_list *)kmalloc(sizeof(struct timer_list), GFP_KERNEL);
	au_timer_t *autimer = (au_timer_t *)kmalloc(sizeof(autimer), GFP_KERNEL);

	init_timer(new_timer);
	new_timer->function = (void *)func;
	new_timer->data = context;
	autimer->timer = new_timer;
	autimer->last_time = 0;
	*timer = (unsigned long)autimer;
}

void linux_add_timer(unsigned long timer, unsigned long ms)
{
	au_timer_t *autimer = (au_timer_t *)timer;
	autimer->timer->expires = jiffies + ((ms * HZ) / 1000);
	if (autimer->timer->expires > autimer->last_time) {
		autimer->last_time = autimer->timer->expires;
		add_timer(autimer->timer);
	} else {
		//PWARN("ignoring late timer, expires %d\n", (int)autimer->timer->expires);
	}
}

void linux_del_timer(unsigned long timer)
{
	au_timer_t *autimer = (au_timer_t *)timer;
	del_timer(autimer->timer);
	kfree((void *)autimer->timer);
	kfree((void *)autimer);
}

unsigned long linux_jiffies()
{
	return jiffies;
}
