/*
 *   Copyright (C) 2005 by Karye
 *   karye@users.sourceforge.net
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include <QtCore/qglobal.h>      // for QMessageLogContext, qInstallMessageH...
#include <cstdio>               // for fprintf, stderr
#include <cstdlib>              // for abort
#include <kaboutdata.h>          // for KAboutData, KAboutLicense, KAboutLic...
#include <KCrash>
#include <klocalizedstring.h>    // for i18n
#include <kmainwindow.h>         // for RESTORE
#include <qapplication.h>        // for QApplication
#include <qbytearray.h>          // for QByteArray
#include <qcommandlineparser.h>  // for QCommandLineParser
#include <qdebug.h>              // for QDebug
#include <qstring.h>             // for QString

#include "kuroo.h"               // for Kuroo
#include "settings.h"            // for KurooConfig

//the original shape of this before we broke it was <major>.<minor>.<patch>_<MMDDYY>_db<dbver> which is parsed in a few places
//this is stored in KurooConfig::hardVersion() immediately on startup, then also in KurooConfig::version() after doing the
//backup and refresh code in KurooInit::KurooInit
//changing the part after '_db' will cause kuroo to backup the db and regenerate from scratch
static const char version[] = "kuroo-1.2.4-250418_db48";

void myMessageOutput(QtMsgType type, const QMessageLogContext &context, const QString &msg)
{
	QByteArray localMsg = msg.toLocal8Bit();
	switch (type) {
	case QtDebugMsg:
		fprintf(stderr, "Debug: %s (%s:%u, %s)\n", localMsg.constData(), context.file, context.line, context.function);
		break;
	case QtInfoMsg:
		fprintf(stderr, "Info: %s (%s:%u, %s)\n", localMsg.constData(), context.file, context.line, context.function);
		break;
	case QtWarningMsg:
		fprintf(stderr, "Warning: %s (%s:%u, %s)\n", localMsg.constData(), context.file, context.line, context.function);
		break;
	case QtCriticalMsg:
		fprintf(stderr, "Critical: %s (%s:%u, %s)\n", localMsg.constData(), context.file, context.line, context.function);
		break;
	case QtFatalMsg:
		fprintf(stderr, "Fatal: %s (%s:%u, %s)\n", localMsg.constData(), context.file, context.line, context.function);
		abort();
	}
}

auto main( int argc, char **argv ) -> int
{
	qInstallMessageHandler(myMessageOutput);
	QApplication app(argc, argv);
	KLocalizedString::setApplicationDomain("org.gentoo.kuroo");
	KAboutData about(QStringLiteral("kuroo"),
					 i18n("Kuroo"),
					 QString::fromUtf8(version),
					 i18n("Frontend to Gentoo Portage"),
					 KAboutLicense::LicenseKey::GPL,
					 i18n("(C) 2006 karye"),
					 QString(),
					 QStringLiteral("https://sourceforge.net/p/kuroo"),
					 QStringLiteral("https://sourceforge.net/p/kuroo/bugs"));
	about.addAuthor(i18n("Amber Schenck"), i18n("Maintainer"), QStringLiteral("galiven@users.sourceforge.net"));
	about.addAuthor(i18n("Karye"), i18n("Original author and maintainer"), QStringLiteral("info@kuroo.org"));
	about.addAuthor(i18n("David C. Manuelda"), i18n("Previous developer and maintainer"), QStringLiteral("StormByte@gmail.com"));
	about.addAuthor(i18n("Matthias Faconneau"), i18n("Developer, port to KDE4"), QStringLiteral("faconneau@users.sourceforge.net"));
	about.addAuthor(i18n("Detlev Casanova"), i18n("Developer, port to KDE4"), QStringLiteral("cazou88@users.sourceforge.net"));
	about.addCredit(i18n("Gombault Damien"), i18n("French translation"), QStringLiteral("desintegr@gmail.com"));
	about.addCredit(i18n("Jan Schnackenberg"), i18n("German translation"), QStringLiteral("jan@schnackenberg.org"));
	about.addCredit(i18n("Alexander Reiterer"), i18n("German translation"), QStringLiteral("alexander.reiterer@tuwien.ac.at"));
	about.addCredit(i18n("Martin Baranski"), i18n("German translation"), QStringLiteral("eagle@eagle-cage.de"));
	about.addCredit(i18n("Matteo Azzali"), i18n("Italian translation"), QStringLiteral("kaioth@tiscalinet.it"));
	about.addCredit(i18n("Alexander N. Sørnes"), i18n("Norwegian translation"), QStringLiteral("alex@thehandofagony.com"));
	about.addCredit(i18n("Konrad Mantorski"), i18n("Polish translation"), QStringLiteral("konrad@mantorski.com"));
	about.addCredit(i18n("Wolfgang Bartelme"), i18n("Kuroo icons"), QStringLiteral("design@bartelme.at"));
	about.addCredit(i18n("Jakob Petsovits"), i18n("Portage version code"), QStringLiteral("jpetso@gmx.at"));
	about.addCredit(i18n("Björn Balazs"), i18n("OpenUsability"), QStringLiteral("B@lazs.de"));
	about.addCredit(i18n("Florian Graessle"), i18n("OpenUsability"), QStringLiteral("holehan@gmx.de"));
	about.setOrganizationDomain("gentoo.org");
	about.setDesktopFileName(QStringLiteral("org.gentoo.kuroo"));
	KAboutData::setApplicationData(about);
	// Nicolas Fella wrote to the mailing list that this should be _after_ KAboutData
	KCrash::initialize();
	
	// documentation in KAboutData.h has this little snippet, though it doesn't set the window icon in wayland :-(
	QApplication::setWindowIcon(QIcon::fromTheme(QStringLiteral("kuroo")));
	
	QCommandLineParser parser;
	parser.addVersionOption();
	parser.addHelpOption();
	about.setupCommandLine(&parser);
	parser.process(app);
	about.processCommandLine(&parser);

	qDebug() << "Kuroo version=" << version;

	KurooConfig::setHardVersion( QString::fromUtf8(version) );
	//KurooConfig::writeConfig();

	// see if we are starting with session management
	if ( app.isSessionRestored() ) {
		kRestoreMainWindows<Kuroo>();
	} else {
		auto *mainWindow = new Kuroo();
		mainWindow->setObjectName( QStringLiteral("kuroo") );
		mainWindow->show();
		//app.connect( app, SIGNAL( lastWindowClosed() ), app, SLOT( quit() ) );
	}

	return QApplication::exec();
}
