/***************************************************************************
 *	Copyright (C) 2004 by karye												*
 *	karye@users.sourceforge.net												*
 *																			*
 *	This program is free software; you can redistribute it and/or modify	*
 *	it under the terms of the GNU General Public License as published by	*
 *	the Free Software Foundation; either version 2 of the License, or		*
 *	(at your option) any later version.										*
 *																			*
 *	This program is distributed in the hope that it will be useful,			*
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of			*
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the			*
 *	GNU General Public License for more details.							*
 *																			*
 *	You should have received a copy of the GNU General Public License		*
 *	along with this program; if not, write to the							*
 *	Free Software Foundation, Inc.,											*
 *	59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.				*
 ***************************************************************************/


#ifndef CATEGORIESLISTVIEW_H
#define CATEGORIESLISTVIEW_H

#include <QMultiHash>
#include <QTreeWidget>
#include <qsize.h>        // for QSize
#include <qvector.h>      // for QVector

/**
 * @class CategoriesListView
 * @short Base class for category listview.
 */
class CategoriesView : public QTreeWidget
{
Q_OBJECT
public:
	CategoriesView( QWidget *parent = nullptr/*, const char *name = 0*/ );
	~CategoriesView();

	/**
	 * @class CategoriesView::CategoryItem
	 * @short Highlight empty category.
	 */
	class CategoryItem : public QTreeWidgetItem
	{
		public:
			CategoryItem( QTreeWidget* parent, const QString& name, QString id );

			inline void				setOn( bool on ) { m_on = on; /*repaint();*/ }
			inline const QString& 	id() const { return m_id; }
			inline const QString& 	name() const { return m_name; }

		protected:
			QString			m_id, m_name;
			bool 			m_on;
			// WARN: paintCell is obsolete, doesn't call through to anything anymore
			void 			paintCell( QPainter *p/*, int column, int width*/, Qt::Alignment alignment );
		// TODO: m_categoryIndex stores full CategoryItems by value rather than pointer, so it has to pretend to be "copyable"
		// even though QObject-based things aren't actually copyable
		//Q_DISABLE_COPY(CategoryItem)
};

	inline CategoryItem*		currentCategory() const { return dynamic_cast<CategoryItem*>( this->currentItem() ); }
	const QString				currentCategoryId() const;

protected Q_SLOTS:
	void						slotStoreFocus( QTreeWidgetItem* current/*, QTreeWidgetItem* previous */);

protected:
	void						restoreFocus( /*const bool& isFiltered */);

protected:

	// Category that has focus
	QString								m_focus;

	// Index of categoris in listview
	QMultiHash<QString, CategoryItem>	m_categoryIndex;

	// Vector containing all categories
	typedef QVector<CategoryItem*>		Categories;
	Categories							categories;

Q_SIGNALS:
	void								categoriesChanged();
};

/**
 * @class CategoriesListView
 * @short Categories listview.
 */
class CategoriesListView : public CategoriesView
{
Q_OBJECT
public:
	CategoriesListView( QWidget *parent = nullptr/*, const char *name = 0 */);
	~CategoriesListView();

	void						init();
	void 						loadCategories( const QStringList& categoriesList/*, bool isFiltered */);
	QSize 						sizeHint() const override;

};

/**
 * @class SubCategoriesListView
 * @short Subcategories listview.
 */
class SubCategoriesListView : public CategoriesView
{
Q_OBJECT
public:
	SubCategoriesListView( QWidget *parent = nullptr/*, const char *name = 0 */);
	~SubCategoriesListView();

	void						init();
	void 						loadCategories( const QStringList& categoriesList );
	QSize						sizeHint() const override;

private:

	// Vector containing all sub-categories
	QVector<QMultiMap<int, QString>>	allSubCategories;
};

#endif
