package rpmdb

func intRef(i ...int) *int {
	if len(i) == 0 {
		return nil
	}
	return &i[0]
}

type commonPackageInfo struct {
	Epoch           *int
	Name            string
	Version         string
	Release         string
	Arch            string
	SourceRpm       string
	Size            int
	License         string
	Vendor          string
	Modularitylabel string
	Summary         string
	SigMD5          string
}

func toPackageInfo(pkgs []*commonPackageInfo) []*PackageInfo {
	pkgList := make([]*PackageInfo, 0, len(pkgs))
	for _, p := range pkgs {
		pkgList = append(pkgList, &PackageInfo{
			Epoch:           p.Epoch,
			Name:            p.Name,
			Version:         p.Version,
			Release:         p.Release,
			Arch:            p.Arch,
			SourceRpm:       p.SourceRpm,
			Size:            p.Size,
			License:         p.License,
			Vendor:          p.Vendor,
			Modularitylabel: p.Modularitylabel,
			Summary:         p.Summary,
			SigMD5:          p.SigMD5,
		})
	}

	return pkgList
}

var (
	// docker run --rm -it centos:5 bash
	// rpm -qa --queryformat "\{%{EPOCH}, \"%{NAME}\", \"%{VERSION}\", \"%{RELEASE}\", \"%{ARCH}\", \"%{SOURCERPM}\", %{SIZE}, \"%{LICENSE}\", \"%{VENDOR}\", \"\", \"%{SUMMARY}\", \"%{SIGMD5}\"\},\n" | sed "s/^{(none)/{intRef()/g" | sed -r 's/^\{([0-9]+),/{intRef(\1),/' | sed "s/(none)/0/g"
	CentOS5Plain = func() []*PackageInfo {
		pkgs := []*commonPackageInfo{
			{intRef(), "setup", "2.5.58", "9.el5", "noarch", "setup-2.5.58-9.el5.src.rpm", 381527, "public domain", "CentOS", "", "A set of system configuration and setup files.", "9d46d21756374c2790de2d98d03ef456"},
			{intRef(), "basesystem", "8.0", "5.1.1.el5.centos", "noarch", "basesystem-8.0-5.1.1.el5.centos.src.rpm", 0, "public domain", "CentOS", "", "The skeleton package which defines a simple CentOS system.", "15c24ea32e8a66ea5000bb968950e511"},
			{intRef(), "cracklib-dicts", "2.8.9", "3.3", "x86_64", "cracklib-2.8.9-3.3.src.rpm", 8043753, "Artistic", "CentOS", "", "The standard CrackLib dictionaries.", "3855449eaef02f3e6a8d0df297f86680"},
			{intRef(), "tzdata", "2016c", "1.el5", "x86_64", "tzdata-2016c-1.el5.src.rpm", 1796593, "Public Domain", "CentOS", "", "Timezone data", "94eb2376a22cc23264310825feeeeecb"},
			{intRef(), "glibc", "2.5", "123.el5_11.3", "x86_64", "glibc-2.5-123.el5_11.3.src.rpm", 11690192, "LGPL", "CentOS", "", "The GNU libc libraries.", "3905753bc4c32e150a71363fa6356f5c"},
			{intRef(), "zlib", "1.2.3", "7.el5", "x86_64", "zlib-1.2.3-7.el5.src.rpm", 88976, "BSD", "CentOS", "", "The zlib compression and decompression library.", "096c61f65c6311a84c554dc4adce5900"},
			{intRef(), "popt", "1.10.2.3", "36.el5_11", "x86_64", "rpm-4.4.2.3-36.el5_11.src.rpm", 142842, "GPLv2+", "CentOS", "", "A C library for parsing command line parameters.", "cefd847e977028e9f6c6ec750ae3a9b3"},
			{intRef(), "glib2", "2.12.3", "4.el5_3.1", "x86_64", "glib2-2.12.3-4.el5_3.1.src.rpm", 2429701, "LGPL", "CentOS", "", "A library of handy utility functions", "e2ff154c67db3c6f1e370ca207b67653"},
			{intRef(), "audit-libs", "1.8", "2.el5", "x86_64", "audit-1.8-2.el5.src.rpm", 156639, "LGPL", "CentOS", "", "Dynamic library for libaudit", "ec272546a69df9f05e72de01468c1b04"},
			{intRef(), "bash", "3.2", "33.el5_11.4", "x86_64", "bash-3.2-33.el5_11.4.src.rpm", 5235475, "GPLv2+", "CentOS", "", "The GNU Bourne Again shell (bash) version 3.2", "77a8895be9b74f9ecc12945f8435346b"},
			{intRef(), "info", "4.8", "14.el5", "x86_64", "texinfo-4.8-14.el5.src.rpm", 285963, "GPL", "CentOS", "", "A stand-alone TTY-based reader for GNU texinfo documentation.", "69620837a17eb142baff1e3ce151aa88"},
			{intRef(), "readline", "5.1", "3.el5", "x86_64", "readline-5.1-3.el5.src.rpm", 384289, "GPL", "CentOS", "", "A library for editing typed command lines.", "06ffca9df95df6c67528e4a79bb2ce92"},
			{intRef(), "nss", "3.19.1", "4.el5_11", "x86_64", "nss-3.19.1-4.el5_11.src.rpm", 3568632, "MPLv2.0", "CentOS", "", "Network Security Services", "378dc06c9b3d51e8b060d78337c4a2e9"},
			{intRef(), "elfutils-libelf", "0.137", "3.el5", "x86_64", "elfutils-0.137-3.el5.src.rpm", 76000, "GPLv2 with exceptions", "CentOS", "", "Library to read and write ELF files", "2096d0df4ef738e730d3a11bcc493e96"},
			{intRef(), "libattr", "2.4.32", "1.1", "x86_64", "attr-2.4.32-1.1.src.rpm", 15560, "LGPL", "CentOS", "", "Dynamic library for extended attribute support.", "249cf790c716cbc7f62d549d0a8c686c"},
			{intRef(), "libstdc++", "4.1.2", "55.el5", "x86_64", "gcc-4.1.2-55.el5.src.rpm", 973680, "GPL", "CentOS", "", "GNU Standard C++ Library", "f8c51e75e3716435e7cc83e5e70b2365"},
			{intRef(), "iproute", "2.6.18", "15.el5", "x86_64", "iproute-2.6.18-15.el5.src.rpm", 2165603, "GNU GPL", "CentOS", "", "Advanced IP routing and network device configuration tools.", "ec8830ade45decc5cdf5d571f551e396"},
			{intRef(), "grep", "2.5.1", "55.el5", "x86_64", "grep-2.5.1-55.el5.src.rpm", 443427, "GPL", "CentOS", "", "The GNU versions of grep pattern matching utilities.", "6b70d99de915f46ee0f1aad653a6276d"},
			{intRef(), "diffutils", "2.8.1", "16.el5", "x86_64", "diffutils-2.8.1-16.el5.src.rpm", 568548, "GPL", "CentOS", "", "A GNU collection of diff utilities.", "b6dc37afa264b2e4c0915389ff5d045c"},
			{intRef(), "gawk", "3.1.5", "16.el5", "x86_64", "gawk-3.1.5-16.el5.src.rpm", 4384895, "GPL", "CentOS", "", "The GNU version of the awk text processing utility.", "925dd9695b7faf60484c6e4ce9b4d14a"},
			{intRef(), "less", "436", "9.el5", "x86_64", "less-436-9.el5.src.rpm", 191053, "GPL", "CentOS", "", "A text file browser similar to more, but better.", "2330420b22dce22ea74aacc1aa3b70aa"},
			{intRef(), "procps", "3.2.7", "26.el5", "x86_64", "procps-3.2.7-26.el5.src.rpm", 413100, "GPL", "CentOS", "", "System and process monitoring utilities.", "11ae6d0435269c9ae15f974db35e10e7"},
			{intRef(), "crontabs", "1.10", "11.el5", "noarch", "crontabs-1.10-11.el5.src.rpm", 2056, "Public Domain", "CentOS", "", "Root crontab files used to schedule the execution of programs.", "1d22356ed33d81cd514588029eb24954"},
			{intRef(), "libxml2", "2.6.26", "2.1.25.el5_11", "x86_64", "libxml2-2.6.26-2.1.25.el5_11.src.rpm", 1646329, "MIT", "CentOS", "", "Library providing XML and HTML support", "7f1cdef9c0e4b1ec67a140f1eb6a4c22"},
			{intRef(), "sgpio", "1.2.0_10", "2.el5", "x86_64", "sgpio-1.2.0_10-2.el5.src.rpm", 15901, "GPLv2+", "CentOS", "", "SGPIO captive backplane tool", "e734392f49b5a8095c72607c42247d54"},
			{intRef(), "mingetty", "1.07", "5.2.2", "x86_64", "mingetty-1.07-5.2.2.src.rpm", 32432, "GPL", "CentOS", "", "A compact getty program for virtual consoles only.", "bef76d70fc457a8873582db9c9ace60c"},
			{intRef(), "libcap", "1.10", "26", "x86_64", "libcap-1.10-26.src.rpm", 56330, "BSD-like and LGPL", "CentOS", "", "Library for getting and setting POSIX.1e capabilities", "cc965ac75b309539889023988bdcd05a"},
			{intRef(), "keyutils-libs", "1.2", "1.el5", "x86_64", "keyutils-1.2-1.el5.src.rpm", 33608, "GPL/LGPL", "CentOS", "", "Key utilities library", "ef18ff023135b0a06c38d008397f5bba"},
			{intRef(10), "centos-release", "5", "11.el5.centos", "x86_64", "centos-release-5-11.el5.centos.src.rpm", 45731, "GPL", "CentOS", "", "CentOS release file", "97ca9e2f526d0f56d12ae333faa7c213"},
			{intRef(), "python-libs", "2.4.3", "56.el5", "x86_64", "python-2.4.3-56.el5.src.rpm", 22018736, "PSF - see LICENSE", "CentOS", "", "The libraries for python runtime", "60cb943297c8197c7d1e4ff887c383b8"},
			{intRef(), "cracklib", "2.8.9", "3.3", "x86_64", "cracklib-2.8.9-3.3.src.rpm", 176409, "Artistic", "CentOS", "", "A password-checking library.", "1fa52fa4ea895f3e9ad49649eb621348"},
			{intRef(), "device-mapper-event", "1.02.67", "2.el5_11.1", "x86_64", "device-mapper-1.02.67-2.el5_11.1.src.rpm", 32432, "GPL", "CentOS", "", "Device-mapper event daemon", "4be45c2008c9438fab4001a19b8fa9b5"},
			{intRef(), "net-tools", "1.60", "83.el5_10", "x86_64", "net-tools-1.60-83.el5_10.src.rpm", 798220, "GPL", "CentOS", "", "Basic networking tools.", "2873ba0b8624debaa53224c0cea7170d"},
			{intRef(), "libutempter", "1.1.4", "4.el5", "x86_64", "libutempter-1.1.4-4.el5.src.rpm", 40484, "LGPL", "CentOS", "", "A privileged helper for utmp/wtmp updates", "ee79f24ada668f277a292f0687128b56"},
			{intRef(2), "tar", "1.15.1", "32.el5_8", "x86_64", "tar-1.15.1-32.el5_8.src.rpm", 1671798, "GPL", "CentOS", "", "A GNU file archiving program", "cdcde2ab088de4fa72387f0151cde06b"},
			{intRef(), "SysVinit", "2.86", "17.el5", "x86_64", "SysVinit-2.86-17.el5.src.rpm", 226009, "GPL", "CentOS", "", "Programs which control basic system processes.", "1805dcc707c129a0a402c4f0fe8a6aac"},
			{intRef(), "e2fsprogs", "1.39", "37.el5", "x86_64", "e2fsprogs-1.39-37.el5.src.rpm", 1985776, "GPL", "CentOS", "", "Utilities for managing the second and third extended (ext2/ext3) filesystems", "cf0d578d38aa8ec14f3e02a545ce45b0"},
			{intRef(), "kpartx", "0.4.7", "64.el5_11", "x86_64", "device-mapper-multipath-0.4.7-64.el5_11.src.rpm", 983150, "GPL", "CentOS", "", "Partition device manager for device-mapper devices.", "f0b09c8a073cb9feeeb1dd3bd67955c4"},
			{intRef(), "device-mapper-multipath", "0.4.7", "64.el5_11", "x86_64", "device-mapper-multipath-0.4.7-64.el5_11.src.rpm", 7193153, "GPL", "CentOS", "", "Tools to manage multipath devices using device-mapper.", "38a44c642f25798d77b1b6491ec7f372"},
			{intRef(), "logrotate", "3.7.4", "14", "x86_64", "logrotate-3.7.4-14.src.rpm", 59821, "GPL", "CentOS", "", "Rotates, compresses, removes and mails system log files.", "17f6b7960a5bb1da0cc09989d59b626d"},
			{intRef(), "MAKEDEV", "3.23", "1.2", "x86_64", "MAKEDEV-3.23-1.2.src.rpm", 660343, "GPL", "CentOS", "", "A program used for creating device files in /dev.", "7bec69c1cfea8240d18a3b84f8ac58e4"},
			{intRef(), "coreutils", "5.97", "34.el5_8.1", "x86_64", "coreutils-5.97-34.el5_8.1.src.rpm", 9045871, "GPLv2+", "CentOS", "", "The GNU core utilities: a set of tools commonly used in shell scripts", "14cda11598e21c143ef917f83b0397bf"},
			{intRef(), "udev", "095", "14.33.el5_11", "x86_64", "udev-095-14.33.el5_11.src.rpm", 5627184, "GPL", "CentOS", "", "A userspace implementation of devfs", "073f597327e25d04ed3c30b1ed12c8b8"},
			{intRef(), "module-init-tools", "3.3", "0.pre3.1.63.el5", "x86_64", "module-init-tools-3.3-0.pre3.1.63.el5.src.rpm", 907394, "GPL", "CentOS", "", "Kernel module management utilities.", "cd01c621f5e98bd9dc2907e89f4d109c"},
			{intRef(), "mcstrans", "0.2.11", "3.el5", "x86_64", "mcstrans-0.2.11-3.el5.src.rpm", 23099, "GPL", "CentOS", "", "SELinux Translation Daemon", "a38b8e3de2e8de30aa5d4aaa2375aa91"},
			{intRef(), "initscripts", "8.45.45", "1.el5.centos", "x86_64", "initscripts-8.45.45-1.el5.centos.src.rpm", 5509410, "GPLv2 and GPLv2+", "CentOS", "", "The inittab file and the /etc/init.d scripts.", "29d10866b790ddd72ee488b572dcc3e6"},
			{intRef(), "rpm-libs", "4.4.2.3", "36.el5_11", "x86_64", "rpm-4.4.2.3-36.el5_11.src.rpm", 2080256, "GPLv2+ and LGPLv2+ with exceptions", "CentOS", "", "Libraries for manipulating RPM packages", "c8c5989d7b50527a8adeda49304da4f1"},
			{intRef(30), "bind-libs", "9.3.6", "25.P1.el5_11.8", "x86_64", "bind-9.3.6-25.P1.el5_11.8.src.rpm", 2023624, "BSD-like", "CentOS", "", "Libraries used by the BIND DNS packages", "545af8d413906490cc89a02da0644f26"},
			{intRef(), "python-elementtree", "1.2.6", "5", "x86_64", "python-elementtree-1.2.6-5.src.rpm", 311637, "PSF", "CentOS", "", "Fast XML parser and writer", "69a450930bd972f88591f61506115204"},
			{intRef(), "m2crypto", "0.16", "9.el5", "x86_64", "m2crypto-0.16-9.el5.src.rpm", 1829395, "BSDish", "CentOS", "", "Support for using OpenSSL in python scripts", "c7d5af816329c171c658e5facf128c02"},
			{intRef(), "yum-metadata-parser", "1.1.2", "4.el5", "x86_64", "yum-metadata-parser-1.1.2-4.el5.src.rpm", 55883, "GPLv2", "CentOS", "", "A fast metadata parser for yum", "fdaedcbe32f2407dc71304f730debd38"},
			{intRef(), "yum", "3.2.22", "40.el5.centos", "noarch", "yum-3.2.22-40.el5.centos.src.rpm", 3429595, "GPLv2+", "CentOS", "", "RPM installer/updater", "1f1a23d92a8f84a34f227253e3243c6e"},
			{intRef(), "libuser", "0.54.7", "3.el5", "x86_64", "libuser-0.54.7-3.el5.src.rpm", 1647843, "LGPL", "CentOS", "", "A user and group account administration library.", "34fc71ea458d96b43809ff706ded8422"},
			{intRef(30), "bind-utils", "9.3.6", "25.P1.el5_11.8", "x86_64", "bind-9.3.6-25.P1.el5_11.8.src.rpm", 300911, "BSD-like", "CentOS", "", "Utilities for querying DNS name servers.", "22c7420b03087f1723a41b38a20b8d58"},
			{intRef(2), "vim-minimal", "7.0.109", "7.2.el5", "x86_64", "vim-7.0.109-7.2.el5.src.rpm", 630565, "freeware", "CentOS", "", "A minimal version of the VIM editor.", "53e32353511355e4f9d53359d7bf9fe2"},
			{intRef(), "rootfiles", "8.1", "1.1.1", "noarch", "rootfiles-8.1-1.1.1.src.rpm", 620, "Public Domain", "CentOS", "", "The basic required files for the root user's directory.", "ba7d3d3b58d6e25f21b35a507d443f5a"},
			{intRef(), "libgcc", "4.1.2", "55.el5", "x86_64", "gcc-4.1.2-55.el5.src.rpm", 84728, "GPL", "CentOS", "", "GCC version 4.1 shared support library", "2154f9619bd613d576bb1fbe5cda8a92"},
			{intRef(), "filesystem", "2.4.0", "3.el5.centos", "x86_64", "filesystem-2.4.0-3.el5.centos.src.rpm", 0, "Public Domain", "CentOS", "", "The basic directory layout for a Linux system.", "6962c3390d1435ec0f34c29907bd072b"},
			{intRef(), "nash", "5.1.19.6", "82.el5", "x86_64", "mkinitrd-5.1.19.6-82.el5.src.rpm", 2455441, "GPL", "CentOS", "", "nash shell", "a2a58a2e04cb645a0d6ea49635051b1f"},
			{intRef(1), "termcap", "5.5", "1.20060701.1", "noarch", "termcap-5.5-1.20060701.1.src.rpm", 807103, "Public Domain", "CentOS", "", "The terminal feature database used by certain applications.", "5ff51068e867ff25dce56110edd2997f"},
			{intRef(), "glibc-common", "2.5", "123.el5_11.3", "x86_64", "glibc-2.5-123.el5_11.3.src.rpm", 66738287, "LGPL", "CentOS", "", "Common binaries and locale data for glibc", "8b78439d1381ca049eb4f76ec60fe85e"},
			{intRef(3), "mktemp", "1.5", "24.el5", "x86_64", "mktemp-1.5-24.el5.src.rpm", 15432, "BSD", "CentOS", "", "A small utility for safely making /tmp files.", "cba9aad217ba3eae67ecf490a317bd4c"},
			{intRef(), "chkconfig", "1.3.30.2", "2.el5", "x86_64", "chkconfig-1.3.30.2-2.el5.src.rpm", 596588, "GPL", "CentOS", "", "A system tool for maintaining the /etc/rc*.d hierarchy.", "89cd767c743dfd937ea66f6a5a77d36b"},
			{intRef(), "nspr", "4.10.8", "2.el5_11", "x86_64", "nspr-4.10.8-2.el5_11.src.rpm", 258472, "MPLv2.0", "CentOS", "", "Netscape Portable Runtime", "a29d4514f13ca32faebba273ca37097a"},
			{intRef(), "bzip2-libs", "1.0.3", "6.el5_5", "x86_64", "bzip2-1.0.3-6.el5_5.src.rpm", 65528, "BSD", "CentOS", "", "Libraries for applications using bzip2", "6fc42e63bd6daad3a36c104a6db90567"},
			{intRef(), "libtermcap", "2.0.8", "46.1", "x86_64", "libtermcap-2.0.8-46.1.src.rpm", 13320, "LGPL", "CentOS", "", "A basic system library for accessing the termcap database.", "090c12069b07e4951e61d82618836ab0"},
			{intRef(), "ncurses", "5.5", "24.20060715", "x86_64", "ncurses-5.5-24.20060715.src.rpm", 3022251, "distributable", "CentOS", "", "A terminal handling library", "676baa19e139faea0d7665f5d2c17877"},
			{intRef(), "libsepol", "1.15.2", "3.el5", "x86_64", "libsepol-1.15.2-3.el5.src.rpm", 245232, "LGPLv2+", "CentOS", "", "SELinux binary policy manipulation library", "c7b4065f45e71c2de576850cacd10410"},
			{intRef(), "sqlite", "3.3.6", "7", "x86_64", "sqlite-3.3.6-7.src.rpm", 411302, "Public Domain", "CentOS", "", "Library that implements an embeddable SQL database engine", "ce4709481838a0ac79b54e689d5f775c"},
			{intRef(), "sed", "4.1.5", "8.el5", "x86_64", "sed-4.1.5-8.el5.src.rpm", 335348, "GPL", "CentOS", "", "A GNU stream text editor.", "2e7d1f476bfb1205a89b94df24c53a9c"},
			{intRef(), "expat", "1.95.8", "11.el5_8", "x86_64", "expat-1.95.8-11.el5_8.src.rpm", 173262, "BSD", "CentOS", "", "A library for parsing XML.", "7bb9c154fde6f6268ed75d62319c4071"},
			{intRef(), "libacl", "2.2.39", "8.el5", "x86_64", "acl-2.2.39-8.el5.src.rpm", 25624, "LGPL", "CentOS", "", "Dynamic library for access control list support.", "e5e62c133abd3f6aedad1714ebc55e8f"},
			{intRef(), "db4", "4.3.29", "10.el5_5.2", "x86_64", "db4-4.3.29-10.el5_5.2.src.rpm", 2118874, "GPL", "CentOS", "", "The Berkeley DB database library (version 4) for C.", "a27a2d93813f90cac11bd41a83616220"},
			{intRef(), "pcre", "6.6", "9.el5", "x86_64", "pcre-6.6-9.el5.src.rpm", 244957, "BSD", "CentOS", "", "Perl-compatible regular expression library", "70cae106b9c2503a47b324e427e6c37b"},
			{intRef(), "hmaccalc", "0.9.6", "4.el5", "x86_64", "hmaccalc-0.9.6-4.el5.src.rpm", 93006, "MIT", "CentOS", "", "Tools for computing and checking HMAC values for files", "897ba1cf439b98f335fd9ccd420662a5"},
			{intRef(), "binutils", "2.17.50.0.6", "26.el5", "x86_64", "binutils-2.17.50.0.6-26.el5.src.rpm", 7416670, "GPL", "CentOS", "", "A GNU collection of binary utilities.", "839deda2ca351da898ff7ba749b1abbf"},
			{intRef(), "cpio", "2.6", "26.el5", "x86_64", "cpio-2.6-26.el5.src.rpm", 285951, "GPL", "CentOS", "", "A GNU archiving program.", "926a88f906512d705cc3fad61586282c"},
			{intRef(), "gzip", "1.3.5", "13.el5.centos", "x86_64", "gzip-1.3.5-13.el5.centos.src.rpm", 181411, "GPL", "CentOS", "", "The GNU data compression program.", "525c87d6a3ff120c446313a802adb7f2"},
			{intRef(), "iputils", "20020927", "46.el5", "x86_64", "iputils-20020927-46.el5.src.rpm", 284557, "BSD", "CentOS", "", "Network monitoring tools including ping.", "eba214bd70f1b9c62f03da76248b2f1f"},
			{intRef(), "libsysfs", "2.1.0", "1.el5", "x86_64", "sysfsutils-2.1.0-1.el5.src.rpm", 144141, "LGPL", "CentOS", "", "Shared library for interfacing with sysfs", "4d9e2a82fd389328976efd9e6b21d656"},
			{intRef(), "cyrus-sasl-lib", "2.1.22", "7.el5_8.1", "x86_64", "cyrus-sasl-2.1.22-7.el5_8.1.src.rpm", 303193, "Freely Distributable", "CentOS", "", "Shared libraries needed by applications which use Cyrus SASL.", "5e2862fd2759bb6f932aeee2717affa6"},
			{intRef(), "gdbm", "1.8.0", "28.el5", "x86_64", "gdbm-1.8.0-28.el5.src.rpm", 47658, "GPL", "CentOS", "", "A GNU set of database routines which use extensible hashing.", "194fb2861f3fd79c87bc4883b8798e11"},
			{intRef(), "ethtool", "6", "4.el5", "x86_64", "ethtool-6-4.el5.src.rpm", 152180, "GPL", "CentOS", "", "Ethernet settings tool for PCI ethernet cards", "085d7f52a7143f3497fc315896c12031"},
			{intRef(), "centos-release-notes", "5.11", "0", "x86_64", "centos-release-notes-5.11-0.src.rpm", 52622, "GPL", "CentOS", "", "CentOS release notes files", "ef63454701728f932f469fc787c1be64"},
			{intRef(), "openssl", "0.9.8e", "39.el5_11", "x86_64", "openssl-0.9.8e-39.el5_11.src.rpm", 4171441, "BSDish", "CentOS", "", "The OpenSSL toolkit", "8371acf8ecd050d83c6d0f7fb9d83337"},
			{intRef(), "python", "2.4.3", "56.el5", "x86_64", "python-2.4.3-56.el5.src.rpm", 74377, "PSF - see LICENSE", "CentOS", "", "An interpreted, interactive, object-oriented programming language.", "ebfb56be33b146ef39180a090e581258"},
			{intRef(), "iscsi-initiator-utils", "6.2.0.872", "16.el5", "x86_64", "iscsi-initiator-utils-6.2.0.872-16.el5.src.rpm", 3137545, "GPL", "CentOS", "", "iSCSI daemon and utility programs", "cb6865ab733a137a34d1a33dcb7e5a82"},
			{intRef(), "dmraid-events", "1.0.0.rc13", "65.el5", "x86_64", "dmraid-1.0.0.rc13-65.el5.src.rpm", 8788, "GPL", "CentOS", "", "dmevent_tool (Device-mapper event tool) and DSO", "53cf5c9a8be71ec46923bec954d22902"},
			{intRef(2), "shadow-utils", "4.0.17", "23.el5", "x86_64", "shadow-utils-4.0.17-23.el5.src.rpm", 2785766, "BSD", "CentOS", "", "Utilities for managing accounts and shadow password files.", "c3d42d1ea6d08cebceb74f9a5807ec82"},
			{intRef(), "psmisc", "22.2", "11", "x86_64", "psmisc-22.2-11.src.rpm", 162941, "BSD/GPL", "CentOS", "", "Utilities for managing processes on your system.", "b72cde94c75b90e96f0001a1231390a1"},
			{intRef(1), "findutils", "4.2.27", "6.el5", "x86_64", "findutils-4.2.27-6.el5.src.rpm", 686644, "GPL", "CentOS", "", "The GNU versions of find utilities (find and xargs).", "5146c9556840d7574a5840741195dad7"},
			{intRef(), "e2fsprogs-libs", "1.39", "37.el5", "x86_64", "e2fsprogs-1.39-37.el5.src.rpm", 233240, "GPL", "CentOS", "", "Ext2/3 filesystem-specific static libraries and headers", "e2167e0544f7bb100912b86d88ef1a2c"},
			{intRef(), "lvm2", "2.02.88", "13.el5", "x86_64", "lvm2-2.02.88-13.el5.src.rpm", 9619149, "GPL", "CentOS", "", "Userland logical volume management tools", "38861e8ab964eaa68990a64ddbbb781a"},
			{intRef(), "dmraid", "1.0.0.rc13", "65.el5", "x86_64", "dmraid-1.0.0.rc13-65.el5.src.rpm", 1697084, "GPL", "CentOS", "", "dmraid (Device-mapper RAID tool and library)", "054319425d654477f40b75ec2bafeeb3"},
			{intRef(), "device-mapper", "1.02.67", "2.el5_11.1", "x86_64", "device-mapper-1.02.67-2.el5_11.1.src.rpm", 2173856, "GPL", "CentOS", "", "device mapper library", "3f24db0f2fed9bc6c3e56a75ca7df7c7"},
			{intRef(), "libselinux", "1.33.4", "5.7.el5.centos", "x86_64", "libselinux-1.33.4-5.7.el5.centos.src.rpm", 92976, "Public domain (uncopyrighted)", "CentOS", "", "SELinux library and simple utilities", "f45c43f3ba43bb55aae87d3edd839d32"},
			{intRef(), "krb5-libs", "1.6.1", "80.el5_11", "x86_64", "krb5-1.6.1-80.el5_11.src.rpm", 1467909, "MIT, freely distributable.", "CentOS", "", "The shared libraries used by Kerberos 5.", "c9d31f74c2bf00ab91324258fa79271e"},
			{intRef(), "pam", "0.99.6.2", "14.el5_11", "x86_64", "pam-0.99.6.2-14.el5_11.src.rpm", 2590520, "GPL or BSD", "CentOS", "", "A security tool which provides authentication for applications", "3207ff6b911d526266f097d6af655cfb"},
			{intRef(), "util-linux", "2.13", "0.59.el5_8", "x86_64", "util-linux-2.13-0.59.el5_8.src.rpm", 4813548, "distributable", "CentOS", "", "A collection of basic system utilities.", "3f100a47feaaba402378d13397f633e6"},
			{intRef(), "mkinitrd", "5.1.19.6", "82.el5", "x86_64", "mkinitrd-5.1.19.6-82.el5.src.rpm", 955929, "GPL", "CentOS", "", "Creates an initial ramdisk image for preloading modules.", "09c213397edd38f2a23718d0139023e9"},
			{intRef(), "rsyslog", "3.22.1", "7.el5", "x86_64", "rsyslog-3.22.1-7.el5.src.rpm", 1277211, "GPLv3+", "CentOS", "", "Enhanced system logging and kernel message trapping daemon", "f00ac74779dd22af83685e19ce3fa387"},
			{intRef(), "rpm", "4.4.2.3", "36.el5_11", "x86_64", "rpm-4.4.2.3-36.el5_11.src.rpm", 3754335, "GPLv2+", "CentOS", "", "The RPM package management system", "4cce99547eb026e86f77bd7372bb0026"},
			{intRef(), "rpm-python", "4.4.2.3", "36.el5_11", "x86_64", "rpm-4.4.2.3-36.el5_11.src.rpm", 133812, "GPLv2+", "CentOS", "", "Python bindings for apps which will manipulate RPM packages", "beee4dcfb0c19258c958c5c7af57bf6a"},
			{intRef(), "python-iniparse", "0.2.3", "6.el5", "noarch", "python-iniparse-0.2.3-6.el5.src.rpm", 134606, "MIT", "CentOS", "", "Python Module for Accessing and Modifying Configuration Data in INI files", "d1ce67108f8e402a0afd3b02e14b0d1f"},
			{intRef(), "python-sqlite", "1.1.7", "1.2.1", "x86_64", "python-sqlite-1.1.7-1.2.1.src.rpm", 120728, "GPL", "CentOS", "", "Python bindings for sqlite.", "50c6d927b1fa3bb93b64dd7ac5236461"},
			{intRef(), "python-urlgrabber", "3.1.0", "6.el5", "noarch", "python-urlgrabber-3.1.0-6.el5.src.rpm", 478912, "LGPLv2+", "CentOS", "", "A high-level cross-protocol url-grabber", "791118e5179a864f6a30b85ce0b44afc"},
			{intRef(), "yum-fastestmirror", "1.1.16", "21.el5.centos", "noarch", "yum-utils-1.1.16-21.el5.centos.src.rpm", 48064, "GPLv2+", "CentOS", "", "Yum plugin which chooses fastest repository from a mirrorlist", "4e2764e66fe0b6e3e40864bbe21dc066"},
			{intRef(), "openldap", "2.3.43", "29.el5_11", "x86_64", "openldap-2.3.43-29.el5_11.src.rpm", 618899, "OpenLDAP", "CentOS", "", "The configuration files, libraries, and documentation for OpenLDAP.", "9ae193e4dd285cbbb2840de2a5b52abb"},
			{intRef(), "passwd", "0.73", "2", "x86_64", "passwd-0.73-2.src.rpm", 27293, "BSD", "CentOS", "", "The passwd utility for setting/changing passwords using PAM", "475e3f87f41e0080f0c1d24feb9f6c79"},
			{intRef(), "libselinux-utils", "1.33.4", "5.7.el5.centos", "x86_64", "libselinux-1.33.4-5.7.el5.centos.src.rpm", 52008, "Public domain (uncopyrighted)", "CentOS", "", "SELinux libselinux utilies", "60b30684b853418598ea90e2f9994c14"},
		}

		return toPackageInfo(pkgs)
	}

	// docker run --rm -it centos:6 bash
	// yum groupinstall -y "Development tools"
	// rpm -qa --queryformat "\{%{EPOCH}, \"%{NAME}\", \"%{VERSION}\", \"%{RELEASE}\", \"%{ARCH}\", \"%{SOURCERPM}\", %{SIZE}, \"%{LICENSE}\", \"%{VENDOR}\", \"\", \"%{SUMMARY}\", \"%{SIGMD5}\"\},\n" | sed "s/^{(none)/{intRef()/g" | sed -r 's/^\{([0-9]+),/{intRef(\1),/' | sed "s/(none)/0/g"
	CentOS6DevTools = func() []*PackageInfo {
		pkgs := []*commonPackageInfo{
			{intRef(), "iproute", "2.6.32", "57.el6", "x86_64", "iproute-2.6.32-57.el6.src.rpm", 963477, "GPLv2+ and Public Domain", "CentOS", "", "Advanced IP routing and network device configuration tools", "68c6859576cf64a7ed187072908587b2"},
			{intRef(), "setup", "2.8.14", "23.el6", "noarch", "setup-2.8.14-23.el6.src.rpm", 666669, "Public Domain", "CentOS", "", "A set of system configuration and setup files", "f58d4ff770a377d42854025b5c64a709"},
			{intRef(), "iputils", "20071127", "24.el6", "x86_64", "iputils-20071127-24.el6.src.rpm", 297243, "BSD with advertising and GPLv2+ and Rdisc", "CentOS", "", "Network monitoring tools including ping", "9429035cf29d52ecbf1eee4a7db120c7"},
			{intRef(), "basesystem", "10.0", "4.el6", "noarch", "basesystem-10.0-4.el6.src.rpm", 0, "Public Domain", "CentOS", "", "The skeleton package which defines a simple Red Hat Enterprise Linux system", "bf36e0d2771604b8ad56d9696b5edd3c"},
			{intRef(), "util-linux-ng", "2.17.2", "12.28.el6_9.2", "x86_64", "util-linux-ng-2.17.2-12.28.el6_9.2.src.rpm", 6139437, "GPLv1+ and GPLv2 and GPLv2+ and LGPLv2+ and MIT and BSD with advertising and Public Domain", "CentOS", "", "A collection of basic system utilities", "ee7e9cba004ab79dce14e988cfb6d157"},
			{intRef(), "tzdata", "2018e", "3.el6", "noarch", "tzdata-2018e-3.el6.src.rpm", 1961609, "Public Domain", "CentOS", "", "Timezone data", "95de917c04de115372e5b75c3874db97"},
			{intRef(), "udev", "147", "2.73.el6_8.2", "x86_64", "udev-147-2.73.el6_8.2.src.rpm", 1280842, "GPLv2", "CentOS", "", "A userspace implementation of devfs", "3e0a5f14576169587dc44a4ebb392442"},
			{intRef(), "openssh-clients", "5.3p1", "124.el6_10", "x86_64", "openssh-5.3p1-124.el6_10.src.rpm", 1352442, "BSD", "CentOS", "", "An open source SSH client applications", "7d7b66b85b2e2e36906eb9c82b8095f2"},
			{intRef(), "git", "1.7.1", "9.el6_9", "x86_64", "git-1.7.1-9.el6_9.src.rpm", 15290753, "GPLv2", "CentOS", "", "Fast Version Control System", "395cf7af32a17cedd20acddfa6e85952"},
			{intRef(), "bash", "4.1.2", "48.el6", "x86_64", "bash-4.1.2-48.el6.src.rpm", 3142529, "GPLv3+", "CentOS", "", "The GNU Bourne Again shell", "fb70e495a46a63523eab6138cae2438d"},
			{intRef(), "systemtap-devel", "2.9", "9.el6", "x86_64", "systemtap-2.9-9.el6.src.rpm", 6327079, "GPLv2+", "CentOS", "", "Programmable system-wide instrumentation system - development headers, tools", "974555cbff98caf78f7b088d7362ec5e"},
			{intRef(), "libcap", "2.16", "5.5.el6", "x86_64", "libcap-2.16-5.5.el6.src.rpm", 64437, "LGPLv2+ or BSD", "CentOS", "", "Library for getting and setting POSIX.1e capabilities", "5d224cdbaa15053a3aa5443a07a45c2b"},
			{intRef(), "hicolor-icon-theme", "0.11", "1.1.el6", "noarch", "hicolor-icon-theme-0.11-1.1.el6.src.rpm", 45406, "GPL+", "CentOS", "", "Basic requirement for icon themes", "c1cf9a985abca13245f6fdbd6b4cb57c"},
			{intRef(), "info", "4.13a", "8.el6", "x86_64", "texinfo-4.13a-8.el6.src.rpm", 329482, "GPLv3+", "CentOS", "", "A stand-alone TTY-based reader for GNU texinfo documentation", "482d0f1ec0690ed672c0c1b594f78583"},
			{intRef(), "libgcj", "4.4.7", "23.el6", "x86_64", "gcc-4.4.7-23.el6.src.rpm", 65012480, "GPLv3+ and GPLv3+ with exceptions and GPLv2+ with exceptions", "CentOS", "", "Java runtime library for gcc", "8522846e5285eb055ec4ae698f001a60"},
			{intRef(), "libacl", "2.2.49", "7.el6_9.1", "x86_64", "acl-2.2.49-7.el6_9.1.src.rpm", 31280, "LGPLv2+", "CentOS", "", "Dynamic library for access control list support", "98166984ca2ff5aabfd1a43fdc89c788"},
			{intRef(), "intltool", "0.41.0", "1.1.el6", "noarch", "intltool-0.41.0-1.1.el6.src.rpm", 167585, "GPLv2 with exceptions", "CentOS", "", "Utility for internationalizing various kinds of data files", "72bad34320d1dc10227e2ae32cbc1d67"},
			{intRef(), "nspr", "4.19.0", "1.el6", "x86_64", "nspr-4.19.0-1.el6.src.rpm", 281968, "MPLv2.0", "CentOS", "", "Netscape Portable Runtime", "18e1b8a44c636953cf82358dc0978d34"},
			{intRef(), "subversion", "1.6.11", "15.el6_7", "x86_64", "subversion-1.6.11-15.el6_7.src.rpm", 12105530, "ASL 1.1", "CentOS", "", "A Modern Concurrent Version Control System", "6366595f694e84409847626985986d56"},
			{intRef(), "libcom_err", "1.41.12", "24.el6", "x86_64", "e2fsprogs-1.41.12-24.el6.src.rpm", 59233, "MIT", "CentOS", "", "Common error description library", "3d4d863ae951dde28e33d4fec21c793d"},
			{intRef(), "gcc-gfortran", "4.4.7", "23.el6", "x86_64", "gcc-4.4.7-23.el6.src.rpm", 14473805, "GPLv3+ and GPLv3+ with exceptions and GPLv2+ with exceptions", "CentOS", "", "Fortran support", "5c76186dccdc67d7cd5afc864e6a92ce"},
			{intRef(), "libsepol", "2.0.41", "4.el6", "x86_64", "libsepol-2.0.41-4.el6.src.rpm", 248680, "LGPLv2+", "CentOS", "", "SELinux binary policy manipulation library", "cbc386b646b94d710dfabe7126ede725"},
			{intRef(), "rpm-build", "4.8.0", "59.el6", "x86_64", "rpm-4.8.0-59.el6.src.rpm", 323213, "GPLv2+", "CentOS", "", "Scripts and executable programs used to build packages", "79d4da891e041d8dd9d9f83ed935da1c"},
			{intRef(), "chkconfig", "1.3.49.5", "1.el6", "x86_64", "chkconfig-1.3.49.5-1.el6.src.rpm", 670580, "GPLv2", "CentOS", "", "A system tool for maintaining the /etc/rc*.d hierarchy", "c5e95feb1e3c73bfd0761981c866a024"},
			{intRef(), "flex", "2.5.35", "9.el6", "x86_64", "flex-2.5.35-9.el6.src.rpm", 736081, "BSD", "CentOS", "", "A tool for creating scanners (text pattern recognizers)", "c952749600445a57fe3f865c47c0d22c"},
			{intRef(), "audit-libs", "2.4.5", "6.el6", "x86_64", "audit-2.4.5-6.el6.src.rpm", 235282, "LGPLv2+", "CentOS", "", "Dynamic library for libaudit", "fec1ad9c60d9d6e870ea6182ba6be06a"},
			{intRef(), "cscope", "15.6", "7.el6", "x86_64", "cscope-15.6-7.el6.src.rpm", 466265, "BSD", "CentOS", "", "C source code tree search and browse tool", "695c44b601b82d29ae3e73ab8dfaf9d1"},
			{intRef(), "readline", "6.0", "4.el6", "x86_64", "readline-6.0-4.el6.src.rpm", 433957, "GPLv3+", "CentOS", "", "A library for editing typed command lines", "d8b0aa8a08f36b43eb4858d76d005c6d"},
			{intRef(), "rcs", "5.7", "37.el6", "x86_64", "rcs-5.7-37.el6.src.rpm", 709927, "GPLv2+", "CentOS", "", "Revision Control System (RCS) file version management tools", "f9340500810916e598bb8c88a2f47f4e"},
			{intRef(), "file-libs", "5.04", "30.el6", "x86_64", "file-5.04-30.el6.src.rpm", 2486624, "BSD", "CentOS", "", "Libraries for applications using libmagic", "e27f1d59098a685689f3f2d099946a85"},
			{intRef(), "diffstat", "1.51", "2.el6", "x86_64", "diffstat-1.51-2.el6.src.rpm", 45568, "MIT", "CentOS", "", "A utility which provides statistics based on the output of diff", "72f972ef4f85141a4c98f69908c5411a"},
			{intRef(1), "dbus-libs", "1.2.24", "9.el6", "x86_64", "dbus-1.2.24-9.el6.src.rpm", 265736, "GPLv2+ or AFL", "CentOS", "", "Libraries for accessing D-BUS", "3b7d65739639c300ab7a69e890240f32"},
			{intRef(1), "doxygen", "1.6.1", "6.el6", "x86_64", "doxygen-1.6.1-6.el6.src.rpm", 9446190, "GPL+", "CentOS", "", "A documentation system for C/C++", "397c080755741ab4bba77e6b18beccd0"},
			{intRef(), "sqlite", "3.6.20", "1.el6_7.2", "x86_64", "sqlite-3.6.20-1.el6_7.2.src.rpm", 640060, "Public Domain", "CentOS", "", "Library that implements an embeddable SQL database engine", "d94a1809a80676af38140d2f0021c5f9"},
			{intRef(), "libuuid", "2.17.2", "12.28.el6_9.2", "x86_64", "util-linux-ng-2.17.2-12.28.el6_9.2.src.rpm", 16304, "BSD", "CentOS", "", "Universally unique ID library", "9e1615002308c631aaee83e16668543f"},
			{intRef(), "xz-libs", "4.999.9", "0.5.beta.20091007git.el6", "x86_64", "xz-4.999.9-0.5.beta.20091007git.el6.src.rpm", 214490, "LGPLv2+", "CentOS", "", "Libraries for decoding LZMA compression", "2afa255c6edf06556b6aea23d72a1874"},
			{intRef(), "libgpg-error", "1.7", "4.el6", "x86_64", "libgpg-error-1.7-4.el6.src.rpm", 214321, "LGPLv2+", "CentOS", "", "Library for error values used by GnuPG components", "a64a2b188d9fb76dc940e809e791aedc"},
			{intRef(), "pcre", "7.8", "7.el6", "x86_64", "pcre-7.8-7.el6.src.rpm", 529027, "BSD", "CentOS", "", "Perl-compatible regular expression library", "60c2d316d863cdcb610cca815e29d894"},
			{intRef(1), "findutils", "4.4.2", "9.el6", "x86_64", "findutils-4.4.2-9.el6.src.rpm", 1442912, "GPLv3+", "CentOS", "", "The GNU versions of find utilities (find and xargs)", "b0ffd83c8576d956eb0af7aaf52bf832"},
			{intRef(), "expat", "2.0.1", "13.el6_8", "x86_64", "expat-2.0.1-13.el6_8.src.rpm", 197794, "MIT", "CentOS", "", "An XML parser library", "7c8e581750011dc5ff891ef8dbb76263"},
			{intRef(), "p11-kit", "0.18.5", "2.el6_5.2", "x86_64", "p11-kit-0.18.5-2.el6_5.2.src.rpm", 262669, "BSD", "CentOS", "", "Library for loading and sharing PKCS#11 modules", "4b6eb29295513687b0dd248091a45be3"},
			{intRef(), "libgcrypt", "1.4.5", "12.el6_8", "x86_64", "libgcrypt-1.4.5-12.el6_8.src.rpm", 536622, "LGPLv2+", "CentOS", "", "A general-purpose cryptography library", "b16c065b3d5f5c351ee3f64c67fb77ce"},
			{intRef(), "libusb", "0.1.12", "23.el6", "x86_64", "libusb-0.1.12-23.el6.src.rpm", 54440, "LGPLv2+", "CentOS", "", "A library which allows userspace access to USB devices", "d1b2530f9e990b4dff32301c7849d417"},
			{intRef(), "pinentry", "0.7.6", "8.el6", "x86_64", "pinentry-0.7.6-8.el6.src.rpm", 147023, "GPLv2+", "CentOS", "", "Collection of simple PIN or passphrase entry dialogs", "6cd8f48fc9404b8c6ce0a0995ec0a3b6"},
			{intRef(), "procps", "3.2.8", "45.el6_9.3", "x86_64", "procps-3.2.8-45.el6_9.3.src.rpm", 475526, "GPLv2+ and LGPLv2+", "CentOS", "", "System and process monitoring utilities", "f21e14ee708e7ff106533791199873cd"},
			{intRef(2), "tar", "1.23", "15.el6_8", "x86_64", "tar-1.23-15.el6_8.src.rpm", 2616465, "GPLv3+", "CentOS", "", "A GNU file archiving program", "369d4ecc0c520fbe7368613a19558afe"},
			{intRef(), "checkpolicy", "2.0.22", "1.el6", "x86_64", "checkpolicy-2.0.22-1.el6.src.rpm", 870239, "GPLv2", "CentOS", "", "SELinux policy compiler", "02392813ec341e8c77a29609f884d937"},
			{intRef(), "db4-utils", "4.7.25", "22.el6", "x86_64", "db4-4.7.25-22.el6.src.rpm", 416398, "Sleepycat and BSD", "CentOS", "", "Command line tools for managing Berkeley DB (version 4) databases", "4ab1657d6f8d178f926a168ad1f40e8c"},
			{intRef(), "which", "2.19", "6.el6", "x86_64", "which-2.19-6.el6.src.rpm", 73004, "GPLv3", "CentOS", "", "Displays where a particular program in your path is located", "0a194d85776ddf95732977c15dde7313"},
			{intRef(), "dash", "0.5.5.1", "4.el6", "x86_64", "dash-0.5.5.1-4.el6.src.rpm", 127277, "BSD", "CentOS", "", "Small and fast POSIX-compliant shell", "34745b4864bf8e42f1181162e3d52d55"},
			{intRef(), "groff", "1.18.1.4", "21.el6", "x86_64", "groff-1.18.1.4-21.el6.src.rpm", 5318766, "GPLv2 and GFDL", "CentOS", "", "A document formatting system", "e4ca537dd1668ea7a00a3476d0bde544"},
			{intRef(), "coreutils-libs", "8.4", "47.el6", "x86_64", "coreutils-8.4-47.el6.src.rpm", 5576, "GPLv3+", "CentOS", "", "Libraries for coreutils", "7ad9b0d44df0e19417e58c922ce145f3"},
			{intRef(), "cracklib", "2.8.16", "4.el6", "x86_64", "cracklib-2.8.16-4.el6.src.rpm", 187265, "LGPLv2+", "CentOS", "", "A password-checking library", "66ba40d1f80041344e9e7f3c2cce0b3c"},
			{intRef(), "coreutils", "8.4", "47.el6", "x86_64", "coreutils-8.4-47.el6.src.rpm", 12873065, "GPLv3+", "CentOS", "", "A set of basic GNU tools commonly used in shell scripts", "84092ccc9b644eb86dc0fb09754f80a1"},
			{intRef(), "module-init-tools", "3.9", "26.el6", "x86_64", "module-init-tools-3.9-26.el6.src.rpm", 1216834, "GPLv2+", "CentOS", "", "Kernel module management utilities.", "cb180e326e41dd50e83a78242742163d"},
			{intRef(), "nss", "3.36.0", "8.el6", "x86_64", "nss-3.36.0-8.el6.src.rpm", 2660886, "MPLv2.0", "CentOS", "", "Network Security Services", "b7517259bc48e009b2f53b9134813cac"},
			{intRef(), "nss-tools", "3.36.0", "8.el6", "x86_64", "nss-3.36.0-8.el6.src.rpm", 2052883, "MPLv2.0", "CentOS", "", "Tools for the Network Security Services", "38296a6aca265f81ed0568689a902153"},
			{intRef(), "mingetty", "1.08", "5.el6", "x86_64", "mingetty-1.08-5.el6.src.rpm", 34586, "GPLv2+", "CentOS", "", "A compact getty program for virtual consoles only", "db0d0ab6289fcee75a1aff64db47e0aa"},
			{intRef(), "krb5-libs", "1.10.3", "65.el6", "x86_64", "krb5-1.10.3-65.el6.src.rpm", 1813468, "MIT", "CentOS", "", "The non-admin shared libraries used by Kerberos 5", "fb39fd3d4667686d8c6a8919cce8518d"},
			{intRef(), "libssh2", "1.4.2", "2.el6_7.1", "x86_64", "libssh2-1.4.2-2.el6_7.1.src.rpm", 325165, "BSD", "CentOS", "", "A library implementing the SSH2 protocol", "75d2094e0838762108ccb666416dc3bf"},
			{intRef(), "rpm-libs", "4.8.0", "59.el6", "x86_64", "rpm-4.8.0-59.el6.src.rpm", 777656, "GPLv2+ and LGPLv2+ with exceptions", "CentOS", "", "Libraries for manipulating RPM packages", "5cc09f970aeaee43fa0ef20bb03d0103"},
			{intRef(), "rpm", "4.8.0", "59.el6", "x86_64", "rpm-4.8.0-59.el6.src.rpm", 2034245, "GPLv2+", "CentOS", "", "The RPM package management system", "be5c4756ff3091f6e8b8cb6864bc8ad2"},
			{intRef(), "gnupg2", "2.0.14", "9.el6_10", "x86_64", "gnupg2-2.0.14-9.el6_10.src.rpm", 6087540, "GPLv3+", "CentOS", "", "Utility for secure communication and data storage", "e3378f38c43677ba3b4c019d0a0be0b0"},
			{intRef(32), "bind-libs", "9.8.2", "0.68.rc1.el6_10.1", "x86_64", "bind-9.8.2-0.68.rc1.el6_10.1.src.rpm", 2340720, "ISC", "CentOS", "", "Libraries used by the BIND DNS packages", "591c20361f155316b65bfb19269682aa"},
			{intRef(), "libsemanage", "2.0.43", "5.1.el6", "x86_64", "libsemanage-2.0.43-5.1.el6.src.rpm", 204223, "LGPLv2+", "CentOS", "", "SELinux binary policy manipulation library", "ddb4929b9a3b18082dd75345fa368c1b"},
			{intRef(), "libutempter", "1.1.5", "4.1.el6", "x86_64", "libutempter-1.1.5-4.1.el6.src.rpm", 40785, "LGPLv2", "CentOS", "", "A privileged helper for utmp/wtmp updates", "9fab105dce8b3f2f7dd9f53d4161c776"},
			{intRef(), "plymouth-core-libs", "0.8.3", "29.el6.centos", "x86_64", "plymouth-0.8.3-29.el6.centos.src.rpm", 177760, "GPLv2+", "CentOS", "", "Plymouth libraries", "c32d0f492fe719154a45ac77da9190de"},
			{intRef(), "libffi", "3.0.5", "3.2.el6", "x86_64", "libffi-3.0.5-3.2.el6.src.rpm", 42881, "BSD", "CentOS", "", "A portable foreign function interface library", "180e75de1468bb9353680e8dabaebeaf"},
			{intRef(), "python-libs", "2.6.6", "66.el6_8", "x86_64", "python-2.6.6-66.el6_8.src.rpm", 22979382, "Python", "CentOS", "", "Runtime libraries for Python", "5af05686c9ebe664b766c9d109e51e41"},
			{intRef(), "pygpgme", "0.1", "18.20090824bzr68.el6", "x86_64", "pygpgme-0.1-18.20090824bzr68.el6.src.rpm", 251432, "LGPLv2+", "CentOS", "", "Python module for working with OpenPGP messages", "98ac89e05c1ff89accffa1803d5966e0"},
			{intRef(), "python-urlgrabber", "3.9.1", "11.el6", "noarch", "python-urlgrabber-3.9.1-11.el6.src.rpm", 323137, "LGPLv2+", "CentOS", "", "A high-level cross-protocol url-grabber", "55fc3e4b6602d49bf3eb55ae688579a7"},
			{intRef(1), "pkgconfig", "0.23", "9.1.el6", "x86_64", "pkgconfig-0.23-9.1.el6.src.rpm", 140091, "GPLv2+", "CentOS", "", "A tool for determining compilation options", "62aa0b6f2b06f987ba56e95d727a948f"},
			{intRef(), "glib2", "2.28.8", "10.el6", "x86_64", "glib2-2.28.8-10.el6.src.rpm", 8061476, "LGPLv2+", "CentOS", "", "A library of handy utility functions", "a2b6d96a84eb5c40d73789bc17c16201"},
			{intRef(), "yum-metadata-parser", "1.1.2", "16.el6", "x86_64", "yum-metadata-parser-1.1.2-16.el6.src.rpm", 58327, "GPLv2", "CentOS", "", "A fast metadata parser for yum", "16a0976518f852b12ed3d1c0285e4bc6"},
			{intRef(), "yum", "3.2.29", "81.el6.centos", "noarch", "yum-3.2.29-81.el6.centos.src.rpm", 4832194, "GPLv2+", "CentOS", "", "RPM package installer/updater/manager", "dd629b7ea24a01ec5e380f78a0d71998"},
			{intRef(), "passwd", "0.77", "7.el6", "x86_64", "passwd-0.77-7.el6.src.rpm", 357699, "BSD or GPLv2+", "CentOS", "", "An utility for setting or changing passwords using PAM", "a2aa313df340ef28a8f4ec28a2f980f7"},
			{intRef(32), "bind-utils", "9.8.2", "0.68.rc1.el6_10.1", "x86_64", "bind-9.8.2-0.68.rc1.el6_10.1.src.rpm", 451000, "ISC", "CentOS", "", "Utilities for querying DNS name servers", "46a22a8044620a77a51e437adabca27d"},
			{intRef(2), "vim-minimal", "7.4.629", "5.el6_8.1", "x86_64", "vim-7.4.629-5.el6_8.1.src.rpm", 909230, "Vim", "CentOS", "", "A minimal version of the VIM editor", "3a8cb69c28f19fc0d374e68814b24d35"},
			{intRef(), "gpg-pubkey", "c105b9de", "4e0fd3a3", "", "", 0, "pubkey", "", "", "gpg(CentOS-6 Key (CentOS 6 Official Signing Key) <centos-6-key@centos.org>)", ""},
			{intRef(), "glibc-common", "2.12", "1.212.el6_10.3", "x86_64", "glibc-2.12-1.212.el6_10.3.src.rpm", 112434659, "LGPLv2+ and LGPLv2+ with exceptions and GPLv2+", "CentOS", "", "Common binaries and locale data for glibc", "42d0277ee6eb50b1443f0aceefa348fa"},
			{intRef(), "fontconfig", "2.8.0", "5.el6", "x86_64", "fontconfig-2.8.0-5.el6.src.rpm", 450597, "MIT", "CentOS", "", "Font configuration and customization library", "5acc9a581a51e30570dc45d8f0bc42f3"},
			{intRef(), "libproxy-bin", "0.3.0", "10.el6", "x86_64", "libproxy-0.3.0-10.el6.src.rpm", 6800, "LGPLv2+", "CentOS", "", "Binary to test libproxy", "f15b1cee00c5d1ce03251c9cd8a9faad"},
			{intRef(), "libproxy", "0.3.0", "10.el6", "x86_64", "libproxy-0.3.0-10.el6.src.rpm", 118439, "LGPLv2+", "CentOS", "", "A library handling all the details of proxy configuration", "60b85650edb37580707ddfd43da0da1c"},
			{intRef(), "mpfr", "2.4.1", "6.el6", "x86_64", "mpfr-2.4.1-6.el6.src.rpm", 377376, "LGPLv2+ and GPLv2+ and GFDL", "CentOS", "", "A C library for multiple-precision floating-point computations", "962fa85bfb231914eb135da3bb9785a7"},
			{intRef(2), "libpng", "1.2.49", "2.el6_7", "x86_64", "libpng-1.2.49-2.el6_7.src.rpm", 654803, "zlib", "CentOS", "", "A library of functions for manipulating PNG image format files", "9b4d9f9547af8d4148433086625becf3"},
			{intRef(), "libtiff", "3.9.4", "21.el6_8", "x86_64", "libtiff-3.9.4-21.el6_8.src.rpm", 966899, "libtiff", "CentOS", "", "Library of functions for manipulating TIFF format image files", "451fc01ad0630c13574eac9f8b814d27"},
			{intRef(), "zip", "3.0", "1.el6_7.1", "x86_64", "zip-3.0-1.el6_7.1.src.rpm", 823164, "BSD", "CentOS", "", "A file compression and packaging utility compatible with PKZIP", "60be41446adb564e824f1c4ed68547a9"},
			{intRef(), "apr", "1.3.9", "5.el6_9.1", "x86_64", "apr-1.3.9-5.el6_9.1.src.rpm", 303173, "ASL 2.0", "CentOS", "", "Apache Portable Runtime library", "444f2c97cb580a034a105d1c75d894a4"},
			{intRef(), "sysvinit-tools", "2.87", "6.dsf.el6", "x86_64", "sysvinit-2.87-6.dsf.el6.src.rpm", 114115, "GPLv2+", "CentOS", "", "Tools used for process and utmp management.", "0c73faafac6b72fe066e3e7426ebc672"},
			{intRef(), "atk", "1.30.0", "1.el6", "x86_64", "atk-1.30.0-1.el6.src.rpm", 957484, "LGPLv2+", "CentOS", "", "Interfaces for accessibility support", "f07e78e7968b9084e3203be0a6ca3bb4"},
			{intRef(), "libICE", "1.0.6", "1.el6", "x86_64", "libICE-1.0.6-1.el6.src.rpm", 117691, "MIT", "CentOS", "", "X.Org X11 ICE runtime library", "9a27acf194a47c2fe666566c06fb2148"},
			{intRef(), "hwdata", "0.233", "20.1.el6", "noarch", "hwdata-0.233-20.1.el6.src.rpm", 5894683, "GPLv2+", "CentOS", "", "Hardware identification and configuration data", "37fdf49e4d7fb4aaac627b2acd369d15"},
			{intRef(), "libSM", "1.2.1", "2.el6", "x86_64", "libSM-1.2.1-2.el6.src.rpm", 78264, "MIT", "CentOS", "", "X.Org X11 SM runtime library", "47019350ac2ca70098f54a6c1fc573e9"},
			{intRef(), "xz-lzma-compat", "4.999.9", "0.5.beta.20091007git.el6", "x86_64", "xz-4.999.9-0.5.beta.20091007git.el6.src.rpm", 21146, "GPLv2+ and LGPLv2+", "CentOS", "", "Older LZMA format compatibility binaries", "4f3dee428bf66173e53600b5d6e7d9a9"},
			{intRef(), "elfutils", "0.164", "2.el6", "x86_64", "elfutils-0.164-2.el6.src.rpm", 677800, "GPLv3+ and (GPLv2+ or LGPLv3+)", "CentOS", "", "A collection of utilities and DSOs to handle compiled objects", "5216dc41ec9fea57403e7a49c7df360e"},
			{intRef(), "jasper-libs", "1.900.1", "22.el6", "x86_64", "jasper-1.900.1-22.el6.src.rpm", 341232, "JasPer", "CentOS", "", "Runtime libraries for jasper", "59c5c852034b4988f15711d0ed43e876"},
			{intRef(), "neon", "0.29.3", "3.el6_4", "x86_64", "neon-0.29.3-3.el6_4.src.rpm", 297415, "LGPLv2+", "CentOS", "", "An HTTP and WebDAV client library", "f70ae9d84ab2b9f8a5503985d12e0231"},
			{intRef(), "alsa-lib", "1.1.0", "4.el6", "x86_64", "alsa-lib-1.1.0-4.el6.src.rpm", 1277969, "LGPLv2+", "CentOS", "", "The Advanced Linux Sound Architecture (ALSA) library", "0149374b0f9ce9e8169a6613f47372f0"},
			{intRef(), "libart_lgpl", "2.3.20", "5.1.el6", "x86_64", "libart_lgpl-2.3.20-5.1.el6.src.rpm", 129211, "LGPLv2+", "CentOS", "", "Library of graphics routines used by libgnomecanvas", "09e165976fd25ceb536f08f04d99601a"},
			{intRef(), "fipscheck", "1.2.0", "7.el6", "x86_64", "fipscheck-1.2.0-7.el6.src.rpm", 28163, "BSD", "CentOS", "", "A library for integrity verification of FIPS validated modules", "eb273b1fce2db4169500b5dcfc7e4e66"},
			{intRef(1), "perl-Pod-Escapes", "1.04", "144.el6", "x86_64", "perl-5.10.1-144.el6.src.rpm", 21092, "GPL+ or Artistic", "CentOS", "", "Perl module for resolving POD escape sequences", "fe03978a389d9991061ca0a8f128f992"},
			{intRef(1), "perl-Pod-Simple", "3.13", "144.el6", "x86_64", "perl-5.10.1-144.el6.src.rpm", 476776, "GPL+ or Artistic", "CentOS", "", "Framework for parsing POD documentation", "f2e0c1759e9a1b5695421946d3c368f3"},
			{intRef(3), "perl-version", "0.77", "144.el6", "x86_64", "perl-5.10.1-144.el6.src.rpm", 51960, "GPL+ or Artistic", "CentOS", "", "Perl extension for Version Objects", "24fadfae3015ef2b7dca2ac6f810198c"},
			{intRef(), "perl-URI", "1.40", "2.el6", "noarch", "perl-URI-1.40-2.el6.src.rpm", 256097, "GPL+ or Artistic", "CentOS", "", "A Perl module implementing URI parsing and manipulation", "6948541dbd43f6d1d4c4b8a57c5708d3"},
			{intRef(), "autoconf", "2.63", "5.1.el6", "noarch", "autoconf-2.63-5.1.el6.src.rpm", 2605343, "GPLv3+ and GFDL", "CentOS", "", "A GNU tool for automatically configuring source code", "70c97849b8629bd056e7a21a7d7d05c4"},
			{intRef(1), "perl-Compress-Raw-Zlib", "2.021", "144.el6", "x86_64", "perl-5.10.1-144.el6.src.rpm", 146581, "GPL+ or Artistic", "CentOS", "", "Low-Level Interface to the zlib compression library", "7810178ad921586b857cf008b47d53d3"},
			{intRef(0), "perl-IO-Compress-Zlib", "2.021", "144.el6", "x86_64", "perl-5.10.1-144.el6.src.rpm", 372202, "GPL+ or Artistic", "CentOS", "", "Perl interface to allow reading and writing of gzip and zip data", "06b8bb581fc1522ff03b3a480b68dff3"},
			{intRef(), "redhat-rpm-config", "9.0.3", "51.el6.centos", "noarch", "redhat-rpm-config-9.0.3-51.el6.centos.src.rpm", 138592, "GPL+", "CentOS", "", "Red Hat specific rpm configuration files", "2d8beaa3415bd4812352e8e393287174"},
			{intRef(), "perl-HTML-Parser", "3.64", "2.el6", "x86_64", "perl-HTML-Parser-3.64-2.el6.src.rpm", 226598, "GPL+ or Artistic", "CentOS", "", "Perl module for parsing HTML", "04c39518b59ac7990462284db01af17d"},
			{intRef(), "gettext", "0.17", "18.el6", "x86_64", "gettext-0.17-18.el6.src.rpm", 6369345, "GPLv3 and LGPLv2+", "CentOS", "", "GNU libraries and utilities for producing multi-lingual messages", "fb61a8cd0412f952819eff3b30a3f3b1"},
			{intRef(), "libxcb", "1.12", "4.el6", "x86_64", "libxcb-1.12-4.el6.src.rpm", 886960, "MIT", "CentOS", "", "A C binding to the X11 protocol", "3cd4018042fcb256ce0943508cb7b534"},
			{intRef(), "ppl", "0.10.2", "11.el6", "x86_64", "ppl-0.10.2-11.el6.src.rpm", 4637284, "GPLv3+", "CentOS", "", "The Parma Polyhedra Library: a library of numerical abstractions", "6d4d70bb01ebe2ffc3a9cf5e27da20cb"},
			{intRef(), "rsync", "3.0.6", "12.el6", "x86_64", "rsync-3.0.6-12.el6.src.rpm", 698678, "GPLv3+", "CentOS", "", "A program for synchronizing files over a network", "5d5894f1a0e11cb5add4effe11921676"},
			{intRef(), "gettext-libs", "0.17", "18.el6", "x86_64", "gettext-0.17-18.el6.src.rpm", 278832, "LGPLv2+", "CentOS", "", "Libraries for gettext", "6fd02734d4bf1eb900c65dadcd767c10"},
			{intRef(), "libgfortran", "4.4.7", "23.el6", "x86_64", "gcc-4.4.7-23.el6.src.rpm", 988064, "GPLv3+ and GPLv3+ with exceptions and GPLv2+ with exceptions", "CentOS", "", "Fortran runtime", "5c44d1612cd16aad0d19c40af49212cf"},
			{intRef(), "kernel-headers", "2.6.32", "754.12.1.el6", "x86_64", "kernel-2.6.32-754.12.1.el6.src.rpm", 2776718, "GPLv2", "CentOS", "", "Header files for the Linux kernel for use by glibc", "5780c616f9ddfffc004b625e7e5675ba"},
			{intRef(), "glibc-devel", "2.12", "1.212.el6_10.3", "x86_64", "glibc-2.12-1.212.el6_10.3.src.rpm", 990251, "LGPLv2+ and LGPLv2+ with exceptions and GPLv2+", "CentOS", "", "Object files for development using standard C libraries.", "be3a69eb29668a3d41bc48e27feaf4d8"},
			{intRef(), "libstdc++-devel", "4.4.7", "23.el6", "x86_64", "gcc-4.4.7-23.el6.src.rpm", 9728819, "GPLv3+ and GPLv3+ with exceptions and GPLv2+ with exceptions", "CentOS", "", "Header files and libraries for C++ development", "105304eb2ae04a3038e0d7e9327d17f6"},
			{intRef(), "libX11", "1.6.4", "3.el6", "x86_64", "libX11-1.6.4-3.el6.src.rpm", 1301928, "MIT", "CentOS", "", "Core X11 protocol client library", "2c87a4537fe52b62f5331864abfab1c8"},
			{intRef(), "libXext", "1.3.3", "1.el6", "x86_64", "libXext-1.3.3-1.el6.src.rpm", 85590, "MIT", "CentOS", "", "X.Org X11 libXext runtime library", "ad1011fc9d46bd36ff97386d96af0b85"},
			{intRef(), "libXfixes", "5.0.3", "1.el6", "x86_64", "libXfixes-5.0.3-1.el6.src.rpm", 23375, "MIT", "CentOS", "", "X Fixes library", "15c515aae8840654a9d2e8767467d3c8"},
			{intRef(), "libXi", "1.7.8", "1.el6", "x86_64", "libXi-1.7.8-1.el6.src.rpm", 66523, "MIT", "CentOS", "", "X.Org X11 libXi runtime library", "a9a16f4ab350184e430427f8bbdda785"},
			{intRef(), "libXtst", "1.2.3", "1.el6", "x86_64", "libXtst-1.2.3-1.el6.src.rpm", 27778, "MIT", "CentOS", "", "X.Org X11 libXtst runtime library", "82406d3f348200d299f62c06c70c0f2c"},
			{intRef(), "libXcursor", "1.1.14", "2.1.el6", "x86_64", "libXcursor-1.1.14-2.1.el6.src.rpm", 41934, "MIT", "CentOS", "", "Cursor management library", "703d84e636a0ffe106a1cfbdbf73ba0b"},
			{intRef(), "libXft", "2.3.2", "1.el6", "x86_64", "libXft-2.3.2-1.el6.src.rpm", 120045, "MIT", "CentOS", "", "X.Org X11 libXft runtime library", "b00354936cfa8d3cfcb761b24255c182"},
			{intRef(), "libXcomposite", "0.4.3", "4.el6", "x86_64", "libXcomposite-0.4.3-4.el6.src.rpm", 31878, "MIT", "CentOS", "", "X Composite Extension library", "41e6049143c54fe8a0af9072c343f9bf"},
			{intRef(), "perl-libwww-perl", "5.833", "5.el6", "noarch", "perl-libwww-perl-5.833-5.el6.src.rpm", 903654, "GPL+ or Artistic", "CentOS", "", "A Perl interface to the World-Wide Web", "16f96b7f7eb81c85801b5cd389991b84"},
			{intRef(), "redhat-logos", "60.0.14", "12.el6.centos", "noarch", "redhat-logos-60.0.14-12.el6.centos.src.rpm", 15816517, "Copyright 1999-2010 the CentOS Project.  All rights reserved.", "CentOS", "", "CentOS-related icons and pictures", "a1c4dbdefe5e3cdaff71a03177d18405"},
			{intRef(), "iptables", "1.4.7", "19.el6", "x86_64", "iptables-1.4.7-19.el6.src.rpm", 861752, "GPLv2", "CentOS", "", "Tools for managing Linux kernel packet filtering capabilities", "b90ad105c729e342f3bc2100f2c67296"},
			{intRef(), "libgcc", "4.4.7", "23.el6", "x86_64", "gcc-4.4.7-23.el6.src.rpm", 117352, "GPLv3+ and GPLv3+ with exceptions and GPLv2+ with exceptions", "CentOS", "", "GCC version 4.4 shared support library", "e7b74ed43c9baf242de1cfbf80f28cbf"},
			{intRef(), "libdrm", "2.4.65", "2.el6", "x86_64", "libdrm-2.4.65-2.el6.src.rpm", 306787, "MIT", "CentOS", "", "Direct Rendering Manager runtime library", "c37e1d3705a0ab67262bb0873fa32b6e"},
			{intRef(), "filesystem", "2.4.30", "3.el6", "x86_64", "filesystem-2.4.30-3.el6.src.rpm", 0, "Public Domain", "CentOS", "", "The basic directory layout for a Linux system", "8328ee9940f142cae12b0ba18a166a7d"},
			{intRef(), "plymouth", "0.8.3", "29.el6.centos", "x86_64", "plymouth-0.8.3-29.el6.centos.src.rpm", 193924, "GPLv2+", "CentOS", "", "Graphical Boot Animation and Logger", "77d86ee21808b2bbbe3ce64fbc82d79c"},
			{intRef(), "initscripts", "9.03.61", "1.el6.centos", "x86_64", "initscripts-9.03.61-1.el6.centos.src.rpm", 5735427, "GPLv2 and GPLv2+", "CentOS", "", "The inittab file and the /etc/init.d scripts", "a7dc57677500a05f5a9c85349765ea8d"},
			{intRef(), "ncurses-base", "5.7", "4.20090207.el6", "x86_64", "ncurses-5.7-4.20090207.el6.src.rpm", 193090, "MIT", "CentOS", "", "Descriptions of common terminals", "26e758f2dfba21d280faece6ec9011b1"},
			{intRef(), "openssh", "5.3p1", "124.el6_10", "x86_64", "openssh-5.3p1-124.el6_10.src.rpm", 787618, "BSD", "CentOS", "", "An open source implementation of SSH protocol versions 1 and 2", "0549cb22418fb5e2a52955ada98700ca"},
			{intRef(), "nss-softokn-freebl", "3.14.3", "23.3.el6_8", "x86_64", "nss-softokn-3.14.3-23.3.el6_8.src.rpm", 490290, "MPLv2.0", "CentOS", "", "Freebl library for the Network Security Services", "2c45c946199a527955ac597c2e0af02c"},
			{intRef(), "perl-Git", "1.7.1", "9.el6_9", "noarch", "git-1.7.1-9.el6_9.src.rpm", 35913, "GPLv2", "CentOS", "", "Perl interface to Git", "bd17e1027f3263de2560453aafb568b4"},
			{intRef(), "ncurses-libs", "5.7", "4.20090207.el6", "x86_64", "ncurses-5.7-4.20090207.el6.src.rpm", 752304, "MIT", "CentOS", "", "Ncurses libraries", "55dd8fcb0fddaa3718c4a9aca1c51470"},
			{intRef(), "kernel-devel", "2.6.32", "754.12.1.el6", "x86_64", "kernel-2.6.32-754.12.1.el6.src.rpm", 27015877, "GPLv2", "CentOS", "", "Development package for building kernel modules to match the kernel", "ee2a19d3e417208b274d5839678c2c73"},
			{intRef(), "libattr", "2.4.44", "7.el6", "x86_64", "attr-2.4.44-7.el6.src.rpm", 18712, "LGPLv2+", "CentOS", "", "Dynamic library for extended attribute support", "c1e774f9e24e226b50a855f49c377de0"},
			{intRef(), "systemtap-client", "2.9", "9.el6", "x86_64", "systemtap-2.9-9.el6.src.rpm", 12275931, "GPLv2+", "CentOS", "", "Programmable system-wide instrumentation system - client", "023f428406fc708b7ff1b47219609b06"},
			{intRef(), "zlib", "1.2.3", "29.el6", "x86_64", "zlib-1.2.3-29.el6.src.rpm", 152305, "zlib and Boost", "CentOS", "", "The zlib compression and decompression library", "17358a2e9d24db6825adfc7a21823891"},
			{intRef(), "gtk2", "2.24.23", "9.el6", "x86_64", "gtk2-2.24.23-9.el6.src.rpm", 12426003, "LGPLv2+", "CentOS", "", "The GIMP ToolKit (GTK+), a library for creating GUIs for X", "46a9c5747cc7bf3b75eec633677e8195"},
			{intRef(), "popt", "1.13", "7.el6", "x86_64", "popt-1.13-7.el6.src.rpm", 83420, "MIT", "CentOS", "", "C library for parsing command line parameters", "cdf20355eaf915d993d7584e2c4ee538"},
			{intRef(), "gettext-devel", "0.17", "18.el6", "x86_64", "gettext-0.17-18.el6.src.rpm", 635713, "LGPLv2+", "CentOS", "", "Development files for gettext", "b5bc3cf13d6cfd67fec7a6c6747891f4"},
			{intRef(), "db4", "4.7.25", "22.el6", "x86_64", "db4-4.7.25-22.el6.src.rpm", 1533095, "Sleepycat and BSD", "CentOS", "", "The Berkeley DB database library (version 4) for C", "5bc080a2706659ba5d200654b537306b"},
			{intRef(), "systemtap", "2.9", "9.el6", "x86_64", "systemtap-2.9-9.el6.src.rpm", 30373, "GPLv2+", "CentOS", "", "Programmable system-wide instrumentation system", "15c06dae0e065b2d524d5cffb6616982"},
			{intRef(), "nss-util", "3.36.0", "1.el6", "x86_64", "nss-util-3.36.0-1.el6.src.rpm", 191928, "MPLv2.0", "CentOS", "", "Network Security Services Utilities Library", "2a9a6cd34706c4ed8560fb50a8bd88d3"},
			{intRef(), "gcc-c++", "4.4.7", "23.el6", "x86_64", "gcc-4.4.7-23.el6.src.rpm", 11429003, "GPLv3+ and GPLv3+ with exceptions and GPLv2+ with exceptions", "CentOS", "", "C++ support for GCC", "72629895facd0975de5dfb25ec7db051"},
			{intRef(), "bzip2-libs", "1.0.5", "7.el6_0", "x86_64", "bzip2-1.0.5-7.el6_0.src.rpm", 67592, "BSD", "CentOS", "", "Libraries for applications using bzip2", "45e62d2d207887437a081283cd979a70"},
			{intRef(), "libtool", "2.2.6", "15.5.el6", "x86_64", "libtool-2.2.6-15.5.el6.src.rpm", 1998350, "GPLv2+ and LGPLv2+ and GFDL", "CentOS", "", "The GNU Portable Library Tool", "7382630bf6c5be8ce5bdbe838e2f4545"},
			{intRef(), "libselinux", "2.0.94", "7.el6", "x86_64", "libselinux-2.0.94-7.el6.src.rpm", 130336, "Public Domain", "CentOS", "", "SELinux library and simple utilities", "481af7162f2cb5714052ae7cf6e7e1ca"},
			{intRef(), "patchutils", "0.3.1", "3.1.el6", "x86_64", "patchutils-0.3.1-3.1.el6.src.rpm", 241488, "GPLv2+", "CentOS", "", "A collection of programs for manipulating patch files", "6579bc74b8c16e9ef5fdc6ed6822300d"},
			{intRef(), "sed", "4.2.1", "10.el6", "x86_64", "sed-4.2.1-10.el6.src.rpm", 542324, "GPLv2+", "CentOS", "", "A GNU stream text editor", "48bcbdc8eab711098683a23048799093"},
			{intRef(), "bison", "2.4.1", "5.el6", "x86_64", "bison-2.4.1-5.el6.src.rpm", 2081518, "GPLv3+", "CentOS", "", "A GNU general-purpose parser generator", "b29fdbd3afaa05248db1124e809db66e"},
			{intRef(), "libidn", "1.18", "2.el6", "x86_64", "libidn-1.18-2.el6.src.rpm", 567612, "LGPLv2+ and GPLv3+ and GFDL", "CentOS", "", "Internationalized Domain Name support library", "fd12d1a9b1bbeb0af06789ca1a91e227"},
			{intRef(), "ctags", "5.8", "2.el6", "x86_64", "ctags-5.8-2.el6.src.rpm", 347508, "GPLv2+ or Public Domain", "CentOS", "", "A C programming language indexing and/or cross-reference tool", "8cd8d127b85bf63a93caa7949ed9c615"},
			{intRef(), "libxml2", "2.7.6", "21.el6_8.1", "x86_64", "libxml2-2.7.6-21.el6_8.1.src.rpm", 1779163, "MIT", "CentOS", "", "Library providing XML and HTML support", "ddd19ff93764586cd73d56a28452d2e8"},
			{intRef(), "swig", "1.3.40", "6.el6", "x86_64", "swig-1.3.40-6.el6.src.rpm", 4290828, "GPLv2+ and LGPLv2+ and BSD", "CentOS", "", "Connects C/C++/Objective C to some high-level programming languages", "471775617eec311a82b9658bbc741a8a"},
			{intRef(), "libstdc++", "4.4.7", "23.el6", "x86_64", "gcc-4.4.7-23.el6.src.rpm", 987096, "GPLv3+ and GPLv3+ with exceptions and GPLv2+ with exceptions", "CentOS", "", "GNU Standard C++ Library", "100188ad74a764ef33ca7b9ed73e2507"},
			{intRef(), "indent", "2.2.10", "7.el6", "x86_64", "indent-2.2.10-7.el6.src.rpm", 252872, "GPLv3+", "CentOS", "", "A GNU program for formatting C code", "aa3c43c7b19aeea54c6f673727b2cd58"},
			{intRef(), "lua", "5.1.4", "4.1.el6", "x86_64", "lua-5.1.4-4.1.el6.src.rpm", 617799, "MIT", "CentOS", "", "Powerful light-weight programming language", "d73fd1effd24a1aeb20fd7b8be255a7d"},
			{intRef(), "byacc", "1.9.20070509", "7.el6", "x86_64", "byacc-1.9.20070509-7.el6.src.rpm", 87968, "Public Domain", "CentOS", "", "Berkeley Yacc, a parser generator", "dba1b6c1ae2677546cc5c83fb0c540be"},
			{intRef(), "gawk", "3.1.7", "10.el6_7.3", "x86_64", "gawk-3.1.7-10.el6_7.3.src.rpm", 2038650, "GPLv3+", "CentOS", "", "The GNU version of the awk text processing utility", "e057b2c0ac4253b726d364c3ce15789c"},
			{intRef(), "libblkid", "2.17.2", "12.28.el6_9.2", "x86_64", "util-linux-ng-2.17.2-12.28.el6_9.2.src.rpm", 136136, "LGPLv2+", "CentOS", "", "Block device ID library", "5bdee9aed72c9be1bf51b6c43d1e35a7"},
			{intRef(), "elfutils-libelf", "0.164", "2.el6", "x86_64", "elfutils-0.164-2.el6.src.rpm", 1011166, "GPLv2+ or LGPLv3+", "CentOS", "", "Library to read and write ELF files", "0b6033f7dac090e7decd459d61714a36"},
			{intRef(), "nss-softokn", "3.14.3", "23.3.el6_8", "x86_64", "nss-softokn-3.14.3-23.3.el6_8.src.rpm", 1103062, "MPLv2.0", "CentOS", "", "Network Security Services Softoken Module", "2acb6fdbf150b397270c2ab0a966b8f0"},
			{intRef(), "grep", "2.20", "6.el6", "x86_64", "grep-2.20-6.el6.src.rpm", 1197904, "GPLv3+", "CentOS", "", "Pattern matching utilities", "54b483d704c87fc97d0433b2ac89ca39"},
			{intRef(), "cpio", "2.10", "13.el6", "x86_64", "cpio-2.10-13.el6.src.rpm", 650433, "GPLv3+", "CentOS", "", "A GNU archiving program", "09fb2b399edb36bc6d8f69f183c90756"},
			{intRef(), "pth", "2.0.7", "9.3.el6", "x86_64", "pth-2.0.7-9.3.el6.src.rpm", 261931, "LGPLv2+", "CentOS", "", "The GNU Portable Threads library", "26e0ade2c2edb655da0922ce0b2ab0a3"},
			{intRef(), "libtasn1", "2.3", "6.el6_5", "x86_64", "libtasn1-2.3-6.el6_5.src.rpm", 443140, "GPLv3+ and LGPLv2+", "CentOS", "", "The ASN.1 library used in GNUTLS", "7dc9e15b02609ef338e49114556f3868"},
			{intRef(), "p11-kit-trust", "0.18.5", "2.el6_5.2", "x86_64", "p11-kit-0.18.5-2.el6_5.2.src.rpm", 178775, "BSD", "CentOS", "", "System trust module from p11-kit", "af4f4b1702c1edb385767deb382df6fd"},
			{intRef(), "libnih", "1.0.1", "8.el6", "x86_64", "libnih-1.0.1-8.el6.src.rpm", 490926, "GPLv2", "CentOS", "", "Lightweight application development library", "b69e9826410913a30efeb2849482ace1"},
			{intRef(), "gmp", "4.3.1", "13.el6", "x86_64", "gmp-4.3.1-13.el6.src.rpm", 657883, "LGPLv2+ and  GPLv3+ and LGPLv3+", "CentOS", "", "A GNU arbitrary precision library", "da0cc35676b00a0f5b216c3ca6187980"},
			{intRef(), "file", "5.04", "30.el6", "x86_64", "file-5.04-30.el6.src.rpm", 56567, "BSD", "CentOS", "", "A utility for determining file types", "5e5e8e881f53ba6a03c8978c037be2ac"},
			{intRef(), "net-tools", "1.60", "114.el6", "x86_64", "net-tools-1.60-114.el6.src.rpm", 778260, "GPL+", "CentOS", "", "Basic networking tools", "b55d83e6309f8af722284dfaef3d08f5"},
			{intRef(), "psmisc", "22.6", "24.el6", "x86_64", "psmisc-22.6-24.el6.src.rpm", 222569, "GPLv2+", "CentOS", "", "Utilities for managing processes on your system", "e54c765f163883606eaba07d0230d3cf"},
			{intRef(), "libselinux-utils", "2.0.94", "7.el6", "x86_64", "libselinux-2.0.94-7.el6.src.rpm", 62739, "Public Domain", "CentOS", "", "SELinux libselinux utilies", "fd6929bd238cf955852465941a9b757d"},
			{intRef(), "bzip2", "1.0.5", "7.el6_0", "x86_64", "bzip2-1.0.5-7.el6_0.src.rpm", 79087, "BSD", "CentOS", "", "A file compression utility", "0cc4395aa298488f73ba4b5836d25668"},
			{intRef(), "cyrus-sasl-lib", "2.1.23", "15.el6_6.2", "x86_64", "cyrus-sasl-2.1.23-15.el6_6.2.src.rpm", 357710, "BSD", "CentOS", "", "Shared libraries needed by applications which use Cyrus SASL", "6199bc3ccc2b414fb57b66bdeccd71d1"},
			{intRef(1), "make", "3.81", "23.el6", "x86_64", "make-3.81-23.el6.src.rpm", 1079569, "GPLv2+", "CentOS", "", "A GNU tool which simplifies the build process for users", "e2f8c3031a4384208a7b7e8b0f6f0f11"},
			{intRef(), "diffutils", "2.8.1", "28.el6", "x86_64", "diffutils-2.8.1-28.el6.src.rpm", 588813, "GPLv2+", "CentOS", "", "A GNU collection of diff utilities", "8ead45c0c09898f39170ea97c521581b"},
			{intRef(), "ncurses", "5.7", "4.20090207.el6", "x86_64", "ncurses-5.7-4.20090207.el6.src.rpm", 386371, "MIT", "CentOS", "", "Ncurses support utilities", "e2e764dda00e1b7bafc865e9a274bacf"},
			{intRef(), "less", "436", "13.el6", "x86_64", "less-436-13.el6.src.rpm", 201909, "GPLv3+", "CentOS", "", "A text file browser similar to more, but better", "1168dd2315ee9f080c3dea0bb45fc786"},
			{intRef(), "gzip", "1.3.12", "24.el6", "x86_64", "gzip-1.3.12-24.el6.src.rpm", 225924, "GPLv2+ and GFDL", "CentOS", "", "The GNU data compression program", "75fbcc40c3e1e745402ca99687f99a14"},
			{intRef(), "cracklib-dicts", "2.8.16", "4.el6", "x86_64", "cracklib-2.8.16-4.el6.src.rpm", 9327207, "LGPLv2+", "CentOS", "", "The standard CrackLib dictionaries", "cadecbc8984f7bc55493c47e0d2c27fe"},
			{intRef(), "pam", "1.1.1", "24.el6", "x86_64", "pam-1.1.1-24.el6.src.rpm", 2419646, "BSD and GPLv2+", "CentOS", "", "An extensible library which provides authentication for applications", "9c5ae01c61f415d34fe4b192d5b5add3"},
			{intRef(), "plymouth-scripts", "0.8.3", "29.el6.centos", "x86_64", "plymouth-0.8.3-29.el6.centos.src.rpm", 11006, "GPLv2+", "CentOS", "", "Plymouth related scripts", "15e520370abb095e2fd36b86d863d737"},
			{intRef(), "ca-certificates", "2018.2.22", "65.1.el6", "noarch", "ca-certificates-2018.2.22-65.1.el6.src.rpm", 2675229, "Public Domain", "CentOS", "", "The Mozilla CA root certificate bundle", "33aa2022c5bfe5613215e68dc02941af"},
			{intRef(), "nss-sysinit", "3.36.0", "8.el6", "x86_64", "nss-3.36.0-8.el6.src.rpm", 32822, "MPLv2.0", "CentOS", "", "System NSS Initialization", "6188b4840cfa3f575e762328501f721f"},
			{intRef(2), "ethtool", "3.5", "6.el6", "x86_64", "ethtool-3.5-6.el6.src.rpm", 283910, "GPLv2", "CentOS", "", "Ethernet settings tool for PCI ethernet cards", "9649daaebc2c5dadae7da90cf4c0613c"},
			{intRef(), "keyutils-libs", "1.4", "5.el6", "x86_64", "keyutils-1.4-5.el6.src.rpm", 36624, "GPLv2+ and LGPLv2+", "CentOS", "", "Key utilities library", "e55a68fb4ada530ec6f52d34e1dee538"},
			{intRef(), "openssl", "1.0.1e", "57.el6", "x86_64", "openssl-1.0.1e-57.el6.src.rpm", 4248338, "OpenSSL", "CentOS", "", "A general purpose cryptography library with TLS implementation", "43a12b9b1a7050d62b52099552ec7963"},
			{intRef(), "libcurl", "7.19.7", "53.el6_9", "x86_64", "curl-7.19.7-53.el6_9.src.rpm", 347536, "MIT", "CentOS", "", "A library for getting files from web servers", "587463a0294856ae3177d97b70f8136c"},
			{intRef(), "curl", "7.19.7", "53.el6_9", "x86_64", "curl-7.19.7-53.el6_9.src.rpm", 357074, "MIT", "CentOS", "", "A utility for getting files from remote servers (FTP, HTTP, and others)", "8ab5f2926ce20347a872179bcb89500b"},
			{intRef(), "openldap", "2.4.40", "16.el6", "x86_64", "openldap-2.4.40-16.el6.src.rpm", 859518, "OpenLDAP", "CentOS", "", "LDAP support libraries", "5fb3cd202d6354c28ebdf420c8eb5916"},
			{intRef(), "gpgme", "1.1.8", "3.el6", "x86_64", "gpgme-1.1.8-3.el6.src.rpm", 729658, "LGPLv2+", "CentOS", "", "GnuPG Made Easy - high level crypto API", "f20b9ddce4a0b8952a50a826c2b5fcfb"},
			{intRef(), "ustr", "1.0.4", "9.1.el6", "x86_64", "ustr-1.0.4-9.1.el6.src.rpm", 273983, "MIT or LGPLv2+ or BSD", "CentOS", "", "String library, very low memory overhead, simple to import", "16060496fca7666123134438f27c9e91"},
			{intRef(2), "shadow-utils", "4.1.5.1", "5.el6", "x86_64", "shadow-utils-4.1.5.1-5.el6.src.rpm", 3466724, "BSD and GPLv2+", "CentOS", "", "Utilities for managing accounts and shadow password files", "3c869910a1ae19bdd1392b4ac1cc4c40"},
			{intRef(), "MAKEDEV", "3.24", "6.el6", "x86_64", "MAKEDEV-3.24-6.el6.src.rpm", 227290, "GPLv2", "CentOS", "", "A program used for creating device files in /dev", "1a999a3ef7117fbf20cf67e726429a60"},
			{intRef(), "gdbm", "1.8.0", "39.el6", "x86_64", "gdbm-1.8.0-39.el6.src.rpm", 48770, "GPLv2+", "CentOS", "", "A GNU set of database routines which use extensible hashing", "f32d22ad07490183bc56bceeb30baa8e"},
			{intRef(), "python", "2.6.6", "66.el6_8", "x86_64", "python-2.6.6-66.el6_8.src.rpm", 79603, "Python", "CentOS", "", "An interpreted, interactive, object-oriented programming language", "649646824189190c3153fcdcc7cf686d"},
			{intRef(), "rpm-python", "4.8.0", "59.el6", "x86_64", "rpm-4.8.0-59.el6.src.rpm", 120906, "GPLv2+", "CentOS", "", "Python bindings for apps which will manipulate RPM packages", "907e2545aaf74bfff58a005542c8daff"},
			{intRef(), "python-pycurl", "7.19.0", "9.el6", "x86_64", "python-pycurl-7.19.0-9.el6.src.rpm", 236939, "LGPLv2+ or MIT", "CentOS", "", "A Python interface to libcurl", "cd1c2c372f3036b9594b942bc57dcffb"},
			{intRef(), "python-iniparse", "0.3.1", "2.1.el6", "noarch", "python-iniparse-0.3.1-2.1.el6.src.rpm", 109284, "MIT", "CentOS", "", "Python Module for Accessing and Modifying Configuration Data in INI files", "1f3ba002bade99b8ba653d1a4ef63337"},
			{intRef(), "gamin", "0.1.10", "9.el6", "x86_64", "gamin-0.1.10-9.el6.src.rpm", 416440, "LGPLv2", "CentOS", "", "Library providing the FAM File Alteration Monitor API", "5156618d3b648e76d643495671fc0abd"},
			{intRef(), "shared-mime-info", "0.70", "6.el6", "x86_64", "shared-mime-info-0.70-6.el6.src.rpm", 1411396, "GPLv2+", "CentOS", "", "Shared MIME information database", "5e6203e5374e767bbb3eecaf04e06836"},
			{intRef(), "libuser", "0.56.13", "8.el6_7", "x86_64", "libuser-0.56.13-8.el6_7.src.rpm", 1882055, "LGPLv2+", "CentOS", "", "A user and group account administration library", "ce682d078b0b7d99df962a8bb7d0c719"},
			{intRef(), "yum-plugin-fastestmirror", "1.1.30", "42.el6_10", "noarch", "yum-utils-1.1.30-42.el6_10.src.rpm", 53961, "GPLv2+", "CentOS", "", "Yum plugin which chooses fastest repository from a mirrorlist", "8d6fd5bad8fe4e6a932cf7aa624ab360"},
			{intRef(), "centos-release", "6", "10.el6.centos.12.3", "x86_64", "centos-release-6-10.el6.centos.12.3.src.rpm", 38232, "GPLv2", "CentOS", "", "CentOS release file", "0a78cbc4776e9fce236b09814009025f"},
			{intRef(), "yum-plugin-ovl", "1.1.30", "42.el6_10", "noarch", "yum-utils-1.1.30-42.el6_10.src.rpm", 22350, "GPLv2+", "CentOS", "", "Yum plugin to work around overlayfs issues", "e6f25f528cb1d4e7af487da25bf648ee"},
			{intRef(), "rootfiles", "8.1", "6.1.el6", "noarch", "rootfiles-8.1-6.1.el6.src.rpm", 599, "Public Domain", "CentOS", "", "The basic required files for the root user's directory", "6a2b3154d2e600c8c90d15b2a61fbd76"},
			{intRef(), "glibc", "2.12", "1.212.el6_10.3", "x86_64", "glibc-2.12-1.212.el6_10.3.src.rpm", 13117447, "LGPLv2+ and LGPLv2+ with exceptions and GPLv2+", "CentOS", "", "The GNU libc libraries", "4c7a81c88f2273ac4eec12312de1dade"},
			{intRef(), "freetype", "2.3.11", "17.el6", "x86_64", "freetype-2.3.11-17.el6.src.rpm", 836907, "FTL or GPLv2+", "CentOS", "", "A free and portable font rendering engine", "081c635819bce2feccc9161a78b45c96"},
			{intRef(), "libjpeg-turbo", "1.2.1", "3.el6_5", "x86_64", "libjpeg-turbo-1.2.1-3.el6_5.src.rpm", 476782, "wxWidgets", "CentOS", "", "A MMX/SSE2 accelerated library for manipulating JPEG image files", "60f8e5d81bd60ede61babd48fadb82df"},
			{intRef(), "libproxy-python", "0.3.0", "10.el6", "x86_64", "libproxy-0.3.0-10.el6.src.rpm", 8320, "LGPLv2+", "CentOS", "", "Binding for libproxy and python", "411e2885bbd0c04c57de835ab6f9a675"},
			{intRef(), "m4", "1.4.13", "5.el6", "x86_64", "m4-1.4.13-5.el6.src.rpm", 560949, "GPLv3+", "CentOS", "", "The GNU macro processor", "920603310fe2915ee14b1d40ce423ba7"},
			{intRef(), "elfutils-libs", "0.164", "2.el6", "x86_64", "elfutils-0.164-2.el6.src.rpm", 648199, "GPLv2+ or LGPLv3+", "CentOS", "", "Libraries to handle compiled objects", "649d6422db36cc364218b19e26605e4f"},
			{intRef(), "avahi-libs", "0.6.25", "17.el6", "x86_64", "avahi-0.6.25-17.el6.src.rpm", 114824, "LGPLv2", "CentOS", "", "Libraries for avahi run-time use", "ef17f65cf598c96ea63183ec6e4fd6ab"},
			{intRef(), "gnutls", "2.12.23", "22.el6", "x86_64", "gnutls-2.12.23-22.el6.src.rpm", 1167530, "GPLv3+ and LGPLv2+", "CentOS", "", "A TLS protocol implementation", "a3daf16b5f5c5d6dd5223df7c456b873"},
			{intRef(), "xz", "4.999.9", "0.5.beta.20091007git.el6", "x86_64", "xz-4.999.9-0.5.beta.20091007git.el6.src.rpm", 488160, "LGPLv2+", "CentOS", "", "LZMA compression utilities", "467d0fcd75321d77e5ee7193a9b422af"},
			{intRef(), "binutils", "2.20.51.0.2", "5.48.el6_10.1", "x86_64", "binutils-2.20.51.0.2-5.48.el6_10.1.src.rpm", 9831328, "GPLv3+", "CentOS", "", "A GNU collection of binary utilities", "a46a671ca94b19e3297ecdc7820ea677"},
			{intRef(), "unzip", "6.0", "5.el6", "x86_64", "unzip-6.0-5.el6.src.rpm", 331766, "BSD", "CentOS", "", "A utility for unpacking zip files", "07aa8e52cc567ff759018dabf4f6434b"},
			{intRef(), "libgomp", "4.4.7", "23.el6", "x86_64", "gcc-4.4.7-23.el6.src.rpm", 127982, "GPLv3+ and GPLv3+ with exceptions and GPLv2+ with exceptions", "CentOS", "", "GCC OpenMP v3.0 shared support library", "592333d5d1adb6e8cae3c37d86f65be6"},
			{intRef(), "patch", "2.6", "8.el6_9", "x86_64", "patch-2.6-8.el6_9.src.rpm", 179790, "GPLv2+", "CentOS", "", "Utility for modifying/upgrading files", "f0a28aacaf3c6db5847a30fbd75995c5"},
			{intRef(), "libpciaccess", "0.13.4", "1.el6", "x86_64", "libpciaccess-0.13.4-1.el6.src.rpm", 40777, "MIT", "CentOS", "", "PCI access library", "3c5695676634ec16e2a140ac2aa2eda5"},
			{intRef(), "apr-util", "1.3.9", "3.el6_0.1", "x86_64", "apr-util-1.3.9-3.el6_0.1.src.rpm", 202360, "ASL 2.0", "CentOS", "", "Apache Portable Runtime Utility library", "a73fea4ca6995c987b6d6c5ebf295cf1"},
			{intRef(1), "cups-libs", "1.4.2", "81.el6_10", "x86_64", "cups-1.4.2-81.el6_10.src.rpm", 667768, "LGPLv2", "CentOS", "", "Common Unix Printing System - libraries", "2863fed90f6f9e697e86a4935890b3da"},
			{intRef(), "cpp", "4.4.7", "23.el6", "x86_64", "gcc-4.4.7-23.el6.src.rpm", 10012603, "GPLv3+ and GPLv3+ with exceptions and GPLv2+ with exceptions", "CentOS", "", "The C Preprocessor", "d2cb382510fb5e37cd8dab0bc8840fe7"},
			{intRef(), "pakchois", "0.4", "3.2.el6", "x86_64", "pakchois-0.4-3.2.el6.src.rpm", 50156, "LGPLv2+", "CentOS", "", "A wrapper library for PKCS#11", "de07ec421f7e21f52ad524495b05f49b"},
			{intRef(), "dbus-glib", "0.86", "6.el6", "x86_64", "dbus-glib-0.86-6.el6.src.rpm", 579611, "AFL and GPLv2+", "CentOS", "", "GLib bindings for D-Bus", "36355c2ebf53d75d3573433be1fe07af"},
			{intRef(), "systemtap-runtime", "2.9", "9.el6", "x86_64", "systemtap-2.9-9.el6.src.rpm", 616098, "GPLv2+", "CentOS", "", "Programmable system-wide instrumentation system - runtime", "4971f8dc2cb537034137070c8218c942"},
			{intRef(), "fipscheck-lib", "1.2.0", "7.el6", "x86_64", "fipscheck-1.2.0-7.el6.src.rpm", 10353, "BSD", "CentOS", "", "Library files for fipscheck", "bc082d2bc191e3367d04c5e9d90b64a9"},
			{intRef(), "pixman", "0.32.8", "1.el6", "x86_64", "pixman-0.32.8-1.el6.src.rpm", 721560, "MIT", "CentOS", "", "Pixel manipulation library", "10f2a88a8d7753eb1b433a7e36e243e7"},
			{intRef(4), "perl-libs", "5.10.1", "144.el6", "x86_64", "perl-5.10.1-144.el6.src.rpm", 1485896, "GPL+ or Artistic", "CentOS", "", "The libraries for the perl runtime", "c8d9e92a17d953b36d59238e45a78695"},
			{intRef(1), "perl-Module-Pluggable", "3.90", "144.el6", "x86_64", "perl-5.10.1-144.el6.src.rpm", 31033, "GPL+ or Artistic", "CentOS", "", "Automatically give your module the ability to have plugins", "86c0ff292dcc2511141221544f4a692a"},
			{intRef(4), "perl", "5.10.1", "144.el6", "x86_64", "perl-5.10.1-144.el6.src.rpm", 35282834, "(GPL+ or Artistic) and (GPLv2+ or Artistic) and Copyright Only and MIT and Public Domain and UCD", "CentOS", "", "Practical Extraction and Report Language", "25ae099cca4c5561c027d40a3d818455"},
			{intRef(1), "perl-Error", "0.17015", "4.el6", "noarch", "perl-Error-0.17015-4.el6.src.rpm", 47372, "GPL+ or Artistic", "CentOS", "", "Error/exception handling in an OO-ish way", "4e1b07c3b809e5fe89325f99fc8c4cf3"},
			{intRef(), "automake", "1.11.1", "4.el6", "noarch", "automake-1.11.1-4.el6.src.rpm", 1520636, "GPLv2+ and GFDL", "CentOS", "", "A GNU tool for automatically creating Makefiles", "fc4c86d7b34501bbca46487558576ed6"},
			{intRef(0), "perl-IO-Compress-Base", "2.021", "144.el6", "x86_64", "perl-5.10.1-144.el6.src.rpm", 138661, "GPL+ or Artistic", "CentOS", "", "Base Class for IO::Compress modules", "3ef8b6a0b11cf795826a54be4178da3e"},
			{intRef(0), "perl-Compress-Zlib", "2.021", "144.el6", "x86_64", "perl-5.10.1-144.el6.src.rpm", 52519, "GPL+ or Artistic", "CentOS", "", "A module providing Perl interfaces to the zlib compression library", "0b32c2c8994e47574b363bb425ce90de"},
			{intRef(), "perl-HTML-Tagset", "3.20", "4.el6", "noarch", "perl-HTML-Tagset-3.20-4.el6.src.rpm", 19692, "GPL+ or Artistic", "CentOS", "", "HTML::Tagset - data tables useful in parsing HTML", "05f5ff07267411d09c637378b82bd9fc"},
			{intRef(), "cvs", "1.11.23", "16.el6", "x86_64", "cvs-1.11.23-16.el6.src.rpm", 1590583, "GPL+ and GPLv2+ and LGPL+", "CentOS", "", "A version control system", "2b017b45821781a4fe60454aa3ad52e2"},
			{intRef(), "libXau", "1.0.6", "4.el6", "x86_64", "libXau-1.0.6-4.el6.src.rpm", 40009, "MIT", "CentOS", "", "Sample Authorization Protocol for X", "556b44d7bf6ae5ea3c2d05e7e3eedb33"},
			{intRef(), "upstart", "0.6.5", "17.el6", "x86_64", "upstart-0.6.5-17.el6.src.rpm", 567729, "GPLv2 and LGPLv2+", "CentOS", "", "An event-driven init system", "f70eae1ce0b394f9309a830fba720d24"},
			{intRef(), "cloog-ppl", "0.15.7", "1.2.el6", "x86_64", "cloog-0.15.7-1.2.el6.src.rpm", 185386, "GPLv2+", "CentOS", "", "Parma Polyhedra Library backend (ppl) based version of the Cloog binaries", "45bf8b3d5ecdf00b8920cae8215cd23f"},
			{intRef(), "gdb", "7.2", "92.el6", "x86_64", "gdb-7.2-92.el6.src.rpm", 5545845, "GPLv3+ and GPLv3+ with exceptions and GPLv2+ and GPLv2+ with exceptions and GPL+ and LGPLv2+ and GFDL and BSD and Public Domain", "CentOS", "", "A GNU source-level debugger for C, C++, Java and other languages", "bd590a46cfb4a39436f5d21d7dc3d111"},
			{intRef(), "libthai", "0.1.12", "3.el6", "x86_64", "libthai-0.1.12-3.el6.src.rpm", 694712, "LGPLv2+", "CentOS", "", "Thai language support routines", "e555d32a374e2e0be18695cb4daba52b"},
			{intRef(), "libedit", "2.11", "4.20080712cvs.1.el6", "x86_64", "libedit-2.11-4.20080712cvs.1.el6.src.rpm", 185648, "BSD", "CentOS", "", "The NetBSD Editline library", "85e16c85ed37d898a4e1512971b0f26f"},
			{intRef(), "glibc-headers", "2.12", "1.212.el6_10.3", "x86_64", "glibc-2.12-1.212.el6_10.3.src.rpm", 2146171, "LGPLv2+ and LGPLv2+ with exceptions and GPLv2+", "CentOS", "", "Header files for development using standard C libraries.", "d05960317274627ff65930e2b13ac43c"},
			{intRef(), "gcc", "4.4.7", "23.el6", "x86_64", "gcc-4.4.7-23.el6.src.rpm", 19499265, "GPLv3+ and GPLv3+ with exceptions and GPLv2+ with exceptions", "CentOS", "", "Various compilers (C, C++, Objective-C, Java, ...)", "d20a4218376de95fdf4f7b64eb54b9dc"},
			{intRef(), "libX11-common", "1.6.4", "3.el6", "noarch", "libX11-1.6.4-3.el6.src.rpm", 1341368, "MIT", "CentOS", "", "Common data for libX11", "9f2fffc04a738fe0c5422492e98e8e7e"},
			{intRef(), "libXrender", "0.9.10", "1.el6", "x86_64", "libXrender-0.9.10-1.el6.src.rpm", 40515, "MIT", "CentOS", "", "X.Org X11 libXrender runtime library", "47d558a87001213505278f36f6789efa"},
			{intRef(), "cairo", "1.8.8", "6.el6_6", "x86_64", "cairo-1.8.8-6.el6_6.src.rpm", 797873, "LGPLv2 or MPLv1.1", "CentOS", "", "A 2D graphics library", "8a2bb9e82c6703437e07d2ddb24dd6e9"},
			{intRef(), "libXrandr", "1.5.1", "1.el6", "x86_64", "libXrandr-1.5.1-1.el6.src.rpm", 43834, "MIT", "CentOS", "", "X.Org X11 libXrandr runtime library", "d0faae2bde7eb356e450f1c8effb19a8"},
			{intRef(), "gdk-pixbuf2", "2.24.1", "6.el6_7", "x86_64", "gdk-pixbuf2-2.24.1-6.el6_7.src.rpm", 2637837, "LGPLv2+ and (LGPLv2+ or MPLv1.1) and Public Domain", "CentOS", "", "An image loading library", "ce2735d6e78fcd90d3315408a8275cc2"},
			{intRef(), "libXdamage", "1.1.3", "4.el6", "x86_64", "libXdamage-1.1.3-4.el6.src.rpm", 24896, "MIT", "CentOS", "", "X Damage extension library", "dd355decbb52211e6b7d3bd0bdebb14b"},
			{intRef(), "libXinerama", "1.1.3", "2.1.el6", "x86_64", "libXinerama-1.1.3-2.1.el6.src.rpm", 12039, "MIT", "CentOS", "", "X.Org X11 libXinerama runtime library", "2116fec43b4a0af1c84ee30d7d1d93fd"},
			{intRef(), "pango", "1.28.1", "11.el6", "x86_64", "pango-1.28.1-11.el6.src.rpm", 1041857, "LGPLv2+", "CentOS", "", "System for layout and rendering of internationalized text", "e159760ef6e92c3edb581e57bc3ed0e1"},
			{intRef(), "mailcap", "2.1.31", "2.el6", "noarch", "mailcap-2.1.31-2.el6.src.rpm", 52877, "Public Domain and MIT", "CentOS", "", "Helper application and MIME type associations for file types", "ad32dcba085cf9233930addac50d9d39"},
			{intRef(), "perl-XML-Parser", "2.36", "7.el6", "x86_64", "perl-XML-Parser-2.36-7.el6.src.rpm", 652307, "GPL+ or Artistic", "CentOS", "", "Perl module for parsing XML files", "5401c94b342490b49f15778da75ae4b5"},
			{intRef(), "policycoreutils", "2.0.83", "30.1.el6_8", "x86_64", "policycoreutils-2.0.83-30.1.el6_8.src.rpm", 3596110, "GPLv2+", "CentOS", "", "SELinux policy core utilities", "b3aad0189a7e858962379b5721fd3244"},
		}

		return toPackageInfo(pkgs)
	}

	// docker run --rm -it centos:6 bash
	// yum groupinstall -y "Development tools"
	// yum install -y rpm-build redhat-rpm-config asciidoc hmaccalc perl-ExtUtils-Embed pesign xmlto
	// yum install -y audit-libs-devel binutils-devel elfutils-devel elfutils-libelf-devel java-devel
	// yum install -y ncurses-devel newt-devel numactl-devel pciutils-devel python-devel zlib-devel
	// yum install -y net-tools bc
	// rpm -qa --queryformat "\{%{EPOCH}, \"%{NAME}\", \"%{VERSION}\", \"%{RELEASE}\", \"%{ARCH}\", \"%{SOURCERPM}\", %{SIZE}, \"%{LICENSE}\", \"%{VENDOR}\", \"\", \"%{SUMMARY}\", \"%{SIGMD5}\"\},\n" | sed "s/^{(none)/{intRef()/g" | sed -r 's/^\{([0-9]+),/{intRef(\1),/' | sed "s/(none)/0/g"
	CentOS6Many = func() []*PackageInfo {
		pkgs := []*commonPackageInfo{
			{intRef(), "iproute", "2.6.32", "57.el6", "x86_64", "iproute-2.6.32-57.el6.src.rpm", 963477, "GPLv2+ and Public Domain", "CentOS", "", "Advanced IP routing and network device configuration tools", "68c6859576cf64a7ed187072908587b2"},
			{intRef(), "setup", "2.8.14", "23.el6", "noarch", "setup-2.8.14-23.el6.src.rpm", 666669, "Public Domain", "CentOS", "", "A set of system configuration and setup files", "f58d4ff770a377d42854025b5c64a709"},
			{intRef(), "iputils", "20071127", "24.el6", "x86_64", "iputils-20071127-24.el6.src.rpm", 297243, "BSD with advertising and GPLv2+ and Rdisc", "CentOS", "", "Network monitoring tools including ping", "9429035cf29d52ecbf1eee4a7db120c7"},
			{intRef(), "basesystem", "10.0", "4.el6", "noarch", "basesystem-10.0-4.el6.src.rpm", 0, "Public Domain", "CentOS", "", "The skeleton package which defines a simple Red Hat Enterprise Linux system", "bf36e0d2771604b8ad56d9696b5edd3c"},
			{intRef(), "util-linux-ng", "2.17.2", "12.28.el6_9.2", "x86_64", "util-linux-ng-2.17.2-12.28.el6_9.2.src.rpm", 6139437, "GPLv1+ and GPLv2 and GPLv2+ and LGPLv2+ and MIT and BSD with advertising and Public Domain", "CentOS", "", "A collection of basic system utilities", "ee7e9cba004ab79dce14e988cfb6d157"},
			{intRef(), "tzdata", "2018e", "3.el6", "noarch", "tzdata-2018e-3.el6.src.rpm", 1961609, "Public Domain", "CentOS", "", "Timezone data", "95de917c04de115372e5b75c3874db97"},
			{intRef(), "udev", "147", "2.73.el6_8.2", "x86_64", "udev-147-2.73.el6_8.2.src.rpm", 1280842, "GPLv2", "CentOS", "", "A userspace implementation of devfs", "3e0a5f14576169587dc44a4ebb392442"},
			{intRef(), "openssh-clients", "5.3p1", "124.el6_10", "x86_64", "openssh-5.3p1-124.el6_10.src.rpm", 1352442, "BSD", "CentOS", "", "An open source SSH client applications", "7d7b66b85b2e2e36906eb9c82b8095f2"},
			{intRef(), "git", "1.7.1", "9.el6_9", "x86_64", "git-1.7.1-9.el6_9.src.rpm", 15290753, "GPLv2", "CentOS", "", "Fast Version Control System", "395cf7af32a17cedd20acddfa6e85952"},
			{intRef(), "bash", "4.1.2", "48.el6", "x86_64", "bash-4.1.2-48.el6.src.rpm", 3142529, "GPLv3+", "CentOS", "", "The GNU Bourne Again shell", "fb70e495a46a63523eab6138cae2438d"},
			{intRef(), "systemtap-devel", "2.9", "9.el6", "x86_64", "systemtap-2.9-9.el6.src.rpm", 6327079, "GPLv2+", "CentOS", "", "Programmable system-wide instrumentation system - development headers, tools", "974555cbff98caf78f7b088d7362ec5e"},
			{intRef(), "libcap", "2.16", "5.5.el6", "x86_64", "libcap-2.16-5.5.el6.src.rpm", 64437, "LGPLv2+ or BSD", "CentOS", "", "Library for getting and setting POSIX.1e capabilities", "5d224cdbaa15053a3aa5443a07a45c2b"},
			{intRef(), "hicolor-icon-theme", "0.11", "1.1.el6", "noarch", "hicolor-icon-theme-0.11-1.1.el6.src.rpm", 45406, "GPL+", "CentOS", "", "Basic requirement for icon themes", "c1cf9a985abca13245f6fdbd6b4cb57c"},
			{intRef(), "info", "4.13a", "8.el6", "x86_64", "texinfo-4.13a-8.el6.src.rpm", 329482, "GPLv3+", "CentOS", "", "A stand-alone TTY-based reader for GNU texinfo documentation", "482d0f1ec0690ed672c0c1b594f78583"},
			{intRef(), "libgcj", "4.4.7", "23.el6", "x86_64", "gcc-4.4.7-23.el6.src.rpm", 65012480, "GPLv3+ and GPLv3+ with exceptions and GPLv2+ with exceptions", "CentOS", "", "Java runtime library for gcc", "8522846e5285eb055ec4ae698f001a60"},
			{intRef(), "libacl", "2.2.49", "7.el6_9.1", "x86_64", "acl-2.2.49-7.el6_9.1.src.rpm", 31280, "LGPLv2+", "CentOS", "", "Dynamic library for access control list support", "98166984ca2ff5aabfd1a43fdc89c788"},
			{intRef(), "intltool", "0.41.0", "1.1.el6", "noarch", "intltool-0.41.0-1.1.el6.src.rpm", 167585, "GPLv2 with exceptions", "CentOS", "", "Utility for internationalizing various kinds of data files", "72bad34320d1dc10227e2ae32cbc1d67"},
			{intRef(), "nspr", "4.19.0", "1.el6", "x86_64", "nspr-4.19.0-1.el6.src.rpm", 281968, "MPLv2.0", "CentOS", "", "Netscape Portable Runtime", "18e1b8a44c636953cf82358dc0978d34"},
			{intRef(), "subversion", "1.6.11", "15.el6_7", "x86_64", "subversion-1.6.11-15.el6_7.src.rpm", 12105530, "ASL 1.1", "CentOS", "", "A Modern Concurrent Version Control System", "6366595f694e84409847626985986d56"},
			{intRef(), "libcom_err", "1.41.12", "24.el6", "x86_64", "e2fsprogs-1.41.12-24.el6.src.rpm", 59233, "MIT", "CentOS", "", "Common error description library", "3d4d863ae951dde28e33d4fec21c793d"},
			{intRef(), "gcc-gfortran", "4.4.7", "23.el6", "x86_64", "gcc-4.4.7-23.el6.src.rpm", 14473805, "GPLv3+ and GPLv3+ with exceptions and GPLv2+ with exceptions", "CentOS", "", "Fortran support", "5c76186dccdc67d7cd5afc864e6a92ce"},
			{intRef(), "libsepol", "2.0.41", "4.el6", "x86_64", "libsepol-2.0.41-4.el6.src.rpm", 248680, "LGPLv2+", "CentOS", "", "SELinux binary policy manipulation library", "cbc386b646b94d710dfabe7126ede725"},
			{intRef(), "rpm-build", "4.8.0", "59.el6", "x86_64", "rpm-4.8.0-59.el6.src.rpm", 323213, "GPLv2+", "CentOS", "", "Scripts and executable programs used to build packages", "79d4da891e041d8dd9d9f83ed935da1c"},
			{intRef(), "chkconfig", "1.3.49.5", "1.el6", "x86_64", "chkconfig-1.3.49.5-1.el6.src.rpm", 670580, "GPLv2", "CentOS", "", "A system tool for maintaining the /etc/rc*.d hierarchy", "c5e95feb1e3c73bfd0761981c866a024"},
			{intRef(), "flex", "2.5.35", "9.el6", "x86_64", "flex-2.5.35-9.el6.src.rpm", 736081, "BSD", "CentOS", "", "A tool for creating scanners (text pattern recognizers)", "c952749600445a57fe3f865c47c0d22c"},
			{intRef(), "audit-libs", "2.4.5", "6.el6", "x86_64", "audit-2.4.5-6.el6.src.rpm", 235282, "LGPLv2+", "CentOS", "", "Dynamic library for libaudit", "fec1ad9c60d9d6e870ea6182ba6be06a"},
			{intRef(), "cscope", "15.6", "7.el6", "x86_64", "cscope-15.6-7.el6.src.rpm", 466265, "BSD", "CentOS", "", "C source code tree search and browse tool", "695c44b601b82d29ae3e73ab8dfaf9d1"},
			{intRef(), "readline", "6.0", "4.el6", "x86_64", "readline-6.0-4.el6.src.rpm", 433957, "GPLv3+", "CentOS", "", "A library for editing typed command lines", "d8b0aa8a08f36b43eb4858d76d005c6d"},
			{intRef(), "rcs", "5.7", "37.el6", "x86_64", "rcs-5.7-37.el6.src.rpm", 709927, "GPLv2+", "CentOS", "", "Revision Control System (RCS) file version management tools", "f9340500810916e598bb8c88a2f47f4e"},
			{intRef(), "file-libs", "5.04", "30.el6", "x86_64", "file-5.04-30.el6.src.rpm", 2486624, "BSD", "CentOS", "", "Libraries for applications using libmagic", "e27f1d59098a685689f3f2d099946a85"},
			{intRef(), "diffstat", "1.51", "2.el6", "x86_64", "diffstat-1.51-2.el6.src.rpm", 45568, "MIT", "CentOS", "", "A utility which provides statistics based on the output of diff", "72f972ef4f85141a4c98f69908c5411a"},
			{intRef(1), "dbus-libs", "1.2.24", "9.el6", "x86_64", "dbus-1.2.24-9.el6.src.rpm", 265736, "GPLv2+ or AFL", "CentOS", "", "Libraries for accessing D-BUS", "3b7d65739639c300ab7a69e890240f32"},
			{intRef(1), "doxygen", "1.6.1", "6.el6", "x86_64", "doxygen-1.6.1-6.el6.src.rpm", 9446190, "GPL+", "CentOS", "", "A documentation system for C/C++", "397c080755741ab4bba77e6b18beccd0"},
			{intRef(), "sqlite", "3.6.20", "1.el6_7.2", "x86_64", "sqlite-3.6.20-1.el6_7.2.src.rpm", 640060, "Public Domain", "CentOS", "", "Library that implements an embeddable SQL database engine", "d94a1809a80676af38140d2f0021c5f9"},
			{intRef(), "libxslt", "1.1.26", "2.el6_3.1", "x86_64", "libxslt-1.1.26-2.el6_3.1.src.rpm", 2223955, "MIT", "CentOS", "", "Library providing the Gnome XSLT engine", "1c1d53132c59a11ee3d4852e8f3ee8f6"},
			{intRef(), "libuuid", "2.17.2", "12.28.el6_9.2", "x86_64", "util-linux-ng-2.17.2-12.28.el6_9.2.src.rpm", 16304, "BSD", "CentOS", "", "Universally unique ID library", "9e1615002308c631aaee83e16668543f"},
			{intRef(), "db4-cxx", "4.7.25", "22.el6", "x86_64", "db4-4.7.25-22.el6.src.rpm", 1647216, "Sleepycat and BSD", "CentOS", "", "The Berkeley DB database library (version 4) for C++", "33ca9ce906a4bc1eafd73c26634a35da"},
			{intRef(), "xz-libs", "4.999.9", "0.5.beta.20091007git.el6", "x86_64", "xz-4.999.9-0.5.beta.20091007git.el6.src.rpm", 214490, "LGPLv2+", "CentOS", "", "Libraries for decoding LZMA compression", "2afa255c6edf06556b6aea23d72a1874"},
			{intRef(), "gdbm-devel", "1.8.0", "39.el6", "x86_64", "gdbm-1.8.0-39.el6.src.rpm", 22251, "GPLv2+", "CentOS", "", "Development libraries and header files for the gdbm library", "2c561cc36f6914309af7e4b0639d8efb"},
			{intRef(), "libgpg-error", "1.7", "4.el6", "x86_64", "libgpg-error-1.7-4.el6.src.rpm", 214321, "LGPLv2+", "CentOS", "", "Library for error values used by GnuPG components", "a64a2b188d9fb76dc940e809e791aedc"},
			{intRef(1), "perl-ExtUtils-ParseXS", "2.2003.0", "144.el6", "x86_64", "perl-5.10.1-144.el6.src.rpm", 62208, "GPL+ or Artistic", "CentOS", "", "Module and a script for converting Perl XS code into C code", "528f67be17b9d33807587fb1dfc4094a"},
			{intRef(), "pcre", "7.8", "7.el6", "x86_64", "pcre-7.8-7.el6.src.rpm", 529027, "BSD", "CentOS", "", "Perl-compatible regular expression library", "60c2d316d863cdcb610cca815e29d894"},
			{intRef(0), "perl-ExtUtils-MakeMaker", "6.55", "144.el6", "x86_64", "perl-5.10.1-144.el6.src.rpm", 622462, "GPL+ or Artistic", "CentOS", "", "Create a module Makefile", "febcb37fb232855986ef8f20e3de685a"},
			{intRef(1), "findutils", "4.4.2", "9.el6", "x86_64", "findutils-4.4.2-9.el6.src.rpm", 1442912, "GPLv3+", "CentOS", "", "The GNU versions of find utilities (find and xargs)", "b0ffd83c8576d956eb0af7aaf52bf832"},
		}

		return toPackageInfo(pkgs)
	}

	// docker run --rm -it centos:6 bash
	// rpm -qa --queryformat "\{%{EPOCH}, \"%{NAME}\", \"%{VERSION}\", \"%{RELEASE}\", \"%{ARCH}\", \"%{SOURCERPM}\", %{SIZE}, \"%{LICENSE}\", \"%{VENDOR}\", \"\", \"%{SUMMARY}\", \"%{SIGMD5}\"\},\n" | sed "s/^{(none)/{intRef()/g" | sed -r 's/^\{([0-9]+),/{intRef(\1),/' | sed "s/(none)/0/g"
	CentOS6Plain = func() []*PackageInfo {
		pkgs := []*commonPackageInfo{
			{intRef(), "setup", "2.8.14", "23.el6", "noarch", "setup-2.8.14-23.el6.src.rpm", 666669, "Public Domain", "CentOS", "", "A set of system configuration and setup files", "f58d4ff770a377d42854025b5c64a709"},
			{intRef(), "basesystem", "10.0", "4.el6", "noarch", "basesystem-10.0-4.el6.src.rpm", 0, "Public Domain", "CentOS", "", "The skeleton package which defines a simple Red Hat Enterprise Linux system", "bf36e0d2771604b8ad56d9696b5edd3c"},
			{intRef(), "tzdata", "2018e", "3.el6", "noarch", "tzdata-2018e-3.el6.src.rpm", 1961609, "Public Domain", "CentOS", "", "Timezone data", "95de917c04de115372e5b75c3874db97"},
			{intRef(), "glibc-common", "2.12", "1.212.el6", "x86_64", "glibc-2.12-1.212.el6.src.rpm", 112436133, "LGPLv2+ and LGPLv2+ with exceptions and GPLv2+", "CentOS", "", "Common binaries and locale data for glibc", "0a5a0dcfa0322659ef60d49681300c77"},
			{intRef(), "glibc", "2.12", "1.212.el6", "x86_64", "glibc-2.12-1.212.el6.src.rpm", 13117447, "LGPLv2+ and LGPLv2+ with exceptions and GPLv2+", "CentOS", "", "The GNU libc libraries", "89e843d7979a50a26e2ea1924ef3e213"},
			{intRef(), "bash", "4.1.2", "48.el6", "x86_64", "bash-4.1.2-48.el6.src.rpm", 3142529, "GPLv3+", "CentOS", "", "The GNU Bourne Again shell", "fb70e495a46a63523eab6138cae2438d"},
			{intRef(), "libcap", "2.16", "5.5.el6", "x86_64", "libcap-2.16-5.5.el6.src.rpm", 64437, "LGPLv2+ or BSD", "CentOS", "", "Library for getting and setting POSIX.1e capabilities", "5d224cdbaa15053a3aa5443a07a45c2b"},
			{intRef(), "info", "4.13a", "8.el6", "x86_64", "texinfo-4.13a-8.el6.src.rpm", 329482, "GPLv3+", "CentOS", "", "A stand-alone TTY-based reader for GNU texinfo documentation", "482d0f1ec0690ed672c0c1b594f78583"},
			{intRef(), "libacl", "2.2.49", "7.el6_9.1", "x86_64", "acl-2.2.49-7.el6_9.1.src.rpm", 31280, "LGPLv2+", "CentOS", "", "Dynamic library for access control list support", "98166984ca2ff5aabfd1a43fdc89c788"},
			{intRef(), "nspr", "4.19.0", "1.el6", "x86_64", "nspr-4.19.0-1.el6.src.rpm", 281968, "MPLv2.0", "CentOS", "", "Netscape Portable Runtime", "18e1b8a44c636953cf82358dc0978d34"},
			{intRef(), "libcom_err", "1.41.12", "24.el6", "x86_64", "e2fsprogs-1.41.12-24.el6.src.rpm", 59233, "MIT", "CentOS", "", "Common error description library", "3d4d863ae951dde28e33d4fec21c793d"},
			{intRef(), "libsepol", "2.0.41", "4.el6", "x86_64", "libsepol-2.0.41-4.el6.src.rpm", 248680, "LGPLv2+", "CentOS", "", "SELinux binary policy manipulation library", "cbc386b646b94d710dfabe7126ede725"},
			{intRef(), "chkconfig", "1.3.49.5", "1.el6", "x86_64", "chkconfig-1.3.49.5-1.el6.src.rpm", 670580, "GPLv2", "CentOS", "", "A system tool for maintaining the /etc/rc*.d hierarchy", "c5e95feb1e3c73bfd0761981c866a024"},
			{intRef(), "audit-libs", "2.4.5", "6.el6", "x86_64", "audit-2.4.5-6.el6.src.rpm", 235282, "LGPLv2+", "CentOS", "", "Dynamic library for libaudit", "fec1ad9c60d9d6e870ea6182ba6be06a"},
			{intRef(), "readline", "6.0", "4.el6", "x86_64", "readline-6.0-4.el6.src.rpm", 433957, "GPLv3+", "CentOS", "", "A library for editing typed command lines", "d8b0aa8a08f36b43eb4858d76d005c6d"},
			{intRef(), "file-libs", "5.04", "30.el6", "x86_64", "file-5.04-30.el6.src.rpm", 2486624, "BSD", "CentOS", "", "Libraries for applications using libmagic", "e27f1d59098a685689f3f2d099946a85"},
			{intRef(1), "dbus-libs", "1.2.24", "9.el6", "x86_64", "dbus-1.2.24-9.el6.src.rpm", 265736, "GPLv2+ or AFL", "CentOS", "", "Libraries for accessing D-BUS", "3b7d65739639c300ab7a69e890240f32"},
			{intRef(), "sqlite", "3.6.20", "1.el6_7.2", "x86_64", "sqlite-3.6.20-1.el6_7.2.src.rpm", 640060, "Public Domain", "CentOS", "", "Library that implements an embeddable SQL database engine", "d94a1809a80676af38140d2f0021c5f9"},
			{intRef(), "libuuid", "2.17.2", "12.28.el6_9.2", "x86_64", "util-linux-ng-2.17.2-12.28.el6_9.2.src.rpm", 16304, "BSD", "CentOS", "", "Universally unique ID library", "9e1615002308c631aaee83e16668543f"},
			{intRef(), "xz-libs", "4.999.9", "0.5.beta.20091007git.el6", "x86_64", "xz-4.999.9-0.5.beta.20091007git.el6.src.rpm", 214490, "LGPLv2+", "CentOS", "", "Libraries for decoding LZMA compression", "2afa255c6edf06556b6aea23d72a1874"},
			{intRef(), "libgpg-error", "1.7", "4.el6", "x86_64", "libgpg-error-1.7-4.el6.src.rpm", 214321, "LGPLv2+", "CentOS", "", "Library for error values used by GnuPG components", "a64a2b188d9fb76dc940e809e791aedc"},
			{intRef(), "pcre", "7.8", "7.el6", "x86_64", "pcre-7.8-7.el6.src.rpm", 529027, "BSD", "CentOS", "", "Perl-compatible regular expression library", "60c2d316d863cdcb610cca815e29d894"},
			{intRef(1), "findutils", "4.4.2", "9.el6", "x86_64", "findutils-4.4.2-9.el6.src.rpm", 1442912, "GPLv3+", "CentOS", "", "The GNU versions of find utilities (find and xargs)", "b0ffd83c8576d956eb0af7aaf52bf832"},
			{intRef(), "expat", "2.0.1", "13.el6_8", "x86_64", "expat-2.0.1-13.el6_8.src.rpm", 197794, "MIT", "CentOS", "", "An XML parser library", "7c8e581750011dc5ff891ef8dbb76263"},
			{intRef(), "p11-kit", "0.18.5", "2.el6_5.2", "x86_64", "p11-kit-0.18.5-2.el6_5.2.src.rpm", 262669, "BSD", "CentOS", "", "Library for loading and sharing PKCS#11 modules", "4b6eb29295513687b0dd248091a45be3"},
			{intRef(), "libgcrypt", "1.4.5", "12.el6_8", "x86_64", "libgcrypt-1.4.5-12.el6_8.src.rpm", 536622, "LGPLv2+", "CentOS", "", "A general-purpose cryptography library", "b16c065b3d5f5c351ee3f64c67fb77ce"},
			{intRef(), "libusb", "0.1.12", "23.el6", "x86_64", "libusb-0.1.12-23.el6.src.rpm", 54440, "LGPLv2+", "CentOS", "", "A library which allows userspace access to USB devices", "d1b2530f9e990b4dff32301c7849d417"},
			{intRef(), "pinentry", "0.7.6", "8.el6", "x86_64", "pinentry-0.7.6-8.el6.src.rpm", 147023, "GPLv2+", "CentOS", "", "Collection of simple PIN or passphrase entry dialogs", "6cd8f48fc9404b8c6ce0a0995ec0a3b6"},
			{intRef(), "procps", "3.2.8", "45.el6_9.3", "x86_64", "procps-3.2.8-45.el6_9.3.src.rpm", 475526, "GPLv2+ and LGPLv2+", "CentOS", "", "System and process monitoring utilities", "f21e14ee708e7ff106533791199873cd"},
			{intRef(2), "tar", "1.23", "15.el6_8", "x86_64", "tar-1.23-15.el6_8.src.rpm", 2616465, "GPLv3+", "CentOS", "", "A GNU file archiving program", "369d4ecc0c520fbe7368613a19558afe"},
			{intRef(), "checkpolicy", "2.0.22", "1.el6", "x86_64", "checkpolicy-2.0.22-1.el6.src.rpm", 870239, "GPLv2", "CentOS", "", "SELinux policy compiler", "02392813ec341e8c77a29609f884d937"},
			{intRef(), "db4-utils", "4.7.25", "22.el6", "x86_64", "db4-4.7.25-22.el6.src.rpm", 416398, "Sleepycat and BSD", "CentOS", "", "Command line tools for managing Berkeley DB (version 4) databases", "4ab1657d6f8d178f926a168ad1f40e8c"},
			{intRef(), "binutils", "2.20.51.0.2", "5.48.el6", "x86_64", "binutils-2.20.51.0.2-5.48.el6.src.rpm", 9831289, "GPLv3+", "CentOS", "", "A GNU collection of binary utilities", "b53b7da97a0716997465f21147815cd5"},
			{intRef(), "which", "2.19", "6.el6", "x86_64", "which-2.19-6.el6.src.rpm", 73004, "GPLv3", "CentOS", "", "Displays where a particular program in your path is located", "0a194d85776ddf95732977c15dde7313"},
			{intRef(), "dash", "0.5.5.1", "4.el6", "x86_64", "dash-0.5.5.1-4.el6.src.rpm", 127277, "BSD", "CentOS", "", "Small and fast POSIX-compliant shell", "34745b4864bf8e42f1181162e3d52d55"},
			{intRef(), "groff", "1.18.1.4", "21.el6", "x86_64", "groff-1.18.1.4-21.el6.src.rpm", 5318766, "GPLv2 and GFDL", "CentOS", "", "A document formatting system", "e4ca537dd1668ea7a00a3476d0bde544"},
			{intRef(), "coreutils-libs", "8.4", "47.el6", "x86_64", "coreutils-8.4-47.el6.src.rpm", 5576, "GPLv3+", "CentOS", "", "Libraries for coreutils", "7ad9b0d44df0e19417e58c922ce145f3"},
			{intRef(), "cracklib", "2.8.16", "4.el6", "x86_64", "cracklib-2.8.16-4.el6.src.rpm", 187265, "LGPLv2+", "CentOS", "", "A password-checking library", "66ba40d1f80041344e9e7f3c2cce0b3c"},
			{intRef(), "coreutils", "8.4", "47.el6", "x86_64", "coreutils-8.4-47.el6.src.rpm", 12873065, "GPLv3+", "CentOS", "", "A set of basic GNU tools commonly used in shell scripts", "84092ccc9b644eb86dc0fb09754f80a1"},
			{intRef(), "module-init-tools", "3.9", "26.el6", "x86_64", "module-init-tools-3.9-26.el6.src.rpm", 1216834, "GPLv2+", "CentOS", "", "Kernel module management utilities.", "cb180e326e41dd50e83a78242742163d"},
			{intRef(), "nss", "3.36.0", "8.el6", "x86_64", "nss-3.36.0-8.el6.src.rpm", 2660886, "MPLv2.0", "CentOS", "", "Network Security Services", "b7517259bc48e009b2f53b9134813cac"},
			{intRef(), "nss-tools", "3.36.0", "8.el6", "x86_64", "nss-3.36.0-8.el6.src.rpm", 2052883, "MPLv2.0", "CentOS", "", "Tools for the Network Security Services", "38296a6aca265f81ed0568689a902153"},
			{intRef(), "mingetty", "1.08", "5.el6", "x86_64", "mingetty-1.08-5.el6.src.rpm", 34586, "GPLv2+", "CentOS", "", "A compact getty program for virtual consoles only", "db0d0ab6289fcee75a1aff64db47e0aa"},
			{intRef(), "krb5-libs", "1.10.3", "65.el6", "x86_64", "krb5-1.10.3-65.el6.src.rpm", 1813468, "MIT", "CentOS", "", "The non-admin shared libraries used by Kerberos 5", "fb39fd3d4667686d8c6a8919cce8518d"},
			{intRef(), "libssh2", "1.4.2", "2.el6_7.1", "x86_64", "libssh2-1.4.2-2.el6_7.1.src.rpm", 325165, "BSD", "CentOS", "", "A library implementing the SSH2 protocol", "75d2094e0838762108ccb666416dc3bf"},
			{intRef(), "rpm-libs", "4.8.0", "59.el6", "x86_64", "rpm-4.8.0-59.el6.src.rpm", 777656, "GPLv2+ and LGPLv2+ with exceptions", "CentOS", "", "Libraries for manipulating RPM packages", "5cc09f970aeaee43fa0ef20bb03d0103"},
			{intRef(), "rpm", "4.8.0", "59.el6", "x86_64", "rpm-4.8.0-59.el6.src.rpm", 2034245, "GPLv2+", "CentOS", "", "The RPM package management system", "be5c4756ff3091f6e8b8cb6864bc8ad2"},
			{intRef(), "gnupg2", "2.0.14", "9.el6_10", "x86_64", "gnupg2-2.0.14-9.el6_10.src.rpm", 6087540, "GPLv3+", "CentOS", "", "Utility for secure communication and data storage", "e3378f38c43677ba3b4c019d0a0be0b0"},
			{intRef(32), "bind-libs", "9.8.2", "0.68.rc1.el6_10.1", "x86_64", "bind-9.8.2-0.68.rc1.el6_10.1.src.rpm", 2340720, "ISC", "CentOS", "", "Libraries used by the BIND DNS packages", "591c20361f155316b65bfb19269682aa"},
			{intRef(), "libsemanage", "2.0.43", "5.1.el6", "x86_64", "libsemanage-2.0.43-5.1.el6.src.rpm", 204223, "LGPLv2+", "CentOS", "", "SELinux binary policy manipulation library", "ddb4929b9a3b18082dd75345fa368c1b"},
			{intRef(), "libutempter", "1.1.5", "4.1.el6", "x86_64", "libutempter-1.1.5-4.1.el6.src.rpm", 40785, "LGPLv2", "CentOS", "", "A privileged helper for utmp/wtmp updates", "9fab105dce8b3f2f7dd9f53d4161c776"},
			{intRef(), "plymouth-core-libs", "0.8.3", "29.el6.centos", "x86_64", "plymouth-0.8.3-29.el6.centos.src.rpm", 177760, "GPLv2+", "CentOS", "", "Plymouth libraries", "c32d0f492fe719154a45ac77da9190de"},
			{intRef(), "libffi", "3.0.5", "3.2.el6", "x86_64", "libffi-3.0.5-3.2.el6.src.rpm", 42881, "BSD", "CentOS", "", "A portable foreign function interface library", "180e75de1468bb9353680e8dabaebeaf"},
			{intRef(), "python-libs", "2.6.6", "66.el6_8", "x86_64", "python-2.6.6-66.el6_8.src.rpm", 22979382, "Python", "CentOS", "", "Runtime libraries for Python", "5af05686c9ebe664b766c9d109e51e41"},
			{intRef(), "pygpgme", "0.1", "18.20090824bzr68.el6", "x86_64", "pygpgme-0.1-18.20090824bzr68.el6.src.rpm", 251432, "LGPLv2+", "CentOS", "", "Python module for working with OpenPGP messages", "98ac89e05c1ff89accffa1803d5966e0"},
			{intRef(), "python-urlgrabber", "3.9.1", "11.el6", "noarch", "python-urlgrabber-3.9.1-11.el6.src.rpm", 323137, "LGPLv2+", "CentOS", "", "A high-level cross-protocol url-grabber", "55fc3e4b6602d49bf3eb55ae688579a7"},
			{intRef(1), "pkgconfig", "0.23", "9.1.el6", "x86_64", "pkgconfig-0.23-9.1.el6.src.rpm", 140091, "GPLv2+", "CentOS", "", "A tool for determining compilation options", "62aa0b6f2b06f987ba56e95d727a948f"},
			{intRef(), "glib2", "2.28.8", "10.el6", "x86_64", "glib2-2.28.8-10.el6.src.rpm", 8061476, "LGPLv2+", "CentOS", "", "A library of handy utility functions", "a2b6d96a84eb5c40d73789bc17c16201"},
			{intRef(), "yum-metadata-parser", "1.1.2", "16.el6", "x86_64", "yum-metadata-parser-1.1.2-16.el6.src.rpm", 58327, "GPLv2", "CentOS", "", "A fast metadata parser for yum", "16a0976518f852b12ed3d1c0285e4bc6"},
			{intRef(), "yum", "3.2.29", "81.el6.centos", "noarch", "yum-3.2.29-81.el6.centos.src.rpm", 4832194, "GPLv2+", "CentOS", "", "RPM package installer/updater/manager", "dd629b7ea24a01ec5e380f78a0d71998"},
			{intRef(), "passwd", "0.77", "7.el6", "x86_64", "passwd-0.77-7.el6.src.rpm", 357699, "BSD or GPLv2+", "CentOS", "", "An utility for setting or changing passwords using PAM", "a2aa313df340ef28a8f4ec28a2f980f7"},
			{intRef(32), "bind-utils", "9.8.2", "0.68.rc1.el6_10.1", "x86_64", "bind-9.8.2-0.68.rc1.el6_10.1.src.rpm", 451000, "ISC", "CentOS", "", "Utilities for querying DNS name servers", "46a22a8044620a77a51e437adabca27d"},
			{intRef(2), "vim-minimal", "7.4.629", "5.el6_8.1", "x86_64", "vim-7.4.629-5.el6_8.1.src.rpm", 909230, "Vim", "CentOS", "", "A minimal version of the VIM editor", "3a8cb69c28f19fc0d374e68814b24d35"},
			{intRef(), "libgcc", "4.4.7", "23.el6", "x86_64", "gcc-4.4.7-23.el6.src.rpm", 117352, "GPLv3+ and GPLv3+ with exceptions and GPLv2+ with exceptions", "CentOS", "", "GCC version 4.4 shared support library", "e7b74ed43c9baf242de1cfbf80f28cbf"},
			{intRef(), "filesystem", "2.4.30", "3.el6", "x86_64", "filesystem-2.4.30-3.el6.src.rpm", 0, "Public Domain", "CentOS", "", "The basic directory layout for a Linux system", "8328ee9940f142cae12b0ba18a166a7d"},
			{intRef(), "ncurses-base", "5.7", "4.20090207.el6", "x86_64", "ncurses-5.7-4.20090207.el6.src.rpm", 193090, "MIT", "CentOS", "", "Descriptions of common terminals", "26e758f2dfba21d280faece6ec9011b1"},
			{intRef(), "nss-softokn-freebl", "3.14.3", "23.3.el6_8", "x86_64", "nss-softokn-3.14.3-23.3.el6_8.src.rpm", 490290, "MPLv2.0", "CentOS", "", "Freebl library for the Network Security Services", "2c45c946199a527955ac597c2e0af02c"},
			{intRef(), "ncurses-libs", "5.7", "4.20090207.el6", "x86_64", "ncurses-5.7-4.20090207.el6.src.rpm", 752304, "MIT", "CentOS", "", "Ncurses libraries", "55dd8fcb0fddaa3718c4a9aca1c51470"},
			{intRef(), "libattr", "2.4.44", "7.el6", "x86_64", "attr-2.4.44-7.el6.src.rpm", 18712, "LGPLv2+", "CentOS", "", "Dynamic library for extended attribute support", "c1e774f9e24e226b50a855f49c377de0"},
			{intRef(), "zlib", "1.2.3", "29.el6", "x86_64", "zlib-1.2.3-29.el6.src.rpm", 152305, "zlib and Boost", "CentOS", "", "The zlib compression and decompression library", "17358a2e9d24db6825adfc7a21823891"},
			{intRef(), "popt", "1.13", "7.el6", "x86_64", "popt-1.13-7.el6.src.rpm", 83420, "MIT", "CentOS", "", "C library for parsing command line parameters", "cdf20355eaf915d993d7584e2c4ee538"},
			{intRef(), "db4", "4.7.25", "22.el6", "x86_64", "db4-4.7.25-22.el6.src.rpm", 1533095, "Sleepycat and BSD", "CentOS", "", "The Berkeley DB database library (version 4) for C", "5bc080a2706659ba5d200654b537306b"},
			{intRef(), "nss-util", "3.36.0", "1.el6", "x86_64", "nss-util-3.36.0-1.el6.src.rpm", 191928, "MPLv2.0", "CentOS", "", "Network Security Services Utilities Library", "2a9a6cd34706c4ed8560fb50a8bd88d3"},
			{intRef(), "bzip2-libs", "1.0.5", "7.el6_0", "x86_64", "bzip2-1.0.5-7.el6_0.src.rpm", 67592, "BSD", "CentOS", "", "Libraries for applications using bzip2", "45e62d2d207887437a081283cd979a70"},
			{intRef(), "libselinux", "2.0.94", "7.el6", "x86_64", "libselinux-2.0.94-7.el6.src.rpm", 130336, "Public Domain", "CentOS", "", "SELinux library and simple utilities", "481af7162f2cb5714052ae7cf6e7e1ca"},
			{intRef(), "sed", "4.2.1", "10.el6", "x86_64", "sed-4.2.1-10.el6.src.rpm", 542324, "GPLv2+", "CentOS", "", "A GNU stream text editor", "48bcbdc8eab711098683a23048799093"},
			{intRef(), "libidn", "1.18", "2.el6", "x86_64", "libidn-1.18-2.el6.src.rpm", 567612, "LGPLv2+ and GPLv3+ and GFDL", "CentOS", "", "Internationalized Domain Name support library", "fd12d1a9b1bbeb0af06789ca1a91e227"},
			{intRef(), "libxml2", "2.7.6", "21.el6_8.1", "x86_64", "libxml2-2.7.6-21.el6_8.1.src.rpm", 1779163, "MIT", "CentOS", "", "Library providing XML and HTML support", "ddd19ff93764586cd73d56a28452d2e8"},
			{intRef(), "libstdc++", "4.4.7", "23.el6", "x86_64", "gcc-4.4.7-23.el6.src.rpm", 987096, "GPLv3+ and GPLv3+ with exceptions and GPLv2+ with exceptions", "CentOS", "", "GNU Standard C++ Library", "100188ad74a764ef33ca7b9ed73e2507"},
			{intRef(), "lua", "5.1.4", "4.1.el6", "x86_64", "lua-5.1.4-4.1.el6.src.rpm", 617799, "MIT", "CentOS", "", "Powerful light-weight programming language", "d73fd1effd24a1aeb20fd7b8be255a7d"},
			{intRef(), "gawk", "3.1.7", "10.el6_7.3", "x86_64", "gawk-3.1.7-10.el6_7.3.src.rpm", 2038650, "GPLv3+", "CentOS", "", "The GNU version of the awk text processing utility", "e057b2c0ac4253b726d364c3ce15789c"},
			{intRef(), "libblkid", "2.17.2", "12.28.el6_9.2", "x86_64", "util-linux-ng-2.17.2-12.28.el6_9.2.src.rpm", 136136, "LGPLv2+", "CentOS", "", "Block device ID library", "5bdee9aed72c9be1bf51b6c43d1e35a7"},
			{intRef(), "elfutils-libelf", "0.164", "2.el6", "x86_64", "elfutils-0.164-2.el6.src.rpm", 1011166, "GPLv2+ or LGPLv3+", "CentOS", "", "Library to read and write ELF files", "0b6033f7dac090e7decd459d61714a36"},
			{intRef(), "nss-softokn", "3.14.3", "23.3.el6_8", "x86_64", "nss-softokn-3.14.3-23.3.el6_8.src.rpm", 1103062, "MPLv2.0", "CentOS", "", "Network Security Services Softoken Module", "2acb6fdbf150b397270c2ab0a966b8f0"},
			{intRef(), "grep", "2.20", "6.el6", "x86_64", "grep-2.20-6.el6.src.rpm", 1197904, "GPLv3+", "CentOS", "", "Pattern matching utilities", "54b483d704c87fc97d0433b2ac89ca39"},
			{intRef(), "cpio", "2.10", "13.el6", "x86_64", "cpio-2.10-13.el6.src.rpm", 650433, "GPLv3+", "CentOS", "", "A GNU archiving program", "09fb2b399edb36bc6d8f69f183c90756"},
			{intRef(), "pth", "2.0.7", "9.3.el6", "x86_64", "pth-2.0.7-9.3.el6.src.rpm", 261931, "LGPLv2+", "CentOS", "", "The GNU Portable Threads library", "26e0ade2c2edb655da0922ce0b2ab0a3"},
			{intRef(), "libtasn1", "2.3", "6.el6_5", "x86_64", "libtasn1-2.3-6.el6_5.src.rpm", 443140, "GPLv3+ and LGPLv2+", "CentOS", "", "The ASN.1 library used in GNUTLS", "7dc9e15b02609ef338e49114556f3868"},
			{intRef(), "p11-kit-trust", "0.18.5", "2.el6_5.2", "x86_64", "p11-kit-0.18.5-2.el6_5.2.src.rpm", 178775, "BSD", "CentOS", "", "System trust module from p11-kit", "af4f4b1702c1edb385767deb382df6fd"},
			{intRef(), "libnih", "1.0.1", "8.el6", "x86_64", "libnih-1.0.1-8.el6.src.rpm", 490926, "GPLv2", "CentOS", "", "Lightweight application development library", "b69e9826410913a30efeb2849482ace1"},
			{intRef(), "gmp", "4.3.1", "13.el6", "x86_64", "gmp-4.3.1-13.el6.src.rpm", 657883, "LGPLv2+ and  GPLv3+ and LGPLv3+", "CentOS", "", "A GNU arbitrary precision library", "da0cc35676b00a0f5b216c3ca6187980"},
			{intRef(), "file", "5.04", "30.el6", "x86_64", "file-5.04-30.el6.src.rpm", 56567, "BSD", "CentOS", "", "A utility for determining file types", "5e5e8e881f53ba6a03c8978c037be2ac"},
			{intRef(), "net-tools", "1.60", "114.el6", "x86_64", "net-tools-1.60-114.el6.src.rpm", 778260, "GPL+", "CentOS", "", "Basic networking tools", "b55d83e6309f8af722284dfaef3d08f5"},
			{intRef(), "psmisc", "22.6", "24.el6", "x86_64", "psmisc-22.6-24.el6.src.rpm", 222569, "GPLv2+", "CentOS", "", "Utilities for managing processes on your system", "e54c765f163883606eaba07d0230d3cf"},
			{intRef(), "libselinux-utils", "2.0.94", "7.el6", "x86_64", "libselinux-2.0.94-7.el6.src.rpm", 62739, "Public Domain", "CentOS", "", "SELinux libselinux utilies", "fd6929bd238cf955852465941a9b757d"},
			{intRef(), "bzip2", "1.0.5", "7.el6_0", "x86_64", "bzip2-1.0.5-7.el6_0.src.rpm", 79087, "BSD", "CentOS", "", "A file compression utility", "0cc4395aa298488f73ba4b5836d25668"},
			{intRef(), "cyrus-sasl-lib", "2.1.23", "15.el6_6.2", "x86_64", "cyrus-sasl-2.1.23-15.el6_6.2.src.rpm", 357710, "BSD", "CentOS", "", "Shared libraries needed by applications which use Cyrus SASL", "6199bc3ccc2b414fb57b66bdeccd71d1"},
			{intRef(1), "make", "3.81", "23.el6", "x86_64", "make-3.81-23.el6.src.rpm", 1079569, "GPLv2+", "CentOS", "", "A GNU tool which simplifies the build process for users", "e2f8c3031a4384208a7b7e8b0f6f0f11"},
			{intRef(), "diffutils", "2.8.1", "28.el6", "x86_64", "diffutils-2.8.1-28.el6.src.rpm", 588813, "GPLv2+", "CentOS", "", "A GNU collection of diff utilities", "8ead45c0c09898f39170ea97c521581b"},
			{intRef(), "ncurses", "5.7", "4.20090207.el6", "x86_64", "ncurses-5.7-4.20090207.el6.src.rpm", 386371, "MIT", "CentOS", "", "Ncurses support utilities", "e2e764dda00e1b7bafc865e9a274bacf"},
			{intRef(), "less", "436", "13.el6", "x86_64", "less-436-13.el6.src.rpm", 201909, "GPLv3+", "CentOS", "", "A text file browser similar to more, but better", "1168dd2315ee9f080c3dea0bb45fc786"},
			{intRef(), "gzip", "1.3.12", "24.el6", "x86_64", "gzip-1.3.12-24.el6.src.rpm", 225924, "GPLv2+ and GFDL", "CentOS", "", "The GNU data compression program", "75fbcc40c3e1e745402ca99687f99a14"},
			{intRef(), "cracklib-dicts", "2.8.16", "4.el6", "x86_64", "cracklib-2.8.16-4.el6.src.rpm", 9327207, "LGPLv2+", "CentOS", "", "The standard CrackLib dictionaries", "cadecbc8984f7bc55493c47e0d2c27fe"},
			{intRef(), "pam", "1.1.1", "24.el6", "x86_64", "pam-1.1.1-24.el6.src.rpm", 2419646, "BSD and GPLv2+", "CentOS", "", "An extensible library which provides authentication for applications", "9c5ae01c61f415d34fe4b192d5b5add3"},
			{intRef(), "plymouth-scripts", "0.8.3", "29.el6.centos", "x86_64", "plymouth-0.8.3-29.el6.centos.src.rpm", 11006, "GPLv2+", "CentOS", "", "Plymouth related scripts", "15e520370abb095e2fd36b86d863d737"},
			{intRef(), "ca-certificates", "2018.2.22", "65.1.el6", "noarch", "ca-certificates-2018.2.22-65.1.el6.src.rpm", 2675229, "Public Domain", "CentOS", "", "The Mozilla CA root certificate bundle", "33aa2022c5bfe5613215e68dc02941af"},
			{intRef(), "nss-sysinit", "3.36.0", "8.el6", "x86_64", "nss-3.36.0-8.el6.src.rpm", 32822, "MPLv2.0", "CentOS", "", "System NSS Initialization", "6188b4840cfa3f575e762328501f721f"},
			{intRef(2), "ethtool", "3.5", "6.el6", "x86_64", "ethtool-3.5-6.el6.src.rpm", 283910, "GPLv2", "CentOS", "", "Ethernet settings tool for PCI ethernet cards", "9649daaebc2c5dadae7da90cf4c0613c"},
			{intRef(), "keyutils-libs", "1.4", "5.el6", "x86_64", "keyutils-1.4-5.el6.src.rpm", 36624, "GPLv2+ and LGPLv2+", "CentOS", "", "Key utilities library", "e55a68fb4ada530ec6f52d34e1dee538"},
			{intRef(), "openssl", "1.0.1e", "57.el6", "x86_64", "openssl-1.0.1e-57.el6.src.rpm", 4248338, "OpenSSL", "CentOS", "", "A general purpose cryptography library with TLS implementation", "43a12b9b1a7050d62b52099552ec7963"},
			{intRef(), "libcurl", "7.19.7", "53.el6_9", "x86_64", "curl-7.19.7-53.el6_9.src.rpm", 347536, "MIT", "CentOS", "", "A library for getting files from web servers", "587463a0294856ae3177d97b70f8136c"},
			{intRef(), "curl", "7.19.7", "53.el6_9", "x86_64", "curl-7.19.7-53.el6_9.src.rpm", 357074, "MIT", "CentOS", "", "A utility for getting files from remote servers (FTP, HTTP, and others)", "8ab5f2926ce20347a872179bcb89500b"},
			{intRef(), "openldap", "2.4.40", "16.el6", "x86_64", "openldap-2.4.40-16.el6.src.rpm", 859518, "OpenLDAP", "CentOS", "", "LDAP support libraries", "5fb3cd202d6354c28ebdf420c8eb5916"},
			{intRef(), "gpgme", "1.1.8", "3.el6", "x86_64", "gpgme-1.1.8-3.el6.src.rpm", 729658, "LGPLv2+", "CentOS", "", "GnuPG Made Easy - high level crypto API", "f20b9ddce4a0b8952a50a826c2b5fcfb"},
			{intRef(), "ustr", "1.0.4", "9.1.el6", "x86_64", "ustr-1.0.4-9.1.el6.src.rpm", 273983, "MIT or LGPLv2+ or BSD", "CentOS", "", "String library, very low memory overhead, simple to import", "16060496fca7666123134438f27c9e91"},
			{intRef(2), "shadow-utils", "4.1.5.1", "5.el6", "x86_64", "shadow-utils-4.1.5.1-5.el6.src.rpm", 3466724, "BSD and GPLv2+", "CentOS", "", "Utilities for managing accounts and shadow password files", "3c869910a1ae19bdd1392b4ac1cc4c40"},
			{intRef(), "MAKEDEV", "3.24", "6.el6", "x86_64", "MAKEDEV-3.24-6.el6.src.rpm", 227290, "GPLv2", "CentOS", "", "A program used for creating device files in /dev", "1a999a3ef7117fbf20cf67e726429a60"},
			{intRef(), "gdbm", "1.8.0", "39.el6", "x86_64", "gdbm-1.8.0-39.el6.src.rpm", 48770, "GPLv2+", "CentOS", "", "A GNU set of database routines which use extensible hashing", "f32d22ad07490183bc56bceeb30baa8e"},
			{intRef(), "python", "2.6.6", "66.el6_8", "x86_64", "python-2.6.6-66.el6_8.src.rpm", 79603, "Python", "CentOS", "", "An interpreted, interactive, object-oriented programming language", "649646824189190c3153fcdcc7cf686d"},
			{intRef(), "rpm-python", "4.8.0", "59.el6", "x86_64", "rpm-4.8.0-59.el6.src.rpm", 120906, "GPLv2+", "CentOS", "", "Python bindings for apps which will manipulate RPM packages", "907e2545aaf74bfff58a005542c8daff"},
			{intRef(), "python-pycurl", "7.19.0", "9.el6", "x86_64", "python-pycurl-7.19.0-9.el6.src.rpm", 236939, "LGPLv2+ or MIT", "CentOS", "", "A Python interface to libcurl", "cd1c2c372f3036b9594b942bc57dcffb"},
			{intRef(), "python-iniparse", "0.3.1", "2.1.el6", "noarch", "python-iniparse-0.3.1-2.1.el6.src.rpm", 109284, "MIT", "CentOS", "", "Python Module for Accessing and Modifying Configuration Data in INI files", "1f3ba002bade99b8ba653d1a4ef63337"},
			{intRef(), "gamin", "0.1.10", "9.el6", "x86_64", "gamin-0.1.10-9.el6.src.rpm", 416440, "LGPLv2", "CentOS", "", "Library providing the FAM File Alteration Monitor API", "5156618d3b648e76d643495671fc0abd"},
			{intRef(), "shared-mime-info", "0.70", "6.el6", "x86_64", "shared-mime-info-0.70-6.el6.src.rpm", 1411396, "GPLv2+", "CentOS", "", "Shared MIME information database", "5e6203e5374e767bbb3eecaf04e06836"},
			{intRef(), "libuser", "0.56.13", "8.el6_7", "x86_64", "libuser-0.56.13-8.el6_7.src.rpm", 1882055, "LGPLv2+", "CentOS", "", "A user and group account administration library", "ce682d078b0b7d99df962a8bb7d0c719"},
			{intRef(), "yum-plugin-fastestmirror", "1.1.30", "42.el6_10", "noarch", "yum-utils-1.1.30-42.el6_10.src.rpm", 53961, "GPLv2+", "CentOS", "", "Yum plugin which chooses fastest repository from a mirrorlist", "8d6fd5bad8fe4e6a932cf7aa624ab360"},
			{intRef(), "centos-release", "6", "10.el6.centos.12.3", "x86_64", "centos-release-6-10.el6.centos.12.3.src.rpm", 38232, "GPLv2", "CentOS", "", "CentOS release file", "0a78cbc4776e9fce236b09814009025f"},
			{intRef(), "yum-plugin-ovl", "1.1.30", "42.el6_10", "noarch", "yum-utils-1.1.30-42.el6_10.src.rpm", 22350, "GPLv2+", "CentOS", "", "Yum plugin to work around overlayfs issues", "e6f25f528cb1d4e7af487da25bf648ee"},
			{intRef(), "rootfiles", "8.1", "6.1.el6", "noarch", "rootfiles-8.1-6.1.el6.src.rpm", 599, "Public Domain", "CentOS", "", "The basic required files for the root user's directory", "6a2b3154d2e600c8c90d15b2a61fbd76"},
		}

		return toPackageInfo(pkgs)
	}

	// docker run --rm -it centos:7 bash
	// yum groupinstall -y "Development tools"
	// rpm -qa --queryformat "\{%{EPOCH}, \"%{NAME}\", \"%{VERSION}\", \"%{RELEASE}\", \"%{ARCH}\", \"%{SOURCERPM}\", %{SIZE}, \"%{LICENSE}\", \"%{VENDOR}\", \"\", \"%{SUMMARY}\", \"%{SIGMD5}\"\},\n" | sed "s/^{(none)/{intRef()/g" | sed -r 's/^\{([0-9]+),/{intRef(\1),/' | sed "s/(none)/0/g"
	CentOS7DevTools = func() []*PackageInfo {
		pkgs := []*commonPackageInfo{
			{intRef(), "libproxy", "0.4.11", "11.el7", "x86_64", "libproxy-0.4.11-11.el7.src.rpm", 164014, "LGPLv2+", "CentOS", "", "A library handling all the details of proxy configuration", "5238ea30a29e5163ee6010d2696acb1d"},
			{intRef(), "tzdata", "2018e", "3.el7", "noarch", "tzdata-2018e-3.el7.src.rpm", 1966505, "Public Domain", "CentOS", "", "Timezone data", "a1e5c65f5b87e33b23419b153155db0a"},
			{intRef(), "gnutls", "3.3.29", "9.el7_6", "x86_64", "gnutls-3.3.29-9.el7_6.src.rpm", 2097819, "GPLv3+ and LGPLv2+", "CentOS", "", "A TLS protocol implementation", "f82fa52300da8811286430201915afce"},
			{intRef(), "nss-softokn-freebl", "3.36.0", "5.el7_5", "x86_64", "nss-softokn-3.36.0-5.el7_5.src.rpm", 565628, "MPLv2.0", "CentOS", "", "Freebl library for the Network Security Services", "cf0360bbf0396334bb569646bbec25d3"},
			{intRef(1), "make", "3.82", "23.el7", "x86_64", "make-3.82-23.el7.src.rpm", 1160684, "GPLv2+", "CentOS", "", "A GNU tool which simplifies the build process for users", "9e5ab17428b36e59a9d071c1fd1c81de"},
			{intRef(), "ncurses", "5.9", "14.20130511.el7_4", "x86_64", "ncurses-5.9-14.20130511.el7_4.src.rpm", 439378, "MIT", "CentOS", "", "Ncurses support utilities", "8ca93e2831102818759a22e22e871268"},
			{intRef(), "openssh-clients", "7.4p1", "16.el7", "x86_64", "openssh-7.4p1-16.el7.src.rpm", 2651616, "BSD", "CentOS", "", "An open source SSH client applications", "4f9a34658e76fe533402c89ee43fec18"},
			{intRef(), "git", "1.8.3.1", "20.el7", "x86_64", "git-1.8.3.1-20.el7.src.rpm", 23232282, "GPLv2", "CentOS", "", "Fast Version Control System", "bad9f26b37b7b8a74a69a038b773def4"},
			{intRef(), "filesystem", "3.2", "25.el7", "x86_64", "filesystem-3.2-25.el7.src.rpm", 0, "Public Domain", "CentOS", "", "The basic directory layout for a Linux system", "93e1c91a2dbd4d3ac61fb70a40a7c228"},
			{intRef(), "neon", "0.30.0", "3.el7", "x86_64", "neon-0.30.0-3.el7.src.rpm", 567772, "LGPLv2+", "CentOS", "", "An HTTP and WebDAV client library", "0c9ebc3ab96eed7a01fa4a1f7099de88"},
			{intRef(), "bzip2", "1.0.6", "13.el7", "x86_64", "bzip2-1.0.6-13.el7.src.rpm", 83791, "BSD", "CentOS", "", "A file compression utility", "704f597825711270b0ba2af51bebecae"},
			{intRef(), "nspr", "4.19.0", "1.el7_5", "x86_64", "nspr-4.19.0-1.el7_5.src.rpm", 287728, "MPLv2.0", "CentOS", "", "Netscape Portable Runtime", "06e8af105f3a5832065a90c5eb12c064"},
			{intRef(), "mokutil", "15", "2.el7.centos", "x86_64", "shim-signed-15-2.el7.centos.src.rpm", 83521, "BSD", "CentOS", "", "Utilities for managing Secure Boot/MoK keys.", "0149c24490b240a4db17b50acafa6746"},
			{intRef(), "popt", "1.13", "16.el7", "x86_64", "popt-1.13-16.el7.src.rpm", 88516, "MIT", "CentOS", "", "C library for parsing command line parameters", "87f5d6bac8f205fb14e2072e4b47ce23"},
			{intRef(1), "emacs-filesystem", "24.3", "22.el7", "noarch", "emacs-24.3-22.el7.src.rpm", 0, "GPLv3+", "CentOS", "", "Emacs filesystem layout", "54c6f52516764089c58bfebe31187c2d"},
			{intRef(), "libffi", "3.0.13", "18.el7", "x86_64", "libffi-3.0.13-18.el7.src.rpm", 47766, "MIT and Public Domain", "CentOS", "", "A portable foreign function interface library", "f05f7a3749342b925fa10ca7de2f0116"},
			{intRef(), "glibc-headers", "2.17", "260.el7_6.4", "x86_64", "glibc-2.17-260.el7_6.4.src.rpm", 2338328, "LGPLv2+ and LGPLv2+ with exceptions and GPLv2+", "CentOS", "", "Header files for development using standard C libraries.", "10577e5ef1696cce687913ba612698fc"},
			{intRef(), "libcap", "2.22", "9.el7", "x86_64", "libcap-2.22-9.el7.src.rpm", 111445, "LGPLv2+", "CentOS", "", "Library for getting and setting POSIX.1e capabilities", "b4cc4b26a62dbbd66143755c54a59938"},
			{intRef(), "gcc", "4.8.5", "36.el7_6.1", "x86_64", "gcc-4.8.5-36.el7_6.1.src.rpm", 39238921, "GPLv3+ and GPLv3+ with exceptions and GPLv2+ with exceptions and LGPLv2+ and BSD", "CentOS", "", "Various compilers (C, C++, Objective-C, Java, ...)", "82a79193e96339bd9ee093bdb0c53c1a"},
			{intRef(), "libsepol", "2.5", "8.1.el7", "x86_64", "libsepol-2.5-8.1.el7.src.rpm", 686568, "LGPLv2+", "CentOS", "", "SELinux binary policy manipulation library", "b107891426af43d06b02a191f16a490c"},
			{intRef(), "systemtap-devel", "3.3", "3.el7", "x86_64", "systemtap-3.3-3.el7.src.rpm", 7910669, "GPLv2+", "CentOS", "", "Programmable system-wide instrumentation system - development headers, tools", "5ca8a6b02cb40f4df3fdba8ea25da170"},
			{intRef(), "ncurses-libs", "5.9", "14.20130511.el7_4", "x86_64", "ncurses-5.9-14.20130511.el7_4.src.rpm", 1028216, "MIT", "CentOS", "", "Ncurses libraries", "bba677eda5184b5127b7817d2c1dec12"},
			{intRef(), "redhat-rpm-config", "9.1.0", "87.el7.centos", "noarch", "redhat-rpm-config-9.1.0-87.el7.centos.src.rpm", 174572, "GPL+", "CentOS", "", "CentOS specific rpm configuration files", "fa59b48b92a6e9b34c9fad064eafb839"},
			{intRef(), "gawk", "4.0.2", "4.el7_3.1", "x86_64", "gawk-4.0.2-4.el7_3.1.src.rpm", 2435978, "GPLv3+ and GPL and LGPLv3+ and LGPL and BSD", "CentOS", "", "The GNU version of the awk text processing utility", "5efa8feaeb12d57cda1a9bcd9c8af917"},
			{intRef(), "gettext-devel", "0.19.8.1", "2.el7", "x86_64", "gettext-0.19.8.1-2.el7.src.rpm", 1492653, "LGPLv2+ and GPLv3+", "CentOS", "", "Development files for gettext", "b6ae16e073cfe4270869d15e37898234"},
			{intRef(), "libselinux", "2.5", "12.el7", "x86_64", "libselinux-2.5-12.el7.src.rpm", 217874, "Public Domain", "CentOS", "", "SELinux library and simple utilities", "7584c834f1404914f4568973c001f28e"},
			{intRef(), "rpm-build", "4.11.3", "35.el7", "x86_64", "rpm-4.11.3-35.el7.src.rpm", 328409, "GPLv2+", "CentOS", "", "Scripts and executable programs used to build packages", "d2e83d31fc34e6cfb1c8ff7948254716"},
			{intRef(), "grep", "2.20", "3.el7", "x86_64", "grep-2.20-3.el7.src.rpm", 1195131, "GPLv3+", "CentOS", "", "Pattern matching utilities", "92bc957d1c4fc3764e134d8f82544abe"},
			{intRef(), "gcc-gfortran", "4.8.5", "36.el7_6.1", "x86_64", "gcc-4.8.5-36.el7_6.1.src.rpm", 16443248, "GPLv3+ and GPLv3+ with exceptions and GPLv2+ with exceptions and LGPLv2+ and BSD", "CentOS", "", "Fortran support", "c470a080bcd80338a0cb02ea194e87c5"},
			{intRef(), "keyutils-libs", "1.5.8", "3.el7", "x86_64", "keyutils-1.5.8-3.el7.src.rpm", 42138, "GPLv2+ and LGPLv2+", "CentOS", "", "Key utilities library", "e31fa37a714787d3d72e0a9316c3a85b"},
			{intRef(), "gcc-c++", "4.8.5", "36.el7_6.1", "x86_64", "gcc-4.8.5-36.el7_6.1.src.rpm", 17257523, "GPLv3+ and GPLv3+ with exceptions and GPLv2+ with exceptions and LGPLv2+ and BSD", "CentOS", "", "C++ support for GCC", "8f929d2776e8a8b345b4f6d4066dfaa0"},
			{intRef(), "libverto", "0.2.5", "4.el7", "x86_64", "libverto-0.2.5-4.el7.src.rpm", 23060, "MIT", "CentOS", "", "Main loop abstraction library", "341bae036f22ce087486f4d9e180caee"},
			{intRef(), "subversion", "1.7.14", "14.el7", "x86_64", "subversion-1.7.14-14.el7.src.rpm", 4844082, "ASL 2.0", "CentOS", "", "A Modern Concurrent Version Control System", "a52a29990c540008b82065babc5d2212"},
			{intRef(), "p11-kit-trust", "0.23.5", "3.el7", "x86_64", "p11-kit-0.23.5-3.el7.src.rpm", 437261, "BSD", "CentOS", "", "System trust module from p11-kit", "2dc2f7ab212401098e8821ce23f21741"},
			{intRef(), "flex", "2.5.37", "6.el7", "x86_64", "flex-2.5.37-6.el7.src.rpm", 757468, "BSD and LGPLv2+", "CentOS", "", "A tool for creating scanners (text pattern recognizers)", "b18d609b317f948f26c59178fa9986cb"},
			{intRef(1), "openssl-libs", "1.0.2k", "12.el7", "x86_64", "openssl-1.0.2k-12.el7.src.rpm", 3200172, "OpenSSL", "CentOS", "", "A general purpose cryptography library with TLS implementation", "9213b8a36ef5936ac90e2630803e659c"},
			{intRef(), "rpm-python", "4.11.3", "35.el7", "x86_64", "rpm-4.11.3-35.el7.src.rpm", 149898, "GPLv2+", "CentOS", "", "Python bindings for apps which will manipulate RPM packages", "ad03ebb739fc7309edd09eba58ab7ee3"},
			{intRef(), "centos-release", "7", "5.1804.4.el7.centos", "x86_64", "centos-release-7-5.1804.4.el7.centos.src.rpm", 40338, "GPLv2", "CentOS", "", "CentOS Linux release file", "e73301588f9dce0441fb403b2fbc09fd"},
			{intRef(1), "doxygen", "1.8.5", "3.el7", "x86_64", "doxygen-1.8.5-3.el7.src.rpm", 15389684, "GPL+", "CentOS", "", "A documentation system for C/C++", "8338fd066b89d4aa9122a545a36ce50e"},
			{intRef(), "xz-libs", "5.2.2", "1.el7", "x86_64", "xz-5.2.2-1.el7.src.rpm", 239967, "LGPLv2+", "CentOS", "", "Libraries for decoding LZMA compression", "7701c542dc8fcd5a38ee52e8786d7cf4"},
			{intRef(), "diffstat", "1.57", "4.el7", "x86_64", "diffstat-1.57-4.el7.src.rpm", 54065, "MIT", "CentOS", "", "A utility which provides statistics based on the output of diff", "84313732304f6bf93f57590b69b8ee70"},
			{intRef(), "libdb", "5.3.21", "24.el7", "x86_64", "libdb-5.3.21-24.el7.src.rpm", 1858008, "BSD and LGPLv2 and Sleepycat", "CentOS", "", "The Berkeley DB database library for C", "24fdfda30ad8e1cc7ebc744eaf85eacf"},
			{intRef(), "byacc", "1.9.20130304", "3.el7", "x86_64", "byacc-1.9.20130304-3.el7.src.rpm", 135161, "Public Domain", "CentOS", "", "Berkeley Yacc, a parser generator", "da257ad856b5658321480e85cb7c60c2"},
			{intRef(), "libgpg-error", "1.12", "3.el7", "x86_64", "libgpg-error-1.12-3.el7.src.rpm", 350865, "LGPLv2+", "CentOS", "", "Library for error values used by GnuPG components", "a3933d4ce43c07fa9305e8765d78d099"},
			{intRef(), "indent", "2.2.11", "13.el7", "x86_64", "indent-2.2.11-13.el7.src.rpm", 359131, "GPLv3+", "CentOS", "", "A GNU program for formatting C code", "54926576db2ea4e79ed3f7dcbbf25a1b"},
			{intRef(), "libgcrypt", "1.5.3", "14.el7", "x86_64", "libgcrypt-1.5.3-14.el7.src.rpm", 597727, "LGPLv2+", "CentOS", "", "A general-purpose cryptography library", "88c64dc7c3aed6a156e4dee5def38300"},
			{intRef(), "lua", "5.1.4", "15.el7", "x86_64", "lua-5.1.4-15.el7.src.rpm", 640319, "MIT", "CentOS", "", "Powerful light-weight programming language", "d5973eb7ca2bb665f5fbb7141f2e27a3"},
			{intRef(), "libuuid", "2.23.2", "52.el7_5.1", "x86_64", "util-linux-2.23.2-52.el7_5.1.src.rpm", 20326, "BSD", "CentOS", "", "Universally unique ID library", "534c564308a92d9c695b64b7a359f36c"},
			{intRef(), "libmount", "2.23.2", "52.el7_5.1", "x86_64", "util-linux-2.23.2-52.el7_5.1.src.rpm", 273965, "LGPLv2+", "CentOS", "", "Device mounting library", "d084f24c36b7968ed5fc4a34940980ab"},
			{intRef(), "shared-mime-info", "1.8", "4.el7", "x86_64", "shared-mime-info-1.8-4.el7.src.rpm", 2379317, "GPLv2+", "CentOS", "", "Shared MIME information database", "167b39408a8c5705f135e968174e2229"},
			{intRef(), "gzip", "1.5", "10.el7", "x86_64", "gzip-1.5-10.el7.src.rpm", 250440, "GPLv3+ and GFDL", "CentOS", "", "The GNU data compression program", "adcb4df7dfe7866e38b5bf42d54d63ae"},
			{intRef(1), "findutils", "4.5.11", "5.el7", "x86_64", "findutils-4.5.11-5.el7.src.rpm", 1855626, "GPLv3+", "CentOS", "", "The GNU versions of find utilities (find and xargs)", "2d571b503447f5ce47a4767f6a7e1986"},
			{intRef(), "diffutils", "3.3", "4.el7", "x86_64", "diffutils-3.3-4.el7.src.rpm", 1065157, "GPLv3+", "CentOS", "", "A GNU collection of diff utilities", "d8c60275b4c6d54e8bf7ee19cf0c4f2b"},
			{intRef(), "expat", "2.1.0", "10.el7_3", "x86_64", "expat-2.1.0-10.el7_3.src.rpm", 208315, "MIT", "CentOS", "", "An XML parser library", "9c5fdceee3715ad8a1b3a9dd711d5b79"},
			{intRef(), "audit-libs", "2.8.1", "3.el7_5.1", "x86_64", "audit-2.8.1-3.el7_5.1.src.rpm", 256370, "LGPLv2+", "CentOS", "", "Dynamic library for libaudit", "437ad738bda99bc9fdc9185f45554467"},
			{intRef(), "pam", "1.1.8", "22.el7", "x86_64", "pam-1.1.8-22.el7.src.rpm", 2630324, "BSD and GPLv2+", "CentOS", "", "An extensible library which provides authentication for applications", "dce355546c72f1425dc04fd64c9bc664"},
			{intRef(), "nss-softokn", "3.36.0", "5.el7_5", "x86_64", "nss-softokn-3.36.0-5.el7_5.src.rpm", 1130670, "MPLv2.0", "CentOS", "", "Network Security Services Softoken Module", "7b8391d5b6843722982a616a8542d58f"},
			{intRef(), "nss", "3.36.0", "7.el7_5", "x86_64", "nss-3.36.0-7.el7_5.src.rpm", 2424993, "MPLv2.0", "CentOS", "", "Network Security Services", "aead60d750112396fc5381c7c6152863"},
			{intRef(), "libassuan", "2.1.0", "3.el7", "x86_64", "libassuan-2.1.0-3.el7.src.rpm", 155391, "LGPLv2+ and GPLv3+", "CentOS", "", "GnuPG IPC library", "c4c94aadc9da58f576b4ef238d0136cb"},
			{intRef(1), "pkgconfig", "0.27.1", "4.el7", "x86_64", "pkgconfig-0.27.1-4.el7.src.rpm", 105522, "GPLv2+", "CentOS", "", "A tool for determining compilation options", "ef58117c9c617599952c0642ff5e8d00"},
			{intRef(), "cyrus-sasl-lib", "2.1.26", "23.el7", "x86_64", "cyrus-sasl-2.1.26-23.el7.src.rpm", 396911, "BSD with advertising", "CentOS", "", "Shared libraries needed by applications which use Cyrus SASL", "6d06f1337b143c25bc7358255d433006"},
			{intRef(), "binutils", "2.27", "28.base.el7_5.1", "x86_64", "binutils-2.27-28.base.el7_5.1.src.rpm", 25149789, "GPLv3+", "CentOS", "", "A GNU collection of binary utilities", "ab389c762bf3df5411b21583f3479dc1"},
			{intRef(), "libcurl", "7.29.0", "46.el7", "x86_64", "curl-7.29.0-46.el7.src.rpm", 435192, "MIT", "CentOS", "", "A library for getting files from web servers", "2d99038a0bef38275774024117436284"},
			{intRef(), "openldap", "2.4.44", "15.el7_5", "x86_64", "openldap-2.4.44-15.el7_5.src.rpm", 1037299, "OpenLDAP", "CentOS", "", "LDAP support libraries", "0c7d41097a0100c3e40ac4aff0d3cdd8"},
			{intRef(), "pinentry", "0.8.1", "17.el7", "x86_64", "pinentry-0.8.1-17.el7.src.rpm", 159929, "GPLv2+", "CentOS", "", "Collection of simple PIN or passphrase entry dialogs", "740fabe008451f11fee22ff086fb13d3"},
			{intRef(), "libsemanage", "2.5", "11.el7", "x86_64", "libsemanage-2.5-11.el7.src.rpm", 302369, "LGPLv2+", "CentOS", "", "SELinux binary policy manipulation library", "d2fd300c5abc3561b16a685b1f2cbc5d"},
			{intRef(), "libutempter", "1.1.6", "4.el7", "x86_64", "libutempter-1.1.6-4.el7.src.rpm", 49749, "LGPLv2+", "CentOS", "", "A privileged helper for utmp/wtmp updates", "8e28747e6a84b84af7d78c84cd8cd5e8"},
			{intRef(), "qrencode-libs", "3.4.1", "3.el7", "x86_64", "qrencode-3.4.1-3.el7.src.rpm", 126732, "LGPLv2+", "CentOS", "", "QR Code encoding library - Shared libraries", "ac1de0c46bc1886e12ea80771951cde2"},
			{intRef(7), "device-mapper", "1.02.146", "4.el7", "x86_64", "lvm2-2.02.177-4.el7.src.rpm", 338922, "GPLv2", "CentOS", "", "Device mapper utility", "1919f9661f5cf5f63fcb4c5112718db7"},
			{intRef(), "procps-ng", "3.3.10", "17.el7_5.2", "x86_64", "procps-ng-3.3.10-17.el7_5.2.src.rpm", 760500, "GPL+ and GPLv2 and GPLv2+ and GPLv3+ and LGPLv2+", "CentOS", "", "System and process monitoring utilities", "57c9fa477ce98fca4e7dcc5ed6b41a7a"},
			{intRef(), "kmod", "20", "21.el7", "x86_64", "kmod-20-21.el7.src.rpm", 243127, "GPLv2+", "CentOS", "", "Linux kernel module management utilities", "47a0704853382255bfbc519fa9dd1a3d"},
			{intRef(1), "dbus", "1.10.24", "7.el7", "x86_64", "dbus-1.10.24-7.el7.src.rpm", 595223, "(GPLv2+ or AFL) and GPLv2+", "CentOS", "", "D-BUS message bus", "ee5c672ee2b669192e40bdf2460a5427"},
			{intRef(), "iputils", "20160308", "10.el7", "x86_64", "iputils-20160308-10.el7.src.rpm", 343497, "BSD and GPLv2+", "CentOS", "", "Network monitoring tools including ping", "dfa219e6db737a1eb1b5cb23b532e858"},
			{intRef(), "gdbm", "1.10", "8.el7", "x86_64", "gdbm-1.10-8.el7.src.rpm", 184322, "GPLv3+", "CentOS", "", "A GNU set of database routines which use extensible hashing", "770bd49f3898dc1cc831fb9ec12d569a"},
			{intRef(), "python", "2.7.5", "69.el7_5", "x86_64", "python-2.7.5-69.el7_5.src.rpm", 80907, "Python", "CentOS", "", "An interpreted, interactive, object-oriented programming language", "6ed4c7b6c1fbc6994dd6e8c6cc8de459"},
			{intRef(), "dbus-python", "1.1.1", "9.el7", "x86_64", "dbus-python-1.1.1-9.el7.src.rpm", 848122, "MIT", "CentOS", "", "D-Bus Python Bindings", "6be4da91886c59ae25e5255ebff0a0a9"},
			{intRef(), "pyliblzma", "0.5.3", "11.el7", "x86_64", "pyliblzma-0.5.3-11.el7.src.rpm", 190112, "LGPLv3+", "CentOS", "", "Python bindings for lzma", "a09aa8e806a3f6285e9fe765d69d3ee9"},
			{intRef(), "python-urlgrabber", "3.10", "8.el7", "noarch", "python-urlgrabber-3.10-8.el7.src.rpm", 500670, "LGPLv2+", "CentOS", "", "A high-level cross-protocol url-grabber", "8a0c6c349d305a6c89d04e4d661e85a9"},
			{intRef(), "pyxattr", "0.5.1", "5.el7", "x86_64", "pyxattr-0.5.1-5.el7.src.rpm", 63304, "LGPLv2+", "CentOS", "", "Extended attributes library wrapper for Python", "e85121da681760c48a2af6ed8aa31b4e"},
			{intRef(), "python-kitchen", "1.1.1", "5.el7", "noarch", "python-kitchen-1.1.1-5.el7.src.rpm", 1465161, "LGPLv2+", "CentOS", "", "Small, useful pieces of code to make python coding easier", "917912ebf6164135f1f9871150432977"},
			{intRef(), "gnupg2", "2.0.22", "5.el7_5", "x86_64", "gnupg2-2.0.22-5.el7_5.src.rpm", 6637796, "GPLv3+", "CentOS", "", "Utility for secure communication and data storage", "a293dfefd6596bfe5c54ba68377e334e"},
			{intRef(), "pygpgme", "0.3", "9.el7", "x86_64", "pygpgme-0.3-9.el7.src.rpm", 197501, "LGPLv2+", "CentOS", "", "Python module for working with OpenPGP messages", "67b8c720f59a063adbbca2392bac0281"},
			{intRef(), "yum", "3.4.3", "158.el7.centos", "noarch", "yum-3.4.3-158.el7.centos.src.rpm", 5814102, "GPLv2+", "CentOS", "", "RPM package installer/updater/manager", "55ef45392a487da6305b69e89cef7cb7"},
			{intRef(), "yum-utils", "1.1.31", "46.el7_5", "noarch", "yum-utils-1.1.31-46.el7_5.src.rpm", 343422, "GPLv2+", "CentOS", "", "Utilities based around the yum package manager", "9a3315c3ee603728ce1908ec404a2c42"},
			{intRef(2), "vim-minimal", "7.4.160", "4.el7", "x86_64", "vim-7.4.160-4.el7.src.rpm", 917640, "Vim", "CentOS", "", "A minimal version of the VIM editor", "495d7fc35ace98f416bef65eecf8af0d"},
			{intRef(), "gpg-pubkey", "f4a80eb5", "53a7ff4b", "", "", 0, "pubkey", "", "", "gpg(CentOS-7 Key (CentOS 7 Official Signing Key) <security@centos.org>)", ""},
			{intRef(), "glibc-common", "2.17", "260.el7_6.4", "x86_64", "glibc-2.17-260.el7_6.4.src.rpm", 120497983, "LGPLv2+ and LGPLv2+ with exceptions and GPLv2+", "CentOS", "", "Common binaries and locale data for glibc", "43f1a899692374dc09c21bae9c759d1c"},
			{intRef(), "libstdc++", "4.8.5", "36.el7_6.1", "x86_64", "gcc-4.8.5-36.el7_6.1.src.rpm", 1077442, "GPLv3+ and GPLv3+ with exceptions and GPLv2+ with exceptions and LGPLv2+ and BSD", "CentOS", "", "GNU Standard C++ Library", "d2b021bf5f895c786ce110a0ab2267d7"},
			{intRef(), "rpm", "4.11.3", "35.el7", "x86_64", "rpm-4.11.3-35.el7.src.rpm", 2622266, "GPLv2+", "CentOS", "", "The RPM package management system", "ba17adc338f442da8a4598244965e521"},
			{intRef(), "file-libs", "5.11", "35.el7", "x86_64", "file-5.11-35.el7.src.rpm", 3076831, "BSD", "CentOS", "", "Libraries for applications using libmagic", "3037fd1f781186bcb0fb919b2aa78f54"},
			{intRef(), "mpfr", "3.1.1", "4.el7", "x86_64", "mpfr-3.1.1-4.el7.src.rpm", 554279, "LGPLv3+ and GPLv3+ and GFDL", "CentOS", "", "A C library for multiple-precision floating-point computations", "4cedb227fd4730529b0874792007692b"},
			{intRef(), "rpm-build-libs", "4.11.3", "35.el7", "x86_64", "rpm-4.11.3-35.el7.src.rpm", 166776, "GPLv2+ and LGPLv2+ with exceptions", "CentOS", "", "Libraries for building and signing RPM packages", "aba616fc3d1a25cfc77eb27db8610457"},
			{intRef(), "fipscheck-lib", "1.4.1", "6.el7", "x86_64", "fipscheck-1.4.1-6.el7.src.rpm", 11466, "BSD", "CentOS", "", "Library files for fipscheck", "93aa3152b03992c02e0ed2759b1d3ac8"},
			{intRef(), "apr", "1.4.8", "3.el7_4.1", "x86_64", "apr-1.4.8-3.el7_4.1.src.rpm", 226686, "ASL 2.0 and BSD with advertising and ISC and BSD", "CentOS", "", "Apache Portable Runtime library", "0534c5c4a8da91bf8e4a4152c0383b46"},
			{intRef(), "libquadmath", "4.8.5", "36.el7_6.1", "x86_64", "gcc-4.8.5-36.el7_6.1.src.rpm", 282811, "GPLv3+ and GPLv3+ with exceptions and GPLv2+ with exceptions and LGPLv2+ and BSD", "CentOS", "", "GCC __float128 shared support library", "80155e9c1179374d45231786bedae2a4"},
			{intRef(), "boost-system", "1.53.0", "27.el7", "x86_64", "boost-1.53.0-27.el7.src.rpm", 32538, "Boost and MIT and Python", "CentOS", "", "Run-Time component of boost system support library", "6df5a05420fbd79bd36e644acc509dfe"},
			{intRef(), "unzip", "6.0", "19.el7", "x86_64", "unzip-6.0-19.el7.src.rpm", 373986, "BSD", "CentOS", "", "A utility for unpacking zip files", "7a396b648ba0120404fe2ef171b79681"},
			{intRef(), "avahi-libs", "0.6.31", "19.el7", "x86_64", "avahi-0.6.31-19.el7.src.rpm", 123808, "LGPLv2+", "CentOS", "", "Libraries for avahi run-time use", "78426c94f63ec36f4a9d27bfd000a7d3"},
			{intRef(), "gettext-libs", "0.19.8.1", "2.el7", "x86_64", "gettext-0.19.8.1-2.el7.src.rpm", 1546952, "LGPLv2+ and GPLv3+", "CentOS", "", "Libraries for gettext", "3164b6e314d214a982d34c7c77d89e16"},
			{intRef(), "json-c", "0.11", "4.el7_0", "x86_64", "json-c-0.11-4.el7_0.src.rpm", 65593, "MIT", "CentOS", "", "A JSON implementation in C", "62bcb642b82d3b3aeb99ba9ba1b40715"},
			{intRef(), "cryptsetup-libs", "2.0.3", "3.el7", "x86_64", "cryptsetup-2.0.3-3.el7.src.rpm", 1219591, "GPLv2+ and LGPLv2+", "CentOS", "", "Cryptsetup shared library", "61b5b318bfcd411aec021e25d9b76070"},
			{intRef(1), "perl-parent", "0.225", "244.el7", "noarch", "perl-parent-0.225-244.el7.src.rpm", 8141, "GPL+ or Artistic", "CentOS", "", "Establish an ISA relationship with base classes at compile time", "d902f1a3426caa19ccbccf09263c54f6"},
			{intRef(), "perl-podlators", "2.5.1", "3.el7", "noarch", "perl-podlators-2.5.1-3.el7.src.rpm", 287679, "GPL+ or Artistic", "CentOS", "", "Format POD source into various output formats", "8e070dd716e3adf3c297291accaa4348"},
			{intRef(1), "perl-Pod-Escapes", "1.04", "294.el7_6", "noarch", "perl-5.16.3-294.el7_6.src.rpm", 21091, "GPL+ or Artistic", "CentOS", "", "Perl module for resolving POD escape sequences", "ad4d7216b06eb48942c30d4bc2b09755"},
			{intRef(), "perl-Encode", "2.51", "7.el7", "x86_64", "perl-Encode-2.51-7.el7.src.rpm", 10176350, "GPL+ or Artistic", "CentOS", "", "Character encodings in Perl", "713ce27ff4434807ff7060b333ed8873"},
			{intRef(4), "perl-libs", "5.16.3", "294.el7_6", "x86_64", "perl-5.16.3-294.el7_6.src.rpm", 1647344, "GPL+ or Artistic", "CentOS", "", "The libraries for the perl runtime", "669be9d414caaf85799b0dfa644c4e81"},
			{intRef(), "perl-Storable", "2.45", "3.el7", "x86_64", "perl-Storable-2.45-3.el7.src.rpm", 181031, "GPL+ or Artistic", "CentOS", "", "Persistence for Perl data structures", "25787b4b609f3ee50d1c7f33d29e0102"},
			{intRef(), "perl-threads", "1.87", "4.el7", "x86_64", "perl-threads-1.87-4.el7.src.rpm", 98615, "GPL+ or Artistic", "CentOS", "", "Perl interpreter-based threads", "29739d2f766c8bd85e85de9f8ccb7367"},
			{intRef(), "perl-Filter", "1.49", "3.el7", "x86_64", "perl-Filter-1.49-3.el7.src.rpm", 148475, "GPL+ or Artistic", "CentOS", "", "Perl source filters", "66b1205b2eef2a89137e530741ed3710"},
			{intRef(), "perl-constant", "1.27", "2.el7", "noarch", "perl-constant-1.27-2.el7.src.rpm", 26364, "GPL+ or Artistic", "CentOS", "", "Perl pragma to declare constants", "f3ecfa1ce48e13496322e341a3f0b72c"},
			{intRef(4), "perl-macros", "5.16.3", "294.el7_6", "x86_64", "perl-5.16.3-294.el7_6.src.rpm", 5134, "GPL+ or Artistic", "CentOS", "", "Macros for rpmbuild", "7c63786a9688ac528e398b27e86198d8"},
			{intRef(), "perl-File-Temp", "0.23.01", "3.el7", "noarch", "perl-File-Temp-0.23.01-3.el7.src.rpm", 158781, "GPL+ or Artistic", "CentOS", "", "Return name and handle of a temporary file safely", "396b0a1af3a203e1cb33c98814d5a6d8"},
			{intRef(), "perl-PathTools", "3.40", "5.el7", "x86_64", "perl-PathTools-3.40-5.el7.src.rpm", 174131, "(GPL+ or Artistic) and BSD", "CentOS", "", "PathTools Perl module (Cwd, File::Spec)", "c597987071b0f81a0029667e83e1ecae"},
			{intRef(1), "perl-Pod-Simple", "3.28", "4.el7", "noarch", "perl-Pod-Simple-3.28-4.el7.src.rpm", 538320, "GPL+ or Artistic", "CentOS", "", "Framework for parsing POD documentation", "8e5edeb936a154537062b776eb41cb7d"},
			{intRef(4), "perl", "5.16.3", "294.el7_6", "x86_64", "perl-5.16.3-294.el7_6.src.rpm", 23552128, "(GPL+ or Artistic) and (GPLv2+ or Artistic) and Copyright Only and MIT and Public Domain and UCD", "CentOS", "", "Practical Extraction and Report Language", "547b1d620cedc114b9a1a681c1f2a4bd"},
			{intRef(1), "perl-Error", "0.17020", "2.el7", "noarch", "perl-Error-0.17020-2.el7.src.rpm", 50015, "(GPL+ or Artistic) and MIT", "CentOS", "", "Error/exception handling in an OO-ish way", "d5f79c530c268d71811519df24ca0c90"},
			{intRef(), "perl-Data-Dumper", "2.145", "3.el7", "x86_64", "perl-Data-Dumper-2.145-3.el7.src.rpm", 99287, "GPL+ or Artistic", "CentOS", "", "Stringify perl data structures, suitable for printing and eval", "3d2fc7733dc839a2d9a67e2805479feb"},
			{intRef(), "perl-XML-Parser", "2.41", "10.el7", "x86_64", "perl-XML-Parser-2.41-10.el7.src.rpm", 642973, "GPL+ or Artistic", "CentOS", "", "Perl module for parsing XML documents", "1998f3b0016cc426351682109fc7c9a8"},
			{intRef(), "automake", "1.13.4", "3.el7", "noarch", "automake-1.13.4-3.el7.src.rpm", 1731909, "GPLv2+ and GFDL and Public Domain and MIT", "CentOS", "", "A GNU tool for automatically creating Makefiles", "211dbf9617136d88757937007240ae9b"},
			{intRef(), "less", "458", "9.el7", "x86_64", "less-458-9.el7.src.rpm", 215376, "GPLv3+", "CentOS", "", "A text file browser similar to more, but better", "2b1222dfb9fbcf5ca3ee2f07d897bb4c"},
			{intRef(), "libgfortran", "4.8.5", "36.el7_6.1", "x86_64", "gcc-4.8.5-36.el7_6.1.src.rpm", 1186168, "GPLv3+ and GPLv3+ with exceptions and GPLv2+ with exceptions and LGPLv2+ and BSD", "CentOS", "", "Fortran runtime", "a210ea42ee5ed4ca8328a4555b0717ab"},
			{intRef(), "cpp", "4.8.5", "36.el7_6.1", "x86_64", "gcc-4.8.5-36.el7_6.1.src.rpm", 15644861, "GPLv3+ and GPLv3+ with exceptions and GPLv2+ with exceptions and LGPLv2+ and BSD", "CentOS", "", "The C Preprocessor", "2044dee2673e3aabfb7df7e4e152227e"},
			{intRef(), "systemd-libs", "219", "62.el7_6.5", "x86_64", "systemd-219-62.el7_6.5.src.rpm", 1263456, "LGPLv2+ and MIT", "CentOS", "", "systemd libraries", "9577513987ddb8ff723a0b94367fc8c7"},
			{intRef(), "trousers", "0.3.14", "2.el7", "x86_64", "trousers-0.3.14-2.el7.src.rpm", 836873, "BSD", "CentOS", "", "TCG's Software Stack v1.2", "7a4bdd0c029c5750be61101d418d4adc"},
			{intRef(), "systemd-sysv", "219", "62.el7_6.5", "x86_64", "systemd-219-62.el7_6.5.src.rpm", 3979, "LGPLv2+", "CentOS", "", "SysV tools for systemd", "cfb0137b4b27e0e16bc2c23388f69b59"},
			{intRef(), "libdwarf", "20130207", "4.el7", "x86_64", "libdwarf-20130207-4.el7.src.rpm", 291679, "LGPLv2", "CentOS", "", "Library to access the DWARF Debugging file format", "2c5a9b992e22a196fcdfe3657972956d"},
			{intRef(), "boost-date-time", "1.53.0", "27.el7", "x86_64", "boost-1.53.0-27.el7.src.rpm", 139706, "Boost and MIT and Python", "CentOS", "", "Run-Time component of boost date-time library", "ded46380fd053f4a72b921730cd0aa44"},
			{intRef(), "systemtap-runtime", "3.3", "3.el7", "x86_64", "systemtap-3.3-3.el7.src.rpm", 1142954, "GPLv2+", "CentOS", "", "Programmable system-wide instrumentation system - runtime", "e133207717622b13eb3018ef60e11638"},
			{intRef(), "libmodman", "2.0.1", "8.el7", "x86_64", "libmodman-2.0.1-8.el7.src.rpm", 58871, "LGPLv2+", "CentOS", "", "A simple library for managing C++ modules (plug-ins)", "d9d56d2a76492f4c2f9c827d7aae6b08"},
			{intRef(), "nettle", "2.7.1", "8.el7", "x86_64", "nettle-2.7.1-8.el7.src.rpm", 765042, "LGPLv2+", "CentOS", "", "A low-level cryptographic library", "5664b34732ae0135d04a6a3e6a5a6b61"},
			{intRef(), "ncurses-base", "5.9", "14.20130511.el7_4", "noarch", "ncurses-5.9-14.20130511.el7_4.src.rpm", 223432, "MIT", "CentOS", "", "Descriptions of common terminals", "c955fdfe1f8f20b1053a34918e44bd58"},
			{intRef(), "gdb", "7.6.1", "114.el7", "x86_64", "gdb-7.6.1-114.el7.src.rpm", 7371507, "GPLv3+ and GPLv3+ with exceptions and GPLv2+ and GPLv2+ with exceptions and GPL+ and LGPLv2+ and BSD and Public Domain", "CentOS", "", "A GNU source-level debugger for C, C++, Fortran, Go and other languages", "1d085bb4c7273a51db58d47b35ab95b4"},
			{intRef(), "bash", "4.2.46", "30.el7", "x86_64", "bash-4.2.46-30.el7.src.rpm", 3667709, "GPLv3+", "CentOS", "", "The GNU Bourne Again shell", "4c9037d4d3139a2c8fd28ed6b27d47da"},
			{intRef(), "libedit", "3.0", "12.20121213cvs.el7", "x86_64", "libedit-3.0-12.20121213cvs.el7.src.rpm", 244257, "BSD", "CentOS", "", "The NetBSD Editline library", "9e248aca4576e17c188396aa91931c63"},
			{intRef(), "chkconfig", "1.7.4", "1.el7", "x86_64", "chkconfig-1.7.4-1.el7.src.rpm", 779531, "GPLv2", "CentOS", "", "A system tool for maintaining the /etc/rc*.d hierarchy", "e126177e9ac5841deca96f4e15340613"},
			{intRef(), "perl-Git", "1.8.3.1", "20.el7", "noarch", "git-1.8.3.1-20.el7.src.rpm", 58567, "GPLv2", "CentOS", "", "Perl interface to Git", "801ebea11b001365abebd04e2da2690f"},
			{intRef(), "setup", "2.8.71", "9.el7", "noarch", "setup-2.8.71-9.el7.src.rpm", 696925, "Public Domain", "CentOS", "", "A set of system configuration and setup files", "3e3fc345cd2ac49c17b8e38df270fd16"},
			{intRef(), "pakchois", "0.4", "10.el7", "x86_64", "pakchois-0.4-10.el7.src.rpm", 29489, "LGPLv2+", "CentOS", "", "A wrapper library for PKCS#11", "7adb828e1d494684471a8c13511131d9"},
			{intRef(), "basesystem", "10.0", "7.el7.centos", "noarch", "basesystem-10.0-7.el7.centos.src.rpm", 0, "Public Domain", "CentOS", "", "The skeleton package which defines a simple CentOS Linux system", "778eb3d19a934f42d1354ed795d06b4f"},
			{intRef(), "subversion-libs", "1.7.14", "14.el7", "x86_64", "subversion-1.7.14-14.el7.src.rpm", 2615823, "ASL 2.0", "CentOS", "", "Libraries for Subversion Version Control system", "ae892db7483602754c8a3e98869e46c6"},
			{intRef(), "zlib", "1.2.7", "17.el7", "x86_64", "zlib-1.2.7-17.el7.src.rpm", 185710, "zlib and Boost", "CentOS", "", "The compression and decompression library", "505b6038db14d2b7d58c5721126cb11c"},
			{intRef(), "efivar-libs", "36", "11.el7_6.1", "x86_64", "efivar-36-11.el7_6.1.src.rpm", 252096, "LGPLv2+", "CentOS", "", "Library to manage UEFI variables", "bce3fe8bb152785ab1475b64339a3984"},
			{intRef(), "nss-util", "3.36.0", "1.el7_5", "x86_64", "nss-util-3.36.0-1.el7_5.src.rpm", 194944, "MPLv2.0", "CentOS", "", "Network Security Services Utilities Library", "ecf6d738049e13862a8cf2a151b78bb5"},
			{intRef(), "systemtap-client", "3.3", "3.el7", "x86_64", "systemtap-3.3-3.el7.src.rpm", 10146436, "GPLv2+", "CentOS", "", "Programmable system-wide instrumentation system - client", "d9129a6c4b403c9ff7f1a38e32add796"},
			{intRef(), "libcom_err", "1.42.9", "12.el7_5", "x86_64", "e2fsprogs-1.42.9-12.el7_5.src.rpm", 60489, "MIT", "CentOS", "", "Common error description library", "4b27bfeb726d091b4317b5e66d6dba7f"},
			{intRef(), "kernel-headers", "3.10.0", "957.10.1.el7", "x86_64", "kernel-3.10.0-957.10.1.el7.src.rpm", 3823832, "GPLv2", "CentOS", "", "Header files for the Linux kernel for use by glibc", "17c5c27d1c0577522a0bd74f96990825"},
			{intRef(), "libattr", "2.4.46", "13.el7", "x86_64", "attr-2.4.46-13.el7.src.rpm", 19896, "LGPLv2+", "CentOS", "", "Dynamic library for extended attribute support", "209b05c7b77a33a4cc176077945833e8"},
			{intRef(), "glibc-devel", "2.17", "260.el7_6.4", "x86_64", "glibc-2.17-260.el7_6.4.src.rpm", 1066078, "LGPLv2+ and LGPLv2+ with exceptions and GPLv2+", "CentOS", "", "Object files for development using standard C libraries.", "cf1d0eead12e91a77c631cb27f0aa8ee"},
			{intRef(), "libacl", "2.2.51", "14.el7", "x86_64", "acl-2.2.51-14.el7.src.rpm", 37056, "LGPLv2+", "CentOS", "", "Dynamic library for access control list support", "c3243d728de0ad95e0c05230e6463bdd"},
			{intRef(), "libquadmath-devel", "4.8.5", "36.el7_6.1", "x86_64", "gcc-4.8.5-36.el7_6.1.src.rpm", 18514, "GPLv3+ and GPLv3+ with exceptions and GPLv2+ with exceptions and LGPLv2+ and BSD", "CentOS", "", "GCC __float128 support", "3cebb989a4fc2a0787f68ae06061d502"},
			{intRef(), "perl-srpm-macros", "1", "8.el7", "noarch", "perl-srpm-macros-1-8.el7.src.rpm", 794, "GPLv3+", "CentOS", "", "RPM macros for building Perl source package from source repository", "6bd0791a143b53addc95cbeb050859c8"},
			{intRef(), "info", "5.1", "5.el7", "x86_64", "texinfo-5.1-5.el7.src.rpm", 494630, "GPLv3+", "CentOS", "", "A stand-alone TTY-based reader for GNU texinfo documentation", "e1f75e40cf956ed3efe7b49c5f6939ec"},
			{intRef(), "gettext-common-devel", "0.19.8.1", "2.el7", "noarch", "gettext-0.19.8.1-2.el7.src.rpm", 397912, "GPLv3+", "CentOS", "", "Common development files for gettext", "08291ef848c478a58968efe973aced3d"},
			{intRef(), "pcre", "8.32", "17.el7", "x86_64", "pcre-8.32-17.el7.src.rpm", 1475532, "BSD", "CentOS", "", "Perl-compatible regular expression library", "04a1926cc29eef0f79f8183eb7907024"},
			{intRef(), "intltool", "0.50.2", "7.el7", "noarch", "intltool-0.50.2-7.el7.src.rpm", 170106, "GPLv2 with exceptions", "CentOS", "", "Utility for internationalizing various kinds of data files", "f5fd5d682337200009190a45fa4f80bf"},
			{intRef(), "sed", "4.2.2", "5.el7", "x86_64", "sed-4.2.2-5.el7.src.rpm", 601208, "GPLv3+", "CentOS", "", "A GNU stream text editor", "4d696dbd3061493258490f40c625adba"},
			{intRef(), "systemtap", "3.3", "3.el7", "x86_64", "systemtap-3.3-3.el7.src.rpm", 206577, "GPLv2+", "CentOS", "", "Programmable system-wide instrumentation system", "7aa25cadecbbb48aaefedd2dab5d8196"},
			{intRef(), "p11-kit", "0.23.5", "3.el7", "x86_64", "p11-kit-0.23.5-3.el7.src.rpm", 1337825, "BSD", "CentOS", "", "Library for loading and sharing PKCS#11 modules", "9a0f49ca3b6d58aeb05780003cae38f1"},
			{intRef(), "libtool", "2.4.2", "22.el7_3", "x86_64", "libtool-2.4.2-22.el7_3.src.rpm", 2323761, "GPLv2+ and LGPLv2+ and GFDL", "CentOS", "", "The GNU Portable Library Tool", "34886e5d2953afff439b5b7e9b1ca3d4"},
			{intRef(1), "gmp", "6.0.0", "15.el7", "x86_64", "gmp-6.0.0-15.el7.src.rpm", 657046, "LGPLv3+ or GPLv2+", "CentOS", "", "A GNU arbitrary precision library", "4cd60ebf94766a46c590e17036ece777"},
			{intRef(), "cscope", "15.8", "10.el7", "x86_64", "cscope-15.8-10.el7.src.rpm", 939744, "BSD and GPLv2+", "CentOS", "", "C source code tree search and browse tool", "3335c17d1185aab5024ffec64854b254"},
			{intRef(), "libtasn1", "4.10", "1.el7", "x86_64", "libtasn1-4.10-1.el7.src.rpm", 424486, "GPLv3+ and LGPLv2+", "CentOS", "", "The ASN.1 library used in GNUTLS", "5bd1b0b4094fac08713f986572a8625e"},
			{intRef(), "patchutils", "0.3.3", "4.el7", "x86_64", "patchutils-0.3.3-4.el7.src.rpm", 260135, "GPLv2+", "CentOS", "", "A collection of programs for manipulating patch files", "f90fbbf401062fd532219b576d061676"},
			{intRef(), "ca-certificates", "2018.2.22", "70.0.el7_5", "noarch", "ca-certificates-2018.2.22-70.0.el7_5.src.rpm", 973960, "Public Domain", "CentOS", "", "The Mozilla CA root certificate bundle", "a95373fc5273e7c757a240316d4430b9"},
			{intRef(), "bison", "3.0.4", "2.el7", "x86_64", "bison-3.0.4-2.el7.src.rpm", 2154174, "GPLv3+", "CentOS", "", "A GNU general-purpose parser generator", "c12d2d7e86d2ede2fe96c8c119c6d06d"},
			{intRef(), "coreutils", "8.22", "21.el7", "x86_64", "coreutils-8.22-21.el7.src.rpm", 14588989, "GPLv3+", "CentOS", "", "A set of basic GNU tools commonly used in shell scripts", "fb8117e7aeb1ab935460fa412277a088"},
			{intRef(), "rpm-sign", "4.11.3", "35.el7", "x86_64", "rpm-4.11.3-35.el7.src.rpm", 17397, "GPLv2+", "CentOS", "", "Package signing support", "8a594161eed57b87386dc0e4d85b03b4"},
			{intRef(), "krb5-libs", "1.15.1", "19.el7", "x86_64", "krb5-1.15.1-19.el7.src.rpm", 1984782, "MIT", "CentOS", "", "The non-admin shared libraries used by Kerberos 5", "9811e62803caaf9375851ed6f16fedd7"},
			{intRef(), "swig", "2.0.10", "5.el7", "x86_64", "swig-2.0.10-5.el7.src.rpm", 4926232, "GPLv3+ and BSD", "CentOS", "", "Connects C/C++/Objective C to some high-level programming languages", "6f2f743e241650625c57d1012b1d7eff"},
			{intRef(), "bzip2-libs", "1.0.6", "13.el7", "x86_64", "bzip2-1.0.6-13.el7.src.rpm", 70093, "BSD", "CentOS", "", "Libraries for applications using bzip2", "0486f0b02e0caa4195761266798e7a26"},
			{intRef(), "rcs", "5.9.0", "5.el7", "x86_64", "rcs-5.9.0-5.el7.src.rpm", 624710, "GPLv3+", "CentOS", "", "Revision Control System (RCS) file version management tools", "fcb64d3f5164834d42cd02954692bfdb"},
			{intRef(), "ctags", "5.8", "13.el7", "x86_64", "ctags-5.8-13.el7.src.rpm", 359724, "GPLv2+ and LGPLv2+ and Public Domain", "CentOS", "", "A C programming language indexing and/or cross-reference tool", "89441c9d19184e2f2a8b14e39b2ef63c"},
			{intRef(), "libxml2", "2.9.1", "6.el7_2.3", "x86_64", "libxml2-2.9.1-6.el7_2.3.src.rpm", 1710062, "MIT", "CentOS", "", "Library providing XML and HTML support", "e439b8934c4e9576eccfbc60d3cd714f"},
			{intRef(), "readline", "6.2", "10.el7", "x86_64", "readline-6.2-10.el7.src.rpm", 460464, "GPLv3+", "CentOS", "", "A library for editing typed command lines", "b552128087365a58b3566631c678752f"},
			{intRef(), "cpio", "2.11", "27.el7", "x86_64", "cpio-2.11-27.el7.src.rpm", 689335, "GPLv3+", "CentOS", "", "A GNU archiving program", "32498692f82296d51663586f08cebafc"},
			{intRef(), "libblkid", "2.23.2", "52.el7_5.1", "x86_64", "util-linux-2.23.2-52.el7_5.1.src.rpm", 261837, "LGPLv2+", "CentOS", "", "Block device ID library", "427f260c9fc06f72d722e5ccdfcf30c5"},
			{intRef(), "glib2", "2.54.2", "2.el7", "x86_64", "glib2-2.54.2-2.el7.src.rpm", 11986873, "LGPLv2+", "CentOS", "", "A library of handy utility functions", "75e476b2fbb8f06f881338c12e379cc3"},
			{intRef(), "sqlite", "3.7.17", "8.el7", "x86_64", "sqlite-3.7.17-8.el7.src.rpm", 814231, "Public Domain", "CentOS", "", "Library that implements an embeddable SQL database engine", "0ceb9434c5a9d281399583087cfe889b"},
			{intRef(), "cracklib", "2.9.0", "11.el7", "x86_64", "cracklib-2.9.0-11.el7.src.rpm", 209610, "LGPLv2+", "CentOS", "", "A password-checking library", "a973d883d6785904e49266428bb3ed90"},
			{intRef(), "libidn", "1.28", "4.el7", "x86_64", "libidn-1.28-4.el7.src.rpm", 630407, "LGPLv2+ and GPLv3+ and GFDL", "CentOS", "", "Internationalized Domain Name support library", "3d582106ac1032c59c1cd4236e5d6415"},
			{intRef(), "libcap-ng", "0.7.5", "4.el7", "x86_64", "libcap-ng-0.7.5-4.el7.src.rpm", 50510, "LGPLv2+", "CentOS", "", "An alternate posix capabilities library", "84efa5ca78408f78e606e7c8d1816759"},
			{intRef(), "cracklib-dicts", "2.9.0", "11.el7", "x86_64", "cracklib-2.9.0-11.el7.src.rpm", 9389116, "LGPLv2+", "CentOS", "", "The standard CrackLib dictionaries", "da657b0facf181893f9a38437092dbbf"},
			{intRef(), "libpwquality", "1.2.3", "5.el7", "x86_64", "libpwquality-1.2.3-5.el7.src.rpm", 332421, "BSD or GPLv2+", "CentOS", "", "A library for password generation and password quality checking", "e40a69f0a1803dfa3736aba2c8ea1fb4"},
			{intRef(), "nss-sysinit", "3.36.0", "7.el7_5", "x86_64", "nss-3.36.0-7.el7_5.src.rpm", 14061, "MPLv2.0", "CentOS", "", "System NSS Initialization", "ace8226abffd1e928af6f4c62cdd450d"},
			{intRef(), "nss-pem", "1.0.3", "4.el7", "x86_64", "nss-pem-1.0.3-4.el7.src.rpm", 201219, "MPLv1.1", "CentOS", "", "PEM file reader for Network Security Services (NSS)", "95cf0973322bf389a612fdbd1e3a6627"},
			{intRef(), "xz", "5.2.2", "1.el7", "x86_64", "xz-5.2.2-1.el7.src.rpm", 798130, "LGPLv2+", "CentOS", "", "LZMA compression utilities", "3e6f05008f7af7931aa8257964527f1a"},
			{intRef(), "lz4", "1.7.5", "2.el7", "x86_64", "lz4-1.7.5-2.el7.src.rpm", 366872, "GPLv2+ and BSD", "CentOS", "", "Extremely fast compression algorithm", "89e45994eabac0ba4ccae7c1005ea06c"},
			{intRef(), "nss-tools", "3.36.0", "7.el7_5", "x86_64", "nss-3.36.0-7.el7_5.src.rpm", 2069571, "MPLv2.0", "CentOS", "", "Tools for the Network Security Services", "dfada2eb06ef6cccbdc69c711c59ae02"},
			{intRef(), "gobject-introspection", "1.50.0", "1.el7", "x86_64", "gobject-introspection-1.50.0-1.el7.src.rpm", 834149, "GPLv2+, LGPLv2+, MIT", "CentOS", "", "Introspection system for GObject-based libraries", "6048b8e810c7b808f4eddca36d12d5b8"},
			{intRef(), "libdb-utils", "5.3.21", "24.el7", "x86_64", "libdb-5.3.21-24.el7.src.rpm", 326487, "BSD and LGPLv2 and Sleepycat", "CentOS", "", "Command line tools for managing Berkeley DB databases", "d06077335f27858ecb06ad48fa2e812d"},
			{intRef(), "kmod-libs", "20", "21.el7", "x86_64", "kmod-20-21.el7.src.rpm", 91800, "LGPLv2+", "CentOS", "", "Libraries to handle kernel module loading and unloading", "d7dea2fe3482829a84830109bf4bcc97"},
			{intRef(), "libssh2", "1.4.3", "10.el7_2.1", "x86_64", "libssh2-1.4.3-10.el7_2.1.src.rpm", 341782, "BSD", "CentOS", "", "A library implementing the SSH2 protocol", "8ee7c5f75e2a2c85536f0f2282347545"},
			{intRef(), "curl", "7.29.0", "46.el7", "x86_64", "curl-7.29.0-46.el7.src.rpm", 540259, "MIT", "CentOS", "", "A utility for getting files from remote servers (FTP, HTTP, and others)", "401357a38601356be337ead6bf4470e0"},
			{intRef(), "libuser", "0.60", "9.el7", "x86_64", "libuser-0.60-9.el7.src.rpm", 1952592, "LGPLv2+", "CentOS", "", "A user and group account administration library", "8fdd92e668b91208dfd916ebdce8a297"},
			{intRef(2), "tar", "1.26", "34.el7", "x86_64", "tar-1.26-34.el7.src.rpm", 2838271, "GPLv3+", "CentOS", "", "A GNU file archiving program", "419ef9b4469e60d4d4d2599c1c8d79ba"},
			{intRef(), "acl", "2.2.51", "14.el7", "x86_64", "acl-2.2.51-14.el7.src.rpm", 201225, "GPLv2+", "CentOS", "", "Access control list utilities", "885771078c3fe1efa43e3e5e0f2b5246"},
			{intRef(), "ustr", "1.0.4", "16.el7", "x86_64", "ustr-1.0.4-16.el7.src.rpm", 285943, "MIT or LGPLv2+ or BSD", "CentOS", "", "String library, very low memory overhead, simple to import", "58bb2ebb64b048e24a4c7d5d51170f55"},
			{intRef(2), "shadow-utils", "4.1.5.1", "24.el7", "x86_64", "shadow-utils-4.1.5.1-24.el7.src.rpm", 3534490, "BSD and GPLv2+", "CentOS", "", "Utilities for managing accounts and shadow password files", "19741e43ee0b142d310393848d75239f"},
			{intRef(1), "hardlink", "1.0", "19.el7", "x86_64", "hardlink-1.0-19.el7.src.rpm", 16545, "GPL+", "CentOS", "", "Create a tree of hardlinks", "3a1bcd8e12bbdbced75d74b5919a039a"},
			{intRef(), "util-linux", "2.23.2", "52.el7_5.1", "x86_64", "util-linux-2.23.2-52.el7_5.1.src.rpm", 8642769, "GPLv2 and GPLv2+ and LGPLv2+ and BSD with advertising and Public Domain", "CentOS", "", "A collection of basic system utilities", "8c402cdaecc9f30a146aa82ff30c8aea"},
			{intRef(), "kpartx", "0.4.9", "119.el7_5.1", "x86_64", "device-mapper-multipath-0.4.9-119.el7_5.1.src.rpm", 41363, "GPL+", "CentOS", "", "Partition device manager for device-mapper devices", "fd61a26347cf0d090798768d7e952759"},
			{intRef(7), "device-mapper-libs", "1.02.146", "4.el7", "x86_64", "lvm2-2.02.177-4.el7.src.rpm", 400551, "LGPLv2", "CentOS", "", "Device-mapper shared library", "f0b566ce4fe22594b7f5f246870d4214"},
			{intRef(), "dracut", "033", "535.el7_5.1", "x86_64", "dracut-033-535.el7_5.1.src.rpm", 898198, "GPLv2+ and LGPLv2+", "CentOS", "", "Initramfs generator using udev", "22ffeb1421964a58b78b87eeb47014a7"},
			{intRef(1), "dbus-libs", "1.10.24", "7.el7", "x86_64", "dbus-1.10.24-7.el7.src.rpm", 362584, "(GPLv2+ or AFL) and GPLv2+", "CentOS", "", "Libraries for accessing D-BUS", "7c544dc4e3020620da275ddf1c0de95c"},
			{intRef(), "elfutils-default-yama-scope", "0.170", "4.el7", "noarch", "elfutils-0.170-4.el7.src.rpm", 1810, "GPLv2+ or LGPLv3+", "CentOS", "", "Default yama attach scope sysctl setting", "186adf73da2641da5799e54b4ccfed5d"},
			{intRef(), "dbus-glib", "0.100", "7.el7", "x86_64", "dbus-glib-0.100-7.el7.src.rpm", 301237, "AFL and GPLv2+", "CentOS", "", "GLib bindings for D-Bus", "fea831a1bb3627640677db4d441a65c6"},
			{intRef(), "python-libs", "2.7.5", "69.el7_5", "x86_64", "python-2.7.5-69.el7_5.src.rpm", 24713084, "Python", "CentOS", "", "Runtime libraries for Python", "879bca5c512f1199f3694bced2464d5c"},
			{intRef(), "libxml2-python", "2.9.1", "6.el7_2.3", "x86_64", "libxml2-2.9.1-6.el7_2.3.src.rpm", 1503050, "MIT", "CentOS", "", "Python bindings for the libxml2 library", "4af3055a710e118b3e7c49cb1df4e784"},
			{intRef(), "python-gobject-base", "3.22.0", "1.el7_4.1", "x86_64", "pygobject3-3.22.0-1.el7_4.1.src.rpm", 1123114, "LGPLv2+ and MIT", "CentOS", "", "Python 2 bindings for GObject Introspection base package", "e7f47d020d89e7ecc10aa699c0f8a4f0"},
			{intRef(), "yum-metadata-parser", "1.1.4", "10.el7", "x86_64", "yum-metadata-parser-1.1.4-10.el7.src.rpm", 58789, "GPLv2", "CentOS", "", "A fast metadata parser for yum", "929f0abd8c5671c69f86fce8d7cdd42d"},
			{intRef(), "python-pycurl", "7.19.0", "19.el7", "x86_64", "python-pycurl-7.19.0-19.el7.src.rpm", 241513, "LGPLv2+ or MIT", "CentOS", "", "A Python interface to libcurl", "d5152315d59bde1faaa203e68de6a64e"},
			{intRef(), "python-iniparse", "0.4", "9.el7", "noarch", "python-iniparse-0.4-9.el7.src.rpm", 115166, "MIT", "CentOS", "", "Python Module for Accessing and Modifying Configuration Data in INI files", "3140d9efca99c4dd5215383bab36bca5"},
			{intRef(), "python-chardet", "2.2.1", "1.el7_1", "noarch", "python-chardet-2.2.1-1.el7_1.src.rpm", 1156541, "LGPLv2", "CentOS", "", "Character encoding auto-detection in Python", "c70da920f85b12ecd25429913e3e861e"},
			{intRef(), "hostname", "3.13", "3.el7", "x86_64", "hostname-3.13-3.el7.src.rpm", 19449, "GPLv2+", "CentOS", "", "Utility to set/show the host name or domain name", "677ae7d92bb29445d1e0b66ecf5cab81"},
			{intRef(), "pth", "2.0.7", "23.el7", "x86_64", "pth-2.0.7-23.el7.src.rpm", 267851, "LGPLv2+", "CentOS", "", "The GNU Portable Threads library", "2d6939b73767d0f96b02677bcbeef2e2"},
			{intRef(), "gpgme", "1.3.2", "5.el7", "x86_64", "gpgme-1.3.2-5.el7.src.rpm", 547534, "LGPLv2+", "CentOS", "", "GnuPG Made Easy - high level crypto API", "d693313ae4184395e902c28ce87d4573"},
			{intRef(), "yum-plugin-fastestmirror", "1.1.31", "46.el7_5", "noarch", "yum-utils-1.1.31-46.el7_5.src.rpm", 53895, "GPLv2+", "CentOS", "", "Yum plugin which chooses fastest repository from a mirrorlist", "d9f1b3ff9204795855f0f6cf09ab40dd"},
			{intRef(32), "bind-license", "9.9.4", "61.el7_5.1", "noarch", "bind-9.9.4-61.el7_5.1.src.rpm", 26831, "ISC", "CentOS", "", "License of the BIND DNS suite", "8983808555b122a7220800907c368f23"},
			{intRef(), "yum-plugin-ovl", "1.1.31", "46.el7_5", "noarch", "yum-utils-1.1.31-46.el7_5.src.rpm", 22399, "GPLv2+", "CentOS", "", "Yum plugin to work around overlayfs issues", "5ee5adb1e57e68d35865f4c94181f473"},
			{intRef(), "passwd", "0.79", "4.el7", "x86_64", "passwd-0.79-4.el7.src.rpm", 429874, "BSD or GPL+", "CentOS", "", "An utility for setting or changing passwords using PAM", "bc5080b013d6a43fb00e57bba7517e0f"},
			{intRef(), "rootfiles", "8.1", "11.el7", "noarch", "rootfiles-8.1-11.el7.src.rpm", 599, "Public Domain", "CentOS", "", "The basic required files for the root user's directory", "b9b9eceee7ad38f868520e80f3404642"},
			{intRef(), "libgcc", "4.8.5", "36.el7_6.1", "x86_64", "gcc-4.8.5-36.el7_6.1.src.rpm", 179328, "GPLv3+ and GPLv3+ with exceptions and GPLv2+ with exceptions and LGPLv2+ and BSD", "CentOS", "", "GCC version 4.8 shared support library", "562e2ee43ee56c57acf69b14f6f1535b"},
			{intRef(), "glibc", "2.17", "260.el7_6.4", "x86_64", "glibc-2.17-260.el7_6.4.src.rpm", 14346480, "LGPLv2+ and LGPLv2+ with exceptions and GPLv2+", "CentOS", "", "The GNU libc libraries", "17e2b4067ebdf2993c799ddce8bb76e0"},
			{intRef(), "elfutils-libelf", "0.172", "2.el7", "x86_64", "elfutils-0.172-2.el7.src.rpm", 913443, "GPLv2+ or LGPLv3+", "CentOS", "", "Library to read and write ELF files", "383dff055bd0bc27b026d1359974e00e"},
			{intRef(), "rpm-libs", "4.11.3", "35.el7", "x86_64", "rpm-4.11.3-35.el7.src.rpm", 611536, "GPLv2+ and LGPLv2+ with exceptions", "CentOS", "", "Libraries for manipulating RPM packages", "2ac2673a85c45699ccf2e5d1a4eb2c4c"},
			{intRef(), "elfutils-libs", "0.172", "2.el7", "x86_64", "elfutils-0.172-2.el7.src.rpm", 793087, "GPLv2+ or LGPLv3+", "CentOS", "", "Libraries to handle compiled objects", "8d24c886e0655d29240085a75a5d7047"},
			{intRef(), "libmpc", "1.0.1", "3.el7", "x86_64", "libmpc-1.0.1-3.el7.src.rpm", 113833, "LGPLv3+ and GFDL", "CentOS", "", "C library for multiple precision complex arithmetic", "835abbe42dbf1e0f94d227d54c01f483"},
			{intRef(), "fipscheck", "1.4.1", "6.el7", "x86_64", "fipscheck-1.4.1-6.el7.src.rpm", 38839, "BSD", "CentOS", "", "A library for integrity verification of FIPS validated modules", "a2e0953263a6285a00476819997c1caf"},
			{intRef(), "libgomp", "4.8.5", "36.el7_6.1", "x86_64", "gcc-4.8.5-36.el7_6.1.src.rpm", 212184, "GPLv3+ and GPLv3+ with exceptions and GPLv2+ with exceptions and LGPLv2+ and BSD", "CentOS", "", "GCC OpenMP v3.0 shared support library", "35872c42fa6e4867151f828230a2636f"},
			{intRef(), "m4", "1.4.16", "10.el7", "x86_64", "m4-1.4.16-10.el7.src.rpm", 525707, "GPLv3+", "CentOS", "", "The GNU macro processor", "a71de093f3eba336d4fe4044fb6db349"},
			{intRef(), "apr-util", "1.5.2", "6.el7", "x86_64", "apr-util-1.5.2-6.el7.src.rpm", 198751, "ASL 2.0", "CentOS", "", "Apache Portable Runtime Utility library", "e4abdd79021510c0a5eb7fdea8389881"},
			{intRef(), "groff-base", "1.22.2", "8.el7", "x86_64", "groff-1.22.2-8.el7.src.rpm", 3453946, "GPLv3+ and GFDL and BSD and MIT", "CentOS", "", "Parts of the groff formatting system required to display manual pages", "591d2a12420abc9fa9934414fc0987f4"},
			{intRef(), "libunistring", "0.9.3", "9.el7", "x86_64", "libunistring-0.9.3-9.el7.src.rpm", 1145761, "LGPLv3+", "CentOS", "", "GNU Unicode string library", "b1b9755e18cb312740d96d89e61ef70e"},
			{intRef(), "libcroco", "0.6.12", "4.el7", "x86_64", "libcroco-0.6.12-4.el7.src.rpm", 320955, "LGPLv2", "CentOS", "", "A CSS2 parsing library", "8f570a2d6cb38d68f7c099b8130a1068"},
			{intRef(), "patch", "2.7.1", "10.el7_5", "x86_64", "patch-2.7.1-10.el7_5.src.rpm", 215449, "GPLv3+", "CentOS", "", "Utility for modifying/upgrading files", "3c1cd445c2a45f19b9bb46c19a4a943c"},
			{intRef(), "zip", "3.0", "11.el7", "x86_64", "zip-3.0-11.el7.src.rpm", 815173, "BSD", "CentOS", "", "A file compression and packaging utility compatible with PKZIP", "f49ae6270a053005b67e417123691ca7"},
			{intRef(), "gettext", "0.19.8.1", "2.el7", "x86_64", "gettext-0.19.8.1-2.el7.src.rpm", 5029340, "GPLv3+ and LGPLv2+", "CentOS", "", "GNU libraries and utilities for producing multi-lingual messages", "e8b96713d3d1d5080233abae6e9c5e10"},
			{intRef(), "perl-HTTP-Tiny", "0.033", "3.el7", "noarch", "perl-HTTP-Tiny-0.033-3.el7.src.rpm", 97210, "GPL+ or Artistic", "CentOS", "", "Small, simple, correct HTTP/1.1 client", "6133c7c1aee2bfeee269de4592b25884"},
			{intRef(), "perl-Pod-Perldoc", "3.20", "4.el7", "noarch", "perl-Pod-Perldoc-3.20-4.el7.src.rpm", 166910, "GPL+ or Artistic", "CentOS", "", "Look up Perl documentation in Pod format", "29a8a33b24ddf4e0e95ad8d2484cc277"},
			{intRef(), "perl-Text-ParseWords", "3.29", "4.el7", "noarch", "perl-Text-ParseWords-3.29-4.el7.src.rpm", 16431, "GPL+ or Artistic", "CentOS", "", "Parse text into an array of tokens or array of arrays", "58a96328cb0e6dd435d15af542467a38"},
			{intRef(), "perl-Pod-Usage", "1.63", "3.el7", "noarch", "perl-Pod-Usage-1.63-3.el7.src.rpm", 44671, "GPL+ or Artistic", "CentOS", "", "Print a usage message from embedded POD documentation", "1db34851ba13011739022f031438e15c"},
			{intRef(), "perl-Socket", "2.010", "4.el7", "x86_64", "perl-Socket-2.010-4.el7.src.rpm", 114497, "GPL+ or Artistic", "CentOS", "", "Networking constants and support functions", "5cda40abac38b41918352e9c75a1e651"},
			{intRef(4), "perl-Time-HiRes", "1.9725", "3.el7", "x86_64", "perl-Time-HiRes-1.9725-3.el7.src.rpm", 94069, "GPL+ or Artistic", "CentOS", "", "High resolution alarm, sleep, gettimeofday, interval timers", "43f7c93d02750221738a8c3bc845a1ca"},
			{intRef(), "perl-Carp", "1.26", "244.el7", "noarch", "perl-Carp-1.26-244.el7.src.rpm", 28276, "GPL+ or Artistic", "CentOS", "", "Alternative warn and die for modules", "0f9ffc1d8836311235869048667422e9"},
			{intRef(), "perl-Exporter", "5.68", "3.el7", "noarch", "perl-Exporter-5.68-3.el7.src.rpm", 56612, "GPL+ or Artistic", "CentOS", "", "Implements default import method for modules", "d6e7deb74b149e4b7aa23a998a127bb4"},
			{intRef(), "perl-Time-Local", "1.2300", "2.el7", "noarch", "perl-Time-Local-1.2300-2.el7.src.rpm", 44062, "GPL+ or Artistic", "CentOS", "", "Efficiently compute time from local and GMT time", "8dba0f2d49849dc3990cfc9e91d5ec6b"},
			{intRef(), "perl-threads-shared", "1.43", "6.el7", "x86_64", "perl-threads-shared-1.43-6.el7.src.rpm", 73972, "GPL+ or Artistic", "CentOS", "", "Perl extension for sharing data structures between threads", "cd3e1ec9dfe8b5e9121fcd53b41aa197"},
			{intRef(), "perl-File-Path", "2.09", "2.el7", "noarch", "perl-File-Path-2.09-2.el7.src.rpm", 50067, "GPL+ or Artistic", "CentOS", "", "Create or remove directory trees", "94d54ef6ce6893ae2a4400c4b0ce79b7"},
			{intRef(), "perl-Scalar-List-Utils", "1.27", "248.el7", "x86_64", "perl-Scalar-List-Utils-1.27-248.el7.src.rpm", 67994, "GPL+ or Artistic", "CentOS", "", "A selection of general-utility scalar and list subroutines", "2fb261c150fc06b1a978809c5a9c706e"},
			{intRef(), "perl-Getopt-Long", "2.40", "3.el7", "noarch", "perl-Getopt-Long-2.40-3.el7.src.rpm", 134846, "GPLv2+ or Artistic", "CentOS", "", "Extended processing of command line options", "e4fc3f2fed36897ae178ecd5d9475217"},
			{intRef(), "perl-Thread-Queue", "3.02", "2.el7", "noarch", "perl-Thread-Queue-3.02-2.el7.src.rpm", 27642, "GPL+ or Artistic", "CentOS", "", "Thread-safe queues", "29d052232cb1b86133bd838739672a15"},
			{intRef(), "perl-TermReadKey", "2.30", "20.el7", "x86_64", "perl-TermReadKey-2.30-20.el7.src.rpm", 59999, "(Copyright only) and (Artistic or GPL+)", "CentOS", "", "A perl module for simple terminal control", "48356fe925819f41d6446ffc09d1b21d"},
			{intRef(), "autoconf", "2.69", "11.el7", "noarch", "autoconf-2.69-11.el7.src.rpm", 2320371, "GPLv2+ and GFDL", "CentOS", "", "A GNU tool for automatically configuring source code", "24e721d7349c86c3df3cf4727e598a54"},
			{intRef(), "perl-Test-Harness", "3.28", "3.el7", "noarch", "perl-Test-Harness-3.28-3.el7.src.rpm", 607239, "GPL+ or Artistic", "CentOS", "", "Run Perl standard test scripts with statistics", "f131d5755e383c37135b023649a236b3"},
			{intRef(), "kernel-debug-devel", "3.10.0", "957.10.1.el7", "x86_64", "kernel-3.10.0-957.10.1.el7.src.rpm", 39270753, "GPLv2", "CentOS", "", "Development package for building kernel modules to match the debug kernel", "420c2ea8fef42dd04e53294121aed12a"},
			{intRef(), "boost-thread", "1.53.0", "27.el7", "x86_64", "boost-1.53.0-27.el7.src.rpm", 95658, "Boost and MIT and Python", "CentOS", "", "Run-Time component of boost thread library", "f93accf1d38b62ca79ba4424d7a603a8"},
			{intRef(), "openssh", "7.4p1", "16.el7", "x86_64", "openssh-7.4p1-16.el7.src.rpm", 1995364, "BSD", "CentOS", "", "An open source implementation of SSH protocol versions 1 and 2", "805b1fb17b3b7a41b2df9fd89b75e377"},
			{intRef(), "elfutils", "0.172", "2.el7", "x86_64", "elfutils-0.172-2.el7.src.rpm", 811817, "GPLv3+ and (GPLv2+ or LGPLv3+)", "CentOS", "", "A collection of utilities and DSOs to handle ELF files and DWARF data", "b405152d92211d83d371d791d702b883"},
			{intRef(), "systemd", "219", "62.el7_6.5", "x86_64", "systemd-219-62.el7_6.5.src.rpm", 24440903, "LGPLv2+ and MIT and GPLv2+", "CentOS", "", "A System and Service Manager", "20127ecb38e5f361019f50f3ae7bc8a7"},
			{intRef(), "rsync", "3.1.2", "4.el7", "x86_64", "rsync-3.1.2-4.el7.src.rpm", 834618, "GPLv3+", "CentOS", "", "A program for synchronizing files over a network", "a50dfc05a59066dea8ab23dda156a45a"},
			{intRef(), "file", "5.11", "35.el7", "x86_64", "file-5.11-35.el7.src.rpm", 67448, "BSD", "CentOS", "", "A utility for determining file types", "d993604425647d31a529f678397db7a4"},
			{intRef(), "dwz", "0.11", "3.el7", "x86_64", "dwz-0.11-3.el7.src.rpm", 225103, "GPLv2+ and GPLv3+", "CentOS", "", "DWARF optimization and duplicate removal tool", "28a047bc76d0ad80e6938a94898129ca"},
			{intRef(), "dyninst", "9.3.1", "2.el7", "x86_64", "dyninst-9.3.1-2.el7.src.rpm", 13049000, "LGPLv2+", "CentOS", "", "An API for Run-time Code Generation", "f32826510026b19cc65dfdfd0cdf8bf6"},
			{intRef(), "libstdc++-devel", "4.8.5", "36.el7_6.1", "x86_64", "gcc-4.8.5-36.el7_6.1.src.rpm", 8436804, "GPLv3+ and GPLv3+ with exceptions and GPLv2+ with exceptions and LGPLv2+ and BSD", "CentOS", "", "Header files and libraries for C++ development", "90dc72456e50813f4a311d17514e21eb"},
		}

		return toPackageInfo(pkgs)
	}

	// docker run --rm -it centos/httpd-24-centos7 bash
	// rpm -qa --queryformat "\{%{EPOCH}, \"%{NAME}\", \"%{VERSION}\", \"%{RELEASE}\", \"%{ARCH}\", \"%{SOURCERPM}\", %{SIZE}, \"%{LICENSE}\", \"%{VENDOR}\", \"\", \"%{SUMMARY}\", \"%{SIGMD5}\"\},\n" | sed "s/^{(none)/{intRef()/g" | sed -r 's/^\{([0-9]+),/{intRef(\1),/' | sed "s/(none)/0/g"
	CentOS7Httpd24 = func() []*PackageInfo {
		pkgs := []*commonPackageInfo{
			{intRef(32), "bind-license", "9.9.4", "73.el7_6", "noarch", "bind-9.9.4-73.el7_6.src.rpm", 26831, "ISC", "CentOS", "", "License of the BIND DNS suite", "98dd71aa6b527468317e3e96e1629ddb"},
			{intRef(), "bash", "4.2.46", "31.el7", "x86_64", "bash-4.2.46-31.el7.src.rpm", 3667773, "GPLv3+", "CentOS", "", "The GNU Bourne Again shell", "ed4d986a9753c79474e6ab726f40d864"},
			{intRef(), "glibc-common", "2.17", "260.el7_6.3", "x86_64", "glibc-2.17-260.el7_6.3.src.rpm", 120499441, "LGPLv2+ and LGPLv2+ with exceptions and GPLv2+", "CentOS", "", "Common binaries and locale data for glibc", "b6d5b5844c699f7d3474795ced771933"},
			{intRef(), "nss-softokn-freebl", "3.36.0", "5.el7_5", "x86_64", "nss-softokn-3.36.0-5.el7_5.src.rpm", 565628, "MPLv2.0", "CentOS", "", "Freebl library for the Network Security Services", "cf0360bbf0396334bb569646bbec25d3"},
			{intRef(), "filesystem", "3.2", "25.el7", "x86_64", "filesystem-3.2-25.el7.src.rpm", 0, "Public Domain", "CentOS", "", "The basic directory layout for a Linux system", "93e1c91a2dbd4d3ac61fb70a40a7c228"},
			{intRef(), "glibc", "2.17", "260.el7_6.3", "x86_64", "glibc-2.17-260.el7_6.3.src.rpm", 14346456, "LGPLv2+ and LGPLv2+ with exceptions and GPLv2+", "CentOS", "", "The GNU libc libraries", "0face5fe8edcb15b63b59d0445a74787"},
			{intRef(), "nspr", "4.19.0", "1.el7_5", "x86_64", "nspr-4.19.0-1.el7_5.src.rpm", 287728, "MPLv2.0", "CentOS", "", "Netscape Portable Runtime", "06e8af105f3a5832065a90c5eb12c064"},
			{intRef(), "popt", "1.13", "16.el7", "x86_64", "popt-1.13-16.el7.src.rpm", 88516, "MIT", "CentOS", "", "C library for parsing command line parameters", "87f5d6bac8f205fb14e2072e4b47ce23"},
			{intRef(), "libcom_err", "1.42.9", "13.el7", "x86_64", "e2fsprogs-1.42.9-13.el7.src.rpm", 60489, "MIT", "CentOS", "", "Common error description library", "d50ec3dda8af48a865da2d9754fa21bd"},
			{intRef(), "libcap", "2.22", "9.el7", "x86_64", "libcap-2.22-9.el7.src.rpm", 111445, "LGPLv2+", "CentOS", "", "Library for getting and setting POSIX.1e capabilities", "b4cc4b26a62dbbd66143755c54a59938"},
			{intRef(), "libstdc++", "4.8.5", "36.el7", "x86_64", "gcc-4.8.5-36.el7.src.rpm", 1077442, "GPLv3+ and GPLv3+ with exceptions and GPLv2+ with exceptions and LGPLv2+ and BSD", "CentOS", "", "GNU Standard C++ Library", "9012d4414e5017937f3b58c73731b054"},
			{intRef(), "info", "5.1", "5.el7", "x86_64", "texinfo-5.1-5.el7.src.rpm", 494630, "GPLv3+", "CentOS", "", "A stand-alone TTY-based reader for GNU texinfo documentation", "e1f75e40cf956ed3efe7b49c5f6939ec"},
			{intRef(), "gawk", "4.0.2", "4.el7_3.1", "x86_64", "gawk-4.0.2-4.el7_3.1.src.rpm", 2435978, "GPLv3+ and GPL and LGPLv3+ and LGPL and BSD", "CentOS", "", "The GNU version of the awk text processing utility", "5efa8feaeb12d57cda1a9bcd9c8af917"},
			{intRef(), "libselinux", "2.5", "14.1.el7", "x86_64", "libselinux-2.5-14.1.el7.src.rpm", 217874, "Public Domain", "CentOS", "", "SELinux library and simple utilities", "36462311e16b3aafc9f29bbaa00bfb6e"},
			{intRef(), "grep", "2.20", "3.el7", "x86_64", "grep-2.20-3.el7.src.rpm", 1195131, "GPLv3+", "CentOS", "", "Pattern matching utilities", "92bc957d1c4fc3764e134d8f82544abe"},
			{intRef(), "keyutils-libs", "1.5.8", "3.el7", "x86_64", "keyutils-1.5.8-3.el7.src.rpm", 42138, "GPLv2+ and LGPLv2+", "CentOS", "", "Key utilities library", "e31fa37a714787d3d72e0a9316c3a85b"},
			{intRef(), "libverto", "0.2.5", "4.el7", "x86_64", "libverto-0.2.5-4.el7.src.rpm", 23060, "MIT", "CentOS", "", "Main loop abstraction library", "341bae036f22ce087486f4d9e180caee"},
			{intRef(), "p11-kit-trust", "0.23.5", "3.el7", "x86_64", "p11-kit-0.23.5-3.el7.src.rpm", 437261, "BSD", "CentOS", "", "System trust module from p11-kit", "2dc2f7ab212401098e8821ce23f21741"},
			{intRef(), "krb5-libs", "1.15.1", "37.el7_6", "x86_64", "krb5-1.15.1-37.el7_6.src.rpm", 2200985, "MIT", "CentOS", "", "The non-admin shared libraries used by Kerberos 5", "240e6ada762f6cdc46d8d6a3bdbadd49"},
			{intRef(), "xz-libs", "5.2.2", "1.el7", "x86_64", "xz-5.2.2-1.el7.src.rpm", 239967, "LGPLv2+", "CentOS", "", "Libraries for decoding LZMA compression", "7701c542dc8fcd5a38ee52e8786d7cf4"},
			{intRef(), "libdb", "5.3.21", "24.el7", "x86_64", "libdb-5.3.21-24.el7.src.rpm", 1858008, "BSD and LGPLv2 and Sleepycat", "CentOS", "", "The Berkeley DB database library for C", "24fdfda30ad8e1cc7ebc744eaf85eacf"},
			{intRef(), "libgpg-error", "1.12", "3.el7", "x86_64", "libgpg-error-1.12-3.el7.src.rpm", 350865, "LGPLv2+", "CentOS", "", "Library for error values used by GnuPG components", "a3933d4ce43c07fa9305e8765d78d099"},
			{intRef(), "libgcrypt", "1.5.3", "14.el7", "x86_64", "libgcrypt-1.5.3-14.el7.src.rpm", 597727, "LGPLv2+", "CentOS", "", "A general-purpose cryptography library", "88c64dc7c3aed6a156e4dee5def38300"},
			{intRef(), "lua", "5.1.4", "15.el7", "x86_64", "lua-5.1.4-15.el7.src.rpm", 640319, "MIT", "CentOS", "", "Powerful light-weight programming language", "d5973eb7ca2bb665f5fbb7141f2e27a3"},
			{intRef(), "libuuid", "2.23.2", "59.el7", "x86_64", "util-linux-2.23.2-59.el7.src.rpm", 20326, "BSD", "CentOS", "", "Universally unique ID library", "71fa9d535bbf86cd9dabb2ef6d6f37ae"},
			{intRef(), "libmount", "2.23.2", "59.el7", "x86_64", "util-linux-2.23.2-59.el7.src.rpm", 278173, "LGPLv2+", "CentOS", "", "Device mounting library", "c513d12f33c07d836ce3ef13e8eba980"},
			{intRef(), "shared-mime-info", "1.8", "4.el7", "x86_64", "shared-mime-info-1.8-4.el7.src.rpm", 2379317, "GPLv2+", "CentOS", "", "Shared MIME information database", "167b39408a8c5705f135e968174e2229"},
			{intRef(), "gzip", "1.5", "10.el7", "x86_64", "gzip-1.5-10.el7.src.rpm", 250440, "GPLv3+ and GFDL", "CentOS", "", "The GNU data compression program", "adcb4df7dfe7866e38b5bf42d54d63ae"},
			{intRef(1), "findutils", "4.5.11", "6.el7", "x86_64", "findutils-4.5.11-6.el7.src.rpm", 1855882, "GPLv3+", "CentOS", "", "The GNU versions of find utilities (find and xargs)", "162b03172dee0f90443197d424bc9be0"},
			{intRef(), "diffutils", "3.3", "4.el7", "x86_64", "diffutils-3.3-4.el7.src.rpm", 1065157, "GPLv3+", "CentOS", "", "A GNU collection of diff utilities", "d8c60275b4c6d54e8bf7ee19cf0c4f2b"},
			{intRef(), "expat", "2.1.0", "10.el7_3", "x86_64", "expat-2.1.0-10.el7_3.src.rpm", 208315, "MIT", "CentOS", "", "An XML parser library", "9c5fdceee3715ad8a1b3a9dd711d5b79"},
			{intRef(), "audit-libs", "2.8.4", "4.el7", "x86_64", "audit-2.8.4-4.el7.src.rpm", 256362, "LGPLv2+", "CentOS", "", "Dynamic library for libaudit", "a57039d27b473fc766688886d53ecce2"},
			{intRef(), "pam", "1.1.8", "22.el7", "x86_64", "pam-1.1.8-22.el7.src.rpm", 2630324, "BSD and GPLv2+", "CentOS", "", "An extensible library which provides authentication for applications", "dce355546c72f1425dc04fd64c9bc664"},
			{intRef(), "nss-softokn", "3.36.0", "5.el7_5", "x86_64", "nss-softokn-3.36.0-5.el7_5.src.rpm", 1130670, "MPLv2.0", "CentOS", "", "Network Security Services Softoken Module", "7b8391d5b6843722982a616a8542d58f"},
			{intRef(), "nss", "3.36.0", "7.1.el7_6", "x86_64", "nss-3.36.0-7.1.el7_6.src.rpm", 2425024, "MPLv2.0", "CentOS", "", "Network Security Services", "208ddd9381e8c7b512ce763bae2e7628"},
			{intRef(), "libassuan", "2.1.0", "3.el7", "x86_64", "libassuan-2.1.0-3.el7.src.rpm", 155391, "LGPLv2+ and GPLv3+", "CentOS", "", "GnuPG IPC library", "c4c94aadc9da58f576b4ef238d0136cb"},
			{intRef(), "nss-tools", "3.36.0", "7.1.el7_6", "x86_64", "nss-3.36.0-7.1.el7_6.src.rpm", 2069587, "MPLv2.0", "CentOS", "", "Tools for the Network Security Services", "53bab8a320b55204b17fc880c326ae26"},
			{intRef(), "gobject-introspection", "1.56.1", "1.el7", "x86_64", "gobject-introspection-1.56.1-1.el7.src.rpm", 854207, "GPLv2+, LGPLv2+, MIT", "CentOS", "", "Introspection system for GObject-based libraries", "35d1071e217a3adca0dc2679a049ca04"},
			{intRef(), "libdb-utils", "5.3.21", "24.el7", "x86_64", "libdb-5.3.21-24.el7.src.rpm", 326487, "BSD and LGPLv2 and Sleepycat", "CentOS", "", "Command line tools for managing Berkeley DB databases", "d06077335f27858ecb06ad48fa2e812d"},
			{intRef(), "kmod-libs", "20", "23.el7", "x86_64", "kmod-20-23.el7.src.rpm", 91848, "LGPLv2+", "CentOS", "", "Libraries to handle kernel module loading and unloading", "8e1fe13756e455026ef91936cf555599"},
			{intRef(), "libssh2", "1.4.3", "12.el7", "x86_64", "libssh2-1.4.3-12.el7.src.rpm", 341886, "BSD", "CentOS", "", "A library implementing the SSH2 protocol", "8629b98c219f01a14e71d0d8bd57965d"},
			{intRef(), "curl", "7.29.0", "51.el7", "x86_64", "curl-7.29.0-51.el7.src.rpm", 540293, "MIT", "CentOS", "", "A utility for getting files from remote servers (FTP, HTTP, and others)", "2008575cc1fdf94eedba4e920908cb38"},
			{intRef(), "rpm", "4.11.3", "35.el7", "x86_64", "rpm-4.11.3-35.el7.src.rpm", 2622266, "GPLv2+", "CentOS", "", "The RPM package management system", "ba17adc338f442da8a4598244965e521"},
			{intRef(), "libuser", "0.60", "9.el7", "x86_64", "libuser-0.60-9.el7.src.rpm", 1952592, "LGPLv2+", "CentOS", "", "A user and group account administration library", "8fdd92e668b91208dfd916ebdce8a297"},
			{intRef(2), "tar", "1.26", "35.el7", "x86_64", "tar-1.26-35.el7.src.rpm", 2838510, "GPLv3+", "CentOS", "", "A GNU file archiving program", "28f3087e5b512d8cc05e3a8a65727686"},
			{intRef(), "acl", "2.2.51", "14.el7", "x86_64", "acl-2.2.51-14.el7.src.rpm", 201225, "GPLv2+", "CentOS", "", "Access control list utilities", "885771078c3fe1efa43e3e5e0f2b5246"},
			{intRef(), "libsmartcols", "2.23.2", "59.el7", "x86_64", "util-linux-2.23.2-59.el7.src.rpm", 164608, "LGPLv2+", "CentOS", "", "Formatting library for ls-like programs.", "7610ea5954b653ea569cc4e187f613db"},
			{intRef(), "libsemanage", "2.5", "14.el7", "x86_64", "libsemanage-2.5-14.el7.src.rpm", 302329, "LGPLv2+", "CentOS", "", "SELinux binary policy manipulation library", "95dadf0074c4bbd2d63b0c6d5e6922fa"},
			{intRef(), "libutempter", "1.1.6", "4.el7", "x86_64", "libutempter-1.1.6-4.el7.src.rpm", 49749, "LGPLv2+", "CentOS", "", "A privileged helper for utmp/wtmp updates", "8e28747e6a84b84af7d78c84cd8cd5e8"},
			{intRef(), "qrencode-libs", "3.4.1", "3.el7", "x86_64", "qrencode-3.4.1-3.el7.src.rpm", 126732, "LGPLv2+", "CentOS", "", "QR Code encoding library - Shared libraries", "ac1de0c46bc1886e12ea80771951cde2"},
			{intRef(), "python-libs", "2.7.5", "76.el7", "x86_64", "python-2.7.5-76.el7.src.rpm", 24713067, "Python", "CentOS", "", "Runtime libraries for Python", "f7aa64ba20585799cc289eddc634d4d8"},
			{intRef(), "python-gobject-base", "3.22.0", "1.el7_4.1", "x86_64", "pygobject3-3.22.0-1.el7_4.1.src.rpm", 1123114, "LGPLv2+ and MIT", "CentOS", "", "Python 2 bindings for GObject Introspection base package", "e7f47d020d89e7ecc10aa699c0f8a4f0"},
			{intRef(), "pyliblzma", "0.5.3", "11.el7", "x86_64", "pyliblzma-0.5.3-11.el7.src.rpm", 190112, "LGPLv3+", "CentOS", "", "Python bindings for lzma", "a09aa8e806a3f6285e9fe765d69d3ee9"},
			{intRef(), "python-urlgrabber", "3.10", "9.el7", "noarch", "python-urlgrabber-3.10-9.el7.src.rpm", 503349, "LGPLv2+", "CentOS", "", "A high-level cross-protocol url-grabber", "00eda8adf14af162fd4dfc89ab3d5e45"},
			{intRef(), "pyxattr", "0.5.1", "5.el7", "x86_64", "pyxattr-0.5.1-5.el7.src.rpm", 63304, "LGPLv2+", "CentOS", "", "Extended attributes library wrapper for Python", "e85121da681760c48a2af6ed8aa31b4e"},
			{intRef(), "python-kitchen", "1.1.1", "5.el7", "noarch", "python-kitchen-1.1.1-5.el7.src.rpm", 1465161, "LGPLv2+", "CentOS", "", "Small, useful pieces of code to make python coding easier", "917912ebf6164135f1f9871150432977"},
			{intRef(), "json-c", "0.11", "4.el7_0", "x86_64", "json-c-0.11-4.el7_0.src.rpm", 65593, "MIT", "CentOS", "", "A JSON implementation in C", "62bcb642b82d3b3aeb99ba9ba1b40715"},
			{intRef(), "procps-ng", "3.3.10", "23.el7", "x86_64", "procps-ng-3.3.10-23.el7.src.rpm", 760565, "GPL+ and GPLv2 and GPLv2+ and GPLv3+ and LGPLv2+", "CentOS", "", "System and process monitoring utilities", "b7dd3953621742926579dc26c71f5b52"},
			{intRef(7), "device-mapper", "1.02.149", "10.el7_6.3", "x86_64", "lvm2-2.02.180-10.el7_6.3.src.rpm", 339987, "GPLv2", "CentOS", "", "Device mapper utility", "c3f668105fef62e64e7e3fd43c3e558f"},
			{intRef(), "cryptsetup-libs", "2.0.3", "3.el7", "x86_64", "cryptsetup-2.0.3-3.el7.src.rpm", 1219591, "GPLv2+ and LGPLv2+", "CentOS", "", "Cryptsetup shared library", "61b5b318bfcd411aec021e25d9b76070"},
			{intRef(), "kmod", "20", "23.el7", "x86_64", "kmod-20-23.el7.src.rpm", 243559, "GPLv2+", "CentOS", "", "Linux kernel module management utilities", "99ea42add73605e4c070a5db69a5818c"},
			{intRef(), "systemd-libs", "219", "62.el7_6.5", "x86_64", "systemd-219-62.el7_6.5.src.rpm", 1263456, "LGPLv2+ and MIT", "CentOS", "", "systemd libraries", "9577513987ddb8ff723a0b94367fc8c7"},
			{intRef(), "systemd", "219", "62.el7_6.5", "x86_64", "systemd-219-62.el7_6.5.src.rpm", 24440903, "LGPLv2+ and MIT and GPLv2+", "CentOS", "", "A System and Service Manager", "20127ecb38e5f361019f50f3ae7bc8a7"},
			{intRef(), "elfutils-default-yama-scope", "0.172", "2.el7", "noarch", "elfutils-0.172-2.el7.src.rpm", 1810, "GPLv2+ or LGPLv3+", "CentOS", "", "Default yama attach scope sysctl setting", "5f2326e3ab4ee0002a6de12404aa1668"},
			{intRef(), "iputils", "20160308", "10.el7", "x86_64", "iputils-20160308-10.el7.src.rpm", 343497, "BSD and GPLv2+", "CentOS", "", "Network monitoring tools including ping", "dfa219e6db737a1eb1b5cb23b532e858"},
			{intRef(), "dbus-glib", "0.100", "7.el7", "x86_64", "dbus-glib-0.100-7.el7.src.rpm", 301237, "AFL and GPLv2+", "CentOS", "", "GLib bindings for D-Bus", "fea831a1bb3627640677db4d441a65c6"},
			{intRef(), "pth", "2.0.7", "23.el7", "x86_64", "pth-2.0.7-23.el7.src.rpm", 267851, "LGPLv2+", "CentOS", "", "The GNU Portable Threads library", "2d6939b73767d0f96b02677bcbeef2e2"},
			{intRef(), "rpm-build-libs", "4.11.3", "35.el7", "x86_64", "rpm-4.11.3-35.el7.src.rpm", 166776, "GPLv2+ and LGPLv2+ with exceptions", "CentOS", "", "Libraries for building and signing RPM packages", "aba616fc3d1a25cfc77eb27db8610457"},
			{intRef(), "gpgme", "1.3.2", "5.el7", "x86_64", "gpgme-1.3.2-5.el7.src.rpm", 547534, "LGPLv2+", "CentOS", "", "GnuPG Made Easy - high level crypto API", "d693313ae4184395e902c28ce87d4573"},
			{intRef(), "yum-plugin-fastestmirror", "1.1.31", "50.el7", "noarch", "yum-utils-1.1.31-50.el7.src.rpm", 53895, "GPLv2+", "CentOS", "", "Yum plugin which chooses fastest repository from a mirrorlist", "27a1d31b36b337a4ab254926cd2665fd"},
			{intRef(), "yum-utils", "1.1.31", "50.el7", "noarch", "yum-utils-1.1.31-50.el7.src.rpm", 345277, "GPLv2+", "CentOS", "", "Utilities based around the yum package manager", "3a596d781e91319c17ba84f82256bf5c"},
			{intRef(), "passwd", "0.79", "4.el7", "x86_64", "passwd-0.79-4.el7.src.rpm", 429874, "BSD or GPL+", "CentOS", "", "An utility for setting or changing passwords using PAM", "bc5080b013d6a43fb00e57bba7517e0f"},
			{intRef(2), "vim-minimal", "7.4.160", "5.el7", "x86_64", "vim-7.4.160-5.el7.src.rpm", 917744, "Vim", "CentOS", "", "A minimal version of the VIM editor", "64a183937eac1409f64824544ef5220d"},
			{intRef(), "gpg-pubkey", "f4a80eb5", "53a7ff4b", "", "", 0, "pubkey", "", "", "gpg(CentOS-7 Key (CentOS 7 Official Signing Key) <security@centos.org>)", ""},
			{intRef(), "lzo", "2.06", "8.el7", "x86_64", "lzo-2.06-8.el7.src.rpm", 165360, "GPLv2+", "CentOS", "", "Data compression library with very fast (de)compression", "fb57116a5f1f0e494abb8db7e7bfb249"},
			{intRef(), "libcroco", "0.6.12", "4.el7", "x86_64", "libcroco-0.6.12-4.el7.src.rpm", 320955, "LGPLv2", "CentOS", "", "A CSS2 parsing library", "8f570a2d6cb38d68f7c099b8130a1068"},
			{intRef(), "libarchive", "3.1.2", "10.el7_2", "x86_64", "libarchive-3.1.2-10.el7_2.src.rpm", 745291, "BSD", "CentOS", "", "A library for handling streaming archive formats", "b06fb8340a6a96b935488dd04d12e645"},
			{intRef(), "gettext", "0.19.8.1", "2.el7", "x86_64", "gettext-0.19.8.1-2.el7.src.rpm", 5029340, "GPLv3+ and LGPLv2+", "CentOS", "", "GNU libraries and utilities for producing multi-lingual messages", "e8b96713d3d1d5080233abae6e9c5e10"},
			{intRef(), "groff-base", "1.22.2", "8.el7", "x86_64", "groff-1.22.2-8.el7.src.rpm", 3453946, "GPLv3+ and GFDL and BSD and MIT", "CentOS", "", "Parts of the groff formatting system required to display manual pages", "591d2a12420abc9fa9934414fc0987f4"},
			{intRef(), "centos-release-scl-rh", "2", "3.el7.centos", "noarch", "centos-release-scl-rh-2-3.el7.centos.src.rpm", 20015, "GPLv2", "CentOS", "", "Software collections from the CentOS SCLo SIG (upstream scl only)", "d991ba7e51ac40b70cfbf486715b5e71"},
			{intRef(), "epel-release", "7", "11", "noarch", "epel-release-7-11.src.rpm", 24834, "GPLv2", "Fedora Project", "", "Extra Packages for Enterprise Linux repository configuration", "049a2f5109a5050367cc4a8ae8f62c33"},
			{intRef(), "gpg-pubkey", "352c64e5", "52ae6884", "", "", 0, "pubkey", "", "", "gpg(Fedora EPEL (7) <epel@fedoraproject.org>)", ""},
			{intRef(), "apr", "1.4.8", "3.el7_4.1", "x86_64", "apr-1.4.8-3.el7_4.1.src.rpm", 226686, "ASL 2.0 and BSD with advertising and ISC and BSD", "CentOS", "", "Apache Portable Runtime library", "0534c5c4a8da91bf8e4a4152c0383b46"},
			{intRef(), "libtool-ltdl", "2.4.2", "22.el7_3", "x86_64", "libtool-2.4.2-22.el7_3.src.rpm", 67814, "LGPLv2+", "CentOS", "", "Runtime libraries for GNU Libtool Dynamic Module Loader", "ff22ccbfa94b59e371c3073511c533c9"},
			{intRef(), "apr-util", "1.5.2", "6.el7", "x86_64", "apr-util-1.5.2-6.el7.src.rpm", 198751, "ASL 2.0", "CentOS", "", "Apache Portable Runtime Utility library", "e4abdd79021510c0a5eb7fdea8389881"},
			{intRef(32), "bind-libs", "9.9.4", "73.el7_6", "x86_64", "bind-9.9.4-73.el7_6.src.rpm", 2720960, "ISC", "CentOS", "", "Libraries used by the BIND DNS packages", "2837cf220fa8ec07d8c79a66dec3d206"},
			{intRef(), "lasso", "2.5.1", "2.el7", "x86_64", "lasso-2.5.1-2.el7.src.rpm", 570546, "GPLv2+", "CentOS", "", "Liberty Alliance Single Sign On", "26df3fc7b46ee8fe0d0c26c39ad476f1"},
			{intRef(), "perl-HTTP-Tiny", "0.033", "3.el7", "noarch", "perl-HTTP-Tiny-0.033-3.el7.src.rpm", 97210, "GPL+ or Artistic", "CentOS", "", "Small, simple, correct HTTP/1.1 client", "6133c7c1aee2bfeee269de4592b25884"},
			{intRef(), "perl-Pod-Perldoc", "3.20", "4.el7", "noarch", "perl-Pod-Perldoc-3.20-4.el7.src.rpm", 166910, "GPL+ or Artistic", "CentOS", "", "Look up Perl documentation in Pod format", "29a8a33b24ddf4e0e95ad8d2484cc277"},
			{intRef(), "perl-Text-ParseWords", "3.29", "4.el7", "noarch", "perl-Text-ParseWords-3.29-4.el7.src.rpm", 16431, "GPL+ or Artistic", "CentOS", "", "Parse text into an array of tokens or array of arrays", "58a96328cb0e6dd435d15af542467a38"},
			{intRef(), "perl-Pod-Usage", "1.63", "3.el7", "noarch", "perl-Pod-Usage-1.63-3.el7.src.rpm", 44671, "GPL+ or Artistic", "CentOS", "", "Print a usage message from embedded POD documentation", "1db34851ba13011739022f031438e15c"},
			{intRef(4), "perl-macros", "5.16.3", "294.el7_6", "x86_64", "perl-5.16.3-294.el7_6.src.rpm", 5134, "GPL+ or Artistic", "CentOS", "", "Macros for rpmbuild", "7c63786a9688ac528e398b27e86198d8"},
			{intRef(), "perl-Exporter", "5.68", "3.el7", "noarch", "perl-Exporter-5.68-3.el7.src.rpm", 56612, "GPL+ or Artistic", "CentOS", "", "Implements default import method for modules", "d6e7deb74b149e4b7aa23a998a127bb4"},
			{intRef(), "perl-Time-Local", "1.2300", "2.el7", "noarch", "perl-Time-Local-1.2300-2.el7.src.rpm", 44062, "GPL+ or Artistic", "CentOS", "", "Efficiently compute time from local and GMT time", "8dba0f2d49849dc3990cfc9e91d5ec6b"},
			{intRef(), "perl-Carp", "1.26", "244.el7", "noarch", "perl-Carp-1.26-244.el7.src.rpm", 28276, "GPL+ or Artistic", "CentOS", "", "Alternative warn and die for modules", "0f9ffc1d8836311235869048667422e9"},
			{intRef(), "perl-threads-shared", "1.43", "6.el7", "x86_64", "perl-threads-shared-1.43-6.el7.src.rpm", 73972, "GPL+ or Artistic", "CentOS", "", "Perl extension for sharing data structures between threads", "cd3e1ec9dfe8b5e9121fcd53b41aa197"},
			{intRef(), "perl-Scalar-List-Utils", "1.27", "248.el7", "x86_64", "perl-Scalar-List-Utils-1.27-248.el7.src.rpm", 67994, "GPL+ or Artistic", "CentOS", "", "A selection of general-utility scalar and list subroutines", "2fb261c150fc06b1a978809c5a9c706e"},
			{intRef(), "perl-File-Temp", "0.23.01", "3.el7", "noarch", "perl-File-Temp-0.23.01-3.el7.src.rpm", 158781, "GPL+ or Artistic", "CentOS", "", "Return name and handle of a temporary file safely", "396b0a1af3a203e1cb33c98814d5a6d8"},
			{intRef(), "perl-Filter", "1.49", "3.el7", "x86_64", "perl-Filter-1.49-3.el7.src.rpm", 148475, "GPL+ or Artistic", "CentOS", "", "Perl source filters", "66b1205b2eef2a89137e530741ed3710"},
			{intRef(), "perl-Getopt-Long", "2.40", "3.el7", "noarch", "perl-Getopt-Long-2.40-3.el7.src.rpm", 134846, "GPLv2+ or Artistic", "CentOS", "", "Extended processing of command line options", "e4fc3f2fed36897ae178ecd5d9475217"},
			{intRef(1), "make", "3.82", "23.el7", "x86_64", "make-3.82-23.el7.src.rpm", 1160684, "GPLv2+", "CentOS", "", "A GNU tool which simplifies the build process for users", "9e5ab17428b36e59a9d071c1fd1c81de"},
			{intRef(), "libcgroup", "0.41", "20.el7", "x86_64", "libcgroup-0.41-20.el7.src.rpm", 137210, "LGPLv2+", "CentOS", "", "Library to control and monitor control groups", "eb76a54af92009d1c72e8efd9d0ac434"},
			{intRef(), "jansson", "2.10", "1.el7", "x86_64", "jansson-2.10-1.el7.src.rpm", 74998, "MIT", "CentOS", "", "C library for encoding, decoding and manipulating JSON data", "47add619dede2cd5e4079de7250ed141"},
			{intRef(), "libselinux-utils", "2.5", "14.1.el7", "x86_64", "libselinux-2.5-14.1.el7.src.rpm", 172645, "Public Domain", "CentOS", "", "SELinux libselinux utilies", "4480bb426dabc45b7b5fd5f4f6a45a52"},
			{intRef(), "libsemanage-python", "2.5", "14.el7", "x86_64", "libsemanage-2.5-14.el7.src.rpm", 451817, "LGPLv2+", "CentOS", "", "semanage python bindings for libsemanage", "44967d0d996064547841a4572ef0f9fb"},
			{intRef(), "python-IPy", "0.75", "6.el7", "noarch", "python-IPy-0.75-6.el7.src.rpm", 121946, "BSD", "CentOS", "", "Python module for handling IPv4 and IPv6 Addresses and Networks", "21be4d5d974076d9817cd9cc616eef0f"},
			{intRef(), "centos-logos", "70.0.6", "3.el7.centos", "noarch", "centos-logos-70.0.6-3.el7.centos.src.rpm", 22548401, "Copyright © 2014 The CentOS Project.  All rights reserved.", "CentOS", "", "CentOS-related icons and pictures", "288c573e626f84c20c6830e833da985c"},
			{intRef(), "audit-libs-python", "2.8.4", "4.el7", "x86_64", "audit-2.8.4-4.el7.src.rpm", 327265, "LGPLv2+", "CentOS", "", "Python bindings for libaudit", "665eb826f1113adf9f656760f589bd55"},
			{intRef(), "httpd24-runtime", "1.1", "18.el7", "x86_64", "httpd24-1.1-18.el7.src.rpm", 20989, "GPLv2+", "CentOS", "", "Package that handles httpd24 Software Collection.", "e9cf25c326a6a03eff6c0d6c7b31a93e"},
			{intRef(), "httpd24-libnghttp2", "1.7.1", "7.el7", "x86_64", "httpd24-nghttp2-1.7.1-7.el7.src.rpm", 142738, "MIT", "CentOS", "", "A library implementing the HTTP/2 protocol", "2b27a006c985eaf902fef4866d4c21cd"},
			{intRef(), "httpd24-httpd", "2.4.34", "7.el7", "x86_64", "httpd24-httpd-2.4.34-7.el7.src.rpm", 4739300, "ASL 2.0", "CentOS", "", "Apache HTTP Server", "4011d451af79d2177345b9112fd36730"},
			{intRef(1), "httpd24-mod_ssl", "2.4.34", "7.el7", "x86_64", "httpd24-httpd-2.4.34-7.el7.src.rpm", 241612, "ASL 2.0", "CentOS", "", "SSL/TLS module for the Apache HTTP Server", "a2b99bfd142913c2b2c2a286f74602ac"},
			{intRef(), "nss_wrapper", "1.1.5", "1.el7", "x86_64", "nss_wrapper-1.1.5-1.el7.src.rpm", 75009, "BSD", "Fedora Project", "", "A wrapper for the user, group and hosts NSS API", "8041aec46ff5013a13657f362b4a334d"},
			{intRef(), "libgcc", "4.8.5", "36.el7", "x86_64", "gcc-4.8.5-36.el7.src.rpm", 179328, "GPLv3+ and GPLv3+ with exceptions and GPLv2+ with exceptions and LGPLv2+ and BSD", "CentOS", "", "GCC version 4.8 shared support library", "96274f04f5bda49af612ee45d8d964c6"},
			{intRef(), "tzdata", "2018i", "1.el7", "noarch", "tzdata-2018i-1.el7.src.rpm", 2007607, "Public Domain", "CentOS", "", "Timezone data", "d85d7853af12b5daefb67cb6fadb8c15"},
			{intRef(), "ncurses-base", "5.9", "14.20130511.el7_4", "noarch", "ncurses-5.9-14.20130511.el7_4.src.rpm", 223432, "MIT", "CentOS", "", "Descriptions of common terminals", "c955fdfe1f8f20b1053a34918e44bd58"},
			{intRef(), "chkconfig", "1.7.4", "1.el7", "x86_64", "chkconfig-1.7.4-1.el7.src.rpm", 779531, "GPLv2", "CentOS", "", "A system tool for maintaining the /etc/rc*.d hierarchy", "e126177e9ac5841deca96f4e15340613"},
			{intRef(), "ncurses", "5.9", "14.20130511.el7_4", "x86_64", "ncurses-5.9-14.20130511.el7_4.src.rpm", 439378, "MIT", "CentOS", "", "Ncurses support utilities", "8ca93e2831102818759a22e22e871268"},
			{intRef(), "setup", "2.8.71", "10.el7", "noarch", "setup-2.8.71-10.el7.src.rpm", 696893, "Public Domain", "CentOS", "", "A set of system configuration and setup files", "df4a32b192e93cc82ed77e10d204ce98"},
			{intRef(), "basesystem", "10.0", "7.el7.centos", "noarch", "basesystem-10.0-7.el7.centos.src.rpm", 0, "Public Domain", "CentOS", "", "The skeleton package which defines a simple CentOS Linux system", "778eb3d19a934f42d1354ed795d06b4f"},
			{intRef(), "zlib", "1.2.7", "18.el7", "x86_64", "zlib-1.2.7-18.el7.src.rpm", 185294, "zlib and Boost", "CentOS", "", "The compression and decompression library", "063a85f8271de9b41e9ac9657962e3ec"},
			{intRef(), "nss-util", "3.36.0", "1.1.el7_6", "x86_64", "nss-util-3.36.0-1.1.el7_6.src.rpm", 199008, "MPLv2.0", "CentOS", "", "Network Security Services Utilities Library", "eacfab0995b7a9d221edd5d23700dff3"},
			{intRef(), "libffi", "3.0.13", "18.el7", "x86_64", "libffi-3.0.13-18.el7.src.rpm", 47766, "MIT and Public Domain", "CentOS", "", "A portable foreign function interface library", "f05f7a3749342b925fa10ca7de2f0116"},
			{intRef(), "libattr", "2.4.46", "13.el7", "x86_64", "attr-2.4.46-13.el7.src.rpm", 19896, "LGPLv2+", "CentOS", "", "Dynamic library for extended attribute support", "209b05c7b77a33a4cc176077945833e8"},
			{intRef(), "libacl", "2.2.51", "14.el7", "x86_64", "acl-2.2.51-14.el7.src.rpm", 37056, "LGPLv2+", "CentOS", "", "Dynamic library for access control list support", "c3243d728de0ad95e0c05230e6463bdd"},
			{intRef(), "ncurses-libs", "5.9", "14.20130511.el7_4", "x86_64", "ncurses-5.9-14.20130511.el7_4.src.rpm", 1028216, "MIT", "CentOS", "", "Ncurses libraries", "bba677eda5184b5127b7817d2c1dec12"},
			{intRef(), "libsepol", "2.5", "10.el7", "x86_64", "libsepol-2.5-10.el7.src.rpm", 686640, "LGPLv2+", "CentOS", "", "SELinux binary policy manipulation library", "cb2c225685a559d68fdb0e07f9b488a2"},
			{intRef(), "pcre", "8.32", "17.el7", "x86_64", "pcre-8.32-17.el7.src.rpm", 1475532, "BSD", "CentOS", "", "Perl-compatible regular expression library", "04a1926cc29eef0f79f8183eb7907024"},
			{intRef(), "sed", "4.2.2", "5.el7", "x86_64", "sed-4.2.2-5.el7.src.rpm", 601208, "GPLv3+", "CentOS", "", "A GNU stream text editor", "4d696dbd3061493258490f40c625adba"},
			{intRef(), "p11-kit", "0.23.5", "3.el7", "x86_64", "p11-kit-0.23.5-3.el7.src.rpm", 1337825, "BSD", "CentOS", "", "Library for loading and sharing PKCS#11 modules", "9a0f49ca3b6d58aeb05780003cae38f1"},
			{intRef(1), "gmp", "6.0.0", "15.el7", "x86_64", "gmp-6.0.0-15.el7.src.rpm", 657046, "LGPLv3+ or GPLv2+", "CentOS", "", "A GNU arbitrary precision library", "4cd60ebf94766a46c590e17036ece777"},
			{intRef(), "libtasn1", "4.10", "1.el7", "x86_64", "libtasn1-4.10-1.el7.src.rpm", 424486, "GPLv3+ and LGPLv2+", "CentOS", "", "The ASN.1 library used in GNUTLS", "5bd1b0b4094fac08713f986572a8625e"},
			{intRef(), "ca-certificates", "2018.2.22", "70.0.el7_5", "noarch", "ca-certificates-2018.2.22-70.0.el7_5.src.rpm", 973960, "Public Domain", "CentOS", "", "The Mozilla CA root certificate bundle", "a95373fc5273e7c757a240316d4430b9"},
			{intRef(), "coreutils", "8.22", "23.el7", "x86_64", "coreutils-8.22-23.el7.src.rpm", 14588674, "GPLv3+", "CentOS", "", "A set of basic GNU tools commonly used in shell scripts", "904a2aeeeaa0d833a0dff696c89bd7f3"},
			{intRef(), "centos-release", "7", "6.1810.2.el7.centos", "x86_64", "centos-release-7-6.1810.2.el7.centos.src.rpm", 41271, "GPLv2", "CentOS", "", "CentOS Linux release file", "9f661612be55e8bb73f78163a41df816"},
			{intRef(), "bzip2-libs", "1.0.6", "13.el7", "x86_64", "bzip2-1.0.6-13.el7.src.rpm", 70093, "BSD", "CentOS", "", "Libraries for applications using bzip2", "0486f0b02e0caa4195761266798e7a26"},
			{intRef(), "elfutils-libelf", "0.172", "2.el7", "x86_64", "elfutils-0.172-2.el7.src.rpm", 913443, "GPLv2+ or LGPLv3+", "CentOS", "", "Library to read and write ELF files", "383dff055bd0bc27b026d1359974e00e"},
			{intRef(), "libxml2", "2.9.1", "6.el7_2.3", "x86_64", "libxml2-2.9.1-6.el7_2.3.src.rpm", 1710062, "MIT", "CentOS", "", "Library providing XML and HTML support", "e439b8934c4e9576eccfbc60d3cd714f"},
			{intRef(), "readline", "6.2", "10.el7", "x86_64", "readline-6.2-10.el7.src.rpm", 460464, "GPLv3+", "CentOS", "", "A library for editing typed command lines", "b552128087365a58b3566631c678752f"},
			{intRef(), "cpio", "2.11", "27.el7", "x86_64", "cpio-2.11-27.el7.src.rpm", 689335, "GPLv3+", "CentOS", "", "A GNU archiving program", "32498692f82296d51663586f08cebafc"},
			{intRef(), "libblkid", "2.23.2", "59.el7", "x86_64", "util-linux-2.23.2-59.el7.src.rpm", 265973, "LGPLv2+", "CentOS", "", "Block device ID library", "1f215e7cf6a8a08fe16ff5c4edf0f68b"},
			{intRef(), "glib2", "2.56.1", "2.el7", "x86_64", "glib2-2.56.1-2.el7.src.rpm", 12166425, "LGPLv2+", "CentOS", "", "A library of handy utility functions", "5ba70ca2db724338484aedc7f9d8a4b9"},
			{intRef(), "sqlite", "3.7.17", "8.el7", "x86_64", "sqlite-3.7.17-8.el7.src.rpm", 814231, "Public Domain", "CentOS", "", "Library that implements an embeddable SQL database engine", "0ceb9434c5a9d281399583087cfe889b"},
			{intRef(), "cracklib", "2.9.0", "11.el7", "x86_64", "cracklib-2.9.0-11.el7.src.rpm", 209610, "LGPLv2+", "CentOS", "", "A password-checking library", "a973d883d6785904e49266428bb3ed90"},
			{intRef(), "libidn", "1.28", "4.el7", "x86_64", "libidn-1.28-4.el7.src.rpm", 630407, "LGPLv2+ and GPLv3+ and GFDL", "CentOS", "", "Internationalized Domain Name support library", "3d582106ac1032c59c1cd4236e5d6415"},
			{intRef(), "libcap-ng", "0.7.5", "4.el7", "x86_64", "libcap-ng-0.7.5-4.el7.src.rpm", 50510, "LGPLv2+", "CentOS", "", "An alternate posix capabilities library", "84efa5ca78408f78e606e7c8d1816759"},
			{intRef(), "cracklib-dicts", "2.9.0", "11.el7", "x86_64", "cracklib-2.9.0-11.el7.src.rpm", 9389116, "LGPLv2+", "CentOS", "", "The standard CrackLib dictionaries", "da657b0facf181893f9a38437092dbbf"},
			{intRef(), "libpwquality", "1.2.3", "5.el7", "x86_64", "libpwquality-1.2.3-5.el7.src.rpm", 332421, "BSD or GPLv2+", "CentOS", "", "A library for password generation and password quality checking", "e40a69f0a1803dfa3736aba2c8ea1fb4"},
			{intRef(), "nss-pem", "1.0.3", "5.el7", "x86_64", "nss-pem-1.0.3-5.el7.src.rpm", 205539, "MPLv1.1", "CentOS", "", "PEM file reader for Network Security Services (NSS)", "24d8ee5cd00822ebc0ceca3e9f0bb906"},
			{intRef(), "nss-sysinit", "3.36.0", "7.1.el7_6", "x86_64", "nss-3.36.0-7.1.el7_6.src.rpm", 14067, "MPLv2.0", "CentOS", "", "System NSS Initialization", "27ff88c4a26101b75c51b31b5e85c435"},
			{intRef(), "xz", "5.2.2", "1.el7", "x86_64", "xz-5.2.2-1.el7.src.rpm", 798130, "LGPLv2+", "CentOS", "", "LZMA compression utilities", "3e6f05008f7af7931aa8257964527f1a"},
			{intRef(), "file-libs", "5.11", "35.el7", "x86_64", "file-5.11-35.el7.src.rpm", 3076831, "BSD", "CentOS", "", "Libraries for applications using libmagic", "3037fd1f781186bcb0fb919b2aa78f54"},
			{intRef(), "lz4", "1.7.5", "2.el7", "x86_64", "lz4-1.7.5-2.el7.src.rpm", 366872, "GPLv2+ and BSD", "CentOS", "", "Extremely fast compression algorithm", "89e45994eabac0ba4ccae7c1005ea06c"},
			{intRef(1), "pkgconfig", "0.27.1", "4.el7", "x86_64", "pkgconfig-0.27.1-4.el7.src.rpm", 105522, "GPLv2+", "CentOS", "", "A tool for determining compilation options", "ef58117c9c617599952c0642ff5e8d00"},
			{intRef(), "cyrus-sasl-lib", "2.1.26", "23.el7", "x86_64", "cyrus-sasl-2.1.26-23.el7.src.rpm", 396911, "BSD with advertising", "CentOS", "", "Shared libraries needed by applications which use Cyrus SASL", "6d06f1337b143c25bc7358255d433006"},
			{intRef(), "binutils", "2.27", "34.base.el7", "x86_64", "binutils-2.27-34.base.el7.src.rpm", 25150000, "GPLv3+", "CentOS", "", "A GNU collection of binary utilities", "b9772c7ef8848f4b5372fa8f52222a6a"},
			{intRef(), "libcurl", "7.29.0", "51.el7", "x86_64", "curl-7.29.0-51.el7.src.rpm", 435192, "MIT", "CentOS", "", "A library for getting files from web servers", "998abd47ebced6ef98be2020cafcf569"},
			{intRef(), "rpm-libs", "4.11.3", "35.el7", "x86_64", "rpm-4.11.3-35.el7.src.rpm", 611536, "GPLv2+ and LGPLv2+ with exceptions", "CentOS", "", "Libraries for manipulating RPM packages", "2ac2673a85c45699ccf2e5d1a4eb2c4c"},
			{intRef(), "openldap", "2.4.44", "21.el7_6", "x86_64", "openldap-2.4.44-21.el7_6.src.rpm", 1037424, "OpenLDAP", "CentOS", "", "LDAP support libraries", "0ccf548b74498ba6b9762498609de967"},
			{intRef(), "pinentry", "0.8.1", "17.el7", "x86_64", "pinentry-0.8.1-17.el7.src.rpm", 159929, "GPLv2+", "CentOS", "", "Collection of simple PIN or passphrase entry dialogs", "740fabe008451f11fee22ff086fb13d3"},
			{intRef(), "ustr", "1.0.4", "16.el7", "x86_64", "ustr-1.0.4-16.el7.src.rpm", 285943, "MIT or LGPLv2+ or BSD", "CentOS", "", "String library, very low memory overhead, simple to import", "58bb2ebb64b048e24a4c7d5d51170f55"},
			{intRef(2), "shadow-utils", "4.1.5.1", "25.el7", "x86_64", "shadow-utils-4.1.5.1-25.el7.src.rpm", 3540592, "BSD and GPLv2+", "CentOS", "", "Utilities for managing accounts and shadow password files", "22c762981f93de691a7f74a7259a1906"},
			{intRef(1), "hardlink", "1.0", "19.el7", "x86_64", "hardlink-1.0-19.el7.src.rpm", 16545, "GPL+", "CentOS", "", "Create a tree of hardlinks", "3a1bcd8e12bbdbced75d74b5919a039a"},
			{intRef(), "gdbm", "1.10", "8.el7", "x86_64", "gdbm-1.10-8.el7.src.rpm", 184322, "GPLv3+", "CentOS", "", "A GNU set of database routines which use extensible hashing", "770bd49f3898dc1cc831fb9ec12d569a"},
			{intRef(), "python", "2.7.5", "76.el7", "x86_64", "python-2.7.5-76.el7.src.rpm", 80907, "Python", "CentOS", "", "An interpreted, interactive, object-oriented programming language", "9882effd17f2a65048dc45ed140ba617"},
			{intRef(), "libxml2-python", "2.9.1", "6.el7_2.3", "x86_64", "libxml2-2.9.1-6.el7_2.3.src.rpm", 1503050, "MIT", "CentOS", "", "Python bindings for the libxml2 library", "4af3055a710e118b3e7c49cb1df4e784"},
			{intRef(), "yum-metadata-parser", "1.1.4", "10.el7", "x86_64", "yum-metadata-parser-1.1.4-10.el7.src.rpm", 58789, "GPLv2", "CentOS", "", "A fast metadata parser for yum", "929f0abd8c5671c69f86fce8d7cdd42d"},
			{intRef(), "python-pycurl", "7.19.0", "19.el7", "x86_64", "python-pycurl-7.19.0-19.el7.src.rpm", 241513, "LGPLv2+ or MIT", "CentOS", "", "A Python interface to libcurl", "d5152315d59bde1faaa203e68de6a64e"},
			{intRef(), "python-iniparse", "0.4", "9.el7", "noarch", "python-iniparse-0.4-9.el7.src.rpm", 115166, "MIT", "CentOS", "", "Python Module for Accessing and Modifying Configuration Data in INI files", "3140d9efca99c4dd5215383bab36bca5"},
			{intRef(), "python-chardet", "2.2.1", "1.el7_1", "noarch", "python-chardet-2.2.1-1.el7_1.src.rpm", 1156541, "LGPLv2", "CentOS", "", "Character encoding auto-detection in Python", "c70da920f85b12ecd25429913e3e861e"},
			{intRef(), "hostname", "3.13", "3.el7", "x86_64", "hostname-3.13-3.el7.src.rpm", 19449, "GPLv2+", "CentOS", "", "Utility to set/show the host name or domain name", "677ae7d92bb29445d1e0b66ecf5cab81"},
			{intRef(), "util-linux", "2.23.2", "59.el7", "x86_64", "util-linux-2.23.2-59.el7.src.rpm", 8447471, "GPLv2 and GPLv2+ and LGPLv2+ and BSD with advertising and Public Domain", "CentOS", "", "A collection of basic system utilities", "6768a3c060a549635213eb4f9e32f08e"},
			{intRef(), "kpartx", "0.4.9", "123.el7", "x86_64", "device-mapper-multipath-0.4.9-123.el7.src.rpm", 41363, "GPL+", "CentOS", "", "Partition device manager for device-mapper devices", "aecde17836ffb541d265eeaeb9ebd1c8"},
			{intRef(7), "device-mapper-libs", "1.02.149", "10.el7_6.3", "x86_64", "lvm2-2.02.180-10.el7_6.3.src.rpm", 400607, "LGPLv2", "CentOS", "", "Device-mapper shared library", "32edc5c3d11c6bb1577f1c0cf8b43a72"},
			{intRef(), "dracut", "033", "554.el7", "x86_64", "dracut-033-554.el7.src.rpm", 903230, "GPLv2+ and LGPLv2+", "CentOS", "", "Initramfs generator using udev", "57e8bd40745c0cba95d9c3f65acb2205"},
			{intRef(), "elfutils-libs", "0.172", "2.el7", "x86_64", "elfutils-0.172-2.el7.src.rpm", 793087, "GPLv2+ or LGPLv3+", "CentOS", "", "Libraries to handle compiled objects", "8d24c886e0655d29240085a75a5d7047"},
			{intRef(1), "dbus-libs", "1.10.24", "12.el7", "x86_64", "dbus-1.10.24-12.el7.src.rpm", 362560, "(GPLv2+ or AFL) and GPLv2+", "CentOS", "", "Libraries for accessing D-BUS", "76ed3fc760940227749d71884cfcaa33"},
			{intRef(1), "dbus", "1.10.24", "12.el7", "x86_64", "dbus-1.10.24-12.el7.src.rpm", 595210, "(GPLv2+ or AFL) and GPLv2+", "CentOS", "", "D-BUS message bus", "0e7061cc90fda4f111245bee037271e4"},
			{intRef(), "dbus-python", "1.1.1", "9.el7", "x86_64", "dbus-python-1.1.1-9.el7.src.rpm", 848122, "MIT", "CentOS", "", "D-Bus Python Bindings", "6be4da91886c59ae25e5255ebff0a0a9"},
			{intRef(), "gnupg2", "2.0.22", "5.el7_5", "x86_64", "gnupg2-2.0.22-5.el7_5.src.rpm", 6637796, "GPLv3+", "CentOS", "", "Utility for secure communication and data storage", "a293dfefd6596bfe5c54ba68377e334e"},
			{intRef(), "rpm-python", "4.11.3", "35.el7", "x86_64", "rpm-4.11.3-35.el7.src.rpm", 149898, "GPLv2+", "CentOS", "", "Python bindings for apps which will manipulate RPM packages", "ad03ebb739fc7309edd09eba58ab7ee3"},
			{intRef(), "pygpgme", "0.3", "9.el7", "x86_64", "pygpgme-0.3-9.el7.src.rpm", 197501, "LGPLv2+", "CentOS", "", "Python module for working with OpenPGP messages", "67b8c720f59a063adbbca2392bac0281"},
			{intRef(), "yum", "3.4.3", "161.el7.centos", "noarch", "yum-3.4.3-161.el7.centos.src.rpm", 5824869, "GPLv2+", "CentOS", "", "RPM package installer/updater/manager", "ae05719cfd4e3b80b3de83da0e7ba4bf"},
			{intRef(), "yum-plugin-ovl", "1.1.31", "50.el7", "noarch", "yum-utils-1.1.31-50.el7.src.rpm", 22399, "GPLv2+", "CentOS", "", "Yum plugin to work around overlayfs issues", "5c82294b6b298d58a036b719cc667e7c"},
			{intRef(), "rootfiles", "8.1", "11.el7", "noarch", "rootfiles-8.1-11.el7.src.rpm", 599, "Public Domain", "CentOS", "", "The basic required files for the root user's directory", "b9b9eceee7ad38f868520e80f3404642"},
			{intRef(), "libgomp", "4.8.5", "36.el7_6.1", "x86_64", "gcc-4.8.5-36.el7_6.1.src.rpm", 212184, "GPLv3+ and GPLv3+ with exceptions and GPLv2+ with exceptions and LGPLv2+ and BSD", "CentOS", "", "GCC OpenMP v3.0 shared support library", "35872c42fa6e4867151f828230a2636f"},
			{intRef(), "libunistring", "0.9.3", "9.el7", "x86_64", "libunistring-0.9.3-9.el7.src.rpm", 1145761, "LGPLv3+", "CentOS", "", "GNU Unicode string library", "b1b9755e18cb312740d96d89e61ef70e"},
			{intRef(), "gettext-libs", "0.19.8.1", "2.el7", "x86_64", "gettext-0.19.8.1-2.el7.src.rpm", 1546952, "LGPLv2+ and GPLv3+", "CentOS", "", "Libraries for gettext", "3164b6e314d214a982d34c7c77d89e16"},
			{intRef(), "bsdtar", "3.1.2", "10.el7_2", "x86_64", "libarchive-3.1.2-10.el7_2.src.rpm", 110459, "BSD", "CentOS", "", "Manipulate tape archives", "ea6cdf65424903f1ab6eb8d33ed7ed31"},
			{intRef(), "unzip", "6.0", "19.el7", "x86_64", "unzip-6.0-19.el7.src.rpm", 373986, "BSD", "CentOS", "", "A utility for unpacking zip files", "7a396b648ba0120404fe2ef171b79681"},
			{intRef(), "scl-utils", "20130529", "19.el7", "x86_64", "scl-utils-20130529-19.el7.src.rpm", 25141, "GPLv2+", "CentOS", "", "Utilities for alternative packaging", "c8feffb6b8bbcf34c2f20ad60ae1cf41"},
			{intRef(), "centos-release-scl", "2", "3.el7.centos", "noarch", "centos-release-scl-2-3.el7.centos.src.rpm", 20042, "GPLv2", "CentOS", "", "Software collections from the CentOS SCLo SIG", "1985cc482b2f414166e33f22808d944f"},
			{intRef(), "gpg-pubkey", "f2ee9d55", "560cfc0a", "", "", 0, "pubkey", "", "", "gpg(CentOS SoftwareCollections SIG (https://wiki.centos.org/SpecialInterestGroup/SCLo) <security@centos.org>)", ""},
			{intRef(1), "openssl-libs", "1.0.2k", "16.el7_6.1", "x86_64", "openssl-1.0.2k-16.el7_6.1.src.rpm", 3204708, "OpenSSL", "CentOS", "", "A general purpose cryptography library with TLS implementation", "f22c63c9514e9f24f023d69bc1f7a959"},
			{intRef(), "libxslt", "1.1.28", "5.el7", "x86_64", "libxslt-1.1.28-5.el7.src.rpm", 497582, "MIT", "CentOS", "", "Library providing the Gnome XSLT engine", "e901ff455779767496a18d1a5f43816a"},
			{intRef(), "xmlsec1", "1.2.20", "7.el7_4", "x86_64", "xmlsec1-1.2.20-7.el7_4.src.rpm", 568638, "MIT", "CentOS", "", "Library providing support for \"XML Signature\" and \"XML Encryption\" standards", "e9a66a76bc5ea38e581df367dd8bb8d9"},
			{intRef(), "GeoIP", "1.5.0", "13.el7", "x86_64", "GeoIP-1.5.0-13.el7.src.rpm", 3912607, "LGPLv2+ and GPLv2+ and CC-BY-SA", "CentOS", "", "Library for country/city/organization to IP address or hostname mapping", "20e073cea8bc347151eceae084e7e7f1"},
			{intRef(), "xmlsec1-openssl", "1.2.20", "7.el7_4", "x86_64", "xmlsec1-1.2.20-7.el7_4.src.rpm", 245128, "MIT", "CentOS", "", "OpenSSL crypto plugin for XML Security Library", "176f351e1b0f4d0bcab524bbd47d4e97"},
			{intRef(1), "perl-parent", "0.225", "244.el7", "noarch", "perl-parent-0.225-244.el7.src.rpm", 8141, "GPL+ or Artistic", "CentOS", "", "Establish an ISA relationship with base classes at compile time", "d902f1a3426caa19ccbccf09263c54f6"},
			{intRef(), "perl-podlators", "2.5.1", "3.el7", "noarch", "perl-podlators-2.5.1-3.el7.src.rpm", 287679, "GPL+ or Artistic", "CentOS", "", "Format POD source into various output formats", "8e070dd716e3adf3c297291accaa4348"},
			{intRef(1), "perl-Pod-Escapes", "1.04", "294.el7_6", "noarch", "perl-5.16.3-294.el7_6.src.rpm", 21091, "GPL+ or Artistic", "CentOS", "", "Perl module for resolving POD escape sequences", "ad4d7216b06eb48942c30d4bc2b09755"},
			{intRef(), "perl-Encode", "2.51", "7.el7", "x86_64", "perl-Encode-2.51-7.el7.src.rpm", 10176350, "GPL+ or Artistic", "CentOS", "", "Character encodings in Perl", "713ce27ff4434807ff7060b333ed8873"},
			{intRef(4), "perl-libs", "5.16.3", "294.el7_6", "x86_64", "perl-5.16.3-294.el7_6.src.rpm", 1647344, "GPL+ or Artistic", "CentOS", "", "The libraries for the perl runtime", "669be9d414caaf85799b0dfa644c4e81"},
			{intRef(), "perl-threads", "1.87", "4.el7", "x86_64", "perl-threads-1.87-4.el7.src.rpm", 98615, "GPL+ or Artistic", "CentOS", "", "Perl interpreter-based threads", "29739d2f766c8bd85e85de9f8ccb7367"},
			{intRef(), "perl-constant", "1.27", "2.el7", "noarch", "perl-constant-1.27-2.el7.src.rpm", 26364, "GPL+ or Artistic", "CentOS", "", "Perl pragma to declare constants", "f3ecfa1ce48e13496322e341a3f0b72c"},
			{intRef(), "perl-Socket", "2.010", "4.el7", "x86_64", "perl-Socket-2.010-4.el7.src.rpm", 114497, "GPL+ or Artistic", "CentOS", "", "Networking constants and support functions", "5cda40abac38b41918352e9c75a1e651"},
			{intRef(4), "perl-Time-HiRes", "1.9725", "3.el7", "x86_64", "perl-Time-HiRes-1.9725-3.el7.src.rpm", 94069, "GPL+ or Artistic", "CentOS", "", "High resolution alarm, sleep, gettimeofday, interval timers", "43f7c93d02750221738a8c3bc845a1ca"},
			{intRef(), "perl-PathTools", "3.40", "5.el7", "x86_64", "perl-PathTools-3.40-5.el7.src.rpm", 174131, "(GPL+ or Artistic) and BSD", "CentOS", "", "PathTools Perl module (Cwd, File::Spec)", "c597987071b0f81a0029667e83e1ecae"},
			{intRef(), "perl-Storable", "2.45", "3.el7", "x86_64", "perl-Storable-2.45-3.el7.src.rpm", 181031, "GPL+ or Artistic", "CentOS", "", "Persistence for Perl data structures", "25787b4b609f3ee50d1c7f33d29e0102"},
			{intRef(), "perl-File-Path", "2.09", "2.el7", "noarch", "perl-File-Path-2.09-2.el7.src.rpm", 50067, "GPL+ or Artistic", "CentOS", "", "Create or remove directory trees", "94d54ef6ce6893ae2a4400c4b0ce79b7"},
			{intRef(1), "perl-Pod-Simple", "3.28", "4.el7", "noarch", "perl-Pod-Simple-3.28-4.el7.src.rpm", 538320, "GPL+ or Artistic", "CentOS", "", "Framework for parsing POD documentation", "8e5edeb936a154537062b776eb41cb7d"},
			{intRef(4), "perl", "5.16.3", "294.el7_6", "x86_64", "perl-5.16.3-294.el7_6.src.rpm", 23552128, "(GPL+ or Artistic) and (GPLv2+ or Artistic) and Copyright Only and MIT and Public Domain and UCD", "CentOS", "", "Practical Extraction and Report Language", "547b1d620cedc114b9a1a681c1f2a4bd"},
			{intRef(1), "openssl", "1.0.2k", "16.el7_6.1", "x86_64", "openssl-1.0.2k-16.el7_6.1.src.rpm", 833657, "OpenSSL", "CentOS", "", "Utilities from the general purpose cryptography library with TLS implementation", "3039d54f5d5482ad72948ae5c9b06f24"},
			{intRef(), "cmake", "2.8.12.2", "2.el7", "x86_64", "cmake-2.8.12.2-2.el7.src.rpm", 28036363, "BSD and MIT and zlib", "CentOS", "", "Cross-platform make system", "fb733f4838e9c493cd3dc4f1242659a2"},
			{intRef(), "libselinux-python", "2.5", "14.1.el7", "x86_64", "libselinux-2.5-14.1.el7.src.rpm", 603236, "Public Domain", "CentOS", "", "SELinux python bindings for libselinux", "cfbad05c1a681e59c0b59d2bfa6663ab"},
			{intRef(), "policycoreutils", "2.5", "29.el7_6.1", "x86_64", "policycoreutils-2.5-29.el7_6.1.src.rpm", 5321515, "GPLv2", "CentOS", "", "SELinux policy core utilities", "5b157e1d9bb389d4df8a002a785ede4b"},
			{intRef(), "setools-libs", "3.3.8", "4.el7", "x86_64", "setools-3.3.8-4.el7.src.rpm", 1917790, "LGPLv2", "CentOS", "", "Policy analysis support libraries for SELinux", "d25e0add9591f438d7e1b6d313f67352"},
			{intRef(), "checkpolicy", "2.5", "8.el7", "x86_64", "checkpolicy-2.5-8.el7.src.rpm", 1288327, "GPLv2", "CentOS", "", "SELinux policy compiler", "a42deadeddfdef88930f5ebcb6e21c8e"},
			{intRef(), "mailcap", "2.1.41", "2.el7", "noarch", "mailcap-2.1.41-2.el7.src.rpm", 63360, "Public Domain and MIT", "CentOS", "", "Helper application and MIME type associations for file types", "d20a25f8d834c1cac92727b8aa8e0f72"},
			{intRef(), "policycoreutils-python", "2.5", "29.el7_6.1", "x86_64", "policycoreutils-2.5-29.el7_6.1.src.rpm", 1299284, "GPLv2", "CentOS", "", "SELinux policy core python utilities", "5127183afef4a2a5d327a2cdffa09a7b"},
			{intRef(), "httpd24-libcurl", "7.61.1", "1.el7", "x86_64", "httpd24-curl-7.61.1-1.el7.src.rpm", 545704, "MIT", "CentOS", "", "A library for getting files from web servers", "05a2f8f385dfc114a46b8b08c676d88e"},
			{intRef(), "httpd24-httpd-tools", "2.4.34", "7.el7", "x86_64", "httpd24-httpd-2.4.34-7.el7.src.rpm", 177148, "ASL 2.0", "CentOS", "", "Tools for use with the Apache HTTP Server", "60c720e48321e50f1788ce5bfbf0e1d6"},
			{intRef(), "httpd24-mod_auth_mellon", "0.13.1", "2.el7", "x86_64", "httpd24-mod_auth_mellon-0.13.1-2.el7.src.rpm", 1108011, "GPLv2+", "CentOS", "", "A SAML 2.0 authentication module for the Apache Httpd Server", "9bf4aaf047a3510cf44d949b9f27eed9"},
			{intRef(), "httpd24", "1.1", "18.el7", "x86_64", "httpd24-1.1-18.el7.src.rpm", 0, "GPLv2+", "CentOS", "", "Package that installs httpd24", "d0d4d214432bf7e782efb96ffa33ca65"},
			{intRef(32), "bind-utils", "9.9.4", "73.el7_6", "x86_64", "bind-9.9.4-73.el7_6.src.rpm", 441412, "ISC", "CentOS", "", "Utilities for querying DNS name servers", "db9fc7150faa212f39fa8a0dfeac5aaa"},
		}

		return toPackageInfo(pkgs)
	}

	// docker run --rm -it centos:7 bash
	// yum groupinstall -y "Development tools"
	// yum install -y rpm-build redhat-rpm-config asciidoc hmaccalc perl-ExtUtils-Embed pesign xmlto
	// yum install -y audit-libs-devel binutils-devel elfutils-devel elfutils-libelf-devel java-devel
	// yum install -y ncurses-devel newt-devel numactl-devel pciutils-devel python-devel zlib-devel
	// yum install -y net-tools bc
	// rpm -qa --queryformat "\{%{EPOCH}, \"%{NAME}\", \"%{VERSION}\", \"%{RELEASE}\", \"%{ARCH}\", \"%{SOURCERPM}\", %{SIZE}, \"%{LICENSE}\", \"%{VENDOR}\", \"\", \"%{SUMMARY}\", \"%{SIGMD5}\"\},\n" | sed "s/^{(none)/{intRef()/g" | sed -r 's/^\{([0-9]+),/{intRef(\1),/' | sed "s/(none)/0/g"
	CentOS7Many = func() []*PackageInfo {
		pkgs := []*commonPackageInfo{
			{intRef(), "gnutls", "3.3.29", "9.el7_6", "x86_64", "gnutls-3.3.29-9.el7_6.src.rpm", 2097819, "GPLv3+ and LGPLv2+", "CentOS", "", "A TLS protocol implementation", "f82fa52300da8811286430201915afce"},
			{intRef(), "nss-softokn-freebl", "3.36.0", "5.el7_5", "x86_64", "nss-softokn-3.36.0-5.el7_5.src.rpm", 565628, "MPLv2.0", "CentOS", "", "Freebl library for the Network Security Services", "cf0360bbf0396334bb569646bbec25d3"},
			{intRef(), "openssh-clients", "7.4p1", "16.el7", "x86_64", "openssh-7.4p1-16.el7.src.rpm", 2651616, "BSD", "CentOS", "", "An open source SSH client applications", "4f9a34658e76fe533402c89ee43fec18"},
			{intRef(), "neon", "0.30.0", "3.el7", "x86_64", "neon-0.30.0-3.el7.src.rpm", 567772, "LGPLv2+", "CentOS", "", "An HTTP and WebDAV client library", "0c9ebc3ab96eed7a01fa4a1f7099de88"},
			{intRef(), "mokutil", "15", "2.el7.centos", "x86_64", "shim-signed-15-2.el7.centos.src.rpm", 83521, "BSD", "CentOS", "", "Utilities for managing Secure Boot/MoK keys.", "0149c24490b240a4db17b50acafa6746"},
			{intRef(), "popt", "1.13", "16.el7", "x86_64", "popt-1.13-16.el7.src.rpm", 88516, "MIT", "CentOS", "", "C library for parsing command line parameters", "87f5d6bac8f205fb14e2072e4b47ce23"},
			{intRef(), "glibc-headers", "2.17", "260.el7_6.4", "x86_64", "glibc-2.17-260.el7_6.4.src.rpm", 2338328, "LGPLv2+ and LGPLv2+ with exceptions and GPLv2+", "CentOS", "", "Header files for development using standard C libraries.", "10577e5ef1696cce687913ba612698fc"},
			{intRef(), "libcap", "2.22", "9.el7", "x86_64", "libcap-2.22-9.el7.src.rpm", 111445, "LGPLv2+", "CentOS", "", "Library for getting and setting POSIX.1e capabilities", "b4cc4b26a62dbbd66143755c54a59938"},
			{intRef(), "systemtap-devel", "3.3", "3.el7", "x86_64", "systemtap-3.3-3.el7.src.rpm", 7910669, "GPLv2+", "CentOS", "", "Programmable system-wide instrumentation system - development headers, tools", "5ca8a6b02cb40f4df3fdba8ea25da170"},
			{intRef(), "ncurses-libs", "5.9", "14.20130511.el7_4", "x86_64", "ncurses-5.9-14.20130511.el7_4.src.rpm", 1028216, "MIT", "CentOS", "", "Ncurses libraries", "bba677eda5184b5127b7817d2c1dec12"},
			{intRef(), "gettext-devel", "0.19.8.1", "2.el7", "x86_64", "gettext-0.19.8.1-2.el7.src.rpm", 1492653, "LGPLv2+ and GPLv3+", "CentOS", "", "Development files for gettext", "b6ae16e073cfe4270869d15e37898234"},
			{intRef(), "libselinux", "2.5", "12.el7", "x86_64", "libselinux-2.5-12.el7.src.rpm", 217874, "Public Domain", "CentOS", "", "SELinux library and simple utilities", "7584c834f1404914f4568973c001f28e"},
			{intRef(), "gcc-gfortran", "4.8.5", "36.el7_6.1", "x86_64", "gcc-4.8.5-36.el7_6.1.src.rpm", 16443248, "GPLv3+ and GPLv3+ with exceptions and GPLv2+ with exceptions and LGPLv2+ and BSD", "CentOS", "", "Fortran support", "c470a080bcd80338a0cb02ea194e87c5"},
			{intRef(), "keyutils-libs", "1.5.8", "3.el7", "x86_64", "keyutils-1.5.8-3.el7.src.rpm", 42138, "GPLv2+ and LGPLv2+", "CentOS", "", "Key utilities library", "e31fa37a714787d3d72e0a9316c3a85b"},
			{intRef(), "subversion", "1.7.14", "14.el7", "x86_64", "subversion-1.7.14-14.el7.src.rpm", 4844082, "ASL 2.0", "CentOS", "", "A Modern Concurrent Version Control System", "a52a29990c540008b82065babc5d2212"},
			{intRef(), "p11-kit-trust", "0.23.5", "3.el7", "x86_64", "p11-kit-0.23.5-3.el7.src.rpm", 437261, "BSD", "CentOS", "", "System trust module from p11-kit", "2dc2f7ab212401098e8821ce23f21741"},
			{intRef(), "rpm-python", "4.11.3", "35.el7", "x86_64", "rpm-4.11.3-35.el7.src.rpm", 149898, "GPLv2+", "CentOS", "", "Python bindings for apps which will manipulate RPM packages", "ad03ebb739fc7309edd09eba58ab7ee3"},
			{intRef(), "centos-release", "7", "5.1804.4.el7.centos", "x86_64", "centos-release-7-5.1804.4.el7.centos.src.rpm", 40338, "GPLv2", "CentOS", "", "CentOS Linux release file", "e73301588f9dce0441fb403b2fbc09fd"},
			{intRef(), "diffstat", "1.57", "4.el7", "x86_64", "diffstat-1.57-4.el7.src.rpm", 54065, "MIT", "CentOS", "", "A utility which provides statistics based on the output of diff", "84313732304f6bf93f57590b69b8ee70"},
			{intRef(), "libdb", "5.3.21", "24.el7", "x86_64", "libdb-5.3.21-24.el7.src.rpm", 1858008, "BSD and LGPLv2 and Sleepycat", "CentOS", "", "The Berkeley DB database library for C", "24fdfda30ad8e1cc7ebc744eaf85eacf"},
			{intRef(), "indent", "2.2.11", "13.el7", "x86_64", "indent-2.2.11-13.el7.src.rpm", 359131, "GPLv3+", "CentOS", "", "A GNU program for formatting C code", "54926576db2ea4e79ed3f7dcbbf25a1b"},
			{intRef(), "libgcrypt", "1.5.3", "14.el7", "x86_64", "libgcrypt-1.5.3-14.el7.src.rpm", 597727, "LGPLv2+", "CentOS", "", "A general-purpose cryptography library", "88c64dc7c3aed6a156e4dee5def38300"},
			{intRef(), "libICE", "1.0.9", "9.el7", "x86_64", "libICE-1.0.9-9.el7.src.rpm", 152213, "MIT", "CentOS", "", "X.Org X11 ICE runtime library", "dfa3c8915ac16deedaa578e08748b045"},
			{intRef(), "libuuid", "2.23.2", "52.el7_5.1", "x86_64", "util-linux-2.23.2-52.el7_5.1.src.rpm", 20326, "BSD", "CentOS", "", "Universally unique ID library", "534c564308a92d9c695b64b7a359f36c"},
			{intRef(1), "cups-libs", "1.6.3", "35.el7", "x86_64", "cups-1.6.3-35.el7.src.rpm", 734207, "LGPLv2 and zlib", "CentOS", "", "CUPS printing system - libraries", "bf85d281f99cae0fad729d864c4d7216"},
			{intRef(), "shared-mime-info", "1.8", "4.el7", "x86_64", "shared-mime-info-1.8-4.el7.src.rpm", 2379317, "GPLv2+", "CentOS", "", "Shared MIME information database", "167b39408a8c5705f135e968174e2229"},
			{intRef(), "atk", "2.28.1", "1.el7", "x86_64", "atk-2.28.1-1.el7.src.rpm", 1273762, "LGPLv2+", "CentOS", "", "Interfaces for accessibility support", "f47aaaa94be176394e9f81d83ad417ed"},
			{intRef(1), "findutils", "4.5.11", "5.el7", "x86_64", "findutils-4.5.11-5.el7.src.rpm", 1855626, "GPLv3+", "CentOS", "", "The GNU versions of find utilities (find and xargs)", "2d571b503447f5ce47a4767f6a7e1986"},
			{intRef(), "dejavu-fonts-common", "2.33", "6.el7", "noarch", "dejavu-fonts-2.33-6.el7.src.rpm", 130455, "Bitstream Vera and Public Domain", "CentOS", "", "Common files for the Dejavu font set", "61a93ec3edc83c900c0730e52ac1d1cb"},
			{intRef(), "expat", "2.1.0", "10.el7_3", "x86_64", "expat-2.1.0-10.el7_3.src.rpm", 208315, "MIT", "CentOS", "", "An XML parser library", "9c5fdceee3715ad8a1b3a9dd711d5b79"},
			{intRef(), "libfontenc", "1.1.3", "3.el7", "x86_64", "libfontenc-1.1.3-3.el7.src.rpm", 55701, "MIT", "CentOS", "", "X.Org X11 libfontenc runtime library", "1473960449a5d9ef0cc86d67bc47dff2"},
			{intRef(), "pam", "1.1.8", "22.el7", "x86_64", "pam-1.1.8-22.el7.src.rpm", 2630324, "BSD and GPLv2+", "CentOS", "", "An extensible library which provides authentication for applications", "dce355546c72f1425dc04fd64c9bc664"},
			{intRef(2), "vim-filesystem", "7.4.160", "5.el7", "x86_64", "vim-7.4.160-5.el7.src.rpm", 0, "Vim", "CentOS", "", "VIM filesystem layout", "85efae230969215df879da9479c80bdf"},
			{intRef(), "nss", "3.36.0", "7.el7_5", "x86_64", "nss-3.36.0-7.el7_5.src.rpm", 2424993, "MPLv2.0", "CentOS", "", "Network Security Services", "aead60d750112396fc5381c7c6152863"},
			{intRef(), "libusbx", "1.0.21", "1.el7", "x86_64", "libusbx-1.0.21-1.el7.src.rpm", 143429, "LGPLv2+", "CentOS", "", "Library for accessing USB devices", "cc8e175b4c336554ea675b4a36cb9e71"},
			{intRef(), "opensc", "0.16.0", "10.20170227git777e2a3.el7", "x86_64", "opensc-0.16.0-10.20170227git777e2a3.el7.src.rpm", 3260615, "LGPLv2+", "CentOS", "", "Smart card library and applications", "b49aa1bc288e2252efc9b56dd16558f2"},
			{intRef(), "poppler-data", "0.4.6", "3.el7", "noarch", "poppler-data-0.4.6-3.el7.src.rpm", 12013394, "BSD and GPLv2 and GPLv3+", "CentOS", "", "Encoding files", "982b1c3e6fbd7e2da6f79ef34b7c778c"},
			{intRef(), "hicolor-icon-theme", "0.12", "7.el7", "noarch", "hicolor-icon-theme-0.12-7.el7.src.rpm", 45788, "GPL+", "CentOS", "", "Basic requirement for icon themes", "54567528a0e83e823d9792db7121f73e"},
			{intRef(), "libX11-common", "1.6.5", "2.el7", "noarch", "libX11-1.6.5-2.el7.src.rpm", 1335662, "MIT", "CentOS", "", "Common data for libX11", "8008bbb9137fcb4eee8f46018f6fc5ff"},
			{intRef(), "lynx", "2.8.8", "0.3.dev15.el7", "x86_64", "lynx-2.8.8-0.3.dev15.el7.src.rpm", 5618384, "GPLv2", "CentOS", "", "A text-based Web browser", "a101416c60d8f2327b3a4db4ecf1fc10"},
			{intRef(), "libdrm", "2.4.91", "3.el7", "x86_64", "libdrm-2.4.91-3.el7.src.rpm", 371903, "MIT", "CentOS", "", "Direct Rendering Manager runtime library", "8214c63a12d3729f360600318c1078ad"},
			{intRef(), "docbook-style-xsl", "1.78.1", "3.el7", "noarch", "docbook-style-xsl-1.78.1-3.el7.src.rpm", 16671587, "DMIT", "CentOS", "", "Norman Walsh's XSL stylesheets for DocBook XML", "e8cf9e6ecce0240a06d31690545e4061"},
			{intRef(), "libXext", "1.3.3", "3.el7", "x86_64", "libXext-1.3.3-3.el7.src.rpm", 87102, "MIT", "CentOS", "", "X.Org X11 libXext runtime library", "fa7e50dec56e01bbe4d33adaec446c9c"},
			{intRef(), "libutempter", "1.1.6", "4.el7", "x86_64", "libutempter-1.1.6-4.el7.src.rpm", 49749, "LGPLv2+", "CentOS", "", "A privileged helper for utmp/wtmp updates", "8e28747e6a84b84af7d78c84cd8cd5e8"},
			{intRef(), "libXt", "1.1.5", "3.el7", "x86_64", "libXt-1.1.5-3.el7.src.rpm", 430105, "MIT", "CentOS", "", "X.Org X11 libXt runtime library", "5f357545763a40cda11cfd5e002cae50"},
			{intRef(7), "device-mapper", "1.02.146", "4.el7", "x86_64", "lvm2-2.02.177-4.el7.src.rpm", 338922, "GPLv2", "CentOS", "", "Device mapper utility", "1919f9661f5cf5f63fcb4c5112718db7"},
			{intRef(), "libXaw", "1.0.13", "4.el7", "x86_64", "libXaw-1.0.13-4.el7.src.rpm", 498789, "MIT", "CentOS", "", "X Athena Widget Set", "42e25e863c3a848f9ffd2ef57b2f75da"},
			{intRef(), "libXcursor", "1.1.15", "1.el7", "x86_64", "libXcursor-1.1.15-1.el7.src.rpm", 47039, "MIT", "CentOS", "", "Cursor management library", "4820a6e82a10283c5d0847f37c189363"},
			{intRef(), "libXinerama", "1.1.3", "2.1.el7", "x86_64", "libXinerama-1.1.3-2.1.el7.src.rpm", 14967, "MIT", "CentOS", "", "X.Org X11 libXinerama runtime library", "114ac67c20604dce0339038e2e34b193"},
			{intRef(1), "dbus", "1.10.24", "7.el7", "x86_64", "dbus-1.10.24-7.el7.src.rpm", 595223, "(GPLv2+ or AFL) and GPLv2+", "CentOS", "", "D-BUS message bus", "ee5c672ee2b669192e40bdf2460a5427"},
			{intRef(), "libXcomposite", "0.4.4", "4.1.el7", "x86_64", "libXcomposite-0.4.4-4.1.el7.src.rpm", 35184, "MIT", "CentOS", "", "X Composite Extension library", "43ff7c9b707c42cfe3723220ca5d31de"},
			{intRef(), "gdbm", "1.10", "8.el7", "x86_64", "gdbm-1.10-8.el7.src.rpm", 184322, "GPLv3+", "CentOS", "", "A GNU set of database routines which use extensible hashing", "770bd49f3898dc1cc831fb9ec12d569a"},
			{intRef(), "fribidi", "1.0.2", "1.el7", "x86_64", "fribidi-1.0.2-1.el7.src.rpm", 309831, "LGPLv2+ and UCD", "CentOS", "", "Library implementing the Unicode Bidirectional Algorithm", "5c0471896bfa74d49b08846246e53ecb"},
			{intRef(), "dbus-python", "1.1.1", "9.el7", "x86_64", "dbus-python-1.1.1-9.el7.src.rpm", 848122, "MIT", "CentOS", "", "D-Bus Python Bindings", "6be4da91886c59ae25e5255ebff0a0a9"},
			{intRef(), "libtool-ltdl", "2.4.2", "22.el7_3", "x86_64", "libtool-2.4.2-22.el7_3.src.rpm", 67814, "LGPLv2+", "CentOS", "", "Runtime libraries for GNU Libtool Dynamic Module Loader", "ff22ccbfa94b59e371c3073511c533c9"},
			{intRef(), "perl-ExtUtils-MakeMaker", "6.68", "3.el7", "noarch", "perl-ExtUtils-MakeMaker-6.68-3.el7.src.rpm", 650532, "GPL+ or Artistic", "CentOS", "", "Create a module Makefile", "bc996c68077ef5ac7a16424aed45c93b"},
			{intRef(), "pyliblzma", "0.5.3", "11.el7", "x86_64", "pyliblzma-0.5.3-11.el7.src.rpm", 190112, "LGPLv3+", "CentOS", "", "Python bindings for lzma", "a09aa8e806a3f6285e9fe765d69d3ee9"},
			{intRef(), "asciidoc", "8.6.8", "5.el7", "noarch", "asciidoc-8.6.8-5.el7.src.rpm", 976747, "GPL+ and GPLv2+", "CentOS", "", "Text based document generation", "aedf8ae4cd21e9fd4053db4069f78238"},
			{intRef(), "pyxattr", "0.5.1", "5.el7", "x86_64", "pyxattr-0.5.1-5.el7.src.rpm", 63304, "LGPLv2+", "CentOS", "", "Extended attributes library wrapper for Python", "e85121da681760c48a2af6ed8aa31b4e"},
			{intRef(), "zlib", "1.2.7", "18.el7", "x86_64", "zlib-1.2.7-18.el7.src.rpm", 185294, "zlib and Boost", "CentOS", "", "The compression and decompression library", "063a85f8271de9b41e9ac9657962e3ec"},
			{intRef(), "python-javapackages", "3.4.1", "11.el7", "noarch", "javapackages-tools-3.4.1-11.el7.src.rpm", 66505, "BSD", "CentOS", "", "Module for handling various files for Java packaging", "9bacc7d18c047dd61e7b205f1bcb8b21"},
			{intRef(), "binutils", "2.27", "34.base.el7", "x86_64", "binutils-2.27-34.base.el7.src.rpm", 25150000, "GPLv3+", "CentOS", "", "A GNU collection of binary utilities", "b9772c7ef8848f4b5372fa8f52222a6a"},
			{intRef(), "yum", "3.4.3", "158.el7.centos", "noarch", "yum-3.4.3-158.el7.centos.src.rpm", 5814102, "GPLv2+", "CentOS", "", "RPM package installer/updater/manager", "55ef45392a487da6305b69e89cef7cb7"},
			{intRef(), "libXtst", "1.2.3", "1.el7", "x86_64", "libXtst-1.2.3-1.el7.src.rpm", 29938, "MIT", "CentOS", "", "X.Org X11 libXtst runtime library", "fa9186e0d600312861a3230199657845"},
			{intRef(), "alsa-lib", "1.1.6", "2.el7", "x86_64", "alsa-lib-1.1.6-2.el7.src.rpm", 1433094, "LGPLv2+", "CentOS", "", "The Advanced Linux Sound Architecture (ALSA) library", "0ec6aefe3b19547f106b945d5e3dfae8"},
			{intRef(), "yum-utils", "1.1.31", "46.el7_5", "noarch", "yum-utils-1.1.31-46.el7_5.src.rpm", 343422, "GPLv2+", "CentOS", "", "Utilities based around the yum package manager", "9a3315c3ee603728ce1908ec404a2c42"},
			{intRef(), "audit-libs-devel", "2.8.4", "4.el7", "x86_64", "audit-2.8.4-4.el7.src.rpm", 88847, "LGPLv2+", "CentOS", "", "Header files for libaudit", "cd7ffd7df89d1ba886b13b454eeb8204"},
			{intRef(2), "vim-minimal", "7.4.160", "4.el7", "x86_64", "vim-7.4.160-4.el7.src.rpm", 917640, "Vim", "CentOS", "", "A minimal version of the VIM editor", "495d7fc35ace98f416bef65eecf8af0d"},
			{intRef(), "python-libs", "2.7.5", "77.el7_6", "x86_64", "python-2.7.5-77.el7_6.src.rpm", 24714266, "Python", "CentOS", "", "Runtime libraries for Python", "7e08db1c34a5fc8223919643ece416dd"},
			{intRef(), "glibc-common", "2.17", "260.el7_6.4", "x86_64", "glibc-2.17-260.el7_6.4.src.rpm", 120497983, "LGPLv2+ and LGPLv2+ with exceptions and GPLv2+", "CentOS", "", "Common binaries and locale data for glibc", "43f1a899692374dc09c21bae9c759d1c"},
			{intRef(), "slang-devel", "2.2.4", "11.el7", "x86_64", "slang-2.2.4-11.el7.src.rpm", 356429, "GPLv2+", "CentOS", "", "Development files for the S-Lang extension language", "487472fcf4fa82fa513f7489fb4fe9af"},
			{intRef(), "rpm", "4.11.3", "35.el7", "x86_64", "rpm-4.11.3-35.el7.src.rpm", 2622266, "GPLv2+", "CentOS", "", "The RPM package management system", "ba17adc338f442da8a4598244965e521"},
			{intRef(), "newt-devel", "0.52.15", "4.el7", "x86_64", "newt-0.52.15-4.el7.src.rpm", 127322, "LGPLv2", "CentOS", "", "Newt windowing toolkit development files", "294745fabf4c8a67684689dae1229d0f"},
			{intRef(), "mpfr", "3.1.1", "4.el7", "x86_64", "mpfr-3.1.1-4.el7.src.rpm", 554279, "LGPLv3+ and GPLv3+ and GFDL", "CentOS", "", "A C library for multiple-precision floating-point computations", "4cedb227fd4730529b0874792007692b"},
			{intRef(), "bc", "1.06.95", "13.el7", "x86_64", "bc-1.06.95-13.el7.src.rpm", 219830, "GPLv2+", "CentOS", "", "GNU's bc (a numeric processing language) and dc (a calculator)", "7ecc821309bbd4e262ddaf9a4c45d005"},
			{intRef(), "fipscheck-lib", "1.4.1", "6.el7", "x86_64", "fipscheck-1.4.1-6.el7.src.rpm", 11466, "BSD", "CentOS", "", "Library files for fipscheck", "93aa3152b03992c02e0ed2759b1d3ac8"},
			{intRef(), "libquadmath", "4.8.5", "36.el7_6.1", "x86_64", "gcc-4.8.5-36.el7_6.1.src.rpm", 282811, "GPLv3+ and GPLv3+ with exceptions and GPLv2+ with exceptions and LGPLv2+ and BSD", "CentOS", "", "GCC __float128 shared support library", "80155e9c1179374d45231786bedae2a4"},
			{intRef(), "unzip", "6.0", "19.el7", "x86_64", "unzip-6.0-19.el7.src.rpm", 373986, "BSD", "CentOS", "", "A utility for unpacking zip files", "7a396b648ba0120404fe2ef171b79681"},
			{intRef(), "gettext-libs", "0.19.8.1", "2.el7", "x86_64", "gettext-0.19.8.1-2.el7.src.rpm", 1546952, "LGPLv2+ and GPLv3+", "CentOS", "", "Libraries for gettext", "3164b6e314d214a982d34c7c77d89e16"},
			{intRef(), "cryptsetup-libs", "2.0.3", "3.el7", "x86_64", "cryptsetup-2.0.3-3.el7.src.rpm", 1219591, "GPLv2+ and LGPLv2+", "CentOS", "", "Cryptsetup shared library", "61b5b318bfcd411aec021e25d9b76070"},
			{intRef(), "perl-podlators", "2.5.1", "3.el7", "noarch", "perl-podlators-2.5.1-3.el7.src.rpm", 287679, "GPL+ or Artistic", "CentOS", "", "Format POD source into various output formats", "8e070dd716e3adf3c297291accaa4348"},
			{intRef(), "perl-Encode", "2.51", "7.el7", "x86_64", "perl-Encode-2.51-7.el7.src.rpm", 10176350, "GPL+ or Artistic", "CentOS", "", "Character encodings in Perl", "713ce27ff4434807ff7060b333ed8873"},
			{intRef(), "perl-Storable", "2.45", "3.el7", "x86_64", "perl-Storable-2.45-3.el7.src.rpm", 181031, "GPL+ or Artistic", "CentOS", "", "Persistence for Perl data structures", "25787b4b609f3ee50d1c7f33d29e0102"},
			{intRef(), "perl-Filter", "1.49", "3.el7", "x86_64", "perl-Filter-1.49-3.el7.src.rpm", 148475, "GPL+ or Artistic", "CentOS", "", "Perl source filters", "66b1205b2eef2a89137e530741ed3710"},
			{intRef(4), "perl-macros", "5.16.3", "294.el7_6", "x86_64", "perl-5.16.3-294.el7_6.src.rpm", 5134, "GPL+ or Artistic", "CentOS", "", "Macros for rpmbuild", "7c63786a9688ac528e398b27e86198d8"},
			{intRef(), "perl-PathTools", "3.40", "5.el7", "x86_64", "perl-PathTools-3.40-5.el7.src.rpm", 174131, "(GPL+ or Artistic) and BSD", "CentOS", "", "PathTools Perl module (Cwd, File::Spec)", "c597987071b0f81a0029667e83e1ecae"},
			{intRef(4), "perl", "5.16.3", "294.el7_6", "x86_64", "perl-5.16.3-294.el7_6.src.rpm", 23552128, "(GPL+ or Artistic) and (GPLv2+ or Artistic) and Copyright Only and MIT and Public Domain and UCD", "CentOS", "", "Practical Extraction and Report Language", "547b1d620cedc114b9a1a681c1f2a4bd"},
			{intRef(), "perl-Data-Dumper", "2.145", "3.el7", "x86_64", "perl-Data-Dumper-2.145-3.el7.src.rpm", 99287, "GPL+ or Artistic", "CentOS", "", "Stringify perl data structures, suitable for printing and eval", "3d2fc7733dc839a2d9a67e2805479feb"},
			{intRef(), "automake", "1.13.4", "3.el7", "noarch", "automake-1.13.4-3.el7.src.rpm", 1731909, "GPLv2+ and GFDL and Public Domain and MIT", "CentOS", "", "A GNU tool for automatically creating Makefiles", "211dbf9617136d88757937007240ae9b"},
			{intRef(), "libgfortran", "4.8.5", "36.el7_6.1", "x86_64", "gcc-4.8.5-36.el7_6.1.src.rpm", 1186168, "GPLv3+ and GPLv3+ with exceptions and GPLv2+ with exceptions and LGPLv2+ and BSD", "CentOS", "", "Fortran runtime", "a210ea42ee5ed4ca8328a4555b0717ab"},
			{intRef(), "systemd-libs", "219", "62.el7_6.5", "x86_64", "systemd-219-62.el7_6.5.src.rpm", 1263456, "LGPLv2+ and MIT", "CentOS", "", "systemd libraries", "9577513987ddb8ff723a0b94367fc8c7"},
			{intRef(), "systemd-sysv", "219", "62.el7_6.5", "x86_64", "systemd-219-62.el7_6.5.src.rpm", 3979, "LGPLv2+", "CentOS", "", "SysV tools for systemd", "cfb0137b4b27e0e16bc2c23388f69b59"},
			{intRef(), "boost-date-time", "1.53.0", "27.el7", "x86_64", "boost-1.53.0-27.el7.src.rpm", 139706, "Boost and MIT and Python", "CentOS", "", "Run-Time component of boost date-time library", "ded46380fd053f4a72b921730cd0aa44"},
			{intRef(), "libmodman", "2.0.1", "8.el7", "x86_64", "libmodman-2.0.1-8.el7.src.rpm", 58871, "LGPLv2+", "CentOS", "", "A simple library for managing C++ modules (plug-ins)", "d9d56d2a76492f4c2f9c827d7aae6b08"},
			{intRef(), "nettle", "2.7.1", "8.el7", "x86_64", "nettle-2.7.1-8.el7.src.rpm", 765042, "LGPLv2+", "CentOS", "", "A low-level cryptographic library", "5664b34732ae0135d04a6a3e6a5a6b61"},
			{intRef(), "ncurses-base", "5.9", "14.20130511.el7_4", "noarch", "ncurses-5.9-14.20130511.el7_4.src.rpm", 223432, "MIT", "CentOS", "", "Descriptions of common terminals", "c955fdfe1f8f20b1053a34918e44bd58"},
			{intRef(), "gdb", "7.6.1", "114.el7", "x86_64", "gdb-7.6.1-114.el7.src.rpm", 7371507, "GPLv3+ and GPLv3+ with exceptions and GPLv2+ and GPLv2+ with exceptions and GPL+ and LGPLv2+ and BSD and Public Domain", "CentOS", "", "A GNU source-level debugger for C, C++, Fortran, Go and other languages", "1d085bb4c7273a51db58d47b35ab95b4"},
			{intRef(), "bash", "4.2.46", "30.el7", "x86_64", "bash-4.2.46-30.el7.src.rpm", 3667709, "GPLv3+", "CentOS", "", "The GNU Bourne Again shell", "4c9037d4d3139a2c8fd28ed6b27d47da"},
			{intRef(), "libedit", "3.0", "12.20121213cvs.el7", "x86_64", "libedit-3.0-12.20121213cvs.el7.src.rpm", 244257, "BSD", "CentOS", "", "The NetBSD Editline library", "9e248aca4576e17c188396aa91931c63"},
			{intRef(), "chkconfig", "1.7.4", "1.el7", "x86_64", "chkconfig-1.7.4-1.el7.src.rpm", 779531, "GPLv2", "CentOS", "", "A system tool for maintaining the /etc/rc*.d hierarchy", "e126177e9ac5841deca96f4e15340613"},
			{intRef(), "perl-Git", "1.8.3.1", "20.el7", "noarch", "git-1.8.3.1-20.el7.src.rpm", 58567, "GPLv2", "CentOS", "", "Perl interface to Git", "801ebea11b001365abebd04e2da2690f"},
			{intRef(), "setup", "2.8.71", "9.el7", "noarch", "setup-2.8.71-9.el7.src.rpm", 696925, "Public Domain", "CentOS", "", "A set of system configuration and setup files", "3e3fc345cd2ac49c17b8e38df270fd16"},
			{intRef(), "pakchois", "0.4", "10.el7", "x86_64", "pakchois-0.4-10.el7.src.rpm", 29489, "LGPLv2+", "CentOS", "", "A wrapper library for PKCS#11", "7adb828e1d494684471a8c13511131d9"},
			{intRef(), "basesystem", "10.0", "7.el7.centos", "noarch", "basesystem-10.0-7.el7.centos.src.rpm", 0, "Public Domain", "CentOS", "", "The skeleton package which defines a simple CentOS Linux system", "778eb3d19a934f42d1354ed795d06b4f"},
			{intRef(), "subversion-libs", "1.7.14", "14.el7", "x86_64", "subversion-1.7.14-14.el7.src.rpm", 2615823, "ASL 2.0", "CentOS", "", "Libraries for Subversion Version Control system", "ae892db7483602754c8a3e98869e46c6"},
			{intRef(), "efivar-libs", "36", "11.el7_6.1", "x86_64", "efivar-36-11.el7_6.1.src.rpm", 252096, "LGPLv2+", "CentOS", "", "Library to manage UEFI variables", "bce3fe8bb152785ab1475b64339a3984"},
			{intRef(), "nss-util", "3.36.0", "1.el7_5", "x86_64", "nss-util-3.36.0-1.el7_5.src.rpm", 194944, "MPLv2.0", "CentOS", "", "Network Security Services Utilities Library", "ecf6d738049e13862a8cf2a151b78bb5"},
			{intRef(), "systemtap-client", "3.3", "3.el7", "x86_64", "systemtap-3.3-3.el7.src.rpm", 10146436, "GPLv2+", "CentOS", "", "Programmable system-wide instrumentation system - client", "d9129a6c4b403c9ff7f1a38e32add796"},
			{intRef(), "libcom_err", "1.42.9", "12.el7_5", "x86_64", "e2fsprogs-1.42.9-12.el7_5.src.rpm", 60489, "MIT", "CentOS", "", "Common error description library", "4b27bfeb726d091b4317b5e66d6dba7f"},
			{intRef(), "kernel-headers", "3.10.0", "957.10.1.el7", "x86_64", "kernel-3.10.0-957.10.1.el7.src.rpm", 3823832, "GPLv2", "CentOS", "", "Header files for the Linux kernel for use by glibc", "17c5c27d1c0577522a0bd74f96990825"},
			{intRef(), "libattr", "2.4.46", "13.el7", "x86_64", "attr-2.4.46-13.el7.src.rpm", 19896, "LGPLv2+", "CentOS", "", "Dynamic library for extended attribute support", "209b05c7b77a33a4cc176077945833e8"},
			{intRef(), "glibc-devel", "2.17", "260.el7_6.4", "x86_64", "glibc-2.17-260.el7_6.4.src.rpm", 1066078, "LGPLv2+ and LGPLv2+ with exceptions and GPLv2+", "CentOS", "", "Object files for development using standard C libraries.", "cf1d0eead12e91a77c631cb27f0aa8ee"},
			{intRef(), "libacl", "2.2.51", "14.el7", "x86_64", "acl-2.2.51-14.el7.src.rpm", 37056, "LGPLv2+", "CentOS", "", "Dynamic library for access control list support", "c3243d728de0ad95e0c05230e6463bdd"},
			{intRef(), "libquadmath-devel", "4.8.5", "36.el7_6.1", "x86_64", "gcc-4.8.5-36.el7_6.1.src.rpm", 18514, "GPLv3+ and GPLv3+ with exceptions and GPLv2+ with exceptions and LGPLv2+ and BSD", "CentOS", "", "GCC __float128 support", "3cebb989a4fc2a0787f68ae06061d502"},
			{intRef(), "perl-srpm-macros", "1", "8.el7", "noarch", "perl-srpm-macros-1-8.el7.src.rpm", 794, "GPLv3+", "CentOS", "", "RPM macros for building Perl source package from source repository", "6bd0791a143b53addc95cbeb050859c8"},
			{intRef(), "info", "5.1", "5.el7", "x86_64", "texinfo-5.1-5.el7.src.rpm", 494630, "GPLv3+", "CentOS", "", "A stand-alone TTY-based reader for GNU texinfo documentation", "e1f75e40cf956ed3efe7b49c5f6939ec"},
			{intRef(), "gettext-common-devel", "0.19.8.1", "2.el7", "noarch", "gettext-0.19.8.1-2.el7.src.rpm", 397912, "GPLv3+", "CentOS", "", "Common development files for gettext", "08291ef848c478a58968efe973aced3d"},
			{intRef(), "pcre", "8.32", "17.el7", "x86_64", "pcre-8.32-17.el7.src.rpm", 1475532, "BSD", "CentOS", "", "Perl-compatible regular expression library", "04a1926cc29eef0f79f8183eb7907024"},
			{intRef(), "intltool", "0.50.2", "7.el7", "noarch", "intltool-0.50.2-7.el7.src.rpm", 170106, "GPLv2 with exceptions", "CentOS", "", "Utility for internationalizing various kinds of data files", "f5fd5d682337200009190a45fa4f80bf"},
			{intRef(), "sed", "4.2.2", "5.el7", "x86_64", "sed-4.2.2-5.el7.src.rpm", 601208, "GPLv3+", "CentOS", "", "A GNU stream text editor", "4d696dbd3061493258490f40c625adba"},
			{intRef(), "systemtap", "3.3", "3.el7", "x86_64", "systemtap-3.3-3.el7.src.rpm", 206577, "GPLv2+", "CentOS", "", "Programmable system-wide instrumentation system", "7aa25cadecbbb48aaefedd2dab5d8196"},
			{intRef(), "p11-kit", "0.23.5", "3.el7", "x86_64", "p11-kit-0.23.5-3.el7.src.rpm", 1337825, "BSD", "CentOS", "", "Library for loading and sharing PKCS#11 modules", "9a0f49ca3b6d58aeb05780003cae38f1"},
			{intRef(), "libtool", "2.4.2", "22.el7_3", "x86_64", "libtool-2.4.2-22.el7_3.src.rpm", 2323761, "GPLv2+ and LGPLv2+ and GFDL", "CentOS", "", "The GNU Portable Library Tool", "34886e5d2953afff439b5b7e9b1ca3d4"},
			{intRef(1), "gmp", "6.0.0", "15.el7", "x86_64", "gmp-6.0.0-15.el7.src.rpm", 657046, "LGPLv3+ or GPLv2+", "CentOS", "", "A GNU arbitrary precision library", "4cd60ebf94766a46c590e17036ece777"},
			{intRef(), "cscope", "15.8", "10.el7", "x86_64", "cscope-15.8-10.el7.src.rpm", 939744, "BSD and GPLv2+", "CentOS", "", "C source code tree search and browse tool", "3335c17d1185aab5024ffec64854b254"},
			{intRef(), "libtasn1", "4.10", "1.el7", "x86_64", "libtasn1-4.10-1.el7.src.rpm", 424486, "GPLv3+ and LGPLv2+", "CentOS", "", "The ASN.1 library used in GNUTLS", "5bd1b0b4094fac08713f986572a8625e"},
			{intRef(), "patchutils", "0.3.3", "4.el7", "x86_64", "patchutils-0.3.3-4.el7.src.rpm", 260135, "GPLv2+", "CentOS", "", "A collection of programs for manipulating patch files", "f90fbbf401062fd532219b576d061676"},
			{intRef(), "ca-certificates", "2018.2.22", "70.0.el7_5", "noarch", "ca-certificates-2018.2.22-70.0.el7_5.src.rpm", 973960, "Public Domain", "CentOS", "", "The Mozilla CA root certificate bundle", "a95373fc5273e7c757a240316d4430b9"},
			{intRef(), "bison", "3.0.4", "2.el7", "x86_64", "bison-3.0.4-2.el7.src.rpm", 2154174, "GPLv3+", "CentOS", "", "A GNU general-purpose parser generator", "c12d2d7e86d2ede2fe96c8c119c6d06d"},
			{intRef(), "coreutils", "8.22", "21.el7", "x86_64", "coreutils-8.22-21.el7.src.rpm", 14588989, "GPLv3+", "CentOS", "", "A set of basic GNU tools commonly used in shell scripts", "fb8117e7aeb1ab935460fa412277a088"},
			{intRef(), "rpm-sign", "4.11.3", "35.el7", "x86_64", "rpm-4.11.3-35.el7.src.rpm", 17397, "GPLv2+", "CentOS", "", "Package signing support", "8a594161eed57b87386dc0e4d85b03b4"},
			{intRef(), "krb5-libs", "1.15.1", "19.el7", "x86_64", "krb5-1.15.1-19.el7.src.rpm", 1984782, "MIT", "CentOS", "", "The non-admin shared libraries used by Kerberos 5", "9811e62803caaf9375851ed6f16fedd7"},
			{intRef(), "swig", "2.0.10", "5.el7", "x86_64", "swig-2.0.10-5.el7.src.rpm", 4926232, "GPLv3+ and BSD", "CentOS", "", "Connects C/C++/Objective C to some high-level programming languages", "6f2f743e241650625c57d1012b1d7eff"},
			{intRef(), "bzip2-libs", "1.0.6", "13.el7", "x86_64", "bzip2-1.0.6-13.el7.src.rpm", 70093, "BSD", "CentOS", "", "Libraries for applications using bzip2", "0486f0b02e0caa4195761266798e7a26"},
			{intRef(), "rcs", "5.9.0", "5.el7", "x86_64", "rcs-5.9.0-5.el7.src.rpm", 624710, "GPLv3+", "CentOS", "", "Revision Control System (RCS) file version management tools", "fcb64d3f5164834d42cd02954692bfdb"},
			{intRef(), "ctags", "5.8", "13.el7", "x86_64", "ctags-5.8-13.el7.src.rpm", 359724, "GPLv2+ and LGPLv2+ and Public Domain", "CentOS", "", "A C programming language indexing and/or cross-reference tool", "89441c9d19184e2f2a8b14e39b2ef63c"},
			{intRef(), "libxml2", "2.9.1", "6.el7_2.3", "x86_64", "libxml2-2.9.1-6.el7_2.3.src.rpm", 1710062, "MIT", "CentOS", "", "Library providing XML and HTML support", "e439b8934c4e9576eccfbc60d3cd714f"},
			{intRef(2), "libpng", "1.5.13", "7.el7_2", "x86_64", "libpng-1.5.13-7.el7_2.src.rpm", 616101, "zlib", "CentOS", "", "A library of functions for manipulating PNG image format files", "f474205e2ea2073937a95b3c55741e46"},
			{intRef(), "readline", "6.2", "10.el7", "x86_64", "readline-6.2-10.el7.src.rpm", 460464, "GPLv3+", "CentOS", "", "A library for editing typed command lines", "b552128087365a58b3566631c678752f"},
			{intRef(), "libjpeg-turbo", "1.2.90", "6.el7", "x86_64", "libjpeg-turbo-1.2.90-6.el7.src.rpm", 350499, "IJG", "CentOS", "", "A MMX/SSE2 accelerated library for manipulating JPEG image files", "34514802e4fea3e664d8beacd7ac6b4d"},
			{intRef(), "cpio", "2.11", "27.el7", "x86_64", "cpio-2.11-27.el7.src.rpm", 689335, "GPLv3+", "CentOS", "", "A GNU archiving program", "32498692f82296d51663586f08cebafc"},
			{intRef(), "libSM", "1.2.2", "2.el7", "x86_64", "libSM-1.2.2-2.el7.src.rpm", 81546, "MIT", "CentOS", "", "X.Org X11 SM runtime library", "4dfa7c7907174b7354c740d44dba2f8d"},
			{intRef(), "libblkid", "2.23.2", "52.el7_5.1", "x86_64", "util-linux-2.23.2-52.el7_5.1.src.rpm", 261837, "LGPLv2+", "CentOS", "", "Block device ID library", "427f260c9fc06f72d722e5ccdfcf30c5"},
			{intRef(), "pcsc-lite-libs", "1.8.8", "8.el7", "x86_64", "pcsc-lite-1.8.8-8.el7.src.rpm", 46708, "BSD", "CentOS", "", "PC/SC Lite libraries", "20ea86c68a20f01f498ba7f7811f3a59"},
			{intRef(), "glib2", "2.54.2", "2.el7", "x86_64", "glib2-2.54.2-2.el7.src.rpm", 11986873, "LGPLv2+", "CentOS", "", "A library of handy utility functions", "75e476b2fbb8f06f881338c12e379cc3"},
			{intRef(1), "libglvnd", "1.0.1", "0.8.git5baa1e5.el7", "x86_64", "libglvnd-1.0.1-0.8.git5baa1e5.el7.src.rpm", 666456, "MIT", "CentOS", "", "The GL Vendor-Neutral Dispatch library", "6641b187badae8ebd191c0e73fc8fa45"},
			{intRef(), "sqlite", "3.7.17", "8.el7", "x86_64", "sqlite-3.7.17-8.el7.src.rpm", 814231, "Public Domain", "CentOS", "", "Library that implements an embeddable SQL database engine", "0ceb9434c5a9d281399583087cfe889b"},
			{intRef(), "xml-common", "0.6.3", "39.el7", "noarch", "sgml-common-0.6.3-39.el7.src.rpm", 46969, "GPL+", "CentOS", "", "Common XML catalog and DTD files", "83630b0f69dd2b423166aa9dd588dbb2"},
			{intRef(), "cracklib", "2.9.0", "11.el7", "x86_64", "cracklib-2.9.0-11.el7.src.rpm", 209610, "LGPLv2+", "CentOS", "", "A password-checking library", "a973d883d6785904e49266428bb3ed90"},
			{intRef(), "libxslt", "1.1.28", "5.el7", "x86_64", "libxslt-1.1.28-5.el7.src.rpm", 497582, "MIT", "CentOS", "", "Library providing the Gnome XSLT engine", "e901ff455779767496a18d1a5f43816a"},
			{intRef(), "libidn", "1.28", "4.el7", "x86_64", "libidn-1.28-4.el7.src.rpm", 630407, "LGPLv2+ and GPLv3+ and GFDL", "CentOS", "", "Internationalized Domain Name support library", "3d582106ac1032c59c1cd4236e5d6415"},
			{intRef(), "libwayland-server", "1.15.0", "1.el7", "x86_64", "wayland-1.15.0-1.el7.src.rpm", 81084, "MIT", "CentOS", "", "Wayland server library", "61bc90fe0aa6792be68e420231fc2572"},
			{intRef(), "dejavu-sans-fonts", "2.33", "6.el7", "noarch", "dejavu-fonts-2.33-6.el7.src.rpm", 5395167, "Bitstream Vera and Public Domain", "CentOS", "", "Variable-width sans-serif font faces", "1e16e93e08e0bf32696b98a4ede090d0"},
			{intRef(), "libcap-ng", "0.7.5", "4.el7", "x86_64", "libcap-ng-0.7.5-4.el7.src.rpm", 50510, "LGPLv2+", "CentOS", "", "An alternate posix capabilities library", "84efa5ca78408f78e606e7c8d1816759"},
			{intRef(), "jasper-libs", "1.900.1", "33.el7", "x86_64", "jasper-1.900.1-33.el7.src.rpm", 350690, "JasPer", "CentOS", "", "Runtime libraries for jasper", "426d4ea10e8e7c3614fb2ce00d457824"},
			{intRef(), "cracklib-dicts", "2.9.0", "11.el7", "x86_64", "cracklib-2.9.0-11.el7.src.rpm", 9389116, "LGPLv2+", "CentOS", "", "The standard CrackLib dictionaries", "da657b0facf181893f9a38437092dbbf"},
			{intRef(1), "xorg-x11-font-utils", "7.5", "21.el7", "x86_64", "xorg-x11-font-utils-7.5-21.el7.src.rpm", 361874, "MIT", "CentOS", "", "X.Org X11 font utilities", "5fc7df3b3e1ae496600ac1de86a1a658"},
			{intRef(), "libpwquality", "1.2.3", "5.el7", "x86_64", "libpwquality-1.2.3-5.el7.src.rpm", 332421, "BSD or GPLv2+", "CentOS", "", "A library for password generation and password quality checking", "e40a69f0a1803dfa3736aba2c8ea1fb4"},
			{intRef(), "ghostscript-fonts", "5.50", "32.el7", "noarch", "ghostscript-fonts-5.50-32.el7.src.rpm", 636425, "GPLv2+ and Hershey and MIT and OFL and Public Domain", "CentOS", "", "Fonts for the Ghostscript PostScript interpreter", "d0b5530fd88d0966a755e7b34c5520b4"},
			{intRef(), "nss-sysinit", "3.36.0", "7.el7_5", "x86_64", "nss-3.36.0-7.el7_5.src.rpm", 14061, "MPLv2.0", "CentOS", "", "System NSS Initialization", "ace8226abffd1e928af6f4c62cdd450d"},
			{intRef(), "lcms2", "2.6", "3.el7", "x86_64", "lcms2-2.6-3.el7.src.rpm", 377192, "MIT", "CentOS", "", "Color Management Engine", "b0922952f79e6678c0cca8095a909a52"},
			{intRef(), "nss-pem", "1.0.3", "4.el7", "x86_64", "nss-pem-1.0.3-4.el7.src.rpm", 201219, "MPLv1.1", "CentOS", "", "PEM file reader for Network Security Services (NSS)", "95cf0973322bf389a612fdbd1e3a6627"},
			{intRef(), "libtiff", "4.0.3", "27.el7_3", "x86_64", "libtiff-4.0.3-27.el7_3.src.rpm", 494377, "libtiff", "CentOS", "", "Library of functions for manipulating TIFF format image files", "8555f1bc1831d96717174589ae1922ec"},
			{intRef(), "xz", "5.2.2", "1.el7", "x86_64", "xz-5.2.2-1.el7.src.rpm", 798130, "LGPLv2+", "CentOS", "", "LZMA compression utilities", "3e6f05008f7af7931aa8257964527f1a"},
			{intRef(), "pcsc-lite-ccid", "1.4.10", "14.el7", "x86_64", "pcsc-lite-ccid-1.4.10-14.el7.src.rpm", 562872, "LGPLv2+", "CentOS", "", "Generic USB CCID smart card reader driver", "9c2f326c2a8b1d35d65df1cf7f780ca2"},
			{intRef(), "coolkey", "1.1.0", "40.el7", "x86_64", "coolkey-1.1.0-40.el7.src.rpm", 304556, "LGPLv2", "CentOS", "", "CoolKey PKCS #11 module", "e82080986a71e3122eea213e7d400d46"},
			{intRef(), "lz4", "1.7.5", "2.el7", "x86_64", "lz4-1.7.5-2.el7.src.rpm", 366872, "GPLv2+ and BSD", "CentOS", "", "Extremely fast compression algorithm", "89e45994eabac0ba4ccae7c1005ea06c"},
			{intRef(), "libthai", "0.1.14", "9.el7", "x86_64", "libthai-0.1.14-9.el7.src.rpm", 704340, "LGPLv2+", "CentOS", "", "Thai language support routines", "50db030da1664a77725f711a80a68299"},
			{intRef(), "systemtap-sdt-devel", "3.3", "3.el7", "x86_64", "systemtap-3.3-3.el7.src.rpm", 178425, "GPLv2+ and Public Domain", "CentOS", "", "Static probe support tools", "08052fa243a5ecbe54ade18168afadd7"},
			{intRef(), "nss-tools", "3.36.0", "7.el7_5", "x86_64", "nss-3.36.0-7.el7_5.src.rpm", 2069571, "MPLv2.0", "CentOS", "", "Tools for the Network Security Services", "dfada2eb06ef6cccbdc69c711c59ae02"},
			{intRef(), "libicu", "50.1.2", "17.el7", "x86_64", "icu-50.1.2-17.el7.src.rpm", 25217861, "MIT and UCD and Public Domain", "CentOS", "", "International Components for Unicode - libraries", "136aed025e02b1a0eaa8ea5a9a1f3c39"},
			{intRef(), "source-highlight", "3.1.6", "6.el7", "x86_64", "source-highlight-3.1.6-6.el7.src.rpm", 3024049, "GPLv3+", "CentOS", "", "Produces a document with syntax highlighting", "b9f0338612470b08e5aa9c230ed8bdc5"},
			{intRef(), "gobject-introspection", "1.50.0", "1.el7", "x86_64", "gobject-introspection-1.50.0-1.el7.src.rpm", 834149, "GPLv2+, LGPLv2+, MIT", "CentOS", "", "Introspection system for GObject-based libraries", "6048b8e810c7b808f4eddca36d12d5b8"},
			{intRef(), "gdbm-devel", "1.10", "8.el7", "x86_64", "gdbm-1.10-8.el7.src.rpm", 42866, "GPLv3+", "CentOS", "", "Development libraries and header files for the gdbm library", "2b68a1b0bb8e948d116351c48e8dd3e4"},
			{intRef(), "libdb-utils", "5.3.21", "24.el7", "x86_64", "libdb-5.3.21-24.el7.src.rpm", 326487, "BSD and LGPLv2 and Sleepycat", "CentOS", "", "Command line tools for managing Berkeley DB databases", "d06077335f27858ecb06ad48fa2e812d"},
			{intRef(), "perl-ExtUtils-Manifest", "1.61", "244.el7", "noarch", "perl-ExtUtils-Manifest-1.61-244.el7.src.rpm", 85359, "GPL+ or Artistic", "CentOS", "", "Utilities to write and check a MANIFEST file", "de7fd112d77843cf3e0298d5d48640ab"},
			{intRef(), "kmod-libs", "20", "21.el7", "x86_64", "kmod-20-21.el7.src.rpm", 91800, "LGPLv2+", "CentOS", "", "Libraries to handle kernel module loading and unloading", "d7dea2fe3482829a84830109bf4bcc97"},
			{intRef(), "graphite2", "1.3.10", "1.el7_3", "x86_64", "graphite2-1.3.10-1.el7_3.src.rpm", 254548, "(LGPLv2+ or GPLv2+ or MPL) and (Netscape or GPLv2+ or LGPLv2+)", "CentOS", "", "Font rendering capabilities for complex non-Roman writing systems", "ac1d0755528ed9fd520d7321bf194d41"},
			{intRef(), "libssh2", "1.4.3", "10.el7_2.1", "x86_64", "libssh2-1.4.3-10.el7_2.1.src.rpm", 341782, "BSD", "CentOS", "", "A library implementing the SSH2 protocol", "8ee7c5f75e2a2c85536f0f2282347545"},
			{intRef(), "centos-indexhtml", "7", "9.el7.centos", "noarch", "centos-indexhtml-7-9.el7.centos.src.rpm", 92560, "Distributable", "CentOS", "", "Browser default start page for CentOS", "eee16f5af0320932f730f5a2732d55bd"},
			{intRef(), "curl", "7.29.0", "46.el7", "x86_64", "curl-7.29.0-46.el7.src.rpm", 540259, "MIT", "CentOS", "", "A utility for getting files from remote servers (FTP, HTTP, and others)", "401357a38601356be337ead6bf4470e0"},
			{intRef(), "pixman", "0.34.0", "1.el7", "x86_64", "pixman-0.34.0-1.el7.src.rpm", 698744, "MIT", "CentOS", "", "Pixel manipulation library", "4e6af972d04a88512ba81d3d303f05be"},
			{intRef(), "libpciaccess", "0.14", "1.el7", "x86_64", "libpciaccess-0.14-1.el7.src.rpm", 45649, "MIT", "CentOS", "", "PCI access library", "b980f4a133f488a872fea9465d6039ed"},
			{intRef(), "libuser", "0.60", "9.el7", "x86_64", "libuser-0.60-9.el7.src.rpm", 1952592, "LGPLv2+", "CentOS", "", "A user and group account administration library", "8fdd92e668b91208dfd916ebdce8a297"},
			{intRef(), "mesa-libgbm", "18.0.5", "4.el7_6", "x86_64", "mesa-18.0.5-4.el7_6.src.rpm", 58336, "MIT", "CentOS", "", "Mesa gbm library", "b3a621f15d077c63e5d2825f232f2985"},
			{intRef(2), "tar", "1.26", "34.el7", "x86_64", "tar-1.26-34.el7.src.rpm", 2838271, "GPLv3+", "CentOS", "", "A GNU file archiving program", "419ef9b4469e60d4d4d2599c1c8d79ba"},
			{intRef(), "docbook-dtds", "1.0", "60.el7", "noarch", "docbook-dtds-1.0-60.el7.src.rpm", 4634764, "Copyright only", "CentOS", "", "SGML and XML document type definitions for DocBook", "9c4a680c0f3fce42b9c9c04c668847bb"},
			{intRef(), "acl", "2.2.51", "14.el7", "x86_64", "acl-2.2.51-14.el7.src.rpm", 201225, "GPLv2+", "CentOS", "", "Access control list utilities", "885771078c3fe1efa43e3e5e0f2b5246"},
			{intRef(), "libXau", "1.0.8", "2.1.el7", "x86_64", "libXau-1.0.8-2.1.el7.src.rpm", 55529, "MIT", "CentOS", "", "Sample Authorization Protocol for X", "042d0ad179921f12e4b33e7123a0142b"},
			{intRef(), "ustr", "1.0.4", "16.el7", "x86_64", "ustr-1.0.4-16.el7.src.rpm", 285943, "MIT or LGPLv2+ or BSD", "CentOS", "", "String library, very low memory overhead, simple to import", "58bb2ebb64b048e24a4c7d5d51170f55"},
			{intRef(), "libX11", "1.6.5", "2.el7", "x86_64", "libX11-1.6.5-2.el7.src.rpm", 1325648, "MIT", "CentOS", "", "Core X11 protocol client library", "381f3dba6b6e3e3bea2241650651740a"},
			{intRef(2), "shadow-utils", "4.1.5.1", "24.el7", "x86_64", "shadow-utils-4.1.5.1-24.el7.src.rpm", 3534490, "BSD and GPLv2+", "CentOS", "", "Utilities for managing accounts and shadow password files", "19741e43ee0b142d310393848d75239f"},
			{intRef(), "libXrender", "0.9.10", "1.el7", "x86_64", "libXrender-0.9.10-1.el7.src.rpm", 46979, "MIT", "CentOS", "", "X.Org X11 libXrender runtime library", "abaf13bfdc95039bfe3c3f2db139e498"},
			{intRef(1), "hardlink", "1.0", "19.el7", "x86_64", "hardlink-1.0-19.el7.src.rpm", 16545, "GPL+", "CentOS", "", "Create a tree of hardlinks", "3a1bcd8e12bbdbced75d74b5919a039a"},
			{intRef(), "gdk-pixbuf2", "2.36.12", "3.el7", "x86_64", "gdk-pixbuf2-2.36.12-3.el7.src.rpm", 2927994, "LGPLv2+", "CentOS", "", "An image loading library", "bb7dabe82de2f808be2e1c12c260d669"},
			{intRef(), "util-linux", "2.23.2", "52.el7_5.1", "x86_64", "util-linux-2.23.2-52.el7_5.1.src.rpm", 8642769, "GPLv2 and GPLv2+ and LGPLv2+ and BSD with advertising and Public Domain", "CentOS", "", "A collection of basic system utilities", "8c402cdaecc9f30a146aa82ff30c8aea"},
			{intRef(), "libXpm", "3.5.12", "1.el7", "x86_64", "libXpm-3.5.12-1.el7.src.rpm", 116599, "MIT", "CentOS", "", "X.Org X11 libXpm runtime library", "513d610e407b9d461e3b358f86b58841"},
			{intRef(), "kpartx", "0.4.9", "119.el7_5.1", "x86_64", "device-mapper-multipath-0.4.9-119.el7_5.1.src.rpm", 41363, "GPL+", "CentOS", "", "Partition device manager for device-mapper devices", "fd61a26347cf0d090798768d7e952759"},
			{intRef(), "libXdamage", "1.1.4", "4.1.el7", "x86_64", "libXdamage-1.1.4-4.1.el7.src.rpm", 29688, "MIT", "CentOS", "", "X Damage extension library", "22c8af639cb4ad5be38747dce56a632b"},
			{intRef(7), "device-mapper-libs", "1.02.146", "4.el7", "x86_64", "lvm2-2.02.177-4.el7.src.rpm", 400551, "LGPLv2", "CentOS", "", "Device-mapper shared library", "f0b566ce4fe22594b7f5f246870d4214"},
			{intRef(), "gd", "2.0.35", "26.el7", "x86_64", "gd-2.0.35-26.el7.src.rpm", 554528, "MIT", "CentOS", "", "A graphics library for quick creation of PNG or JPEG images", "9d62ce2e46a14f71589c82e752d428e5"},
			{intRef(), "dracut", "033", "535.el7_5.1", "x86_64", "dracut-033-535.el7_5.1.src.rpm", 898198, "GPLv2+ and LGPLv2+", "CentOS", "", "Initramfs generator using udev", "22ffeb1421964a58b78b87eeb47014a7"},
			{intRef(), "gtk-update-icon-cache", "3.22.30", "3.el7", "x86_64", "gtk3-3.22.30-3.el7.src.rpm", 59878, "LGPLv2+", "CentOS", "", "Icon theme caching utility", "ebb747dd17ade2bce6534a352b8748b8"},
			{intRef(), "libXrandr", "1.5.1", "2.el7", "x86_64", "libXrandr-1.5.1-2.el7.src.rpm", 48626, "MIT", "CentOS", "", "X.Org X11 libXrandr runtime library", "08ffc02cefbf48f4e52f7a6ac4f8dc24"},
			{intRef(1), "dbus-libs", "1.10.24", "7.el7", "x86_64", "dbus-1.10.24-7.el7.src.rpm", 362584, "(GPLv2+ or AFL) and GPLv2+", "CentOS", "", "Libraries for accessing D-BUS", "7c544dc4e3020620da275ddf1c0de95c"},
			{intRef(), "libXi", "1.7.9", "1.el7", "x86_64", "libXi-1.7.9-1.el7.src.rpm", 70019, "MIT", "CentOS", "", "X.Org X11 libXi runtime library", "9801d7674c51b5de5fc926702853c7fb"},
			{intRef(), "elfutils-default-yama-scope", "0.170", "4.el7", "noarch", "elfutils-0.170-4.el7.src.rpm", 1810, "GPLv2+ or LGPLv3+", "CentOS", "", "Default yama attach scope sysctl setting", "186adf73da2641da5799e54b4ccfed5d"},
			{intRef(), "libXxf86vm", "1.1.4", "1.el7", "x86_64", "libXxf86vm-1.1.4-1.el7.src.rpm", 25680, "MIT", "CentOS", "", "X.Org X11 libXxf86vm runtime library", "cf59e013b0def500ab01ceffff473372"},
			{intRef(), "mesa-libGL", "18.0.5", "4.el7_6", "x86_64", "mesa-18.0.5-4.el7_6.src.rpm", 493744, "MIT", "CentOS", "", "Mesa libGL runtime libraries and DRI drivers", "78aa552da99eeaf6eb74fb7d830330ef"},
			{intRef(), "dbus-glib", "0.100", "7.el7", "x86_64", "dbus-glib-0.100-7.el7.src.rpm", 301237, "AFL and GPLv2+", "CentOS", "", "GLib bindings for D-Bus", "fea831a1bb3627640677db4d441a65c6"},
			{intRef(1), "libglvnd-egl", "1.0.1", "0.8.git5baa1e5.el7", "x86_64", "libglvnd-1.0.1-0.8.git5baa1e5.el7.src.rpm", 84096, "MIT", "CentOS", "", "EGL support for libglvnd", "d5065f9c6151d7df91ee0049843b7f67"},
			{intRef(), "cairo", "1.15.12", "3.el7", "x86_64", "cairo-1.15.12-3.el7.src.rpm", 1894412, "LGPLv2 or MPLv1.1", "CentOS", "", "A 2D graphics library", "436d55090051a62834a6f51448e87b6c"},
			{intRef(), "pango", "1.42.4", "1.el7", "x86_64", "pango-1.42.4-1.el7.src.rpm", 755391, "LGPLv2+", "CentOS", "", "System for layout and rendering of internationalized text", "073ea3cd24c952084ac30b40ff16043b"},
			{intRef(), "libxml2-python", "2.9.1", "6.el7_2.3", "x86_64", "libxml2-2.9.1-6.el7_2.3.src.rpm", 1503050, "MIT", "CentOS", "", "Python bindings for the libxml2 library", "4af3055a710e118b3e7c49cb1df4e784"},
			{intRef(), "gtk2", "2.24.31", "1.el7", "x86_64", "gtk2-2.24.31-1.el7.src.rpm", 13494501, "LGPLv2+", "CentOS", "", "The GIMP ToolKit (GTK+), a library for creating GUIs for X", "c77e31729af1a79df088cc9ec6817db1"},
			{intRef(), "graphviz", "2.30.1", "21.el7", "x86_64", "graphviz-2.30.1-21.el7.src.rpm", 4019006, "EPL", "CentOS", "", "Graph Visualization Tools", "38b18062976a1542c162fa5cc26baa31"},
			{intRef(), "python-gobject-base", "3.22.0", "1.el7_4.1", "x86_64", "pygobject3-3.22.0-1.el7_4.1.src.rpm", 1123114, "LGPLv2+ and MIT", "CentOS", "", "Python 2 bindings for GObject Introspection base package", "e7f47d020d89e7ecc10aa699c0f8a4f0"},
			{intRef(1), "perl-ExtUtils-ParseXS", "3.18", "3.el7", "noarch", "perl-ExtUtils-ParseXS-3.18-3.el7.src.rpm", 178634, "GPL+ or Artistic", "CentOS", "", "Module and a script for converting Perl XS code into C code", "5e2d79fd842dffc418eee0c494c154ee"},
			{intRef(), "yum-metadata-parser", "1.1.4", "10.el7", "x86_64", "yum-metadata-parser-1.1.4-10.el7.src.rpm", 58789, "GPLv2", "CentOS", "", "A fast metadata parser for yum", "929f0abd8c5671c69f86fce8d7cdd42d"},
			{intRef(0), "perl-ExtUtils-Install", "1.58", "294.el7_6", "noarch", "perl-5.16.3-294.el7_6.src.rpm", 76408, "GPL+ or Artistic", "CentOS", "", "Install files from here to there", "58a2205cc1b45007edc7b5e99e9b1919"},
			{intRef(), "python-pycurl", "7.19.0", "19.el7", "x86_64", "python-pycurl-7.19.0-19.el7.src.rpm", 241513, "LGPLv2+ or MIT", "CentOS", "", "A Python interface to libcurl", "d5152315d59bde1faaa203e68de6a64e"},
			{intRef(0), "perl-ExtUtils-Embed", "1.30", "294.el7_6", "noarch", "perl-5.16.3-294.el7_6.src.rpm", 17136, "GPL+ or Artistic", "CentOS", "", "Utilities for embedding Perl in C/C++ applications", "ae1cc1ad9326ec0f67b471dcafa444ab"},
			{intRef(), "python-iniparse", "0.4", "9.el7", "noarch", "python-iniparse-0.4-9.el7.src.rpm", 115166, "MIT", "CentOS", "", "Python Module for Accessing and Modifying Configuration Data in INI files", "3140d9efca99c4dd5215383bab36bca5"},
			{intRef(), "xmlto", "0.0.25", "7.el7", "x86_64", "xmlto-0.0.25-7.el7.src.rpm", 108322, "GPLv2+", "CentOS", "", "A tool for converting XML files to various formats", "a8c546288b0e9d1c6ce783fcaf87daad"},
			{intRef(), "python-chardet", "2.2.1", "1.el7_1", "noarch", "python-chardet-2.2.1-1.el7_1.src.rpm", 1156541, "LGPLv2", "CentOS", "", "Character encoding auto-detection in Python", "c70da920f85b12ecd25429913e3e861e"},
			{intRef(), "hmaccalc", "0.9.13", "4.el7", "x86_64", "hmaccalc-0.9.13-4.el7.src.rpm", 102013, "BSD", "CentOS", "", "Tools for computing and checking HMAC values for files", "bae644ffd0fdfbcccd9685c333f91476"},
			{intRef(), "hostname", "3.13", "3.el7", "x86_64", "hostname-3.13-3.el7.src.rpm", 19449, "GPLv2+", "CentOS", "", "Utility to set/show the host name or domain name", "677ae7d92bb29445d1e0b66ecf5cab81"},
			{intRef(), "zlib-devel", "1.2.7", "18.el7", "x86_64", "zlib-1.2.7-18.el7.src.rpm", 135536, "zlib and Boost", "CentOS", "", "Header files and libraries for Zlib development", "b2cb796d4a3938d3659cd6cd89584e57"},
			{intRef(), "pth", "2.0.7", "23.el7", "x86_64", "pth-2.0.7-23.el7.src.rpm", 267851, "LGPLv2+", "CentOS", "", "The GNU Portable Threads library", "2d6939b73767d0f96b02677bcbeef2e2"},
			{intRef(), "python-lxml", "3.2.1", "4.el7", "x86_64", "python-lxml-3.2.1-4.el7.src.rpm", 2722705, "BSD", "CentOS", "", "ElementTree-like Python bindings for libxml2 and libxslt", "551e22015379cdc27348499a00a3c7ca"},
			{intRef(), "javapackages-tools", "3.4.1", "11.el7", "noarch", "javapackages-tools-3.4.1-11.el7.src.rpm", 160151, "BSD", "CentOS", "", "Macros and scripts for Java packaging support", "e9320fd84907047334534191b1829353"},
			{intRef(), "gpgme", "1.3.2", "5.el7", "x86_64", "gpgme-1.3.2-5.el7.src.rpm", 547534, "LGPLv2+", "CentOS", "", "GnuPG Made Easy - high level crypto API", "d693313ae4184395e902c28ce87d4573"},
			{intRef(), "xorg-x11-fonts-Type1", "7.5", "9.el7", "noarch", "xorg-x11-fonts-7.5-9.el7.src.rpm", 883992, "MIT and Lucida and Public Domain", "CentOS", "", "Type1 fonts provided by the X Window System", "1be2bbefd0ff0dad1e2d72bd3c1d1075"},
			{intRef(), "yum-plugin-fastestmirror", "1.1.31", "46.el7_5", "noarch", "yum-utils-1.1.31-46.el7_5.src.rpm", 53895, "GPLv2+", "CentOS", "", "Yum plugin which chooses fastest repository from a mirrorlist", "d9f1b3ff9204795855f0f6cf09ab40dd"},
			{intRef(), "xz-devel", "5.2.2", "1.el7", "x86_64", "xz-5.2.2-1.el7.src.rpm", 168793, "LGPLv2+", "CentOS", "", "Devel libraries & headers for liblzma", "5c84b76faef54c4d1140637f303b9eac"},
			{intRef(32), "bind-license", "9.9.4", "61.el7_5.1", "noarch", "bind-9.9.4-61.el7_5.1.src.rpm", 26831, "ISC", "CentOS", "", "License of the BIND DNS suite", "8983808555b122a7220800907c368f23"},
			{intRef(), "lksctp-tools", "1.0.17", "2.el7", "x86_64", "lksctp-tools-1.0.17-2.el7.src.rpm", 232487, "GPLv2 and GPLv2+ and LGPLv2 and MIT", "CentOS", "", "User-space access to Linux Kernel SCTP", "335ef58f575dc5fd412a3514e39dd5c3"},
			{intRef(), "copy-jdk-configs", "3.3", "10.el7_5", "noarch", "copy-jdk-configs-3.3-10.el7_5.src.rpm", 16264, "BSD", "CentOS", "", "JDKs configuration files copier", "7027a10264c21f5b904ff16be487103c"},
			{intRef(), "yum-plugin-ovl", "1.1.31", "46.el7_5", "noarch", "yum-utils-1.1.31-46.el7_5.src.rpm", 22399, "GPLv2+", "CentOS", "", "Yum plugin to work around overlayfs issues", "5ee5adb1e57e68d35865f4c94181f473"},
			{intRef(), "passwd", "0.79", "4.el7", "x86_64", "passwd-0.79-4.el7.src.rpm", 429874, "BSD or GPL+", "CentOS", "", "An utility for setting or changing passwords using PAM", "bc5080b013d6a43fb00e57bba7517e0f"},
			{intRef(), "rootfiles", "8.1", "11.el7", "noarch", "rootfiles-8.1-11.el7.src.rpm", 599, "Public Domain", "CentOS", "", "The basic required files for the root user's directory", "b9b9eceee7ad38f868520e80f3404642"},
			{intRef(), "slang", "2.2.4", "11.el7", "x86_64", "slang-2.2.4-11.el7.src.rpm", 1881912, "GPLv2+", "CentOS", "", "The shared library for the S-Lang extension language", "82cf9088d54b58cb0ac91f8c2e033fb5"},
			{intRef(), "libgcc", "4.8.5", "36.el7_6.1", "x86_64", "gcc-4.8.5-36.el7_6.1.src.rpm", 179328, "GPLv3+ and GPLv3+ with exceptions and GPLv2+ with exceptions and LGPLv2+ and BSD", "CentOS", "", "GCC version 4.8 shared support library", "562e2ee43ee56c57acf69b14f6f1535b"},
			{intRef(), "glibc", "2.17", "260.el7_6.4", "x86_64", "glibc-2.17-260.el7_6.4.src.rpm", 14346480, "LGPLv2+ and LGPLv2+ with exceptions and GPLv2+", "CentOS", "", "The GNU libc libraries", "17e2b4067ebdf2993c799ddce8bb76e0"},
			{intRef(), "elfutils-libelf", "0.172", "2.el7", "x86_64", "elfutils-0.172-2.el7.src.rpm", 913443, "GPLv2+ or LGPLv3+", "CentOS", "", "Library to read and write ELF files", "383dff055bd0bc27b026d1359974e00e"},
			{intRef(), "rpm-libs", "4.11.3", "35.el7", "x86_64", "rpm-4.11.3-35.el7.src.rpm", 611536, "GPLv2+ and LGPLv2+ with exceptions", "CentOS", "", "Libraries for manipulating RPM packages", "2ac2673a85c45699ccf2e5d1a4eb2c4c"},
			{intRef(), "elfutils-libs", "0.172", "2.el7", "x86_64", "elfutils-0.172-2.el7.src.rpm", 793087, "GPLv2+ or LGPLv3+", "CentOS", "", "Libraries to handle compiled objects", "8d24c886e0655d29240085a75a5d7047"},
			{intRef(), "libmpc", "1.0.1", "3.el7", "x86_64", "libmpc-1.0.1-3.el7.src.rpm", 113833, "LGPLv3+ and GFDL", "CentOS", "", "C library for multiple precision complex arithmetic", "835abbe42dbf1e0f94d227d54c01f483"},
			{intRef(), "fipscheck", "1.4.1", "6.el7", "x86_64", "fipscheck-1.4.1-6.el7.src.rpm", 38839, "BSD", "CentOS", "", "A library for integrity verification of FIPS validated modules", "a2e0953263a6285a00476819997c1caf"},
			{intRef(), "net-tools", "2.0", "0.24.20131004git.el7", "x86_64", "net-tools-2.0-0.24.20131004git.el7.src.rpm", 939930, "GPLv2+", "CentOS", "", "Basic networking tools", "8da21eddba8648fde043386ed2b2ed95"},
			{intRef(), "libgomp", "4.8.5", "36.el7_6.1", "x86_64", "gcc-4.8.5-36.el7_6.1.src.rpm", 212184, "GPLv3+ and GPLv3+ with exceptions and GPLv2+ with exceptions and LGPLv2+ and BSD", "CentOS", "", "GCC OpenMP v3.0 shared support library", "35872c42fa6e4867151f828230a2636f"},
			{intRef(), "m4", "1.4.16", "10.el7", "x86_64", "m4-1.4.16-10.el7.src.rpm", 525707, "GPLv3+", "CentOS", "", "The GNU macro processor", "a71de093f3eba336d4fe4044fb6db349"},
			{intRef(), "apr-util", "1.5.2", "6.el7", "x86_64", "apr-util-1.5.2-6.el7.src.rpm", 198751, "ASL 2.0", "CentOS", "", "Apache Portable Runtime Utility library", "e4abdd79021510c0a5eb7fdea8389881"},
			{intRef(), "groff-base", "1.22.2", "8.el7", "x86_64", "groff-1.22.2-8.el7.src.rpm", 3453946, "GPLv3+ and GFDL and BSD and MIT", "CentOS", "", "Parts of the groff formatting system required to display manual pages", "591d2a12420abc9fa9934414fc0987f4"},
			{intRef(), "libunistring", "0.9.3", "9.el7", "x86_64", "libunistring-0.9.3-9.el7.src.rpm", 1145761, "LGPLv3+", "CentOS", "", "GNU Unicode string library", "b1b9755e18cb312740d96d89e61ef70e"},
			{intRef(), "libcroco", "0.6.12", "4.el7", "x86_64", "libcroco-0.6.12-4.el7.src.rpm", 320955, "LGPLv2", "CentOS", "", "A CSS2 parsing library", "8f570a2d6cb38d68f7c099b8130a1068"},
			{intRef(), "patch", "2.7.1", "10.el7_5", "x86_64", "patch-2.7.1-10.el7_5.src.rpm", 215449, "GPLv3+", "CentOS", "", "Utility for modifying/upgrading files", "3c1cd445c2a45f19b9bb46c19a4a943c"},
			{intRef(), "zip", "3.0", "11.el7", "x86_64", "zip-3.0-11.el7.src.rpm", 815173, "BSD", "CentOS", "", "A file compression and packaging utility compatible with PKZIP", "f49ae6270a053005b67e417123691ca7"},
			{intRef(), "gettext", "0.19.8.1", "2.el7", "x86_64", "gettext-0.19.8.1-2.el7.src.rpm", 5029340, "GPLv3+ and LGPLv2+", "CentOS", "", "GNU libraries and utilities for producing multi-lingual messages", "e8b96713d3d1d5080233abae6e9c5e10"},
			{intRef(), "perl-HTTP-Tiny", "0.033", "3.el7", "noarch", "perl-HTTP-Tiny-0.033-3.el7.src.rpm", 97210, "GPL+ or Artistic", "CentOS", "", "Small, simple, correct HTTP/1.1 client", "6133c7c1aee2bfeee269de4592b25884"},
			{intRef(), "perl-Pod-Perldoc", "3.20", "4.el7", "noarch", "perl-Pod-Perldoc-3.20-4.el7.src.rpm", 166910, "GPL+ or Artistic", "CentOS", "", "Look up Perl documentation in Pod format", "29a8a33b24ddf4e0e95ad8d2484cc277"},
			{intRef(), "perl-Text-ParseWords", "3.29", "4.el7", "noarch", "perl-Text-ParseWords-3.29-4.el7.src.rpm", 16431, "GPL+ or Artistic", "CentOS", "", "Parse text into an array of tokens or array of arrays", "58a96328cb0e6dd435d15af542467a38"},
			{intRef(), "perl-Pod-Usage", "1.63", "3.el7", "noarch", "perl-Pod-Usage-1.63-3.el7.src.rpm", 44671, "GPL+ or Artistic", "CentOS", "", "Print a usage message from embedded POD documentation", "1db34851ba13011739022f031438e15c"},
			{intRef(), "perl-Socket", "2.010", "4.el7", "x86_64", "perl-Socket-2.010-4.el7.src.rpm", 114497, "GPL+ or Artistic", "CentOS", "", "Networking constants and support functions", "5cda40abac38b41918352e9c75a1e651"},
			{intRef(4), "perl-Time-HiRes", "1.9725", "3.el7", "x86_64", "perl-Time-HiRes-1.9725-3.el7.src.rpm", 94069, "GPL+ or Artistic", "CentOS", "", "High resolution alarm, sleep, gettimeofday, interval timers", "43f7c93d02750221738a8c3bc845a1ca"},
			{intRef(), "perl-Carp", "1.26", "244.el7", "noarch", "perl-Carp-1.26-244.el7.src.rpm", 28276, "GPL+ or Artistic", "CentOS", "", "Alternative warn and die for modules", "0f9ffc1d8836311235869048667422e9"},
			{intRef(), "perl-Exporter", "5.68", "3.el7", "noarch", "perl-Exporter-5.68-3.el7.src.rpm", 56612, "GPL+ or Artistic", "CentOS", "", "Implements default import method for modules", "d6e7deb74b149e4b7aa23a998a127bb4"},
			{intRef(), "perl-Time-Local", "1.2300", "2.el7", "noarch", "perl-Time-Local-1.2300-2.el7.src.rpm", 44062, "GPL+ or Artistic", "CentOS", "", "Efficiently compute time from local and GMT time", "8dba0f2d49849dc3990cfc9e91d5ec6b"},
			{intRef(), "perl-threads-shared", "1.43", "6.el7", "x86_64", "perl-threads-shared-1.43-6.el7.src.rpm", 73972, "GPL+ or Artistic", "CentOS", "", "Perl extension for sharing data structures between threads", "cd3e1ec9dfe8b5e9121fcd53b41aa197"},
			{intRef(), "perl-File-Path", "2.09", "2.el7", "noarch", "perl-File-Path-2.09-2.el7.src.rpm", 50067, "GPL+ or Artistic", "CentOS", "", "Create or remove directory trees", "94d54ef6ce6893ae2a4400c4b0ce79b7"},
			{intRef(), "perl-Scalar-List-Utils", "1.27", "248.el7", "x86_64", "perl-Scalar-List-Utils-1.27-248.el7.src.rpm", 67994, "GPL+ or Artistic", "CentOS", "", "A selection of general-utility scalar and list subroutines", "2fb261c150fc06b1a978809c5a9c706e"},
			{intRef(), "perl-Getopt-Long", "2.40", "3.el7", "noarch", "perl-Getopt-Long-2.40-3.el7.src.rpm", 134846, "GPLv2+ or Artistic", "CentOS", "", "Extended processing of command line options", "e4fc3f2fed36897ae178ecd5d9475217"},
			{intRef(), "perl-Thread-Queue", "3.02", "2.el7", "noarch", "perl-Thread-Queue-3.02-2.el7.src.rpm", 27642, "GPL+ or Artistic", "CentOS", "", "Thread-safe queues", "29d052232cb1b86133bd838739672a15"},
			{intRef(), "perl-TermReadKey", "2.30", "20.el7", "x86_64", "perl-TermReadKey-2.30-20.el7.src.rpm", 59999, "(Copyright only) and (Artistic or GPL+)", "CentOS", "", "A perl module for simple terminal control", "48356fe925819f41d6446ffc09d1b21d"},
			{intRef(), "autoconf", "2.69", "11.el7", "noarch", "autoconf-2.69-11.el7.src.rpm", 2320371, "GPLv2+ and GFDL", "CentOS", "", "A GNU tool for automatically configuring source code", "24e721d7349c86c3df3cf4727e598a54"},
			{intRef(), "perl-Test-Harness", "3.28", "3.el7", "noarch", "perl-Test-Harness-3.28-3.el7.src.rpm", 607239, "GPL+ or Artistic", "CentOS", "", "Run Perl standard test scripts with statistics", "f131d5755e383c37135b023649a236b3"},
			{intRef(), "kernel-debug-devel", "3.10.0", "957.10.1.el7", "x86_64", "kernel-3.10.0-957.10.1.el7.src.rpm", 39270753, "GPLv2", "CentOS", "", "Development package for building kernel modules to match the debug kernel", "420c2ea8fef42dd04e53294121aed12a"},
			{intRef(), "boost-thread", "1.53.0", "27.el7", "x86_64", "boost-1.53.0-27.el7.src.rpm", 95658, "Boost and MIT and Python", "CentOS", "", "Run-Time component of boost thread library", "f93accf1d38b62ca79ba4424d7a603a8"},
			{intRef(), "openssh", "7.4p1", "16.el7", "x86_64", "openssh-7.4p1-16.el7.src.rpm", 1995364, "BSD", "CentOS", "", "An open source implementation of SSH protocol versions 1 and 2", "805b1fb17b3b7a41b2df9fd89b75e377"},
			{intRef(), "elfutils", "0.172", "2.el7", "x86_64", "elfutils-0.172-2.el7.src.rpm", 811817, "GPLv3+ and (GPLv2+ or LGPLv3+)", "CentOS", "", "A collection of utilities and DSOs to handle ELF files and DWARF data", "b405152d92211d83d371d791d702b883"},
			{intRef(), "systemd", "219", "62.el7_6.5", "x86_64", "systemd-219-62.el7_6.5.src.rpm", 24440903, "LGPLv2+ and MIT and GPLv2+", "CentOS", "", "A System and Service Manager", "20127ecb38e5f361019f50f3ae7bc8a7"},
			{intRef(), "rsync", "3.1.2", "4.el7", "x86_64", "rsync-3.1.2-4.el7.src.rpm", 834618, "GPLv3+", "CentOS", "", "A program for synchronizing files over a network", "a50dfc05a59066dea8ab23dda156a45a"},
			{intRef(), "file", "5.11", "35.el7", "x86_64", "file-5.11-35.el7.src.rpm", 67448, "BSD", "CentOS", "", "A utility for determining file types", "d993604425647d31a529f678397db7a4"},
			{intRef(), "dwz", "0.11", "3.el7", "x86_64", "dwz-0.11-3.el7.src.rpm", 225103, "GPLv2+ and GPLv3+", "CentOS", "", "DWARF optimization and duplicate removal tool", "28a047bc76d0ad80e6938a94898129ca"},
			{intRef(), "dyninst", "9.3.1", "2.el7", "x86_64", "dyninst-9.3.1-2.el7.src.rpm", 13049000, "LGPLv2+", "CentOS", "", "An API for Run-time Code Generation", "f32826510026b19cc65dfdfd0cdf8bf6"},
			{intRef(), "libstdc++-devel", "4.8.5", "36.el7_6.1", "x86_64", "gcc-4.8.5-36.el7_6.1.src.rpm", 8436804, "GPLv3+ and GPLv3+ with exceptions and GPLv2+ with exceptions and LGPLv2+ and BSD", "CentOS", "", "Header files and libraries for C++ development", "90dc72456e50813f4a311d17514e21eb"},
			{intRef(), "audit-libs", "2.8.4", "4.el7", "x86_64", "audit-2.8.4-4.el7.src.rpm", 256362, "LGPLv2+", "CentOS", "", "Dynamic library for libaudit", "a57039d27b473fc766688886d53ecce2"},
			{intRef(), "giflib", "4.1.6", "9.el7", "x86_64", "giflib-4.1.6-9.el7.src.rpm", 90124, "MIT", "CentOS", "", "Library for manipulating GIF format image files", "eb75dac60250315a841f867a62fb741a"},
			{intRef(1), "java-1.8.0-openjdk-devel", "1.8.0.212.b04", "0.el7_6", "x86_64", "java-1.8.0-openjdk-1.8.0.212.b04-0.el7_6.src.rpm", 42212373, "ASL 1.1 and ASL 2.0 and BSD and BSD with advertising and GPL+ and GPLv2 and GPLv2 with exceptions and IJG and LGPLv2+ and MIT and MPLv2.0 and Public Domain and W3C and zlib", "CentOS", "", "OpenJDK Development Environment 8", "45a0d21f2578d85ae098d09ee9bd1fd5"},
			{intRef(), "elfutils-devel", "0.172", "2.el7", "x86_64", "elfutils-0.172-2.el7.src.rpm", 306300, "GPLv2+ or LGPLv3+", "CentOS", "", "Development libraries to handle compiled objects", "669b26d711acb84682eaeb8b9aae0ae3"},
			{intRef(), "pciutils-libs", "3.5.1", "3.el7", "x86_64", "pciutils-3.5.1-3.el7.src.rpm", 72691, "GPLv2+", "CentOS", "", "Linux PCI library", "ea4cb9fe5e425fffdd0ab23f74ad2da8"},
			{intRef(), "python", "2.7.5", "77.el7_6", "x86_64", "python-2.7.5-77.el7_6.src.rpm", 80907, "Python", "CentOS", "", "An interpreted, interactive, object-oriented programming language", "55d1e1f07dadda414e965325e7b8c0aa"},
			{intRef(), "newt", "0.52.15", "4.el7", "x86_64", "newt-0.52.15-4.el7.src.rpm", 185196, "LGPLv2", "CentOS", "", "A library for text mode user interfaces", "0d10a144053e012313ebe3967f229985"},
			{intRef(), "numactl-devel", "2.0.9", "7.el7", "x86_64", "numactl-2.0.9-7.el7.src.rpm", 25174, "LGPLv2 and GPLv2", "CentOS", "", "Development package for building Applications that use numa", "2e4e68c0bac12b338933af591394fc20"},
			{intRef(), "python-devel", "2.7.5", "77.el7_6", "x86_64", "python-2.7.5-77.el7_6.src.rpm", 1104386, "Python", "CentOS", "", "The libraries and header files needed for Python development", "e3647e534025d35fabf5382bdb27c0f4"},
			{intRef(), "ncurses-devel", "5.9", "14.20130511.el7_4", "x86_64", "ncurses-5.9-14.20130511.el7_4.src.rpm", 2253171, "MIT", "CentOS", "", "Development files for the ncurses library", "7aff4b04dd41595d646914e778a1f8ab"},
			{intRef(), "libproxy", "0.4.11", "11.el7", "x86_64", "libproxy-0.4.11-11.el7.src.rpm", 164014, "LGPLv2+", "CentOS", "", "A library handling all the details of proxy configuration", "5238ea30a29e5163ee6010d2696acb1d"},
			{intRef(), "tzdata", "2018e", "3.el7", "noarch", "tzdata-2018e-3.el7.src.rpm", 1966505, "Public Domain", "CentOS", "", "Timezone data", "a1e5c65f5b87e33b23419b153155db0a"},
			{intRef(1), "make", "3.82", "23.el7", "x86_64", "make-3.82-23.el7.src.rpm", 1160684, "GPLv2+", "CentOS", "", "A GNU tool which simplifies the build process for users", "9e5ab17428b36e59a9d071c1fd1c81de"},
			{intRef(), "ncurses", "5.9", "14.20130511.el7_4", "x86_64", "ncurses-5.9-14.20130511.el7_4.src.rpm", 439378, "MIT", "CentOS", "", "Ncurses support utilities", "8ca93e2831102818759a22e22e871268"},
			{intRef(), "git", "1.8.3.1", "20.el7", "x86_64", "git-1.8.3.1-20.el7.src.rpm", 23232282, "GPLv2", "CentOS", "", "Fast Version Control System", "bad9f26b37b7b8a74a69a038b773def4"},
			{intRef(), "filesystem", "3.2", "25.el7", "x86_64", "filesystem-3.2-25.el7.src.rpm", 0, "Public Domain", "CentOS", "", "The basic directory layout for a Linux system", "93e1c91a2dbd4d3ac61fb70a40a7c228"},
			{intRef(), "bzip2", "1.0.6", "13.el7", "x86_64", "bzip2-1.0.6-13.el7.src.rpm", 83791, "BSD", "CentOS", "", "A file compression utility", "704f597825711270b0ba2af51bebecae"},
			{intRef(), "nspr", "4.19.0", "1.el7_5", "x86_64", "nspr-4.19.0-1.el7_5.src.rpm", 287728, "MPLv2.0", "CentOS", "", "Netscape Portable Runtime", "06e8af105f3a5832065a90c5eb12c064"},
			{intRef(1), "emacs-filesystem", "24.3", "22.el7", "noarch", "emacs-24.3-22.el7.src.rpm", 0, "GPLv3+", "CentOS", "", "Emacs filesystem layout", "54c6f52516764089c58bfebe31187c2d"},
			{intRef(), "libffi", "3.0.13", "18.el7", "x86_64", "libffi-3.0.13-18.el7.src.rpm", 47766, "MIT and Public Domain", "CentOS", "", "A portable foreign function interface library", "f05f7a3749342b925fa10ca7de2f0116"},
			{intRef(), "gcc", "4.8.5", "36.el7_6.1", "x86_64", "gcc-4.8.5-36.el7_6.1.src.rpm", 39238921, "GPLv3+ and GPLv3+ with exceptions and GPLv2+ with exceptions and LGPLv2+ and BSD", "CentOS", "", "Various compilers (C, C++, Objective-C, Java, ...)", "82a79193e96339bd9ee093bdb0c53c1a"},
			{intRef(), "libsepol", "2.5", "8.1.el7", "x86_64", "libsepol-2.5-8.1.el7.src.rpm", 686568, "LGPLv2+", "CentOS", "", "SELinux binary policy manipulation library", "b107891426af43d06b02a191f16a490c"},
			{intRef(), "redhat-rpm-config", "9.1.0", "87.el7.centos", "noarch", "redhat-rpm-config-9.1.0-87.el7.centos.src.rpm", 174572, "GPL+", "CentOS", "", "CentOS specific rpm configuration files", "fa59b48b92a6e9b34c9fad064eafb839"},
			{intRef(), "gawk", "4.0.2", "4.el7_3.1", "x86_64", "gawk-4.0.2-4.el7_3.1.src.rpm", 2435978, "GPLv3+ and GPL and LGPLv3+ and LGPL and BSD", "CentOS", "", "The GNU version of the awk text processing utility", "5efa8feaeb12d57cda1a9bcd9c8af917"},
			{intRef(), "rpm-build", "4.11.3", "35.el7", "x86_64", "rpm-4.11.3-35.el7.src.rpm", 328409, "GPLv2+", "CentOS", "", "Scripts and executable programs used to build packages", "d2e83d31fc34e6cfb1c8ff7948254716"},
			{intRef(), "grep", "2.20", "3.el7", "x86_64", "grep-2.20-3.el7.src.rpm", 1195131, "GPLv3+", "CentOS", "", "Pattern matching utilities", "92bc957d1c4fc3764e134d8f82544abe"},
			{intRef(), "gcc-c++", "4.8.5", "36.el7_6.1", "x86_64", "gcc-4.8.5-36.el7_6.1.src.rpm", 17257523, "GPLv3+ and GPLv3+ with exceptions and GPLv2+ with exceptions and LGPLv2+ and BSD", "CentOS", "", "C++ support for GCC", "8f929d2776e8a8b345b4f6d4066dfaa0"},
			{intRef(), "libverto", "0.2.5", "4.el7", "x86_64", "libverto-0.2.5-4.el7.src.rpm", 23060, "MIT", "CentOS", "", "Main loop abstraction library", "341bae036f22ce087486f4d9e180caee"},
			{intRef(), "flex", "2.5.37", "6.el7", "x86_64", "flex-2.5.37-6.el7.src.rpm", 757468, "BSD and LGPLv2+", "CentOS", "", "A tool for creating scanners (text pattern recognizers)", "b18d609b317f948f26c59178fa9986cb"},
			{intRef(1), "openssl-libs", "1.0.2k", "12.el7", "x86_64", "openssl-1.0.2k-12.el7.src.rpm", 3200172, "OpenSSL", "CentOS", "", "A general purpose cryptography library with TLS implementation", "9213b8a36ef5936ac90e2630803e659c"},
			{intRef(1), "doxygen", "1.8.5", "3.el7", "x86_64", "doxygen-1.8.5-3.el7.src.rpm", 15389684, "GPL+", "CentOS", "", "A documentation system for C/C++", "8338fd066b89d4aa9122a545a36ce50e"},
			{intRef(), "xz-libs", "5.2.2", "1.el7", "x86_64", "xz-5.2.2-1.el7.src.rpm", 239967, "LGPLv2+", "CentOS", "", "Libraries for decoding LZMA compression", "7701c542dc8fcd5a38ee52e8786d7cf4"},
			{intRef(), "byacc", "1.9.20130304", "3.el7", "x86_64", "byacc-1.9.20130304-3.el7.src.rpm", 135161, "Public Domain", "CentOS", "", "Berkeley Yacc, a parser generator", "da257ad856b5658321480e85cb7c60c2"},
			{intRef(), "libgpg-error", "1.12", "3.el7", "x86_64", "libgpg-error-1.12-3.el7.src.rpm", 350865, "LGPLv2+", "CentOS", "", "Library for error values used by GnuPG components", "a3933d4ce43c07fa9305e8765d78d099"},
			{intRef(), "freetype", "2.8", "12.el7_6.1", "x86_64", "freetype-2.8-12.el7_6.1.src.rpm", 824497, "(FTL or GPLv2+) and BSD and MIT and Public Domain and zlib with acknowledgement", "CentOS", "", "A free and portable font rendering engine", "76a6a154ceceeecc7df8ae70cc1bb6a8"},
			{intRef(), "lua", "5.1.4", "15.el7", "x86_64", "lua-5.1.4-15.el7.src.rpm", 640319, "MIT", "CentOS", "", "Powerful light-weight programming language", "d5973eb7ca2bb665f5fbb7141f2e27a3"},
			{intRef(), "mesa-libglapi", "18.0.5", "4.el7_6", "x86_64", "mesa-18.0.5-4.el7_6.src.rpm", 204320, "MIT", "CentOS", "", "Mesa shared glapi", "5024231c094c8ca262bd9112e9fd662e"},
			{intRef(), "libmount", "2.23.2", "52.el7_5.1", "x86_64", "util-linux-2.23.2-52.el7_5.1.src.rpm", 273965, "LGPLv2+", "CentOS", "", "Device mounting library", "d084f24c36b7968ed5fc4a34940980ab"},
			{intRef(), "libxshmfence", "1.2", "1.el7", "x86_64", "libxshmfence-1.2-1.el7.src.rpm", 11160, "MIT", "CentOS", "", "X11 shared memory fences", "a4d00670969adcf52e88026b2605d5a8"},
			{intRef(), "gzip", "1.5", "10.el7", "x86_64", "gzip-1.5-10.el7.src.rpm", 250440, "GPLv3+ and GFDL", "CentOS", "", "The GNU data compression program", "adcb4df7dfe7866e38b5bf42d54d63ae"},
			{intRef(), "fontpackages-filesystem", "1.44", "8.el7", "noarch", "fontpackages-1.44-8.el7.src.rpm", 0, "Public Domain", "CentOS", "", "Directories used by font packages", "dbc95bc0a2568eb82f59c004462d90ab"},
			{intRef(), "diffutils", "3.3", "4.el7", "x86_64", "diffutils-3.3-4.el7.src.rpm", 1065157, "GPLv3+", "CentOS", "", "A GNU collection of diff utilities", "d8c60275b4c6d54e8bf7ee19cf0c4f2b"},
			{intRef(), "fontconfig", "2.13.0", "4.3.el7", "x86_64", "fontconfig-2.13.0-4.3.el7.src.rpm", 666475, "MIT and Public Domain and UCD", "CentOS", "", "Font configuration and customization library", "6db8946075bf52456fa1ab75d05830e0"},
			{intRef(), "urw-fonts", "2.4", "16.el7", "noarch", "urw-fonts-2.4-16.el7.src.rpm", 4389400, "GPL+", "CentOS", "", "Free versions of the 35 standard PostScript fonts.", "eb65fa0afd148b6f8da9ccf2de0082fc"},
			{intRef(), "nss-softokn", "3.36.0", "5.el7_5", "x86_64", "nss-softokn-3.36.0-5.el7_5.src.rpm", 1130670, "MPLv2.0", "CentOS", "", "Network Security Services Softoken Module", "7b8391d5b6843722982a616a8542d58f"},
			{intRef(), "jbigkit-libs", "2.0", "11.el7", "x86_64", "jbigkit-2.0-11.el7.src.rpm", 105882, "GPLv2+", "CentOS", "", "JBIG1 lossless image compression library", "1ed20c50fdc4614601403474e8240db8"},
			{intRef(), "libassuan", "2.1.0", "3.el7", "x86_64", "libassuan-2.1.0-3.el7.src.rpm", 155391, "LGPLv2+ and GPLv3+", "CentOS", "", "GnuPG IPC library", "c4c94aadc9da58f576b4ef238d0136cb"},
			{intRef(), "pcsc-lite", "1.8.8", "8.el7", "x86_64", "pcsc-lite-1.8.8-8.el7.src.rpm", 634433, "BSD", "CentOS", "", "PC/SC Lite smart card framework and applications", "420daf792bbfe95e2dd13a8087d72eb2"},
			{intRef(), "pyparsing", "1.5.6", "9.el7", "noarch", "pyparsing-1.5.6-9.el7.src.rpm", 375948, "MIT", "CentOS", "", "An object-oriented approach to text processing", "61b409d502448705a714d86288ee231a"},
			{intRef(), "boost-regex", "1.53.0", "27.el7", "x86_64", "boost-1.53.0-27.el7.src.rpm", 2122578, "Boost and MIT and Python", "CentOS", "", "Run-Time component of boost regular expression library", "b6571ff0209881eb78ca1700bba3edd2"},
			{intRef(1), "pkgconfig", "0.27.1", "4.el7", "x86_64", "pkgconfig-0.27.1-4.el7.src.rpm", 105522, "GPLv2+", "CentOS", "", "A tool for determining compilation options", "ef58117c9c617599952c0642ff5e8d00"},
			{intRef(), "libwayland-client", "1.15.0", "1.el7", "x86_64", "wayland-1.15.0-1.el7.src.rpm", 64076, "MIT", "CentOS", "", "Wayland client library", "0c6b8b38625e1cec4ecba0f23702a86b"},
			{intRef(), "cyrus-sasl-lib", "2.1.26", "23.el7", "x86_64", "cyrus-sasl-2.1.26-23.el7.src.rpm", 396911, "BSD with advertising", "CentOS", "", "Shared libraries needed by applications which use Cyrus SASL", "6d06f1337b143c25bc7358255d433006"},
			{intRef(), "harfbuzz", "1.7.5", "2.el7", "x86_64", "harfbuzz-1.7.5-2.el7.src.rpm", 709362, "MIT", "CentOS", "", "Text shaping library", "bb9904035f958cfff8a59f4e79ffeaa0"},
			{intRef(), "libcurl", "7.29.0", "46.el7", "x86_64", "curl-7.29.0-46.el7.src.rpm", 435192, "MIT", "CentOS", "", "A library for getting files from web servers", "2d99038a0bef38275774024117436284"},
			{intRef(), "hwdata", "0.252", "9.1.el7", "x86_64", "hwdata-0.252-9.1.el7.src.rpm", 13726739, "GPLv2+", "CentOS", "", "Hardware identification and configuration data", "c1fa015c9d2928f58e264380e730eae1"},
			{intRef(), "openldap", "2.4.44", "15.el7_5", "x86_64", "openldap-2.4.44-15.el7_5.src.rpm", 1037299, "OpenLDAP", "CentOS", "", "LDAP support libraries", "0c7d41097a0100c3e40ac4aff0d3cdd8"},
			{intRef(), "sgml-common", "0.6.3", "39.el7", "noarch", "sgml-common-0.6.3-39.el7.src.rpm", 170882, "GPL+", "CentOS", "", "Common SGML catalog and DTD files", "a867d2e1f95cfad7ecefbabf4f6461d5"},
			{intRef(), "pinentry", "0.8.1", "17.el7", "x86_64", "pinentry-0.8.1-17.el7.src.rpm", 159929, "GPLv2+", "CentOS", "", "Collection of simple PIN or passphrase entry dialogs", "740fabe008451f11fee22ff086fb13d3"},
			{intRef(), "libxcb", "1.13", "1.el7", "x86_64", "libxcb-1.13-1.el7.src.rpm", 1011680, "MIT", "CentOS", "", "A C binding to the X11 protocol", "923a4fbb3fc66c81699e3c0354c2f8d2"},
			{intRef(), "libsemanage", "2.5", "11.el7", "x86_64", "libsemanage-2.5-11.el7.src.rpm", 302369, "LGPLv2+", "CentOS", "", "SELinux binary policy manipulation library", "d2fd300c5abc3561b16a685b1f2cbc5d"},
			{intRef(), "libXfixes", "5.0.3", "1.el7", "x86_64", "libXfixes-5.0.3-1.el7.src.rpm", 26903, "MIT", "CentOS", "", "X Fixes library", "f7535a332d39d22e1853767cb29c38b7"},
			{intRef(), "qrencode-libs", "3.4.1", "3.el7", "x86_64", "qrencode-3.4.1-3.el7.src.rpm", 126732, "LGPLv2+", "CentOS", "", "QR Code encoding library - Shared libraries", "ac1de0c46bc1886e12ea80771951cde2"},
			{intRef(), "libXmu", "1.1.2", "2.el7", "x86_64", "libXmu-1.1.2-2.el7.src.rpm", 172054, "MIT", "CentOS", "", "X.Org X11 libXmu/libXmuu runtime libraries", "30004e6ab36e030aae8da010bfc697f9"},
			{intRef(), "procps-ng", "3.3.10", "17.el7_5.2", "x86_64", "procps-ng-3.3.10-17.el7_5.2.src.rpm", 760500, "GPL+ and GPLv2 and GPLv2+ and GPLv3+ and LGPLv2+", "CentOS", "", "System and process monitoring utilities", "57c9fa477ce98fca4e7dcc5ed6b41a7a"},
			{intRef(), "ghostscript", "9.07", "31.el7_6.10", "x86_64", "ghostscript-9.07-31.el7_6.10.src.rpm", 17357534, "AGPLv3+ and Redistributable, no modification permitted", "CentOS", "", "A PostScript interpreter and renderer", "f02a30002d050db973563bc1e24ee05e"},
			{intRef(), "kmod", "20", "21.el7", "x86_64", "kmod-20-21.el7.src.rpm", 243127, "GPLv2+", "CentOS", "", "Linux kernel module management utilities", "47a0704853382255bfbc519fa9dd1a3d"},
			{intRef(), "libXft", "2.3.2", "2.el7", "x86_64", "libXft-2.3.2-2.el7.src.rpm", 125933, "MIT", "CentOS", "", "X.Org X11 libXft runtime library", "dd826dcfb492ed42a1d822c74eec0195"},
			{intRef(1), "libglvnd-glx", "1.0.1", "0.8.git5baa1e5.el7", "x86_64", "libglvnd-1.0.1-0.8.git5baa1e5.el7.src.rpm", 657304, "MIT", "CentOS", "", "GLX support for libglvnd", "196e956024a79b196c8ab662badf9371"},
			{intRef(), "iputils", "20160308", "10.el7", "x86_64", "iputils-20160308-10.el7.src.rpm", 343497, "BSD and GPLv2+", "CentOS", "", "Network monitoring tools including ping", "dfa219e6db737a1eb1b5cb23b532e858"},
			{intRef(), "mesa-libEGL", "18.0.5", "4.el7_6", "x86_64", "mesa-18.0.5-4.el7_6.src.rpm", 232401, "MIT", "CentOS", "", "Mesa libEGL runtime libraries", "6d3c4169b130edb2bb172fa3b8d70e8e"},
			{intRef(), "librsvg2", "2.40.20", "1.el7", "x86_64", "librsvg2-2.40.20-1.el7.src.rpm", 308951, "LGPLv2+", "CentOS", "", "An SVG library based on cairo", "c6c998edaaee8f2e448cf14713063d4b"},
			{intRef(), "libdb-devel", "5.3.21", "24.el7", "x86_64", "libdb-5.3.21-24.el7.src.rpm", 129076, "BSD and LGPLv2 and Sleepycat", "CentOS", "", "C development files for the Berkeley DB library", "32730435809421188228bfcc6b926ad0"},
			{intRef(4), "perl-devel", "5.16.3", "294.el7_6", "x86_64", "perl-5.16.3-294.el7_6.src.rpm", 1930081, "GPL+ or Artistic", "CentOS", "", "Header #files for use in perl development", "c7a84d1e5f0e38b67d556a6f9b0cd4f5"},
			{intRef(), "python-urlgrabber", "3.10", "8.el7", "noarch", "python-urlgrabber-3.10-8.el7.src.rpm", 500670, "LGPLv2+", "CentOS", "", "A high-level cross-protocol url-grabber", "8a0c6c349d305a6c89d04e4d661e85a9"},
			{intRef(), "pesign", "0.109", "10.el7", "x86_64", "pesign-0.109-10.el7.src.rpm", 368213, "GPLv2", "CentOS", "", "Signing utility for UEFI binaries", "394bdc1b74dff953dc42958ec13bac6c"},
			{intRef(), "python-kitchen", "1.1.1", "5.el7", "noarch", "python-kitchen-1.1.1-5.el7.src.rpm", 1465161, "LGPLv2+", "CentOS", "", "Small, useful pieces of code to make python coding easier", "917912ebf6164135f1f9871150432977"},
			{intRef(), "elfutils-libelf-devel", "0.172", "2.el7", "x86_64", "elfutils-0.172-2.el7.src.rpm", 32387, "GPLv2+ or LGPLv3+", "CentOS", "", "Development support for libelf", "af2405af86a547ddda01157a288a1384"},
			{intRef(), "gnupg2", "2.0.22", "5.el7_5", "x86_64", "gnupg2-2.0.22-5.el7_5.src.rpm", 6637796, "GPLv3+", "CentOS", "", "Utility for secure communication and data storage", "a293dfefd6596bfe5c54ba68377e334e"},
			{intRef(), "ttmkfdir", "3.0.9", "42.el7", "x86_64", "ttmkfdir-3.0.9-42.el7.src.rpm", 105418, "LGPLv2+", "CentOS", "", "Utility to create fonts.scale files for truetype fonts", "2560840e51852145eaf4c4fd7837f338"},
			{intRef(), "pygpgme", "0.3", "9.el7", "x86_64", "pygpgme-0.3-9.el7.src.rpm", 197501, "LGPLv2+", "CentOS", "", "Python module for working with OpenPGP messages", "67b8c720f59a063adbbca2392bac0281"},
			{intRef(), "tzdata-java", "2019a", "1.el7", "noarch", "tzdata-2019a-1.el7.src.rpm", 374277, "Public Domain", "CentOS", "", "Timezone data for Java", "5679fbe22d2eee8156d3f0671a11bd59"},
			{intRef(1), "java-1.8.0-openjdk-headless", "1.8.0.212.b04", "0.el7_6", "x86_64", "java-1.8.0-openjdk-1.8.0.212.b04-0.el7_6.src.rpm", 108711819, "ASL 1.1 and ASL 2.0 and BSD and BSD with advertising and GPL+ and GPLv2 and GPLv2 with exceptions and IJG and LGPLv2+ and MIT and MPLv2.0 and Public Domain and W3C and zlib", "CentOS", "", "OpenJDK Headless Runtime Environment 8", "db70b8dce71856397e8b5f5bbb1ad779"},
			{intRef(1), "java-1.8.0-openjdk", "1.8.0.212.b04", "0.el7_6", "x86_64", "java-1.8.0-openjdk-1.8.0.212.b04-0.el7_6.src.rpm", 541844, "ASL 1.1 and ASL 2.0 and BSD and BSD with advertising and GPL+ and GPLv2 and GPLv2 with exceptions and IJG and LGPLv2+ and MIT and MPLv2.0 and Public Domain and W3C and zlib", "CentOS", "", "OpenJDK Runtime Environment 8", "93f38377ee9fd5b0450f6be108fd0ad4"},
			{intRef(), "binutils-devel", "2.27", "34.base.el7", "x86_64", "binutils-2.27-34.base.el7.src.rpm", 4540848, "GPLv3+", "CentOS", "", "BFD and opcodes static and dynamic libraries and header files", "3aecd6ac92eabeb8aa5261707f33f562"},
			{intRef(), "gpg-pubkey", "f4a80eb5", "53a7ff4b", "", "", 0, "pubkey", "", "", "gpg(CentOS-7 Key (CentOS 7 Official Signing Key) <security@centos.org>)", ""},
			{intRef(), "pciutils", "3.5.1", "3.el7", "x86_64", "pciutils-3.5.1-3.el7.src.rpm", 200397, "GPLv2+", "CentOS", "", "PCI bus related utilities", "4dc44058001b78bab74eb1a1d227faaa"},
			{intRef(), "libstdc++", "4.8.5", "36.el7_6.1", "x86_64", "gcc-4.8.5-36.el7_6.1.src.rpm", 1077442, "GPLv3+ and GPLv3+ with exceptions and GPLv2+ with exceptions and LGPLv2+ and BSD", "CentOS", "", "GNU Standard C++ Library", "d2b021bf5f895c786ce110a0ab2267d7"},
			{intRef(), "numactl-libs", "2.0.9", "7.el7", "x86_64", "numactl-2.0.9-7.el7.src.rpm", 50752, "LGPLv2 and GPLv2", "CentOS", "", "libnuma libraries", "1d7a96cb51e28a7a66454f92d54e57fe"},
			{intRef(), "file-libs", "5.11", "35.el7", "x86_64", "file-5.11-35.el7.src.rpm", 3076831, "BSD", "CentOS", "", "Libraries for applications using libmagic", "3037fd1f781186bcb0fb919b2aa78f54"},
			{intRef(), "pciutils-devel", "3.5.1", "3.el7", "x86_64", "pciutils-3.5.1-3.el7.src.rpm", 78345, "GPLv2+", "CentOS", "", "Linux PCI development library", "7e6f5aac02e0286f01e3cfba2ccbfd9f"},
			{intRef(), "rpm-build-libs", "4.11.3", "35.el7", "x86_64", "rpm-4.11.3-35.el7.src.rpm", 166776, "GPLv2+ and LGPLv2+ with exceptions", "CentOS", "", "Libraries for building and signing RPM packages", "aba616fc3d1a25cfc77eb27db8610457"},
			{intRef(), "apr", "1.4.8", "3.el7_4.1", "x86_64", "apr-1.4.8-3.el7_4.1.src.rpm", 226686, "ASL 2.0 and BSD with advertising and ISC and BSD", "CentOS", "", "Apache Portable Runtime library", "0534c5c4a8da91bf8e4a4152c0383b46"},
			{intRef(), "boost-system", "1.53.0", "27.el7", "x86_64", "boost-1.53.0-27.el7.src.rpm", 32538, "Boost and MIT and Python", "CentOS", "", "Run-Time component of boost system support library", "6df5a05420fbd79bd36e644acc509dfe"},
			{intRef(), "avahi-libs", "0.6.31", "19.el7", "x86_64", "avahi-0.6.31-19.el7.src.rpm", 123808, "LGPLv2+", "CentOS", "", "Libraries for avahi run-time use", "78426c94f63ec36f4a9d27bfd000a7d3"},
			{intRef(), "json-c", "0.11", "4.el7_0", "x86_64", "json-c-0.11-4.el7_0.src.rpm", 65593, "MIT", "CentOS", "", "A JSON implementation in C", "62bcb642b82d3b3aeb99ba9ba1b40715"},
			{intRef(1), "perl-parent", "0.225", "244.el7", "noarch", "perl-parent-0.225-244.el7.src.rpm", 8141, "GPL+ or Artistic", "CentOS", "", "Establish an ISA relationship with base classes at compile time", "d902f1a3426caa19ccbccf09263c54f6"},
			{intRef(1), "perl-Pod-Escapes", "1.04", "294.el7_6", "noarch", "perl-5.16.3-294.el7_6.src.rpm", 21091, "GPL+ or Artistic", "CentOS", "", "Perl module for resolving POD escape sequences", "ad4d7216b06eb48942c30d4bc2b09755"},
			{intRef(4), "perl-libs", "5.16.3", "294.el7_6", "x86_64", "perl-5.16.3-294.el7_6.src.rpm", 1647344, "GPL+ or Artistic", "CentOS", "", "The libraries for the perl runtime", "669be9d414caaf85799b0dfa644c4e81"},
			{intRef(), "perl-threads", "1.87", "4.el7", "x86_64", "perl-threads-1.87-4.el7.src.rpm", 98615, "GPL+ or Artistic", "CentOS", "", "Perl interpreter-based threads", "29739d2f766c8bd85e85de9f8ccb7367"},
			{intRef(), "perl-constant", "1.27", "2.el7", "noarch", "perl-constant-1.27-2.el7.src.rpm", 26364, "GPL+ or Artistic", "CentOS", "", "Perl pragma to declare constants", "f3ecfa1ce48e13496322e341a3f0b72c"},
			{intRef(), "perl-File-Temp", "0.23.01", "3.el7", "noarch", "perl-File-Temp-0.23.01-3.el7.src.rpm", 158781, "GPL+ or Artistic", "CentOS", "", "Return name and handle of a temporary file safely", "396b0a1af3a203e1cb33c98814d5a6d8"},
			{intRef(1), "perl-Pod-Simple", "3.28", "4.el7", "noarch", "perl-Pod-Simple-3.28-4.el7.src.rpm", 538320, "GPL+ or Artistic", "CentOS", "", "Framework for parsing POD documentation", "8e5edeb936a154537062b776eb41cb7d"},
			{intRef(1), "perl-Error", "0.17020", "2.el7", "noarch", "perl-Error-0.17020-2.el7.src.rpm", 50015, "(GPL+ or Artistic) and MIT", "CentOS", "", "Error/exception handling in an OO-ish way", "d5f79c530c268d71811519df24ca0c90"},
			{intRef(), "perl-XML-Parser", "2.41", "10.el7", "x86_64", "perl-XML-Parser-2.41-10.el7.src.rpm", 642973, "GPL+ or Artistic", "CentOS", "", "Perl module for parsing XML documents", "1998f3b0016cc426351682109fc7c9a8"},
			{intRef(), "less", "458", "9.el7", "x86_64", "less-458-9.el7.src.rpm", 215376, "GPLv3+", "CentOS", "", "A text file browser similar to more, but better", "2b1222dfb9fbcf5ca3ee2f07d897bb4c"},
			{intRef(), "cpp", "4.8.5", "36.el7_6.1", "x86_64", "gcc-4.8.5-36.el7_6.1.src.rpm", 15644861, "GPLv3+ and GPLv3+ with exceptions and GPLv2+ with exceptions and LGPLv2+ and BSD", "CentOS", "", "The C Preprocessor", "2044dee2673e3aabfb7df7e4e152227e"},
			{intRef(), "trousers", "0.3.14", "2.el7", "x86_64", "trousers-0.3.14-2.el7.src.rpm", 836873, "BSD", "CentOS", "", "TCG's Software Stack v1.2", "7a4bdd0c029c5750be61101d418d4adc"},
			{intRef(), "libdwarf", "20130207", "4.el7", "x86_64", "libdwarf-20130207-4.el7.src.rpm", 291679, "LGPLv2", "CentOS", "", "Library to access the DWARF Debugging file format", "2c5a9b992e22a196fcdfe3657972956d"},
			{intRef(), "systemtap-runtime", "3.3", "3.el7", "x86_64", "systemtap-3.3-3.el7.src.rpm", 1142954, "GPLv2+", "CentOS", "", "Programmable system-wide instrumentation system - runtime", "e133207717622b13eb3018ef60e11638"},
		}

		return toPackageInfo(pkgs)
	}

	// docker run --rm -it centos:7 bash
	// rpm -qa --queryformat "\{%{EPOCH}, \"%{NAME}\", \"%{VERSION}\", \"%{RELEASE}\", \"%{ARCH}\", \"%{SOURCERPM}\", %{SIZE}, \"%{LICENSE}\", \"%{VENDOR}\", \"\", \"%{SUMMARY}\", \"%{SIGMD5}\"\},\n" | sed "s/^{(none)/{intRef()/g" | sed -r 's/^\{([0-9]+),/{intRef(\1),/' | sed "s/(none)/0/g"
	CentOS7Plain = func() []*PackageInfo {
		pkgs := []*commonPackageInfo{
			{intRef(), "tzdata", "2018e", "3.el7", "noarch", "tzdata-2018e-3.el7.src.rpm", 1966505, "Public Domain", "CentOS", "", "Timezone data", "a1e5c65f5b87e33b23419b153155db0a"},
			{intRef(), "nss-softokn-freebl", "3.36.0", "5.el7_5", "x86_64", "nss-softokn-3.36.0-5.el7_5.src.rpm", 565628, "MPLv2.0", "CentOS", "", "Freebl library for the Network Security Services", "cf0360bbf0396334bb569646bbec25d3"},
			{intRef(), "ncurses", "5.9", "14.20130511.el7_4", "x86_64", "ncurses-5.9-14.20130511.el7_4.src.rpm", 439378, "MIT", "CentOS", "", "Ncurses support utilities", "8ca93e2831102818759a22e22e871268"},
			{intRef(), "glibc-common", "2.17", "222.el7", "x86_64", "glibc-2.17-222.el7.src.rpm", 120325207, "LGPLv2+ and LGPLv2+ with exceptions and GPLv2+", "CentOS", "", "Common binaries and locale data for glibc", "862cbbb5262d965c2a57fb8cc3b02b99"},
			{intRef(), "filesystem", "3.2", "25.el7", "x86_64", "filesystem-3.2-25.el7.src.rpm", 0, "Public Domain", "CentOS", "", "The basic directory layout for a Linux system", "93e1c91a2dbd4d3ac61fb70a40a7c228"},
			{intRef(), "glibc", "2.17", "222.el7", "x86_64", "glibc-2.17-222.el7.src.rpm", 14223248, "LGPLv2+ and LGPLv2+ with exceptions and GPLv2+", "CentOS", "", "The GNU libc libraries", "b20747760e86c79af25fa522f4844460"},
			{intRef(), "nspr", "4.19.0", "1.el7_5", "x86_64", "nspr-4.19.0-1.el7_5.src.rpm", 287728, "MPLv2.0", "CentOS", "", "Netscape Portable Runtime", "06e8af105f3a5832065a90c5eb12c064"},
			{intRef(), "popt", "1.13", "16.el7", "x86_64", "popt-1.13-16.el7.src.rpm", 88516, "MIT", "CentOS", "", "C library for parsing command line parameters", "87f5d6bac8f205fb14e2072e4b47ce23"},
			{intRef(), "libffi", "3.0.13", "18.el7", "x86_64", "libffi-3.0.13-18.el7.src.rpm", 47766, "MIT and Public Domain", "CentOS", "", "A portable foreign function interface library", "f05f7a3749342b925fa10ca7de2f0116"},
			{intRef(), "libcap", "2.22", "9.el7", "x86_64", "libcap-2.22-9.el7.src.rpm", 111445, "LGPLv2+", "CentOS", "", "Library for getting and setting POSIX.1e capabilities", "b4cc4b26a62dbbd66143755c54a59938"},
			{intRef(), "libsepol", "2.5", "8.1.el7", "x86_64", "libsepol-2.5-8.1.el7.src.rpm", 686568, "LGPLv2+", "CentOS", "", "SELinux binary policy manipulation library", "b107891426af43d06b02a191f16a490c"},
			{intRef(), "ncurses-libs", "5.9", "14.20130511.el7_4", "x86_64", "ncurses-5.9-14.20130511.el7_4.src.rpm", 1028216, "MIT", "CentOS", "", "Ncurses libraries", "bba677eda5184b5127b7817d2c1dec12"},
			{intRef(), "gawk", "4.0.2", "4.el7_3.1", "x86_64", "gawk-4.0.2-4.el7_3.1.src.rpm", 2435978, "GPLv3+ and GPL and LGPLv3+ and LGPL and BSD", "CentOS", "", "The GNU version of the awk text processing utility", "5efa8feaeb12d57cda1a9bcd9c8af917"},
			{intRef(), "libselinux", "2.5", "12.el7", "x86_64", "libselinux-2.5-12.el7.src.rpm", 217874, "Public Domain", "CentOS", "", "SELinux library and simple utilities", "7584c834f1404914f4568973c001f28e"},
			{intRef(), "grep", "2.20", "3.el7", "x86_64", "grep-2.20-3.el7.src.rpm", 1195131, "GPLv3+", "CentOS", "", "Pattern matching utilities", "92bc957d1c4fc3764e134d8f82544abe"},
			{intRef(), "keyutils-libs", "1.5.8", "3.el7", "x86_64", "keyutils-1.5.8-3.el7.src.rpm", 42138, "GPLv2+ and LGPLv2+", "CentOS", "", "Key utilities library", "e31fa37a714787d3d72e0a9316c3a85b"},
			{intRef(), "libverto", "0.2.5", "4.el7", "x86_64", "libverto-0.2.5-4.el7.src.rpm", 23060, "MIT", "CentOS", "", "Main loop abstraction library", "341bae036f22ce087486f4d9e180caee"},
			{intRef(), "p11-kit-trust", "0.23.5", "3.el7", "x86_64", "p11-kit-0.23.5-3.el7.src.rpm", 437261, "BSD", "CentOS", "", "System trust module from p11-kit", "2dc2f7ab212401098e8821ce23f21741"},
			{intRef(1), "openssl-libs", "1.0.2k", "12.el7", "x86_64", "openssl-1.0.2k-12.el7.src.rpm", 3200172, "OpenSSL", "CentOS", "", "A general purpose cryptography library with TLS implementation", "9213b8a36ef5936ac90e2630803e659c"},
			{intRef(), "centos-release", "7", "5.1804.4.el7.centos", "x86_64", "centos-release-7-5.1804.4.el7.centos.src.rpm", 40338, "GPLv2", "CentOS", "", "CentOS Linux release file", "e73301588f9dce0441fb403b2fbc09fd"},
			{intRef(), "xz-libs", "5.2.2", "1.el7", "x86_64", "xz-5.2.2-1.el7.src.rpm", 239967, "LGPLv2+", "CentOS", "", "Libraries for decoding LZMA compression", "7701c542dc8fcd5a38ee52e8786d7cf4"},
			{intRef(), "libdb", "5.3.21", "24.el7", "x86_64", "libdb-5.3.21-24.el7.src.rpm", 1858008, "BSD and LGPLv2 and Sleepycat", "CentOS", "", "The Berkeley DB database library for C", "24fdfda30ad8e1cc7ebc744eaf85eacf"},
			{intRef(), "libgpg-error", "1.12", "3.el7", "x86_64", "libgpg-error-1.12-3.el7.src.rpm", 350865, "LGPLv2+", "CentOS", "", "Library for error values used by GnuPG components", "a3933d4ce43c07fa9305e8765d78d099"},
			{intRef(), "libgcrypt", "1.5.3", "14.el7", "x86_64", "libgcrypt-1.5.3-14.el7.src.rpm", 597727, "LGPLv2+", "CentOS", "", "A general-purpose cryptography library", "88c64dc7c3aed6a156e4dee5def38300"},
			{intRef(), "lua", "5.1.4", "15.el7", "x86_64", "lua-5.1.4-15.el7.src.rpm", 640319, "MIT", "CentOS", "", "Powerful light-weight programming language", "d5973eb7ca2bb665f5fbb7141f2e27a3"},
			{intRef(), "libuuid", "2.23.2", "52.el7_5.1", "x86_64", "util-linux-2.23.2-52.el7_5.1.src.rpm", 20326, "BSD", "CentOS", "", "Universally unique ID library", "534c564308a92d9c695b64b7a359f36c"},
			{intRef(), "libmount", "2.23.2", "52.el7_5.1", "x86_64", "util-linux-2.23.2-52.el7_5.1.src.rpm", 273965, "LGPLv2+", "CentOS", "", "Device mounting library", "d084f24c36b7968ed5fc4a34940980ab"},
			{intRef(), "shared-mime-info", "1.8", "4.el7", "x86_64", "shared-mime-info-1.8-4.el7.src.rpm", 2379317, "GPLv2+", "CentOS", "", "Shared MIME information database", "167b39408a8c5705f135e968174e2229"},
			{intRef(), "gzip", "1.5", "10.el7", "x86_64", "gzip-1.5-10.el7.src.rpm", 250440, "GPLv3+ and GFDL", "CentOS", "", "The GNU data compression program", "adcb4df7dfe7866e38b5bf42d54d63ae"},
			{intRef(1), "findutils", "4.5.11", "5.el7", "x86_64", "findutils-4.5.11-5.el7.src.rpm", 1855626, "GPLv3+", "CentOS", "", "The GNU versions of find utilities (find and xargs)", "2d571b503447f5ce47a4767f6a7e1986"},
			{intRef(), "diffutils", "3.3", "4.el7", "x86_64", "diffutils-3.3-4.el7.src.rpm", 1065157, "GPLv3+", "CentOS", "", "A GNU collection of diff utilities", "d8c60275b4c6d54e8bf7ee19cf0c4f2b"},
			{intRef(), "expat", "2.1.0", "10.el7_3", "x86_64", "expat-2.1.0-10.el7_3.src.rpm", 208315, "MIT", "CentOS", "", "An XML parser library", "9c5fdceee3715ad8a1b3a9dd711d5b79"},
			{intRef(), "audit-libs", "2.8.1", "3.el7_5.1", "x86_64", "audit-2.8.1-3.el7_5.1.src.rpm", 256370, "LGPLv2+", "CentOS", "", "Dynamic library for libaudit", "437ad738bda99bc9fdc9185f45554467"},
			{intRef(), "pam", "1.1.8", "22.el7", "x86_64", "pam-1.1.8-22.el7.src.rpm", 2630324, "BSD and GPLv2+", "CentOS", "", "An extensible library which provides authentication for applications", "dce355546c72f1425dc04fd64c9bc664"},
			{intRef(), "nss-softokn", "3.36.0", "5.el7_5", "x86_64", "nss-softokn-3.36.0-5.el7_5.src.rpm", 1130670, "MPLv2.0", "CentOS", "", "Network Security Services Softoken Module", "7b8391d5b6843722982a616a8542d58f"},
			{intRef(), "nss", "3.36.0", "7.el7_5", "x86_64", "nss-3.36.0-7.el7_5.src.rpm", 2424993, "MPLv2.0", "CentOS", "", "Network Security Services", "aead60d750112396fc5381c7c6152863"},
			{intRef(), "libassuan", "2.1.0", "3.el7", "x86_64", "libassuan-2.1.0-3.el7.src.rpm", 155391, "LGPLv2+ and GPLv3+", "CentOS", "", "GnuPG IPC library", "c4c94aadc9da58f576b4ef238d0136cb"},
			{intRef(), "file-libs", "5.11", "33.el7", "x86_64", "file-5.11-33.el7.src.rpm", 3077690, "BSD", "CentOS", "", "Libraries for applications using libmagic", "14a5e5d0610ce2de13f59b3996fa21ba"},
			{intRef(1), "pkgconfig", "0.27.1", "4.el7", "x86_64", "pkgconfig-0.27.1-4.el7.src.rpm", 105522, "GPLv2+", "CentOS", "", "A tool for determining compilation options", "ef58117c9c617599952c0642ff5e8d00"},
			{intRef(), "cyrus-sasl-lib", "2.1.26", "23.el7", "x86_64", "cyrus-sasl-2.1.26-23.el7.src.rpm", 396911, "BSD with advertising", "CentOS", "", "Shared libraries needed by applications which use Cyrus SASL", "6d06f1337b143c25bc7358255d433006"},
			{intRef(), "binutils", "2.27", "28.base.el7_5.1", "x86_64", "binutils-2.27-28.base.el7_5.1.src.rpm", 25149789, "GPLv3+", "CentOS", "", "A GNU collection of binary utilities", "ab389c762bf3df5411b21583f3479dc1"},
			{intRef(), "libcurl", "7.29.0", "46.el7", "x86_64", "curl-7.29.0-46.el7.src.rpm", 435192, "MIT", "CentOS", "", "A library for getting files from web servers", "2d99038a0bef38275774024117436284"},
			{intRef(), "rpm-libs", "4.11.3", "32.el7", "x86_64", "rpm-4.11.3-32.el7.src.rpm", 611384, "GPLv2+ and LGPLv2+ with exceptions", "CentOS", "", "Libraries for manipulating RPM packages", "04a21eb64e7b2495dfe0ef704165506e"},
			{intRef(), "openldap", "2.4.44", "15.el7_5", "x86_64", "openldap-2.4.44-15.el7_5.src.rpm", 1037299, "OpenLDAP", "CentOS", "", "LDAP support libraries", "0c7d41097a0100c3e40ac4aff0d3cdd8"},
			{intRef(), "pinentry", "0.8.1", "17.el7", "x86_64", "pinentry-0.8.1-17.el7.src.rpm", 159929, "GPLv2+", "CentOS", "", "Collection of simple PIN or passphrase entry dialogs", "740fabe008451f11fee22ff086fb13d3"},
			{intRef(), "libsemanage", "2.5", "11.el7", "x86_64", "libsemanage-2.5-11.el7.src.rpm", 302369, "LGPLv2+", "CentOS", "", "SELinux binary policy manipulation library", "d2fd300c5abc3561b16a685b1f2cbc5d"},
			{intRef(), "libutempter", "1.1.6", "4.el7", "x86_64", "libutempter-1.1.6-4.el7.src.rpm", 49749, "LGPLv2+", "CentOS", "", "A privileged helper for utmp/wtmp updates", "8e28747e6a84b84af7d78c84cd8cd5e8"},
			{intRef(), "qrencode-libs", "3.4.1", "3.el7", "x86_64", "qrencode-3.4.1-3.el7.src.rpm", 126732, "LGPLv2+", "CentOS", "", "QR Code encoding library - Shared libraries", "ac1de0c46bc1886e12ea80771951cde2"},
			{intRef(7), "device-mapper", "1.02.146", "4.el7", "x86_64", "lvm2-2.02.177-4.el7.src.rpm", 338922, "GPLv2", "CentOS", "", "Device mapper utility", "1919f9661f5cf5f63fcb4c5112718db7"},
			{intRef(), "procps-ng", "3.3.10", "17.el7_5.2", "x86_64", "procps-ng-3.3.10-17.el7_5.2.src.rpm", 760500, "GPL+ and GPLv2 and GPLv2+ and GPLv3+ and LGPLv2+", "CentOS", "", "System and process monitoring utilities", "57c9fa477ce98fca4e7dcc5ed6b41a7a"},
			{intRef(), "cryptsetup-libs", "1.7.4", "4.el7", "x86_64", "cryptsetup-1.7.4-4.el7.src.rpm", 969908, "GPLv2+ and LGPLv2+", "CentOS", "", "Cryptsetup shared library", "979897acf07d1ab7db8a0380a8c8ecf5"},
			{intRef(), "kmod", "20", "21.el7", "x86_64", "kmod-20-21.el7.src.rpm", 243127, "GPLv2+", "CentOS", "", "Linux kernel module management utilities", "47a0704853382255bfbc519fa9dd1a3d"},
			{intRef(), "systemd-libs", "219", "57.el7_5.3", "x86_64", "systemd-219-57.el7_5.3.src.rpm", 1263552, "LGPLv2+ and MIT", "CentOS", "", "systemd libraries", "2ce20481cfc413a926533b156f1f329f"},
			{intRef(), "systemd", "219", "57.el7_5.3", "x86_64", "systemd-219-57.el7_5.3.src.rpm", 24402038, "LGPLv2+ and MIT and GPLv2+", "CentOS", "", "A System and Service Manager", "ba2398624fcb2872d952e548aeddcab1"},
			{intRef(1), "dbus", "1.10.24", "7.el7", "x86_64", "dbus-1.10.24-7.el7.src.rpm", 595223, "(GPLv2+ or AFL) and GPLv2+", "CentOS", "", "D-BUS message bus", "ee5c672ee2b669192e40bdf2460a5427"},
			{intRef(), "iputils", "20160308", "10.el7", "x86_64", "iputils-20160308-10.el7.src.rpm", 343497, "BSD and GPLv2+", "CentOS", "", "Network monitoring tools including ping", "dfa219e6db737a1eb1b5cb23b532e858"},
			{intRef(), "gdbm", "1.10", "8.el7", "x86_64", "gdbm-1.10-8.el7.src.rpm", 184322, "GPLv3+", "CentOS", "", "A GNU set of database routines which use extensible hashing", "770bd49f3898dc1cc831fb9ec12d569a"},
			{intRef(), "python", "2.7.5", "69.el7_5", "x86_64", "python-2.7.5-69.el7_5.src.rpm", 80907, "Python", "CentOS", "", "An interpreted, interactive, object-oriented programming language", "6ed4c7b6c1fbc6994dd6e8c6cc8de459"},
			{intRef(), "dbus-python", "1.1.1", "9.el7", "x86_64", "dbus-python-1.1.1-9.el7.src.rpm", 848122, "MIT", "CentOS", "", "D-Bus Python Bindings", "6be4da91886c59ae25e5255ebff0a0a9"},
			{intRef(), "pyliblzma", "0.5.3", "11.el7", "x86_64", "pyliblzma-0.5.3-11.el7.src.rpm", 190112, "LGPLv3+", "CentOS", "", "Python bindings for lzma", "a09aa8e806a3f6285e9fe765d69d3ee9"},
			{intRef(), "python-urlgrabber", "3.10", "8.el7", "noarch", "python-urlgrabber-3.10-8.el7.src.rpm", 500670, "LGPLv2+", "CentOS", "", "A high-level cross-protocol url-grabber", "8a0c6c349d305a6c89d04e4d661e85a9"},
			{intRef(), "pyxattr", "0.5.1", "5.el7", "x86_64", "pyxattr-0.5.1-5.el7.src.rpm", 63304, "LGPLv2+", "CentOS", "", "Extended attributes library wrapper for Python", "e85121da681760c48a2af6ed8aa31b4e"},
			{intRef(), "python-kitchen", "1.1.1", "5.el7", "noarch", "python-kitchen-1.1.1-5.el7.src.rpm", 1465161, "LGPLv2+", "CentOS", "", "Small, useful pieces of code to make python coding easier", "917912ebf6164135f1f9871150432977"},
			{intRef(), "gnupg2", "2.0.22", "5.el7_5", "x86_64", "gnupg2-2.0.22-5.el7_5.src.rpm", 6637796, "GPLv3+", "CentOS", "", "Utility for secure communication and data storage", "a293dfefd6596bfe5c54ba68377e334e"},
			{intRef(), "rpm-python", "4.11.3", "32.el7", "x86_64", "rpm-4.11.3-32.el7.src.rpm", 149714, "GPLv2+", "CentOS", "", "Python bindings for apps which will manipulate RPM packages", "a39b0ef5a60188e60fde5785a09226f4"},
			{intRef(), "pygpgme", "0.3", "9.el7", "x86_64", "pygpgme-0.3-9.el7.src.rpm", 197501, "LGPLv2+", "CentOS", "", "Python module for working with OpenPGP messages", "67b8c720f59a063adbbca2392bac0281"},
			{intRef(), "yum", "3.4.3", "158.el7.centos", "noarch", "yum-3.4.3-158.el7.centos.src.rpm", 5814102, "GPLv2+", "CentOS", "", "RPM package installer/updater/manager", "55ef45392a487da6305b69e89cef7cb7"},
			{intRef(), "yum-utils", "1.1.31", "46.el7_5", "noarch", "yum-utils-1.1.31-46.el7_5.src.rpm", 343422, "GPLv2+", "CentOS", "", "Utilities based around the yum package manager", "9a3315c3ee603728ce1908ec404a2c42"},
			{intRef(2), "vim-minimal", "7.4.160", "4.el7", "x86_64", "vim-7.4.160-4.el7.src.rpm", 917640, "Vim", "CentOS", "", "A minimal version of the VIM editor", "495d7fc35ace98f416bef65eecf8af0d"},
			{intRef(), "libgcc", "4.8.5", "28.el7_5.1", "x86_64", "gcc-4.8.5-28.el7_5.1.src.rpm", 179328, "GPLv3+ and GPLv3+ with exceptions and GPLv2+ with exceptions and LGPLv2+ and BSD", "CentOS", "", "GCC version 4.8 shared support library", "518679e12b8d890c8c90b82d96d45a7a"},
			{intRef(), "ncurses-base", "5.9", "14.20130511.el7_4", "noarch", "ncurses-5.9-14.20130511.el7_4.src.rpm", 223432, "MIT", "CentOS", "", "Descriptions of common terminals", "c955fdfe1f8f20b1053a34918e44bd58"},
			{intRef(), "bash", "4.2.46", "30.el7", "x86_64", "bash-4.2.46-30.el7.src.rpm", 3667709, "GPLv3+", "CentOS", "", "The GNU Bourne Again shell", "4c9037d4d3139a2c8fd28ed6b27d47da"},
			{intRef(), "chkconfig", "1.7.4", "1.el7", "x86_64", "chkconfig-1.7.4-1.el7.src.rpm", 779531, "GPLv2", "CentOS", "", "A system tool for maintaining the /etc/rc*.d hierarchy", "e126177e9ac5841deca96f4e15340613"},
			{intRef(), "setup", "2.8.71", "9.el7", "noarch", "setup-2.8.71-9.el7.src.rpm", 696925, "Public Domain", "CentOS", "", "A set of system configuration and setup files", "3e3fc345cd2ac49c17b8e38df270fd16"},
			{intRef(), "basesystem", "10.0", "7.el7.centos", "noarch", "basesystem-10.0-7.el7.centos.src.rpm", 0, "Public Domain", "CentOS", "", "The skeleton package which defines a simple CentOS Linux system", "778eb3d19a934f42d1354ed795d06b4f"},
			{intRef(), "zlib", "1.2.7", "17.el7", "x86_64", "zlib-1.2.7-17.el7.src.rpm", 185710, "zlib and Boost", "CentOS", "", "The compression and decompression library", "505b6038db14d2b7d58c5721126cb11c"},
			{intRef(), "nss-util", "3.36.0", "1.el7_5", "x86_64", "nss-util-3.36.0-1.el7_5.src.rpm", 194944, "MPLv2.0", "CentOS", "", "Network Security Services Utilities Library", "ecf6d738049e13862a8cf2a151b78bb5"},
			{intRef(), "libcom_err", "1.42.9", "12.el7_5", "x86_64", "e2fsprogs-1.42.9-12.el7_5.src.rpm", 60489, "MIT", "CentOS", "", "Common error description library", "4b27bfeb726d091b4317b5e66d6dba7f"},
			{intRef(), "libattr", "2.4.46", "13.el7", "x86_64", "attr-2.4.46-13.el7.src.rpm", 19896, "LGPLv2+", "CentOS", "", "Dynamic library for extended attribute support", "209b05c7b77a33a4cc176077945833e8"},
			{intRef(), "libacl", "2.2.51", "14.el7", "x86_64", "acl-2.2.51-14.el7.src.rpm", 37056, "LGPLv2+", "CentOS", "", "Dynamic library for access control list support", "c3243d728de0ad95e0c05230e6463bdd"},
			{intRef(), "libstdc++", "4.8.5", "28.el7_5.1", "x86_64", "gcc-4.8.5-28.el7_5.1.src.rpm", 1077442, "GPLv3+ and GPLv3+ with exceptions and GPLv2+ with exceptions and LGPLv2+ and BSD", "CentOS", "", "GNU Standard C++ Library", "59b268ee5d8a252656d7d30a7bed03bc"},
			{intRef(), "info", "5.1", "5.el7", "x86_64", "texinfo-5.1-5.el7.src.rpm", 494630, "GPLv3+", "CentOS", "", "A stand-alone TTY-based reader for GNU texinfo documentation", "e1f75e40cf956ed3efe7b49c5f6939ec"},
			{intRef(), "pcre", "8.32", "17.el7", "x86_64", "pcre-8.32-17.el7.src.rpm", 1475532, "BSD", "CentOS", "", "Perl-compatible regular expression library", "04a1926cc29eef0f79f8183eb7907024"},
			{intRef(), "sed", "4.2.2", "5.el7", "x86_64", "sed-4.2.2-5.el7.src.rpm", 601208, "GPLv3+", "CentOS", "", "A GNU stream text editor", "4d696dbd3061493258490f40c625adba"},
			{intRef(), "p11-kit", "0.23.5", "3.el7", "x86_64", "p11-kit-0.23.5-3.el7.src.rpm", 1337825, "BSD", "CentOS", "", "Library for loading and sharing PKCS#11 modules", "9a0f49ca3b6d58aeb05780003cae38f1"},
			{intRef(1), "gmp", "6.0.0", "15.el7", "x86_64", "gmp-6.0.0-15.el7.src.rpm", 657046, "LGPLv3+ or GPLv2+", "CentOS", "", "A GNU arbitrary precision library", "4cd60ebf94766a46c590e17036ece777"},
			{intRef(), "libtasn1", "4.10", "1.el7", "x86_64", "libtasn1-4.10-1.el7.src.rpm", 424486, "GPLv3+ and LGPLv2+", "CentOS", "", "The ASN.1 library used in GNUTLS", "5bd1b0b4094fac08713f986572a8625e"},
			{intRef(), "ca-certificates", "2018.2.22", "70.0.el7_5", "noarch", "ca-certificates-2018.2.22-70.0.el7_5.src.rpm", 973960, "Public Domain", "CentOS", "", "The Mozilla CA root certificate bundle", "a95373fc5273e7c757a240316d4430b9"},
			{intRef(), "coreutils", "8.22", "21.el7", "x86_64", "coreutils-8.22-21.el7.src.rpm", 14588989, "GPLv3+", "CentOS", "", "A set of basic GNU tools commonly used in shell scripts", "fb8117e7aeb1ab935460fa412277a088"},
			{intRef(), "krb5-libs", "1.15.1", "19.el7", "x86_64", "krb5-1.15.1-19.el7.src.rpm", 1984782, "MIT", "CentOS", "", "The non-admin shared libraries used by Kerberos 5", "9811e62803caaf9375851ed6f16fedd7"},
			{intRef(), "bzip2-libs", "1.0.6", "13.el7", "x86_64", "bzip2-1.0.6-13.el7.src.rpm", 70093, "BSD", "CentOS", "", "Libraries for applications using bzip2", "0486f0b02e0caa4195761266798e7a26"},
			{intRef(), "elfutils-libelf", "0.170", "4.el7", "x86_64", "elfutils-0.170-4.el7.src.rpm", 936931, "GPLv2+ or LGPLv3+", "CentOS", "", "Library to read and write ELF files", "6a261ac45f947b29f2bbfe7e538aac20"},
			{intRef(), "libxml2", "2.9.1", "6.el7_2.3", "x86_64", "libxml2-2.9.1-6.el7_2.3.src.rpm", 1710062, "MIT", "CentOS", "", "Library providing XML and HTML support", "e439b8934c4e9576eccfbc60d3cd714f"},
			{intRef(), "readline", "6.2", "10.el7", "x86_64", "readline-6.2-10.el7.src.rpm", 460464, "GPLv3+", "CentOS", "", "A library for editing typed command lines", "b552128087365a58b3566631c678752f"},
			{intRef(), "cpio", "2.11", "27.el7", "x86_64", "cpio-2.11-27.el7.src.rpm", 689335, "GPLv3+", "CentOS", "", "A GNU archiving program", "32498692f82296d51663586f08cebafc"},
			{intRef(), "libblkid", "2.23.2", "52.el7_5.1", "x86_64", "util-linux-2.23.2-52.el7_5.1.src.rpm", 261837, "LGPLv2+", "CentOS", "", "Block device ID library", "427f260c9fc06f72d722e5ccdfcf30c5"},
			{intRef(), "glib2", "2.54.2", "2.el7", "x86_64", "glib2-2.54.2-2.el7.src.rpm", 11986873, "LGPLv2+", "CentOS", "", "A library of handy utility functions", "75e476b2fbb8f06f881338c12e379cc3"},
			{intRef(), "sqlite", "3.7.17", "8.el7", "x86_64", "sqlite-3.7.17-8.el7.src.rpm", 814231, "Public Domain", "CentOS", "", "Library that implements an embeddable SQL database engine", "0ceb9434c5a9d281399583087cfe889b"},
			{intRef(), "cracklib", "2.9.0", "11.el7", "x86_64", "cracklib-2.9.0-11.el7.src.rpm", 209610, "LGPLv2+", "CentOS", "", "A password-checking library", "a973d883d6785904e49266428bb3ed90"},
			{intRef(), "libidn", "1.28", "4.el7", "x86_64", "libidn-1.28-4.el7.src.rpm", 630407, "LGPLv2+ and GPLv3+ and GFDL", "CentOS", "", "Internationalized Domain Name support library", "3d582106ac1032c59c1cd4236e5d6415"},
			{intRef(), "libcap-ng", "0.7.5", "4.el7", "x86_64", "libcap-ng-0.7.5-4.el7.src.rpm", 50510, "LGPLv2+", "CentOS", "", "An alternate posix capabilities library", "84efa5ca78408f78e606e7c8d1816759"},
			{intRef(), "cracklib-dicts", "2.9.0", "11.el7", "x86_64", "cracklib-2.9.0-11.el7.src.rpm", 9389116, "LGPLv2+", "CentOS", "", "The standard CrackLib dictionaries", "da657b0facf181893f9a38437092dbbf"},
			{intRef(), "libpwquality", "1.2.3", "5.el7", "x86_64", "libpwquality-1.2.3-5.el7.src.rpm", 332421, "BSD or GPLv2+", "CentOS", "", "A library for password generation and password quality checking", "e40a69f0a1803dfa3736aba2c8ea1fb4"},
			{intRef(), "nss-sysinit", "3.36.0", "7.el7_5", "x86_64", "nss-3.36.0-7.el7_5.src.rpm", 14061, "MPLv2.0", "CentOS", "", "System NSS Initialization", "ace8226abffd1e928af6f4c62cdd450d"},
			{intRef(), "nss-pem", "1.0.3", "4.el7", "x86_64", "nss-pem-1.0.3-4.el7.src.rpm", 201219, "MPLv1.1", "CentOS", "", "PEM file reader for Network Security Services (NSS)", "95cf0973322bf389a612fdbd1e3a6627"},
			{intRef(), "xz", "5.2.2", "1.el7", "x86_64", "xz-5.2.2-1.el7.src.rpm", 798130, "LGPLv2+", "CentOS", "", "LZMA compression utilities", "3e6f05008f7af7931aa8257964527f1a"},
			{intRef(), "lz4", "1.7.5", "2.el7", "x86_64", "lz4-1.7.5-2.el7.src.rpm", 366872, "GPLv2+ and BSD", "CentOS", "", "Extremely fast compression algorithm", "89e45994eabac0ba4ccae7c1005ea06c"},
			{intRef(), "nss-tools", "3.36.0", "7.el7_5", "x86_64", "nss-3.36.0-7.el7_5.src.rpm", 2069571, "MPLv2.0", "CentOS", "", "Tools for the Network Security Services", "dfada2eb06ef6cccbdc69c711c59ae02"},
			{intRef(), "gobject-introspection", "1.50.0", "1.el7", "x86_64", "gobject-introspection-1.50.0-1.el7.src.rpm", 834149, "GPLv2+, LGPLv2+, MIT", "CentOS", "", "Introspection system for GObject-based libraries", "6048b8e810c7b808f4eddca36d12d5b8"},
			{intRef(), "libdb-utils", "5.3.21", "24.el7", "x86_64", "libdb-5.3.21-24.el7.src.rpm", 326487, "BSD and LGPLv2 and Sleepycat", "CentOS", "", "Command line tools for managing Berkeley DB databases", "d06077335f27858ecb06ad48fa2e812d"},
			{intRef(), "kmod-libs", "20", "21.el7", "x86_64", "kmod-20-21.el7.src.rpm", 91800, "LGPLv2+", "CentOS", "", "Libraries to handle kernel module loading and unloading", "d7dea2fe3482829a84830109bf4bcc97"},
			{intRef(), "libssh2", "1.4.3", "10.el7_2.1", "x86_64", "libssh2-1.4.3-10.el7_2.1.src.rpm", 341782, "BSD", "CentOS", "", "A library implementing the SSH2 protocol", "8ee7c5f75e2a2c85536f0f2282347545"},
			{intRef(), "curl", "7.29.0", "46.el7", "x86_64", "curl-7.29.0-46.el7.src.rpm", 540259, "MIT", "CentOS", "", "A utility for getting files from remote servers (FTP, HTTP, and others)", "401357a38601356be337ead6bf4470e0"},
			{intRef(), "rpm", "4.11.3", "32.el7", "x86_64", "rpm-4.11.3-32.el7.src.rpm", 2621900, "GPLv2+", "CentOS", "", "The RPM package management system", "de491f66be27f9af821bd34c3873dca4"},
			{intRef(), "libuser", "0.60", "9.el7", "x86_64", "libuser-0.60-9.el7.src.rpm", 1952592, "LGPLv2+", "CentOS", "", "A user and group account administration library", "8fdd92e668b91208dfd916ebdce8a297"},
			{intRef(2), "tar", "1.26", "34.el7", "x86_64", "tar-1.26-34.el7.src.rpm", 2838271, "GPLv3+", "CentOS", "", "A GNU file archiving program", "419ef9b4469e60d4d4d2599c1c8d79ba"},
			{intRef(), "acl", "2.2.51", "14.el7", "x86_64", "acl-2.2.51-14.el7.src.rpm", 201225, "GPLv2+", "CentOS", "", "Access control list utilities", "885771078c3fe1efa43e3e5e0f2b5246"},
			{intRef(), "ustr", "1.0.4", "16.el7", "x86_64", "ustr-1.0.4-16.el7.src.rpm", 285943, "MIT or LGPLv2+ or BSD", "CentOS", "", "String library, very low memory overhead, simple to import", "58bb2ebb64b048e24a4c7d5d51170f55"},
			{intRef(2), "shadow-utils", "4.1.5.1", "24.el7", "x86_64", "shadow-utils-4.1.5.1-24.el7.src.rpm", 3534490, "BSD and GPLv2+", "CentOS", "", "Utilities for managing accounts and shadow password files", "19741e43ee0b142d310393848d75239f"},
			{intRef(1), "hardlink", "1.0", "19.el7", "x86_64", "hardlink-1.0-19.el7.src.rpm", 16545, "GPL+", "CentOS", "", "Create a tree of hardlinks", "3a1bcd8e12bbdbced75d74b5919a039a"},
			{intRef(), "util-linux", "2.23.2", "52.el7_5.1", "x86_64", "util-linux-2.23.2-52.el7_5.1.src.rpm", 8642769, "GPLv2 and GPLv2+ and LGPLv2+ and BSD with advertising and Public Domain", "CentOS", "", "A collection of basic system utilities", "8c402cdaecc9f30a146aa82ff30c8aea"},
			{intRef(), "kpartx", "0.4.9", "119.el7_5.1", "x86_64", "device-mapper-multipath-0.4.9-119.el7_5.1.src.rpm", 41363, "GPL+", "CentOS", "", "Partition device manager for device-mapper devices", "fd61a26347cf0d090798768d7e952759"},
			{intRef(7), "device-mapper-libs", "1.02.146", "4.el7", "x86_64", "lvm2-2.02.177-4.el7.src.rpm", 400551, "LGPLv2", "CentOS", "", "Device-mapper shared library", "f0b566ce4fe22594b7f5f246870d4214"},
			{intRef(), "dracut", "033", "535.el7_5.1", "x86_64", "dracut-033-535.el7_5.1.src.rpm", 898198, "GPLv2+ and LGPLv2+", "CentOS", "", "Initramfs generator using udev", "22ffeb1421964a58b78b87eeb47014a7"},
			{intRef(), "elfutils-libs", "0.170", "4.el7", "x86_64", "elfutils-0.170-4.el7.src.rpm", 747527, "GPLv2+ or LGPLv3+", "CentOS", "", "Libraries to handle compiled objects", "385aa8222aad0bf1e9499e79284f77d7"},
			{intRef(1), "dbus-libs", "1.10.24", "7.el7", "x86_64", "dbus-1.10.24-7.el7.src.rpm", 362584, "(GPLv2+ or AFL) and GPLv2+", "CentOS", "", "Libraries for accessing D-BUS", "7c544dc4e3020620da275ddf1c0de95c"},
			{intRef(), "elfutils-default-yama-scope", "0.170", "4.el7", "noarch", "elfutils-0.170-4.el7.src.rpm", 1810, "GPLv2+ or LGPLv3+", "CentOS", "", "Default yama attach scope sysctl setting", "186adf73da2641da5799e54b4ccfed5d"},
			{intRef(), "dbus-glib", "0.100", "7.el7", "x86_64", "dbus-glib-0.100-7.el7.src.rpm", 301237, "AFL and GPLv2+", "CentOS", "", "GLib bindings for D-Bus", "fea831a1bb3627640677db4d441a65c6"},
			{intRef(), "python-libs", "2.7.5", "69.el7_5", "x86_64", "python-2.7.5-69.el7_5.src.rpm", 24713084, "Python", "CentOS", "", "Runtime libraries for Python", "879bca5c512f1199f3694bced2464d5c"},
			{intRef(), "libxml2-python", "2.9.1", "6.el7_2.3", "x86_64", "libxml2-2.9.1-6.el7_2.3.src.rpm", 1503050, "MIT", "CentOS", "", "Python bindings for the libxml2 library", "4af3055a710e118b3e7c49cb1df4e784"},
			{intRef(), "python-gobject-base", "3.22.0", "1.el7_4.1", "x86_64", "pygobject3-3.22.0-1.el7_4.1.src.rpm", 1123114, "LGPLv2+ and MIT", "CentOS", "", "Python 2 bindings for GObject Introspection base package", "e7f47d020d89e7ecc10aa699c0f8a4f0"},
			{intRef(), "yum-metadata-parser", "1.1.4", "10.el7", "x86_64", "yum-metadata-parser-1.1.4-10.el7.src.rpm", 58789, "GPLv2", "CentOS", "", "A fast metadata parser for yum", "929f0abd8c5671c69f86fce8d7cdd42d"},
			{intRef(), "python-pycurl", "7.19.0", "19.el7", "x86_64", "python-pycurl-7.19.0-19.el7.src.rpm", 241513, "LGPLv2+ or MIT", "CentOS", "", "A Python interface to libcurl", "d5152315d59bde1faaa203e68de6a64e"},
			{intRef(), "python-iniparse", "0.4", "9.el7", "noarch", "python-iniparse-0.4-9.el7.src.rpm", 115166, "MIT", "CentOS", "", "Python Module for Accessing and Modifying Configuration Data in INI files", "3140d9efca99c4dd5215383bab36bca5"},
			{intRef(), "python-chardet", "2.2.1", "1.el7_1", "noarch", "python-chardet-2.2.1-1.el7_1.src.rpm", 1156541, "LGPLv2", "CentOS", "", "Character encoding auto-detection in Python", "c70da920f85b12ecd25429913e3e861e"},
			{intRef(), "hostname", "3.13", "3.el7", "x86_64", "hostname-3.13-3.el7.src.rpm", 19449, "GPLv2+", "CentOS", "", "Utility to set/show the host name or domain name", "677ae7d92bb29445d1e0b66ecf5cab81"},
			{intRef(), "pth", "2.0.7", "23.el7", "x86_64", "pth-2.0.7-23.el7.src.rpm", 267851, "LGPLv2+", "CentOS", "", "The GNU Portable Threads library", "2d6939b73767d0f96b02677bcbeef2e2"},
			{intRef(), "rpm-build-libs", "4.11.3", "32.el7", "x86_64", "rpm-4.11.3-32.el7.src.rpm", 166664, "GPLv2+ and LGPLv2+ with exceptions", "CentOS", "", "Libraries for building and signing RPM packages", "5733e3c03981b025cab7ea343257e528"},
			{intRef(), "gpgme", "1.3.2", "5.el7", "x86_64", "gpgme-1.3.2-5.el7.src.rpm", 547534, "LGPLv2+", "CentOS", "", "GnuPG Made Easy - high level crypto API", "d693313ae4184395e902c28ce87d4573"},
			{intRef(), "yum-plugin-fastestmirror", "1.1.31", "46.el7_5", "noarch", "yum-utils-1.1.31-46.el7_5.src.rpm", 53895, "GPLv2+", "CentOS", "", "Yum plugin which chooses fastest repository from a mirrorlist", "d9f1b3ff9204795855f0f6cf09ab40dd"},
			{intRef(32), "bind-license", "9.9.4", "61.el7_5.1", "noarch", "bind-9.9.4-61.el7_5.1.src.rpm", 26831, "ISC", "CentOS", "", "License of the BIND DNS suite", "8983808555b122a7220800907c368f23"},
			{intRef(), "yum-plugin-ovl", "1.1.31", "46.el7_5", "noarch", "yum-utils-1.1.31-46.el7_5.src.rpm", 22399, "GPLv2+", "CentOS", "", "Yum plugin to work around overlayfs issues", "5ee5adb1e57e68d35865f4c94181f473"},
			{intRef(), "passwd", "0.79", "4.el7", "x86_64", "passwd-0.79-4.el7.src.rpm", 429874, "BSD or GPL+", "CentOS", "", "An utility for setting or changing passwords using PAM", "bc5080b013d6a43fb00e57bba7517e0f"},
			{intRef(), "rootfiles", "8.1", "11.el7", "noarch", "rootfiles-8.1-11.el7.src.rpm", 599, "Public Domain", "CentOS", "", "The basic required files for the root user's directory", "b9b9eceee7ad38f868520e80f3404642"},
		}

		return toPackageInfo(pkgs)
	}

	// docker run --rm -it centos/python-35-centos7 bash
	// rpm -qa --queryformat "\{%{EPOCH}, \"%{NAME}\", \"%{VERSION}\", \"%{RELEASE}\", \"%{ARCH}\", \"%{SOURCERPM}\", %{SIZE}, \"%{LICENSE}\", \"%{VENDOR}\", \"\", \"%{SUMMARY}\", \"%{SIGMD5}\"\},\n" | sed "s/^{(none)/{intRef()/g" | sed -r 's/^\{([0-9]+),/{intRef(\1),/' | sed "s/(none)/0/g"
	CentOS7Python35 = func() []*PackageInfo {
		pkgs := []*commonPackageInfo{
			{intRef(), "fontconfig-devel", "2.13.0", "4.3.el7", "x86_64", "fontconfig-2.13.0-4.3.el7.src.rpm", 111050, "MIT and Public Domain and UCD", "CentOS", "", "Font configuration and customization library", "2adc764bcb07849e3c9b3313bea0a5d9"},
			{intRef(32), "bind-license", "9.9.4", "73.el7_6", "noarch", "bind-9.9.4-73.el7_6.src.rpm", 26831, "ISC", "CentOS", "", "License of the BIND DNS suite", "98dd71aa6b527468317e3e96e1629ddb"},
			{intRef(), "less", "458", "9.el7", "x86_64", "less-458-9.el7.src.rpm", 215376, "GPLv3+", "CentOS", "", "A text file browser similar to more, but better", "2b1222dfb9fbcf5ca3ee2f07d897bb4c"},
			{intRef(), "libxml2-devel", "2.9.1", "6.el7_2.3", "x86_64", "libxml2-2.9.1-6.el7_2.3.src.rpm", 9213393, "MIT", "CentOS", "", "Libraries, includes, etc. to develop XML and HTML applications", "511b950a717dfb131d815aec2127a0e0"},
			{intRef(), "bash", "4.2.46", "31.el7", "x86_64", "bash-4.2.46-31.el7.src.rpm", 3667773, "GPLv3+", "CentOS", "", "The GNU Bourne Again shell", "ed4d986a9753c79474e6ab726f40d864"},
			{intRef(), "util-linux", "2.23.2", "59.el7_6.1", "x86_64", "util-linux-2.23.2-59.el7_6.1.src.rpm", 8460215, "GPLv2 and GPLv2+ and LGPLv2+ and BSD with advertising and Public Domain", "CentOS", "", "A collection of basic system utilities", "556ac8615ad253912ea1608b63d984d9"},
			{intRef(), "glibc-common", "2.17", "260.el7_6.3", "x86_64", "glibc-2.17-260.el7_6.3.src.rpm", 120499441, "LGPLv2+ and LGPLv2+ with exceptions and GPLv2+", "CentOS", "", "Common binaries and locale data for glibc", "b6d5b5844c699f7d3474795ced771933"},
			{intRef(), "libxslt", "1.1.28", "5.el7", "x86_64", "libxslt-1.1.28-5.el7.src.rpm", 497582, "MIT", "CentOS", "", "Library providing the Gnome XSLT engine", "e901ff455779767496a18d1a5f43816a"},
			{intRef(), "nss-softokn-freebl", "3.36.0", "5.el7_5", "x86_64", "nss-softokn-3.36.0-5.el7_5.src.rpm", 565628, "MPLv2.0", "CentOS", "", "Freebl library for the Network Security Services", "cf0360bbf0396334bb569646bbec25d3"},
			{intRef(), "glibc-headers", "2.17", "260.el7_6.3", "x86_64", "glibc-2.17-260.el7_6.3.src.rpm", 2338328, "LGPLv2+ and LGPLv2+ with exceptions and GPLv2+", "CentOS", "", "Header files for development using standard C libraries.", "7a029bcdf31a7f69269ee2783bc1e278"},
			{intRef(), "filesystem", "3.2", "25.el7", "x86_64", "filesystem-3.2-25.el7.src.rpm", 0, "Public Domain", "CentOS", "", "The basic directory layout for a Linux system", "93e1c91a2dbd4d3ac61fb70a40a7c228"},
			{intRef(), "gcc", "4.8.5", "36.el7_6.1", "x86_64", "gcc-4.8.5-36.el7_6.1.src.rpm", 39238921, "GPLv3+ and GPLv3+ with exceptions and GPLv2+ with exceptions and LGPLv2+ and BSD", "CentOS", "", "Various compilers (C, C++, Objective-C, Java, ...)", "82a79193e96339bd9ee093bdb0c53c1a"},
			{intRef(), "glibc", "2.17", "260.el7_6.3", "x86_64", "glibc-2.17-260.el7_6.3.src.rpm", 14346456, "LGPLv2+ and LGPLv2+ with exceptions and GPLv2+", "CentOS", "", "The GNU libc libraries", "0face5fe8edcb15b63b59d0445a74787"},
			{intRef(), "autoconf", "2.69", "11.el7", "noarch", "autoconf-2.69-11.el7.src.rpm", 2320371, "GPLv2+ and GFDL", "CentOS", "", "A GNU tool for automatically configuring source code", "24e721d7349c86c3df3cf4727e598a54"},
			{intRef(), "nspr", "4.19.0", "1.el7_5", "x86_64", "nspr-4.19.0-1.el7_5.src.rpm", 287728, "MPLv2.0", "CentOS", "", "Netscape Portable Runtime", "06e8af105f3a5832065a90c5eb12c064"},
			{intRef(), "libX11-common", "1.6.5", "2.el7", "noarch", "libX11-1.6.5-2.el7.src.rpm", 1335662, "MIT", "CentOS", "", "Common data for libX11", "8008bbb9137fcb4eee8f46018f6fc5ff"},
			{intRef(), "popt", "1.13", "16.el7", "x86_64", "popt-1.13-16.el7.src.rpm", 88516, "MIT", "CentOS", "", "C library for parsing command line parameters", "87f5d6bac8f205fb14e2072e4b47ce23"},
			{intRef(), "libXpm", "3.5.12", "1.el7", "x86_64", "libXpm-3.5.12-1.el7.src.rpm", 116599, "MIT", "CentOS", "", "X.Org X11 libXpm runtime library", "513d610e407b9d461e3b358f86b58841"},
			{intRef(), "libcom_err", "1.42.9", "13.el7", "x86_64", "e2fsprogs-1.42.9-13.el7.src.rpm", 60489, "MIT", "CentOS", "", "Common error description library", "d50ec3dda8af48a865da2d9754fa21bd"},
			{intRef(), "gd", "2.0.35", "26.el7", "x86_64", "gd-2.0.35-26.el7.src.rpm", 554528, "MIT", "CentOS", "", "A graphics library for quick creation of PNG or JPEG images", "9d62ce2e46a14f71589c82e752d428e5"},
			{intRef(), "libcap", "2.22", "9.el7", "x86_64", "libcap-2.22-9.el7.src.rpm", 111445, "LGPLv2+", "CentOS", "", "Library for getting and setting POSIX.1e capabilities", "b4cc4b26a62dbbd66143755c54a59938"},
			{intRef(), "libXt", "1.1.5", "3.el7", "x86_64", "libXt-1.1.5-3.el7.src.rpm", 430105, "MIT", "CentOS", "", "X.Org X11 libXt runtime library", "5f357545763a40cda11cfd5e002cae50"},
			{intRef(), "libkadm5", "1.15.1", "37.el7_6", "x86_64", "krb5-1.15.1-37.el7_6.src.rpm", 221096, "MIT", "CentOS", "", "Kerberos 5 Administrative libraries", "aed2ac504c09afd6fcb8fad11088de5d"},
			{intRef(), "info", "5.1", "5.el7", "x86_64", "texinfo-5.1-5.el7.src.rpm", 494630, "GPLv3+", "CentOS", "", "A stand-alone TTY-based reader for GNU texinfo documentation", "e1f75e40cf956ed3efe7b49c5f6939ec"},
			{intRef(), "keyutils-libs-devel", "1.5.8", "3.el7", "x86_64", "keyutils-1.5.8-3.el7.src.rpm", 29186, "GPLv2+ and LGPLv2+", "CentOS", "", "Development package for building Linux key management utilities", "89e25181b24bb62f927ee72c767b9f31"},
			{intRef(), "gawk", "4.0.2", "4.el7_3.1", "x86_64", "gawk-4.0.2-4.el7_3.1.src.rpm", 2435978, "GPLv3+ and GPL and LGPLv3+ and LGPL and BSD", "CentOS", "", "The GNU version of the awk text processing utility", "5efa8feaeb12d57cda1a9bcd9c8af917"},
			{intRef(), "libgcrypt-devel", "1.5.3", "14.el7", "x86_64", "libgcrypt-1.5.3-14.el7.src.rpm", 220369, "LGPLv2+ and GPLv2+", "CentOS", "", "Development files for the libgcrypt package", "51c351529dc1eb4c14f45aa9b8f310c6"},
			{intRef(), "libselinux", "2.5", "14.1.el7", "x86_64", "libselinux-2.5-14.1.el7.src.rpm", 217874, "Public Domain", "CentOS", "", "SELinux library and simple utilities", "36462311e16b3aafc9f29bbaa00bfb6e"},
			{intRef(), "openssh-clients", "7.4p1", "16.el7", "x86_64", "openssh-7.4p1-16.el7.src.rpm", 2651616, "BSD", "CentOS", "", "An open source SSH client applications", "4f9a34658e76fe533402c89ee43fec18"},
			{intRef(), "grep", "2.20", "3.el7", "x86_64", "grep-2.20-3.el7.src.rpm", 1195131, "GPLv3+", "CentOS", "", "Pattern matching utilities", "92bc957d1c4fc3764e134d8f82544abe"},
			{intRef(), "git", "1.8.3.1", "20.el7", "x86_64", "git-1.8.3.1-20.el7.src.rpm", 23232282, "GPLv2", "CentOS", "", "Fast Version Control System", "bad9f26b37b7b8a74a69a038b773def4"},
			{intRef(), "keyutils-libs", "1.5.8", "3.el7", "x86_64", "keyutils-1.5.8-3.el7.src.rpm", 42138, "GPLv2+ and LGPLv2+", "CentOS", "", "Key utilities library", "e31fa37a714787d3d72e0a9316c3a85b"},
			{intRef(), "libselinux-devel", "2.5", "14.1.el7", "x86_64", "libselinux-2.5-14.1.el7.src.rpm", 194101, "Public Domain", "CentOS", "", "Header files and libraries used to build SELinux", "93a41b1ad6b78570561586803d3f643e"},
			{intRef(), "libverto", "0.2.5", "4.el7", "x86_64", "libverto-0.2.5-4.el7.src.rpm", 23060, "MIT", "CentOS", "", "Main loop abstraction library", "341bae036f22ce087486f4d9e180caee"},
			{intRef(), "krb5-devel", "1.15.1", "37.el7_6", "x86_64", "krb5-1.15.1-37.el7_6.src.rpm", 796853, "MIT", "CentOS", "", "Development files needed to compile Kerberos 5 programs", "9f15acf86c251fa9fd5b4072721cb400"},
			{intRef(), "p11-kit-trust", "0.23.5", "3.el7", "x86_64", "p11-kit-0.23.5-3.el7.src.rpm", 437261, "BSD", "CentOS", "", "System trust module from p11-kit", "2dc2f7ab212401098e8821ce23f21741"},
			{intRef(1), "mariadb-devel", "5.5.60", "1.el7_5", "x86_64", "mariadb-5.5.60-1.el7_5.src.rpm", 3478237, "GPLv2 with exceptions and LGPLv2 and BSD", "CentOS", "", "Files for development of MariaDB/MySQL applications", "16d50cc8947cfc0db22f34c601783af3"},
			{intRef(), "gd-devel", "2.0.35", "26.el7", "x86_64", "gd-2.0.35-26.el7.src.rpm", 300168, "MIT", "CentOS", "", "The development libraries and header files for gd", "44fca0e2303a521754c3824346dc9578"},
			{intRef(), "krb5-libs", "1.15.1", "37.el7_6", "x86_64", "krb5-1.15.1-37.el7_6.src.rpm", 2200985, "MIT", "CentOS", "", "The non-admin shared libraries used by Kerberos 5", "240e6ada762f6cdc46d8d6a3bdbadd49"},
			{intRef(), "gcc-c++", "4.8.5", "36.el7_6.1", "x86_64", "gcc-4.8.5-36.el7_6.1.src.rpm", 17257523, "GPLv3+ and GPLv3+ with exceptions and GPLv2+ with exceptions and LGPLv2+ and BSD", "CentOS", "", "C++ support for GCC", "8f929d2776e8a8b345b4f6d4066dfaa0"},
			{intRef(), "xz-libs", "5.2.2", "1.el7", "x86_64", "xz-5.2.2-1.el7.src.rpm", 239967, "LGPLv2+", "CentOS", "", "Libraries for decoding LZMA compression", "7701c542dc8fcd5a38ee52e8786d7cf4"},
			{intRef(), "gdb", "7.6.1", "114.el7", "x86_64", "gdb-7.6.1-114.el7.src.rpm", 7371507, "GPLv3+ and GPLv3+ with exceptions and GPLv2+ and GPLv2+ with exceptions and GPL+ and LGPLv2+ and BSD and Public Domain", "CentOS", "", "A GNU source-level debugger for C, C++, Fortran, Go and other languages", "1d085bb4c7273a51db58d47b35ab95b4"},
			{intRef(), "libdb", "5.3.21", "24.el7", "x86_64", "libdb-5.3.21-24.el7.src.rpm", 1858008, "BSD and LGPLv2 and Sleepycat", "CentOS", "", "The Berkeley DB database library for C", "24fdfda30ad8e1cc7ebc744eaf85eacf"},
			{intRef(), "lsof", "4.87", "6.el7", "x86_64", "lsof-4.87-6.el7.src.rpm", 949035, "zlib and Sendmail and LGPLv2+", "CentOS", "", "A utility which lists open files on a Linux/UNIX system", "b922bef40badac88a7a924c9436a1da2"},
			{intRef(), "libgpg-error", "1.12", "3.el7", "x86_64", "libgpg-error-1.12-3.el7.src.rpm", 350865, "LGPLv2+", "CentOS", "", "Library for error values used by GnuPG components", "a3933d4ce43c07fa9305e8765d78d099"},
			{intRef(), "patch", "2.7.1", "10.el7_5", "x86_64", "patch-2.7.1-10.el7_5.src.rpm", 215449, "GPLv3+", "CentOS", "", "Utility for modifying/upgrading files", "3c1cd445c2a45f19b9bb46c19a4a943c"},
			{intRef(), "libgcrypt", "1.5.3", "14.el7", "x86_64", "libgcrypt-1.5.3-14.el7.src.rpm", 597727, "LGPLv2+", "CentOS", "", "A general-purpose cryptography library", "88c64dc7c3aed6a156e4dee5def38300"},
			{intRef(), "bzip2", "1.0.6", "13.el7", "x86_64", "bzip2-1.0.6-13.el7.src.rpm", 83791, "BSD", "CentOS", "", "A file compression utility", "704f597825711270b0ba2af51bebecae"},
			{intRef(), "lua", "5.1.4", "15.el7", "x86_64", "lua-5.1.4-15.el7.src.rpm", 640319, "MIT", "CentOS", "", "Powerful light-weight programming language", "d5973eb7ca2bb665f5fbb7141f2e27a3"},
			{intRef(), "which", "2.20", "7.el7", "x86_64", "which-2.20-7.el7.src.rpm", 76962, "GPLv3", "CentOS", "", "Displays where a particular program in your path is located", "5b0eff1739daa46a80e9908f39167c29"},
			{intRef(), "apr", "1.4.8", "3.el7_4.1", "x86_64", "apr-1.4.8-3.el7_4.1.src.rpm", 226686, "ASL 2.0 and BSD with advertising and ISC and BSD", "CentOS", "", "Apache Portable Runtime library", "0534c5c4a8da91bf8e4a4152c0383b46"},
			{intRef(), "libquadmath", "4.8.5", "36.el7_6.1", "x86_64", "gcc-4.8.5-36.el7_6.1.src.rpm", 282811, "GPLv3+ and GPLv3+ with exceptions and GPLv2+ with exceptions and LGPLv2+ and BSD", "CentOS", "", "GCC __float128 shared support library", "80155e9c1179374d45231786bedae2a4"},
			{intRef(), "shared-mime-info", "1.8", "4.el7", "x86_64", "shared-mime-info-1.8-4.el7.src.rpm", 2379317, "GPLv2+", "CentOS", "", "Shared MIME information database", "167b39408a8c5705f135e968174e2229"},
			{intRef(), "apr-devel", "1.4.8", "3.el7_4.1", "x86_64", "apr-1.4.8-3.el7_4.1.src.rpm", 789433, "ASL 2.0 and BSD with advertising and ISC and BSD", "CentOS", "", "APR library development kit", "b14b1f61d17c5630c5e94a5f3bb537a2"},
			{intRef(), "gzip", "1.5", "10.el7", "x86_64", "gzip-1.5-10.el7.src.rpm", 250440, "GPLv3+ and GFDL", "CentOS", "", "The GNU data compression program", "adcb4df7dfe7866e38b5bf42d54d63ae"},
			{intRef(), "rh-python35-python-libs", "3.5.1", "11.el7", "x86_64", "rh-python35-python-3.5.1-11.el7.src.rpm", 40247447, "Python", "CentOS", "", "Python 3 runtime libraries", "af3f9f34eba1fbb750e4c7b02aead906"},
			{intRef(1), "findutils", "4.5.11", "6.el7", "x86_64", "findutils-4.5.11-6.el7.src.rpm", 1855882, "GPLv3+", "CentOS", "", "The GNU versions of find utilities (find and xargs)", "162b03172dee0f90443197d424bc9be0"},
			{intRef(), "rh-python35-python-pip", "7.1.0", "2.el7", "noarch", "rh-python35-python-pip-7.1.0-2.el7.src.rpm", 8198956, "MIT", "CentOS", "", "A tool for installing and managing Python packages", "a16a319283e963a6e6ccea12b12729cc"},
			{intRef(), "diffutils", "3.3", "4.el7", "x86_64", "diffutils-3.3-4.el7.src.rpm", 1065157, "GPLv3+", "CentOS", "", "A GNU collection of diff utilities", "d8c60275b4c6d54e8bf7ee19cf0c4f2b"},
			{intRef(), "libmnl", "1.0.3", "7.el7", "x86_64", "libmnl-1.0.3-7.el7.src.rpm", 51847, "LGPLv2+", "CentOS", "", "A minimalistic Netlink library", "a73c0dd6533ee63b1c100c01c3cf2fc6"},
			{intRef(), "expat", "2.1.0", "10.el7_3", "x86_64", "expat-2.1.0-10.el7_3.src.rpm", 208315, "MIT", "CentOS", "", "An XML parser library", "9c5fdceee3715ad8a1b3a9dd711d5b79"},
			{intRef(), "iptables", "1.4.21", "28.el7", "x86_64", "iptables-1.4.21-28.el7.src.rpm", 1555961, "GPLv2", "CentOS", "", "Tools for managing Linux kernel packet filtering capabilities", "0fd17d33af23274a15a51cf40469d38b"},
			{intRef(), "audit-libs", "2.8.4", "4.el7", "x86_64", "audit-2.8.4-4.el7.src.rpm", 256362, "LGPLv2+", "CentOS", "", "Dynamic library for libaudit", "a57039d27b473fc766688886d53ecce2"},
			{intRef(), "atlas", "3.10.1", "12.el7", "x86_64", "atlas-3.10.1-12.el7.src.rpm", 21813025, "BSD", "CentOS", "", "Automatically Tuned Linear Algebra Software", "8686de5cd4f5c0bfd52097bbd1d60751"},
			{intRef(), "pam", "1.1.8", "22.el7", "x86_64", "pam-1.1.8-22.el7.src.rpm", 2630324, "BSD and GPLv2+", "CentOS", "", "An extensible library which provides authentication for applications", "dce355546c72f1425dc04fd64c9bc664"},
			{intRef(), "apr-util-ldap", "1.5.2", "6.el7", "x86_64", "apr-util-1.5.2-6.el7.src.rpm", 11408, "ASL 2.0", "CentOS", "", "APR utility library LDAP support", "2c30586907d30bdfa248992cb1340698"},
			{intRef(), "nss-softokn", "3.36.0", "5.el7_5", "x86_64", "nss-softokn-3.36.0-5.el7_5.src.rpm", 1130670, "MPLv2.0", "CentOS", "", "Network Security Services Softoken Module", "7b8391d5b6843722982a616a8542d58f"},
			{intRef(), "hunspell", "1.3.2", "15.el7", "x86_64", "hunspell-1.3.2-15.el7.src.rpm", 583000, "LGPLv2+ or GPLv2+ or MPLv1.1", "CentOS", "", "A spell checker and morphological analyzer library", "5fe6f58315041f1d1f75c0b50bac1039"},
			{intRef(), "nss", "3.36.0", "7.1.el7_6", "x86_64", "nss-3.36.0-7.1.el7_6.src.rpm", 2425024, "MPLv2.0", "CentOS", "", "Network Security Services", "208ddd9381e8c7b512ce763bae2e7628"},
			{intRef(), "dwz", "0.11", "3.el7", "x86_64", "dwz-0.11-3.el7.src.rpm", 225103, "GPLv2+ and GPLv3+", "CentOS", "", "DWARF optimization and duplicate removal tool", "28a047bc76d0ad80e6938a94898129ca"},
			{intRef(), "libassuan", "2.1.0", "3.el7", "x86_64", "libassuan-2.1.0-3.el7.src.rpm", 155391, "LGPLv2+ and GPLv3+", "CentOS", "", "GnuPG IPC library", "c4c94aadc9da58f576b4ef238d0136cb"},
			{intRef(), "policycoreutils", "2.5", "29.el7_6.1", "x86_64", "policycoreutils-2.5-29.el7_6.1.src.rpm", 5321515, "GPLv2", "CentOS", "", "SELinux policy core utilities", "5b157e1d9bb389d4df8a002a785ede4b"},
			{intRef(), "libcgroup", "0.41", "20.el7", "x86_64", "libcgroup-0.41-20.el7.src.rpm", 137210, "LGPLv2+", "CentOS", "", "Library to control and monitor control groups", "eb76a54af92009d1c72e8efd9d0ac434"},
			{intRef(), "iso-codes", "3.46", "2.el7", "noarch", "iso-codes-3.46-2.el7.src.rpm", 14504447, "LGPLv2+", "CentOS", "", "ISO code lists and translations", "fdc8f7128246bcb6b43ff39516165f85"},
			{intRef(), "libsemanage-python", "2.5", "14.el7", "x86_64", "libsemanage-2.5-14.el7.src.rpm", 451817, "LGPLv2+", "CentOS", "", "semanage python bindings for libsemanage", "44967d0d996064547841a4572ef0f9fb"},
			{intRef(), "nss-tools", "3.36.0", "7.1.el7_6", "x86_64", "nss-3.36.0-7.1.el7_6.src.rpm", 2069587, "MPLv2.0", "CentOS", "", "Tools for the Network Security Services", "53bab8a320b55204b17fc880c326ae26"},
			{intRef(), "jansson", "2.10", "1.el7", "x86_64", "jansson-2.10-1.el7.src.rpm", 74998, "MIT", "CentOS", "", "C library for encoding, decoding and manipulating JSON data", "47add619dede2cd5e4079de7250ed141"},
			{intRef(), "perl-srpm-macros", "1", "8.el7", "noarch", "perl-srpm-macros-1-8.el7.src.rpm", 794, "GPLv3+", "CentOS", "", "RPM macros for building Perl source package from source repository", "6bd0791a143b53addc95cbeb050859c8"},
			{intRef(), "gobject-introspection", "1.56.1", "1.el7", "x86_64", "gobject-introspection-1.56.1-1.el7.src.rpm", 854207, "GPLv2+, LGPLv2+, MIT", "CentOS", "", "Introspection system for GObject-based libraries", "35d1071e217a3adca0dc2679a049ca04"},
			{intRef(), "scl-utils-build", "20130529", "19.el7", "x86_64", "scl-utils-20130529-19.el7.src.rpm", 18361, "GPLv2+", "CentOS", "", "RPM build macros for alternative packaging", "8efb7c3ee86a3c7462b6609cc3f3eefa"},
			{intRef(), "libdb-utils", "5.3.21", "24.el7", "x86_64", "libdb-5.3.21-24.el7.src.rpm", 326487, "BSD and LGPLv2 and Sleepycat", "CentOS", "", "Command line tools for managing Berkeley DB databases", "d06077335f27858ecb06ad48fa2e812d"},
			{intRef(), "rh-python35-python-virtualenv", "13.1.2", "2.el7", "noarch", "rh-python35-python-virtualenv-13.1.2-2.el7.src.rpm", 2764777, "MIT", "CentOS", "", "Tool to create isolated Python environments", "e916e6a3f5f1e85dda8290f33094c903"},
			{intRef(), "kmod-libs", "20", "23.el7", "x86_64", "kmod-20-23.el7.src.rpm", 91848, "LGPLv2+", "CentOS", "", "Libraries to handle kernel module loading and unloading", "8e1fe13756e455026ef91936cf555599"},
			{intRef(), "initscripts", "9.49.46", "1.el7", "x86_64", "initscripts-9.49.46-1.el7.src.rpm", 1519938, "GPLv2 and GPLv2+", "CentOS", "", "The inittab file and the /etc/init.d scripts", "86366cfcc12b419e3dae819319489fb4"},
			{intRef(), "libssh2", "1.4.3", "12.el7", "x86_64", "libssh2-1.4.3-12.el7.src.rpm", 341886, "BSD", "CentOS", "", "A library implementing the SSH2 protocol", "8629b98c219f01a14e71d0d8bd57965d"},
			{intRef(), "cyrus-sasl-devel", "2.1.26", "23.el7", "x86_64", "cyrus-sasl-2.1.26-23.el7.src.rpm", 1024214, "BSD with advertising", "CentOS", "", "Files needed for developing applications with Cyrus SASL", "29cc91e7f9c9bfd27e5d1c795acf26e2"},
			{intRef(), "curl", "7.29.0", "51.el7", "x86_64", "curl-7.29.0-51.el7.src.rpm", 540293, "MIT", "CentOS", "", "A utility for getting files from remote servers (FTP, HTTP, and others)", "2008575cc1fdf94eedba4e920908cb38"},
			{intRef(), "libdb-devel", "5.3.21", "24.el7", "x86_64", "libdb-5.3.21-24.el7.src.rpm", 129076, "BSD and LGPLv2 and Sleepycat", "CentOS", "", "C development files for the Berkeley DB library", "32730435809421188228bfcc6b926ad0"},
			{intRef(), "rpm", "4.11.3", "35.el7", "x86_64", "rpm-4.11.3-35.el7.src.rpm", 2622266, "GPLv2+", "CentOS", "", "The RPM package management system", "ba17adc338f442da8a4598244965e521"},
			{intRef(), "mailcap", "2.1.41", "2.el7", "noarch", "mailcap-2.1.41-2.el7.src.rpm", 63360, "Public Domain and MIT", "CentOS", "", "Helper application and MIME type associations for file types", "d20a25f8d834c1cac92727b8aa8e0f72"},
			{intRef(), "libuser", "0.60", "9.el7", "x86_64", "libuser-0.60-9.el7.src.rpm", 1952592, "LGPLv2+", "CentOS", "", "A user and group account administration library", "8fdd92e668b91208dfd916ebdce8a297"},
			{intRef(), "audit-libs-python", "2.8.4", "4.el7", "x86_64", "audit-2.8.4-4.el7.src.rpm", 327265, "LGPLv2+", "CentOS", "", "Python bindings for libaudit", "665eb826f1113adf9f656760f589bd55"},
			{intRef(2), "tar", "1.26", "35.el7", "x86_64", "tar-1.26-35.el7.src.rpm", 2838510, "GPLv3+", "CentOS", "", "A GNU file archiving program", "28f3087e5b512d8cc05e3a8a65727686"},
			{intRef(), "httpd24-runtime", "1.1", "18.el7", "x86_64", "httpd24-1.1-18.el7.src.rpm", 20989, "GPLv2+", "CentOS", "", "Package that handles httpd24 Software Collection.", "e9cf25c326a6a03eff6c0d6c7b31a93e"},
			{intRef(), "acl", "2.2.51", "14.el7", "x86_64", "acl-2.2.51-14.el7.src.rpm", 201225, "GPLv2+", "CentOS", "", "Access control list utilities", "885771078c3fe1efa43e3e5e0f2b5246"},
			{intRef(), "httpd24-libnghttp2", "1.7.1", "7.el7", "x86_64", "httpd24-nghttp2-1.7.1-7.el7.src.rpm", 142738, "MIT", "CentOS", "", "A library implementing the HTTP/2 protocol", "2b27a006c985eaf902fef4866d4c21cd"},
			{intRef(), "httpd24-httpd", "2.4.34", "7.el7", "x86_64", "httpd24-httpd-2.4.34-7.el7.src.rpm", 4739300, "ASL 2.0", "CentOS", "", "Apache HTTP Server", "4011d451af79d2177345b9112fd36730"},
			{intRef(), "libsemanage", "2.5", "14.el7", "x86_64", "libsemanage-2.5-14.el7.src.rpm", 302329, "LGPLv2+", "CentOS", "", "SELinux binary policy manipulation library", "95dadf0074c4bbd2d63b0c6d5e6922fa"},
			{intRef(1), "httpd24-mod_ssl", "2.4.34", "7.el7", "x86_64", "httpd24-httpd-2.4.34-7.el7.src.rpm", 241612, "ASL 2.0", "CentOS", "", "SSL/TLS module for the Apache HTTP Server", "a2b99bfd142913c2b2c2a286f74602ac"},
			{intRef(), "libutempter", "1.1.6", "4.el7", "x86_64", "libutempter-1.1.6-4.el7.src.rpm", 49749, "LGPLv2+", "CentOS", "", "A privileged helper for utmp/wtmp updates", "8e28747e6a84b84af7d78c84cd8cd5e8"},
			{intRef(), "httpd24-mod_session", "2.4.34", "7.el7", "x86_64", "httpd24-httpd-2.4.34-7.el7.src.rpm", 110671, "ASL 2.0", "CentOS", "", "Session interface for the Apache HTTP Server", "82767fc9af5a1076ba85a10c42f8dbd0"},
			{intRef(), "qrencode-libs", "3.4.1", "3.el7", "x86_64", "qrencode-3.4.1-3.el7.src.rpm", 126732, "LGPLv2+", "CentOS", "", "QR Code encoding library - Shared libraries", "ac1de0c46bc1886e12ea80771951cde2"},
			{intRef(), "httpd24-httpd-devel", "2.4.34", "7.el7", "x86_64", "httpd24-httpd-2.4.34-7.el7.src.rpm", 836817, "ASL 2.0", "CentOS", "", "Development interfaces for the Apache HTTP server", "ea134c6ed64ee7578b79ad88ceb3290e"},
			{intRef(), "python-libs", "2.7.5", "76.el7", "x86_64", "python-2.7.5-76.el7.src.rpm", 24713067, "Python", "CentOS", "", "Runtime libraries for Python", "f7aa64ba20585799cc289eddc634d4d8"},
			{intRef(1), "enchant", "1.6.0", "8.el7", "x86_64", "enchant-1.6.0-8.el7.src.rpm", 130894, "LGPLv2+", "CentOS", "", "An Enchanting Spell Checking Library", "c692bc9199d4900f54f1734dedd0cb49"},
			{intRef(), "atlas-devel", "3.10.1", "12.el7", "x86_64", "atlas-3.10.1-12.el7.src.rpm", 2201996, "BSD", "CentOS", "", "Development libraries for ATLAS", "0372d52cdeec402eba19a2107de574e9"},
			{intRef(), "libtool-ltdl", "2.4.2", "22.el7_3", "x86_64", "libtool-2.4.2-22.el7_3.src.rpm", 67814, "LGPLv2+", "CentOS", "", "Runtime libraries for GNU Libtool Dynamic Module Loader", "ff22ccbfa94b59e371c3073511c533c9"},
			{intRef(), "python-gobject-base", "3.22.0", "1.el7_4.1", "x86_64", "pygobject3-3.22.0-1.el7_4.1.src.rpm", 1123114, "LGPLv2+ and MIT", "CentOS", "", "Python 2 bindings for GObject Introspection base package", "e7f47d020d89e7ecc10aa699c0f8a4f0"},
			{intRef(), "pyliblzma", "0.5.3", "11.el7", "x86_64", "pyliblzma-0.5.3-11.el7.src.rpm", 190112, "LGPLv3+", "CentOS", "", "Python bindings for lzma", "a09aa8e806a3f6285e9fe765d69d3ee9"},
			{intRef(), "python-urlgrabber", "3.10", "9.el7", "noarch", "python-urlgrabber-3.10-9.el7.src.rpm", 503349, "LGPLv2+", "CentOS", "", "A high-level cross-protocol url-grabber", "00eda8adf14af162fd4dfc89ab3d5e45"},
			{intRef(), "pyxattr", "0.5.1", "5.el7", "x86_64", "pyxattr-0.5.1-5.el7.src.rpm", 63304, "LGPLv2+", "CentOS", "", "Extended attributes library wrapper for Python", "e85121da681760c48a2af6ed8aa31b4e"},
			{intRef(), "python-kitchen", "1.1.1", "5.el7", "noarch", "python-kitchen-1.1.1-5.el7.src.rpm", 1465161, "LGPLv2+", "CentOS", "", "Small, useful pieces of code to make python coding easier", "917912ebf6164135f1f9871150432977"},
			{intRef(), "json-c", "0.11", "4.el7_0", "x86_64", "json-c-0.11-4.el7_0.src.rpm", 65593, "MIT", "CentOS", "", "A JSON implementation in C", "62bcb642b82d3b3aeb99ba9ba1b40715"},
			{intRef(), "procps-ng", "3.3.10", "23.el7", "x86_64", "procps-ng-3.3.10-23.el7.src.rpm", 760565, "GPL+ and GPLv2 and GPLv2+ and GPLv3+ and LGPLv2+", "CentOS", "", "System and process monitoring utilities", "b7dd3953621742926579dc26c71f5b52"},
			{intRef(7), "device-mapper", "1.02.149", "10.el7_6.3", "x86_64", "lvm2-2.02.180-10.el7_6.3.src.rpm", 339987, "GPLv2", "CentOS", "", "Device mapper utility", "c3f668105fef62e64e7e3fd43c3e558f"},
			{intRef(), "cryptsetup-libs", "2.0.3", "3.el7", "x86_64", "cryptsetup-2.0.3-3.el7.src.rpm", 1219591, "GPLv2+ and LGPLv2+", "CentOS", "", "Cryptsetup shared library", "61b5b318bfcd411aec021e25d9b76070"},
			{intRef(), "kmod", "20", "23.el7", "x86_64", "kmod-20-23.el7.src.rpm", 243559, "GPLv2+", "CentOS", "", "Linux kernel module management utilities", "99ea42add73605e4c070a5db69a5818c"},
			{intRef(), "systemd-libs", "219", "62.el7_6.5", "x86_64", "systemd-219-62.el7_6.5.src.rpm", 1263456, "LGPLv2+ and MIT", "CentOS", "", "systemd libraries", "9577513987ddb8ff723a0b94367fc8c7"},
			{intRef(), "systemd", "219", "62.el7_6.5", "x86_64", "systemd-219-62.el7_6.5.src.rpm", 24440903, "LGPLv2+ and MIT and GPLv2+", "CentOS", "", "A System and Service Manager", "20127ecb38e5f361019f50f3ae7bc8a7"},
			{intRef(), "elfutils-default-yama-scope", "0.172", "2.el7", "noarch", "elfutils-0.172-2.el7.src.rpm", 1810, "GPLv2+ or LGPLv3+", "CentOS", "", "Default yama attach scope sysctl setting", "5f2326e3ab4ee0002a6de12404aa1668"},
			{intRef(), "iputils", "20160308", "10.el7", "x86_64", "iputils-20160308-10.el7.src.rpm", 343497, "BSD and GPLv2+", "CentOS", "", "Network monitoring tools including ping", "dfa219e6db737a1eb1b5cb23b532e858"},
			{intRef(), "dbus-glib", "0.100", "7.el7", "x86_64", "dbus-glib-0.100-7.el7.src.rpm", 301237, "AFL and GPLv2+", "CentOS", "", "GLib bindings for D-Bus", "fea831a1bb3627640677db4d441a65c6"},
			{intRef(), "pth", "2.0.7", "23.el7", "x86_64", "pth-2.0.7-23.el7.src.rpm", 267851, "LGPLv2+", "CentOS", "", "The GNU Portable Threads library", "2d6939b73767d0f96b02677bcbeef2e2"},
			{intRef(), "rpm-build-libs", "4.11.3", "35.el7", "x86_64", "rpm-4.11.3-35.el7.src.rpm", 166776, "GPLv2+ and LGPLv2+ with exceptions", "CentOS", "", "Libraries for building and signing RPM packages", "aba616fc3d1a25cfc77eb27db8610457"},
			{intRef(), "gpgme", "1.3.2", "5.el7", "x86_64", "gpgme-1.3.2-5.el7.src.rpm", 547534, "LGPLv2+", "CentOS", "", "GnuPG Made Easy - high level crypto API", "d693313ae4184395e902c28ce87d4573"},
			{intRef(), "yum-plugin-fastestmirror", "1.1.31", "50.el7", "noarch", "yum-utils-1.1.31-50.el7.src.rpm", 53895, "GPLv2+", "CentOS", "", "Yum plugin which chooses fastest repository from a mirrorlist", "27a1d31b36b337a4ab254926cd2665fd"},
			{intRef(), "yum-utils", "1.1.31", "50.el7", "noarch", "yum-utils-1.1.31-50.el7.src.rpm", 345277, "GPLv2+", "CentOS", "", "Utilities based around the yum package manager", "3a596d781e91319c17ba84f82256bf5c"},
			{intRef(), "passwd", "0.79", "4.el7", "x86_64", "passwd-0.79-4.el7.src.rpm", 429874, "BSD or GPL+", "CentOS", "", "An utility for setting or changing passwords using PAM", "bc5080b013d6a43fb00e57bba7517e0f"},
			{intRef(2), "vim-minimal", "7.4.160", "5.el7", "x86_64", "vim-7.4.160-5.el7.src.rpm", 917744, "Vim", "CentOS", "", "A minimal version of the VIM editor", "64a183937eac1409f64824544ef5220d"},
			{intRef(), "gpg-pubkey", "f4a80eb5", "53a7ff4b", "", "", 0, "pubkey", "", "", "gpg(CentOS-7 Key (CentOS 7 Official Signing Key) <security@centos.org>)", ""},
			{intRef(), "lzo", "2.06", "8.el7", "x86_64", "lzo-2.06-8.el7.src.rpm", 165360, "GPLv2+", "CentOS", "", "Data compression library with very fast (de)compression", "fb57116a5f1f0e494abb8db7e7bfb249"},
			{intRef(), "libcroco", "0.6.12", "4.el7", "x86_64", "libcroco-0.6.12-4.el7.src.rpm", 320955, "LGPLv2", "CentOS", "", "A CSS2 parsing library", "8f570a2d6cb38d68f7c099b8130a1068"},
			{intRef(), "libarchive", "3.1.2", "10.el7_2", "x86_64", "libarchive-3.1.2-10.el7_2.src.rpm", 745291, "BSD", "CentOS", "", "A library for handling streaming archive formats", "b06fb8340a6a96b935488dd04d12e645"},
			{intRef(), "gettext", "0.19.8.1", "2.el7", "x86_64", "gettext-0.19.8.1-2.el7.src.rpm", 5029340, "GPLv3+ and LGPLv2+", "CentOS", "", "GNU libraries and utilities for producing multi-lingual messages", "e8b96713d3d1d5080233abae6e9c5e10"},
			{intRef(), "groff-base", "1.22.2", "8.el7", "x86_64", "groff-1.22.2-8.el7.src.rpm", 3453946, "GPLv3+ and GFDL and BSD and MIT", "CentOS", "", "Parts of the groff formatting system required to display manual pages", "591d2a12420abc9fa9934414fc0987f4"},
			{intRef(), "centos-release-scl-rh", "2", "3.el7.centos", "noarch", "centos-release-scl-rh-2-3.el7.centos.src.rpm", 20015, "GPLv2", "CentOS", "", "Software collections from the CentOS SCLo SIG (upstream scl only)", "d991ba7e51ac40b70cfbf486715b5e71"},
			{intRef(), "gpg-pubkey", "f2ee9d55", "560cfc0a", "", "", 0, "pubkey", "", "", "gpg(CentOS SoftwareCollections SIG (https://wiki.centos.org/SpecialInterestGroup/SCLo) <security@centos.org>)", ""},
			{intRef(), "libuuid", "2.23.2", "59.el7_6.1", "x86_64", "util-linux-2.23.2-59.el7_6.1.src.rpm", 20326, "BSD", "CentOS", "", "Universally unique ID library", "5e791f417b4a43383a56b82a066be7a8"},
			{intRef(), "mpfr", "3.1.1", "4.el7", "x86_64", "mpfr-3.1.1-4.el7.src.rpm", 554279, "LGPLv3+ and GPLv3+ and GFDL", "CentOS", "", "A C library for multiple-precision floating-point computations", "4cedb227fd4730529b0874792007692b"},
			{intRef(), "libstdc++", "4.8.5", "36.el7_6.1", "x86_64", "gcc-4.8.5-36.el7_6.1.src.rpm", 1077442, "GPLv3+ and GPLv3+ with exceptions and GPLv2+ with exceptions and LGPLv2+ and BSD", "CentOS", "", "GNU Standard C++ Library", "d2b021bf5f895c786ce110a0ab2267d7"},
			{intRef(2), "libpng", "1.5.13", "7.el7_2", "x86_64", "libpng-1.5.13-7.el7_2.src.rpm", 616101, "zlib", "CentOS", "", "A library of functions for manipulating PNG image format files", "f474205e2ea2073937a95b3c55741e46"},
			{intRef(2), "libpng-devel", "1.5.13", "7.el7_2", "x86_64", "libpng-1.5.13-7.el7_2.src.rpm", 216267, "zlib", "CentOS", "", "Development tools for programs to manipulate PNG image format files", "ce0bee9bad720d02a716b4c49471bfcb"},
			{intRef(), "libblkid", "2.23.2", "59.el7_6.1", "x86_64", "util-linux-2.23.2-59.el7_6.1.src.rpm", 265973, "LGPLv2+", "CentOS", "", "Block device ID library", "88156585f9251e493e252b9a691c7e30"},
			{intRef(), "libjpeg-turbo", "1.2.90", "6.el7", "x86_64", "libjpeg-turbo-1.2.90-6.el7.src.rpm", 350499, "IJG", "CentOS", "", "A MMX/SSE2 accelerated library for manipulating JPEG image files", "34514802e4fea3e664d8beacd7ac6b4d"},
			{intRef(), "rh-nodejs10-runtime", "3.2", "2.el7", "x86_64", "rh-nodejs10-3.2-2.el7.src.rpm", 3921, "MIT", "CentOS", "", "Package that handles rh-nodejs10 Software Collection.", "ff46d4aba9f7980851a9bce73a8bea51"},
			{intRef(), "libXau", "1.0.8", "2.1.el7", "x86_64", "libXau-1.0.8-2.1.el7.src.rpm", 55529, "MIT", "CentOS", "", "Sample Authorization Protocol for X", "042d0ad179921f12e4b33e7123a0142b"},
			{intRef(), "xorg-x11-proto-devel", "2018.4", "1.el7", "noarch", "xorg-x11-proto-devel-2018.4-1.el7.src.rpm", 1835516, "MIT", "CentOS", "", "X.Org X11 Protocol headers", "ca0a72d5caf58a6e4de1f82668ede103"},
			{intRef(), "libxcb-devel", "1.13", "1.el7", "x86_64", "libxcb-1.13-1.el7.src.rpm", 2717470, "MIT", "CentOS", "", "Development files for libxcb", "da252204ac03df04fc7bef8755095c0d"},
			{intRef(), "rh-nodejs10-npm", "6.4.1", "10.10.0.2.el7", "x86_64", "rh-nodejs10-nodejs-10.10.0-2.el7.src.rpm", 16465524, "MIT and ASL 2.0 and ISC and BSD", "CentOS", "", "Node.js Package Manager", "d33d2a18fea63776538dbf85e033dbeb"},
			{intRef(), "dejavu-fonts-common", "2.33", "6.el7", "noarch", "dejavu-fonts-2.33-6.el7.src.rpm", 130455, "Bitstream Vera and Public Domain", "CentOS", "", "Common files for the Dejavu font set", "61a93ec3edc83c900c0730e52ac1d1cb"},
			{intRef(), "fontconfig", "2.13.0", "4.3.el7", "x86_64", "fontconfig-2.13.0-4.3.el7.src.rpm", 666475, "MIT and Public Domain and UCD", "CentOS", "", "Font configuration and customization library", "6db8946075bf52456fa1ab75d05830e0"},
			{intRef(), "postgresql", "9.2.24", "1.el7_5", "x86_64", "postgresql-9.2.24-1.el7_5.src.rpm", 16706464, "PostgreSQL", "CentOS", "", "PostgreSQL client programs", "9833dd46d3c5c8cb5d537faee328db58"},
			{intRef(), "cpp", "4.8.5", "36.el7_6.1", "x86_64", "gcc-4.8.5-36.el7_6.1.src.rpm", 15644861, "GPLv3+ and GPLv3+ with exceptions and GPLv2+ with exceptions and LGPLv2+ and BSD", "CentOS", "", "The C Preprocessor", "2044dee2673e3aabfb7df7e4e152227e"},
			{intRef(), "libstdc++-devel", "4.8.5", "36.el7_6.1", "x86_64", "gcc-4.8.5-36.el7_6.1.src.rpm", 8436804, "GPLv3+ and GPLv3+ with exceptions and GPLv2+ with exceptions and LGPLv2+ and BSD", "CentOS", "", "Header files and libraries for C++ development", "90dc72456e50813f4a311d17514e21eb"},
			{intRef(), "fipscheck-lib", "1.4.1", "6.el7", "x86_64", "fipscheck-1.4.1-6.el7.src.rpm", 11466, "BSD", "CentOS", "", "Library files for fipscheck", "93aa3152b03992c02e0ed2759b1d3ac8"},
			{intRef(1), "perl-parent", "0.225", "244.el7", "noarch", "perl-parent-0.225-244.el7.src.rpm", 8141, "GPL+ or Artistic", "CentOS", "", "Establish an ISA relationship with base classes at compile time", "d902f1a3426caa19ccbccf09263c54f6"},
			{intRef(), "perl-podlators", "2.5.1", "3.el7", "noarch", "perl-podlators-2.5.1-3.el7.src.rpm", 287679, "GPL+ or Artistic", "CentOS", "", "Format POD source into various output formats", "8e070dd716e3adf3c297291accaa4348"},
			{intRef(1), "perl-Pod-Escapes", "1.04", "294.el7_6", "noarch", "perl-5.16.3-294.el7_6.src.rpm", 21091, "GPL+ or Artistic", "CentOS", "", "Perl module for resolving POD escape sequences", "ad4d7216b06eb48942c30d4bc2b09755"},
			{intRef(), "perl-Encode", "2.51", "7.el7", "x86_64", "perl-Encode-2.51-7.el7.src.rpm", 10176350, "GPL+ or Artistic", "CentOS", "", "Character encodings in Perl", "713ce27ff4434807ff7060b333ed8873"},
			{intRef(4), "perl-libs", "5.16.3", "294.el7_6", "x86_64", "perl-5.16.3-294.el7_6.src.rpm", 1647344, "GPL+ or Artistic", "CentOS", "", "The libraries for the perl runtime", "669be9d414caaf85799b0dfa644c4e81"},
			{intRef(4), "perl-Time-HiRes", "1.9725", "3.el7", "x86_64", "perl-Time-HiRes-1.9725-3.el7.src.rpm", 94069, "GPL+ or Artistic", "CentOS", "", "High resolution alarm, sleep, gettimeofday, interval timers", "43f7c93d02750221738a8c3bc845a1ca"},
			{intRef(), "perl-threads", "1.87", "4.el7", "x86_64", "perl-threads-1.87-4.el7.src.rpm", 98615, "GPL+ or Artistic", "CentOS", "", "Perl interpreter-based threads", "29739d2f766c8bd85e85de9f8ccb7367"},
			{intRef(), "perl-Carp", "1.26", "244.el7", "noarch", "perl-Carp-1.26-244.el7.src.rpm", 28276, "GPL+ or Artistic", "CentOS", "", "Alternative warn and die for modules", "0f9ffc1d8836311235869048667422e9"},
			{intRef(), "perl-constant", "1.27", "2.el7", "noarch", "perl-constant-1.27-2.el7.src.rpm", 26364, "GPL+ or Artistic", "CentOS", "", "Perl pragma to declare constants", "f3ecfa1ce48e13496322e341a3f0b72c"},
			{intRef(4), "perl-macros", "5.16.3", "294.el7_6", "x86_64", "perl-5.16.3-294.el7_6.src.rpm", 5134, "GPL+ or Artistic", "CentOS", "", "Macros for rpmbuild", "7c63786a9688ac528e398b27e86198d8"},
			{intRef(), "perl-File-Temp", "0.23.01", "3.el7", "noarch", "perl-File-Temp-0.23.01-3.el7.src.rpm", 158781, "GPL+ or Artistic", "CentOS", "", "Return name and handle of a temporary file safely", "396b0a1af3a203e1cb33c98814d5a6d8"},
			{intRef(), "perl-PathTools", "3.40", "5.el7", "x86_64", "perl-PathTools-3.40-5.el7.src.rpm", 174131, "(GPL+ or Artistic) and BSD", "CentOS", "", "PathTools Perl module (Cwd, File::Spec)", "c597987071b0f81a0029667e83e1ecae"},
			{intRef(1), "perl-Pod-Simple", "3.28", "4.el7", "noarch", "perl-Pod-Simple-3.28-4.el7.src.rpm", 538320, "GPL+ or Artistic", "CentOS", "", "Framework for parsing POD documentation", "8e5edeb936a154537062b776eb41cb7d"},
			{intRef(4), "perl", "5.16.3", "294.el7_6", "x86_64", "perl-5.16.3-294.el7_6.src.rpm", 23552128, "(GPL+ or Artistic) and (GPLv2+ or Artistic) and Copyright Only and MIT and Public Domain and UCD", "CentOS", "", "Practical Extraction and Report Language", "547b1d620cedc114b9a1a681c1f2a4bd"},
			{intRef(), "perl-TermReadKey", "2.30", "20.el7", "x86_64", "perl-TermReadKey-2.30-20.el7.src.rpm", 59999, "(Copyright only) and (Artistic or GPL+)", "CentOS", "", "A perl module for simple terminal control", "48356fe925819f41d6446ffc09d1b21d"},
			{intRef(), "perl-Test-Harness", "3.28", "3.el7", "noarch", "perl-Test-Harness-3.28-3.el7.src.rpm", 607239, "GPL+ or Artistic", "CentOS", "", "Run Perl standard test scripts with statistics", "f131d5755e383c37135b023649a236b3"},
			{intRef(), "expat-devel", "2.1.0", "10.el7_3", "x86_64", "expat-2.1.0-10.el7_3.src.rpm", 182002, "MIT", "CentOS", "", "Libraries and header files to develop applications using expat", "5ae674da13c0df4ac2068c2bf9d75bc8"},
			{intRef(), "libcom_err-devel", "1.42.9", "13.el7", "x86_64", "e2fsprogs-1.42.9-13.el7.src.rpm", 17196, "MIT", "CentOS", "", "Common error description library", "6111c715fe35d0a553d57964ab2790df"},
			{intRef(), "tzdata", "2018i", "1.el7", "noarch", "tzdata-2018i-1.el7.src.rpm", 2007607, "Public Domain", "CentOS", "", "Timezone data", "d85d7853af12b5daefb67cb6fadb8c15"},
			{intRef(), "xz-devel", "5.2.2", "1.el7", "x86_64", "xz-5.2.2-1.el7.src.rpm", 168793, "LGPLv2+", "CentOS", "", "Devel libraries & headers for liblzma", "5c84b76faef54c4d1140637f303b9eac"},
			{intRef(), "ncurses-base", "5.9", "14.20130511.el7_4", "noarch", "ncurses-5.9-14.20130511.el7_4.src.rpm", 223432, "MIT", "CentOS", "", "Descriptions of common terminals", "c955fdfe1f8f20b1053a34918e44bd58"},
			{intRef(), "libsmartcols", "2.23.2", "59.el7_6.1", "x86_64", "util-linux-2.23.2-59.el7_6.1.src.rpm", 164608, "LGPLv2+", "CentOS", "", "Formatting library for ls-like programs.", "0aa54f1c264794e869936732ee817499"},
			{intRef(), "chkconfig", "1.7.4", "1.el7", "x86_64", "chkconfig-1.7.4-1.el7.src.rpm", 779531, "GPLv2", "CentOS", "", "A system tool for maintaining the /etc/rc*.d hierarchy", "e126177e9ac5841deca96f4e15340613"},
			{intRef(), "openssh", "7.4p1", "16.el7", "x86_64", "openssh-7.4p1-16.el7.src.rpm", 1995364, "BSD", "CentOS", "", "An open source implementation of SSH protocol versions 1 and 2", "805b1fb17b3b7a41b2df9fd89b75e377"},
			{intRef(), "ncurses", "5.9", "14.20130511.el7_4", "x86_64", "ncurses-5.9-14.20130511.el7_4.src.rpm", 439378, "MIT", "CentOS", "", "Ncurses support utilities", "8ca93e2831102818759a22e22e871268"},
			{intRef(), "kernel-headers", "3.10.0", "957.10.1.el7", "x86_64", "kernel-3.10.0-957.10.1.el7.src.rpm", 3823832, "GPLv2", "CentOS", "", "Header files for the Linux kernel for use by glibc", "17c5c27d1c0577522a0bd74f96990825"},
			{intRef(), "setup", "2.8.71", "10.el7", "noarch", "setup-2.8.71-10.el7.src.rpm", 696893, "Public Domain", "CentOS", "", "A set of system configuration and setup files", "df4a32b192e93cc82ed77e10d204ce98"},
			{intRef(), "glibc-devel", "2.17", "260.el7_6.3", "x86_64", "glibc-2.17-260.el7_6.3.src.rpm", 1066078, "LGPLv2+ and LGPLv2+ with exceptions and GPLv2+", "CentOS", "", "Object files for development using standard C libraries.", "9b23df2963af54dc8f68fc378b521d04"},
			{intRef(), "basesystem", "10.0", "7.el7.centos", "noarch", "basesystem-10.0-7.el7.centos.src.rpm", 0, "Public Domain", "CentOS", "", "The skeleton package which defines a simple CentOS Linux system", "778eb3d19a934f42d1354ed795d06b4f"},
			{intRef(), "m4", "1.4.16", "10.el7", "x86_64", "m4-1.4.16-10.el7.src.rpm", 525707, "GPLv3+", "CentOS", "", "The GNU macro processor", "a71de093f3eba336d4fe4044fb6db349"},
			{intRef(), "zlib", "1.2.7", "18.el7", "x86_64", "zlib-1.2.7-18.el7.src.rpm", 185294, "zlib and Boost", "CentOS", "", "The compression and decompression library", "063a85f8271de9b41e9ac9657962e3ec"},
			{intRef(), "pcre-devel", "8.32", "17.el7", "x86_64", "pcre-8.32-17.el7.src.rpm", 1505771, "BSD", "CentOS", "", "Development files for pcre", "30c6f68dd21b31e7d1f569ea8d4c003a"},
			{intRef(), "nss-util", "3.36.0", "1.1.el7_6", "x86_64", "nss-util-3.36.0-1.1.el7_6.src.rpm", 199008, "MPLv2.0", "CentOS", "", "Network Security Services Utilities Library", "eacfab0995b7a9d221edd5d23700dff3"},
			{intRef(), "libX11", "1.6.5", "2.el7", "x86_64", "libX11-1.6.5-2.el7.src.rpm", 1325648, "MIT", "CentOS", "", "Core X11 protocol client library", "381f3dba6b6e3e3bea2241650651740a"},
			{intRef(), "libffi", "3.0.13", "18.el7", "x86_64", "libffi-3.0.13-18.el7.src.rpm", 47766, "MIT and Public Domain", "CentOS", "", "A portable foreign function interface library", "f05f7a3749342b925fa10ca7de2f0116"},
			{intRef(), "libX11-devel", "1.6.5", "2.el7", "x86_64", "libX11-1.6.5-2.el7.src.rpm", 1100103, "MIT", "CentOS", "", "Development files for libX11", "2b5d00f3b5fd53acbddb9f25e64bccb0"},
			{intRef(), "libattr", "2.4.46", "13.el7", "x86_64", "attr-2.4.46-13.el7.src.rpm", 19896, "LGPLv2+", "CentOS", "", "Dynamic library for extended attribute support", "209b05c7b77a33a4cc176077945833e8"},
			{intRef(), "libXext", "1.3.3", "3.el7", "x86_64", "libXext-1.3.3-3.el7.src.rpm", 87102, "MIT", "CentOS", "", "X.Org X11 libXext runtime library", "fa7e50dec56e01bbe4d33adaec446c9c"},
			{intRef(), "libacl", "2.2.51", "14.el7", "x86_64", "acl-2.2.51-14.el7.src.rpm", 37056, "LGPLv2+", "CentOS", "", "Dynamic library for access control list support", "c3243d728de0ad95e0c05230e6463bdd"},
			{intRef(), "libXpm-devel", "3.5.12", "1.el7", "x86_64", "libXpm-3.5.12-1.el7.src.rpm", 68654, "MIT", "CentOS", "", "X.Org X11 libXpm development package", "dffb341ed28f76a352383da376a9491f"},
			{intRef(), "ncurses-libs", "5.9", "14.20130511.el7_4", "x86_64", "ncurses-5.9-14.20130511.el7_4.src.rpm", 1028216, "MIT", "CentOS", "", "Ncurses libraries", "bba677eda5184b5127b7817d2c1dec12"},
			{intRef(), "rsync", "3.1.2", "4.el7", "x86_64", "rsync-3.1.2-4.el7.src.rpm", 834618, "GPLv3+", "CentOS", "", "A program for synchronizing files over a network", "a50dfc05a59066dea8ab23dda156a45a"},
			{intRef(), "libsepol", "2.5", "10.el7", "x86_64", "libsepol-2.5-10.el7.src.rpm", 686640, "LGPLv2+", "CentOS", "", "SELinux binary policy manipulation library", "cb2c225685a559d68fdb0e07f9b488a2"},
			{intRef(), "libgpg-error-devel", "1.12", "3.el7", "x86_64", "libgpg-error-1.12-3.el7.src.rpm", 29702, "LGPLv2+", "CentOS", "", "Development files for the libgpg-error package", "1b2e1179768d1299fa77ffb94b9460a4"},
			{intRef(), "pcre", "8.32", "17.el7", "x86_64", "pcre-8.32-17.el7.src.rpm", 1475532, "BSD", "CentOS", "", "Perl-compatible regular expression library", "04a1926cc29eef0f79f8183eb7907024"},
			{intRef(), "libedit", "3.0", "12.20121213cvs.el7", "x86_64", "libedit-3.0-12.20121213cvs.el7.src.rpm", 244257, "BSD", "CentOS", "", "The NetBSD Editline library", "9e248aca4576e17c188396aa91931c63"},
			{intRef(), "sed", "4.2.2", "5.el7", "x86_64", "sed-4.2.2-5.el7.src.rpm", 601208, "GPLv3+", "CentOS", "", "A GNU stream text editor", "4d696dbd3061493258490f40c625adba"},
			{intRef(), "perl-Git", "1.8.3.1", "20.el7", "noarch", "git-1.8.3.1-20.el7.src.rpm", 58567, "GPLv2", "CentOS", "", "Perl interface to Git", "801ebea11b001365abebd04e2da2690f"},
			{intRef(), "p11-kit", "0.23.5", "3.el7", "x86_64", "p11-kit-0.23.5-3.el7.src.rpm", 1337825, "BSD", "CentOS", "", "Library for loading and sharing PKCS#11 modules", "9a0f49ca3b6d58aeb05780003cae38f1"},
			{intRef(), "libsepol-devel", "2.5", "10.el7", "x86_64", "libsepol-2.5-10.el7.src.rpm", 126580, "LGPLv2+", "CentOS", "", "Header files and libraries used to build policy manipulation tools", "7737c3c3ec497910db7bd7b4060732dd"},
			{intRef(1), "gmp", "6.0.0", "15.el7", "x86_64", "gmp-6.0.0-15.el7.src.rpm", 657046, "LGPLv3+ or GPLv2+", "CentOS", "", "A GNU arbitrary precision library", "4cd60ebf94766a46c590e17036ece777"},
			{intRef(), "libverto-devel", "0.2.5", "4.el7", "x86_64", "libverto-0.2.5-4.el7.src.rpm", 25871, "MIT", "CentOS", "", "Development files for libverto", "474e3916ae4568f0ec63415ecbd0afb1"},
			{intRef(), "libtasn1", "4.10", "1.el7", "x86_64", "libtasn1-4.10-1.el7.src.rpm", 424486, "GPLv3+ and LGPLv2+", "CentOS", "", "The ASN.1 library used in GNUTLS", "5bd1b0b4094fac08713f986572a8625e"},
			{intRef(1), "openssl-devel", "1.0.2k", "16.el7_6.1", "x86_64", "openssl-1.0.2k-16.el7_6.1.src.rpm", 3268426, "OpenSSL", "CentOS", "", "Files for development of applications which will use OpenSSL", "87aef62e847562a11beef5bd1a9015e9"},
			{intRef(), "ca-certificates", "2018.2.22", "70.0.el7_5", "noarch", "ca-certificates-2018.2.22-70.0.el7_5.src.rpm", 973960, "Public Domain", "CentOS", "", "The Mozilla CA root certificate bundle", "a95373fc5273e7c757a240316d4430b9"},
			{intRef(), "libxslt-devel", "1.1.28", "5.el7", "x86_64", "libxslt-1.1.28-5.el7.src.rpm", 2502662, "MIT", "CentOS", "", "Development files for libxslt", "e8193a392742a5dcc7c5a354016671b7"},
			{intRef(), "coreutils", "8.22", "23.el7", "x86_64", "coreutils-8.22-23.el7.src.rpm", 14588674, "GPLv3+", "CentOS", "", "A set of basic GNU tools commonly used in shell scripts", "904a2aeeeaa0d833a0dff696c89bd7f3"},
			{intRef(), "automake", "1.13.4", "3.el7", "noarch", "automake-1.13.4-3.el7.src.rpm", 1731909, "GPLv2+ and GFDL and Public Domain and MIT", "CentOS", "", "A GNU tool for automatically creating Makefiles", "211dbf9617136d88757937007240ae9b"},
			{intRef(), "centos-release", "7", "6.1810.2.el7.centos", "x86_64", "centos-release-7-6.1810.2.el7.centos.src.rpm", 41271, "GPLv2", "CentOS", "", "CentOS Linux release file", "9f661612be55e8bb73f78163a41df816"},
			{intRef(), "postgresql-devel", "9.2.24", "1.el7_5", "x86_64", "postgresql-9.2.24-1.el7_5.src.rpm", 4147268, "PostgreSQL", "CentOS", "", "PostgreSQL development header files and libraries", "598c5d3566b7b597b2aecf573607a361"},
			{intRef(), "bzip2-libs", "1.0.6", "13.el7", "x86_64", "bzip2-1.0.6-13.el7.src.rpm", 70093, "BSD", "CentOS", "", "Libraries for applications using bzip2", "0486f0b02e0caa4195761266798e7a26"},
			{intRef(), "wget", "1.14", "18.el7", "x86_64", "wget-1.14-18.el7.src.rpm", 2055557, "GPLv3+", "CentOS", "", "A utility for retrieving files using the HTTP or FTP protocols", "5b4e39845da3994d420e53a1e428cb01"},
			{intRef(), "elfutils-libelf", "0.172", "2.el7", "x86_64", "elfutils-0.172-2.el7.src.rpm", 913443, "GPLv2+ or LGPLv3+", "CentOS", "", "Library to read and write ELF files", "383dff055bd0bc27b026d1359974e00e"},
			{intRef(), "libcurl-devel", "7.29.0", "51.el7", "x86_64", "curl-7.29.0-51.el7.src.rpm", 638377, "MIT", "CentOS", "", "Files needed for building applications with libcurl", "1da33e8aa33d0315ccdb3479b0d32719"},
			{intRef(), "libxml2", "2.9.1", "6.el7_2.3", "x86_64", "libxml2-2.9.1-6.el7_2.3.src.rpm", 1710062, "MIT", "CentOS", "", "Library providing XML and HTML support", "e439b8934c4e9576eccfbc60d3cd714f"},
			{intRef(), "sqlite-devel", "3.7.17", "8.el7", "x86_64", "sqlite-3.7.17-8.el7.src.rpm", 374846, "Public Domain", "CentOS", "", "Development tools for the sqlite3 embeddable SQL database engine", "842f7f00f0ee23d7063b54d49708133e"},
			{intRef(), "readline", "6.2", "10.el7", "x86_64", "readline-6.2-10.el7.src.rpm", 460464, "GPLv3+", "CentOS", "", "A library for editing typed command lines", "b552128087365a58b3566631c678752f"},
			{intRef(1), "make", "3.82", "23.el7", "x86_64", "make-3.82-23.el7.src.rpm", 1160684, "GPLv2+", "CentOS", "", "A GNU tool which simplifies the build process for users", "9e5ab17428b36e59a9d071c1fd1c81de"},
			{intRef(), "cpio", "2.11", "27.el7", "x86_64", "cpio-2.11-27.el7.src.rpm", 689335, "GPLv3+", "CentOS", "", "A GNU archiving program", "32498692f82296d51663586f08cebafc"},
			{intRef(), "rh-python35-runtime", "2.0", "2.el7", "x86_64", "rh-python35-2.0-2.el7.src.rpm", 21152, "GPLv2+", "CentOS", "", "Package that handles rh-python35 Software Collection.", "af5138e84487e1e4404caad6f07ba23e"},
			{intRef(), "apr-util", "1.5.2", "6.el7", "x86_64", "apr-util-1.5.2-6.el7.src.rpm", 198751, "ASL 2.0", "CentOS", "", "Apache Portable Runtime Utility library", "e4abdd79021510c0a5eb7fdea8389881"},
			{intRef(), "glib2", "2.56.1", "2.el7", "x86_64", "glib2-2.56.1-2.el7.src.rpm", 12166425, "LGPLv2+", "CentOS", "", "A library of handy utility functions", "5ba70ca2db724338484aedc7f9d8a4b9"},
			{intRef(), "libgfortran", "4.8.5", "36.el7_6.1", "x86_64", "gcc-4.8.5-36.el7_6.1.src.rpm", 1186168, "GPLv3+ and GPLv3+ with exceptions and GPLv2+ with exceptions and LGPLv2+ and BSD", "CentOS", "", "Fortran runtime", "a210ea42ee5ed4ca8328a4555b0717ab"},
			{intRef(), "sqlite", "3.7.17", "8.el7", "x86_64", "sqlite-3.7.17-8.el7.src.rpm", 814231, "Public Domain", "CentOS", "", "Library that implements an embeddable SQL database engine", "0ceb9434c5a9d281399583087cfe889b"},
			{intRef(), "rh-python35-python-setuptools", "18.0.1", "2.el7", "noarch", "rh-python35-python-setuptools-18.0.1-2.el7.src.rpm", 1785163, "Python or ZPLv2.0", "CentOS", "", "Easily build and distribute Python packages", "56ecb94133907de44994814a67ad2a35"},
			{intRef(), "cracklib", "2.9.0", "11.el7", "x86_64", "cracklib-2.9.0-11.el7.src.rpm", 209610, "LGPLv2+", "CentOS", "", "A password-checking library", "a973d883d6785904e49266428bb3ed90"},
			{intRef(), "rh-python35-python", "3.5.1", "11.el7", "x86_64", "rh-python35-python-3.5.1-11.el7.src.rpm", 42812, "Python", "CentOS", "", "Version 3 of the Python programming language aka Python 3000", "8288cc1578c3556fb1a7542175393166"},
			{intRef(), "libidn", "1.28", "4.el7", "x86_64", "libidn-1.28-4.el7.src.rpm", 630407, "LGPLv2+ and GPLv3+ and GFDL", "CentOS", "", "Internationalized Domain Name support library", "3d582106ac1032c59c1cd4236e5d6415"},
			{intRef(), "libnfnetlink", "1.0.1", "4.el7", "x86_64", "libnfnetlink-1.0.1-4.el7.src.rpm", 47123, "GPLv2+", "CentOS", "", "Netfilter netlink userspace library", "5ea6c06bb1c97e6a62cfa4bba82b6126"},
			{intRef(), "libnetfilter_conntrack", "1.0.6", "1.el7_3", "x86_64", "libnetfilter_conntrack-1.0.6-1.el7_3.src.rpm", 143566, "GPLv2+", "CentOS", "", "Netfilter conntrack userspace library", "d39e408e9135443bab55466af0a08bad"},
			{intRef(), "libcap-ng", "0.7.5", "4.el7", "x86_64", "libcap-ng-0.7.5-4.el7.src.rpm", 50510, "LGPLv2+", "CentOS", "", "An alternate posix capabilities library", "84efa5ca78408f78e606e7c8d1816759"},
			{intRef(), "iproute", "4.11.0", "14.el7", "x86_64", "iproute-4.11.0-14.el7.src.rpm", 1788731, "GPLv2+ and Public Domain", "CentOS", "", "Advanced IP routing and network device configuration tools", "27cd8889841504c76694a98251c274bd"},
			{intRef(), "cracklib-dicts", "2.9.0", "11.el7", "x86_64", "cracklib-2.9.0-11.el7.src.rpm", 9389116, "LGPLv2+", "CentOS", "", "The standard CrackLib dictionaries", "da657b0facf181893f9a38437092dbbf"},
			{intRef(), "libquadmath-devel", "4.8.5", "36.el7_6.1", "x86_64", "gcc-4.8.5-36.el7_6.1.src.rpm", 18514, "GPLv3+ and GPLv3+ with exceptions and GPLv2+ with exceptions and LGPLv2+ and BSD", "CentOS", "", "GCC __float128 support", "3cebb989a4fc2a0787f68ae06061d502"},
			{intRef(), "libpwquality", "1.2.3", "5.el7", "x86_64", "libpwquality-1.2.3-5.el7.src.rpm", 332421, "BSD or GPLv2+", "CentOS", "", "A library for password generation and password quality checking", "e40a69f0a1803dfa3736aba2c8ea1fb4"},
			{intRef(), "apr-util-openssl", "1.5.2", "6.el7", "x86_64", "apr-util-1.5.2-6.el7.src.rpm", 19896, "ASL 2.0", "CentOS", "", "APR utility library OpenSSL crytpo support", "8c4e65262569d8b100622fd143f88d1b"},
			{intRef(), "nss-pem", "1.0.3", "5.el7", "x86_64", "nss-pem-1.0.3-5.el7.src.rpm", 205539, "MPLv1.1", "CentOS", "", "PEM file reader for Network Security Services (NSS)", "24d8ee5cd00822ebc0ceca3e9f0bb906"},
			{intRef(), "hunspell-en-US", "0.20121024", "6.el7", "noarch", "hunspell-en-0.20121024-6.el7.src.rpm", 549311, "LGPLv2+ and LGPLv2 and BSD", "CentOS", "", "US English hunspell dictionaries", "7a675e23f446e224883ea7e334905b61"},
			{intRef(), "nss-sysinit", "3.36.0", "7.1.el7_6", "x86_64", "nss-3.36.0-7.1.el7_6.src.rpm", 14067, "MPLv2.0", "CentOS", "", "System NSS Initialization", "27ff88c4a26101b75c51b31b5e85c435"},
			{intRef(), "libselinux-utils", "2.5", "14.1.el7", "x86_64", "libselinux-2.5-14.1.el7.src.rpm", 172645, "Public Domain", "CentOS", "", "SELinux libselinux utilies", "4480bb426dabc45b7b5fd5f4f6a45a52"},
			{intRef(), "xz", "5.2.2", "1.el7", "x86_64", "xz-5.2.2-1.el7.src.rpm", 798130, "LGPLv2+", "CentOS", "", "LZMA compression utilities", "3e6f05008f7af7931aa8257964527f1a"},
			{intRef(), "zip", "3.0", "11.el7", "x86_64", "zip-3.0-11.el7.src.rpm", 815173, "BSD", "CentOS", "", "A file compression and packaging utility compatible with PKZIP", "f49ae6270a053005b67e417123691ca7"},
			{intRef(), "file-libs", "5.11", "35.el7", "x86_64", "file-5.11-35.el7.src.rpm", 3076831, "BSD", "CentOS", "", "Libraries for applications using libmagic", "3037fd1f781186bcb0fb919b2aa78f54"},
			{intRef(), "xml-common", "0.6.3", "39.el7", "noarch", "sgml-common-0.6.3-39.el7.src.rpm", 46969, "GPL+", "CentOS", "", "Common XML catalog and DTD files", "83630b0f69dd2b423166aa9dd588dbb2"},
			{intRef(), "lz4", "1.7.5", "2.el7", "x86_64", "lz4-1.7.5-2.el7.src.rpm", 366872, "GPLv2+ and BSD", "CentOS", "", "Extremely fast compression algorithm", "89e45994eabac0ba4ccae7c1005ea06c"},
			{intRef(), "libselinux-python", "2.5", "14.1.el7", "x86_64", "libselinux-2.5-14.1.el7.src.rpm", 603236, "Public Domain", "CentOS", "", "SELinux python bindings for libselinux", "cfbad05c1a681e59c0b59d2bfa6663ab"},
			{intRef(), "checkpolicy", "2.5", "8.el7", "x86_64", "checkpolicy-2.5-8.el7.src.rpm", 1288327, "GPLv2", "CentOS", "", "SELinux policy compiler", "a42deadeddfdef88930f5ebcb6e21c8e"},
			{intRef(1), "openssl", "1.0.2k", "16.el7_6.1", "x86_64", "openssl-1.0.2k-16.el7_6.1.src.rpm", 833657, "OpenSSL", "CentOS", "", "Utilities from the general purpose cryptography library with TLS implementation", "3039d54f5d5482ad72948ae5c9b06f24"},
			{intRef(), "python-IPy", "0.75", "6.el7", "noarch", "python-IPy-0.75-6.el7.src.rpm", 121946, "BSD", "CentOS", "", "Python module for handling IPv4 and IPv6 Addresses and Networks", "21be4d5d974076d9817cd9cc616eef0f"},
			{intRef(1), "pkgconfig", "0.27.1", "4.el7", "x86_64", "pkgconfig-0.27.1-4.el7.src.rpm", 105522, "GPLv2+", "CentOS", "", "A tool for determining compilation options", "ef58117c9c617599952c0642ff5e8d00"},
			{intRef(), "redhat-rpm-config", "9.1.0", "87.el7.centos", "noarch", "redhat-rpm-config-9.1.0-87.el7.centos.src.rpm", 174572, "GPL+", "CentOS", "", "CentOS specific rpm configuration files", "fa59b48b92a6e9b34c9fad064eafb839"},
			{intRef(), "rh-python35-python-devel", "3.5.1", "11.el7", "x86_64", "rh-python35-python-3.5.1-11.el7.src.rpm", 600179, "Python", "CentOS", "", "Libraries and header files needed for Python 3 development", "512bbb0dff507c54dacb5c42f418f1e1"},
			{intRef(), "cyrus-sasl-lib", "2.1.26", "23.el7", "x86_64", "cyrus-sasl-2.1.26-23.el7.src.rpm", 396911, "BSD with advertising", "CentOS", "", "Shared libraries needed by applications which use Cyrus SASL", "6d06f1337b143c25bc7358255d433006"},
			{intRef(), "sysvinit-tools", "2.88", "14.dsf.el7", "x86_64", "sysvinit-2.88-14.dsf.el7.src.rpm", 109118, "GPLv2+", "CentOS", "", "Tools used for process and utmp management.", "92a1f861a25257bf73735dc5360940f2"},
			{intRef(), "binutils", "2.27", "34.base.el7", "x86_64", "binutils-2.27-34.base.el7.src.rpm", 25150000, "GPLv3+", "CentOS", "", "A GNU collection of binary utilities", "b9772c7ef8848f4b5372fa8f52222a6a"},
			{intRef(), "cyrus-sasl", "2.1.26", "23.el7", "x86_64", "cyrus-sasl-2.1.26-23.el7.src.rpm", 144492, "BSD with advertising", "CentOS", "", "The Cyrus SASL library", "b9a1f034949fbe65363802143dc7b70c"},
			{intRef(), "libcurl", "7.29.0", "51.el7", "x86_64", "curl-7.29.0-51.el7.src.rpm", 435192, "MIT", "CentOS", "", "A library for getting files from web servers", "998abd47ebced6ef98be2020cafcf569"},
			{intRef(), "openldap-devel", "2.4.44", "21.el7_6", "x86_64", "openldap-2.4.44-21.el7_6.src.rpm", 3847489, "OpenLDAP", "CentOS", "", "LDAP development libraries and header files", "72bd71a4cfb88a9b29dde89c14abc78c"},
			{intRef(), "rpm-libs", "4.11.3", "35.el7", "x86_64", "rpm-4.11.3-35.el7.src.rpm", 611536, "GPLv2+ and LGPLv2+ with exceptions", "CentOS", "", "Libraries for manipulating RPM packages", "2ac2673a85c45699ccf2e5d1a4eb2c4c"},
			{intRef(), "apr-util-devel", "1.5.2", "6.el7", "x86_64", "apr-util-1.5.2-6.el7.src.rpm", 298354, "ASL 2.0", "CentOS", "", "APR utility library development kit", "7e75b92e20e410da142bc2f471923c10"},
			{intRef(), "openldap", "2.4.44", "21.el7_6", "x86_64", "openldap-2.4.44-21.el7_6.src.rpm", 1037424, "OpenLDAP", "CentOS", "", "LDAP support libraries", "0ccf548b74498ba6b9762498609de967"},
			{intRef(), "setools-libs", "3.3.8", "4.el7", "x86_64", "setools-3.3.8-4.el7.src.rpm", 1917790, "LGPLv2", "CentOS", "", "Policy analysis support libraries for SELinux", "d25e0add9591f438d7e1b6d313f67352"},
			{intRef(), "policycoreutils-python", "2.5", "29.el7_6.1", "x86_64", "policycoreutils-2.5-29.el7_6.1.src.rpm", 1299284, "GPLv2", "CentOS", "", "SELinux policy core python utilities", "5127183afef4a2a5d327a2cdffa09a7b"},
			{intRef(), "pinentry", "0.8.1", "17.el7", "x86_64", "pinentry-0.8.1-17.el7.src.rpm", 159929, "GPLv2+", "CentOS", "", "Collection of simple PIN or passphrase entry dialogs", "740fabe008451f11fee22ff086fb13d3"},
			{intRef(), "httpd24-libcurl", "7.61.1", "1.el7", "x86_64", "httpd24-curl-7.61.1-1.el7.src.rpm", 545704, "MIT", "CentOS", "", "A library for getting files from web servers", "05a2f8f385dfc114a46b8b08c676d88e"},
			{intRef(), "httpd24-httpd-tools", "2.4.34", "7.el7", "x86_64", "httpd24-httpd-2.4.34-7.el7.src.rpm", 177148, "ASL 2.0", "CentOS", "", "Tools for use with the Apache HTTP Server", "60c720e48321e50f1788ce5bfbf0e1d6"},
			{intRef(), "ustr", "1.0.4", "16.el7", "x86_64", "ustr-1.0.4-16.el7.src.rpm", 285943, "MIT or LGPLv2+ or BSD", "CentOS", "", "String library, very low memory overhead, simple to import", "58bb2ebb64b048e24a4c7d5d51170f55"},
			{intRef(), "httpd24-mod_auth_kerb", "5.4", "33.el7", "x86_64", "httpd24-mod_auth_kerb-5.4-33.el7.src.rpm", 67371, "BSD and MIT and ASL 2.0", "CentOS", "", "Kerberos authentication module for HTTP", "957f324562e80273620836e6e7bec685"},
			{intRef(2), "shadow-utils", "4.1.5.1", "25.el7", "x86_64", "shadow-utils-4.1.5.1-25.el7.src.rpm", 3540592, "BSD and GPLv2+", "CentOS", "", "Utilities for managing accounts and shadow password files", "22c762981f93de691a7f74a7259a1906"},
			{intRef(), "httpd24-mod_ldap", "2.4.34", "7.el7", "x86_64", "httpd24-httpd-2.4.34-7.el7.src.rpm", 134213, "ASL 2.0", "CentOS", "", "LDAP authentication modules for the Apache HTTP Server", "576646b86df818081cc40ff10aec3746"},
			{intRef(1), "hardlink", "1.0", "19.el7", "x86_64", "hardlink-1.0-19.el7.src.rpm", 16545, "GPL+", "CentOS", "", "Create a tree of hardlinks", "3a1bcd8e12bbdbced75d74b5919a039a"},
			{intRef(), "httpd24", "1.1", "18.el7", "x86_64", "httpd24-1.1-18.el7.src.rpm", 0, "GPLv2+", "CentOS", "", "Package that installs httpd24", "d0d4d214432bf7e782efb96ffa33ca65"},
			{intRef(), "gdbm", "1.10", "8.el7", "x86_64", "gdbm-1.10-8.el7.src.rpm", 184322, "GPLv3+", "CentOS", "", "A GNU set of database routines which use extensible hashing", "770bd49f3898dc1cc831fb9ec12d569a"},
			{intRef(), "rh-python35", "2.0", "2.el7", "x86_64", "rh-python35-2.0-2.el7.src.rpm", 0, "GPLv2+", "CentOS", "", "Package that installs rh-python35", "d1b82d1d26c028d05688f17f7ea8e291"},
			{intRef(), "python", "2.7.5", "76.el7", "x86_64", "python-2.7.5-76.el7.src.rpm", 80907, "Python", "CentOS", "", "An interpreted, interactive, object-oriented programming language", "9882effd17f2a65048dc45ed140ba617"},
			{intRef(), "gcc-gfortran", "4.8.5", "36.el7_6.1", "x86_64", "gcc-4.8.5-36.el7_6.1.src.rpm", 16443248, "GPLv3+ and GPLv3+ with exceptions and GPLv2+ with exceptions and LGPLv2+ and BSD", "CentOS", "", "Fortran support", "c470a080bcd80338a0cb02ea194e87c5"},
			{intRef(), "libxml2-python", "2.9.1", "6.el7_2.3", "x86_64", "libxml2-2.9.1-6.el7_2.3.src.rpm", 1503050, "MIT", "CentOS", "", "Python bindings for the libxml2 library", "4af3055a710e118b3e7c49cb1df4e784"},
			{intRef(), "nss_wrapper", "1.0.3", "1.el7", "x86_64", "nss_wrapper-1.0.3-1.el7.src.rpm", 61024, "BSD", "CentOS", "", "A wrapper for the user, group and hosts NSS API", "d252eb9182b02a1d2d896a7047cc008a"},
			{intRef(), "libffi-devel", "3.0.13", "18.el7", "x86_64", "libffi-3.0.13-18.el7.src.rpm", 27170, "MIT and Public Domain", "CentOS", "", "Development files for libffi", "6d06f43e9777c945f8d508bdbd897bfa"},
			{intRef(), "yum-metadata-parser", "1.1.4", "10.el7", "x86_64", "yum-metadata-parser-1.1.4-10.el7.src.rpm", 58789, "GPLv2", "CentOS", "", "A fast metadata parser for yum", "929f0abd8c5671c69f86fce8d7cdd42d"},
			{intRef(), "python-pycurl", "7.19.0", "19.el7", "x86_64", "python-pycurl-7.19.0-19.el7.src.rpm", 241513, "LGPLv2+ or MIT", "CentOS", "", "A Python interface to libcurl", "d5152315d59bde1faaa203e68de6a64e"},
			{intRef(), "python-iniparse", "0.4", "9.el7", "noarch", "python-iniparse-0.4-9.el7.src.rpm", 115166, "MIT", "CentOS", "", "Python Module for Accessing and Modifying Configuration Data in INI files", "3140d9efca99c4dd5215383bab36bca5"},
			{intRef(), "python-chardet", "2.2.1", "1.el7_1", "noarch", "python-chardet-2.2.1-1.el7_1.src.rpm", 1156541, "LGPLv2", "CentOS", "", "Character encoding auto-detection in Python", "c70da920f85b12ecd25429913e3e861e"},
			{intRef(), "hostname", "3.13", "3.el7", "x86_64", "hostname-3.13-3.el7.src.rpm", 19449, "GPLv2+", "CentOS", "", "Utility to set/show the host name or domain name", "677ae7d92bb29445d1e0b66ecf5cab81"},
			{intRef(), "kpartx", "0.4.9", "123.el7", "x86_64", "device-mapper-multipath-0.4.9-123.el7.src.rpm", 41363, "GPL+", "CentOS", "", "Partition device manager for device-mapper devices", "aecde17836ffb541d265eeaeb9ebd1c8"},
			{intRef(7), "device-mapper-libs", "1.02.149", "10.el7_6.3", "x86_64", "lvm2-2.02.180-10.el7_6.3.src.rpm", 400607, "LGPLv2", "CentOS", "", "Device-mapper shared library", "32edc5c3d11c6bb1577f1c0cf8b43a72"},
			{intRef(), "dracut", "033", "554.el7", "x86_64", "dracut-033-554.el7.src.rpm", 903230, "GPLv2+ and LGPLv2+", "CentOS", "", "Initramfs generator using udev", "57e8bd40745c0cba95d9c3f65acb2205"},
			{intRef(), "elfutils-libs", "0.172", "2.el7", "x86_64", "elfutils-0.172-2.el7.src.rpm", 793087, "GPLv2+ or LGPLv3+", "CentOS", "", "Libraries to handle compiled objects", "8d24c886e0655d29240085a75a5d7047"},
			{intRef(1), "dbus-libs", "1.10.24", "12.el7", "x86_64", "dbus-1.10.24-12.el7.src.rpm", 362560, "(GPLv2+ or AFL) and GPLv2+", "CentOS", "", "Libraries for accessing D-BUS", "76ed3fc760940227749d71884cfcaa33"},
			{intRef(1), "dbus", "1.10.24", "12.el7", "x86_64", "dbus-1.10.24-12.el7.src.rpm", 595210, "(GPLv2+ or AFL) and GPLv2+", "CentOS", "", "D-BUS message bus", "0e7061cc90fda4f111245bee037271e4"},
			{intRef(), "dbus-python", "1.1.1", "9.el7", "x86_64", "dbus-python-1.1.1-9.el7.src.rpm", 848122, "MIT", "CentOS", "", "D-Bus Python Bindings", "6be4da91886c59ae25e5255ebff0a0a9"},
			{intRef(), "gnupg2", "2.0.22", "5.el7_5", "x86_64", "gnupg2-2.0.22-5.el7_5.src.rpm", 6637796, "GPLv3+", "CentOS", "", "Utility for secure communication and data storage", "a293dfefd6596bfe5c54ba68377e334e"},
			{intRef(), "rpm-python", "4.11.3", "35.el7", "x86_64", "rpm-4.11.3-35.el7.src.rpm", 149898, "GPLv2+", "CentOS", "", "Python bindings for apps which will manipulate RPM packages", "ad03ebb739fc7309edd09eba58ab7ee3"},
			{intRef(), "pygpgme", "0.3", "9.el7", "x86_64", "pygpgme-0.3-9.el7.src.rpm", 197501, "LGPLv2+", "CentOS", "", "Python module for working with OpenPGP messages", "67b8c720f59a063adbbca2392bac0281"},
			{intRef(), "yum", "3.4.3", "161.el7.centos", "noarch", "yum-3.4.3-161.el7.centos.src.rpm", 5824869, "GPLv2+", "CentOS", "", "RPM package installer/updater/manager", "ae05719cfd4e3b80b3de83da0e7ba4bf"},
			{intRef(), "yum-plugin-ovl", "1.1.31", "50.el7", "noarch", "yum-utils-1.1.31-50.el7.src.rpm", 22399, "GPLv2+", "CentOS", "", "Yum plugin to work around overlayfs issues", "5c82294b6b298d58a036b719cc667e7c"},
			{intRef(), "rootfiles", "8.1", "11.el7", "noarch", "rootfiles-8.1-11.el7.src.rpm", 599, "Public Domain", "CentOS", "", "The basic required files for the root user's directory", "b9b9eceee7ad38f868520e80f3404642"},
			{intRef(), "libgomp", "4.8.5", "36.el7_6.1", "x86_64", "gcc-4.8.5-36.el7_6.1.src.rpm", 212184, "GPLv3+ and GPLv3+ with exceptions and GPLv2+ with exceptions and LGPLv2+ and BSD", "CentOS", "", "GCC OpenMP v3.0 shared support library", "35872c42fa6e4867151f828230a2636f"},
			{intRef(), "libunistring", "0.9.3", "9.el7", "x86_64", "libunistring-0.9.3-9.el7.src.rpm", 1145761, "LGPLv3+", "CentOS", "", "GNU Unicode string library", "b1b9755e18cb312740d96d89e61ef70e"},
			{intRef(), "gettext-libs", "0.19.8.1", "2.el7", "x86_64", "gettext-0.19.8.1-2.el7.src.rpm", 1546952, "LGPLv2+ and GPLv3+", "CentOS", "", "Libraries for gettext", "3164b6e314d214a982d34c7c77d89e16"},
			{intRef(), "bsdtar", "3.1.2", "10.el7_2", "x86_64", "libarchive-3.1.2-10.el7_2.src.rpm", 110459, "BSD", "CentOS", "", "Manipulate tape archives", "ea6cdf65424903f1ab6eb8d33ed7ed31"},
			{intRef(), "unzip", "6.0", "19.el7", "x86_64", "unzip-6.0-19.el7.src.rpm", 373986, "BSD", "CentOS", "", "A utility for unpacking zip files", "7a396b648ba0120404fe2ef171b79681"},
			{intRef(), "scl-utils", "20130529", "19.el7", "x86_64", "scl-utils-20130529-19.el7.src.rpm", 25141, "GPLv2+", "CentOS", "", "Utilities for alternative packaging", "c8feffb6b8bbcf34c2f20ad60ae1cf41"},
			{intRef(), "centos-release-scl", "2", "3.el7.centos", "noarch", "centos-release-scl-2-3.el7.centos.src.rpm", 20042, "GPLv2", "CentOS", "", "Software collections from the CentOS SCLo SIG", "1985cc482b2f414166e33f22808d944f"},
			{intRef(1), "openssl-libs", "1.0.2k", "16.el7_6.1", "x86_64", "openssl-1.0.2k-16.el7_6.1.src.rpm", 3204708, "OpenSSL", "CentOS", "", "A general purpose cryptography library with TLS implementation", "f22c63c9514e9f24f023d69bc1f7a959"},
			{intRef(), "zlib-devel", "1.2.7", "18.el7", "x86_64", "zlib-1.2.7-18.el7.src.rpm", 135536, "zlib and Boost", "CentOS", "", "Header files and libraries for Zlib development", "b2cb796d4a3938d3659cd6cd89584e57"},
			{intRef(), "libgcc", "4.8.5", "36.el7_6.1", "x86_64", "gcc-4.8.5-36.el7_6.1.src.rpm", 179328, "GPLv3+ and GPLv3+ with exceptions and GPLv2+ with exceptions and LGPLv2+ and BSD", "CentOS", "", "GCC version 4.8 shared support library", "562e2ee43ee56c57acf69b14f6f1535b"},
			{intRef(), "libmpc", "1.0.1", "3.el7", "x86_64", "libmpc-1.0.1-3.el7.src.rpm", 113833, "LGPLv3+ and GFDL", "CentOS", "", "C library for multiple precision complex arithmetic", "835abbe42dbf1e0f94d227d54c01f483"},
			{intRef(), "freetype", "2.8", "12.el7_6.1", "x86_64", "freetype-2.8-12.el7_6.1.src.rpm", 824497, "(FTL or GPLv2+) and BSD and MIT and Public Domain and zlib with acknowledgement", "CentOS", "", "A free and portable font rendering engine", "76a6a154ceceeecc7df8ae70cc1bb6a8"},
			{intRef(), "freetype-devel", "2.8", "12.el7_6.1", "x86_64", "freetype-2.8-12.el7_6.1.src.rpm", 2980705, "(FTL or GPLv2+) and BSD and MIT and Public Domain and zlib with acknowledgement", "CentOS", "", "FreeType development libraries and header files", "843a18fdfc8318aadb50ae99c4b26db9"},
			{intRef(), "postgresql-libs", "9.2.24", "1.el7_5", "x86_64", "postgresql-9.2.24-1.el7_5.src.rpm", 683495, "PostgreSQL", "CentOS", "", "The shared libraries required for any PostgreSQL clients", "e50b47bd3ad3c7182ef059679e65063d"},
			{intRef(), "fontpackages-filesystem", "1.44", "8.el7", "noarch", "fontpackages-1.44-8.el7.src.rpm", 0, "Public Domain", "CentOS", "", "Directories used by font packages", "dbc95bc0a2568eb82f59c004462d90ab"},
			{intRef(), "libICE", "1.0.9", "9.el7", "x86_64", "libICE-1.0.9-9.el7.src.rpm", 152213, "MIT", "CentOS", "", "X.Org X11 ICE runtime library", "dfa3c8915ac16deedaa578e08748b045"},
			{intRef(), "libxcb", "1.13", "1.el7", "x86_64", "libxcb-1.13-1.el7.src.rpm", 1011680, "MIT", "CentOS", "", "A C binding to the X11 protocol", "923a4fbb3fc66c81699e3c0354c2f8d2"},
			{intRef(), "libXau-devel", "1.0.8", "2.1.el7", "x86_64", "libXau-1.0.8-2.1.el7.src.rpm", 6519, "MIT", "CentOS", "", "Development files for libXau", "979289dbf99315769bc6e25d65ff5785"},
			{intRef(), "libSM", "1.2.2", "2.el7", "x86_64", "libSM-1.2.2-2.el7.src.rpm", 81546, "MIT", "CentOS", "", "X.Org X11 SM runtime library", "4dfa7c7907174b7354c740d44dba2f8d"},
			{intRef(), "rh-nodejs10-nodejs", "10.10.0", "2.el7", "x86_64", "rh-nodejs10-nodejs-10.10.0-2.el7.src.rpm", 28549171, "MIT and ASL 2.0 and ISC and BSD", "CentOS", "", "JavaScript runtime", "fe80cb3915b76a3a87d759e97863ba82"},
			{intRef(), "dejavu-sans-fonts", "2.33", "6.el7", "noarch", "dejavu-fonts-2.33-6.el7.src.rpm", 5395167, "Bitstream Vera and Public Domain", "CentOS", "", "Variable-width sans-serif font faces", "1e16e93e08e0bf32696b98a4ede090d0"},
			{intRef(), "libjpeg-turbo-devel", "1.2.90", "6.el7", "x86_64", "libjpeg-turbo-1.2.90-6.el7.src.rpm", 321845, "IJG", "CentOS", "", "Headers for the libjpeg-turbo library", "4b777dad00ceabbf43f675ed0e083888"},
			{intRef(), "libmount", "2.23.2", "59.el7_6.1", "x86_64", "util-linux-2.23.2-59.el7_6.1.src.rpm", 278173, "LGPLv2+", "CentOS", "", "Device mounting library", "4dcc53827852b39ebf10f2c4b75d434d"},
			{intRef(1), "mariadb-libs", "5.5.60", "1.el7_5", "x86_64", "mariadb-5.5.60-1.el7_5.src.rpm", 4615006, "GPLv2 with exceptions and LGPLv2 and BSD", "CentOS", "", "The shared libraries required for MariaDB/MySQL clients", "ea4df3c126431072efb3a6c68978f213"},
			{intRef(), "libuuid-devel", "2.23.2", "59.el7_6.1", "x86_64", "util-linux-2.23.2-59.el7_6.1.src.rpm", 17258, "BSD", "CentOS", "", "Universally unique ID library", "d24c84344c7795e2fa9de5841ac65784"},
			{intRef(), "fipscheck", "1.4.1", "6.el7", "x86_64", "fipscheck-1.4.1-6.el7.src.rpm", 38839, "BSD", "CentOS", "", "A library for integrity verification of FIPS validated modules", "a2e0953263a6285a00476819997c1caf"},
			{intRef(), "perl-HTTP-Tiny", "0.033", "3.el7", "noarch", "perl-HTTP-Tiny-0.033-3.el7.src.rpm", 97210, "GPL+ or Artistic", "CentOS", "", "Small, simple, correct HTTP/1.1 client", "6133c7c1aee2bfeee269de4592b25884"},
			{intRef(), "perl-Pod-Perldoc", "3.20", "4.el7", "noarch", "perl-Pod-Perldoc-3.20-4.el7.src.rpm", 166910, "GPL+ or Artistic", "CentOS", "", "Look up Perl documentation in Pod format", "29a8a33b24ddf4e0e95ad8d2484cc277"},
			{intRef(), "perl-Text-ParseWords", "3.29", "4.el7", "noarch", "perl-Text-ParseWords-3.29-4.el7.src.rpm", 16431, "GPL+ or Artistic", "CentOS", "", "Parse text into an array of tokens or array of arrays", "58a96328cb0e6dd435d15af542467a38"},
			{intRef(), "perl-Pod-Usage", "1.63", "3.el7", "noarch", "perl-Pod-Usage-1.63-3.el7.src.rpm", 44671, "GPL+ or Artistic", "CentOS", "", "Print a usage message from embedded POD documentation", "1db34851ba13011739022f031438e15c"},
			{intRef(), "perl-Socket", "2.010", "4.el7", "x86_64", "perl-Socket-2.010-4.el7.src.rpm", 114497, "GPL+ or Artistic", "CentOS", "", "Networking constants and support functions", "5cda40abac38b41918352e9c75a1e651"},
			{intRef(), "perl-Filter", "1.49", "3.el7", "x86_64", "perl-Filter-1.49-3.el7.src.rpm", 148475, "GPL+ or Artistic", "CentOS", "", "Perl source filters", "66b1205b2eef2a89137e530741ed3710"},
			{intRef(), "perl-Storable", "2.45", "3.el7", "x86_64", "perl-Storable-2.45-3.el7.src.rpm", 181031, "GPL+ or Artistic", "CentOS", "", "Persistence for Perl data structures", "25787b4b609f3ee50d1c7f33d29e0102"},
			{intRef(), "perl-Exporter", "5.68", "3.el7", "noarch", "perl-Exporter-5.68-3.el7.src.rpm", 56612, "GPL+ or Artistic", "CentOS", "", "Implements default import method for modules", "d6e7deb74b149e4b7aa23a998a127bb4"},
			{intRef(), "perl-Time-Local", "1.2300", "2.el7", "noarch", "perl-Time-Local-1.2300-2.el7.src.rpm", 44062, "GPL+ or Artistic", "CentOS", "", "Efficiently compute time from local and GMT time", "8dba0f2d49849dc3990cfc9e91d5ec6b"},
			{intRef(), "perl-threads-shared", "1.43", "6.el7", "x86_64", "perl-threads-shared-1.43-6.el7.src.rpm", 73972, "GPL+ or Artistic", "CentOS", "", "Perl extension for sharing data structures between threads", "cd3e1ec9dfe8b5e9121fcd53b41aa197"},
			{intRef(), "perl-File-Path", "2.09", "2.el7", "noarch", "perl-File-Path-2.09-2.el7.src.rpm", 50067, "GPL+ or Artistic", "CentOS", "", "Create or remove directory trees", "94d54ef6ce6893ae2a4400c4b0ce79b7"},
			{intRef(), "perl-Scalar-List-Utils", "1.27", "248.el7", "x86_64", "perl-Scalar-List-Utils-1.27-248.el7.src.rpm", 67994, "GPL+ or Artistic", "CentOS", "", "A selection of general-utility scalar and list subroutines", "2fb261c150fc06b1a978809c5a9c706e"},
			{intRef(), "perl-Getopt-Long", "2.40", "3.el7", "noarch", "perl-Getopt-Long-2.40-3.el7.src.rpm", 134846, "GPLv2+ or Artistic", "CentOS", "", "Extended processing of command line options", "e4fc3f2fed36897ae178ecd5d9475217"},
			{intRef(1), "perl-Error", "0.17020", "2.el7", "noarch", "perl-Error-0.17020-2.el7.src.rpm", 50015, "(GPL+ or Artistic) and MIT", "CentOS", "", "Error/exception handling in an OO-ish way", "d5f79c530c268d71811519df24ca0c90"},
			{intRef(), "perl-Data-Dumper", "2.145", "3.el7", "x86_64", "perl-Data-Dumper-2.145-3.el7.src.rpm", 99287, "GPL+ or Artistic", "CentOS", "", "Stringify perl data structures, suitable for printing and eval", "3d2fc7733dc839a2d9a67e2805479feb"},
			{intRef(), "perl-Thread-Queue", "3.02", "2.el7", "noarch", "perl-Thread-Queue-3.02-2.el7.src.rpm", 27642, "GPL+ or Artistic", "CentOS", "", "Thread-safe queues", "29d052232cb1b86133bd838739672a15"},
		}

		return toPackageInfo(pkgs)
	}

	// docker run --rm -it centos:8 bash
	// yum module install -y container-tools
	// yum groupinstall -y "Development tools"
	// yum -y install nodejs podman-docker
	// rpm -qa --queryformat "\{%{EPOCH}, \"%{NAME}\", \"%{VERSION}\", \"%{RELEASE}\", \"%{ARCH}\", \"%{SOURCERPM}\", %{SIZE}, \"%{LICENSE}\", \"%{VENDOR}\", \"\", \"%{SUMMARY}\", \"%{SIGMD5}\"\},\n" | sed "s/^{(none)/{intRef()/g" | sed -r 's/^\{([0-9]+),/{intRef(\1),/' | sed "s/(none)/0/g"
	CentOS8Modularitylabel = func() []*PackageInfo {
		pkgs := []*commonPackageInfo{
			{intRef(), "strace", "4.24", "9.el8", "x86_64", "strace-4.24-9.el8.src.rpm", 2176244, "LGPL-2.1+ and GPL-2.0+", "CentOS", "", "Tracks and displays system calls associated with a running process", "0d086902619eb98efc1c415c910d2064"},
			{intRef(), "libxcb", "1.13.1", "1.el8", "x86_64", "libxcb-1.13.1-1.el8.src.rpm", 1028472, "MIT", "CentOS", "", "A C binding to the X11 protocol", "9871866c5dc03e2f1adf8c7cc84d8a1d"},
			{intRef(), "centos-gpg-keys", "8.2", "2.2004.0.1.el8", "noarch", "centos-release-8.2-2.2004.0.1.el8.src.rpm", 3370, "GPLv2", "CentOS", "", "CentOS Linux RPM keys", "dab4971ddc24244c18495d20a726c7e5"},
			{intRef(), "byacc", "1.9.20170709", "4.el8", "x86_64", "byacc-1.9.20170709-4.el8.src.rpm", 253862, "Public Domain", "CentOS", "", "Berkeley Yacc, a parser generator", "2ac10dd9212145ba63cdddbdef2c3b37"},
			{intRef(), "libXrender", "0.9.10", "7.el8", "x86_64", "libXrender-0.9.10-7.el8.src.rpm", 52595, "MIT", "CentOS", "", "X.Org X11 libXrender runtime library", "aae5a4b6679fda4aec635060b59c5d69"},
			{intRef(), "filesystem", "3.8", "2.el8", "x86_64", "filesystem-3.8-2.el8.src.rpm", 0, "Public Domain", "CentOS", "", "The basic directory layout for a Linux system", "b3b5004f3478d8b6692e392260597d96"},
			{intRef(), "python3-gobject", "3.28.3", "1.el8", "x86_64", "pygobject3-3.28.3-1.el8.src.rpm", 16432, "LGPLv2+ and MIT", "CentOS", "", "Python 3 bindings for GObject Introspection", "e39c91da4276d4750650a122ddc348a0"},
			{intRef(), "pcre2", "10.32", "1.el8", "x86_64", "pcre2-10.32-1.el8.src.rpm", 667046, "BSD", "CentOS", "", "Perl-compatible regular expression library", "2e9a6d375bb106ef8dbb9f37fa960b17"},
			{intRef(), "containernetworking-plugins", "0.8.3", "5.module_el8.2.0+305+5e198a41", "x86_64", "containernetworking-plugins-0.8.3-5.module_el8.2.0+305+5e198a41.src.rpm", 74349750, "ASL 2.0", "CentOS", "container-tools:rhel8:8020020200507003530:0d58ad57", "CNI network plugins", "6ad893e194038ec7da6a8f003a7c98ff"},
			{intRef(), "glibc-common", "2.28", "101.el8", "x86_64", "glibc-2.28-101.el8.src.rpm", 9531204, "LGPLv2+ and LGPLv2+ with exceptions and GPLv2+ and GPLv2+ with exceptions and BSD and Inner-Net and ISC and Public Domain and GFDL", "CentOS", "", "Common binaries and locale data for glibc", "59fd06b7b2ff75b002fbdc1de4e84b8f"},
			{intRef(), "toolbox", "0.0.7", "1.module_el8.2.0+305+5e198a41", "noarch", "toolbox-0.0.7-1.module_el8.2.0+305+5e198a41.src.rpm", 18199, "ASL 2.0", "CentOS", "container-tools:rhel8:8020020200507003530:0d58ad57", "Script to launch privileged container with podman", "722b85aec942262c54d88ba600c837e1"},
			{intRef(), "zlib", "1.2.11", "13.el8", "x86_64", "zlib-1.2.11-13.el8.src.rpm", 195551, "zlib and Boost", "CentOS", "", "The compression and decompression library", "69fe14fcd3773a42ca40b318d33fa6f5"},
			{intRef(), "udica", "0.2.1", "2.module_el8.2.0+305+5e198a41", "noarch", "udica-0.2.1-2.module_el8.2.0+305+5e198a41.src.rpm", 106242, "GPLv3+", "CentOS", "container-tools:rhel8:8020020200507003530:0d58ad57", "A tool for generating SELinux security policies for containers", "d029413d13ba866b7978766fb3476360"},
			{intRef(), "libgpg-error", "1.31", "1.el8", "x86_64", "libgpg-error-1.31-1.el8.src.rpm", 902818, "LGPLv2+", "CentOS", "", "Library for error values used by GnuPG components", "e84b78135beff56ee4835fed39d7401c"},
			{intRef(), "perl-Carp", "1.42", "396.el8", "noarch", "perl-Carp-1.42-396.el8.src.rpm", 41852, "GPL+ or Artistic", "CentOS", "", "Alternative warn and die for modules", "4b23c1071bd0bd3f46aa4bc06d3cc6bb"},
			{intRef(), "libxcrypt", "4.1.1", "4.el8", "x86_64", "libxcrypt-4.1.1-4.el8.src.rpm", 194420, "LGPLv2+ and BSD and Public Domain", "CentOS", "", "Extended crypt library for DES, MD5, Blowfish and others", "5d0ae947dd72ad1f5b27c8ab209e9b12"},
			{intRef(3), "perl-Scalar-List-Utils", "1.49", "2.el8", "x86_64", "perl-Scalar-List-Utils-1.49-2.el8.src.rpm", 124632, "GPL+ or Artistic", "CentOS", "", "A selection of general-utility scalar and list subroutines", "ae9fd554ccfa042dc074779abf5f2e58"},
			{intRef(), "libstdc++", "8.3.1", "5.el8.0.2", "x86_64", "gcc-8.3.1-5.el8.0.2.src.rpm", 1855607, "GPLv3+ and GPLv3+ with exceptions and GPLv2+ with exceptions and LGPLv2+ and BSD", "CentOS", "", "GNU Standard C++ Library", "cb453d84e5e5758ed5e5cb27247ad32b"},
			{intRef(), "libgomp", "8.3.1", "5.el8.0.2", "x86_64", "gcc-8.3.1-5.el8.0.2.src.rpm", 331057, "GPLv3+ and GPLv3+ with exceptions and GPLv2+ with exceptions and LGPLv2+ and BSD", "CentOS", "", "GCC OpenMP v4.5 shared support library", "2356db2ff13d8d90ede9c428e79327b1"},
			{intRef(), "readline", "7.0", "10.el8", "x86_64", "readline-7.0-10.el8.src.rpm", 466321, "GPLv3+", "CentOS", "", "A library for editing typed command lines", "53395adc92c214ce3ce3f63c19180d19"},
			{intRef(), "m4", "1.4.18", "7.el8", "x86_64", "m4-1.4.18-7.el8.src.rpm", 379304, "GPLv3+", "CentOS", "", "The GNU macro processor", "65b81ee9fe7f8936bafa29901124e3a4"},
			{intRef(), "libacl", "2.2.53", "1.el8", "x86_64", "acl-2.2.53-1.el8.src.rpm", 59272, "LGPLv2+", "CentOS", "", "Dynamic library for access control list support", "cd43f03193723ee093e7705f1c5c6f39"},
			{intRef(1), "cups-libs", "2.2.6", "33.el8", "x86_64", "cups-2.2.6-33.el8.src.rpm", 947548, "LGPLv2 and zlib", "CentOS", "", "CUPS printing system - libraries", "c9c51dc9b8cb075f8c6e28289b8e7c39"},
			{intRef(), "libmount", "2.32.1", "22.el8", "x86_64", "util-linux-2.32.1-22.el8.src.rpm", 398154, "LGPLv2+", "CentOS", "", "Device mounting library", "bc5feadfa4bfce68811894bb73f493c4"},
			{intRef(), "libXcursor", "1.1.15", "3.el8", "x86_64", "libXcursor-1.1.15-3.el8.src.rpm", 48759, "MIT", "CentOS", "", "Cursor management library", "1f160b358d95dc012768764ef017ab26"},
			{intRef(), "libsmartcols", "2.32.1", "22.el8", "x86_64", "util-linux-2.32.1-22.el8.src.rpm", 244258, "LGPLv2+", "CentOS", "", "Formatting library for ls-like programs.", "6533915dd27fba66917d6cdca81a67fc"},
			{intRef(), "libXt", "1.1.5", "12.el8", "x86_64", "libXt-1.1.5-12.el8.src.rpm", 597617, "MIT", "CentOS", "", "X.Org X11 libXt runtime library", "3238eebe16e74f4f20a0d43492842ab2"},
			{intRef(), "p11-kit", "0.23.14", "5.el8_0", "x86_64", "p11-kit-0.23.14-5.el8_0.src.rpm", 1394732, "BSD", "CentOS", "", "Library for loading and sharing PKCS#11 modules", "77b1d075374eae3c0fa1a2ffa004120a"},
			{intRef(0), "perl-Errno", "1.28", "416.el8", "x86_64", "perl-5.26.3-416.el8.src.rpm", 9495, "GPL+ or Artistic", "CentOS", "", "System errno constants", "f0355f3f5fcd20f240b9a59e0f0dff88"},
			{intRef(), "libunistring", "0.9.9", "3.el8", "x86_64", "libunistring-0.9.9-3.el8.src.rpm", 1855932, "GPLv2+ or LGPLv3+", "CentOS", "", "GNU Unicode string library", "7769c82a0a6aa692f4778f71f6fdec0c"},
			{intRef(), "perl-File-Path", "2.15", "2.el8", "noarch", "perl-File-Path-2.15-2.el8.src.rpm", 64920, "GPL+ or Artistic", "CentOS", "", "Create or remove directory trees", "26125ecafbd51e07539b96c2699308bd"},
			{intRef(1), "gdbm-libs", "1.18", "1.el8", "x86_64", "gdbm-1.18-1.el8.src.rpm", 135248, "GPLv3+", "CentOS", "", "Libraries files for gdbm", "8efb9ab97fa0223d11d0857febae479d"},
			{intRef(1), "perl-threads", "2.21", "2.el8", "x86_64", "perl-threads-2.21-2.el8.src.rpm", 109155, "GPL+ or Artistic", "CentOS", "", "Perl interpreter-based threads", "1a94ed19723e65a29945e59d8dfb5cf4"},
			{intRef(), "lzo", "2.08", "14.el8", "x86_64", "lzo-2.08-14.el8.src.rpm", 198757, "GPLv2+", "CentOS", "", "Data compression library with very fast (de)compression", "2e8c7912284d0c40f98a93649366097c"},
			{intRef(), "perl-IO-Socket-IP", "0.39", "5.el8", "noarch", "perl-IO-Socket-IP-0.39-5.el8.src.rpm", 99525, "GPL+ or Artistic", "CentOS", "", "Drop-in replacement for IO::Socket::INET supporting both IPv4 and IPv6", "9db2eaec9a186e8a7b92990683a12044"},
			{intRef(1), "dbus-libs", "1.12.8", "10.el8_2", "x86_64", "dbus-1.12.8-10.el8_2.src.rpm", 380576, "(GPLv2+ or AFL) and GPLv2+", "CentOS", "", "Libraries for accessing D-BUS", "e895b9b8d703c114cb15c7564df6de96"},
			{intRef(), "xml-common", "0.6.3", "50.el8", "noarch", "sgml-common-0.6.3-50.el8.src.rpm", 80250, "GPL+", "CentOS", "", "Common XML catalog and DTD files", "23245dba9afaea298d6823327bd91846"},
			{intRef(), "procps-ng", "3.3.15", "1.el8", "x86_64", "procps-ng-3.3.15-1.el8.src.rpm", 938380, "GPL+ and GPLv2 and GPLv2+ and GPLv3+ and LGPLv2+", "CentOS", "", "System and process monitoring utilities", "030ea9850814d2c60c5af0ddb2f69c72"},
			{intRef(), "efivar-libs", "36", "1.el8", "x86_64", "efivar-36-1.el8.src.rpm", 250520, "LGPLv2.1", "CentOS", "", "Library to manage UEFI variables", "6ab0b388f81dbc1e254733bb3c0c1b56"},
			{intRef(1), "gdbm", "1.18", "1.el8", "x86_64", "gdbm-1.18-1.el8.src.rpm", 399977, "GPLv3+", "CentOS", "", "A GNU set of database routines which use extensible hashing", "69ce5734e1fd54cdd298b4667d76616f"},
			{intRef(), "javapackages-filesystem", "5.3.0", "1.module_el8.0.0+11+5b8c10bd", "noarch", "javapackages-tools-5.3.0-1.module_el8.0.0+11+5b8c10bd.src.rpm", 1935, "BSD", "CentOS", "javapackages-runtime:201801:8000020190530193251:278695df", "Java packages filesystem layout", "d3dc80ecbcf4fa910c46092359774f53"},
			{intRef(), "libfdisk", "2.32.1", "22.el8", "x86_64", "util-linux-2.32.1-22.el8.src.rpm", 438722, "LGPLv2+", "CentOS", "", "Partitioning library for fdisk-like programs.", "886e9927f7ebfcc4c907a870f40d9c09"},
			{intRef(1), "mokutil", "0.3.0", "9.el8", "x86_64", "mokutil-0.3.0-9.el8.src.rpm", 89577, "GPLv3+", "CentOS", "", "Tool to manage UEFI Secure Boot MoK Keys", "ee486a44f8f3c8cc57795039bbd3516d"},
			{intRef(), "snappy", "1.1.7", "5.el8", "x86_64", "snappy-1.1.7-5.el8.src.rpm", 58789, "BSD", "CentOS", "", "Fast compression and decompression library", "459dcaf6d9799509b291148959e3a2ac"},
			{intRef(1), "perl-Error", "0.17025", "2.el8", "noarch", "perl-Error-0.17025-2.el8.src.rpm", 71565, "(GPL+ or Artistic) and MIT", "CentOS", "", "Error/exception handling in an OO-ish way", "235ea1ae7ab96868efe5090a4eb7bf1b"},
			{intRef(), "libksba", "1.3.5", "7.el8", "x86_64", "libksba-1.3.5-7.el8.src.rpm", 342935, "(LGPLv3+ or GPLv2+) and GPLv3+", "CentOS", "", "CMS and X.509 library", "74d2b871d0f179895daee53422a52b94"},
			{intRef(1), "perl-Pod-Escapes", "1.07", "395.el8", "noarch", "perl-Pod-Escapes-1.07-395.el8.src.rpm", 25763, "GPL+ or Artistic", "CentOS", "", "Resolve POD escape sequences", "06bdeb217473ea5c468916d046240269"},
			{intRef(), "libmnl", "1.0.4", "6.el8", "x86_64", "libmnl-1.0.4-6.el8.src.rpm", 53687, "LGPLv2+", "CentOS", "", "A minimalistic Netlink library", "003fa2a9079721368b658c63d8832e20"},
			{intRef(), "boost-timer", "1.66.0", "7.el8", "x86_64", "boost-1.66.0-7.el8.src.rpm", 26226, "Boost and MIT and Python", "CentOS", "", "Run-time component of boost timer library", "2d577eb03f153513ffef05c01ee67481"},
			{intRef(), "libseccomp", "2.4.1", "1.el8", "x86_64", "libseccomp-2.4.1-1.el8.src.rpm", 402960, "LGPLv2", "CentOS", "", "Enhanced seccomp library", "6237a23621d5be842f2db331714e7215"},
			{intRef(), "cpp", "8.3.1", "5.el8.0.2", "x86_64", "gcc-8.3.1-5.el8.0.2.src.rpm", 29644557, "GPLv3+ and GPLv3+ with exceptions and GPLv2+ with exceptions and LGPLv2+ and BSD", "CentOS", "", "The C Preprocessor", "0578fcab8e65239fc0b5000723873379"},
			{intRef(), "libnsl2", "1.2.0", "2.20180605git4a062cf.el8", "x86_64", "libnsl2-1.2.0-2.20180605git4a062cf.el8.src.rpm", 147122, "BSD and LGPLv2+", "CentOS", "", "Public client interface library for NIS(YP) and NIS+", "7be9bc3ac6e7e4d56de085c410a8e20d"},
			{intRef(), "jasper-libs", "2.0.14", "4.el8", "x86_64", "jasper-2.0.14-4.el8.src.rpm", 389890, "JasPer", "CentOS", "", "Runtime libraries for jasper", "af4ce44daf52307ad50a9a5f767c1f75"},
			{intRef(), "crypto-policies", "20191128", "2.git23e1bf1.el8", "noarch", "crypto-policies-20191128-2.git23e1bf1.el8.src.rpm", 190228, "LGPLv2+", "CentOS", "", "System-wide crypto policies", "a6cdcd759a11870401ea0021c2a1496c"},
			{intRef(), "nss-sysinit", "3.53.1", "11.el8_2", "x86_64", "nss-3.53.1-11.el8_2.src.rpm", 14402, "MPLv2.0", "CentOS", "", "System NSS Initialization", "094e2ee6a5c5ce63f17098268e8dc066"},
			{intRef(), "libdb", "5.3.28", "37.el8", "x86_64", "libdb-5.3.28-37.el8.src.rpm", 2515048, "BSD and LGPLv2 and Sleepycat", "CentOS", "", "The Berkeley DB database library for C", "293f618e23dd98b899b17e526e052883"},
			{intRef(), "fipscheck-lib", "1.5.0", "4.el8", "x86_64", "fipscheck-1.5.0-4.el8.src.rpm", 12433, "BSD", "CentOS", "", "Library files for fipscheck", "9c891d4da9388c333268d7f2e3a9bcb1"},
			{intRef(), "gnutls", "3.6.8", "11.el8_2", "x86_64", "gnutls-3.6.8-11.el8_2.src.rpm", 2687433, "GPLv3+ and LGPLv2+", "CentOS", "", "A TLS protocol implementation", "9d8812427df36d252b285cdc3ea1bfc9"},
			{intRef(), "docbook-dtds", "1.0", "69.el8", "noarch", "docbook-dtds-1.0-69.el8.src.rpm", 8665150, "Copyright only", "CentOS", "", "SGML and XML document type definitions for DocBook", "dc4fb72600ded71ab95ce49f0d2fa312"},
			{intRef(), "ima-evm-utils", "1.1", "5.el8", "x86_64", "ima-evm-utils-1.1-5.el8.src.rpm", 123538, "GPLv2", "CentOS", "", "IMA/EVM support utilities", "c5dccadfe8fd2c247d6785651fb218c5"},
			{intRef(), "dnf-plugins-core", "4.0.12", "4.el8_2", "noarch", "dnf-plugins-core-4.0.12-4.el8_2.src.rpm", 16900, "GPLv2+", "CentOS", "", "Core Plugins for DNF", "bfbf1db7325a3274ab6244f3ba83e824"},
			{intRef(), "cyrus-sasl-lib", "2.1.27", "1.el8", "x86_64", "cyrus-sasl-2.1.27-1.el8.src.rpm", 734978, "BSD with advertising", "CentOS", "", "Shared libraries needed by applications which use Cyrus SASL", "7a3b1a78bd4838e0f6324f5bba9c148e"},
			{intRef(), "perl-Term-Cap", "1.17", "395.el8", "noarch", "perl-Term-Cap-1.17-395.el8.src.rpm", 29850, "GPL+ or Artistic", "CentOS", "", "Perl termcap interface", "a36d892cdd3e4c40b70ebf3ebd307ece"},
			{intRef(32), "bind-export-libs", "9.11.13", "5.el8_2", "x86_64", "bind-9.11.13-5.el8_2.src.rpm", 3067121, "MPLv2.0", "CentOS", "", "ISC libs for DHCP application", "47a7f0c3b04ea913f0366b75ba4a17ca"},
			{intRef(), "libsecret", "0.18.6", "1.el8", "x86_64", "libsecret-0.18.6-1.el8.src.rpm", 520362, "LGPLv2+", "CentOS", "", "Library for storing and retrieving passwords and other secrets", "debd45dfee226d419b9cb0f66242aa5b"},
			{intRef(), "libsolv", "0.7.7", "1.el8", "x86_64", "libsolv-0.7.7-1.el8.src.rpm", 789176, "BSD", "CentOS", "", "Package dependency solver", "b742b6c750a836cf3ac712046b1479fa"},
			{intRef(), "libicu", "60.3", "2.el8_1", "x86_64", "icu-60.3-2.el8_1.src.rpm", 33716339, "MIT and UCD and Public Domain", "CentOS", "", "International Components for Unicode - libraries", "a63aabcbc386c22aefd79cc72114aab0"},
			{intRef(), "gnupg2", "2.2.9", "1.el8", "x86_64", "gnupg2-2.2.9-1.el8.src.rpm", 9632671, "GPLv3+", "CentOS", "", "Utility for secure communication and data storage", "0b4519bf8b18306203885e2108c81aca"},
			{intRef(), "git-core", "2.18.4", "2.el8_2", "x86_64", "git-2.18.4-2.el8_2.src.rpm", 24912186, "GPLv2", "CentOS", "", "Core package of git with minimal functionality", "ba87010c46d7b15221de570dca65076d"},
			{intRef(), "python3-libdnf", "0.39.1", "6.el8_2", "x86_64", "libdnf-0.39.1-6.el8_2.src.rpm", 3666814, "LGPLv2+", "CentOS", "", "Python 3 bindings for the libdnf library.", "38a109680c9ff4ce9d1c8f3597c346c4"},
			{intRef(), "libxcrypt-devel", "4.1.1", "4.el8", "x86_64", "libxcrypt-4.1.1-4.el8.src.rpm", 24771, "LGPLv2+ and BSD and Public Domain", "CentOS", "", "Development files for libxcrypt", "aec174890471f284aa6e03020ff76443"},
			{intRef(), "dnf-data", "4.2.17", "7.el8_2", "noarch", "dnf-4.2.17-7.el8_2.src.rpm", 36254, "GPLv2+ and GPLv2 and GPL", "CentOS", "", "Common data and configuration files for DNF", "9e2017577948cc0fb98c7907611ee902"},
			{intRef(1), "perl-Pod-Simple", "3.35", "395.el8", "noarch", "perl-Pod-Simple-3.35-395.el8.src.rpm", 543719, "GPL+ or Artistic", "CentOS", "", "Framework for parsing POD documentation", "4634886be1066e76224760899bd706d5"},
			{intRef(8), "device-mapper", "1.02.169", "3.el8", "x86_64", "lvm2-2.03.08-3.el8.src.rpm", 355102, "GPLv2", "CentOS", "", "Device mapper utility", "4e2e93bf6d5fc21511b94184b03be6b9"},
			{intRef(), "perl-Pod-Perldoc", "3.28", "396.el8", "noarch", "perl-Pod-Perldoc-3.28-396.el8.src.rpm", 169228, "GPL+ or Artistic", "CentOS", "", "Look up Perl documentation in Pod format", "fc8b1db50f574413212744f6f888c3d2"},
			{intRef(), "elfutils-libs", "0.178", "7.el8", "x86_64", "elfutils-0.178-7.el8.src.rpm", 717567, "GPLv2+ or LGPLv3+", "CentOS", "", "Libraries to handle compiled objects", "11d18e7fceb1aacb16e96dbebdf78557"},
			{intRef(), "perl-URI", "1.73", "3.el8", "noarch", "perl-URI-1.73-3.el8.src.rpm", 216452, "GPL+ or Artistic", "CentOS", "", "A Perl module implementing URI parsing and manipulation", "b66d85e3853e61853c8334dd507a77a0"},
			{intRef(), "iputils", "20180629", "2.el8", "x86_64", "iputils-20180629-2.el8.src.rpm", 351665, "BSD and GPLv2+", "CentOS", "", "Network monitoring tools including ping", "ca9a89c22af93423852ecbd08d3abf59"},
			{intRef(), "git", "2.18.4", "2.el8_2", "x86_64", "git-2.18.4-2.el8_2.src.rpm", 399381, "GPLv2", "CentOS", "", "Fast Version Control System", "c8a5db38b49ac1f39df666f5fbf3c6e8"},
			{intRef(), "systemd-udev", "239", "31.el8_2.2", "x86_64", "systemd-239-31.el8_2.2.src.rpm", 7940076, "LGPLv2+", "CentOS", "", "Rule-based device node and kernel event manager", "36012002d579569d43dfa958dbbaf95c"},
			{intRef(), "zstd", "1.4.2", "2.el8", "x86_64", "zstd-1.4.2-2.el8.src.rpm", 1550971, "BSD and GPLv2", "CentOS", "", "Zstd compression library", "a2f153ccbdd781670799448ffb0829bf"},
			{intRef(), "rpm-build-libs", "4.14.2", "37.el8", "x86_64", "rpm-4.14.2-37.el8.src.rpm", 215992, "GPLv2+ and LGPLv2+ with exceptions", "CentOS", "", "Libraries for building and signing RPM packages", "f60199bf9d32dec92807da5945d7eff8"},
			{intRef(), "rust-srpm-macros", "5", "2.el8", "noarch", "rust-srpm-macros-5-2.el8.src.rpm", 1131, "MIT", "CentOS", "", "RPM macros for building Rust source packages", "d96b3587cf204ff53b4b377f1f61bcc1"},
			{intRef(), "yum", "4.2.17", "7.el8_2", "noarch", "dnf-4.2.17-7.el8_2.src.rpm", 70885, "GPLv2+ and GPLv2 and GPL", "CentOS", "", "Package manager", "315b9809e784fb291a584f24244e641e"},
			{intRef(), "perl-srpm-macros", "1", "25.el8", "noarch", "perl-srpm-macros-1-25.el8.src.rpm", 794, "GPLv3+", "CentOS", "", "RPM macros for building Perl source package from source repository", "0293bcef8257b954e640b14fa1fbfac0"},
			{intRef(2), "vim-minimal", "8.0.1763", "13.el8", "x86_64", "vim-8.0.1763-13.el8.src.rpm", 1420484, "Vim and MIT", "CentOS", "", "A minimal version of the VIM editor", "ed73caeb8c3746e48ced83354900908b"},
			{intRef(), "lua", "5.3.4", "11.el8", "x86_64", "lua-5.3.4-11.el8.src.rpm", 638964, "MIT", "CentOS", "", "Powerful light-weight programming language", "3a2519a18aee2cdf287f971f9d5b9938"},
			{intRef(), "rootfiles", "8.1", "22.el8", "noarch", "rootfiles-8.1-22.el8.src.rpm", 599, "Public Domain", "CentOS", "", "The basic required files for the root user's directory", "8396ccef88edaffce4a8d5eb1ac4aa66"},
			{intRef(), "libstdc++-devel", "8.3.1", "5.el8.0.2", "x86_64", "gcc-8.3.1-5.el8.0.2.src.rpm", 11807826, "GPLv3+ and GPLv3+ with exceptions and GPLv2+ with exceptions and LGPLv2+ and BSD", "CentOS", "", "Header files and libraries for C++ development", "7e8a8c7fbeb75afd0f9c05f52ee1f803"},
			{intRef(1), "nodejs", "10.21.0", "3.module_el8.2.0+391+8da3adc6", "x86_64", "nodejs-10.21.0-3.module_el8.2.0+391+8da3adc6.src.rpm", 31483781, "MIT and ASL 2.0 and ISC and BSD", "CentOS", "nodejs:10:8020020200707141642:6a468ee4", "JavaScript runtime", "bac7919c2369f944f9da510bbd01370b"},
			{intRef(), "libipt", "1.6.1", "8.el8", "x86_64", "libipt-1.6.1-8.el8.src.rpm", 108543, "BSD", "CentOS", "", "Intel Processor Trace Decoder Library", "68edc160e21686087d8486cdf060073e"},
			{intRef(), "libnl3", "3.5.0", "1.el8", "x86_64", "libnl3-3.5.0-1.el8.src.rpm", 994296, "LGPLv2", "CentOS", "", "Convenience library for kernel netlink sockets", "09d3b77ea984c402b4514b6ec1d2a3f2"},
			{intRef(), "xorg-x11-fonts-ISO8859-1-100dpi", "7.5", "19.el8", "noarch", "xorg-x11-fonts-7.5-19.el8.src.rpm", 1070824, "MIT and Lucida and Public Domain", "CentOS", "", "A set of 100dpi ISO-8859-1 fonts for X", "db0cbd7441bae6fe18d87986701c4b0a"},
			{intRef(), "slirp4netns", "0.4.2", "3.git21fdece.module_el8.2.0+305+5e198a41", "x86_64", "slirp4netns-0.4.2-3.git21fdece.module_el8.2.0+305+5e198a41.src.rpm", 173387, "GPLv2", "CentOS", "container-tools:rhel8:8020020200507003530:0d58ad57", "slirp for network namespaces", "d1512dc9ad64ed638dfe18f9005d1d64"},
			{intRef(), "libatomic_ops", "7.6.2", "3.el8", "x86_64", "libatomic_ops-7.6.2-3.el8.src.rpm", 76822, "GPLv2 and MIT", "CentOS", "", "Atomic memory update operations", "6f1ef12f2494056f5cbafb60b23d483c"},
			{intRef(), "gdk-pixbuf2", "2.36.12", "5.el8", "x86_64", "gdk-pixbuf2-2.36.12-5.el8.src.rpm", 2657576, "LGPLv2+", "CentOS", "", "An image loading library", "3006826ddf4e875c1f21a9da6579bd20"},
			{intRef(), "libXxf86vm", "1.1.4", "9.el8", "x86_64", "libXxf86vm-1.1.4-9.el8.src.rpm", 26424, "MIT", "CentOS", "", "X.Org X11 libXxf86vm runtime library", "2e674cae50d672e67f5afc5b46afe913"},
			{intRef(), "libnfnetlink", "1.0.1", "13.el8", "x86_64", "libnfnetlink-1.0.1-13.el8.src.rpm", 52395, "GPLv2+", "CentOS", "", "Netfilter netlink userspace library", "bb5e0b1f9a6925de1920ae5ff8c10497"},
			{intRef(), "urw-base35-c059-fonts", "20170801", "10.el8", "noarch", "urw-base35-fonts-20170801-10.el8.src.rpm", 1460348, "AGPLv3", "CentOS", "", "C059 font family [part of Level 2 Core Font Set]", "e6f22d6ea059da79d67e5a1905949c53"},
			{intRef(), "abattis-cantarell-fonts", "0.0.25", "4.el8", "noarch", "abattis-cantarell-fonts-0.0.25-4.el8.src.rpm", 302227, "OFL", "CentOS", "", "Cantarell, a Humanist sans-serif font family", "4ccb64635fd6cf43ffa27d993682e961"},
			{intRef(), "urw-base35-nimbus-roman-fonts", "20170801", "10.el8", "noarch", "urw-base35-fonts-20170801-10.el8.src.rpm", 1429675, "AGPLv3", "CentOS", "", "Nimbus Roman font family [part of Level 2 Core Font Set]", "7c8915d94c2c51421ab11e2bd324f4a8"},
			{intRef(), "audit", "3.0", "0.17.20191104git1c2f876.el8", "x86_64", "audit-3.0-0.17.20191104git1c2f876.el8.src.rpm", 671305, "GPLv2+", "CentOS", "", "User space tools for kernel auditing", "0113e7aa0ecd79593c59f5c0767930a8"},
			{intRef(), "urw-base35-z003-fonts", "20170801", "10.el8", "noarch", "urw-base35-fonts-20170801-10.el8.src.rpm", 400088, "AGPLv3", "CentOS", "", "Z003 font family [part of Level 2 Core Font Set]", "0b8313e3b9335f0786c2adf4c8eedb6e"},
			{intRef(), "libteam", "1.29", "1.el8_2.2", "x86_64", "libteam-1.29-1.el8_2.2.src.rpm", 106092, "LGPLv2+", "CentOS", "", "Library for controlling team network device", "a7c28fbd534a1f3d606810e26a813b6c"},
			{intRef(), "lcms2", "2.9", "2.el8", "x86_64", "lcms2-2.9-2.el8.src.rpm", 399313, "MIT", "CentOS", "", "Color Management Engine", "23d5f2bf5897a440de253fe694e0f57f"},
			{intRef(), "python3-slip", "0.6.4", "11.el8", "noarch", "python-slip-0.6.4-11.el8.src.rpm", 61396, "GPLv2+", "CentOS", "", "Convenience, extension and workaround code for Python 3.x", "64a739ce401335acb5f47be73cc285cb"},
			{intRef(), "gdk-pixbuf2-modules", "2.36.12", "5.el8", "x86_64", "gdk-pixbuf2-2.36.12-5.el8.src.rpm", 315856, "LGPLv2+", "CentOS", "", "Additional image modules for gdk-pixbuf", "e5d50b0e3a219515e3750449e63cb1e2"},
			{intRef(), "python3-decorator", "4.2.1", "2.el8", "noarch", "python-decorator-4.2.1-2.el8.src.rpm", 47871, "BSD", "CentOS", "", "Module to simplify usage of decorators in python3", "c9b5be20911b95f447dccfcde8ae924c"},
			{intRef(), "annobin", "8.90", "1.el8.0.1", "x86_64", "annobin-8.90-1.el8.0.1.src.rpm", 432502, "GPLv3+", "CentOS", "", "Binary annotation plugin for GCC", "fe2dce9fda6c5d2717dc0b7cff64dcfb"},
			{intRef(), "mozjs60", "60.9.0", "4.el8", "x86_64", "mozjs60-60.9.0-4.el8.src.rpm", 23728688, "MPLv2.0 and MPLv1.1 and BSD and GPLv2+ and GPLv3+ and LGPLv2+ and AFL and ASL 2.0", "CentOS", "", "SpiderMonkey JavaScript library", "4cf8dc678ff7a070c0beed5c2bd1eb48"},
			{intRef(), "gtk-update-icon-cache", "3.22.30", "5.el8", "x86_64", "gtk3-3.22.30-5.el8.src.rpm", 60214, "LGPLv2+", "CentOS", "", "Icon theme caching utility", "b2303d1366f20f7657e1aa9f1fe12536"},
			{intRef(), "libstemmer", "0", "10.585svn.el8", "x86_64", "libstemmer-0-10.585svn.el8.src.rpm", 361957, "BSD", "CentOS", "", "C stemming algorithm library", "f91149f5eb21116fdc4789fc495b0fff"},
			{intRef(), "go-srpm-macros", "2", "16.el8", "noarch", "go-srpm-macros-2-16.el8.src.rpm", 18532, "GPLv3+", "CentOS", "", "RPM macros for building Golang packages for various architectures", "53657f791c19de2ab00fbc5b47f4db14"},
			{intRef(), "libmodman", "2.0.1", "17.el8", "x86_64", "libmodman-2.0.1-17.el8.src.rpm", 69479, "LGPLv2+", "CentOS", "", "A simple library for managing C++ modules (plug-ins)", "12f69442d1ad2b256dc4942385f4b9d2"},
			{intRef(), "gtk2", "2.24.32", "4.el8", "x86_64", "gtk2-2.24.32-4.el8.src.rpm", 13811558, "LGPLv2+", "CentOS", "", "GTK+ graphical user interface library", "5bd06487b19605b2597b36abc2b5e409"},
			{intRef(), "libsoup", "2.62.3", "1.el8", "x86_64", "libsoup-2.62.3-1.el8.src.rpm", 1594291, "LGPLv2", "CentOS", "", "Soup, an HTTP library implementation", "37fbd6acde3d2bdbe7a56817a4d204dd"},
			{intRef(), "efi-srpm-macros", "3", "2.el8", "noarch", "efi-rpm-macros-3-2.el8.src.rpm", 39223, "GPLv3+", "CentOS", "", "Common SRPM Macros for building EFI-related packages", "4d80b86588aa4a74fe6eb44d16bcaf95"},
			{intRef(), "teamd", "1.29", "1.el8_2.2", "x86_64", "libteam-1.29-1.el8_2.2.src.rpm", 278854, "LGPLv2+", "CentOS", "", "Team network device control daemon", "75a737805827b98dcfda249a412b8582"},
			{intRef(), "source-highlight", "3.1.8", "16.el8", "x86_64", "source-highlight-3.1.8-16.el8.src.rpm", 3516067, "GPLv3+", "CentOS", "", "Produces a document with syntax highlighting", "2b50dd617a2d2acc5392b2d82efa8313"},
			{intRef(), "diffutils", "3.6", "6.el8", "x86_64", "diffutils-3.6-6.el8.src.rpm", 1369962, "GPLv3+", "CentOS", "", "A GNU collection of diff utilities", "f1d7b5001daf0d2122265c67b7fa2b7f"},
			{intRef(), "systemtap-runtime", "4.2", "6.el8", "x86_64", "systemtap-4.2-6.el8.src.rpm", 1393756, "GPLv2+", "CentOS", "", "Programmable system-wide instrumentation system - runtime", "41c00680766a1fbe16c50151e309ea4e"},
			{intRef(), "selinux-policy-targeted", "3.14.3", "41.el8_2.8", "noarch", "selinux-policy-3.14.3-41.el8_2.8.src.rpm", 52586852, "GPLv2+", "CentOS", "", "SELinux targeted base policy", "8cdf37012815e8631b0c0017bac8fd36"},
			{intRef(), "graphviz", "2.40.1", "40.el8", "x86_64", "graphviz-2.40.1-40.el8.src.rpm", 7807623, "EPL-1.0", "CentOS", "", "Graph Visualization Tools", "5b33ec1eeca8e3be76dd6b699dcc4b09"},
			{intRef(), "checkpolicy", "2.9", "1.el8", "x86_64", "checkpolicy-2.9-1.el8.src.rpm", 1768984, "GPLv2", "CentOS", "", "SELinux policy compiler", "8e3e7a230d9e8e9ff28589d183dde3b4"},
			{intRef(), "gdb", "8.2", "11.el8", "x86_64", "gdb-8.2-11.el8.src.rpm", 363845, "GPLv3+ and GPLv3+ with exceptions and GPLv2+ and GPLv2+ with exceptions and GPL+ and LGPLv2+ and LGPLv3+ and BSD and Public Domain and GFDL", "CentOS", "", "A stub package for GNU source-level debugger", "3d9b9041ca72902b62ac16da16eb1218"},
			{intRef(1), "NetworkManager-libnm", "1.22.8", "5.el8_2", "x86_64", "NetworkManager-1.22.8-5.el8_2.src.rpm", 9207215, "LGPLv2+", "CentOS", "", "Libraries for adding NetworkManager support to applications (new API).", "0b863ffd1eeafe78345c429a1787cb4e"},
			{intRef(), "intltool", "0.51.0", "11.el8", "noarch", "intltool-0.51.0-11.el8.src.rpm", 173158, "GPLv2 with exceptions", "CentOS", "", "Utility for internationalizing various kinds of data files", "3494f518cfb0a0183dd4cdfe1ff7d9d1"},
			{intRef(), "python3-psutil", "5.4.3", "10.el8", "x86_64", "python-psutil-5.4.3-10.el8.src.rpm", 2041226, "BSD", "CentOS", "", "A process and system utilities module for Python", "770656179e036634f65365cfce9a3759"},
			{intRef(), "patchutils", "0.3.4", "10.el8", "x86_64", "patchutils-0.3.4-10.el8.src.rpm", 285091, "GPLv2+", "CentOS", "", "A collection of programs for manipulating patch files", "083a92406e86ee42857ebca75295195f"},
			{intRef(), "criu", "3.12", "9.module_el8.2.0+305+5e198a41", "x86_64", "criu-3.12-9.module_el8.2.0+305+5e198a41.src.rpm", 1332765, "GPLv2", "CentOS", "container-tools:rhel8:8020020200507003530:0d58ad57", "Tool for Checkpoint/Restore in User-space", "400403b888cd7a13bba76f6595c5189a"},
			{intRef(), "libgcc", "8.3.1", "5.el8.0.2", "x86_64", "gcc-8.3.1-5.el8.0.2.src.rpm", 190232, "GPLv3+ and GPLv3+ with exceptions and GPLv2+ with exceptions and LGPLv2+ and BSD", "CentOS", "", "GCC version 8 shared support library", "6d2466b49e2e159a9ffbc965d87539b5"},
			{intRef(), "libXau", "1.0.8", "13.el8", "x86_64", "libXau-1.0.8-13.el8.src.rpm", 59505, "MIT", "CentOS", "", "Sample Authorization Protocol for X", "ede016aae66ea3f1ab621753c8d55b43"},
			{intRef(), "python3-pip-wheel", "9.0.3", "16.el8", "noarch", "python-pip-9.0.3-16.el8.src.rpm", 1255748, "MIT and Python and ASL 2.0 and BSD and ISC and LGPLv2 and MPLv2.0 and (ASL 2.0 or BSD)", "CentOS", "", "The pip wheel", "c7af612313ed1ff4b0831c8e2a811ef4"},
			{intRef(), "libX11-common", "1.6.8", "3.el8", "noarch", "libX11-1.6.8-3.el8.src.rpm", 1339608, "MIT", "CentOS", "", "Common data for libX11", "200569d0076f1d935efdc55ea11ae82b"},
			{intRef(), "centos-release", "8.2", "2.2004.0.1.el8", "x86_64", "centos-release-8.2-2.2004.0.1.el8.src.rpm", 25430, "GPLv2", "CentOS", "", "CentOS Linux release file", "e36bc671f0e117ac535f4c1b3256c926"},
			{intRef(), "libXext", "1.3.3", "9.el8", "x86_64", "libXext-1.3.3-9.el8.src.rpm", 97990, "MIT", "CentOS", "", "X.Org X11 libXext runtime library", "c036a82276c1005192b59328e86471cc"},
			{intRef(), "setup", "2.12.2", "5.el8", "noarch", "setup-2.12.2-5.el8.src.rpm", 724831, "Public Domain", "CentOS", "", "A set of system configuration and setup files", "d55f10b24cc01ff684bb2380bcc453b8"},
			{intRef(), "cairo", "1.15.12", "3.el8", "x86_64", "cairo-1.15.12-3.el8.src.rpm", 1881396, "LGPLv2 or MPLv1.1", "CentOS", "", "A 2D graphics library", "1ecc3b8c339aef6694167d1bfc9ff802"},
			{intRef(), "basesystem", "11", "5.el8", "noarch", "basesystem-11-5.el8.src.rpm", 0, "Public Domain", "CentOS", "", "The skeleton package which defines a simple centos system", "f988bd6b76df626c40d6e051ee7629c9"},
			{intRef(), "python3-cairo", "1.16.3", "6.el8", "x86_64", "pycairo-1.16.3-6.el8.src.rpm", 324125, "MPLv1.1 or LGPLv2", "CentOS", "", "Python 3 bindings for the cairo library", "9ba0ca0e7cac019c8a2447d9afaac896"},
			{intRef(), "ncurses-base", "6.1", "7.20180224.el8", "noarch", "ncurses-6.1-7.20180224.el8.src.rpm", 290089, "MIT", "CentOS", "", "Descriptions of common terminals", "b159cfc17482713cecfbfd8e031a91f2"},
			{intRef(), "setroubleshoot-plugins", "3.3.11", "2.el8", "noarch", "setroubleshoot-plugins-3.3.11-2.el8.src.rpm", 2621479, "GPLv2+", "CentOS", "", "Analysis plugins for use with setroubleshoot", "32c2b288a9104e797ff5282ff580f958"},
			{intRef(), "libselinux", "2.9", "3.el8", "x86_64", "libselinux-2.9-3.el8.src.rpm", 305912, "Public Domain", "CentOS", "", "SELinux library and simple utilities", "6ab87f69ce519580828da70906fc05a2"},
			{intRef(), "cockpit-system", "211.3", "1.el8", "noarch", "cockpit-211.3-1.el8.src.rpm", 2428779, "LGPLv2+", "CentOS", "", "Cockpit admin interface package for configuring and troubleshooting a system", "931511098764abee25a2c3df3d5757ca"},
			{intRef(), "glibc-minimal-langpack", "2.28", "101.el8", "x86_64", "glibc-2.28-101.el8.src.rpm", 0, "LGPLv2+ and LGPLv2+ with exceptions and GPLv2+ and GPLv2+ with exceptions and BSD and Inner-Net and ISC and Public Domain and GFDL", "CentOS", "", "Minimal language packs for glibc.", "83a4dc4baf1266a1818a32d3115c3697"},
			{intRef(2), "conmon", "2.0.6", "1.module_el8.2.0+305+5e198a41", "x86_64", "conmon-2.0.6-1.module_el8.2.0+305+5e198a41.src.rpm", 85817, "ASL 2.0", "CentOS", "container-tools:rhel8:8020020200507003530:0d58ad57", "OCI container runtime monitor", "d4c9339cc1806348f06d201bcc0a9d6b"},
			{intRef(), "glibc", "2.28", "101.el8", "x86_64", "glibc-2.28-101.el8.src.rpm", 17885631, "LGPLv2+ and LGPLv2+ with exceptions and GPLv2+ and GPLv2+ with exceptions and BSD and Inner-Net and ISC and Public Domain and GFDL", "CentOS", "", "The GNU libc libraries", "5cb33fd16d51470238a7a4ec125016ba"},
			{intRef(), "cockpit-podman", "12", "1.module_el8.2.0+305+5e198a41", "noarch", "cockpit-podman-12-1.module_el8.2.0+305+5e198a41.src.rpm", 4244724, "LGPLv2+", "CentOS", "container-tools:rhel8:8020020200507003530:0d58ad57", "Cockpit component for Podman containers", "aa3d1bfdc3d19408d9311869915308c9"},
			{intRef(), "libsepol", "2.9", "1.el8", "x86_64", "libsepol-2.9-1.el8.src.rpm", 996264, "LGPLv2+", "CentOS", "", "SELinux binary policy manipulation library", "38605757e0d48009beca6e82c6bdc8d5"},
			{intRef(), "buildah", "1.11.6", "7.module_el8.2.0+305+5e198a41", "x86_64", "buildah-1.11.6-7.module_el8.2.0+305+5e198a41.src.rpm", 37059509, "ASL 2.0", "CentOS", "container-tools:rhel8:8020020200507003530:0d58ad57", "A command line tool used for creating OCI Images", "bcd13dc985fcb5ac5cc45fc5338c281a"},
			{intRef(), "xz-libs", "5.2.4", "3.el8", "x86_64", "xz-5.2.4-3.el8.src.rpm", 194799, "Public Domain", "CentOS", "", "Libraries for decoding LZMA compression", "96bf8a25552450c4992aeb824bb55c87"},
			{intRef(), "python-podman-api", "1.2.0", "0.2.gitd0a45fe.module_el8.2.0+305+5e198a41", "noarch", "python-podman-api-1.2.0-0.2.gitd0a45fe.module_el8.2.0+305+5e198a41.src.rpm", 120489, "LGPLv2", "CentOS", "container-tools:rhel8:8020020200507003530:0d58ad57", "Podman API", "caa77010c68fe9c81b926a5fac967a9f"},
			{intRef(), "libcap", "2.26", "3.el8", "x86_64", "libcap-2.26-3.el8.src.rpm", 124170, "GPLv2", "CentOS", "", "Library for getting and setting POSIX.1e capabilities", "5fe96e3190b4c9ae5bfcdb5268e7a69f"},
			{intRef(), "podman-docker", "1.6.4", "10.module_el8.2.0+305+5e198a41", "noarch", "podman-1.6.4-10.module_el8.2.0+305+5e198a41.src.rpm", 5996, "ASL 2.0", "CentOS", "container-tools:rhel8:8020020200507003530:0d58ad57", "Emulate Docker CLI using podman", "939dd8968dd7c58426516083e28bb950"},
			{intRef(), "info", "6.5", "6.el8", "x86_64", "texinfo-6.5-6.el8.src.rpm", 386513, "GPLv3+", "CentOS", "", "A stand-alone TTY-based reader for GNU texinfo documentation", "e368cb98b8ae037ff471725c2cea363e"},
			{intRef(4), "perl-libs", "5.26.3", "416.el8", "x86_64", "perl-5.26.3-416.el8.src.rpm", 6122645, "(GPL+ or Artistic) and HSRL and MIT and UCD", "CentOS", "", "The libraries for the perl run-time", "6a77515e3417539336ed3f49908f83c7"},
			{intRef(), "libcom_err", "1.45.4", "3.el8", "x86_64", "e2fsprogs-1.45.4-3.el8.src.rpm", 61921, "MIT", "CentOS", "", "Common error description library", "ec30a7983356b3de9e83f1dda2eecd8b"},
			{intRef(), "urw-base35-fonts-common", "20170801", "10.el8", "noarch", "urw-base35-fonts-20170801-10.el8.src.rpm", 38217, "AGPLv3", "CentOS", "", "Common files of the (URW)++ Level 2 Core Font Set", "b9c0233504dbc87da36397e6a3d37efc"},
			{intRef(), "libxml2", "2.9.7", "7.el8", "x86_64", "libxml2-2.9.7-7.el8.src.rpm", 1752506, "MIT", "CentOS", "", "Library providing XML and HTML support", "18a89ea76ea4278d068b24672a91f048"},
			{intRef(), "nss-util", "3.53.1", "11.el8_2", "x86_64", "nss-3.53.1-11.el8_2.src.rpm", 220716, "MPLv2.0", "CentOS", "", "Network Security Services Utilities Library", "f4a99f6bc7e92ebfb9ae2470de3f6de4"},
			{intRef(), "expat", "2.2.5", "3.el8", "x86_64", "expat-2.2.5-3.el8.src.rpm", 314068, "MIT", "CentOS", "", "An XML parser library", "0599f3c75b448b9c88e478b764361704"},
			{intRef(1), "perl-parent", "0.237", "1.el8", "noarch", "perl-parent-0.237-1.el8.src.rpm", 9187, "GPL+ or Artistic", "CentOS", "", "Establish an ISA relationship with base classes at compile time", "3a1e8a3596d9a3ca9021a40c141adca3"},
			{intRef(), "libuuid", "2.32.1", "22.el8", "x86_64", "util-linux-2.32.1-22.el8.src.rpm", 34832, "BSD", "CentOS", "", "Universally unique ID library", "490942418c894fbc00e1bed23dd61096"},
			{intRef(), "boost-system", "1.66.0", "7.el8", "x86_64", "boost-1.66.0-7.el8.src.rpm", 21994, "Boost and MIT and Python", "CentOS", "", "Run-time component of boost system support library", "7c273dcab4dfb6745e50c9397474d67b"},
			{intRef(), "chkconfig", "1.11", "1.el8", "x86_64", "chkconfig-1.11-1.el8.src.rpm", 791234, "GPLv2", "CentOS", "", "A system tool for maintaining the /etc/rc*.d hierarchy", "33ae796d1b96f631732e6553d5e67e26"},
			{intRef(), "libICE", "1.0.9", "15.el8", "x86_64", "libICE-1.0.9-15.el8.src.rpm", 205821, "MIT", "CentOS", "", "X.Org X11 ICE runtime library", "240847d47f854c3eeef95daf071359da"},
			{intRef(1), "gmp", "6.1.2", "10.el8", "x86_64", "gmp-6.1.2-10.el8.src.rpm", 1678740, "LGPLv3+ or GPLv2+", "CentOS", "", "A GNU arbitrary precision library", "d97c6201b7617cb78474f88a10f7da34"},
			{intRef(), "unzip", "6.0", "43.el8", "x86_64", "unzip-6.0-43.el8.src.rpm", 423365, "BSD", "CentOS", "", "A utility for unpacking zip files", "941c099b3eb137f41a3cd17932dd9edf"},
			{intRef(), "libattr", "2.4.48", "3.el8", "x86_64", "attr-2.4.48-3.el8.src.rpm", 27346, "LGPLv2+", "CentOS", "", "Dynamic library for extended attribute support", "bdc22fe0cdcb03a5014e6275c6539003"},
			{intRef(), "libcroco", "0.6.12", "4.el8_2.1", "x86_64", "libcroco-0.6.12-4.el8_2.1.src.rpm", 330811, "LGPLv2", "CentOS", "", "A CSS2 parsing library", "15aa0ca9c1889bdda4537f342f8df60d"},
			{intRef(), "coreutils-single", "8.30", "7.el8_2.1", "x86_64", "coreutils-8.30-7.el8_2.1.src.rpm", 1356273, "GPLv3+", "CentOS", "", "coreutils multicall binary", "68abfec2549d36f258bd2c2e513fbddc"},
			{intRef(), "avahi-libs", "0.7", "19.el8", "x86_64", "avahi-0.7-19.el8.src.rpm", 162712, "LGPLv2+", "CentOS", "", "Libraries for avahi run-time use", "3666aa23d780de207324079dd88ad562"},
			{intRef(), "libblkid", "2.32.1", "22.el8", "x86_64", "util-linux-2.32.1-22.el8.src.rpm", 339680, "LGPLv2+", "CentOS", "", "Block device ID library", "081c1805273069f7933c6a2bc62f2a5b"},
			{intRef(), "libmpc", "1.0.2", "9.el8", "x86_64", "libmpc-1.0.2-9.el8.src.rpm", 154016, "LGPLv3+ and GFDL", "CentOS", "", "C library for multiple precision complex arithmetic", "2c68aca463647ae457efae8232d80a86"},
			{intRef(), "libcap-ng", "0.7.9", "5.el8", "x86_64", "libcap-ng-0.7.9-5.el8.src.rpm", 51278, "LGPLv2+", "CentOS", "", "An alternate posix capabilities library", "dbb245f0f1383ee7e20511a832a9caeb"},
			{intRef(), "libXfixes", "5.0.3", "7.el8", "x86_64", "libXfixes-5.0.3-7.el8.src.rpm", 29263, "MIT", "CentOS", "", "X Fixes library", "56d72ffa985b1d479bf170e442b840e9"},
			{intRef(), "libffi", "3.1", "21.el8", "x86_64", "libffi-3.1-21.el8.src.rpm", 68404, "MIT", "CentOS", "", "A portable foreign function interface library", "75f0d6ae08ec69f1d9c600ce05bfad37"},
			{intRef(), "gettext-libs", "0.19.8.1", "17.el8", "x86_64", "gettext-0.19.8.1-17.el8.src.rpm", 1612648, "LGPLv2+ and GPLv3+", "CentOS", "", "Libraries for gettext", "8967e0fd86f922ac3fe079bb64ec85c4"},
			{intRef(), "libzstd", "1.4.2", "2.el8", "x86_64", "zstd-1.4.2-2.el8.src.rpm", 703765, "BSD and GPLv2", "CentOS", "", "Zstd shared library", "6cc827a8449f9baed1dbad4f5992d2a1"},
			{intRef(), "libSM", "1.2.3", "1.el8", "x86_64", "libSM-1.2.3-1.el8.src.rpm", 93427, "MIT", "CentOS", "", "X.Org X11 SM runtime library", "47f227b29e3abb1260b8f1f178b588ca"},
			{intRef(), "lz4-libs", "1.8.1.2", "4.el8", "x86_64", "lz4-1.8.1.2-4.el8.src.rpm", 97367, "GPLv2+ and BSD", "CentOS", "", "Libaries for lz4", "a5e16dd8650587ebf1e85350bb7fb063"},
			{intRef(), "libXmu", "1.1.2", "12.el8", "x86_64", "libXmu-1.1.2-12.el8.src.rpm", 196334, "MIT", "CentOS", "", "X.Org X11 libXmu/libXmuu runtime libraries", "74c7427d544d1ce1d0173e93b2a035e8"},
			{intRef(), "libgcrypt", "1.8.3", "4.el8", "x86_64", "libgcrypt-1.8.3-4.el8.src.rpm", 1547061, "LGPLv2+", "CentOS", "", "A general-purpose cryptography library", "453cb9a4dcfba2847247a6004adb8a40"},
			{intRef(4), "perl-macros", "5.26.3", "416.el8", "x86_64", "perl-5.26.3-416.el8.src.rpm", 5184, "GPL+ or Artistic", "CentOS", "", "Macros for rpmbuild", "ae436db14693a9b49ecd5d02ef877a13"},
			{intRef(), "cracklib", "2.9.6", "15.el8", "x86_64", "cracklib-2.9.6-15.el8.src.rpm", 239047, "LGPLv2+", "CentOS", "", "A password-checking library", "3ced50b8ce5c91849d67c8e6f75f5cd1"},
			{intRef(4), "perl-Socket", "2.027", "3.el8", "x86_64", "perl-Socket-2.027-3.el8.src.rpm", 127014, "GPL+ or Artistic", "CentOS", "", "Networking constants and support functions", "6805c843a6f77fb26c9213320e0c54d1"},
			{intRef(), "libidn2", "2.2.0", "1.el8", "x86_64", "libidn2-2.2.0-1.el8.src.rpm", 287762, "(GPLv2+ or LGPLv3+) and GPLv3+", "CentOS", "", "Library to support IDNA2008 internationalized domain names", "206b757710d3924958eeb8bd423243bd"},
			{intRef(), "perl-Unicode-Normalize", "1.25", "396.el8", "x86_64", "perl-Unicode-Normalize-1.25-396.el8.src.rpm", 637517, "GPL+ or Artistic", "CentOS", "", "Unicode Normalization Forms", "8461c73c94cc877ae2103add615572f8"},
			{intRef(), "file-libs", "5.33", "13.el8", "x86_64", "file-5.33-13.el8.src.rpm", 6382974, "BSD", "CentOS", "", "Libraries for applications using libmagic", "99f7be5b7e16f6c8881a5f0c7201d727"},
			{intRef(0), "perl-IO", "1.38", "416.el8", "x86_64", "perl-5.26.3-416.el8.src.rpm", 140303, "GPL+ or Artistic", "CentOS", "", "Perl input/output modules", "24ea3921fbc00498cc592aa5fa5513c4"},
			{intRef(), "keyutils-libs", "1.5.10", "6.el8", "x86_64", "keyutils-1.5.10-6.el8.src.rpm", 43926, "GPLv2+ and LGPLv2+", "CentOS", "", "Key utilities library", "abf58d0542bb7f9113858fce1e2aa9b4"},
			{intRef(), "perl-constant", "1.33", "396.el8", "noarch", "perl-constant-1.33-396.el8.src.rpm", 27104, "GPL+ or Artistic", "CentOS", "", "Perl pragma to declare constants", "dbb827cf7e21d66176d41c7280d1da8b"},
			{intRef(), "p11-kit-trust", "0.23.14", "5.el8_0", "x86_64", "p11-kit-0.23.14-5.el8_0.src.rpm", 508547, "BSD", "CentOS", "", "System trust module from p11-kit", "801f7a4fef4a8c568c336fdfb3031b21"},
			{intRef(), "perl-threads-shared", "1.58", "2.el8", "x86_64", "perl-threads-shared-1.58-2.el8.src.rpm", 78170, "GPL+ or Artistic", "CentOS", "", "Perl extension for sharing data structures between threads", "f7ca1f9a2bb956e0fcb5c0921352b23c"},
			{intRef(), "pcre", "8.42", "4.el8", "x86_64", "pcre-8.42-4.el8.src.rpm", 518067, "BSD", "CentOS", "", "Perl-compatible regular expression library", "9c05cc476028e5b01edd29104ec2e9b3"},
			{intRef(), "perl-MIME-Base64", "3.15", "396.el8", "x86_64", "perl-MIME-Base64-3.15-396.el8.src.rpm", 41435, "(GPL+ or Artistic) and MIT", "CentOS", "", "Encoding and decoding of Base64 and quoted-printable strings", "8ed7ed3ebaf11f7a51f28e8602e60020"},
			{intRef(), "systemd-libs", "239", "31.el8_2.2", "x86_64", "systemd-239-31.el8_2.2.src.rpm", 4497790, "LGPLv2+ and MIT", "CentOS", "", "systemd libraries", "a316b3f18d3fd2ae1a08e2a877f41ae7"},
			{intRef(1), "perl-Time-Local", "1.280", "1.el8", "noarch", "perl-Time-Local-1.280-1.el8.src.rpm", 59906, "GPL+ or Artistic", "CentOS", "", "Efficiently compute time from local and GMT time", "d9ff68ed7dce4a5bb9a13c85270f9908"},
			{intRef(1), "dbus-tools", "1.12.8", "10.el8_2", "x86_64", "dbus-1.12.8-10.el8_2.src.rpm", 125490, "(GPLv2+ or AFL) and GPLv2+", "CentOS", "", "D-BUS Tools and Utilities", "7d9ee2812adf87829a32fb0b026fe137"},
			{intRef(), "perl-File-Temp", "0.230.600", "1.el8", "noarch", "perl-File-Temp-0.230.600-1.el8.src.rpm", 164723, "GPL+ or Artistic", "CentOS", "", "Return name and handle of a temporary file safely", "4a996a6bcd7b0cda3b3d8dc5b4e712fe"},
			{intRef(), "libusbx", "1.0.22", "1.el8", "x86_64", "libusbx-1.0.22-1.el8.src.rpm", 151177, "LGPLv2+", "CentOS", "", "Library for accessing USB devices", "8ccdca006e6ce8c7c62ac19db25e7734"},
			{intRef(), "patch", "2.7.6", "11.el8", "x86_64", "patch-2.7.6-11.el8.src.rpm", 266609, "GPLv3+", "CentOS", "", "Utility for modifying/upgrading files", "0d454699b91476374cc9397d264731f3"},
			{intRef(), "ca-certificates", "2019.2.32", "80.0.el8_1", "noarch", "ca-certificates-2019.2.32-80.0.el8_1.src.rpm", 993761, "Public Domain", "CentOS", "", "The Mozilla CA root certificate bundle", "2f7ed719e139f40f7be253d44b555743"},
			{intRef(), "file", "5.33", "13.el8", "x86_64", "file-5.33-13.el8.src.rpm", 93224, "BSD", "CentOS", "", "A utility for determining file types", "814793b843457f406144e67030207c01"},
			{intRef(), "squashfs-tools", "4.3", "19.el8", "x86_64", "squashfs-tools-4.3-19.el8.src.rpm", 502829, "GPLv2+", "CentOS", "", "Utility for the creation of squashfs filesystems", "23db010940aed94245b743afe8aabfec"},
			{intRef(), "libXrandr", "1.5.1", "7.el8", "x86_64", "libXrandr-1.5.1-7.el8.src.rpm", 53138, "MIT", "CentOS", "", "X.Org X11 libXrandr runtime library", "a40ba84685d7ff94db6b4b8ef61ccf0c"},
			{intRef(), "libsemanage", "2.9", "2.el8", "x86_64", "libsemanage-2.9-2.el8.src.rpm", 477962, "LGPLv2+", "CentOS", "", "SELinux binary policy manipulation library", "dabbf5bee55cc3853ae33c8e739c59b1"},
			{intRef(), "libXi", "1.7.9", "7.el8", "x86_64", "libXi-1.7.9-7.el8.src.rpm", 98363, "MIT", "CentOS", "", "X.Org X11 libXi runtime library", "fbc24d32e386398b59e16e2af61e1219"},
			{intRef(), "libutempter", "1.1.6", "14.el8", "x86_64", "libutempter-1.1.6-14.el8.src.rpm", 52637, "LGPLv2+", "CentOS", "", "A privileged helper for utmp/wtmp updates", "d8773aceac7955c99c3d89a95568c180"},
			{intRef(), "atk", "2.28.1", "1.el8", "x86_64", "atk-2.28.1-1.el8.src.rpm", 1294970, "LGPLv2+", "CentOS", "", "Interfaces for accessibility support", "d03abb8e862140158230d990e67ed0da"},
			{intRef(), "acl", "2.2.53", "1.el8", "x86_64", "acl-2.2.53-1.el8.src.rpm", 205740, "GPLv2+", "CentOS", "", "Access control list utilities", "483792b8b5f9eb8be7dc4407733118d0"},
			{intRef(), "adobe-mappings-cmap-deprecated", "20171205", "3.el8", "noarch", "adobe-mappings-cmap-20171205-3.el8.src.rpm", 596942, "BSD", "CentOS", "", "Deprecated CMap resources for Adobe's character collections", "6e60575dedf19a3ce375fc2166d1a146"},
			{intRef(), "nettle", "3.4.1", "1.el8", "x86_64", "nettle-3.4.1-1.el8.src.rpm", 683185, "LGPLv3+ or GPLv2+", "CentOS", "", "A low-level cryptographic library", "6bf682f2d691cb408aa61a2601d0f922"},
			{intRef(), "perl-Digest", "1.17", "395.el8", "noarch", "perl-Digest-1.17-395.el8.src.rpm", 26685, "GPL+ or Artistic", "CentOS", "", "Modules that calculate message digests", "3288d677385ef645566a458dae131af2"},
			{intRef(), "libcomps", "0.1.11", "4.el8", "x86_64", "libcomps-0.1.11-4.el8.src.rpm", 217067, "GPLv2+", "CentOS", "", "Comps XML file manipulation library", "61f1c2ca663b8226890e72a85e94f058"},
			{intRef(), "perl-Net-SSLeay", "1.88", "1.el8", "x86_64", "perl-Net-SSLeay-1.88-1.el8.src.rpm", 1368791, "Artistic 2.0", "CentOS", "", "Perl extension for using OpenSSL", "d060dd8a7177e91cee9f963348ab8305"},
			{intRef(1), "findutils", "4.6.0", "20.el8", "x86_64", "findutils-4.6.0-20.el8.src.rpm", 1816673, "GPLv3+", "CentOS", "", "The GNU versions of find utilities (find and xargs)", "5ca7479cd7d1cdbe30746ce1c53d6bc1"},
			{intRef(), "perl-TermReadKey", "2.37", "7.el8", "x86_64", "perl-TermReadKey-2.37-7.el8.src.rpm", 66206, "(Copyright only) and (Artistic or GPL+)", "CentOS", "", "A perl module for simple terminal control", "1ca4492dd5719a92713930073a8fc89c"},
			{intRef(), "cpio", "2.12", "8.el8", "x86_64", "cpio-2.12-8.el8.src.rpm", 989536, "GPLv3+", "CentOS", "", "A GNU archiving program", "7ddd4e1548f05e3884f62166277480e7"},
			{intRef(1), "valgrind", "3.15.0", "11.el8", "x86_64", "valgrind-3.15.0-11.el8.src.rpm", 32152354, "GPLv2+", "CentOS", "", "Tool for finding memory management bugs in programs", "c6d75d1d561bd01756f629d43920f8dd"},
			{intRef(), "ipcalc", "0.2.4", "4.el8", "x86_64", "ipcalc-0.2.4-4.el8.src.rpm", 67705, "GPLv2+", "CentOS", "", "IP network address calculator", "4564057e3ae90a2296967da77f71cb87"},
			{intRef(1), "perl-Storable", "3.11", "3.el8", "x86_64", "perl-Storable-3.11-3.el8.src.rpm", 221862, "GPL+ or Artistic", "CentOS", "", "Persistence for Perl data structures", "7575af2e91c5f435c050e1b3f453c9f3"},
			{intRef(), "libnghttp2", "1.33.0", "3.el8_2.1", "x86_64", "nghttp2-1.33.0-3.el8_2.1.src.rpm", 168044, "MIT", "CentOS", "", "A library implementing the HTTP/2 protocol", "54279556ffeb8d7a527c7d6d157965e3"},
			{intRef(), "perl-Mozilla-CA", "20160104", "7.el8", "noarch", "perl-Mozilla-CA-20160104-7.el8.src.rpm", 5716, "MPLv2.0", "CentOS", "", "Mozilla's CA cert bundle in PEM format", "e517a072a8b0ef27b42a5472efb1bd17"},
			{intRef(), "iptables-libs", "1.8.4", "10.el8_2.1", "x86_64", "iptables-1.8.4-10.el8_2.1.src.rpm", 201920, "GPLv2 and Artistic 2.0 and ISC", "CentOS", "", "iptables libraries", "5e4704a82f9b3370b941fddcb34c8031"},
			{intRef(), "libXaw", "1.0.13", "10.el8", "x86_64", "libXaw-1.0.13-10.el8.src.rpm", 523837, "MIT", "CentOS", "", "X Athena Widget Set", "2017f62bc27ca372978960ff77416260"},
			{intRef(), "libsigsegv", "2.11", "5.el8", "x86_64", "libsigsegv-2.11-5.el8.src.rpm", 47034, "GPLv2+", "CentOS", "", "Library for handling page faults in user mode", "5625ecad632168eecb8960821b0abeca"},
			{intRef(), "libXdamage", "1.1.4", "14.el8", "x86_64", "libXdamage-1.1.4-14.el8.src.rpm", 30432, "MIT", "CentOS", "", "X Damage extension library", "5a6d7d7ded7b105fd1e101c7cba314e9"},
			{intRef(), "libverto", "0.3.0", "5.el8", "x86_64", "libverto-0.3.0-5.el8.src.rpm", 28244, "MIT", "CentOS", "", "Main loop abstraction library", "4b04a34ead82e762637fae7d79adf926"},
			{intRef(), "libidn", "1.34", "5.el8", "x86_64", "libidn-1.34-5.el8.src.rpm", 713211, "LGPLv2+ and GPLv3+ and GFDL", "CentOS", "", "Internationalized Domain Name support library", "093d16171fe387733094b8aead460e49"},
			{intRef(), "libtirpc", "1.1.4", "4.el8", "x86_64", "libtirpc-1.1.4-4.el8.src.rpm", 381964, "SISSL and BSD", "CentOS", "", "Transport Independent RPC Library", "8b4eec2ef36446061ee886a625584830"},
			{intRef(), "boost-thread", "1.66.0", "7.el8", "x86_64", "boost-1.66.0-7.el8.src.rpm", 182682, "Boost and MIT and Python", "CentOS", "", "Run-time component of boost thread library", "ebbe0723621f5c00877eec30e7a4f61c"},
			{intRef(1), "openssl-libs", "1.1.1c", "15.el8", "x86_64", "openssl-1.1.1c-15.el8.src.rpm", 3744176, "OpenSSL", "CentOS", "", "A general purpose cryptography library with TLS implementation", "035906e9d19890fe1f95b1e8d4b1a890"},
			{intRef(), "nss-softokn-freebl", "3.53.1", "11.el8_2", "x86_64", "nss-3.53.1-11.el8_2.src.rpm", 634614, "MPLv2.0", "CentOS", "", "Freebl library for the Network Security Services", "2dc45b206ef714ffe6914e0af559ddf8"},
			{intRef(), "platform-python-setuptools", "39.2.0", "5.el8", "noarch", "python-setuptools-39.2.0-5.el8.src.rpm", 2930163, "MIT", "CentOS", "", "Easily build and distribute Python 3 packages", "864c0cf264ad1ece7b0df732cc247d99"},
			{intRef(), "nss", "3.53.1", "11.el8_2", "x86_64", "nss-3.53.1-11.el8_2.src.rpm", 1996163, "MPLv2.0", "CentOS", "", "Network Security Services", "e5a5911dab0c66f9fd756f7845278a8d"},
			{intRef(), "python3-libs", "3.6.8", "23.el8", "x86_64", "python3-3.6.8-23.el8.src.rpm", 32187857, "Python", "CentOS", "", "Python runtime libraries", "24366dc108d8d78edaf5a0297620741c"},
			{intRef(), "nss-tools", "3.53.1", "11.el8_2", "x86_64", "nss-3.53.1-11.el8_2.src.rpm", 2286837, "MPLv2.0", "CentOS", "", "Tools for the Network Security Services", "ec266d08839e08a1330f1e015fd8ce53"},
			{intRef(), "libpwquality", "1.4.0", "9.el8", "x86_64", "libpwquality-1.4.0-9.el8.src.rpm", 384791, "BSD or GPLv2+", "CentOS", "", "A library for password generation and password quality checking", "5b6c127112b13b51d39696fe3eb6ae50"},
			{intRef(), "perl-Term-ANSIColor", "4.06", "396.el8", "noarch", "perl-Term-ANSIColor-4.06-396.el8.src.rpm", 89627, "GPL+ or Artistic", "CentOS", "", "Color screen output using ANSI escape sequences", "c0fc2f73ed2179d13f01ef64e42008ad"},
			{intRef(), "util-linux", "2.32.1", "22.el8", "x86_64", "util-linux-2.32.1-22.el8.src.rpm", 11560494, "GPLv2 and GPLv2+ and LGPLv2+ and BSD with advertising and Public Domain", "CentOS", "", "A collection of basic system utilities", "f6b9e0111f2f92d2a3cfe3091237a049"},
			{intRef(), "fipscheck", "1.5.0", "4.el8", "x86_64", "fipscheck-1.5.0-4.el8.src.rpm", 48547, "BSD", "CentOS", "", "A library for integrity verification of FIPS validated modules", "33be21c19bde710c0b22e48c6ab19c91"},
			{intRef(), "glib2", "2.56.4", "8.el8", "x86_64", "glib2-2.56.4-8.el8.src.rpm", 12272168, "LGPLv2+", "CentOS", "", "A library of handy utility functions", "5cd98b131a43e30a121edb9f26697a57"},
			{intRef(), "sgml-common", "0.6.3", "50.el8", "noarch", "sgml-common-0.6.3-50.el8.src.rpm", 172077, "GPL+", "CentOS", "", "Common SGML catalog and DTD files", "a9b3cbd855749f5b6d03ef44439af469"},
			{intRef(), "iproute", "5.3.0", "1.el8", "x86_64", "iproute-5.3.0-1.el8.src.rpm", 1894954, "GPLv2+ and Public Domain", "CentOS", "", "Advanced IP routing and network device configuration tools", "d8fba74d3a00fae03acc06cf82eac90b"},
			{intRef(), "docbook-style-xsl", "1.79.2", "7.el8", "noarch", "docbook-style-xsl-1.79.2-7.el8.src.rpm", 16311851, "DMIT", "CentOS", "", "Norman Walsh's XSL stylesheets for DocBook XML", "abdf03995a730a98874e10c8596b0047"},
			{intRef(), "kmod", "25", "16.el8", "x86_64", "kmod-25-16.el8.src.rpm", 243998, "GPLv2+", "CentOS", "", "Linux kernel module management utilities", "57442c44df12acfc5a6164cbd3780511"},
			{intRef(), "python3-dnf-plugins-core", "4.0.12", "4.el8_2", "noarch", "dnf-plugins-core-4.0.12-4.el8_2.src.rpm", 663140, "GPLv2+", "CentOS", "", "Core Plugins for DNF", "095f7428ed2f94967651d5e4671db45f"},
			{intRef(), "curl", "7.61.1", "12.el8", "x86_64", "curl-7.61.1-12.el8.src.rpm", 709006, "MIT", "CentOS", "", "A utility for getting files from remote servers (FTP, HTTP, and others)", "c305531060535f28771ea538447cc0a9"},
			{intRef(), "pkgconf-m4", "1.4.2", "1.el8", "noarch", "pkgconf-1.4.2-1.el8.src.rpm", 14187, "GPLv2+ with exceptions", "CentOS", "", "m4 macros for pkgconf", "398894375a82b4e1d5f7f0b38d3bec99"},
			{intRef(), "openldap", "2.4.46", "11.el8_1", "x86_64", "openldap-2.4.46-11.el8_1.src.rpm", 1388793, "OpenLDAP", "CentOS", "", "LDAP support libraries", "435b641c2259efbca591c3692fb3ee4f"},
			{intRef(), "ncurses", "6.1", "7.20180224.el8", "x86_64", "ncurses-6.1-7.20180224.el8.src.rpm", 600396, "MIT", "CentOS", "", "Ncurses support utilities", "93bfe403590240ca37823287b0cb4669"},
			{intRef(), "python3-libcomps", "0.1.11", "4.el8", "x86_64", "libcomps-0.1.11-4.el8.src.rpm", 147027, "GPLv2+", "CentOS", "", "Python 3 bindings for libcomps library", "5b6f724e320aac06d2dbb21171455287"},
			{intRef(1), "make", "4.2.1", "10.el8", "x86_64", "make-4.2.1-10.el8.src.rpm", 1435840, "GPLv3+", "CentOS", "", "A GNU tool which simplifies the build process for users", "5e35f9e2dafcf4717463a5cccd6363eb"},
			{intRef(), "libarchive", "3.3.2", "8.el8_1", "x86_64", "libarchive-3.3.2-8.el8_1.src.rpm", 1139914, "BSD", "CentOS", "", "A library for handling streaming archive formats", "3e813ad4d33701372cb8dcbf7e13bab6"},
			{intRef(), "libxslt", "1.1.32", "4.el8", "x86_64", "libxslt-1.1.32-4.el8.src.rpm", 751661, "MIT", "CentOS", "", "Library providing the Gnome XSLT engine", "540f694353a8460979a80d2c66480f21"},
			{intRef(), "rpm-libs", "4.14.2", "37.el8", "x86_64", "rpm-4.14.2-37.el8.src.rpm", 722464, "GPLv2+ and LGPLv2+ with exceptions", "CentOS", "", "Libraries for manipulating RPM packages", "80d6f6cd5681e87b27b4271bd26a45ab"},
			{intRef(), "libpkgconf", "1.4.2", "1.el8", "x86_64", "pkgconf-1.4.2-1.el8.src.rpm", 68093, "ISC", "CentOS", "", "Backend library for pkgconf", "c1d476e8247ae535374ebffccfb2ca5c"},
			{intRef(), "libyaml", "0.1.7", "5.el8", "x86_64", "libyaml-0.1.7-5.el8.src.rpm", 136478, "MIT", "CentOS", "", "YAML 1.1 parser and emitter written in C", "7c8bef326f7edfab7928fbaaeee8c652"},
			{intRef(), "npth", "1.5", "4.el8", "x86_64", "npth-1.5-4.el8.src.rpm", 47909, "LGPLv2+", "CentOS", "", "The New GNU Portable Threads library", "b944c570e2db54dd8f8d3ab4d7b7990d"},
			{intRef(), "gpgme", "1.10.0", "6.el8.0.1", "x86_64", "gpgme-1.10.0-6.el8.0.1.src.rpm", 741097, "LGPLv2+", "CentOS", "", "GnuPG Made Easy - high level crypto API", "23c61df97de39d050a78bfe752b271fe"},
			{intRef(), "libdnf", "0.39.1", "6.el8_2", "x86_64", "libdnf-0.39.1-6.el8_2.src.rpm", 2123765, "LGPLv2+", "CentOS", "", "Library providing simplified C and Python API to libsolv", "fffff8e4851ab715207b238f149bbbdd"},
			{intRef(), "python3-hawkey", "0.39.1", "6.el8_2", "x86_64", "libdnf-0.39.1-6.el8_2.src.rpm", 263176, "LGPLv2+", "CentOS", "", "Python 3 bindings for the hawkey library", "b91b61327f7e96d7cfc60c43131e344c"},
			{intRef(), "libreport-filesystem", "2.9.5", "10.el8", "x86_64", "libreport-2.9.5-10.el8.src.rpm", 0, "GPLv2+", "CentOS", "", "Filesystem layout for libreport", "688a38734bfb36dcd3aa27147e733f48"},
			{intRef(12), "dhcp-common", "4.3.6", "40.el8", "noarch", "dhcp-4.3.6-40.el8.src.rpm", 301814, "ISC", "CentOS", "", "Common files used by ISC dhcp client, server and relay agent", "cff0499703ca84229ef6cdb351b72248"},
			{intRef(1), "dbus-daemon", "1.12.8", "10.el8_2", "x86_64", "dbus-1.12.8-10.el8_2.src.rpm", 560056, "(GPLv2+ or AFL) and GPLv2+", "CentOS", "", "D-BUS message bus", "5650c215b979d90eda12fbc0f3029c02"},
			{intRef(8), "device-mapper-libs", "1.02.169", "3.el8", "x86_64", "lvm2-2.03.08-3.el8.src.rpm", 416167, "LGPLv2", "CentOS", "", "Device-mapper shared library", "5efcaa95e0028e060cbb5e8654429d5c"},
			{intRef(), "elfutils-default-yama-scope", "0.178", "7.el8", "noarch", "elfutils-0.178-7.el8.src.rpm", 1810, "GPLv2+ or LGPLv3+", "CentOS", "", "Default yama attach scope sysctl setting", "d610d0c6277635bcb67b234bc252e3ab"},
			{intRef(), "systemd-pam", "239", "31.el8_2.2", "x86_64", "systemd-239-31.el8_2.2.src.rpm", 902504, "LGPLv2+ and MIT and GPLv2+", "CentOS", "", "systemd PAM module", "c7bcefdda4c92a3ddce3361e73a6fb04"},
			{intRef(1), "dbus", "1.12.8", "10.el8_2", "x86_64", "dbus-1.12.8-10.el8_2.src.rpm", 0, "(GPLv2+ or AFL) and GPLv2+", "CentOS", "", "D-BUS message bus", "0527c3681f77bd4775f23298020c568c"},
			{intRef(12), "dhcp-client", "4.3.6", "40.el8", "x86_64", "dhcp-4.3.6-40.el8.src.rpm", 530732, "ISC", "CentOS", "", "Provides the ISC DHCP client daemon and dhclient-script", "188f208523e4eff06746a39355aae3f9"},
			{intRef(), "libkcapi-hmaccalc", "1.1.1", "16_1.el8", "x86_64", "libkcapi-1.1.1-16_1.el8.src.rpm", 35165, "BSD or GPLv2", "CentOS", "", "Drop-in replacements for hmaccalc provided by the libkcapi package", "386b48136f3955caddf97c0526e0fee0"},
			{intRef(), "dracut", "049", "70.git20200228.el8", "x86_64", "dracut-049-70.git20200228.el8.src.rpm", 1046582, "GPLv2+ and LGPLv2+", "CentOS", "", "Initramfs generator using udev", "6ebe6c201c226b6df68e38a6d9e2c0c3"},
			{intRef(), "dracut-squash", "049", "70.git20200228.el8", "x86_64", "dracut-049-70.git20200228.el8.src.rpm", 3054, "GPLv2+ and LGPLv2+", "CentOS", "", "dracut module to build an initramfs with most files in a squashfs image", "477bd22fb0a32db1be3435b173f6e8a7"},
			{intRef(), "python3-rpm", "4.14.2", "37.el8", "x86_64", "rpm-4.14.2-37.el8.src.rpm", 430929, "GPLv2+", "CentOS", "", "Python 3 bindings for apps which will manipulate RPM packages", "3784956e0a76e04444fc82b6fe6a6d44"},
			{intRef(), "dnf", "4.2.17", "7.el8_2", "noarch", "dnf-4.2.17-7.el8_2.src.rpm", 1670640, "GPLv2+ and GPLv2 and GPL", "CentOS", "", "Package manager", "d11048989f73ec620c1e1d1df594e893"},
			{intRef(), "kexec-tools", "2.0.20", "14.el8", "x86_64", "kexec-tools-2.0.20-14.el8.src.rpm", 1222009, "GPLv2", "CentOS", "", "The kexec/kdump userspace component", "a826184bec8587e019019077336578fa"},
			{intRef(2), "tar", "1.30", "4.el8", "x86_64", "tar-1.30-4.el8.src.rpm", 2914728, "GPLv3+", "CentOS", "", "A GNU file archiving program", "9cf6795c42907b004d73bbb404a037fc"},
			{intRef(), "hostname", "3.20", "6.el8", "x86_64", "hostname-3.20-6.el8.src.rpm", 43979, "GPLv2+", "CentOS", "", "Utility to set/show the host name or domain name", "6f9f94b9611168a6baa489eeae0d1f05"},
			{intRef(), "langpacks-en", "1.0", "12.el8", "noarch", "langpacks-1.0-12.el8.src.rpm", 400, "GPLv2+", "CentOS", "", "English langpacks meta-package", "170167632b0fe83fe2b3f2b1296109e2"},
			{intRef(), "gpg-pubkey", "8483c65d", "5ccc5b19", "", "", 0, "pubkey", "", "", "gpg(CentOS (CentOS Official Signing Key) <security@centos.org>)", ""},
			{intRef(1), "npm", "6.14.4", "1.10.21.0.3.module_el8.2.0+391+8da3adc6", "x86_64", "nodejs-10.21.0-3.module_el8.2.0+391+8da3adc6.src.rpm", 16351222, "MIT and ASL 2.0 and ISC and BSD", "CentOS", "nodejs:10:8020020200707141642:6a468ee4", "Node.js Package Manager", "c36d61a68ea091683d7a2c88e19239c1"},
			{intRef(), "python3-libselinux", "2.9", "3.el8", "x86_64", "libselinux-2.9-3.el8.src.rpm", 894878, "Public Domain", "CentOS", "", "SELinux python 3 bindings for libselinux", "51b9d30308d8f844e831c8c67f64da43"},
			{intRef(2), "libpng", "1.6.34", "5.el8", "x86_64", "libpng-1.6.34-5.el8.src.rpm", 235304, "zlib", "CentOS", "", "A library of functions for manipulating PNG image format files", "de6b9bd5cb672b910a696870cb3364ab"},
			{intRef(), "freetype", "2.9.1", "4.el8", "x86_64", "freetype-2.9.1-4.el8.src.rpm", 828847, "(FTL or GPLv2+) and BSD and MIT and Public Domain and zlib with acknowledgement", "CentOS", "", "A free and portable font rendering engine", "40b8383daf8e9f5e8b53a5919f6b971b"},
			{intRef(), "gobject-introspection", "1.56.1", "1.el8", "x86_64", "gobject-introspection-1.56.1-1.el8.src.rpm", 880467, "GPLv2+, LGPLv2+, MIT", "CentOS", "", "Introspection system for GObject-based libraries", "47a0a168d455401af3b21154c4379f7e"},
			{intRef(), "libnl3-cli", "3.5.0", "1.el8", "x86_64", "libnl3-3.5.0-1.el8.src.rpm", 838214, "LGPLv2", "CentOS", "", "Command line interface utils for libnl3", "c493b4042c637a29bb6edbdcec94a7af"},
			{intRef(), "shared-mime-info", "1.9", "3.el8", "x86_64", "shared-mime-info-1.9-3.el8.src.rpm", 2442451, "GPLv2+", "CentOS", "", "Shared MIME information database", "1d8fd8be23dbda89657c931d08c58c20"},
			{intRef(), "python3-audit", "3.0", "0.17.20191104git1c2f876.el8", "x86_64", "audit-3.0-0.17.20191104git1c2f876.el8.src.rpm", 333023, "LGPLv2+", "CentOS", "", "Python3 bindings for libaudit", "1f24c77d9e5d1506980ef058839f8b55"},
			{intRef(), "libnftnl", "1.1.5", "4.el8", "x86_64", "libnftnl-1.1.5-4.el8.src.rpm", 222320, "GPLv2+", "CentOS", "", "Library for low-level interaction with nftables Netlink's API over libmnl", "ce6abcd6e4f23c881f5826300051f776"},
			{intRef(), "json-glib", "1.4.4", "1.el8", "x86_64", "json-glib-1.4.4-1.el8.src.rpm", 530243, "LGPLv2+", "CentOS", "", "Library for JavaScript Object Notation format", "6e797785105429c5b9286ee029901480"},
			{intRef(), "fontpackages-filesystem", "1.44", "22.el8", "noarch", "fontpackages-1.44-22.el8.src.rpm", 0, "Public Domain", "CentOS", "", "Directories used by font packages", "40b0272ca2fdd533a2d188077e86c1d2"},
			{intRef(), "fontconfig", "2.13.1", "3.el8", "x86_64", "fontconfig-2.13.1-3.el8.src.rpm", 729249, "MIT and Public Domain and UCD", "CentOS", "", "Font configuration and customization library", "7d1a8b0f5c8e099735456e200b67f2ac"},
			{intRef(), "gsettings-desktop-schemas", "3.32.0", "4.el8", "x86_64", "gsettings-desktop-schemas-3.32.0-4.el8.src.rpm", 4216968, "LGPLv2+", "CentOS", "", "A collection of GSettings schemas", "a8b43ba220c7f37604f349107e8a48b8"},
			{intRef(), "libnetfilter_conntrack", "1.0.6", "5.el8", "x86_64", "libnetfilter_conntrack-1.0.6-5.el8.src.rpm", 165190, "GPLv2+", "CentOS", "", "Netfilter conntrack userspace library", "6f1a47011471e28c9e9b1d457bb7d1af"},
			{intRef(1), "nftables", "0.9.3", "12.el8_2.1", "x86_64", "nftables-0.9.3-12.el8_2.1.src.rpm", 811211, "GPLv2", "CentOS", "", "Netfilter Tables userspace utillites", "a4249e7eadec0b6b5a394f0dfb43c12e"},
			{intRef(), "PackageKit-glib", "1.1.12", "4.el8", "x86_64", "PackageKit-1.1.12-4.el8.src.rpm", 635676, "GPLv2+ and LGPLv2+", "CentOS", "", "GLib libraries for accessing PackageKit", "48b85207c2f0c2dfdea979080ab1ed4f"},
			{intRef(), "python3-setools", "4.2.2", "2.el8", "x86_64", "setools-4.2.2-2.el8.src.rpm", 2691942, "GPLv2", "CentOS", "", "Policy analysis tools for SELinux", "f1780e8c8614757ffdb6de8fb95969c9"},
			{intRef(), "python3-six", "1.11.0", "8.el8", "noarch", "python-six-1.11.0-8.el8.src.rpm", 100282, "MIT", "CentOS", "", "Python 2 and 3 compatibility utilities", "0a21bf51853479b94dd494443fa29ecc"},
			{intRef(), "python3-libxml2", "2.9.7", "7.el8", "x86_64", "libxml2-2.9.7-7.el8.src.rpm", 1358319, "MIT", "CentOS", "", "Python 3 bindings for the libxml2 library", "ff6ace818557281219b38c89002a3b66"},
			{intRef(), "platform-python-pip", "9.0.3", "16.el8", "noarch", "python-pip-9.0.3-16.el8.src.rpm", 7749727, "MIT and Python and ASL 2.0 and BSD and ISC and LGPLv2 and MPLv2.0 and (ASL 2.0 or BSD)", "CentOS", "", "A tool for installing and managing Python3 packages", "ba48280ea2dddea9aa4daea4c7f0476c"},
			{intRef(), "python3-pip", "9.0.3", "16.el8", "noarch", "python-pip-9.0.3-16.el8.src.rpm", 2856, "MIT and Python and ASL 2.0 and BSD and ISC and LGPLv2 and MPLv2.0 and (ASL 2.0 or BSD)", "CentOS", "", "A tool for installing and managing Python3 packages", "9d841a5884ea83fe48acfbedc1a6a031"},
			{intRef(), "polkit", "0.115", "11.el8", "x86_64", "polkit-0.115-11.el8.src.rpm", 469075, "LGPLv2+", "CentOS", "", "An authorization framework", "dc299ecae1fea8869cf2016d17ca5b0d"},
			{intRef(), "libvarlink", "18", "3.el8", "x86_64", "libvarlink-18-3.el8.src.rpm", 132453, "ASL 2.0", "CentOS", "", "Varlink C Library", "b4cab07c57e8c29bbf30d65be9b834df"},
			{intRef(), "libssh-config", "0.9.0", "4.el8", "noarch", "libssh-0.9.0-4.el8.src.rpm", 357, "LGPLv2+", "CentOS", "", "Configuration files for libssh", "13da181c86507f0a35851137e2bee8c2"},
			{intRef(), "libndp", "1.7", "3.el8", "x86_64", "libndp-1.7-3.el8.src.rpm", 80786, "LGPLv2+", "CentOS", "", "Library for Neighbor Discovery Protocol", "ab60f69f198d81caff5be4c21d917fc2"},
			{intRef(), "libproxy", "0.4.15", "5.2.el8", "x86_64", "libproxy-0.4.15-5.2.el8.src.rpm", 201294, "LGPLv2+", "CentOS", "", "A library handling all the details of proxy configuration", "fdff6157401876336cd97ea9b89179c1"},
			{intRef(), "cockpit-bridge", "211.3", "1.el8", "x86_64", "cockpit-211.3-1.el8.src.rpm", 1062966, "LGPLv2+", "CentOS", "", "Cockpit bridge server-side component", "a0fef54235e1f3e7b8a0daed5fb74a97"},
			{intRef(), "libappstream-glib", "0.7.14", "3.el8", "x86_64", "libappstream-glib-0.7.14-3.el8.src.rpm", 1072064, "LGPLv2+", "CentOS", "", "Library for AppStream metadata", "481422c7bbc789282fcd1fc99baeaa09"},
			{intRef(), "libdaemon", "0.14", "15.el8", "x86_64", "libdaemon-0.14-15.el8.src.rpm", 64469, "LGPLv2+", "CentOS", "", "Library for writing UNIX daemons", "56a6f57270c27db065ca9ec72986bbe1"},
			{intRef(), "fuse3-libs", "3.2.1", "12.el8", "x86_64", "fuse-2.9.7-12.el8.src.rpm", 285282, "LGPLv2+", "CentOS", "", "File System in Userspace (FUSE) v3 libraries", "b5341d342ba87f98eaabdd5ef133f4c9"},
			{intRef(1), "containers-common", "0.1.40", "11.module_el8.2.0+377+92552693", "x86_64", "skopeo-0.1.40-11.module_el8.2.0+377+92552693.src.rpm", 44763, "ASL 2.0", "CentOS", "container-tools:rhel8:8020020200603213325:0d58ad57", "Configuration files for working with image signatures", "5179ea94e6969bdd1651ac7a60f0f442"},
			{intRef(), "policycoreutils", "2.9", "9.el8", "x86_64", "policycoreutils-2.9-9.el8.src.rpm", 672959, "GPLv2", "CentOS", "", "SELinux policy core utilities", "662e8cb5f32acb3c63d31826f4cda4d5"},
			{intRef(), "selinux-policy", "3.14.3", "41.el8_2.8", "noarch", "selinux-policy-3.14.3-41.el8_2.8.src.rpm", 24923, "GPLv2+", "CentOS", "", "SELinux policy configuration", "c63080adf6f87df7bd9010caa046a2aa"},
			{intRef(), "dbus-glib", "0.110", "2.el8", "x86_64", "dbus-glib-0.110-2.el8.src.rpm", 367078, "AFL and GPLv2+", "CentOS", "", "GLib bindings for D-Bus", "c0b0c110491e9c33bd9b80f79d142cad"},
			{intRef(), "python3-slip-dbus", "0.6.4", "11.el8", "noarch", "python-slip-0.6.4-11.el8.src.rpm", 71273, "GPLv2+", "CentOS", "", "Convenience functions for dbus services in Python 3.x", "7a2ef135f3203c60c116876c5f694d30"},
			{intRef(), "python3-policycoreutils", "2.9", "9.el8", "noarch", "policycoreutils-2.9-9.el8.src.rpm", 5632728, "GPLv2", "CentOS", "", "SELinux policy core python3 interfaces", "909d7eda3e2c0d870465b8eadae59409"},
			{intRef(2), "container-selinux", "2.124.0", "1.module_el8.2.0+305+5e198a41", "noarch", "container-selinux-2.124.0-1.module_el8.2.0+305+5e198a41.src.rpm", 45036, "GPLv2", "CentOS", "container-tools:rhel8:8020020200507003530:0d58ad57", "SELinux policies for container runtimes", "531384de780793a28141f1e1b8f0ec15"},
			{intRef(1), "NetworkManager", "1.22.8", "5.el8_2", "x86_64", "NetworkManager-1.22.8-5.el8_2.src.rpm", 6568601, "GPLv2+ and LGPLv2+", "CentOS", "", "Network connection manager and user applications", "a5c33b6d79b38d356a126f4acb788c2d"},
			{intRef(), "python3-systemd", "234", "8.el8", "x86_64", "python-systemd-234-8.el8.src.rpm", 259121, "LGPLv2+", "CentOS", "", "Python module wrapping systemd functionality", "e9933a06b9ba596ce84034e4e2acb5a0"},
			{intRef(), "protobuf-c", "1.3.0", "4.el8", "x86_64", "protobuf-c-1.3.0-4.el8.src.rpm", 57650, "BSD", "CentOS", "", "C bindings for Google's Protocol Buffers", "fe484866da911be329826bbb2ab54a14"},
			{intRef(), "libnet", "1.1.6", "15.el8", "x86_64", "libnet-1.1.6-15.el8.src.rpm", 174574, "BSD", "CentOS", "", "C library for portable packet creation and injection", "0e2889323e1240042193c9f2888d12a3"},
			{intRef(), "flex", "2.6.1", "9.el8", "x86_64", "flex-2.6.1-9.el8.src.rpm", 931867, "BSD and LGPLv2+", "CentOS", "", "A tool for creating scanners (text pattern recognizers)", "e72058ebaebda1c37c2e6761b582a0ea"},
			{intRef(), "rpm-sign", "4.14.2", "37.el8", "x86_64", "rpm-4.14.2-37.el8.src.rpm", 18996, "GPLv2+", "CentOS", "", "Package signing support", "cfb9864a7f6db67b51a5ea7912ba9ae5"},
			{intRef(), "diffstat", "1.61", "7.el8", "x86_64", "diffstat-1.61-7.el8.src.rpm", 65815, "MIT", "CentOS", "", "A utility which provides statistics based on the output of diff", "f1cde6823c3d9c3679bada1e9fc1e9a8"},
			{intRef(), "pkgconf-pkg-config", "1.4.2", "1.el8", "x86_64", "pkgconf-1.4.2-1.el8.src.rpm", 3094, "ISC", "CentOS", "", "pkgconf shim to provide /usr/bin/pkg-config", "6a5f9694de9b98ca71ed47b605ba977a"},
			{intRef(), "boost-regex", "1.66.0", "7.el8", "x86_64", "boost-1.66.0-7.el8.src.rpm", 1185250, "Boost and MIT and Python", "CentOS", "", "Run-time component of boost regular expression library", "ba6f0f94829f05645542848fe3cc6374"},
			{intRef(), "openssh-clients", "8.0p1", "4.el8_1", "x86_64", "openssh-8.0p1-4.el8_1.src.rpm", 3567696, "BSD", "CentOS", "", "An open source SSH client applications", "78ac3735eebc747b931cc7712cc0e46f"},
			{intRef(), "git-core-doc", "2.18.4", "2.el8_2", "noarch", "git-2.18.4-2.el8_2.src.rpm", 11778148, "GPLv2", "CentOS", "", "Documentation files for git-core", "b5e58ac9279ac3502dc6f53feecb64c1"},
			{intRef(), "glibc-headers", "2.28", "101.el8", "x86_64", "glibc-2.28-101.el8.src.rpm", 2029912, "LGPLv2+ and LGPLv2+ with exceptions and GPLv2+ and GPLv2+ with exceptions and BSD and Inner-Net and ISC and Public Domain and GFDL", "CentOS", "", "Header files for development using standard C libraries.", "14929ccf48957fe03d69f587efe9d942"},
			{intRef(), "glibc-devel", "2.28", "101.el8", "x86_64", "glibc-2.28-101.el8.src.rpm", 1273916, "LGPLv2+ and LGPLv2+ with exceptions and GPLv2+ and GPLv2+ with exceptions and BSD and Inner-Net and ISC and Public Domain and GFDL", "CentOS", "", "Object files for development using standard C libraries.", "e522ba17ee3f0d9e73ba581ec0dbf5d9"},
			{intRef(4), "perl-Encode", "2.97", "3.el8", "x86_64", "perl-Encode-2.97-3.el8.src.rpm", 10200695, "(GPL+ or Artistic) and Artistic 2.0 and UCD", "CentOS", "", "Character encodings in Perl", "9c13163ba1dba6de88e23119fbbb07c7"},
			{intRef(1), "perl-Getopt-Long", "2.50", "4.el8", "noarch", "perl-Getopt-Long-2.50-4.el8.src.rpm", 139724, "GPLv2+ or Artistic", "CentOS", "", "Extended processing of command line options", "8402f4c723f38564bab926f618d4a638"},
			{intRef(4), "perl-Pod-Usage", "1.69", "395.el8", "noarch", "perl-Pod-Usage-1.69-395.el8.src.rpm", 49721, "GPL+ or Artistic", "CentOS", "", "Print a usage message from embedded POD documentation", "7ca3785d080ee4a20cb030cff8d14650"},
			{intRef(), "perl-HTTP-Tiny", "0.074", "1.el8", "noarch", "perl-HTTP-Tiny-0.074-1.el8.src.rpm", 149162, "GPL+ or Artistic", "CentOS", "", "Small, simple, correct HTTP/1.1 client", "450794533f6287021ea470ccdb71a39a"},
			{intRef(), "perl-libnet", "3.11", "3.el8", "noarch", "perl-libnet-3.11-3.el8.src.rpm", 277135, "GPL+ or Artistic", "CentOS", "", "Perl clients for various network protocols", "84cf9f5f0d65cd76fdc6916aa342c1f3"},
			{intRef(), "autoconf", "2.69", "27.el8", "noarch", "autoconf-2.69-27.el8.src.rpm", 2323003, "GPLv2+ and GFDL", "CentOS", "", "A GNU tool for automatically configuring source code", "9c5483a0a68d7fd73aec444ac2a9d631"},
			{intRef(), "perl-Git", "2.18.4", "2.el8_2", "noarch", "git-2.18.4-2.el8_2.src.rpm", 64807, "GPLv2", "CentOS", "", "Perl interface to Git", "1186b063cb093bd7e40f35d6d6d2edd5"},
			{intRef(), "gettext-common-devel", "0.19.8.1", "17.el8", "noarch", "gettext-0.19.8.1-17.el8.src.rpm", 397912, "GPLv3+", "CentOS", "", "Common development files for gettext", "0383d8a86aba24f71729c918d949d967"},
			{intRef(), "bzip2", "1.0.6", "26.el8", "x86_64", "bzip2-1.0.6-26.el8.src.rpm", 93460, "BSD", "CentOS", "", "A file compression utility", "3a97e3a32e5b761e2596c25ba1185da3"},
			{intRef(2), "vim-filesystem", "8.0.1763", "13.el8", "noarch", "vim-8.0.1763-13.el8.src.rpm", 40, "Vim and MIT", "CentOS", "", "VIM filesystem layout", "747cf7810d58760c910bf51f0fae0722"},
			{intRef(), "tbb", "2018.2", "9.el8", "x86_64", "tbb-2018.2-9.el8.src.rpm", 470470, "ASL 2.0", "CentOS", "", "The Threading Building Blocks library abstracts low-level threading details", "010463610cbe8b7daf3b87f767aec59f"},
			{intRef(), "qt5-srpm-macros", "5.12.5", "3.el8", "noarch", "qt5-5.12.5-3.el8.src.rpm", 0, "GPLv3", "CentOS", "", "RPM macros for source Qt5 packages", "0bc556815ac52e15d6781ce597c4b96b"},
			{intRef(), "python-srpm-macros", "3", "38.el8", "noarch", "python-rpm-macros-3-38.el8.src.rpm", 3747, "MIT", "CentOS", "", "RPM macros for building Python source packages", "45706b5613b15f635784edf97af6c60d"},
			{intRef(), "openjpeg2", "2.3.1", "6.el8", "x86_64", "openjpeg2-2.3.1-6.el8.src.rpm", 356438, "BSD and MIT", "CentOS", "", "C-Library for JPEG 2000", "fd031a9e20ff4c4add143f023808cf9c"},
			{intRef(), "ocaml-srpm-macros", "5", "4.el8", "noarch", "ocaml-srpm-macros-5-4.el8.src.rpm", 737, "GPLv2+", "CentOS", "", "OCaml architecture macros", "287a7361faafcb3820cffcce8958f73b"},
			{intRef(), "copy-jdk-configs", "3.7", "1.el8", "noarch", "copy-jdk-configs-3.7-1.el8.src.rpm", 16774, "BSD", "CentOS", "", "JDKs configuration files copier", "79e5b1b1ef9ba3524430003a0249db16"},
			{intRef(), "libwebp", "1.0.0", "1.el8", "x86_64", "libwebp-1.0.0-1.el8.src.rpm", 860976, "BSD", "CentOS", "", "Library and tools for the WebP graphics format", "ce1657ded05ef1b5513e792902d195d4"},
			{intRef(), "libpaper", "1.1.24", "22.el8", "x86_64", "libpaper-1.1.24-22.el8.src.rpm", 90939, "GPLv2", "CentOS", "", "Library and tools for handling papersize", "4970014f161418aee1c0460e8ce027e7"},
			{intRef(), "mcpp", "2.7.2", "20.el8", "x86_64", "mcpp-2.7.2-20.el8.src.rpm", 53794, "BSD", "CentOS", "", "Alternative C/C++ preprocessor", "b10475c4ce5d13728b96c76f68834d1e"},
			{intRef(), "libijs", "0.35", "5.el8", "x86_64", "libijs-0.35-5.el8.src.rpm", 60640, "AGPLv3+", "CentOS", "", "IJS Raster Image Transport Protocol Library", "86aa14b655428b65d88a68e8b0014c2e"},
			{intRef(1), "xorg-x11-font-utils", "7.5", "40.el8", "x86_64", "xorg-x11-font-utils-7.5-40.el8.src.rpm", 383068, "MIT", "CentOS", "", "X.Org X11 font utilities", "d5bc48bd0c692587d63902ee41027983"},
			{intRef(), "libdatrie", "0.2.9", "7.el8", "x86_64", "libdatrie-0.2.9-7.el8.src.rpm", 62583, "LGPLv2+", "CentOS", "", "Implementation of Double-Array structure for representing trie", "e8238df1f04b3284ea4c6b4a5c1c79ca"},
			{intRef(), "libbabeltrace", "1.5.4", "2.el8", "x86_64", "babeltrace-1.5.4-2.el8.src.rpm", 580059, "MIT and GPLv2", "CentOS", "", "Common Trace Format Babel Tower", "51054f4281a03f7ca4fc6ab166226309"},
			{intRef(), "gc", "7.6.4", "3.el8", "x86_64", "gc-7.6.4-3.el8.src.rpm", 229024, "BSD", "CentOS", "", "A garbage collector for C and C++", "d16a971513963ab4666f0aef05b733b6"},
			{intRef(), "gdb-headless", "8.2", "11.el8", "x86_64", "gdb-8.2-11.el8.src.rpm", 11897980, "GPLv3+ and GPLv3+ with exceptions and GPLv2+ and GPLv2+ with exceptions and GPL+ and LGPLv2+ and LGPLv3+ and BSD and Public Domain and GFDL", "CentOS", "", "A GNU source-level debugger for C, C++, Fortran, Go and other languages", "8f41d147d932e8814bf33bfab71117ec"},
			{intRef(), "libXxf86misc", "1.0.4", "1.el8", "x86_64", "libXxf86misc-1.0.4-1.el8.src.rpm", 32012, "MIT", "CentOS", "", "X.Org X11 libXxf86misc runtime library", "c1728eb43ebd6d96c74380d3ec4c60c1"},
			{intRef(), "urw-base35-bookman-fonts", "20170801", "10.el8", "noarch", "urw-base35-fonts-20170801-10.el8.src.rpm", 1428518, "AGPLv3", "CentOS", "", "URW Bookman font family [part of Level 2 Core Font Set]", "2cbb1a3a39ef1185fe9a193d003951ba"},
			{intRef(), "urw-base35-d050000l-fonts", "20170801", "10.el8", "noarch", "urw-base35-fonts-20170801-10.el8.src.rpm", 86572, "AGPLv3", "CentOS", "", "D050000L font family [part of Level 2 Core Font Set]", "500e4022d7801715bb69cc1f1e1603c3"},
			{intRef(), "urw-base35-nimbus-mono-ps-fonts", "20170801", "10.el8", "noarch", "urw-base35-fonts-20170801-10.el8.src.rpm", 1099749, "AGPLv3", "CentOS", "", "Nimbus Mono PS font family [part of Level 2 Core Font Set]", "c64c1ed1e447e132e92adfd76aa3870e"},
			{intRef(), "urw-base35-nimbus-sans-fonts", "20170801", "10.el8", "noarch", "urw-base35-fonts-20170801-10.el8.src.rpm", 2469317, "AGPLv3", "CentOS", "", "Nimbus Sans font family [part of Level 2 Core Font Set]", "dffcdf61c430e626a40ce66d8ba2fb93"},
			{intRef(), "urw-base35-standard-symbols-ps-fonts", "20170801", "10.el8", "noarch", "urw-base35-fonts-20170801-10.el8.src.rpm", 45184, "AGPLv3", "CentOS", "", "Standard Symbols PS font family [part of Level 2 Core Font Set]", "3d036fbd5ec99facbd7416bcff86c58f"},
			{intRef(), "urw-base35-fonts", "20170801", "10.el8", "noarch", "urw-base35-fonts-20170801-10.el8.src.rpm", 5478, "AGPLv3", "CentOS", "", "Core Font Set containing 35 freely distributable fonts from (URW)++", "90b579587fc764dac0351a2f2e04f3c4"},
			{intRef(), "libXcomposite", "0.4.4", "14.el8", "x86_64", "libXcomposite-0.4.4-14.el8.src.rpm", 35952, "MIT", "CentOS", "", "X Composite Extension library", "994c8bec984cc544058db44ccad111b2"},
			{intRef(), "jbigkit-libs", "2.1", "14.el8", "x86_64", "jbigkit-2.1-14.el8.src.rpm", 109677, "GPLv2+", "CentOS", "", "JBIG1 lossless image compression library", "c63a249918e68c8fb851261731080349"},
			{intRef(), "gd", "2.2.5", "6.el8", "x86_64", "gd-2.2.5-6.el8.src.rpm", 441251, "MIT", "CentOS", "", "A graphics library for quick creation of PNG or JPEG images", "9c15a09e74685686b18d9c0d74d66a40"},
			{intRef(), "jbig2dec-libs", "0.14", "4.el8_2", "x86_64", "jbig2dec-0.14-4.el8_2.src.rpm", 146739, "GPLv2", "CentOS", "", "A decoder implementation of the JBIG2 image compression format", "1d39c56c6e36e5e6817edd9ade7d8ccb"},
			{intRef(), "gcc", "8.3.1", "5.el8.0.2", "x86_64", "gcc-8.3.1-5.el8.0.2.src.rpm", 61698975, "GPLv3+ and GPLv3+ with exceptions and GPLv2+ with exceptions and LGPLv2+ and BSD", "CentOS", "", "Various compilers (C, C++, Objective-C, ...)", "3380ab4e7a454e71dc39e99c060dfcda"},
			{intRef(), "gcc-gdb-plugin", "8.3.1", "5.el8.0.2", "x86_64", "gcc-8.3.1-5.el8.0.2.src.rpm", 341891, "GPLv3+ and GPLv3+ with exceptions and GPLv2+ with exceptions and LGPLv2+ and BSD", "CentOS", "", "GCC plugin for GDB", "a8c49d042ac44224cc64b62f7abe7c09"},
			{intRef(), "hicolor-icon-theme", "0.17", "2.el8", "noarch", "hicolor-icon-theme-0.17-2.el8.src.rpm", 73932, "GPLv2+", "CentOS", "", "Basic requirement for icon themes", "7fbb3efb70b3a17ca40662e89407d14b"},
			{intRef(), "graphite2", "1.3.10", "10.el8", "x86_64", "graphite2-1.3.10-10.el8.src.rpm", 270396, "(LGPLv2+ or GPLv2+ or MPL) and (Netscape or GPLv2+ or LGPLv2+)", "CentOS", "", "Font rendering capabilities for complex non-Roman writing systems", "8bd9e8164c9b128b70bb6b6a28ecc04e"},
			{intRef(), "google-droid-sans-fonts", "20120715", "13.el8", "noarch", "google-droid-fonts-20120715-13.el8.src.rpm", 6229278, "ASL 2.0", "CentOS", "", "A humanist sans serif typeface", "ae5337d5ef62351fcad39143150dd6bf"},
			{intRef(), "ghc-srpm-macros", "1.4.2", "7.el8", "noarch", "ghc-srpm-macros-1.4.2-7.el8.src.rpm", 414, "GPLv2+", "CentOS", "", "RPM macros for building Haskell source packages", "0c2ffae7ef28a3e4ad323b8bf7838fc2"},
			{intRef(), "pango", "1.42.4", "6.el8", "x86_64", "pango-1.42.4-6.el8.src.rpm", 958244, "LGPLv2+", "CentOS", "", "System for layout and rendering of internationalized text", "531f8754411e2ac78d581288aeb362a4"},
			{intRef(), "librsvg2", "2.42.7", "3.el8", "x86_64", "librsvg2-2.42.7-3.el8.src.rpm", 1830479, "LGPLv2+", "CentOS", "", "An SVG library based on cairo", "9044e527cfe93f742fe2a036af87e085"},
			{intRef(), "elfutils", "0.178", "7.el8", "x86_64", "elfutils-0.178-7.el8.src.rpm", 2835784, "GPLv3+ and (GPLv2+ or LGPLv3+) and GFDL", "CentOS", "", "A collection of utilities and DSOs to handle ELF files and DWARF data", "8b87d360b4ffd7df560785e5703dee4f"},
			{intRef(), "dwz", "0.12", "9.el8", "x86_64", "dwz-0.12-9.el8.src.rpm", 232103, "GPLv2+ and GPLv3+", "CentOS", "", "DWARF optimization and duplicate removal tool", "266fad1920ff941eb837f3c7e4dc7f13"},
			{intRef(), "ctags", "5.8", "22.el8", "x86_64", "ctags-5.8-22.el8.src.rpm", 417116, "GPLv2+ and LGPLv2+ and Public Domain", "CentOS", "", "A C programming language indexing and/or cross-reference tool", "a87d90ee8142a2da4af1f660bda3c9a4"},
			{intRef(), "boost-date-time", "1.66.0", "7.el8", "x86_64", "boost-1.66.0-7.el8.src.rpm", 79610, "Boost and MIT and Python", "CentOS", "", "Run-time component of boost date-time library", "d1dea33d84265129c2ad02ff8422f1ea"},
			{intRef(), "dyninst", "10.1.0", "4.el8", "x86_64", "dyninst-10.1.0-4.el8.src.rpm", 14686061, "LGPLv2+", "CentOS", "", "An API for Run-time Code Generation", "07f0fe3fb0c74973a2ee2958b489f883"},
			{intRef(), "systemtap-client", "4.2", "6.el8", "x86_64", "systemtap-4.2-6.el8.src.rpm", 11184527, "GPLv2+", "CentOS", "", "Programmable system-wide instrumentation system - client", "51dee002b452dfcf0290f8a8a01dfb4c"},
			{intRef(), "libgs", "9.25", "5.el8_1.1", "x86_64", "ghostscript-9.25-5.el8_1.1.src.rpm", 20878358, "AGPLv3+", "CentOS", "", "Library providing Ghostcript's core functionality", "2fc9a1af1b846b7f37bcd8f7f8600c7f"},
			{intRef(), "asciidoc", "8.6.10", "0.5.20180627gitf7c2274.el8", "noarch", "asciidoc-8.6.10-0.5.20180627gitf7c2274.el8.src.rpm", 808989, "GPL+ and GPLv2+", "CentOS", "", "Text based document generation", "4c89126d49351c94d2800206a5f395d4"},
			{intRef(), "rpm-build", "4.14.2", "37.el8", "x86_64", "rpm-4.14.2-37.el8.src.rpm", 299895, "GPLv2+", "CentOS", "", "Scripts and executable programs used to build packages", "d18977dc69c214825a0d5f1a02357f0d"},
			{intRef(), "gcc-c++", "8.3.1", "5.el8.0.2", "x86_64", "gcc-8.3.1-5.el8.0.2.src.rpm", 32170609, "GPLv3+ and GPLv3+ with exceptions and GPLv2+ with exceptions and LGPLv2+ and BSD", "CentOS", "", "C++ support for GCC", "60a4ffca0b458dca68f713a67aaec773"},
			{intRef(), "jna", "4.5.1", "5.el8", "x86_64", "jna-4.5.1-5.el8.src.rpm", 416128, "(LGPLv2 or ASL 2.0) and ASL 2.0", "CentOS", "", "Pure Java access to native libraries", "4fd2761d587cf99ca78b74e969e77bd5"},
			{intRef(1), "valgrind-devel", "3.15.0", "11.el8", "x86_64", "valgrind-3.15.0-11.el8.src.rpm", 474728, "GPLv2+", "CentOS", "", "Development files for valgrind aware programs", "76a96e6804285186ff92d60bfeaf5c47"},
			{intRef(), "pesign", "0.112", "25.el8", "x86_64", "pesign-0.112-25.el8.src.rpm", 1060595, "GPLv2", "CentOS", "", "Signing utility for UEFI binaries", "f4491a847d7be47db09614cb0226c3a1"},
			{intRef(), "bison", "3.0.4", "10.el8", "x86_64", "bison-3.0.4-10.el8.src.rpm", 2199622, "GPLv3+", "CentOS", "", "A GNU general-purpose parser generator", "9ecc659af7d30ee806bcaf9e9bacb006"},
			{intRef(), "runc", "1.0.0", "65.rc10.module_el8.2.0+305+5e198a41", "x86_64", "runc-1.0.0-65.rc10.module_el8.2.0+305+5e198a41.src.rpm", 10250793, "ASL 2.0", "CentOS", "container-tools:rhel8:8020020200507003530:0d58ad57", "CLI for running Open Containers", "4d92762146537e6954ae16a4aa312bda"},
			{intRef(), "python3-setuptools-wheel", "39.2.0", "5.el8", "noarch", "python-setuptools-39.2.0-5.el8.src.rpm", 347696, "MIT", "CentOS", "", "The setuptools wheel", "86b613f3e1fbf33755d1b34d3e1cb098"},
			{intRef(), "ltrace", "0.7.91", "28.el8", "x86_64", "ltrace-0.7.91-28.el8.src.rpm", 339426, "GPLv2+", "CentOS", "", "Tracks runtime library calls from dynamically linked executables", "9c613ec34bce83ea49ba584736ce5f69"},
			{intRef(), "libX11", "1.6.8", "3.el8", "x86_64", "libX11-1.6.8-3.el8.src.rpm", 1343952, "MIT", "CentOS", "", "Core X11 protocol client library", "b9d3598f257da1a2c32400472e6563c8"},
			{intRef(), "centos-repos", "8.2", "2.2004.0.1.el8", "x86_64", "centos-release-8.2-2.2004.0.1.el8.src.rpm", 9660, "GPLv2", "CentOS", "", "CentOS Linux package repositories", "7b5da31594f465b7bc3a6a027cb297dd"},
			{intRef(), "cairo-gobject", "1.15.12", "3.el8", "x86_64", "cairo-1.15.12-3.el8.src.rpm", 36928, "LGPLv2 or MPLv1.1", "CentOS", "", "GObject bindings for cairo", "017b0cb9b1022342c47dcaf7dbe31960"},
			{intRef(), "tzdata", "2020a", "1.el8", "noarch", "tzdata-2020a-1.el8.src.rpm", 1904256, "Public Domain", "CentOS", "", "Timezone data", "041caaf2c8e98a0493cc1011b1d4e12b"},
			{intRef(), "setroubleshoot-server", "3.3.22", "2.el8", "x86_64", "setroubleshoot-3.3.22-2.el8.src.rpm", 1347731, "GPLv2+", "CentOS", "", "SELinux troubleshoot server", "208ed008e86409465b3112998aa3780d"},
			{intRef(), "ncurses-libs", "6.1", "7.20180224.el8", "x86_64", "ncurses-6.1-7.20180224.el8.src.rpm", 1120040, "MIT", "CentOS", "", "Ncurses libraries", "e22fb3c519b0d6c0b1544ade616647a1"},
			{intRef(), "podman", "1.6.4", "10.module_el8.2.0+305+5e198a41", "x86_64", "podman-1.6.4-10.module_el8.2.0+305+5e198a41.src.rpm", 57521087, "ASL 2.0", "CentOS", "container-tools:rhel8:8020020200507003530:0d58ad57", "Manage Pods, Containers and Container Images", "bc16fa130090c5c2fd59e7a25b946084"},
			{intRef(), "bash", "4.4.19", "10.el8", "x86_64", "bash-4.4.19-10.el8.src.rpm", 6930068, "GPLv3+", "CentOS", "", "The GNU Bourne Again shell", "8b1a1d644da5e718f7a1cd76769e3f83"},
			{intRef(1), "skopeo", "0.1.40", "11.module_el8.2.0+377+92552693", "x86_64", "skopeo-0.1.40-11.module_el8.2.0+377+92552693.src.rpm", 22732055, "ASL 2.0", "CentOS", "container-tools:rhel8:8020020200603213325:0d58ad57", "Inspect container images and repositories on registries", "31f63884caef684cd779b1230a5e6074"},
			{intRef(), "bzip2-libs", "1.0.6", "26.el8", "x86_64", "bzip2-1.0.6-26.el8.src.rpm", 77229, "BSD", "CentOS", "", "Libraries for applications using bzip2", "dd9abf1cc65154142a808c839f4dda55"},
			{intRef(), "perl-Exporter", "5.72", "396.el8", "noarch", "perl-Exporter-5.72-396.el8.src.rpm", 55719, "GPL+ or Artistic", "CentOS", "", "Implements default import method for modules", "4f56f0f2098313624fdcd567a5c644fc"},
			{intRef(), "elfutils-libelf", "0.178", "7.el8", "x86_64", "elfutils-0.178-7.el8.src.rpm", 920699, "GPLv2+ or LGPLv3+", "CentOS", "", "Library to read and write ELF files", "cd8b135e1df7ffbfb140b41ab26f87aa"},
			{intRef(), "nspr", "4.25.0", "2.el8_2", "x86_64", "nspr-4.25.0-2.el8_2.src.rpm", 317302, "MPLv2.0", "CentOS", "", "Netscape Portable Runtime", "1a03fabf3824c5fa596f3160a89d3aa1"},
			{intRef(), "sqlite-libs", "3.26.0", "6.el8", "x86_64", "sqlite-3.26.0-6.el8.src.rpm", 1162241, "Public Domain", "CentOS", "", "Shared library for the sqlite3 embeddable SQL database engine.", "123153073f9818beb07847d632f6bbb7"},
			{intRef(), "libjpeg-turbo", "1.5.3", "10.el8", "x86_64", "libjpeg-turbo-1.5.3-10.el8.src.rpm", 638420, "IJG", "CentOS", "", "A MMX/SSE2/SIMD accelerated library for manipulating JPEG image files", "62d877376edaea71b13fa70dbb4537dc"},
			{intRef(), "popt", "1.16", "14.el8", "x86_64", "popt-1.16-14.el8.src.rpm", 128374, "MIT", "CentOS", "", "C library for parsing command line parameters", "91f1fc1e5257f088cf83fcde67e1597f"},
			{intRef(), "perl-Text-ParseWords", "3.30", "395.el8", "noarch", "perl-Text-ParseWords-3.30-395.el8.src.rpm", 13101, "GPL+ or Artistic", "CentOS", "", "Parse text into an array of tokens or array of arrays", "bf3b84c83bffa76bd3af79141fe72394"},
			{intRef(), "json-c", "0.13.1", "0.2.el8", "x86_64", "json-c-0.13.1-0.2.el8.src.rpm", 73898, "MIT", "CentOS", "", "JSON implementation in C", "eb194624bd2d5fd5966ea9a6927525c7"},
			{intRef(1), "emacs-filesystem", "26.1", "5.el8", "noarch", "emacs-26.1-5.el8.src.rpm", 0, "GPLv3+ and CC0-1.0", "CentOS", "", "Emacs filesystem layout", "94218736a9d0203615311c0eb679b00a"},
			{intRef(), "sed", "4.5", "1.el8", "x86_64", "sed-4.5-1.el8.src.rpm", 776854, "GPLv3+", "CentOS", "", "A GNU stream text editor", "3ddec62cedf49c241d6ab77613daaf0a"},
			{intRef(), "libXpm", "3.5.12", "8.el8", "x86_64", "libXpm-3.5.12-8.el8.src.rpm", 120823, "MIT", "CentOS", "", "X.Org X11 libXpm runtime library", "f4896be49953be1b482184c8cb9f6e8c"},
			{intRef(), "audit-libs", "3.0", "0.17.20191104git1c2f876.el8", "x86_64", "audit-3.0-0.17.20191104git1c2f876.el8.src.rpm", 283708, "LGPLv2+", "CentOS", "", "Dynamic library for libaudit", "85d66fe7059cab257dff0b1a99e59479"},
			{intRef(), "zip", "3.0", "23.el8", "x86_64", "zip-3.0-23.el8.src.rpm", 842885, "BSD", "CentOS", "", "A file compression and packaging utility compatible with PKZIP", "100278a94f1f36e5918a09b79dbfa0a1"},
			{intRef(), "lua-libs", "5.3.4", "11.el8", "x86_64", "lua-5.3.4-11.el8.src.rpm", 351728, "MIT", "CentOS", "", "Libraries for lua", "3ec414ac7e8a8f90968c86da820336cb"},
			{intRef(), "boost-chrono", "1.66.0", "7.el8", "x86_64", "boost-1.66.0-7.el8.src.rpm", 38826, "Boost and MIT and Python", "CentOS", "", "Run-time component of boost chrono library", "7dfaa398e6e340e60522782a24ff012d"},
			{intRef(), "gzip", "1.9", "9.el8", "x86_64", "gzip-1.9-9.el8.src.rpm", 426515, "GPLv3+ and GFDL", "CentOS", "", "The GNU data compression program", "d540a593d8d1a3e01a685dbe12294a37"},
			{intRef(), "perl-Text-Tabs+Wrap", "2013.0523", "395.el8", "noarch", "perl-Text-Tabs+Wrap-2013.0523-395.el8.src.rpm", 24825, "TTWL", "CentOS", "", "Expand tabs and do simple line wrapping", "68fe36e969406f0e90c6ee13b59b03e3"},
			{intRef(), "libassuan", "2.5.1", "3.el8", "x86_64", "libassuan-2.5.1-3.el8.src.rpm", 202763, "LGPLv2+ and GPLv3+", "CentOS", "", "GnuPG IPC library", "44d9976e6073475aebe11bef3ff8dc33"},
			{intRef(), "perl-PathTools", "3.74", "1.el8", "x86_64", "perl-PathTools-3.74-1.el8.src.rpm", 182821, "(GPL+ or Artistic) and BSD", "CentOS", "", "PathTools Perl module (Cwd, File::Spec)", "7cbc1c39bf3c5c7425bf73e5235d00d3"},
			{intRef(), "libtasn1", "4.13", "3.el8", "x86_64", "libtasn1-4.13-3.el8.src.rpm", 168725, "GPLv3+ and LGPLv2+", "CentOS", "", "The ASN.1 library used in GNUTLS", "6fb9739221265fd08745407c915f792d"},
			{intRef(4), "perl-interpreter", "5.26.3", "416.el8", "x86_64", "perl-5.26.3-416.el8.src.rpm", 14381705, "(GPL+ or Artistic) and (GPLv2+ or Artistic) and BSD and Public Domain and UCD", "CentOS", "", "Standalone executable Perl interpreter", "e6124e5f67e224ff12cd136480af150a"},
			{intRef(), "grep", "3.1", "6.el8", "x86_64", "grep-3.1-6.el8.src.rpm", 835205, "GPLv3+", "CentOS", "", "Pattern matching utilities", "94ed80f88be0c0b77082a51035c5e53f"},
			{intRef(), "perl-Data-Dumper", "2.167", "399.el8", "x86_64", "perl-Data-Dumper-2.167-399.el8.src.rpm", 106523, "GPL+ or Artistic", "CentOS", "", "Stringify perl data structures, suitable for printing and eval", "e7ff09d4e428471fd85587fbe9260fca"},
			{intRef(12), "dhcp-libs", "4.3.6", "40.el8", "x86_64", "dhcp-4.3.6-40.el8.src.rpm", 161256, "ISC", "CentOS", "", "Shared libraries used by ISC dhcp client and server", "1f247bfb2d46e59b1fe552896cfcf0d9"},
			{intRef(), "libtool-ltdl", "2.4.6", "25.el8", "x86_64", "libtool-2.4.6-25.el8.src.rpm", 71434, "LGPLv2+", "CentOS", "", "Runtime libraries for GNU Libtool Dynamic Module Loader", "e5553e445ddba10c79d2f430439f24cb"},
			{intRef(), "xz", "5.2.4", "3.el8", "x86_64", "xz-5.2.4-3.el8.src.rpm", 432832, "GPLv2+ and Public Domain", "CentOS", "", "LZMA compression utilities", "2acee73f3a4b9738e0133043726a13f7"},
			{intRef(), "libXinerama", "1.1.4", "1.el8", "x86_64", "libXinerama-1.1.4-1.el8.src.rpm", 15719, "MIT", "CentOS", "", "X.Org X11 libXinerama runtime library", "a58f333735db17b93d106a10499d219c"},
			{intRef(2), "shadow-utils", "4.6", "8.el8", "x86_64", "shadow-utils-4.6-8.el8.src.rpm", 5368080, "BSD and GPLv2+", "CentOS", "", "Utilities for managing accounts and shadow password files", "211de7a014813ec64ce2bb8800b336b6"},
			{intRef(), "adobe-mappings-cmap", "20171205", "3.el8", "noarch", "adobe-mappings-cmap-20171205-3.el8.src.rpm", 13746679, "BSD", "CentOS", "", "CMap resources for Adobe's character collections", "5904158c62f46bb518f6445e49044e60"},
			{intRef(), "mpfr", "3.1.6", "1.el8", "x86_64", "mpfr-3.1.6-1.el8.src.rpm", 612625, "LGPLv3+ and GPLv3+ and GFDL", "CentOS", "", "A C library for multiple-precision floating-point computations", "7504c4f1b7aab1cf22f4896a6c35e91f"},
			{intRef(), "perl-Digest-MD5", "2.55", "396.el8", "x86_64", "perl-Digest-MD5-2.55-396.el8.src.rpm", 56718, "(GPL+ or Artistic) and BSD", "CentOS", "", "Perl interface to the MD5 algorithm", "baf4430def945ca345c13bd147f60e7c"},
			{intRef(), "libmetalink", "0.1.3", "7.el8", "x86_64", "libmetalink-0.1.3-7.el8.src.rpm", 76960, "MIT", "CentOS", "", "Metalink library written in C", "a97ac81965ec113b881058f001c770f6"},
			{intRef(), "perl-XML-Parser", "2.44", "11.el8", "x86_64", "perl-XML-Parser-2.44-11.el8.src.rpm", 643439, "GPL+ or Artistic", "CentOS", "", "Perl module for parsing XML documents", "4bd2f5451a8bca4aff405aafb164b0c2"},
			{intRef(2), "ethtool", "5.0", "2.el8", "x86_64", "ethtool-5.0-2.el8.src.rpm", 502623, "GPLv2", "CentOS", "", "Settings tool for Ethernet NICs", "ec93a9c27813a31ebce90966632b8edd"},
			{intRef(), "perl-Thread-Queue", "3.13", "1.el8", "noarch", "perl-Thread-Queue-3.13-1.el8.src.rpm", 29787, "GPL+ or Artistic", "CentOS", "", "Thread-safe queues", "6f6fe2a2414cdc6fe4ef25c47b079c62"},
			{intRef(14), "libpcap", "1.9.0", "3.el8", "x86_64", "libpcap-1.9.0-3.el8.src.rpm", 424251, "BSD with advertising", "CentOS", "", "A system-independent interface for user-level packet capture", "4d2a5ad234af8d3ebe34fc032aa8dfdd"},
			{intRef(), "gettext", "0.19.8.1", "17.el8", "x86_64", "gettext-0.19.8.1-17.el8.src.rpm", 5412553, "GPLv3+ and LGPLv2+", "CentOS", "", "GNU libraries and utilities for producing multi-lingual messages", "b0c5f450b122286eb0d173e9c1aaa23d"},
			{intRef(), "gawk", "4.2.1", "1.el8", "x86_64", "gawk-4.2.1-1.el8.src.rpm", 2717614, "GPLv3+ and GPLv2+ and LGPLv2+ and BSD", "CentOS", "", "The GNU version of the AWK text processing utility", "c47002ae2e1dddb7747b3a6e813e821d"},
			{intRef(), "boost-filesystem", "1.66.0", "7.el8", "x86_64", "boost-1.66.0-7.el8.src.rpm", 113690, "Boost and MIT and Python", "CentOS", "", "Run-time component of boost filesystem library", "567afe4cd00c330378e08ab25d56a087"},
			{intRef(), "krb5-libs", "1.17", "18.el8", "x86_64", "krb5-1.17-18.el8.src.rpm", 2259532, "MIT", "CentOS", "", "The non-admin shared libraries used by Kerberos 5", "800e946bba1c5c102ccdb4159c046ce6"},
			{intRef(), "nss-softokn", "3.53.1", "11.el8_2", "x86_64", "nss-3.53.1-11.el8_2.src.rpm", 1917766, "MPLv2.0", "CentOS", "", "Network Security Services Softoken Module", "0245449c679c87662bab54d4f57b8956"},
			{intRef(), "platform-python", "3.6.8", "23.el8", "x86_64", "python3-3.6.8-23.el8.src.rpm", 41790, "Python", "CentOS", "", "Internal interpreter of the Python programming language", "5674cc4a141996200466adabc7e51f5c"},
			{intRef(), "perl-Fedora-VSP", "0.001", "9.el8", "noarch", "perl-Fedora-VSP-0.001-9.el8.src.rpm", 40886, "GPLv3+", "CentOS", "", "Perl version normalization for RPM", "b411c33d7f444d300f5d6f2e04b8a647"},
			{intRef(), "pam", "1.3.1", "8.el8", "x86_64", "pam-1.3.1-8.el8.src.rpm", 2857052, "BSD and GPLv2+", "CentOS", "", "An extensible library which provides authentication for applications", "f83fb370c1196ddc5e979ddf009b40b2"},
			{intRef(), "openssh", "8.0p1", "4.el8_1", "x86_64", "openssh-8.0p1-4.el8_1.src.rpm", 2255675, "BSD", "CentOS", "", "An open source implementation of SSH protocol version 2", "cbc935bae411018f5812f57975b259f3"},
			{intRef(), "kmod-libs", "25", "16.el8", "x86_64", "kmod-25-16.el8.src.rpm", 126640, "LGPLv2+", "CentOS", "", "Libraries to handle kernel module loading and unloading", "09bde70809bfa4bf789c0ad78f31dc15"},
			{intRef(1), "python3-dateutil", "2.6.1", "6.el8", "noarch", "python-dateutil-2.6.1-6.el8.src.rpm", 596677, "BSD", "CentOS", "", "Powerful extensions to the standard datetime module", "e1f1417da4b0cd49312a7874c640c923"},
			{intRef(), "libcurl-minimal", "7.61.1", "12.el8", "x86_64", "curl-7.61.1-12.el8.src.rpm", 551776, "MIT", "CentOS", "", "Conservatively configured build of libcurl for minimal installations", "97c7f1247354d8a7561b5a26dd184ebb"},
			{intRef(1), "openssl", "1.1.1c", "15.el8", "x86_64", "openssl-1.1.1c-15.el8.src.rpm", 1175523, "OpenSSL", "CentOS", "", "Utilities from the general purpose cryptography library with TLS implementation", "a2d45c750a52617caa18d0f23c80534e"},
			{intRef(), "libdb-utils", "5.3.28", "37.el8", "x86_64", "libdb-5.3.28-37.el8.src.rpm", 536911, "BSD and LGPLv2 and Sleepycat", "CentOS", "", "Command line tools for managing Berkeley DB databases", "e4324aa7441337796a429e9f92455ede"},
			{intRef(), "lksctp-tools", "1.0.18", "3.el8", "x86_64", "lksctp-tools-1.0.18-3.el8.src.rpm", 258226, "GPLv2 and GPLv2+ and LGPLv2 and MIT", "CentOS", "", "User-space access to Linux Kernel SCTP", "45e25b87423e136e109fbf221707e2de"},
			{intRef(), "rpm", "4.14.2", "37.el8", "x86_64", "rpm-4.14.2-37.el8.src.rpm", 2084270, "GPLv2+", "CentOS", "", "The RPM package management system", "89f36b3bdf3dfd109031309fa40d02fc"},
			{intRef(), "pkgconf", "1.4.2", "1.el8", "x86_64", "pkgconf-1.4.2-1.el8.src.rpm", 63271, "ISC", "CentOS", "", "Package compiler and linker metadata toolkit", "c5c251d6f960076fcbae5a6e61fe23d1"},
			{intRef(), "libmodulemd1", "1.8.16", "0.2.8.2.1", "x86_64", "libmodulemd-2.8.2-1.el8.src.rpm", 546039, "MIT", "CentOS", "", "Compatibility package for libmodulemd 1.x", "6b71512e8c92424e2bbf51ac393fb29c"},
			{intRef(), "libedit", "3.1", "23.20170329cvs.el8", "x86_64", "libedit-3.1-23.20170329cvs.el8.src.rpm", 247168, "BSD", "CentOS", "", "The NetBSD Editline library", "bcd179f588923a78be4b41d5ec7a7a39"},
			{intRef(), "librepo", "1.11.0", "2.el8", "x86_64", "librepo-1.11.0-2.el8.src.rpm", 262664, "LGPLv2+", "CentOS", "", "Repodata downloading library", "35df699b2d7ba82e8c358c892968f8a4"},
			{intRef(), "kernel-headers", "4.18.0", "193.28.1.el8_2", "x86_64", "kernel-4.18.0-193.28.1.el8_2.src.rpm", 4958028, "GPLv2 and Redistributable, no modification permitted", "CentOS", "", "Header files for the Linux kernel for use by glibc", "cb027fddb5bfd3e90f75340b4b86a00b"},
			{intRef(), "python3-gpg", "1.10.0", "6.el8.0.1", "x86_64", "gpgme-1.10.0-6.el8.0.1.src.rpm", 1295107, "LGPLv2+", "CentOS", "", "gpgme bindings for Python 3", "f1bb67f41c054b7a241452a9db8c72f4"},
			{intRef(), "groff-base", "1.22.3", "18.el8", "x86_64", "groff-1.22.3-18.el8.src.rpm", 4216866, "GPLv3+ and GFDL and BSD and MIT", "CentOS", "", "Parts of the groff formatting system required to display manual pages", "6ed091baa7cbbcf9205fce2f24fa48cb"},
			{intRef(1), "dbus-common", "1.12.8", "10.el8_2", "noarch", "dbus-1.12.8-10.el8_2.src.rpm", 11131, "(GPLv2+ or AFL) and GPLv2+", "CentOS", "", "D-BUS message bus configuration", "77d77f4770ed1b5442e40d6d8fd10d58"},
			{intRef(), "perl-podlators", "4.11", "1.el8", "noarch", "perl-podlators-4.11-1.el8.src.rpm", 287639, "(GPL+ or Artistic) and FSFAP", "CentOS", "", "Format POD source into various output formats", "bd2b28862f36b3cec0e1d5fb74d5edb7"},
			{intRef(), "cryptsetup-libs", "2.2.2", "1.el8", "x86_64", "cryptsetup-2.2.2-1.el8.src.rpm", 1871402, "GPLv2+ and LGPLv2+", "CentOS", "", "Cryptsetup shared library", "97b1e392c1b2c834b1233239b95bd0cc"},
			{intRef(), "perl-IO-Socket-SSL", "2.066", "4.el8", "noarch", "perl-IO-Socket-SSL-2.066-4.el8.src.rpm", 618705, "(GPL+ or Artistic) and MPLv2.0", "CentOS", "", "Perl library for transparent SSL", "896f66d11d6b66dd0fe2f849b41febed"},
			{intRef(), "systemd", "239", "31.el8_2.2", "x86_64", "systemd-239-31.el8_2.2.src.rpm", 11073343, "LGPLv2+ and MIT and GPLv2+", "CentOS", "", "System and Service Manager", "e8944e79d0bd5f79c2746813db1244ee"},
			{intRef(), "automake", "1.16.1", "6.el8", "noarch", "automake-1.16.1-6.el8.src.rpm", 1805946, "GPLv2+ and GFDL and Public Domain and MIT", "CentOS", "", "A GNU tool for automatically creating Makefiles", "82fb5fbeda4c984163fa56c0d87f88d2"},
			{intRef(), "libkcapi", "1.1.1", "16_1.el8", "x86_64", "libkcapi-1.1.1-16_1.el8.src.rpm", 82828, "BSD or GPLv2", "CentOS", "", "User space interface to the Linux Kernel Crypto API", "d765e8153b4adaa1b9d759650ef97cce"},
			{intRef(), "gettext-devel", "0.19.8.1", "17.el8", "x86_64", "gettext-0.19.8.1-17.el8.src.rpm", 1552453, "LGPLv2+ and GPLv3+", "CentOS", "", "Development files for gettext", "bdc16337577ae7ba9540b00bfd49fe75"},
			{intRef(), "dracut-network", "049", "70.git20200228.el8", "x86_64", "dracut-049-70.git20200228.el8.src.rpm", 160704, "GPLv2+ and LGPLv2+", "CentOS", "", "dracut modules to build a dracut initramfs with network support", "e91224a402dbebc12de2db7fb4747e4a"},
			{intRef(), "tzdata-java", "2020d", "1.el8", "noarch", "tzdata-2020d-1.el8.src.rpm", 374148, "Public Domain", "CentOS", "", "Timezone data for Java", "9833c70520367c6f321b6341c761888e"},
			{intRef(), "python3-dnf", "4.2.17", "7.el8_2", "noarch", "dnf-4.2.17-7.el8_2.src.rpm", 1829655, "GPLv2+ and GPLv2 and GPL", "CentOS", "", "Python 3 interface to DNF", "32a841a42030f8a4768761feb148c477"},
			{intRef(), "python3-rpm-macros", "3", "38.el8", "noarch", "python-rpm-macros-3-38.el8.src.rpm", 1888, "MIT", "CentOS", "", "RPM macros for building Python 3 packages", "8b93d0ccafe759434fb7d7ceb2edf241"},
			{intRef(), "binutils", "2.30", "73.el8", "x86_64", "binutils-2.30-73.el8.src.rpm", 24856745, "GPLv3+", "CentOS", "", "A GNU collection of binary utilities", "f6fcd230323217f591c9f96172dae25f"},
			{intRef(), "openblas-srpm-macros", "2", "2.el8", "noarch", "openblas-srpm-macros-2-2.el8.src.rpm", 104, "MIT", "CentOS", "", "OpenBLAS architecture macros", "43aa3a337edfc79098c8ad423b5e49df"},
			{intRef(), "less", "530", "1.el8", "x86_64", "less-530-1.el8.src.rpm", 344874, "GPLv3+ or BSD", "CentOS", "", "A text file browser similar to more, but better", "c95ece51b133ab6ce794a40e777072d2"},
			{intRef(1), "java-1.8.0-openjdk-headless", "1.8.0.272.b10", "1.el8_2", "x86_64", "java-1.8.0-openjdk-1.8.0.272.b10-1.el8_2.src.rpm", 122536636, "ASL 1.1 and ASL 2.0 and BSD and BSD with advertising and GPL+ and GPLv2 and GPLv2 with exceptions and IJG and LGPLv2+ and MIT and MPLv2.0 and Public Domain and W3C and zlib", "CentOS", "", "OpenJDK Headless Runtime Environment 8", "630fabeef18ed842ccc128308d599f97"},
			{intRef(1), "nodejs-full-i18n", "10.21.0", "3.module_el8.2.0+391+8da3adc6", "x86_64", "nodejs-10.21.0-3.module_el8.2.0+391+8da3adc6.src.rpm", 27531792, "MIT and ASL 2.0 and ISC and BSD", "CentOS", "nodejs:10:8020020200707141642:6a468ee4", "Non-English locale data for Node.js", "06c123561d40c914ddb262744ad30403"},
			{intRef(), "libmcpp", "2.7.2", "20.el8", "x86_64", "mcpp-2.7.2-20.el8.src.rpm", 153388, "BSD", "CentOS", "", "Alternative C/C++ preprocessor (library build)", "1ff95fab40bb4954f1975ed036efc0ed"},
			{intRef(), "polkit-libs", "0.115", "11.el8", "x86_64", "polkit-0.115-11.el8.src.rpm", 273132, "LGPLv2+", "CentOS", "", "Libraries for polkit", "7672876acc63c1a5bb3557efa504a28d"},
			{intRef(), "libfontenc", "1.1.3", "8.el8", "x86_64", "libfontenc-1.1.3-8.el8.src.rpm", 57021, "MIT", "CentOS", "", "X.Org X11 libfontenc runtime library", "9e9c07ea8ff2599ca7a271a05b92641d"},
			{intRef(), "jansson", "2.11", "3.el8", "x86_64", "jansson-2.11-3.el8.src.rpm", 88783, "MIT", "CentOS", "", "C library for encoding, decoding and manipulating JSON data", "9d5d6daaf6c3446d67377e05733ffff6"},
			{intRef(), "libthai", "0.1.27", "2.el8", "x86_64", "libthai-0.1.27-2.el8.src.rpm", 774997, "LGPLv2+", "CentOS", "", "Thai language support routines", "801f8cc0e2979cb4c0fd989e1ff4f752"},
			{intRef(), "python3-libsemanage", "2.9", "2.el8", "x86_64", "libsemanage-2.9-2.el8.src.rpm", 474221, "LGPLv2+", "CentOS", "", "semanage python 3 bindings for libsemanage", "58b692104949c34e4bef119bc50572ed"},
			{intRef(5), "guile", "2.0.14", "7.el8", "x86_64", "guile-2.0.14-7.el8.src.rpm", 12159310, "LGPLv3+", "CentOS", "", "A GNU implementation of Scheme for application extensibility", "575e446ce886def641e2b1c46db9480d"},
			{intRef(), "libselinux-utils", "2.9", "3.el8", "x86_64", "libselinux-2.9-3.el8.src.rpm", 369120, "Public Domain", "CentOS", "", "SELinux libselinux utilities", "e5f89c3b213f75dc0ad47cf38fa3c62b"},
			{intRef(), "xorg-x11-server-utils", "7.7", "27.el8", "x86_64", "xorg-x11-server-utils-7.7-27.el8.src.rpm", 502730, "MIT", "CentOS", "", "X.Org X11 X server utilities", "4563ac1d6facd98936be75bc47d27bc6"},
			{intRef(), "initscripts", "10.00.6", "1.el8_2.2", "x86_64", "initscripts-10.00.6-1.el8_2.2.src.rpm", 1086847, "GPLv2", "CentOS", "", "Basic support for legacy System V init scripts", "68037cad2d7decc7152a2a830a617101"},
			{intRef(), "urw-base35-gothic-fonts", "20170801", "10.el8", "noarch", "urw-base35-fonts-20170801-10.el8.src.rpm", 1216672, "AGPLv3", "CentOS", "", "URW Gothic font family [part of Level 2 Core Font Set]", "8ad6b1809fd9281e8207ed1461f2b13e"},
			{intRef(), "pixman", "0.38.4", "1.el8", "x86_64", "pixman-0.38.4-1.el8.src.rpm", 693167, "MIT", "CentOS", "", "Pixel manipulation library", "52874a10b07010f0637b5baed2c973a0"},
			{intRef(), "urw-base35-p052-fonts", "20170801", "10.el8", "noarch", "urw-base35-fonts-20170801-10.el8.src.rpm", 1557171, "AGPLv3", "CentOS", "", "P052 font family [part of Level 2 Core Font Set]", "5b3ff8d7697534b661d9c1f860da20a8"},
			{intRef(), "iptables", "1.8.4", "10.el8_2.1", "x86_64", "iptables-1.8.4-10.el8_2.1.src.rpm", 1974841, "GPLv2 and Artistic 2.0 and ISC", "CentOS", "", "Tools for managing Linux kernel packet filtering capabilities", "0d2f72efc52e7c4f832ce90c2c2f60c9"},
			{intRef(), "libXft", "2.3.2", "10.el8", "x86_64", "libXft-2.3.2-10.el8.src.rpm", 136069, "MIT", "CentOS", "", "X.Org X11 libXft runtime library", "b3852262ea8ac451b9b70811ad24b3ad"},
			{intRef(), "python3-gobject-base", "3.28.3", "1.el8", "x86_64", "pygobject3-3.28.3-1.el8.src.rpm", 1116879, "LGPLv2+ and MIT", "CentOS", "", "Python 3 bindings for GObject Introspection base package", "88c929c11f0f0bb88151837a2f678ab0"},
			{intRef(), "libtiff", "4.0.9", "17.el8", "x86_64", "libtiff-4.0.9-17.el8.src.rpm", 517321, "libtiff", "CentOS", "", "Library of functions for manipulating TIFF format image files", "0ee8f3c426cb7563d9a3e746ca326c21"},
			{intRef(), "python3-setuptools", "39.2.0", "5.el8", "noarch", "python-setuptools-39.2.0-5.el8.src.rpm", 460967, "MIT", "CentOS", "", "Easily build and distribute Python 3 packages", "52d6465c67eae824ada6c5e92a907755"},
			{intRef(), "isl", "0.16.1", "6.el8", "x86_64", "isl-0.16.1-6.el8.src.rpm", 3270833, "MIT", "CentOS", "", "Integer point manipulation library", "7e07c5719518014f7be119290bd3c713"},
			{intRef(), "python36", "3.6.8", "2.module_el8.1.0+245+c39af44f", "x86_64", "python36-3.6.8-2.module_el8.1.0+245+c39af44f.src.rpm", 13131, "Python", "CentOS", "python36:3.6:8010020191115015851:a920e634", "Interpreter of the Python programming language", "95108af09ca88c7a879b8944c1ea710c"},
			{intRef(), "systemtap-devel", "4.2", "6.el8", "x86_64", "systemtap-4.2-6.el8.src.rpm", 8837735, "GPLv2+", "CentOS", "", "Programmable system-wide instrumentation system - development headers, tools", "b3858f2a3d8ba7a970de8f9403386c94"},
			{intRef(), "polkit-pkla-compat", "0.1", "12.el8", "x86_64", "polkit-pkla-compat-0.1-12.el8.src.rpm", 95728, "LGPLv2+", "CentOS", "", "Rules for polkit to add compatibility with pklocalauthority", "e6e8e4547233398c4a8f8b76699f329c"},
			{intRef(), "harfbuzz", "1.7.5", "3.el8", "x86_64", "harfbuzz-1.7.5-3.el8.src.rpm", 802826, "MIT", "CentOS", "", "Text shaping library", "f0573459cfad84d39db6f740bbfa3c27"},
			{intRef(), "libssh", "0.9.0", "4.el8", "x86_64", "libssh-0.9.0-4.el8.src.rpm", 822758, "LGPLv2+", "CentOS", "", "A library implementing the SSH protocol", "11c4cb20187b1f700433f09f71a07fbc"},
			{intRef(), "fribidi", "1.0.4", "8.el8", "x86_64", "fribidi-1.0.4-8.el8.src.rpm", 319733, "LGPLv2+ and UCD", "CentOS", "", "Library implementing the Unicode Bidirectional Algorithm", "33029da0f21054974b35183e06cbbfae"},
			{intRef(), "glib-networking", "2.56.1", "1.1.el8", "x86_64", "glib-networking-2.56.1-1.1.el8.src.rpm", 531110, "LGPLv2+", "CentOS", "", "Networking support for GLib", "7debb73c4c82ac6110fd601795c495df"},
			{intRef(), "elfutils-debuginfod-client", "0.178", "7.el8", "x86_64", "elfutils-0.178-7.el8.src.rpm", 35910, "GPLv3+ and (GPLv2+ or LGPLv3+)", "CentOS", "", "Library and command line client for build-id HTTP ELF/DWARF server", "69f1c1ff3b9e5f68bae6021e477c0f57"},
			{intRef(), "PackageKit", "1.1.12", "4.el8", "x86_64", "PackageKit-1.1.12-4.el8.src.rpm", 2873887, "GPLv2+ and LGPLv2+", "CentOS", "", "Package management service", "860e8e95e76e8bbe4f2cde9ad6884e25"},
			{intRef(), "redhat-rpm-config", "122", "1.el8", "noarch", "redhat-rpm-config-122-1.el8.src.rpm", 140370, "GPL+", "CentOS", "", "CentOS specific rpm configuration files", "62c4294340df4afa7d71731bada14840"},
			{intRef(), "fuse-overlayfs", "0.7.2", "5.module_el8.2.0+305+5e198a41", "x86_64", "fuse-overlayfs-0.7.2-5.module_el8.2.0+305+5e198a41.src.rpm", 119420, "GPLv3+", "CentOS", "container-tools:rhel8:8020020200507003530:0d58ad57", "FUSE overlay+shiftfs implementation for rootless containers", "957975bbf3b5a67b2fd63166c1995f54"},
			{intRef(), "boost-atomic", "1.66.0", "7.el8", "x86_64", "boost-1.66.0-7.el8.src.rpm", 8666, "Boost and MIT and Python", "CentOS", "", "Run-time component of boost atomic library", "71322ed1eff7c9fce98c01c6694b8842"},
			{intRef(), "rpm-plugin-selinux", "4.14.2", "37.el8", "x86_64", "rpm-4.14.2-37.el8.src.rpm", 11880, "GPLv2+", "CentOS", "", "Rpm plugin for SELinux functionality", "849a9ad2f2627e164f64debb8c26bd88"},
			{intRef(), "adobe-mappings-pdf", "20180407", "1.el8", "noarch", "adobe-mappings-pdf-20180407-1.el8.src.rpm", 4398414, "BSD", "CentOS", "", "PDF mapping resources from Adobe", "a27029d067dc090e174ebae3e994403a"},
			{intRef(), "python3-dbus", "1.2.4", "15.el8", "x86_64", "dbus-python-1.2.4-15.el8.src.rpm", 489189, "MIT", "CentOS", "", "D-Bus bindings for python3", "f42bb07e95635c208f12900eadbd1d76"},
			{intRef(), "systemtap", "4.2", "6.el8", "x86_64", "systemtap-4.2-6.el8.src.rpm", 0, "GPLv2+", "CentOS", "", "Programmable system-wide instrumentation system", "e3d3d1933cd3fec834c755d0a79e02be"},
			{intRef(), "policycoreutils-python-utils", "2.9", "9.el8", "noarch", "policycoreutils-2.9-9.el8.src.rpm", 140042, "GPLv2", "CentOS", "", "SELinux policy core python utilities", "e02aeeaa3348304882f2660c8b36d49e"},
			{intRef(), "libtool", "2.4.6", "25.el8", "x86_64", "libtool-2.4.6-25.el8.src.rpm", 2687501, "GPLv2+ and LGPLv2+ and GFDL", "CentOS", "", "The GNU Portable Library Tool", "46ac6d2c69ea159d33657a2cd9525be4"},
			{intRef(1), "NetworkManager-team", "1.22.8", "5.el8_2", "x86_64", "NetworkManager-1.22.8-5.el8_2.src.rpm", 45480, "GPLv2+ and LGPLv2+", "CentOS", "", "Team device plugin for NetworkManager", "daeae3b298529f075f9710b99714c167"},
			{intRef(), "perl-generators", "1.10", "9.el8", "noarch", "perl-generators-1.10-9.el8.src.rpm", 21852, "GPL+", "CentOS", "", "RPM Perl dependencies generators", "69b03038244cb6e7847b9fb393847939"},
			{intRef(), "ostree-libs", "2019.6", "2.el8", "x86_64", "ostree-2019.6-2.el8.src.rpm", 981292, "LGPLv2+", "CentOS", "", "Development headers for ostree", "08bf79c23ad2ddaf45cb2bca744530d1"},
		}

		return toPackageInfo(pkgs)
	}

	// docker run --platform s390x --rm -it registry.access.redhat.com/ubi8/ubi bash
	// rpm -qa --queryformat "\{%{EPOCH}, \"%{NAME}\", \"%{VERSION}\", \"%{RELEASE}\", \"%{ARCH}\", \"%{SOURCERPM}\", %{SIZE}, \"%{LICENSE}\", \"%{VENDOR}\", \"\", \"%{SUMMARY}\", \"%{SIGMD5}\"\},\n" | sed "s/^{(none)/{intRef()/g" | sed -r 's/^\{([0-9]+),/{intRef(\1),/' | sed "s/(none)/0/g"
	UBI8s390x = func() []*PackageInfo {
		pkgs := []*commonPackageInfo{
			{intRef(), "tzdata", "2022a", "1.el8", "noarch", "tzdata-2022a-1.el8.src.rpm", 1891990, "Public Domain", "Red Hat, Inc.", "", "Timezone data", "1058856cc44cf659afb42c6a481bffba"},
			{intRef(), "python3-pip-wheel", "9.0.3", "22.el8", "noarch", "python-pip-9.0.3-22.el8.src.rpm", 929805, "MIT and Python and ASL 2.0 and BSD and ISC and LGPLv2 and MPLv2.0 and (ASL 2.0 or BSD)", "Red Hat, Inc.", "", "The pip wheel", "eb3719e7a6391740511c3b91956067e5"},
			{intRef(), "redhat-release", "8.6", "0.1.el8", "s390x", "redhat-release-8.6-0.1.el8.src.rpm", 56548, "GPLv2", "Red Hat, Inc.", "", "Red Hat Enterprise Linux release file", "6c24b2fe641ea2c8043202f6cbe33561"},
			{intRef(), "filesystem", "3.8", "6.el8", "s390x", "filesystem-3.8-6.el8.src.rpm", 0, "Public Domain", "Red Hat, Inc.", "", "The basic directory layout for a Linux system", "cec06159982912515d9098e4eef7cdeb"},
			{intRef(), "publicsuffix-list-dafsa", "20180723", "1.el8", "noarch", "publicsuffix-list-20180723-1.el8.src.rpm", 64502, "MPLv2.0", "Red Hat, Inc.", "", "Cross-vendor public domain suffix database in DAFSA form", "193f2f7ffa5e990141b941b47dfbab79"},
			{intRef(), "pcre2", "10.32", "2.el8", "s390x", "pcre2-10.32-2.el8.src.rpm", 466150, "BSD", "Red Hat, Inc.", "", "Perl-compatible regular expression library", "796db70cca475d86f3847b815df6c5b4"},
			{intRef(), "ncurses-libs", "6.1", "9.20180224.el8", "s390x", "ncurses-6.1-9.20180224.el8.src.rpm", 977472, "MIT", "Red Hat, Inc.", "", "Ncurses libraries", "27d587ba6aa5ef29547480bf0af34b67"},
			{intRef(), "glibc-common", "2.28", "189.1.el8", "s390x", "glibc-2.28-189.1.el8.src.rpm", 8623535, "LGPLv2+ and LGPLv2+ with exceptions and GPLv2+ and GPLv2+ with exceptions and BSD and Inner-Net and ISC and Public Domain and GFDL", "Red Hat, Inc.", "", "Common binaries and locale data for glibc", "15a01b9af506200fc155efc78b935c1f"},
			{intRef(), "bash", "4.4.20", "3.el8", "s390x", "bash-4.4.20-3.el8.src.rpm", 7065310, "GPLv3+", "Red Hat, Inc.", "", "The GNU Bourne Again shell", "5a3650abd17ec3d20616e6c924729df9"},
			{intRef(), "zlib", "1.2.11", "18.el8_5", "s390x", "zlib-1.2.11-18.el8_5.src.rpm", 212335, "zlib and Boost", "Red Hat, Inc.", "", "The compression and decompression library", "a6c4557b8b1c8a70724d4cccf43d7a6a"},
			{intRef(), "bzip2-libs", "1.0.6", "26.el8", "s390x", "bzip2-1.0.6-26.el8.src.rpm", 84197, "BSD", "Red Hat, Inc.", "", "Libraries for applications using bzip2", "1ba5e68e3d5b1745f13cc2a1c7af90e8"},
			{intRef(), "sqlite-libs", "3.26.0", "15.el8", "s390x", "sqlite-3.26.0-15.el8.src.rpm", 1265377, "Public Domain", "Red Hat, Inc.", "", "Shared library for the sqlite3 embeddable SQL database engine.", "88b5f9b1cf5022e0bb40b7b7931f5cf3"},
			{intRef(), "libcap", "2.48", "2.el8", "s390x", "libcap-2.48-2.el8.src.rpm", 165085, "BSD or GPLv2", "Red Hat, Inc.", "", "Library for getting and setting POSIX.1e capabilities", "b4ecdd2cd7e79ed1316417ba8ac2639d"},
			{intRef(), "libzstd", "1.4.4", "1.el8", "s390x", "zstd-1.4.4-1.el8.src.rpm", 709973, "BSD and GPLv2", "Red Hat, Inc.", "", "Zstd shared library", "1df9a004666be3ea4a9370d3001f249c"},
			{intRef(), "libxml2", "2.9.7", "13.el8", "s390x", "libxml2-2.9.7-13.el8.src.rpm", 1829754, "MIT", "Red Hat, Inc.", "", "Library providing XML and HTML support", "ee0c253ae8695de6131f71e25d0be5a9"},
			{intRef(), "expat", "2.2.5", "8.el8", "s390x", "expat-2.2.5-8.el8.src.rpm", 379588, "MIT", "Red Hat, Inc.", "", "An XML parser library", "892470470470ec1dd6c4ba7a908190f9"},
			{intRef(), "libcom_err", "1.45.6", "4.el8", "s390x", "e2fsprogs-1.45.6-4.el8.src.rpm", 61025, "MIT", "Red Hat, Inc.", "", "Common error description library", "a646db5a7880e3f07020b185f60c8602"},
			{intRef(), "libuuid", "2.32.1", "35.el8", "s390x", "util-linux-2.32.1-35.el8.src.rpm", 34312, "BSD", "Red Hat, Inc.", "", "Universally unique ID library", "cb2502af2557a2297611636294153456"},
			{intRef(1), "gmp", "6.1.2", "10.el8", "s390x", "gmp-6.1.2-10.el8.src.rpm", 1475820, "LGPLv3+ or GPLv2+", "Red Hat, Inc.", "", "A GNU arbitrary precision library", "fcd905dcdfcc990633d085a634f31b72"},
			{intRef(), "libacl", "2.2.53", "1.el8", "s390x", "acl-2.2.53-1.el8.src.rpm", 55680, "LGPLv2+", "Red Hat, Inc.", "", "Dynamic library for access control list support", "6f8ca1573e06715993a3d9cf72602cef"},
			{intRef(), "libblkid", "2.32.1", "35.el8", "s390x", "util-linux-2.32.1-35.el8.src.rpm", 359904, "LGPLv2+", "Red Hat, Inc.", "", "Block device ID library", "8693a1b1a8471a986c58c223514034fd"},
			{intRef(), "sed", "4.5", "5.el8", "s390x", "sed-4.5-5.el8.src.rpm", 781054, "GPLv3+", "Red Hat, Inc.", "", "A GNU stream text editor", "1111dbe960e6272f11fb33452700e04d"},
			{intRef(), "libstdc++", "8.5.0", "10.el8", "s390x", "gcc-8.5.0-10.el8.src.rpm", 2072324, "GPLv3+ and GPLv3+ with exceptions and GPLv2+ with exceptions and LGPLv2+ and BSD", "Red Hat, Inc.", "", "GNU Standard C++ Library", "d855b0d1abb7eba4e7860a878d5dea1a"},
			{intRef(), "p11-kit", "0.23.22", "1.el8", "s390x", "p11-kit-0.23.22-1.el8.src.rpm", 1755223, "BSD", "Red Hat, Inc.", "", "Library for loading and sharing PKCS#11 modules", "e59200a12df95c92976cbe033caa0a9a"},
			{intRef(), "libunistring", "0.9.9", "3.el8", "s390x", "libunistring-0.9.9-3.el8.src.rpm", 1788492, "GPLv2+ or LGPLv3+", "Red Hat, Inc.", "", "GNU Unicode string library", "84a77119e1d8c017049b4f70b1976bd6"},
			{intRef(), "libgcrypt", "1.8.5", "6.el8", "s390x", "libgcrypt-1.8.5-6.el8.src.rpm", 905470, "LGPLv2+", "Red Hat, Inc.", "", "A general-purpose cryptography library", "97f6da2f1b4b81af7bd421045bf95df3"},
			{intRef(), "libcap-ng", "0.7.11", "1.el8", "s390x", "libcap-ng-0.7.11-1.el8.src.rpm", 51126, "LGPLv2+", "Red Hat, Inc.", "", "An alternate posix capabilities library", "0d6520d0f9700d1c36526d87fad8a195"},
			{intRef(), "lz4-libs", "1.8.3", "3.el8_4", "s390x", "lz4-1.8.3-3.el8_4.src.rpm", 128927, "GPLv2+ and BSD", "Red Hat, Inc.", "", "Libaries for lz4", "85fd7ea50bb9eec24659b57441697b84"},
			{intRef(1), "gdbm-libs", "1.18", "1.el8", "s390x", "gdbm-1.18-1.el8.src.rpm", 135944, "GPLv3+", "Red Hat, Inc.", "", "Libraries files for gdbm", "83ff9ed81bee653f4228af50fe53a3da"},
			{intRef(), "libtasn1", "4.13", "3.el8", "s390x", "libtasn1-4.13-3.el8.src.rpm", 175293, "GPLv3+ and LGPLv2+", "Red Hat, Inc.", "", "The ASN.1 library used in GNUTLS", "5eea350ae52bf84742ccd778eafd82c2"},
			{intRef(), "pcre", "8.42", "6.el8", "s390x", "pcre-8.42-6.el8.src.rpm", 327843, "BSD", "Red Hat, Inc.", "", "Perl-compatible regular expression library", "9ab303f2f391cbadf05080a15b974af8"},
			{intRef(), "systemd-libs", "239", "58.el8", "s390x", "systemd-239-58.el8.src.rpm", 4673382, "LGPLv2+ and MIT", "Red Hat, Inc.", "", "systemd libraries", "594af46fbeff806adc68f60acf239425"},
			{intRef(), "ca-certificates", "2021.2.50", "80.0.el8_4", "noarch", "ca-certificates-2021.2.50-80.0.el8_4.src.rpm", 930318, "Public Domain", "Red Hat, Inc.", "", "The Mozilla CA root certificate bundle", "b2865ad3f1041196e64c2aafab77f0be"},
			{intRef(), "libusbx", "1.0.23", "4.el8", "s390x", "libusbx-1.0.23-4.el8.src.rpm", 157907, "LGPLv2+", "Red Hat, Inc.", "", "Library for accessing USB devices", "85b925baa2ea12e2f5061f4a71f9c4a6"},
			{intRef(), "libsemanage", "2.9", "8.el8", "s390x", "libsemanage-2.9-8.el8.src.rpm", 319800, "LGPLv2+", "Red Hat, Inc.", "", "SELinux binary policy manipulation library", "c7f1e7b5402b07c46196db7798fc78b0"},
			{intRef(), "libutempter", "1.1.6", "14.el8", "s390x", "libutempter-1.1.6-14.el8.src.rpm", 51557, "LGPLv2+", "Red Hat, Inc.", "", "A privileged helper for utmp/wtmp updates", "60aeddb64e2e5f777680b085e6de31ed"},
			{intRef(), "libfdisk", "2.32.1", "35.el8", "s390x", "util-linux-2.32.1-35.el8.src.rpm", 454514, "LGPLv2+", "Red Hat, Inc.", "", "Partitioning library for fdisk-like programs.", "0a5b231f3e16fde7f143e76dfe2f3adf"},
			{intRef(), "gzip", "1.9", "13.el8_5", "s390x", "gzip-1.9-13.el8_5.src.rpm", 364906, "GPLv3+ and GFDL", "Red Hat, Inc.", "", "The GNU data compression program", "aed1fce3e0ce0e2554d2f811a536eeff"},
			{intRef(), "cracklib-dicts", "2.9.6", "15.el8", "s390x", "cracklib-2.9.6-15.el8.src.rpm", 9815016, "LGPLv2+", "Red Hat, Inc.", "", "The standard CrackLib dictionaries", "4f52aae907f4fcc4ecace03ae358ad83"},
			{intRef(), "mpfr", "3.1.6", "1.el8", "s390x", "mpfr-3.1.6-1.el8.src.rpm", 659537, "LGPLv3+ and GPLv3+ and GFDL", "Red Hat, Inc.", "", "A C library for multiple-precision floating-point computations", "fc66607bc6990ce55c2f50f0f6f95b72"},
			{intRef(), "libcomps", "0.1.18", "1.el8", "s390x", "libcomps-0.1.18-1.el8.src.rpm", 238804, "GPLv2+", "Red Hat, Inc.", "", "Comps XML file manipulation library", "866d4554a1fae21563e5955da119cbea"},
			{intRef(), "brotli", "1.0.6", "3.el8", "s390x", "brotli-1.0.6-3.el8.src.rpm", 1577676, "MIT", "Red Hat, Inc.", "", "Lossless compression algorithm", "faab99cf563fafd9e2ce4222e07f98e7"},
			{intRef(), "libnl3", "3.5.0", "1.el8", "s390x", "libnl3-3.5.0-1.el8.src.rpm", 1028840, "LGPLv2", "Red Hat, Inc.", "", "Convenience library for kernel netlink sockets", "1b303adc88673e6f8d022435d0d161fa"},
			{intRef(), "libsigsegv", "2.11", "5.el8", "s390x", "libsigsegv-2.11-5.el8.src.rpm", 46082, "GPLv2+", "Red Hat, Inc.", "", "Library for handling page faults in user mode", "e182243c0efafda080dbe6d11b6055aa"},
			{intRef(), "libverto", "0.3.0", "5.el8", "s390x", "libverto-0.3.0-5.el8.src.rpm", 27740, "MIT", "Red Hat, Inc.", "", "Main loop abstraction library", "77fcf5ca7c3de76c1af2e27e693d7ff7"},
			{intRef(), "libtirpc", "1.1.4", "6.el8", "s390x", "libtirpc-1.1.4-6.el8.src.rpm", 236366, "SISSL and BSD", "Red Hat, Inc.", "", "Transport Independent RPC Library", "8e0ce7cc1225ea2eae06193d418d18eb"},
			{intRef(1), "openssl-libs", "1.1.1k", "6.el8_5", "s390x", "openssl-1.1.1k-6.el8_5.src.rpm", 3364026, "OpenSSL and ASL 2.0", "Red Hat, Inc.", "", "A general purpose cryptography library with TLS implementation", "7fc92bf8e37568ca57c0d14da1164f52"},
			{intRef(), "crypto-policies-scripts", "20211116", "1.gitae470d6.el8", "noarch", "crypto-policies-20211116-1.gitae470d6.el8.src.rpm", 189741, "LGPLv2+", "Red Hat, Inc.", "", "Tool to switch between crypto policies", "d361aea97ec27d92e7fb1a41dc9261c1"},
			{intRef(), "platform-python", "3.6.8", "45.el8", "s390x", "python3-3.6.8-45.el8.src.rpm", 41118, "Python", "Red Hat, Inc.", "", "Internal interpreter of the Python programming language", "9f1224e173992afb2e721c6518b7b219"},
			{intRef(), "libdb", "5.3.28", "42.el8_4", "s390x", "libdb-5.3.28-42.el8_4.src.rpm", 1906952, "BSD and LGPLv2 and Sleepycat", "Red Hat, Inc.", "", "The Berkeley DB database library for C", "89f159cca4fe8f531e7fa469fca168f7"},
			{intRef(), "pam", "1.3.1", "16.el8", "s390x", "pam-1.3.1-16.el8.src.rpm", 2613560, "BSD and GPLv2+", "Red Hat, Inc.", "", "An extensible library which provides authentication for applications", "2758a471e5fd319555afeb95c6636b58"},
			{intRef(), "util-linux", "2.32.1", "35.el8", "s390x", "util-linux-2.32.1-35.el8.src.rpm", 11563941, "GPLv2 and GPLv2+ and LGPLv2+ and BSD with advertising and Public Domain", "Red Hat, Inc.", "", "A collection of basic system utilities", "230d7f09a3e19a94d1487bea9c36023b"},
			{intRef(), "gnutls", "3.6.16", "4.el8", "s390x", "gnutls-3.6.16-4.el8.src.rpm", 2838279, "GPLv3+ and LGPLv2+", "Red Hat, Inc.", "", "A TLS protocol implementation", "38243046f641da8eb31ca15036a426ff"},
			{intRef(), "json-glib", "1.4.4", "1.el8", "s390x", "json-glib-1.4.4-1.el8.src.rpm", 537299, "LGPLv2+", "Red Hat, Inc.", "", "Library for JavaScript Object Notation format", "acfad1507faa5bae78096af59e87ec4b"},
			{intRef(), "python3-iniparse", "0.4", "31.el8", "noarch", "python-iniparse-0.4-31.el8.src.rpm", 108846, "MIT and Python", "Red Hat, Inc.", "", "Python Module for Accessing and Modifying Configuration Data in INI files", "29f73a3875d2d7f187c2796233b96724"},
			{intRef(), "python3-dbus", "1.2.4", "15.el8", "s390x", "dbus-python-1.2.4-15.el8.src.rpm", 523093, "MIT", "Red Hat, Inc.", "", "D-Bus bindings for python3", "efca01f653ce8f3e724caa8f233c3006"},
			{intRef(), "python3-gobject-base", "3.28.3", "2.el8", "s390x", "pygobject3-3.28.3-2.el8.src.rpm", 1128653, "LGPLv2+ and MIT", "Red Hat, Inc.", "", "Python 3 bindings for GObject Introspection base package", "e1739de5dcda13ff4f51253fe8c6c2ca"},
			{intRef(), "cyrus-sasl-lib", "2.1.27", "6.el8_5", "s390x", "cyrus-sasl-2.1.27-6.el8_5.src.rpm", 736746, "BSD with advertising", "Red Hat, Inc.", "", "Shared libraries needed by applications which use Cyrus SASL", "f67fd6a6a0b44a1b9fc49952c7b64c8f"},
			{intRef(), "libuser", "0.62", "24.el8", "s390x", "libuser-0.62-24.el8.src.rpm", 1968316, "LGPLv2+", "Red Hat, Inc.", "", "A user and group account administration library", "21f8182086b92c7ddc12dcbf359dda94"},
			{intRef(), "usermode", "1.113", "2.el8", "s390x", "usermode-1.113-2.el8.src.rpm", 853880, "GPLv2+", "Red Hat, Inc.", "", "Tools for certain user account management tasks", "4330571c497ffd65696d6ad1d875b91c"},
			{intRef(), "python3-ethtool", "0.14", "5.el8", "s390x", "python-ethtool-0.14-5.el8.src.rpm", 95616, "GPLv2", "Red Hat, Inc.", "", "Python module to interface with ethtool", "1088b62fa5d8858b31d495f5d43816ee"},
			{intRef(), "python3-chardet", "3.0.4", "7.el8", "noarch", "python-chardet-3.0.4-7.el8.src.rpm", 925538, "LGPLv2", "Red Hat, Inc.", "", "Character encoding auto-detection in Python 3", "3ada1fbcb564b91e61de7f32130c471a"},
			{intRef(), "python3-idna", "2.5", "5.el8", "noarch", "python-idna-2.5-5.el8.src.rpm", 521503, "BSD and Python and Unicode", "Red Hat, Inc.", "", "Internationalized Domain Names in Applications (IDNA)", "6ac5832b8bd214a6d2abc0072c97ab94"},
			{intRef(), "python3-pysocks", "1.6.8", "3.el8", "noarch", "python-pysocks-1.6.8-3.el8.src.rpm", 77054, "BSD", "Red Hat, Inc.", "", "A Python SOCKS client module", "69d81ceb864a79a1b73a3c9826d9d6a7"},
			{intRef(), "python3-requests", "2.20.0", "2.1.el8_1", "noarch", "python-requests-2.20.0-2.1.el8_1.src.rpm", 377832, "ASL 2.0", "Red Hat, Inc.", "", "HTTP library, written in Python, for human beings", "86b829d8cb0e3dc32d5c0d3a992456f4"},
			{intRef(), "python3-syspurpose", "1.28.29", "3.el8", "s390x", "subscription-manager-1.28.29-3.el8.src.rpm", 165137, "GPLv2", "Red Hat, Inc.", "", "A commandline utility for declaring system syspurpose", "14a0f2622a8580fefafc6a6a809dc4f8"},
			{intRef(8), "device-mapper", "1.02.181", "3.el8", "s390x", "lvm2-2.03.14-3.el8.src.rpm", 360430, "GPLv2", "Red Hat, Inc.", "", "Device mapper utility", "175b00e2679094919cbb6b3211073ead"},
			{intRef(), "cryptsetup-libs", "2.3.7", "2.el8", "s390x", "cryptsetup-2.3.7-2.el8.src.rpm", 2149647, "GPLv2+ and LGPLv2+", "Red Hat, Inc.", "", "Cryptsetup shared library", "a31ab06ba963183e3d40dcca3c10dc0b"},
			{intRef(), "elfutils-libs", "0.186", "1.el8", "s390x", "elfutils-0.186-1.el8.src.rpm", 749343, "GPLv2+ or LGPLv3+", "Red Hat, Inc.", "", "Libraries to handle compiled objects", "63bee1402b4e5746b60f06260f43d0ae"},
			{intRef(1), "dbus-daemon", "1.12.8", "18.el8", "s390x", "dbus-1.12.8-18.el8.src.rpm", 569168, "(GPLv2+ or AFL) and GPLv2+", "Red Hat, Inc.", "", "D-BUS message bus", "934af9cbbb54a8e3d6e770a5f7748fb0"},
			{intRef(), "systemd", "239", "58.el8", "s390x", "systemd-239-58.el8.src.rpm", 11171575, "LGPLv2+ and MIT and GPLv2+", "Red Hat, Inc.", "", "System and Service Manager", "42eea01e56f3ece787b702b0e5ba67be"},
			{intRef(), "libarchive", "3.3.3", "3.el8_5", "s390x", "libarchive-3.3.3-3.el8_5.src.rpm", 879679, "BSD", "Red Hat, Inc.", "", "A library for handling streaming archive formats", "e25b0b013f5abb587bcfd0027353851e"},
			{intRef(), "ima-evm-utils", "1.3.2", "12.el8", "s390x", "ima-evm-utils-1.3.2-12.el8.src.rpm", 141033, "GPLv2", "Red Hat, Inc.", "", "IMA/EVM support utilities", "0ed54cdc9767c4d7c9836eefe791fc22"},
			{intRef(), "npth", "1.5", "4.el8", "s390x", "npth-1.5-4.el8.src.rpm", 47245, "LGPLv2+", "Red Hat, Inc.", "", "The New GNU Portable Threads library", "885cc27cd427a0988f8db098ae80b31b"},
			{intRef(), "gpgme", "1.13.1", "11.el8", "s390x", "gpgme-1.13.1-11.el8.src.rpm", 1023808, "LGPLv2+ and GPLv3+", "Red Hat, Inc.", "", "GnuPG Made Easy - high level crypto API", "6b93c9c67963f937600b8dc955353bba"},
			{intRef(), "libssh-config", "0.9.6", "3.el8", "noarch", "libssh-0.9.6-3.el8.src.rpm", 277, "LGPLv2+", "Red Hat, Inc.", "", "Configuration files for libssh", "af3dd754ef71c5189a6062566267526f"},
			{intRef(), "libcurl", "7.61.1", "22.el8", "s390x", "curl-7.61.1-22.el8.src.rpm", 617928, "MIT", "Red Hat, Inc.", "", "A library for getting files from web servers", "f15334bbb12a8985bf54fafbfc5cefcc"},
			{intRef(), "python3-librepo", "1.14.2", "1.el8", "s390x", "librepo-1.14.2-1.el8.src.rpm", 183126, "LGPLv2+", "Red Hat, Inc.", "", "Python 3 bindings for the librepo library", "47691646d9cae6c7224dbdf5c52800c9"},
			{intRef(), "rpm", "4.14.3", "23.el8", "s390x", "rpm-4.14.3-23.el8.src.rpm", 2084813, "GPLv2+", "Red Hat, Inc.", "", "The RPM package management system", "4924469df702e9ef6a82dddbde0e5cee"},
			{intRef(), "libmodulemd", "2.13.0", "1.el8", "s390x", "libmodulemd-2.13.0-1.el8.src.rpm", 691082, "MIT", "Red Hat, Inc.", "", "Module metadata manipulation library", "e0ac8ec3213df0d336434002067c2b7d"},
			{intRef(), "libdnf", "0.63.0", "8.el8", "s390x", "libdnf-0.63.0-8.el8.src.rpm", 2404251, "LGPLv2+", "Red Hat, Inc.", "", "Library providing simplified C and Python API to libsolv", "6c29397fc08915a1e724e56e935b8744"},
			{intRef(), "python3-hawkey", "0.63.0", "8.el8", "s390x", "libdnf-0.63.0-8.el8.src.rpm", 297200, "LGPLv2+", "Red Hat, Inc.", "", "Python 3 bindings for the hawkey library", "b4eb1e600a69c97fcf3535cc652afcbf"},
			{intRef(), "python3-rpm", "4.14.3", "23.el8", "s390x", "rpm-4.14.3-23.el8.src.rpm", 423965, "GPLv2+", "Red Hat, Inc.", "", "Python 3 bindings for apps which will manipulate RPM packages", "04957da6de7eb9e0d149e1d835b70d74"},
			{intRef(), "libreport-filesystem", "2.9.5", "15.el8", "s390x", "libreport-2.9.5-15.el8.src.rpm", 0, "GPLv2+", "Red Hat, Inc.", "", "Filesystem layout for libreport", "470d5469057401340bf4a084538d3f5e"},
			{intRef(), "python3-dnf", "4.7.0", "8.el8", "noarch", "dnf-4.7.0-8.el8.src.rpm", 1905194, "GPLv2+", "Red Hat, Inc.", "", "Python 3 interface to DNF", "67413af9a15637c155b24f03ecb5253f"},
			{intRef(), "python3-dnf-plugins-core", "4.0.21", "11.el8", "noarch", "dnf-plugins-core-4.0.21-11.el8.src.rpm", 825827, "GPLv2+", "Red Hat, Inc.", "", "Core Plugins for DNF", "1d7d2d173293228c0e2020942f89148f"},
			{intRef(), "subscription-manager", "1.28.29", "3.el8", "s390x", "subscription-manager-1.28.29-3.el8.src.rpm", 4550333, "GPLv2", "Red Hat, Inc.", "", "Tools and libraries for subscription and repository management", "3d438c79ab3760341c0f56af75168650"},
			{intRef(), "gdb-gdbserver", "8.2", "18.el8", "s390x", "gdb-8.2-18.el8.src.rpm", 782013, "GPLv3+ and GPLv3+ with exceptions and GPLv2+ and GPLv2+ with exceptions and GPL+ and LGPLv2+ and LGPLv3+ and BSD and Public Domain and GFDL", "Red Hat, Inc.", "", "A standalone server for GDB (the GNU source-level debugger)", "0ac37efac0f1ed87f2fc1fb41f8167f0"},
			{intRef(2), "vim-minimal", "8.0.1763", "16.el8_5.13", "s390x", "vim-8.0.1763-16.el8_5.13.src.rpm", 1341404, "Vim and MIT", "Red Hat, Inc.", "", "A minimal version of the VIM editor", "07212dd0b68bd41418cf64ff782d2804"},
			{intRef(), "langpacks-en", "1.0", "12.el8", "noarch", "langpacks-1.0-12.el8.src.rpm", 400, "GPLv2+", "Red Hat, Inc.", "", "English langpacks meta-package", "1130f256caf9531f47bf94dc9c89a697"},
			{intRef(), "gpg-pubkey", "fd431d51", "4ae0493b", "", "", 0, "pubkey", "", "", "gpg(Red Hat, Inc. (release key 2) <security@redhat.com>)", ""},
			{intRef(), "libgcc", "8.5.0", "10.el8", "s390x", "gcc-8.5.0-10.el8.src.rpm", 160544, "GPLv3+ and GPLv3+ with exceptions and GPLv2+ with exceptions and LGPLv2+ and BSD", "Red Hat, Inc.", "", "GCC version 8 shared support library", "933cc0a00a93c0edb647945bbf0821a5"},
			{intRef(), "python3-setuptools-wheel", "39.2.0", "6.el8", "noarch", "python-setuptools-39.2.0-6.el8.src.rpm", 347720, "MIT", "Red Hat, Inc.", "", "The setuptools wheel", "f7a0c30562c572b2b169a51cdd75df84"},
			{intRef(), "subscription-manager-rhsm-certificates", "1.28.29", "3.el8", "s390x", "subscription-manager-1.28.29-3.el8.src.rpm", 9716, "GPLv2", "Red Hat, Inc.", "", "Certificates required to communicate with a Red Hat Unified Entitlement Platform", "3d1574cc47043a6d11a0fd9da088ec37"},
			{intRef(), "setup", "2.12.2", "6.el8", "noarch", "setup-2.12.2-6.el8.src.rpm", 724837, "Public Domain", "Red Hat, Inc.", "", "A set of system configuration and setup files", "109811c80a28ad746bcd12d4d0c06794"},
			{intRef(), "basesystem", "11", "5.el8", "noarch", "basesystem-11-5.el8.src.rpm", 0, "Public Domain", "Red Hat, Inc.", "", "The skeleton package which defines a simple Red Hat Enterprise Linux system", "31bc067a6462aacd3b891681bdb27512"},
			{intRef(), "ncurses-base", "6.1", "9.20180224.el8", "noarch", "ncurses-6.1-9.20180224.el8.src.rpm", 290089, "MIT", "Red Hat, Inc.", "", "Descriptions of common terminals", "daf3b68d63963e4e84f6658a2984e1ff"},
			{intRef(), "libselinux", "2.9", "5.el8", "s390x", "libselinux-2.9-5.el8.src.rpm", 185616, "Public Domain", "Red Hat, Inc.", "", "SELinux library and simple utilities", "bc723e63488942a9e79bd87c13e1591a"},
			{intRef(), "glibc-minimal-langpack", "2.28", "189.1.el8", "s390x", "glibc-2.28-189.1.el8.src.rpm", 0, "LGPLv2+ and LGPLv2+ with exceptions and GPLv2+ and GPLv2+ with exceptions and BSD and Inner-Net and ISC and Public Domain and GFDL", "Red Hat, Inc.", "", "Minimal language packs for glibc.", "4e75d0fe7929b021b0775e7ed6fe346a"},
			{intRef(), "glibc", "2.28", "189.1.el8", "s390x", "glibc-2.28-189.1.el8.src.rpm", 5258281, "LGPLv2+ and LGPLv2+ with exceptions and GPLv2+ and GPLv2+ with exceptions and BSD and Inner-Net and ISC and Public Domain and GFDL", "Red Hat, Inc.", "", "The GNU libc libraries", "f2068bc59b0590e1aaec093cc9b3832e"},
			{intRef(), "libsepol", "2.9", "3.el8", "s390x", "libsepol-2.9-3.el8.src.rpm", 802488, "LGPLv2+", "Red Hat, Inc.", "", "SELinux binary policy manipulation library", "46486cb696f59e758c999566bf5ec490"},
			{intRef(), "xz-libs", "5.2.4", "4.el8_6", "s390x", "xz-5.2.4-4.el8_6.src.rpm", 172471, "Public Domain", "Red Hat, Inc.", "", "Libraries for decoding LZMA compression", "900d196b8fd0fc90b4d8c24b51a5d808"},
			{intRef(), "libgpg-error", "1.31", "1.el8", "s390x", "libgpg-error-1.31-1.el8.src.rpm", 902914, "LGPLv2+", "Red Hat, Inc.", "", "Library for error values used by GnuPG components", "d2929d3570d08dbc4f848c5d9652e795"},
			{intRef(), "info", "6.5", "7.el8", "s390x", "texinfo-6.5-7.el8.src.rpm", 413222, "GPLv3+", "Red Hat, Inc.", "", "A stand-alone TTY-based reader for GNU texinfo documentation", "91559f751e22c755d8d6b2115130e1d9"},
			{intRef(), "libxcrypt", "4.1.1", "6.el8", "s390x", "libxcrypt-4.1.1-6.el8.src.rpm", 185564, "LGPLv2+ and BSD and Public Domain", "Red Hat, Inc.", "", "Extended crypt library for DES, MD5, Blowfish and others", "7abd8dae681838bd9e35d648f2045602"},
			{intRef(), "popt", "1.18", "1.el8", "s390x", "popt-1.18-1.el8.src.rpm", 134450, "MIT", "Red Hat, Inc.", "", "C library for parsing command line parameters", "2cca803a53883566713ac58156f47138"},
			{intRef(), "elfutils-libelf", "0.186", "1.el8", "s390x", "elfutils-0.186-1.el8.src.rpm", 1027205, "GPLv2+ or LGPLv3+", "Red Hat, Inc.", "", "Library to read and write ELF files", "54c0a8532cf7ac99214ddd154acc6627"},
			{intRef(), "json-c", "0.13.1", "3.el8", "s390x", "json-c-0.13.1-3.el8.src.rpm", 72698, "MIT", "Red Hat, Inc.", "", "JSON implementation in C", "350d10fa2845be36fd9babe37f33a318"},
			{intRef(), "libffi", "3.1", "23.el8", "s390x", "libffi-3.1-23.el8.src.rpm", 53124, "MIT", "Red Hat, Inc.", "", "A portable foreign function interface library", "57c503043d7d17df4777907e6c323600"},
			{intRef(), "readline", "7.0", "10.el8", "s390x", "readline-7.0-10.el8.src.rpm", 505065, "GPLv3+", "Red Hat, Inc.", "", "A library for editing typed command lines", "8cd29bcf434ee6c0398e25a4a8a45ea2"},
			{intRef(), "libattr", "2.4.48", "3.el8", "s390x", "attr-2.4.48-3.el8.src.rpm", 26266, "LGPLv2+", "Red Hat, Inc.", "", "Dynamic library for extended attribute support", "eaf56ed40c57769be9b81e7bf0ff605f"},
			{intRef(), "coreutils-single", "8.30", "12.el8", "s390x", "coreutils-8.30-12.el8.src.rpm", 1497657, "GPLv3+", "Red Hat, Inc.", "", "coreutils multicall binary", "40acfa3615bb7d315815bcef896e8a2b"},
			{intRef(), "libmount", "2.32.1", "35.el8", "s390x", "util-linux-2.32.1-35.el8.src.rpm", 418282, "LGPLv2+", "Red Hat, Inc.", "", "Device mounting library", "c2bee9d424d6dd02bdf4800067e38809"},
			{intRef(), "libsmartcols", "2.32.1", "35.el8", "s390x", "util-linux-2.32.1-35.el8.src.rpm", 260194, "LGPLv2+", "Red Hat, Inc.", "", "Formatting library for ls-like programs.", "cb227a394b8c62bb2d4f59bdba42be88"},
			{intRef(), "lua-libs", "5.3.4", "12.el8", "s390x", "lua-5.3.4-12.el8.src.rpm", 276232, "MIT", "Red Hat, Inc.", "", "Libraries for lua", "491f27bbd55f435bb1061fdac161a457"},
			{intRef(), "chkconfig", "1.19.1", "1.el8", "s390x", "chkconfig-1.19.1-1.el8.src.rpm", 827014, "GPLv2", "Red Hat, Inc.", "", "A system tool for maintaining the /etc/rc*.d hierarchy", "70fb9e42a55e992a9df28929a5848541"},
			{intRef(), "libidn2", "2.2.0", "1.el8", "s390x", "libidn2-2.2.0-1.el8.src.rpm", 278258, "(GPLv2+ or LGPLv3+) and GPLv3+", "Red Hat, Inc.", "", "Library to support IDNA2008 internationalized domain names", "c6541dd1fcd76038170314bf47f111f8"},
			{intRef(), "file-libs", "5.33", "20.el8", "s390x", "file-5.33-20.el8.src.rpm", 6394213, "BSD", "Red Hat, Inc.", "", "Libraries for applications using libmagic", "c297e7f1fc192552221e205a26c1aad1"},
			{intRef(), "audit-libs", "3.0.7", "2.el8.2", "s390x", "audit-3.0.7-2.el8.2.src.rpm", 311170, "LGPLv2+", "Red Hat, Inc.", "", "Dynamic library for libaudit", "529f6eeab45a10aaf25247e878ec7201"},
			{intRef(), "libassuan", "2.5.1", "3.el8", "s390x", "libassuan-2.5.1-3.el8.src.rpm", 199299, "LGPLv2+ and GPLv3+", "Red Hat, Inc.", "", "GnuPG IPC library", "d736d854811fde43a638567619ff859c"},
			{intRef(), "keyutils-libs", "1.5.10", "9.el8", "s390x", "keyutils-1.5.10-9.el8.src.rpm", 43486, "GPLv2+ and LGPLv2+", "Red Hat, Inc.", "", "Key utilities library", "c01b279f51978fd6eb46e65740252cf0"},
			{intRef(), "p11-kit-trust", "0.23.22", "1.el8", "s390x", "p11-kit-0.23.22-1.el8.src.rpm", 479001, "BSD", "Red Hat, Inc.", "", "System trust module from p11-kit", "848c2bbccc3db39ed4057bacb13c5f3b"},
			{intRef(), "grep", "3.1", "6.el8", "s390x", "grep-3.1-6.el8.src.rpm", 843669, "GPLv3+", "Red Hat, Inc.", "", "Pattern matching utilities", "9fdf8d7b863c12c8ba427984f93eaaa9"},
			{intRef(1), "dbus-libs", "1.12.8", "18.el8", "s390x", "dbus-1.12.8-18.el8.src.rpm", 400512, "(GPLv2+ or AFL) and GPLv2+", "Red Hat, Inc.", "", "Libraries for accessing D-BUS", "4a82f02554ca0dfcca4cfcbbc292df61"},
			{intRef(1), "dbus-tools", "1.12.8", "18.el8", "s390x", "dbus-1.12.8-18.el8.src.rpm", 122507, "(GPLv2+ or AFL) and GPLv2+", "Red Hat, Inc.", "", "D-BUS Tools and Utilities", "8199d51f2e504816bd44805a2db94968"},
			{intRef(1), "gdbm", "1.18", "1.el8", "s390x", "gdbm-1.18-1.el8.src.rpm", 413369, "GPLv3+", "Red Hat, Inc.", "", "A GNU set of database routines which use extensible hashing", "6c062a12b6ab4e1853c89f1d37b63e55"},
			{intRef(2), "shadow-utils", "4.6", "16.el8", "s390x", "shadow-utils-4.6-16.el8.src.rpm", 4167677, "BSD and GPLv2+", "Red Hat, Inc.", "", "Utilities for managing accounts and shadow password files", "c12df68c5fef7859881d8af4654ba912"},
			{intRef(), "libpsl", "0.20.2", "6.el8", "s390x", "libpsl-0.20.2-6.el8.src.rpm", 70372, "MIT", "Red Hat, Inc.", "", "C library for the Publix Suffix List", "3c166e9937f047996d0f46cf090e6b60"},
			{intRef(), "which", "2.21", "17.el8", "s390x", "which-2.21-17.el8.src.rpm", 86670, "GPLv3", "Red Hat, Inc.", "", "Displays where a particular program in your path is located", "2226d103b42871541faad78b784c2200"},
			{intRef(), "cracklib", "2.9.6", "15.el8", "s390x", "cracklib-2.9.6-15.el8.src.rpm", 241319, "LGPLv2+", "Red Hat, Inc.", "", "A password-checking library", "46e4dff165a874aa5b16f23e739b099a"},
			{intRef(), "acl", "2.2.53", "1.el8", "s390x", "acl-2.2.53-1.el8.src.rpm", 209884, "GPLv2+", "Red Hat, Inc.", "", "Access control list utilities", "aa0bfa3ab554c508b2b6f26f9be0d673"},
			{intRef(), "nettle", "3.4.1", "7.el8", "s390x", "nettle-3.4.1-7.el8.src.rpm", 591705, "LGPLv3+ or GPLv2+", "Red Hat, Inc.", "", "A low-level cryptographic library", "cd3272e873bd256b75ee0c7997cf7fe9"},
			{intRef(), "libksba", "1.3.5", "7.el8", "s390x", "libksba-1.3.5-7.el8.src.rpm", 351823, "(LGPLv3+ or GPLv2+) and GPLv3+", "Red Hat, Inc.", "", "CMS and X.509 library", "044b90f8dc9bc1ae7eadacfe7bf26728"},
			{intRef(), "libnghttp2", "1.33.0", "3.el8_2.1", "s390x", "nghttp2-1.33.0-3.el8_2.1.src.rpm", 167740, "MIT", "Red Hat, Inc.", "", "A library implementing the HTTP/2 protocol", "86b28474cbed029c964a89872740a0ee"},
			{intRef(), "libseccomp", "2.5.2", "1.el8", "s390x", "libseccomp-2.5.2-1.el8.src.rpm", 177341, "LGPLv2", "Red Hat, Inc.", "", "Enhanced seccomp library", "d7c227cdfc31676125bed5ee01fddc28"},
			{intRef(), "gawk", "4.2.1", "4.el8", "s390x", "gawk-4.2.1-4.el8.src.rpm", 2768302, "GPLv3+ and GPLv2+ and LGPLv2+ and BSD", "Red Hat, Inc.", "", "The GNU version of the AWK text processing utility", "8a58f1f5dd6e0294457abc66868d4917"},
			{intRef(), "libnsl2", "1.2.0", "2.20180605git4a062cf.el8", "s390x", "libnsl2-1.2.0-2.20180605git4a062cf.el8.src.rpm", 152210, "BSD and LGPLv2+", "Red Hat, Inc.", "", "Public client interface library for NIS(YP) and NIS+", "176de9c0f3594cb7c21b7783913d45c9"},
			{intRef(), "krb5-libs", "1.18.2", "14.el8", "s390x", "krb5-1.18.2-14.el8.src.rpm", 2291939, "MIT", "Red Hat, Inc.", "", "The non-admin shared libraries used by Kerberos 5", "dccc5a9c0988f5eed5e2ec87c1b6f252"},
			{intRef(), "crypto-policies", "20211116", "1.gitae470d6.el8", "noarch", "crypto-policies-20211116-1.gitae470d6.el8.src.rpm", 73257, "LGPLv2+", "Red Hat, Inc.", "", "System-wide crypto policies", "89f5d5fb9c9db9e0e36df7071320cc3f"},
			{intRef(), "platform-python-setuptools", "39.2.0", "6.el8", "noarch", "python-setuptools-39.2.0-6.el8.src.rpm", 2930503, "MIT", "Red Hat, Inc.", "", "Easily build and distribute Python 3 packages", "5934b19fab076ef2ddcf46b5b547dccb"},
			{intRef(), "python3-libs", "3.6.8", "45.el8", "s390x", "python3-3.6.8-45.el8.src.rpm", 32457398, "Python", "Red Hat, Inc.", "", "Python runtime libraries", "4d941bbfab88cc8eb1b0c0d0e1e7e325"},
			{intRef(), "libpwquality", "1.4.4", "3.el8", "s390x", "libpwquality-1.4.4-3.el8.src.rpm", 399312, "BSD or GPLv2+", "Red Hat, Inc.", "", "A library for password generation and password quality checking", "aeca2b505e49039af71e01a56084d7a5"},
			{intRef(), "python3-six", "1.11.0", "8.el8", "noarch", "python-six-1.11.0-8.el8.src.rpm", 100282, "MIT", "Red Hat, Inc.", "", "Python 2 and 3 compatibility utilities", "3a46b593df0c4e15061a4340cb8503bd"},
			{intRef(1), "python3-dateutil", "2.6.1", "6.el8", "noarch", "python-dateutil-2.6.1-6.el8.src.rpm", 596677, "BSD", "Red Hat, Inc.", "", "Powerful extensions to the standard datetime module", "229da5815b24a30b853e16880b69a193"},
			{intRef(), "glib2", "2.56.4", "158.el8", "s390x", "glib2-2.56.4-158.el8.src.rpm", 12460592, "LGPLv2+", "Red Hat, Inc.", "", "A library of handy utility functions", "8d5617e5ac172672a565a5d226a8fbfb"},
			{intRef(), "librhsm", "0.0.3", "4.el8", "s390x", "librhsm-0.0.3-4.el8.src.rpm", 75824, "LGPLv2+", "Red Hat, Inc.", "", "Red Hat Subscription Manager library", "64134a4e923912545547151fbb443986"},
			{intRef(), "dbus-glib", "0.110", "2.el8", "s390x", "dbus-glib-0.110-2.el8.src.rpm", 379662, "AFL and GPLv2+", "Red Hat, Inc.", "", "GLib bindings for D-Bus", "c5321c80e4237f2b07a514e58db398d6"},
			{intRef(), "gobject-introspection", "1.56.1", "1.el8", "s390x", "gobject-introspection-1.56.1-1.el8.src.rpm", 898391, "GPLv2+, LGPLv2+, MIT", "Red Hat, Inc.", "", "Introspection system for GObject-based libraries", "7f9b6c5f64ff633b7163a4a2286fd530"},
			{intRef(), "virt-what", "1.18", "13.el8", "s390x", "virt-what-1.18-13.el8.src.rpm", 47599, "GPLv2+", "Red Hat, Inc.", "", "Detect if we are running in a virtual machine", "1867430df5004ab8e2e526bcaeb35e2e"},
			{intRef(), "openldap", "2.4.46", "18.el8", "s390x", "openldap-2.4.46-18.el8.src.rpm", 1042339, "OpenLDAP", "Red Hat, Inc.", "", "LDAP support libraries", "e5f12ca2789e19f99a2a7f478bb03688"},
			{intRef(), "passwd", "0.80", "4.el8", "s390x", "passwd-0.80-4.el8.src.rpm", 441893, "BSD or GPL+", "Red Hat, Inc.", "", "An utility for setting or changing passwords using PAM", "fa7746b748fa5c385fac3f0c3fb0ed9e"},
			{intRef(), "libdb-utils", "5.3.28", "42.el8_4", "s390x", "libdb-5.3.28-42.el8_4.src.rpm", 393695, "BSD and LGPLv2 and Sleepycat", "Red Hat, Inc.", "", "Command line tools for managing Berkeley DB databases", "9204541b54f49485dcf24d4745895273"},
			{intRef(), "python3-libcomps", "0.1.18", "1.el8", "s390x", "libcomps-0.1.18-1.el8.src.rpm", 151158, "GPLv2+", "Red Hat, Inc.", "", "Python 3 bindings for libcomps library", "83f033fe32ffa373ed42e87a29eaac39"},
			{intRef(), "python3-decorator", "4.2.1", "2.el8", "noarch", "python-decorator-4.2.1-2.el8.src.rpm", 47871, "BSD", "Red Hat, Inc.", "", "Module to simplify usage of decorators in python3", "8e25edc54aba4c26c4962fe3df53cca3"},
			{intRef(), "python3-inotify", "0.9.6", "13.el8", "noarch", "python-inotify-0.9.6-13.el8.src.rpm", 248598, "MIT", "Red Hat, Inc.", "", "Monitor filesystem events with Python under Linux", "7edde5ba6f70c5bdd543fd295738b6f2"},
			{intRef(), "python3-urllib3", "1.24.2", "5.el8", "noarch", "python-urllib3-1.24.2-5.el8.src.rpm", 620045, "MIT", "Red Hat, Inc.", "", "Python3 HTTP library with thread-safe connection pooling and file post", "3e72ec4338ad0bb966222d5dcb2333b7"},
			{intRef(), "python3-cloud-what", "1.28.29", "3.el8", "s390x", "subscription-manager-1.28.29-3.el8.src.rpm", 71971, "GPLv2", "Red Hat, Inc.", "", "Python package for detection of public cloud provider", "7583c04746608946010913171021d2a5"},
			{intRef(), "kmod-libs", "25", "19.el8", "s390x", "kmod-25-19.el8.src.rpm", 134664, "LGPLv2+", "Red Hat, Inc.", "", "Libraries to handle kernel module loading and unloading", "74ecc416eac6acf7f36bd1742d9c285e"},
			{intRef(8), "device-mapper-libs", "1.02.181", "3.el8", "s390x", "lvm2-2.03.14-3.el8.src.rpm", 435871, "LGPLv2", "Red Hat, Inc.", "", "Device-mapper shared library", "ea89335ec4bae64bc7cdb81fb5c6e8f5"},
			{intRef(), "elfutils-default-yama-scope", "0.186", "1.el8", "noarch", "elfutils-0.186-1.el8.src.rpm", 1810, "GPLv2+ or LGPLv3+", "Red Hat, Inc.", "", "Default yama attach scope sysctl setting", "87b23f2ecf9788e683e7ac5acd538bb2"},
			{intRef(1), "dbus-common", "1.12.8", "18.el8", "noarch", "dbus-1.12.8-18.el8.src.rpm", 11131, "(GPLv2+ or AFL) and GPLv2+", "Red Hat, Inc.", "", "D-BUS message bus configuration", "b8007efcad40343a4013764d4ee2a3c9"},
			{intRef(), "systemd-pam", "239", "58.el8", "s390x", "systemd-239-58.el8.src.rpm", 937656, "LGPLv2+ and MIT and GPLv2+", "Red Hat, Inc.", "", "systemd PAM module", "ce586872f8fd616afd3af881c9c26912"},
			{intRef(1), "dbus", "1.12.8", "18.el8", "s390x", "dbus-1.12.8-18.el8.src.rpm", 0, "(GPLv2+ or AFL) and GPLv2+", "Red Hat, Inc.", "", "D-BUS message bus", "04aa526487f167a4a97275136309b767"},
			{intRef(), "tpm2-tss", "2.3.2", "4.el8", "s390x", "tpm2-tss-2.3.2-4.el8.src.rpm", 1145142, "BSD", "Red Hat, Inc.", "", "TPM2.0 Software Stack", "541c7e7ac2f1b43b796875f2e6d7a4b5"},
			{intRef(), "libyaml", "0.1.7", "5.el8", "s390x", "libyaml-0.1.7-5.el8.src.rpm", 127126, "MIT", "Red Hat, Inc.", "", "YAML 1.1 parser and emitter written in C", "2eee61dd3f9ba4709670fa5d22afb917"},
			{intRef(), "gnupg2", "2.2.20", "2.el8", "s390x", "gnupg2-2.2.20-2.el8.src.rpm", 10294211, "GPLv3+", "Red Hat, Inc.", "", "Utility for secure communication and data storage", "f9e4eddfcd74ed1848afd212aa181bcc"},
			{intRef(), "python3-gpg", "1.13.1", "11.el8", "s390x", "gpgme-1.13.1-11.el8.src.rpm", 1436090, "LGPLv2+ and GPLv3+", "Red Hat, Inc.", "", "gpgme bindings for Python 3", "f8afd1cd7600967814570a1ae1d65965"},
			{intRef(), "libssh", "0.9.6", "3.el8", "s390x", "libssh-0.9.6-3.el8.src.rpm", 529471, "LGPLv2+", "Red Hat, Inc.", "", "A library implementing the SSH protocol", "e033939d7ea30816df54be5e1ff158cf"},
			{intRef(), "librepo", "1.14.2", "1.el8", "s390x", "librepo-1.14.2-1.el8.src.rpm", 212388, "LGPLv2+", "Red Hat, Inc.", "", "Repodata downloading library", "d197b4b9a0a212f942f73bffe0f19a5e"},
			{intRef(), "curl", "7.61.1", "22.el8", "s390x", "curl-7.61.1-22.el8.src.rpm", 704757, "MIT", "Red Hat, Inc.", "", "A utility for getting files from remote servers (FTP, HTTP, and others)", "a46df6680004ddb40039067114be71b0"},
			{intRef(), "rpm-libs", "4.14.3", "23.el8", "s390x", "rpm-4.14.3-23.el8.src.rpm", 783184, "GPLv2+ and LGPLv2+ with exceptions", "Red Hat, Inc.", "", "Libraries for manipulating RPM packages", "16a03a8ba4373a4ffaec8b6503c338e8"},
			{intRef(), "libsolv", "0.7.20", "1.el8", "s390x", "libsolv-0.7.20-1.el8.src.rpm", 875691, "BSD", "Red Hat, Inc.", "", "Package dependency solver", "42c5d2fc3b3c7bbb0f5f58d524bc0aa1"},
			{intRef(), "python3-libdnf", "0.63.0", "8.el8", "s390x", "libdnf-0.63.0-8.el8.src.rpm", 4025722, "LGPLv2+", "Red Hat, Inc.", "", "Python 3 bindings for the libdnf library.", "58e4fa8fe76c8f573d1590dd8a935200"},
			{intRef(), "rpm-build-libs", "4.14.3", "23.el8", "s390x", "rpm-4.14.3-23.el8.src.rpm", 223408, "GPLv2+ and LGPLv2+ with exceptions", "Red Hat, Inc.", "", "Libraries for building and signing RPM packages", "22f56df678dde2fdb47b7fd0471b7008"},
			{intRef(), "python3-subscription-manager-rhsm", "1.28.29", "3.el8", "s390x", "subscription-manager-1.28.29-3.el8.src.rpm", 424353, "GPLv2", "Red Hat, Inc.", "", "A Python library to communicate with a Red Hat Unified Entitlement Platform", "20efc9e73f419aa3bb7e612b9639870d"},
			{intRef(), "dnf-data", "4.7.0", "8.el8", "noarch", "dnf-4.7.0-8.el8.src.rpm", 38535, "GPLv2+", "Red Hat, Inc.", "", "Common data and configuration files for DNF", "177c70907234cd187da4154e74d40ef3"},
			{intRef(), "dnf", "4.7.0", "8.el8", "noarch", "dnf-4.7.0-8.el8.src.rpm", 2086072, "GPLv2+", "Red Hat, Inc.", "", "Package manager", "9db4297efa5f683da42850d895e00deb"},
			{intRef(), "dnf-plugin-subscription-manager", "1.28.29", "3.el8", "s390x", "subscription-manager-1.28.29-3.el8.src.rpm", 93730, "GPLv2", "Red Hat, Inc.", "", "Subscription Manager plugins for DNF", "1b9ae260fe7e33740d5165bdedda4661"},
			{intRef(), "yum", "4.7.0", "8.el8", "noarch", "dnf-4.7.0-8.el8.src.rpm", 76588, "GPLv2+", "Red Hat, Inc.", "", "Package manager", "ebf15c2e794278a36ab7af293bfa2463"},
			{intRef(2), "tar", "1.30", "5.el8", "s390x", "tar-1.30-5.el8.src.rpm", 2937393, "GPLv3+", "Red Hat, Inc.", "", "A GNU file archiving program", "59ad7788360a2252611add2efbb858f0"},
			{intRef(1), "findutils", "4.6.0", "20.el8", "s390x", "findutils-4.6.0-20.el8.src.rpm", 1838089, "GPLv3+", "Red Hat, Inc.", "", "The GNU versions of find utilities (find and xargs)", "531b7d79a84fa700b60771b326c77bf4"},
			{intRef(), "rootfiles", "8.1", "22.el8", "noarch", "rootfiles-8.1-22.el8.src.rpm", 599, "Public Domain", "Red Hat, Inc.", "", "The basic required files for the root user's directory", "95e9e04d2d86a4595e51bb74119338c6"},
			{intRef(), "gpg-pubkey", "d4082792", "5b32db75", "", "", 0, "pubkey", "", "", "gpg(Red Hat, Inc. (auxiliary key) <security@redhat.com>)", ""},
		}

		return toPackageInfo(pkgs)
	}

	// docker run --rm -it registry.suse.com/bci/bci-minimal:15.3 bash
	// rpm -qa --queryformat "\{%{EPOCH}, \"%{NAME}\", \"%{VERSION}\", \"%{RELEASE}\", \"%{ARCH}\", \"%{SOURCERPM}\", %{SIZE}, \"%{LICENSE}\", \"%{VENDOR}\", \"\", \"%{SUMMARY}\", \"%{SIGMD5}\"\},\n" | sed "s/^{(none)/{intRef()/g" | sed -r 's/^\{([0-9]+),/{intRef(\1),/' | sed "s/(none)/0/g"
	SLE15WithNDB = func() []*PackageInfo {
		pkgs := []*commonPackageInfo{
			{intRef(), "system-user-root", "20190513", "3.3.1", "noarch", "system-user-root-20190513-3.3.1.src.rpm", 186, "MIT", "SUSE LLC <https://www.suse.com/>", "", "System user and group root", "2c914632f9c42c48931f4ed8182b59a2"},
			{intRef(), "filesystem", "15.0", "11.3.2", "x86_64", "filesystem-15.0-11.3.2.src.rpm", 535, "MIT", "SUSE LLC <https://www.suse.com/>", "", "Basic Directory Layout", "8575edea6a9d5cf23b0c3a9cf65f5c71"},
			{intRef(), "glibc", "2.31", "9.3.2", "x86_64", "glibc-2.31-9.3.2.src.rpm", 6183407, "LGPL-2.1-or-later AND LGPL-2.1-or-later WITH GCC-exception-2.0 AND GPL-2.0-or-later", "SUSE LLC <https://www.suse.com/>", "", "Standard Shared Libraries (from the GNU C Library)", "832606b4923996b2868e2d687e188e0a"},
			{intRef(), "libpcre1", "8.45", "20.10.1", "x86_64", "pcre-8.45-20.10.1.src.rpm", 938295, "BSD-3-Clause", "SUSE LLC <https://www.suse.com/>", "", "A library for Perl-compatible regular expressions", "3ee5f6578bffab82f6447ce0866d4ad2"},
			{intRef(), "libgmp10", "6.1.2", "4.6.1", "x86_64", "gmp-6.1.2-4.6.1.src.rpm", 711445, "LGPL-3.0-or-later OR GPL-2.0-or-later", "SUSE LLC <https://www.suse.com/>", "", "Shared library for the GNU MP Library", "4521ddeda36e13c7c03d202bcbdd1635"},
			{intRef(), "libgcc_s1", "11.2.1+git610", "1.3.9", "x86_64", "gcc11-11.2.1+git610-1.3.9.src.rpm", 101024, "GPL-3.0-or-later WITH GCC-exception-3.1", "SUSE LLC <https://www.suse.com/>", "", "C compiler runtime library", "c16da8b31ebc5ed3b25403ed3033615d"},
			{intRef(), "libcap2", "2.26", "4.6.1", "x86_64", "libcap-2.26-4.6.1.src.rpm", 39224, "BSD-3-Clause or GPL-2.0", "SUSE LLC <https://www.suse.com/>", "", "Library for Capabilities (linux-privs) Support", "9a3cfe53cd724b794d7db73e20ffc446"},
			{intRef(), "libstdc++6", "11.2.1+git610", "1.3.9", "x86_64", "gcc11-11.2.1+git610-1.3.9.src.rpm", 2161776, "GPL-3.0-or-later WITH GCC-exception-3.1", "SUSE LLC <https://www.suse.com/>", "", "The standard C++ shared library", "4d690fedab92e99e0e799fc66d6c917a"},
			{intRef(), "libncurses6", "6.1", "5.9.1", "x86_64", "ncurses-6.1-5.9.1.src.rpm", 1116008, "MIT", "SUSE LLC <https://www.suse.com/>", "", "Terminal control library", "3fe02c56f76525f254fd405f9ea2acd4"},
			{intRef(), "terminfo-base", "6.1", "5.9.1", "x86_64", "ncurses-6.1-5.9.1.src.rpm", 1179602, "MIT", "SUSE LLC <https://www.suse.com/>", "", "A terminal descriptions database", "c3595e3ee328100d80a4c7b2b7bbe441"},
			{intRef(), "libattr1", "2.4.47", "2.19", "x86_64", "attr-2.4.47-2.19.src.rpm", 46233, "GPL-2.0-or-later AND LGPL-2.1-or-later", "SUSE LLC <https://www.suse.com/>", "", "A dynamic library for filesystem extended attribute support", "4076970e6f886b0720829e01446863d6"},
			{intRef(), "libselinux1", "3.0", "1.31", "x86_64", "libselinux-3.0-1.31.src.rpm", 159424, "SUSE-Public-Domain", "SUSE LLC <https://www.suse.com/>", "", "SELinux runtime library", "390011b0d24e3c543e0ba274d5fed66a"},
			{intRef(), "libreadline7", "7.0", "19.6.1", "x86_64", "bash-4.4-19.6.1.src.rpm", 396195, "GPL-3.0-or-later", "SUSE LLC <https://www.suse.com/>", "", "The Readline Library", "9c2c4e1efdccd9a40344ec0a4011b6d7"},
			{intRef(), "bash", "4.4", "19.6.1", "x86_64", "bash-4.4-19.6.1.src.rpm", 1114706, "GPL-3.0-or-later", "SUSE LLC <https://www.suse.com/>", "", "The GNU Bourne-Again Shell", "6e08d527bb9ceeec2821351806422886"},
			{intRef(), "libacl1", "2.2.52", "4.3.1", "x86_64", "acl-2.2.52-4.3.1.src.rpm", 35424, "GPL-2.0+ and LGPL-2.1+", "SUSE LLC <https://www.suse.com/>", "", "A dynamic library for accessing POSIX Access Control Lists", "88298f8be6d0a6fdc190913475582aed"},
			{intRef(), "coreutils", "8.32", "3.2.1", "x86_64", "coreutils-8.32-3.2.1.src.rpm", 6488992, "GPL-3.0-or-later", "SUSE LLC <https://www.suse.com/>", "", "GNU Core Utilities", "f65db0eb9b127b01db3b377ad50958d0"},
			{intRef(), "sles-release", "15.3", "55.4.1", "x86_64", "sles-release-15.3-55.4.1.src.rpm", 342491, "MIT", "SUSE LLC <https://www.suse.com/>", "", "SUSE Linux Enterprise Server 15 SP3", "33d7ae539dde6bf923257fd08366806f"},
			{intRef(), "ca-certificates-mozilla-prebuilt", "2.44", "21.1", "noarch", "ca-certificates-mozilla-prebuilt-2.44-21.1.src.rpm", 836576, "MPL-2.0", "SUSE LLC <https://www.suse.com/>", "", "Pre-built CA certificates for OpenSSL", "0efd3b127865f5737dc656d36d3974a4"},
			{intRef(), "libgpg-error0", "1.29", "1.8", "x86_64", "libgpg-error-1.29-1.8.src.rpm", 565983, "GPL-2.0-or-later AND LGPL-2.1-or-later", "SUSE LLC <https://www.suse.com/>", "", "Library That Defines Common Error Values for All GnuPG Components", "8234710df3df24d8e188db1e877e5411"},
			{intRef(), "libpopt0", "1.16", "3.22", "x86_64", "popt-1.16-3.22.src.rpm", 124686, "MIT", "SUSE LLC <https://www.suse.com/>", "", "A C library for parsing command line parameters", "fb89834a85c7bc6d36a7d0bb92bb7d6e"},
			{intRef(), "file-magic", "5.32", "7.14.1", "noarch", "file-5.32-7.14.1.src.rpm", 5916442, "BSD-2-Clause", "SUSE LLC <https://www.suse.com/>", "", "Database for libmagic to help identify files", "c62f735607e8d1ca531a4fa6ebfa6ebe"},
			{intRef(), "libbz2-1", "1.0.6", "5.11.1", "x86_64", "bzip2-1.0.6-5.11.1.src.rpm", 120168, "BSD-3-Clause", "SUSE LLC <https://www.suse.com/>", "", "The bzip2 runtime library", "99cd2e04ea02f2f1bdfefdb33f538ea7"},
			{intRef(), "liblua5_3-5", "5.3.6", "3.6.1", "x86_64", "lua53-5.3.6-3.6.1.src.rpm", 237296, "MIT", "SUSE LLC <https://www.suse.com/>", "", "The Lua integration library", "24d36e34dde0f0a697d73cbafb0de5fd"},
			{intRef(), "liblzma5", "5.2.3", "4.3.1", "x86_64", "xz-5.2.3-4.3.1.src.rpm", 235576, "SUSE-Public-Domain", "SUSE LLC <https://www.suse.com/>", "", "Lempel–Ziv–Markov chain algorithm compression library", "05c135f27cd9be1ff19110c7495187c5"},
			{intRef(), "libz1", "1.2.11", "3.21.1", "x86_64", "zlib-1.2.11-3.21.1.src.rpm", 110685, "Zlib", "SUSE LLC <https://www.suse.com/>", "", "Library implementing the DEFLATE compression algorithm", "0287d06ad8d707ef12da44205a4fa559"},
			{intRef(), "libzstd1", "1.4.4", "1.6.1", "x86_64", "zstd-1.4.4-1.6.1.src.rpm", 682141, "BSD-3-Clause AND GPL-2.0-only", "SUSE LLC <https://www.suse.com/>", "", "Zstd compression library", "673eb06865824393b012aa4b144cd858"},
			{intRef(), "libmagic1", "5.32", "7.14.1", "x86_64", "file-5.32-7.14.1.src.rpm", 138472, "BSD-2-Clause", "SUSE LLC <https://www.suse.com/>", "", "Library for heuristic file type identification", "cfdbe836881a4f04e081c2766ec41e70"},
			{intRef(), "libdw1", "0.168", "4.5.3", "x86_64", "elfutils-0.168-4.5.3.src.rpm", 294456, "SUSE-GPL-2.0-with-OSI-exception", "SUSE LLC <https://www.suse.com/>", "", "Library to access DWARF debugging information", "3132144092d764e1452a1fce402de682"},
			{intRef(), "libebl-plugins", "0.168", "4.5.3", "x86_64", "elfutils-0.168-4.5.3.src.rpm", 372800, "SUSE-GPL-2.0-with-OSI-exception", "SUSE LLC <https://www.suse.com/>", "", "Architecture backends for libebl", "77d2257b85891c90490b2eded7d1ba07"},
			{intRef(), "libelf1", "0.168", "4.5.3", "x86_64", "elfutils-0.168-4.5.3.src.rpm", 96880, "SUSE-GPL-2.0-with-OSI-exception", "SUSE LLC <https://www.suse.com/>", "", "Library to read and write ELF files", "b9143a8e73c930603842e0a6b8c4d628"},
			{intRef(), "libcrypt1", "4.4.15", "2.51", "x86_64", "libxcrypt-4.4.15-2.51.src.rpm", 265241, "LGPL-2.1-or-later AND BSD-2-Clause AND BSD-3-Clause AND SUSE-Public-Domain", "SUSE LLC <https://www.suse.com/>", "", "Extended crypt library for DES, MD5, Blowfish and others", "cb7fdcf8d1cdc296ab30e887498347d5"},
			{intRef(), "perl-base", "5.26.1", "15.87", "x86_64", "perl-5.26.1-15.87.src.rpm", 4299811, "Artistic-1.0 or GPL-2.0+", "SUSE LLC <https://www.suse.com/>", "", "The Perl interpreter", "ccf54da23f8a7670ffc96afbfe03a03d"},
			{intRef(), "libgcrypt20", "1.8.2", "8.39.1", "x86_64", "libgcrypt-1.8.2-8.39.1.src.rpm", 1198761, "GPL-2.0+ AND LGPL-2.1+", "SUSE LLC <https://www.suse.com/>", "", "The GNU Crypto Library", "8b1a4e30dd4a5fbccd3c0aeec4f192f3"},
			{intRef(), "rpm-config-SUSE", "1", "5.6.1", "noarch", "rpm-config-SUSE-1-5.6.1.src.rpm", 38001, "GPL-2.0-or-later", "SUSE LLC <https://www.suse.com/>", "", "SUSE specific RPM configuration files", "86783c2a6c00c83f786ccf32e4b9b0df"},
			{intRef(), "rpm-ndb", "4.14.3", "40.1", "x86_64", "rpm-ndb-4.14.3-40.1.src.rpm", 3132579, "GPL-2.0-or-later", "SUSE LLC <https://www.suse.com/>", "", "The RPM Package Manager, ndb only version", "e8f3d423d58de0bfcffd2014cb5088fd"},
		}

		return toPackageInfo(pkgs)
	}

	// docker run --rm -it fedora:35 bash
	// rpm -qa --queryformat "\{%{EPOCH}, \"%{NAME}\", \"%{VERSION}\", \"%{RELEASE}\", \"%{ARCH}\", \"%{SOURCERPM}\", %{SIZE}, \"%{LICENSE}\", \"%{VENDOR}\", \"\", \"%{SUMMARY}\", \"%{SIGMD5}\"\},\n" | sed "s/^{(none)/{intRef()/g" | sed -r 's/^\{([0-9]+),/{intRef(\1),/' | sed "s/(none)/0/g"
	Fedora35WithSQLite3 = func() []*PackageInfo {
		pkgs := []*commonPackageInfo{
			{intRef(), "libgcc", "11.2.1", "1.fc35", "x86_64", "gcc-11.2.1-1.fc35.src.rpm", 194980, "GPLv3+ and GPLv3+ with exceptions and GPLv2+ with exceptions and LGPLv2+ and BSD", "Fedora Project", "", "GCC version 11 shared support library", "34477f066df520bcc8baf572e3923490"},
			{intRef(), "crypto-policies", "20210819", "1.gitd0fdcfb.fc35", "noarch", "crypto-policies-20210819-1.gitd0fdcfb.fc35.src.rpm", 86107, "LGPLv2+", "Fedora Project", "", "System-wide crypto policies", "8b615ce53d4afc9bd0c8d2fce41c7a3b"},
			{intRef(), "tzdata", "2021e", "1.fc35", "noarch", "tzdata-2021e-1.fc35.src.rpm", 1800709, "Public Domain", "Fedora Project", "", "Timezone data", "c45f9f773d61e48a37530109463e4601"},
			{intRef(), "fedora-release-identity-container", "35", "35", "noarch", "fedora-release-35-35.src.rpm", 1512, "MIT", "Fedora Project", "", "Package providing the identity for Fedora Container Base Image", "98b1c485f19c02796c78a283f57e0201"},
			{intRef(), "python-setuptools-wheel", "57.4.0", "1.fc35", "noarch", "python-setuptools-57.4.0-1.fc35.src.rpm", 596568, "MIT and (BSD or ASL 2.0)", "Fedora Project", "", "The setuptools wheel", "d607a75ddbf395abbf12517f48ab08b5"},
			{intRef(), "publicsuffix-list-dafsa", "20210518", "2.fc35", "noarch", "publicsuffix-list-20210518-2.fc35.src.rpm", 68815, "MPLv2.0", "Fedora Project", "", "Cross-vendor public domain suffix database in DAFSA form", "66e8314a663c969e724c10861c660788"},
			{intRef(), "pcre2-syntax", "10.37", "4.fc35", "noarch", "pcre2-10.37-4.fc35.src.rpm", 222822, "BSD", "Fedora Project", "", "Documentation for PCRE2 regular expressions", "a015c1f8b22a7a74972ff28a9cb653b4"},
			{intRef(), "ncurses-base", "6.2", "8.20210508.fc35", "noarch", "ncurses-6.2-8.20210508.fc35.src.rpm", 307293, "MIT", "Fedora Project", "", "Descriptions of common terminals", "6fc56839d735263a9cfc0886c8d62377"},
			{intRef(), "libssh-config", "0.9.6", "1.fc35", "noarch", "libssh-0.9.6-1.fc35.src.rpm", 277, "LGPLv2+", "Fedora Project", "", "Configuration files for libssh", "197f4d57cb353d8586957553e84c8980"},
			{intRef(), "libreport-filesystem", "2.15.2", "6.fc35", "noarch", "libreport-2.15.2-6.fc35.src.rpm", 0, "GPLv2+", "Fedora Project", "", "Filesystem layout for libreport", "050efa6d25216fc05f0a73ae4fb98fac"},
			{intRef(), "dnf-data", "4.9.0", "1.fc35", "noarch", "dnf-4.9.0-1.fc35.src.rpm", 38568, "GPLv2+", "Fedora Project", "", "Common data and configuration files for DNF", "68f6e6b1c26b6ab5ad529fa556e7a9dc"},
			{intRef(), "fedora-gpg-keys", "35", "1", "noarch", "fedora-repos-35-1.src.rpm", 118311, "MIT", "Fedora Project", "", "Fedora RPM keys", "ee1123a814f64132a29d25320d02d69e"},
			{intRef(), "fedora-release-container", "35", "35", "noarch", "fedora-release-35-35.src.rpm", 0, "MIT", "Fedora Project", "", "Base package for Fedora container specific default configurations", "c4e34ee575908ebd0d0ad973595bc990"},
			{intRef(), "fedora-repos", "35", "1", "noarch", "fedora-repos-35-1.src.rpm", 4597, "MIT", "Fedora Project", "", "Fedora package repositories", "274b046d52c4a96de0c27a55dae7014d"},
			{intRef(), "fedora-release-common", "35", "35", "noarch", "fedora-release-35-35.src.rpm", 17557, "MIT", "Fedora Project", "", "Fedora release files", "981a0ca37122c2e7b76319274cf22a24"},
			{intRef(), "setup", "2.13.9.1", "2.fc35", "noarch", "setup-2.13.9.1-2.fc35.src.rpm", 736053, "Public Domain", "Fedora Project", "", "A set of system configuration and setup files", "64f8b1c53f47bf22f25324c6e700cc72"},
			{intRef(), "filesystem", "3.14", "7.fc35", "x86_64", "filesystem-3.14-7.fc35.src.rpm", 106, "Public Domain", "Fedora Project", "", "The basic directory layout for a Linux system", "1c0ec91e10e7d2346448d02ce12c4d47"},
			{intRef(), "basesystem", "11", "12.fc35", "noarch", "basesystem-11-12.fc35.src.rpm", 0, "Public Domain", "Fedora Project", "", "The skeleton package which defines a simple Fedora system", "af618e2cf6a1583a176bb5e45b0a5068"},
			{intRef(), "bash", "5.1.8", "2.fc35", "x86_64", "bash-5.1.8-2.fc35.src.rpm", 7739604, "GPLv3+", "Fedora Project", "", "The GNU Bourne Again shell", "b5f0cc215104983edd613065e450c0e3"},
			{intRef(), "ncurses-libs", "6.2", "8.20210508.fc35", "x86_64", "ncurses-6.2-8.20210508.fc35.src.rpm", 996375, "MIT", "Fedora Project", "", "Ncurses libraries", "2f994f573415c7206570f1717dcf626b"},
			{intRef(), "glibc-common", "2.34", "8.fc35", "x86_64", "glibc-2.34-8.fc35.src.rpm", 1089462, "LGPLv2+ and LGPLv2+ with exceptions and GPLv2+ and GPLv2+ with exceptions and BSD and Inner-Net and ISC and Public Domain and GFDL", "Fedora Project", "", "Common binaries and locale data for glibc", "2d88c26d38ada9eb1f8b45b7689ff4ac"},
			{intRef(), "glibc-minimal-langpack", "2.34", "8.fc35", "x86_64", "glibc-2.34-8.fc35.src.rpm", 0, "LGPLv2+ and LGPLv2+ with exceptions and GPLv2+ and GPLv2+ with exceptions and BSD and Inner-Net and ISC and Public Domain and GFDL", "Fedora Project", "", "Minimal language packs for glibc.", "08eea10dc5ac6e3f9143420d3a4dbc25"},
			{intRef(), "glibc", "2.34", "8.fc35", "x86_64", "glibc-2.34-8.fc35.src.rpm", 6237291, "LGPLv2+ and LGPLv2+ with exceptions and GPLv2+ and GPLv2+ with exceptions and BSD and Inner-Net and ISC and Public Domain and GFDL", "Fedora Project", "", "The GNU libc libraries", "89244b1bc4f73c57c212389a496bf8af"},
			{intRef(), "zlib", "1.2.11", "30.fc35", "x86_64", "zlib-1.2.11-30.fc35.src.rpm", 203449, "zlib and Boost", "Fedora Project", "", "Compression and decompression library", "fc88ceb52649360b8d871331220b0f6c"},
			{intRef(), "bzip2-libs", "1.0.8", "9.fc35", "x86_64", "bzip2-1.0.8-9.fc35.src.rpm", 78660, "BSD", "Fedora Project", "", "Libraries for applications using bzip2", "0d16644a1c137a65208e4cf440c4d832"},
			{intRef(), "xz-libs", "5.2.5", "7.fc35", "x86_64", "xz-5.2.5-7.fc35.src.rpm", 181437, "Public Domain", "Fedora Project", "", "Libraries for decoding LZMA compression", "a020e7464332c7f3873c34907eacff50"},
			{intRef(), "libzstd", "1.5.0", "2.fc35", "x86_64", "zstd-1.5.0-2.fc35.src.rpm", 1028163, "BSD and GPLv2", "Fedora Project", "", "Zstd shared library", "c56a81660a167665720337b2d0c97070"},
			{intRef(), "sqlite-libs", "3.36.0", "3.fc35", "x86_64", "sqlite-3.36.0-3.fc35.src.rpm", 1334505, "Public Domain", "Fedora Project", "", "Shared library for the sqlite3 embeddable SQL database engine.", "eebb4534e82961cc46ef159d4072bca8"},
			{intRef(1), "gmp", "6.2.0", "7.fc35", "x86_64", "gmp-6.2.0-7.fc35.src.rpm", 809478, "LGPLv3+ or GPLv2+", "Fedora Project", "", "GNU arbitrary precision library", "90e5db3d624adbdb7d59835ab68fb865"},
			{intRef(), "libcap", "2.48", "3.fc35", "x86_64", "libcap-2.48-3.fc35.src.rpm", 180511, "BSD or GPLv2", "Fedora Project", "", "Library for getting and setting POSIX.1e capabilities", "fa377911e225ceb086daf490d0c94ab7"},
			{intRef(), "popt", "1.18", "6.fc35", "x86_64", "popt-1.18-6.fc35.src.rpm", 130256, "MIT", "Fedora Project", "", "C library for parsing command line parameters", "43cab6097d127d0d309a21fb5e811d4c"},
			{intRef(), "libgpg-error", "1.43", "1.fc35", "x86_64", "libgpg-error-1.43-1.fc35.src.rpm", 851181, "LGPLv2+", "Fedora Project", "", "Library for error values used by GnuPG components", "2acf9c28449a9ab3cbe56c5109f5b957"},
			{intRef(), "libxml2", "2.9.12", "6.fc35", "x86_64", "libxml2-2.9.12-6.fc35.src.rpm", 1923894, "MIT", "Fedora Project", "", "Library providing XML and HTML support", "58e6c4b8f0daf9cb8cf8355e9855a6ce"},
			{intRef(), "libcom_err", "1.46.3", "1.fc35", "x86_64", "e2fsprogs-1.46.3-1.fc35.src.rpm", 68441, "MIT", "Fedora Project", "", "Common error description library", "eb56b897164901257a96b54c9f5442f7"},
			{intRef(), "libstdc++", "11.2.1", "1.fc35", "x86_64", "gcc-11.2.1-1.fc35.src.rpm", 2476520, "GPLv3+ and GPLv3+ with exceptions and GPLv2+ with exceptions and LGPLv2+ and BSD", "Fedora Project", "", "GNU Standard C++ Library", "0d3819bb7da679b2f0c68209b7550a82"},
			{intRef(), "libxcrypt", "4.4.26", "4.fc35", "x86_64", "libxcrypt-4.4.26-4.fc35.src.rpm", 275090, "LGPLv2+ and BSD and Public Domain", "Fedora Project", "", "Extended crypt library for descrypt, md5crypt, bcrypt, and others", "c9fa7936f993432ab8574bdbec6c006e"},
			{intRef(), "lua-libs", "5.4.3", "2.fc35", "x86_64", "lua-5.4.3-2.fc35.src.rpm", 555606, "MIT", "Fedora Project", "", "Libraries for lua", "708062639b1ec59c853e3949ecb655b4"},
			{intRef(), "elfutils-libelf", "0.185", "5.fc35", "x86_64", "elfutils-0.185-5.fc35.src.rpm", 992174, "GPLv2+ or LGPLv3+", "Fedora Project", "", "Library to read and write ELF files", "34da460b225bcbd122e42ed8c6ca2415"},
			{intRef(), "file-libs", "5.40", "9.fc35", "x86_64", "file-5.40-9.fc35.src.rpm", 8529778, "BSD", "Fedora Project", "", "Libraries for applications using libmagic", "5cefc7d6d4ee8e91b61b7dbef6f9626d"},
			{intRef(), "libattr", "2.5.1", "3.fc35", "x86_64", "attr-2.5.1-3.fc35.src.rpm", 29341, "LGPLv2+", "Fedora Project", "", "Dynamic library for extended attribute support", "a17b4483f4a01b5eb6d01f41842da285"},
			{intRef(), "libacl", "2.3.1", "2.fc35", "x86_64", "acl-2.3.1-2.fc35.src.rpm", 41090, "LGPLv2+", "Fedora Project", "", "Dynamic library for access control list support", "7f91f200578996693fbd6ba53b745348"},
			{intRef(), "libffi", "3.1", "29.fc35", "x86_64", "libffi-3.1-29.fc35.src.rpm", 56872, "MIT", "Fedora Project", "", "A portable foreign function interface library", "b4412b95f19fdc9fc212a6219d9bd508"},
			{intRef(), "p11-kit", "0.23.22", "4.fc35", "x86_64", "p11-kit-0.23.22-4.fc35.src.rpm", 1659536, "BSD", "Fedora Project", "", "Library for loading and sharing PKCS#11 modules", "2e7cda635f37cded07d6b2c83065ba9a"},
			{intRef(), "libunistring", "0.9.10", "14.fc35", "x86_64", "libunistring-0.9.10-14.fc35.src.rpm", 1642923, "GPLv2+ or LGPLv3+", "Fedora Project", "", "GNU Unicode string library", "fb4cd52955e96d4fe989ece3153119fa"},
			{intRef(), "libidn2", "2.3.2", "3.fc35", "x86_64", "libidn2-2.3.2-3.fc35.src.rpm", 291720, "(GPLv2+ or LGPLv3+) and GPLv3+", "Fedora Project", "", "Library to support IDNA2008 internationalized domain names", "88d523b2e03784d487b1fa9d4b91c5b8"},
			{intRef(), "libuuid", "2.37.2", "1.fc35", "x86_64", "util-linux-2.37.2-1.fc35.src.rpm", 34389, "BSD", "Fedora Project", "", "Universally unique ID library", "338b40b1919cd29fb68a0a49af5a9ee7"},
			{intRef(), "readline", "8.1", "3.fc35", "x86_64", "readline-8.1-3.fc35.src.rpm", 492684, "GPLv3+", "Fedora Project", "", "A library for editing typed command lines", "aaf61524862c6beeb77380c15d658a93"},
			{intRef(), "libassuan", "2.5.5", "3.fc35", "x86_64", "libassuan-2.5.5-3.fc35.src.rpm", 171069, "LGPLv2+ and GPLv3+", "Fedora Project", "", "GnuPG IPC library", "d4a9b9e1572325bf831fd6a85f2dc939"},
			{intRef(), "expat", "2.4.1", "2.fc35", "x86_64", "expat-2.4.1-2.fc35.src.rpm", 295041, "MIT", "Fedora Project", "", "An XML parser library", "1daae32a6bd9b6d68dce05f6cc47415a"},
			{intRef(), "json-c", "0.15", "2.fc35", "x86_64", "json-c-0.15-2.fc35.src.rpm", 79583, "MIT", "Fedora Project", "", "JSON implementation in C", "a8a4268ee30918be55244041058454dc"},
			{intRef(), "keyutils-libs", "1.6.1", "3.fc35", "x86_64", "keyutils-1.6.1-3.fc35.src.rpm", 55801, "GPLv2+ and LGPLv2+", "Fedora Project", "", "Key utilities library", "535a1f1ca9e121cd9574c95b07820bb7"},
			{intRef(), "libsigsegv", "2.13", "3.fc35", "x86_64", "libsigsegv-2.13-3.fc35.src.rpm", 50250, "GPLv2+", "Fedora Project", "", "Library for handling page faults in user mode", "f9aa2f7104c70b331c044360a70664f4"},
			{intRef(), "libsmartcols", "2.37.2", "1.fc35", "x86_64", "util-linux-2.37.2-1.fc35.src.rpm", 135371, "LGPLv2+", "Fedora Project", "", "Formatting library for ls-like programs.", "c4d9a07d0ade1c261ce99b1d7a258180"},
			{intRef(), "libtasn1", "4.16.0", "6.fc35", "x86_64", "libtasn1-4.16.0-6.fc35.src.rpm", 183868, "GPLv3+ and LGPLv2+", "Fedora Project", "", "The ASN.1 library used in GNUTLS", "de5f598de38a61b7288a400c1398b8e3"},
			{intRef(), "pcre", "8.45", "1.fc35", "x86_64", "pcre-8.45-1.fc35.src.rpm", 539220, "BSD", "Fedora Project", "", "Perl-compatible regular expression library", "5514c5f24b4d2325f93065903618e59e"},
			{intRef(), "grep", "3.6", "4.fc35", "x86_64", "grep-3.6-4.fc35.src.rpm", 857744, "GPLv3+", "Fedora Project", "", "Pattern matching utilities", "156e66ef6b4acfcc051a5c68590478c1"},
			{intRef(1), "gdbm-libs", "1.22", "1.fc35", "x86_64", "gdbm-1.22-1.fc35.src.rpm", 128594, "GPLv3+", "Fedora Project", "", "Libraries files for gdbm", "44c8f9a2f908cd026db026cff5bcb9dc"},
			{intRef(), "libsepol", "3.3", "2.fc35", "x86_64", "libsepol-3.3-2.fc35.src.rpm", 755891, "LGPLv2+", "Fedora Project", "", "SELinux binary policy manipulation library", "2315ac21f0edf70323532a17badac156"},
			{intRef(), "libcomps", "0.1.18", "1.fc35", "x86_64", "libcomps-0.1.18-1.fc35.src.rpm", 214999, "GPLv2+", "Fedora Project", "", "Comps XML file manipulation library", "8b7103efef0dde4c254212396768bdc6"},
			{intRef(), "libpsl", "0.21.1", "4.fc35", "x86_64", "libpsl-0.21.1-4.fc35.src.rpm", 78520, "MIT", "Fedora Project", "", "C library for the Publix Suffix List", "418fc994aecd296a0e9f3a900a46f7c5"},
			{intRef(), "mpdecimal", "2.5.1", "2.fc35", "x86_64", "mpdecimal-2.5.1-2.fc35.src.rpm", 246955, "BSD", "Fedora Project", "", "Library for general decimal arithmetic", "5a959017ea843dc9cd834137e17dcbc5"},
			{intRef(), "libgcrypt", "1.9.4", "1.fc35", "x86_64", "libgcrypt-1.9.4-1.fc35.src.rpm", 1392828, "LGPLv2+", "Fedora Project", "", "A general-purpose cryptography library", "3de42e7502c099e402c6eff1fd477153"},
			{intRef(), "libksba", "1.6.0", "2.fc35", "x86_64", "libksba-1.6.0-2.fc35.src.rpm", 401600, "(LGPLv3+ or GPLv2+) and GPLv3+", "Fedora Project", "", "CMS and X.509 library", "18e1c1cdf40e9811107c906a74ee3817"},
			{intRef(), "mpfr", "4.1.0", "8.fc35", "x86_64", "mpfr-4.1.0-8.fc35.src.rpm", 802431, "LGPLv3+", "Fedora Project", "", "C library for multiple-precision floating-point computations", "de62fb0bd1ab4954a5a3e601e4f95b58"},
			{intRef(), "gawk", "5.1.0", "4.fc35", "x86_64", "gawk-5.1.0-4.fc35.src.rpm", 1684030, "GPLv3+ and GPLv2+ and LGPLv2+ and BSD", "Fedora Project", "", "The GNU version of the AWK text processing utility", "4c2057f0c12df0632024f9b90f075c28"},
			{intRef(), "nettle", "3.7.3", "2.fc35", "x86_64", "nettle-3.7.3-2.fc35.src.rpm", 735221, "LGPLv3+ or GPLv2+", "Fedora Project", "", "A low-level cryptographic library", "4a7d4cbe7785c81277e7dba802d858f7"},
			{intRef(), "alternatives", "1.19", "1.fc35", "x86_64", "chkconfig-1.19-1.fc35.src.rpm", 63264, "GPLv2", "Fedora Project", "", "A tool to maintain symbolic links determining default commands", "5cd393a56ec2efe3a8dd3f9abe55f043"},
			{intRef(), "p11-kit-trust", "0.23.22", "4.fc35", "x86_64", "p11-kit-0.23.22-4.fc35.src.rpm", 451087, "BSD", "Fedora Project", "", "System trust module from p11-kit", "70a611a55f6a52dbf5e6966daff44ae9"},
			{intRef(), "gnutls", "3.7.2", "2.fc35", "x86_64", "gnutls-3.7.2-2.fc35.src.rpm", 3141270, "GPLv3+ and LGPLv2+", "Fedora Project", "", "A TLS protocol implementation", "10504fd7a0e1fe097092920f6f761f2e"},
			{intRef(), "libbrotli", "1.0.9", "6.fc35", "x86_64", "brotli-1.0.9-6.fc35.src.rpm", 784274, "MIT", "Fedora Project", "", "Library for brotli lossless compression algorithm", "d45caad095b8be733739ddc36bbf1a3b"},
			{intRef(), "libcap-ng", "0.8.2", "6.fc35", "x86_64", "libcap-ng-0.8.2-6.fc35.src.rpm", 75012, "LGPLv2+", "Fedora Project", "", "Alternate posix capabilities library", "ff4a1b7b3bbd3aa7ea01faac637a5fe9"},
			{intRef(), "audit-libs", "3.0.6", "1.fc35", "x86_64", "audit-3.0.6-1.fc35.src.rpm", 307177, "LGPLv2+", "Fedora Project", "", "Dynamic library for libaudit", "4a16d6b199563985221afd8823981737"},
			{intRef(), "libdb", "5.3.28", "50.fc35", "x86_64", "libdb-5.3.28-50.fc35.src.rpm", 1922782, "BSD and LGPLv2 and Sleepycat", "Fedora Project", "", "The Berkeley DB database library for C", "035be9f77c8bd1ff1d600a3712c99e9b"},
			{intRef(), "libeconf", "0.4.0", "2.fc35", "x86_64", "libeconf-0.4.0-2.fc35.src.rpm", 46171, "MIT", "Fedora Project", "", "Enhanced config file parser library", "1d425d5dc6879bacad175542465b0706"},
			{intRef(), "libgomp", "11.2.1", "1.fc35", "x86_64", "gcc-11.2.1-1.fc35.src.rpm", 413740, "GPLv3+ and GPLv3+ with exceptions and GPLv2+ with exceptions and LGPLv2+ and BSD", "Fedora Project", "", "GCC OpenMP v4.5 shared support library", "c47d93388cbf7161da2bec5d7ba96da5"},
			{intRef(), "libnghttp2", "1.45.1", "1.fc35", "x86_64", "nghttp2-1.45.1-1.fc35.src.rpm", 162468, "MIT", "Fedora Project", "", "A library implementing the HTTP/2 protocol", "0d6fd9b7c3f410856cb296843f1840b7"},
			{intRef(), "libverto", "0.3.2", "2.fc35", "x86_64", "libverto-0.3.2-2.fc35.src.rpm", 30277, "MIT", "Fedora Project", "", "Main loop abstraction library", "cbb25f656c2c22c557ee83ad0d80dfb1"},
			{intRef(), "libyaml", "0.2.5", "6.fc35", "x86_64", "libyaml-0.2.5-6.fc35.src.rpm", 138211, "MIT", "Fedora Project", "", "YAML 1.1 parser and emitter written in C", "f9ff1a0ffc3d4cc9ecaa76ba39afeb4d"},
			{intRef(), "lz4-libs", "1.9.3", "3.fc35", "x86_64", "lz4-1.9.3-3.fc35.src.rpm", 145387, "GPLv2+ and BSD", "Fedora Project", "", "Libaries for lz4", "35f02696af287b52c0a8ccd356299a79"},
			{intRef(), "npth", "1.6", "7.fc35", "x86_64", "npth-1.6-7.fc35.src.rpm", 50531, "LGPLv2+", "Fedora Project", "", "The New GNU Portable Threads library", "bdf927678bcfaadeba6fa20b62f90734"},
			{intRef(), "pcre2", "10.37", "4.fc35", "x86_64", "pcre2-10.37-4.fc35.src.rpm", 633138, "BSD", "Fedora Project", "", "Perl-compatible regular expression library", "4a0a07ff00922aaafb5701a06810da7f"},
			{intRef(), "libselinux", "3.3", "1.fc35", "x86_64", "libselinux-3.3-1.fc35.src.rpm", 169365, "Public Domain", "Fedora Project", "", "SELinux library and simple utilities", "70515806983a64a7e56cc2c0fa15b3f1"},
			{intRef(), "sed", "4.8", "8.fc35", "x86_64", "sed-4.8-8.fc35.src.rpm", 813479, "GPLv3+", "Fedora Project", "", "A GNU stream text editor", "18a95fcfe58463e71717b520d2412a8a"},
			{intRef(), "libsemanage", "3.3", "1.fc35", "x86_64", "libsemanage-3.3-1.fc35.src.rpm", 303824, "LGPLv2+", "Fedora Project", "", "SELinux binary policy manipulation library", "1bf5012ddec7e2232047c99fd9746df4"},
			{intRef(2), "shadow-utils", "4.9", "7.fc35", "x86_64", "shadow-utils-4.9-7.fc35.src.rpm", 3836461, "BSD and GPLv2+", "Fedora Project", "", "Utilities for managing accounts and shadow password files", "7527e3c5c074744d47d53c6c8b688caa"},
			{intRef(2), "vim-minimal", "8.2.3642", "1.fc35", "x86_64", "vim-8.2.3642-1.fc35.src.rpm", 1529767, "Vim and MIT", "Fedora Project", "", "A minimal version of the VIM editor", "76d6ee9a527357f3cc8f7d2d21656285"},
			{intRef(), "elfutils-default-yama-scope", "0.185", "5.fc35", "noarch", "elfutils-0.185-5.fc35.src.rpm", 1810, "GPLv2+ or LGPLv3+", "Fedora Project", "", "Default yama attach scope sysctl setting", "1a4ed61c32688e5f2f93ce33501c4a16"},
			{intRef(), "elfutils-libs", "0.185", "5.fc35", "x86_64", "elfutils-0.185-5.fc35.src.rpm", 709117, "GPLv2+ or LGPLv3+", "Fedora Project", "", "Libraries to handle compiled objects", "2c1ab9675c840b1b828052157ed4cc9f"},
			{intRef(), "coreutils-common", "8.32", "31.fc35", "x86_64", "coreutils-8.32-31.fc35.src.rpm", 10880210, "GPLv3+", "Fedora Project", "", "coreutils common optional components", "1199baa383f946e4b01413a52cf5dbfe"},
			{intRef(1), "openssl-libs", "1.1.1l", "2.fc35", "x86_64", "openssl-1.1.1l-2.fc35.src.rpm", 3855396, "OpenSSL and ASL 2.0", "Fedora Project", "", "A general purpose cryptography library with TLS implementation", "3131af4dea3019fcfd0f040e7af67dcb"},
			{intRef(), "coreutils", "8.32", "31.fc35", "x86_64", "coreutils-8.32-31.fc35.src.rpm", 6040898, "GPLv3+", "Fedora Project", "", "A set of basic GNU tools commonly used in shell scripts", "31edc65f732179c418c5bfc50d097aa9"},
			{intRef(), "ca-certificates", "2021.2.50", "3.fc35", "noarch", "ca-certificates-2021.2.50-3.fc35.src.rpm", 939948, "Public Domain", "Fedora Project", "", "The Mozilla CA root certificate bundle", "aff01f0137ea9195efe407be332a9c66"},
			{intRef(), "krb5-libs", "1.19.2", "2.fc35", "x86_64", "krb5-1.19.2-2.fc35.src.rpm", 2198421, "MIT", "Fedora Project", "", "The non-admin shared libraries used by Kerberos 5", "8cb025e5ab47044fb45e62bd7d5abe89"},
			{intRef(), "libtirpc", "1.3.2", "1.fc35", "x86_64", "libtirpc-1.3.2-1.fc35.src.rpm", 208122, "SISSL and BSD", "Fedora Project", "", "Transport Independent RPC Library", "774bb16f7bceb07e236edd74893c3da2"},
			{intRef(), "libnsl2", "1.3.0", "4.fc35", "x86_64", "libnsl2-1.3.0-4.fc35.src.rpm", 130270, "BSD and LGPLv2+", "Fedora Project", "", "Public client interface library for NIS(YP) and NIS+", "ec90470cea464c30c21887ef0b6b5f78"},
			{intRef(), "zchunk-libs", "1.1.15", "2.fc35", "x86_64", "zchunk-1.1.15-2.fc35.src.rpm", 90820, "BSD and MIT", "Fedora Project", "", "Zchunk library", "66829a62bfa036541114223927236c56"},
			{intRef(), "libfsverity", "1.4", "6.fc35", "x86_64", "fsverity-utils-1.4-6.fc35.src.rpm", 29672, "BSD", "Fedora Project", "", "Development package for fsverity-utils", "ca39a98e52bfd390757594501bfe9158"},
			{intRef(), "cyrus-sasl-lib", "2.1.27", "13.fc35", "x86_64", "cyrus-sasl-2.1.27-13.fc35.src.rpm", 2409736, "BSD with advertising", "Fedora Project", "", "Shared libraries needed by applications which use Cyrus SASL", "244cc72621a079a0444957ed952b692b"},
			{intRef(), "openldap", "2.4.59", "3.fc35", "x86_64", "openldap-2.4.59-3.fc35.src.rpm", 718501, "OpenLDAP", "Fedora Project", "", "LDAP support libraries", "0e77658e50841dafd342dee9da801f8c"},
			{intRef(), "gnupg2", "2.3.3", "1.fc35", "x86_64", "gnupg2-2.3.3-1.fc35.src.rpm", 9244445, "GPLv3+", "Fedora Project", "", "Utility for secure communication and data storage", "d87bc19f1b0546ab44302ed9a56b081f"},
			{intRef(), "gpgme", "1.15.1", "6.fc35", "x86_64", "gpgme-1.15.1-6.fc35.src.rpm", 573957, "LGPLv2+", "Fedora Project", "", "GnuPG Made Easy - high level crypto API", "55d1815dcc9dd566a95999a31334702c"},
			{intRef(), "libssh", "0.9.6", "1.fc35", "x86_64", "libssh-0.9.6-1.fc35.src.rpm", 513049, "LGPLv2+", "Fedora Project", "", "A library implementing the SSH protocol", "89d0f0ed09d6da4ba2ab98275c98f173"},
			{intRef(), "libcurl", "7.79.1", "1.fc35", "x86_64", "curl-7.79.1-1.fc35.src.rpm", 681030, "MIT", "Fedora Project", "", "A library for getting files from web servers", "d990c31b9e1e543f45e954204899997a"},
			{intRef(), "tpm2-tss", "3.1.0", "3.fc35", "x86_64", "tpm2-tss-3.1.0-3.fc35.src.rpm", 2227128, "BSD and TCGL", "Fedora Project", "", "TPM2.0 Software Stack", "37a74040701bebdccc86ea28632895c4"},
			{intRef(), "ima-evm-utils", "1.3.2", "3.fc35", "x86_64", "ima-evm-utils-1.3.2-3.fc35.src.rpm", 141126, "GPLv2", "Fedora Project", "", "IMA/EVM support utilities", "0981d3adc3feddae5d198d0df691e21a"},
			{intRef(), "curl", "7.79.1", "1.fc35", "x86_64", "curl-7.79.1-1.fc35.src.rpm", 723076, "MIT", "Fedora Project", "", "A utility for getting files from remote servers (FTP, HTTP, and others)", "8015ff9223a971c77a8721daa8f50796"},
			{intRef(), "python-pip-wheel", "21.2.3", "4.fc35", "noarch", "python-pip-21.2.3-4.fc35.src.rpm", 1220638, "MIT and Python and ASL 2.0 and BSD and ISC and LGPLv2 and MPLv2.0 and (ASL 2.0 or BSD)", "Fedora Project", "", "The pip wheel", "af8fd2c05e3b0764b53c3eb21b3f430f"},
			{intRef(), "python3", "3.10.0", "1.fc35", "x86_64", "python3.10-3.10.0-1.fc35.src.rpm", 33090, "Python", "Fedora Project", "", "Python 3.10 interpreter", "7bc13f0cc3c75abac26b8f89dc0d4eae"},
			{intRef(), "python3-libs", "3.10.0", "1.fc35", "x86_64", "python3.10-3.10.0-1.fc35.src.rpm", 33027906, "Python", "Fedora Project", "", "Python runtime libraries", "2a4c661cf66d7ad3d55601c742451dbc"},
			{intRef(), "python3-libcomps", "0.1.18", "1.fc35", "x86_64", "libcomps-0.1.18-1.fc35.src.rpm", 146971, "GPLv2+", "Fedora Project", "", "Python 3 bindings for libcomps library", "9344fc5a5274824bbb2cebd5e7341c23"},
			{intRef(), "python3-gpg", "1.15.1", "6.fc35", "x86_64", "gpgme-1.15.1-6.fc35.src.rpm", 1394334, "LGPLv2+", "Fedora Project", "", "gpgme bindings for Python 3", "87da740b4d4cd47f726b5876769db9df"},
			{intRef(), "gzip", "1.10", "5.fc35", "x86_64", "gzip-1.10-5.fc35.src.rpm", 357298, "GPLv3+ and GFDL", "Fedora Project", "", "GNU data compression program", "0c334e5cdc70ca2c94149b74b6771d59"},
			{intRef(), "cracklib", "2.9.6", "27.fc35", "x86_64", "cracklib-2.9.6-27.fc35.src.rpm", 251474, "LGPLv2+", "Fedora Project", "", "A password-checking library", "be873d814043808a88cd6ac0b984d7cd"},
			{intRef(), "libpwquality", "1.4.4", "6.fc35", "x86_64", "libpwquality-1.4.4-6.fc35.src.rpm", 415452, "BSD or GPLv2+", "Fedora Project", "", "A library for password generation and password quality checking", "2c4c69665af0cbea0ea4405af255304d"},
			{intRef(), "pam", "1.5.2", "5.fc35", "x86_64", "pam-1.5.2-5.fc35.src.rpm", 1947788, "BSD and GPLv2+", "Fedora Project", "", "An extensible library which provides authentication for applications", "370c1852e1e67c174496bb910457993d"},
			{intRef(), "libblkid", "2.37.2", "1.fc35", "x86_64", "util-linux-2.37.2-1.fc35.src.rpm", 230761, "LGPLv2+", "Fedora Project", "", "Block device ID library", "e8870cb597c3762b975d144dc65b5f8b"},
			{intRef(), "libmount", "2.37.2", "1.fc35", "x86_64", "util-linux-2.37.2-1.fc35.src.rpm", 311125, "LGPLv2+", "Fedora Project", "", "Device mounting library", "e6e2c13e7b0243aae0291e0b3a19b85c"},
			{intRef(), "glib2", "2.70.1", "1.fc35", "x86_64", "glib2-2.70.1-1.fc35.src.rpm", 13474770, "LGPLv2+", "Fedora Project", "", "A library of handy utility functions", "f005d25b77fe62d6c79d2e0b0f208e1a"},
			{intRef(), "librepo", "1.14.2", "1.fc35", "x86_64", "librepo-1.14.2-1.fc35.src.rpm", 241974, "LGPLv2+", "Fedora Project", "", "Repodata downloading library", "11553e792c5ce5f412bfcaa52b0c36f7"},
			{intRef(), "libarchive", "3.5.2", "2.fc35", "x86_64", "libarchive-3.5.2-2.fc35.src.rpm", 907245, "BSD", "Fedora Project", "", "A library for handling streaming archive formats", "e1660078d5ded318f65782bd9ba959f8"},
			{intRef(), "rpm-libs", "4.17.0", "1.fc35", "x86_64", "rpm-4.17.0-1.fc35.src.rpm", 775132, "GPLv2+ and LGPLv2+ with exceptions", "Fedora Project", "", "Libraries for manipulating RPM packages", "0f8e469cb6f09b48ceff0b49c7056d53"},
			{intRef(), "rpm", "4.17.0", "1.fc35", "x86_64", "rpm-4.17.0-1.fc35.src.rpm", 2948898, "GPLv2+", "Fedora Project", "", "The RPM package management system", "5b2ab90a61c97f9530725e88a786f2b0"},
			{intRef(), "libmodulemd", "2.13.0", "3.fc35", "x86_64", "libmodulemd-2.13.0-3.fc35.src.rpm", 733689, "MIT", "Fedora Project", "", "Module metadata manipulation library", "af896aabcbe9a142f5b808bd493ef218"},
			{intRef(), "libsolv", "0.7.19", "3.fc35", "x86_64", "libsolv-0.7.19-3.fc35.src.rpm", 897882, "BSD", "Fedora Project", "", "Package dependency solver", "4a3052d6738d16a276ae1930ecf26268"},
			{intRef(), "libdnf", "0.64.0", "1.fc35", "x86_64", "libdnf-0.64.0-1.fc35.src.rpm", 2045581, "LGPLv2+", "Fedora Project", "", "Library providing simplified C and Python API to libsolv", "101ec0844aef916129ce40d7874d5522"},
			{intRef(), "python3-libdnf", "0.64.0", "1.fc35", "x86_64", "libdnf-0.64.0-1.fc35.src.rpm", 3766775, "LGPLv2+", "Fedora Project", "", "Python 3 bindings for the libdnf library.", "69eca99cdd94d9b163461c7431405599"},
			{intRef(), "python3-hawkey", "0.64.0", "1.fc35", "x86_64", "libdnf-0.64.0-1.fc35.src.rpm", 310028, "LGPLv2+", "Fedora Project", "", "Python 3 bindings for the hawkey library", "33dcf93d1a4744ec72c46c19538a2c1a"},
			{intRef(), "rpm-build-libs", "4.17.0", "1.fc35", "x86_64", "rpm-4.17.0-1.fc35.src.rpm", 199518, "GPLv2+ and LGPLv2+ with exceptions", "Fedora Project", "", "Libraries for building RPM packages", "cfecb17fdb2063daabd812d01505f38b"},
			{intRef(), "rpm-sign-libs", "4.17.0", "1.fc35", "x86_64", "rpm-4.17.0-1.fc35.src.rpm", 40492, "GPLv2+ and LGPLv2+ with exceptions", "Fedora Project", "", "Libraries for signing RPM packages", "b260ded7bef805d0f2bfcbb9a9dd9527"},
			{intRef(), "python3-rpm", "4.17.0", "1.fc35", "x86_64", "rpm-4.17.0-1.fc35.src.rpm", 378257, "GPLv2+", "Fedora Project", "", "Python 3 bindings for apps which will manipulate RPM packages", "281a4a0daa66204d5b83e515d8b9d00d"},
			{intRef(), "python3-dnf", "4.9.0", "1.fc35", "noarch", "dnf-4.9.0-1.fc35.src.rpm", 1898937, "GPLv2+", "Fedora Project", "", "Python 3 interface to DNF", "f4d42de5bb07769bfa44f9ec6ea826ad"},
			{intRef(), "dnf", "4.9.0", "1.fc35", "noarch", "dnf-4.9.0-1.fc35.src.rpm", 2203005, "GPLv2+", "Fedora Project", "", "Package manager", "f754b5918e3d8774ae9522e55818a09b"},
			{intRef(), "yum", "4.9.0", "1.fc35", "noarch", "dnf-4.9.0-1.fc35.src.rpm", 22042, "GPLv2+", "Fedora Project", "", "Package manager", "557ae4521569b4fa4599dcf24ea48d97"},
			{intRef(), "sudo", "1.9.7p2", "2.fc35", "x86_64", "sudo-1.9.7p2-2.fc35.src.rpm", 4324216, "ISC", "Fedora Project", "", "Allows restricted root access for specified users", "b5f88dba9550cee464dc45d595a98c09"},
			{intRef(2), "tar", "1.34", "2.fc35", "x86_64", "tar-1.34-2.fc35.src.rpm", 3156278, "GPLv3+", "Fedora Project", "", "GNU file archiving program", "052b560465e9eebf3c4f7986aa10ee0b"},
			{intRef(), "fedora-repos-modular", "35", "1", "noarch", "fedora-repos-35-1.src.rpm", 4042, "MIT", "Fedora Project", "", "Fedora modular package repositories", "f80de6667a3eeb0688c854b07128f815"},
			{intRef(), "rootfiles", "8.1", "30.fc35", "noarch", "rootfiles-8.1-30.fc35.src.rpm", 817, "Public Domain", "Fedora Project", "", "The basic required files for the root user's directory", "2a5798670386c2b0ac45b20d55599029"},
			{intRef(), "gpg-pubkey", "9867c58f", "601c49ca", "", "", 0, "pubkey", "", "", "Fedora (35) <fedora-35-primary@fedoraproject.org> public key", ""},
		}

		return toPackageInfo(pkgs)
	}

	// docker run --rm -it fedora:35 bash
	// dnf -y install https://repo.mongodb.org/yum/redhat/8/mongodb-org/5.0/x86_64/RPMS/mongodb-org-shell-5.0.9-1.el8.x86_64.rpm
	// dnf -y install https://repo.mongodb.org/yum/redhat/8/mongodb-org/5.0/x86_64/RPMS/mongodb-cli-1.25.0.x86_64.rpm
	// rpm -qa --queryformat "\{%{EPOCH}, \"%{NAME}\", \"%{VERSION}\", \"%{RELEASE}\", \"%{ARCH}\", \"%{SOURCERPM}\", %{SIZE}, \"%{LICENSE}\", \"%{VENDOR}\", \"\", \"%{SUMMARY}\", \"%{SIGMD5}\"\},\n" | sed "s/^{(none)/{intRef()/g" | sed -r 's/^\{([0-9]+),/{intRef(\1),/' | sed "s/(none)/0/g"
	Fedora35PlusMongoDBWithSQLite3 = func() []*PackageInfo {
		pkgs := []*commonPackageInfo{
			{intRef(), "libgcc", "11.2.1", "9.fc35", "x86_64", "gcc-11.2.1-9.fc35.src.rpm", 198596, "GPLv3+ and GPLv3+ with exceptions and GPLv2+ with exceptions and LGPLv2+ and BSD", "Fedora Project", "", "GCC version 11 shared support library", "405419f81bb99a7298ea5a8eb55feded"},
			{intRef(), "crypto-policies", "20210819", "1.gitd0fdcfb.fc35", "noarch", "crypto-policies-20210819-1.gitd0fdcfb.fc35.src.rpm", 86107, "LGPLv2+", "Fedora Project", "", "System-wide crypto policies", "8b615ce53d4afc9bd0c8d2fce41c7a3b"},
			{intRef(), "tzdata", "2021e", "1.fc35", "noarch", "tzdata-2021e-1.fc35.src.rpm", 1800709, "Public Domain", "Fedora Project", "", "Timezone data", "c45f9f773d61e48a37530109463e4601"},
			{intRef(), "fedora-release-identity-container", "35", "36", "noarch", "fedora-release-35-36.src.rpm", 1512, "MIT", "Fedora Project", "", "Package providing the identity for Fedora Container Base Image", "20f0f0024639d7bc795669ed6c8e4ec9"},
			{intRef(2), "vim-data", "8.2.4386", "1.fc35", "noarch", "vim-8.2.4386-1.fc35.src.rpm", 17949, "Vim and MIT", "Fedora Project", "", "Shared data for Vi and Vim", "10ffd9509e7452784970d57b7a36a0f3"},
			{intRef(), "pcre2-syntax", "10.39", "1.fc35", "noarch", "pcre2-10.39-1.fc35.src.rpm", 229322, "BSD", "Fedora Project", "", "Documentation for PCRE2 regular expressions", "0f5ddd45f69f25609dad74a6be3bae36"},
			{intRef(), "python-setuptools-wheel", "57.4.0", "1.fc35", "noarch", "python-setuptools-57.4.0-1.fc35.src.rpm", 596568, "MIT and (BSD or ASL 2.0)", "Fedora Project", "", "The setuptools wheel", "d607a75ddbf395abbf12517f48ab08b5"},
			{intRef(), "publicsuffix-list-dafsa", "20210518", "2.fc35", "noarch", "publicsuffix-list-20210518-2.fc35.src.rpm", 68815, "MPLv2.0", "Fedora Project", "", "Cross-vendor public domain suffix database in DAFSA form", "66e8314a663c969e724c10861c660788"},
			{intRef(), "ncurses-base", "6.2", "8.20210508.fc35", "noarch", "ncurses-6.2-8.20210508.fc35.src.rpm", 307293, "MIT", "Fedora Project", "", "Descriptions of common terminals", "6fc56839d735263a9cfc0886c8d62377"},
			{intRef(), "libssh-config", "0.9.6", "1.fc35", "noarch", "libssh-0.9.6-1.fc35.src.rpm", 277, "LGPLv2+", "Fedora Project", "", "Configuration files for libssh", "197f4d57cb353d8586957553e84c8980"},
			{intRef(), "libreport-filesystem", "2.15.2", "6.fc35", "noarch", "libreport-2.15.2-6.fc35.src.rpm", 0, "GPLv2+", "Fedora Project", "", "Filesystem layout for libreport", "050efa6d25216fc05f0a73ae4fb98fac"},
			{intRef(), "dnf-data", "4.9.0", "1.fc35", "noarch", "dnf-4.9.0-1.fc35.src.rpm", 38568, "GPLv2+", "Fedora Project", "", "Common data and configuration files for DNF", "68f6e6b1c26b6ab5ad529fa556e7a9dc"},
			{intRef(), "fedora-gpg-keys", "35", "1", "noarch", "fedora-repos-35-1.src.rpm", 118311, "MIT", "Fedora Project", "", "Fedora RPM keys", "ee1123a814f64132a29d25320d02d69e"},
			{intRef(), "fedora-release-container", "35", "36", "noarch", "fedora-release-35-36.src.rpm", 0, "MIT", "Fedora Project", "", "Base package for Fedora container specific default configurations", "0750fc36a5609b4362ca162142874003"},
			{intRef(), "fedora-repos", "35", "1", "noarch", "fedora-repos-35-1.src.rpm", 4597, "MIT", "Fedora Project", "", "Fedora package repositories", "274b046d52c4a96de0c27a55dae7014d"},
			{intRef(), "fedora-release-common", "35", "36", "noarch", "fedora-release-35-36.src.rpm", 17657, "MIT", "Fedora Project", "", "Fedora release files", "bc6c1693bb5ef325dcb913853ee3500e"},
			{intRef(), "setup", "2.13.9.1", "2.fc35", "noarch", "setup-2.13.9.1-2.fc35.src.rpm", 736053, "Public Domain", "Fedora Project", "", "A set of system configuration and setup files", "64f8b1c53f47bf22f25324c6e700cc72"},
			{intRef(), "filesystem", "3.14", "7.fc35", "x86_64", "filesystem-3.14-7.fc35.src.rpm", 106, "Public Domain", "Fedora Project", "", "The basic directory layout for a Linux system", "1c0ec91e10e7d2346448d02ce12c4d47"},
			{intRef(), "basesystem", "11", "12.fc35", "noarch", "basesystem-11-12.fc35.src.rpm", 0, "Public Domain", "Fedora Project", "", "The skeleton package which defines a simple Fedora system", "af618e2cf6a1583a176bb5e45b0a5068"},
			{intRef(), "bash", "5.1.8", "2.fc35", "x86_64", "bash-5.1.8-2.fc35.src.rpm", 7739604, "GPLv3+", "Fedora Project", "", "The GNU Bourne Again shell", "b5f0cc215104983edd613065e450c0e3"},
			{intRef(), "ncurses-libs", "6.2", "8.20210508.fc35", "x86_64", "ncurses-6.2-8.20210508.fc35.src.rpm", 996375, "MIT", "Fedora Project", "", "Ncurses libraries", "2f994f573415c7206570f1717dcf626b"},
			{intRef(), "glibc-common", "2.34", "25.fc35", "x86_64", "glibc-2.34-25.fc35.src.rpm", 1089465, "LGPLv2+ and LGPLv2+ with exceptions and GPLv2+ and GPLv2+ with exceptions and BSD and Inner-Net and ISC and Public Domain and GFDL", "Fedora Project", "", "Common binaries and locale data for glibc", "76ce5e08081a9349defaf84e33783c9a"},
			{intRef(), "glibc-minimal-langpack", "2.34", "25.fc35", "x86_64", "glibc-2.34-25.fc35.src.rpm", 0, "LGPLv2+ and LGPLv2+ with exceptions and GPLv2+ and GPLv2+ with exceptions and BSD and Inner-Net and ISC and Public Domain and GFDL", "Fedora Project", "", "Minimal language packs for glibc.", "6aff6629d5ecf1f1d50929426d2ee243"},
			{intRef(), "glibc", "2.34", "25.fc35", "x86_64", "glibc-2.34-25.fc35.src.rpm", 6262755, "LGPLv2+ and LGPLv2+ with exceptions and GPLv2+ and GPLv2+ with exceptions and BSD and Inner-Net and ISC and Public Domain and GFDL", "Fedora Project", "", "The GNU libc libraries", "83bf2ef2a0b74c2e106a8abd488cae4c"},
			{intRef(), "zlib", "1.2.11", "30.fc35", "x86_64", "zlib-1.2.11-30.fc35.src.rpm", 203449, "zlib and Boost", "Fedora Project", "", "Compression and decompression library", "fc88ceb52649360b8d871331220b0f6c"},
			{intRef(), "bzip2-libs", "1.0.8", "9.fc35", "x86_64", "bzip2-1.0.8-9.fc35.src.rpm", 78660, "BSD", "Fedora Project", "", "Libraries for applications using bzip2", "0d16644a1c137a65208e4cf440c4d832"},
			{intRef(), "xz-libs", "5.2.5", "7.fc35", "x86_64", "xz-5.2.5-7.fc35.src.rpm", 181437, "Public Domain", "Fedora Project", "", "Libraries for decoding LZMA compression", "a020e7464332c7f3873c34907eacff50"},
			{intRef(), "libzstd", "1.5.2", "1.fc35", "x86_64", "zstd-1.5.2-1.fc35.src.rpm", 738483, "BSD and GPLv2", "Fedora Project", "", "Zstd shared library", "69ba361b97b48c0fa076ff189e890fb0"},
			{intRef(), "sqlite-libs", "3.36.0", "3.fc35", "x86_64", "sqlite-3.36.0-3.fc35.src.rpm", 1334505, "Public Domain", "Fedora Project", "", "Shared library for the sqlite3 embeddable SQL database engine.", "eebb4534e82961cc46ef159d4072bca8"},
			{intRef(1), "gmp", "6.2.0", "7.fc35", "x86_64", "gmp-6.2.0-7.fc35.src.rpm", 809478, "LGPLv3+ or GPLv2+", "Fedora Project", "", "GNU arbitrary precision library", "90e5db3d624adbdb7d59835ab68fb865"},
			{intRef(), "libcap", "2.48", "3.fc35", "x86_64", "libcap-2.48-3.fc35.src.rpm", 180511, "BSD or GPLv2", "Fedora Project", "", "Library for getting and setting POSIX.1e capabilities", "fa377911e225ceb086daf490d0c94ab7"},
			{intRef(), "popt", "1.18", "6.fc35", "x86_64", "popt-1.18-6.fc35.src.rpm", 130256, "MIT", "Fedora Project", "", "C library for parsing command line parameters", "43cab6097d127d0d309a21fb5e811d4c"},
			{intRef(), "libgpg-error", "1.43", "1.fc35", "x86_64", "libgpg-error-1.43-1.fc35.src.rpm", 851181, "LGPLv2+", "Fedora Project", "", "Library for error values used by GnuPG components", "2acf9c28449a9ab3cbe56c5109f5b957"},
			{intRef(), "libxml2", "2.9.12", "6.fc35", "x86_64", "libxml2-2.9.12-6.fc35.src.rpm", 1923894, "MIT", "Fedora Project", "", "Library providing XML and HTML support", "58e6c4b8f0daf9cb8cf8355e9855a6ce"},
			{intRef(), "libcom_err", "1.46.3", "1.fc35", "x86_64", "e2fsprogs-1.46.3-1.fc35.src.rpm", 68441, "MIT", "Fedora Project", "", "Common error description library", "eb56b897164901257a96b54c9f5442f7"},
			{intRef(), "libstdc++", "11.2.1", "9.fc35", "x86_64", "gcc-11.2.1-9.fc35.src.rpm", 2481086, "GPLv3+ and GPLv3+ with exceptions and GPLv2+ with exceptions and LGPLv2+ and BSD", "Fedora Project", "", "GNU Standard C++ Library", "385c15a432fef2b80dcc433073bbcaa4"},
			{intRef(), "libxcrypt", "4.4.28", "1.fc35", "x86_64", "libxcrypt-4.4.28-1.fc35.src.rpm", 275394, "LGPLv2+ and BSD and Public Domain", "Fedora Project", "", "Extended crypt library for descrypt, md5crypt, bcrypt, and others", "b53236060f750ee37972ae14ce1bd7e8"},
			{intRef(), "lua-libs", "5.4.4", "1.fc35", "x86_64", "lua-5.4.4-1.fc35.src.rpm", 555678, "MIT", "Fedora Project", "", "Libraries for lua", "7f20f5e135e45a95e02b95559feee819"},
			{intRef(), "file-libs", "5.40", "9.fc35", "x86_64", "file-5.40-9.fc35.src.rpm", 8529778, "BSD", "Fedora Project", "", "Libraries for applications using libmagic", "5cefc7d6d4ee8e91b61b7dbef6f9626d"},
			{intRef(), "elfutils-libelf", "0.186", "1.fc35", "x86_64", "elfutils-0.186-1.fc35.src.rpm", 1027449, "GPLv2+ or LGPLv3+", "Fedora Project", "", "Library to read and write ELF files", "98d6aee2b22f80d792e9f89666bf156a"},
			{intRef(), "libattr", "2.5.1", "3.fc35", "x86_64", "attr-2.5.1-3.fc35.src.rpm", 29341, "LGPLv2+", "Fedora Project", "", "Dynamic library for extended attribute support", "a17b4483f4a01b5eb6d01f41842da285"},
			{intRef(), "libacl", "2.3.1", "2.fc35", "x86_64", "acl-2.3.1-2.fc35.src.rpm", 41090, "LGPLv2+", "Fedora Project", "", "Dynamic library for access control list support", "7f91f200578996693fbd6ba53b745348"},
			{intRef(), "libffi", "3.1", "29.fc35", "x86_64", "libffi-3.1-29.fc35.src.rpm", 56872, "MIT", "Fedora Project", "", "A portable foreign function interface library", "b4412b95f19fdc9fc212a6219d9bd508"},
			{intRef(), "p11-kit", "0.23.22", "4.fc35", "x86_64", "p11-kit-0.23.22-4.fc35.src.rpm", 1659536, "BSD", "Fedora Project", "", "Library for loading and sharing PKCS#11 modules", "2e7cda635f37cded07d6b2c83065ba9a"},
			{intRef(), "libunistring", "0.9.10", "14.fc35", "x86_64", "libunistring-0.9.10-14.fc35.src.rpm", 1642923, "GPLv2+ or LGPLv3+", "Fedora Project", "", "GNU Unicode string library", "fb4cd52955e96d4fe989ece3153119fa"},
			{intRef(), "libidn2", "2.3.2", "3.fc35", "x86_64", "libidn2-2.3.2-3.fc35.src.rpm", 291720, "(GPLv2+ or LGPLv3+) and GPLv3+", "Fedora Project", "", "Library to support IDNA2008 internationalized domain names", "88d523b2e03784d487b1fa9d4b91c5b8"},
			{intRef(), "readline", "8.1", "3.fc35", "x86_64", "readline-8.1-3.fc35.src.rpm", 492684, "GPLv3+", "Fedora Project", "", "A library for editing typed command lines", "aaf61524862c6beeb77380c15d658a93"},
			{intRef(), "libuuid", "2.37.4", "1.fc35", "x86_64", "util-linux-2.37.4-1.fc35.src.rpm", 34389, "BSD", "Fedora Project", "", "Universally unique ID library", "f5b027f1cda706fb4b19fd96f22e8578"},
			{intRef(), "libassuan", "2.5.5", "3.fc35", "x86_64", "libassuan-2.5.5-3.fc35.src.rpm", 171069, "LGPLv2+ and GPLv3+", "Fedora Project", "", "GnuPG IPC library", "d4a9b9e1572325bf831fd6a85f2dc939"},
			{intRef(), "json-c", "0.15", "2.fc35", "x86_64", "json-c-0.15-2.fc35.src.rpm", 79583, "MIT", "Fedora Project", "", "JSON implementation in C", "a8a4268ee30918be55244041058454dc"},
			{intRef(), "keyutils-libs", "1.6.1", "3.fc35", "x86_64", "keyutils-1.6.1-3.fc35.src.rpm", 55801, "GPLv2+ and LGPLv2+", "Fedora Project", "", "Key utilities library", "535a1f1ca9e121cd9574c95b07820bb7"},
			{intRef(), "libsigsegv", "2.13", "3.fc35", "x86_64", "libsigsegv-2.13-3.fc35.src.rpm", 50250, "GPLv2+", "Fedora Project", "", "Library for handling page faults in user mode", "f9aa2f7104c70b331c044360a70664f4"},
			{intRef(), "libtasn1", "4.16.0", "6.fc35", "x86_64", "libtasn1-4.16.0-6.fc35.src.rpm", 183868, "GPLv3+ and LGPLv2+", "Fedora Project", "", "The ASN.1 library used in GNUTLS", "de5f598de38a61b7288a400c1398b8e3"},
			{intRef(), "pcre", "8.45", "1.fc35", "x86_64", "pcre-8.45-1.fc35.src.rpm", 539220, "BSD", "Fedora Project", "", "Perl-compatible regular expression library", "5514c5f24b4d2325f93065903618e59e"},
			{intRef(), "grep", "3.6", "4.fc35", "x86_64", "grep-3.6-4.fc35.src.rpm", 857744, "GPLv3+", "Fedora Project", "", "Pattern matching utilities", "156e66ef6b4acfcc051a5c68590478c1"},
			{intRef(), "expat", "2.4.4", "1.fc35", "x86_64", "expat-2.4.4-1.fc35.src.rpm", 300148, "MIT", "Fedora Project", "", "An XML parser library", "27b822f915271dd7ff7e03b14a4fa471"},
			{intRef(1), "gdbm-libs", "1.22", "1.fc35", "x86_64", "gdbm-1.22-1.fc35.src.rpm", 128594, "GPLv3+", "Fedora Project", "", "Libraries files for gdbm", "44c8f9a2f908cd026db026cff5bcb9dc"},
			{intRef(), "libsepol", "3.3", "2.fc35", "x86_64", "libsepol-3.3-2.fc35.src.rpm", 755891, "LGPLv2+", "Fedora Project", "", "SELinux binary policy manipulation library", "2315ac21f0edf70323532a17badac156"},
			{intRef(), "libsmartcols", "2.37.4", "1.fc35", "x86_64", "util-linux-2.37.4-1.fc35.src.rpm", 135379, "LGPLv2+", "Fedora Project", "", "Formatting library for ls-like programs.", "2d46924a9ddc8ab88d9004d8657b5c75"},
			{intRef(), "libcomps", "0.1.18", "1.fc35", "x86_64", "libcomps-0.1.18-1.fc35.src.rpm", 214999, "GPLv2+", "Fedora Project", "", "Comps XML file manipulation library", "8b7103efef0dde4c254212396768bdc6"},
			{intRef(), "libpsl", "0.21.1", "4.fc35", "x86_64", "libpsl-0.21.1-4.fc35.src.rpm", 78520, "MIT", "Fedora Project", "", "C library for the Publix Suffix List", "418fc994aecd296a0e9f3a900a46f7c5"},
			{intRef(), "mpdecimal", "2.5.1", "2.fc35", "x86_64", "mpdecimal-2.5.1-2.fc35.src.rpm", 246955, "BSD", "Fedora Project", "", "Library for general decimal arithmetic", "5a959017ea843dc9cd834137e17dcbc5"},
			{intRef(), "libgcrypt", "1.9.4", "1.fc35", "x86_64", "libgcrypt-1.9.4-1.fc35.src.rpm", 1392828, "LGPLv2+", "Fedora Project", "", "A general-purpose cryptography library", "3de42e7502c099e402c6eff1fd477153"},
			{intRef(), "libksba", "1.6.0", "2.fc35", "x86_64", "libksba-1.6.0-2.fc35.src.rpm", 401600, "(LGPLv3+ or GPLv2+) and GPLv3+", "Fedora Project", "", "CMS and X.509 library", "18e1c1cdf40e9811107c906a74ee3817"},
			{intRef(), "mpfr", "4.1.0", "8.fc35", "x86_64", "mpfr-4.1.0-8.fc35.src.rpm", 802431, "LGPLv3+", "Fedora Project", "", "C library for multiple-precision floating-point computations", "de62fb0bd1ab4954a5a3e601e4f95b58"},
			{intRef(), "gawk", "5.1.0", "4.fc35", "x86_64", "gawk-5.1.0-4.fc35.src.rpm", 1684030, "GPLv3+ and GPLv2+ and LGPLv2+ and BSD", "Fedora Project", "", "The GNU version of the AWK text processing utility", "4c2057f0c12df0632024f9b90f075c28"},
			{intRef(), "nettle", "3.7.3", "2.fc35", "x86_64", "nettle-3.7.3-2.fc35.src.rpm", 735221, "LGPLv3+ or GPLv2+", "Fedora Project", "", "A low-level cryptographic library", "4a7d4cbe7785c81277e7dba802d858f7"},
			{intRef(), "alternatives", "1.19", "1.fc35", "x86_64", "chkconfig-1.19-1.fc35.src.rpm", 63264, "GPLv2", "Fedora Project", "", "A tool to maintain symbolic links determining default commands", "5cd393a56ec2efe3a8dd3f9abe55f043"},
			{intRef(), "p11-kit-trust", "0.23.22", "4.fc35", "x86_64", "p11-kit-0.23.22-4.fc35.src.rpm", 451087, "BSD", "Fedora Project", "", "System trust module from p11-kit", "70a611a55f6a52dbf5e6966daff44ae9"},
			{intRef(), "gnutls", "3.7.2", "2.fc35", "x86_64", "gnutls-3.7.2-2.fc35.src.rpm", 3141270, "GPLv3+ and LGPLv2+", "Fedora Project", "", "A TLS protocol implementation", "10504fd7a0e1fe097092920f6f761f2e"},
			{intRef(), "libbrotli", "1.0.9", "6.fc35", "x86_64", "brotli-1.0.9-6.fc35.src.rpm", 784274, "MIT", "Fedora Project", "", "Library for brotli lossless compression algorithm", "d45caad095b8be733739ddc36bbf1a3b"},
			{intRef(), "libdb", "5.3.28", "50.fc35", "x86_64", "libdb-5.3.28-50.fc35.src.rpm", 1922782, "BSD and LGPLv2 and Sleepycat", "Fedora Project", "", "The Berkeley DB database library for C", "035be9f77c8bd1ff1d600a3712c99e9b"},
			{intRef(), "libeconf", "0.4.0", "2.fc35", "x86_64", "libeconf-0.4.0-2.fc35.src.rpm", 46171, "MIT", "Fedora Project", "", "Enhanced config file parser library", "1d425d5dc6879bacad175542465b0706"},
			{intRef(), "libnghttp2", "1.45.1", "1.fc35", "x86_64", "nghttp2-1.45.1-1.fc35.src.rpm", 162468, "MIT", "Fedora Project", "", "A library implementing the HTTP/2 protocol", "0d6fd9b7c3f410856cb296843f1840b7"},
			{intRef(), "libverto", "0.3.2", "2.fc35", "x86_64", "libverto-0.3.2-2.fc35.src.rpm", 30277, "MIT", "Fedora Project", "", "Main loop abstraction library", "cbb25f656c2c22c557ee83ad0d80dfb1"},
			{intRef(), "libyaml", "0.2.5", "6.fc35", "x86_64", "libyaml-0.2.5-6.fc35.src.rpm", 138211, "MIT", "Fedora Project", "", "YAML 1.1 parser and emitter written in C", "f9ff1a0ffc3d4cc9ecaa76ba39afeb4d"},
			{intRef(), "lz4-libs", "1.9.3", "3.fc35", "x86_64", "lz4-1.9.3-3.fc35.src.rpm", 145387, "GPLv2+ and BSD", "Fedora Project", "", "Libaries for lz4", "35f02696af287b52c0a8ccd356299a79"},
			{intRef(), "npth", "1.6", "7.fc35", "x86_64", "npth-1.6-7.fc35.src.rpm", 50531, "LGPLv2+", "Fedora Project", "", "The New GNU Portable Threads library", "bdf927678bcfaadeba6fa20b62f90734"},
			{intRef(), "libcap-ng", "0.8.2", "8.fc35", "x86_64", "libcap-ng-0.8.2-8.fc35.src.rpm", 75012, "LGPLv2+", "Fedora Project", "", "Alternate posix capabilities library", "41da9333ecb5667832c9b2a40e9e2323"},
			{intRef(), "audit-libs", "3.0.7", "2.fc35", "x86_64", "audit-3.0.7-2.fc35.src.rpm", 307185, "LGPLv2+", "Fedora Project", "", "Dynamic library for libaudit", "4f04d7b4481f98f23873438aff0f1190"},
			{intRef(), "libgomp", "11.2.1", "9.fc35", "x86_64", "gcc-11.2.1-9.fc35.src.rpm", 412830, "GPLv3+ and GPLv3+ with exceptions and GPLv2+ with exceptions and LGPLv2+ and BSD", "Fedora Project", "", "GCC OpenMP v4.5 shared support library", "eddb7af4f13d79fd62e26393cc237b95"},
			{intRef(), "pcre2", "10.39", "1.fc35", "x86_64", "pcre2-10.39-1.fc35.src.rpm", 633146, "BSD", "Fedora Project", "", "Perl-compatible regular expression library", "75d614b14d53b96c247f382adbd56d7c"},
			{intRef(), "libselinux", "3.3", "1.fc35", "x86_64", "libselinux-3.3-1.fc35.src.rpm", 169365, "Public Domain", "Fedora Project", "", "SELinux library and simple utilities", "70515806983a64a7e56cc2c0fa15b3f1"},
			{intRef(), "sed", "4.8", "8.fc35", "x86_64", "sed-4.8-8.fc35.src.rpm", 813479, "GPLv3+", "Fedora Project", "", "A GNU stream text editor", "18a95fcfe58463e71717b520d2412a8a"},
			{intRef(), "libsemanage", "3.3", "1.fc35", "x86_64", "libsemanage-3.3-1.fc35.src.rpm", 303824, "LGPLv2+", "Fedora Project", "", "SELinux binary policy manipulation library", "1bf5012ddec7e2232047c99fd9746df4"},
			{intRef(2), "shadow-utils", "4.9", "9.fc35", "x86_64", "shadow-utils-4.9-9.fc35.src.rpm", 3841041, "BSD and GPLv2+", "Fedora Project", "", "Utilities for managing accounts and shadow password files", "dbe74af1d330be8c97b1c1679bf99fd4"},
			{intRef(2), "vim-minimal", "8.2.4386", "1.fc35", "x86_64", "vim-8.2.4386-1.fc35.src.rpm", 1545045, "Vim and MIT", "Fedora Project", "", "A minimal version of the VIM editor", "40e6d96dc0a6a12f2b7000f8d084c452"},
			{intRef(), "elfutils-default-yama-scope", "0.186", "1.fc35", "noarch", "elfutils-0.186-1.fc35.src.rpm", 1810, "GPLv2+ or LGPLv3+", "Fedora Project", "", "Default yama attach scope sysctl setting", "bbb81d7e155c1afebc041bf4bc9ee9cf"},
			{intRef(), "elfutils-libs", "0.186", "1.fc35", "x86_64", "elfutils-0.186-1.fc35.src.rpm", 682389, "GPLv2+ or LGPLv3+", "Fedora Project", "", "Libraries to handle compiled objects", "1d5405205f47cfbce358e4125249fe64"},
			{intRef(), "coreutils-common", "8.32", "31.fc35", "x86_64", "coreutils-8.32-31.fc35.src.rpm", 10880210, "GPLv3+", "Fedora Project", "", "coreutils common optional components", "1199baa383f946e4b01413a52cf5dbfe"},
			{intRef(1), "openssl-libs", "1.1.1l", "2.fc35", "x86_64", "openssl-1.1.1l-2.fc35.src.rpm", 3855396, "OpenSSL and ASL 2.0", "Fedora Project", "", "A general purpose cryptography library with TLS implementation", "3131af4dea3019fcfd0f040e7af67dcb"},
			{intRef(), "coreutils", "8.32", "31.fc35", "x86_64", "coreutils-8.32-31.fc35.src.rpm", 6040898, "GPLv3+", "Fedora Project", "", "A set of basic GNU tools commonly used in shell scripts", "31edc65f732179c418c5bfc50d097aa9"},
			{intRef(), "ca-certificates", "2021.2.52", "1.0.fc35", "noarch", "ca-certificates-2021.2.52-1.0.fc35.src.rpm", 972279, "Public Domain", "Fedora Project", "", "The Mozilla CA root certificate bundle", "b115c20bc79221913009ca791a3bc1cb"},
			{intRef(), "krb5-libs", "1.19.2", "2.fc35", "x86_64", "krb5-1.19.2-2.fc35.src.rpm", 2198421, "MIT", "Fedora Project", "", "The non-admin shared libraries used by Kerberos 5", "8cb025e5ab47044fb45e62bd7d5abe89"},
			{intRef(), "libtirpc", "1.3.2", "1.fc35", "x86_64", "libtirpc-1.3.2-1.fc35.src.rpm", 208122, "SISSL and BSD", "Fedora Project", "", "Transport Independent RPC Library", "774bb16f7bceb07e236edd74893c3da2"},
			{intRef(), "libnsl2", "1.3.0", "4.fc35", "x86_64", "libnsl2-1.3.0-4.fc35.src.rpm", 130270, "BSD and LGPLv2+", "Fedora Project", "", "Public client interface library for NIS(YP) and NIS+", "ec90470cea464c30c21887ef0b6b5f78"},
			{intRef(), "zchunk-libs", "1.1.15", "2.fc35", "x86_64", "zchunk-1.1.15-2.fc35.src.rpm", 90820, "BSD and MIT", "Fedora Project", "", "Zchunk library", "66829a62bfa036541114223927236c56"},
			{intRef(), "libfsverity", "1.4", "6.fc35", "x86_64", "fsverity-utils-1.4-6.fc35.src.rpm", 29672, "BSD", "Fedora Project", "", "Development package for fsverity-utils", "ca39a98e52bfd390757594501bfe9158"},
			{intRef(), "cyrus-sasl-lib", "2.1.27", "13.fc35", "x86_64", "cyrus-sasl-2.1.27-13.fc35.src.rpm", 2409736, "BSD with advertising", "Fedora Project", "", "Shared libraries needed by applications which use Cyrus SASL", "244cc72621a079a0444957ed952b692b"},
			{intRef(), "openldap", "2.4.59", "3.fc35", "x86_64", "openldap-2.4.59-3.fc35.src.rpm", 718501, "OpenLDAP", "Fedora Project", "", "LDAP support libraries", "0e77658e50841dafd342dee9da801f8c"},
			{intRef(), "gnupg2", "2.3.4", "1.fc35", "x86_64", "gnupg2-2.3.4-1.fc35.src.rpm", 9261756, "GPLv3+", "Fedora Project", "", "Utility for secure communication and data storage", "17a955e67bbecd0b76006762ec23f01d"},
			{intRef(), "gpgme", "1.15.1", "6.fc35", "x86_64", "gpgme-1.15.1-6.fc35.src.rpm", 573957, "LGPLv2+", "Fedora Project", "", "GnuPG Made Easy - high level crypto API", "55d1815dcc9dd566a95999a31334702c"},
			{intRef(), "libssh", "0.9.6", "1.fc35", "x86_64", "libssh-0.9.6-1.fc35.src.rpm", 513049, "LGPLv2+", "Fedora Project", "", "A library implementing the SSH protocol", "89d0f0ed09d6da4ba2ab98275c98f173"},
			{intRef(), "libcurl", "7.79.1", "1.fc35", "x86_64", "curl-7.79.1-1.fc35.src.rpm", 681030, "MIT", "Fedora Project", "", "A library for getting files from web servers", "d990c31b9e1e543f45e954204899997a"},
			{intRef(), "tpm2-tss", "3.1.0", "3.fc35", "x86_64", "tpm2-tss-3.1.0-3.fc35.src.rpm", 2227128, "BSD and TCGL", "Fedora Project", "", "TPM2.0 Software Stack", "37a74040701bebdccc86ea28632895c4"},
			{intRef(), "ima-evm-utils", "1.3.2", "3.fc35", "x86_64", "ima-evm-utils-1.3.2-3.fc35.src.rpm", 141126, "GPLv2", "Fedora Project", "", "IMA/EVM support utilities", "0981d3adc3feddae5d198d0df691e21a"},
			{intRef(), "curl", "7.79.1", "1.fc35", "x86_64", "curl-7.79.1-1.fc35.src.rpm", 723076, "MIT", "Fedora Project", "", "A utility for getting files from remote servers (FTP, HTTP, and others)", "8015ff9223a971c77a8721daa8f50796"},
			{intRef(), "python-pip-wheel", "21.2.3", "4.fc35", "noarch", "python-pip-21.2.3-4.fc35.src.rpm", 1220638, "MIT and Python and ASL 2.0 and BSD and ISC and LGPLv2 and MPLv2.0 and (ASL 2.0 or BSD)", "Fedora Project", "", "The pip wheel", "af8fd2c05e3b0764b53c3eb21b3f430f"},
			{intRef(), "python3", "3.10.2", "1.fc35", "x86_64", "python3.10-3.10.2-1.fc35.src.rpm", 33090, "Python", "Fedora Project", "", "Python 3.10 interpreter", "d8ce50c8b3fec3df48b0c77bfb947d19"},
			{intRef(), "python3-libs", "3.10.2", "1.fc35", "x86_64", "python3.10-3.10.2-1.fc35.src.rpm", 33109642, "Python", "Fedora Project", "", "Python runtime libraries", "d43ba03aa5f297a8ece98865b450286c"},
			{intRef(), "python3-libcomps", "0.1.18", "1.fc35", "x86_64", "libcomps-0.1.18-1.fc35.src.rpm", 146971, "GPLv2+", "Fedora Project", "", "Python 3 bindings for libcomps library", "9344fc5a5274824bbb2cebd5e7341c23"},
			{intRef(), "python3-gpg", "1.15.1", "6.fc35", "x86_64", "gpgme-1.15.1-6.fc35.src.rpm", 1394334, "LGPLv2+", "Fedora Project", "", "gpgme bindings for Python 3", "87da740b4d4cd47f726b5876769db9df"},
			{intRef(), "gzip", "1.10", "5.fc35", "x86_64", "gzip-1.10-5.fc35.src.rpm", 357298, "GPLv3+ and GFDL", "Fedora Project", "", "GNU data compression program", "0c334e5cdc70ca2c94149b74b6771d59"},
			{intRef(), "cracklib", "2.9.6", "27.fc35", "x86_64", "cracklib-2.9.6-27.fc35.src.rpm", 251474, "LGPLv2+", "Fedora Project", "", "A password-checking library", "be873d814043808a88cd6ac0b984d7cd"},
			{intRef(), "libpwquality", "1.4.4", "6.fc35", "x86_64", "libpwquality-1.4.4-6.fc35.src.rpm", 415452, "BSD or GPLv2+", "Fedora Project", "", "A library for password generation and password quality checking", "2c4c69665af0cbea0ea4405af255304d"},
			{intRef(), "pam", "1.5.2", "7.fc35", "x86_64", "pam-1.5.2-7.fc35.src.rpm", 1947749, "BSD and GPLv2+", "Fedora Project", "", "An extensible library which provides authentication for applications", "e3509e3b6393fc5853d9ffa08fde7c5c"},
			{intRef(), "libblkid", "2.37.4", "1.fc35", "x86_64", "util-linux-2.37.4-1.fc35.src.rpm", 230753, "LGPLv2+", "Fedora Project", "", "Block device ID library", "71cc31b85c8cb80de2a5f103089fa040"},
			{intRef(), "libmount", "2.37.4", "1.fc35", "x86_64", "util-linux-2.37.4-1.fc35.src.rpm", 311117, "LGPLv2+", "Fedora Project", "", "Device mounting library", "aa6091a17f0ae6f463354f30a25cf7a8"},
			{intRef(), "glib2", "2.70.4", "1.fc35", "x86_64", "glib2-2.70.4-1.fc35.src.rpm", 13585533, "LGPLv2+", "Fedora Project", "", "A library of handy utility functions", "653d77040158b640acea7b1931e8e633"},
			{intRef(), "librepo", "1.14.2", "1.fc35", "x86_64", "librepo-1.14.2-1.fc35.src.rpm", 241974, "LGPLv2+", "Fedora Project", "", "Repodata downloading library", "11553e792c5ce5f412bfcaa52b0c36f7"},
			{intRef(), "libarchive", "3.5.2", "2.fc35", "x86_64", "libarchive-3.5.2-2.fc35.src.rpm", 907245, "BSD", "Fedora Project", "", "A library for handling streaming archive formats", "e1660078d5ded318f65782bd9ba959f8"},
			{intRef(), "rpm", "4.17.0", "4.fc35", "x86_64", "rpm-4.17.0-4.fc35.src.rpm", 2948906, "GPLv2+", "Fedora Project", "", "The RPM package management system", "4d1f115d98b805430eb3a964f07656d0"},
			{intRef(), "rpm-libs", "4.17.0", "4.fc35", "x86_64", "rpm-4.17.0-4.fc35.src.rpm", 775140, "GPLv2+ and LGPLv2+ with exceptions", "Fedora Project", "", "Libraries for manipulating RPM packages", "5d501dd9ef6c4069a77c7ff072f2e466"},
			{intRef(), "libsolv", "0.7.19", "3.fc35", "x86_64", "libsolv-0.7.19-3.fc35.src.rpm", 897882, "BSD", "Fedora Project", "", "Package dependency solver", "4a3052d6738d16a276ae1930ecf26268"},
			{intRef(), "libmodulemd", "2.14.0", "1.fc35", "x86_64", "libmodulemd-2.14.0-1.fc35.src.rpm", 736677, "MIT", "Fedora Project", "", "Module metadata manipulation library", "ccb5cd06a276f59ec90ad39f81193d74"},
			{intRef(), "libdnf", "0.64.0", "1.fc35", "x86_64", "libdnf-0.64.0-1.fc35.src.rpm", 2045581, "LGPLv2+", "Fedora Project", "", "Library providing simplified C and Python API to libsolv", "101ec0844aef916129ce40d7874d5522"},
			{intRef(), "python3-libdnf", "0.64.0", "1.fc35", "x86_64", "libdnf-0.64.0-1.fc35.src.rpm", 3766775, "LGPLv2+", "Fedora Project", "", "Python 3 bindings for the libdnf library.", "69eca99cdd94d9b163461c7431405599"},
			{intRef(), "python3-hawkey", "0.64.0", "1.fc35", "x86_64", "libdnf-0.64.0-1.fc35.src.rpm", 310028, "LGPLv2+", "Fedora Project", "", "Python 3 bindings for the hawkey library", "33dcf93d1a4744ec72c46c19538a2c1a"},
			{intRef(), "rpm-build-libs", "4.17.0", "4.fc35", "x86_64", "rpm-4.17.0-4.fc35.src.rpm", 199518, "GPLv2+ and LGPLv2+ with exceptions", "Fedora Project", "", "Libraries for building RPM packages", "d08d601abf47c55029d557fa9fc94ee9"},
			{intRef(), "rpm-sign-libs", "4.17.0", "4.fc35", "x86_64", "rpm-4.17.0-4.fc35.src.rpm", 40492, "GPLv2+ and LGPLv2+ with exceptions", "Fedora Project", "", "Libraries for signing RPM packages", "57618e7007593175546f8afb197aa44e"},
			{intRef(), "python3-rpm", "4.17.0", "4.fc35", "x86_64", "rpm-4.17.0-4.fc35.src.rpm", 378243, "GPLv2+", "Fedora Project", "", "Python 3 bindings for apps which will manipulate RPM packages", "50f2032b29f43d046290260f446c18e3"},
			{intRef(), "python3-dnf", "4.9.0", "1.fc35", "noarch", "dnf-4.9.0-1.fc35.src.rpm", 1898937, "GPLv2+", "Fedora Project", "", "Python 3 interface to DNF", "f4d42de5bb07769bfa44f9ec6ea826ad"},
			{intRef(), "dnf", "4.9.0", "1.fc35", "noarch", "dnf-4.9.0-1.fc35.src.rpm", 2203005, "GPLv2+", "Fedora Project", "", "Package manager", "f754b5918e3d8774ae9522e55818a09b"},
			{intRef(), "yum", "4.9.0", "1.fc35", "noarch", "dnf-4.9.0-1.fc35.src.rpm", 22042, "GPLv2+", "Fedora Project", "", "Package manager", "557ae4521569b4fa4599dcf24ea48d97"},
			{intRef(), "sudo", "1.9.7p2", "2.fc35", "x86_64", "sudo-1.9.7p2-2.fc35.src.rpm", 4324216, "ISC", "Fedora Project", "", "Allows restricted root access for specified users", "b5f88dba9550cee464dc45d595a98c09"},
			{intRef(2), "tar", "1.34", "2.fc35", "x86_64", "tar-1.34-2.fc35.src.rpm", 3156278, "GPLv3+", "Fedora Project", "", "GNU file archiving program", "052b560465e9eebf3c4f7986aa10ee0b"},
			{intRef(), "fedora-repos-modular", "35", "1", "noarch", "fedora-repos-35-1.src.rpm", 4042, "MIT", "Fedora Project", "", "Fedora modular package repositories", "f80de6667a3eeb0688c854b07128f815"},
			{intRef(), "rootfiles", "8.1", "30.fc35", "noarch", "rootfiles-8.1-30.fc35.src.rpm", 817, "Public Domain", "Fedora Project", "", "The basic required files for the root user's directory", "2a5798670386c2b0ac45b20d55599029"},
			{intRef(), "gpg-pubkey", "9867c58f", "601c49ca", "", "", 0, "pubkey", "", "", "Fedora (35) <fedora-35-primary@fedoraproject.org> public key", ""},
			{intRef(1), "openssl", "1.1.1l", "2.fc35", "x86_64", "openssl-1.1.1l-2.fc35.src.rpm", 1197622, "OpenSSL and ASL 2.0", "Fedora Project", "", "Utilities from the general purpose cryptography library with TLS implementation", "b6c0dcb35a2754dd4133e84b0d75707f"},
			{intRef(), "mongodb-org-shell", "5.0.9", "1.el8", "x86_64", "mongodb-org-database-5.0.9-1.el8.src.rpm", 58465553, "SSPL", "", "", "MongoDB shell client", "500fa1e55d5679de5a7e133cae01ca39"},
			{intRef(0), "mongocli", "1.25.0", "1", "x86_64", "mongocli-1.25.0-1.src.rpm", 26181632, "Apache 2.0", "MongoDB Inc.", "", "MongoDB Command Line Interface (mongocli) is a tool for managing your MongoDB cloud services, like MongoDB Atlas, MongoDB Cloud Manager, and MongoDB Ops Manager.", "d27e4376bba413f3d1cf741b7fba314f"},
		}

		return toPackageInfo(pkgs)
	}

	// docker run --rm -it rockylinux:9 bash
	// rpm -qa --queryformat "\{%{EPOCH}, \"%{NAME}\", \"%{VERSION}\", \"%{RELEASE}\", \"%{ARCH}\", \"%{SOURCERPM}\", %{SIZE}, \"%{LICENSE}\", \"%{VENDOR}\", \"\", \"%{SUMMARY}\", \"%{SIGMD5}\"\},\n" | sed "s/^{(none)/{intRef()/g" | sed -r 's/^\{([0-9]+),/{intRef(\1),/' | sed "s/(none)/0/g"
	Rockylinux9WithSQLite3 = func() []*PackageInfo {
		pkgs := []*commonPackageInfo{
			{intRef(), "libgcc", "11.4.1", "2.1.el9", "aarch64", "gcc-11.4.1-2.1.el9.src.rpm", 226740, "GPLv3+ and GPLv3+ with exceptions and GPLv2+ with exceptions and LGPLv2+ and BSD", "Rocky Enterprise Software Foundation", "", "GCC version 11 shared support library", "bffcb37f7529c515ef63883129d90b1f"},
			{intRef(), "crypto-policies", "20230731", "1.git94f0e2c.el9_3.1", "noarch", "crypto-policies-20230731-1.git94f0e2c.el9_3.1.src.rpm", 86024, "LGPLv2+", "Rocky Enterprise Software Foundation", "", "System-wide crypto policies", "ff248c90791a5e77b33d0bd0be1166f6"},
			{intRef(), "tzdata", "2023c", "1.el9", "noarch", "tzdata-2023c-1.el9.src.rpm", 1712614, "Public Domain", "Rocky Enterprise Software Foundation", "", "Timezone data", "afb53e41ded87d71c56bb333bea308d1"},
			{intRef(), "rocky-gpg-keys", "9.3", "1.1.el9", "noarch", "rocky-release-9.3-1.1.el9.src.rpm", 4909, "BSD-3-Clause", "Rocky Enterprise Software Foundation", "", "Rocky RPM GPG Keys", "268d46075f8cef8d07cfd365c256585a"},
			{intRef(), "rocky-release", "9.3", "1.1.el9", "noarch", "rocky-release-9.3-1.1.el9.src.rpm", 21579, "BSD-3-Clause", "Rocky Enterprise Software Foundation", "", "Rocky Linux release files", "1b0773de3a751485f12730ac0f1c957a"},
			{intRef(), "rocky-repos", "9.3", "1.1.el9", "noarch", "rocky-release-9.3-1.1.el9.src.rpm", 15763, "BSD-3-Clause", "Rocky Enterprise Software Foundation", "", "Rocky Linux Package Repositories", "5e95693b8c8cff926e259ed0f1fab0e1"},
			{intRef(), "setup", "2.13.7", "9.el9", "noarch", "setup-2.13.7-9.el9.src.rpm", 725907, "Public Domain", "Rocky Enterprise Software Foundation", "", "A set of system configuration and setup files", "9c862c5c59f3ee7a366165f2000a6676"},
			{intRef(), "filesystem", "3.16", "2.el9", "aarch64", "filesystem-3.16-2.el9.src.rpm", 106, "Public Domain", "Rocky Enterprise Software Foundation", "", "The basic directory layout for a Linux system", "21d54cdc4d037ecdac2bf3b117b0b40f"},
			{intRef(), "basesystem", "11", "13.el9", "noarch", "basesystem-11-13.el9.src.rpm", 0, "Public Domain", "Rocky Enterprise Software Foundation", "", "The skeleton package which defines a simple Rocky Linux system", "1e84df7870e25521bc4b1abba9c14ec5"},
			{intRef(), "libreport-filesystem", "2.15.2", "6.el9.rocky.0.2", "noarch", "libreport-2.15.2-6.el9.rocky.0.2.src.rpm", 0, "GPLv2+", "Rocky Enterprise Software Foundation", "", "Filesystem layout for libreport", "0e563cd5090f2b096938974a50c84736"},
			{intRef(), "dnf-data", "4.14.0", "8.el9", "noarch", "dnf-4.14.0-8.el9.src.rpm", 39048, "GPLv2+", "Rocky Enterprise Software Foundation", "", "Common data and configuration files for DNF", "cfe649b9edc6e391e86b1e9efc6eee41"},
			{intRef(), "ncurses-base", "6.2", "10.20210508.el9", "noarch", "ncurses-6.2-10.20210508.el9.src.rpm", 307293, "MIT", "Rocky Enterprise Software Foundation", "", "Descriptions of common terminals", "de956b53147623c910c9764db2b0cc63"},
			{intRef(), "bash", "5.1.8", "6.el9_1", "aarch64", "bash-5.1.8-6.el9_1.src.rpm", 7716255, "GPLv3+", "Rocky Enterprise Software Foundation", "", "The GNU Bourne Again shell", "5c9682e9d5d238178e2d44f6b6def390"},
			{intRef(), "ncurses-libs", "6.2", "10.20210508.el9", "aarch64", "ncurses-6.2-10.20210508.el9.src.rpm", 1229847, "MIT", "Rocky Enterprise Software Foundation", "", "Ncurses libraries", "8d1e3bde131dd5ebf329fc977a88da1a"},
			{intRef(), "glibc-minimal-langpack", "2.34", "83.el9.7", "aarch64", "glibc-2.34-83.el9.7.src.rpm", 0, "LGPLv2+ and LGPLv2+ with exceptions and GPLv2+ and GPLv2+ with exceptions and BSD and Inner-Net and ISC and Public Domain and GFDL", "Rocky Enterprise Software Foundation", "", "Minimal language packs for glibc.", "34055e4493208dfc763f160a265f390e"},
			{intRef(), "glibc-common", "2.34", "83.el9.7", "aarch64", "glibc-2.34-83.el9.7.src.rpm", 1353373, "LGPLv2+ and LGPLv2+ with exceptions and GPLv2+ and GPLv2+ with exceptions and BSD and Inner-Net and ISC and Public Domain and GFDL", "Rocky Enterprise Software Foundation", "", "Common binaries and locale data for glibc", "4adcd449910ed930afbb1318855c2134"},
			{intRef(), "glibc", "2.34", "83.el9.7", "aarch64", "glibc-2.34-83.el9.7.src.rpm", 6400596, "LGPLv2+ and LGPLv2+ with exceptions and GPLv2+ and GPLv2+ with exceptions and BSD and Inner-Net and ISC and Public Domain and GFDL", "Rocky Enterprise Software Foundation", "", "The GNU libc libraries", "75890336303e584c642cc3ab39a78421"},
			{intRef(), "zlib", "1.2.11", "40.el9", "aarch64", "zlib-1.2.11-40.el9.src.rpm", 235473, "zlib and Boost", "Rocky Enterprise Software Foundation", "", "Compression and decompression library", "6ee45a47968bcc0e9503f6d838869861"},
			{intRef(), "bzip2-libs", "1.0.8", "8.el9", "aarch64", "bzip2-1.0.8-8.el9.src.rpm", 74508, "BSD", "Rocky Enterprise Software Foundation", "", "Libraries for applications using bzip2", "673ee2b55be4f2160ea7fe8cc1c8eb99"},
			{intRef(), "xz-libs", "5.2.5", "8.el9_0", "aarch64", "xz-5.2.5-8.el9_0.src.rpm", 205837, "Public Domain", "Rocky Enterprise Software Foundation", "", "Libraries for decoding LZMA compression", "6241b242fb2506646fc17b4ea4d70a36"},
			{intRef(), "libzstd", "1.5.1", "2.el9", "aarch64", "zstd-1.5.1-2.el9.src.rpm", 750403, "BSD and GPLv2", "Rocky Enterprise Software Foundation", "", "Zstd shared library", "43ba45ad84facdcb85db7644db885cc7"},
			{intRef(), "libxcrypt", "4.4.18", "3.el9", "aarch64", "libxcrypt-4.4.18-3.el9.src.rpm", 270596, "LGPLv2+ and BSD and Public Domain", "Rocky Enterprise Software Foundation", "", "Extended crypt library for descrypt, md5crypt, bcrypt, and others", "f20b927c77d43204ad49a1658ca6bdb9"},
			{intRef(), "sqlite-libs", "3.34.1", "6.el9_1", "aarch64", "sqlite-3.34.1-6.el9_1.src.rpm", 1294872, "Public Domain", "Rocky Enterprise Software Foundation", "", "Shared library for the sqlite3 embeddable SQL database engine.", "4e6424f80c5da717f7b3a75ecb270475"},
			{intRef(), "popt", "1.18", "8.el9", "aarch64", "popt-1.18-8.el9.src.rpm", 142488, "MIT", "Rocky Enterprise Software Foundation", "", "C library for parsing command line parameters", "f1bcac9605780279ad496fe7f5bbefc0"},
			{intRef(), "libuuid", "2.37.4", "15.el9", "aarch64", "util-linux-2.37.4-15.el9.src.rpm", 70549, "BSD", "Rocky Enterprise Software Foundation", "", "Universally unique ID library", "da36aa0244195f494a73f0f2fbdc1a0b"},
			{intRef(), "libcap", "2.48", "9.el9_2", "aarch64", "libcap-2.48-9.el9_2.src.rpm", 507655, "BSD or GPLv2", "Rocky Enterprise Software Foundation", "", "Library for getting and setting POSIX.1e capabilities", "c39fd61c4678c25fb312f452057bc160"},
			{intRef(), "libgpg-error", "1.42", "5.el9", "aarch64", "libgpg-error-1.42-5.el9.src.rpm", 918488, "LGPLv2+", "Rocky Enterprise Software Foundation", "", "Library for error values used by GnuPG components", "716b574d895e0587b004db9d0fa138ef"},
			{intRef(), "libstdc++", "11.4.1", "2.1.el9", "aarch64", "gcc-11.4.1-2.1.el9.src.rpm", 2516515, "GPLv3+ and GPLv3+ with exceptions and GPLv2+ with exceptions and LGPLv2+ and BSD", "Rocky Enterprise Software Foundation", "", "GNU Standard C++ Library", "34e5e0ae84cde3525187937ee693ec99"},
			{intRef(), "libxml2", "2.9.13", "4.el9", "aarch64", "libxml2-2.9.13-4.el9.src.rpm", 2027796, "MIT", "Rocky Enterprise Software Foundation", "", "Library providing XML and HTML support", "7242aec061d6fa3f2d50cb925a8bfea5"},
			{intRef(), "readline", "8.1", "4.el9", "aarch64", "readline-8.1-4.el9.src.rpm", 508892, "GPLv3+", "Rocky Enterprise Software Foundation", "", "A library for editing typed command lines", "92f93dc0138cf3bffb6a7b4d22a73caf"},
			{intRef(), "libsmartcols", "2.37.4", "15.el9", "aarch64", "util-linux-2.37.4-15.el9.src.rpm", 163235, "LGPLv2+", "Rocky Enterprise Software Foundation", "", "Formatting library for ls-like programs.", "e30c255fe8dead6846d432f511564f94"},
			{intRef(), "elfutils-libelf", "0.189", "3.el9", "aarch64", "elfutils-0.189-3.el9.src.rpm", 1050977, "GPLv2+ or LGPLv3+", "Rocky Enterprise Software Foundation", "", "Library to read and write ELF files", "4f3b53c3a154157f81f63cbc0c5207f1"},
			{intRef(), "libattr", "2.5.1", "3.el9", "aarch64", "attr-2.5.1-3.el9.src.rpm", 70269, "LGPLv2+", "Rocky Enterprise Software Foundation", "", "Dynamic library for extended attribute support", "c43217519565f8e56acf1cd104991f04"},
			{intRef(), "libacl", "2.3.1", "3.el9", "aarch64", "acl-2.3.1-3.el9.src.rpm", 69722, "LGPLv2+", "Rocky Enterprise Software Foundation", "", "Dynamic library for access control list support", "1565969b9038d8c30d34b6e8b3d70229"},
			{intRef(), "libffi", "3.4.2", "8.el9", "aarch64", "libffi-3.4.2-8.el9.src.rpm", 90073, "MIT", "Rocky Enterprise Software Foundation", "", "A portable foreign function interface library", "27875dc7d4c89ebdefa01e04451a95e3"},
			{intRef(), "p11-kit", "0.24.1", "2.el9", "aarch64", "p11-kit-0.24.1-2.el9.src.rpm", 1787358, "BSD", "Rocky Enterprise Software Foundation", "", "Library for loading and sharing PKCS#11 modules", "468eae53a5754af6c2edba392fcbff74"},
			{intRef(), "libcom_err", "1.46.5", "3.el9", "aarch64", "e2fsprogs-1.46.5-3.el9.src.rpm", 113953, "MIT", "Rocky Enterprise Software Foundation", "", "Common error description library", "0bc5925c24e0498ef61c18e03773e962"},
			{intRef(), "lua-libs", "5.4.4", "4.el9", "aarch64", "lua-5.4.4-4.el9.src.rpm", 336139, "MIT", "Rocky Enterprise Software Foundation", "", "Libraries for lua", "a87e4bbb9724e154f7416608184d84c1"},
			{intRef(), "libassuan", "2.5.5", "3.el9", "aarch64", "libassuan-2.5.5-3.el9.src.rpm", 220101, "LGPLv2+ and GPLv3+", "Rocky Enterprise Software Foundation", "", "GnuPG IPC library", "881549b7f4ac965be8f786f3a6b3aa2f"},
			{intRef(), "libgcrypt", "1.10.0", "10.el9_2", "aarch64", "libgcrypt-1.10.0-10.el9_2.src.rpm", 1040058, "LGPLv2+", "Rocky Enterprise Software Foundation", "", "A general-purpose cryptography library", "e12cbe68b820b96bb7e916182743a919"},
			{intRef(), "file-libs", "5.39", "14.el9", "aarch64", "file-5.39-14.el9.src.rpm", 8102676, "BSD", "Rocky Enterprise Software Foundation", "", "Libraries for applications using libmagic", "603befdf1fc96a660ead1dc971a1a642"},
			{intRef(), "alternatives", "1.24", "1.el9", "aarch64", "chkconfig-1.24-1.el9.src.rpm", 91718, "GPL-2.0-only", "Rocky Enterprise Software Foundation", "", "A tool to maintain symbolic links determining default commands", "1495aeab8f11b9203581866e6f82f8d0"},
			{intRef(), "lz4-libs", "1.9.3", "5.el9", "aarch64", "lz4-1.9.3-5.el9.src.rpm", 137195, "GPLv2+ and BSD", "Rocky Enterprise Software Foundation", "", "Libaries for lz4", "49e119aa5c22f61c78f49124bee345d2"},
			{intRef(), "libcap-ng", "0.8.2", "7.el9", "aarch64", "libcap-ng-0.8.2-7.el9.src.rpm", 164940, "LGPLv2+", "Rocky Enterprise Software Foundation", "", "Alternate posix capabilities library", "62630f1089446dda9fe4a84802858555"},
			{intRef(), "audit-libs", "3.0.7", "104.el9", "aarch64", "audit-3.0.7-104.el9.src.rpm", 296953, "LGPLv2+", "Rocky Enterprise Software Foundation", "", "Dynamic library for libaudit", "cc98f304853252a160d9b205582a0f25"},
			{intRef(), "libsigsegv", "2.13", "4.el9", "aarch64", "libsigsegv-2.13-4.el9.src.rpm", 99362, "GPLv2+", "Rocky Enterprise Software Foundation", "", "Library for handling page faults in user mode", "1b2ff67d29316f1a5d2fc2df191f4c6c"},
			{intRef(), "libunistring", "0.9.10", "15.el9", "aarch64", "libunistring-0.9.10-15.el9.src.rpm", 1630283, "GPLv2+ or LGPLv3+", "Rocky Enterprise Software Foundation", "", "GNU Unicode string library", "f171d3b2e551e8b27f35fd8bade795ea"},
			{intRef(1), "gmp", "6.2.0", "13.el9", "aarch64", "gmp-6.2.0-13.el9.src.rpm", 671932, "LGPLv3+ or GPLv2+", "Rocky Enterprise Software Foundation", "", "A GNU arbitrary precision library", "02714203d54ab936f5a375b964022ab2"},
			{intRef(), "keyutils-libs", "1.6.3", "1.el9", "aarch64", "keyutils-1.6.3-1.el9.src.rpm", 100107, "GPLv2+ and LGPLv2+", "Rocky Enterprise Software Foundation", "", "Key utilities library", "300278f4c1056e876eb8222057705d9e"},
			{intRef(), "libsepol", "3.5", "1.el9", "aarch64", "libsepol-3.5-1.el9.src.rpm", 763019, "LGPLv2+", "Rocky Enterprise Software Foundation", "", "SELinux binary policy manipulation library", "0052712847b99ef052b24466e0745b12"},
			{intRef(), "libtasn1", "4.16.0", "8.el9_1", "aarch64", "libtasn1-4.16.0-8.el9_1.src.rpm", 224068, "GPLv3+ and LGPLv2+", "Rocky Enterprise Software Foundation", "", "The ASN.1 library used in GNUTLS", "d0265c50ec7c164e9b52525bf08a9d75"},
			{intRef(), "p11-kit-trust", "0.24.1", "2.el9", "aarch64", "p11-kit-0.24.1-2.el9.src.rpm", 541423, "BSD", "Rocky Enterprise Software Foundation", "", "System trust module from p11-kit", "c8ef4d6963afdbd40743b04ec8a44b06"},
			{intRef(), "pcre", "8.44", "3.el9.3", "aarch64", "pcre-8.44-3.el9.3.src.rpm", 567424, "BSD", "Rocky Enterprise Software Foundation", "", "Perl-compatible regular expression library", "6f29bde7ad63f666d0d2c79d2231198f"},
			{intRef(), "grep", "3.6", "5.el9", "aarch64", "grep-3.6-5.el9.src.rpm", 837104, "GPLv3+", "Rocky Enterprise Software Foundation", "", "Pattern matching utilities", "48a84a9e0602fd93869b82a268ca176d"},
			{intRef(), "json-c", "0.14", "11.el9", "aarch64", "json-c-0.14-11.el9.src.rpm", 74554, "MIT", "Rocky Enterprise Software Foundation", "", "JSON implementation in C", "351e39ec7a1c629cad744ec6b3cee72a"},
			{intRef(), "expat", "2.5.0", "1.el9", "aarch64", "expat-2.5.0-1.el9.src.rpm", 341477, "MIT", "Rocky Enterprise Software Foundation", "", "An XML parser library", "4bfa9e923d8a8721111540706e87d8e5"},
			{intRef(1), "gdbm-libs", "1.19", "4.el9", "aarch64", "gdbm-1.19-4.el9.src.rpm", 173386, "GPLv3+", "Rocky Enterprise Software Foundation", "", "Libraries files for gdbm", "6d26c07d6a1dd26b623e61e25fabe978"},
			{intRef(), "libcomps", "0.1.18", "1.el9", "aarch64", "libcomps-0.1.18-1.el9.src.rpm", 227119, "GPLv2+", "Rocky Enterprise Software Foundation", "", "Comps XML file manipulation library", "b28c73806c5298b198d69a3cc8fac11f"},
			{intRef(), "mpfr", "4.1.0", "7.el9", "aarch64", "mpfr-4.1.0-7.el9.src.rpm", 608531, "LGPLv3+", "Rocky Enterprise Software Foundation", "", "C library for multiple-precision floating-point computations", "6166147d76df349652b64c79e3e93b96"},
			{intRef(), "gawk", "5.1.0", "6.el9", "aarch64", "gawk-5.1.0-6.el9.src.rpm", 2416174, "GPLv3+ and GPLv2+ and LGPLv2+ and BSD", "Rocky Enterprise Software Foundation", "", "The GNU version of the AWK text processing utility", "ee8d1dcd221d71492b3f1826666af149"},
			{intRef(), "libidn2", "2.3.0", "7.el9", "aarch64", "libidn2-2.3.0-7.el9.src.rpm", 257420, "(GPLv2+ or LGPLv3+) and GPLv3+", "Rocky Enterprise Software Foundation", "", "Library to support IDNA2008 internationalized domain names", "9f1beeb1b9facf2f71b7dc48946ba26e"},
			{intRef(), "libksba", "1.5.1", "6.el9_1", "aarch64", "libksba-1.5.1-6.el9_1.src.rpm", 394246, "(LGPLv3+ or GPLv2+) and GPLv3+", "Rocky Enterprise Software Foundation", "", "CMS and X.509 library", "034ea3c5de7dbdab80e4377b226c17a3"},
			{intRef(), "libverto", "0.3.2", "3.el9", "aarch64", "libverto-0.3.2-3.el9.src.rpm", 71189, "MIT", "Rocky Enterprise Software Foundation", "", "Main loop abstraction library", "e4d77f54c6b777ac7d4c0e30a8426834"},
			{intRef(), "libnghttp2", "1.43.0", "5.el9_3.1", "aarch64", "nghttp2-1.43.0-5.el9_3.1.src.rpm", 202308, "MIT", "Rocky Enterprise Software Foundation", "", "A library implementing the HTTP/2 protocol", "0ac6b01bc48a78d0cd5ff77b6074547f"},
			{intRef(), "nettle", "3.8", "3.el9_0", "aarch64", "nettle-3.8-3.el9_0.src.rpm", 970774, "LGPLv3+ or GPLv2+", "Rocky Enterprise Software Foundation", "", "A low-level cryptographic library", "21be9045776b338eefbf2bbe731b27a2"},
			{intRef(), "gnutls", "3.7.6", "23.el9", "aarch64", "gnutls-3.7.6-23.el9.src.rpm", 3200275, "GPLv3+ and LGPLv2+", "Rocky Enterprise Software Foundation", "", "A TLS protocol implementation", "c5d1a2a0c034984ce61834896baee2d5"},
			{intRef(), "npth", "1.6", "8.el9", "aarch64", "npth-1.6-8.el9.src.rpm", 95499, "LGPLv2+", "Rocky Enterprise Software Foundation", "", "The New GNU Portable Threads library", "a3820265626f69d1f19044affaae5e06"},
			{intRef(), "libyaml", "0.2.5", "7.el9", "aarch64", "libyaml-0.2.5-7.el9.src.rpm", 137819, "MIT", "Rocky Enterprise Software Foundation", "", "YAML 1.1 parser and emitter written in C", "31b90f7ee134a0a9a308e8f392515196"},
			{intRef(), "libdb", "5.3.28", "53.el9", "aarch64", "libdb-5.3.28-53.el9.src.rpm", 1758454, "BSD and LGPLv2 and Sleepycat", "Rocky Enterprise Software Foundation", "", "The Berkeley DB database library for C", "2d814fa3e6001e8a793d6296f5013eb6"},
			{intRef(), "libgomp", "11.4.1", "2.1.el9", "aarch64", "gcc-11.4.1-2.1.el9.src.rpm", 400522, "GPLv3+ and GPLv3+ with exceptions and GPLv2+ with exceptions and LGPLv2+ and BSD", "Rocky Enterprise Software Foundation", "", "GCC OpenMP v4.5 shared support library", "a796471cbc342297ac34b510642fef69"},
			{intRef(), "libatomic", "11.4.1", "2.1.el9", "aarch64", "gcc-11.4.1-2.1.el9.src.rpm", 70586, "GPLv3+ and GPLv3+ with exceptions and GPLv2+ with exceptions and LGPLv2+ and BSD", "Rocky Enterprise Software Foundation", "", "The GNU Atomic library", "70d578a6fb78fd73c94bf24dc2e80281"},
			{intRef(), "libeconf", "0.4.1", "3.el9_2", "aarch64", "libeconf-0.4.1-3.el9_2.src.rpm", 74171, "MIT", "Rocky Enterprise Software Foundation", "", "Enhanced config file parser library", "b8c8b40407fdb26b6d16299e6534e5a1"},
			{intRef(), "elfutils-default-yama-scope", "0.189", "3.el9", "noarch", "elfutils-0.189-3.el9.src.rpm", 1810, "GPLv2+ or LGPLv3+", "Rocky Enterprise Software Foundation", "", "Default yama attach scope sysctl setting", "5c06e4b3809294db74d2060112309115"},
			{intRef(), "elfutils-libs", "0.189", "3.el9", "aarch64", "elfutils-0.189-3.el9.src.rpm", 762805, "GPLv2+ or LGPLv3+", "Rocky Enterprise Software Foundation", "", "Libraries to handle compiled objects", "f3fd3a77e973bc0bbcb4489c0f3f160f"},
			{intRef(), "python3-setuptools-wheel", "53.0.0", "12.el9", "noarch", "python-setuptools-53.0.0-12.el9.src.rpm", 562584, "MIT and (BSD or ASL 2.0)", "Rocky Enterprise Software Foundation", "", "The setuptools wheel", "6ff272eb8366a638ff9039ee0cc7d16b"},
			{intRef(), "pcre2-syntax", "10.40", "2.el9", "noarch", "pcre2-10.40-2.el9.src.rpm", 234324, "BSD", "Rocky Enterprise Software Foundation", "", "Documentation for PCRE2 regular expressions", "1573427a88007995fe7918f38a324e48"},
			{intRef(), "pcre2", "10.40", "2.el9", "aarch64", "pcre2-10.40-2.el9.src.rpm", 665738, "BSD", "Rocky Enterprise Software Foundation", "", "Perl-compatible regular expression library", "746b53b42790d4107309080b76c5492f"},
			{intRef(), "libselinux", "3.5", "1.el9", "aarch64", "libselinux-3.5-1.el9.src.rpm", 205117, "Public Domain", "Rocky Enterprise Software Foundation", "", "SELinux library and simple utilities", "97fdb7579dc0138a08eb757b0078eb13"},
			{intRef(), "coreutils-single", "8.32", "34.el9", "aarch64", "coreutils-8.32-34.el9.src.rpm", 1515017, "GPLv3+", "Rocky Enterprise Software Foundation", "", "coreutils multicall binary", "8f5a2d7edfbb52044554921a8cb0c55c"},
			{intRef(), "libblkid", "2.37.4", "15.el9", "aarch64", "util-linux-2.37.4-15.el9.src.rpm", 270521, "LGPLv2+", "Rocky Enterprise Software Foundation", "", "Block device ID library", "4170ff47814e89f5da18499b524f2d44"},
			{intRef(), "libmount", "2.37.4", "15.el9", "aarch64", "util-linux-2.37.4-15.el9.src.rpm", 363173, "LGPLv2+", "Rocky Enterprise Software Foundation", "", "Device mounting library", "b134a6cb2a19243e58abd47753dcf6c0"},
			{intRef(), "glib2", "2.68.4", "11.el9", "aarch64", "glib2-2.68.4-11.el9.src.rpm", 13869159, "LGPLv2+", "Rocky Enterprise Software Foundation", "", "A library of handy utility functions", "35d213646ff525b9e3a0c0e3cf5e366a"},
			{intRef(), "sed", "4.8", "9.el9", "aarch64", "sed-4.8-9.el9.src.rpm", 833863, "GPLv3+", "Rocky Enterprise Software Foundation", "", "A GNU stream text editor", "601492e9b20b558d627e352088aab4e3"},
			{intRef(), "ca-certificates", "2023.2.60_v7.0.306", "90.1.el9_2", "noarch", "ca-certificates-2023.2.60_v7.0.306-90.1.el9_2.src.rpm", 2354652, "Public Domain", "Rocky Enterprise Software Foundation", "", "The Mozilla CA root certificate bundle", "e0ef61291e4078224c329f3c7825d8d3"},
			{intRef(1), "openssl-libs", "3.0.7", "24.el9", "aarch64", "openssl-3.0.7-24.el9.src.rpm", 6423244, "ASL 2.0", "Rocky Enterprise Software Foundation", "", "A general purpose cryptography library with TLS implementation", "17012166b07eaa970ef47145c1a56bea"},
			{intRef(), "systemd-libs", "252", "18.el9", "aarch64", "systemd-252-18.el9.src.rpm", 1976512, "LGPLv2+ and MIT", "Rocky Enterprise Software Foundation", "", "systemd libraries", "7cf100d06fe904fbd95a8c1174aace0d"},
			{intRef(), "krb5-libs", "1.21.1", "1.el9", "aarch64", "krb5-1.21.1-1.el9.src.rpm", 2704396, "MIT", "Rocky Enterprise Software Foundation", "", "The non-admin shared libraries used by Kerberos 5", "1426ef6fb8c7682bfd0e877d15f267ec"},
			{intRef(), "libcurl-minimal", "7.76.1", "26.el9_3.2.0.1", "aarch64", "curl-7.76.1-26.el9_3.2.0.1.src.rpm", 476878, "MIT", "Rocky Enterprise Software Foundation", "", "Conservatively configured build of libcurl for minimal installations", "a2ddc4b701a0bb0078a65e1d9a6206ef"},
			{intRef(), "elfutils-debuginfod-client", "0.189", "3.el9", "aarch64", "elfutils-0.189-3.el9.src.rpm", 142867, "GPLv3+ and (GPLv2+ or LGPLv3+)", "Rocky Enterprise Software Foundation", "", "Library and command line client for build-id HTTP ELF/DWARF server", "c705a41c5f470a36572326051e48af16"},
			{intRef(), "binutils-gold", "2.35.2", "42.el9", "aarch64", "binutils-2.35.2-42.el9.src.rpm", 2763171, "GPLv3+", "Rocky Enterprise Software Foundation", "", "The GOLD linker, a faster alternative to the BFD linker", "58747bd8ec42d5c730a56c48b78b818f"},
			{intRef(), "binutils", "2.35.2", "42.el9", "aarch64", "binutils-2.35.2-42.el9.src.rpm", 27445909, "GPLv3+", "Rocky Enterprise Software Foundation", "", "A GNU collection of binary utilities", "8fc84814c2ebb929f4715456806cb7a4"},
			{intRef(), "curl-minimal", "7.76.1", "26.el9_3.2.0.1", "aarch64", "curl-7.76.1-26.el9_3.2.0.1.src.rpm", 249065, "MIT", "Rocky Enterprise Software Foundation", "", "Conservatively configured build of curl for minimal installations", "bee688ae4dba3fe953616331aff1453d"},
			{intRef(), "cyrus-sasl-lib", "2.1.27", "21.el9", "aarch64", "cyrus-sasl-2.1.27-21.el9.src.rpm", 2456576, "BSD with advertising", "Rocky Enterprise Software Foundation", "", "Shared libraries needed by applications which use Cyrus SASL", "b1ad1f80de616d24fa27f6c105c9bf6e"},
			{intRef(), "util-linux-core", "2.37.4", "15.el9", "aarch64", "util-linux-2.37.4-15.el9.src.rpm", 2214247, "GPLv2 and GPLv2+ and LGPLv2+ and BSD with advertising and Public Domain", "Rocky Enterprise Software Foundation", "", "The most essential utilities from the util-linux suite.", "fbbab424e6ce5a66d9969f8a40999c40"},
			{intRef(), "libarchive", "3.5.3", "4.el9", "aarch64", "libarchive-3.5.3-4.el9.src.rpm", 930486, "BSD", "Rocky Enterprise Software Foundation", "", "A library for handling streaming archive formats", "4260b6f1bbd759d1e525558706869b5a"},
			{intRef(), "rpm", "4.16.1.3", "25.el9", "aarch64", "rpm-4.16.1.3-25.el9.src.rpm", 2993997, "GPLv2+", "Rocky Enterprise Software Foundation", "", "The RPM package management system", "4b87cd297e444464857c2954627d78c9"},
			{intRef(), "rpm-libs", "4.16.1.3", "25.el9", "aarch64", "rpm-4.16.1.3-25.el9.src.rpm", 817620, "GPLv2+ and LGPLv2+ with exceptions", "Rocky Enterprise Software Foundation", "", "Libraries for manipulating RPM packages", "8d5728a8746edf4f86e7808d51fec834"},
			{intRef(), "libsolv", "0.7.24", "2.el9", "aarch64", "libsolv-0.7.24-2.el9.src.rpm", 879330, "BSD", "Rocky Enterprise Software Foundation", "", "Package dependency solver", "cb969680ca5173970f779e2d7e6a5303"},
			{intRef(), "libmodulemd", "2.13.0", "2.el9", "aarch64", "libmodulemd-2.13.0-2.el9.src.rpm", 736095, "MIT", "Rocky Enterprise Software Foundation", "", "Module metadata manipulation library", "0d0c6ec761c1fd6cc07e5b167e815ab1"},
			{intRef(), "rpm-build-libs", "4.16.1.3", "25.el9", "aarch64", "rpm-4.16.1.3-25.el9.src.rpm", 202398, "GPLv2+ and LGPLv2+ with exceptions", "Rocky Enterprise Software Foundation", "", "Libraries for building RPM packages", "8faeddedcbf5fe937a65a4fcae205730"},
			{intRef(1), "openssl", "3.0.7", "24.el9", "aarch64", "openssl-3.0.7-24.el9.src.rpm", 1853491, "ASL 2.0", "Rocky Enterprise Software Foundation", "", "Utilities from the general purpose cryptography library with TLS implementation", "7fb066b523e625e60a1602af0020b846"},
			{intRef(), "libevent", "2.1.12", "6.el9", "aarch64", "libevent-2.1.12-6.el9.src.rpm", 1135978, "BSD and ISC", "Rocky Enterprise Software Foundation", "", "Abstract asynchronous event notification library", "67878004385b13238449fd2944c0dd0c"},
			{intRef(), "openldap", "2.6.3", "1.el9", "aarch64", "openldap-2.6.3-1.el9.src.rpm", 1044490, "OpenLDAP", "Rocky Enterprise Software Foundation", "", "LDAP support libraries", "3d9909889f1c0f6efe9d2ef7a84321cd"},
			{intRef(), "gnupg2", "2.3.3", "4.el9", "aarch64", "gnupg2-2.3.3-4.el9.src.rpm", 9762309, "GPLv3+", "Rocky Enterprise Software Foundation", "", "Utility for secure communication and data storage", "3274d3b792b522addc862ea7e0fddeb4"},
			{intRef(), "gpgme", "1.15.1", "6.el9", "aarch64", "gpgme-1.15.1-6.el9.src.rpm", 616153, "LGPLv2+ and GPLv3+", "Rocky Enterprise Software Foundation", "", "GnuPG Made Easy - high level crypto API", "9cc1d55c337d947ca68cacf21a611501"},
			{intRef(), "librepo", "1.14.5", "1.el9", "aarch64", "librepo-1.14.5-1.el9.src.rpm", 232244, "LGPLv2+", "Rocky Enterprise Software Foundation", "", "Repodata downloading library", "51d56e68c9396ae9d57d41e0a54df69c"},
			{intRef(), "libdnf", "0.69.0", "6.el9_3", "aarch64", "libdnf-0.69.0-6.el9_3.src.rpm", 2002357, "LGPLv2+", "Rocky Enterprise Software Foundation", "", "Library providing simplified C and Python API to libsolv", "10544f09748b4e793045fcce6e64ad00"},
			{intRef(), "python3-pip-wheel", "21.2.3", "7.el9", "noarch", "python-pip-21.2.3-7.el9.src.rpm", 1220855, "MIT and Python and ASL 2.0 and BSD and ISC and LGPLv2 and MPLv2.0 and (ASL 2.0 or BSD)", "Rocky Enterprise Software Foundation", "", "The pip wheel", "1e32dd07ba486d0e37b9cefa0dd9ab1e"},
			{intRef(), "python3", "3.9.18", "1.el9_3", "aarch64", "python3.9-3.9.18-1.el9_3.src.rpm", 85877, "Python", "Rocky Enterprise Software Foundation", "", "Python 3.9 interpreter", "5a78fefe98eedef39bf484a856fa37b9"},
			{intRef(), "python3-libs", "3.9.18", "1.el9_3", "aarch64", "python3.9-3.9.18-1.el9_3.src.rpm", 34486753, "Python", "Rocky Enterprise Software Foundation", "", "Python runtime libraries", "8f386691baf4d6009937edfb4976427c"},
			{intRef(), "python3-libdnf", "0.69.0", "6.el9_3", "aarch64", "libdnf-0.69.0-6.el9_3.src.rpm", 3934799, "LGPLv2+", "Rocky Enterprise Software Foundation", "", "Python 3 bindings for the libdnf library.", "5031b5c82c4ecb3e95f90a8ea63b66c7"},
			{intRef(), "python3-hawkey", "0.69.0", "6.el9_3", "aarch64", "libdnf-0.69.0-6.el9_3.src.rpm", 379963, "LGPLv2+", "Rocky Enterprise Software Foundation", "", "Python 3 bindings for the hawkey library", "69047cecf499a698ff2a3af0e4f05e2f"},
			{intRef(), "python3-libcomps", "0.1.18", "1.el9", "aarch64", "libcomps-0.1.18-1.el9.src.rpm", 155170, "GPLv2+", "Rocky Enterprise Software Foundation", "", "Python 3 bindings for libcomps library", "ff6d6d4e40c5c07fb8fcdac070d8e407"},
			{intRef(), "python3-gpg", "1.15.1", "6.el9", "aarch64", "gpgme-1.15.1-6.el9.src.rpm", 1459878, "LGPLv2+ and GPLv3+", "Rocky Enterprise Software Foundation", "", "gpgme bindings for Python 3", "e668edb9d65f51f7f528636b2e41e6da"},
			{intRef(), "libfdisk", "2.37.4", "15.el9", "aarch64", "util-linux-2.37.4-15.el9.src.rpm", 371619, "LGPLv2+", "Rocky Enterprise Software Foundation", "", "Partitioning library for fdisk-like programs.", "5bb166584370736c8691c46f057fe175"},
			{intRef(), "gzip", "1.12", "1.el9", "aarch64", "gzip-1.12-1.el9.src.rpm", 421717, "GPLv3+ and GFDL", "Rocky Enterprise Software Foundation", "", "The GNU data compression program", "296cb3a8f8eca2e717148c7ca312e417"},
			{intRef(), "cracklib", "2.9.6", "27.el9", "aarch64", "cracklib-2.9.6-27.el9.src.rpm", 431178, "LGPLv2+", "Rocky Enterprise Software Foundation", "", "A password-checking library", "43f63d5d9b8bc981aabdbda1e73c1344"},
			{intRef(), "cracklib-dicts", "2.9.6", "27.el9", "aarch64", "cracklib-2.9.6-27.el9.src.rpm", 9815154, "LGPLv2+", "Rocky Enterprise Software Foundation", "", "The standard CrackLib dictionaries", "4200739f84e8d208241f8426f7f6f774"},
			{intRef(), "libpwquality", "1.4.4", "8.el9", "aarch64", "libpwquality-1.4.4-8.el9.src.rpm", 612232, "BSD or GPLv2+", "Rocky Enterprise Software Foundation", "", "A library for password generation and password quality checking", "11c81a0901f33974b3fedc3b5d8ab290"},
			{intRef(), "pam", "1.5.1", "15.el9", "aarch64", "pam-1.5.1-15.el9.src.rpm", 4634416, "BSD and GPLv2+", "Rocky Enterprise Software Foundation", "", "An extensible library which provides authentication for applications", "5b3013602ad8d0a6d59bdbb965e24224"},
			{intRef(), "libuser", "0.63", "13.el9", "aarch64", "libuser-0.63-13.el9.src.rpm", 2602027, "LGPLv2+", "Rocky Enterprise Software Foundation", "", "A user and group account administration library", "7c4721b693979f0a08aafd82ad050f2a"},
			{intRef(), "passwd", "0.80", "12.el9", "aarch64", "passwd-0.80-12.el9.src.rpm", 478407, "BSD or GPL+", "Rocky Enterprise Software Foundation", "", "An utility for setting or changing passwords using PAM", "85852cc4f75bc75248966c7b63a3a2fd"},
			{intRef(), "libsemanage", "3.5", "2.el9", "aarch64", "libsemanage-3.5-2.el9.src.rpm", 306742, "LGPLv2+", "Rocky Enterprise Software Foundation", "", "SELinux binary policy manipulation library", "7744c5ce07e1173a56d690d5bb34b0fc"},
			{intRef(2), "shadow-utils", "4.9", "8.el9", "aarch64", "shadow-utils-4.9-8.el9.src.rpm", 4258909, "BSD and GPLv2+", "Rocky Enterprise Software Foundation", "", "Utilities for managing accounts and shadow password files", "20d34e73a9f6b699fc49e54033c59aa2"},
			{intRef(), "libutempter", "1.2.1", "6.el9", "aarch64", "libutempter-1.2.1-6.el9.src.rpm", 165513, "LGPLv2+", "Rocky Enterprise Software Foundation", "", "A privileged helper for utmp/wtmp updates", "8b5f8b9736171546fc574b6317b182c4"},
			{intRef(), "util-linux", "2.37.4", "15.el9", "aarch64", "util-linux-2.37.4-15.el9.src.rpm", 13939331, "GPLv2 and GPLv2+ and LGPLv2+ and BSD with advertising and Public Domain", "Rocky Enterprise Software Foundation", "", "A collection of basic system utilities", "320ffa477978e81b69d24f71d505e649"},
			{intRef(), "tpm2-tss", "3.2.2", "2.el9", "aarch64", "tpm2-tss-3.2.2-2.el9.src.rpm", 2480560, "BSD", "Rocky Enterprise Software Foundation", "", "TPM2.0 Software Stack", "28b40028ed4184b60a60fdd326630849"},
			{intRef(), "ima-evm-utils", "1.4", "4.el9", "aarch64", "ima-evm-utils-1.4-4.el9.src.rpm", 187511, "GPLv2", "Rocky Enterprise Software Foundation", "", "IMA/EVM support utilities", "65ce199c9352fe74e7b497d3c6d4b83b"},
			{intRef(), "rpm-sign-libs", "4.16.1.3", "25.el9", "aarch64", "rpm-4.16.1.3-25.el9.src.rpm", 68380, "GPLv2+ and LGPLv2+ with exceptions", "Rocky Enterprise Software Foundation", "", "Libraries for signing RPM packages", "cb957727484120aa31f607eaa8c7c9c8"},
			{intRef(), "python3-rpm", "4.16.1.3", "25.el9", "aarch64", "rpm-4.16.1.3-25.el9.src.rpm", 242679, "GPLv2+", "Rocky Enterprise Software Foundation", "", "Python 3 bindings for apps which will manipulate RPM packages", "d3fdd5370c034a3695334d26c0eeffda"},
			{intRef(), "python3-dnf", "4.14.0", "8.el9", "noarch", "dnf-4.14.0-8.el9.src.rpm", 1924491, "GPLv2+", "Rocky Enterprise Software Foundation", "", "Python 3 interface to DNF", "02da4e125b4a1fd7aa18f68179e63508"},
			{intRef(), "dnf", "4.14.0", "8.el9", "noarch", "dnf-4.14.0-8.el9.src.rpm", 2425263, "GPLv2+", "Rocky Enterprise Software Foundation", "", "Package manager", "2f670ab4961e529daeec98829e03c05a"},
			{intRef(), "yum", "4.14.0", "8.el9", "noarch", "dnf-4.14.0-8.el9.src.rpm", 78878, "GPLv2+", "Rocky Enterprise Software Foundation", "", "Package manager", "ceff95310e4939dc28080530b971e663"},
			{intRef(), "usermode", "1.114", "4.el9", "aarch64", "usermode-1.114-4.el9.src.rpm", 933249, "GPLv2+", "Rocky Enterprise Software Foundation", "", "Tools for certain user account management tasks", "0e9b8aa105ba9759085723067b31957c"},
			{intRef(), "crypto-policies-scripts", "20230731", "1.git94f0e2c.el9_3.1", "noarch", "crypto-policies-20230731-1.git94f0e2c.el9_3.1.src.rpm", 238108, "LGPLv2+", "Rocky Enterprise Software Foundation", "", "Tool to switch between crypto policies", "02add95125405bdcf8ad01ec6d657619"},
			{intRef(), "libusbx", "1.0.26", "1.el9", "aarch64", "libusbx-1.0.26-1.el9.src.rpm", 181918, "LGPLv2+", "Rocky Enterprise Software Foundation", "", "Library for accessing USB devices", "0519340e5e712ee4ad7d6e673a24ae8f"},
			{intRef(1), "findutils", "4.8.0", "6.el9", "aarch64", "findutils-4.8.0-6.el9.src.rpm", 1805102, "GPLv3+", "Rocky Enterprise Software Foundation", "", "The GNU versions of find utilities (find and xargs)", "b5c5008dd621d16bec54b5443ae94567"},
			{intRef(2), "tar", "1.34", "6.el9_1", "aarch64", "tar-1.34-6.el9_1.src.rpm", 3174805, "GPLv3+", "Rocky Enterprise Software Foundation", "", "GNU file archiving program", "fd7ada9e2ea8fbe846d917eaccbcb0bb"},
			{intRef(2), "vim-minimal", "8.2.2637", "20.el9_1", "aarch64", "vim-8.2.2637-20.el9_1.src.rpm", 1462417, "Vim and MIT", "Rocky Enterprise Software Foundation", "", "A minimal version of the VIM editor", "d143cdf202fce6dd37d18f44fcf0152d"},
			{intRef(), "hostname", "3.23", "6.el9", "aarch64", "hostname-3.23-6.el9.src.rpm", 91672, "GPLv2+", "Rocky Enterprise Software Foundation", "", "Utility to set/show the host name or domain name", "8d8cdc55f002f536f30631f92b73d81f"},
			{intRef(), "less", "590", "2.el9_2", "aarch64", "less-590-2.el9_2.src.rpm", 478520, "GPLv3+ or BSD", "Rocky Enterprise Software Foundation", "", "A text file browser similar to more, but better", "c25b8061a8b855fceb0482ce5d660cc2"},
			{intRef(), "rootfiles", "8.1", "31.el9", "noarch", "rootfiles-8.1-31.el9.src.rpm", 817, "Public Domain", "Rocky Enterprise Software Foundation", "", "The basic required files for the root user's directory", "257b067816a2842693e462d0fbbdd46f"},
		}

		return toPackageInfo(pkgs)
	}

	// rpm -ql python --dbpath /path/to/testdata/centos5-plain | awk '{printf "\"%s\",\n", $1}'
	CentOS5PythonInstalledFileNames = []string{
		"/usr/bin/pydoc",
		"/usr/bin/python",
		"/usr/bin/python2",
		"/usr/bin/python2.4",
		"/usr/share/doc/python-2.4.3",
		"/usr/share/doc/python-2.4.3/LICENSE",
		"/usr/share/doc/python-2.4.3/README",
		"/usr/share/man/man1/python.1.gz",
	}

	// rpm -ql glibc --dbpath /path/to/testdata/centos6-plain | awk '{printf "\"%s\",\n", $1}'
	CentOS6GlibcInstalledFileNames = []string{
		"/etc/gai.conf",
		"/etc/ld.so.cache",
		"/etc/ld.so.conf",
		"/etc/ld.so.conf.d",
		"/etc/localtime",
		"/etc/nsswitch.conf",
		"/etc/rpc",
		"/lib64/ld-2.12.so",
		"/lib64/ld-linux-x86-64.so.2",
		"/lib64/libBrokenLocale-2.12.so",
		"/lib64/libBrokenLocale.so.1",
		"/lib64/libSegFault.so",
		"/lib64/libanl-2.12.so",
		"/lib64/libanl.so.1",
		"/lib64/libc-2.12.so",
		"/lib64/libc.so.6",
		"/lib64/libcidn-2.12.so",
		"/lib64/libcidn.so.1",
		"/lib64/libcrypt-2.12.so",
		"/lib64/libcrypt.so.1",
		"/lib64/libdl-2.12.so",
		"/lib64/libdl.so.2",
		"/lib64/libm-2.12.so",
		"/lib64/libm.so.6",
		"/lib64/libnsl-2.12.so",
		"/lib64/libnsl.so.1",
		"/lib64/libnss_compat-2.12.so",
		"/lib64/libnss_compat.so.2",
		"/lib64/libnss_dns-2.12.so",
		"/lib64/libnss_dns.so.2",
		"/lib64/libnss_files-2.12.so",
		"/lib64/libnss_files.so.2",
		"/lib64/libnss_hesiod-2.12.so",
		"/lib64/libnss_hesiod.so.2",
		"/lib64/libnss_nis-2.12.so",
		"/lib64/libnss_nis.so.2",
		"/lib64/libnss_nisplus-2.12.so",
		"/lib64/libnss_nisplus.so.2",
		"/lib64/libpthread-2.12.so",
		"/lib64/libpthread.so.0",
		"/lib64/libresolv-2.12.so",
		"/lib64/libresolv.so.2",
		"/lib64/librt-2.12.so",
		"/lib64/librt.so.1",
		"/lib64/libthread_db-1.0.so",
		"/lib64/libthread_db.so.1",
		"/lib64/libutil-2.12.so",
		"/lib64/libutil.so.1",
		"/lib64/rtkaio",
		"/lib64/rtkaio/librt.so.1",
		"/lib64/rtkaio/librtkaio-2.12.so",
		"/sbin/ldconfig",
		"/sbin/sln",
		"/usr/lib64/gconv",
		"/usr/lib64/gconv/ANSI_X3.110.so",
		"/usr/lib64/gconv/ARMSCII-8.so",
		"/usr/lib64/gconv/ASMO_449.so",
		"/usr/lib64/gconv/BIG5.so",
		"/usr/lib64/gconv/BIG5HKSCS.so",
		"/usr/lib64/gconv/BRF.so",
		"/usr/lib64/gconv/CP10007.so",
		"/usr/lib64/gconv/CP1125.so",
		"/usr/lib64/gconv/CP1250.so",
		"/usr/lib64/gconv/CP1251.so",
		"/usr/lib64/gconv/CP1252.so",
		"/usr/lib64/gconv/CP1253.so",
		"/usr/lib64/gconv/CP1254.so",
		"/usr/lib64/gconv/CP1255.so",
		"/usr/lib64/gconv/CP1256.so",
		"/usr/lib64/gconv/CP1257.so",
		"/usr/lib64/gconv/CP1258.so",
		"/usr/lib64/gconv/CP737.so",
		"/usr/lib64/gconv/CP775.so",
		"/usr/lib64/gconv/CP932.so",
		"/usr/lib64/gconv/CSN_369103.so",
		"/usr/lib64/gconv/CWI.so",
		"/usr/lib64/gconv/DEC-MCS.so",
		"/usr/lib64/gconv/EBCDIC-AT-DE-A.so",
		"/usr/lib64/gconv/EBCDIC-AT-DE.so",
		"/usr/lib64/gconv/EBCDIC-CA-FR.so",
		"/usr/lib64/gconv/EBCDIC-DK-NO-A.so",
		"/usr/lib64/gconv/EBCDIC-DK-NO.so",
		"/usr/lib64/gconv/EBCDIC-ES-A.so",
		"/usr/lib64/gconv/EBCDIC-ES-S.so",
		"/usr/lib64/gconv/EBCDIC-ES.so",
		"/usr/lib64/gconv/EBCDIC-FI-SE-A.so",
		"/usr/lib64/gconv/EBCDIC-FI-SE.so",
		"/usr/lib64/gconv/EBCDIC-FR.so",
		"/usr/lib64/gconv/EBCDIC-IS-FRISS.so",
		"/usr/lib64/gconv/EBCDIC-IT.so",
		"/usr/lib64/gconv/EBCDIC-PT.so",
		"/usr/lib64/gconv/EBCDIC-UK.so",
		"/usr/lib64/gconv/EBCDIC-US.so",
		"/usr/lib64/gconv/ECMA-CYRILLIC.so",
		"/usr/lib64/gconv/EUC-CN.so",
		"/usr/lib64/gconv/EUC-JISX0213.so",
		"/usr/lib64/gconv/EUC-JP-MS.so",
		"/usr/lib64/gconv/EUC-JP.so",
		"/usr/lib64/gconv/EUC-KR.so",
		"/usr/lib64/gconv/EUC-TW.so",
		"/usr/lib64/gconv/GB18030.so",
		"/usr/lib64/gconv/GBBIG5.so",
		"/usr/lib64/gconv/GBGBK.so",
		"/usr/lib64/gconv/GBK.so",
		"/usr/lib64/gconv/GEORGIAN-ACADEMY.so",
		"/usr/lib64/gconv/GEORGIAN-PS.so",
		"/usr/lib64/gconv/GOST_19768-74.so",
		"/usr/lib64/gconv/GREEK-CCITT.so",
		"/usr/lib64/gconv/GREEK7-OLD.so",
		"/usr/lib64/gconv/GREEK7.so",
		"/usr/lib64/gconv/HP-GREEK8.so",
		"/usr/lib64/gconv/HP-ROMAN8.so",
		"/usr/lib64/gconv/HP-ROMAN9.so",
		"/usr/lib64/gconv/HP-THAI8.so",
		"/usr/lib64/gconv/HP-TURKISH8.so",
		"/usr/lib64/gconv/IBM037.so",
		"/usr/lib64/gconv/IBM038.so",
		"/usr/lib64/gconv/IBM1004.so",
		"/usr/lib64/gconv/IBM1008.so",
		"/usr/lib64/gconv/IBM1008_420.so",
		"/usr/lib64/gconv/IBM1025.so",
		"/usr/lib64/gconv/IBM1026.so",
		"/usr/lib64/gconv/IBM1046.so",
		"/usr/lib64/gconv/IBM1047.so",
		"/usr/lib64/gconv/IBM1097.so",
		"/usr/lib64/gconv/IBM1112.so",
		"/usr/lib64/gconv/IBM1122.so",
		"/usr/lib64/gconv/IBM1123.so",
		"/usr/lib64/gconv/IBM1124.so",
		"/usr/lib64/gconv/IBM1129.so",
		"/usr/lib64/gconv/IBM1130.so",
		"/usr/lib64/gconv/IBM1132.so",
		"/usr/lib64/gconv/IBM1133.so",
		"/usr/lib64/gconv/IBM1137.so",
		"/usr/lib64/gconv/IBM1140.so",
		"/usr/lib64/gconv/IBM1141.so",
		"/usr/lib64/gconv/IBM1142.so",
		"/usr/lib64/gconv/IBM1143.so",
		"/usr/lib64/gconv/IBM1144.so",
		"/usr/lib64/gconv/IBM1145.so",
		"/usr/lib64/gconv/IBM1146.so",
		"/usr/lib64/gconv/IBM1147.so",
		"/usr/lib64/gconv/IBM1148.so",
		"/usr/lib64/gconv/IBM1149.so",
		"/usr/lib64/gconv/IBM1153.so",
		"/usr/lib64/gconv/IBM1154.so",
		"/usr/lib64/gconv/IBM1155.so",
		"/usr/lib64/gconv/IBM1156.so",
		"/usr/lib64/gconv/IBM1157.so",
		"/usr/lib64/gconv/IBM1158.so",
		"/usr/lib64/gconv/IBM1160.so",
		"/usr/lib64/gconv/IBM1161.so",
		"/usr/lib64/gconv/IBM1162.so",
		"/usr/lib64/gconv/IBM1163.so",
		"/usr/lib64/gconv/IBM1164.so",
		"/usr/lib64/gconv/IBM1166.so",
		"/usr/lib64/gconv/IBM1167.so",
		"/usr/lib64/gconv/IBM12712.so",
		"/usr/lib64/gconv/IBM1364.so",
		"/usr/lib64/gconv/IBM1371.so",
		"/usr/lib64/gconv/IBM1388.so",
		"/usr/lib64/gconv/IBM1390.so",
		"/usr/lib64/gconv/IBM1399.so",
		"/usr/lib64/gconv/IBM16804.so",
		"/usr/lib64/gconv/IBM256.so",
		"/usr/lib64/gconv/IBM273.so",
		"/usr/lib64/gconv/IBM274.so",
		"/usr/lib64/gconv/IBM275.so",
		"/usr/lib64/gconv/IBM277.so",
		"/usr/lib64/gconv/IBM278.so",
		"/usr/lib64/gconv/IBM280.so",
		"/usr/lib64/gconv/IBM281.so",
		"/usr/lib64/gconv/IBM284.so",
		"/usr/lib64/gconv/IBM285.so",
		"/usr/lib64/gconv/IBM290.so",
		"/usr/lib64/gconv/IBM297.so",
		"/usr/lib64/gconv/IBM420.so",
		"/usr/lib64/gconv/IBM423.so",
		"/usr/lib64/gconv/IBM424.so",
		"/usr/lib64/gconv/IBM437.so",
		"/usr/lib64/gconv/IBM4517.so",
		"/usr/lib64/gconv/IBM4899.so",
		"/usr/lib64/gconv/IBM4909.so",
		"/usr/lib64/gconv/IBM4971.so",
		"/usr/lib64/gconv/IBM500.so",
		"/usr/lib64/gconv/IBM5347.so",
		"/usr/lib64/gconv/IBM803.so",
		"/usr/lib64/gconv/IBM850.so",
		"/usr/lib64/gconv/IBM851.so",
		"/usr/lib64/gconv/IBM852.so",
		"/usr/lib64/gconv/IBM855.so",
		"/usr/lib64/gconv/IBM856.so",
		"/usr/lib64/gconv/IBM857.so",
		"/usr/lib64/gconv/IBM860.so",
		"/usr/lib64/gconv/IBM861.so",
		"/usr/lib64/gconv/IBM862.so",
		"/usr/lib64/gconv/IBM863.so",
		"/usr/lib64/gconv/IBM864.so",
		"/usr/lib64/gconv/IBM865.so",
		"/usr/lib64/gconv/IBM866.so",
		"/usr/lib64/gconv/IBM866NAV.so",
		"/usr/lib64/gconv/IBM868.so",
		"/usr/lib64/gconv/IBM869.so",
		"/usr/lib64/gconv/IBM870.so",
		"/usr/lib64/gconv/IBM871.so",
		"/usr/lib64/gconv/IBM874.so",
		"/usr/lib64/gconv/IBM875.so",
		"/usr/lib64/gconv/IBM880.so",
		"/usr/lib64/gconv/IBM891.so",
		"/usr/lib64/gconv/IBM901.so",
		"/usr/lib64/gconv/IBM902.so",
		"/usr/lib64/gconv/IBM903.so",
		"/usr/lib64/gconv/IBM9030.so",
		"/usr/lib64/gconv/IBM904.so",
		"/usr/lib64/gconv/IBM905.so",
		"/usr/lib64/gconv/IBM9066.so",
		"/usr/lib64/gconv/IBM918.so",
		"/usr/lib64/gconv/IBM921.so",
		"/usr/lib64/gconv/IBM922.so",
		"/usr/lib64/gconv/IBM930.so",
		"/usr/lib64/gconv/IBM932.so",
		"/usr/lib64/gconv/IBM933.so",
		"/usr/lib64/gconv/IBM935.so",
		"/usr/lib64/gconv/IBM937.so",
		"/usr/lib64/gconv/IBM939.so",
		"/usr/lib64/gconv/IBM943.so",
		"/usr/lib64/gconv/IBM9448.so",
		"/usr/lib64/gconv/IEC_P27-1.so",
		"/usr/lib64/gconv/INIS-8.so",
		"/usr/lib64/gconv/INIS-CYRILLIC.so",
		"/usr/lib64/gconv/INIS.so",
		"/usr/lib64/gconv/ISIRI-3342.so",
		"/usr/lib64/gconv/ISO-2022-CN-EXT.so",
		"/usr/lib64/gconv/ISO-2022-CN.so",
		"/usr/lib64/gconv/ISO-2022-JP-3.so",
		"/usr/lib64/gconv/ISO-2022-JP.so",
		"/usr/lib64/gconv/ISO-2022-KR.so",
		"/usr/lib64/gconv/ISO-IR-197.so",
		"/usr/lib64/gconv/ISO-IR-209.so",
		"/usr/lib64/gconv/ISO646.so",
		"/usr/lib64/gconv/ISO8859-1.so",
		"/usr/lib64/gconv/ISO8859-10.so",
		"/usr/lib64/gconv/ISO8859-11.so",
		"/usr/lib64/gconv/ISO8859-13.so",
		"/usr/lib64/gconv/ISO8859-14.so",
		"/usr/lib64/gconv/ISO8859-15.so",
		"/usr/lib64/gconv/ISO8859-16.so",
		"/usr/lib64/gconv/ISO8859-2.so",
		"/usr/lib64/gconv/ISO8859-3.so",
		"/usr/lib64/gconv/ISO8859-4.so",
		"/usr/lib64/gconv/ISO8859-5.so",
		"/usr/lib64/gconv/ISO8859-6.so",
		"/usr/lib64/gconv/ISO8859-7.so",
		"/usr/lib64/gconv/ISO8859-8.so",
		"/usr/lib64/gconv/ISO8859-9.so",
		"/usr/lib64/gconv/ISO8859-9E.so",
		"/usr/lib64/gconv/ISO_10367-BOX.so",
		"/usr/lib64/gconv/ISO_11548-1.so",
		"/usr/lib64/gconv/ISO_2033.so",
		"/usr/lib64/gconv/ISO_5427-EXT.so",
		"/usr/lib64/gconv/ISO_5427.so",
		"/usr/lib64/gconv/ISO_5428.so",
		"/usr/lib64/gconv/ISO_6937-2.so",
		"/usr/lib64/gconv/ISO_6937.so",
		"/usr/lib64/gconv/JOHAB.so",
		"/usr/lib64/gconv/KOI-8.so",
		"/usr/lib64/gconv/KOI8-R.so",
		"/usr/lib64/gconv/KOI8-RU.so",
		"/usr/lib64/gconv/KOI8-T.so",
		"/usr/lib64/gconv/KOI8-U.so",
		"/usr/lib64/gconv/LATIN-GREEK-1.so",
		"/usr/lib64/gconv/LATIN-GREEK.so",
		"/usr/lib64/gconv/MAC-CENTRALEUROPE.so",
		"/usr/lib64/gconv/MAC-IS.so",
		"/usr/lib64/gconv/MAC-SAMI.so",
		"/usr/lib64/gconv/MAC-UK.so",
		"/usr/lib64/gconv/MACINTOSH.so",
		"/usr/lib64/gconv/MIK.so",
		"/usr/lib64/gconv/NATS-DANO.so",
		"/usr/lib64/gconv/NATS-SEFI.so",
		"/usr/lib64/gconv/PT154.so",
		"/usr/lib64/gconv/RK1048.so",
		"/usr/lib64/gconv/SAMI-WS2.so",
		"/usr/lib64/gconv/SHIFT_JISX0213.so",
		"/usr/lib64/gconv/SJIS.so",
		"/usr/lib64/gconv/T.61.so",
		"/usr/lib64/gconv/TCVN5712-1.so",
		"/usr/lib64/gconv/TIS-620.so",
		"/usr/lib64/gconv/TSCII.so",
		"/usr/lib64/gconv/UHC.so",
		"/usr/lib64/gconv/UNICODE.so",
		"/usr/lib64/gconv/UTF-16.so",
		"/usr/lib64/gconv/UTF-32.so",
		"/usr/lib64/gconv/UTF-7.so",
		"/usr/lib64/gconv/VISCII.so",
		"/usr/lib64/gconv/gconv-modules",
		"/usr/lib64/gconv/gconv-modules.cache",
		"/usr/lib64/gconv/libCNS.so",
		"/usr/lib64/gconv/libGB.so",
		"/usr/lib64/gconv/libISOIR165.so",
		"/usr/lib64/gconv/libJIS.so",
		"/usr/lib64/gconv/libJISX0213.so",
		"/usr/lib64/gconv/libKSC.so",
		"/usr/lib64/libmemusage.so",
		"/usr/lib64/libpcprofile.so",
		"/usr/libexec/getconf",
		"/usr/libexec/getconf/POSIX_V6_LP64_OFF64",
		"/usr/libexec/getconf/POSIX_V7_LP64_OFF64",
		"/usr/sbin/glibc_post_upgrade.x86_64",
		"/usr/sbin/iconvconfig",
		"/usr/sbin/iconvconfig.x86_64",
		"/usr/share/doc/glibc-2.12",
		"/usr/share/doc/glibc-2.12/BUGS",
		"/usr/share/doc/glibc-2.12/CONFORMANCE",
		"/usr/share/doc/glibc-2.12/COPYING",
		"/usr/share/doc/glibc-2.12/COPYING.LIB",
		"/usr/share/doc/glibc-2.12/FAQ",
		"/usr/share/doc/glibc-2.12/INSTALL",
		"/usr/share/doc/glibc-2.12/LICENSES",
		"/usr/share/doc/glibc-2.12/NEWS",
		"/usr/share/doc/glibc-2.12/NOTES",
		"/usr/share/doc/glibc-2.12/PROJECTS",
		"/usr/share/doc/glibc-2.12/README",
		"/usr/share/doc/glibc-2.12/README.hesiod",
		"/usr/share/doc/glibc-2.12/README.libm",
		"/var/cache/ldconfig",
		"/var/cache/ldconfig/aux-cache",
	}

	CentOS8NodejsInstalledFileNames = []string{
		"/usr/bin/node",
		"/usr/lib/.build-id",
		"/usr/lib/.build-id/4c",
		"/usr/lib/.build-id/4c/1f16ab7e1bbbf8f56dfbf1a098b6e6a6a9bbe4",
		"/usr/lib/node_modules",
		"/usr/lib/rpm/fileattrs/nodejs_native.attr",
		"/usr/lib/rpm/nodejs_native.req",
		"/usr/share/doc/nodejs",
		"/usr/share/doc/nodejs/AUTHORS",
		"/usr/share/doc/nodejs/CHANGELOG.md",
		"/usr/share/doc/nodejs/COLLABORATOR_GUIDE.md",
		"/usr/share/doc/nodejs/GOVERNANCE.md",
		"/usr/share/doc/nodejs/README.md",
		"/usr/share/licenses/nodejs",
		"/usr/share/licenses/nodejs/LICENSE",
		"/usr/share/man/man1/node.1.gz",
		"/usr/share/node",
		"/usr/share/systemtap",
		"/usr/share/systemtap/tapset",
		"/usr/share/systemtap/tapset/node.stp",
	}

	Mariner2CurlInstalledFileNames = []string{
		"/usr/bin/curl",
		"/usr/bin/curl-config",
		"/usr/share/licenses/curl",
		"/usr/share/licenses/curl/COPYING",
		"/usr/share/man/man1/curl-config.1.gz",
		"/usr/share/man/man1/curl.1.gz",
	}

	LibuuidInstalledFileNames = []string{
		"/usr/lib/.build-id",
		"/usr/lib/.build-id/df/ef6a880adac817216ab9866779a0725e017647",
		"/usr/lib64/libuuid.so.1",
		"/usr/lib64/libuuid.so.1.3.0",
		"/usr/share/licenses/libuuid",
		"/usr/share/licenses/libuuid/COPYING",
		"/usr/share/licenses/libuuid/COPYING.BSD-3",
	}

	CentOS5PythonInstalledFiles = []FileInfo{
		{Path: "/usr/bin/pydoc", Mode: 0x81ed, Digest: "ba5dcf9eb37e464cec372640192d64ad", Size: 82, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/bin/python", Mode: 0x81ed, Digest: "43c97c09763ed41358120d0f87df1a99", Size: 4736, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/bin/python2", Mode: 0xa1ff, Digest: "", Size: 6, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/bin/python2.4", Mode: 0x81ed, Digest: "43c97c09763ed41358120d0f87df1a99", Size: 4736, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/share/doc/python-2.4.3", Mode: 0x41ed, Digest: "", Size: 4096, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/share/doc/python-2.4.3/LICENSE", Mode: 0x81a4, Digest: "0c5ffad47181d838261b710c0dfbdf9a", Size: 13487, Username: "root", Groupname: "root", Flags: 2},
		{Path: "/usr/share/doc/python-2.4.3/README", Mode: 0x81a4, Digest: "3cb4a1f3f7b33acf3886c399979166db", Size: 50765, Username: "root", Groupname: "root", Flags: 2},
		{Path: "/usr/share/man/man1/python.1.gz", Mode: 0x81a4, Digest: "5a6410bfbd042581f41d9045caf829e3", Size: 5307, Username: "root", Groupname: "root", Flags: 2},
	}

	CentOS6GlibcInstalledFiles = []FileInfo{
		{Path: "/etc/gai.conf", Mode: 0x81a4, Digest: "e3b0c44298fc1c149afbf4c8996fb92427ae41e4649b934ca495991b7852b855", Size: 0, Username: "root", Groupname: "root", Flags: 25},
		{Path: "/etc/ld.so.cache", Mode: 0x81a4, Digest: "e3b0c44298fc1c149afbf4c8996fb92427ae41e4649b934ca495991b7852b855", Size: 0, Username: "root", Groupname: "root", Flags: 89},
		{Path: "/etc/ld.so.conf", Mode: 0x81a4, Digest: "239c865e4c0746a01f82b03d38d620853bab2a2ba8e81d6f5606c503e0ea379f", Size: 28, Username: "root", Groupname: "root", Flags: 17},
		{Path: "/etc/ld.so.conf.d", Mode: 0x41ed, Digest: "", Size: 4096, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/etc/localtime", Mode: 0xa1ff, Digest: "", Size: 29, Username: "root", Groupname: "root", Flags: 89},
		{Path: "/etc/nsswitch.conf", Mode: 0x81a4, Digest: "e86e1d6d7941c53222aee80794345866d6534f757b1c28759430efdec905500a", Size: 1688, Username: "root", Groupname: "root", Flags: 17},
		{Path: "/etc/rpc", Mode: 0x81a4, Digest: "990b547bc1bf0b1042d6ed34dba6ea8c512e8f0cd2bc322833f74371f7f6a6f9", Size: 1615, Username: "root", Groupname: "root", Flags: 17},
		{Path: "/lib64/ld-2.12.so", Mode: 0x81ed, Digest: "5801d263321dd810fffbc89f51f16cd59233041df80b85ea633c44dae62c8fab", Size: 159312, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/lib64/ld-linux-x86-64.so.2", Mode: 0xa1ff, Digest: "", Size: 10, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/lib64/libBrokenLocale-2.12.so", Mode: 0x81ed, Digest: "9fbd10ca830fd7dbe72dde5fb917ace196a946741f762046ad9fbe43acb2c220", Size: 8936, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/lib64/libBrokenLocale.so.1", Mode: 0xa1ff, Digest: "", Size: 23, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/lib64/libSegFault.so", Mode: 0x81ed, Digest: "4004f025cb3bc66af2786606340caa82f82fefea0efd394ca526beb1ef5e2062", Size: 22384, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/lib64/libanl-2.12.so", Mode: 0x81ed, Digest: "1fb24ef62b142a46339c28b2933b8cb6c505a4a6a34bdc74a0b3892c435acbb1", Size: 19872, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/lib64/libanl.so.1", Mode: 0xa1ff, Digest: "", Size: 14, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/lib64/libc-2.12.so", Mode: 0x81ed, Digest: "1c7f16207c1c58a5f8e0aaa93709ee2c18180f291a420ec2677b2fa745413b86", Size: 1924768, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/lib64/libc.so.6", Mode: 0xa1ff, Digest: "", Size: 12, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/lib64/libcidn-2.12.so", Mode: 0x81ed, Digest: "ba13a9012c9a823afd127b413dba8a29aa4da8c4bc972586c938493dbe82147a", Size: 201632, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/lib64/libcidn.so.1", Mode: 0xa1ff, Digest: "", Size: 15, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/lib64/libcrypt-2.12.so", Mode: 0x81ed, Digest: "a9c28f92c26cea64950f9bf6bdf43c78d7701099e7147d6a536b2b004c376eaf", Size: 40872, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/lib64/libcrypt.so.1", Mode: 0xa1ff, Digest: "", Size: 16, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/lib64/libdl-2.12.so", Mode: 0x81ed, Digest: "4edf32b1960f925ef3e70be0fcc04cc86ff0d7f691e346410a771e2d37f5290c", Size: 20024, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/lib64/libdl.so.2", Mode: 0xa1ff, Digest: "", Size: 13, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/lib64/libm-2.12.so", Mode: 0x81ed, Digest: "e4dc1e279bf38ea1c8ef1e9c32af12dc3bb88bee8efc6ee1bbd0f87d03716885", Size: 596864, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/lib64/libm.so.6", Mode: 0xa1ff, Digest: "", Size: 12, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/lib64/libnsl-2.12.so", Mode: 0x81ed, Digest: "6427011bdb9c4f4751951b267e3c68922cabed988cb03142774c12b61855d34d", Size: 113904, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/lib64/libnsl.so.1", Mode: 0xa1ff, Digest: "", Size: 14, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/lib64/libnss_compat-2.12.so", Mode: 0x81ed, Digest: "e8ea7d360461591e9f7b9def227604ed2eece427bf2db42c5509e3e7f16a5c6a", Size: 43288, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/lib64/libnss_compat.so.2", Mode: 0xa1ff, Digest: "", Size: 21, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/lib64/libnss_dns-2.12.so", Mode: 0x81ed, Digest: "5f0db745d142aa4ef0e1f2e0feae08260ac6024748a26df6d1aaeebdfd1c8f9e", Size: 27896, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/lib64/libnss_dns.so.2", Mode: 0xa1ff, Digest: "", Size: 18, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/lib64/libnss_files-2.12.so", Mode: 0x81ed, Digest: "46346e35c392d5e22a9f4329210233df0861a20f40ecc6b5ca9e90255882d9ba", Size: 66432, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/lib64/libnss_files.so.2", Mode: 0xa1ff, Digest: "", Size: 20, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/lib64/libnss_hesiod-2.12.so", Mode: 0x81ed, Digest: "0e4b0d96abf6b094ec08121c2493b69b2dd0eda8bcb1fe7149c0690823521cde", Size: 24624, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/lib64/libnss_hesiod.so.2", Mode: 0xa1ff, Digest: "", Size: 21, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/lib64/libnss_nis-2.12.so", Mode: 0x81ed, Digest: "90e04e609d512d819a55ae435c236fc2eff8c2c87c69c60f99f981bcaa5ac738", Size: 53040, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/lib64/libnss_nis.so.2", Mode: 0xa1ff, Digest: "", Size: 18, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/lib64/libnss_nisplus-2.12.so", Mode: 0x81ed, Digest: "de10fd080fd9808b4beceb41d674aa47321ec6f02c2a3cd01702028ec1d75528", Size: 62184, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/lib64/libnss_nisplus.so.2", Mode: 0xa1ff, Digest: "", Size: 22, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/lib64/libpthread-2.12.so", Mode: 0x81ed, Digest: "54b3744fff87694a0b2b9ddc772c5806b638c003640202cd1cad264a5103a5cf", Size: 143280, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/lib64/libpthread.so.0", Mode: 0xa1ff, Digest: "", Size: 18, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/lib64/libresolv-2.12.so", Mode: 0x81ed, Digest: "890897d9703e15c08e272202ee9940eb59f5fdb8f252b5dc3cf91ecbc19ff2eb", Size: 111440, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/lib64/libresolv.so.2", Mode: 0xa1ff, Digest: "", Size: 17, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/lib64/librt-2.12.so", Mode: 0x81ed, Digest: "cf6da056af3bef5de32618ca5f38f1295e69a6e0c6522c2feff7e3b5c182f7bf", Size: 44472, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/lib64/librt.so.1", Mode: 0xa1ff, Digest: "", Size: 13, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/lib64/libthread_db-1.0.so", Mode: 0x81ed, Digest: "cfaa36f8a51e0cbf4a894cfa5ca1683eca312cbd64d42b917f17eea982c55060", Size: 34488, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/lib64/libthread_db.so.1", Mode: 0xa1ff, Digest: "", Size: 19, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/lib64/libutil-2.12.so", Mode: 0x81ed, Digest: "3aa80508ee35c3d255b4ba17f12931e01d0f9aac3698df02dc84a2178db52502", Size: 15056, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/lib64/libutil.so.1", Mode: 0xa1ff, Digest: "", Size: 15, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/lib64/rtkaio", Mode: 0x41ed, Digest: "", Size: 4096, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/lib64/rtkaio/librt.so.1", Mode: 0xa1ff, Digest: "", Size: 17, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/lib64/rtkaio/librtkaio-2.12.so", Mode: 0x81ed, Digest: "951b3377616277234a1462e27572171eb2069733722b2b9606cf335d8bd48346", Size: 53616, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/sbin/ldconfig", Mode: 0x81ed, Digest: "111e3902b6dec11574a24cfa666d23c9a75e74c62bf11de26654377082344d59", Size: 899192, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/sbin/sln", Mode: 0x81ed, Digest: "c741c4c888ccef43c2f90e08464f2e4657d150082b976f35b5527f34f78bb9a4", Size: 692840, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv", Mode: 0x41ed, Digest: "", Size: 12288, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/ANSI_X3.110.so", Mode: 0x81ed, Digest: "992040064638dafed8945e441cf2c925d12e667fc9acaf54e124e7fca177bc5f", Size: 25464, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/ARMSCII-8.so", Mode: 0x81ed, Digest: "7dc62f0563563361d78ac32694b5903d760b59345dfa55b5d7ad1b7467f78274", Size: 12944, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/ASMO_449.so", Mode: 0x81ed, Digest: "7390c7ba183f9a2102aa5c04f826882db39aabbac79de91b37d6efec169f6d39", Size: 12960, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/BIG5.so", Mode: 0x81ed, Digest: "146b7a7ccb12cf4a2e27566a5a4dce847f20010e7d981f0ff06379a323791e5b", Size: 91304, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/BIG5HKSCS.so", Mode: 0x81ed, Digest: "65d447e43e655a644515604b2c92121ffd15fede3231d7d6bacfc462008e2810", Size: 238256, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/BRF.so", Mode: 0x81ed, Digest: "ec1d6b1685048b677df6378ffa0e984e57d138ad2199e506a3e0cb03599056b3", Size: 12952, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/CP10007.so", Mode: 0x81ed, Digest: "a96b8eca29b7b0b8484b9c0f28a2381fae020299e3ccf52a4258dd029d3196bb", Size: 12960, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/CP1125.so", Mode: 0x81ed, Digest: "5573c2ee993954c1a8d0853f745c1b21855fa8bbdbbbebc069a2de18f24bded7", Size: 12952, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/CP1250.so", Mode: 0x81ed, Digest: "5653bdd519861f0f05a9072969803a7ae58b6de60741b679826e378d4d50dfe3", Size: 12952, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/CP1251.so", Mode: 0x81ed, Digest: "d3e10311e66f7839663ce9888e8924db6dfaae919affb4af0b3b5a77d5a27c93", Size: 12952, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/CP1252.so", Mode: 0x81ed, Digest: "277d3822481d4533f7a983fe71676dce7ac025a1145c957ce4c2dbb834aa43ca", Size: 12952, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/CP1253.so", Mode: 0x81ed, Digest: "c63580cf22bff65ecf9072a2c9eb2832c90d3876d967e1ca67ec745fe1f7122c", Size: 12952, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/CP1254.so", Mode: 0x81ed, Digest: "faca2ff2b99aabaf0d696aa08bc3686d2a89e61fa85a76e750755682fe306409", Size: 12952, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/CP1255.so", Mode: 0x81ed, Digest: "338b7322d5aebe8e9d8a0876f2f2b2613180ddb6821ada8c9d59ae6c148a3ab0", Size: 17168, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/CP1256.so", Mode: 0x81ed, Digest: "98bf7d67403c41b59c23a5cb63f213e7decc5ee3ca4bd0b0a093845d671fa003", Size: 12952, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/CP1257.so", Mode: 0x81ed, Digest: "1680a47946bf3808ce0434376ec9f0bf33e3f4af5445dd200b156d3938087a87", Size: 12952, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/CP1258.so", Mode: 0x81ed, Digest: "e8b3e91c65dcbc5c54c8d583f3427171b5c2a8a03e2dcaa4c7381dc0c7a41588", Size: 17184, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/CP737.so", Mode: 0x81ed, Digest: "f6fecc63454898243bd114991d6f890c8b665f7442278a55118a67f632c14ae1", Size: 13224, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/CP775.so", Mode: 0x81ed, Digest: "574c40d9eef48f40034514f18f1481210296208ab030f2e2df39028f91ec340d", Size: 13248, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/CP932.so", Mode: 0x81ed, Digest: "21e10c94b10e5e8930b390d5c72cc6bcfaba1527a3658f1fa604d71c21a493b9", Size: 99280, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/CSN_369103.so", Mode: 0x81ed, Digest: "7bb7dd0446bca606171a852e0c2f15c06d0136825bd9f37bbe5a30c52445f4b9", Size: 12960, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/CWI.so", Mode: 0x81ed, Digest: "eba40bdbbacb29dfcfdf28e0865fb54c1de0c9d6ba6e6aa9d747677514c30699", Size: 12952, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/DEC-MCS.so", Mode: 0x81ed, Digest: "cb95724f9abbf14d053b1e8b913e172bbbdcdb4b57a14bff187ac4940be6b4e9", Size: 12960, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/EBCDIC-AT-DE-A.so", Mode: 0x81ed, Digest: "634f3a05f24553556c77d7ed86cce3d9404c37e432b3e9ab73e3b0fc14e3af0c", Size: 12936, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/EBCDIC-AT-DE.so", Mode: 0x81ed, Digest: "a96baef26e88533e517e0e043a29e9808e11420bec15c794856c7fb2ff8f9545", Size: 12936, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/EBCDIC-CA-FR.so", Mode: 0x81ed, Digest: "ae56e185d73e120782834247462e92dd7cc8b8ffce8d0c89580b688f67f7cafd", Size: 12936, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/EBCDIC-DK-NO-A.so", Mode: 0x81ed, Digest: "057ebb516a0549fb96d8dd3b9cf5ea547d871c52ed4639bc3dd88ef80174253e", Size: 12936, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/EBCDIC-DK-NO.so", Mode: 0x81ed, Digest: "0f07ec7632fdb6bac0767581c0db164348a65f4c3cb21f35c8906356f0753616", Size: 12936, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/EBCDIC-ES-A.so", Mode: 0x81ed, Digest: "37b83762526754a9c3ba1a1446fd7daac1c98c6f6b77cb8d30c74646ea2da7f0", Size: 12968, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/EBCDIC-ES-S.so", Mode: 0x81ed, Digest: "247aec091ff8cea3f34abaa5a56a1d8a255cfe18198b8655188d0aceb2648c50", Size: 12936, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/EBCDIC-ES.so", Mode: 0x81ed, Digest: "542fa1b1b7c19cfffabe77cae3e9fa6be727a49a837aeed2e0990a9f684fec95", Size: 12960, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/EBCDIC-FI-SE-A.so", Mode: 0x81ed, Digest: "5220d14db7feb86c460ac16286244e000afc64ac166adf999389f9cc8c35a67a", Size: 12936, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/EBCDIC-FI-SE.so", Mode: 0x81ed, Digest: "25ca67407e0f7abf98b603dd87e568fbb9c1b6c6cde730f32d2999e4651fdaa5", Size: 12936, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/EBCDIC-FR.so", Mode: 0x81ed, Digest: "dd7baf45053b567de38cc985046e876111f09b29282f93454d2885e4e71d1a8b", Size: 12928, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/EBCDIC-IS-FRISS.so", Mode: 0x81ed, Digest: "4b719eb6dbd58760b59cb30feb924f8800c3805b297f01c09570387f0aad74b1", Size: 12976, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/EBCDIC-IT.so", Mode: 0x81ed, Digest: "958b23145c8e97de5cb5b43165091633a6508f77ddf8d849ecb1a93b0f299f25", Size: 12928, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/EBCDIC-PT.so", Mode: 0x81ed, Digest: "1a80a6659cf8df1315fb600288f4c8ba52cc1dc1ef75b02b08466397d4256f8b", Size: 12928, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/EBCDIC-UK.so", Mode: 0x81ed, Digest: "55d379a230ddbae7dd5be337958b65f23fe9c17b152e37b5ba856e2b81e0c93f", Size: 12960, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/EBCDIC-US.so", Mode: 0x81ed, Digest: "74a8e0b607ec48835ff5b6691806925bf50e235d2cf36c5cbc7c9bb0d451af92", Size: 12928, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/ECMA-CYRILLIC.so", Mode: 0x81ed, Digest: "87280bf714578f7ee7ae931d4457d327ce438fd1e63d84dba88fb264bc818226", Size: 12968, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/EUC-CN.so", Mode: 0x81ed, Digest: "cf0e798fa10f1866f48810c4b54ef9d0ab0285663f9deb48f492f0d0b2200320", Size: 17424, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/EUC-JISX0213.so", Mode: 0x81ed, Digest: "3367511f74cb96ffe94ca4432a668a628f73279090dc3e4466abc8f1d108a089", Size: 17256, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/EUC-JP-MS.so", Mode: 0x81ed, Digest: "24c8a2e1add46e6e07d05bf3ff94531435b7b995fea5ef6776cc6c3c5e311dbd", Size: 87032, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/EUC-JP.so", Mode: 0x81ed, Digest: "d6b2d70ea33a80db73f0e9cc2fbff1575b816244a78727bebde8c9b66bb95b61", Size: 17416, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/EUC-KR.so", Mode: 0x81ed, Digest: "b9e006366263fcc3096c00e3d94a4eb3c2f4ce3faf33da418ef9ee85ee9ccda2", Size: 13088, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/EUC-TW.so", Mode: 0x81ed, Digest: "25f5bf41bfe18d1633475c03861b84c07495ded4e270733ca87d574a92e3f8e4", Size: 26424, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/GB18030.so", Mode: 0x81ed, Digest: "d0c7d23d78b09291c01ec003fe0764a36e60b1b5075e697c18639e68d120eb46", Size: 185064, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/GBBIG5.so", Mode: 0x81ed, Digest: "1fe9eaffe72dfddc7900ab5877a7eed10be6f627ee4af997528042fa4df41f1f", Size: 57968, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/GBGBK.so", Mode: 0x81ed, Digest: "033dc5034a3ff23c3413624e0413e8015408cb3a21029a72947e6ac09624bdaa", Size: 12816, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/GBK.so", Mode: 0x81ed, Digest: "d79f116d58aea8c5df541623adcc8e6811f5ae283eb1392946319e1307d93266", Size: 115784, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/GEORGIAN-ACADEMY.so", Mode: 0x81ed, Digest: "c2f3cfd87f5fa5fbde3fe20b7f6186dcb1c74c3eabd59e0b8275f91be988d689", Size: 12976, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/GEORGIAN-PS.so", Mode: 0x81ed, Digest: "484a36b7a05637efa06f6775c9c8ddfed3a15a6bbabd8079c49a25b9cd0e1210", Size: 12968, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/GOST_19768-74.so", Mode: 0x81ed, Digest: "b5a9cace944419f5820950cf26ca6f509e5e7da58af110474f9825ea91a96396", Size: 12968, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/GREEK-CCITT.so", Mode: 0x81ed, Digest: "14a7fb15f4419f3bdee2c681f075124003f827590bb7b2d5ae26cd60ac4d4f36", Size: 12968, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/GREEK7-OLD.so", Mode: 0x81ed, Digest: "70b8ca5220d415feb253520c226e43cda0cb319d1b1057d1192ddb836d41e21c", Size: 12960, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/GREEK7.so", Mode: 0x81ed, Digest: "a8943672394fb71f7de6fee62993dfad1c96e74b29385e19bf6c3e7bc73bc2c6", Size: 12952, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/HP-GREEK8.so", Mode: 0x81ed, Digest: "e771ff71b9a54067824f09c3603d6108ed78ee750c14eb74641aed4b654dd976", Size: 12960, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/HP-ROMAN8.so", Mode: 0x81ed, Digest: "32c8e4474b042405a537fa2b30f07b6b2abad7cb2398735baceed24416ce303c", Size: 12960, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/HP-ROMAN9.so", Mode: 0x81ed, Digest: "e900ff11c922b3123f3664aabf631c214f7da428d46c9efb069c11cec8e9ff27", Size: 12960, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/HP-THAI8.so", Mode: 0x81ed, Digest: "786301b5e375b262dab6d5e8dccf296defeeb53ae86f53998feb9e3b77dd5357", Size: 12960, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/HP-TURKISH8.so", Mode: 0x81ed, Digest: "445058b3fb9ab2f8b7c25dac1267d3d4826870e8909665d28f27bedadd5fbb35", Size: 12968, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM037.so", Mode: 0x81ed, Digest: "425cfb98c889c268373afc5bee6e01631852731fe15d9778ed7628842b05d992", Size: 12920, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM038.so", Mode: 0x81ed, Digest: "3aca8582c85265063718a45b4ac6dfddda1ccfae5b4c30dee16311865e39151c", Size: 12920, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM1004.so", Mode: 0x81ed, Digest: "43992ca4d2cfb4bd3b9cb4fca07571b79e1fb4eefaff97db880264df39347a4a", Size: 12960, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM1008.so", Mode: 0x81ed, Digest: "3b49768acd1985cfe2e76be8fa22d6cdd56c08fbccce906d4d0ee59c3f92ba04", Size: 12960, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM1008_420.so", Mode: 0x81ed, Digest: "d6bbbbe54f83d86c306b8c2092342aea5fc56b82fcd88b52e2e8d0e3dfec3070", Size: 8832, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM1025.so", Mode: 0x81ed, Digest: "f3c0e550cba622ef91b8a53102c5e4270c601673a0e54868bebce4cbf4a0b0e1", Size: 12960, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM1026.so", Mode: 0x81ed, Digest: "cc0184d5d65a555fee6b2c41c483c67330b110f6ed0bcc1ac650913f63db3a13", Size: 12960, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM1046.so", Mode: 0x81ed, Digest: "10a711755299fd7dff764acada6e364c32f12e4f73db01ad74cb247df1c521ec", Size: 12960, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM1047.so", Mode: 0x81ed, Digest: "52e4c9937da94d4aa599db2233df63879a1de7e7e212c696d109dc5454e14324", Size: 12928, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM1097.so", Mode: 0x81ed, Digest: "82b3616fe06b95b5a1d3f8dc1b2dca7d0b1beb0a8d3b73f68ac591e6abd3d457", Size: 12960, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM1112.so", Mode: 0x81ed, Digest: "6018d7ebe00e71a0784d6862d818cbaa4b48e274600ba30bd6493dc84479ebd6", Size: 12960, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM1122.so", Mode: 0x81ed, Digest: "c0faa948eb9584ac6b7ea44124e3980e37bb8256d2363f34186a36fee1187903", Size: 12960, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM1123.so", Mode: 0x81ed, Digest: "7b951fb124b3dad91ef6796715735ec492789f473cc386fdad22c5ec72c9eb27", Size: 12960, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM1124.so", Mode: 0x81ed, Digest: "efdd56158d1ad11d152f92fe3642d1239db58592ed8e89f9ca7aa443de741333", Size: 12960, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM1129.so", Mode: 0x81ed, Digest: "c40628c4672899313ef256ab1b9b65a037d2a3065ef57a049bee38f39a54c5a3", Size: 12960, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM1130.so", Mode: 0x81ed, Digest: "c04f69b5a23d4ea56a17465a1bef21b3bf7e85cd6d414652f802914f00ca08f5", Size: 12960, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM1132.so", Mode: 0x81ed, Digest: "5b8483c7b724ee0e976d40f7ac5ad288c5db60a295845741d34bca82c54332ae", Size: 12960, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM1133.so", Mode: 0x81ed, Digest: "204539c9917c49dcd4eafb3a2a9e50d8a25eb49f737c934bf5b2fa24753136be", Size: 12960, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM1137.so", Mode: 0x81ed, Digest: "cc23a512c915cbc9310b60ae63535eb327eb7c3835a124e1a7c21fc86ec76109", Size: 12960, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM1140.so", Mode: 0x81ed, Digest: "186fe8305040be8011b767dabd9d31b418f2099b96b6ec05329863d3d7b23ee5", Size: 12960, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM1141.so", Mode: 0x81ed, Digest: "53bb28773a26f735c76ec165e44be23dec8cb3adffe01b5ea0d27f1a2ffe3f4a", Size: 12960, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM1142.so", Mode: 0x81ed, Digest: "a9b84e13f5fc5e62e7c13a714efa90f72f8861eb13fe6f43d1ee8c65b96c1084", Size: 12960, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM1143.so", Mode: 0x81ed, Digest: "1063d45e72751c4fd3b9fa664df207c33c2a1a0663b6670063a136c02cb0ad71", Size: 12960, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM1144.so", Mode: 0x81ed, Digest: "f6af55bcdb776bee236ce28cedfa285b52b0b91a6e7e1659dd3e47ef8e0922bb", Size: 12960, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM1145.so", Mode: 0x81ed, Digest: "a5ba79f319f8838f4f7df383fffaaa68bc92bd85c9ef9b539ccf32dd2314b8f6", Size: 12960, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM1146.so", Mode: 0x81ed, Digest: "d3e38b5b995fcd653a2a5f898ed6e599c062a05d9dee248838a622533a23a982", Size: 12960, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM1147.so", Mode: 0x81ed, Digest: "8ef03c4cdb54b359415856dc0c0316a1c642090bd4601a0cd703cac368e59e88", Size: 12960, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM1148.so", Mode: 0x81ed, Digest: "ea4aff633ee245e1a2db9c31dcbc9d16afeaf79b75164963ae9316f4d555f764", Size: 12960, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM1149.so", Mode: 0x81ed, Digest: "348b8c947876ead5581452264ce8604eed0897966d65869b904dcba44f9fca54", Size: 12960, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM1153.so", Mode: 0x81ed, Digest: "0bf8e00624c78b6ff3ab224467769b91949f2460408c5aa071020d30d84858e8", Size: 12960, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM1154.so", Mode: 0x81ed, Digest: "4bdbda5dbbc19e94801c2d6222a7571a16db4e423df7c10f1482bf1f3945bf5a", Size: 12960, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM1155.so", Mode: 0x81ed, Digest: "5b9fdbccf79d1c2de443b77fc288d3422aa8271508ce216ecc0ff4f98bac11b8", Size: 12960, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM1156.so", Mode: 0x81ed, Digest: "db9bcc97b8cff1f0135e5f3ef24a176b6567418ec6b566ef4f0d23e9ba1d3e6d", Size: 12960, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM1157.so", Mode: 0x81ed, Digest: "bb166d507b85022dcb6443cf64b99ad925af7abfbbc6f51513917205e58dfcc6", Size: 12960, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM1158.so", Mode: 0x81ed, Digest: "e431c36d8d9c9a4d446b9bd55a2a184604b0f5704c30d7ad4852bf11f7c40e80", Size: 12960, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM1160.so", Mode: 0x81ed, Digest: "be64038146a91f2e00f47978a71db7cd7b100b779cf1a9a90e6ea247478b4f21", Size: 12960, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM1161.so", Mode: 0x81ed, Digest: "7984ab267971c7a4f915add42406c7ff9c5d688ed696a9c66067448000a8379f", Size: 12960, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM1162.so", Mode: 0x81ed, Digest: "2a83ff0bfb676ed8c782ce444df6b15226043e9702054c2ea2a24d9de5570b97", Size: 12960, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM1163.so", Mode: 0x81ed, Digest: "9b4e36dbcb7ac1a3a4ff15caabcf05a8dd33833f61b04964a1f30bf5e31c7f53", Size: 12960, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM1164.so", Mode: 0x81ed, Digest: "dcabaa9bf3da292d2cb6ad8757e404bbfce1f1f13288a721da441ed3bb6d2a7c", Size: 12960, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM1166.so", Mode: 0x81ed, Digest: "66b516cf29c0fbbae9191e06bd6bb25e8a9a58673e63376cf310b97559d9aa70", Size: 12960, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM1167.so", Mode: 0x81ed, Digest: "8525cdad406c2e0c549c5d9f133208406e31419bf5f6f9157e6497c4e3623a70", Size: 12960, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM12712.so", Mode: 0x81ed, Digest: "643576aef3e0b66fa45e8a489abfb0150760a4e49c15615c7ac2345ffedbbb7e", Size: 12960, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM1364.so", Mode: 0x81ed, Digest: "6dde92b7e4a7fdbe575c78a03fdb83794df8e9320abb8fdd7f9eeecb045be5cb", Size: 152408, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM1371.so", Mode: 0x81ed, Digest: "8427ccba754f7cca355ec077dde8ce72cdbfdcb2e3c1e1d7223f29d8d5bb963b", Size: 127832, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM1388.so", Mode: 0x81ed, Digest: "884199a6b3fa141bd8b94265703035f5e3f6502008262c16f485d3278d012b1a", Size: 172888, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM1390.so", Mode: 0x81ed, Digest: "534ca78d2edfd037db714d8a239bbcb39da8cf16b75b62f596c94d1c42388b80", Size: 230336, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM1399.so", Mode: 0x81ed, Digest: "a82165f90de1e0a17256dd69ebe4ed06894798acd0d31b82ad776e12bbe5eb44", Size: 230336, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM16804.so", Mode: 0x81ed, Digest: "43c8d40e2b0bab213802f180a0af912ee63fd2edf6e5536c6808d90b09cc2f6d", Size: 12960, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM256.so", Mode: 0x81ed, Digest: "9e20014decc14566834d77a1405e5a1ed9ad271b458c7af2c4df01894eb7ab7c", Size: 12952, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM273.so", Mode: 0x81ed, Digest: "a1b169f9ddd7d47553a0f6cee28e6fdf18ac4f5dc5a1b9895ed3fad739e91c78", Size: 12952, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM274.so", Mode: 0x81ed, Digest: "53a314fc378368913c87de2e8c47c567146748bf9093109379c07b8eaf7c3709", Size: 12920, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM275.so", Mode: 0x81ed, Digest: "a4a9b2d0707aefa4554b7546efb0d961af6730d44cb75df70470a3fd4ff25ee4", Size: 12920, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM277.so", Mode: 0x81ed, Digest: "5db16f5a7e8f62adacffd1380257b51c3a6c3773ac9a4b0a0177e6465f18c2a0", Size: 12952, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM278.so", Mode: 0x81ed, Digest: "a4ec72a8882cae30d4b76052aaef0cfc6d5a5e2c23651d940d02dbc3a079607c", Size: 12952, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM280.so", Mode: 0x81ed, Digest: "db0d8c15f27daaf8b752691427e21e66dbc016897ff9d8dde830547c4b02ac9f", Size: 12952, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM281.so", Mode: 0x81ed, Digest: "453e38c625370c63085d0ead3fb6090416c13c43eb419269c8a1193ff420165b", Size: 12952, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM284.so", Mode: 0x81ed, Digest: "5e1711bf7ca5c630da1fc48e516a4327b663b29bd02d6d7b9c545ac0c25df0ad", Size: 12952, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM285.so", Mode: 0x81ed, Digest: "7d07f890be8fd1c0e81ed93cbd6e1c786e3628b4474974c3fce54408604f3eee", Size: 12952, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM290.so", Mode: 0x81ed, Digest: "71222daa6b904a3de3085af13a9914f4bc13db150fa5be5928c06b9c3d7d0448", Size: 12952, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM297.so", Mode: 0x81ed, Digest: "e1f83ff9d83028a035430bafffc41d22f1113b07d57361e69692f12d226d85c5", Size: 12952, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM420.so", Mode: 0x81ed, Digest: "652ed558ecccdb463bc3b575bd79e1b06cc6bccab04b5d40c44d047574e32c40", Size: 12952, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM423.so", Mode: 0x81ed, Digest: "467915473b431902f24145cbfe0612ef924f3c5f70cc0bd30dae432406bacb51", Size: 12920, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM424.so", Mode: 0x81ed, Digest: "347f38466c512d069d48aecca495a2ccdb4275dd1e2bd66c7de0993ea6ad93e3", Size: 12952, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM437.so", Mode: 0x81ed, Digest: "66459200279e8f1ac2ce9d607eff83a04086dca85f15adc4a2539321ee8db2f7", Size: 12952, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM4517.so", Mode: 0x81ed, Digest: "e5e6b724a0e5926c312d325b8ef0cc69a09b50c87b03359b1668aecd4a24aa8c", Size: 12960, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM4899.so", Mode: 0x81ed, Digest: "ebdffa371b29d24276cf186d5c46b527c6c2044dce5cca563cfc3b445bb4ff44", Size: 12960, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM4909.so", Mode: 0x81ed, Digest: "c62376159e5cda2f24b41938e7bfb41c0172e87b6512aff164b086b4637b0e40", Size: 12960, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM4971.so", Mode: 0x81ed, Digest: "8948e0297c9f304499acb6c9f1771100bc42d148c43da51b002dfce042fc2916", Size: 12960, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM500.so", Mode: 0x81ed, Digest: "f6ef759e605a3caf647594c20a85b1320126ec8a82bf5e19ca3f7a3efa4e6809", Size: 12920, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM5347.so", Mode: 0x81ed, Digest: "66c3957d3205ee50937ef999723b9111c43a80fd007125ecf6f5998b5dfc27cb", Size: 12960, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM803.so", Mode: 0x81ed, Digest: "c30646f442d3a91a3f1653c0de9fee1a5082316eb2d30ae05caf895a4f8ce1b2", Size: 12952, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM850.so", Mode: 0x81ed, Digest: "582e37871f0e561c6f2aa0c19cd6c71ab5fcb305cbcf75f598d09c8ccfeada6c", Size: 12952, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM851.so", Mode: 0x81ed, Digest: "817dfc538b05a9d68a192816f920fb25e3e958afa3db624ee0f249a9df931074", Size: 12952, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM852.so", Mode: 0x81ed, Digest: "fbb4757ac5ef44330b973491711fa0402af065a96b559026a9b1a558355bcbbd", Size: 12952, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM855.so", Mode: 0x81ed, Digest: "e64bd0c29d66dc6d970408f259bfcc30a46b526033a2c405db039ae31f9d2144", Size: 12952, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM856.so", Mode: 0x81ed, Digest: "ca536d16ed76de3fc7ec756e6b4a7741864b2e5ded7cc78fe431dec9f9aa68cb", Size: 12952, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM857.so", Mode: 0x81ed, Digest: "414258a6c41cace42dc502f05555e0c837994619a557da1f92358245012c94bd", Size: 12952, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM860.so", Mode: 0x81ed, Digest: "c5f96217983e7d8a686145fb24f48e708e35b586d967c0198c8a4ac3aa8356bb", Size: 12952, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM861.so", Mode: 0x81ed, Digest: "9a3e5292ecdac5020bfdb7c1f5c94912af865f1020334d8d8bf0895ba3276978", Size: 12952, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM862.so", Mode: 0x81ed, Digest: "56e7dc94a966c36ee7dcc0acea0fdae2c29e3628fed9434909c10c1bd6325c71", Size: 12952, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM863.so", Mode: 0x81ed, Digest: "eda3501dbaca3c5664cd4c638455350cdbf53544f942db318e43801bf276d32e", Size: 12952, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM864.so", Mode: 0x81ed, Digest: "61e561884ca05e2ce3b116adc3b07573d99fbc3805ee61db51923356ef4e8f51", Size: 12952, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM865.so", Mode: 0x81ed, Digest: "f9c10f386c08409fd56820c7f923126f89d48423602502651baf786fd30cdfdd", Size: 12952, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM866.so", Mode: 0x81ed, Digest: "58dbf7ed066c6a6443b929647db41a9364ac3325d5156d0a3d3d2d5eec59a03a", Size: 12952, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM866NAV.so", Mode: 0x81ed, Digest: "10d7efe3ad25e4d2e9354f07e0ba4aed7b0b5d33ce862cd0a47a9ea9065046b0", Size: 12960, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM868.so", Mode: 0x81ed, Digest: "fd2fddf0e49182fb20d9346465431afef951cc2100b4e88a36d60f97d1e8551b", Size: 12952, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM869.so", Mode: 0x81ed, Digest: "f815721dbc62432a2859e7dcfff94eb48c74a81a95cf99ed2a387748126acc3e", Size: 12952, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM870.so", Mode: 0x81ed, Digest: "c7370be8e0cdc6ae1d5411d543888e3e926fc5cf007d4212b0ab55f6b18b6fc9", Size: 12920, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM871.so", Mode: 0x81ed, Digest: "c49d504a3389553cfd32b62ddb4f5615c6d3d042e9ee53a2f49255f92b75a204", Size: 12920, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM874.so", Mode: 0x81ed, Digest: "ee18f36056ccdc3ea0ea51170b7b13cc4b3d177ad03ff0f87fb8faeaf1815a0c", Size: 12952, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM875.so", Mode: 0x81ed, Digest: "c3819c91502c3fe5a97bddd8b24cc1591c86af22b45592b0632887cd27255cdc", Size: 12952, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM880.so", Mode: 0x81ed, Digest: "db40834acd51008dfb6401f02f4a605be5bb3c2ba276d7f2b194e5a789f91d05", Size: 12952, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM891.so", Mode: 0x81ed, Digest: "efea7096e7f631c8802ef8524f6989a2a19fcd4eed21c03afa4d4dfe349f7a91", Size: 12920, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM901.so", Mode: 0x81ed, Digest: "524c7644ba6ef81246c00c43eb293d35e65974ca475ec6223e100f2d6dbe2a83", Size: 12952, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM902.so", Mode: 0x81ed, Digest: "a95583091f5d6b4faecba4d438f69b2a6f56873bf952510527d42c0ae5e80f3a", Size: 12952, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM903.so", Mode: 0x81ed, Digest: "ab1005eef92d77c337e4fade4df20e46bfd652e723e509f23fcf59cdbf401c3f", Size: 12920, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM9030.so", Mode: 0x81ed, Digest: "602800323179c22541d81e0fcd4a9c087c9163f8d54bc4cb8bbd5946f2a19ea8", Size: 12960, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM904.so", Mode: 0x81ed, Digest: "e9553a1bcc614fed1c588ba17e7d7268dd882df4e5b43f8af90f60f747bb0c73", Size: 12920, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM905.so", Mode: 0x81ed, Digest: "6675d58a2186807342842f4dc8c34003f8100f3d909276e1ca4d8f59d2f9c335", Size: 12920, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM9066.so", Mode: 0x81ed, Digest: "e53025857073d03318b256aaf3fb6701f2ccafa86b23e7a64af5499c5a2ae1db", Size: 12960, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM918.so", Mode: 0x81ed, Digest: "dc4530d17ccb16db0fc95f0a72d7b62586dc58f9d3613073fa2746f1b7fee8d9", Size: 12952, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM921.so", Mode: 0x81ed, Digest: "01d7f9d67c172a8215c87b9a0ab33e50fa0fcc78e4cbf59099bfb1185ba90fe1", Size: 12952, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM922.so", Mode: 0x81ed, Digest: "dd81e29d1fbb15ba3b2e27c3cd2cecca2836e89e6e9979a17ff47c455a859845", Size: 12952, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM930.so", Mode: 0x81ed, Digest: "b9a0eb5c7a44351e3ec991f465a614840339660f6346c5874380399f15a705e4", Size: 107344, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM932.so", Mode: 0x81ed, Digest: "7cd334612653e34c25cc11a793d446f9744447d7465f9aadac71afdc4fe75158", Size: 70464, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM933.so", Mode: 0x81ed, Digest: "60a36caa21eec08a70af57799f53dcbdda8154ccd455b6a231bdf6dae8562317", Size: 127824, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM935.so", Mode: 0x81ed, Digest: "859e645a585f3b2300cf9c541451394ed87ed5a115afa0132915f19a37a42c30", Size: 95056, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM937.so", Mode: 0x81ed, Digest: "b606fb90bdc098198390dea0d4c4b61b96fc005c3a06e1f5188ac748438bcdb6", Size: 115536, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM939.so", Mode: 0x81ed, Digest: "e65c7203ddea0f9be74999df1c80087ea64b8f2aaa5b228600aed6fd443956c3", Size: 107344, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM943.so", Mode: 0x81ed, Digest: "651026e2945739d24343c68c8fb4825b1c972cd62139198574457847be7f7f99", Size: 70464, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IBM9448.so", Mode: 0x81ed, Digest: "562033b7733a9da270fc3dcb63742798efd7ae2eff37876e5f53703431453806", Size: 12960, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/IEC_P27-1.so", Mode: 0x81ed, Digest: "ed3c73c1b322a56fd16ba0f59f67621801179d08b25d300550d15c7be10f651c", Size: 12960, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/INIS-8.so", Mode: 0x81ed, Digest: "2bb9fcf9e07d668cd730eba6f39a8f30910e38f431d246094f618903b36880de", Size: 12952, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/INIS-CYRILLIC.so", Mode: 0x81ed, Digest: "5a63beb032390f4bdeff271e0685635106e73e082d7ffc6a3e51136050bdd7f7", Size: 12968, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/INIS.so", Mode: 0x81ed, Digest: "2f70b714ca058b863236c952d45346c97d65499cf921eb6beade6533aeda1860", Size: 12952, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/ISIRI-3342.so", Mode: 0x81ed, Digest: "55549d34b791ab9c1b5ff7e5bd1eaeb6449c50d4cae015287fc8617e9a7486dc", Size: 12960, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/ISO-2022-CN-EXT.so", Mode: 0x81ed, Digest: "233e18a5023aa322aefa3ba30eca3dab207d36c52d7c26e1e9c0fca438e306ad", Size: 47488, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/ISO-2022-CN.so", Mode: 0x81ed, Digest: "a34beadeabbb7ff9c2608bb59175162b1c27f24a6f2d721264e750d8f35219e1", Size: 34664, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/ISO-2022-JP-3.so", Mode: 0x81ed, Digest: "95d8505170dc3ed0194108658ccd565816c4a24510d7fbdaf76df4db2225c266", Size: 25696, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/ISO-2022-JP.so", Mode: 0x81ed, Digest: "8a66e38b3e2ca397701511f05a3b43ff36b51ad4dec14189c303579c75462d43", Size: 43080, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/ISO-2022-KR.so", Mode: 0x81ed, Digest: "fc56396c6fe5ece00c86491fde33ac978e8a5c86f2d167f4a8d3d69e5c1548ab", Size: 17168, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/ISO-IR-197.so", Mode: 0x81ed, Digest: "dcbd420049c2fcb02d5496861a6f195b26aa94263e331f2e82957897bb361e1c", Size: 12960, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/ISO-IR-209.so", Mode: 0x81ed, Digest: "776a0f104862ab9cdbceab290e1bfe20dda86548ebc5fe338fed3394aa70a82c", Size: 12960, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/ISO646.so", Mode: 0x81ed, Digest: "e158d39fb07417594261d46f1b04bb7873a8d0f94aa615b8d3a278d3d4eb0381", Size: 29424, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/ISO8859-1.so", Mode: 0x81ed, Digest: "968ce054a2b284d2b80c18d87d560869c215f2e58984d9edd0dc426c1b6693a0", Size: 12864, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/ISO8859-10.so", Mode: 0x81ed, Digest: "4f99ab6758ca01cd6c1a982439e9c3463a7db05a6f4c6a6c29924a0cd9474efb", Size: 12960, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/ISO8859-11.so", Mode: 0x81ed, Digest: "25d87660e0b6223423e5a8e480f677f9fd36739afec428773c6b0919a3753517", Size: 12960, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/ISO8859-13.so", Mode: 0x81ed, Digest: "c0d5bb8abc2369663190e5cf33b168188c399883ab967370cd9b1dce41b8543e", Size: 12960, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/ISO8859-14.so", Mode: 0x81ed, Digest: "fd5abc392ac4a2e008a82718a2678f2f63aec733dcd5760cc0c45e2e1edd3472", Size: 12960, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/ISO8859-15.so", Mode: 0x81ed, Digest: "079b2dfe86d2e4ad7b8b447dc1413f4ea09363a1bb7cc5d6e504f97a54f1a29b", Size: 12960, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/ISO8859-16.so", Mode: 0x81ed, Digest: "3258073881081b32c4ff3d423f3b9b88c41f963c7fc7911fb2d8819267396741", Size: 12960, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/ISO8859-2.so", Mode: 0x81ed, Digest: "c9ec4ca3666de17a0935abc96984b5519c714082dc8b2d3433fc74071cfbc55d", Size: 12928, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/ISO8859-3.so", Mode: 0x81ed, Digest: "2761782840c784b52cd88247ebed268723f60b75d39e3688fa3da7319677d708", Size: 12928, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/ISO8859-4.so", Mode: 0x81ed, Digest: "4b52d71d5735ddd7b36699591656aeb522749d0fac03bd8fad66b6cb57875911", Size: 12928, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/ISO8859-5.so", Mode: 0x81ed, Digest: "caa9d83d16a6ae7bcc09492457b40401b472d1e43f62aa0d8babc936bee920f6", Size: 12960, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/ISO8859-6.so", Mode: 0x81ed, Digest: "0a2b787fe59d9c46db27a8c79b85c95d7fd59d8c302a39b607c559d779abf22a", Size: 12928, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/ISO8859-7.so", Mode: 0x81ed, Digest: "80a387435ca8b52f775a4f50d38643cb45fc7a8623351d96413c01ef2903b293", Size: 12960, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/ISO8859-8.so", Mode: 0x81ed, Digest: "3cc7b590c220df601961dddf543811a0b0fd5c4c9b42a3b2c878985c1e03fbf0", Size: 12960, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/ISO8859-9.so", Mode: 0x81ed, Digest: "fa01dcd538bf4a73b8f4ee3c18dbaba6e3d48b9d5ac397756bea8279191fd0cd", Size: 12928, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/ISO8859-9E.so", Mode: 0x81ed, Digest: "538379737c13e3b5bfaedd36bf361a1cf9cac0e4bd3b8ec8cb12b8ee3e2ce6a8", Size: 12960, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/ISO_10367-BOX.so", Mode: 0x81ed, Digest: "2fa45a5407a0baed9f340e81d49a3af887cba6187b32f7889df1dcb13f366721", Size: 12968, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/ISO_11548-1.so", Mode: 0x81ed, Digest: "f9fc05588f8cf6cb87108d09ac2dedea4001c85725937867d54fdce0b9c78322", Size: 12872, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/ISO_2033.so", Mode: 0x81ed, Digest: "b615c2a0b3754ab22e26647f23d3bf8cf715e09ab6b448b0db5ee93a45beed72", Size: 12960, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/ISO_5427-EXT.so", Mode: 0x81ed, Digest: "bb178689f4fbc6386035e4a07044aa935a427c366fbd3d455bff1d0478b8b342", Size: 12968, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/ISO_5427.so", Mode: 0x81ed, Digest: "b5848cf336f2c5f2f4aff2163d4702ab5aa0070f20ebb5bb8da64d55c6a2314f", Size: 12960, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/ISO_5428.so", Mode: 0x81ed, Digest: "12262d118494ad6e9c73c37b6127dcfeeb8a456f76c4cd2f1d4abf319d2a05a8", Size: 12960, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/ISO_6937-2.so", Mode: 0x81ed, Digest: "827221fe4800c76fe93de4b740a6c32166543f34dff466c720cf5dabd963f62c", Size: 21224, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/ISO_6937.so", Mode: 0x81ed, Digest: "f2db0c236632d3935758cd971a84981e103f1ca78c78466f4c3a87d7e3b5082e", Size: 25320, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/JOHAB.so", Mode: 0x81ed, Digest: "b32a2ab97f86f448d18d6382b1f2dd0f7e22b39fdb9be66dc278efab10fd8b77", Size: 17448, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/KOI-8.so", Mode: 0x81ed, Digest: "f7c2285643cf5a88f9c254d69737359970183e6e8654f659b60a086fa429c338", Size: 12920, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/KOI8-R.so", Mode: 0x81ed, Digest: "ab9d2abe4a0a5d5cd24ff48c055cbae39bdd5ed6e670dc29e20dbd5c77a675c0", Size: 12952, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/KOI8-RU.so", Mode: 0x81ed, Digest: "404b0886fbfd342b045f7910e47190e1f20485472a3e453bb6a2136160da7964", Size: 12960, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/KOI8-T.so", Mode: 0x81ed, Digest: "0dac37cd0ce494f3536174ac50395e9cb0cd7da98782a4a4e3ef44d5ec715dd5", Size: 12952, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/KOI8-U.so", Mode: 0x81ed, Digest: "a8c038e01d28901f229d133a7d1dcccda933eb8d9164b5261afa2d0849cbc740", Size: 12952, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/LATIN-GREEK-1.so", Mode: 0x81ed, Digest: "f087836226ad927459559461054dd761a977e94e04812069fa3ae7d32c3e17a2", Size: 12968, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/LATIN-GREEK.so", Mode: 0x81ed, Digest: "14fc03ff22fee470dfd217126fd02e87e6d8188c73559c04197d6919988d3d03", Size: 12968, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/MAC-CENTRALEUROPE.so", Mode: 0x81ed, Digest: "e9a4bf52d8525390ce68b5864bb6ba1d9a49297eaa3b1b71527ece549cbb07d7", Size: 12976, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/MAC-IS.so", Mode: 0x81ed, Digest: "93bb4ddfb8a74c1e6f315d4dec5ceb0028050e084b0595ab9c0cd04bb17b4c90", Size: 12952, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/MAC-SAMI.so", Mode: 0x81ed, Digest: "17caa423bb02429d1118e8419e4287cf38b07c968be890ee1af900cccf4c640a", Size: 12960, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/MAC-UK.so", Mode: 0x81ed, Digest: "19dc5563f0528392697f6fa20743ef2eb7469e615bcf63cce8dd8fa08aab62c0", Size: 12952, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/MACINTOSH.so", Mode: 0x81ed, Digest: "7fe9811d921f9d33931f0fbbf2c7fe4ee7884f08167ab514efa7a1d824bf34f0", Size: 12960, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/MIK.so", Mode: 0x81ed, Digest: "1755c53332dc5eecfc6fe957f5edb5079ffd3b3a7e61b5d9d96f9841110987e1", Size: 12952, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/NATS-DANO.so", Mode: 0x81ed, Digest: "89c3ee6c5a5282b968049b5cb12cc98c9c89899746f9797fe4e341d9e3546d2b", Size: 12960, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/NATS-SEFI.so", Mode: 0x81ed, Digest: "022d837a738a37b74e3efab3c2563bb28442b9098b91bc709ddc6b9e69a52f35", Size: 12960, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/PT154.so", Mode: 0x81ed, Digest: "00b432e9fb6736c990aaccccaecb211aab3acb0d671d88bcc7008b220895d6cb", Size: 12952, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/RK1048.so", Mode: 0x81ed, Digest: "01a5f2e5c6f61cfcf1155278a22eacc10113e6a42d3507d15f32790afc558539", Size: 12952, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/SAMI-WS2.so", Mode: 0x81ed, Digest: "421d015c53f05140da80170e12be6231006a090549aa039470960b9216ca1acc", Size: 12960, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/SHIFT_JISX0213.so", Mode: 0x81ed, Digest: "5e1792695e6af92932768d2ff788a0ffc20512d8e622d0da29fdfa606dc44dbc", Size: 17264, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/SJIS.so", Mode: 0x81ed, Digest: "07b9a79ceb2160ad9d36c816ee3dd95dc4056d779bc4b923d6cb3ca6c3460a8b", Size: 95072, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/T.61.so", Mode: 0x81ed, Digest: "90ccfefe08a9c813d65fb75b25da1735b75c4ec780ab31d84923a3cf61c7f825", Size: 21216, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/TCVN5712-1.so", Mode: 0x81ed, Digest: "d0a2b63df0846a77277c27264d55cf5421d5e23d6d00d692514ea325e22e46b1", Size: 17240, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/TIS-620.so", Mode: 0x81ed, Digest: "e2f4ef81de5e7edaf8b75d3e0458d976f6ba17466bbc897541f11c941084ba42", Size: 12960, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/TSCII.so", Mode: 0x81ed, Digest: "5a188ed2be198af3f8ce939bf546cc824319a882bafc310fc51717ce090dbef9", Size: 21256, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/UHC.so", Mode: 0x81ed, Digest: "486dfa345d12e83ae83595ca20d1899fc91f57cdb455da4fdad2a420d3a16c1b", Size: 74616, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/UNICODE.so", Mode: 0x81ed, Digest: "a934e46295f94147b9a70ab07b3e9c97730cc92e99606d34c036dac807a8ec4b", Size: 12960, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/UTF-16.so", Mode: 0x81ed, Digest: "249e5e492a813de719a51fcf92f4776f35aac93407de143c725ae7c09db6b0db", Size: 13016, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/UTF-32.so", Mode: 0x81ed, Digest: "690df77765c61dfc55f27b3f3d124923dd6e2d707e37fd751311c7800e0a648b", Size: 13016, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/UTF-7.so", Mode: 0x81ed, Digest: "97d9709f586804ea4743cd82e159ffd3e9d9cdd8e87dd3eae092c15a3402c4c5", Size: 21200, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/VISCII.so", Mode: 0x81ed, Digest: "cc4284e8b380fc4db4c96c69432fe9eb52d3b12f7f7b421a9c50766e1a023de2", Size: 12952, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/gconv-modules", Mode: 0x81a4, Digest: "caa9c4b59d20b59f1430cb0d8a40100ce96251b2594129e8253ae10aad7019ae", Size: 56103, Username: "root", Groupname: "root", Flags: 17},
		{Path: "/usr/lib64/gconv/gconv-modules.cache", Mode: 0x81a4, Digest: "e3b0c44298fc1c149afbf4c8996fb92427ae41e4649b934ca495991b7852b855", Size: 0, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/libCNS.so", Mode: 0x81ed, Digest: "fda992879a3ed74e8a18d3b41922465903acac3e72ff8e27b166b67075056e3a", Size: 472600, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/libGB.so", Mode: 0x81ed, Digest: "0c65e1ffa767b92f43a2d2043e0e2fd60d5d491c76733c374e7a5cf046d28646", Size: 70344, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/libISOIR165.so", Mode: 0x81ed, Digest: "5884d9ab2010f025826259ac2fb3ee826c17eb0dda58d9545880a792f7c00966", Size: 61824, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/libJIS.so", Mode: 0x81ed, Digest: "fb592d1dd7624dfdf76da191fdeb382c4da6c61ec53f506eb40b174dfe38ed61", Size: 103168, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/libJISX0213.so", Mode: 0x81ed, Digest: "56c3c442aa81192a78996ec64b378141fb0da42ff05fdcbbea3afb4ae9db15cc", Size: 123384, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/gconv/libKSC.so", Mode: 0x81ed, Digest: "16207e9305df9960417e127ac99c9dcdb66eaf8b1d1948461d4658d7be916c86", Size: 49632, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/libmemusage.so", Mode: 0x81ed, Digest: "2430154ed9e5af1995446e43f009787f40b3fc8a00c61c53779d9c6d47a2c2ed", Size: 23808, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/libpcprofile.so", Mode: 0x81ed, Digest: "017eeba9d6a4e77ebf61dc6d3df85f4ee5479c592b69762d492a51c786f70933", Size: 9272, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/libexec/getconf", Mode: 0x41ed, Digest: "", Size: 4096, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/libexec/getconf/POSIX_V6_LP64_OFF64", Mode: 0x81ed, Digest: "bad7760cfdca61b76cae487e050e063c92725084c87917d467900fe5e9ed7a72", Size: 22808, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/libexec/getconf/POSIX_V7_LP64_OFF64", Mode: 0x81ed, Digest: "bad7760cfdca61b76cae487e050e063c92725084c87917d467900fe5e9ed7a72", Size: 22808, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/sbin/glibc_post_upgrade.x86_64", Mode: 0x81c0, Digest: "31449eff6ea6cb37ce03099141f27f71aa992089970aa94a7b359ab363a8c129", Size: 692856, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/sbin/iconvconfig", Mode: 0x81ed, Digest: "7b3ad7c4db6e9419b197c90b3b7596961e1e0b0613d2ce4169ca8e6098d19e3e", Size: 23056, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/sbin/iconvconfig.x86_64", Mode: 0x81ed, Digest: "7b3ad7c4db6e9419b197c90b3b7596961e1e0b0613d2ce4169ca8e6098d19e3e", Size: 23056, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/share/doc/glibc-2.12", Mode: 0x41ed, Digest: "", Size: 4096, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/share/doc/glibc-2.12/BUGS", Mode: 0x81a4, Digest: "9b2d4b25c8600508e1d148feeaed5da04a13daf988d5854012aebcc37fd84ef6", Size: 2954, Username: "root", Groupname: "root", Flags: 2},
		{Path: "/usr/share/doc/glibc-2.12/CONFORMANCE", Mode: 0x81a4, Digest: "64753d94e291cb24b0dceec2e32bd6a0bba35b8d74a6d64d9ea345705088dc88", Size: 6598, Username: "root", Groupname: "root", Flags: 2},
		{Path: "/usr/share/doc/glibc-2.12/COPYING", Mode: 0x81a4, Digest: "204d8eff92f95aac4df6c8122bc1505f468f3a901e5a4cc08940e0ede1938994", Size: 18009, Username: "root", Groupname: "root", Flags: 2},
		{Path: "/usr/share/doc/glibc-2.12/COPYING.LIB", Mode: 0x81a4, Digest: "ca0061fc1381a3ab242310e4b3f56389f28e3d460eb2fd822ed7a21c6f030532", Size: 26536, Username: "root", Groupname: "root", Flags: 2},
		{Path: "/usr/share/doc/glibc-2.12/FAQ", Mode: 0x81a4, Digest: "21705edf8c77e38140afe51b3e0b762afcf464b0c210fe51a85a789191e3c7ef", Size: 84483, Username: "root", Groupname: "root", Flags: 2},
		{Path: "/usr/share/doc/glibc-2.12/INSTALL", Mode: 0x81a4, Digest: "7905f20dc8bf7a0e0d375ee3b70c67f2eba7c52a3436ae8f16b1732abc3dc13b", Size: 22140, Username: "root", Groupname: "root", Flags: 2},
		{Path: "/usr/share/doc/glibc-2.12/LICENSES", Mode: 0x81a4, Digest: "291c84cdb0ece86692c555bb6b61d6229c95c1d83c14751360ace8de17ad5b79", Size: 10728, Username: "root", Groupname: "root", Flags: 2},
		{Path: "/usr/share/doc/glibc-2.12/NEWS", Mode: 0x81a4, Digest: "5bd87874e54b96440f697edcd1cb37cab75b48393605241cb994819bfc324f23", Size: 72289, Username: "root", Groupname: "root", Flags: 2},
		{Path: "/usr/share/doc/glibc-2.12/NOTES", Mode: 0x81a4, Digest: "179b5fe54f02281b21f133cc8757db265af2b403b8978641e7ad6bf8f1ca9e13", Size: 10702, Username: "root", Groupname: "root", Flags: 2},
		{Path: "/usr/share/doc/glibc-2.12/PROJECTS", Mode: 0x81a4, Digest: "b3ea3fbdc20ac7e3477fccb2a3e7e81d4f7e429644379a91455c39b785202a52", Size: 4155, Username: "root", Groupname: "root", Flags: 2},
		{Path: "/usr/share/doc/glibc-2.12/README", Mode: 0x81a4, Digest: "7ea3e20637ecd9ddbc5ca7b23d41003437cf3dd50e309b95aeb239018582fb2b", Size: 4838, Username: "root", Groupname: "root", Flags: 2},
		{Path: "/usr/share/doc/glibc-2.12/README.hesiod", Mode: 0x81a4, Digest: "71b88fbf6a491615b8370677f6cf49d518aa5d85bfba3da36252bc0e43939729", Size: 5672, Username: "root", Groupname: "root", Flags: 2},
		{Path: "/usr/share/doc/glibc-2.12/README.libm", Mode: 0x81a4, Digest: "b4f55a722fcfa05d23ebea15705084d1c307617458414dc758bcf7ffdb009703", Size: 31517, Username: "root", Groupname: "root", Flags: 2},
		{Path: "/var/cache/ldconfig", Mode: 0x41c0, Digest: "", Size: 4096, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/var/cache/ldconfig/aux-cache", Mode: 0x8180, Digest: "e3b0c44298fc1c149afbf4c8996fb92427ae41e4649b934ca495991b7852b855", Size: 0, Username: "root", Groupname: "root", Flags: 89},
	}

	CentOS8NodejsInstalledFiles = []FileInfo{
		{Path: "/usr/bin/node", Mode: 0x81ed, Digest: "b894371f1e9e4fc29eb10f06537f7e9d95302b75f43797b63ded087c56a2a1b4", Size: 31194672, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib/.build-id", Mode: 0x41ed, Digest: "", Size: 0, Username: "root", Groupname: "root", Flags: 4096},
		{Path: "/usr/lib/.build-id/4c", Mode: 0x41ed, Digest: "", Size: 0, Username: "root", Groupname: "root", Flags: 4096},
		{Path: "/usr/lib/.build-id/4c/1f16ab7e1bbbf8f56dfbf1a098b6e6a6a9bbe4", Mode: 0xa1ff, Digest: "", Size: 24, Username: "root", Groupname: "root", Flags: 4096},
		{Path: "/usr/lib/node_modules", Mode: 0x41ed, Digest: "", Size: 0, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib/rpm/fileattrs/nodejs_native.attr", Mode: 0x81a4, Digest: "991876f5bd2383feac22a58397641096d62045629f6e107cb9d211e626d7e226", Size: 119, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib/rpm/nodejs_native.req", Mode: 0x81ed, Digest: "2eb0511016eb87b70278b06c3af7ed39986ac3dc43cad8239060e69c764c3a07", Size: 70, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/share/doc/nodejs", Mode: 0x41ed, Digest: "", Size: 0, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/share/doc/nodejs/AUTHORS", Mode: 0x81a4, Digest: "692b7cc46bbc9630c6e0d7b5692ea9dd970e5a578264813568fa30ec11e193ee", Size: 87615, Username: "root", Groupname: "root", Flags: 2},
		{Path: "/usr/share/doc/nodejs/CHANGELOG.md", Mode: 0x81a4, Digest: "994b0861c5f28d4a57101d746024333b12e09f3d6b825d04ea3f34f36efd0952", Size: 52452, Username: "root", Groupname: "root", Flags: 2},
		{Path: "/usr/share/doc/nodejs/COLLABORATOR_GUIDE.md", Mode: 0x81a4, Digest: "17d2103cb77e34fe60ba703dfba90769ff8d6b3e9be75cad598d0286ee7b84bc", Size: 36320, Username: "root", Groupname: "root", Flags: 2},
		{Path: "/usr/share/doc/nodejs/GOVERNANCE.md", Mode: 0x81a4, Digest: "55e859154a49870e8a5b41ef216fedcc2a7d375b2f87167b25e30f408209226e", Size: 8676, Username: "root", Groupname: "root", Flags: 2},
		{Path: "/usr/share/doc/nodejs/README.md", Mode: 0x81a4, Digest: "ff363d76470542a790a00688d377a0164e6ace34f87042f9bcdd23e78b5b0c26", Size: 26749, Username: "root", Groupname: "root", Flags: 2},
		{Path: "/usr/share/licenses/nodejs", Mode: 0x41ed, Digest: "", Size: 0, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/share/licenses/nodejs/LICENSE", Mode: 0x81a4, Digest: "2b0fe00a83916d0290c8531db25a827e18d01e7c4bf000e9a0f2e826604ba41e", Size: 68764, Username: "root", Groupname: "root", Flags: 128},
		{Path: "/usr/share/man/man1/node.1.gz", Mode: 0x81a4, Digest: "492c658669822df426eeeeffcd33572f9801f06ddfec2a5ab6ee622943d613ea", Size: 4849, Username: "root", Groupname: "root", Flags: 2},
		{Path: "/usr/share/node", Mode: 0x41ed, Digest: "", Size: 0, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/share/systemtap", Mode: 0x41ed, Digest: "", Size: 0, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/share/systemtap/tapset", Mode: 0x41ed, Digest: "", Size: 0, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/share/systemtap/tapset/node.stp", Mode: 0x81a4, Digest: "1df168860f170690b202271d7f604514e64831523db62e7bae3f9a85cbbd7bf4", Size: 3495, Username: "root", Groupname: "root", Flags: 0},
	}

	Mariner2CurlInstalledFiles = []FileInfo{
		{Path: "/usr/bin/curl", Mode: 0x81ed, Digest: "7ef6b3fd854d4ed189dd0aab2ffcf5e6bd58018ff69b97e1b86e1df7b86f3df0", Size: 267672, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/bin/curl-config", Mode: 0x81ed, Digest: "f044fcec45baaaaa7f2de43646719812a1fc75eb3582ae4c9bfc2fd0aa873afc", Size: 7515, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/share/licenses/curl", Mode: 0x41ed, Digest: "", Size: 0, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/share/licenses/curl/COPYING", Mode: 0x81a4, Digest: "6fd1a1c008b5ef4c4741dd188c3f8af6944c14c25afa881eb064f98fb98358e7", Size: 1088, Username: "root", Groupname: "root", Flags: 128},
		{Path: "/usr/share/man/man1/curl-config.1.gz", Mode: 0x81a4, Digest: "e9346781b9d38f416138255182c0ff5818f09adaaf38f08abfd545fa149f77b6", Size: 1763, Username: "root", Groupname: "root", Flags: 2},
		{Path: "/usr/share/man/man1/curl.1.gz", Mode: 0x81a4, Digest: "6a937dea68ab20b8bf9d376bb68fcfe06608a4ddb7d7725986c7ad7451817912", Size: 47985, Username: "root", Groupname: "root", Flags: 2},
	}

	LibuuidInstalledFiles = []FileInfo{
		{Path: "/usr/lib/.build-id", Mode: 0x41ed, Digest: "", Size: 0, Username: "root", Groupname: "root", Flags: 4096},
		{Path: "/usr/lib/.build-id/df/ef6a880adac817216ab9866779a0725e017647", Mode: 0xa1ff, Digest: "", Size: 38, Username: "root", Groupname: "root", Flags: 4096},
		{Path: "/usr/lib64/libuuid.so.1", Mode: 0xa1ff, Digest: "", Size: 16, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib64/libuuid.so.1.3.0", Mode: 0x81ed, Digest: "3c8f59a0e39cce53501045859f5c9a86b82fedd558a2f4110ab1823047b43745", Size: 33496, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/share/licenses/libuuid", Mode: 0x41ed, Digest: "", Size: 0, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/share/licenses/libuuid/COPYING", Mode: 0x81a4, Digest: "f051c5b2e7ed25acac4b79d22b9e5df80447c16f9ec543d6ed21c213c576f6aa", Size: 217, Username: "root", Groupname: "root", Flags: 128},
		{Path: "/usr/share/licenses/libuuid/COPYING.BSD-3", Mode: 0x81a4, Digest: "9b718a9460fed5952466421235bc79eb49d4e9eacc920d7a9dd6285ab8fd6c6d", Size: 1391, Username: "root", Groupname: "root", Flags: 128},
	}

	Rockylinux9HostnameFiles = []FileInfo{
		{Path: "/usr/bin/dnsdomainname", Mode: 0xa1ff, Digest: "", Size: 8, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/bin/domainname", Mode: 0xa1ff, Digest: "", Size: 8, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/bin/hostname", Mode: 0x81ed, Digest: "94ef5f552485167bbb790ab79b0ffb8f1b7b5f6e99d3a2d6becfbe77115a22c5", Size: 69256, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/bin/nisdomainname", Mode: 0xa1ff, Digest: "", Size: 8, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/bin/ypdomainname", Mode: 0xa1ff, Digest: "", Size: 8, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/lib/.build-id", Mode: 0x41ed, Digest: "", Size: 0, Username: "root", Groupname: "root", Flags: 4096},
		{Path: "/usr/lib/.build-id/32", Mode: 0x41ed, Digest: "", Size: 0, Username: "root", Groupname: "root", Flags: 4096},
		{Path: "/usr/lib/.build-id/32/2d92d5e3853c3ef53bcbdee3666eff79eeca92", Mode: 0xa1ff, Digest: "", Size: 28, Username: "root", Groupname: "root", Flags: 4096},
		{Path: "/usr/lib/systemd/system/nis-domainname.service", Mode: 0x81a4, Digest: "571aea8d633c0a9a08a453a387768cfa7e819c52a0cf6302036c0cc8711907ee", Size: 378, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/libexec/hostname", Mode: 0x41ed, Digest: "", Size: 0, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/libexec/hostname/nis-domainname", Mode: 0x81ed, Digest: "e34871863f59e9f0eee4674217d11d4ae46fe20b6496170787f0e45170997a4a", Size: 166, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/share/doc/hostname", Mode: 0x41ed, Digest: "", Size: 0, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/share/doc/hostname/COPYRIGHT", Mode: 0x81a4, Digest: "303020aa9550b292aac5980b98d8bbeb7c8162b044f6d3608b81d906ae67ed36", Size: 831, Username: "root", Groupname: "root", Flags: 2},
		{Path: "/usr/share/licenses/hostname", Mode: 0x41ed, Digest: "", Size: 0, Username: "root", Groupname: "root", Flags: 0},
		{Path: "/usr/share/licenses/hostname/gpl-2.0.txt", Mode: 0x81a4, Digest: "8177f97513213526df2cf6184d8ff986c675afb514d4e68a404010521b880643", Size: 18092, Username: "root", Groupname: "root", Flags: 128},
		{Path: "/usr/share/man/man1/dnsdomainname.1.gz", Mode: 0xa1ff, Digest: "", Size: 13, Username: "root", Groupname: "root", Flags: 2},
		{Path: "/usr/share/man/man1/domainname.1.gz", Mode: 0xa1ff, Digest: "", Size: 13, Username: "root", Groupname: "root", Flags: 2},
		{Path: "/usr/share/man/man1/hostname.1.gz", Mode: 0x81a4, Digest: "e7472cf4dc265034adbb60c69912d55a67de4bcc3d0adfa21425e876ad9e18c8", Size: 2837, Username: "root", Groupname: "root", Flags: 2},
		{Path: "/usr/share/man/man1/nisdomainname.1.gz", Mode: 0xa1ff, Digest: "", Size: 13, Username: "root", Groupname: "root", Flags: 2},
		{Path: "/usr/share/man/man1/ypdomainname.1.gz", Mode: 0xa1ff, Digest: "", Size: 13, Username: "root", Groupname: "root", Flags: 2},
	}

	Rockylinux9HostnameFileNames = []string{
		"/usr/bin/dnsdomainname",
		"/usr/bin/domainname",
		"/usr/bin/hostname",
		"/usr/bin/nisdomainname",
		"/usr/bin/ypdomainname",
		"/usr/lib/.build-id",
		"/usr/lib/.build-id/32",
		"/usr/lib/.build-id/32/2d92d5e3853c3ef53bcbdee3666eff79eeca92",
		"/usr/lib/systemd/system/nis-domainname.service",
		"/usr/libexec/hostname",
		"/usr/libexec/hostname/nis-domainname",
		"/usr/share/doc/hostname",
		"/usr/share/doc/hostname/COPYRIGHT",
		"/usr/share/licenses/hostname",
		"/usr/share/licenses/hostname/gpl-2.0.txt",
		"/usr/share/man/man1/dnsdomainname.1.gz",
		"/usr/share/man/man1/domainname.1.gz",
		"/usr/share/man/man1/hostname.1.gz",
		"/usr/share/man/man1/nisdomainname.1.gz",
		"/usr/share/man/man1/ypdomainname.1.gz",
	}
)
