/*
** Copyright (C) 2001-2025 Zabbix SIA
**
** Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated
** documentation files (the "Software"), to deal in the Software without restriction, including without limitation the
** rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to
** permit persons to whom the Software is furnished to do so, subject to the following conditions:
**
** The above copyright notice and this permission notice shall be included in all copies or substantial portions
** of the Software.
**
** THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
** WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR
** COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
** TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
** SOFTWARE.
**/

package tlsconfig

import (
	"testing"

	"github.com/google/go-cmp/cmp"
)

func TestNewTLSConnectionType(t *testing.T) {
	t.Parallel()

	type args struct {
		tlsType string
	}

	tests := []struct {
		name    string
		args    args
		want    TLSConnectionType
		wantErr bool
	}{
		{
			name:    "+no_tls",
			args:    args{tlsType: "disabled"},
			want:    Disabled,
			wantErr: false,
		},
		{
			name:    "+required",
			args:    args{tlsType: "required"},
			want:    Required,
			wantErr: false,
		},
		{
			name:    "+verify_ca",
			args:    args{tlsType: "verify_ca"},
			want:    VerifyCA,
			wantErr: false,
		},
		{
			name:    "+verify_full",
			args:    args{tlsType: "verify_full"},
			want:    VerifyFull,
			wantErr: false,
		},
		{
			name:    "+case-insensitive",
			args:    args{tlsType: "VERIFY_FULL"},
			want:    VerifyFull,
			wantErr: false,
		},
		{
			name:    "+leading/trailing whitespace",
			args:    args{tlsType: "  required  "},
			want:    Required,
			wantErr: false,
		},
		{
			name:    "+emptyDefaultsToDisabled",
			args:    args{tlsType: ""},
			want:    Disabled,
			wantErr: false,
		},
		{
			name:    "+whitespace-onlyDefaultsToDisabled",
			args:    args{tlsType: "   "},
			want:    Disabled,
			wantErr: false,
		},
		{
			name:    "-invalid connection type",
			args:    args{tlsType: "invalid_type"},
			want:    "",
			wantErr: true,
		},
		{
			name:    "-partialMatch",
			args:    args{tlsType: "verify"},
			want:    "",
			wantErr: true,
		},
	}

	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			t.Parallel()

			got, err := NewTLSConnectionType(tt.args.tlsType)
			if (err != nil) != tt.wantErr {
				t.Fatalf("NewTLSConnectionType() error = %v, wantErr %v", err, tt.wantErr)
			}

			if diff := cmp.Diff(tt.want, got); diff != "" {
				t.Fatalf("NewTLSConnectionType() = %s", diff)
			}
		})
	}
}
