package ntlmssp

import (
	"bytes"
	"testing"

	"github.com/stretchr/testify/assert"
)

func TestGenerateClientChallenge(t *testing.T) {
	tables := []struct {
		want int
		err  error
	}{
		{8, nil},
	}

	for _, table := range tables {
		got, err := generateClientChallenge()
		assert.Equal(t, table.want, len(got))
		assert.Equal(t, table.err, err)

		dup, err := generateClientChallenge()
		assert.Equal(t, table.want, len(dup))
		assert.Equal(t, table.err, err)

		assert.NotEqual(t, got, dup)
	}
}

func TestGenerateExportedSessionKey(t *testing.T) {
	tables := []struct {
		want int
		err  error
	}{
		{16, nil},
	}

	for _, table := range tables {
		got, err := generateExportedSessionKey()
		assert.Equal(t, table.want, len(got))
		assert.Equal(t, table.err, err)

		dup, err := generateExportedSessionKey()
		assert.Equal(t, table.want, len(dup))
		assert.Equal(t, table.err, err)

		assert.NotEqual(t, got, dup)
	}
}

func TestClientProcessChallengeMessage(t *testing.T) {
	tables := []struct {
		client           *Client
		challengeMessage []byte
		channelBindings  *ChannelBindings
		want             []byte
		err              error
	}{
		// The MS-NLMP document gives examples of challenge messages
		// and its resulting authenticate messages for various different
		// scenarios however the message flags do not match and the
		// document does not explain why these flags should be changed.
		// Different flags need to be massaged in each example so it
		// smells of lack of testing. Happy to be corrected here!
		{
			&Client{
				domain:      "Domain",
				password:    "Password",
				username:    "User",
				workstation: "COMPUTER",
				version: &Version{
					ProductMajorVersion: 5,
					ProductMinorVersion: 1,
					ProductBuild:        2600,
					NTLMRevisionCurrent: NTLMSSPRevisionW2K3,
				},
			},
			[]byte{
				0x4e, 0x54, 0x4c, 0x4d, 0x53, 0x53, 0x50, 0x00,
				0x02, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x0c, 0x00,
				0x38, 0x00, 0x00, 0x00, 0x33, 0x82, 0x02, 0xe2,
				0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x06, 0x00, 0x70, 0x17, 0x00, 0x00, 0x00, 0x0f,
				0x53, 0x00, 0x65, 0x00, 0x72, 0x00, 0x76, 0x00,
				0x65, 0x00, 0x72, 0x00,
			},
			nil,
			[]byte{
				0x4e, 0x54, 0x4c, 0x4d, 0x53, 0x53, 0x50, 0x00,
				0x03, 0x00, 0x00, 0x00, 0x18, 0x00, 0x18, 0x00,
				0x6c, 0x00, 0x00, 0x00, 0x18, 0x00, 0x18, 0x00,
				0x84, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x0c, 0x00,
				0x48, 0x00, 0x00, 0x00, 0x08, 0x00, 0x08, 0x00,
				0x54, 0x00, 0x00, 0x00, 0x10, 0x00, 0x10, 0x00,
				0x5c, 0x00, 0x00, 0x00, 0x10, 0x00, 0x10, 0x00,
				0x9c, 0x00, 0x00, 0x00, 0x31, 0x82, 0x02, 0xe2, // Last four bytes originally 0x35, 0x82, 0x80, 0xe2
				0x05, 0x01, 0x28, 0x0a, 0x00, 0x00, 0x00, 0x0f,
				0x44, 0x00, 0x6f, 0x00, 0x6d, 0x00, 0x61, 0x00,
				0x69, 0x00, 0x6e, 0x00, 0x55, 0x00, 0x73, 0x00,
				0x65, 0x00, 0x72, 0x00, 0x43, 0x00, 0x4f, 0x00,
				0x4d, 0x00, 0x50, 0x00, 0x55, 0x00, 0x54, 0x00,
				0x45, 0x00, 0x52, 0x00, 0x98, 0xde, 0xf7, 0xb8,
				0x7f, 0x88, 0xaa, 0x5d, 0xaf, 0xe2, 0xdf, 0x77,
				0x96, 0x88, 0xa1, 0x72, 0xde, 0xf1, 0x1c, 0x7d,
				0x5c, 0xcd, 0xef, 0x13, 0x67, 0xc4, 0x30, 0x11,
				0xf3, 0x02, 0x98, 0xa2, 0xad, 0x35, 0xec, 0xe6,
				0x4f, 0x16, 0x33, 0x1c, 0x44, 0xbd, 0xbe, 0xd9,
				0x27, 0x84, 0x1f, 0x94, 0x51, 0x88, 0x22, 0xb1,
				0xb3, 0xf3, 0x50, 0xc8, 0x95, 0x86, 0x82, 0xec,
				0xbb, 0x3e, 0x3c, 0xb7,
			},
			nil,
		},
		{
			&Client{
				domain:      "Domain",
				password:    "Password",
				username:    "User",
				workstation: "COMPUTER",
				version: &Version{
					ProductMajorVersion: 5,
					ProductMinorVersion: 1,
					ProductBuild:        2600,
					NTLMRevisionCurrent: NTLMSSPRevisionW2K3,
				},
			},
			[]byte{
				0x4e, 0x54, 0x4c, 0x4d, 0x53, 0x53, 0x50, 0x00,
				0x02, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x0c, 0x00,
				0x38, 0x00, 0x00, 0x00, 0x33, 0x82, 0x0a, 0x82,
				0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x06, 0x00, 0x70, 0x17, 0x00, 0x00, 0x00, 0x0f,
				0x53, 0x00, 0x65, 0x00, 0x72, 0x00, 0x76, 0x00,
				0x65, 0x00, 0x72, 0x00,
			},
			nil,
			[]byte{
				0x4e, 0x54, 0x4c, 0x4d, 0x53, 0x53, 0x50, 0x00,
				0x03, 0x00, 0x00, 0x00, 0x18, 0x00, 0x18, 0x00,
				0x6c, 0x00, 0x00, 0x00, 0x18, 0x00, 0x18, 0x00,
				0x84, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x0c, 0x00,
				0x48, 0x00, 0x00, 0x00, 0x08, 0x00, 0x08, 0x00,
				0x54, 0x00, 0x00, 0x00, 0x10, 0x00, 0x10, 0x00,
				0x5c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x9c, 0x00, 0x00, 0x00, 0x31, 0x82, 0x0a, 0x82, // Last four bytes originally 0x35, 0x82, 0x08, 0x82
				0x05, 0x01, 0x28, 0x0a, 0x00, 0x00, 0x00, 0x0f,
				0x44, 0x00, 0x6f, 0x00, 0x6d, 0x00, 0x61, 0x00,
				0x69, 0x00, 0x6e, 0x00, 0x55, 0x00, 0x73, 0x00,
				0x65, 0x00, 0x72, 0x00, 0x43, 0x00, 0x4f, 0x00,
				0x4d, 0x00, 0x50, 0x00, 0x55, 0x00, 0x54, 0x00,
				0x45, 0x00, 0x52, 0x00, 0xaa, 0xaa, 0xaa, 0xaa,
				0xaa, 0xaa, 0xaa, 0xaa, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x75, 0x37, 0xf8, 0x03,
				0xae, 0x36, 0x71, 0x28, 0xca, 0x45, 0x82, 0x04,
				0xbd, 0xe7, 0xca, 0xf8, 0x1e, 0x97, 0xed, 0x26,
				0x83, 0x26, 0x72, 0x32,
			},
			nil,
		},
		{
			&Client{
				compatibilityLevel: 3,
				domain:             "Domain",
				password:           "Password",
				username:           "User",
				workstation:        "COMPUTER",
				version: &Version{
					ProductMajorVersion: 5,
					ProductMinorVersion: 1,
					ProductBuild:        2600,
					NTLMRevisionCurrent: NTLMSSPRevisionW2K3,
				},
			},
			[]byte{
				0x4e, 0x54, 0x4c, 0x4d, 0x53, 0x53, 0x50, 0x00,
				0x02, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x0c, 0x00,
				0x38, 0x00, 0x00, 0x00, 0x33, 0x82, 0x8a, 0xe2,
				0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x24, 0x00, 0x24, 0x00, 0x44, 0x00, 0x00, 0x00,
				0x06, 0x00, 0x70, 0x17, 0x00, 0x00, 0x00, 0x0f,
				0x53, 0x00, 0x65, 0x00, 0x72, 0x00, 0x76, 0x00,
				0x65, 0x00, 0x72, 0x00, 0x02, 0x00, 0x0c, 0x00,
				0x44, 0x00, 0x6f, 0x00, 0x6d, 0x00, 0x61, 0x00,
				0x69, 0x00, 0x6e, 0x00, 0x01, 0x00, 0x0c, 0x00,
				0x53, 0x00, 0x65, 0x00, 0x72, 0x00, 0x76, 0x00,
				0x65, 0x00, 0x72, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			nil,
			[]byte{
				0x4e, 0x54, 0x4c, 0x4d, 0x53, 0x53, 0x50, 0x00,
				0x03, 0x00, 0x00, 0x00, 0x18, 0x00, 0x18, 0x00,
				0x6c, 0x00, 0x00, 0x00, 0x54, 0x00, 0x54, 0x00,
				0x84, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x0c, 0x00,
				0x48, 0x00, 0x00, 0x00, 0x08, 0x00, 0x08, 0x00,
				0x54, 0x00, 0x00, 0x00, 0x10, 0x00, 0x10, 0x00,
				0x5c, 0x00, 0x00, 0x00, 0x10, 0x00, 0x10, 0x00,
				0xd8, 0x00, 0x00, 0x00, 0x31, 0x82, 0x8a, 0xe2, // Last four bytes originally 0x35, 0x82, 0x88, 0xe2
				0x05, 0x01, 0x28, 0x0a, 0x00, 0x00, 0x00, 0x0f,
				0x44, 0x00, 0x6f, 0x00, 0x6d, 0x00, 0x61, 0x00,
				0x69, 0x00, 0x6e, 0x00, 0x55, 0x00, 0x73, 0x00,
				0x65, 0x00, 0x72, 0x00, 0x43, 0x00, 0x4f, 0x00,
				0x4d, 0x00, 0x50, 0x00, 0x55, 0x00, 0x54, 0x00,
				0x45, 0x00, 0x52, 0x00, 0x86, 0xc3, 0x50, 0x97,
				0xac, 0x9c, 0xec, 0x10, 0x25, 0x54, 0x76, 0x4a,
				0x57, 0xcc, 0xcc, 0x19, 0xaa, 0xaa, 0xaa, 0xaa,
				0xaa, 0xaa, 0xaa, 0xaa, 0x68, 0xcd, 0x0a, 0xb8,
				0x51, 0xe5, 0x1c, 0x96, 0xaa, 0xbc, 0x92, 0x7b,
				0xeb, 0xef, 0x6a, 0x1c, 0x01, 0x01, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0xaa, 0xaa, 0xaa, 0xaa,
				0xaa, 0xaa, 0xaa, 0xaa, 0x00, 0x00, 0x00, 0x00,
				0x02, 0x00, 0x0c, 0x00, 0x44, 0x00, 0x6f, 0x00,
				0x6d, 0x00, 0x61, 0x00, 0x69, 0x00, 0x6e, 0x00,
				0x01, 0x00, 0x0c, 0x00, 0x53, 0x00, 0x65, 0x00,
				0x72, 0x00, 0x76, 0x00, 0x65, 0x00, 0x72, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0xc5, 0xda, 0xd2, 0x54, 0x4f, 0xc9, 0x79, 0x90,
				0x94, 0xce, 0x1c, 0xe9, 0x0b, 0xc9, 0xd0, 0x3e,
			},
			nil,
		},
	}

	// Mock the functions that return values based on either random
	// numbers or the current time so they return fixed values

	oldClientChallenge := generateClientChallenge
	defer func() { generateClientChallenge = oldClientChallenge }()
	generateClientChallenge = func() ([]byte, error) {
		return bytes.Repeat([]byte{0xaa}, 8), nil
	}

	oldExportedSessionKey := generateExportedSessionKey
	defer func() { generateExportedSessionKey = oldExportedSessionKey }()
	generateExportedSessionKey = func() ([]byte, error) {
		return bytes.Repeat([]byte{0x55}, 16), nil
	}

	oldCurrentTime := currentTime
	defer func() { currentTime = oldCurrentTime }()
	currentTime = func() ([]byte, error) {
		return bytes.Repeat([]byte{0x00}, 8), nil
	}

	for _, table := range tables {
		got, err := table.client.processChallengeMessage(table.challengeMessage, table.channelBindings)
		assert.Equal(t, table.err, err)
		if err == nil {
			assert.Equal(t, table.want, got)
		}
	}
}

func TestClient(t *testing.T) {
	tables := []struct {
		domain, username, password, workstation                                  string
		version                                                                  *Version
		negotiate, challenge, fingerprint, authenticate, request, wrap, response []byte
	}{
		{
			"Domain",
			"User",
			"Password",
			"COMPUTER",
			&Version{
				ProductMajorVersion: 5,
				ProductMinorVersion: 1,
				ProductBuild:        2600,
				NTLMRevisionCurrent: NTLMSSPRevisionW2K3,
			},
			[]byte{
				0x4e, 0x54, 0x4c, 0x4d, 0x53, 0x53, 0x50, 0x00,
				0x01, 0x00, 0x00, 0x00, 0x31, 0xb0, 0x88, 0xe2,
				0x06, 0x00, 0x06, 0x00, 0x28, 0x00, 0x00, 0x00,
				0x08, 0x00, 0x08, 0x00, 0x2e, 0x00, 0x00, 0x00,
				0x05, 0x01, 0x28, 0x0a, 0x00, 0x00, 0x00, 0x0f,
				0x44, 0x6f, 0x6d, 0x61, 0x69, 0x6e, 0x43, 0x4f,
				0x4d, 0x50, 0x55, 0x54, 0x45, 0x52,
			},
			[]byte{
				0x4e, 0x54, 0x4c, 0x4d, 0x53, 0x53, 0x50, 0x00,
				0x02, 0x00, 0x00, 0x00, 0x2f, 0x82, 0x88, 0xe2,
				0x38, 0x00, 0x00, 0x00, 0x33, 0x82, 0x8a, 0xe2,
				0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x24, 0x00, 0x24, 0x00, 0x44, 0x00, 0x00, 0x00,
				0x06, 0x00, 0x70, 0x17, 0x00, 0x00, 0x00, 0x0f,
				0x53, 0x00, 0x65, 0x00, 0x72, 0x00, 0x76, 0x00,
				0x65, 0x00, 0x72, 0x00, 0x02, 0x00, 0x0c, 0x00,
				0x44, 0x00, 0x6f, 0x00, 0x6d, 0x00, 0x61, 0x00,
				0x69, 0x00, 0x6e, 0x00, 0x01, 0x00, 0x0c, 0x00,
				0x53, 0x00, 0x65, 0x00, 0x72, 0x00, 0x76, 0x00,
				0x65, 0x00, 0x72, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			[]byte{
				0xe3, 0xca, 0x49, 0x27, 0x1e, 0x50, 0x89, 0xcc,
				0x48, 0xce, 0x82, 0x10, 0x9f, 0x13, 0x24, 0xf4,
				0x1d, 0xbe, 0xdd, 0xc2, 0x9a, 0x77, 0x74, 0x10,
				0xc7, 0x38, 0xf7, 0x86, 0x8c, 0x4f, 0xf4, 0x05,
			},
			[]byte{
				0x4e, 0x54, 0x4c, 0x4d, 0x53, 0x53, 0x50, 0x00,
				0x03, 0x00, 0x00, 0x00, 0x18, 0x00, 0x18, 0x00,
				0x6c, 0x00, 0x00, 0x00, 0x68, 0x00, 0x68, 0x00,
				0x84, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x0c, 0x00,
				0x48, 0x00, 0x00, 0x00, 0x08, 0x00, 0x08, 0x00,
				0x54, 0x00, 0x00, 0x00, 0x10, 0x00, 0x10, 0x00,
				0x5c, 0x00, 0x00, 0x00, 0x10, 0x00, 0x10, 0x00,
				0xec, 0x00, 0x00, 0x00, 0x31, 0x82, 0x8a, 0xe2,
				0x05, 0x01, 0x28, 0x0a, 0x00, 0x00, 0x00, 0x0f,
				0x44, 0x00, 0x6f, 0x00, 0x6d, 0x00, 0x61, 0x00,
				0x69, 0x00, 0x6e, 0x00, 0x55, 0x00, 0x73, 0x00,
				0x65, 0x00, 0x72, 0x00, 0x43, 0x00, 0x4f, 0x00,
				0x4d, 0x00, 0x50, 0x00, 0x55, 0x00, 0x54, 0x00,
				0x45, 0x00, 0x52, 0x00, 0x86, 0xc3, 0x50, 0x97,
				0xac, 0x9c, 0xec, 0x10, 0x25, 0x54, 0x76, 0x4a,
				0x57, 0xcc, 0xcc, 0x19, 0xaa, 0xaa, 0xaa, 0xaa,
				0xaa, 0xaa, 0xaa, 0xaa, 0x04, 0x10, 0xc4, 0x7a,
				0xcf, 0x19, 0x97, 0x89, 0xde, 0x7f, 0x20, 0x11,
				0x95, 0x7a, 0xea, 0x50, 0x01, 0x01, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0xaa, 0xaa, 0xaa, 0xaa,
				0xaa, 0xaa, 0xaa, 0xaa, 0x00, 0x00, 0x00, 0x00,
				0x02, 0x00, 0x0c, 0x00, 0x44, 0x00, 0x6f, 0x00,
				0x6d, 0x00, 0x61, 0x00, 0x69, 0x00, 0x6e, 0x00,
				0x01, 0x00, 0x0c, 0x00, 0x53, 0x00, 0x65, 0x00,
				0x72, 0x00, 0x76, 0x00, 0x65, 0x00, 0x72, 0x00,
				0x0a, 0x00, 0x10, 0x00, 0x6e, 0xa1, 0x9d, 0xf0,
				0x66, 0xda, 0x46, 0x22, 0x05, 0x1f, 0x9c, 0x4f,
				0x92, 0xc6, 0xdf, 0x74, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0xe5, 0x69, 0x95, 0x1d,
				0x15, 0xd4, 0x73, 0x5f, 0x49, 0xe1, 0x4c, 0xf9,
				0xa7, 0xd3, 0xe6, 0x72,
			},
			[]byte("test req"),
			[]byte{
				0x01, 0x00, 0x00, 0x00, 0xbc, 0xe3, 0x23, 0xa1,
				0x72, 0x06, 0x23, 0x78, 0x00, 0x00, 0x00, 0x00,
				0x70, 0x80, 0x1e, 0x11, 0xfe, 0x6b, 0x3a, 0xad,
			},
			[]byte("test res"),
		},
	}

	// Mock the functions that return values based on either random
	// numbers or the current time so they return fixed values

	oldClientChallenge := generateClientChallenge
	defer func() { generateClientChallenge = oldClientChallenge }()
	generateClientChallenge = func() ([]byte, error) {
		return bytes.Repeat([]byte{0xaa}, 8), nil
	}

	oldExportedSessionKey := generateExportedSessionKey
	defer func() { generateExportedSessionKey = oldExportedSessionKey }()
	generateExportedSessionKey = func() ([]byte, error) {
		return bytes.Repeat([]byte{0x55}, 16), nil
	}

	oldCurrentTime := currentTime
	defer func() { currentTime = oldCurrentTime }()
	currentTime = func() ([]byte, error) {
		return bytes.Repeat([]byte{0x00}, 8), nil
	}

	for _, table := range tables {
		client, err := NewClient(SetDomain(table.domain), SetUserInfo(table.username, table.password), SetWorkstation(table.workstation), SetVersion(table.version))
		assert.Nil(t, err)

		negotiate, err := client.Authenticate(nil, nil)
		assert.Nil(t, err)
		assert.Equal(t, table.negotiate, negotiate)

		cbt := &ChannelBindings{
			ApplicationData: concat([]byte(TLSServerEndPoint+":"), table.fingerprint),
		}

		authenticate, err := client.Authenticate(table.challenge, cbt)
		assert.Nil(t, err)
		assert.Equal(t, table.authenticate, authenticate)

		assert.Equal(t, true, client.Complete())

		clientSession := client.SecuritySession()
		assert.NotNil(t, clientSession)

		message, signature, err := clientSession.Wrap(table.request)
		assert.Nil(t, err)

		assert.Equal(t, table.wrap, concat(signature, message))

		serverSession, err := newSecuritySession(client.negotiatedFlags, bytes.Repeat([]byte{0x55}, 16), sourceServer)
		assert.Nil(t, err)

		unwrapped, err := serverSession.Unwrap(message, signature)
		assert.Nil(t, err)

		assert.Equal(t, table.request, unwrapped)

		message, signature, err = serverSession.Wrap(table.response)
		assert.Nil(t, err)

		unwrapped, err = clientSession.Unwrap(message, signature)
		assert.Nil(t, err)
		assert.Equal(t, table.response, unwrapped)
	}
}
