using System.Collections;
using System.CommandLine;
using System.Reflection;
using System.Text;
using Microsoft.CodeAnalysis;
using Microsoft.CodeAnalysis.CSharp;
using Microsoft.CodeAnalysis.CSharp.Syntax;
using Microsoft.Dafny;
using Scripts;
using NullabilityInfo = System.Reflection.NullabilityInfo;
using Type = System.Type;

namespace IntegrationTests;

public class SyntaxDeserializerGenerator : SyntaxAstVisitor {

  private readonly HashSet<Type> typesWithHardcodedDeserializer = [
    typeof(Token),
    typeof(Specification<>),
    typeof(CasePattern<>),
    typeof(MultiSelectExpr),
    typeof(AllocateArray),
    typeof(Function)
  ];

  private ClassDeclarationSyntax deserializeClass = (ClassDeclarationSyntax)SyntaxFactory.ParseMemberDeclaration(@"
partial class SyntaxDeserializer {}")!;
  private readonly List<StatementSyntax> deserializeObjectCases = new();
  protected static Dictionary<Type, IDictionary<string, int>> ParameterToSchemaPositions = new();

  public static Command GetCommand() {
    var result = new Command("generate-syntax-deserializer", "");
    var fileArgument = new Argument<FileInfo>();
    result.AddArgument(fileArgument);
    result.SetHandler((outputFile) => Handle(outputFile.FullName), fileArgument);
    return result;
  }

  public static async Task Handle(string outputFile) {
    var program = typeof(TopLevelDecl);
    var generator = new SyntaxDeserializerGenerator();
    generator.VisitTypesFromRoots([program, typeof(SourceOrigin), typeof(TokenRangeOrigin)]);

    var deserializerUnit = SyntaxFactory.ParseCompilationUnit(@"
// <auto-generated/>
using System;
using System.Collections.Generic;
using Microsoft.Boogie;
using BinaryExprOpcode = Microsoft.Dafny.BinaryExpr.Opcode;
");
    var ns = SyntaxFactory.NamespaceDeclaration(SyntaxFactory.ParseName("Microsoft.Dafny"));
    var deserializeObjectSyntax = (MethodDeclarationSyntax)SyntaxFactory
      .ParseMemberDeclaration("private object ReadObject(System.Type actualType) {}")!;
    deserializeObjectSyntax = deserializeObjectSyntax
      .AddBodyStatements(generator.deserializeObjectCases.ToArray())
      .AddBodyStatements(SyntaxFactory.ParseStatement("throw new Exception($\"Cannot read object of type {actualType.FullName}\");"));

    generator.deserializeClass = generator.deserializeClass.AddMembers(deserializeObjectSyntax);
    ns = ns.AddMembers(generator.deserializeClass);

    deserializerUnit = deserializerUnit.WithMembers(deserializerUnit.Members.Add(ns));
    var contents = deserializerUnit.NormalizeWhitespace(eol: "\n").ToFullString();
    await File.WriteAllTextAsync(outputFile, contents);
  }

  protected override void HandleClass(Type type) {
    var ownedFieldPosition = 0;
    var baseType = GetBaseType(type);
    if (baseType != null && baseType != typeof(ValueType) && baseType != typeof(object)) {
      ownedFieldPosition = ParameterToSchemaPositions[baseType].Count;
    }

    var parameterToSchemaPosition = new Dictionary<string, int>();
    var schemaToConstructorPosition = new Dictionary<int, int>();
    ParameterToSchemaPositions[type] = parameterToSchemaPosition;
    var statements = new StringBuilder();

    VisitParameters(type, (index, parameter) => {
      var parameterType = parameter.ParameterType;
      if (ExcludedTypes.Contains(parameterType)) {
        statements.AppendLine($"{parameterType} parameter{index} = null;");
        return;
      }

      if (parameter.GetCustomAttribute<BackEdge>() != null) {
        parameterToSchemaPosition[parameter.Name!] = -1;
        schemaToConstructorPosition[-1] = index;
        return;
      }

      var memberBelongsToBase = DoesMemberBelongToBase(type, parameter, baseType);
      if (memberBelongsToBase) {
        if (!ParameterToSchemaPositions[baseType!]
              .TryGetValue(parameter.Name!, out var schemaPosition)) {
          throw new Exception(
            $"parameter '{parameter.Name}' of '{type.Name}' should have been in parent type '{baseType}' constructor, but was not found");
        }

        schemaToConstructorPosition[schemaPosition] = index;
        parameterToSchemaPosition[parameter.Name!] = schemaPosition;
        return;
      }

      var schemaPosition2 = ownedFieldPosition++;
      parameterToSchemaPosition[parameter.Name!] = schemaPosition2;
      schemaToConstructorPosition[schemaPosition2] = index;
    });

    if (baseType != null && baseType != typeof(ValueType) && baseType != typeof(object)) {
      foreach (var (baseParamName, baseSchemaIndex) in ParameterToSchemaPositions[baseType]) {
        if (!schemaToConstructorPosition.ContainsKey(baseSchemaIndex)) {
          throw new Exception($"Constructor for type {type.Name} is missing a parameter for field/property {baseParamName} inherited from {baseType.Name} - add one or use {nameof(SyntaxBaseType)}");
        }
      }
    }
    GenerateReadMethod(type, schemaToConstructorPosition, statements);
  }

  protected override void HandleEnum(Type type) {
    var typeString = ToGenericTypeString(type, nestedDot: true);
    var typeString2 = ToGenericTypeString(type, nestedDot: false);
    var deserializer = SyntaxFactory.ParseMemberDeclaration($@"
private {typeString} Read{typeString2}() {{
  int ordinal = ReadInt32();
  return ({typeString})ordinal;
}}")!;
    deserializeClass = deserializeClass.WithMembers(deserializeClass.Members.Add(deserializer));
  }

  private void GenerateReadMethod(Type type, IDictionary<int, int> schemaToConstructorPosition,
    StringBuilder statements) {
    if (type.IsAbstract) {
      return;
    }

    var constructor = GetParseConstructor(type);
    if (constructor == null) {
      return;
    }
    var parameters = constructor.GetParameters();

    var typeString = ToGenericTypeString(type);
    var readMethodName = $"Read{typeString}";
    var readOptionMethodName = $"Read{ToGenericTypeString(type, suffix: "Option")}";

    if (typesWithHardcodedDeserializer.Contains(type.WithoutGenericArguments())) {
      return;
    }

    // If some fields should not be serialized, the keys of schemaToConstructorPosition may not be contiguous,
    // so we iterate in ascending order rather than by indexing from 0 to its size.
    foreach (var (schemaPosition, constructorIndex) in schemaToConstructorPosition.OrderBy(p => p.Key)) {
      var parameter = parameters[constructorIndex];
      var nullabilityContext = new NullabilityInfoContext();
      var nullabilityInfo = nullabilityContext.Create(parameter);
      var parameterTypeReadCall = GetReadTypeCall(parameter.ParameterType, nullabilityInfo);
      if (schemaPosition == -1) {
        statements.AppendLine($"{parameter.ParameterType} parameter{constructorIndex} = null;");
      } else {
        statements.AppendLine(
          $"var parameter{constructorIndex} = {parameterTypeReadCall};");
      }
    }
    AddReadMethodForType(parameters, statements, typeString, readMethodName);
    AddReadOptionMethodForType(typeString, readMethodName, readOptionMethodName);
    deserializeObjectCases.Add(SyntaxFactory.ParseStatement($@"
if (actualType == typeof({typeString})) {{
  return {readMethodName}();
}}
"));

  }

  private string GetReadTypeCall(Type parameterType, NullabilityInfo nullability) {
    string parameterTypeReadCall;
    var newType = MappedTypes.GetValueOrDefault(parameterType, parameterType);
    var optionString = nullability.ReadState == NullabilityState.Nullable ? "Option" : "";
    if (newType.IsArray) {
      var elementType = newType.GetGenericArguments()[0];
      var elementRead = GetReadTypeCall(elementType, nullability.ElementType!);
      var elementTypeString = ToGenericTypeString(elementType, false, false);
      return $"ReadArray{optionString}<{elementTypeString}>(() => {elementRead})";
    }

    if (newType.IsGenericType && newType.IsAssignableTo(typeof(IEnumerable))) {
      var elementType = newType.GetGenericArguments()[0];
      var elementRead = GetReadTypeCall(elementType, nullability.GenericTypeArguments[0]);
      var elementTypeString = ToGenericTypeString(elementType, false, false);
      return $"ReadList{optionString}<{elementTypeString}>(() => {elementRead})";
    }

    if (newType.IsAbstract || newType == typeof(object)) {
      var abstractTypeString = ToGenericTypeString(parameterType, true, false);
      parameterTypeReadCall = $"ReadAbstract{optionString}<{abstractTypeString}>()";
    } else {
      var objectTypeString = ToGenericTypeString(parameterType, true, false, suffix: optionString);
      parameterTypeReadCall = $"Read{objectTypeString}()";
    }

    return parameterTypeReadCall;
  }

  private void AddReadOptionMethodForType(string typeString, string readMethodName, string readOptionMethodName) {
    var typedDeserialize = SyntaxFactory.ParseMemberDeclaration(@$"
 public {typeString} {readOptionMethodName}() {{
  if (ReadIsNull()) {{
     return default;
  }}
  return {readMethodName}();
}}")!;
    deserializeClass = deserializeClass.WithMembers(deserializeClass.Members.Add(typedDeserialize));
  }

  private void AddReadMethodForType(ParameterInfo[] parameters, StringBuilder statements, string typeString,
    string methodName) {
    var parametersString = string.Join(", ", Enumerable.Range(0, parameters.Length).Select(index =>
      $"parameter{index}"));
    var typedDeserialize = SyntaxFactory.ParseMemberDeclaration(@$"
 public {typeString} {methodName}() {{
  {statements}
  return new {typeString}({parametersString});
}}")!;
    deserializeClass = deserializeClass.WithMembers(deserializeClass.Members.Add(typedDeserialize));
  }
}