//
// Syd: rock-solid application kernel
// benches/sandbox/from_str.rs: Sandbox FromStr microbenchmarks
//
// Copyright (c) 2025 Ali Polatel <alip@chesswob.org>
//
// SPDX-License-Identifier: GPL-3.0

use std::{hint::black_box, str::FromStr};

use brunch::{benches, Bench};
use syd::sandbox::{Action, Capability};

const ACTION_LOOP: u64 = 1 << 22;
const CAP_LOOP: u64 = 1 << 16;

// List of valid actions.
const ACTION_STR: &[&str] = &[
    "abort", "allow", "deny", "exit", "filter", "kill", "panic", "stop", "warn",
];

// List of valid capabilities.
const CAP_STR: &[&str] = &[
    "all", // alias for CAP_GLOB.
    "chattr",
    "chdir",
    "chgrp",
    "chmod",
    "chown",
    "chroot",
    "create",
    "crypt",
    "delete",
    "exec",
    "force",
    "ioctl",
    "lock",
    "mem",
    "mkdev",
    "mkdir",
    "mkfifo",
    "mktemp",
    "net", // alias for CAP_NET.
    "net/bind",
    "net/connect",
    "net/sendfd",
    "pid",
    "proxy",
    "read",
    "readdir",
    "rename",
    "rmdir",
    "stat",
    "symlink",
    "tpe",
    "truncate",
    "utime",
    "write",
];

fn bench_action_from_str_valid() {
    for action in ACTION_STR {
        let _ = black_box(Action::from_str(action));
    }
}

fn bench_action_from_str_invalid() {
    let _ = black_box(Action::from_str("foo"));
    let _ = black_box(Action::from_str("bar"));
    let _ = black_box(Action::from_str("baz"));
}

fn bench_action_from_str_long(long: &str) {
    let _ = black_box(Action::from_str(long));
}

fn bench_capability_from_str_valid() {
    for action in CAP_STR {
        let _ = black_box(Capability::from_str(action));
    }
}

fn bench_capability_from_str_invalid() {
    let _ = black_box(Capability::from_str("foo"));
    let _ = black_box(Capability::from_str("bar"));
    let _ = black_box(Capability::from_str("baz"));
}

fn bench_capability_from_str_long(long: &str) {
    let _ = black_box(Capability::from_str(long));
}

fn main() {
    benches!(
        inline:

        Bench::new("syd::sandbox::Capability::from_str.valid").run(|| {
            for _ in 0..CAP_LOOP {
                bench_capability_from_str_valid();
            }
        }),

        Bench::new("syd::sandbox::Capability::from_str.invalid").run(|| {
            for _ in 0..CAP_LOOP {
                bench_capability_from_str_invalid();
            }
        }),

        Bench::new("syd::sandbox::Capability::from_str.long").run_seeded_with(
            || "x".repeat(64_000_000),
            |long| {
                for _ in 0..CAP_LOOP {
                    bench_capability_from_str_long(&long)
                }
        }),

        Bench::new("syd::sandbox::Action::from_str.valid").run(|| {
            for _ in 0..ACTION_LOOP {
                bench_action_from_str_valid();
            }
        }),

        Bench::new("syd::sandbox::Action::from_str.invalid").run(|| {
            for _ in 0..ACTION_LOOP {
                bench_action_from_str_invalid();
            }
        }),

        Bench::new("syd::sandbox::Action::from_str.long").run_seeded_with(
            || "x".repeat(64_000_000),
            |long| {
                for _ in 0..ACTION_LOOP {
                    bench_action_from_str_long(&long)
                }
        }),
    );
}
