<?php

namespace MediaWiki\Session;

use Psr\Log\LoggerAwareInterface;

/**
 * This is a session store abstraction layer, which can be used to read
 * and write sessions to configured backend(s). Backends can be single or
 * multiple, depending on the configuration of the site.
 *
 * Sessions are read or written to the store(s), depending on the session
 * type. The get(), set() and delete() methods are in charge of deciding
 * the relevant store to fetch, write, or delete content from. Sessions
 * can be strongly persistent or weakly persistent based on their type.
 *
 * @anchor SessionStore-storage-expectations
 *
 *  ## Storage expectations
 *
 *  The SessionManager should be configured with a very fast storage system that is
 *  optimized for holding key-value pairs. It expects:
 *
 *  - Low latencies. Session data is read or written to during nearly all web requests from
 *    people who have contributed to or otherwise engaged with the site, including those not
 *    logged in with a registered account.
 *
 *  - Locally writable data. The data must be writable from both primary and secondary
 *    data centres.
 *
 *  - Locally latest reads. Writes must by default be immediately consistent within
 *    the local data centre, and visible to other reads from web servers in that data centre.
 *
 *  - Replication. The data must be eventually consistent across all data centres. Writes
 *    are either synced to all remote data centres, or locally overwritten by another write
 *    that is.
 *
 *  The SessionManager uses `set()` and `delete()` for write operations, which should be
 *  synchronous in the local data centre, and replicate asynchronously to any others.
 *
 *  Anonymous sessions (used mainly for login/signup) can share the same storage as logged-in
 *  sessions, or can use a separate store. On high-traffic wikis, separating the two can help
 *  reduce storage needs, and isolate logged-in users from problems caused by spikes of
 *  anonymous sessions (which can be generated by web crawlers and are easier to DoS).
 *
 *  The loss of anonymous sessions is less disruptive than the loss of authenticated sessions
 *  (will typically only affect users who are right in the middle of login or signup) and so
 *  a store with less redundancy or persistence can be used.
 *
 * @ingroup Session
 * @since 1.45
 */
interface SessionStore extends LoggerAwareInterface {

	/**
	 * Retrieves the session data for a given session ID. Should
	 * return false if the session is not found.
	 *
	 * @param SessionInfo $sessionInfo
	 *
	 * @return mixed
	 */
	public function get( SessionInfo $sessionInfo );

	/**
	 * Write the session data for a given key to the session store.
	 *
	 * @param SessionInfo $sessionInfo
	 * @param mixed $value
	 * @param int $exptime
	 * @param int $flags Should be one of BagOStuff::WRITE_* constants
	 */
	public function set( SessionInfo $sessionInfo, $value, $exptime = 0, $flags = 0 ): void;

	/**
	 * Delete the session data for a given ID from the session store.
	 *
	 * @param SessionInfo $sessionInfo
	 */
	public function delete( SessionInfo $sessionInfo ): void;

	/**
	 * Will be called during shutdown.
	 *
	 * @return void
	 */
	public function shutdown(): void;
}
