/**
 * Implementation of the CVSS v3.0 specification (https://www.first.org/cvss/v3.0/specification-document).
 */
export declare class CVSS30 {
    private _metrics;
    /**
     * Construct a CVSS v3.0 object, and parse the vector if provided.
     * If not, the Base metrics is set to the default values (score = 0).
     *
     * @param vector The vector to parse.
     * @throws When the vector is invalid.
     */
    constructor(vector?: string);
    /**
     * Parse the provided vector.
     *
     * @param vector The vector to parse.
     * @throws When the vector is invalid.
     */
    private parse;
    /**
     * Return the vector string representation of the CVSS v3.0 object.
     *
     * @return The vector string representation.
     */
    Vector(): string;
    /**
     * Get the metric value given its value (e.g. 'AV').
     *
     * @param metric The metric to get the value of.
     * @return The corresponding metric value.
     * @throws Metric does not exist.
     */
    Get(metric: string): string;
    /**
     * Set the metric value given its key and value (e.g. 'AV' and 'L').
     *
     * @param metric The metric to set the value of.
     * @param value The corresponding metric value.
     * @throws Metric does not exist or has an invalid value.
     */
    Set(metric: string, value: string): void;
    /**
     * Compute the CVSS v3.0 Impact of the current object, given its metrics and their
     * corresponding values.
     *
     * The implementation internals are largely based upon https://github.com/pandatix/go-cvss
     * submodule 30.
     *
     * @return The score (between 0.0 and 10.0 both included).
     */
    Impact(): number;
    /**
     * Compute the CVSS v3.0 Exploitability of the current object, given its metrics and their
     * corresponding values.
     *
     * The implementation internals are largely based upon https://github.com/pandatix/go-cvss
     * submodule 30.
     *
     * @return The score (between 0.0 and 10.0 both included).
     */
    Exploitability(): number;
    /**
     * Compute the CVSS v3.0 Base Score of the current object, given its metrics and their
     * corresponding values.
     *
     * The implementation internals are largely based upon https://github.com/pandatix/go-cvss
     * submodule 30.
     *
     * @return The score (between 0.0 and 10.0 both included).
     */
    BaseScore(): number;
    /**
     * Compute the CVSS v3.0 Temporal Score of the current object, given its metrics and their
     * corresponding values.
     *
     * The implementation internals are largely based upon https://github.com/pandatix/go-cvss
     * submodule 30.
     *
     * @return The score (between 0.0 and 10.0 both included).
     */
    TemporalScore(): number;
    /**
     * Compute the CVSS v3.0 Environmental Score of the current object, given its metrics and their
     * corresponding values.
     *
     * The implementation internals are largely based upon https://github.com/pandatix/go-cvss
     * submodule 30.
     *
     * @return The score (between 0.0 and 10.0 both included).
     */
    EnvironmentalScore(): number;
    private getReal;
    private static roundup;
    /**
     * Give the corresponding rating of the provided score.
     *
     * @param score The score to rate.
     * @return The rating.
     * @throws When the score is out of bounds.
     */
    static Rating(score: number): 'CRITICAL' | 'HIGH' | 'MEDIUM' | 'LOW' | 'NONE';
}
