# Licensed to the Software Freedom Conservancy (SFC) under one
# or more contributor license agreements.  See the NOTICE file
# distributed with this work for additional information
# regarding copyright ownership.  The SFC licenses this file
# to you under the Apache License, Version 2.0 (the
# "License"); you may not use this file except in compliance
# with the License.  You may obtain a copy of the License at
#
#   http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing,
# software distributed under the License is distributed on an
# "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
# KIND, either express or implied.  See the License for the
# specific language governing permissions and limitations
# under the License.


"""This script recursively scans the `selenium` package directory and generates an API listing.

Recursively scans the `selenium` package directory to find all modules,
then generates the `py/docs/source/api.rst` file containing a listing of all
modules in separate sections. The `api.rst` file is later used by
`sphinx-autogen` to generate sphinx autodoc stub pages used in the Python API
documentation. See `py/tox.ini` for how it is invoked.
"""

import os
import site


def find_modules(package_name):
    modules = []
    for dirpath, _, filenames in os.walk(package_name):
        for filename in filenames:
            if filename.endswith(".py") and not filename.startswith("__"):
                module_name = (
                    os.path.join(dirpath, filename)
                    .removeprefix(site.getsitepackages()[-1])
                    .removeprefix(os.sep)
                    .removesuffix(".py")
                    .replace(os.sep, ".")
                )
                modules.append(module_name)
    return sorted(set(modules))


if __name__ == "__main__":
    # Support running via bazel run (uses BUILD_WORKSPACE_DIRECTORY)
    workspace = os.environ.get("BUILD_WORKSPACE_DIRECTORY", "")
    base_dir = os.path.join(workspace, "py") if workspace else "."
    os.chdir(base_dir)

    package_name = "selenium"
    output_file = os.path.join("docs", "source", "api.rst")
    print(f"generating module list for sphinx autodoc in: {output_file}\n")
    modules = [module for module in find_modules(package_name) if ".devtools." not in module]
    base_modules = [mod for mod in sorted({module.rsplit(".", 1)[0] for module in modules}) if mod != package_name]
    print("found sections:")
    for base_module in base_modules:
        print(f"    {base_module}")
    with open(output_file, "w") as f:
        f.write(
            """\
..
    this file was auto-generated by `generate_api_module_listing.py`
    DO NOT EDIT

:orphan:

======================
Selenium Documentation
======================
"""
        )
        for base_module in base_modules:
            content_section = base_module.split(".", 1)[1]
            separator = "-" * len(content_section)
            f.write(
                f"""
{content_section}
{separator}

.. currentmodule:: {base_module}
.. autosummary::
   :toctree: {base_module.replace(".", "_")}

"""
            )
            for module in modules:
                if base_module in module:
                    if len(module.split(".")) - len(base_module.split(".")) == 1:
                        f.write(f"   {module}\n")
        f.write(
            """
Indices and tables

* :ref:`genindex`
* :ref:`modindex`
* :ref:`search`
"""
        )
