import { IDisposable } from '@lumino/disposable';
import { Poll } from '@lumino/polling';
import { ISignal } from '@lumino/signaling';
import { Builder, BuildManager } from './builder';
import { NbConvert, NbConvertManager } from './nbconvert';
import { Contents, ContentsManager } from './contents';
import { Event } from './event';
import { KernelSpec, KernelSpecManager } from './kernelspec';
import { Session, SessionManager } from './session';
import { Setting, SettingManager } from './setting';
import { Terminal, TerminalManager } from './terminal';
import { ServerConnection } from './serverconnection';
import { User } from './user';
import { Workspace, WorkspaceManager } from './workspace';
/**
 * A Jupyter services manager.
 */
export declare class ServiceManager implements ServiceManager.IManager {
    /**
     * Construct a new services provider.
     */
    constructor(options?: ServiceManager.IOptions);
    /**
     * A signal emitted when there is a connection failure with the kernel.
     */
    get connectionFailure(): ISignal<this, Error>;
    /**
     * Test whether the service manager is disposed.
     */
    get isDisposed(): boolean;
    /**
     * Dispose of the resources used by the manager.
     */
    dispose(): void;
    /**
     * The server settings of the manager.
     */
    readonly serverSettings: ServerConnection.ISettings;
    /**
     * Get the session manager instance.
     */
    readonly sessions: SessionManager;
    /**
     * Get the session manager instance.
     */
    readonly kernelspecs: KernelSpecManager;
    /**
     * Get the setting manager instance.
     */
    readonly settings: SettingManager;
    /**
     * The builder for the manager.
     */
    readonly builder: BuildManager;
    /**
     * Get the contents manager instance.
     */
    readonly contents: ContentsManager;
    /**
     * The event manager instance.
     */
    readonly events: Event.IManager;
    /**
     * Get the terminal manager instance.
     */
    readonly terminals: TerminalManager;
    /**
     * Get the user manager instance.
     */
    readonly user: User.IManager;
    /**
     * Get the workspace manager instance.
     */
    readonly workspaces: WorkspaceManager;
    /**
     * Get the nbconvert manager instance.
     */
    readonly nbconvert: NbConvertManager;
    /**
     * Test whether the manager is ready.
     */
    get isReady(): boolean;
    /**
     * A promise that fulfills when the manager is ready.
     */
    get ready(): Promise<void>;
    private _onConnectionFailure;
    private _isDisposed;
    private _readyPromise;
    private _connectionFailure;
    private _isReady;
}
/**
 * The namespace for `ServiceManager` statics.
 */
export declare namespace ServiceManager {
    /**
     * A service manager interface.
     */
    interface IManager extends IDisposable {
        /**
         * The builder for the manager.
         */
        readonly builder: Builder.IManager;
        /**
         * The contents manager for the manager.
         */
        readonly contents: Contents.IManager;
        /**
         * The events service manager.
         *
         * #### Notes
         * The events manager is optional until JupyterLab 4.
         */
        readonly events?: Event.IManager;
        /**
         * Test whether the manager is ready.
         */
        readonly isReady: boolean;
        /**
         * A promise that fulfills when the manager is initially ready.
         */
        readonly ready: Promise<void>;
        /**
         * The server settings of the manager.
         */
        readonly serverSettings: ServerConnection.ISettings;
        /**
         * The session manager for the manager.
         */
        readonly sessions: Session.IManager;
        /**
         * The session manager for the manager.
         */
        readonly kernelspecs: KernelSpec.IManager;
        /**
         * The setting manager for the manager.
         */
        readonly settings: Setting.IManager;
        /**
         * The terminals manager for the manager.
         */
        readonly terminals: Terminal.IManager;
        /**
         * The user manager for the manager.
         */
        readonly user?: User.IManager;
        /**
         * The workspace manager for the manager.
         */
        readonly workspaces: Workspace.IManager;
        /**
         * The nbconvert manager for the manager.
         */
        readonly nbconvert: NbConvert.IManager;
        /**
         * A signal emitted when there is a connection failure with the server.
         */
        readonly connectionFailure: ISignal<IManager, Error>;
    }
    /**
     * The options used to create a service manager.
     */
    interface IOptions {
        /**
         * The server settings of the manager.
         */
        readonly serverSettings?: ServerConnection.ISettings;
        /**
         * The default drive for the contents manager.
         */
        readonly defaultDrive?: Contents.IDrive;
        /**
         * When the manager stops polling the API. Defaults to `when-hidden`.
         */
        standby?: Poll.Standby;
    }
}
