// -*- C++ -*-
#include "Rivet/Analysis.hh"
#include "Rivet/Projections/FinalState.hh"
#include "Rivet/Projections/UnstableParticles.hh"
#include "Rivet/Projections/Beam.hh"

namespace Rivet {


  /// @brief gamma, pi0 and eta spectra at 35 and 44 GeV
  class JADE_1990_I282847 : public Analysis {
  public:

    /// Constructor
    RIVET_DEFAULT_ANALYSIS_CTOR(JADE_1990_I282847);


    /// @name Analysis methods
    /// @{

    /// Book histograms and initialise projections before the run
    void init() {

      declare(Beam(), "Beams");
      declare(FinalState(), "FS");
      declare(UnstableParticles(), "UFS");

      // Book histograms
      size_t ih = 0;
      for (double eVal : allowedEnergies()) {
        const string en = toString(round(eVal));
        if (isCompatibleWithSqrtS(eVal))  _sqs = en;
        book(_h[en+"gamma"], 1+ih, 1, 1);
        book(_h[en+"pi0"],   3+ih, 1, 1);
        if (ih == 0)  book(_h[en+"eta"], 5, 1, 1);
        ++ih;
      }
      raiseBeamErrorIf(_sqs.empty());
    }


    /// Perform the per-event analysis
    void analyze(const Event& event) {
      // Get beams and average beam momentum
      const ParticlePair& beams = apply<Beam>(event, "Beams").beams();
      const double meanBeamMom = 0.5*(beams.first.p3().mod() + beams.second.p3().mod());
      // gamma
      const FinalState& fs = apply<FinalState>(event, "FS");
      for (const Particle& p : fs.particles(Cuts::pid==22)) {
      	_h[_sqs+"gamma"]->fill(p.E()/meanBeamMom);
      }
      // pi0, eta
      const UnstableParticles& ufs = apply<UnstableParticles>(event, "UFS");
      for (const Particle& p : ufs.particles(Cuts::pid==111 || Cuts::pid==221)) {
      	const double modp = p.p3().mod();
      	const double xE = p.E()/meanBeamMom;
      	const double beta = modp/p.E();
      	if (p.pid()==111) {
      	  _h[_sqs+"pi0"]->fill(xE, 1.0/beta);
        }
      	else if (_sqs == "35"s) {
      	  _h[_sqs+"eta"]->fill(xE, 1.0/beta);
        }
      }
    }


    /// Normalise histograms etc., after the run
    void finalize() {
      for (auto& item : _h) {
        const double en = stod(item.first.substr(0,2));
        scale(item.second, crossSection()*sqr(en)/microbarn/sumOfWeights());
      }
    }

    /// @}

    /// @name Histograms
    /// @{
    map<string,Histo1DPtr> _h;
    string _sqs = "";
    /// @}

  };


  RIVET_DECLARE_PLUGIN(JADE_1990_I282847);
}
