// -*- C++ -*-
#include "Rivet/Analysis.hh"
#include "Rivet/Projections/UnstableParticles.hh"

namespace Rivet {


  /// @brief Upsilon production at 7,8 TeV
  class LHCB_2015_I1392456 : public Analysis {
  public:

    /// Constructor
    RIVET_DEFAULT_ANALYSIS_CTOR(LHCB_2015_I1392456);


    /// @name Analysis methods
    /// @{

    /// Book histograms and initialise projections before the run
    void init() {
      declare(UnstableParticles(), "UFS");

      for (double eVal : allowedEnergies()) {
        const int en = int(eVal+0.5);
        if (isCompatibleWithSqrtS(eVal))  _sqs = en;

        int ih = en==7000? 0 : 10;
        for (size_t iups=0; iups<3; ++iups) {
          book(_h_Ups[en+iups], {2.0,2.5,3.0,3.5,4.0,4.5});
          for (size_t iy=0; iy<5; ++iy) {
              book(_h_Ups[en+iups]->bin(iy+1), 1+iups+ih, 1, 1+iy);
          }
          book(_h_Ups_pT[en+iups], 4+ih, 1, 1+iups);
          book(_h_Ups_y [en+iups], 5+ih, 1, 1+iups);
        }
      }
      raiseBeamErrorIf(_sqs==0);
    }


    /// Perform the per-event analysis
    void analyze(const Event& event) {
      // Final state of unstable particles to get particle spectra
      const UnstableParticles& ufs = apply<UnstableParticles>(event, "UFS");
      for (const Particle& p : ufs.particles(Cuts::pid==553 or Cuts::pid==100553 or Cuts::pid==200553)) {
        const double absrap = p.absrap();
        const double xp = p.perp();
        if (absrap<2.0 || absrap>4.5) continue;
        unsigned int iups=p.pid()/100000;
        _h_Ups_pT[_sqs+iups]->fill(xp);
        if (xp<30.) _h_Ups_y[_sqs+iups]->fill(absrap);
        _h_Ups[_sqs+iups]->fill(absrap,xp);
      }
    }


    /// Normalise histograms etc., after the run
    void finalize() {
      // 1/2 due rapidity folding +/-
      const double factor = 0.5*crossSection()/picobarn/sumOfWeights();
      // branching ratios for upsilon
      vector<double> br = {0.0248,0.0193,0.0218};

      for (double eVal : allowedEnergies()) {
        const int en = int(eVal+0.5);
        for (size_t iups=0; iups<3; ++iups) {
          scale(_h_Ups_pT[en+iups], factor*br[iups]);
          scale(_h_Ups_y[en+iups], factor/1000.*br[iups]);
          scale(_h_Ups[en+iups], factor*br[iups]);
          divByGroupWidth(_h_Ups[en+iups]);
        }
        size_t ih = en == 8000? 10 : 0;
        for (size_t ix=0; ix<_h_Ups[en+0]->numBins(); ++ix) {
          Estimate1DPtr tmp;
          book(tmp, 6+ih, 1, 1+ix);
          divide(_h_Ups[en+1]->bin(ix+1), _h_Ups[en+0]->bin(ix+1), tmp);
          book(tmp, 7+ih, 1, 1+ix);
          divide(_h_Ups[en+2]->bin(ix+1), _h_Ups[en+0]->bin(ix+1), tmp);
          book(tmp, 8+ih, 1, 1+ix);
          divide(_h_Ups[en+2]->bin(ix+1), _h_Ups[en+1]->bin(ix+1), tmp);
        }
        for (size_t iups=1; iups<3; ++iups) {
          Estimate1DPtr tmp;
          book(tmp, 9+ih, 1, iups);
          divide(_h_Ups_pT[en+iups], _h_Ups_pT[en+0], tmp);
          book(tmp, 10+ih,1,iups);
          divide(_h_Ups_y[en+iups], _h_Ups_y[en+0], tmp);
        }
        Estimate1DPtr tmp;
        book(tmp, 9+ih, 1, 3);
        divide(_h_Ups_pT[en+2], _h_Ups_pT[en+1], tmp);
        book(tmp, 10+ih, 1, 3);
        divide(_h_Ups_y[en+2], _h_Ups_y[en+1], tmp);
      }
    }

    /// @}


    /// @name Histograms
    /// @{
    map<int,Histo1DPtr> _h_Ups_pT, _h_Ups_y;
    map<int,Histo1DGroupPtr> _h_Ups;
    int _sqs = 0;
    /// @}


  };


  RIVET_DECLARE_PLUGIN(LHCB_2015_I1392456);

}
