// -*- C++ -*-
#include "Rivet/Analysis.hh"
#include "Rivet/Projections/LeptonFinder.hh"
#include "Rivet/Projections/DileptonFinder.hh"

namespace Rivet {


  /// @brief Forward W boson production in pp collisions at 7 TeV
  class LHCB_2014_I1311488 : public Analysis {
  public:

    /// Constructor
    RIVET_DEFAULT_ANALYSIS_CTOR(LHCB_2014_I1311488);


    /// @name Analysis methods
    ///@{

    /// Book histograms and initialise projections before the run
    void init() {

      // Initialise and register projections
      Cut cuts = (Cuts::pT >= 20.0*GeV && Cuts::absetaIn(2.0, 4.5) && Cuts::abspid == PID::MUON);
      LeptonFinder wFinder(cuts, 0.0);
      LeptonFinder wFinder_Fsr(cuts, 0.1);
      declare(wFinder, "LeptonFinder");
      declare(wFinder_Fsr, "LeptonFinder_Fsr");

      // Book histograms
      book(_h_wPlusProdEta, 2, 1, 1);
      book(_h_wMinusProdEta, 2, 1, 2);
      book(_h_wPlusProdEta_Fsr, 2, 1, 3);
      book(_h_wMinusProdEta_Fsr, 2, 1, 4);
      book(_h_wRatioEta, 3, 1, 1);
      book(_h_wChargeAsymEta, 4, 1, 1);
      book(_h_wRatioEta_Fsr, 3, 1, 2);
      book(_h_wChargeAsymEta_Fsr, 4, 1, 2);
    }


    /// Perform the per-event analysis
    void analyze(const Event& event) {

      // Searching for W bosons
      const DressedLeptons w = apply<LeptonFinder>(event, "LeptonFinder").dressedLeptons();
      if (w.size() > 0){
        const Particle& mu = w[0];
        (mu.charge3() > 0 ? _h_wPlusProdEta : _h_wMinusProdEta)->fill(mu.abseta());
      }

    // Searching for W bosons + FSR photon
      const DressedLeptons w_Fsr = apply<LeptonFinder>(event, "LeptonFinder_Fsr").dressedLeptons();
      if (w_Fsr.size() > 0) {
        const Particle& mu = w_Fsr[0];
        (mu.charge3() > 0 ? _h_wPlusProdEta_Fsr : _h_wMinusProdEta_Fsr)->fill(mu.abseta());
      }
    }


    /// Normalise histograms etc., after the run
    void finalize() {

      // Scaling of the histograms and correction on FSR
      const double xs = crossSection()/picobarn;
      const double sf = xs/sumOfWeights()/2.0;

      scale(_h_wPlusProdEta, sf);
      scale(_h_wPlusProdEta_Fsr, sf);
      scale(_h_wMinusProdEta, sf);
      scale(_h_wMinusProdEta_Fsr, sf);

      //Calculation of ratio and charge asymmetry
      divide(_h_wPlusProdEta, _h_wMinusProdEta, _h_wRatioEta);
      divide(_h_wPlusProdEta_Fsr, _h_wMinusProdEta_Fsr, _h_wRatioEta_Fsr);
      asymm(_h_wPlusProdEta, _h_wMinusProdEta, _h_wChargeAsymEta);
      asymm(_h_wPlusProdEta_Fsr, _h_wPlusProdEta_Fsr, _h_wChargeAsymEta_Fsr);
      scale(_h_wChargeAsymEta, 100);
      scale(_h_wChargeAsymEta_Fsr, 100);
    }

    ///@}


    /// @name Histograms
    ///@{
    Histo1DPtr _h_wPlusProdEta;
    Histo1DPtr _h_wMinusProdEta;
    Histo1DPtr _h_wPlusProdEta_Fsr;
    Histo1DPtr _h_wMinusProdEta_Fsr;
    Estimate1DPtr _h_wRatioEta;
    Estimate1DPtr _h_wChargeAsymEta;
    Estimate1DPtr _h_wRatioEta_Fsr;
    Estimate1DPtr _h_wChargeAsymEta_Fsr;
    ///@}


  };


  RIVET_DECLARE_PLUGIN(LHCB_2014_I1311488);

}
