// -*- C++ -*-
#include "Rivet/Analysis.hh"
#include "Rivet/Projections/FinalState.hh"

namespace Rivet {


  /// @brief R measurement
  class LENA_1982_I179431 : public Analysis {
  public:

    /// Constructor
    RIVET_DEFAULT_ANALYSIS_CTOR(LENA_1982_I179431);


    /// @name Analysis methods
    /// @{

    /// Book histograms and initialise projections before the run
    void init() {
      // Initialise and register projections
      declare(FinalState(), "FS");

      // Book histograms
      for (size_t i=0; i<2; ++i) {
        const auto& ref = refData(i+1, 1, 1);
        book(_h_hadrons[i], "d0"+to_string(i+1)+"_sigma_hadrons", ref);
        book(_h_muons[i], "d0"+to_string(i+1)+"_sigma_muons",     ref);
      }
      const auto& ref = refData<YODA::BinnedEstimate<string>>(3,1,1);
      book(_c_hadrons, "d03_sigma_hadrons", ref);
      book(_c_muons,   "d03_sigma_muons",   ref);
      for (const string& en : _c_hadrons.binning().edges<0>()) {
        double eval = stod(en);
        if (isCompatibleWithSqrtS(eval)) {
          _sqs = en; break;
        }
      }
      raiseBeamErrorIf(_sqs.empty() && !inRange(sqrtS()/GeV, 7.4, 9.43));
    }


    /// Perform the per-event analysis
    void analyze(const Event& event) {
      const FinalState& fs = apply<FinalState>(event, "FS");

      map<long,int> nCount;
      int ntotal(0);
      for (const Particle& p : fs.particles()) {
        nCount[p.pid()] += 1;
        ++ntotal;
      }
      // mu+mu- + photons
      if (nCount[-13]==1 and nCount[13]==1 && ntotal==2+nCount[22]) {
        _h_muons[0]->fill(sqrtS()/GeV);
        _h_muons[1]->fill(sqrtS()/GeV);
        _c_muons->fill(_sqs);
      }
      // everything else
      else {
        _h_hadrons[0]->fill(sqrtS()/GeV);
        _h_hadrons[1]->fill(sqrtS()/GeV);
        _c_hadrons->fill(_sqs);
      }
    }


    /// Normalise histograms etc., after the run
    void finalize() {
      scale({_c_hadrons, _c_muons}, crossSectionPerEvent());
      scale(_h_hadrons, crossSectionPerEvent());
      scale(_h_muons, crossSectionPerEvent());
      for (size_t ix=0; ix<2; ++ix) {
        Estimate1DPtr mult;
        book(mult, ix+1, 1, 1);
        divide(_h_hadrons[ix], _h_muons[ix], mult);
      }
      BinnedEstimatePtr<string> mult;
      book(mult, 3, 1, 1);
      divide(_c_hadrons,_c_muons,mult);
    }

    /// @}


    /// @name Histograms
    /// @{
    BinnedHistoPtr<string> _c_hadrons, _c_muons;
    Histo1DPtr _h_hadrons[2], _h_muons[2];
    string _sqs = "";
    /// @}


  };


  RIVET_DECLARE_PLUGIN(LENA_1982_I179431);


}
