/*
** Copyright (C) 1998 Steve Borho <steve@borho.org>
**  
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
** 
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
** 
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software 
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/


#include "mount.h"
#include <errno.h>

static char    *getFileName();
static int      createDefaultPropertiesFile(allInfo *ai);
static int      makeRecursivePath(char *path);

bool            loadPropString(
        WMPropList *p,
        char *key,
        char *valbuf,
        size_t valbuf_size)
{
    WMPropList *keyProp, *valProp;
    bool string_found = FALSE;

    keyProp = WMCreatePLString(key);
    if (keyProp)
    {
        valProp = WMGetFromPLDictionary(p, keyProp);
        if (valProp && WMIsPLString(valProp))
        {
            safeCopy(valbuf, WMGetFromPLString(valProp), valbuf_size);
            string_found = TRUE;
        }

        WMReleasePropList(keyProp);
    }

    return string_found;
}

void            readProperties(allInfo *ai)
{
    char        buf[256];
    int         cur_mp;
    WMPropList  *props;
    struct MPInfo  *mp;

    props = WMReadPropListFromFile(getFileName());
    if (props == NULL)
    {
        if (createDefaultPropertiesFile(ai))
        {
            props = WMReadPropListFromFile(getFileName());
            firstTimeUser = TRUE;
        }
        else
            return;
    }

    loadPropString(props, "txtfont", txtfont, STR);
    loadPropString(props, "txtcolor", txtcolor, STR);
    loadPropString(props, "capcolor", capcolor, STR);
    loadPropString(props, "capfont", capfont, STR);
    loadPropString(props, "mountcmd", mountcmd, STR);
    loadPropString(props, "umountcmd", umountcmd, STR);
    loadPropString(props, "ejectcmd", ejectcmd, STR);
    loadPropString(props, "opencmd", opencmd, STR);
    loadPropString(props, "initial_mount", initialMnt, STR);

    loadPropString(props, "icondir", icondir, STR);

    if (loadPropString(props, "cdicon", buf, sizeof(buf)))
        setIcon(ICO_CD, buf, ai);
    if (loadPropString(props, "flicon", buf, sizeof(buf)))
        setIcon(ICO_FLOPPY, buf, ai);
    if (loadPropString(props, "zipicon", buf, sizeof(buf)))
        setIcon(ICO_ZIP, buf, ai);
    if (loadPropString(props, "hdicon", buf, sizeof(buf)))
        setIcon(ICO_HD, buf, ai);
    if (loadPropString(props, "dosicon", buf, sizeof(buf)))
        setIcon(ICO_DOS, buf, ai);
    if (loadPropString(props, "macicon", buf, sizeof(buf)))
        setIcon(ICO_MAC, buf, ai);
    if (loadPropString(props, "unkicon", buf, sizeof(buf)))
        setIcon(ICO_OTHER, buf, ai);
    if (loadPropString(props, "usr1icon", buf, sizeof(buf)))
        setIcon(ICO_USR1, buf, ai);
    if (loadPropString(props, "usr2icon", buf, sizeof(buf)))
        setIcon(ICO_USR2, buf, ai);

    if (loadPropString(props, "idleCycle", buf, sizeof(buf)))
    {
        if (!strcasecmp(buf, "yes"))
            ai->idleCycle = TRUE;
        else
            ai->idleCycle = FALSE;
    }

    if (loadPropString(props, "errorChars", buf, sizeof(buf)))
    {
        if (!strcasecmp(buf, "yes"))
            ai->showErrorChars = TRUE;
        else
            ai->showErrorChars = FALSE;
    }

    if (loadPropString(props, "newbie", buf, sizeof(buf)))
    {
        if (!strcasecmp(buf, "yes"))
            ai->newbieMessages = TRUE;
        else
            ai->newbieMessages = FALSE;
    }

    if (loadPropString(props, "descriptive", buf, sizeof(buf)))
    {
        if (!strcasecmp(buf, "yes"))
            ai->descriptChar = TRUE;
        else
            ai->descriptChar = FALSE;
    }

    for (cur_mp = 0; cur_mp < ai->numMounts; cur_mp++)
    {
        WMPropList *nameProp, *mpProp;

        mp = ai->mpi[cur_mp];
        sprintf(buf, "mountpoint.%s", mp->mountStr);

        nameProp = WMCreatePLString(buf);
        if (nameProp == NULL)
            continue;

        mpProp = WMGetFromPLDictionary(props, nameProp);
        if (mpProp && WMIsPLDictionary(mpProp))
        {
            loadPropString(mpProp, "mountname", mp->name, NAME_LEN);
            if (loadPropString(mpProp, "icontype", buf, sizeof(buf)))
            {
                if (!strcmp(buf, "cd"))
                    mp->icon = ICO_CD;
                else if (!strcmp(buf, "floppy"))
                    mp->icon = ICO_FLOPPY;
                else if (!strcmp(buf, "zip"))
                    mp->icon = ICO_ZIP;
                else if (!strcmp(buf, "hd"))
                    mp->icon = ICO_HD;
                else if (!strcmp(buf, "dos"))
                    mp->icon = ICO_DOS;
                else if (!strcmp(buf, "mac"))
                    mp->icon = ICO_MAC;
                else if (!strcmp(buf, "other"))
                    mp->icon = ICO_OTHER;
                else if (!strcmp(buf, "usr1"))
                    mp->icon = ICO_USR1;
                else if (!strcmp(buf, "usr2"))
                    mp->icon = ICO_USR2;
                else
                    fprintf(stderr, "Unknown icontype %s\n", buf);
            }

            if (loadPropString(mpProp, "showcapacity", buf, sizeof(buf)))
            {
                if (!strcmp(buf, "none"))
                    mp->showCap = CAPACITY_NONE;
                else if (!strcmp(buf, "pfull"))
                    mp->showCap = CAPACITY_PFULL;
                else if (!strcmp(buf, "pempty"))
                    mp->showCap = CAPACITY_PEMPTY;
                else if (!strcmp(buf, "mfull"))
                    mp->showCap = CAPACITY_MFULL;
                else if (!strcmp(buf, "mempty"))
                    mp->showCap = CAPACITY_MEMPTY;
            }

            if (loadPropString(mpProp, "ignore", buf, sizeof(buf)))
            {
                if (!strcmp(buf, "yes"))
                    mp->ignored = TRUE;
                else
                    mp->ignored = FALSE;
            }

            if (loadPropString(mpProp, "eject", buf, sizeof(buf)))
            {
                if (!strcmp(buf, "yes"))
                    mp->ejectFlag = TRUE;
                else
                    mp->ejectFlag = FALSE;
            }

        }
        WMReleasePropList(nameProp);
    }

    WMReleasePropList(props);
}

bool            savePropString(
        WMPropList *p,
        char *key,
        char *valbuf,
        size_t valbuf_size)
{
    WMPropList *keyProp, *valProp;
    bool string_found = FALSE;

    keyProp = WMCreatePLString(key);
    if (keyProp)
    {
        valProp = WMGetFromPLDictionary(p, keyProp);
        if (valProp && WMIsPLString(valProp))
        {
            string_found = TRUE;
            if (strcmp(valbuf, WMGetFromPLString(valProp)) == 0)
            {
                WMReleasePropList(keyProp);
                return string_found;
            }

            WMRemoveFromPLDictionary(p, keyProp);
        }

        if (*valbuf)
        {
            valProp = WMCreatePLString(valbuf);
            WMPutInPLDictionary(p, keyProp, valProp);
            WMReleasePropList(valProp);
        }

        WMReleasePropList(keyProp);
    }

    return string_found;
}

void            saveProperties(allInfo *ai)
{
    char        buf[256];
    int         cur_mp, i;
    WMPropList *props;
    struct MPInfo  *mp;

    props = WMReadPropListFromFile(getFileName());
    if (props == NULL)
    {
        static int recursion = 0;

        if (recursion)
        {
            printf("FATAL: Unable to write properties file\n");
            exit(1);
        }
        else
            recursion++;

        if (createDefaultPropertiesFile(ai) == FALSE)
            return;
    }

    savePropString(props, "txtfont", txtfont, STR);
    savePropString(props, "txtcolor", txtcolor, STR);
    savePropString(props, "capcolor", capcolor, STR);
    savePropString(props, "capfont", capfont, STR);
    savePropString(props, "mountcmd", mountcmd, STR);
    savePropString(props, "umountcmd", umountcmd, STR);
    savePropString(props, "ejectcmd", ejectcmd, STR);
    savePropString(props, "opencmd", opencmd, STR);
    savePropString(props, "initial_mount", initialMnt, STR);

    /*
     * If the icondir is the same as the compiled in default,
     * then remove it from the config file. (in case mount.app
     * gets recompiled with a new ICONDIR)
     */
    if (strcmp(icondir, ICONDIR) == 0)
    {
        WMPropList *keyProp;

        keyProp = WMCreatePLString("icondir");
        if (keyProp && WMGetFromPLDictionary(props, keyProp))
            WMRemoveFromPLDictionary(props, keyProp);
        WMReleasePropList(keyProp);
    }
    else
        savePropString(props, "icondir", icondir, STR);

    for (i = 0 ; i < MAX_ICONS ; i++)
    {
        if (ai->iconFileName[i][0] == 0)
            continue;

        safeCopy(buf, ai->iconFileName[i], sizeof(buf));

        switch (i)
        {
        case ICO_CD:
            savePropString(props, "cdicon", buf, sizeof(buf));
            break;
        case ICO_FLOPPY:
            savePropString(props, "flicon", buf, sizeof(buf));
            break;
        case ICO_ZIP:
            savePropString(props, "zipicon", buf, sizeof(buf));
            break;
        case ICO_HD:
            savePropString(props, "hdicon", buf, sizeof(buf));
            break;
        case ICO_DOS:
            savePropString(props, "dosicon", buf, sizeof(buf));
            break;
        case ICO_MAC:
            savePropString(props, "macicon", buf, sizeof(buf));
            break;
        case ICO_OTHER:
            savePropString(props, "unkicon", buf, sizeof(buf));
            break;
        case ICO_USR1:
            savePropString(props, "usr1icon", buf, sizeof(buf));
            break;
        case ICO_USR2:
            savePropString(props, "usr2icon", buf, sizeof(buf));
            break;
        }
    }

    if (ai->idleCycle)
        savePropString(props, "idleCycle", "yes", 4);
    else
        savePropString(props, "idleCycle", "no", 3);

    if (ai->showErrorChars)
        savePropString(props, "errorChars", "yes", 4);
    else
        savePropString(props, "errorChars", "no", 3);

    if (ai->newbieMessages)
        savePropString(props, "newbie", "yes", 4);
    else
        savePropString(props, "newbie", "no", 3);

    if (ai->descriptChar)
        savePropString(props, "descriptive", "yes", 4);
    else
        savePropString(props, "descriptive", "no", 3);

    for (cur_mp = 0; cur_mp < ai->numMounts; cur_mp++)
    {
        WMPropList *nameProp, *mpProp;

        mp = ai->mpi[cur_mp];
        sprintf(buf, "mountpoint.%s", mp->mountStr);

        nameProp = WMCreatePLString(buf);
        if (nameProp == NULL)
            continue;

        mpProp = WMGetFromPLDictionary(props, nameProp);
        if (mpProp && WMIsPLDictionary(mpProp))
        {
            WMRemoveFromPLDictionary(props, nameProp);
        }
        {
            WMPropList *key, *value;

            key = WMCreatePLString("mountname");
            value = WMCreatePLString(mp->name);
            mpProp = WMCreatePLDictionary(key, value, NULL);
        }

        switch(mp->icon)
        {
        case ICO_CD:
            strcpy(buf, "cd");
            break;
        case ICO_FLOPPY:
            strcpy(buf, "floppy");
            break;
        case ICO_ZIP:
            strcpy(buf, "zip");
            break;
        case ICO_HD:
            strcpy(buf, "hd");
            break;
        case ICO_DOS:
            strcpy(buf, "dos");
            break;
        case ICO_MAC:
            strcpy(buf, "mac");
            break;
        default:
        case ICO_OTHER:
            strcpy(buf, "other");
            break;
        case ICO_USR1:
            strcpy(buf, "usr1");
            break;
        case ICO_USR2:
            strcpy(buf, "usr2");
            break;
        }
        savePropString(mpProp, "icontype", buf, sizeof(buf));

        switch (mp->showCap)
        {
        case CAPACITY_NONE:
            strcpy(buf, "none");
            break;
        case CAPACITY_PFULL:
            strcpy(buf, "pfull");
            break;
        case CAPACITY_PEMPTY:
            strcpy(buf, "pempty");
            break;
        case CAPACITY_MFULL:
            strcpy(buf, "mfull");
            break;
        case CAPACITY_MEMPTY:
            strcpy(buf, "mempty");
            break;
        }
        savePropString(mpProp, "showcapacity", buf, sizeof(buf));

        if (mp->ignored)
            savePropString(mpProp, "ignore", "yes", 4);
        else
            savePropString(mpProp, "ignore", "no", 3);

        if (mp->ejectFlag)
            savePropString(mpProp, "eject", "yes", 4);
        else
            savePropString(mpProp, "eject", "no", 3);

        WMPutInPLDictionary(props, nameProp, mpProp);

        WMReleasePropList(nameProp);
        WMReleasePropList(mpProp);
    }

    if (WMWritePropListToFile(props, getFileName(), FALSE) == FALSE)
        printf("Error occurred synchronizing proplist\n");

    WMReleasePropList(props);
}

/*
   Assumes src is valid string pointer which zero terminates somewhere
   in it's allocated space.  It will copy up to dest_len-1 characters
   from src to dest and then make sure dest is zero terminated.
   if dest_len == 0, then *dest will be set to 0 anyway.
 */
void            safeCopy(char *dest, char *src, int dest_len)
{
    for (; *src; src++, dest++)
        if (--dest_len <= 0)
            break;
        else
            *dest = *src;
    *dest = 0;
}

char    *getFileName()
{
    static char buf[256] = "";
    char       *str, *home;

    /* Calculate only once, then always return the same name */
    if (*buf)
        return buf;

    /* Hacked by Michael Bueker: The config file simply gets
     * written to $HOME/.mount.app/ instead of a dir related
     * to GNUstep.
     */
    if (((home = getenv("HOME")) == 0) || strlen(home) > 100)
    {
        printf("$HOME not valid\n");
        exit(1);
    }

    sprintf(buf, "%s/.mount.app/mount.app", home);

    return buf;
}

int      createDefaultPropertiesFile(allInfo *ai)
{
    FILE *fp;

    fp = fopen(getFileName(), "w");
    if (fp == NULL)
    {
        if (errno == ENOENT)
        {
            if (makeRecursivePath(getFileName()) == 0)
                fp = fopen(getFileName(), "w");
            else
            {
                fprintf(stderr, "Unable to open new properties file\n");
                return FALSE;
            }
        }
        else
        {
            fprintf(stderr, "Unable to open new properties file\n");
            return FALSE;
        }
    }

    fputs("{ }\n", fp);
    fclose(fp);

    saveProperties(ai);

    return TRUE;
}

int       makeRecursivePath(char *path)
{
    char  buf[256];
    int   i;

    /* make copy on stack, then truncate last dirname */
    strcpy(buf, path);
    for (i = strlen(buf) - 1; i > 0 ; i--)
        if (buf[i] == '/')
            break;
    if (i <= 0)
        return -1;
    else
        buf[i] = 0;

    if (mkdir(buf, 0777) == -1)
    {
        if (errno == ENOENT)
        {
            if (makeRecursivePath(buf) == 0)
                return mkdir(buf, 0777);
            else
                return -1;
        }
        else
            return -1;
    }
    else
        return 0;
}
