/**
 * \file text3.C
 * This file is part of LyX, the document processor.
 * Licence details can be found in the file COPYING.
 *
 * \author Asger Alstrup
 * \author Lars Gullik Bjnnes
 * \author Alfredo Braunstein
 * \author Angus Leeming
 * \author John Levon
 * \author Andr Pnitz
 *
 * Full author contact details are available in file CREDITS.
 */

#include <config.h>

#include "lyxtext.h"

#include "BranchList.h"
#include "FloatList.h"
#include "FuncStatus.h"
#include "buffer.h"
#include "buffer_funcs.h"
#include "bufferparams.h"
#include "BufferView.h"
#include "cursor.h"
#include "coordcache.h"
#include "CutAndPaste.h"
#include "debug.h"
#include "dispatchresult.h"
#include "factory.h"
#include "funcrequest.h"
#include "gettext.h"
#include "intl.h"
#include "language.h"
#include "LyXAction.h"
#include "lyxfunc.h"
#include "lyxlex.h"
#include "lyxrc.h"
#include "lyxrow.h"
#include "paragraph.h"
#include "paragraph_funcs.h"
#include "ParagraphParameters.h"
#include "undo.h"
#include "vspace.h"

#include "frontends/Dialogs.h"
#include "frontends/LyXView.h"

#include "insets/insetcommand.h"
#include "insets/insetfloatlist.h"
#include "insets/insetnewline.h"
#include "insets/insetquotes.h"
#include "insets/insetspecialchar.h"
#include "insets/insettext.h"

#include "support/lstrings.h"
#include "support/lyxlib.h"
#include "support/convert.h"
#include "support/lyxtime.h"

#include "mathed/math_hullinset.h"
#include "mathed/math_macrotemplate.h"

#include <boost/current_function.hpp>

#include <clocale>
#include <sstream>

using lyx::pos_type;

using lyx::cap::copySelection;
using lyx::cap::cutSelection;
using lyx::cap::pasteSelection;
using lyx::cap::replaceSelection;

using lyx::support::isStrUnsignedInt;
using lyx::support::token;

using std::endl;
using std::string;
using std::istringstream;


extern string current_layout;


namespace {

	// globals...
	LyXFont freefont(LyXFont::ALL_IGNORE);
	bool toggleall = false;


	void toggleAndShow(LCursor & cur, LyXText * text,
		LyXFont const & font, bool toggleall = true)
	{
		text->toggleFree(cur, font, toggleall);

		if (font.language() != ignore_language ||
				font.number() != LyXFont::IGNORE) {
			Paragraph & par = cur.paragraph();
			text->bidi.computeTables(par, cur.buffer(), cur.textRow());
			if (cur.boundary() !=
					text->bidi.isBoundary(cur.buffer(), par,
							cur.pos(),
							text->real_current_font))
				text->setCursor(cur, cur.pit(), cur.pos(),
						false, !cur.boundary());
		}
	}


	void moveCursor(LCursor & cur, bool selecting)
	{
		if (selecting || cur.mark())
			cur.setSelection();
		if (!cur.selection())
			cur.bv().haveSelection(false);
		cur.bv().switchKeyMap();
	}


	void finishChange(LCursor & cur, bool selecting)
	{
		finishUndo();
		moveCursor(cur, selecting);
	}


	void mathDispatch(LCursor & cur, FuncRequest const & cmd, bool display)
	{
		recordUndo(cur);
		string sel = cur.selectionAsString(false);
		//lyxerr << "selection is: '" << sel << "'" << endl;

		// It may happen that sel is empty but there is a selection
		replaceSelection(cur);

		if (sel.empty()) {
			const int old_pos = cur.pos();
			cur.insert(new MathHullInset("simple"));
			BOOST_ASSERT(old_pos == cur.pos());
			cur.nextInset()->edit(cur, true);
			// don't do that also for LFUN_MATH_MODE
			// unless you want end up with always changing
			// to mathrm when opening an inlined inset --
			// I really hate "LyXfunc overloading"...
			if (display)
				cur.dispatch(FuncRequest(LFUN_MATH_DISPLAY));
			// Avoid an unnecessary undo step if cmd.argument
			// is empty
			if (!cmd.argument.empty())
				cur.dispatch(FuncRequest(LFUN_INSERT_MATH,
				                         cmd.argument));
		} else {
			// create a macro if we see "\\newcommand"
			// somewhere, and an ordinary formula
			// otherwise
			istringstream is(sel);
			if (sel.find("\\newcommand") == string::npos
			    && sel.find("\\def") == string::npos)
			{
				MathHullInset * formula = new MathHullInset;
				LyXLex lex(0, 0);
				lex.setStream(is);
				formula->read(cur.buffer(), lex);
				if (formula->getType() == "none")
					// Don't create pseudo formulas if
					// delimiters are left out
					formula->mutate("simple");
				cur.insert(formula);
			} else
				cur.insert(new MathMacroTemplate(is));
		}
		cur.message(N_("Math editor mode"));
	}

} // namespace anon



namespace bv_funcs {

string const freefont2string()
{
	string data;
	if (font2string(freefont, toggleall, data))
		return data;
	return string();
}

}

bool LyXText::cursorPrevious(LCursor & cur)
{
	pos_type cpos = cur.pos();
	lyx::pit_type cpar = cur.pit();

	int x = cur.x_target();

	bool updated = setCursorFromCoordinates(cur, x, 0);
	if (updated)
		cur.bv().update();
	cur.dispatch(FuncRequest(cur.selection()? LFUN_UPSEL: LFUN_UP));

	if (cpar == cur.pit() && cpos == cur.pos()) {
		// we have a row which is taller than the workarea. The
		// simplest solution is to move to the previous row instead.
		cur.dispatch(FuncRequest(cur.selection()? LFUN_UPSEL: LFUN_UP));
	}

	cur.bv().updateScrollbar();
	finishUndo();
	return updated;
}


bool LyXText::cursorNext(LCursor & cur)
{
	pos_type cpos = cur.pos();
	lyx::pit_type cpar = cur.pit();

	int x = cur.x_target();
	bool updated = setCursorFromCoordinates(cur, x, cur.bv().workHeight() - 1);
	if (updated)
		cur.bv().update();
	cur.dispatch(FuncRequest(cur.selection()? LFUN_DOWNSEL: LFUN_DOWN));

	if (cpar == cur.pit() && cpos == cur.pos()) {
		// we have a row which is taller than the workarea. The
		// simplest solution is to move to the next row instead.
		cur.dispatch(FuncRequest(cur.selection()? LFUN_DOWNSEL: LFUN_DOWN));
	}

	cur.bv().updateScrollbar();
	finishUndo();
	return updated;
}


namespace {

void specialChar(LCursor & cur, InsetSpecialChar::Kind kind)
{
	recordUndo(cur);
	lyx::cap::replaceSelection(cur);
	cur.insert(new InsetSpecialChar(kind));
	cur.posRight();
}


bool doInsertInset(LCursor & cur, LyXText * text,
	FuncRequest const & cmd, bool edit, bool pastesel)
{
	InsetBase * inset = createInset(&cur.bv(), cmd);
	if (!inset)
		return false;

	recordUndo(cur);
	bool gotsel = false;
	if (cur.selection()) {
		cur.bv().owner()->dispatch(FuncRequest(LFUN_CUT));
		gotsel = true;
	}
	text->insertInset(cur, inset);

	if (edit)
		inset->edit(cur, true);

	if (gotsel && pastesel) {
		cur.bv().owner()->dispatch(FuncRequest(LFUN_PASTE));
		// reset first par to default
		if (cur.lastpit() != 0 || cur.lastpos() != 0) {
			LyXLayout_ptr const layout =
				cur.buffer().params().getLyXTextClass().defaultLayout();
			cur.text()->paragraphs().begin()->layout(layout);
		}
	}
	return true;
}


void update(LCursor & cur)
{
	//we don't call update(true, false) directly to save a metrics call
	if (cur.bv().fitCursor())
		cur.bv().update(Update::Force);
}


} // anon namespace


void LyXText::number(LCursor & cur)
{
	LyXFont font(LyXFont::ALL_IGNORE);
	font.setNumber(LyXFont::TOGGLE);
	toggleAndShow(cur, this, font);
}


bool LyXText::isRTL(Paragraph const & par) const
{
	return par.isRightToLeftPar(bv()->buffer()->params());
}


void LyXText::dispatch(LCursor & cur, FuncRequest & cmd)
{
	lyxerr[Debug::ACTION] << "LyXText::dispatch: cmd: " << cmd << endl;

	BOOST_ASSERT(cur.text() == this);
	BufferView * bv = &cur.bv();
	CursorSlice oldTopSlice = cur.top();
	bool oldBoundary = cur.boundary();
	bool sel = cur.selection();
	// Signals that, even if needsUpdate == false, an update of the
	// cursor paragraph is required
	bool singleParUpdate = lyxaction.funcHasFlag(cmd.action,
		LyXAction::SingleParUpdate);
	// Signals that a full-screen update is required
	bool needsUpdate = !(lyxaction.funcHasFlag(cmd.action, 
		LyXAction::NoUpdate) || singleParUpdate);
	// Remember the old paragraph metric (_outer_ paragraph!)
	Dimension olddim = cur.bottom().paragraph().dim();

	switch (cmd.action) {

	case LFUN_APPENDIX: {
		Paragraph & par = cur.paragraph();
		bool start = !par.params().startOfAppendix();

#ifdef WITH_WARNINGS
#warning The code below only makes sense at top level.
// Should LFUN_APPENDIX be restricted to top-level paragraphs?
#endif
		// ensure that we have only one start_of_appendix in this document
		for (pit_type tmp = 0, end = pars_.size(); tmp != end; ++tmp) {
			if (pars_[tmp].params().startOfAppendix()) {
				recUndo(tmp);
				pars_[tmp].params().startOfAppendix(false);
				break;
			}
		}

		recordUndo(cur);
		par.params().startOfAppendix(start);

		// we can set the refreshing parameters now
		updateCounters(cur.buffer());
		break;
	}

	case LFUN_DELETE_WORD_FORWARD:
		cur.clearSelection();
		deleteWordForward(cur);
		finishChange(cur, false);
		break;

	case LFUN_DELETE_WORD_BACKWARD:
		cur.clearSelection();
		deleteWordBackward(cur);
		finishChange(cur, false);
		break;

	case LFUN_DELETE_LINE_FORWARD:
		cur.clearSelection();
		deleteLineForward(cur);
		finishChange(cur, false);
		break;

	case LFUN_WORDRIGHT:
	case LFUN_WORDRIGHTSEL:
		cur.selHandle(cmd.action == LFUN_WORDRIGHTSEL);
		if (isRTL(cur.paragraph()))
			needsUpdate = cursorLeftOneWord(cur);
		else
			needsUpdate = cursorRightOneWord(cur);
		break;

	case LFUN_WORDLEFT:
	case LFUN_WORDLEFTSEL:
		cur.selHandle(cmd.action == LFUN_WORDLEFTSEL);
		if (isRTL(cur.paragraph()))
			needsUpdate = cursorRightOneWord(cur);
		else
			needsUpdate = cursorLeftOneWord(cur);
		break;


	case LFUN_BEGINNINGBUF:
	case LFUN_BEGINNINGBUFSEL:
		cur.selHandle(cmd.action == LFUN_BEGINNINGBUFSEL);
		if (cur.depth() == 1) {
			needsUpdate = cursorTop(cur);
		} else {
			cur.undispatched();
		}
		break;

	case LFUN_ENDBUF:
	case LFUN_ENDBUFSEL:
		cur.selHandle(cmd.action == LFUN_ENDBUFSEL);
		if (cur.depth() == 1) {
			needsUpdate = cursorBottom(cur);
		} else {
			cur.undispatched();
		}
		break;

	case LFUN_RIGHT:
	case LFUN_RIGHTSEL:
		//lyxerr << BOOST_CURRENT_FUNCTION
		//       << " LFUN_RIGHT[SEL]:\n" << cur << endl;
		cur.selHandle(cmd.action == LFUN_RIGHTSEL);
		if (isRTL(cur.paragraph()))
			needsUpdate = cursorLeft(cur);
		else
			needsUpdate = cursorRight(cur);

		if (!needsUpdate && oldTopSlice == cur.top()
				&& cur.boundary() == oldBoundary) {
			cur.undispatched();
			cmd = FuncRequest(LFUN_FINISHED_RIGHT);
		}
		break;

	case LFUN_LEFT:
	case LFUN_LEFTSEL:
		//lyxerr << "handle LFUN_LEFT[SEL]:\n" << cur << endl;
		cur.selHandle(cmd.action == LFUN_LEFTSEL);
		if (isRTL(cur.paragraph()))
			needsUpdate = cursorRight(cur);
		else
			needsUpdate = cursorLeft(cur);

		if (!needsUpdate && oldTopSlice == cur.top()
			&& cur.boundary() == oldBoundary) {
			cur.undispatched();
			cmd = FuncRequest(LFUN_FINISHED_LEFT);
		}
		break;

	case LFUN_UP:
	case LFUN_UPSEL:
		update(cur);
		//lyxerr << "handle LFUN_UP[SEL]:\n" << cur << endl;
		cur.selHandle(cmd.action == LFUN_UPSEL);

		needsUpdate = cursorUp(cur);
		if (!needsUpdate && oldTopSlice == cur.top()
			  && cur.boundary() == oldBoundary) {
			cur.undispatched();
			cmd = FuncRequest(LFUN_FINISHED_UP);
		}
		break;

	case LFUN_DOWN:
	case LFUN_DOWNSEL:
		update(cur);
		//lyxerr << "handle LFUN_DOWN[SEL]:\n" << cur << endl;
		cur.selHandle(cmd.action == LFUN_DOWNSEL);
		needsUpdate = cursorDown(cur);
		if (!needsUpdate && oldTopSlice == cur.top() &&
		    cur.boundary() == oldBoundary)
		{
			cur.undispatched();
			cmd = FuncRequest(LFUN_FINISHED_DOWN);
		}
		break;

	case LFUN_UP_PARAGRAPH:
	case LFUN_UP_PARAGRAPHSEL:
		cur.selHandle(cmd.action == LFUN_UP_PARAGRAPHSEL);
		needsUpdate = cursorUpParagraph(cur);
		break;

	case LFUN_DOWN_PARAGRAPH:
	case LFUN_DOWN_PARAGRAPHSEL:
		cur.selHandle(cmd.action == LFUN_DOWN_PARAGRAPHSEL);
		needsUpdate = cursorDownParagraph(cur);
		break;

	case LFUN_PRIOR:
	case LFUN_PRIORSEL:
		update(cur);
		cur.selHandle(cmd.action == LFUN_PRIORSEL);
		if (cur.pit() == 0 && cur.textRow().pos() == 0) {
			cur.undispatched();
			cmd = FuncRequest(LFUN_FINISHED_UP);
		} else {
			needsUpdate = cursorPrevious(cur);
		}
		break;

	case LFUN_NEXT:
	case LFUN_NEXTSEL:
		update(cur);
		cur.selHandle(cmd.action == LFUN_NEXTSEL);
		if (cur.pit() == cur.lastpit()
			  && cur.textRow().endpos() == cur.lastpos()) {
			cur.undispatched();
			cmd = FuncRequest(LFUN_FINISHED_DOWN);
		} else {
			needsUpdate = cursorNext(cur);
		}
		break;

	case LFUN_HOME:
	case LFUN_HOMESEL:
		update(cur);
		cur.selHandle(cmd.action == LFUN_HOMESEL);
		needsUpdate = cursorHome(cur);
		break;

	case LFUN_END:
	case LFUN_ENDSEL:
		update(cur);
		cur.selHandle(cmd.action == LFUN_ENDSEL);
		needsUpdate = cursorEnd(cur);
		break;

	case LFUN_WORDSEL: {
		selectWord(cur, lyx::WHOLE_WORD);
		finishChange(cur, true);
		break;
	}

	case LFUN_BREAKLINE: {
		// Not allowed by LaTeX (labels or empty par)
		if (cur.pos() > cur.paragraph().beginOfBody()) {
			if (!cur.selection())
				recordUndo(cur);
			lyx::cap::replaceSelection(cur);
			cur.insert(new InsetNewline);
			cur.posRight();
			moveCursor(cur, false);
		}
		break;
	}

	case LFUN_DELETE:
		if (!cur.selection()) {
			if (cur.pos() == cur.paragraph().size())
				// Par boundary, force full-screen update
				singleParUpdate = false;
			needsUpdate = Delete(cur);
			cur.resetAnchor();
			// It is possible to make it a lot faster still
			// just comment out the line below...
		} else {
			cutSelection(cur, true, false);
			singleParUpdate = false;
		}
		moveCursor(cur, false);
		break;

	case LFUN_DELETE_SKIP:
		// Reverse the effect of LFUN_BREAKPARAGRAPH_SKIP.
		if (!cur.selection()) {
			if (cur.pos() == cur.lastpos()) {
				cursorRight(cur);
				cursorLeft(cur);
			}
			Delete(cur);
			cur.resetAnchor();
		} else {
			cutSelection(cur, true, false);
		}
		break;


	case LFUN_BACKSPACE:
		if (!cur.selection()) {
			if (bv->owner()->getIntl().getTransManager().backspace()) {
				// Par boundary, full-screen update
				if (cur.pos() == 0)
					singleParUpdate = false;
				needsUpdate = backspace(cur);
				cur.resetAnchor();
				// It is possible to make it a lot faster still
				// just comment out the line below...
			}
		} else {
			cutSelection(cur, true, false);
			singleParUpdate = false;
		}
		bv->switchKeyMap();
		break;

	case LFUN_BACKSPACE_SKIP:
		// Reverse the effect of LFUN_BREAKPARAGRAPH_SKIP.
		if (!cur.selection()) {
#ifdef WITH_WARNINGS
#warning look here
#endif
			//CursorSlice cur = cursor();
			backspace(cur);
			//anchor() = cur;
		} else {
			cutSelection(cur, true, false);
		}
		break;

	case LFUN_BREAKPARAGRAPH:
		lyx::cap::replaceSelection(cur);
		breakParagraph(cur, 0);
		cur.resetAnchor();
		bv->switchKeyMap();
		break;

	case LFUN_BREAKPARAGRAPHKEEPLAYOUT:
		lyx::cap::replaceSelection(cur);
		breakParagraph(cur, 1);
		cur.resetAnchor();
		bv->switchKeyMap();
		break;

	case LFUN_BREAKPARAGRAPH_SKIP: {
		// When at the beginning of a paragraph, remove
		// indentation.  Otherwise, do the same as LFUN_BREAKPARAGRAPH.
		lyx::cap::replaceSelection(cur);
		if (cur.pos() == 0)
			cur.paragraph().params().labelWidthString(string());
		else
			breakParagraph(cur, 0);
		cur.resetAnchor();
		bv->switchKeyMap();
		break;
	}

	case LFUN_PARAGRAPH_SPACING: {
		Paragraph & par = cur.paragraph();
		Spacing::Space cur_spacing = par.params().spacing().getSpace();
		string cur_value = "1.0";
		if (cur_spacing == Spacing::Other)
			cur_value = par.params().spacing().getValueAsString();

		istringstream is(cmd.argument);
		string tmp;
		is >> tmp;
		Spacing::Space new_spacing = cur_spacing;
		string new_value = cur_value;
		if (tmp.empty()) {
			lyxerr << "Missing argument to `paragraph-spacing'"
			       << endl;
		} else if (tmp == "single") {
			new_spacing = Spacing::Single;
		} else if (tmp == "onehalf") {
			new_spacing = Spacing::Onehalf;
		} else if (tmp == "double") {
			new_spacing = Spacing::Double;
		} else if (tmp == "other") {
			new_spacing = Spacing::Other;
			string tmpval = "0.0";
			is >> tmpval;
			lyxerr << "new_value = " << tmpval << endl;
			if (tmpval != "0.0")
				new_value = tmpval;
		} else if (tmp == "default") {
			new_spacing = Spacing::Default;
		} else {
			lyxerr << _("Unknown spacing argument: ")
			       << cmd.argument << endl;
		}
		if (cur_spacing != new_spacing || cur_value != new_value)
			par.params().spacing(Spacing(new_spacing, new_value));
		break;
	}

	case LFUN_INSET_INSERT: {
		recordUndo(cur);
		InsetBase * inset = createInset(bv, cmd);
		if (inset) {
			insertInset(cur, inset);
			cur.posRight();
		}
		break;
	}

	case LFUN_INSET_DISSOLVE:
		needsUpdate = dissolveInset(cur);
		break;

	case LFUN_INSET_SETTINGS:
		cur.inset().showInsetDialog(bv);
		break;

	case LFUN_SPACE_INSERT:
		if (cur.paragraph().layout()->free_spacing)
			insertChar(cur, ' ');
		else {
			doInsertInset(cur, this, cmd, false, false);
			cur.posRight();
		}
		moveCursor(cur, false);
		break;

	case LFUN_HYPHENATION:
		specialChar(cur, InsetSpecialChar::HYPHENATION);
		break;

	case LFUN_LIGATURE_BREAK:
		specialChar(cur, InsetSpecialChar::LIGATURE_BREAK);
		break;

	case LFUN_LDOTS:
		specialChar(cur, InsetSpecialChar::LDOTS);
		break;

	case LFUN_END_OF_SENTENCE:
		specialChar(cur, InsetSpecialChar::END_OF_SENTENCE);
		break;

	case LFUN_MENU_SEPARATOR:
		specialChar(cur, InsetSpecialChar::MENU_SEPARATOR);
		break;

	case LFUN_UPCASE_WORD:
		changeCase(cur, LyXText::text_uppercase);
		break;

	case LFUN_LOWCASE_WORD:
		changeCase(cur, LyXText::text_lowercase);
		break;

	case LFUN_CAPITALIZE_WORD:
		changeCase(cur, LyXText::text_capitalization);
		break;

	case LFUN_TRANSPOSE_CHARS:
		charsTranspose(cur);
		break;

	case LFUN_PASTE:
		cur.message(_("Paste"));
		lyx::cap::replaceSelection(cur);
		if (isStrUnsignedInt(cmd.argument))
			pasteSelection(cur, convert<unsigned int>(cmd.argument));
		else
			pasteSelection(cur, 0);
		cur.clearSelection(); // bug 393
		bv->switchKeyMap();
		finishUndo();
		break;

	case LFUN_CUT:
		cutSelection(cur, true, true);
		cur.message(_("Cut"));
		break;

	case LFUN_COPY:
		copySelection(cur);
		cur.message(_("Copy"));
		break;

	case LFUN_GETXY:
		cur.message(convert<string>(cursorX(cur.top(), cur.boundary())) + ' '
		          + convert<string>(cursorY(cur.top(), cur.boundary())));
		break;

	case LFUN_SETXY: {
		int x = 0;
		int y = 0;
		istringstream is(cmd.argument);
		is >> x >> y;
		if (!is)
			lyxerr << "SETXY: Could not parse coordinates in '"
			       << cmd.argument << std::endl;
		else
			setCursorFromCoordinates(cur, x, y);
		break;
	}

	case LFUN_GETFONT:
		if (current_font.shape() == LyXFont::ITALIC_SHAPE)
			cur.message("E");
		else if (current_font.shape() == LyXFont::SMALLCAPS_SHAPE)
			cur.message("N");
		else
			cur.message("0");
		break;

	case LFUN_GETLAYOUT:
		cur.message(cur.paragraph().layout()->name());
		break;

	case LFUN_LAYOUT: {
		lyxerr[Debug::INFO] << "LFUN_LAYOUT: (arg) "
		  << cmd.argument << endl;

		// Derive layout number from given argument (string)
		// and current buffer's textclass (number)
		LyXTextClass const & tclass = bv->buffer()->params().getLyXTextClass();
		string layout = cmd.argument.empty() ?
			tclass.defaultLayoutName() :
			cmd.argument;
		bool hasLayout = tclass.hasLayout(layout);

		// If the entry is obsolete, use the new one instead.
		if (hasLayout) {
			string const & obs = tclass[layout]->obsoleted_by();
			if (!obs.empty())
				layout = obs;
		}

		if (!hasLayout) {
			cur.errorMessage(string(N_("Layout ")) + layout +
				N_(" not known"));
			break;
		}

		bool change_layout = (current_layout != layout);

		if (!change_layout && cur.selection() &&
			cur.selBegin().pit() != cur.selEnd().pit())
		{
			pit_type spit = cur.selBegin().pit();
			pit_type epit = cur.selEnd().pit() + 1;
			while (spit != epit) {
				if (pars_[spit].layout()->name() != current_layout) {
					change_layout = true;
					break;
				}
				++spit;
			}
		}

		if (change_layout) {
			current_layout = layout;
			setLayout(cur, layout);
			bv->owner()->setLayout(layout);
			bv->switchKeyMap();
		}
		break;
	}

	case LFUN_PASTESELECTION: {
		cur.clearSelection();
		string const clip = bv->getClipboard();
		if (!clip.empty()) {
			recordUndo(cur);
			if (cmd.argument == "paragraph")
				insertStringAsParagraphs(cur, clip);
			else
				insertStringAsLines(cur, clip);
		}
		break;
	}

	case LFUN_QUOTE: {
		Paragraph & par = cur.paragraph();
		lyx::pos_type pos = cur.pos();
		BufferParams const & bufparams = bv->buffer()->params();
		LyXLayout_ptr const & style = par.layout();
		if (!style->pass_thru
		    && par.getFontSettings(bufparams, pos).language()->lang() != "hebrew") {
			// this avoids a double undo
			// FIXME: should not be needed, ideally
			if (!cur.selection())
				recordUndo(cur);
			lyx::cap::replaceSelection(cur);
			pos = cur.pos();
			char c;
			if (pos == 0)
				c = ' ';
			else if (cur.prevInset() && cur.prevInset()->isSpace())
				c = ' ';
			else
				c = par.getChar(pos - 1);
			string arg = cmd.argument;
			if (arg == "single")
				cur.insert(new InsetQuotes(c,
				    bufparams.quotes_language,
				    InsetQuotes::SingleQ));
			else
				cur.insert(new InsetQuotes(c,
				    bufparams.quotes_language,
				    InsetQuotes::DoubleQ));
			cur.posRight();
		}
		else
			bv->owner()->dispatch(FuncRequest(LFUN_SELFINSERT, "\""));
		break;
	}

	case LFUN_DATE_INSERT: 
		if (cmd.argument.empty())
			bv->owner()->dispatch(FuncRequest(LFUN_SELFINSERT,
				lyx::formatted_time(lyx::current_time())));
		else
			bv->owner()->dispatch(FuncRequest(LFUN_SELFINSERT,
				lyx::formatted_time(lyx::current_time(), cmd.argument)));
		break;

	case LFUN_MOUSE_TRIPLE:
		if (cmd.button() == mouse_button::button1) {
			cursorHome(cur);
			cur.resetAnchor();
			cursorEnd(cur);
			cur.setSelection();
			bv->cursor() = cur;
			bv->haveSelection(cur.selection());
		}
		break;

	case LFUN_MOUSE_DOUBLE:
		if (cmd.button() == mouse_button::button1) {
			selectWord(cur, lyx::WHOLE_WORD_STRICT);
			bv->cursor() = cur;
			bv->haveSelection(cur.selection());
		}
		break;

	// Single-click on work area
	case LFUN_MOUSE_PRESS: {
		// Right click on a footnote flag opens float menu
		if (cmd.button() == mouse_button::button3)
			cur.clearSelection();

		// Middle button press pastes if we have a selection
		// We do this here as if the selection was inside an inset
		// it could get cleared on the unlocking of the inset so
		// we have to check this first
		bool paste_internally = false;
		if (cmd.button() == mouse_button::button2 && cur.selection()) {
			bv->owner()->dispatch(FuncRequest(LFUN_COPY));
			paste_internally = true;
		}

		bv->mouseSetCursor(cur);

		// Insert primary selection with middle mouse
		// if there is a local selection in the current buffer,
		// insert this
		if (cmd.button() == mouse_button::button2) {
			if (paste_internally)
				bv->owner()->dispatch(FuncRequest(LFUN_PASTE));
			else
				bv->owner()->dispatch(FuncRequest(LFUN_PASTESELECTION, "paragraph"));
		}

		break;
	}

	case LFUN_MOUSE_MOTION: {
		// Only use motion with button 1
		//if (cmd.button() != mouse_button::button1)
		//	return false;

		// ignore motions deeper nested than the real anchor
		LCursor & bvcur = cur.bv().cursor();
		if (bvcur.anchor_.hasPart(cur)) {
			CursorSlice old = bvcur.top();

			int const wh = bv->workHeight();
			int const y = std::max(0, std::min(wh - 1, cmd.y));

			setCursorFromCoordinates(cur, cmd.x, y);
			cur.x_target() = cmd.x;
			if (cmd.y >= wh)
				cursorDown(cur);
			else if (cmd.y < 0)
				cursorUp(cur);
			// This is to allow jumping over large insets
			if (cur.top() == old) {
				if (cmd.y >= wh)
					cursorDown(cur);
				else if (cmd.y < 0)
					cursorUp(cur);
			}

			if (cur.top() == old)
				cur.noUpdate();
			else {
				// don't set anchor_
				bvcur.setCursor(cur);
				bvcur.selection() = true;
				//lyxerr << "MOTION: " << bv->cursor() << endl;
			}

		} else
			cur.undispatched();
		break;
	}

	case LFUN_MOUSE_RELEASE: {
		if (cmd.button() == mouse_button::button2)
			break;

		// finish selection
		if (cmd.button() == mouse_button::button1)
			bv->haveSelection(cur.selection());

		bv->switchKeyMap();
		bv->owner()->updateMenubar();
		bv->owner()->updateToolbars();
		break;
	}

	case LFUN_SELFINSERT: {
		if (cmd.argument.empty())
			break;

		// Automatically delete the currently selected
		// text and replace it with what is being
		// typed in now. Depends on lyxrc settings
		// "auto_region_delete", which defaults to
		// true (on).

		if (lyxrc.auto_region_delete) {
			if (cur.selection())
				cutSelection(cur, false, false);
			bv->haveSelection(false);
		}

		cur.clearSelection();
		LyXFont const old_font = real_current_font;

		string::const_iterator cit = cmd.argument.begin();
		string::const_iterator end = cmd.argument.end();
		for (; cit != end; ++cit)
			bv->owner()->getIntl().getTransManager().
				TranslateAndInsert(*cit, this);

		cur.resetAnchor();
		moveCursor(cur, false);
		bv->updateScrollbar();
		break;
	}

	case LFUN_URL: {
		InsetCommandParams p("url");
		string const data = InsetCommandMailer::params2string("url", p);
		bv->owner()->getDialogs().show("url", data, 0);
		break;
	}

	case LFUN_HTMLURL: {
		InsetCommandParams p("htmlurl");
		string const data = InsetCommandMailer::params2string("url", p);
		bv->owner()->getDialogs().show("url", data, 0);
		break;
	}

	case LFUN_INSERT_LABEL: {
		// Try to generate a valid label
		string const contents = cmd.argument.empty() ?
			cur.getPossibleLabel() : cmd.argument;

		InsetCommandParams p("label", contents);
		string const data = InsetCommandMailer::params2string("label", p);

		if (cmd.argument.empty()) {
			bv->owner()->getDialogs().show("label", data, 0);
		} else {
			FuncRequest fr(LFUN_INSET_INSERT, data);
			dispatch(cur, fr);
		}
		break;
	}


#if 0
	case LFUN_INSET_LIST:
	case LFUN_INSET_THEOREM:
	case LFUN_INSET_CAPTION:
#endif
	case LFUN_INSERT_NOTE:
	case LFUN_INSERT_CHARSTYLE:
	case LFUN_INSERT_BOX:
	case LFUN_INSERT_BRANCH:
	case LFUN_INSERT_BIBITEM:
	case LFUN_INSET_ERT:
	case LFUN_INSET_FOOTNOTE:
	case LFUN_INSET_MARGINAL:
	case LFUN_INSET_OPTARG:
	case LFUN_ENVIRONMENT_INSERT:
		// Open the inset, and move the current selection
		// inside it.
		doInsertInset(cur, this, cmd, true, true);
		cur.posRight();
		break;

	case LFUN_TABULAR_INSERT:
		// if there were no arguments, just open the dialog
		if (doInsertInset(cur, this, cmd, false, true))
			cur.posRight();
		else
			bv->owner()->getDialogs().show("tabularcreate");

		break;

	case LFUN_INSET_FLOAT:
	case LFUN_INSET_WIDE_FLOAT:
	case LFUN_INSET_WRAP:
		doInsertInset(cur, this, cmd, true, true);
		cur.posRight();
		// FIXME: the "Caption" name should not be hardcoded,
		// but given by the float definition.
		cur.dispatch(FuncRequest(LFUN_LAYOUT, "Caption"));
		break;

	case LFUN_INDEX_INSERT: {
		InsetBase * inset = createInset(&cur.bv(), cmd);
		if (!inset)
			break;

		recordUndo(cur);
		cur.clearSelection();
		insertInset(cur, inset);
		inset->edit(cur, true);
		cur.posRight();
		break;
	}

	case LFUN_INDEX_PRINT:
	case LFUN_TOC_INSERT:
	case LFUN_HFILL:
	case LFUN_INSERT_LINE:
	case LFUN_INSERT_PAGEBREAK:
		// do nothing fancy
		doInsertInset(cur, this, cmd, false, false);
		cur.posRight();
		break;

	case LFUN_DEPTH_MIN:
		changeDepth(cur, DEC_DEPTH);
		break;

	case LFUN_DEPTH_PLUS:
		changeDepth(cur, INC_DEPTH);
		break;

	case LFUN_MATH_DISPLAY:
		mathDispatch(cur, cmd, true);
		break;

	case LFUN_MATH_IMPORT_SELECTION:
	case LFUN_MATH_MODE:
		if (cmd.argument == "on")
			// don't pass "on" as argument
			mathDispatch(cur, FuncRequest(LFUN_MATH_MODE), false);
		else
			mathDispatch(cur, cmd, false);
		break;

	case LFUN_MATH_MACRO:
		if (cmd.argument.empty())
			cur.errorMessage(N_("Missing argument"));
		else {
			string s = cmd.argument;
			string const s1 = token(s, ' ', 1);
			int const nargs = s1.empty() ? 0 : convert<int>(s1);
			string const s2 = token(s, ' ', 2);
			string const type = s2.empty() ? "newcommand" : s2;
			cur.insert(new MathMacroTemplate(token(s, ' ', 0), nargs, type));
			//cur.nextInset()->edit(cur, true);
		}
		break;

	// passthrough hat and underscore outside mathed:
	case LFUN_SUBSCRIPT:
		mathDispatch(cur, FuncRequest(LFUN_SELFINSERT, "_"), false);
		break;
	case LFUN_SUPERSCRIPT:
		mathDispatch(cur, FuncRequest(LFUN_SELFINSERT, "^"), false);
		break;

	case LFUN_INSERT_MATH:
	case LFUN_INSERT_MATRIX:
	case LFUN_MATH_DELIM:
	case LFUN_MATH_BIGDELIM: {
		cur.insert(new MathHullInset("simple"));
		cur.dispatch(FuncRequest(LFUN_RIGHT));
		cur.dispatch(cmd);
		break;
	}

	case LFUN_EMPH: {
		LyXFont font(LyXFont::ALL_IGNORE);
		font.setEmph(LyXFont::TOGGLE);
		toggleAndShow(cur, this, font);
		break;
	}

	case LFUN_BOLD: {
		LyXFont font(LyXFont::ALL_IGNORE);
		font.setSeries(LyXFont::BOLD_SERIES);
		toggleAndShow(cur, this, font);
		break;
	}

	case LFUN_NOUN: {
		LyXFont font(LyXFont::ALL_IGNORE);
		font.setNoun(LyXFont::TOGGLE);
		toggleAndShow(cur, this, font);
		break;
	}

	case LFUN_CODE: {
		LyXFont font(LyXFont::ALL_IGNORE);
		font.setFamily(LyXFont::TYPEWRITER_FAMILY); // no good
		toggleAndShow(cur, this, font);
		break;
	}

	case LFUN_SANS: {
		LyXFont font(LyXFont::ALL_IGNORE);
		font.setFamily(LyXFont::SANS_FAMILY);
		toggleAndShow(cur, this, font);
		break;
	}

	case LFUN_ROMAN: {
		LyXFont font(LyXFont::ALL_IGNORE);
		font.setFamily(LyXFont::ROMAN_FAMILY);
		toggleAndShow(cur, this, font);
		break;
	}

	case LFUN_DEFAULT: {
		LyXFont font(LyXFont::ALL_INHERIT, ignore_language);
		toggleAndShow(cur, this, font);
		break;
	}

	case LFUN_UNDERLINE: {
		LyXFont font(LyXFont::ALL_IGNORE);
		font.setUnderbar(LyXFont::TOGGLE);
		toggleAndShow(cur, this, font);
		break;
	}

	case LFUN_FONT_SIZE: {
		LyXFont font(LyXFont::ALL_IGNORE);
		font.setLyXSize(cmd.argument);
		toggleAndShow(cur, this, font);
		break;
	}

	case LFUN_LANGUAGE: {
		Language const * lang = languages.getLanguage(cmd.argument);
		if (!lang)
			break;
		LyXFont font(LyXFont::ALL_IGNORE);
		font.setLanguage(lang);
		toggleAndShow(cur, this, font);
		bv->switchKeyMap();
		break;
	}

	case LFUN_FREEFONT_APPLY:
		toggleAndShow(cur, this, freefont, toggleall);
		cur.message(_("Character set"));
		break;

	// Set the freefont using the contents of \param data dispatched from
	// the frontends and apply it at the current cursor location.
	case LFUN_FREEFONT_UPDATE: {
		LyXFont font;
		bool toggle;
		if (bv_funcs::string2font(cmd.argument, font, toggle)) {
			freefont = font;
			toggleall = toggle;
			toggleAndShow(cur, this, freefont, toggleall);
			cur.message(_("Character set"));
		}
		break;
	}

	case LFUN_FINISHED_LEFT:
		lyxerr[Debug::DEBUG] << "handle LFUN_FINISHED_LEFT:\n" << cur << endl;
		break;

	case LFUN_FINISHED_RIGHT:
		lyxerr[Debug::DEBUG] << "handle LFUN_FINISHED_RIGHT:\n" << cur << endl;
		++cur.pos();
		break;

	case LFUN_FINISHED_UP:
		lyxerr[Debug::DEBUG] << "handle LFUN_FINISHED_UP:\n" << cur << endl;
		cursorUp(cur);
		break;

	case LFUN_FINISHED_DOWN:
		lyxerr[Debug::DEBUG] << "handle LFUN_FINISHED_DOWN:\n" << cur << endl;
		cursorDown(cur);
		break;

	case LFUN_LAYOUT_PARAGRAPH: {
		string data;
		params2string(cur.paragraph(), data);
		data = "show\n" + data;
		bv->owner()->getDialogs().show("paragraph", data);
		break;
	}

	case LFUN_PARAGRAPH_UPDATE: {
		if (!bv->owner()->getDialogs().visible("paragraph"))
			break;
		string data;
		params2string(cur.paragraph(), data);

		// Will the paragraph accept changes from the dialog?
		bool const accept = !cur.inset().forceDefaultParagraphs(cur.idx());

		data = "update " + convert<string>(accept) + '\n' + data;
		bv->owner()->getDialogs().update("paragraph", data);
		break;
	}

	case LFUN_UMLAUT:
	case LFUN_CIRCUMFLEX:
	case LFUN_GRAVE:
	case LFUN_ACUTE:
	case LFUN_TILDE:
	case LFUN_CEDILLA:
	case LFUN_MACRON:
	case LFUN_DOT:
	case LFUN_UNDERDOT:
	case LFUN_UNDERBAR:
	case LFUN_CARON:
	case LFUN_SPECIAL_CARON:
	case LFUN_BREVE:
	case LFUN_TIE:
	case LFUN_HUNG_UMLAUT:
	case LFUN_CIRCLE:
	case LFUN_OGONEK:
		bv->owner()->getLyXFunc().handleKeyFunc(cmd.action);
		if (!cmd.argument.empty())
			bv->owner()->getIntl().getTransManager()
				.TranslateAndInsert(cmd.argument[0], this);
		break;

	case LFUN_FLOAT_LIST: {
		LyXTextClass const & tclass = bv->buffer()->params().getLyXTextClass();
		if (tclass.floats().typeExist(cmd.argument)) {
			// not quite sure if we want this...
			recordUndo(cur);
			cur.clearSelection();
			breakParagraph(cur);

			if (cur.lastpos() != 0) {
				cursorLeft(cur);
				breakParagraph(cur);
			}

			setLayout(cur, tclass.defaultLayoutName());
			setParagraph(cur, Spacing(), LYX_ALIGN_LAYOUT, string(), 0);
			insertInset(cur, new InsetFloatList(cmd.argument));
			cur.posRight();
		} else {
			lyxerr << "Non-existent float type: "
			       << cmd.argument << endl;
		}
		break;
	}

	case LFUN_ACCEPT_CHANGE: {
		acceptChange(cur);
		break;
	}

	case LFUN_REJECT_CHANGE: {
		rejectChange(cur);
		break;
	}

	case LFUN_THESAURUS_ENTRY: {
		string arg = cmd.argument;
		if (arg.empty()) {
			arg = cur.selectionAsString(false);
			// FIXME
			if (arg.size() > 100 || arg.empty()) {
				// Get word or selection
				selectWordWhenUnderCursor(cur, lyx::WHOLE_WORD);
				arg = cur.selectionAsString(false);
			}
		}
		bv->owner()->getDialogs().show("thesaurus", arg);
		break;
	}

	case LFUN_PARAGRAPH_APPLY: {
		// Given data, an encoding of the ParagraphParameters
		// generated in the Paragraph dialog, this function sets
		// the current paragraph appropriately.
		istringstream is(cmd.argument);
		LyXLex lex(0, 0);
		lex.setStream(is);
		ParagraphParameters params;
		params.read(lex);
		setParagraph(cur,
					 params.spacing(),
					 params.align(),
					 params.labelWidthString(),
					 params.noindent());
		cur.message(_("Paragraph layout set"));
		break;
	}

	case LFUN_INSET_DIALOG_SHOW: {
		InsetBase * inset = cur.nextInset();
		if (inset) {
			FuncRequest fr(LFUN_INSET_DIALOG_SHOW);
			inset->dispatch(cur, fr);
		}
		break;
	}

	case LFUN_ESCAPE:
		if (cur.selection()) {
			cur.selection() = false;
		} else {
			cur.undispatched();
			cmd = FuncRequest(LFUN_FINISHED_RIGHT);
		}
		break;

	default:
		lyxerr[Debug::ACTION] 
			<< BOOST_CURRENT_FUNCTION
			<< ": Command " << cmd 
			<< " not DISPATCHED by LyXText" << endl;
		cur.undispatched();
		break;
	}

	if (singleParUpdate)
		// Inserting characters does not change par height
		if (cur.bottom().paragraph().dim().height() 
		    == olddim.height()) {
			// if so, update _only_ this paragraph
			cur.bv().update(Update::SinglePar |
					Update::FitCursor |
					Update::MultiParSel);
			cur.noUpdate();
			return;
		} else
			needsUpdate = true;
	if (!needsUpdate
	    && &oldTopSlice.inset() == &cur.inset()
	    && oldTopSlice.idx() == cur.idx()
	    && !sel
	    && !cur.selection())
		cur.noUpdate();
	else
		cur.needsUpdate();
}


bool LyXText::getStatus(LCursor & cur, FuncRequest const & cmd,
			FuncStatus & flag) const
{
	BOOST_ASSERT(cur.text() == this);

	LyXFont const & font = real_current_font;
	bool enable = true;
	InsetBase::Code code = InsetBase::NO_CODE;

	switch (cmd.action) {

	case LFUN_DEPTH_MIN:
		enable = changeDepthAllowed(cur, DEC_DEPTH);
		break;

	case LFUN_DEPTH_PLUS:
		enable = changeDepthAllowed(cur, INC_DEPTH);
		break;

	case LFUN_APPENDIX:
		flag.setOnOff(cur.paragraph().params().startOfAppendix());
		return true;

	case LFUN_INSERT_BIBITEM:
		enable = (cur.paragraph().layout()->labeltype == LABEL_BIBLIO);
		break;

	case LFUN_DIALOG_SHOW_NEW_INSET:
		if (cmd.argument == "bibitem")
			code = InsetBase::BIBITEM_CODE;
		else if (cmd.argument == "bibtex")
			code = InsetBase::BIBTEX_CODE;
		else if (cmd.argument == "box")
			code = InsetBase::BOX_CODE;
		else if (cmd.argument == "branch")
			code = InsetBase::BRANCH_CODE;
		else if (cmd.argument == "citation")
			code = InsetBase::CITE_CODE;
		else if (cmd.argument == "ert")
			code = InsetBase::ERT_CODE;
		else if (cmd.argument == "external")
			code = InsetBase::EXTERNAL_CODE;
		else if (cmd.argument == "float")
			code = InsetBase::FLOAT_CODE;
		else if (cmd.argument == "graphics")
			code = InsetBase::GRAPHICS_CODE;
		else if (cmd.argument == "include")
			code = InsetBase::INCLUDE_CODE;
		else if (cmd.argument == "index")
			code = InsetBase::INDEX_CODE;
		else if (cmd.argument == "label")
			code = InsetBase::LABEL_CODE;
		else if (cmd.argument == "note")
			code = InsetBase::NOTE_CODE;
		else if (cmd.argument == "ref")
			code = InsetBase::REF_CODE;
		else if (cmd.argument == "toc")
			code = InsetBase::TOC_CODE;
		else if (cmd.argument == "url")
			code = InsetBase::URL_CODE;
		else if (cmd.argument == "vspace")
			code = InsetBase::VSPACE_CODE;
		else if (cmd.argument == "wrap")
			code = InsetBase::WRAP_CODE;
		break;

	case LFUN_INSET_ERT:
		code = InsetBase::ERT_CODE;
		break;
	case LFUN_INSET_FOOTNOTE:
		code = InsetBase::FOOT_CODE;
		break;
	case LFUN_TABULAR_INSERT:
		code = InsetBase::TABULAR_CODE;
		break;
	case LFUN_INSET_MARGINAL:
		code = InsetBase::MARGIN_CODE;
		break;
	case LFUN_INSET_FLOAT:
	case LFUN_INSET_WIDE_FLOAT:
		code = InsetBase::FLOAT_CODE;
		break;
	case LFUN_INSET_WRAP:
		code = InsetBase::WRAP_CODE;
		break;
	case LFUN_FLOAT_LIST:
		code = InsetBase::FLOAT_LIST_CODE;
		break;
#if 0
	case LFUN_INSET_LIST:
		code = InsetBase::LIST_CODE;
		break;
	case LFUN_INSET_THEOREM:
		code = InsetBase::THEOREM_CODE;
		break;
#endif
	case LFUN_INSET_CAPTION:
		code = InsetBase::CAPTION_CODE;
		break;
	case LFUN_INSERT_NOTE:
		code = InsetBase::NOTE_CODE;
		break;
	case LFUN_INSERT_CHARSTYLE:
		code = InsetBase::CHARSTYLE_CODE;
		if (cur.buffer().params().getLyXTextClass().charstyles().empty())
			enable = false;
		break;
	case LFUN_INSERT_BOX:
		code = InsetBase::BOX_CODE;
		break;
	case LFUN_INSERT_BRANCH:
		code = InsetBase::BRANCH_CODE;
		if (cur.buffer().getMasterBuffer()->params().branchlist().empty())
			enable = false;
		break;
	case LFUN_INSERT_LABEL:
		code = InsetBase::LABEL_CODE;
		break;
	case LFUN_INSET_OPTARG:
		code = InsetBase::OPTARG_CODE;
		enable = numberOfOptArgs(cur.paragraph())
			< cur.paragraph().layout()->optionalargs;
		break;
	case LFUN_ENVIRONMENT_INSERT:
		code = InsetBase::BOX_CODE;
		break;
	case LFUN_INDEX_INSERT:
		code = InsetBase::INDEX_CODE;
		break;
	case LFUN_INDEX_PRINT:
		code = InsetBase::INDEX_PRINT_CODE;
		break;
	case LFUN_TOC_INSERT:
		code = InsetBase::TOC_CODE;
		break;
	case LFUN_HTMLURL:
	case LFUN_URL:
		code = InsetBase::URL_CODE;
		break;
	case LFUN_QUOTE:
		// always allow this, since we will inset a raw quote
		// if an inset is not allowed.
		break;
	case LFUN_HYPHENATION:
	case LFUN_LIGATURE_BREAK:
	case LFUN_HFILL:
	case LFUN_MENU_SEPARATOR:
	case LFUN_LDOTS:
	case LFUN_END_OF_SENTENCE:
		code = InsetBase::SPECIALCHAR_CODE;
		break;
	case LFUN_SPACE_INSERT:
		// slight hack: we know this is allowed in math mode
		if (cur.inTexted())
			code = InsetBase::SPACE_CODE;
		break;

#ifdef WITH_WARNINGS
#warning This LFUN is not used anymore and should be nuked (JMarc 29/10/2005)
#endif
#if 0
	case LFUN_INSET_DIALOG_SHOW: {
		InsetBase * inset = cur.nextInset();
		enable = inset;
		if (inset) {
			code = inset->lyxCode();
			if (!(code == InsetBase::INCLUDE_CODE
				|| code == InsetBase::BIBTEX_CODE
				|| code == InsetBase::FLOAT_LIST_CODE
				|| code == InsetBase::TOC_CODE))
				enable = false;
		}
		break;
	}
#endif

	case LFUN_INSET_MODIFY:
		// We need to disable this, because we may get called for a
		// tabular cell via
		// InsetTabular::getStatus() -> InsetText::getStatus()
		// and we don't handle LFUN_INSET_MODIFY.
		enable = false;
		break;

	case LFUN_EMPH:
		flag.setOnOff(font.emph() == LyXFont::ON);
		return true;

	case LFUN_NOUN:
		flag.setOnOff(font.noun() == LyXFont::ON);
		return true;

	case LFUN_BOLD:
		flag.setOnOff(font.series() == LyXFont::BOLD_SERIES);
		return true;

	case LFUN_SANS:
		flag.setOnOff(font.family() == LyXFont::SANS_FAMILY);
		return true;

	case LFUN_ROMAN:
		flag.setOnOff(font.family() == LyXFont::ROMAN_FAMILY);
		return true;

	case LFUN_CODE:
		flag.setOnOff(font.family() == LyXFont::TYPEWRITER_FAMILY);
		return true;

	case LFUN_CUT:
	case LFUN_COPY:
		enable = cur.selection();
		break;

	case LFUN_PASTE:
		enable = lyx::cap::numberOfSelections() > 0;
		break;

	case LFUN_INSET_DISSOLVE:
		enable = !isMainText() && cur.inset().nargs() == 1;
		break;

	case LFUN_DELETE_WORD_FORWARD:
	case LFUN_DELETE_WORD_BACKWARD:
	case LFUN_DELETE_LINE_FORWARD:
	case LFUN_WORDRIGHT:
	case LFUN_WORDLEFT:
	case LFUN_RIGHT:
	case LFUN_RIGHTSEL:
	case LFUN_LEFT:
	case LFUN_LEFTSEL:
	case LFUN_UP:
	case LFUN_UPSEL:
	case LFUN_DOWN:
	case LFUN_DOWNSEL:
	case LFUN_UP_PARAGRAPHSEL:
	case LFUN_DOWN_PARAGRAPHSEL:
	case LFUN_PRIORSEL:
	case LFUN_NEXTSEL:
	case LFUN_HOMESEL:
	case LFUN_ENDSEL:
	case LFUN_WORDRIGHTSEL:
	case LFUN_WORDLEFTSEL:
	case LFUN_WORDSEL:
	case LFUN_UP_PARAGRAPH:
	case LFUN_DOWN_PARAGRAPH:
	case LFUN_PRIOR:
	case LFUN_NEXT:
	case LFUN_HOME:
	case LFUN_END:
	case LFUN_BREAKLINE:
	case LFUN_DELETE:
	case LFUN_DELETE_SKIP:
	case LFUN_BACKSPACE:
	case LFUN_BACKSPACE_SKIP:
	case LFUN_BREAKPARAGRAPH:
	case LFUN_BREAKPARAGRAPHKEEPLAYOUT:
	case LFUN_BREAKPARAGRAPH_SKIP:
	case LFUN_PARAGRAPH_SPACING:
	case LFUN_INSET_INSERT:
	case LFUN_UPCASE_WORD:
	case LFUN_LOWCASE_WORD:
	case LFUN_CAPITALIZE_WORD:
	case LFUN_TRANSPOSE_CHARS:
	case LFUN_GETXY:
	case LFUN_SETXY:
	case LFUN_GETFONT:
	case LFUN_GETLAYOUT:
	case LFUN_LAYOUT:
	case LFUN_PASTESELECTION:
	case LFUN_DATE_INSERT:
	case LFUN_SELFINSERT:
	case LFUN_INSERT_LINE:
	case LFUN_INSERT_PAGEBREAK:
	case LFUN_MATH_DISPLAY:
	case LFUN_MATH_IMPORT_SELECTION:
	case LFUN_MATH_MODE:
	case LFUN_MATH_MACRO:
	case LFUN_INSERT_MATH:
	case LFUN_INSERT_MATRIX:
	case LFUN_MATH_DELIM:
	case LFUN_MATH_BIGDELIM:
	case LFUN_SUBSCRIPT:
	case LFUN_SUPERSCRIPT:
	case LFUN_DEFAULT:
	case LFUN_UNDERLINE:
	case LFUN_FONT_SIZE:
	case LFUN_LANGUAGE:
	case LFUN_FREEFONT_APPLY:
	case LFUN_FREEFONT_UPDATE:
	case LFUN_LAYOUT_PARAGRAPH:
	case LFUN_PARAGRAPH_UPDATE:
	case LFUN_UMLAUT:
	case LFUN_CIRCUMFLEX:
	case LFUN_GRAVE:
	case LFUN_ACUTE:
	case LFUN_TILDE:
	case LFUN_CEDILLA:
	case LFUN_MACRON:
	case LFUN_DOT:
	case LFUN_UNDERDOT:
	case LFUN_UNDERBAR:
	case LFUN_CARON:
	case LFUN_SPECIAL_CARON:
	case LFUN_BREVE:
	case LFUN_TIE:
	case LFUN_HUNG_UMLAUT:
	case LFUN_CIRCLE:
	case LFUN_OGONEK:
	case LFUN_ACCEPT_CHANGE:
	case LFUN_REJECT_CHANGE:
	case LFUN_THESAURUS_ENTRY:
	case LFUN_PARAGRAPH_APPLY:
	case LFUN_ESCAPE:
	case LFUN_ENDBUF:
	case LFUN_BEGINNINGBUF:
	case LFUN_BEGINNINGBUFSEL:
	case LFUN_ENDBUFSEL:
		// these are handled in our dispatch()
		enable = true;
		break;

	default:
		return false;
	}

	if (code != InsetBase::NO_CODE
	    && (cur.empty() || !cur.inset().insetAllowed(code)))
		enable = false;

	flag.enabled(enable);
	return true;
}
