/**
 * \file math_biginset.C
 * This file is part of LyX, the document processor.
 * Licence details can be found in the file COPYING.
 *
 * \author Andr Pnitz
 *
 * Full author contact details are available in file CREDITS.
 */

#include <config.h>

#include "math_biginset.h"
#include "math_support.h"
#include "math_mathmlstream.h"
#include "math_streamstr.h"

#include "support/lstrings.h"


using std::string;
using std::auto_ptr;


MathBigInset::MathBigInset(string const & name, string const & delim)
	: name_(name), delim_(delim)
{}


string MathBigInset::name() const
{
	return name_;
}


auto_ptr<InsetBase> MathBigInset::doClone() const
{
	return auto_ptr<InsetBase>(new MathBigInset(*this));
}


MathBigInset::size_type MathBigInset::size() const
{
	// order: big Big bigg Bigg biggg Biggg
	//        0   1   2    3    4     5
	return name_[0] == 'B' ?
		2 * (name_.size() - 4) + 1:
		2 * (name_.size() - 4);
}


double MathBigInset::increase() const
{
	// The formula used in amsmath.sty is
	// 1.2 * (1.0 + size() * 0.5) - 1.0.
	// We use a smaller step and a bigger offset because our base size
	// is different.
	return (size() + 1) * 0.3;
}


void MathBigInset::metrics(MetricsInfo & mi, Dimension & dim) const
{
	double const h = mathed_char_ascent(mi.base.font, 'I');
	double const f = increase();
	dim_.wid = 6;
	dim_.asc = int(h + f * h);
	dim_.des = int(f * h);
	dim = dim_;
}


void MathBigInset::draw(PainterInfo & pi, int x, int y) const
{
	// mathed_draw_deco does not use the leading backslash, so remove it
	// (but don't use ltrim if this is the backslash delimiter).
	// Replace \| by \Vert (equivalent in LaTeX), since mathed_draw_deco
	// would treat it as |.
	string const delim = (delim_ == "\\|") ? "Vert" :
		(delim_ == "\\\\") ? "\\" : lyx::support::ltrim(delim_, "\\");
	mathed_draw_deco(pi, x + 1, y - dim_.ascent(), 4, dim_.height(),
	                 delim);
	setPosCache(pi, x, y);
}


void MathBigInset::write(WriteStream & os) const
{
	os << '\\' << name_ << delim_;
	if (delim_[0] == '\\')
		os.pendingSpace(true);
}


void MathBigInset::normalize(NormalStream & os) const
{
	os << '[' << name_ << ' ' <<  delim_ << ']';
}


void MathBigInset::infoize2(std::ostream & os) const
{
	os << name_;
}


bool MathBigInset::isBigInsetDelim(string const & delim)
{
	// mathed_draw_deco must handle these
	static char const * const delimiters[] = {
		"(", ")", "\\{", "\\}", "\\lbrace", "\\rbrace", "[", "]",
		"|", "/", "\\slash", "\\|", "\\vert", "\\Vert", "'",
		"\\\\", "\\backslash",
		"\\langle", "\\lceil", "\\lfloor",
		"\\rangle", "\\rceil", "\\rfloor",
		"\\downarrow", "\\Downarrow",
		"\\uparrow", "\\Uparrow",
		"\\updownarrow", "\\Updownarrow", ""
	};
	return (lyx::support::findToken(delimiters, delim) >= 0);
}
