/**
 * \file insetlabel.C
 * This file is part of LyX, the document processor.
 * Licence details can be found in the file COPYING.
 *
 * \author Lars Gullik Bjnnes
 *
 * Full author contact details are available in file CREDITS.
 */

#include <config.h>

#include "insetlabel.h"

#include "buffer.h"
#include "BufferView.h"
#include "dispatchresult.h"
#include "funcrequest.h"
#include "InsetList.h"
#include "lyxtext.h"
#include "paragraph.h"
#include "pariterator.h"
#include "sgml.h"

#include "frontends/LyXView.h"

#include "support/lstrings.h"
#include "support/lyxalgo.h"
#include "support/std_ostream.h"

using lyx::support::escape;

using std::string;
using std::ostream;
using std::vector;


InsetLabel::InsetLabel(InsetCommandParams const & p)
	: InsetCommand(p, "label")
{}


std::auto_ptr<InsetBase> InsetLabel::doClone() const
{
	return std::auto_ptr<InsetBase>(new InsetLabel(params()));
}


void InsetLabel::getLabelList(Buffer const &, std::vector<string> & list) const
{
	list.push_back(getContents());
}


string const InsetLabel::getScreenLabel(Buffer const &) const
{
	return getContents();
}


void InsetLabel::doDispatch(LCursor & cur, FuncRequest & cmd)
{
	switch (cmd.action) {

	case LFUN_INSET_MODIFY: {
		InsetCommandParams p;
		InsetCommandMailer::string2params("label", cmd.argument, p);
		if (p.getCmdName().empty()) {
			cur.noUpdate();
			break;
		}
		if (p.getContents() != params().getContents())
			cur.bv().buffer()->changeRefsIfUnique(params().getContents(),
						       p.getContents(), InsetBase::REF_CODE);
		setParams(p);
		break;
	}

	default:
		InsetCommand::doDispatch(cur, cmd);
		break;
	}
}


int InsetLabel::latex(Buffer const &, ostream & os,
		      OutputParams const &) const
{
	os << escape(getCommand());
	return 0;
}


int InsetLabel::plaintext(Buffer const &, ostream & os,
		      OutputParams const &) const
{
	os << '<' << getContents()  << '>';
	return 0;
}


int InsetLabel::linuxdoc(Buffer const & buf, ostream & os,
			 OutputParams const & runparams) const
{
	os << "<label id=\"" << sgml::cleanID(buf, runparams, getContents()) << "\" >";
	return 0;
}


int InsetLabel::docbook(Buffer const & buf, ostream & os,
			OutputParams const & runparams) const
{
	os << "<!-- anchor id=\"" << sgml::cleanID(buf, runparams, getContents()) << "\" -->";
	return 0;
}
