//  BMPx - The Dumb Music Player
//  Copyright (C) 2005 BMPx development team.
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License Version 2
//  as published by the Free Software Foundation.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//  --
//
//  The BMPx project hereby grants permission for non-GPL compatible GStreamer
//  plugins to be used and distributed together with GStreamer and BMPx. This
//  permission is above and beyond the permissions granted by the GPL license
//  BMPx is covered by.

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif //HAVE_CONFIG_H

#include <glibmm.h>
#include <glibmm/i18n.h>
#include <gtkmm.h>
#include <libglademm.h>

#include <cstring>
#include <iostream>
#include <sstream>
#include <string>

#include <boost/algorithm/string.hpp>
#include <boost/format.hpp>

#include "util.hh"
#include "util_file.hh"
#include "uri++.hh"
#include "debug.hh"
#include "stock.hh"
#include "paths.hh"
#include "main.hh"
#include "network.hh"

#include "x_mood.hh"
#include "x_vfs.hh"
#include "x_mcsbind.hh"

#include "ui_toolbox.hh"
#include "ui-part-downloads.hh"

namespace
{

#define MOOD_ACTION_REMOVE_SELECTED   "action-remove-selected"
#define MOOD_ACTION_REMOVE_FINISHED   "action-remove-finished"
#define MOOD_ACTION_REMOVE_ABORTED    "action-remove-aborted"
#define MOOD_ACTION_REMOVE_QUEUED     "action-remove-queued"
#define MOOD_ACTION_REMOVE_OFFLINE    "action-remove-offline"

#define MOOD_ACTION_DOWNLOAD_FILES    "action-download-files"
#define MOOD_ACTION_ADD_AS_ALBUM      "action-import-tracks-as-album"

#define MOOD_ACTION_CANCEL            "action-cancel"
#define MOOD_ACTION_RETRY             "action-retry"

  const char *ui_menu_mood = 
  "<ui>"
  "  <menubar name='popup-moo'>"
  "  <menu action='dummy' name='menu-moo-searches'>"
  "    <menuitem action='" MOOD_ACTION_DOWNLOAD_FILES "'/>"
  "  </menu>"
  "  <menu action='dummy' name='menu-moo-downloads'>"
  "    <menuitem action='" MOOD_ACTION_ADD_AS_ALBUM "'/>"
  "    <separator name='sep1'/>"
  "    <menuitem action='" MOOD_ACTION_RETRY "'/>"
  "    <menuitem action='" MOOD_ACTION_CANCEL "'/>"
  "    <separator name='sep2'/>"
  "    <menuitem action='" MOOD_ACTION_REMOVE_ABORTED "'/>"
  "    <menuitem action='" MOOD_ACTION_REMOVE_SELECTED "'/>"
  "    <menuitem action='" MOOD_ACTION_REMOVE_FINISHED "'/>"
  "  </menu>"
  "  <menu action='dummy' name='menu-moo-uploads'>"
  "    <menuitem action='" MOOD_ACTION_RETRY "'/>"
  "    <menuitem action='" MOOD_ACTION_CANCEL "'/>"
  "    <separator name='sep3'/>"
  "    <menuitem action='" MOOD_ACTION_REMOVE_ABORTED "'/>"
  "    <menuitem action='" MOOD_ACTION_REMOVE_SELECTED "'/>"
  "    <menuitem action='" MOOD_ACTION_REMOVE_FINISHED "'/>"
  "  </menu>"
  "  </menubar>"
  "</ui>";

  bool new_tab_block = false;

  void
  log_view (Gtk::Window *window, bool show)
  {
    if (show)
      window->present ();
    else
      window->hide ();
  }

  enum
  {
    TAB_DOWNLOADS,
    TAB_UPLOADS
  };

  void
  calc_position (int& x, int& y, bool& push_in, int x_in, int y_in) 
  {
    push_in = true;
    x = x_in; 
    y = y_in; 
  }

  Gtk::Widget*
  get_popup (Glib::RefPtr<Gtk::UIManager> ui_manager,
             Glib::ustring                path) 
  {
      Gtk::Widget *menu_item;
      menu_item = ui_manager->get_widget (path);
      if ( menu_item )
          return dynamic_cast<Gtk::MenuItem*>(menu_item)->get_submenu ();
      else
          return 0;
  }
}

namespace Bmp
{
  namespace UiPart
  {

    guint
    Downloads::add_ui ()
    {
      return 0;
    };

    Downloads::~Downloads ()
    {
    }

    Downloads::Downloads (Glib::RefPtr<Gnome::Glade::Xml> const& xml, Glib::RefPtr<Gtk::UIManager> ui_manager) 

        : PlaybackSource (PlaybackSource::NONE), UiPart::Base (xml, ui_manager), n_tabs (0)

         
    {
        if (!Network::check_connected())
          {
            m_ref_xml->get_widget ("mood-vbox-main")->set_sensitive (false);
            return;
          }

        // --- Misc Images

        Gtk::Image *i;
        m_ref_xml->get_widget ("i_xfer_up", i);
        i->set (Gtk::StockID (BMP_STOCK_XFER_UP), Gtk::ICON_SIZE_SMALL_TOOLBAR);
        m_ref_xml->get_widget ("i_xfer_down", i);
        i->set (Gtk::StockID (BMP_STOCK_XFER_DOWN), Gtk::ICON_SIZE_SMALL_TOOLBAR);

        // --- UIManager + Actions
        m_actions = Gtk::ActionGroup::create ("MooDriverUI"); 

        m_actions->add ( Gtk::Action::create ("dummy", "dummy", "dummy"));

        m_actions->add ( Gtk::Action::create (MOOD_ACTION_REMOVE_SELECTED,   _("Remove Selected")),
                              sigc::bind (sigc::mem_fun (this, &Bmp::UiPart::Downloads::remove), R_SELECTED));
        m_actions->add ( Gtk::Action::create (MOOD_ACTION_REMOVE_FINISHED,   _("Remove Finished")),
                              sigc::bind (sigc::mem_fun (this, &Bmp::UiPart::Downloads::remove), R_FINISHED));
        m_actions->add ( Gtk::Action::create (MOOD_ACTION_REMOVE_ABORTED,    _("Remove Aborted")),
                              sigc::bind (sigc::mem_fun (this, &Bmp::UiPart::Downloads::remove), R_ABORTED));
        m_actions->add ( Gtk::Action::create (MOOD_ACTION_REMOVE_QUEUED,     _("Remove Queued")),
                              sigc::bind (sigc::mem_fun (this, &Bmp::UiPart::Downloads::remove), R_QUEUED));
        m_actions->add ( Gtk::Action::create (MOOD_ACTION_REMOVE_OFFLINE,    _("Remove Offlined")),
                              sigc::bind (sigc::mem_fun (this, &Bmp::UiPart::Downloads::remove), R_OFFLINE));

        m_actions->add ( Gtk::Action::create (MOOD_ACTION_DOWNLOAD_FILES,
                              Gtk::StockID (BMP_STOCK_DOWNLOAD),
                              _("Download Selected")),
                              sigc::mem_fun (this, &Bmp::UiPart::Downloads::start_transfers));

        m_actions->add ( Gtk::Action::create (MOOD_ACTION_ADD_AS_ALBUM,
                              Gtk::StockID (BMP_STOCK_ADD_MUSIC),
                              _("Add Selected as Album...")),
                              sigc::mem_fun (this, &Bmp::UiPart::Downloads::import_tracks_as_album));

        m_actions->add ( Gtk::Action::create (MOOD_ACTION_CANCEL,
                              Gtk::StockID (GTK_STOCK_CANCEL),
                              _("Cancel")),
                              sigc::mem_fun (this, &Bmp::UiPart::Downloads::cancel));

        m_actions->add ( Gtk::Action::create (MOOD_ACTION_RETRY,
                              Gtk::StockID (GTK_STOCK_REFRESH),
                              _("Retry")),
                              sigc::mem_fun (this, &Bmp::UiPart::Downloads::retry));

        m_ui_manager->insert_action_group (m_actions);
        m_ui_manager->add_ui_from_string (ui_menu_mood);

        m_actions->get_action (MOOD_ACTION_REMOVE_SELECTED)->set_sensitive (false);
        m_actions->get_action (MOOD_ACTION_REMOVE_FINISHED)->set_sensitive (false);
        m_actions->get_action (MOOD_ACTION_REMOVE_ABORTED)->set_sensitive  (false);
        m_actions->get_action (MOOD_ACTION_REMOVE_QUEUED)->set_sensitive   (false);
        m_actions->get_action (MOOD_ACTION_REMOVE_OFFLINE)->set_sensitive  (false);
        m_actions->get_action (MOOD_ACTION_CANCEL)->set_sensitive  (false);
        m_actions->get_action (MOOD_ACTION_RETRY)->set_sensitive  (false);

        char *headers[] =
        {
             N_("User"),
             N_("Filename"), 
             N_("Transferred"), 
             N_("Size"),
             N_("Speed"),
             N_("State"),
             N_("Directory"),
        };

        // --- Downloads View/Store
        m_ref_xml->get_widget ("xfer_view", xfer_view[DOWN]);
        for (unsigned int n = 0; n < G_N_ELEMENTS(headers); ++n)
        {
          Gtk::TreeViewColumn *column = 0;
          if (n == 2)
            {
              Gtk::CellRendererProgress *cell = Gtk::manage ( new Gtk::CellRendererProgress() );
              xfer_view[DOWN]->append_column (_(headers[n]), *cell);
              column = xfer_view[DOWN]->get_column (n);
              column->add_attribute (*cell, "value", n);
              column->set_resizable (false);
              column->set_sort_column (n);
              column->set_expand (false);
              column->set_fixed_width (100);
              column->set_sizing (Gtk::TREE_VIEW_COLUMN_FIXED);
            }
          else
            {
              Gtk::CellRendererText *cell = Gtk::manage ( new Gtk::CellRendererText() );
              xfer_view[DOWN]->append_column (_(headers[n]), *cell);
              column = xfer_view[DOWN]->get_column (n);
              column->add_attribute (*cell, "text", n);
              column->set_resizable (true);
              column->set_sort_column (n);
              column->set_expand (false);
            }
        }
        xfer_store[DOWN] = Gtk::ListStore::create (cr_xfer);
        xfer_store[DOWN]->signal_row_inserted ().connect (sigc::mem_fun (this, &Bmp::UiPart::Downloads::downloads_row_inserted));
        xfer_store[DOWN]->signal_row_deleted  ().connect (sigc::mem_fun (this, &Bmp::UiPart::Downloads::downloads_row_deleted));
        xfer_view[DOWN]->set_model (xfer_store[DOWN]);
        xfer_view[DOWN]->get_selection()->signal_changed().connect (sigc::mem_fun (this, &Downloads::downloads_selection_changed));
        xfer_view[DOWN]->get_selection()->set_mode (Gtk::SELECTION_MULTIPLE);
        xfer_view[DOWN]->signal_event().connect (sigc::mem_fun (this, &Bmp::UiPart::Downloads::popup_xfer_down_menu));

        // --- Uploads View/Store
        m_ref_xml->get_widget ("xfer_view_uploads", xfer_view[UP]);
        for (unsigned int n = 0; n < G_N_ELEMENTS(headers); ++n)
        {
          Gtk::TreeViewColumn *column = 0;
          if (n == 2)
            {
              Gtk::CellRendererProgress *cell = Gtk::manage ( new Gtk::CellRendererProgress() );
              xfer_view[UP]->append_column (_(headers[n]), *cell);
              column = xfer_view[UP]->get_column (n);
              column->add_attribute (*cell, "value", n);
              column->set_resizable (false);
              column->set_sort_column (n);
              column->set_expand (false);
              column->set_fixed_width (100);
              column->set_sizing (Gtk::TREE_VIEW_COLUMN_FIXED);
            }
          else
            {
              Gtk::CellRendererText *cell = Gtk::manage ( new Gtk::CellRendererText() );
              xfer_view[UP]->append_column (_(headers[n]), *cell);
              column = xfer_view[UP]->get_column (n);
              column->add_attribute (*cell, "text", n);
              column->set_resizable (true);
              column->set_sort_column (n);
              column->set_expand (false);
            }
        }
    
        xfer_store[UP] = Gtk::ListStore::create (cr_xfer);
        xfer_store[UP]->signal_row_inserted ().connect (sigc::mem_fun (this, &Bmp::UiPart::Downloads::uploads_row_inserted));
        xfer_store[UP]->signal_row_deleted  ().connect (sigc::mem_fun (this, &Bmp::UiPart::Downloads::uploads_row_deleted));
        xfer_view[UP]->set_model (xfer_store[UP]);
        xfer_view[UP]->get_selection()->signal_changed().connect (sigc::mem_fun (this, &Downloads::uploads_selection_changed));
        xfer_view[UP]->get_selection()->set_mode (Gtk::SELECTION_MULTIPLE);
        xfer_view[UP]->signal_event().connect (sigc::mem_fun (this, &Bmp::UiPart::Downloads::popup_xfer_up_menu));


        m_ref_xml->get_widget ("notebook_transfers", notebook_transfers);
        notebook_transfers->signal_switch_page().connect (sigc::mem_fun (this, &Bmp::UiPart::Downloads::notebook_transfers_switch_page));
        notebook_transfers->set_sensitive (false);

        // Search stuff (connect to moodriver signals, setup widgets)
        m_ref_xml->get_widget ("notebook_searches", notebook_searches);
        notebook_searches->signal_switch_page().connect (sigc::mem_fun (this, &Bmp::UiPart::Downloads::notebook_searches_switch_page));
        notebook_searches->remove_page (); 

        m_ref_xml->get_widget ("e_search", e_search);
        e_search->signal_activate().connect (sigc::mem_fun (this, &Downloads::start_search));
        e_search->set_sensitive (false);

        mood->s_search().connect
              (sigc::mem_fun (this, &Downloads::s_search_callback));
        mood->s_transfer_update().connect
              (sigc::mem_fun (this, &Downloads::s_transfer_update_callback));
        mood->s_search_result().connect
              (sigc::mem_fun (this, &Downloads::s_search_result_callback));
        mood->s_status_message().connect
              (sigc::mem_fun (this, &Downloads::mood_status_message));
        mood->s_login().connect
              (sigc::mem_fun (this, &Downloads::mood_login));
        mood->s_server_state().connect
              (sigc::mem_fun (this, &Downloads::mood_server_state));
        mood->s_disconnected().connect
              (sigc::mem_fun (this, &Downloads::mood_disconnect));
        mood->s_connected().connect
              (sigc::mem_fun (this, &Downloads::mood_connect));


        mood_adjustment = dynamic_cast<Gtk::ScrolledWindow *>(m_ref_xml->get_widget ("mood-sw"))->get_vadjustment();
        
        mcs_bind->bind_entry (dynamic_cast<Gtk::Entry *>(m_ref_xml->get_widget ("mood-hostname")), "mood", "hostname");
        mcs_bind->bind_entry (dynamic_cast<Gtk::Entry *>(m_ref_xml->get_widget ("mood-port")), "mood", "port");
        mcs_bind->bind_entry (dynamic_cast<Gtk::Entry *>(m_ref_xml->get_widget ("mood-password")), "mood", "password");

        m_ref_xml->get_widget ("mood-enable", mood_enable);
        mcs->key_set<bool>("bmp", "enable-mood", false);
        mcs->subscribe ("Preferences", "bmp", "enable-mood",
							sigc::mem_fun(this, &Bmp::UiPart::Downloads::mcs_enable_mood_changed));

        mood_buffer = Gtk::TextBuffer::create ();
        tag_b = mood_buffer->create_tag();
        tag_b->property_weight() = Pango::WEIGHT_BOLD; 
        tag_i = mood_buffer->create_tag();
        tag_i->property_style() = Pango::STYLE_ITALIC; 

        m_ref_xml->get_widget ("mood-textview", mood_textview);
        mood_textview->set_buffer (mood_buffer);

        mcs_bind->bind_toggle_button (mood_enable, "bmp", "enable-mood");

        dynamic_cast<Gtk::Button *>(m_ref_xml->get_widget ("mood-show-log"))->signal_clicked().connect
            (sigc::bind (sigc::ptr_fun (&log_view), dynamic_cast<Gtk::Window *>(m_ref_xml->get_widget ("mood-logview")), true));
        dynamic_cast<Gtk::Button *>(m_ref_xml->get_widget ("mood-hide-log"))->signal_clicked().connect
            (sigc::bind (sigc::ptr_fun (&log_view), dynamic_cast<Gtk::Window *>(m_ref_xml->get_widget ("mood-logview")), false));

    }

    //---------------- Searches 
    void
    Downloads::start_search ()
    {
      mood->search_start (Moo::SEARCH_GLOBAL, e_search->get_text().c_str());
    }

    bool
    Downloads::popup_searches_menu (GdkEvent *ev)
    {
      using namespace Gtk;

      GdkEventButton *event = 0;
      if (ev->type == GDK_BUTTON_PRESS)
        {
          event = reinterpret_cast<GdkEventButton *>(ev);
          if ((event->button == 3) && (event->type == GDK_BUTTON_PRESS))
            {
              Menu *menu = dynamic_cast<Menu *>(get_popup (m_ui_manager, "/popup-moo/menu-moo-searches"));
              menu->popup (event->button, event->time);
              return true;
            }

        } 
      return false;
    }

    void
    Downloads::s_search_callback (std::string name, unsigned int ticket)
    {
      contexts_lock.lock ();
      SearchContextsT::iterator sti = contexts.find (ticket); 
      contexts_lock.unlock ();

      if (sti == contexts.end())
        {
            new_tab_block = true;

            SearchContext *ctx = new SearchContext (name, cr_search, 0); 
            int tab = notebook_searches->append_page (ctx->sw, ctx->hbox);
            ctx->m_tab = tab;

            contexts_lock.lock ();
            contexts.insert (std::make_pair (ticket, ctx));
            contexts_by_tab.insert (std::make_pair (tab, ctx));
            contexts_lock.unlock ();

            ctx->conn1 = ctx->button.signal_clicked().connect
                        (sigc::bind (sigc::mem_fun (this, &Bmp::UiPart::Downloads::cancel_search), tab, ticket)); 
            ctx->conn2 = ctx->treeview.signal_row_activated().connect
                        (sigc::bind (sigc::mem_fun (this, &Bmp::UiPart::Downloads::start_transfer), ctx));
            ctx->conn3 = ctx->treeview.signal_event().connect
                         (sigc::mem_fun (this, &Bmp::UiPart::Downloads::popup_searches_menu));

            new_tab_block = false;
        }
    }

    void
    Downloads::cancel_search (int tab, unsigned int ticket)
    {
      notebook_searches->set_sensitive (false);
      mood->search_cancel (ticket);

      SearchContextsT::iterator sti = contexts.find (ticket); 

      SearchContext *ctx = sti->second; 
      ctx->conn1.disconnect ();
      ctx->conn2.disconnect ();
      ctx->conn3.disconnect ();

      contexts.erase (sti);
      contexts_by_tab.erase (tab);

      notebook_searches->remove_page (tab);

      while (ctx->append)
        while (gtk_events_pending()) gtk_main_iteration ();

      delete ctx;

      notebook_searches->set_sensitive (true);
    }

    void
    Downloads::notebook_searches_switch_page (GtkNotebookPage *p, guint page)
    {
      if (new_tab_block) return;

      contexts_lock.lock ();
      SearchContextsByTabT::iterator sti = contexts_by_tab.find (page);
      SearchContext *ctx = sti->second; 
      ctx->label.set_text (ctx->name);
      contexts_lock.unlock ();
    }

    void
    Downloads::s_search_result_callback (Moo::SearchResult result)
    {
      using namespace Gtk;
      using namespace Glib;

      contexts_lock.lock ();
      SearchContextsT::iterator sti = contexts.find (result.ticket); 
      contexts_lock.unlock ();

      if (sti != contexts.end())
      {
          SearchContext *ctx = sti->second; 

          ustring label_text;

          label_text  .append ("<u>")
                      .append (ctx->name)
                      .append ("</u>");

          ctx->label.set_markup (label_text);

          for (Moo::FileList::const_iterator i = result.files.begin(); i != result.files.end(); ++i)
          {
              using boost::algorithm::split;
              using boost::algorithm::is_any_of;
              using namespace std;

              contexts_lock.lock ();
              if  (contexts.find (result.ticket) == contexts.end())
                {
                  contexts_lock.unlock ();
                  ctx->append = false;
                  break; 
                }
              contexts_lock.unlock ();

              ctx->append = true;

              TreeModel::iterator m_iter = ctx->store->append();
              (*m_iter)[cr_search.username] = result.user;
              (*m_iter)[cr_search.path]     = i->name; 
              (*m_iter)[cr_search.filesize] = i->size; 
              (*m_iter)[cr_search.slot]     = result.slot; 
              (*m_iter)[cr_search.speed]    = result.speed; 
              (*m_iter)[cr_search.queue]    = result.queue; 

              string splitstring = i->name.c_str();
              vector<string> subs;
              split (subs, splitstring, is_any_of ("/\\"));

              (*m_iter)[cr_search.basename] = subs[subs.size()-1];
              (*m_iter)[cr_search.basepath] = i->name.substr (0, i->name.size() - subs[subs.size()-1].size());

              ctx->append = false;
          }
      }
    }

    //---------------- Transfers
    void
    Downloads::notebook_transfers_switch_page (GtkNotebookPage *p, guint page)
    {
      if (notebook_transfers->get_current_page() == TAB_UPLOADS)
      {
        set_selection_based_controls_sensitive (xfer_view[UP]->get_selection()->count_selected_rows());
        actions_set_sensitive (!xfer_store[UP]->children().empty());
      }
      else
      if (notebook_transfers->get_current_page() == TAB_DOWNLOADS)
      {
        set_selection_based_controls_sensitive (xfer_view[DOWN]->get_selection()->count_selected_rows());
        actions_set_sensitive (!xfer_store[DOWN]->children().empty());
      }
    }

    bool
    Downloads::popup_xfer_down_menu (GdkEvent *ev)
    {
      using namespace Gtk;

      GdkEventButton *event = 0;
      if (ev->type == GDK_BUTTON_PRESS)
        {
          event = reinterpret_cast<GdkEventButton *>(ev);
          if ((event->button == 3) && (event->type == GDK_BUTTON_PRESS))
            {
              Menu *menu = dynamic_cast<Menu *>(get_popup (m_ui_manager, "/popup-moo/menu-moo-downloads"));
              m_actions->get_action (MOOD_ACTION_ADD_AS_ALBUM)->set_sensitive (xfer_view[DOWN]->get_selection()->count_selected_rows());
              menu->popup (event->button, event->time);
              return true;
            }
        } 
      return false;
    }

    bool
    Downloads::popup_xfer_up_menu (GdkEvent *ev)
    {
      using namespace Gtk;

      GdkEventButton *event = 0;
      if (ev->type == GDK_BUTTON_PRESS)
        {
          event = reinterpret_cast<GdkEventButton *>(ev);
          if ((event->button == 3) && (event->type == GDK_BUTTON_PRESS))
            {
              Menu *menu = dynamic_cast<Menu *>(get_popup (m_ui_manager, "/popup-moo/menu-moo-uploads"));
              menu->popup (event->button, event->time);
              return true;
            }
        } 
      return false;
    }

    void
    Downloads::import_tracks_as_album ()
    {
      using namespace Glib;
      using namespace Gtk;
      using namespace Moo;
      using namespace std;

      Bmp::VUri list;
      typedef std::vector<XFERKey> XFERKeysT;
      XFERKeysT keys;

      TreeSelection::ListHandle_Path paths = xfer_view[DOWN]->get_selection()->get_selected_rows ();
      string download_basepath = mood->get_key ("transfers", "download-dir");

      for (TreeSelection::ListHandle_Path::const_iterator path  = paths.begin (),
                                                          end   = paths.end ();
                                                          path != end; ++path) 
        {
            TreeModel::iterator x_iter = xfer_store[DOWN]->get_iter (*path);
            TransferState state = (*x_iter)[cr_xfer.state];
            if (state != XFER_STATE_FINISHED)
              {
                Gtk::MessageDialog dialog (_("You can only import tracks as an Album if all transfers you have selected are Finished!"),
                                              true, Gtk::MESSAGE_ERROR, Gtk::BUTTONS_OK, true);
                dialog.run ();
                return;
              }

            ustring basename = (*x_iter)[cr_xfer.basename];
            ustring user = (*x_iter)[cr_xfer.user];
            ustring path = (*x_iter)[cr_xfer.path];

            XFERKey key (user.c_str(), path.c_str());
            list.push_back (Glib::filename_to_uri (Glib::build_filename (download_basepath, basename)));
        }
      s_request_import_.emit (list); 
      remove (R_SELECTED);  
    }

    void
    Downloads::start_transfers ()
    {
      using namespace Glib;
      using namespace Gtk;
      using namespace Moo;

      SearchContextsByTabT::iterator sti = contexts_by_tab.find (notebook_searches->get_current_page ()); 
      SearchContext *ctx = sti->second; 

      TreeSelection::ListHandle_Path paths = ctx->treeview.get_selection()->get_selected_rows ();
      for (TreeSelection::ListHandle_Path::const_iterator path  = paths.begin (),
                                                          end   = paths.end ();
                                                          path != end; ++path) 
        {
          TreeModel::const_iterator m_iter = ctx->store->get_iter (*path);
          ustring username = (*m_iter)[cr_search.username];
          ustring path = (*m_iter)[cr_search.path];
          XFERKey key (username.c_str(), path.c_str());
          mood->download_start (key);
        }
    }
    
    void
    Downloads::start_transfer (const Gtk::TreeModel::Path&    treepath,
                               Gtk::TreeViewColumn           *column,
                               SearchContext                 *ctx) 
    {
      using namespace Gtk;
      using namespace Glib;
      using namespace Moo;
      using namespace boost;
      using namespace std;
      using boost::algorithm::split;
      using boost::algorithm::is_any_of;

      ctx->treeview.get_selection()->unselect_all ();

      TreeModel::iterator m_iter  = ctx->store->get_iter (treepath);
      ustring username            = (*m_iter)[cr_search.username];
      ustring path                = (*m_iter)[cr_search.path];

      XFERKey key (username.c_str(), path.c_str());
      mood->download_start (key);
    }

    void
    Downloads::s_transfer_update_callback (Moo::XFERKey key, Moo::Transfer xfer)
    {
      using namespace Gtk;
      using namespace Glib;
      using namespace Moo;
      using namespace boost;
      using namespace std;
      using boost::algorithm::split;
      using boost::algorithm::is_any_of;

      static format frate ("%.2f KB/s");
      static format mbyte ("%.2f MB");
      static format kbyte ("%.2f KB");
      static format bytes ("%u Bytes");

      const double MBYTE = 1048576;
      const double KBYTE = 1024;

      int map = xfer.upload ? 1:0;

      XFERMapT::iterator i;
      i = xfermap_schedule_remove[map].find (key);
      if (i != xfermap_schedule_remove[map].end())
        {
          xfermap_schedule_remove[map].erase (i);
          xfermap[map].erase (key);
          return;
        }
     
      unsigned int size = xfer.size;

      bool new_xfer = false;
      i = xfermap[map].find (key);
      TreeModel::iterator x_iter;
      if (i == xfermap[map].end())
        {
          x_iter = xfer_store[map]->append ();
          ustring username = xfer.user; 
          ustring path = xfer.path; 
          XFERKey key (username.c_str(), path.c_str());
          xfermap[map].insert (make_pair (key, x_iter));
          new_xfer = true;
          (*x_iter)[cr_xfer.user] = xfer.user; 
          (*x_iter)[cr_xfer.path] = xfer.path; 
          (*x_iter)[cr_xfer.size] = size; 
        }
      else
        {
          x_iter = i->second; 
        }

      (*x_iter)[cr_xfer.position]   = xfer.position;
      (*x_iter)[cr_xfer.rate]       = (frate % (xfer.rate/1024.)).str();
      (*x_iter)[cr_xfer.state_str]  = Driver::xfer_status_description (TransferState (xfer.state));
      (*x_iter)[cr_xfer.state]      = TransferState (xfer.state);

      if (xfer.position > 0)
        {
          (*x_iter)[cr_xfer.percentage] = int ((xfer.position*100) / xfer.size);
        }
      else
        {
          (*x_iter)[cr_xfer.percentage] = 0; 
        }

      string size_str; 
      if (size > MBYTE)
        {
          size_str = (mbyte % (size / MBYTE)).str();
        }
      else
      if (size > KBYTE)
        {
          size_str = (kbyte % (size / KBYTE)).str();
        }
      else
        {
          size_str = (bytes % size).str();
        }

      (*x_iter)[cr_xfer.size_str] = size_str; 

      string splitstring = xfer.path.c_str();
      vector<string> subs;
      split (subs, splitstring, is_any_of ("/\\"));
      (*x_iter)[cr_xfer.basename] = subs[subs.size()-1];
      (*x_iter)[cr_xfer.basepath] = xfer.path.substr (0, xfer.path.size() - subs[subs.size()-1].size());

#if 0
      if ((xfer.state == XFER_STATE_FINISHED) && mcs->key_get<bool>("mood","enqueue-finished") && !new_xfer && !xfer.upload)
      {
      }
#endif
    }

    void
    Downloads::retry ()
    {
      using namespace Gtk;
      using namespace Moo;

      for (TreeNodeChildren::iterator x_iter = xfer_store[DOWN]->children().begin(); x_iter != xfer_store[DOWN]->children().end(); ++x_iter)
      {
        TransferState state = (*x_iter)[cr_xfer.state]; 
        if ((xfer_view[DOWN]->get_selection()->is_selected (x_iter)) && ( state != XFER_STATE_RUNNING ))
          {
            Glib::ustring user = (*x_iter)[cr_xfer.user];
            Glib::ustring path = (*x_iter)[cr_xfer.path];
            XFERKey key (user.c_str(), path.c_str());
            mood->download_start (key);
          }
      }
    }

    void
    Downloads::remove  (RemoveAction action)
    {
        using namespace Moo;
        using namespace std;
        using namespace Gtk;
        using namespace Glib;

        typedef vector<TreeModel::RowReference> ModelRowRefsT;
        ModelRowRefsT refs;

        int map = notebook_transfers->get_current_page();

        for (TreeNodeChildren::iterator x_iter = xfer_store[map]->children().begin(); x_iter != xfer_store[map]->children().end(); ++x_iter)
        {
          refs.push_back (Gtk::TreeModel::RowReference (xfer_store[map], Gtk::TreeModel::Path (x_iter)));
        }

        RefPtr<Gtk::TreeView::Selection> selection = xfer_view[map]->get_selection ();
        for (ModelRowRefsT::iterator i = refs.begin(); i != refs.end(); ++i)
        {
            Gtk::TreeModel::iterator x_iter = xfer_store[map]->get_iter (i->get_path());

            TransferState state = (*x_iter)[cr_xfer.state];
            ustring user        = (*x_iter)[cr_xfer.user];
            ustring path        = (*x_iter)[cr_xfer.path];

            XFERKey key (user.c_str(), path.c_str());

            bool erase = false;

            switch (action)
            {
              case R_SELECTED:
              {
                if (selection->is_selected (x_iter) && (state != XFER_STATE_RUNNING))
                    erase = true;
                break;
              }
         
              case R_FINISHED:
              {
                if (state == XFER_STATE_FINISHED)
                  erase = true;
                break;
              }

              case R_ABORTED:
              {
                if (state == XFER_STATE_ABORTED)
                  erase = true;
                break;
              }

              case R_QUEUED:
              {
                if (state == XFER_STATE_QUEUED)
                  erase = true;
                break;
              }

              case R_OFFLINE:
              {
                if (state == XFER_STATE_UNABLE_TO_CONNECT)
                  erase = true;
                break;
              }
            }

          if (erase)
          {
            xfermap_schedule_remove[map].insert (make_pair (key, x_iter));
            xfer_store[map]->erase (x_iter);
            mood->transfer_abort  (key, bool(map?true:false));
            mood->transfer_remove (key, bool(map?true:false));
          }
        }
    }

    void
    Downloads::cancel ()
    {
      using namespace Gtk;

      int map = notebook_transfers->get_current_page();

      for (TreeNodeChildren::iterator x_iter = xfer_store[map]->children().begin(); x_iter != xfer_store[map]->children().end(); ++x_iter)
      {
        if (xfer_view[map]->get_selection()->is_selected (x_iter))
          {
            Glib::ustring user = (*x_iter)[cr_xfer.user];
            Glib::ustring path = (*x_iter)[cr_xfer.path];
            Moo::XFERKey key (user.c_str(), path.c_str());
            mood->transfer_abort (key, bool(map?true:false));
          }
      }
    }

    void 
    Downloads::actions_set_sensitive (bool sensitive)
    {
      m_actions->get_action (MOOD_ACTION_REMOVE_FINISHED)->set_sensitive (sensitive);
      m_actions->get_action (MOOD_ACTION_REMOVE_ABORTED)->set_sensitive  (sensitive);
      m_actions->get_action (MOOD_ACTION_REMOVE_QUEUED)->set_sensitive   (sensitive);
      m_actions->get_action (MOOD_ACTION_REMOVE_OFFLINE)->set_sensitive  (sensitive);
    }

    void
    Downloads::set_selection_based_controls_sensitive (bool sensitive)
    {
      m_actions->get_action (MOOD_ACTION_CANCEL)->set_sensitive  (sensitive);

      if (notebook_transfers->get_current_page() == TAB_DOWNLOADS)
          m_actions->get_action (MOOD_ACTION_RETRY)->set_sensitive (sensitive);
      else
          m_actions->get_action (MOOD_ACTION_RETRY)->set_sensitive (false);

      m_actions->get_action (MOOD_ACTION_REMOVE_SELECTED)->set_sensitive (sensitive);
    }

    // Selection changing + row insertion/removal for _downloads_
    void
    Downloads::downloads_selection_changed ()
    {
      if (notebook_transfers->get_current_page() == TAB_DOWNLOADS)
        set_selection_based_controls_sensitive (xfer_view[DOWN]->get_selection()->count_selected_rows());

      if (xfer_view[DOWN]->get_selection()->count_selected_rows() == 1)
      {
        PathList paths = xfer_view[DOWN]->get_selection()->get_selected_rows ();
        Gtk::TreeModel::iterator m_iter = xfer_store[DOWN]->get_iter (paths[0]);
        if ((*m_iter)[cr_xfer.state] == Moo::XFER_STATE_FINISHED)
          {
            m_caps = Caps (m_caps | PlaybackSource::CAN_PLAY);
            m_caps = Caps (m_caps | PlaybackSource::CAN_SEEK);
            s_caps_.emit (m_caps);
          }
        else
          {
            m_caps = Caps (m_caps & ~PlaybackSource::CAN_PLAY);
            m_caps = Caps (m_caps & ~PlaybackSource::CAN_SEEK);
            s_caps_.emit (m_caps);
          }
      }
      else
      {
        m_caps = Caps (m_caps & ~PlaybackSource::CAN_PLAY);
        m_caps = Caps (m_caps & ~PlaybackSource::CAN_SEEK);
        s_caps_.emit (m_caps);
      }
    }

    Glib::ustring
    Downloads::get_uri ()
    {
      PathList paths = xfer_view[DOWN]->get_selection()->get_selected_rows ();
      Gtk::TreeModel::iterator m_iter = xfer_store[DOWN]->get_iter (paths[0]);
      Glib::ustring basename ((*m_iter)[cr_xfer.basename]);
      std::string download_dir = mood->get_key ("transfers", "download-dir");
      SimpleTrackInfo sti;
      sti.title = basename;
      s_track_info_.emit (Glib::Markup::escape_text (basename), sti); 
      return Glib::filename_to_uri (Glib::build_filename (download_dir, basename.c_str()));
    }

    void
    Downloads::downloads_row_inserted (const Gtk::TreeModel::Path& path, const Gtk::TreeModel::iterator& iter)
    {
      if (notebook_transfers->get_current_page() == TAB_DOWNLOADS)
        actions_set_sensitive (!xfer_store[DOWN]->children().empty());
    }

    void
    Downloads::downloads_row_deleted (const Gtk::TreeModel::Path& path)
    {
      if (notebook_transfers->get_current_page() == TAB_DOWNLOADS)
        actions_set_sensitive (!xfer_store[DOWN]->children().empty());
    }

    // Selection changing + row insertion/removal for _uploads_
    void
    Downloads::uploads_selection_changed ()
    {
      if (notebook_transfers->get_current_page() == TAB_UPLOADS)
        set_selection_based_controls_sensitive (xfer_view[UP]->get_selection()->count_selected_rows());
    }

    void
    Downloads::uploads_row_inserted (const Gtk::TreeModel::Path& path, const Gtk::TreeModel::iterator& iter)
    {
      if (notebook_transfers->get_current_page() == TAB_UPLOADS)
        actions_set_sensitive (!xfer_store[UP]->children().empty());
    }

    void
    Downloads::uploads_row_deleted (const Gtk::TreeModel::Path& path)
    {
      if (notebook_transfers->get_current_page() == TAB_UPLOADS)
        actions_set_sensitive (!xfer_store[UP]->children().empty());
    }

    //---------------- SearchContext class
    Downloads::SearchContext::SearchContext (std::string const& search_name, const Gtk::TreeModel::ColumnRecord& record, int tab)
      : m_tab (tab), name (search_name), append (false)
    {
      image.set (Gtk::Stock::CLOSE, Gtk::ICON_SIZE_SMALL_TOOLBAR);
      image.set_size_request (16,16);

      button.add (image);
      button.set_size_request (22,18);
      button.show_all();

      label.set_text (name);
      label.show_all ();

      hbox.pack_start (label, false, false, 1);
      hbox.pack_start (button, false, false, 1);
      hbox.property_homogeneous() = false;
      hbox.property_spacing() = 2;
      hbox.show_all ();

      char *headers[] = {
           N_("User"),
           N_("Filename"), 
           N_("Filesize"), 
           N_("Slot"),
           N_("Speed"),
           N_("Queue Size"),
           N_("Directory"),
      };

    for (unsigned int n = 0; n < G_N_ELEMENTS(headers); ++n)
      {
        Gtk::TreeViewColumn   *column = 0;
        Gtk::CellRendererText *cell = Gtk::manage ( new Gtk::CellRendererText() );
        treeview.append_column (_(headers[n]), *cell);
        column = treeview.get_column  (n);
        column->add_attribute         (*cell, "text", n);
        column->set_resizable         (true);
        column->set_sort_column       (n);
        column->set_expand            (true);
      }

      store = Gtk::ListStore::create (record);
      treeview.set_model (store);
      treeview.get_selection()->set_mode (Gtk::SELECTION_MULTIPLE);
      sw.add (treeview);
      sw.show_all ();
      sw.set_policy (Gtk::POLICY_AUTOMATIC, Gtk::POLICY_ALWAYS);
    }

    void
    Downloads::mcs_enable_mood_changed  (MCS_CB_DEFAULT_SIGNATURE)
    {
      bool enabled = boost::get<bool>(value); 

      if (enabled)
        {
          mood_enable->set_sensitive (false);
          mood->credentials (mcs->key_get<std::string>("mood", "hostname"),
                             mcs->key_get<std::string>("mood", "port"),
                             mcs->key_get<std::string>("mood", "password"));
          if (!mood->connect ())
            {
              mood_buffer->insert_with_tag (mood_buffer->end(), Util::get_timestr (time (NULL), 0), tag_b);
              mood_buffer->insert_with_tag (mood_buffer->end(), " (!!) ", tag_b);
              mood_buffer->insert (mood_buffer->end(), _("Unable to connect to museek daemon at "));
              mood_buffer->insert (mood_buffer->end(), mcs->key_get<std::string>("mood","hostname"));
              mood_buffer->insert (mood_buffer->end(), ":");
              mood_buffer->insert (mood_buffer->end(), mcs->key_get<std::string>("mood","port"));
              mood_buffer->insert (mood_buffer->end(), "\n");
              mood_adjustment->set_value ( mood_adjustment->get_upper() - mood_adjustment->get_page_size() );
              gtk_adjustment_value_changed (GTK_ADJUSTMENT (mood_adjustment->gobj()));
              mood_enable->set_active (false);
            }
        }
      else
        {
          mood->disconnect ();
          e_search->set_sensitive (false);
          notebook_transfers->set_sensitive (false);
          xfer_store[UP]->clear();
          xfer_store[DOWN]->clear();
        }
    }

    void
    Downloads::mood_connect ()
    {
      mood_enable->set_sensitive (true);
    }
   
    void
    Downloads::mood_disconnect ()
    {
      mood_enable->set_sensitive (true);
      mood_enable->set_active (false);

      mood_buffer->insert_with_tag (mood_buffer->end(), Util::get_timestr (time (NULL), 0), tag_b);
      mood_buffer->insert_with_tag (mood_buffer->end(), " (!!) ", tag_b);
      mood_buffer->insert (mood_buffer->end(), _("Disconnected from the museek daemon.\n")); 
      mood_adjustment->set_value ( mood_adjustment->get_upper() - mood_adjustment->get_page_size() );
    }

    void
    Downloads::mood_status_message (bool server_peer, std::string message) 
    {
      mood_buffer->insert_with_tag (mood_buffer->end(), Util::get_timestr (time (NULL), 0), tag_b);
      mood_buffer->insert_with_tag (mood_buffer->end(), " (II) ", tag_b);

      mood_buffer->insert (mood_buffer->end(), server_peer?"PEER:\n":"SERVER:\n");
      mood_buffer->insert_with_tag (mood_buffer->end(), message, tag_i);
      mood_buffer->insert (mood_buffer->end(), "\n");
      mood_adjustment->set_value ( mood_adjustment->get_upper() - mood_adjustment->get_page_size() );
    }

    void
    Downloads::mood_server_state (bool connected, std::string user)
    {
      e_search->set_sensitive (connected);
      notebook_transfers->set_sensitive (connected);

      mood_buffer->insert_with_tag (mood_buffer->end(), Util::get_timestr (time (NULL), 0), tag_b);
      mood_buffer->insert_with_tag (mood_buffer->end(), " (II) ", tag_b);

      if (connected)
        {
          mood_buffer->insert (mood_buffer->end(), "Connected as "); 
          mood_buffer->insert (mood_buffer->end(), user); 
          mood_buffer->insert (mood_buffer->end(), ".\n"); 
          mood_adjustment->set_value ( mood_adjustment->get_upper() - mood_adjustment->get_page_size() );
        }
      else
        {
          mood_buffer->insert (mood_buffer->end(), "Disconnected.\n");
          mood_adjustment->set_value ( mood_adjustment->get_upper() - mood_adjustment->get_page_size() );
        }
    }

    void
    Downloads::mood_login (bool ok, std::string message)
    {
      if (ok)
        {
          mood_buffer->insert_with_tag (mood_buffer->end(), Util::get_timestr (time (NULL), 0), tag_b);
          mood_buffer->insert_with_tag (mood_buffer->end(), " (II) ", tag_b);

          mood_buffer->insert (mood_buffer->end(), "Login OK.\n"); 
          mood_adjustment->set_value ( mood_adjustment->get_upper() - mood_adjustment->get_page_size() );
        }
      else
        {
          mood_buffer->insert_with_tag (mood_buffer->end(), Util::get_timestr (time (NULL), 0), tag_b);
          mood_buffer->insert_with_tag (mood_buffer->end(), " (WW) ", tag_b);

          mood_buffer->insert (mood_buffer->end(), "Login NOT OK: ");
          mood_buffer->insert (mood_buffer->end(), message); 
          mood_buffer->insert (mood_buffer->end(), "\n"); 
          mood_adjustment->set_value ( mood_adjustment->get_upper() - mood_adjustment->get_page_size() );
        }
    }

  } // namespace UiPart
} // namespace Bmp
