//  BMPx - The Dumb Music Player
//  Copyright (C) 2005-2006 BMPx development team.
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License Version 2
//  as published by the Free Software Foundation.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//  --
//
//  The BMPx project hereby grants permission for non-GPL compatible GStreamer
//  plugins to be used and distributed together with GStreamer and BMPx. This
//  permission is above and beyond the permissions granted by the GPL license
//  BMPx is covered by.

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif //HAVE_CONFIG_H

#include <cstring>
#include <glibmm.h>

#include "main.hh"
#include "network.hh"

#include <dbus/dbus.h>
#include <dbus/dbus-glib.h>
#include <bmp/dbus.hh>

#include "ui_toolbox.hh"
#include "playershell.hh"
#include "service_core.hh"

#define TYPE_C_SERVICE_CORE (CServiceCore::get_type ())
#define C_SERVICE_CORE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), TYPE_C_SERVICE_CORE, CServiceCore))

namespace Bmp
{
  char const* mpris_identity = "BMP2";

  struct CServiceCoreClass
  {
    GObjectClass parent;
  };

  struct ServiceCore::CServiceCore
  {
    GObject parent;
    ServiceCore *core;

    enum
      {
        SIGNAL_STARTUP_COMPLETE,
        SIGNAL_SHUTDOWN_COMPLETE,
        SIGNAL_QUIT,
        N_SIGNALS,
      };

    static guint signals[N_SIGNALS];

    static gpointer parent_class;

    static GType
    get_type ();

    static CServiceCore *
    create (ServiceCore & core);

    static void
    class_init (gpointer klass,
                gpointer class_data);

    static GObject *
    constructor (GType                   type,
                 guint                   n_construct_properties,
                 GObjectConstructParam * construct_properties);

    static gboolean
    identity (CServiceCore * self,
              char **        identity,
              GError **      error);

    static gboolean
    ui_raise (CServiceCore * self,
              GError **      error);

    static gboolean
    volume_set (CServiceCore * self,
                int            volume,
                GError **      error);

    static gboolean
    volume_get (CServiceCore * self,
                int *          volume,
                GError **      error);

    static gboolean
    startup (CServiceCore * self,
             int            no_network,
             GError **      error);

    static gboolean
    playlist_add_uri_list (CServiceCore * self,
                           char **        uris,
                           gboolean       playback,
                           GError **      error);

    static gboolean
    last_fm_play_uri (CServiceCore * self,
                      char *         uri,
                      GError **      error);

    static gboolean
    get_current_source (CServiceCore * self,
                        int *          source,
                        GError **      error);

    static gboolean
    get_source_caps (CServiceCore * self,
                     int            source,
                     int *          caps,
                     GError **      error);

    static gboolean
    get_metadata_from_source (CServiceCore * self,
                              int            source,
                              GHashTable **  metadata,
                              GError **      error);

    static gboolean
    go_next (CServiceCore * self,
             GError **      error);

    static gboolean
    go_prev (CServiceCore * self,
             GError **      error);

    static gboolean
    pause  (CServiceCore * self,
             GError **      error);

    static gboolean
    play   (CServiceCore * self,
            GError **      error);

    static gboolean
    stop   (CServiceCore * self,
            GError **      error);


    // Used Internally by ServiceCore
    static void
    initialize_dbus (CServiceCore * self);

    static void
    startup_complete (CServiceCore * self);
  
    static void
    shutdown_complete (CServiceCore * self);

    static void
    quit (CServiceCore * self);
  };

  gpointer ServiceCore::CServiceCore::parent_class       = 0;
  guint    ServiceCore::CServiceCore::signals[N_SIGNALS] = { 0 };


// HACK: Hackery to rename functions in glue
#define c_service_core_startup                  startup
#define c_service_core_identity                 identity
#define c_service_core_ui_raise                 ui_raise
#define c_service_core_volume_get               volume_get
#define c_service_core_volume_set               volume_set
#define c_service_core_get_current_source       get_current_source
#define c_service_core_get_source_caps          get_source_caps
#define c_service_core_get_metadata_from_source get_metadata_from_source
#define c_service_core_playlist_add_uri_list    playlist_add_uri_list
#define c_service_core_last_fm_play_uri         last_fm_play_uri

#define c_service_core_go_next                  go_next
#define c_service_core_go_prev                  go_prev
#define c_service_core_pause                    pause
#define c_service_core_stop                     stop
#define c_service_core_play                     play

#include "service_core_glue.h"
#include "bmp-marshalers.h"

  void
  ServiceCore::CServiceCore::class_init (gpointer klass,
                                         gpointer class_data)
  {
    parent_class = g_type_class_peek_parent (klass);

    GObjectClass *gobject_class = reinterpret_cast<GObjectClass*>(klass);
    gobject_class->constructor  = &CServiceCore::constructor;

    signals[SIGNAL_STARTUP_COMPLETE] =
      g_signal_new ("startup-complete",
                    G_OBJECT_CLASS_TYPE (G_OBJECT_CLASS (klass)),
                    GSignalFlags (G_SIGNAL_RUN_LAST | G_SIGNAL_DETAILED),
                    0,
                    NULL, NULL,
                    g_cclosure_marshal_VOID__VOID, G_TYPE_NONE, 0);

    signals[SIGNAL_SHUTDOWN_COMPLETE] =
      g_signal_new ("shutdown-complete",
                    G_OBJECT_CLASS_TYPE (G_OBJECT_CLASS (klass)),
                    GSignalFlags (G_SIGNAL_RUN_LAST | G_SIGNAL_DETAILED),
                    0,
                    NULL, NULL,
                    g_cclosure_marshal_VOID__VOID, G_TYPE_NONE, 0);

    signals[SIGNAL_QUIT] =
      g_signal_new ("quit",
                    G_OBJECT_CLASS_TYPE (G_OBJECT_CLASS (klass)),
                    GSignalFlags (G_SIGNAL_RUN_LAST | G_SIGNAL_DETAILED),
                    0,
                    NULL, NULL,
                    g_cclosure_marshal_VOID__VOID, G_TYPE_NONE, 0);
  }

  GObject *
  ServiceCore::CServiceCore::constructor (GType                  type,
                                          guint                  n_construct_properties,
                                          GObjectConstructParam *construct_properties)
  {
    GObject *object = G_OBJECT_CLASS (parent_class)->constructor (type, n_construct_properties, construct_properties);

    return object;
  }

  ServiceCore::CServiceCore *
  ServiceCore::CServiceCore::create (ServiceCore &core)
  {
    dbus_g_object_type_install_info (TYPE_C_SERVICE_CORE, &dbus_glib_c_service_core_object_info);

    CServiceCore *self = C_SERVICE_CORE (g_object_new (TYPE_C_SERVICE_CORE, NULL));
    self->core = &core;

    // Set up DBus connection
    ServiceCore::CServiceCore::initialize_dbus (self);

    return self;
  }

  GType
  ServiceCore::CServiceCore::get_type ()
  {
    static GType type = 0;

    if (G_UNLIKELY (type == 0))
      {
        static GTypeInfo const type_info =
          {
            sizeof (CServiceCoreClass),
            NULL,
            NULL,
            &class_init,
            NULL,
            NULL,
            sizeof (CServiceCore),
            0,
            NULL
          };

        type = g_type_register_static (G_TYPE_OBJECT, "CServiceCore", &type_info, GTypeFlags (0));
      }

    return type;
  }

  gboolean
  ServiceCore::CServiceCore::identity (CServiceCore * self,
                                       char **        identity,
                                       GError **      error)
  {
    *identity = g_strdup (mpris_identity);
    return TRUE;
  }

  gboolean
  ServiceCore::CServiceCore::ui_raise (CServiceCore * self,
                                       GError **      error)
  {
    self->core->shell->raise_ui();
    return TRUE;
  }

  gboolean
  ServiceCore::CServiceCore::volume_set (CServiceCore * self,
                                         int            volume,
                                         GError **      error)
  {
    mcs->key_set<int> ("bmp", "volume", volume);
    return TRUE;
  }

  gboolean
  ServiceCore::CServiceCore::volume_get (CServiceCore  *self,
                                         int *          volume,
                                         GError **      error)
  {
    *volume = mcs->key_get<int> ("bmp", "volume");
    return TRUE;
  }

  gboolean
  ServiceCore::CServiceCore::go_next  (CServiceCore  *self,
                                       GError **      error)
  {
    self->core->shell->next_ext (); 
    return TRUE;
  }

  gboolean
  ServiceCore::CServiceCore::go_prev  (CServiceCore  *self,
                                       GError **      error)
  {
    self->core->shell->prev_ext (); 
    return TRUE;
  }

  gboolean
  ServiceCore::CServiceCore::pause   (CServiceCore  *self,
                                      GError **      error)
  {
    self->core->shell->pause_ext (); 
    return TRUE;
  }

  gboolean
  ServiceCore::CServiceCore::stop   (CServiceCore  *self,
                                     GError **      error)
  {
    self->core->shell->stop_ext (); 
    return TRUE;
  }

  gboolean
  ServiceCore::CServiceCore::play   (CServiceCore  *self,
                                     GError **      error)
  {
    self->core->shell->play_ext (); 
    return TRUE;
  }

  gboolean
  ServiceCore::CServiceCore::startup  (CServiceCore * self,
                                       int            no_network, 
                                       GError **      error)
  {
    self->core->_disable_network = (no_network == 1) ? true : false; 
    return TRUE;
  }

  gboolean
  ServiceCore::CServiceCore::last_fm_play_uri (CServiceCore * self,
                                               char *         uri,
                                               GError **      error)
  {
    self->core->shell->play_lastfm_uri (uri);
    return TRUE;
  }

  gboolean
  ServiceCore::CServiceCore::playlist_add_uri_list (CServiceCore * self,
                                                    char **        uris,
                                                    gboolean       playback,
                                                    GError **      error)
  {
    VUri list;
    VFS::VFS::strv_to_uri_list (list, uris, false);
    self->core->shell->play_uris (list, playback);
    return TRUE;
  }

  gboolean
  ServiceCore::CServiceCore::get_current_source (CServiceCore * self,
                                                 int *          source,
                                                 GError **      error)
  {
    *source = self->core->shell->get_current_source ();
    return TRUE;
  }

  gboolean
  ServiceCore::CServiceCore::get_source_caps (CServiceCore * self,
                                              int            source,
                                              int *          caps,
                                              GError **      error)
  {
      try
        {
          *caps = int (self->core->shell->get_source_caps (PlaybackSourceID (source)));
          return TRUE;
        }
      catch (PlayerShell::InvalidSourceError& cxe)
        {
          *error = g_error_new (g_quark_from_static_string ("BMP-CServiceCore"), 0, "Invalid Source Specified");
          return FALSE;
        }
      catch (...)
        {
          *error = g_error_new (g_quark_from_static_string ("BMP-CServiceCore"), 0, "Unknown Error");
          return FALSE;
        }
  }

  gboolean
  ServiceCore::CServiceCore::get_metadata_from_source (CServiceCore * self,
                                                       int            source,
                                                       GHashTable **  metadata,
                                                       GError **      error)
  {
      try
        {
          *metadata = self->core->shell->get_metadata_from_source (PlaybackSourceID (source));
          return TRUE;
        }
      catch (PlayerShell::InvalidSourceError & cxe)
        {
          *error = g_error_new (g_quark_from_static_string ("BMP-CServiceCore"), 0, "Invalid Source Specified");
          return FALSE;
        }
      catch (PlayerShell::NoMetadataError & cxe)
        {
          *error = g_error_new (g_quark_from_static_string ("BMP-CServiceCore"), 0, "No Metadata Available");
          return FALSE;
        }
      catch (...)
        {
          *error = g_error_new (g_quark_from_static_string ("BMP-CServiceCore"), 0, "Unknown Error");
          return FALSE;
        }
  }

  void
  ServiceCore::CServiceCore::startup_complete (CServiceCore *self)
  {
    g_signal_emit (self, signals[SIGNAL_STARTUP_COMPLETE], 0);
  }

  void
  ServiceCore::CServiceCore::shutdown_complete (CServiceCore *self)
  {
    g_signal_emit (self, signals[SIGNAL_SHUTDOWN_COMPLETE], 0);
  }

  void
  ServiceCore::CServiceCore::quit (CServiceCore *self)
  {
    g_signal_emit (self, signals[SIGNAL_QUIT], 0);
  }

  void
  ServiceCore::CServiceCore::initialize_dbus (CServiceCore *self)
  {
    DBusGConnection * session_bus;
    DBusGProxy *      session_bus_proxy;
    gboolean          connected = FALSE;
    guint             request_name_result;
    GError *          error = NULL;

    session_bus = dbus_g_bus_get (DBUS_BUS_SESSION, &error);
    if (!session_bus)
      {
        g_message ("Running: %s (DBUS Error: '%s')", G_OBJECT_TYPE_NAME(self), error->message);
        g_error_free (error);
        return;
      }

    session_bus_proxy = dbus_g_proxy_new_for_name (session_bus,
                                                   "org.freedesktop.DBus",
                                                   "/org/freedesktop/DBus",
                                                   "org.freedesktop.DBus");

    if (!dbus_g_proxy_call (session_bus_proxy, "RequestName", &error,
                            G_TYPE_STRING, BMP_DBUS_SERVICE,
                            G_TYPE_UINT, 0,
                            G_TYPE_INVALID,
                            G_TYPE_UINT, &request_name_result,
                            G_TYPE_INVALID))
      {
        g_error ("%s: Failed RequestName request: %s", G_STRFUNC, error->message);
        g_error_free (error);
        error = NULL;
      }

    switch (request_name_result)
      {
        case DBUS_REQUEST_NAME_REPLY_PRIMARY_OWNER:
          {
            dbus_g_connection_register_g_object (session_bus, BMP_DBUS_PATH, G_OBJECT(self));
            connected = true;
            break;
          }

        case DBUS_REQUEST_NAME_REPLY_EXISTS:
          {
            connected = false;
            break;
          }
        }

    if (connected)
      {
        g_message ("Running: %s (DBus:%s)", G_OBJECT_TYPE_NAME(self), BMP_DBUS_SERVICE);
      }
    else
      {
        g_message ("Running: %s", G_OBJECT_TYPE_NAME(self));
      }
  }

  ServiceCore::ServiceCore ()
    : c_service_core (0),
      shell (0),
      _disable_network (false)
  {
    c_service_core = CServiceCore::create (*this);
  }

  ServiceCore::~ServiceCore ()
  {
    if (c_service_core)
      {
        CServiceCore::shutdown_complete (c_service_core);
        g_object_unref (c_service_core);
      }
  }

  void
  ServiceCore::quit ()
  {
    CServiceCore::quit (c_service_core);
  }

  void
  ServiceCore::ui_stop ()
  {
    delete shell;
  }

  void
  ServiceCore::ui_start ()
  {
    if (_disable_network)
      Network::disable ();

    g_log (G_LOG_DOMAIN, G_LOG_LEVEL_INFO, "Starting UI");
    shell = PlayerShell::create ();
    gtk_widget_set_style (GTK_WIDGET (shell->gobj()), Util::style_get());

    shell->present ();
    while (gtk_events_pending()) gtk_main_iteration ();

    CServiceCore::startup_complete (c_service_core);
    s_startup_complete_.emit ();
  }
} // Bmp namespace
