//  BMPx - The Dumb Music Player
//  Copyright (C) 2005-2006 BMPx development team.
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//  --
//
//  The BMPx project hereby grants permission for non GPL-compatible GStreamer
//  plugins to be used and distributed together with GStreamer and BMPx. This
//  permission is above and beyond the permissions granted by the GPL license
//  BMPx is covered by.

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif //HAVE_CONFIG_H

#include <iostream>

#include <libxml/parser.h>
#include <glibmm.h>

#include "podcast_types.hh"
#include "podcast-libxml2-sax.hh"

using namespace Bmp::PodCastTools;
using namespace std;

namespace
{
  const char * A_URL            = "url";
  const char * A_LENGTH         = "length";
  const char * A_IS_PERMA_LINK  = "isPermaLink";
  const char * A_TYPE           = "type";

  enum Element
  {
    E_NONE                = 0,
    E_CHANNEL             = 1 << 0,
    E_TITLE               = 1 << 1,
    E_DESCRIPTION         = 1 << 2,
    E_LINK                = 1 << 3,
    E_LAST_BUILD_DATE     = 1 << 4,
    E_PUB_DATE            = 1 << 5,
    E_GENERATOR           = 1 << 6,
    E_ITEM                = 1 << 7,
    E_ENCLOSURE           = 1 << 8,
    E_GUID                = 1 << 10,
    E_IMAGE               = 1 << 11,
    E_URL                 = 1 << 12,
    E_COPYRIGHT           = 1 << 13,
  };

  struct PodCastParserContext
  {
    int             m_state;
    PodCastItem     m_item;
    PodCast       & m_podcast;
    bool            m_element;

    PodCastParserContext (PodCast & cast) : m_state (E_NONE), m_podcast(cast), m_element(false) {}
  };

  typedef std::map < Glib::ustring, Glib::ustring > ElementAttributes; 

#define SET_STATE(e)    ((context->m_state |= e))
#define CLEAR_STATE(e)  ((context->m_state &= ~e))
#define STATE(e)        ((context->m_state & e) != 0)

  void
  item_overlay (Bmp::PodCastTools::PodCastItem & item, Bmp::PodCastTools::PodCastOverlayItem const& overlay_item)
  {
    item.listened_to    = overlay_item.listened_to;
    item.downloaded     = overlay_item.downloaded;
    item.localfilename  = overlay_item.localfilename;
  }
  
  //////////////////////////////////////////////////////////////////////////////

  xmlEntityPtr
  get_entity      (void * ctxptr, const xmlChar *name)
  {   
    return xmlGetPredefinedEntity (name);
  } 

  void
  start_document  (void * ctxptr)
  {
    PodCastParserContext * context = reinterpret_cast<PodCastParserContext *>(ctxptr);
  }

  void
  end_document    (void * ctxptr)
  {
    PodCastParserContext * context = reinterpret_cast<PodCastParserContext *>(ctxptr);
  }

  void
  start_element   (void * ctxptr, const xmlChar * _name, const xmlChar ** _attributes)
  {
    PodCastParserContext * context = reinterpret_cast<PodCastParserContext *>(ctxptr);

    Glib::ustring name ((const char*)_name); 
    ElementAttributes attributes; 

    context->m_element = true;

    for (int i = 0; _attributes && _attributes[i] != NULL; i += 2)
    {
      attributes.insert (std::make_pair (Glib::ustring ((const char*)_attributes[i]), Glib::ustring ((const char*)_attributes[i+1])));
    }

    if (name == "channel") 
    {
      SET_STATE(E_CHANNEL);
      return;
    }

    if (name == "title")
    {
      SET_STATE(E_TITLE);
      return;
    }

    if (name == "description") 
    {
      SET_STATE(E_DESCRIPTION);
      return;
    }

    if (name == "link")
    {
      SET_STATE(E_LINK);
      return;
    }

    if (name == "lastBuildDate")
    {
      SET_STATE(E_LAST_BUILD_DATE);
      return;
    }

    if (name == "pubDate")
    {
      SET_STATE(E_PUB_DATE);
      return;
    } 

    if (name == "generator")
    {
      SET_STATE(E_GENERATOR); 
      return;
    }

    if (name == "item")
    {
      SET_STATE(E_ITEM);
      context->m_item = PodCastItem();
      return;
    }

    if (name == "enclosure")
    {
      SET_STATE(E_ENCLOSURE);

      if (attributes.find (A_URL) != attributes.end())
          context->m_item.enclosure_url = attributes.find(A_URL)->second;

      if (attributes.find (A_LENGTH) != attributes.end())
          context->m_item.enclosure_length = strtol (attributes.find(A_LENGTH)->second.c_str(), 0, 10);

      if (attributes.find (A_TYPE) != attributes.end())
          context->m_item.enclosure_type = attributes.find(A_TYPE)->second;

      return;
    }

    if (name == "guid")
    {
      SET_STATE(E_GUID);

      if (attributes.find (A_IS_PERMA_LINK) != attributes.end())
          context->m_item.guid_permalink = bool (attributes.find(A_IS_PERMA_LINK)->second == "true");

      return;
    }

    if (name == "image")
    {
      SET_STATE(E_IMAGE);
      return;
    }

    if (name == "url")
    {
      SET_STATE(E_URL);
      return;
    }
  }

  void
  end_element     (void * ctxptr, const xmlChar * _name)
  {
    PodCastParserContext * context = reinterpret_cast<PodCastParserContext *>(ctxptr);

    Glib::ustring name ((const char*)(_name));

    context->m_element = false;

    if (name == "channel") 
    {
      CLEAR_STATE(E_CHANNEL);
      return;
    }

    if (name == "title") 
    {
      CLEAR_STATE(E_TITLE);
      return;
    }

    if (name == "description")  
    {
      CLEAR_STATE(E_DESCRIPTION);
      return;
    }

    if (name == "link")
    {
      CLEAR_STATE(E_LINK);
      return;
    }

    if (name == "lastBuildDate") 
    {
      CLEAR_STATE(E_LAST_BUILD_DATE);
      return;
    }

    if (name == "pubDate")
    {
      CLEAR_STATE(E_PUB_DATE);
      return;
    }

    if (name == "generator")
    {
      CLEAR_STATE(E_GENERATOR);
      return;
    }

    if (name == "item")
    {
      CLEAR_STATE(E_ITEM);

      if (!context->m_item.enclosure_url)
      {
        std::cerr << " No Enclosure URL for Item " << std::endl;
        return;
      }

      if (context->m_item.guid_value)
        {
          MOverlayItems::const_iterator overlay = context->m_podcast.overlay_items.find (context->m_item.guid_value.get());
          PodCastOverlayItem overlay_item;

          if (overlay != context->m_podcast.overlay_items.end())
              overlay_item = overlay->second;
          else
              context->m_podcast.overlay_items.insert (std::make_pair (context->m_item.guid_value.get(), overlay_item));

          item_overlay (context->m_item, overlay_item);
        }

      context->m_podcast.items.push_back (context->m_item);
      return;
    }

    if (name == "enclosure")
    {
      CLEAR_STATE(E_ENCLOSURE);
      return;
    }

    if (name == "guid")
    {
      CLEAR_STATE(E_GUID);
      return;
    }

    if (name == "image")
    {
      CLEAR_STATE(E_IMAGE);
      return;
    }

    if (name == "url")
    {
      CLEAR_STATE(E_URL);
      return;
    }
  }

  void
  whitespace      (void * ctxptr, const xmlChar * _text, int length)
  {
  }

  void
  pcdata          (void * ctxptr, const xmlChar * _text, int length)
  {
    PodCastParserContext * context = reinterpret_cast<PodCastParserContext *>(ctxptr);

    if (!context->m_element)
      return;

    Glib::ustring text (std::string((const char*)_text).substr(0, length));

    if (!STATE(E_CHANNEL)) return;

    if (!STATE(E_ITEM) && !STATE(E_IMAGE))
    {
        if (STATE(E_TITLE))
          {
            context->m_podcast.title = text;
          }
        else
        if (STATE(E_DESCRIPTION))
          {
            context->m_podcast.description = text;
          }
        else
        if (STATE(E_LINK))
          {
            context->m_podcast.link = text;
          }
        else
        if (STATE(E_LAST_BUILD_DATE))
          {
            context->m_podcast.last_build_date = text;
          }
        else
        if (STATE(E_PUB_DATE))
          {
            context->m_podcast.pub_date = text;
          }
        else
        if (STATE(E_GENERATOR))
          {
            context->m_podcast.generator = text;
          }
        return;
    }

    if (!STATE(E_ITEM) && STATE(E_IMAGE))
    {
      if (STATE(E_URL))
        {
          context->m_podcast.image_url = text;
        }
      else
      if (STATE(E_TITLE))
        {
          context->m_podcast.image_title = text;
        }
      else
      if (STATE(E_LINK))
        {
          context->m_podcast.image_link = text;
        }
      else
      if (STATE(E_DESCRIPTION))
        {
          context->m_podcast.image_description = text;
        }
      return;
    }

    if (STATE(E_ITEM))
    {
        if (STATE(E_TITLE))
          {
            context->m_item.title = text;
          }
        else
        if (STATE(E_DESCRIPTION))
          {
            context->m_item.description = text;
          }
        else
        if (STATE(E_LINK))
          {
            context->m_item.link = text;
          }
        else
        if (STATE(E_COPYRIGHT))
          {
            context->m_item.copyright = text;
          }
        else
        if (STATE(E_PUB_DATE))
          {
            context->m_item.pub_date = text;
          }
        else
        if (STATE(E_GUID))
          {
            context->m_item.guid_value = text;
          }
      }
  }

  void
  warning         (void * ctxptr, const char * message, ...)
  {
    PodCastParserContext * context = reinterpret_cast<PodCastParserContext *>(ctxptr);
  }

  void
  error           (void * ctxptr, const char * message, ...)  
  {
    PodCastParserContext * context = reinterpret_cast<PodCastParserContext *>(ctxptr);
  }

  void
  fatal_error     (void * ctxptr, const char * message, ...)
  {
    PodCastParserContext * context = reinterpret_cast<PodCastParserContext *>(ctxptr);
  }
    
  xmlSAXHandler PodCastParser =
  {
      (internalSubsetSAXFunc)0,                           // internalSubset 
      (isStandaloneSAXFunc)0,                             // isStandalone 
      (hasInternalSubsetSAXFunc)0,                        // hasInternalSubset 
      (hasExternalSubsetSAXFunc)0,                        // hasExternalSubset 
      (resolveEntitySAXFunc)0,                            // resolveEntity 
      (getEntitySAXFunc)get_entity,                       // getEntity 
      (entityDeclSAXFunc)0,                               // entityDecl 
      (notationDeclSAXFunc)0,                             // notationDecl 
      (attributeDeclSAXFunc)0,                            // attributeDecl 
      (elementDeclSAXFunc)0,                              // elementDecl 
      (unparsedEntityDeclSAXFunc)0,                       // unparsedEntityDecl 
      (setDocumentLocatorSAXFunc)0,                       // setDocumentLocator 
      (startDocumentSAXFunc)start_document,               // startDocument 
      (endDocumentSAXFunc)end_document,                   // endDocument 
      (startElementSAXFunc)start_element,                 // startElement 
      (endElementSAXFunc)end_element,                     // endElement 
      (referenceSAXFunc)0,                                // reference 
      (charactersSAXFunc)pcdata,                          // characters 
      (ignorableWhitespaceSAXFunc)whitespace,             // ignorableWhitespace 
      (processingInstructionSAXFunc)0,                    // processingInstruction 
      (commentSAXFunc)0,                                  // comment 
      (warningSAXFunc)warning,                            // warning 
      (errorSAXFunc)error,                                // error 
      (fatalErrorSAXFunc)fatal_error,                     // fatalError 
  };
}

namespace Bmp
{
  namespace PodCastTools
  {
    int podcast_rss_parse (PodCast & cast, std::string const& data)
    {
        PodCastParserContext context (cast); 
        int rc = xmlSAXUserParseMemory (&PodCastParser, reinterpret_cast<void *>(&context), data.c_str(), strlen(data.c_str()));
        return rc;
    }
  }
}
