//  BMPx - The Dumb Music Player
//  Copyright (C) 2005 BMPx development team.
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License Version 2
//  as published by the Free Software Foundation.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//  --
//
//  The BMPx project hereby grants permission for non-GPL compatible GStreamer
//  plugins to be used and distributed together with GStreamer and BMPx. This
//  permission is above and beyond the permissions granted by the GPL license
//  BMPx is covered by.

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif //HAVE_CONFIG_H

#include <gtkmm.h>
#include <glibmm/i18n.h>
#include <cstdlib>
#include <cstring>
#include <iostream>
#include <sstream>

#include <boost/format.hpp>
#include <boost/optional.hpp>

#include <ne_basic.h>
#include <ne_compress.h>
#include <ne_socket.h>
#include <ne_utils.h>
#include <ne_props.h>
#include <ne_session.h>
#include <ne_request.h>
#include <ne_uri.h>
#include <ne_auth.h>

#include "main.hh"
#include "debug.hh"
#include "paths.hh"
#include "stock.hh"
#include "audio.hh"
#include "util_file.hh"
#include "util.hh"
#include "ui_toolbox.hh"

#include "x_library.hh"
#include "x_vfs.hh"

#include "mbxml.hh"
#include "library-ui-modify.hh"
#include "ld.hh"

namespace
{
  void
  cbe_set_sensitive (Gtk::CheckButton *cb, Gtk::ComboBoxEntry *cbe)
  {
      cbe->set_sensitive (cb->get_active());
  }

  void
  fcb_set_sensitive (Gtk::CheckButton *cb, Gtk::FileChooserButton *fcb)
  {
      fcb->set_sensitive (cb->get_active());
  }
  
  bool processed = false;

  void
  set_processed ()
  {
    processed = true;
  }

  void 
  set_progress_bar (int position, Gtk::ProgressBar *pb, Bmp::Audio::ProcessorBase *processor)
  {
    pb->property_fraction() = (1.*position) / 135.;
  }

  void
  set_cbox_sensitive (Gtk::CheckButton *cb, Gtk::ComboBoxEntry *cb_entry)
  {
    cb_entry->set_sensitive (cb->get_active());
  }

  Gtk::Widget*
  get_popup (Glib::RefPtr<Gtk::UIManager> ui_manager,
             Glib::ustring                path) 
  {
      Gtk::Widget *menu_item;
      menu_item = ui_manager->get_widget (path);
      if ( menu_item )
          return dynamic_cast<Gtk::MenuItem*>(menu_item)->get_submenu ();
      else
          return 0;
  }

#define ACTION_UPDATE     "action-update"
#define ACTION_MOVE_UP    "action-move-up"
#define ACTION_MOVE_DOWN  "action-move-down"

#define ACTION_REMOVE     "action-remove"

#define ACTION_RELAYOUT_NONE          "action-relayout-none"
#define ACTION_RELAYOUT_TRACKNUMBERS  "action-relayout-tracknumber"
#define ACTION_RELAYOUT_LEVENSHTEIN   "action-relayout-levenshtein"

#define ACTION_RESTORE                "action-restore"
#define ACTION_USE_FILENAMES          "action-use-filenames"

#ifdef HAVE_OFA
#  define ACTION_SCAN_RELEASE "action-scan-release"
#  define ACTION_SCAN_FILE    "action-scan-file"
#  define ACTION_SCAN_ALL     "action-scan-all"
#  define ACTION_SUBMIT_PUIDS "action-submit-puids"

  const char *modify_menu = 
  "<ui>"
  "   <menubar name='popup-modify'>"
  "   <menu action='dummy' name='menu-modify'>"
  "     <menuitem action='" ACTION_SCAN_RELEASE "'/>"
  "     <menuitem action='" ACTION_SCAN_FILE "'/>"
  "     <menuitem action='" ACTION_SCAN_ALL "'/>"
  "     <menuitem action='" ACTION_SUBMIT_PUIDS "'/>"
  "     <separator name='sep1'/>"
  "     <menuitem action='" ACTION_REMOVE "'/>"
  "     <separator name='sep2'/>"
  "     <menuitem action='" ACTION_RELAYOUT_NONE "'/>"
  "     <menuitem action='" ACTION_RELAYOUT_TRACKNUMBERS "'/>"
  "     <menuitem action='" ACTION_RELAYOUT_LEVENSHTEIN "'/>"
  "     <separator name='sep3'/>"
  "     <menuitem action='" ACTION_USE_FILENAMES "'/>"
  "     <menuitem action='" ACTION_RESTORE "'/>"
  "   </menu>"
  "   </menubar>"
  "</ui>";

#else

  const char *modify_menu = 
  "<ui>"
  "   <menubar name='popup-modify'>"
  "   <menu action='dummy' name='menu-modify'>"
  "     <menuitem action='" ACTION_REMOVE "'/>"
  "     <separator name='sep1'/>"
  "     <menuitem action='" ACTION_RELAYOUT_NONE "'/>"
  "     <menuitem action='" ACTION_RELAYOUT_TRACKNUMBERS "'/>"
  "     <menuitem action='" ACTION_RELAYOUT_LEVENSHTEIN "'/>"
  "     <separator name='sep2'/>"
  "     <menuitem action='" ACTION_USE_FILENAMES "'/>"
  "     <menuitem action='" ACTION_RESTORE "'/>"
  "   </menu>"
  "   </menubar>"
  "</ui>";

#endif //HAVE_OFA

}

namespace Bmp
{
      LibraryUiModify::LibraryUiModify (BaseObjectType                       * obj,
                                        Glib::RefPtr<Gnome::Glade::Xml> const& xml)

              : Gtk::Dialog         (obj),
                m_ref_xml           (xml),

                m_album_unknown     (Gdk::Pixbuf::create_from_file
                                            (Glib::build_filename (BMP_IMAGE_DIR, BMP_COVER_IMAGE_DEFAULT))),

                m_current_ordering  (ORDER_AUTO)

      {
          set_title (_("Edit Album(s) - BMP"));

          // Widgets
          m_ref_xml->get_widget ("merge_cb_artist", merge_cb_artist);
          m_ref_xml->get_widget ("merge_cb_album", merge_cb_album);

          m_ref_xml->get_widget ("cb_genre", cb_genre);
          m_ref_xml->get_widget ("cb_modify_genre", cb_modify_genre);
          cb_genre_entry = cb_genre->get_entry ();
          cb_modify_genre->signal_toggled().connect
            (sigc::bind (sigc::ptr_fun (&set_cbox_sensitive), cb_modify_genre, cb_genre));
                                                                               
          m_ref_xml->get_widget ("merge_coverart", merge_coverart);
          m_ref_xml->get_widget ("merge_artist_id", merge_artist_id);
          m_ref_xml->get_widget ("merge_release_id", merge_release_id);
          m_ref_xml->get_widget ("merge_asin", merge_asin);
          m_ref_xml->get_widget ("merge_score",	merge_score);
          m_ref_xml->get_widget ("view_release", view_release);
          m_ref_xml->get_widget ("view_local", view_local);
          m_ref_xml->get_widget ("progress_processing",	progress_processing);
          m_ref_xml->get_widget ("l_tracks", l_tracks);

          m_ref_xml->get_widget ("cancel", cancel);
          m_ref_xml->get_widget ("ok", ok);
          m_ref_xml->get_widget ("b_update", b_update);
          m_ref_xml->get_widget ("b_move_up", b_move_up);
          m_ref_xml->get_widget ("b_move_down", b_move_down);

          std::vector<Gtk::Widget*> chainwidgets; //FIXME: this doesn't work, but why? -- deadchip
          chainwidgets.push_back (merge_cb_artist);
          chainwidgets.push_back (merge_cb_album);
          chainwidgets.push_back (b_update);
          chainwidgets.push_back (cb_genre);
          chainwidgets.push_back (cb_modify_genre);
          chainwidgets.push_back (cancel);
          chainwidgets.push_back (ok);
          chainwidgets.push_back (merge_artist_id);
          chainwidgets.push_back (merge_release_id);
          chainwidgets.push_back (merge_asin);
          chainwidgets.push_back (merge_score);
          chainwidgets.push_back (view_release);
          chainwidgets.push_back (view_local);
          set_focus_chain (chainwidgets);

          m_ui_manager = Gtk::UIManager::create ();
          m_actions = Gtk::ActionGroup::create ("ActionsLibraryModify"); 

          m_actions->add ( Gtk::Action::create ("dummy", "dummy"));

          m_actions->add ( Gtk::Action::create (ACTION_UPDATE,
                                    Gtk::StockID (GTK_STOCK_NETWORK),
                                    _("Search")),
                                    sigc::mem_fun (this, &Bmp::LibraryUiModify::on_action_update));

          m_actions->add ( Gtk::Action::create (ACTION_MOVE_UP,
                                    Gtk::StockID (GTK_STOCK_GO_UP),
                                    _("Move Up")),
                                    sigc::mem_fun (this, &Bmp::LibraryUiModify::on_action_up));

          m_actions->add ( Gtk::Action::create (ACTION_MOVE_DOWN,
                                    Gtk::StockID (GTK_STOCK_GO_DOWN),
                                    _("Move Down")),
                                    sigc::mem_fun (this, &Bmp::LibraryUiModify::on_action_down));

          m_actions->get_action (ACTION_UPDATE)->connect_proxy (*b_update);
          m_actions->get_action (ACTION_MOVE_UP)->connect_proxy (*b_move_up);
          m_actions->get_action (ACTION_MOVE_DOWN)->connect_proxy (*b_move_down);

          m_actions->get_action (ACTION_MOVE_UP)->set_sensitive (false);
          m_actions->get_action (ACTION_MOVE_DOWN)->set_sensitive (false);

          m_actions->add ( Gtk::Action::create (ACTION_REMOVE,
                                    Gtk::StockID (BMP_STOCK_DELETE),
                                    _("Remove Track From List")),
                                    sigc::mem_fun (this, &Bmp::LibraryUiModify::on_action_remove));

          m_actions->add ( Gtk::Action::create (ACTION_RELAYOUT_NONE,
                                    Gtk::StockID (BMP_STOCK_SCAN),
                                    _("Re-Layout Tracks Using No Ordering")),
                  (sigc::bind (sigc::mem_fun (this, &Bmp::LibraryUiModify::on_action_relayout),
                    ORDER_LINEAR)));

          m_actions->add ( Gtk::Action::create (ACTION_RELAYOUT_TRACKNUMBERS,
                                    Gtk::StockID (BMP_STOCK_SCAN),
                                    _("Re-Layout Tracks Using Tracknumbers")),
                  (sigc::bind (sigc::mem_fun (this, &Bmp::LibraryUiModify::on_action_relayout),
                    ORDER_TRACKNUMBERS)));

          m_actions->add ( Gtk::Action::create (ACTION_RELAYOUT_LEVENSHTEIN,
                                    Gtk::StockID (BMP_STOCK_SCAN),
                                    _("Re-Layout Tracks Using Name Matching")),
                  (sigc::bind (sigc::mem_fun (this, &Bmp::LibraryUiModify::on_action_relayout),
                    ORDER_LEVENSHTEIN)));

          m_actions->add ( Gtk::ToggleAction::create (ACTION_USE_FILENAMES,
                                    Gtk::StockID (GTK_STOCK_FILE),
                                    _("Display/Use Filenames Rather than Track Names")),
                  (sigc::mem_fun (this, &Bmp::LibraryUiModify::on_action_use_filenames_toggled)));

          m_action_use_filenames =
              Glib::RefPtr<Gtk::ToggleAction>::cast_static(m_actions->get_action (ACTION_USE_FILENAMES));

          m_actions->add ( Gtk::Action::create (ACTION_RESTORE,
                                    Gtk::StockID (GTK_STOCK_REFRESH),
                                    _("Restore Original Track List")),
                  (sigc::mem_fun (this, &Bmp::LibraryUiModify::on_action_restore)));

#ifdef HAVE_OFA

          m_actions->add ( Gtk::Action::create (ACTION_SCAN_RELEASE,
                                  Gtk::StockID (GTK_STOCK_FIND),
                                  _("Find Release based on Selected File")),
                                  Gtk::AccelKey (""), 
                                  sigc::mem_fun (this, &Bmp::LibraryUiModify::on_action_scan));

          m_actions->add ( Gtk::Action::create (ACTION_SCAN_FILE,
                                  Gtk::StockID (BMP_STOCK_SCAN),
                                  _("Find PUID for selected File")),
                                  Gtk::AccelKey (""), 
                                  sigc::mem_fun (this, &Bmp::LibraryUiModify::on_action_scan_file));

          m_actions->add ( Gtk::Action::create (ACTION_SCAN_ALL,
                                  Gtk::StockID (GTK_STOCK_FIND),
                                  _("Find PUIDs for all Files")),
                                  Gtk::AccelKey (""), 
                                  sigc::mem_fun (this, &Bmp::LibraryUiModify::on_action_scan_all));

          m_actions->add ( Gtk::Action::create (ACTION_SUBMIT_PUIDS,
                                  Gtk::StockID (GTK_STOCK_NETWORK),
                                  _("Submit found PUID <-> Track ID Pairs")),
                                  Gtk::AccelKey (""), 
                                  sigc::mem_fun (this, &Bmp::LibraryUiModify::on_action_submit_puids));

          m_actions->get_action (ACTION_SCAN_RELEASE)->set_sensitive (false);
          m_actions->get_action (ACTION_SCAN_FILE)->set_sensitive (false);
          m_actions->get_action (ACTION_SCAN_ALL)->set_sensitive (false);

#endif //HAVE_OFA

          m_ui_manager->insert_action_group (m_actions);
          m_ui_manager->add_ui_from_string (modify_menu);

          view_local->signal_event().connect
            (sigc::mem_fun (this, &Bmp::LibraryUiModify::popup_rhs_menu));

          merge_cb_artist_entry = merge_cb_artist->get_entry ();
          merge_cb_album_entry  = merge_cb_album->get_entry ();
          merge_cb_artist_entry->set_activates_default (true);
          merge_cb_album_entry->set_activates_default (true);
          merge_cb_artist_entry->set_text (_("<select here>"));
          merge_cb_album_entry->set_text ("");
          merge_coverart->set (m_album_unknown);

          struct { 
              char    *title;
              double   xalign;
              int	     width;
          } headers_tracks[] = {
              { _("#"),	    0.5, 0,  },
              { _("Title"), 0.0, 100 },
          };

          struct { 
              char    *title;
              double   xalign;
              int	     width;
          } headers_tracks_local[] = {
              { _("#"),	    0.5, 0,  },
              { "",         0.5, 24, },
              { _("Title"), 0.0, 100 },
          };


          // Release
          for (unsigned int n = 0; n < G_N_ELEMENTS (headers_tracks); ++n)
            {
              Gtk::CellRendererText *cell = Gtk::manage ( new Gtk::CellRendererText() );
              cell->property_xalign() = headers_tracks[n].xalign;
              view_release->append_column (headers_tracks[n].title, *cell);
              Gtk::TreeViewColumn *column = view_release->get_column (n);
              column->add_attribute (*cell, "text", n);
              column->set_resizable (true);
            }

          store_release = Gtk::ListStore::create (tracks_release);
          view_release->set_model (store_release);
          view_release->get_selection()->set_mode (Gtk::SELECTION_NONE);

          // Local
          {
            Gtk::CellRendererToggle *cell = Gtk::manage ( new Gtk::CellRendererToggle() );
            view_local->append_column ("", *cell);
            Gtk::TreeViewColumn *column = view_local->get_column (0);
            column->add_attribute (*cell, "active", 0);
            column->set_resizable (false);
            cell->signal_toggled().connect
              (sigc::mem_fun (this, &Bmp::LibraryUiModify::on_mod_toggled));
          }

          for (unsigned int n = 0; n < G_N_ELEMENTS (headers_tracks_local); ++n)
          {
            if (n == 0)
              {
                Gtk::CellRendererText *cell = Gtk::manage ( new Gtk::CellRendererText() );
                cell->property_xalign() = headers_tracks_local[n].xalign;
                view_local->append_column (headers_tracks_local[n].title, *cell);
                Gtk::TreeViewColumn *column = view_local->get_column (n+1);
                column->set_cell_data_func (*cell,
                  sigc::mem_fun(this, &Bmp::LibraryUiModify::cell_data_func_number));
              }

#ifdef HAVE_OFA

            else
            if (n == 1)
              {
                Gtk::CellRendererPixbuf *cell = Gtk::manage ( new Gtk::CellRendererPixbuf() );
                view_local->append_column ("", *cell);
                Gtk::TreeViewColumn *column = view_local->get_column (n+1);
                column->set_cell_data_func (*cell,
                  sigc::mem_fun(this, &Bmp::LibraryUiModify::cell_data_func_puidstate));
                column->set_expand (false);
              }

#endif //HAVE_OFA

            else
              {
                Gtk::CellRendererText *cell = Gtk::manage ( new Gtk::CellRendererText() );
                cell->property_xalign() = headers_tracks_local[n].xalign;
                view_local->append_column (headers_tracks_local[n].title, *cell);
                Gtk::TreeViewColumn *column = view_local->get_column (n+1);
                column->add_attribute (*cell, "text", 2);
              }

          }

          store_local = Gtk::ListStore::create (tracks_local);
          view_local->set_model (store_local);
          view_local->get_selection()->set_mode (Gtk::SELECTION_SINGLE);
          view_local->get_selection()->signal_changed().connect
            (sigc::mem_fun (this, &Bmp::LibraryUiModify::local_selection_changed));

          ok->set_sensitive (false);
      }

      LibraryUiModify*
      LibraryUiModify::create ()
      {
        const std::string path (BMP_GLADE_DIR G_DIR_SEPARATOR_S "library-ui-modify-import.glade");
        Glib::RefPtr<Gnome::Glade::Xml> glade_xml = Gnome::Glade::Xml::create (path);
        LibraryUiModify *p = 0;
        glade_xml->get_widget_derived ("dialog-window", p);
        return p;
      }

      void
      LibraryUiModify::cell_data_func_number (Gtk::CellRenderer* cell_, const Gtk::TreeModel::iterator& iter)
      {
        Gtk::CellRendererText *cell = reinterpret_cast<Gtk::CellRendererText*>(cell_);
        static boost::format format_int ("%d");
        unsigned int nr = (*iter)[tracks_local.tracknumber];
        if (nr == 0)
          cell->property_text() = "";
        else
          cell->property_text() = (format_int % nr).str();
      }

      void
      LibraryUiModify::on_mod_toggled (const Glib::ustring& path)
      {
        Gtk::TreeModel::iterator m_iter = store_local->get_iter (path);
        (*m_iter)[tracks_local.modify] = !(*m_iter)[tracks_local.modify];
      }

     void
      LibraryUiModify::local_selection_changed ()
      {
        bool selected = view_local->get_selection()->count_selected_rows ();
        m_actions->get_action (ACTION_MOVE_UP)->set_sensitive (selected);
        m_actions->get_action (ACTION_MOVE_DOWN)->set_sensitive (selected);

#ifdef HAVE_OFA

        m_actions->get_action (ACTION_SCAN_RELEASE)->set_sensitive (selected);
        m_actions->get_action (ACTION_SCAN_FILE)->set_sensitive (selected);

#endif //HAVE_OFA

      }

      void
      LibraryUiModify::on_action_remove ()
      {
        using namespace std; 
        using namespace Gtk; 
        using namespace Glib; 
        using namespace Bmp::Library; 

#ifdef HAVE_HAL

        TreeModel::iterator const& m_iter = view_local->get_selection()->get_selected ();
        ustring volume_udi ((*m_iter)[tracks_local.volume_udi]);
        ustring volume_relative_path ((*m_iter)[tracks_local.volume_relative_path]);

        TreeModel::Path path (store_local->get_path (m_iter));        
        store_local->erase (m_iter);
        TreeModel::iterator const& n_iter (store_local->get_iter (path)); 
        store_local->insert (n_iter);

        for (VUriVrpPair::iterator i = m_list.begin(), e = m_list.end(); i != e; ++i)
        {
          if ((i->first == volume_udi) && (i->second == volume_relative_path))
            {
              m_list.erase (i);
              break;
            }
        }

#else

        TreeModel::iterator const& m_iter = view_local->get_selection()->get_selected ();
        ustring location ((*m_iter)[tracks_local.location]);

        TreeModel::Path path (store_local->get_path (m_iter));        
        store_local->erase (m_iter);
        TreeModel::iterator const& n_iter (store_local->get_iter (path)); 
        store_local->insert (n_iter);

        for (VUri::iterator i = m_list.begin(), e = m_list.end(); i != e; ++i)
        {
          if ((*i) == location)
            {
              m_list.erase (i);
              break;
            }
        }

#endif //HAVE_HAL

      }

      void
      LibraryUiModify::on_action_relayout (OrderingType ordering_type)
      {
        populate_local_store (ordering_type); 
      }

      void
      LibraryUiModify::on_action_restore ()
      {
        m_list = m_list_original; 
        populate_local_store (m_current_ordering);
      }

      void
      LibraryUiModify::on_action_up ()
      {
        Gtk::TreeModel::iterator m_iter = view_local->get_selection()->get_selected ();
        if (m_iter == store_local->children().begin()) return;
        Gtk::TreeModel::iterator d_Iter = m_iter;
        d_Iter--; // yes, this is slow, but it doesn't matter much as a release doesn't have that many tracks
        store_local->iter_swap (m_iter, d_Iter);
      }

      void
      LibraryUiModify::on_action_down ()
      {
        Gtk::TreeModel::iterator m_iter = view_local->get_selection()->get_selected ();
        if (m_iter == store_local->children().end()) return;
        Gtk::TreeModel::iterator d_Iter = m_iter;
        d_Iter++;
        store_local->iter_swap (m_iter, d_Iter);
      }

      void
      LibraryUiModify::populate_data (MusicBrainzXML::MBReleaseV& releases)
      {
          using namespace Bmp::MusicBrainzXML;
          using namespace Gtk;
          using namespace std;
          using namespace Glib;

          merge_cb_artist_store = ListStore::create (artists);
          merge_cb_album_store  = ListStore::create (albums);
          merge_map.clear ();
          merge_coverart->set (m_album_unknown);
          merge_artist_id->set_text   ("");
          merge_release_id->set_text  ("");
          merge_asin->set_text        ("");
          merge_score->set_text       ("");
          l_tracks->set_text          ("");
          store_release->clear ();
          store_local->clear ();
          merge_cb_artist_entry->set_text ("");
          merge_cb_album_entry->set_text ("");

          for (MBReleaseV::const_iterator r_iter  = releases.begin (),
                                          r_end   = releases.end ();
                                          r_iter != r_end; ++r_iter)
          {
            MBRelease const& release = (*r_iter);

            merge_releases.insert (make_pair (release.release_id, release));
            ArtistReleaseMap::iterator map_iter = merge_map.find (release.artist_id);

            if (map_iter != merge_map.end ())
            {
              ReleaseIdV &id_v = map_iter->second;
              id_v.push_back (release.release_id);
            }
            else
            {
              TreeModel::iterator m_iter = merge_cb_artist_store->append ();
              (*m_iter)[artists.name]	= release.artist_name;
              (*m_iter)[artists.id]	= release.artist_id;

              ReleaseIdV id_v; 
              id_v.push_back (release.release_id);
              merge_map.insert (make_pair (release.artist_id, id_v)); 
            }
          }

          merge_cb_artist->set_model  (merge_cb_artist_store);
          merge_cb_album->set_model   (merge_cb_album_store);

          merge_cb_artist->set_active (0);
          merge_cb_album->set_active (0);
          ok->set_sensitive (true);
      }

      bool
      LibraryUiModify::popup_rhs_menu (GdkEvent *ev)
      {
        using namespace Gtk;

        GdkEventButton *event = 0;
        if (ev->type == GDK_BUTTON_PRESS)
          {
            event = reinterpret_cast<GdkEventButton *>(ev);
            if ((event->button == 3) && (event->type == GDK_BUTTON_PRESS) && (store_local->children().size()))
              {
                Menu *menu = dynamic_cast<Menu *>(get_popup (m_ui_manager, "/popup-modify/menu-modify"));
                menu->popup (event->button, event->time);
                return true;
              }

          } 
        return false;
      }

#ifdef HAVE_OFA

      void
      LibraryUiModify::cell_data_func_puidstate (Gtk::CellRenderer* cell, const Gtk::TreeModel::iterator& iter)
      {
        if (Glib::ustring((*iter)[tracks_local.location]).empty())
          {
            cell->property_visible() = false;
            return;
          }
        else
          {
            cell->property_visible() = true;
          }

        PuidState state = (*iter)[tracks_local.puidstate];
        Glib::RefPtr<Gdk::Pixbuf> icon;

        switch (state)
          {
            case P_NONE:
                icon = render_icon (Gtk::StockID (BMP_STOCK_ERROR), Gtk::ICON_SIZE_SMALL_TOOLBAR);
                break;

            case P_PUID:
                icon = render_icon (Gtk::StockID (BMP_STOCK_APPROVE_PARTIAL), Gtk::ICON_SIZE_SMALL_TOOLBAR);
                break;

            case P_PUID_METADATA:
                icon = render_icon (Gtk::StockID (BMP_STOCK_APPROVE), Gtk::ICON_SIZE_SMALL_TOOLBAR);
                break;
          }

        Gtk::CellRendererPixbuf *cell_pb = dynamic_cast<Gtk::CellRendererPixbuf *>(cell);
        cell_pb->property_pixbuf () = icon;
      }

      int
      LibraryUiModify::http_auth (gpointer data, const char *realm, 
                                int attempts, char *username, char *password)
      {
        strncpy (username, mcs->key_get<std::string>("musicbrainz", "username").c_str(), NE_ABUFSIZ);
        strncpy (password, mcs->key_get<std::string>("musicbrainz", "password").c_str(), NE_ABUFSIZ);
        return attempts;
      }

      void
      LibraryUiModify::set_local_puidstate (Gtk::TreeModel::iterator& m_iter, PuidState state)
      {
        using namespace Gtk;

        (*m_iter)[tracks_local.puidstate] = state;
        m_actions->get_action (ACTION_SUBMIT_PUIDS)->set_sensitive (true);
      }

      void
      LibraryUiModify::on_action_submit_puids ()
      {
          using namespace Glib;
          using namespace Gtk;
          using namespace Bmp::DB;
          using namespace Bmp::Library;
          using namespace std;

          typedef std::vector<unsigned int> PositionV;

          ne_session *sess = 0;   
          ne_request *reqs = 0;

          TreeNodeChildren const& nodes_release = store_release->children();
          TreeNodeChildren const& nodes_local   = store_local->children();

          unsigned int max_nodes = nodes_release.size();
          if (nodes_local.size() < max_nodes) max_nodes = nodes_local.size();

          TreeNodeChildren::const_iterator iter_l = nodes_release.begin();
          TreeNodeChildren::const_iterator iter_r = nodes_local.begin();

          PositionV posv;
          for (unsigned int n = 0; n < max_nodes; ++n, ++iter_r)
          {
            if (PuidState((*iter_r)[tracks_local.puidstate]) == P_PUID) 
            {
                posv.push_back (n);
            }
          }

          if (!posv.size()) return;

          sess = ne_session_create ("http", "musicbrainz.org", 80);
          if (!sess) return; 

          set_sensitive (false);
          ne_set_server_auth (sess, http_auth, NULL);
          ne_set_useragent (sess, PACKAGE"/"VERSION);

          progress_processing->set_fraction (0.0);
          progress_processing->set_text ("");

          double n_children = posv.size(); 
          double nth_child = 0;

          for (PositionV::iterator v  = posv.begin();
                                   v != posv.end(); ++v)
          {
              Gtk::TreePath path;

              path.append_index (*v);
              Gtk::TreeModel::iterator iter_r = store_local->get_iter (path);

              path = Gtk::TreePath();
              path.append_index (*v);
              Gtk::TreeModel::iterator iter_l = store_release->get_iter (path);

              MusicBrainzXML::MBTrack const& track = (*iter_l)[tracks_release.track];

              ustring puid = (*iter_r)[tracks_local.puid];
              ustring trackid = track.track_id;
  
              ustring uri ("http://musicbrainz.org/ws/1/track/");
              ustring data ("client=BMP-0.30&puid=");
                      data.append (trackid).append ("+").append(puid);

              debug ("puid-submit", "Data: %s", data.c_str());

              reqs = ne_request_create (sess, "POST", uri.c_str());

#ifdef NE_FEATURE_I18N                  
              ne_set_request_flag (reqs, NE_REQFLAG_IDEMPOTENT, 0);
#endif //NE_FEATURE_I18N

              ne_add_request_header (reqs, "Content-type", "application/x-www-form-urlencoded");
              ne_set_request_body_buffer (reqs, data.c_str(), data.size());
              int result = ne_request_dispatch (reqs);
              int status = ne_get_status (reqs)->code;
              ne_request_destroy (reqs);

              std::string error_message = ne_get_error (sess);

              debug ("puid-submit", "NeonError: %s", error_message.c_str());
              debug ("puid-submit", "NeonState: %d", status); 
              debug ("puid-submit", "NeonResult: %d", result); 

              nth_child += 1;
              progress_processing->set_fraction (double((1.*nth_child)/(1.*n_children)));
              progress_processing->set_text ((boost::format ("%d / %d") % int(nth_child) % int(n_children)).str());

              while (gtk_events_pending()) gtk_main_iteration ();

              //FIXME: Deal with errors
              switch (result)
                {
                  case NE_OK:
                    break;

                  case NE_CONNECT:
                    continue;  

                  case NE_TIMEOUT:
                    continue;  

                  case NE_AUTH:
                    return;  
                
                  default: break;
                }

              switch (status)
                {
                  case 200: set_local_puidstate (iter_r, P_PUID_METADATA); break;
                  case 400: break;
                  case 401: break;
                  case 404: break;
                  default: break;
                }
          }

          progress_processing->set_fraction (0.0);
          progress_processing->set_text ("");
          set_sensitive (true);
          ne_session_destroy (sess);
      }

      void
      LibraryUiModify::on_action_scan_all ()
      {
          using namespace Bmp::MusicBrainzXML;
          using namespace Gtk;
          using namespace std;
          using namespace Glib;

          m_actions->get_action (ACTION_SCAN_RELEASE)->set_sensitive (false);
          m_actions->get_action (ACTION_SCAN_FILE)->set_sensitive (false);
          m_actions->get_action (ACTION_SCAN_ALL)->set_sensitive (false);
          m_actions->get_action (ACTION_UPDATE)->set_sensitive (false);

          merge_cb_artist->set_sensitive (false);
          merge_cb_album->set_sensitive (false);

          ok->set_sensitive (false);
          cancel->set_sensitive (false);

          set_sensitive (false);

          progress_processing->set_sensitive (true);
          progress_processing->property_fraction() = 0.; 

          view_local->get_selection()->unselect_all();

          for (TreeNodeChildren::iterator m_iter  = store_local->children().begin();
                                          m_iter != store_local->children().end(); ++m_iter)
          {
              if (PuidState((*m_iter)[tracks_local.puidstate]) != P_NONE) continue;
              if (ustring((*m_iter)[tracks_local.location]).empty()) continue;

              view_local->get_selection()->select(m_iter);

              Bmp::Audio::ProcessorPUID *processor = new Bmp::Audio::ProcessorPUID;
              processed = false;
              processor->signal_eos().connect (sigc::ptr_fun (&set_processed));
              processor->signal_position().connect (sigc::bind (sigc::ptr_fun (&set_progress_bar), progress_processing, processor));
              processor->set_uri (ustring((*m_iter)[tracks_local.location]));
              processor->run ();

              while (!processed)
                {
                  while (gtk_events_pending())
                    {
                      gtk_main_iteration();
                    }
                }

              boost::optional<ustring> puid    = processor->get_puid ();
              boost::optional<ustring> artist  = processor->get_artist ();
              boost::optional<ustring> title   = processor->get_title ();

              delete processor;

              if (puid)
                {
                  set_local_puidstate (m_iter, P_PUID);
                  (*m_iter)[tracks_local.puid] = puid.get();
                  MBTrackV tracks;
                  mb_tracks_by_puid (puid.get(), tracks);
                  for (unsigned int n = 0; n < tracks.size(); ++n)
                    {
                      MBTrack const& track = tracks[n];
                      if (track.release_id && track.release_id.get() == m_release.release_id)
                        {
                          set_local_puidstate (m_iter, P_PUID_METADATA);
                          (*m_iter)[tracks_local.title]       = track.track_title; 
                          (*m_iter)[tracks_local.tracknumber] = track.tracknumber; 
                          (*m_iter)[tracks_local.track]       = track;
                          break;
                        }
                    }
                }
          }

          m_actions->get_action (ACTION_SCAN_RELEASE)->set_sensitive (true);
          m_actions->get_action (ACTION_SCAN_ALL)->set_sensitive (true);
          m_actions->get_action (ACTION_UPDATE)->set_sensitive (true);

          merge_cb_artist->set_sensitive (true);
          merge_cb_album->set_sensitive (true);

          ok->set_sensitive (true);
          cancel->set_sensitive (true);

          set_sensitive (true);

          progress_processing->set_sensitive (false);
          progress_processing->property_fraction() = 0.;
      }

      void
      LibraryUiModify::on_action_scan_file ()
      {
          using namespace Bmp::MusicBrainzXML;
          using namespace Gtk;
          using namespace std;
          using namespace Glib;

          if (view_local->get_selection()->count_selected_rows() < 1) return;

          m_actions->get_action (ACTION_SCAN_RELEASE)->set_sensitive (false);
          m_actions->get_action (ACTION_SCAN_FILE)->set_sensitive (false);
          m_actions->get_action (ACTION_SCAN_ALL)->set_sensitive (false);
          m_actions->get_action (ACTION_UPDATE)->set_sensitive (false);

          merge_cb_artist->set_sensitive (false);
          merge_cb_album->set_sensitive (false);

          ok->set_sensitive (false);
          cancel->set_sensitive (false);

          set_sensitive (false);

          progress_processing->set_sensitive (true);
          progress_processing->property_fraction() = 0.; 

          Gtk::TreeModel::iterator m_iter = view_local->get_selection()->get_selected ();
          Bmp::Audio::ProcessorPUID *processor = new Bmp::Audio::ProcessorPUID;

          processed = false;
          processor->signal_eos().connect (sigc::ptr_fun (&set_processed));
          processor->signal_position().connect (sigc::bind (sigc::ptr_fun (&set_progress_bar), progress_processing, processor));
          processor->set_uri (ustring((*m_iter)[tracks_local.location]));
          processor->run ();
        
          while (!processed)
          {
            while (gtk_events_pending()) gtk_main_iteration();
          }

          boost::optional<ustring> puid    = processor->get_puid ();
          boost::optional<ustring> artist  = processor->get_artist ();
          boost::optional<ustring> title   = processor->get_title ();

          delete processor;

          if (puid)
            {
              set_local_puidstate (m_iter, P_PUID);
              MBTrackV tracks;
              mb_tracks_by_puid (puid.get(), tracks);
              for (unsigned int n = 0; n < tracks.size(); ++n)
                {
                  MBTrack const& track = tracks[n];
                  if (track.release_id && track.release_id.get() == m_release.release_id)
                    {
                      set_local_puidstate (m_iter, P_PUID_METADATA);
                      (*m_iter)[tracks_local.puid]        = puid.get();
                      (*m_iter)[tracks_local.title]       = track.track_title; 
                      (*m_iter)[tracks_local.tracknumber] = track.tracknumber; 
                      (*m_iter)[tracks_local.track]       = track;
                      break;
                    }
                }
            }

          m_actions->get_action (ACTION_SCAN_RELEASE)->set_sensitive (true);
          m_actions->get_action (ACTION_SCAN_ALL)->set_sensitive (true);
          m_actions->get_action (ACTION_UPDATE)->set_sensitive (true);

          merge_cb_artist->set_sensitive (true);
          merge_cb_album->set_sensitive (true);

          ok->set_sensitive (true);
          cancel->set_sensitive (true);

          set_sensitive (true);

          progress_processing->set_sensitive (false);
          progress_processing->property_fraction() = 0.;
      }

      void
      LibraryUiModify::on_action_scan ()
      {
          using namespace Bmp::MusicBrainzXML;
          using namespace Gtk;
          using namespace std;
          using namespace Glib;

          set_sensitive (false);

          progress_processing->set_sensitive (true);
          progress_processing->property_fraction() = 0.;

          Gtk::TreeModel::iterator m_iter = view_local->get_selection()->get_selected ();
          Bmp::Audio::ProcessorPUID *processor = new Bmp::Audio::ProcessorPUID;

          processed = false;
          processor->signal_eos().connect (sigc::ptr_fun (&set_processed));
          processor->signal_position().connect
            (sigc::bind (sigc::ptr_fun (&set_progress_bar), progress_processing, processor));
          processor->set_uri (ustring((*m_iter)[tracks_local.location]));
          processor->run ();
        
          while (!processed)
          {
              while (gtk_events_pending())
              {
                gtk_main_iteration();
              }
          }

          boost::optional<ustring> puid = processor->get_puid ();

          delete processor;

          progress_processing->set_sensitive (false);
          progress_processing->property_fraction() = 0.;

          if (puid)
            {
              MBTrackV tracks; 
              mb_tracks_by_puid (puid.get(), tracks);
  
              MBReleaseV releases;    
              for (MBTrackV::const_iterator i = tracks.begin(); i != tracks.end(); ++i)
                {
                  if (i->release_id)
                    {
                      mb_releases_by_id (i->release_id.get(), releases);
                    }
                }
              populate_data (releases);
            }
          set_sensitive (true);
      }

#endif //HAVE_OFA

      void
      LibraryUiModify::on_action_update ()
      {
        using namespace Bmp::MusicBrainzXML;
        using namespace Gtk;
        using namespace std;
        using namespace Glib;

        set_sensitive (false);
        MBReleaseV releases; 
        mb_query_releases (merge_cb_artist_entry->get_text (), merge_cb_album_entry->get_text (), releases);
        populate_data (releases);
        populate_local_store (ORDER_AUTO);
        set_sensitive (true);
      }

      void
      LibraryUiModify::on_action_use_filenames_toggled ()
      {
        if (m_action_use_filenames->get_active())
          {
            m_actions->get_action (ACTION_RELAYOUT_TRACKNUMBERS)->set_sensitive (false);
            m_current_ordering = ORDER_LEVENSHTEIN;
            populate_local_store (m_current_ordering); 
          }
        else
          {
            m_actions->get_action (ACTION_RELAYOUT_TRACKNUMBERS)->set_sensitive (true);
            m_current_ordering = ORDER_AUTO;
            populate_local_store (m_current_ordering); 
          }
      }

      void
      LibraryUiModify::populate_local_store (OrderingType ordering_type_specified)
      {
          using namespace Bmp::DB;
          using namespace Bmp::Library;
          using namespace Bmp::MusicBrainzXML;
          using namespace std;
          using namespace Gtk;
          using namespace Glib;

          store_local->clear ();

          TreeModel::Children const& nodes = store_release->children();

          if (nodes.size() == 0)
            {
              for (unsigned int n = 0 ; n < m_list.size(); ++n) store_local->append ();
              ordering_type_specified = ORDER_LINEAR;
            }
          else
            {
              for (unsigned int n = 0 ; n < nodes.size(); ++n) store_local->append ();
            }

          VRows vector;

#ifdef HAVE_HAL
          for (::Bmp::VUriVrpPair::const_iterator i = m_list.begin (); i != m_list.end(); ++i)
          {
            VAttributes attrs; 
            UriVrpPair const& pair (*i);

            attrs.push_back (Attribute (DB::EXACT,
                                        MetadatumId (DATUM_HAL_VOLUME_UDI),
                                        ustring(pair.first))); 

            attrs.push_back (Attribute (DB::EXACT,
                                        MetadatumId (DATUM_VOLUME_RELATIVE_PATH),
                                        ustring(pair.second)));
            library->query (attrs, vector);
          }

#else
          for (::Bmp::VUri::const_iterator i = m_list.begin (); i != m_list.end(); ++i)
          {
            VAttributes attrs; 
            attrs.push_back (Attribute (DB::EXACT, MetadatumId (DATUM_LOCATION), (*i)));
            library->query (attrs, vector);
          }
#endif //HAVE_HAL

          // Determine the ordering type
          std::set<unsigned int> tracknumbers;
  
          OrderingType ordering_type = (ordering_type_specified == ORDER_AUTO)  ? ORDER_TRACKNUMBERS 
                                                                                : ordering_type_specified;

          if (ordering_type_specified == ORDER_AUTO)
          {
              for (VRows::const_iterator v = vector.begin (), end = vector.end (); v != end; ++v) 
              { 
                  Track track (*v);

                  oustring sort_name;

                  if (m_action_use_filenames->get_active())
                    sort_name = track.title;
                  else
                    sort_name = Glib::path_get_basename (Glib::filename_from_uri (track.location.get()));

                  if (!sort_name)
                  {
                    ordering_type = ORDER_LINEAR;
                    break;
                  }

                  if (track.tracknumber &&
                        (tracknumbers.find (track.tracknumber.get()) != tracknumbers.end()))
                  {
                    ordering_type = ORDER_LINEAR;
                    break;
                  }

                  if (track.tracknumber && track.tracknumber.get() == 0)
                  {
                    ordering_type = ORDER_LEVENSHTEIN;      
                    break;
                  }

                if (track.tracknumber) tracknumbers.insert (track.tracknumber.get());

              }
          }

          m_current_ordering = ordering_type;

          int path_index = 0;
          for (VRows::const_iterator v = vector.begin (), end = vector.end (); v != end; ++v) 
          { 
              Track track (*v);

              int index = -1;

              switch (ordering_type)
              {                 
                case ORDER_LEVENSHTEIN:
                {
                  ustring::size_type ld = ustring::npos; 

                  for (TreeModel::Children::const_iterator n = nodes.begin() ; n != nodes.end() ; ++n)
                  {
                      ustring::size_type
                        calculated_ld = ld_distance<Glib::ustring> (track.title.get(), (*n)[tracks_release.title]);

                      if (calculated_ld < ld)
                        {
                          ld = calculated_ld;
                          index = store_release->get_path (n).get_indices().data()[0];
                        }
                  };
                  break;
                }

                case ORDER_TRACKNUMBERS:
                {
                  index = track.tracknumber.get() - 1;
                  break;
                }

                case ORDER_LINEAR:
                {
                  index = path_index++; 
                  break;
                }

                case ORDER_AUTO: // Handled
                  break;
              }

              TreeModel::Path path;
              path.append_index (index);

              TreeModel::iterator iter_r = store_local->get_iter (path);

              (*iter_r)[tracks_local.location]  = track.location.get(); 

#ifdef HAVE_HAL
              (*iter_r)[tracks_local.volume_udi] = track.volume_udi.get();
              (*iter_r)[tracks_local.volume_relative_path]  = track.volume_relative_path.get();
#endif //HAVE_HAL

#ifdef HAVE_OFA
              set_local_puidstate (iter_r, track.puid ? P_PUID_METADATA : P_NONE); 
#endif //HAVE_OFA

              if (m_action_use_filenames->get_active())  
                {
                  (*iter_r)[tracks_local.tracknumber]   = 0;
                  (*iter_r)[tracks_local.title]         = ustring (path_get_basename
                                                                  (filename_from_uri (track.location.get())));
                }
              else
                {
                  (*iter_r)[tracks_local.tracknumber] = track.tracknumber ? track.tracknumber.get() : 0;
                  (*iter_r)[tracks_local.title] =
                    (!track.title || !track.title.get().size()) ? ustring (path_get_basename 
                                                                          (filename_from_uri (track.location.get())))
                                                                : track.title.get();
                }

              (*iter_r)[tracks_local.modify] = true;
          }

          view_local->columns_autosize ();
      }

      void
      LibraryUiModify::album_changed ()
      {
          using namespace Bmp::DB;
          using namespace Bmp::Library;
          using namespace Bmp::MusicBrainzXML;
          using namespace std;
          using namespace Gtk;
          using namespace Glib;

          if (merge_cb_album->get_active_row_number () < 0) return;

          set_sensitive (false);

          //--- Set Up Release 
          TreeModel::const_iterator m_iter  = merge_cb_album->get_active ();
          ReleaseMap::iterator r_iter       = merge_releases.find ((*m_iter)[albums.id]);
          MBRelease const& release          = r_iter->second;
          try {
              if (release.asin)
                {
                  Glib::RefPtr<Gdk::Pixbuf> cover;
                  Amazon::fetch( release.asin.get(), cover );
                  merge_coverart->set (cover->scale_simple (128, 128, Gdk::INTERP_BILINEAR));
                  merge_asin->set_text (release.asin.get());
                }
              else throw 0;
            }
          catch (...)
            { 
              merge_coverart->set (m_album_unknown);
            }

          merge_artist_id->set_text   (release.artist_id);
          merge_release_id->set_text  (release.release_id);

          if (release.score)
            {
              merge_score->set_text ((boost::format ("%d") % release.score.get()).str());
            }

          if (release.track_list)
            {
              l_tracks->set_text ((boost::format ("%d") % release.track_list.get()).str());
            }

          MBReleaseV releases;
          mb_releases_by_id (release.release_id, releases);

          m_release = releases[0];
          MBTrackV const& tracks (releases[0].tracks.get());

          store_release->clear ();

          for (MBTrackV::const_iterator i = tracks.begin() ; i != tracks.end() ; ++i)
          {
            TreeModel::iterator iter_l              = store_release->append();
            (*iter_l)[tracks_release.title]         = i->track_title; 
            (*iter_l)[tracks_release.tracknumber]   = i->tracknumber; 
            (*iter_l)[tracks_release.track]         = (*i);
          }

          view_release->columns_autosize ();
          populate_local_store (ORDER_AUTO);

#ifdef HAVE_OFA
          m_actions->get_action (ACTION_SCAN_ALL)->set_sensitive (true);
#endif //HAVE_OFA

          ok->set_sensitive (true);
          set_sensitive (true);
      }

      void
      LibraryUiModify::artist_changed ()
      {
          using namespace Gtk;
          using namespace MusicBrainzXML;

          if (merge_cb_artist->get_active_row_number () < 0) return;

          set_sensitive (false);

          // Get releases from selected artist
          TreeModel::const_iterator m_iter = merge_cb_artist->get_active ();
          ArtistReleaseMap::iterator map_iter = merge_map.find ((*m_iter)[artists.id]);
          ReleaseIdV & id_v = map_iter->second;
          merge_cb_album_store->clear ();

          typedef std::set<Glib::ustring> ReleaseSet;
          ReleaseSet release_set;

          for (ReleaseIdV::const_iterator r_iter  = id_v.begin (),
                                          r_end   = id_v.end ();
                                          r_iter != r_end; ++r_iter)
            {
              // Iter from releases map
              ReleaseMap::iterator map_iter = merge_releases.find (*r_iter);

              // Get the release based on id 
              MBRelease const& release = map_iter->second; 

              if (release_set.find(release.release_id) != release_set.end()) continue;
              release_set.insert (release.release_id);

              TreeModel::iterator c_iter = merge_cb_album_store->append ();
              (*c_iter)[albums.name] = release.release_title;
              (*c_iter)[albums.id] = release.release_id;
            }
      
          merge_cb_album_entry->set_text("");
          store_release->clear();
          populate_local_store (ORDER_LINEAR);

#ifdef HAVE_OFA
          m_actions->get_action (ACTION_SCAN_FILE)->set_sensitive (false);
          m_actions->get_action (ACTION_SCAN_ALL)->set_sensitive (false);
#endif

          ok->set_sensitive (false);

          if (merge_cb_album_store->children().size() == 1)  
            {
              merge_cb_album->set_active (0);
            }

          set_sensitive (true);
      }

#ifdef HAVE_HAL
      int
      LibraryUiModify::run (MusicBrainzXML::MBReleaseV const& releases,
                            Bmp::VUriVrpPair const& list,
                            ModifyTracks & modify_tracks)
#else
      int
      LibraryUiModify::run (MusicBrainzXML::MBReleaseV const& releases,
                            Bmp::VUri const& list,
                            ModifyTracks & modify_tracks)
#endif
      {
          using namespace Gtk;
          using namespace Glib;
          using namespace Bmp::MusicBrainzXML;
          using namespace Bmp::Library;
          using namespace std;

          m_list = list;
          m_list_original = list;

          merge_cb_artist_store = ListStore::create (artists);
          merge_cb_album_store  = ListStore::create (albums);

          for (MBReleaseV::const_iterator r_iter  = releases.begin(),
                                          r_end   = releases.end();
                                          r_iter != r_end; ++r_iter)
          {
            MBRelease const& release = (*r_iter);
            merge_releases.insert (std::make_pair (release.release_id, release));
        
            ArtistReleaseMap::iterator map_iter = merge_map.find (release.artist_id);

            if (map_iter != merge_map.end ())
              {
                ReleaseIdV &id_v = map_iter->second;
                id_v.push_back (release.release_id);
              }
            else
              {
                ReleaseIdV id_v; 
                TreeModel::iterator m_iter = merge_cb_artist_store->append ();
                (*m_iter)[artists.name]	= release.artist_name;
                (*m_iter)[artists.id]	= release.artist_id;
                id_v.push_back (release.release_id);
                merge_map.insert (std::make_pair (release.artist_id, id_v)); 
              }
          }

          // Setup 'Artist' ComboBox 
          merge_cb_artist->set_model (merge_cb_artist_store);
          merge_cb_artist->signal_changed().connect
            (sigc::mem_fun (this, &Bmp::LibraryUiModify::artist_changed));
          merge_cb_artist->set_text_column (0);
    
          // Setup 'Album' ComboBox 
          merge_cb_album->set_model (merge_cb_album_store);
          merge_cb_album->signal_changed().connect
            (sigc::mem_fun (this, &Bmp::LibraryUiModify::album_changed));
          merge_cb_album->set_text_column (0);

          if (!merge_cb_artist_store->children().empty())
              merge_cb_artist->set_active (0);
          if (!merge_cb_album_store->children().empty())
              merge_cb_album->set_active (0);
          else
              populate_local_store (m_current_ordering);

          if (Gtk::Dialog::run ()  == GTK_RESPONSE_OK)
            {
              TreeNodeChildren const& nodes_release = store_release->children();
              TreeNodeChildren const& nodes_local   = store_local->children();

              unsigned int max_nodes = nodes_release.size();
              if (nodes_local.size() < max_nodes) max_nodes = nodes_local.size();

              TreeNodeChildren::const_iterator iter_l = nodes_release.begin();
              TreeNodeChildren::const_iterator iter_r = nodes_local.begin();

              boost::optional<ustring> genre;
              unsigned int date = 0;

              if (cb_modify_genre->get_active())
                genre = cb_genre_entry->get_text ();

              if (m_release.date)
                sscanf (m_release.date.get().c_str(), "%04u", &date);

              for (unsigned int n = 0; n < max_nodes; ++n)
              {
                  if ((*iter_r)[tracks_local.modify]) 
                  {
                      UTrack u;

                      MBTrack const& track ((*iter_l)[tracks_release.track]);

                      mb_track_to_update_track (track, u);
                      mb_release_to_update_track (m_release, u);

                      u.location_cur = (*iter_r)[tracks_local.location];
                      u.genre        = genre; 
                      u.date         = date;
                      u.new_item     = true;

#ifdef HAVE_OFA
                      if (PuidState((*iter_r)[tracks_local.puidstate]) == P_PUID_METADATA) 
                        u.puid = (*iter_r)[tracks_local.puid];
#endif //HAVE_OFA

#ifdef HAVE_HAL
                      u.volume_udi            = (*iter_r)[tracks_local.volume_udi];
                      u.volume_relative_path  = (*iter_r)[tracks_local.volume_relative_path];
#endif //HAVE_HAL
                      modify_tracks.tracks.push_back (u);
                  }
                ++iter_l;
                ++iter_r;
              }

              modify_tracks.album.asin        = m_release.asin;
              modify_tracks.album.artist      = m_release.artist_name;
              modify_tracks.album.album       = m_release.release_title;
              modify_tracks.album.release_id  = m_release.release_id;
              modify_tracks.album.artist_id   = m_release.artist_id;
              modify_tracks.album.sortname    = m_release.artist_sort_name; 
              modify_tracks.album.tracks      = modify_tracks.tracks.size(); 
              modify_tracks.album.new_item    = true;
              modify_tracks.album.date        = date;

              modify_tracks.album.key         = ::Bmp::Library::Album::create_key (modify_tracks.album);
              modify_tracks.album.tpl         = ::Bmp::Library::Album::create_tpl (modify_tracks.album);

              return GTK_RESPONSE_OK;

            }
          else
            {
              return GTK_RESPONSE_CANCEL;
            }
      }
}
