//  BMPx - The Dumb Music Player
//  Copyright (C) 2005-2006 BMPx development team.
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//  --
//
//  The BMPx project hereby grants permission for non GPL-compatible GStreamer
//  plugins to be used and distributed together with GStreamer and BMPx. This
//  permission is above and beyond the permissions granted by the GPL license
//  BMPx is covered by.

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif //HAVE_CONFIG_H

#include <glib/gi18n.h>
#include <gdk/gdk.h>
#include <gdk/gdkkeysyms.h>
#include <cairomm/cairomm.h>
#include <boost/format.hpp>
#include <cmath>

#include "main.hh"
#include "paths.hh"

#include <build.h>
#include <revision.h>

#include "ui_toolbox.hh"
#include "dialog-about.hh"


namespace
{
  const int mask_alpha_threshold = 128;

  // Animation settings

  const int animation_fps = 24;
  const int animation_frame_period_ms = 1000 / animation_fps;

  const char  *text_font          = "Sans";
  const int    text_size_px       = 15;
  const double text_colour[3]     = { 1.0, 1.0, 1.0 };
  const double text_bottom_margin = 16;
  const double text_fade_in_time  = 0.8;
  const double text_fade_out_time = 0.5;
  const double text_hold_time     = 1.3;
  const double text_time          = text_fade_in_time + text_fade_out_time + text_hold_time;
  const double text_full_alpha    = 0.90;

  const double initial_delay      = 2.0;

  // Credits text
  // Each list of names is sorted in dictionary order, first according
  // to last name, then according to first name.

  const char *credit_text[] =
  {
      N_("<big><b>BMP is brought to you by</b></big>"),
      "Chong Kai Xiong",
      "Milosz Derezynski",
      "David Le Brun",
      N_("<big><b>With contributions from</b></big>"),
      "Saleem Abdulrasool",
      "Olivier Blin",
      "Roman Bogorodskiy",
      "Shlomi Fish",
      "Robert Kruus",
      "Kenneth Ostby",
      "Mart Raudsepp",
      "Martin Schlemmer",
      N_("<big><b>Translations by</b></big>"),
      "Afrikaan: Martin Schlemmer",
      "Brazilian Portuguese: Renato Atilio",
      "Czech: Ondrej Novy",
      "Dutch: Niels Abspoel",
      "Estonian: Hannes Tarien",
      "Finnish: Tero Ratilainen",
      "French: David Le Brun, Olivier Blin",
      "Georgian: George Machitidze",
      "German: Milosz Derezynski",
      "Greek: Stavros Giannouris",
      "Hungarian: Balint Erdosi",
      "Italian: Andrea Scialpi",
      "Japanese: Nobuyuki Ito",
      "Lithuanian: Ernestas Liubarskij",
      "Polish: Jaroslaw Foksa, Jacek Wolszczak",
      "Romanian: Sabin Iacob",
      "Russian: Youri",
      "Simplified Chinese: Tao Wei, Huang Huan, Chong Kai Xiong",
      "Spanish: Andres Suarez",
      "Swedish: Peter Asplund",
      "Thai: Prach Pongpanich",
  };

  const unsigned int n_lines = G_N_ELEMENTS (credit_text);

  const double total_animation_time = n_lines * text_time;
  const double start_time = initial_delay;
  const double end_time   = start_time + total_animation_time;

  inline double
  cos_smooth (double x)
  {
      return (1.0 - std::cos (x * G_PI)) / 2.0;
  }

  const char *
  get_text_at_time (double time)
  {
      if (time >= start_time && time <= end_time)
      {
          unsigned int line = static_cast<unsigned int> ((time - start_time) / text_time);
          return credit_text[line];
      }
      else
      {
          return 0;
      }
  }

  double
  get_text_alpha_at_time (double time)
  {
      if (time >= start_time && time <= end_time)
      {
          double offset = std::fmod ((time - start_time), text_time);

          if (offset < text_fade_in_time)
          {
              return text_full_alpha * cos_smooth (offset / text_fade_in_time);
          }
          else if (offset < text_fade_in_time + text_hold_time)
          {
              return text_full_alpha;
          }
          else
          {
              return text_full_alpha * cos_smooth (1.0 - (offset - text_fade_in_time - text_hold_time) / text_fade_out_time);
          }
      }
      else
      {
          return 0;
      }
  }
}

namespace Bmp
{
  AboutDialog::AboutDialog ()
  {
/*
      static boost::format svn_rev (" (%s-R%s)");
      m_version.append (VERSION);
      if (std::strlen (RV_REVISION))
      {
        m_version.append ((svn_rev % RV_LAST_CHANGED_DATE % RV_REVISION).str());
      }
*/

      set_title (_("About BMP"));
      Util::window_set_icon_list (GTK_WIDGET (gobj ()), "player");

      set_position (Gtk::WIN_POS_CENTER);
      set_resizable (false);

      set_type_hint (Gdk::WINDOW_TYPE_HINT_DIALOG);
      set_decorated (false);

      set_app_paintable (true);
      add_events (Gdk::ALL_EVENTS_MASK);

      gchar const* filename = BMP_IMAGE_DIR G_DIR_SEPARATOR_S "logo-about.png";
      m_background = Gdk::Pixbuf::create_from_file (filename);

      set_size_request (m_background->get_width (), m_background->get_height ());

      m_has_alpha = Util::has_alpha ();

      if (!m_has_alpha)
      {
          Glib::RefPtr<Gdk::Pixmap> pixmap;
          Glib::RefPtr<Gdk::Bitmap> mask;
          m_background->render_pixmap_and_mask (pixmap, mask, mask_alpha_threshold);
          shape_combine_mask (mask, 0, 0);
      }
      else
      {
          set_colormap (Util::get_rgba_colormap());
      }

      m_timer.stop ();
      m_timer.reset ();
  }

  bool
  AboutDialog::on_key_press_event (GdkEventKey *event)
  {
      if (event->keyval == GDK_Escape)
          hide ();

      return false;
  }

  bool
  AboutDialog::on_button_press_event (GdkEventButton *event)
  {
      if (event->button == 1)
          hide ();

      return false;
  }

  bool
  AboutDialog::on_expose_event (GdkEventExpose *event)
  {
      draw_frame ();

      return false;
  }

  bool
  AboutDialog::on_delete_event (GdkEventAny *event)
  {
      hide ();
      return true;
  }

  void
  AboutDialog::on_map ()
  {
      Gtk::Window::on_map ();

      if (m_timer.elapsed () == 0.0)
      {
          m_timer.start ();

          m_update_connection = Glib::signal_timeout ().connect (sigc::mem_fun (this, &AboutDialog::update_frame),
                                                               animation_frame_period_ms);
      }
  }

  void
  AboutDialog::on_unmap ()
  {
      if (m_timer.elapsed () > 0.0)
      {
          m_update_connection.disconnect ();

          m_timer.stop ();
          m_timer.reset ();
      }

      Gtk::Window::on_unmap ();
  }

  void
  AboutDialog::draw_frame ()
  {
      Cairo::RefPtr<Cairo::Context> cr = Util::create_gdk_cairo_context (get_window ());

      if (m_has_alpha)
      {
          cr->set_source_rgba (.0, .0, .0, .0);
          cr->paint ();
      }

      Util::set_cairo_source_pixbuf (cr, m_background, 0, 0);
      cr->set_operator (Cairo::OPERATOR_SOURCE);
      cr->paint ();

      double current_time = m_timer.elapsed ();

      if (current_time >= start_time && current_time <= end_time)
      {
          Pango::FontDescription font_desc (text_font);

          int text_size_pt = static_cast<int> ((text_size_px * 72) / Util::screen_get_y_resolution (Gdk::Screen::get_default ()));
          font_desc.set_size (text_size_pt * PANGO_SCALE);

          char const* text  = get_text_at_time (current_time);
          double      alpha = get_text_alpha_at_time (current_time);

          Glib::RefPtr<Pango::Layout> layout = Glib::wrap (pango_cairo_create_layout (cr->cobj ()));
          layout->set_font_description (font_desc);

          layout->set_markup (_(text));

          int width, height;
          layout->get_pixel_size (width, height);

          cr->move_to ((m_background->get_width () - width) / 2,
                        m_background->get_height () - height - text_bottom_margin);
          cr->set_source_rgba (text_colour[0], text_colour[1], text_colour[2], alpha);
          cr->set_operator (Cairo::OPERATOR_ATOP);

          pango_cairo_show_layout (cr->cobj (), layout->gobj ());
      }

/*
    Glib::RefPtr<Pango::Layout> layout = Glib::wrap (pango_cairo_create_layout (cr->cobj ()));
    Pango::FontDescription font_desc (text_font); 
    pango_font_description_set_absolute_size (font_desc.gobj(), 10 * PANGO_SCALE);
    layout->set_font_description (font_desc);
    layout->set_markup (m_version); 
    cr->set_operator( Cairo::OPERATOR_ATOP );
    cr->move_to( 50 , m_background->get_height() - 14 ); 
    cr->set_source_rgba( 1. , 1. , 1. , 0.85 ); 
    pango_cairo_show_layout (cr->cobj(), layout->gobj());
*/
  }

  bool
  AboutDialog::update_frame ()
  {
      queue_draw ();

      return true;
  }

} // namespace Bmp
