//  BMPx - The Dumb Music Player
//  Copyright (C) 2005-2006 BMPx development team.
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//  --
//
//  The BMPx project hereby grants permission for non GPL-compatible GStreamer
//  plugins to be used and distributed together with GStreamer and BMPx. This
//  permission is above and beyond the permissions granted by the GPL license
//  BMPx is covered by.

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif //HAVE_CONFIG_H

#include <iostream>
#include <sstream>

#include <glibmm.h>
#include <glib/gstdio.h>

#include <sqlite3_bmp.h>

#include <boost/tuple/tuple.hpp>
#include <boost/variant.hpp>
#include <boost/format.hpp>

#include "debug.hh"
#include "database.hh"

namespace
{
  const char* sql_valuenames[] =
  {
      "BOOL", 
      "INTEGER",
      "REAL",
      "TEXT"
  };

  static boost::format format_int  ("%llu");
  static boost::format format_bool ("%d");
  static boost::format format_real ("%f");

  static boost::format sql_error_f ("SQLite Error: %s, SQLite Error Code: %d, SQL String: '%s'");

  static boost::format sql_value_int_f ("'%s' %s DEFAULT 0 ");
  static boost::format sql_value_str_f ("'%s' %s DEFAULT NULL ");

  static boost::format sql_value_eq_f ("%s='%s' ");

  std::string attribute_names (Bmp::DB::ValueMap const& map) 
  {
    using namespace Bmp::DB;

    std::string attrs;

    for (ValueMap::const_iterator i = map.begin(), i_last = --map.end(); i != map.end(); ++i)
    {
      attrs.append(i->first);
      if (i != i_last)
        attrs.append(" , ");
    }
    return attrs;
  }

  std::string variant_deserialize (Bmp::DB::Variant const& value)
  {
    static boost::format sql_format_value_f ("'%s'");

    using namespace Bmp;
    using namespace Bmp::DB;
    using namespace Glib; 

    switch (value.which())
    {
      case VALUE_TYPE_STRING:
        return (sql_format_value_f % sql_uprintf ("%q", boost::get<std::string>(value).c_str())).str();

      case VALUE_TYPE_INT:
        return (sql_format_value_f % (format_int % boost::get<uint64_t>(value)).str()).str();

      case VALUE_TYPE_BOOL:
        return (sql_format_value_f % (format_bool % boost::get<bool>(value)).str()).str();

      case VALUE_TYPE_REAL:
        return (sql_format_value_f % (format_real % boost::get<double>(value)).str()).str();
    }

    return std::string(); // borken GCC!
  }

  int statement_prepare (sqlite3_stmt **pStmt, Glib::ustring const& sql, sqlite3 *sqlite)
  {
      const char *  tail;
      int           counter =0;

      int status = sqlite3_prepare (sqlite,
                                sql.c_str(), 
                                strlen (sql.c_str()),
                                pStmt,
                                &tail);      
  
      if (status == SQLITE_SCHEMA)
        {
          sqlite3_reset (*pStmt);
          int status = sqlite3_prepare (sqlite,
                                sql.c_str(), 
                                strlen (sql.c_str()),
                                pStmt,
                                &tail);      
        }

      if (status)
        {
          g_warning ("SQL Error: '%s', SQL Statement: '%s'", sqlite3_errmsg (sqlite), sql.c_str ());
          sqlite3_reset (*pStmt);
        };

      return status;
  }

  void get_row_from_statement (sqlite3_stmt *pStmt, Bmp::DB::Row& row, Bmp::DB::ValueMap const& map)
  {
      using namespace Bmp::DB;
      using namespace Glib; 
      using namespace std;

      unsigned int counter = 0;
      for (ValueMap::const_iterator i = map.begin (); i != map.end (); ++i)
      {
        switch (i->second)
          {
            case VALUE_TYPE_BOOL:
              row.insert (make_pair (i->first, bool(sqlite3_column_int (pStmt, counter++))));
              break;

            case VALUE_TYPE_INT:
              row.insert (make_pair (i->first, uint64_t(sqlite3_column_int (pStmt, counter++))));
              break;

            case VALUE_TYPE_REAL:
              row.insert (make_pair (i->first, double(sqlite3_column_double (pStmt, counter++))));
              break;

            case VALUE_TYPE_STRING:
              {
                const char * value = (const char*)sqlite3_column_text (pStmt, counter++);        
                if (value && strlen(value))
                  {
                    row.insert (make_pair (i->first, ustring(value)));
                    break;
                  }
              }
            default: break;
          }
      }
  }

  void print_compiled_statement (sqlite3_stmt *stmt)
  {
    int count = sqlite3_bind_parameter_count (stmt);
    for (int n = 0; n < count; ++n)
    {
      g_message ("Parameter %d: %s", n, sqlite3_bind_parameter_name (stmt, n));
    }
  }

};

namespace Bmp
{
  Glib::ustring
  sql_uprintf (const char *format, ...)
  {
    va_list args;
    va_start (args, format);
    char *v = sqlite3_vmprintf (format, args);
    va_end (args);
    Glib::ustring r (v);
    sqlite3_free (v);
    return r;
  }

  namespace DB
  {
      Glib::ustring attributes_deserialize (Bmp::DB::VAttributes const& attributes)
      {
          using namespace Bmp;
          using namespace Bmp::DB;
          using namespace Glib;

          std::string sql;

          for (VAttributes::const_iterator i = attributes.begin(), i_last = --attributes.end(); i != attributes.end(); ++i)
          {
            MatchStyle    const   match_style     (boost::get<0>(*i));
            Variant       const&  value           (boost::get<2>(*i));
            std::string   const&  attribute_name  (boost::get<1>(*i));
          
            bool use_op = true;

            if (match_style == FUZZY)
              {
                  switch (value.which())
                    {
                      case VALUE_TYPE_STRING: 
                        {
                          sql.append (sql_uprintf (" %s LIKE '%%%q%%' ",
                              attribute_name.c_str(), locale_to_utf8 (boost::get<std::string>(value)).c_str())); 
                          break;
                        }

                      case VALUE_TYPE_INT:
                        {
                          sql .append (attribute_name)  
                              .append (" LIKE '%")
                              .append ((format_int % boost::get<uint64_t>(value)).str()).append ("%' ");
                          break;
                        }

                      case VALUE_TYPE_BOOL:
                        {
                          sql .append (attribute_name)
                              .append (" LIKE '%")
                              .append ((format_int % uint64_t(boost::get<bool>(value))).str()).append ("%' ");
                          break;
                        }
                    }
              }
            else
            if (match_style == NOT_NULL)
              {
                sql.append (sql_uprintf (" %s NOT NULL ", attribute_name.c_str()));
                use_op = false;
              }
            else
            if (match_style == IS_NULL)
              {
                sql.append (sql_uprintf (" (%s IS NULL OR length(%s) = 0) ",
                    attribute_name.c_str(), attribute_name.c_str()));
                use_op = false;
              }
            else
              {
                std::string op;

                switch (match_style)
                {
                  case EXACT:
                  {
                    op = "="; break;
                  }

                  case LESSER_THAN:
                  {
                    op = "<"; break;
                  }

                  case GREATER_THAN:
                  {
                    op = ">"; break;
                  }

                  case LESSER_THAN_OR_EQUAL:
                  {
                    op = "<="; break;
                  }

                  case GREATER_THAN_OR_EQUAL:
                  {
                    op = ">="; break;
                  }

                  case IS_NULL:
                  case NOT_NULL:
                  case FUZZY:
                  case NOOP:
                  {
                      // do nothing, all are handled above (and NOOP needs nothing)
                  }
                }

                if (use_op)
                  {
                    switch (value.which())
                      {
                        case VALUE_TYPE_STRING:
                          {
                            sql.append (sql_uprintf (" %s %s '%q' ",
                                attribute_name.c_str(), op.c_str(), locale_to_utf8 (boost::get<std::string>(value)).c_str())); 
                            break;
                          }

                        case VALUE_TYPE_INT:
                          {
                            sql .append (attribute_name)
                                .append (" ")
                                .append (op)
                                .append (" '")
                                .append ((format_int % boost::get<uint64_t>(value)).str()).append ("' ");
                            break;
                          }

                        case VALUE_TYPE_BOOL:
                          {
                            sql .append (attribute_name)
                                .append (" ")
                                .append (op)
                                .append (" '")
                                .append ((format_int % int(boost::get<bool>(value))).str()).append ("' ");
                            break;
                          }
                      }
                  }
              }
          if (i != i_last)
            sql.append (" AND ");
        }
        return sql;
      }

      void
      Database::sqlite_trace   (void         *data,
                                const char   *sqltext)
      {
        g_message (G_STRLOC ": %s", sqltext);
      }

      unsigned int
      Database::sqlite_exec_simple (Glib::ustring const& sql) 
      {
        int            status = 0;
        sqlite3_stmt  *pStmt  = 0;
        const char    *tail   = 0;
        unsigned int   rows   = 0;

        debug ("sql", "exec-simple: '%s'", sql.c_str());

        statement_prepare (&pStmt, sql, sqlite);

        if (status != SQLITE_OK)
          {
            throw SqlError (FORMAT_ERROR(sqlite3_errmsg (sqlite), sqlite3_errcode (sqlite), sql));
          }

        for (status = -1; status != SQLITE_DONE; )
          {
            status = sqlite3_step (pStmt);
            if (status == SQLITE_BUSY) continue;
            if (status == SQLITE_DONE) break;
            if (status == SQLITE_ROW) { rows++; continue; }
            if ((status == SQLITE_ERROR) || (status == SQLITE_MISUSE))
              {
                int errcode = sqlite3_errcode (sqlite);
                if (errcode != 0)
                  {
                    sqlite3_reset (pStmt);
                    throw SqlError (FORMAT_ERROR(sqlite3_errmsg (sqlite), sqlite3_errcode (sqlite), sql));
                  }
                else break;
              }
          }
        sqlite3_finalize (pStmt);
        return rows;
      }

      Database::Database (std::string const& name, std::string const& path, DbOpenMode openmode)
        : name (name), path (path)
      {
        using namespace std;
        using namespace Glib;

        string db_filename (build_filename (path, name) + ".mlib");        

        if (file_test (db_filename, FILE_TEST_EXISTS) && openmode == DB_TRUNCATE)
          {
            g_unlink (db_filename.c_str());
          }

        if (sqlite3_open (db_filename.c_str(), &sqlite))
          {
            throw DbInitError ((boost::format("Unable to open database at '%s'") % db_filename).str());
          }

#ifdef SQLITE_TRACE
          sqlite3_trace (sqlite, Database::sqlite_trace, this);
#endif //SQLITE_TRACE

      };

      Database::~Database ()
      {
        sqlite3_close (sqlite);
      };

      void
      Database::create_attribute_view (std::string const& name,
                                       std::string const& view_name,
                                       std::string const& attribute,
                                       ValueMap    const& map)  
      {
        static boost::format sql_view_f ("CREATE VIEW IF NOT EXISTS %s.%s AS SELECT DISTINCT %s FROM %s;");

        mapmap.insert (std::make_pair (name, map));
        std::string sql ((sql_view_f % name % view_name % attribute % name).str());
        sqlite_exec_simple (sql);
      }

      void
      Database::create_view_manual (std::string const& name,
                                    std::string const& view_name,
                                    std::string const& select,
                                    ValueMap    const& map)
      {
        static boost::format sql_view_f ("CREATE VIEW IF NOT EXISTS %s.%s AS %s;");

        mapmap.insert (std::make_pair (view_name, map));
        std::string sql ((sql_view_f % name % view_name % select).str());
        sqlite_exec_simple (sql);
      }

      void
      Database::add  (std::string   const& name,
                      Bmp::DB::Row  const& row)
      {
        if (mapmap.find (name) == mapmap.end ())
          g_error ("No value-map for table '%s'", name.c_str());

        ValueMap const& map (mapmap.find(name)->second);

        std::string sql_keys,
                    sql_values;

        sql_keys .append ("(");
        sql_values .append ("("); 

        for (ValueMap::const_iterator i = map.begin(), i_last = --map.end(); i != map.end(); ++i)
        {
          Row::const_iterator r (row.find (i->first));
          if (r != row.end())
            {
              sql_keys.append (i->first);
              sql_values.append (variant_deserialize (r->second));

              if (i != i_last)
                {
                  sql_keys.append(" , ");
                  sql_values.append(" , ");
                }
            }
        }

        sql_keys.append (")");
        sql_values.append (")");

        static boost::format sql_insert_f ("INSERT INTO %s %s VALUES %s;");
        sqlite_exec_simple ((sql_insert_f % name % sql_keys % sql_values).str());
      }

      void
      Database::del   (std::string const&     name,
                        VAttributes  const& attributes)
      {
        static boost::format sql_delete_f ("DELETE FROM %s WHERE %s;"); 
        sqlite_exec_simple ((sql_delete_f % name % attributes_deserialize (attributes)).str());
      }

      void 
      Database::project (std::string const& name,
                         std::string const& project, 
                         VRows            & vector,
                         VAttributes const&  attributes)
      {
        if (mapmap.find (name) == mapmap.end ())
          g_error ("No value-map for table '%s'", name.c_str());

        static boost::format sql_project_f ("SELECT DISTINCT %s FROM %s;");
        static boost::format sql_project_where_f ("SELECT DISTINCT %s FROM %s WHERE %s;");

        ValueMap const& map (mapmap.find(name)->second);
    
        std::string sql;

        if (!attributes.empty ())
            sql.append ((sql_project_where_f  % attribute_names(map)
                                              % name 
                                              % attributes_deserialize (attributes)).str());
        else
            sql.append ((sql_project_f        % attribute_names(map)
                                              % name).str());

        sqlite3_stmt *pStmt = 0;

        if (statement_prepare (&pStmt, sql, sqlite))
          throw SqlError (FORMAT_ERROR(sqlite3_errmsg (sqlite), sqlite3_errcode (sqlite), sql));

        for (int status = -1 ; status != SQLITE_DONE ;)
          {
            status = sqlite3_step (pStmt);
            switch (status)
              {
                case SQLITE_BUSY: continue;
                case SQLITE_DONE: break;
                case SQLITE_ROW:            
                {
                  Row row;
                  get_row_from_statement (pStmt, row, map);
                  vector.push_back (row);
                  break;
                }
                case SQLITE_ERROR:  sqlite3_reset (pStmt); return; 
                case SQLITE_MISUSE: sqlite3_reset (pStmt); return; 
              } 
          } 
        sqlite3_finalize (pStmt);
      }

      void 
      Database::set     (std::string const& name,
                         std::string const& pkey,
                         std::string const& pkey_value,
                         VAttributes const& attributes)
      {
        std::string set;
            
        for (VAttributes::const_iterator i = attributes.begin(), i_last = --attributes.end(); i != attributes.end(); ++i)
        {
          std::string const& attribute_name = boost::get<1>(*i);
          Variant     const& value          = boost::get<2>(*i);

          switch (value.which())
          {
            case VALUE_TYPE_STRING:
                set.append ((sql_value_eq_f % attribute_name % sql_uprintf("%q", boost::get<std::string>(value).c_str())).str());
                break;

            case VALUE_TYPE_INT:
                set.append ((sql_value_eq_f % attribute_name % (format_int % boost::get<uint64_t>(value)).str()).str());
                break;

            case VALUE_TYPE_BOOL:
                set.append ((sql_value_eq_f % attribute_name % (format_int % int(boost::get<bool>(value))).str()).str());
                break;
          }

          if (i != i_last)
            set.append (" , ");
        }

        static boost::format sql_update_f ("UPDATE %s SET %s WHERE %s='%s';");
        sqlite_exec_simple ((sql_update_f % name % set % pkey % pkey_value).str());
      }

      void 
      Database::set (std::string    const& name,
                     Glib::ustring  const& where_clause,
                     VAttributes    const& attributes)
      {
        std::string set; 

        for (VAttributes::const_iterator i = attributes.begin(), i_last = --attributes.end(); i != attributes.end() ; ++i)
        {
          std::string const& attribute_name = boost::get<1>(*i);
          Variant     const& value          = boost::get<2>(*i);

          switch (value.which())
          {
            case VALUE_TYPE_STRING:
                set.append ((sql_value_eq_f % attribute_name % sql_uprintf("%q", boost::get<std::string>(value).c_str())).str());
                break;

            case VALUE_TYPE_INT:
                set.append ((sql_value_eq_f % attribute_name % (format_int % boost::get<uint64_t>(value)).str()).str());
                break;

            case VALUE_TYPE_BOOL:
                set.append ((sql_value_eq_f % attribute_name % (format_int % int(boost::get<bool>(value))).str()).str());
                break;
          }

          if (i != i_last)
            set.append (" , ");
        }

        static boost::format sql_update_where_f ("UPDATE %s SET %s %s;"); 
        sqlite_exec_simple ((sql_update_where_f % name % set % where_clause).str());
      }

      void 
      Database::get (std::string const& name, sqlite_int64 rowid, Row & row)
      {
        if (mapmap.find(name) == mapmap.end())
          g_error ("No value-map for table '%s'", name.c_str());

        ValueMap const& map = mapmap.find(name)->second;

        static boost::format sql_select_by_rowid_f ("SELECT %s FROM %s WHERE rowid='%lld';");
        std::string sql ((sql_select_by_rowid_f % attribute_names (map) % name % rowid).str());

        sqlite3_stmt *pStmt = 0;

        if (statement_prepare (&pStmt, sql, sqlite))
          throw SqlError (FORMAT_ERROR(sqlite3_errmsg (sqlite), sqlite3_errcode (sqlite), sql));

        for (int status = -1 ; status != SQLITE_DONE ;)
          {
            switch (sqlite3_step (pStmt))
              {
                case SQLITE_BUSY:   continue;
                case SQLITE_DONE:   break;
                case SQLITE_ERROR:  sqlite3_reset (pStmt); return;
                case SQLITE_MISUSE: sqlite3_reset (pStmt); return;
                case SQLITE_ROW:            
                  {
                    get_row_from_statement (pStmt, row, map);
                    sqlite3_finalize (pStmt);
                    return;
                  }
              } 
          } 
        sqlite3_finalize (pStmt);
      }

      void 
      Database::get (std::string const& name,
                     VRows            & vector,
                     VAttributes const& attributes,
                     std::string const& suffix)
      {
        if (mapmap.find (name) == mapmap.end ())
          g_error ("No value-map for table '%s'", name.c_str());

        ValueMap const& map = mapmap.find(name)->second;

        static boost::format sql_select_where_f ("SELECT %s FROM %s WHERE %s %s;");
        static boost::format sql_select_f ("SELECT %s FROM %s %s;");
      
        std::string sql;

        if (!attributes.empty ())
            sql.append((sql_select_where_f
                        % attribute_names (map)
                        % name
                        % attributes_deserialize (attributes)
                        % suffix).str());
        else
            sql.append((sql_select_f
                        % attribute_names (map)
                        % name
                        % suffix).str());

        debug ("sql", "get: '%s'", sql.c_str());

        sqlite3_stmt *pStmt = 0;

        if (statement_prepare (&pStmt, sql, sqlite))
          throw SqlError (FORMAT_ERROR(sqlite3_errmsg (sqlite), sqlite3_errcode (sqlite), sql));
 
        for (int status = -1 ; status != SQLITE_DONE ;)
          {
            switch (sqlite3_step (pStmt))
              {
                case SQLITE_BUSY:   continue;
                case SQLITE_DONE:   break;
                case SQLITE_MISUSE: sqlite3_reset (pStmt); return;
                case SQLITE_ERROR:  sqlite3_reset (pStmt); return;
                case SQLITE_ROW:            
                  {
                    Bmp::DB::Row row;
                    get_row_from_statement (pStmt, row, map);
                    vector.push_back (row);
                    break;
                  }
              } 
          } 
          sqlite3_finalize (pStmt);
      }

      void
      Database::insert_map_for_table (std::string const& name, ValueMap const& map)
      {
        TableMapMap::iterator i = mapmap.find (name);
        if (i != mapmap.end())
          mapmap.erase (i);
        mapmap.insert (std::make_pair (name, map));
      }

      void
      Database::create_table (std::string const&  name,
                              std::string const&  pkey,
                              ValueMap    const&  map,
                              TableRowMode        row_mode)
      {
        static boost::format sql_create_table_f ("CREATE TABLE IF NOT EXISTS %s (%s);");        
        static boost::format sql_pkey_f ("'%s' %s NOT NULL PRIMARY KEY ");
        static boost::format sql_pkey_replace_f ("'%s' %s NOT NULL PRIMARY KEY ");

        mapmap.insert (std::make_pair (name, map));
        std::string create; 

        for (ValueMap::const_iterator i = map.begin(), i_last = --map.end(); i != map.end(); ++i)
          {
            if (!pkey.compare (i->first))
              {
                if (row_mode == ROW_MODE_REPLACE)
                  create.append ((sql_pkey_replace_f % i->first % sql_valuenames[i->second]).str());
                else
                  create.append ((sql_pkey_f % i->first % sql_valuenames[i->second]).str());
              }
            else
              {
                if (i->second == VALUE_TYPE_STRING)
                  create.append ((sql_value_str_f % i->first % sql_valuenames[i->second]).str());
                else
                  create.append ((sql_value_int_f % i->first % sql_valuenames[i->second]).str());
              }

            if (i != i_last)
              create.append (" , ");
          }
        sqlite_exec_simple ((sql_create_table_f % name % create).str());
      }

      void
      Database::drop_table (std::string const& name)
      {
        static boost::format sql_drop_table_f ("DROP TABLE %s;");
        sqlite_exec_simple ((sql_drop_table_f % name).str());
      }

      bool
      Database::table_exists (std::string const& name)
      {
        static boost::format sql_table_exists_f ("SELECT name FROM sqlite_master WHERE name='%s';");
        return (0 != sqlite_exec_simple ((sql_table_exists_f % name).str()));
      }

      bool
      Database::attr_exists (std::string   const& name, 
                             std::string   const& key,
                             Glib::ustring const& value)
      {
        static boost::format sql_attribute_exists_f ("SELECT %s FROM %s WHERE %s='%s';");
        return (0 != sqlite_exec_simple ((sql_attribute_exists_f % key % name % key % sql_uprintf("%q", value.c_str())).str()));
      }
  } 
}; 
