/*
 *  makegtxt.c  --  part of the build process of CPT 4.0
 * 
 *  Author: Jeff Edwards
 * 
 *  This program is hereby placed in the public domain, June 2000,
 *  ---Jeff Edwards---
 *
 * =================================================================
 *
 * This program reads an ASCII text file and builds an equivalent
 * 'C' include file containing the text as a character array, for 
 *  use in a GTK+ text widget.
 *
 * The input file format supports a few special features:
 *     a) Lines beginning with ';' are comments and are ignored.
 *     b) Other lines are parsed into tokens and whitespace. In this
 *        process, any control characters other than new-line characters
 *        are discarded. The trailing whitespace string on each line is
 *        discarded, but embedded space characters are output exactly
 *        as encountered. A single space characters is inserted between
 *        any two tokens that would otherwise be adjacent.
 *     c) The special token '<br>' (same as the HTML break tag) is 
 *        recognized and causes an new-line character to be output 
 *        instead of the tag itself.
 *
 *  Usage:  makegtxt <infile> <outfile> <arrayname>
 *
 *  ----------------------------------------------------------------
 *   Rev 00  05/07/00  From 'makectxt.c'. LGE.
 *  ----------------------------------------------------------------  
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

void output1(int out_char);

FILE *infile, *outfile;

#define STATE_BOL 0
#define STATE_AFTER_TAG 1
#define STATE_SCANNING_TOKEN 2
#define STATE_SCANNING_SPACE 3

int main (int argc, char *argv[])
{
   int c,i,n;
   int state;
   int space_count;
   int token_length;
   int token_was_last;
   char filename[81];
   char token[81];

   if (argc == 4) goto argcok;

error:
   printf("Convert a mostly-ASCII text file to a 'C' language include file\n");
   printf("containing the text as a character array in a format suitable for\n");
   printf("use in a GTK+ text widget. Please read the source code for details.\n");
   printf("Usage: makegtxt <infile> <outfile> <arrayname>\n");
   exit(1);

argcok:

   infile = fopen(argv[1],"rb");
   if (infile == NULL) goto error;

   strcpy(filename,argv[2]);   /* output file name */
   n = strlen(filename);
   for (i=0; i<n; i++) if (filename[i] == '.') break;
   if (i == n) strcat(filename,".c");
   /* 
    *  The remove() forces a new date on the output file. Under DJGPP,
    *  sometimes 'make' acts funny without it.
    */
   remove(filename);
   outfile = fopen(filename,"wb");
   if (outfile == NULL) {
      printf ("Unable to open %s\n",filename);
      goto error;
   }

   fprintf(outfile,"/* Generated from '%s' by 'makegtxt'. */\n",argv[1]);
   fprintf(outfile," char %s[]={\n ",argv[3]);

   token_was_last = 0;

nextline:
   state = STATE_BOL;
   space_count = 0;
   token_length = 0;
   
nextchar:
   c = getc(infile);
   if ((c < ' ') && (c != '\n') && (c != EOF)) goto nextchar;
   if ((state == STATE_BOL) && (c == ';')) {
      while (1) {
	 c = getc(infile);
	 if (c == EOF) goto eof1;
	 if (c == '\n') goto nextchar;
      }
   }
   if ((state == STATE_BOL) || (state == STATE_AFTER_TAG)) {
      if (c == EOF) goto eof1;
      if (c == '\n') goto nextchar;
      if (c == ' ') {
	 state = STATE_SCANNING_SPACE;
	 space_count = 1;
	 goto nextchar;
      }
      state = STATE_SCANNING_TOKEN;
      token_length = 1;
      token[0] = c;
      goto nextchar;
   }
   if (state == STATE_SCANNING_SPACE) {
      if (c == EOF) goto eof1;
      if (c == '\n') goto nextline;
      if (c == ' ') {
	 space_count++;
	 goto nextchar;
      }
      while (space_count) { 
         output1(' ');
         space_count--;
      }
      token[0] = c;
      token_length = 1;
      state = STATE_SCANNING_TOKEN;
      goto nextchar;
   } 
   if (state == STATE_SCANNING_TOKEN) {
      if (c == EOF) {
	 if (token_was_last) output1(' ');
	 for (i=0;i<token_length;i++) output1(token[i]);
	 goto eof1;
      }
      if (c == '\n') {
	 if (token_was_last) output1(' ');
	 for (i=0;i<token_length;i++) output1(token[i]);
	 token_was_last = 1;
	 goto nextline;
      }
      if (c == ' ') {
	 if (token_was_last) output1(' ');
	 for (i=0;i<token_length;i++) output1(token[i]);
	 token_was_last = 1;
	 space_count = 1;
	 goto nextline;
      }
      token[token_length++] = c;
      /*
       *  Here's where we process possibly embedded '<br>' tags.
       */
      if (token_length >= 4) {
	 if (memcmp(token+token_length-4,"<br>",4) == 0) {
            if (token_length > 4) {
	       if (token_was_last) output1(' ');
	       for (i=0;i<token_length-4;i++) output1(token[i]);
	    }
	    output1('\n');
            token_length -= 4;
	    token_was_last = 0;
            state = STATE_AFTER_TAG;
	 }
      }
      goto nextchar;
   }
    
   /*
    *  Currently, we only get here via a 'goto'.
    */
eof1:
   output1(0);
   fclose(infile);
   fprintf(outfile,"};\n");
   fclose(outfile);

   exit(0);
   
}  /* end of main() */

/*------------------------------------------------------------------*/
void output1(int c) /* output one character */
{
   static int charcnt=0;
   
   if (charcnt > 65) {
      fprintf(outfile,",\n ");
      charcnt = 1;
   }
   else if (charcnt > 1) {
      fprintf(outfile,",");
      charcnt++;
   }
   fprintf(outfile,"%d",c);

   charcnt++;
   if (c>9)  charcnt++;
   if (c>99) charcnt++;

   return;
   
}  /* end of output1 */
