/*=====================================================================
 *  do_hart.c  -   compute the Hart solution for CPT4 for Linux.
 * 
 *  Author:  This program is an adaptation of a public domain program
 *           'decp.c', designed by George W. Hart and coded by Jian Wang.
 *           This adaptation was written by Jeff Edwards.
 * 
 *  This adaptation is hereby placed in the public domain, June 2000,
 *  ---Jeff Edwards---
 * 
 * =====================================================================
 * 
 *  This version of the Hart algorithm runs as a child process of CPT4,
 *  and communicates via a set of pipes whose file descriptors are passed
 *  as parameters on the command lines.
 *
 *  Data passed in both directions will consist messages in the form:
 * 
 *     ddmmmm\n
 * 
 *  where 'dd'    is a two digit decimal message length, n, in ASCII;
 *        'mmmm'  is the (n-1) character message body; and
 *        '/n'     is the newline character.
 *  
 *  Before starting us up, CPT4 will write the puzzle word list to
 *  the CPT-to-Hart pipe; the word list will consist of a sequence of
 *  words in the message format, all of which begin with one of the characters
 *  'A' through 'Z', and some of which may contain embedded apostrophe 
 *  characters. The last word in the word list will be followed by the
 *  message "05!EOL\n".
 * 
 *  If the word list if found to be acceptable, we proceed with the solution
 *  algorithm and back each new solution vector as message in the form:
 * 
 *         36!C=ddd,P=vvvvvvvvvvvvvvvvvvvvvvvvvv\n
 * 
 *  where ddd is a three digit 'score' (the number of words in the puzzle
 *  that occur in the dictionarly, and the first v in vvv... is the plaintext
 *  letter currently assigned to cipher text 'A', and so forth. For ciphertext
 *  letters that do not (yet) have a plaintext letter assigned, the
 *  corresponding 'x' will contain ' '. After writing each solution vector,
 *  we wait for a reply back from CPT4, which will consist of one of the
 *  following:
 * 
 *       04!OK\n
 *       05!DIE\n
 * 
 *  If we get an OK reply, we continue on, otherwise we issue the message:
 * 
 *       nn!ERROR:mmmmm\n
 * 
 *  and exit with completion code one. If we get all the way through our
 *  algorithm, we issue the message:
 * 
 *       06!DONE\n
 * 
 *  after the last reply from CPT4 and exit with completion code zero.
 * 
 */
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <setjmp.h>  
#include "decipher.h"   /* the word list */
#define TRUE_ONE     1
#define FALSE        0
#define LEVEL        2  /* sort LEVEL+1 levels */
#define MAXLEN      16  /* longest word length */
#define MAXCRYPT   200  /* Max # of words in cryptogram */
#define THRESHOLD   10  /* Max # of words in a pat set */

#define DEBUG 0

/*
 *  Local typedefs
 */
typedef unsigned char UCHAR;

/*
 *  Each WORDGRP structure is an element of a group of words with the
 *  same repeat character pattern.
 */
typedef struct WordGrp {        /* Group of words with same pattern */
        int             pos;
        int             row;
        char            word[MAXLEN+1];
        struct WordGrp  *next;
}       WORDGRP;

/*
 *  Each WORDLST structure is the head of a chain of WORDGRP structures
 *  of words with the same repeat character.
 */
typedef struct WordLst {        /* List of word groups */
        int             num;
        int             pos;
        int             pa_row;
        int             pa_pos;
        struct WordGrp  *wgp;
        struct WordLst  *next;
}       WORDLST;

/*
 *  Forward function declarations
 */
WORDGRP *alloc_grp();
WORDLST *alloc_lst();
int     consistent(char *w1, char *w2, UCHAR *NewP, UCHAR *NewQ);
int     count(char *word);
void    form_pattern();         /* form and sort word list for each pat in crypto */
void    free_grp(WORDGRP *Wgp);
void    free_lst(WORDLST *Wlp, int k);
void    free_pat(int Is);
int     getmsg(int pipe, char *buf, int bufsize);
int     in_common(char *w1, char *w2);
void    make_word_list();       /* read cryptogram */
int     match(char *w1, char *w2);
int     match2(char *w1, char *x, char *w2, char *y);
int     new_pattern(char *word, WORDGRP *Wgp);   /* test if a word pat is new */
void    output_solution(UCHAR *P, int score);
void    putmsg(int pipe, char *msg);
void    read_dict();            /* read plain word dictionary */
void    reorder(int Is);
WORDGRP *same_grp(int k, int num, WORDGRP *Wgp);
void    solve(int Is, UCHAR *P, UCHAR *Q, int score, int Top);

/*
 * Globals
 */
UCHAR P0[91], Q0[91];   /* P: permutation, Q: CharMap */

extern unsigned char *decipher[];  /* dictionary */
extern int decipher_size; /* number of words in dictionary */

int     NL,             /* # of plain words in dict., a/w *L */
        NC = 0,         /* # of code words, a/w Ch[] */
        NS,             /* # of distinct pats in dict & crypt., a/w I[], K[] */
        K[MAXCRYPT],    /* K[l] - sorted word order index */
        I[MAXCRYPT],    /* I[l] - parent level of l */
        DUP[MAXCRYPT],  /* flag for duplicated words */
        NUM[MAXCRYPT],  /* # of words in a restricted pat set */
        LEN[MAXCRYPT],  /* length of a pattern set */
        NC_USE = 0,     /* # of coded words actually used */
        Mark[MAXCRYPT], /* flag for used words, while constructing restr. pat set */
        MARK[MAXCRYPT], /* global flag for words used as roots */
        EXTRA[MAXCRYPT],/* flag for the last 'solve' */
        High = 0;       /* the highest score */
char    N[MAXCRYPT][MAXCRYPT];  /* # of letters in common btw Wi & Wj */
char    Ch[MAXCRYPT][MAXLEN+1]; /* input words */
WORDGRP *WGP[MAXCRYPT],         /* word list at each level */
        *SET[MAXCRYPT];         /* original plain word list for coded words */
WORDLST *Pat[MAXCRYPT],         /* restricted pat set, given parent set */
        *L;                     /* dictionary list */

int in_pipe;
int out_pipe;
jmp_buf CPT_Jump_Buffer;

int main(int argc, char *argv[])
{
   int i;

   if (argc < 3) exit(1);
   in_pipe  = atoi(argv[1]);
   out_pipe = atoi(argv[2]);
#if (DEBUG)   
   printf("do_hart: entering main()\n");
#endif
   
   make_word_list();
#if (DEBUG)   
      printf("do_hart: received %d word wordlist\n", NC);
#endif
   if (NC < 1) {
      putmsg(out_pipe,"!ERROR: no words to decode.");
      exit(1);
   }
   
   for (i='A'; i<='Z'; i++)        /* initialize the permutation */
           P0[i] = Q0[i] = FALSE;  /* unmarked */
   for (i=0; i<MAXCRYPT; i++)
           MARK[i] = EXTRA[i] = FALSE;
   read_dict();
 
   /*
    *  If the solution code escapes via a longjmp(),
    *  the following setjmp() returns non-zero.
    */
   if (setjmp(CPT_Jump_Buffer) == 0) {
        form_pattern();
        solve(0, P0, Q0, 0, TRUE_ONE);   /* initial call */
   }
   putmsg(out_pipe,"!DONE");
   exit(0);
   
}  /* end of main() */

/*                                                         File: DECP.C
   Cryptogram deciphering program coded by Jian Wang for G. Hart. 
   Version of April 20, 1992.  Requires word list file: DECIPHER.DCT
   See: G. Hart "To Decode Short Cryptograms" Communications of ACM,
   September, 1994, p. 102, for detailed description of the algorithm.
   Note: different word-order heuristics were employed in different 
   versions; I am not sure if this agrees completely with the version
   in the paper.  One difference is that this first organizes the
   dictionary into pattern sets, as mentioned in footnote 1.
   This program is hereby placed in the public domain, 
                                September 1994,  ---George W. Hart--- */

/*  Notes from LGE
 * 
 *  We start with a dictionary of frequently occuring words; the basic
 *  idea is to assign plain text characters to cipher characters in
 *  such a way as to maximize the number of puzzle words that would
 *  occur in the dictionary.
 *
 *  The article text says that the tree is organized as follows:
 *  "For the top-level word, we choose the cipher word for which the
 *  number of distinct letters divided by the size of its pattern set
 *  is maximum. For each lower level, we choose the cipher word having 
 *  the largest number of distinct letters in common with all the cipher
 *  words above."
 * 
 *  "The algorithm consists of the following steps:
 *   1. Input the cipher text and parse it into words.
 *   2. Group the words by pattern, dropping repetitions.
 *   3. Scan the dictionary and construct pattern sets of plain words
 *      for each cipher word. Ignore any for which the pattern set
 *      is empty.
 *   4. Order the cipher words using one or more of the heuristics ...
 *      given above.
 *   5. Call SOLVE(...) described later."
 * 
 *  We call a mapping from cipher text to plain text a 'permutation'.
 *  the solution consists essentially of using the pattern sets to
 *  create all possible consistent permutations, and counting the
 *  number of cipher words that are in the dictionary for each 
 *  permutation. We assume that 'better' solutions are those with
 *  more cipher words appearing in the dictionary.
 * 
 * --------------------------------------------------------
 * 
 *  We rely on 'L' being initialized to NULL.
 *  Mark[] and MARK[] only used in reorder().
 *  DUP[] is set once in form_pattern() and used only in reorder().
 * 
 */
 
/*
 *  This gets called 1 time  from read_dict() and 
 *                   2 times from reorder().
 */
WORDGRP *alloc_grp()
{
        WORDGRP *wgp;

        if ((wgp = (WORDGRP *)malloc(sizeof(WORDGRP))) == NULL) {
/*              fprintf(stderr, "Out of memory - WORDGRP\n"); */
                exit(1);
        }
        return wgp;
}

/*
 *  This gets called 2 times from read_dict() and 
 *                   3 times from reorder().
 */
WORDLST *alloc_lst()
{
        WORDLST *wlp;

        if ((wlp = (WORDLST *)malloc(sizeof(WORDLST))) == NULL) {
/*              fprintf(stderr, "Out of memory - WORDLST\n"); */
                exit(1);
        }
        return wlp;
}


/*
 *  Now this function processes the included array instead
 *  of reading a file.
 * 
 *  We allocate a WORDGRP structure for each word
 */
void    read_dict()
{
        char    word[MAXLEN+1];
        WORDGRP *Wgp;
        WORDLST *Wlp, *l_end=NULL;

        /*
	 *  'decipher_size' is the number of words in dictionary array.
	 */
        for (NL=0; NL<decipher_size; NL++) {
        
            if (strlen((const char *)decipher[NL]) >= MAXLEN) {
/*              fprintf(stderr, "Dictionary word '%s' too long.\n", */
/*                      decipher[NL]);                              */
                exit(1);
            }
            strcpy(word,(const char *)decipher[NL]);
            Wgp = alloc_grp();
            Wgp->row = 0;                       /* initialize as the first row */
            strcpy(Wgp->word, word);            /* word group unit */
            Wgp->next = NULL;

            if (L == NULL) {            /* first word */
                L = l_end = alloc_lst();
                L->num = 1;
                L->wgp = Wgp;
                L->next = NULL;
            } else
                if (new_pattern(word, Wgp)) {           /* new pattern */
                    Wgp->pos = 0;                       /* first word */
                    Wlp = alloc_lst();
                    Wlp->num = 1;
                    Wlp->wgp = Wgp;
                    Wlp->next = NULL;
                    l_end->next = Wlp;          /* add to the end */
                    l_end = Wlp;
                }
        }
}

/*
 *  This routine checks to see if a word has a pattern we haven't
 *  processed before, and if it does not, adds it to the appropriate
 *  existing pattern list.
 */
int     new_pattern(char *word, WORDGRP *Wgp)   /* test if a word pat is new */
{
        int     k;
        WORDLST *wlp;
        WORDGRP *wgp;

        for (wlp=L; wlp!=NULL; wlp=wlp->next) {
            if (match(word, wlp->wgp->word)) {          /* new pattern ? */
                k = 1;
                for (wgp=wlp->wgp; wgp->next!=NULL; wgp=wgp->next)
                        k++;
                Wgp->pos = wgp->pos + 1;        /* pos of word */
                wlp->num = k + 1;       /* num of words in the set */
                wgp->next = Wgp;                /* add to the head */
                return FALSE;
            }
        }
        return TRUE_ONE;
}

/*
 *  Read the wordlist from the pipe.
 */
void make_word_list()
{
   int  i,n;
   char buff[81];
   
   NC = 0;
   
   while (1) {
#if (DEBUG > 1)   
      printf("do_hart: calling getmsg() in make_word_list().\n");
#endif
      n = getmsg(in_pipe, buff, 81);
      if (strcmp(buff,"!EOL") == 0) break;

      for (i=0;i<n;i++) if ((buff[i] < 'A') || (buff[i] > 'Z')) break; 
      if (i != n) continue;
      memcpy(Ch[NC++],buff,n+1);
   }
} /* end of make_word_list() */

/*
 *  Determines whether or not two words have the same repeating character
 *  pattern. This is central to the whole approach to solution.
 */
int     match(char *w1, char *w2)
{
        int     i, j, Len;
		
		Len = strlen(w1);

        if (Len != (int)strlen(w2))
                return FALSE;
        for (i=1; i < Len; i++)
                for (j=0; j<i; j++)
                        if ((w1[i]==w1[j]) != (w2[i]==w2[j]))  /* pat comp */
                                return FALSE;
        return TRUE_ONE;
}

void    free_grp(WORDGRP *Wgp)
{
        WORDGRP *wgp;

        while (Wgp != NULL) {
                wgp=Wgp->next;
                free((void *)Wgp);
                Wgp = wgp;
        }
}

int     count(char *word)
{
        int i, n = 0, P[91], Len;
   
        Len = strlen(word);
        for (i='A'; i<='Z'; i++)
            P[i] = FALSE;               /* flag for unused letter */
        for (i=0; i<Len; i++)
                if (!P[(int)word[i]]) {      /* unrepeated letter */
                    n++;
                    P[(int)word[i]] = TRUE_ONE;
                }
        return n;
}

int     in_common(char *w1, char *w2)
{
        char tmp1[2*MAXLEN+1];

        strcpy(tmp1, w1);         /* don't destroy original word */
        return (count(w1) + count(w2) - count(strcat(tmp1, w2)));
}

/*
 *  Gets called only once, from first_Hart();
 */
void form_pattern()
{
   int i,j,OK;
   WORDLST *wlp;
        /*
	 *  NC is the number of words we're decrypting.
	 */
        for (i=0; i<NC-1; i++)             /* form Nij */
            for (j=i+1; j<NC; j++)
                N[i][j] = N[j][i] = in_common(Ch[i], Ch[j]);
   
        for (j=0; j<NC; j++) {          /* precomputing flags and word pat list */
            DUP[j] = FALSE;
            for (i=0; !DUP[j] && i<j; i++)              /* duplication flag */
                if (strcmp(Ch[j], Ch[i]) == 0)
                        DUP[j] = TRUE_ONE;
            OK = FALSE;
	    /*
	     *  'L' points to the first element of the dictionary word list.
	     */
            for (wlp=L; !OK && wlp!=NULL; wlp=wlp->next) /* check word in dict */
		/*
		 *  'match()' checks for a repeated character pattern match
		 */
                if (match(Ch[j], wlp->wgp->word)) {
                    OK = TRUE_ONE;
                    SET[j] = wlp->wgp; /* assign list for each word */
                                /* and add a flag word to help extra search */
                    LEN[j] = wlp->num;
                }
            if (!OK || DUP[j])          /* no matched pat set */
                LEN[j] = 0;
            else
                NC_USE++;
        }

        /* release unused memory */
        for (wlp = L; wlp!= NULL; wlp=wlp->next) {
                OK = TRUE_ONE;                      /* unused mem */
                for (j=0; j<NC; j++)
                        if (match(Ch[j], wlp->wgp->word))
                                OK = FALSE;     /* use it */
                if (OK)
                        free_grp(wlp->wgp);
        }
        reorder(0);
   
}  /* end of form_pattern() */

void    free_lst(WORDLST *Wlp, int k)
{
        WORDLST *wlp;

        while (Wlp != NULL) {
                wlp = Wlp->next;
/*              if (Wlp->num != LEN[k]) {               */
                        free_grp(Wlp->wgp);
                        free((void *)Wlp);
/*              }                                       */
                Wlp = wlp;
        }
}

void    free_pat(int Is)
{
        int     i;

        for (i=Is; i<=NS; i++)
                free_lst(Pat[K[i]], K[i]);
}

WORDGRP *same_grp(int k, int num, WORDGRP *Wgp)
{
        int     OK;
        WORDGRP *wgp1, *wgp2;
        WORDLST *wlp;

        if (num == LEN[k])              /* save memory */
                return SET[k];          /* Wgp must be the whole set */
        for (wlp = Pat[k]; wlp != NULL; wlp = wlp->next) {
            if (wlp->num == num) {
                OK = TRUE_ONE;
                wgp2 = Wgp;
                for (wgp1 = wlp->wgp; OK && wgp1 != NULL; wgp1 = wgp1->next) {
                    if (strcmp(wgp1->word, wgp2->word) != 0)
                        OK = FALSE;
                    wgp2 = wgp2->next;
                }
                if (OK)
                    return wlp->wgp;
            }
        }
        return NULL;
}

/*
 *  gets called once from form_pattern() and once from solve().
 */
void reorder(int Is)
{
   int    i, j, k, Sum, II, KK, best_x, Mark2[MAXCRYPT], Len;
   float  best, rate;
   WORDGRP *wgp, *wgp2, *Wgp, *g_end, *PP;
   WORDLST *wlp, *Pat2, *l_end;

   g_end = NULL; /* mollify gcc about g_end being uninitialized */
   l_end = NULL; /* mollify gcc about l_end being uninitialized */
   
   if (Is != 0) {
      if (Is == NS+1) {
/*       fprintf(stderr, "Exiting from reorder()!\n"); */
/*       my_exit();                                    */
         longjmp(CPT_Jump_Buffer,1);
      }
/*    fprintf(stderr, "Re-ordering at level %2d - ", Is);   */
      free_pat(Is);
   }

   for (i=0; i<NC; i++) {                  /* initialize flag */
      if (DUP[i] || LEN[i] == 0 || MARK[i]) Mark[i] = TRUE_ONE;
      else                                  Mark[i] = FALSE;
      Mark2[i] = 0;
   }
   
   best = 0;
   KK = 0;     /* guess, to mollify gcc about KK being uninitialized */
   for (j=0; j<NC; j++) {
      if (!Mark[j]) {
         rate = count(Ch[j])/(float)LEN[j];
         if (rate > best) {
            /* the Is_th word in new order */
            best = rate;
            KK = j;
         }
      }
   }
   
   K[Is] = KK;
   Mark[KK]= TRUE_ONE;
   Pat[KK] = alloc_lst();
   WGP[KK] = Pat[KK]->wgp = SET[KK];  /* first list */
   Pat[KK]->num = LEN[KK];
   Pat[KK]->pos = Pat[KK]->pa_row = Pat[KK]->pa_pos = 0;
   Pat[KK]->next = NULL;
   I[Is] = 0;                         /* avoid null pointer for root */
/* fprintf(stderr, "root word  is  %s ", Ch[K[Is]]);       */

   for (k=Is+1; k<NC_USE; k++) {
      best_x = 0;
      Len = NL;
      for (j=0; j<NC; j++) {              /* get K[] */
         if (!MARK[j] && !Mark[j]) {     /* unused words */
            Sum = 0;
            for (i=Is; i<k; i++) Sum += N[j][K[i]];  /* from the marked words */
            if (k == NC_USE - 1) KK = j;
            if (Sum > best_x || (Sum == best_x && LEN[j] < Len)) {
               best_x = Sum;
               KK = j;
               Len = LEN[j];
            }
         }
      }

      K[k] = KK;          /* index for current crypt word */
      Mark[KK] = TRUE_ONE;            /* mark K[NS] */

      best_x = 0;                         /* get I[] */
      II = Is;                            /* this to molify GCC */
      for (i=Is; i<k; i++) {
         if (!Mark2[K[i]] && N[KK][K[i]] > best_x) {
            II = i;
            best_x = N[KK][K[i]];
         }
      }
      if (best_x == 0) {          /* if no parent matched */
         II = Is;                /* choose root */
         Mark2[KK] = TRUE_ONE;       /* not used as parent */
      }
      I[k] = II;
      II = K[I[k]];
      NUM[KK] = 0;                /* # of words in restricted set */
      Pat[KK] = NULL;
      i = 0;                      /* i - count the position of parent w */
      for (wlp=Pat[II]; wlp != NULL; wlp = wlp->next) {
         for (wgp2=wlp->wgp; wgp2!=NULL; wgp2=wgp2->next) {
            Wgp = NULL;
            j=0;
            for (wgp = SET[KK]; wgp != NULL; wgp = wgp->next) {
               if (match2(Ch[II],wgp2->word,Ch[KK],wgp->word)) {
                  NUM[KK]++;
                  if (Wgp == NULL) {  /* create word subset */
                     Wgp = g_end = alloc_grp();
                     Wgp->row = i;             /* record the row pos */
                     Wgp->pos = j++;
                     strcpy(Wgp->word, wgp->word);
                     Wgp->next = NULL;
                  }
		  else {
                     PP = alloc_grp();
                     PP->row = i;              /* row pos */
                     PP->pos = j++;
                     strcpy(PP->word, wgp->word);
                     PP->next = NULL;
                     g_end->next = PP; /* add to the tail */
                     g_end = PP;
                  }
               }
            }
            if (Wgp != NULL) {
               if (Pat[KK] == NULL) {  /* create pat set */
                  Pat[KK] = l_end = alloc_lst();
                  Pat[KK]->num = j;           /* # of words in the lsit */
                  Pat[KK]->pos = i++;         /* row pos */
                  Pat[KK]->pa_row = wlp->pos; /* parent row pos */
                  Pat[KK]->pa_pos = wgp2->pos;/* parent word pos in that row */
                  Pat[KK]->wgp = Wgp;
                  Pat[KK]->next = NULL;
               }
	       else {
                  Pat2 = alloc_lst();
                  Pat2->num = j;              /* # of words in the list */
                  Pat2->pos = i++;            /* row pos */
                  Pat2->pa_row = wlp->pos;    /* parent row pos */
                  Pat2->pa_pos = wgp2->pos;   /* parent pos in that row */
/*                if ((PP = same_grp(KK, j, Wgp)) != NULL) {
                     free_grp(Wgp);     save memory for identical grp
                     Pat2->wgp = PP;
                  } else                      */
                  Pat2->wgp = Wgp;
                  Pat2->next = NULL;
                  l_end->next = Pat2;
                  l_end = Pat2;
               }
            }
         }
      }
      if (NUM[KK] > THRESHOLD) Mark2[KK] = TRUE_ONE;
   }
   if (Is == 0) {
      NS = k - 1;
      I[k] = 0;                       /* avoid null pointer */
   }
   
}  /* end of reorder() */

/*
 *  Called only from reorder
 */
int     match2(char *w1, char *x, char *w2, char *y)
{
        int     i, j, Len1 = strlen(w1), Len2 = strlen(w2);

        for (i=0; i<Len1; i++)
                for (j=0; j<Len2; j++)
                        if ((w1[i]==w2[j]) != (x[i]==y[j]))
                                return FALSE;
        return TRUE_ONE;
}

/*
 *  'Is' seems to be some kind of recursion level. The calling program
 *       sets it to 0, and each recursion sets it to Is+1.
 */
void    solve(int Is, UCHAR *P, UCHAR *Q, int score, int Top)
{
   int i, OK;
   WORDGRP *wgp;
   WORDLST *wlp;
   UCHAR NewP[91], NewQ[91];

#if (DEBUG > 2)
   printf("do_hart: entering solve()\n");
#endif
   
   if (0 /* quit_Hart_command */) {
      longjmp(CPT_Jump_Buffer,1);
   }
   else {
      /*
       *  If the recursion level is greater than the number of
       *  unique patterns ...
       */
      if (Is > NS || (score + NS - Is + 1) < High) {
         /*
	  * compare score and give possible answers
	  */
         if (score >= High) {
            output_solution(P, score);
            High = score;
         }
      } else {
         for (wgp=WGP[K[Is]]; wgp!=NULL; wgp=wgp->next) {
            for (i=Is+1; i<=NS; i++) {
               if (I[i] == Is) {
                  if (EXTRA[Is]) {
                     EXTRA[i] = TRUE_ONE;
                     WGP[K[i]] = SET[K[i]];
                  }
                  else {
                     OK = FALSE;
                     for (wlp=Pat[K[i]]; !OK && wlp!= NULL; wlp=wlp->next) {
                        if (wlp->pa_row == wgp->row &&  /* match row */
                            wlp->pa_pos == wgp->pos) {  /* match pos */
                               WGP[K[i]] = wlp->wgp;
                               OK = TRUE_ONE;
                        }       /* find the pat subset */
                     }
                     EXTRA[i] = 0;
                  }
               }
	    }
            for (i='A'; i<='Z'; i++) {      /* new permut */
               NewP[i] = P[i];
               NewQ[i] = Q[i];
            }
            if (consistent(Ch[K[Is]], wgp->word, NewP, NewQ)) {
               solve(Is+1, NewP, NewQ, score+1, 0);
	    }
                 /* next word with parent word at level I[] */
         }       /* for its child call */
#if (0)
         if (Top == 1 && Is < LEVEL) {   /* sorting the top 3 level */
            MARK[K[Is]] = TRUE_ONE;     /* level Is has been used up */
            reorder(Is+1);
            solve(Is+1, P, Q, score, 1);
         } else {
#endif		 
         for (i=Is+1; i<=NS; i++) {
            if (I[i] == Is) {
               WGP[K[i]] = SET[K[i]];
               EXTRA[i] = TRUE_ONE;
            }
	 }
         solve(Is+1, P, Q, score, 0);
#if (0)   
         }
#endif   
      }
   }
} /* end of solve() */

/*
 *  See if a new premutation is consistant. 
 */
int     consistent(char *w1, char *w2, UCHAR *NewP, UCHAR *NewQ)
{
        int     i, OK = TRUE_ONE, Len = strlen(w1);

        for (i = 0; OK && i != Len; i++) {
                if (!NewP[(int)w1[i]]) {             /* char w1[i] not filled yet */
                        if (NewQ[(int)w2[i]])        /* char w2[i] not filled yet */
                                OK = FALSE;     /* continue */
                        else {
                                NewP[(int)w1[i]] = w2[i];    /* fill the char */
                                NewQ[(int)w2[i]] = TRUE_ONE;     /* mark w2[i] */
                        }
                } else
                        if (NewP[(int)w1[i]] != w2[i])       /* if mark not correct */
                                OK = FALSE;             /* not consistent yet */
        }
        return OK;
   
} /* end of consistent() */

/*
 *  Write a message to the output pipe. Adds the byte count on to the
 *  beginning and a newline character to the end.
 */
void putmsg(int pipe, char *msg)
{
   int n;
   char buff[128];
   
   n = strlen(msg);
   sprintf(buff,"%02d",n+1);
   memcpy(buff+2,msg,n);
   buff[n+2] = '\n';
   write (pipe,buff,n+3);

} /* end of putmsg() */

/*
 *  Read a message from the input pipe, stripping the byte count and
 *  newline character and adding a terminal NULL. Exits if something
 *  bad happens.
 * 
 *  Returns the actual message length.
 */
int getmsg(int pipe, char *buf, int bufsize)
{
   int c,m,n;
   char length[2];
   
#if (DEBUG > 2)   
   printf("do_hart: entering getmsg()\n");
#endif
   
   m = read (pipe,length,2);
#if (DEBUG > 2)   
   printf("do_hart: completed read of message length in getmsg()\n");
#endif
   if (m != 2) {
#if (DEBUG > 2)   
      printf("do_hart: (m != 2) in getmsg()\n");
#endif
      goto something_bad;
   }
   c = length[0];
   if ((c < '0') || (c > '9')) {
#if (DEBUG > 2)   
      printf("do_hart: 1st digit of incoming message length = 0x%02X\n",c);
#endif
      goto something_bad;
   }
   n = (c - '0')*10;
   c = length[1];
   if ((c < '0') || (c > '9')) {
#if (DEBUG > 2)   
      printf("do_hart: 2nd digit of incoming message length = 0x%02X\n",c);
#endif
      goto something_bad;
   }
   n += (c - '0');
   if ((n < 2) || (n > bufsize)) {
#if (DEBUG > 2)   
      printf("do_hart: bad message size = %d\n",n);
#endif
      goto something_bad;
   }
   m = read (pipe,buf,n);
   if (m != n) {
#if (DEBUG > 2)   
      printf("do_hart: expected %d, got %d characters in getmsg()\n",n,m);
#endif
      goto something_bad;
   }
   n--;    
   if (*(buf+n) != '\n') {
#if (DEBUG > 2)   
      printf("do_hart: missing newline in getmsg()\n");
#endif
      goto something_bad;
   }
   *(buf+n) = 0;
#if (DEBUG > 1)
   printf("do_hart: received %d char msg: '%s'\n",n,buf);
#endif   
   return n;
   
something_bad:
#if (DEBUG)
   printf("do_hart: error detected in getmsg() - exiting.\n");
#endif   
   exit(1);
   
} /* end of getmsg() */

/*
 *  Output a solution message to the output pipe, and wait for a reply.
 */
void output_solution(UCHAR *P, int score)
{
   int i,c;
   char buff[36];
   char *cpo;
   
   sprintf(buff,"!C=%03d,P=",score);
   buff[35]=0;
   cpo = buff+9;
   
   P += 'A';
   for (i=0;i<26;i++) {
      c = *P++;
      if ((c < 'A') || (c > 'Z')) c = ' ';
      *cpo++ = c;
   }
   putmsg(out_pipe, buff);
#if (DEBUG)   
   printf("do_hart: sent solution '%s'\n",buff);
#endif
   i = getmsg(in_pipe,buff,36);
#if (DEBUG)   
   printf("do_hart: got reply: '%s'\n",buff);
#endif
   if (strcmp(buff,"!OK") != 0) exit(1);
   return;   
   
} /* end of output_solution() */
