/***************************************************************************
    copyright            : (C) 2003-2006 by Robby Stephenson
    email                : robby@periapsis.org
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of version 2 of the GNU General Public License as  *
 *   published by the Free Software Foundation;                            *
 *                                                                         *
 ***************************************************************************/


#include <config.h>

#include "searchmanager.h"

#include "z3950searcher.h"
#include "srusearcher.h"
#include "entrezsearcher.h"
#include "ieeexploresearcher.h"
#include "woksearcher.h"

#include <kglobal.h>
#include <kconfig.h>
#include <klocale.h>
#include <iostream>

#include <config.h>

using namespace std;

searchManager* searchManager::s_self = 0;

searchManager::searchManager() : QObject()
{
    searchersList.setAutoDelete(true);
    loadSearchers();
    m_loadDefaults = false;

    m_keyMap.insert(Title,      i18n("Title"));
    m_keyMap.insert(Author,     i18n("Author"));
    m_keyMap.insert(Keyword,    i18n("Keywords"));
    m_keyMap.insert(Subject,    i18n("Subject"));
    m_keyMap.insert(All,        i18n("All fields"));
    m_keyMap.insert(Journal,        i18n("Journal"));
    m_keyMap.insert(Year,    i18n("Year"));
	m_keyMap.insert(Topic,    i18n("Topic"));
}

searchManager::~searchManager()
{}

void searchManager::loadSearchers()
{
    //  myDebug() << "Manager::loadFetchers()" << endl;
    searchersList.clear();

    KConfig* config = KGlobal::config();
    if(config->hasGroup(QString::fromLatin1("Data Sources")))
    {
        KConfigGroupSaver saver(config, QString::fromLatin1("Data Sources"));
        int nSources = config->readNumEntry("Sources Count", 0);
        for(int i = 0; i < nSources; ++i)
        {
            QString group = QString::fromLatin1("Data Source %1").arg(i);
            searcher* f = createSearcher(config, group);
            if(f)
            {
                searchersList.append(f);
            }
        }
        m_loadDefaults = false;
    }
    else
    { // add default sources
        //	  m_searchers = defaultSearchers();
        m_loadDefaults = true;
    }
}

QStringList searchManager::sources()
{
    QStringList sources;
    //searcher* it = m_searchers.first();

    for ( uint i = 0; i < searchersList.count(); ++i )
    {
        if ( searchersList.at(i) )
            sources << searchersList.at(i)->source();

    }
    return sources;
}



searcher* searchManager::createSearcher(KConfig* config_, const QString& group_)
{
    if(!config_->hasGroup(group_))
    {
        cerr << "Manager::createFetcher() - no config group for " << group_.ascii() << endl;
        return 0;
    }

    KConfigGroupSaver groupSaver(config_, group_);

    int type = config_->readNumEntry("Type", Unknown);
    if(type == Unknown)
    {
        cerr << "Manager::createFetcher() - unknown type " << type << ", skipping" << endl;
        return 0;
    }

    searcher* f = 0;
    switch(type)
    {
    case Z3950:
#if HAVE_YAZ
        f = new Z3950Searcher(this);
#endif
        break;
    case SRU:
        f = new SRUSearcher(this);
        break;
    case Entrez:
        f = new EntrezSearcher(this);
        break;
    case IEEEXplore:
        f = new IEEEXploreSearcher(this);
        break;
	 case WOK:
        f = new WOKSearcher(this);
        break;
    case Unknown:
    default:
        break;
    }
    if(f)
    {
        f->readConfig(config_, group_);
    }
    return f;
}


SearchKey searchManager::searchKey(const QString& key_) const
{
    for(SearchKeyMap::ConstIterator it = m_keyMap.begin(); it != m_keyMap.end(); ++it)
    {
        if(it.data() == key_)
        {
            return it.key();
        }
    }
}

QString searchManager::searchKeyString(const SearchKey& key_) const
{
    for(SearchKeyMap::ConstIterator it = m_keyMap.begin(); it != m_keyMap.end(); ++it)
    {
        if(it.key() == key_)
        {
            return it.data();
        }
    }
}

int searchManager::maxSearchTerms(const QString& source_)
{
	searcher *s;
	for ( s = searchersList.first(); s; s = searchersList.next() )
	{
		if(source_ == s->source())
		{
			return 	s->maxSearchTerms();
		}
	}
}


QStringList searchManager::searchKeyList(const QString& source_)
{
    searcher *s;
    for ( s = searchersList.first(); s; s = searchersList.next() )
    {
        if(source_ == s->source())
        {
            return 	s->searchKey();
        }
    }
}

void searchManager::startSearch(const QString& source, SearchKey key1, SearchKey key2, SearchKey key3 , const QString& value1, const QString& value2, const QString& value3, int operator1, int operator2)
{
    cerr << "Starting searching " << source.ascii() << "\n";

    searcher *s;
    for ( s = searchersList.first(); s; s = searchersList.next() )
    {
        if(source == s->source())
        {
            connect(s, SIGNAL(signalResultFound( BibEntry* ) ), SIGNAL(signalResultFound( BibEntry * ) ) );
            connect(s, SIGNAL(signalQueryResult(unsigned int )), SIGNAL(signalQueryResult( unsigned int ) ) );
            connect(s, SIGNAL(signalMessage( QString, int) ), SIGNAL(signalMessage( QString, int) ) );
            connect(s, SIGNAL(signalDone( searcher* ) ), SLOT(slotSearcherDone( searcher *) ) );
            s->search(key1, key2, key3, value1, value2, value3, operator1, operator2);
            break;
        }
    }
}

void searchManager::stop()
{
    for ( searcher *s = searchersList.first(); s; s = searchersList.next() )
    {
        if(s->isSearching())
        {
            s->stop();
        }
    }
}

void searchManager::retrieveRange(unsigned int start, unsigned int end)
{
    searcher *s;
    for ( s = searchersList.first(); s; s = searchersList.next() )
    {
        if (s->isWaitingRetrieveRange())
            s->retrieveRange(start, end);
    }

}


QString searchManager::typeName(Type type_)
{
    switch(type_)
    {
#if HAVE_YAZ
    case Z3950: return Z3950Searcher::defaultName();
#endif
    case Entrez: return EntrezSearcher::defaultName();
    case SRU: return SRUSearcher::defaultName();
	case IEEEXplore: return IEEEXploreSearcher::defaultName();
	case WOK: return WOKSearcher::defaultName();
    default: break;
    }
    return QString::null;
}

void searchManager::slotSearcherDone(searcher *searcher_)
{
    searcher_->disconnect(); // disconnect all signals
    emit signalDone();
}

SearchMap searchManager::sourceMap()
{
    SearchMap map;
#if HAVE_YAZ
    map.insert(Z3950, Z3950Searcher::defaultName());
#endif
    map.insert(SRU, SRUSearcher::defaultName());
    map.insert(Entrez, EntrezSearcher::defaultName());
    map.insert(IEEEXplore, IEEEXploreSearcher::defaultName());
	map.insert(WOK, WOKSearcher::defaultName());
    return map;
}


//#include "../searchmanager.moc"
