//
// C++ Implementation: kbibremotesearchdlg
//
// Description: 
//
//
// Author: Thach Nguyen <thach.nguyen@rmit.edu.au>, (C) 2006
//
// Copyright: See COPYING file that comes with this distribution
//
//
#include "kbib.h"
#include "kbibdoc.h"
#include "kbibremotesearchdlg.h"
#include "kbiblistviewitem.h"
#include "queryresultdlg.h"

#include <klocale.h>
#include <kiconloader.h>
#include <kmessagebox.h>
#include <kaccelmanager.h>
#include <qlayout.h>
#include <qhbox.h>
#include <qvgroupbox.h>
#include <qsplitter.h>
#include <qwhatsthis.h>



KBibRemoteSearchDlg::KBibRemoteSearchDlg(QWidget *parent, const char *name)
	:KDialogBase( parent, name , true, i18n("Remote Database Search"), KDialogBase::Close, KDialogBase::Close, false )
{

	m_started = false;
//	m_manager = new searchManager();
	QWidget* mainWidget = new QWidget(this, "Remote Search");
	setMainWidget(mainWidget);
	QVBoxLayout* topLayout = new QVBoxLayout(mainWidget, 0, KDialog::spacingHint());

	QVGroupBox* queryBox = new QVGroupBox(i18n("Search Query"), mainWidget, "Query Box");
	topLayout->addWidget(queryBox);

	QHBox* box0 = new QHBox(queryBox, "box0");
	box0->setSpacing(KDialog::spacingHint());
	
	QLabel* label = new QLabel(i18n("Search s&ource:"), box0);
	dataSource = new KComboBox(box0);
	dataSource->insertStringList(searchManager::self()->sources());
	label->setBuddy(dataSource);
	QWhatsThis::add(dataSource, i18n("Select the database to search"));
	
	QHBox* box1 = new QHBox(queryBox, "box1");
	box1->setSpacing(KDialog::spacingHint());

	searchTerms1 = new KLineEdit(box1);
	QWhatsThis::add(searchTerms1, i18n("Enter a search value."));
	label = new QLabel(i18n("within:"), box1);
	
	searchKey1 = new KComboBox(box1);
	QWhatsThis::add(searchKey1, i18n("Choose the type of search"));
	
	QHBox* box3 = new QHBox(queryBox, "box3");
	box3->setSpacing(KDialog::spacingHint());

	operator1 = new KComboBox(box3);
	operator1->insertItem("AND");
	operator1->insertItem("OR");
	operator1->insertItem("AND NOT");
	
	searchTerms2 = new KLineEdit(box3);
	QWhatsThis::add(searchTerms2, i18n("Enter a search value."));
	label = new QLabel(i18n("within:"), box3);
	
	searchKey2 = new KComboBox(box3);
	QWhatsThis::add(searchKey2, i18n("Choose the type of search"));
	
	QHBox* box5 = new QHBox(queryBox, "box5");
	box5->setSpacing(KDialog::spacingHint());

	operator2 = new KComboBox(box5);
	operator2->insertItem("AND");
	operator2->insertItem("OR");
	operator2->insertItem("AND NOT");
	searchTerms3 = new KLineEdit(box5);
	QWhatsThis::add(searchTerms3, i18n("Enter a search value."));
	label = new QLabel(i18n("within:"), box5);
	
	searchKey3 = new KComboBox(box5);

	QWhatsThis::add(searchKey2, i18n("Choose the type of search"));

	
	QHBox* box6 = new QHBox(queryBox, "box6");
	box6->setStretchFactor(new QWidget(box6), 1);
	searchButton = new KPushButton(i18n("&Search"), box6);
	searchButton->setGuiItem(KGuiItem(i18n("&Search"),
							 SmallIconSet(QString::fromLatin1("find"))));
	connect(searchButton, SIGNAL(clicked()), SLOT(slotSearchClick()));
	QWhatsThis::add(searchButton, i18n("Click to start or stop the search"));
	searchButton->polish();

	QSplitter* split = new QSplitter(QSplitter::Vertical, mainWidget);
	topLayout->addWidget(split);

	listView = new KListView(split);
	listView->setShowSortIndicator(true);
	listView->setSorting(10);
	listView->setAllColumnsShowFocus(true);
	listView->setSelectionMode(QListView::Extended);

	listView->addColumn(i18n("Author"));
	listView->setColumnWidthMode(0, QListView::Manual);
	listView->addColumn(i18n("Year"));
	listView->addColumn(i18n("Title"));
	listView->setColumnWidth (0, 200);
	listView->setColumnWidth (1, 50);
	
	connect(listView, SIGNAL(clicked(QListViewItem*)), SLOT(slotEntryClick(QListViewItem*)));
	connect(listView, SIGNAL(selectionChanged ()), SLOT(slotSelectionChanged()));

	entryPreview = new QTextEdit(split);
	entryPreview->setReadOnly(TRUE);
	
	QVGroupBox* optionBox = new QVGroupBox(i18n("When adding references"), mainWidget, "Option Box");
	topLayout->addWidget(optionBox);
	ignoreExistingSelect = new QCheckBox(i18n("Do not add existing references"), optionBox);
	createKeySelect = new QCheckBox(i18n("Fix duplicated citation keys"), optionBox); 
	ignoreExistingSelect->setChecked(true);
	createKeySelect->setChecked(true);
	
	QHBox* box7 = new QHBox(mainWidget);
	topLayout->addWidget(box7);
	box7->setSpacing(KDialog::spacingHint());

	addButton = new KPushButton(i18n("&Add References"), box7);
	addButton->setGuiItem(KGuiItem(i18n("&Add References"),
							 SmallIconSet(QString::fromLatin1("add"))));
	addButton->setEnabled(false);
	
	
	this->setMinimumWidth(600);
	
	searchTerms1->setFocus();
	
	connect(dataSource, SIGNAL(activated(const QString&)), SLOT(slotSourceSelected(const QString&)));
	
	connect(addButton, SIGNAL(clicked()), SLOT(slotAddEntryClick()));
	
	clearButton = new KPushButton(KStdGuiItem::clear(), box7);
	
	connect(clearButton, SIGNAL(clicked()), SLOT(slotClearClick()));
	
	connect(searchManager::self(), SIGNAL(signalResultFound(BibEntry*)),
			SLOT(slotResultFound(BibEntry*)));
	
	connect(searchManager::self(), SIGNAL(signalQueryResult(unsigned int)),
			SLOT(slotQueryResult(unsigned int)));
	connect(searchManager::self(), SIGNAL(signalMessage(QString, int ) ), SLOT(slotMessage(QString, int ) ) );
	connect(searchManager::self(), SIGNAL(signalDone() ), SLOT(slotSearchDone() ) );
	
	slotSourceSelected(dataSource->currentText());
	KAcceleratorManager::manage(mainWidget);
}


KBibRemoteSearchDlg::~KBibRemoteSearchDlg()
{
	//delete Bib entry
	for(QListViewItem* item_ = listView->firstChild(); item_; item_ = item_->nextSibling())
	{
		KBibListViewItem *item = dynamic_cast<KBibListViewItem*>(item_);
		BibEntry *e =  item->getBibEntry();
		delete e;
	}
	//clear list view
	listView->clear();
}

void KBibRemoteSearchDlg::slotSourceSelected(const QString& source){
	searchKey1->clear();
	searchKey2->clear();
	searchKey3->clear();
	searchKey1->insertStringList(searchManager::self()->searchKeyList( source));
	searchKey2->insertStringList(searchManager::self()->searchKeyList( source));
	searchKey3->insertStringList(searchManager::self()->searchKeyList( source));
	
	int maxSearchTerms = searchManager::self()->maxSearchTerms(source);
	
	if (maxSearchTerms < 3){
		searchKey3->setEnabled(false);
		searchTerms3->setEnabled(false);
		operator2->setEnabled(false);
	}
	else{
		searchKey3->setEnabled(true);
		searchTerms3->setEnabled(true);
		operator2->setEnabled(true);
	}
		
	if (maxSearchTerms < 2){
		searchKey2->setEnabled(false);
		searchTerms2->setEnabled(false);
		operator1->setEnabled(false);
	}
	else{
		searchKey2->setEnabled(true);
		searchTerms2->setEnabled(true);
		operator1->setEnabled(true);
	}
	
}

void KBibRemoteSearchDlg::stop()
{
	if(m_started) {
		searchManager::self()->stop();
		slotSearchDone();
	}	
}

void KBibRemoteSearchDlg::slotSearchClick()
{
	if(m_started) {
		searchManager::self()->stop();
		slotSearchDone();
	} else {
		QString value1 = searchTerms1->text().simplifyWhiteSpace();
		QString value2 = searchTerms2->text().simplifyWhiteSpace();
		QString value3 = searchTerms3->text().simplifyWhiteSpace();
		if (!value1.isEmpty() || !value2.isEmpty() || !value3.isEmpty()){
			m_started = true;
			searchButton->setGuiItem(KGuiItem(i18n("&STOP"),
								   SmallIconSet(QString::fromLatin1("cancel"))));
			QApplication::setOverrideCursor( Qt::waitCursor );
			searchManager::self()->startSearch(dataSource->currentText(), searchManager::self()->searchKey(searchKey1->currentText()), searchManager::self()->searchKey(searchKey2->currentText()), searchManager::self()->searchKey(searchKey3->currentText()), value1, value2, value3, operator1->currentItem(), operator2->currentItem() );
		}
	}
}


void KBibRemoteSearchDlg::slotResultFound(BibEntry* bib){
	
	if (bib)
	{
		KBibApp *win = (KBibApp *) (this->parent() );
		KBibDoc *doc = win->getDocument();	
		QStringList displayFields;
		displayFields << "author" << "year" << "title";
		KBibListViewItem * item = new KBibListViewItem(listView, doc, new BibEntry(*bib), displayFields, 0 );
		delete bib;
	}	
}

void KBibRemoteSearchDlg::slotQueryResult(unsigned int num){
	if (num > 0){
		QApplication::restoreOverrideCursor();
		queryResultDlg *queryResult = new queryResultDlg(num);
		if (queryResult->exec() == QDialog::Accepted){
			int start, end;
			start = queryResult->getStart();
			end = queryResult->getStop();
			if (end < start)
				end = start;
			searchManager::self()->retrieveRange(start, end);
			QApplication::setOverrideCursor( Qt::waitCursor );
		}
		else
			searchManager::self()->retrieveRange(0, 0);
		delete queryResult;
	}
}

void KBibRemoteSearchDlg::slotAddEntryClick()
{
	KBibApp *win = (KBibApp *) (this->parent() );
	KBibDoc *doc = win->getDocument();	

	for(QListViewItem* item_ = listView->firstChild(); item_; item_ = item_->nextSibling())
	{
		KBibListViewItem *item = dynamic_cast<KBibListViewItem*>(item_);
		if (listView->isSelected(item))
		{
			BibEntry *e =  item->getBibEntry();
			doc->addEntry(new BibEntry(*e), ignoreExistingSelect->isChecked(), createKeySelect->isChecked());
		}
	}
}

void KBibRemoteSearchDlg::slotClearClick(){
	//Remove bib pointers first
	
	for(QListViewItem* item_ = listView->firstChild(); item_; item_ = item_->nextSibling())
	{
		KBibListViewItem *item = dynamic_cast<KBibListViewItem*>(item_);
		BibEntry *e =  item->getBibEntry();
		delete e;
	}
	listView->clear();
	entryPreview->clear();
	searchTerms1->setFocus();

}


void KBibRemoteSearchDlg::slotEntryClick(QListViewItem* item_) {
	if(!item_) {
		addButton->setEnabled(false);
		entryPreview->clear();
		return;
	}


	addButton->setEnabled(true);
	KBibListViewItem *item = dynamic_cast<KBibListViewItem*>(item_);

	BibEntry *entry = item->getBibEntry();
	
	KBibApp *win = (KBibApp *) (this->parent() );
	KBibDoc *doc = win->getDocument();	
	
	entryPreview->setText(doc->getEntryList()->detailedEntry(entry) );
	
	
}

void KBibRemoteSearchDlg::slotSelectionChanged()
{

	QListViewItem *i =  listView->currentItem();
    if ( !i )
        return;

    if ( !i->isSelected() ) {
        return;
    }
	slotEntryClick(i);


}



void KBibRemoteSearchDlg::slotMessage(QString message, int type){
	QApplication::restoreOverrideCursor();
	if (message.isEmpty() )
		return;
	if (type == 0){	//error
		KMessageBox::sorry(this, message);
	}
	else if (type == 1){	//warning
		KMessageBox::information(this, message);	
	}
	
}	

void KBibRemoteSearchDlg::slotSearchDone(){
	QApplication::restoreOverrideCursor();
	m_started = false;
	searchButton->setGuiItem(KGuiItem(i18n("&Search"),
							   SmallIconSet(QString::fromLatin1("find"))));
}

#include "kbibremotesearchdlg.moc"
