/*
 * Copyright (c) 1983, 1993
 *	The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/param.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/utsname.h>
#include <sys/wait.h>
#include <errno.h>

#include <unistd.h>
#include <syslog.h>
#include <netdb.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <stdarg.h>
#include <pwd.h>
#include <grp.h>
#include "pathnames.h"

static void
usage (void)
{
  syslog (LOG_ERR, "usage: fingerd [-slumMpSw] [-P filename] [-t timeout]");
  exit (2);
}

static void
fatal (const char *msg, int use_errno, int tolog, int toclient)
{
  const char *err = "";
  const char *sep = "";
  if (use_errno)
    {
      err = strerror (errno);
      sep = ": ";
    }
  if (tolog)
    syslog (LOG_ERR, "%s%s%s\n", msg, sep, err);
  if (toclient)
    fprintf (stderr, "fingerd: %s%s%s\r\n", msg, sep, err);
  else
    fprintf (stderr, "fingerd: Internal error\r\n");
  exit (1);
}

static void
err (const char *fmt, ...)
{
  va_list ap;
  va_start (ap, fmt);

  vsyslog (LOG_ERR, fmt, ap);
  va_end (ap);
  exit (1);
  /* NOTREACHED */
}

static void
timeout (int sig)
{
  (void) sig;
  errno = ETIMEDOUT;
  fatal ("Input timeout", 0, 1, 1);
}


int
main (int argc, char *argv[])
{
#if 0
  FILE *fp;
  int p[2];
#endif
  int ch, ac = 2;
  int logging, secure, user_required, short_list;
  int welcome, patience = 60;
#define	ENTRIES	50
  char **ap, *av[ENTRIES + 1], **comp, line[8192], *prog, *lp, *hname = "";
  char hostbuf[MAXHOSTNAMELEN];
  struct sockaddr_storage ss;
  int sval;

  openlog ("fingerd", LOG_PID, LOG_DAEMON);

  /* Make sure we run not as root */
  if (!getuid () || !geteuid ())
    {
      struct passwd *pwd = getpwnam ("nobody");
      if (pwd)
	{
	  initgroups (pwd->pw_name, pwd->pw_gid);
	  setgid (pwd->pw_gid);
	  setuid (pwd->pw_uid);
	}
      seteuid (0);		/* this should fail */
      if (!getuid () || !geteuid ())
	fatal ("setuid: couldn't drop root", 0, 1, 0);
    }

  prog = _PATH_FINGER;
  welcome = logging = secure = user_required = short_list = 0;
  opterr = 0;
  while ((ch = getopt (argc, argv, "sluSmMpP:wt:")) != -1)
    switch (ch)
      {
      case 'l':
	logging = 1;
	break;
      case 'P':
	prog = optarg;
	break;
      case 's':
	secure = 1;
	break;
      case 'u':
	user_required = 1;
	break;
      case 'S':
	short_list = 1;
	av[ac++] = "-s";
	break;
      case 'm':
	av[ac++] = "-m";
	break;
      case 'M':
	av[ac++] = "-M";
	break;
      case 'p':
	av[ac++] = "-p";
	break;
      case 'w':
	welcome = 1;
	break;
      case 't':
	patience = atoi (optarg);
	break;
      case 'h':
      case '?':
      default:
	usage ();
      }

  /*
   * Hang up after a while so people can't DoS by leaving lots of
   * open sockets about.
   */
  if (patience != 0)
    {
      signal (SIGALRM, timeout);
      alarm (patience);
    }


  if (logging || welcome)
    {
      sval = sizeof (ss);
      if (getpeername (0, (struct sockaddr *) &ss, &sval) < 0)
	err ("getpeername: %s", strerror (errno));
    }

  if (logging)
    {
      if (getnameinfo ((struct sockaddr *) &ss, sval, hostbuf,
		       sizeof (hostbuf), NULL, 0, 0) != 0)
	strncpy (hostbuf, "?", sizeof (hostbuf));
      hname = hostbuf;
    }

  if (fgets (line, sizeof (line), stdin) == NULL)
    {
      if (logging)
	syslog(LOG_NOTICE, "query from %s: %s", hname,
	       feof(stdin) ? "EOF" : strerror(errno));
      exit(1);
    }

  line[sizeof (line) - 1] = '\0';

  if (logging)
    syslog(LOG_NOTICE, "query from %s: `%.*s'", hname,
	   (int)strcspn(line, "\r\n"), line);

  if (welcome)
    {
      char buf[256];
      struct hostent *hp;
      struct utsname utsname;

      uname (&utsname);
      gethostname (buf, sizeof (buf));
      /* If the request comes from a IPv6 address, give back this
	 name, else give back the IPv4 address. */
      if (ss.ss_family == AF_INET6)
	hp = gethostbyname2 (buf, AF_INET6);
      else
	hp = gethostbyname2 (buf, AF_INET);
      if (hp != NULL)
	{
	  /* paranoia: dns spoofing? */
	  strncpy (buf, hp->h_name, sizeof (buf));
	  buf[sizeof (buf) - 1] = 0;
	}

      printf ("\r\nWelcome to %s version %s at %s !\r\n\n",
	      utsname.sysname, utsname.release, buf);
      fflush (stdout);
      switch (fork ())
	{
	case -1:		/* fork failed, oh well */
	  break;
	case 0:		/* child */
	  execl (_PATH_UPTIME, _PATH_UPTIME, NULL);
	  _exit (1);
	default:		/* parent */
	  wait (NULL);
	  break;
	}
      fflush (stdout);
      printf ("\r\n");
      fflush (stdout);
    }

  /*
   * Note: we assume that finger(1) will treat "--" as end of
   * command args (ie: that it uses getopt(3)).
   */
  av[ac++] = "--";
  comp = &av[1];
  for (lp = line, ap = &av[ac]; ac < ENTRIES;)
    {
      if ((*ap = strtok (lp, " \t\r\n")) == NULL)
	break;
      lp = NULL;
      if (secure && strchr (*ap, '@'))
	{
	  (void) puts ("forwarding service denied\r");
	  exit (1);
	}

      ch = strlen (*ap);
      while ((*ap)[ch - 1] == '@')
	(*ap)[--ch] = '\0';
      if (**ap == '\0')
	continue;

      /* RFC1196: "/[Ww]" == "-l" */
      if ((*ap)[0] == '/' && ((*ap)[1] == 'W' || (*ap)[1] == 'w'))
	{
	  if (!short_list)
	    {
	      av[1] = "-l";
	      comp = &av[0];
	    }
	}
      else
	{
	  ap++;
	  ac++;
	}
    }
  av[ENTRIES - 1] = NULL;

  if ((lp = strrchr (prog, '/')))
    *comp = ++lp;
  else
    *comp = prog;

  if (user_required)
    {
      for (ap = comp + 1; strcmp ("--", *(ap++)););
      if (*ap == NULL)
	{
	  (void) puts ("must provide username\r");
	  exit (1);
	}
    }

/* Yay! we don't need to do this any more - finger does it for us */
#if 0
  if (pipe (p) < 0)
    err ("pipe: %s", strerror (errno));

  switch (fork ())
    {
    case 0: /* Child */
      close (p[0]);
      if (p[1] != 1)
	{
	  dup2 (p[1], 1);
	  close (p[1]);
	}
#endif
      execv (prog, comp);
      err ("execv: %s: %s", prog, strerror (errno));
      exit (1);
#if 0
    case -1:
      err ("fork: %s", strerror (errno));
    }
  close (p[1]); /* Parent */

  /* convert \n to \r\n. This should be an option to finger... */
  if (!(fp = fdopen (p[0], "r")))
    err ("fdopen: %s", strerror (errno));
  while ((ch = getc (fp)) != EOF)
    {
      if (ch == '\n')
	putchar ('\r');
      putchar (ch);
    }
  exit (0);
#endif
}
