#! /bin/perl -w 
#
# Administration Tool for Sun's NIS+ Nameservice
#
# Nistool is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License as
# published by the Free Software Foundation; either version 2 of the
# License, or (at your option) any later version.
#
# Nistool is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# General Public License for more details.
# 
# You should have received a copy of the GNU General Public
# License along with the NIS+ Server; see the file COPYING.  If
# not, write to the Free Software Foundation, Inc., 675 Mass Ave,
# Cambridge, MA 02139, USA.
#
# Version: 0.1alpha
# Author: Ralf Lehmann <ralfl@darwin.muc.de>
#


use Tk;
use Tk::DialogBox;
use Tk::Dialog;
use Nisplus;

## Add path to nisplus binaries
$0 =~ /(.*)\/.+/;
my $p = $1;
$ENV{'PATH'} .= ":/usr/lib/nis:$p";


## Variable showing the current operation
my $OPERATION;
## Error Strings
my @ERROR_STRINGS = ('Error', 'error', 'cannot', 'can\'t', 'principal not found',
	'group not found', 'nable', 'usage', 'Principal .* not found',
	'must be fully qualified', 'Can\'t', 'is master for', 'already serves',
	'denied', 'Couldn\'t', 'insufficent');

## The width of Entry lines
my $W=30;
## Number of Rights in one Frame 
my $NR=4;


my $top = MainWindow->new();

##
## The Menu
##
$menu = $top->Frame(-relief => 'raised', -bd => 2);
$menu->pack(-side => 'top', -fill => 'x');
my $menu_tool = $menu->Menubutton(-text => 'Tool')->pack(-side => 'left');
my $menu_obj = $menu->Menubutton(-text => 'Object');
my $menu_help = $menu->Menubutton(-text => 'Help');
#my $menu_options = $menu->Menubutton(-text => 'Options')->pack(-side => 'left');

## Tool Menu
$menu_tool->command(-label => 'Exit',
			-command => sub {exit 0});
##
## State Line
##
my $STATE = 'Just started';
my $BUSY = 'busy';
my $state_f = $top->Frame();
$state_f->Label(-text => 'State: ', -width=>'8', -anchor=>'e')->pack(-side => 'left');
$state_f->Label(-textvariable => \$STATE, -fg=>'red')->pack(-side => 'left');
$state_f->pack(-side => 'top', -expand => 'no', -fill => 'x');

##
## The Directory location
##
my $LOCATION = `domainname`;
chop ($LOCATION);
$LOCATION = Nisplus::absname($LOCATION); 
my $ROOTDIR = $LOCATION;
while (Nisplus::is_rootdomain($ROOTDIR)){ $ROOTDIR = Nisplus::strip_first($ROOTDIR);}
my $location = $top->Frame()->pack(-side => 'top', -fill => 'x');
$location->Label(-text => 'Location: ', -anchor=>'e')->pack(-side => 'left');
$location->Label(-textvariable => \$LOCATION)->pack(-side => 'left');

##
## The Viewer Window
##
my $view_frame = $top->Frame()
	-> pack(-side => 'left', -expand => 'yes', -fill => 'both');
my $VIEW = $view_frame->ScrlListbox()
	->pack(-side => 'left', -expand => 'yes', -fill => 'both');
$VIEW->bind("<Double-Button-1>" => sub{open_obj()});
$VIEW->bind("<Button-3>" => sub{properties($VIEW->curselection)});


## Object Menu
$menu_obj->command(-label => 'Properties...',
	-command => sub{properties($VIEW->curselection)});
my $menu_obj_domain = $menu_obj->cget(-menu)->Menu;
$menu_obj->cascade(-label => 'Domain', -menu => $menu_obj_domain);
$menu_obj->entryconfigure('Domain', -menu => $menu_obj_domain);
$menu_obj_domain->command(-label => 'Dump to Files...',
	-command => sub{dump_domain($VIEW->curselection)});
$menu_obj_domain->command(-label => 'Load from Files...',
	-command => sub{load_domain($VIEW->curselection)});
$menu_obj_domain->separator();
$menu_obj_domain->command(-label => 'New...', -command => sub {new_domain()});
$menu_obj_domain->command(-label => 'Destroy',
	-command => sub{destroy_dom($VIEW->curselection)});
my $menu_obj_dir=$menu_obj->cget(-menu)->Menu;
$menu_obj->cascade(-label=>'Directory', -menu=>$menu_obj_dir);
$menu_obj->entryconfigure('Directory', -menu=>$menu_obj_dir);
$menu_obj_dir->command(-label=>'Update', -command=>sub{nis_ping()});
$menu_obj_dir->command(-label=>'New...', -command=>sub{new_directory()});
$menu_obj_dir->command(-label=>'Destroy',
	-command => sub{destroy_dir($VIEW->curselection)});
my $menu_obj_tbl=$menu_obj->cget(-menu)->Menu;
$menu_obj->cascade(-label=>'Table', -menu=>$menu_obj_tbl);
$menu_obj->entryconfigure('Table', -menu=>$menu_obj_tbl);
$menu_obj_tbl->command(-label=>'Dump to File...',
	-command => sub{$n=$VIEW->curselection; dump_table($VIEW->get($n))});
$menu_obj_tbl->command(-label=>'Load from File...',
	-command => sub{my $n=$VIEW->curselection; 
		load_table($VIEW->get($n))});
$menu_obj_tbl->separator();
$menu_obj_tbl->command(-label=>'New...',
	-command => sub{new_table()});
$menu_obj_tbl->command(-label=>'Destroy',
	-command => sub{destroy_table($VIEW->curselection)});
my $menu_obj_grp=$menu_obj->cget(-menu)->Menu;
$menu_obj->cascade(-label=>'Group', -menu=>$menu_obj_grp);
$menu_obj->entryconfigure('Group', -menu=>$menu_obj_grp);
$menu_obj_grp->command(-label=>'Add...',
	-command => sub{add_group()});
$menu_obj_grp->command(-label=>'Destroy',
	-command => sub{my $n=$VIEW->curselection; destroy_group($VIEW->get($n))});

$menu_obj->pack(-side => 'left');

## Help Menu
$menu_help->command(-label => 'Tool', 
			-command => sub {show_help('Tool')});
$menu_help->pack(-side => 'right');


##
## Buttons on the right
##
my $r_f_but = $top->Frame();
my $r_but_open = $r_f_but->Button(-text => 'Open', -width => 7,
	-command => sub{open_obj()});
$r_but_open->pack();
my $r_but_exit = $r_f_but->Button(-text => 'Exit', -width => 7,
	-command => sub{ exit 0;})->pack();
$r_f_but->pack(-side => "right", -fill => "y");


##
## The Mainloop
##
init_main();
MainLoop;


##
## The main functions
##
sub show_help($)
{

	if (! defined($_[0])) {
		my $Close_Warn = NISDialog->new($top, 'Close');
		my $ret = $Close_Warn->out('Help', 'No Arguments given to Helpfunction!');
		return;
	}
	my $helpfile = $_[0] . ".hlp";
	if (! -f $helpfile ) {
		my $Close_Warn = NISDialog->new($top, 'Close');
		my $ret = $Close_Warn->out('Help', "Can't find Helpfile: $helpfile");
		return;
	}
	my $help_win = $top->Toplevel(-title => "Help: $_[0]");
	my $help_frame = $help_win->Frame();
	my $help_text = $help_frame->Text(-wrap => 'none');
	my $help_xs = $help_frame->Scrollbar(-orient => 'horizontal',
			-command => [xview => $help_text]);
	my $help_ys = $help_frame->Scrollbar(-command => [yview => $help_text]);
	$help_text->configure(-yscrollcommand => [set => $help_ys]);
	$help_text->configure(-xscrollcommand => [set => $help_xs]);
	$help_ys->pack(-side => 'left' , -fill => 'y');
	$help_xs->pack(-side => 'bottom' , -fill => 'x');
	$help_text->pack(-expand => 'yes', -fill => 'both', -side => 'left');
	$help_frame->pack(-expand => 'yes', -fill => 'both');

	if (! open( F, $helpfile)) {
		my $Close_Warn = NISDialog->new($top, 'Close');
		my $ret = $Close_Warn->out('Help', "Can't open Helpfile: $helpfile");
		return;
	}
	while (<F>) {
		$help_text->insert("end" , $_);
	}
	close F;

	$help_text->configure(-state => 'disabled');
	my $help_fclose = $help_win->Frame();
	my $help_bclose = $help_fclose->Button(-text => 'Close', 
		-command => sub {$help_win->destroy()});
	$help_bclose->pack();
	$help_fclose->pack(-expand => 'no', -side => 'bottom');
}


##
## inititalise first appearance of the tool
##
sub init_main{
	my $pid;

	if ($LOCATION eq 'noname') { # no domainname yet
		disable_new_props();
		return;
	}
	$VIEW->insert("end", $LOCATION);
	$SIG{CHLD} = sub{ $pid = wait; sig_handler($pid); };
}


##
## make a real objectname of a selection in the selection box
##
sub norm_sel {
	my $sel = shift;

	if ($sel eq '.') { $sel = $LOCATION; return $sel;}
	if ($sel eq '..') { $LOCATION =~ /(.*?)\.(.+)/; $sel = $2; return $sel;}
	# strip trailing D, T
	if ($sel =~ /[DTGL]\s+(\w+)/) { $sel = $1};
	if (!($sel =~ /.*\./)) { $sel .= "." . $LOCATION;}
	return ($sel);
}


##
## opens an object. A directory is listed in main window, a table gets a new one
## 
sub open_obj {
	if ($STATE eq $BUSY) { show_ok("Still busy!\nPlease wait."); return;}
	my $sel = $VIEW->curselection;
	if (! defined ($sel)) { $sel = $LOCATION;}
	else { $sel = $VIEW->get($sel);}
	$sel = norm_sel($sel);
	my $obj = Nisplus->new($sel);
	if ($obj->getattr()) { $LOCATION= Nisplus::strip_first($LOCATION);return;}
	my $t = $obj->{'Type'};
	if ($t eq 'DIRECTORY') {
		$LOCATION = $sel;
		list_dir($LOCATION);
	}
	elsif ($t eq 'TABLE') {
		show_table($obj);
	}
	elsif ($t eq 'GROUP'){
		show_group($obj);
	}
}

##
## list a directory
##
sub list_dir{
	my $l = shift;
	if ($STATE eq $BUSY) {return;};
	con_child(LS_R_PIPE, LS_W_PIPE, LS_ER_PIPE, LS_EW_PIPE,
		\&list_cont, 'nisls', $l); 
	$STATE = $BUSY;
	$VIEW->delete(0, "end"); # Can't do it in the wait routine, sorry
}


##
## list_cont put the contents in the ScrolListbox
##
sub list_cont {
	my $d = shift;
	my $dump = <$d>;
	my $err_p = shift;
	my $err = <$err_p>;
	my $obj;
	if (defined($err) && $err =~ /unreachable/) {
		show_ok($err);
		$STATE = "";
		$VIEW->insert("end", '.');
		$VIEW->insert("end", '..');
		return;
	}
	if (defined($dump)){
		$dump =~ /(.*):/;
		$VIEW->insert("end", '.');
		if ($1 ne $ROOTDIR) {$VIEW->insert("end", '..');}
	}
	while (<$d>) {
		chop ;
		$obj = Nisplus->new($_ . "." .$LOCATION);
		$obj->getattr();
		if (!defined($obj->{'Type'})) { next; }
		if ($obj->{'Type'} eq 'DIRECTORY') {
			$VIEW->insert("end", "D  " . $_);
		}
		elsif ($obj->{'Type'} eq 'TABLE') {
			$VIEW->insert("end", "T  " . $_);
		}
		elsif ($obj->{'Type'} eq 'GROUP') {
			$VIEW->insert("end", "G  " . $_);
		}
		elsif ($obj->{'Type'} eq 'LINK') {
			$VIEW->insert("end", "L  " . $_);
		}
	}
	$STATE = "";
}


##
## create a new domain
##
sub new_domain {
	my $domain;
	my $master;
	my $netpass;
	my $button;
	my $WARN;
	my %RET;
	my $is_root;

	if (defined ($LOCATION)) { $master = Nisplus::mymachinename();}
	while (!defined($domain) || !defined($master) || $WARN ne ""){
		$button = dialog_new_domdir("New domain", \$domain, \$master);
		if ($button eq 'Cancel') { return;}
		$WARN = "";
		if (!defined($domain)) {
			$WARN = "You have to specify a Domainname.\n";
		}
		if (defined($domain)) {
			if (!($domain =~ /.*\./)){ $domain .= '.' . $LOCATION;}
			%RET = Nisplus::check_domainname($domain);
			if($RET{'RET'}){
				if (defined ($RET{'TEXT'})){
					$WARN .= $RET{"TEXT"}; $RET{'TEXT'} = '';
				}
				else {
					$WARN .= "Invalid domainname.\n";
				}
			}	
		}
		if (!defined($master)) {
			$WARN .= "You have to specify a Master server.\n"
		}
		if (defined($master)) {
			%RET = Nisplus::check_server($master);
			if ($RET{"RET"}){
				if (defined ($RET{"TEXT"})){
					$WARN .= $RET{"TEXT"}; $RET{"TEXT"} = '';
				}
				else {
					$WARN .= "Invalid masterserver.\n";
				}
			}	
		}
		if ($WARN){
			show_ok($WARN);
		}
	}
	$domain = Nisplus::absname($domain);
	if (!Nisplus::is_rootdomain($domain)) {
		while (!defined($netpass)){
			($button, $netpass) = dialog_new_netpassword("Network password for $master");
		}
		if ($button eq 'Cancel') { return;}
	}
	if (!defined($netpass)){ $netpass="";};
	$VIEW->delete(0, "end"); # Can't do it in the wait routine, sorry
	con_child(LS_R_PIPE, LS_W_PIPE, LS_ER_PIPE, LS_EW_PIPE,
		\&generic_list_cont, 'new_domain.pl', "$domain $master $netpass");
	$STATE = $BUSY;
}


##
## Show a Dialog and ask for the new Domain
##
sub dialog_new_domdir{
	my $text = shift;
	my $domain = shift;
	my $master = shift;
	my $label_text;

	if ($text =~ /domain/) { $label_text = "Domainname"}
	else { $label_text = "Directoryname"; }
	my $d = $top->DialogBox(-title => $text,
		-buttons => ["Create", "Cancel"]);
	my $f_domain = $d->add(Frame);
	$f_domain->Button(-text => 'Help', 
		-command => sub{show_help('Domainname')})
		->pack(-side => 'right');
	my $du=$f_domain->Entry(-textvariable => $domain, -width => 20,)
		->pack(-side =>'right');
	$f_domain->Label(-text => "Domainname:", -justify=>'right')
		->pack(-side => 'right');

	$f_domain->pack(-fill=>'x');
	my $f_master = $d->add(Frame);
	$f_master->Button(-text => 'Help', 
		-command => sub{show_help('Masterserver')})
		->pack(-side => 'right');
	$f_master->Entry(-textvariable => $master, -width => 20)
		->pack(-side =>'right');
	$f_master->Label(-text => 'Master server:', -justify=>'right')
		->pack(-side => 'right');
	$f_master->pack(-fill=>'x');
	my $b = $d->Show();
	return ($b);
}

##
## Show Dialog netpassword
##
sub dialog_new_netpassword{
	my $s = shift;
	my $netpass;
	my $d = $top->DialogBox(-title => $s, 
		-buttons => ["Ok", "Cancel"]);
	my $f_passwd = $d->add(Frame);
	$f_passwd->Label(-text => 'Net. Password: ')
		->pack(-side => 'left');
	my $p=$f_passwd->Entry(-textvariable => \$netpass, -width => 20)
		->pack(-side =>'left');
	$f_passwd->Button(-text => 'Help', 
		-command => sub{show_help('Netpassword')})
		->pack(-side => 'left');
	$f_passwd->pack();
	my $b = $d->Show();
	return ($b, $netpass);
}

##
## create a new directory
##
sub new_directory{
	my $dir;
	my $master;
	my $button;
	my $warn;
	my %RET;

	if (defined($LOCATION)) { $master = Nisplus::mymachinename();}
	while(!defined($dir) || !defined($master) || $warn ne ""){
		$button = dialog_new_domdir("New domain", \$dir, \$master);
		if ($button eq 'Cancel') { return;}
		$warn = "";
		if (!defined($dir)) {
			$warn = "Please give a directory name.";
		}
		if (defined($dir)){
			if (!($dir =~ /.*\./)) { $dir .= '.' . $LOCATION;}
			%RET = Nisplus::check_domainname($dir);
			if ($RET{'RET'}) {
				if (defined($RET{'TEXT'})) {
					$warn = $RET{'TEXT'};
				} else {
					$warn .= "Invalid directory name.\n";
				}
			}
		}
		if (!defined($master)) {
			$warn .= "Please name a master server.\n";
		}
		if (defined($master)){
			%RET = Nisplus::check_server($master);
			if ($RET{'RET'}) {
				if (defined($RET{'TEXT'})) {
					$warn = $RET{'TEXT'};
				} else {
					$warn .= "Invalid master server.\n";
				}
			}
		}
		if ($warn){ show_ok($warn);}
	}
	$VIEW->delete(0, "end"); # Can't do it in the wait routine, sorry
	con_child(LS_R_PIPE, LS_W_PIPE, LS_ER_PIPE, LS_EW_PIPE,
		\&generic_list_cont, 'nis_mkdir.pl', "-m $master  $dir");
	$STATE = $BUSY;
}


##
## create a new table
##
sub new_table{
	my $table;
	my $colnum;
	my $button;
	$button = new_table_dialog(\$table, \$colnum);
	return if ($button eq 'Cancel');
	if (!($table =~ /.*\./)) { $table = $table . '.' . $LOCATION;}
	$button = create_table_dialog($table, $colnum);
	return if ($button eq 'Cancel');
}

##
## new table dialog
##
sub new_table_dialog{
	my $table = shift;
	my $colnum = shift;
	my $w='18';
	my $r;
	
	my $win = $top->DialogBox(-title => "Create new table",
		-buttons => ['Create', 'Cancel']);
	my $f = $win->add(Frame)->pack(-anchor=>'w');
	$f->Label(-text=>'Tablename:', -width=>$w, -anchor=>'e')
		->pack(-side => 'left');
	$f->Entry(-textvariable => $table, -width =>'30')->pack();
	$f = $win->add(Frame)->pack(-anchor=>'w');
	$f->Label(-text=>'Number of Columns:', -width=>$w, -anchor=>'e')
		->pack(-side=>'left');
	$f->Entry(-textvariable=>$colnum, -width=>'30')->pack();
	while (!defined($$table) || !defined($$colnum)){
		$r = $win->Show();
		return $r if ($r eq 'Cancel');
		if (!defined($$table)){show_ok("You have to give a tablename.");}
		if (!defined($$colnum)){show_ok("You have to specify the number of columns.");}
		if (defined($$colnum) && $colnum <= 0){
			show_ok("Please give a numeric value greater 0 for number of columns.");
			undef($$colnum);
		}
	}
	return($r);
}


##
## create table dialog
##
sub create_table_dialog{
	my $table = shift;
	my $colnum = shift;
	my $button;
	my $w = '10';
	my $group;
	my $separator=' ';
	my $r;
	my $i;
	my @col;

	my $win = $top->DialogBox(-title=>"Create table: $table",
		-buttons=>['Create', 'Cancel']);
 	my $f = $win->add(Frame)->pack(-anchor=>'w');
	$f->Label(-text=>'Group:', -width=>$w, -anchor=>'e')
		->pack(-side => 'left');
	$f->Entry(-textvariable=>\$group, -width=>'30')->pack();
 	$f = $win->add(Frame)->pack(-anchor=>'w');
	$f->Label(-text=>'Separator:', -width=>$w, -anchor=>'e')
		->pack(-side => 'left');
	$f->Entry(-textvariable=>\$separator, -width=>'30')->pack();
	$f = $win->add(Frame)->pack(-anchor=>'w');
	$f->Label(-text=>'Table rights:')->pack(-side=>'left');
	my $tbl_r_hash = rights2hash(Nisplus::get_default_rights());
	show_me_rights($f, $tbl_r_hash);
	$f = $win->add(Frame)->pack(-anchor=>'w');
	my @f; my $num = $NR-1;
	for ($i=0; $i<$colnum; $i++){
		if (!defined($f[$i%$num])){
			$f[$i%$num] = $f->Frame()
			->pack(-anchor=>'w');
		}
		my $f1 = $f[$i%$num]->Frame(-relief=>'sunken',-borderwidth=>2)
			->pack(-side=>'left');
		$f1->Label(-text=>"Column #$i:")->pack(-side=>'left', -pady=>'1m');
		$col[$i] = new_column($f1);
	}
	my $check=1;
	while ($check || !check_separator($separator)){
	  	$r = $win->Show();
		return $r if ($r eq 'Cancel');
		$check = !check_cols(\@col, $colnum);
	}
	my $args = new_table_args($table, $group, $separator, $tbl_r_hash, \@col);
	$VIEW->delete(0, "end"); # Can't do it in the wait routine, sorry
	con_child(LS_R_PIPE, LS_W_PIPE, LS_ER_PIPE, LS_EW_PIPE,
		\&generic_list_cont, 'nis_tbladm.pl', "$args");
	$STATE = $BUSY;
	return $r;
}

##
## dipslay a new column
##
sub new_column{
	my $f = shift;
	my %h;
	my $m='10';
	my $f1 = $f->Frame()->pack(-anchor=>'w');
	$f1->Label(-text=>'Col. name:', -anchor=>'e', -width=>$m)-> pack(-side=>'left');
	$f1->Entry(-textvariable=>\$h{'Name'}, -width=>$m)->pack(-side=>'left');
	$f1 = $f->Frame()->pack(-anchor=>'w');
	$h{'S'}=1;
	$f1->Checkbutton(-text=>'Seachable', -variable=>\$h{'S'})->pack(-anchor=>'w');
	$h{'C'}=1;
	$f1->Checkbutton(-text=>'Case sensitive', -variable=>\$h{'C'})->pack(-anchor=>'w');
	my $f2 = $f->Frame()->pack(-anchor=>'w');
	$h{'Rights'} = rights2hash("----------------");
	show_me_rights($f2, $h{'Rights'});
	return(\%h);
}

##
## create the nisplus command to create a table
##
sub new_table_args{
	my $table = shift;
	my $group = shift;
	my $sep = shift;
	my $tbl_rights = shift;
	my $cols = shift;
	my $base = Nisplus::get_first($table);
	my $defaults = "-D access=" . righthash2mod($tbl_rights);
	my $s = " $defaults -c -s \\\'$sep\\\'  $base";$s .= '_tbl ';
	my $i; my $j; my $k;
	my $hash;
	my $col;
	my $rights; my $r;
	my @p = ('n', 'o', 'g', 'w');

	$i = 0;
	while (defined($cols->[$i])){
		$hash = $cols->[$i];
		undef($col);
		$col = 'S' if ($hash->{'S'});
		if (!$hash->{'C'}) {
			if (defined($col)){ $col .= 'I';}
			else {$col = 'I';}
		}
		if (defined($col)){ $col .= ',';}
		else {$col = ",";}
		$col .= righthash2mod($hash->{'Rights'});
		$s .= "$hash->{'Name'}=$col ";
		$i++;
	}
	$s .= "$table";
	return $s;

}


##
## see if all columns a plausible
##
sub check_cols{
	my $c = shift;
	my $n = shift;
	my $i;
	my $h;
	
	for ($i=0; $i<$n; $i++){
		$h = $c->[$i];
		if (!defined($h->{'Name'})){
			show_ok("Please name all columns.");
			return 0;
		}
	}
	return 1;
}

##
## see if it is a valid separator
##
sub check_separator{
	my $sep = shift;
	my @s = split(//, $sep);
	if ($sep eq '\n') {
		show_ok("Separator can't be a newline character.");
		return 0;
	}
	if ($#s > 0 ) {
		show_ok("Separator can be only one character.");
		return 0;
	}
	return 1;
}


##
## destroy a table
##
sub destroy_table{
	my $sel = shift;
	
	if (!defined($sel)){
		show_ok("Please choose a Table to destroy.");
		return;
	}
	$sel = $VIEW->get($sel);
	$sel = norm_sel($sel);
	my $r = show_ok_cancel("Do You realy want to destroy table:\n\n$sel");
	if ($r ne 'Ok') { return;}
	$VIEW->delete(0, "end"); # Can't do it in the wait routine, sorry
	con_child(LS_R_PIPE, LS_W_PIPE, LS_ER_PIPE, LS_EW_PIPE,
		\&generic_list_cont, './del_table.pl', "$sel");
	$STATE = $BUSY;
}



##
## destroy a directory
##
sub destroy_dir{
	my $sel = shift;
	
	if (!defined($sel)){
		show_ok("Please choose a Directory to destroy.");
		return;
	}
	$sel = $VIEW->get($sel);
	$sel = norm_sel($sel);
	my $r = show_ok_cancel("Do You really want to destroy Directory:\n\n$sel");
	if ($r ne 'Ok') { return;}
	$VIEW->delete(0, "end"); # Can't do it in the wait routine, sorry
	con_child(LS_R_PIPE, LS_W_PIPE, LS_ER_PIPE, LS_EW_PIPE,
		\&generic_list_cont, './del_dir.pl', "$sel");
	$STATE = $BUSY;
}


##
## destroy a domain
##
sub destroy_dom{
	my $sel = shift;
	
	if (!defined($sel)){
		show_ok("Please choose a Domain to destroy.");
		return;
	}
	$sel = $VIEW->get($sel);
	$sel = norm_sel($sel);
	my $r = show_ok_cancel("Do You really want to destroy Domain:\n\n$sel");
	if ($r ne 'Ok') { return;}
	$VIEW->delete(0, "end"); # Can't do it in the wait routine, sorry
	con_child(LS_R_PIPE, LS_W_PIPE, LS_ER_PIPE, LS_EW_PIPE,
		\&generic_list_cont, './del_domain.pl', "$sel");
	$STATE = $BUSY;
}

##
## dump a table
##
sub dump_table{
	my $sel = shift;
	my $button;
	my $file;
	my $withheader;
	my $append;
	
	if (!defined($sel)){
		show_ok("Please choose a Table to dump.");
		return;
	}
	$sel = norm_sel($sel);
	while (!defined($file)){
		($button, $file, $withheader, $append) = dump_table_dialog($sel);
		if ($button eq 'Cancel'){ return;}
		if (!defined($file)){ show_ok("You have to specify a Filename");}
	}
	my $opt="";
	$opt = 'h' if $withheader;
	$opt .= 'a' if $append;
	if ($opt ne ""){ $opt = '-' . $opt;}
	if (-e $file && -f $file && $append == 0) {
		my $ret = show_ok_cancel("File $file exists. Overwrite it?");
		return if ($ret eq 'Cancel');
	}
	$OPERATION = "Dumping Table $sel";
	con_child(LS_R_PIPE, LS_W_PIPE, LS_ER_PIPE, LS_EW_PIPE,
		\&ready_cont, './dump_table.pl', "$opt $sel $file");
}

##
## dump table menu
##
sub dump_table_dialog{
	my $sel = shift;
	my $file;
	my $withheader=0;
	my $append=0;

	my $win = $top->DialogBox(-title => "Dump table to file",
		-buttons => ['Dump', 'Cancel']);
	my $f = $win->add(Frame);
	my $f1 = $f->Frame();
	$f1->Label(-text => "Table:")->pack( -anchor => 'e');
	$f1->Label(-text => "Dumpfile:")->pack(-anchor => 'e');
	$f1->pack(-side => 'left');
	$f1 = $f->Frame();
	$f1->Label(-text => "$sel")->pack(-anchor => 'w');
	$f1->Entry(-textvariable => \$file, -width => 30)->pack();
	$f1->pack(-side=>'left');
	$f->pack(-fill=>'x');
	$f = $win->add(Frame);
	$f->pack(-anchor=>'w', -pady => '4m');
	$f->Label(-text => "Options:")->pack(-anchor => 'w');
	$f->Checkbutton(-text => "Dump with Headerline", 
		-variable => \$withheader)->pack(-anchor=>'w');
	$f->Checkbutton(-text => "Append to Dumpfile", 
		-variable => \$append)->pack(-anchor=>'w');
	my $r = $win->Show();
	return ($r, $file,  $withheader, $append);
}


##
## load a table from a file
##
sub load_table{
	my $sel = shift;
	my $view = shift;
	my $button;
	my $file;
	my $option="";
	
	if (!defined($sel) || $sel eq '.'){
		show_ok("Please choose a table to load from a file.");
		return;
	}
	$sel = norm_sel($sel);
	while (!defined($file)){
		($button, $file, $option) = load_table_dialog($sel);
		if ($button eq 'Cancel'){ return;}
		if (!defined($file)){ show_ok("You have to specify a Filename");}
	}
	if (-e $file && !(-f $file)) {
		show_ok("File $file is not a file.");
		return;
	}
	if (!(-e $file)){
		show_ok("File $file does not exist.");
		return;
	}
	$OPERATION = "Loading `table $sel from file $file";
	if (!defined($view)){
		con_child(LS_R_PIPE, LS_W_PIPE, LS_ER_PIPE, LS_EW_PIPE,
			\&ready_cont, './load_table.pl', "$option $sel $file");
	}
	else {
		my $obj = Nisplus->new($sel);
		con_child(LS_R_PIPE, LS_W_PIPE, LS_ER_PIPE, LS_EW_PIPE,
			\&generic_list_table_cont, './load_table.pl', "$option $sel $file", $view, $obj);
	}
}

##
## load table menu
##
sub load_table_dialog{
	my $sel = shift;
	my $file;
	my $option="";

	my $win = $top->DialogBox(-title => "Load table from file",
		-buttons => ['Load', 'Cancel']);
	my $f = $win->add(Frame);
	my $f1 = $f->Frame();
	$f1->Label(-text => "Table:")->pack( -anchor => 'e');
	$f1->Label(-text => "Sourcefile:")->pack(-anchor => 'e');
	$f1->pack(-side => 'left');
	$f1 = $f->Frame();
	$f1->Label(-text => "$sel")->pack(-anchor => 'w');
	$f1->Entry(-textvariable => \$file, -width => 30)->pack();
	$f1->pack(-side=>'left');
	$f->pack(-fill=>'x');
	$f = $win->add(Frame);
	$f->pack(-anchor=>'w', -pady => '4m');
	$f->Label(-text => "Options:")->pack(-anchor => 'w');
	$f->Radiobutton(-text => "Merge file with table", 
		-variable => \$option, -value => '-m')->pack(-anchor=>'w');
	$f->Radiobutton(-text => "Add file to table", 
		-variable => \$option, -value => '-a')->pack(-anchor=>'w');
	$f->Radiobutton(-text => "Replace file with table", 
		-variable => \$option, -value => '-r')->pack(-anchor=>'w');
	my $r = $win->Show();
	return ($r, $file,  $option);
}
##
## load a domain from files
##
sub load_domain{
	my $sel = shift;
	my $button;
	my $file;
	my $option="";
	
	if (!defined($sel)){
		show_ok("Please choose a domain to load from files.");
		return;
	}
	$sel = $VIEW->get($sel);
	$sel = norm_sel($sel);
	while (!defined($file)){
		($button, $file, $option) = load_domain_dialog($sel);
		if ($button eq 'Cancel'){ return;}
		if (!defined($file)){ show_ok("You have to specify a directory name");}
	}
	if (-e $file && !(-d $file)) {
		show_ok("Directory $file is not a directory.");
		return;
	}
	if (!(-e $file)){
		show_ok("Directory $file does not exist.");
		return;
	}
	$OPERATION = "Loading `domain $sel from directory $file";
	con_child(LS_R_PIPE, LS_W_PIPE, LS_ER_PIPE, LS_EW_PIPE,
		\&ready_cont, './load_domain.pl', "$option $sel $file");
}

##
## load domain menu
##
sub load_domain_dialog{
	my $sel = shift;
	my $file;
	my $option="";

	my $win = $top->DialogBox(-title => "Load domain from directory",
		-buttons => ['Load', 'Cancel']);
	my $f = $win->add(Frame);
	my $f1 = $f->Frame();
	$f1->Label(-text => "Domain:")->pack( -anchor => 'e');
	$f1->Label(-text => "Source directory:")->pack(-anchor => 'e');
	$f1->pack(-side => 'left');
	$f1 = $f->Frame();
	$f1->Label(-text => "$sel")->pack(-anchor => 'w');
	$f1->Entry(-textvariable => \$file, -width => 30)->pack();
	$f1->pack(-side=>'left');
	$f->pack(-fill=>'x');
	$f = $win->add(Frame);
	$f->pack(-anchor=>'w', -pady => '4m');
	$f->Label(-text => "Options for standard tables only:")->pack(-anchor => 'w');
	$f->Radiobutton(-text => "Merge files with tables", 
		-variable => \$option, -value => '-m')->pack(-anchor=>'w');
	$f->Radiobutton(-text => "Add files to tables", 
		-variable => \$option, -value => '-a')->pack(-anchor=>'w');
	$f->Radiobutton(-text => "Replace files with tables", 
		-variable => \$option, -value => '-r')->pack(-anchor=>'w');
	my $r = $win->Show();
	return ($r, $file,  $option);
}



##
## dump a domain
##
sub dump_domain{
	my $sel = shift;
	my $button;
	my $dir;
	my $create;
	
	if (!defined($sel)){
		$sel=$LOCATION;
	}
	else {
		$sel = $VIEW->get($sel);
		$sel = norm_sel($sel);
	}
	while (!defined($dir)){
		($button, $dir, $create) = dump_domain_dialog($sel);
		if ($button eq 'Cancel'){ return;}
		if (!defined($dir)){ show_ok("You have to specify a dump directory.");}
	}
	my $opt="";
	$opt = 'c' if $create;
	if ($opt ne ""){ $opt = '-' . $opt;}
	if (!(-e $dir) && !$create) {
		my $ret = show_ok_cancel("Directory $dir does not exist. Create it?");
		return if ($ret eq 'Cancel');
		$opt = '-c';
	}
	$OPERATION="Dumping domain $sel";
	con_child(LS_R_PIPE, LS_W_PIPE, LS_ER_PIPE, LS_EW_PIPE,
		\&ready_cont, './dump_domain.pl', "$opt $sel $dir");
}

##
## dump domain menu
##
sub dump_domain_dialog{
	my $sel = shift;
	my $dir;
	my $create=0;

	my $win = $top->DialogBox(-title => "Dump domain to directory",
		-buttons => ['Dump', 'Cancel']);
	my $f = $win->add(Frame);
	my $f1 = $f->Frame();
	$f1->Label(-text => "Domain:")->pack( -anchor => 'e');
	$f1->Label(-text => "Dump directory:")->pack(-anchor => 'e');
	$f1->pack(-side => 'left');
	$f1 = $f->Frame();
	$f1->Label(-text => "$sel")->pack(-anchor => 'w');
	$f1->Entry(-textvariable => \$dir, -width => 30)->pack();
	$f1->pack(-side=>'left');
	$f->pack(-fill=>'x');
	$f = $win->add(Frame);
	$f->pack(-anchor=>'w', -pady => '4m');
	$f->Label(-text => "Options:")->pack(-anchor => 'w');
	$f->Checkbutton(-text => "Create dump directory", 
		-variable => \$create)->pack(-anchor=>'w');
		my $r = $win->Show();
	return ($r, $dir,  $create);
}

##
## show properties of an object
##
sub properties{
	my $sel = shift;
	my $o;
	my $RETSTR;
	
	if (!defined($sel)){
		show_ok("Please choose a NIS+ object.");
		return;
	}
	$sel = $VIEW->get($sel);
	$sel = norm_sel($sel);
	$o = Nisplus->new($sel);
	$o->getattr();
	if ($RETSTR = $o->get_error()){
		show_ok("$RETSTR");
		return;
	}
	if ($o->{'Type'} eq 'TABLE'){
		show_table_props($o);
	}
	elsif ($o->{'Type'} eq 'GROUP'){
		show_group_props($o);
	}
	elsif ($o->{'Type'} eq 'DIRECTORY'){
		show_dir_props($o);
	}
}


##
## create a child and connect it with a pipe to the
## parent
##
sub con_child {
	my $R_PIPE = shift;	# read pipe
	my $W_PIPE = shift;	# write pipe
	my $ER_PIPE = shift;	# STDERR read Pipe
	my $EW_PIPE = shift;	# STDERR write Pipe
	my $tool_funct = shift;	# Tool function reading the pipes
	my $nis_func = shift;	# NIS+ function to call
	my $args = shift; 	# Argument to NIS+ function
	my $view = shift;	# if we have to list something, this is where to list
	my $obj = shift;	# object to be listed
	my $pid;		# new prozess id

	$| = 1;
	if (!pipe ($R_PIPE, $W_PIPE)){print STDERR "Can't open pipe\n";}
	pipe ($ER_PIPE, $EW_PIPE);
	if ($pid = fork) {
		close $W_PIPE;
		close $EW_PIPE;
		sig_handler($pid, $tool_funct, $R_PIPE, $ER_PIPE, $view, $obj);
	}
	else {
		if (defined $pid) {
			close $R_PIPE;
			close $ER_PIPE;
			open(STDOUT, ">&$W_PIPE");
			open(STDERR, ">&$EW_PIPE");
			if (!exec ("$nis_func $args")) {
			exec ("echo Error:Can't find program $nis_func");}
		}
		else {
			show_ok("Can't fork $!"); return 1;
		}
	}
	return 0;
}


##
## signal handler
##
sub sig_handler {
	my $pid = shift;
	my $funct = shift;
	my %pids;
	my $R_PIPE;
	my $ER_PIPE;
	my $view;
	my $obj;

	# we asign a function to a pid
	if (defined($funct)) { 
		$R_PIPE = shift;
		$ER_PIPE = shift;
		$view = shift;
		$obj = shift;
		$pids->{$pid} = $funct;
		$pids->{"$pid R_PIPE"} = $R_PIPE;
		$pids->{"$pid ER_PIPE"} = $ER_PIPE;
		$pids->{"$pid view"} = $view;
		$pids->{"$pid obj"} = $obj;
		return 0;
	}
	# we got a signal, call function
	if (!defined ($pids->{$pid})) {
		#print STDERR "Got SIGCHLD from Process $pid\n";
		return 0;
	}
	$funct = $pids->{$pid};
	$R_PIPE = $pids->{"$pid R_PIPE"};
	$ER_PIPE = $pids->{"$pid ER_PIPE"};
	$view = $pids->{"$pid view"};
	$obj = $pids->{"$pid obj"};
	if (defined($view)){
		&$funct($R_PIPE, $ER_PIPE, $view, $obj);
	} else {
		&$funct($R_PIPE, $ER_PIPE);
	}
	undef ($pids->{$pid});
	undef ($pids->{"$pid R_PIPE"});
	undef ($pids->{"$pid ER_PIPE"});
	undef ($pids->{"$pid view"});
	undef ($pids->{"$pid obj"});
	return 0;
}


##
## show a group
##
sub show_group{
	my $obj = shift;

	$STATE = $BUSY;
	my $win = $top->Toplevel(-title=>"NIS+ Group: $obj->{'Name'}");
	my $menu = $win->Frame(-relief => 'raised', -bd => 2);
	$menu->pack(-side => 'top', -fill => 'x');
	my $f = $win->Frame()->pack(-anchor=>'w', -fill=>'both', -expand => 'yes');
	$f->Label(-text => "Group members:", -anchor=>'w')
		->pack(-anchor=>'w');
	my $view = $f->ScrlListbox(-selectmode=>'extended', -width=>'30')
		->pack(-side => 'left', -expand => 'yes', -fill => 'both');
	list_group($obj, $view);
	# Tool menue
	my $menu_tool = $menu->Menubutton(-text => 'Tool')->pack(-side => 'left');
	$menu_tool->command(-label => 'Exit',
			-command => sub {$win->destroy()});
	# Group menue
	my $menu_table = $menu->Menubutton(-text => 'Group')->pack(-side => 'left');
	$menu_table->command(-label => 'Properties...', 
			-command => sub{show_group_props($obj)});
	$menu_table->command(-label=>'Reload group', 
		-command=>sub{$view->delete(0, 'end');
			$obj->getattr(); list_group($obj, $view)});
	# Member menue
	my $menu_table = $menu->Menubutton(-text => 'Member')->pack(-side => 'left');
	$menu_table->command(-label => 'Add...', 
			-command => sub{add_member($obj, $view)});
	$menu_table->command(-label=>'Delete', 
		-command=>sub{delete_member($obj, $view)});
	
	$STATE = "";
}

##
## list the members
##
sub list_group{
	my $obj = shift;
	my $view = shift;
	my @mem = split(/ /, $obj->{'Members'});
	foreach (@mem){$view->insert('end', $_);}
}

##
## add a group member
##
sub add_member{
	my $obj = shift;
	my $view = shift;
	my $mem;
	my $win = $top->DialogBox(-title=>"Add member to NIS+ group $obj->{'Name'}",
		-buttons=>['Add', 'Cancel']);
	my $f = $win->add(Frame)->pack();
	$f->Label(-text=>'New member:', -width=>'12', -anchor=>'e')->pack(-side=>'left');
	$f->Entry(-textvariable=>\$mem, -width=>'30')->pack(-side=>'left');
	while (!defined($mem)){
		my $r = $win->Show();
		return if ($r eq 'Cancel');
		if (!defined($mem)){ show_ok("Please give a member name.");}
	}
	my $group = $obj->groupname();
	my $arg = "-a $group $mem";
	$view->delete(0, "end"); # Can't do it in the wait routine, sorry
	con_child(LS_R_PIPE, LS_W_PIPE, LS_ER_PIPE, LS_EW_PIPE,
		\&generic_list_group_cont, 'nis_grpadm.pl', "$arg", $view, $obj);	
}


##
## delete a member
##
sub delete_member{
	my $obj = shift;
	my $view = shift;

	my $sel; my $last;
	my @sel = $view->curselection;
	if (!defined(@sel)) {
		show_ok("Please choose a member.");
		return;
	}
	my $groupname = $obj->groupname();
	my $arg = "-r $groupname ";
	my $i;
	foreach $i (@sel){
		$sel = $view->get($i);
		$arg .= "$sel ";
	}
	$view->delete(0, "end"); # Can't do it in the wait routine, sorry
	con_child(LS_R_PIPE, LS_W_PIPE, LS_ER_PIPE, LS_EW_PIPE,
		\&generic_list_group_cont, 'nis_grpadm.pl', "$arg", $view, $obj);
}

##
## add a NIS+ group
##
sub add_group{
	my $group;
	my $win = $top->DialogBox(-title=>"Create new NIS+ group",
		-buttons=>['Create', 'Cancel']);
	my $f = $win->add(Frame)->pack(-anchor=>'w');
	$f->Label(-text=>'Groupname:')->pack(-side=>'left');
	$f->Entry(-textvariable=>\$group, -width=>$W)->pack(-side=>'left');
	while(!defined($group)){
		my $r = $win->Show();
		return if($r eq 'Cancel');
		if (!defined($group)){
			show_ok("Please give a groupname.");
		}
	}
	if (Nisplus::get_first($LOCATION) eq 'groups_dir') {
		$VIEW->delete(0, "end"); # Can't do it in the wait routine, sorry
		con_child(LS_R_PIPE, LS_W_PIPE, LS_ER_PIPE, LS_EW_PIPE,
			\&generic_list_cont, 'nis_grpadm.pl', "-c $group");
		$STATE = $BUSY;
	}
	else {
		con_child(LS_R_PIPE, LS_W_PIPE, LS_ER_PIPE, LS_EW_PIPE,
			\&dummy_cont, 'nis_grpadm.pl', "-c $group");
	}
}

##
## destroy a NIS+ group
##
sub destroy_group{
	my $group = shift;
	if (!defined($group) || $group eq '.' || $group eq '..'){
		show_ok("Please choose a group to destroy.");
		return;
	}
	if (!($group =~ /^G(.*)/)){
		show_ok("$group is not a NIS+ group.");
		return;
	}
	$group = norm_sel($group);
	my $g = Nisplus->new($group);
	$group = $g->groupname();
	$VIEW->delete(0, "end"); # Can't do it in the wait routine, sorry
	con_child(LS_R_PIPE, LS_W_PIPE, LS_ER_PIPE, LS_EW_PIPE,
		\&generic_list_cont, 'nis_grpadm.pl', "-d $group");
	$STATE = $BUSY;
}


##
## show a table
##
sub show_table {
	my $obj = shift;
	my $i;
	my $colnames = "\t";

	$STATE=$BUSY;
	my $win = $top->Toplevel();
	$win->configure(-title => "Table: $obj->{'Name'}");
	my $menu = $win->Frame(-relief => 'raised', -bd => 2);
	$menu->pack(-side => 'top', -fill => 'x');

	# Column names
	my $frame_col = $win->Frame()->pack(-side=>'top', -fill=>'x');
	for ($i=0; $i<$obj->{'Colnum'}; $i++) {
		$colnames .= $obj->{"Col$i"}->{'Name'} . "\t";
	}
	$frame_col->Label(-text => $colnames)->pack(-side=>'left');
	# Listbox
	my $view_f = $win->Frame();
	my $view = $view_f->ScrlListbox(-selectmode=>'extended')
		->pack(-side => 'left', -expand => 'yes', -fill => 'both');
	$view_f->pack(-expand => 'yes', -fill => 'both');
	list_table($obj, $view);
	show_table_menu($obj, $win, $menu, $view);
	$view->bind("<Double-Button-1>", sub{change_table_entry($view, $obj)});
	$STATE="";
}


##
## show the table menu
##
sub show_table_menu{
	my $obj = shift;
	my $win = shift;
	my $menu = shift;
	my $view = shift;

	# The generic menu for all tables
	# Tool menue
	my $menu_tool = $menu->Menubutton(-text => 'Tool')->pack(-side => 'left');
	$menu_tool->command(-label => 'Exit',
			-command => sub {$win->destroy()});
	# Table menue
	my $menu_table = $menu->Menubutton(-text => 'Table')->pack(-side => 'left');
	$menu_table->command(-label => 'Properties...', 
			-command => sub{show_table_props($obj)});
	$menu_table->command(-label=>'Reload table', 
		-command=>sub{$view->delete(0, 'end');list_table($obj, $view)});
	$menu_table->separator();
	$menu_table->command(-label=>'Load from file...',
		-command=>sub{$view->delete(0, 'end'); load_table($obj->get_name(), $view)});
	$menu_table->command(-label=>'Dump to file...',
		-command=>sub{dump_table($obj->get_name())});
	# Entry menue
	my $menu_entry = $menu->Menubutton(-text=>'Entry')->pack(-side=>'left');
	$menu_entry->command(-label=>'New...',
		-command=>sub{new_table_entry($view, $obj)});
	$menu_entry->command(-label=>'Change...', 
		-command=>sub{change_table_entry($view, $obj)});
	$menu_entry->command(-label=>'Delete', -command=>sub{del_table_entry($view, $obj)});

	my $name = Nisplus::get_first($obj->get_name());
	if ($name eq 'passwd'){
		my $menu_entry = $menu->Menubutton(-text=>'User')->pack(-side=>'left');
		$menu_entry->command(-label=>'New Credentail...',
			-command=>sub{new_user_cred($view, $obj)});
	} elsif ($name eq 'hosts'){
		my $menu_entry = $menu->Menubutton(-text=>'Host')->pack(-side=>'left');
		$menu_entry->command(-label=>'New Credential...',
			-command=>sub{new_host_cred($view, $obj)});
	}	
}

##
## create a new user credential
##
sub new_user_cred{
	my $view = shift;
	my $obj = shift;
	my @sel = $view->curselection();
	if (!defined(@sel)){
		show_ok("Please select a user."); 
		return;
	}
	my $sel; my $password=""; my $ret;
	foreach (@sel){
		$sel = $view->get($_);
		my @user = split(/:/, $sel);
		($ret, $password) = dialog_new_netpassword("Password for user: $user[0]");
		if ($ret eq 'Cancel') { next;}
		my $domain = Nisplus::strip_first($obj->get_name());
		$domain = Nisplus::strip_first($domain);
		my $args = "$user[0].$domain $user[2] $password";
		$view->delete(0, "end");
		$OPERATION = "Credential added for user: $user[0]";
		con_child(R_PIPE, R_PIPE, ER_PIPE, EW_PIPE,
			\&generic_list_table_cont, 'new_cred.pl', "$args", $view, $obj);
	}
}
##
## create a new host credential
##
sub new_host_cred{
	my $view = shift;
	my $obj = shift;
	my @sel = $view->curselection();
	if (!defined(@sel)){
		show_ok("Please select a host."); 
		return;
	}
	my $sel; my $password=""; my $ret;
	foreach (@sel){
		$sel = $view->get($_);
		my @host = split(/ /, $sel);
		($ret, $password) = dialog_new_netpassword("Password for host: $host[0]");
		if ($ret eq 'Cancel') { next;}
		my $domain = Nisplus::strip_first($obj->get_name());
		$domain = Nisplus::strip_first($domain);
		my $args = "$host[0].$domain 0 $password";
		$OPERATION = "Credentail added for host: $host[0]";
		con_child(R_PIPE, W_PIPE, ER_PIPE, EW_PIPE,
			\&ready_cont, 'new_cred.pl', "$args");
	}
}

##
## new table entry
##
sub new_table_entry{
	my $view = shift;
	my $obj = shift;
	my $entry = $obj->emptyentry();
	table_entry_dialog($obj, $entry, $view);
}


##
## change a table entry
##
sub change_table_entry{
	my $view = shift;
	my $obj = shift;

	my @sel = $view->curselection;
	if (!defined($sel[0])) {
		show_ok("Please choose an entry to change.");
		return;
	}
	if ($#sel > 0){
		show_ok("Please select only one entry to change.");
		return;
	}
	my $sel = $view->get($sel[0]);
	chop ($sel);
	my $entry = $obj->get_entry($sel);
	table_entry_dialog($obj, $entry, $view);
}


##
## generic table entry dialog
##
sub table_entry_dialog{
	my $obj = shift;
	my $entry = shift;
	my $view = shift;
	my $m = '10';

	my %old = %$entry;
	my $d = $top->DialogBox(-title=>'Table entry', 
		-buttons=>['Apply', 'Cancel']);
	my $f = $d->add(Frame)->pack(-anchor=>'w');
	$f->Label(-text=>'Owner:', -anchor=>'e', -width=>$m)->pack(-side=>'left');
	$f->Entry(-textvariable=>\$entry->{'Owner'}, -width=>'30')->pack();
	$f = $d->add(Frame)->pack(-anchor=>'w');
	$f->Label(-text=>'Group:', -anchor=>'e', -width=>$m)->pack(-side=>'left');
	$f->Entry(-textvariable=>\$entry->{'Group'}, -width=>'30')->pack();
	$f = $d->add(Frame)->pack(-anchor=>'w');
	my $rights = rights2hash($entry->{'Access'});
	show_me_rights($f, $rights);
	my $i=0; my $colname; my $label;
	while (defined($entry->{"Col$i"})){
		$f = $d->add(Frame)->pack(-anchor=>'w');
		$label = $f->Label(-text=>$obj->{"Col$i"}->{'Name'}, -anchor=>'e', -width=>$m)
			->pack(-side=>'left');
		$f->Entry(-textvariable=>\$entry->{"Col$i"}, -width=>'30')->pack();
		if ($obj->iskeycol($i)) {$label->configure(-fg=>'red');}
		$i++;
	}
	my $r = $d->Show();
	return if ($r eq 'Cancel');
	$i = 0; my $change = 0; my $args;
	my $new_entry = 0;
	if ($old{'Owner'} eq ''){ $new_entry = 1;}
	if ($new_entry) { $args = '-a ';}
	else { $args = '-m ';}
	while (defined($entry->{"Col$i"})){
		if ($old{"Col$i"} ne $entry->{"Col$i"}){
			my $colname = $obj->{"Col$i"}->{"Name"};
			my $new = $entry->{"Col$i"};
			$change = 1;
			$args .= "$colname=\'$new\' ";
		}
		$i++;
	}
	my $indexedname = $obj->indexedname(\%old);
	if ($change){
		if ($new_entry){ $args .= "$obj->{'Name'}";}
		else { $args .= $indexedname; }
		con_child(ALS_R_PIPE, ALS_W_PIPE, ALS_ER_PIPE, ALS_EW_PIPE,
			\&dummy_cont, 'nistbladm', "$args");
	}
	if ($old{'Owner'} ne $entry->{'Owner'}){
		$change = 1;
		$indexedname = $obj->indexedname($entry);
		$args = "$entry->{'Owner'} $indexedname";
		con_child(BLS_R_PIPE, BLS_W_PIPE, BLS_ER_PIPE, BLS_EW_PIPE,
			\&dummy_cont, 'nischown', "$args");
	}
	if ($old{'Group'} ne $entry->{'Group'}){
		$change = 1;
		$indexedname = $obj->indexedname($entry);
		$args = "$entry->{'Group'} $indexedname";
		con_child(CLS_R_PIPE, CLS_W_PIPE, CLS_ER_PIPE, CLS_EW_PIPE,
			\&dummy_cont, 'nischgrp', "$args");
	}
	$entry->{'Access'} = hash2rights($rights);
	if ($old{'Access'} ne $entry->{'Access'}){
		$change = 1;
		$indexedname = $obj->indexedname($entry);
		my $rs = righthash2mod($rights);
		$args = "$rs $indexedname";
		con_child(DLS_R_PIPE, DLS_W_PIPE, DLS_ER_PIPE, DLS_EW_PIPE,
			\&dummy_cont, 'nischmod', "$args");
	}
	$args = Nisplus::strip_first($obj->{'Name'});
	$view->delete(0, "end"); # Can't do it in the wait routine, sorry
	con_child(NLS_R_PIPE, NLS_W_PIPE, NLS_ER_PIPE, NLS_EW_PIPE,
		\&generic_list_table_cont, 'nisping', "$args", $view, $obj);
}


##
## delete a table entry
##
sub del_table_entry{
	my $view = shift;
	my $obj = shift;
	my @sel = $view->curselection();
	if (!defined(@sel)) {
		show_ok("Please choose an entry.");
		return;
	}
	my $i; my $last=0;
	foreach $i (@sel){
		$last = 1 if ($i == $sel[$#sel]);
		$sel = $view->get($i);
		my $indexedname = $obj->indexedname($obj->get_entry($sel));
		my $arg = "-r $indexedname";
		if ($last) {
			$view->delete(0, "end"); # Can't do it in the wait routine, sorry
			con_child(LS_R_PIPE, LS_W_PIPE, LS_ER_PIPE, LS_EW_PIPE,
			\&generic_list_table_cont, 'nis_tbladm.pl', "$arg", $view, $obj);
		} else {
			con_child(LS_R_PIPE, LS_W_PIPE, LS_ER_PIPE, LS_EW_PIPE,
			\&dummy_cont, 'nis_tbladm.pl', "$arg");
		}
	}
}


##
## show and change Table props
##
sub show_table_props {
	my $o = shift;	
	my @hashes;

	my $win = $top->DialogBox(-title => "Properties of table $o->{'Name'}",
		-buttons=>['Apply', 'Cancel']);
	$o->getattr();
	#
	# Table frame
	#
	my $tbl_frame_owner = $win->add(Frame)->pack(-anchor=>'w');
	$tbl_frame_owner->Label(-text=>"Owner : ", -width=>'7',
		-anchor=>'e')
		->pack(-side =>'left');
	my $owner = $o->{'Owner'};
	$tbl_frame_owner->Entry(-textvariable => \$owner,
		-width=>'30')
		->pack(-side=>'left');
	my $tbl_frame_group = $win->add(Frame)->pack(-anchor=>'w');
	$tbl_frame_group->Label(-text=>"Group : ", -width=>'7',
		-anchor=>'e')
		->pack(-side =>'left');
	my $group = $o->{'Group'};
	$tbl_frame_group->Entry(-textvariable => \$group,
		-width=>'30')
		->pack(-side=>'left');
	my $tbl_rights_frame = $win->add(Frame)->pack(-anchor=>'w');
	my $tbl_rights = rights2hash($o->{'Access'});
	show_me_rights($tbl_rights_frame, $tbl_rights);
	#
	#Column Frames
	#
	my $i;
	my @col_rights;
	my $f = $win->add(Frame)->pack();
	my @f;
	for ($i=0; $i<$o->{'Colnum'}; $i++) {
		if (!defined($f[$i%$NR])){
			$f[$i%$NR] = $f->Frame()->pack(-anchor=>'w');
		}
		my $mf = $f[$i%$NR]->Frame(-relief=>'sunken', -borderwidth=>2)
			->pack(-side=>'left');
		my $col = $o->{"Col$i"};
		$mf->Label(-text=>"$col->{'Name'}", 
			-width=>'10')->pack(-side=>'left');
		my $attr = attr2str($col->{'Attr'});
		$mf->Label(-text=>"Attr.: $attr", -anchor=>'w')
			->pack(-anchor=>'w');
		my $col_r = rights2hash($col->{'Access'});
		$col_rights[$i] = $col_r;
		show_me_rights($mf, $col_r);
	}
	my $r = $win->Show();
	return if ($r eq 'Cancel');
	set_table_attributes($o, $owner, $group, $tbl_rights, \@col_rights);
}

##
## show and change Group props
##
sub show_group_props{
	my $o = shift;

	my $win = $top->DialogBox(-title=>"Properties of NIS+ group $o->{'Name'}",
		-buttons=>['Apply', 'Cancel']);
	$o->getattr(); # for safety
	my $f = $win->add(Frame)->pack(-anchor=>'w');
	$f->Label(-text=>"Owner : ", -width=>'7',
		-anchor=>'e')
		->pack(-side =>'left');
	my $owner = $o->{'Owner'};
	$f->Entry(-textvariable => \$owner,
		-width=>'30')
		->pack(-side=>'left');
	$f = $win->add(Frame)->pack(-anchor=>'w');
	$f->Label(-text=>"Group : ", -width=>'7',
		-anchor=>'e')
		->pack(-side =>'left');
	my $group = $o->{'Group'};
	$f->Entry(-textvariable => \$group,
		-width=>'30')
		->pack(-side=>'left');
	$f = $win->add(Frame)->pack(-anchor=>'w');
	my $o_rights = rights2hash($o->{'Access'});
	show_me_rights($f, $o_rights);
	my $r = $win->Show();
	return if ($r eq 'Cancel');
	my $name = $o->get_name();
	if ($owner ne $o->{'Owner'}){
		$OPERATION = "Changed owner of  $name to $owner";
		con_child(A_R_PIPE, A_W_PIPE, A_ER_PIPE, A_EW_PIPE,
			\&ready_cont, 'nischown', "$owner $name");
	}
	if ($group ne $o->{'Group'}){
		$OPERATION = "Changed group of  $name to $group";
		con_child(B_R_PIPE, B_W_PIPE, B_ER_PIPE, B_EW_PIPE,
			\&ready_cont, 'nischgrp', "$group $name");
	}
	my $r = hash2rights($o_rights);
	if ($r ne $o->{'Access'}){
		my $s = righthash2mod($o_rights);
		$OPERATION = "Changed permissions of  $name to $group";
		con_child(C_R_PIPE, C_W_PIPE, C_ER_PIPE, C_EW_PIPE,
			\&ready_cont, 'nischmod', "$s $name");

	}
}

##
## show dir props
##
sub show_dir_props{
	my $obj = shift;
	my $m=12;

	$obj->getattr();
	my $owner = $obj->owner();
	my $group = $obj->{'Group'};
	my $win = $top->DialogBox(-title=>"Properties of directory $obj->{'Name'}",
		-buttons=>['Apply', 'Add Replica', 'Delete Replica', 'Cancel']);
	my $f = $win->add(Frame)->pack(-anchor=>'w');
	$f->Label(-text=>'Master Server:', -anchor=>'e', -width=>$m)
		->pack(-side=>'left');
	$f->Label(-textvariable=>\$obj->{'Master'}, -anchor=>'w')
		->pack(-side=>'left');
	$f = $win->add(Frame)->pack(-anchor=>'w');
	$f->Label(-text=>'Owner:', -anchor=>'e', -width=>$m)
		->pack(-side=>'left');
	$f->Entry(-textvariable=>\$owner, -width=>$W)->pack(-side=>'left');
	$f = $win->add(Frame)->pack(-anchor=>'w');
	$f->Label(-text=>'Group:', -anchor=>'e', -width=>$m)
		->pack(-side=>'left');
	$f->Entry(-textvariable=>\$group, -width=>$W)->pack(-side=>'left');
	$f = $win->add(Frame)->pack(-anchor=>'w');
	my $rights = rights2hash($obj->{'Access'});
	show_me_rights($f, $rights);
	$f = $win->add(Frame)->pack(-anchor=>'w', -fill=>'both', -expand=>'yes');
	$f->Label(-text => "Servers:", -anchor=>'w')->pack(-side=>'left');
	$f->Label(-textvariable=>\$STATE, -fg=>'red')->pack(-side=>'left');
	$f = $win->add(Frame)->pack(-anchor=>'w', -fill=>'both', -expand=>'yes');
	my $view = $f->ScrlListbox(-selectmode=>'extended', -width=>$W)
		->pack(-side => 'left', -expand => 'yes', -fill => 'both');
	list_server($obj, $view);

	my $cont =1; my $r;
	while ($cont){
		$cont = 0;
		$r = $win->Show();
		if ($r eq 'Add Replica'){
			$cont = 1;
			add_replica($obj, $view);
		}
		if ($r eq 'Delete Replica'){
			$cont = 1;
			del_replica($obj, $view);
		}
	}
	my $name = $obj->name();
	return if ($r eq 'Cancel');
	if ($owner ne $obj->owner()){
		$OPERATION = "Changed owner of  $name to $owner";
		con_child(A_R_PIPE, A_W_PIPE, A_ER_PIPE, A_EW_PIPE,
			\&ready_cont, 'nischown', "$owner $name");
	}
	if ($group ne $obj->{'Group'}){
		$OPERATION = "Changed group of  $name to $group";
		con_child(B_R_PIPE, B_W_PIPE, B_ER_PIPE, B_EW_PIPE,
			\&ready_cont, 'nischgrp', "$group $name");
	}
	my $r = hash2rights($rights);
	if ($r ne $obj->{'Access'}){
		my $s = righthash2mod($rights);
		$OPERATION = "Changed permissions of  $name to $group";
		con_child(C_R_PIPE, C_W_PIPE, C_ER_PIPE, C_EW_PIPE,
			\&ready_cont, 'nischmod', "$s $name");

	}

}

##
## add a replicaserver 
##
sub add_replica{
	my $obj = shift;
	my $view = shift;
	my $m = 12;
	my $replica;
	my $r;
	my $name = $obj->get_name();
	my $first = Nisplus::get_first($name);

	my $win = $top->DialogBox(-text=>'Add replica server',
		-buttons=>['Add', 'Cancel']);
	$f = $win->add(Frame)->pack(-anchor=>'w');
	$f->Label(-text=>'New Replica:', -anchor=>'e', -width=>$m)
		->pack(-side=>'left');
	$f->Entry(-textvariable=>\$replica, -width=>$W)->pack(-side=>'left');
	$r = $win->Show();
	return if ($r eq 'Cancel');
	$view->delete(0, "end");
	$OPERATION = "Added replica $replica to directory $obj->{'Name'}";
	if ($first ne 'org_dir' && $first ne 'groups_dir'){
		$OPERATION .= "\nDon't forget to execute\n  nisinit -c -H <server>\non all clients of domain $name!";
	}
	con_child(LS_R_PIPE, LS_W_PIPE, LS_ER_PIPE, LS_EW_PIPE,
		\&generic_list_server_cont, 'nis_mkdir.pl', "-s $replica $name",
		$view, $obj);	
	$STATE = $BUSY;
}

##
## delete a replicaserver 
##
sub del_replica{
	my $obj = shift;
	my $view = shift;
	my $m = 12;
	my $r;

	my $sel = $view->curselection();
	if (!defined($sel)){
		show_ok("Please select a replica to delete.");
		return;
	}
	$sel = $view->get($sel);
	my $win = $top->DialogBox(-text=>'Delete replica server',
		-buttons=>['Delete', 'Force delete', 'Cancel']);
	my $name = $obj->get_name();
	my $first = Nisplus::get_first($name);
	my $f = $win->add(Frame)->pack();
	$f->Label(-text=>"Do you really want to delete $sel from $name?")->pack();
	$r = $win->Show();
	return if ($r eq 'Cancel');
	my $opt = "-s";
	if ($r eq 'Force delete'){
		$opt = '-f ' . $opt;
	}
	$view->delete(0, "end");
	$OPERATION = "Removed replica $sel from directory $obj->{'Name'}";
	if ($first ne 'org_dir' && $first ne 'groups_dir'){
		$OPERATION .= "\nDon't forget to execute\n  nisinit -c -H <server>\non all clients of domain $name!";
	}
	con_child(LS_R_PIPE, LS_W_PIPE, LS_ER_PIPE, LS_EW_PIPE,
		\&generic_list_server_cont, 'nis_rmdir.pl', "$opt $sel $name",
		$view, $obj);	
	$STATE = $BUSY;
}

##
## ping a directory
##
sub nis_ping{
	my $sel = $VIEW->curselection();
	if (!defined($sel)){
		show_ok("Please select a directory to update.");
		return;
	}
	$sel = $VIEW->get($sel);
	$sel = norm_sel($sel);
	$OPERATION = "Pinging directory $sel.";
	con_child(LS_R_PIPE, LS_W_PIPE, LS_ER_PIPE, LS_EW_PIPE,
		\&ready_cont, 'nisping', "$sel");	
}


##
## list servers of a directory object
##
sub list_server{
	my $obj = shift;
	my $view = shift;
	my @serv = split(/ /, $obj->{'Servers'});
	foreach (@serv){
		$view->insert('end', $_);
	}
	$STATE = "";
}

##
## show access rights in a frame
##
sub show_me_rights{
	my $frame = shift;
	my $h = shift;
	my $m = '7';
	my $i; my $j;
	my @p = ('O', 'G', 'W', 'N');
	my @r = ('r', 'm', 'c', 'd');
	my @l = ('Owner:', 'Group:', 'World:', 'Nobody:');
	for($i=0; $i<4; $i++){
		my $f = $frame->Frame()->pack(-anchor=>'w');
		$f->Label(-text=>$l[$i], -anchor=>'e', -width=>$m)
			->pack(-side=>'left');
		for($j=0; $j<4; $j++) {
			$f->Checkbutton(-text=>"$r[$j] ", -variable=>\$h->{"$p[$i]$r[$j]"},
				-selectcolor=>'green')
				->pack(-side=>'left');
		}
	}

}


##
## set the attributes of a table
##
sub set_table_attributes {
	my $o = shift;
	my $owner = shift;
	my $group = shift;
	my $tbl_rights = shift;
	my $col_rights = shift;
	my $ret;
	my $retstr;
	my $name = $o->name();
	
	if ($o->owner() ne $owner){
		$OPERATION = "Changed owner of  $name to $owner";
		con_child(A_R_PIPE, A_W_PIPE, A_ER_PIPE, A_EW_PIPE,
			\&ready_cont, 'nis_chown.pl', "$owner $name");
	}
	if ($o->{'Group'} ne $group){
		$OPERATION = "Changed group of  $name to $group";
		con_child(B_R_PIPE, B_W_PIPE, B_ER_PIPE, B_EW_PIPE,
			\&ready_cont, 'nis_chgrp.pl', "$group $name");
	}
	my $rights = hash2rights($tbl_rights);
	if ($rights ne $o->{'Access'}) {
		$OPERATION = "Changed mod of  $name to $rights";
		$rights = righthash2mod($tbl_rights);
		con_child(C_R_PIPE, C_W_PIPE, C_ER_PIPE, C_EW_PIPE,
			\&ready_cont, 'nis_chmod.pl', "$rights $name");	
	}
	my $i; my $change=""; my $mod; my $colname;
	for ($i=0; $i<$o->{'Colnum'}; $i++) {
		$rights = hash2rights(${$col_rights}[$i]);
		if ($rights ne $o->{"Col$i"}->{'Access'}) {
			$mod = righthash2mod(${$col_rights}[$i]);
			$colname = $o->{"Col$i"}->{'Name'};
			$change .= "$colname=$mod ";
		}
	}
	if ($change ne ""){
		$change = "-u " . $change;
		$OPERATION = "Changed columns rights of table $name";
		con_child(D_R_PIPE, D_W_PIPE, D_ER_PIPE, D_EW_PIPE,
			\&ready_cont, 'nis_tbladm.pl', "$change $name");

	}
	
}

##
## change a hash of access rights to a string like r---rmcdrmcdr---
##
sub hash2rights {
	my $h = shift;
	my @r;
	my $i; my $j;
	my @p = ('N', 'O', 'G', 'W');

	for ($i=0; $i<4 ; $i++) {
		$j = 4*$i;
		($h->{"$p[$i]r"} == 1) ?  ($r[$j]='r') : ($r[$j]='-');
		($h->{"$p[$i]m"} == 1) ?  ($r[$j+1]='m') : ($r[$j+1]='-');
		($h->{"$p[$i]c"} == 1) ?  ($r[$j+2]='c') : ($r[$j+2]='-');
		($h->{"$p[$i]d"} == 1) ?  ($r[$j+3]='d') : ($r[$j+3]='-');
	}
	return ( join('', @r));
}


##
## produce a anonymous hash with given rights
##
sub rights2hash{
	my $rstr = shift;
	my %h;
	my @r = split(//, $rstr);
	my $i; my $j;
	my @p = ('N', 'O', 'G', 'W');
	for($i=0; $i<4; $i++){
		$j=4*$i;
		if($r[$j] eq 'r') { $h{"$p[$i]r"} = 1;} else { $h{"$p[$i]r"} = 0;}
		if($r[$j+1] eq 'm') { $h{"$p[$i]m"} = 1;} else { $h{"$p[$i]m"} = 0;}
		if($r[$j+2] eq 'c') { $h{"$p[$i]c"} = 1;} else { $h{"$p[$i]c"} = 0;}
		if($r[$j+3] eq 'd') { $h{"$p[$i]d"} = 1;} else { $h{"$p[$i]d"} = 0;}
	}
	return (\%h);
}

##
## rights printed to be used as argument to nischmod
##
sub righthash2mod{
	my $rights = shift;
	my $ri;
	my @r;
	my $j; my $k;
	my @p = ('n', 'o', 'g', 'w');
	$ri = hash2rights($rights);
	@r = split(//, $ri);
	$ri = "";
	for ($j=0; $j<4; $j++){
		$k = 4*$j;
		$ri .= "$p[$j]=";
		$ri .= "$r[$k]" if ($r[$k] ne '-');
		$ri .= "$r[$k+1]" if ($r[$k+1] ne '-');
		$ri .= "$r[$k+2]" if ($r[$k+2] ne '-');
		$ri .= "$r[$k+3]" if ($r[$k+3] ne '-');
		if($j<3){ $ri .= ',';}
	}
	return $ri;
}



##
## List a NIS+ table
##
sub list_table{
	my $obj = shift;
	my $v = shift;

	my @l = `niscat $obj->{'Name'}`;
	while (@l) {
		$v->insert("end", shift (@l));
	}
	$STATE = "";
}

##
## debuging print hash
##
sub printhash {
	my $h = shift;
	my $key;
	my $val;
	foreach $key ( keys %{$h}) {
		$val = $h->{"$key"};
		print "$key: $val\n";
	}
}

##
## Show a Dialogbox with OK Button
##
sub show_ok{
	my $win = $top->Dialog(-title=>'Notice', 
		-text=>$_[0], -buttons=>['Ok'],
		-bitmap=>'info');
	my $ret = $win->Show();
	$win->destroy();
	return $ret;
}

##
## Show Ok or Cancel
##
sub show_ok_cancel{
	my $win = $top->Dialog(-title=>'Notice', 
		-text=>$_[0], , -bitmap=>'info',
		-buttons=>['Ok', 'Cancel']);	
	my $ret = $win->Show();
	$win->destroy();
	return $ret;
}

##
## find error messages in a stream
##
sub find_errs {
	my $STREAM = shift;
	my $errs="";

	$| = 1;
	while (<$STREAM>){
		$errs .= $_ if (is_error_string($_)); 
	}
	return $errs;
}


##
## check for strings that are errors
##
sub is_error_string{
	my $s = shift;
	my $i;
	foreach $i (@ERROR_STRINGS){
		if ($s =~ /$i/) { return 1;}
	}
	return 0;
}

##
## get the result of new domain when we got a signal
##
sub generic_list_cont {
	my $stdi = shift;
	my $stde = shift;
	my $errs;

	if ($errs = find_errs($stde)) {
		show_ok($errs);
	}
	close($stdi); close($stde);
	con_child(NISLS_R_PIPE, NISLS_W_PIPE, NISLS_ER_PIPE, NISLS_EW_PIPE,
		\&list_cont, 'nisls', $LOCATION); 
}

##
## list the table again
##
sub generic_list_table_cont {
	my $stdi = shift;
	my $stde = shift;
	my $view = shift;
	my $obj = shift;
	my $errs;

	if ($errs = find_errs($stde)) {
		show_ok($errs);
	}
	list_table($obj, $view);
	close($stdi); close($stde);
}

##
## list the group again
##
sub generic_list_group_cont {
	my $stdi = shift;
	my $stde = shift;
	my $view = shift;
	my $obj = shift;
	my $errs;

	if ($errs = find_errs($stde)) {
		show_ok($errs);
	}
	$obj->getattr();
	list_group($obj, $view);
	close($stdi); close($stde);
}

##
## list the servers again
##
sub generic_list_server_cont {
	my $stdi = shift;
	my $stde = shift;
	my $view = shift;
	my $obj = shift;
	my $errs;

	if ($errs = find_errs($stde)) {
		show_ok($errs);
	}
	$obj->getattr();
	list_server($obj, $view);
	if ($OPERATION){
		show_ok($OPERATION);
		$OPERATION = "";
	}
	close($stdi); close($stde);
}

##
## do nothing signal handler, just show errors
##
sub dummy_cont{
	my $stdi = shift;
	my $stde = shift;
	my $errs;

	if ($errs = find_errs($stde)) {
		show_ok($errs);
	}
	close($stdi); close($stde);
}

##
## show a ready box
##
sub ready_cont{
	my $stdi = shift;
	my $stde = shift;
	my $errs;

	if ($errs = find_errs($stde)) {
		show_ok($errs);
		close($stdi); close($stde);
		return;
	}
	close($stdi); close($stde);
	if ($OPERATION){
		show_ok("$OPERATION \nfinished.") if ($OPERATION ne "");
		$OPERATION = "";
	}
}


##
## attributes to a short String
##
sub attr2str{
	my $s = shift;

	my $r = "";
	$r .= "Key, " if ($s =~/SEARCHABLE/);
	$r .= "Text, " if ($s =~/TEXTUAL/);
	$r .= "Case sens., " if ($s =~/SENSITIVE/);
	return $r;
}
