/* Copyright (c) 1997, 1998 Thorsten Kukuk
   Author: Thorsten Kukuk <kukuk@vt.uni-paderborn.de>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#define _GNU_SOURCE

#include <fnmatch.h>
#ifdef HAVE_GETOPT_H
#include <getopt.h>
#else
#include "lib/compat/getopt.h"
#endif
#include <locale.h>
#include <libintl.h>
#include <stdio.h>
#include <regex.h>
#include <string.h>
#include <locale.h>
#include <libintl.h>
#include <rpcsvc/nis.h>

#ifndef _
#define _(String) gettext (String)
#endif

struct match_t {
  unsigned long col;
  bool_t more;
  regex_t reg;
  char *pattern;
};
typedef struct match_t match_t;

static match_t *matches;

static int raw_binary = 0;
static int cnt_only = 0;
static int print_object = 0;
static unsigned long cnt_match = 0;

static int
print_obj (const_nis_name name __attribute__ ((unused)),
	   const nis_object *obj, const void *udata)
{
  const char *separator = (const char *) udata;
  unsigned int j;
  match_t *tmp = matches;
  char buf[NIS_MAXNAMELEN + 1];

  while (tmp->more)
    {
      j = tmp->col;
      memcpy (buf, ENTRY_VAL (obj, j), ENTRY_LEN (obj, j));
      if (ENTRY_LEN (obj, j) < NIS_MAXNAMELEN)
	buf[ENTRY_LEN (obj, j)] = '\0';
      else
	buf[NIS_MAXNAMELEN] = '\0';
      if (regexec (&tmp->reg, buf, 0, NULL, 0))
	{
	  if (fnmatch (tmp->pattern, buf, 0) == FNM_NOMATCH)
	    return 0;
	}
      ++tmp;
    }

  ++cnt_match;
  if (cnt_only)
    return 0;
  else if (print_object)
    nis_print_object (obj);
  else
    {
      for (j = 0; j < obj->EN_data.en_cols.en_cols_len; ++j)
	{
	  if (ENTRY_VAL (obj, j) != NULL)
	    {
	      if ((obj->EN_data.en_cols.en_cols_val->ec_flags & EN_BINARY)
		  && !raw_binary)
		fputs (_("*BINARY*"), stdout);
	      else
		{
		  fprintf (stdout, "%.*s", (int) ENTRY_LEN (obj, j),
			   ENTRY_VAL (obj, j));
		}
	    }
	  if (j + 1 < obj->EN_data.en_cols.en_cols_len)
	    fprintf (stdout, "%c", *separator);
	}
      fputs ("\n", stdout);
    }
  return 0;
}

/* Print the version information.  */
static inline void
print_version (void)
{
  fprintf (stdout, "nisgrep (%s) %s\n", PACKAGE, VERSION);
  fprintf (stdout, gettext ("\
Copyright (C) %s Thorsten Kukuk.\n\
This is free software; see the source for copying conditions.  There is NO\n\
warranty; not even for MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.\n\
"), "1998");
  /* fprintf (stdout, _("Written by %s.\n"), "Thorsten Kukuk"); */
}

static inline void
print_usage (void)
{
  fputs (_("Usage: nisgrep [-AMPchov] [-s sep] ... tablename\n"), stdout);
}

static void
print_help (void)
{
  print_usage ();
  fputs (_("nisgrep - regex searching in NIS+ tables\n\n"),
         stdout);

  fputs (_("  -A             Use ALL_RESULTS flag for lookups\n"), stdout);
  fputs (_("  -M             Query master server only\n"), stdout);
  fputs (_("  -P             Follow concatenation path\n"), stdout);
  fputs (_("  -c             Print only count of found matches\n"), stdout);
  fputs (_("  -h             Print names of columns\n"), stdout);
  fputs (_("  -o             Display internal representation of object\n"),
	 stdout);
  fputs (_("  -s sep         Use \"sep\" to separate the table columns\n"),
	 stdout);
  fputs (_("  --help         Give this help list\n"), stdout);
  fputs (_("  --usage        Give a short usage message\n"), stdout);
  fputs (_("  --version      Print program version\n"), stdout);
}

static inline void
print_error (void)
{
  const char *program = "nisgrep";

  fprintf (stderr,
           _("Try `%s --help' or `%s --usage' for more information.\n"),
           program, program);
}

int
main (int argc, char *argv[])
{
  nis_result *res = NULL;
  char separator = '\t';
  int display_header = 0;
  int new_sep = 0;
  int flags = EXPAND_NAME;
  char *buf, *cp;

  setlocale (LC_MESSAGES, "");
  bindtextdomain (PACKAGE, LOCALEDIR);
  textdomain (PACKAGE);

  while (1)
    {
      int c;
      int option_index = 0;
      static struct option long_options[] =
      {
	{"version", no_argument, NULL, '\255'},
	{"usage", no_argument, NULL, '\254'},
	{"help", no_argument, NULL, '\253'},
	{NULL, 0, NULL, '\0'}
      };

      c = getopt_long (argc, argv, "AMPchos:v", long_options, &option_index);
      if (c == (-1))
	break;
      switch (c)
	{
	case 'o':
	  if (display_header)
	    {
	      fputs (_ ("You could not use -h and -o !\n"), stderr);
	      print_error ();
	      return 1;
	    }
	  print_object = 1;
	  break;
	case 'h':
	  if (print_object)
	    {
	      fputs (_ ("You could not use -h and -o !\n"), stderr);
	      print_error ();
	      return 1;
	    }
	  display_header = 1;
	  break;
	case 's':
	  new_sep = 1;
	  separator = optarg[0];
	  break;
	case 'A':
	  flags |= ALL_RESULTS;
	case 'P':
	  flags |= FOLLOW_PATH;
	  break;
	case 'M':
	  flags |= MASTER_ONLY;
	  break;
	case 'v':
	  ++raw_binary;
	  break;
	case 'c':
	  cnt_only = 1;
	  break;
        case '\253':
          print_help ();
          return 0;
        case '\255':
          print_version ();
          return 0;
        case '\254':
          print_usage ();
          return 0;
        default:
          print_error ();
          return 2;
	}
    }

  argc -= optind;
  argv += optind;

  if (argc < 2)
    {
      fputs (_("nisgrep: To few parameters\n"), stderr);
      print_error ();
      return 2;
    }

  res = nis_lookup (argv[argc - 1], flags);

  if (res == NULL || NIS_RES_STATUS (res) != NIS_SUCCESS)
    {
      if (res == NULL)
	fputs (_("Out of memory!\n"), stderr);
      else
	{
	  nis_perror (NIS_RES_STATUS (res), "nisgrep");
	  nis_freeresult (res);
	}
      return 2;
    }

  if (__type_of (NIS_RES_OBJECT (res)) != NIS_TABLE_OBJ)
    {
      fprintf (stderr, _("%s.%s is not a table!\n"),
	       NIS_RES_OBJECT (res)->zo_name,
	       NIS_RES_OBJECT (res)->zo_domain);
      nis_freeresult (res);
      return 2;
    }

  if (argc == 2 && strchr (argv[0], '=') == NULL)
    {
      /* Quick case: No "=", only one search argument */
      match_t *tmp;
      int result;

      matches = calloc (2, sizeof (match_t));
      if (matches == NULL)
	{
	  fputs (_("Out of memory!\n"), stderr);
	  return 2;
	}
      matches->pattern = argv[0];
      matches->col = 0;
      matches->more = TRUE;
      memset (&matches->reg, 0, sizeof (regex_t));
      result = regcomp (&matches->reg, argv[0], 0);
      if (result)
	{
	  char buf2[1024];
	  regerror (result, &matches->reg, buf2, 1024);
	  fprintf (stderr, _("Can't compile regular expression: %s\n"), buf2);
	  return 2;
	}
      tmp = matches + 1;
      memset (tmp, 0, sizeof (match_t));
    }
  else
    {
      /* Create a list with patterns */
      unsigned long count = 0;
      long i;

      matches = NULL;
      for (i = 0; i < argc - 1; ++i)
	{
	  unsigned long j;

	  cp = strchr (argv[i], '=');
	  if (cp == NULL)
	    {
	      fprintf (stderr, _("%s: badly formatted search pattern\n"),
		       argv[i]);
	      return 2;
	    }

	  for (j = 0; j < NIS_RES_OBJECT (res)->TA_data.ta_cols.ta_cols_len;
	       ++j)
	    {
	      if (!strncasecmp (
	       NIS_RES_OBJECT (res)->TA_data.ta_cols.ta_cols_val[j].tc_name,
				 argv[i], cp - argv[i]))
		{
		  match_t *tmp;
		  int result;

		  if (matches == NULL)
		    matches = calloc (1, (count + 2) * sizeof (match_t));
		  else
		    matches = realloc (matches, (count + 2) *
				       sizeof (match_t));
		  tmp = matches + count;
		  tmp->col = j;
		  tmp->more = TRUE;
		  memset (&tmp->reg, 0, sizeof (regex_t));
		  tmp->pattern = cp + 1;
		  result = regcomp (&tmp->reg, cp + 1, 0);
		  if (result)
		    {
		      char buf2[1024];
		      regerror (result, &tmp->reg, buf2, 1024);
		      fprintf (stderr,
			       _("Can't compile regular expression: %s\n"),
			       buf2);
		      return 2;
		    }
		  ++tmp;
		  memset (tmp, 0, sizeof (match_t));
		  ++count;
		  break;
		}
	    }
	  if (j == NIS_RES_OBJECT (res)->TA_data.ta_cols.ta_cols_len)
	    {
	      *cp = '\0';
	      fprintf (stderr, _("table '%s' has no such column '%s'\n"),
		       argv[argc - 1], argv[i]);
	    }
	}
    }

  if (!new_sep)
    {
      /* Get the separator character */
      separator = res->objects.objects_val->TA_data.ta_sep;
    }
  /* Should we print the header line with the name of the columns ? */
  if (display_header && !print_object && !cnt_only)
    {
      unsigned int i;

      fprintf (stdout, "# ");
      for (i = 0; i < NIS_RES_OBJECT (res)->TA_data.ta_cols.ta_cols_len; ++i)
	{
	  fprintf (stdout, "%s",
		   NIS_RES_OBJECT (res)->TA_data.ta_cols.ta_cols_val[i].tc_name);
	  if (i == NIS_RES_OBJECT (res)->TA_data.ta_cols.ta_cols_len - 1)
	    fputs ("\n", stdout);
	  else
	    fprintf (stdout, "%c", separator);
	}
    }

  buf = alloca (strlen (NIS_RES_OBJECT (res)->zo_name) +
		strlen (NIS_RES_OBJECT (res)->zo_domain) + 10);
  if (buf == NULL)
    {
      fputs (_("Out of memory!\n"), stderr);
      return 2;
    }
  cp = stpcpy (buf, "[],");
  cp = stpcpy (cp, NIS_RES_OBJECT (res)->zo_name);
  *cp++ = '.';
  strcpy (cp, NIS_RES_OBJECT (res)->zo_domain);
  nis_freeresult (res);

  /* Now we try to get the contents of the table */
  res = nis_list (buf, flags, print_obj, (void *) &separator);
  if (res == NULL ||
      (res->status != NIS_SUCCESS && res->status != NIS_CBRESULTS))
    {
      if (res == NULL)
	{
	  fputs (_("Out of memory!\n"), stderr);
	  return 2;
	}
      else
	{
	  if (res->status != NIS_PARTIAL)
	    {
	      nis_perror (res->status, "nisgrep");
	      nis_freeresult (res);
	      return 2;
	    }
	}
    }
  nis_freeresult (res);

  if (cnt_only)
    fprintf (stdout, "%lu\n", cnt_match);

  if (cnt_match)
    return 0;
  else
    return 1;
}
