/*
 *	Interface GTK+ Theme Engine
 *
 *  interface_theme_main.c:
 *		Theme class stuff. rc-file parsing, data and pixmap caching etc.
 *
 *	Copyright  2000 Johan Hanson <johan@tiq.com>.
 *	
 *	This library is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU Library General Public
 *	License as published by the Free Software Foundation; either
 *	version 2 of the License, or (at your option) any later version.
 *	
 *	This library is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *	Library General Public License for more details.
 *	
 *	You should have received a copy of the GNU Library General Public
 *	License along with this library; if not, write to the 
 *	Free Software Foundation, Inc., 59 Temple Place - Suite 330, 
 *	Boston, MA  02111-1307  USA.
 */

#include "Interface_theme.h"
#include <gmodule.h>
#include <gtk/gtkrc.h>

#ifndef DEBUG
#define	DEBUG 0
#endif

/* Theme functions to export */
void theme_init (GtkThemeEngine *engine);
void theme_exit (void);

/* static functions */
static InterfaceThemeData *	interface_data_new	(void);
static void				interface_data_ref	(InterfaceThemeData *data);
static void				interface_data_unref	(InterfaceThemeData *data);
static InterfaceThemeData *	interface_data_copy	(InterfaceThemeData *src);
static InterfaceThemeData *	interface_data_filter(InterfaceThemeData *orig, guint usage);
static void				interface_data_merge	(InterfaceThemeData *dest, InterfaceThemeData *src);

#define interface_mask_ref(i) \
	interface_mask_refcnt[interface_images[i].mask]++;

static void				interface_mask_unref (InterfaceImageType i);

static gint				interface_parse_int	(GScanner *scanner, guint *token);
static gfloat			interface_parse_float (GScanner *scanner, guint *token);
static gboolean			interface_parse_boolean (GScanner *scanner, guint *token);
static InterfaceKnobStyle	interface_parse_knob (GScanner *scanner, guint *token,
										 InterfaceKnobStyle pos, InterfaceKnobStyle neg);
static InterfaceShadowType	interface_parse_shadow (GScanner *scanner, guint *token,
										 InterfaceShadowType pos, InterfaceShadowType neg);

static guint			interface_parse_rc_style (GScanner *scanner, GtkRcStyle *rc_style);
static void				interface_merge_rc_style (GtkRcStyle *dest, GtkRcStyle *src);
static void				interface_rc_style_to_style (GtkStyle *style, GtkRcStyle *rc_style);
static void				interface_destroy_rc_style (GtkRcStyle *rc_style);

static void				interface_duplicate_style (GtkStyle *dest, GtkStyle *src);
static void				interface_realize_style (GtkStyle *style);
static void				interface_unrealize_style (GtkStyle *style);
static void				interface_destroy_style (GtkStyle *style);


/* debugging */
#if DEBUG
gint	debug_pixmap_cnt = 0;
gint	debug_data_cnt = 0;
#endif

/* masks */
GdkBitmap * interface_masks[INTERFACE_MASK_LAST] = { NULL, NULL, NULL, NULL, NULL };
gint		interface_mask_refcnt[INTERFACE_MASK_LAST] = { 0, 0, 0, 0, 0 };

/* parsing */
typedef struct {
	const gchar *name;
	guint		 token;
} InterfaceSymbol;

static InterfaceSymbol interface_symbols[] = {
	{ "true",	INTERFACE_TOKEN_TRUE	},
	{ "false",	INTERFACE_TOKEN_FALSE	},
	{ "yes",	INTERFACE_TOKEN_TRUE	},
	{ "no",		INTERFACE_TOKEN_FALSE	},
	{ "y",		INTERFACE_TOKEN_TRUE	},
	{ "n",		INTERFACE_TOKEN_FALSE	},
	{ "on",		INTERFACE_TOKEN_TRUE	},
	{ "off",	INTERFACE_TOKEN_FALSE	},
	
	{ "stepper_ends",		INTERFACE_TOKEN_STEPPER_ENDS		},
	{ "stepper_arrows",		INTERFACE_TOKEN_STEPPER_ARROWS		},
	{ "stepper_box",		INTERFACE_TOKEN_STEPPER_BOX			},
	{ "scrollbar_width",	INTERFACE_TOKEN_SCROLLBAR_WIDTH		},
	{ "scrollbar_dimple",	INTERFACE_TOKEN_SCROLLBAR_KNOB		},
	{ "scrollbar_knob",		INTERFACE_TOKEN_SCROLLBAR_KNOB		},
	{ "paned_knob",			INTERFACE_TOKEN_PANED_KNOB			},
	{ "handle_knob",		INTERFACE_TOKEN_HANDLE_KNOB			},
	{ "thin",				INTERFACE_TOKEN_THIN				},
	{ "shine",				INTERFACE_TOKEN_SHINE				},
	{ "shade",				INTERFACE_TOKEN_SHADE				},
	{ "buttondef_shadow",	INTERFACE_TOKEN_BUTTONDEF_SHADOW	},
	{ "tooltip_shadow",		INTERFACE_TOKEN_TOOLTIP_SHADOW		},

	{ "none",				INTERFACE_TOKEN_NONE				},
	{ "normal",				INTERFACE_TOKEN_NONE				},
	{ "dimple",				INTERFACE_TOKEN_DIMPLE				},
	{ "lines",				INTERFACE_TOKEN_LINES				},
	{ "holes",				INTERFACE_TOKEN_HOLES				},
	{ "buds",				INTERFACE_TOKEN_BUDS				},

	{ "NORMAL",				INTERFACE_TOKEN_NORMAL				},
	{ "PRELIGHT",			INTERFACE_TOKEN_PRELIGHT			},
	{ "ACTIVE",				INTERFACE_TOKEN_ACTIVE				},
	{ "SELECTED",			INTERFACE_TOKEN_SELECTED			},
	{ "INSENSITIVE",		INTERFACE_TOKEN_INSENSITIVE			},
	
	{ "in",					INTERFACE_TOKEN_SHADOW_IN			},
	{ "out",				INTERFACE_TOKEN_SHADOW_OUT			},
	{ "menu",				INTERFACE_TOKEN_SHADOW_MENU			},
	{ "etched_in",			INTERFACE_TOKEN_SHADOW_ETCHED_IN	},
	{ "etched_out",			INTERFACE_TOKEN_SHADOW_ETCHED_OUT	},
	{ "button_in",			INTERFACE_TOKEN_SHADOW_BUTTON_IN	},
	{ "none_out",			INTERFACE_TOKEN_SHADOW_NONE_OUT		},
	{ "none_in",			INTERFACE_TOKEN_SHADOW_NONE_IN		},
	{ "in_focused",			INTERFACE_TOKEN_SHADOW_IN_FOCUSED	},
	{ "thin_out",			INTERFACE_TOKEN_SHADOW_THIN_OUT		},
	{ "thin_in",			INTERFACE_TOKEN_SHADOW_THIN_IN		},
	{ "note",				INTERFACE_TOKEN_SHADOW_NOTE			}
};

#define INTERFACE_NSYMBOLS	44

/* InterfaceThemeData */
static InterfaceThemeData *
interface_data_new (void) {
	InterfaceThemeData *data;
	gint i;
	
	data = g_malloc(sizeof(InterfaceThemeData));
	if (data) {
		data->ref_count			= 1;
		data->stepper_ends		= FALSE;
		data->stepper_arrows	= FALSE;
		data->stepper_box		= TRUE;
		data->scrollbar_width	= INTERFACE_DEFAULT_SCROLLBAR_WIDTH;
		data->scrollbar_knob	= INTERFACE_KNOB_NONE;
		data->handle_knob		= INTERFACE_KNOB_NONE;
		data->paned_knob		= INTERFACE_KNOB_NONE;
		data->buttondef_shadow	= INTERFACE_SHADOW_IN;
		data->tooltip_shadow	= INTERFACE_SHADOW_NOTE;
		data->thin				= FALSE;
		for (i=0; i<5; i++) {
			data->shine[i]		= (guint16)(INTERFACE_DEFAULT_SHINE * INTERFACE_SHADE_MUL);
			data->shade[i]		= (guint16)(INTERFACE_DEFAULT_SHADE * INTERFACE_SHADE_MUL);
		}
		data->npixmaps			= 0;
		data->dirty				= 0;
		data->config			= 0;
		for (i=0; i<INTERFACE_IMAGE_LAST; i++)
			data->pixmaps[i] = NULL;
	}
  #if DEBUG
	debug_data_cnt++;
	g_print("creating data, data=%d\n", debug_data_cnt);
  #endif
	return data;
}

static void
interface_data_ref (InterfaceThemeData *data) {
	g_return_if_fail (data != NULL);
	data->ref_count++;
}

static void
interface_data_unref (InterfaceThemeData *data) {
	gint i, j;
	
	g_return_if_fail (data != NULL);
	
	data->ref_count--;
	if (data->ref_count <= 0) {
		if (data->npixmaps) {
			for (i=0; i<INTERFACE_IMAGE_LAST; i++) {
				if (data->pixmaps[i]) {
				  #if DEBUG
					debug_pixmap_cnt--;
				  #endif
					gdk_pixmap_unref(data->pixmaps[i]);
					interface_mask_unref(i);
					data->pixmaps[i] = NULL;
				}
			}
		}
	  #if DEBUG
		debug_data_cnt--;
		g_print("killing data, count=%d\n", debug_data_cnt);
	  #endif
		
		g_free(data);
	}
}

static InterfaceThemeData *
interface_data_copy (InterfaceThemeData *src) {
	InterfaceThemeData	*dst;
	gint i;

	g_return_val_if_fail(src != NULL, NULL);

	dst = interface_data_new();
	if (dst) {
		*dst = *src;		/* bitwise copy */
		dst->ref_count = 1;
		
		if (dst->npixmaps) {
			for (i=0; i<INTERFACE_IMAGE_LAST; i++) {
				if (dst->pixmaps[i]) {
					gdk_pixmap_ref(dst->pixmaps[i]);
					interface_mask_ref(i);
				}
			}
		}
	}
	return dst;
}

static InterfaceThemeData *
interface_data_filter (InterfaceThemeData * orig, guint usage) {
	InterfaceThemeData	*data;
	gint			i;
  
  #if DEBUG
	g_print("%s\n", __PRETTY_FUNCTION__);
  #endif
	
	if (!usage || !orig)
		return orig;

	data = orig;
	if (orig->npixmaps) {
		if (data->ref_count != 1) {
			data = interface_data_copy (orig);
			interface_data_unref (orig);
		}
	
		for (i=0; i<INTERFACE_IMAGE_LAST; i++) {
			if (data->pixmaps[i] != NULL) {
				if (interface_images[i].color_usage & usage) {
				  #if DEBUG
					debug_pixmap_cnt--;
				  #endif
					gdk_pixmap_unref (data->pixmaps[i]);
					interface_mask_unref (i);
					data->pixmaps[i] = NULL;
					data->npixmaps--;
				}
			}
		}
		data->dirty = 0;
	} else {
		data->dirty = usage;
	}
	return data;
}

static void
interface_data_merge (InterfaceThemeData *dest, InterfaceThemeData *src) {
	gint i;
	guint diff = src->config & ~dest->config;

	if (diff & INTERFACE_DATA_STEPPER_ENDS)
		dest->stepper_ends = src->stepper_ends;
	if (diff & INTERFACE_DATA_STEPPER_ARROWS)
		dest->stepper_arrows = src->stepper_arrows;
	if (diff & INTERFACE_DATA_STEPPER_BOX)
		dest->stepper_box = src->stepper_box;
	if (diff & INTERFACE_DATA_SCROLLBAR_WIDTH)
		dest->scrollbar_width = src->scrollbar_width;
	if (diff & INTERFACE_DATA_SCROLLBAR_KNOB)
		dest->scrollbar_knob = src->scrollbar_knob;
	if (diff & INTERFACE_DATA_PANED_KNOB)
		dest->paned_knob = src->paned_knob;
	if (diff & INTERFACE_DATA_HANDLE_KNOB)
		dest->handle_knob = src->handle_knob;
	if (diff & INTERFACE_DATA_BUTTONDEF_SHADOW)
		dest->buttondef_shadow = src->buttondef_shadow;
	if (diff & INTERFACE_DATA_TOOLTIP_SHADOW)
		dest->tooltip_shadow = src->tooltip_shadow;
	if (diff & INTERFACE_DATA_THIN)
		dest->thin = src->thin;
	for (i=0; i<5; i++) {
		if (diff & (INTERFACE_DATA_SHADE<<i))
			dest->shade[i] = src->shade[i];
		if (diff & (INTERFACE_DATA_SHINE<<i))
			dest->shine[i] = src->shine[i];
	}

	dest->config |= diff;
}

/*** Images ***/

/*	Global array of image masks. Only one per original XPM.
	I put the reference counts here so that I will be able to know when the bitmap is destroyed.
*/
static void
interface_mask_unref (InterfaceImageType i) {
	gint j;

	j = interface_images[i].mask;
	interface_mask_refcnt[j]--;
	if (interface_mask_refcnt[j] == 0 && interface_masks[j]) {
		gdk_bitmap_unref (interface_masks[j]);
		interface_masks[j] = NULL;
	}
}

GdkPixmap * interface_pixmap_get (GdkWindow *window, GtkStyle *style,
							 const GtkStyle *style2, InterfaceImageType image_type)
{
	InterfaceThemeData	*data, *tmp;
	InterfaceImage		*image;
	GdkPixmap		*pixmap;
	GdkBitmap		**bitmap_ptr;
	char			**xpm;
	guint			usage;

	g_return_val_if_fail(window != NULL, NULL);
	g_return_val_if_fail(style  != NULL, NULL);
	g_return_val_if_fail(style2 != NULL, NULL);
	g_return_val_if_fail(style->engine_data != NULL, NULL);
	
	data = style->engine_data;

	if ((pixmap = data->pixmaps[image_type])) {
		return pixmap;
	} else {
		image	= &(interface_images[image_type]);
		xpm		= interface_xpm_copy(image->xpm);
		interface_xpm_remap (style, style2, image->state_type, image->remap, xpm);
		if (interface_mask_get(image_type)) {
			bitmap_ptr = NULL;
		} else {
			bitmap_ptr = &(interface_masks[interface_images[image_type].mask]);
		}
		pixmap	= gdk_pixmap_create_from_xpm_d (window, bitmap_ptr, NULL, xpm);
		
		if (pixmap) {
			interface_mask_ref(image_type);
		  #if DEBUG
			debug_pixmap_cnt++;
			g_print("create pixmap    pm=%d\n", debug_pixmap_cnt);
		  #endif
		}

		usage = interface_images[image_type].color_usage;
		if (data->dirty & usage) {
			tmp = data;
			data = interface_data_copy(tmp);
			interface_data_unref(tmp);
			style->engine_data = data;
			data->dirty = 0;
		}
		
		data->pixmaps[image_type] = pixmap;
		data->npixmaps++;
		interface_xpm_free(xpm);
	}
	
	return pixmap;
}


/* RC Style */
static gint
interface_parse_int (GScanner *scanner, guint *token) {
	*token = g_scanner_peek_next_token(scanner);
	if (*token != G_TOKEN_EQUAL_SIGN)
		return 0;
	g_scanner_get_next_token(scanner);
	if ((*token = g_scanner_get_next_token(scanner)) != G_TOKEN_INT)
		return 0;
	*token = G_TOKEN_NONE;
	return scanner->value.v_int;
}

static gfloat
interface_parse_float (GScanner *scanner, guint *token) {
	*token = g_scanner_peek_next_token(scanner);
	if (*token != G_TOKEN_EQUAL_SIGN)
		return 0;
	g_scanner_get_next_token(scanner);
	if ((*token = g_scanner_get_next_token(scanner)) != G_TOKEN_FLOAT)
		return 0;
	*token = G_TOKEN_NONE;
	return scanner->value.v_float;
}

static gboolean
interface_parse_boolean (GScanner *scanner, guint *token) {
	gboolean	ret;
	
	*token = g_scanner_peek_next_token(scanner);
	if (*token != G_TOKEN_EQUAL_SIGN)
		return FALSE;
	g_scanner_get_next_token(scanner);

	*token = g_scanner_get_next_token(scanner);
	ret = (*token==INTERFACE_TOKEN_TRUE) ? TRUE : FALSE;
	
	if (*token == INTERFACE_TOKEN_TRUE || *token == INTERFACE_TOKEN_FALSE)
		*token = G_TOKEN_NONE;
	return ret;
}

static InterfaceKnobStyle
interface_parse_knob (GScanner *scanner, guint *token, InterfaceKnobStyle pos, InterfaceKnobStyle neg) {
	guint	t;
	
	*token = g_scanner_peek_next_token(scanner);
	if (*token != G_TOKEN_EQUAL_SIGN)
		return neg;
	g_scanner_get_next_token(scanner);
	*token = G_TOKEN_NONE;
	switch (g_scanner_get_next_token(scanner)) {
	  case INTERFACE_TOKEN_TRUE:	return pos;
	  case INTERFACE_TOKEN_FALSE: return neg;
	  case INTERFACE_TOKEN_DIMPLE: return INTERFACE_KNOB_DIMPLE;
	  case INTERFACE_TOKEN_LINES: return INTERFACE_KNOB_LINES;
	  case INTERFACE_TOKEN_HOLES: return INTERFACE_KNOB_HOLES;
	  case INTERFACE_TOKEN_BUDS: return INTERFACE_KNOB_BUDS;
	}
	return neg;
}

static InterfaceShadowType
interface_parse_shadow(GScanner *scanner, guint *token, InterfaceShadowType pos, InterfaceShadowType neg) {
	guint	t;
	
	*token = g_scanner_peek_next_token(scanner);
	if (*token != G_TOKEN_EQUAL_SIGN)
		return neg;
	g_scanner_get_next_token(scanner);
	*token = G_TOKEN_NONE;
	switch (g_scanner_get_next_token(scanner)) {
	  case INTERFACE_TOKEN_TRUE:				return pos;
	  case INTERFACE_TOKEN_FALSE:				return neg;
	  case INTERFACE_TOKEN_NORMAL:				return INTERFACE_SHADOW_NONE;
	  case INTERFACE_TOKEN_NONE: 				return INTERFACE_SHADOW_NONE;
	  case INTERFACE_TOKEN_SHADOW_IN:			return INTERFACE_SHADOW_IN;
	  case INTERFACE_TOKEN_SHADOW_OUT:			return INTERFACE_SHADOW_OUT;
	  case INTERFACE_TOKEN_SHADOW_MENU:			return INTERFACE_SHADOW_MENU;
	  case INTERFACE_TOKEN_SHADOW_ETCHED_IN:	return INTERFACE_SHADOW_ETCHED_IN;
	  case INTERFACE_TOKEN_SHADOW_ETCHED_OUT:	return INTERFACE_SHADOW_ETCHED_OUT;
	  case INTERFACE_TOKEN_SHADOW_BUTTON_IN:	return INTERFACE_SHADOW_BUTTON_IN;
	  case INTERFACE_TOKEN_SHADOW_NONE_OUT:		return INTERFACE_SHADOW_NONE_OUT;
	  case INTERFACE_TOKEN_SHADOW_NONE_IN:		return INTERFACE_SHADOW_NONE_IN;
	  case INTERFACE_TOKEN_SHADOW_IN_FOCUSED:	return INTERFACE_SHADOW_IN_FOCUSED;
	  case INTERFACE_TOKEN_SHADOW_THIN_OUT:		return INTERFACE_SHADOW_THIN_OUT;
	  case INTERFACE_TOKEN_SHADOW_THIN_IN:		return INTERFACE_SHADOW_THIN_IN;
	  case INTERFACE_TOKEN_SHADOW_NOTE:			return INTERFACE_SHADOW_NOTE;
	}
	return neg;
}

static GtkStateType
interface_parse_state_type(GScanner *scanner, guint *token) {
	GtkStateType state_type;
	
	g_scanner_get_next_token(scanner); /* [ */
	*token = g_scanner_peek_next_token(scanner);
	switch (*token) {
	  case INTERFACE_TOKEN_NORMAL: state_type = GTK_STATE_NORMAL; break;
	  case INTERFACE_TOKEN_ACTIVE: state_type = GTK_STATE_ACTIVE; break;
	  case INTERFACE_TOKEN_PRELIGHT: state_type = GTK_STATE_PRELIGHT; break;
	  case INTERFACE_TOKEN_SELECTED: state_type = GTK_STATE_SELECTED; break;
	  case INTERFACE_TOKEN_INSENSITIVE: state_type = GTK_STATE_INSENSITIVE; break;
	  default:
		return GTK_STATE_NORMAL;
	}
	g_scanner_get_next_token(scanner); /* state */
	g_scanner_get_next_token(scanner); /* ] */
	
	*token = G_TOKEN_NONE;
	return state_type;
}

static guint
interface_parse_rc_style (GScanner *scanner, GtkRcStyle *rc_style) {
	static GQuark	scope = 0;
	InterfaceThemeData	*data = NULL;
	guint			token, old_scope, i;
	gint			t;

  #if DEBUG
	g_print("%s\n", __PRETTY_FUNCTION__);
  #endif

	if (!scope)
		scope = g_quark_from_string("theme_engine");
	old_scope = g_scanner_set_scope(scanner, scope);

	/* add keywords */
	if (!g_scanner_lookup_symbol(scanner, interface_symbols[0].name)) {
		g_scanner_freeze_symbol_table(scanner);
		for (i=0; i<INTERFACE_NSYMBOLS; i++)
			g_scanner_scope_add_symbol (scanner, scope, interface_symbols[i].name,
										GUINT_TO_POINTER(interface_symbols[i].token));
		g_scanner_thaw_symbol_table(scanner);
	}
	
	/* create default data */
	data = interface_data_new ();
	
	/* parse */
	token = g_scanner_get_next_token(scanner);
	while (token != G_TOKEN_RIGHT_CURLY) {
		switch (token) {
		  case INTERFACE_TOKEN_STEPPER_ENDS:
			data->stepper_ends = interface_parse_boolean(scanner, &token);
			data->config |= INTERFACE_DATA_STEPPER_ENDS;
			break;
		  
		  case INTERFACE_TOKEN_STEPPER_ARROWS:
			data->stepper_arrows = interface_parse_boolean(scanner, &token);
			data->config |= INTERFACE_DATA_STEPPER_ARROWS;
			if (!(data->config & INTERFACE_DATA_STEPPER_BOX))
				data->stepper_box = !(data->stepper_arrows);
			break;

		  case INTERFACE_TOKEN_STEPPER_BOX:
			data->stepper_box = interface_parse_boolean(scanner, &token);
			data->config |= INTERFACE_DATA_STEPPER_BOX;
			break;

		  case INTERFACE_TOKEN_SCROLLBAR_WIDTH:
			t = interface_parse_int (scanner, &token);
			data->scrollbar_width = CLAMP(t, 4, 31);
			data->config |= INTERFACE_DATA_SCROLLBAR_WIDTH;
			break;

		  case INTERFACE_TOKEN_SCROLLBAR_KNOB:
			data->scrollbar_knob = interface_parse_knob(scanner, &token, INTERFACE_KNOB_DIMPLE, INTERFACE_KNOB_NONE);
			data->config |= INTERFACE_DATA_SCROLLBAR_KNOB;
			break;

		  case INTERFACE_TOKEN_PANED_KNOB:
			data->paned_knob = interface_parse_knob(scanner, &token, INTERFACE_KNOB_LINES, INTERFACE_KNOB_NONE);
			data->config |= INTERFACE_DATA_PANED_KNOB;
			break;

		  case INTERFACE_TOKEN_HANDLE_KNOB:
			data->handle_knob = interface_parse_knob(scanner, &token, INTERFACE_KNOB_BUDS, INTERFACE_KNOB_NONE);
			data->config |= INTERFACE_DATA_HANDLE_KNOB;
			break;

		  case INTERFACE_TOKEN_BUTTONDEF_SHADOW:
			data->buttondef_shadow = interface_parse_shadow(scanner, &token, INTERFACE_SHADOW_IN, INTERFACE_SHADOW_NONE);
			data->config |= INTERFACE_DATA_BUTTONDEF_SHADOW;
			break;

		  case INTERFACE_TOKEN_TOOLTIP_SHADOW:
			data->tooltip_shadow = interface_parse_shadow(scanner, &token, INTERFACE_SHADOW_NOTE, INTERFACE_SHADOW_NONE);
			data->config |= INTERFACE_DATA_TOOLTIP_SHADOW;
			break;

		  case INTERFACE_TOKEN_SHINE:
			token = g_scanner_peek_next_token(scanner);
			if (token == G_TOKEN_LEFT_BRACE) {
				i = interface_parse_state_type(scanner, &token);
				if (token == G_TOKEN_NONE) {
					t = interface_parse_float(scanner, &token) * INTERFACE_SHADE_MUL;
					data->shine[i] = CLAMP(t, 0, INTERFACE_SHADE_MUL * 2);
					data->config |= (INTERFACE_DATA_SHINE << i);
				}
			} else {
				t = interface_parse_float(scanner, &token) * INTERFACE_SHADE_MUL;
				t = CLAMP(t, 0, INTERFACE_SHADE_MUL * 2);
				for (i=0; i<5; i++) {
					data->shine[i] = t;
					data->config |= (INTERFACE_DATA_SHINE << i);
				}
			}
			break;

		  case INTERFACE_TOKEN_SHADE:
			token = g_scanner_peek_next_token(scanner);
			if (token == G_TOKEN_LEFT_BRACE) {
				i = interface_parse_state_type(scanner, &token);
				if (token == G_TOKEN_NONE) {
					t = interface_parse_float(scanner, &token) * INTERFACE_SHADE_MUL;
					data->shade[i] = CLAMP(t, 0, INTERFACE_SHADE_MUL);
					data->config |= INTERFACE_DATA_SHADE << i;
				}
			} else {
				t = interface_parse_float(scanner, &token) * INTERFACE_SHADE_MUL;
				t = CLAMP(t, 0, INTERFACE_SHADE_MUL);
				for (i=0; i<5; i++) {
					data->shade[i] = t;
					data->config |= INTERFACE_DATA_SHADE << i;
				}
			}
			break;

		  case INTERFACE_TOKEN_THIN:
			data->thin = interface_parse_boolean(scanner, &token);
			data->config |= INTERFACE_DATA_THIN;
			break;

		  default:
			token = G_TOKEN_RIGHT_CURLY;
			break;
		}
		
		if (token != G_TOKEN_NONE && token != G_TOKEN_RIGHT_CURLY) {
			if (data)
				g_free(data);
			g_print("rc style error\n");
			return token;
		}
		
		token = g_scanner_get_next_token (scanner);
	}
	g_scanner_set_scope(scanner, old_scope);

	/* set up */
	rc_style->engine_data = data;

	return G_TOKEN_NONE;
}


static void
interface_merge_rc_style (GtkRcStyle *dest, GtkRcStyle *src) {
	InterfaceThemeData	*data;
	guint			usage, i;

  #if DEBUG
	g_print("%s\n", __PRETTY_FUNCTION__);
  #endif

	if (src->engine_data) {
		if (dest->engine_data == NULL) {
			data = src->engine_data;
			
			interface_data_ref(data);
			usage = 0;	
			for (i=0; i<5; i++) {
				if (!interface_color_equal (&src->bg[i], &dest->bg[i]))
					usage |= INTERFACE_USE(BG, i);
				if (!interface_color_equal (&src->fg[i], &dest->fg[i]))
					usage |= INTERFACE_USE(FG, i);
				if (!interface_color_equal (&src->text[i], &dest->text[i]))
					usage |= INTERFACE_USE(TEXT, i);
				if (!interface_color_equal (&src->base[i], &dest->base[i]))
					usage |= INTERFACE_USE(BASE, i);
			}
			if (usage)
				data = interface_data_filter (data, usage);
			
			dest->engine_data = data;
		} else {
			interface_data_merge (dest->engine_data, src->engine_data);
		}
	}
}


static void
interface_rc_style_to_style (GtkStyle *style, GtkRcStyle *rc_style) {
  #if DEBUG
	g_print("%s\n", __PRETTY_FUNCTION__);
  #endif
	
	if (style->engine_data && (style->klass==&interface_class || style->klass==&interface_thin_class))
		interface_data_unref(style->engine_data);
	style->klass = &interface_class;
	
	if ((style->engine_data = rc_style->engine_data)) {
		interface_data_ref (style->engine_data);
		if (((InterfaceThemeData *)style->engine_data)->thin)
			style->klass = &interface_thin_class;
	}
}

static void
interface_duplicate_style (GtkStyle *dest, GtkStyle *src) {
  #if DEBUG
	g_print("%s\n", __PRETTY_FUNCTION__);
  #endif
	if ((dest->engine_data = src->engine_data))
		interface_data_ref (dest->engine_data);
}

static void
interface_realize_style (GtkStyle *style) {
	GdkGCValuesMask	gc_values_mask = GDK_GC_FOREGROUND | GDK_GC_FONT;
	GdkGCValues		gc_values;
	InterfaceThemeData	*data;
	GtkRcStyle		*rc_style;
	gint			i, usage;
	extern GdkFont	*default_font;
	gdouble			shine[5], shade[5];

  #if DEBUG
	g_print("%s\n", __PRETTY_FUNCTION__);
  #endif
	
	if (!default_font)
		default_font = gdk_font_load ("-adobe-helvetica-medium-r-normal--*-120-*-*-*-*-*-*");

	if (style->font->type == GDK_FONT_FONT)
		gc_values.font = style->font;
	else if (style->font->type == GDK_FONT_FONTSET)
		gc_values.font = default_font;

	data = style->engine_data;
	rc_style = style->rc_style;
	if (data) {
		for (i=0; i<5; i++) {
			if (!interface_color_equal (&style->bg[i], &rc_style->bg[i]))
				usage |= INTERFACE_USE(BG, i);
			if (!interface_color_equal (&style->fg[i], &rc_style->fg[i]))
				usage |= INTERFACE_USE(FG, i);
			if (!interface_color_equal (&style->text[i], &rc_style->text[i]))
				usage |= INTERFACE_USE(TEXT, i);
			if (!interface_color_equal (&style->base[i], &rc_style->base[i]))
				usage |= INTERFACE_USE(BASE, i);
			
			shine[i] = data->shine[i] / INTERFACE_SHADE_MUL;
			shade[i] = data->shade[i] / INTERFACE_SHADE_MUL;
		}
		if (usage) {
			data = interface_data_filter (data, usage);
			style->engine_data = data;
		}
	} else {
		for (i=0; i<5; i++) {
			shine[i] = INTERFACE_DEFAULT_SHINE;
			shade[i] = INTERFACE_DEFAULT_SHADE;
		}
	}	

	for (i=0; i<5; i++) {
		/* Release old GC */
		gtk_gc_release (style->light_gc [i]);
		gtk_gc_release (style->dark_gc [i]);
		
		/* Compute new colors */
		interface_shade_color (&style->bg[i], &style->light[i], shine[i]);
		interface_shade_color (&style->bg[i], &style->dark[i],  shade[i]);
		if (i == GTK_STATE_INSENSITIVE && data && data->thin) {
			interface_blend_color (&style->light[GTK_STATE_INSENSITIVE], &style->bg[GTK_STATE_NORMAL],
							  0.5, &style->light[GTK_STATE_INSENSITIVE]);
			interface_blend_color (&style->dark[GTK_STATE_INSENSITIVE], &style->bg[GTK_STATE_NORMAL],
							  0.5, &style->dark[GTK_STATE_INSENSITIVE]);
		}
		interface_blend_color (&style->light[i], &style->dark[i], 0.5, &style->mid[i]);
		
		/* Allocate new gc */
		if (!gdk_color_alloc (style->colormap, &style->light [i]))
			g_warning ("unable to allocate color #%02x%02x%02x\n",
					   style->light[i].red>>8, style->light[i].green>>8,
					   style->light[i].blue>>8);
		if (!gdk_color_alloc (style->colormap, &style->dark [i]))
			g_warning ("unable to allocate color #%02x%02x%02x\n",
					   style->dark[i].red>>8, style->dark[i].green>>8,
					   style->dark[i].blue>>8);
		if (!gdk_color_alloc (style->colormap, &style->mid [i]))
			g_warning ("unable to allocate color #%02x%02x%02x\n",
					   style->mid[i].red>>8, style->mid[i].green>>8,
					   style->mid[i].blue>>8);
		
		gc_values.foreground = style->light [i];
		style->light_gc [i] = gtk_gc_get (style->depth, style->colormap,
										  &gc_values, gc_values_mask);
		
		gc_values.foreground = style->dark[i];
		style->dark_gc [i] = gtk_gc_get (style->depth, style->colormap,
										  &gc_values, gc_values_mask);
		
		gc_values.foreground = style->mid [i];
		style->mid_gc [i] = gtk_gc_get (style->depth, style->colormap,
										  &gc_values, gc_values_mask);
	}
}

static void
interface_unrealize_style (GtkStyle *style) {
  #if DEBUG
	g_print("%s\n", __PRETTY_FUNCTION__);
  #endif
}

static void
interface_destroy_rc_style (GtkRcStyle *rc_style) {
  #if DEBUG
	g_print("%s\n", __PRETTY_FUNCTION__);
  #endif
	if (rc_style->engine_data)
		interface_data_unref (rc_style->engine_data);
}

static void
interface_destroy_style (GtkStyle *style) {
  #if DEBUG
	g_print("%s\n", __PRETTY_FUNCTION__);
  #endif
	if (style->engine_data)
		interface_data_unref (style->engine_data);
}


/*
	Init and Exit
*/

void theme_init (GtkThemeEngine *engine) {
	engine->parse_rc_style    = interface_parse_rc_style;
	engine->merge_rc_style    = interface_merge_rc_style;
	engine->rc_style_to_style = interface_rc_style_to_style;
	engine->duplicate_style   = interface_duplicate_style;
	engine->realize_style     = interface_realize_style;
	engine->unrealize_style   = interface_unrealize_style;
	engine->destroy_rc_style  = interface_destroy_rc_style;
	engine->destroy_style     = interface_destroy_style;
	engine->set_background    = NULL;

	interface_patches_install();

  #if DEBUG
	g_print("theme_init()\n");
  #endif
}

void theme_exit (void)
{
	interface_patches_remove();

  #if DEBUG
	g_print("theme_exit()\n");
	g_print("pm = %d\n", debug_pixmap_cnt);
  #endif
}


/*
 * The following function will be called by GTK+ when the module
 * is loaded and checks to see if we are compatible with the
 * version of GTK+ that loads us.
 */
G_MODULE_EXPORT const gchar* g_module_check_init (GModule *module);

const gchar* g_module_check_init (GModule *module) {
	return gtk_check_version (GTK_MAJOR_VERSION,
							  GTK_MINOR_VERSION,
							  GTK_MICRO_VERSION - GTK_INTERFACE_AGE);
}


/* end */

