/* X-Chat
 * Copyright (C) 1998 Peter Zelezny.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include "xchat.h"
#include "menu.h"
#include <fcntl.h>
#include <sys/utsname.h>
#include "cfgfiles.h"
#include "gtkutil.h"


extern struct session *current_tab;
extern struct xchatprefs prefs;
extern GdkColor colors[];

extern GdkFont *my_font_load (char *fontname);
extern int tcp_send_len (struct server *serv, char *buf, int len);
extern int tcp_send (struct server *serv, char *buf);
extern void fkeys_edit (void);
extern void ctcp_gui_load (GtkWidget * list);
extern void ctcp_gui_save (GtkWidget * igad);
extern void editlist_gui_open (GSList * list, char *, char *, char *, char *help);
extern void ctcp_gui_open (void);
extern void notify_opengui (void);
extern void chanlist_opengui (struct server *serv);
extern void url_opengui (GtkWidget *, gpointer);
extern void cmd_help (struct session *sess, char *tbuf, char *word[], char *word_eol[]);
extern void xchat_cleanup (GtkWidget * wid, gpointer sess);
extern struct server *new_server (void);
extern struct session *new_session (struct server *, char *from);
extern void PrintText (struct session *, char *);
extern void open_rawlog (struct server *);
extern void open_dcc_recv_window (GtkWidget * wid, gpointer sess);
extern void open_dcc_send_window (GtkWidget * wid, gpointer sess);
extern void open_dcc_chat_window (GtkWidget * wid, gpointer sess);
extern void open_server_list (GtkWidget * wid, gpointer sess);
extern void settings_opengui (struct session *sess);
extern void show_lastlog_window (void);
extern void search_lastlog (struct session *_sess, char *search);
extern char *find_selected_nick (struct session *sess);
extern int handle_command (char *cmd, struct session *sess, int, int);
extern void zvt_clear (GtkWidget * wid);
extern int cmd_unloadall (struct session *sess, char *tbuf, char *word[], char *word_eol[]);
extern int module_unload (char *name, struct session *sess);
extern void module_glist (struct session *sess);
extern void pevent_dialog_show ();

struct session *menu_sess = 0;
GSList *popup_list = 0;
GSList *button_list = 0;
GSList *command_list = 0;
GSList *ctcp_list = 0;
GSList *replace_list;


/* This structure will hold the pointers to the previus "menu levels" so that we can go back */
struct submenu_list
{
   struct submenu_list *next;
   GtkWidget *menu;
};
/* Pointer to first element */
struct submenu_list *first;


void
popup_cmd (GtkWidget * igad, char *cmd)
{
   int fin = 0, j = 0;
   char *buf;
   char *nick;

   if (menu_sess->is_dialog)
      nick = menu_sess->channel;
   else
      nick = find_selected_nick (menu_sess);
   if (!nick)
      nick = "";

   buf = malloc (strlen (nick) + strlen (cmd) + 256);

   while (!fin)
   {
      switch (*cmd)
      {
      case 0:
         fin = 1;
         buf[j] = 0;
         break;
      case '%':
         cmd++;
         switch (*cmd)
         {
         case 's':
            buf[j] = 0;
            strcat (buf, nick);
            j = strlen (buf);
            break;
         default:
            cmd--;
            buf[j] = *cmd;
            j++;
         }
         break;
      default:
         buf[j] = *cmd;
         j++;
      }
      cmd++;
   }

   handle_command (buf, menu_sess, FALSE, FALSE);
   free (buf);
}

void
menu_quick_item (char *cmd, char *label, GtkWidget * menu, int flags)
{
   GtkWidget *item;
   if (!label)
      item = gtk_menu_item_new ();
   else
      item = gtk_menu_item_new_with_label (label);
   gtk_menu_append (GTK_MENU (menu), item);
   if (cmd)
      gtk_signal_connect (GTK_OBJECT (item), "activate",
                          GTK_SIGNAL_FUNC (popup_cmd), cmd);
   if (flags & (1 << 0))
      gtk_widget_set_sensitive (GTK_WIDGET (item), FALSE);
   gtk_widget_show (item);
}

GtkWidget *
menu_quick_sub (char *name, GtkWidget * menu)
{
   struct submenu_list *temp;
   GtkWidget *sub_menu;
   GtkWidget *sub_item;

   if (!name)
      return menu;

   /* Code to add a submenu */
   sub_menu = gtk_menu_new ();

   sub_item = gtk_menu_item_new_with_label (name);
   gtk_menu_append (GTK_MENU (menu), sub_item);
   gtk_widget_show (sub_item);

   gtk_menu_item_set_submenu (GTK_MENU_ITEM (sub_item), sub_menu);

   /* We create a new element in our linked list */
   temp = malloc (sizeof (struct submenu_list));
   temp->menu = sub_menu;
   temp->next = first;
   first = temp;

   return sub_menu;
}

GtkWidget *
menu_quick_endsub ()
{
   /* Just delete the first element in the linked list pointed to by first */
   struct submenu_list *temp;
   temp = first;
   first = first->next;
   free (temp);
   temp = NULL;
   return (first->menu);
}

void
menu_popup (struct session *sess, GdkEventButton * event, char *nick)
{

   struct submenu_list *temp;
   GtkWidget *menu;
   GtkWidget *tempmenu;
   GSList *list = popup_list;
   struct popup *pop;

   /* Create the first element in out linked list, the main popup menu */
   temp = malloc (sizeof (struct submenu_list));

   menu = gtk_menu_new ();

   /* And set the values in the first element */
   tempmenu = menu;
   temp->menu = menu;
   temp->next = NULL;
   first = temp;

   menu_quick_item (0, nick, menu, 1);
   menu_quick_item (0, 0, menu, 1);

   while (list)
   {
      pop = (struct popup *) list->data;
      if (!strncasecmp (pop->name, "SUB", 3))
         tempmenu = menu_quick_sub (pop->cmd, tempmenu);
      else if (!strncasecmp (pop->name, "ENDSUB", 6))
      {
         if (tempmenu != menu)
            tempmenu = menu_quick_endsub ();
         /* If we get here and tempmenu equals menu that means we havent got any submenus to exit from */
      } else if (!strncasecmp (pop->name, "SEP", 3))
         menu_quick_item (0, 0, tempmenu, 1);
      else
         menu_quick_item (pop->cmd, pop->name, tempmenu, 0);

      list = list->next;

   }

   gtk_menu_popup (GTK_MENU (menu), NULL, NULL, NULL, NULL,
          event->button, event->time);

   /* Let's clean up the linked list from mem */
   while (first != NULL)
   {
      temp = first;
      first = first->next;
      free (temp);
   }
   temp = NULL;
}

void
menu_open_server_list (GtkWidget * wid, gpointer none)
{
   open_server_list (0, menu_sess);
}

void
menu_lastlog (GtkWidget * wid, gpointer none)
{
   show_lastlog_window ();
   search_lastlog (menu_sess, NULL);
}

void
menu_settings (GtkWidget * wid, gpointer sess)
{
   settings_opengui (menu_sess);
}

void
menu_newserver_window (GtkWidget * wid, gpointer sess)
{
   int old = prefs.tabchannels;
   struct server *serv;

   prefs.tabchannels = 0;
   serv = new_server ();
   if (serv)
      new_session (serv, 0);
   prefs.tabchannels = old;
}

void
menu_newchannel_window (GtkWidget * wid, gpointer sess)
{
   int old = prefs.tabchannels;

   prefs.tabchannels = 0;
   new_session (menu_sess->server, 0);
   prefs.tabchannels = old;
}

void
menu_newserver_tab (GtkWidget * wid, gpointer sess)
{
   int old = prefs.tabchannels;
   struct server *serv;

   prefs.tabchannels = 1;
   serv = new_server ();
   if (serv)
      new_session (serv, 0);
   prefs.tabchannels = old;
}

void
menu_newchannel_tab (GtkWidget * wid, gpointer sess)
{
   int old = prefs.tabchannels;

   prefs.tabchannels = 1;
   new_session (menu_sess->server, 0);
   prefs.tabchannels = old;
}

void
menu_rawlog (GtkWidget * wid, gpointer sess)
{
   open_rawlog (menu_sess->server);
}

void
menu_autorejoin (GtkWidget * wid, gpointer sess)
{
   prefs.autorejoin = !prefs.autorejoin;
}

void
menu_autoreconnect (GtkWidget * wid, gpointer sess)
{
   prefs.autoreconnect = !prefs.autoreconnect;
}

void
menu_autodialog (GtkWidget * wid, gpointer sess)
{
   prefs.autodialog = !prefs.autodialog;
}

void
menu_autodccchat (GtkWidget * wid, gpointer sess)
{
   prefs.autodccchat = !prefs.autodccchat;
}

void
menu_autodccsend (GtkWidget * wid, gpointer sess)
{
   prefs.autodccsend = !prefs.autodccsend;
   if (prefs.autodccsend)
   {
      if (!strcasecmp (g_get_home_dir (), prefs.dccdir))
      {
         gtkutil_simpledialog ("*WARNING*\n"
                               "Auto accepting DCC to your home directory\n"
                               "can be dangerous and is exploitable. Eg:\n"
                               "Someone could send you a .bash_profile");
      }
   }
}

void
menu_saveexit (GtkWidget * wid, gpointer sess)
{
   prefs.autosave = !prefs.autosave;
}

void
menu_close (GtkWidget * wid, gpointer sess)
{
   gtk_widget_destroy (menu_sess->window);
}

void
menu_flushbuffer (GtkWidget * wid, gpointer sess)
{
   if (!menu_sess->zvt)
      gtk_text_backward_delete ((GtkText *) menu_sess->textgad,
                                gtk_text_get_length ((GtkText *) ((struct session *) menu_sess)->textgad));
#ifdef USE_GNOME
   else
      zvt_clear (menu_sess->textgad);
#endif
}

GtkWidget *freq = 0;

void
close_savebuffer (void)
{
   gtk_widget_destroy (freq);
   freq = 0;
}

void
savebuffer_req_done (GtkWidget * wid, struct session *sess)
{
   int fh;
   char file[128];
   strcpy (file, gtk_file_selection_get_filename (GTK_FILE_SELECTION (freq)));
   close_savebuffer ();

   fh = open (file, O_TRUNC | O_WRONLY | O_CREAT, 0600);
   if (fh != -1)
   {
      char *buf = gtk_editable_get_chars ((GtkEditable *) sess->textgad, 0, -1);
      write (fh, buf, strlen (buf));
      g_free (buf);
      close (fh);
   } else
      gtkutil_simpledialog ("Cannot write to that file.");
}

void
menu_savebuffer (GtkWidget * wid, gpointer sess)
{
   if (menu_sess->zvt)
   {
      gtkutil_simpledialog ("That feature doesn't work for zvt.");
      return;
   }
   if (freq)
      close_savebuffer ();
   freq = gtk_file_selection_new ("Select an output filename");
   gtk_signal_connect_object (GTK_OBJECT (GTK_FILE_SELECTION (freq)->cancel_button),
                              "clicked", (GtkSignalFunc) close_savebuffer, 0);
   gtk_signal_connect (GTK_OBJECT (GTK_FILE_SELECTION (freq)->ok_button),
                       "clicked", (GtkSignalFunc) savebuffer_req_done, menu_sess);
   gtk_widget_show (freq);
}

void
menu_wallops (GtkWidget * wid, gpointer sess)
{
   char tbuf[128];
   prefs.wallops = !prefs.wallops;
   if (((struct session *) menu_sess)->server->connected)
   {
      if (prefs.wallops)
         sprintf (tbuf, "MODE %s +w\r\n", ((struct session *) menu_sess)->server->nick);
      else
         sprintf (tbuf, "MODE %s -w\r\n", ((struct session *) menu_sess)->server->nick);
      tcp_send (((struct session *) menu_sess)->server, tbuf);
   }
}

void
menu_servernotice (GtkWidget * wid, gpointer sess)
{
   char tbuf[128];
   prefs.servernotice = !prefs.servernotice;
   if (((struct session *) menu_sess)->server->connected)
   {
      if (prefs.servernotice)
         sprintf (tbuf, "MODE %s +s\r\n", ((struct session *) menu_sess)->server->nick);
      else
         sprintf (tbuf, "MODE %s -s\r\n", ((struct session *) menu_sess)->server->nick);
      tcp_send (((struct session *) menu_sess)->server, tbuf);
   }
}

void
menu_away (GtkWidget * wid, gpointer sess)
{
   char tbuf[128];
   prefs.away = !prefs.away;
   if (((struct session *) menu_sess)->server->connected)
   {
      if (prefs.away)
      {
         sprintf (tbuf, "AWAY :%s\r\n", prefs.awayreason);
         tcp_send (((struct session *) menu_sess)->server, tbuf);
      } else
         tcp_send_len (((struct session *) menu_sess)->server, "AWAY\r\n", 6);
   }
}

void
menu_invisible (GtkWidget * wid, gpointer sess)
{
   char tbuf[128];
   prefs.invisible = !prefs.invisible;
   if (((struct session *) menu_sess)->server->connected)
   {
      if (prefs.invisible)
         sprintf (tbuf, "MODE %s +i\r\n", ((struct session *) menu_sess)->server->nick);
      else
         sprintf (tbuf, "MODE %s -i\r\n", ((struct session *) menu_sess)->server->nick);
      tcp_send (((struct session *) menu_sess)->server, tbuf);
   }
}

void
menu_help (GtkWidget * wid, gpointer sess)
{
   cmd_help (menu_sess, 0, 0, 0);
}

void
menu_savedefault (GtkWidget * wid, gpointer sess)
{
   if (save_config ())
      gtkutil_simpledialog ("Settings saved.");
}

void
menu_chanlist (GtkWidget * wid, gpointer sess)
{
   chanlist_opengui (((struct session *) menu_sess)->server);
}

/*void goto_url()
   {
   popen("netscape -remote 'openURL(http://xchat.linuxpower.org)'", "r"); 
   } */

#ifndef __DATE__
#define __DATE__ "<unknown>"
#endif

#ifdef USE_GNOME

void
menu_about (GtkWidget * wid, gpointer sess)
{
   GtkWidget *about;
   const gchar *author[] =
   {"Peter Zelezny <zed@linuxpower.org>", 0};
   char buf[512];
   struct utsname un;

   uname (&un);

   snprintf (buf, sizeof buf,
             "This software comes with NO WARRANTY at all, please\n"
             "refer to COPYING file for more details.\n"
             "X-Chat Project Started:   Jul 27 1998\n"
             "This Binary Compiled:   " __DATE__ "\n"
             "GTK+ Version: %d.%d.%d\n"
             "Currently Running on:   %s %s [%s]\n"
             "Author EMail:   zed@linuxpower.org\n"
             "X-Chat Web Page:   http://xchat.linuxpower.org",
             gtk_major_version, gtk_minor_version, gtk_micro_version,
   un.sysname, un.release, un.machine);

   about = gnome_about_new ("X-Chat",
                            VERSION,
          "(C) 1998-99 Peter Zelezny",
                            author,
                            buf,
                            0);
   gtk_widget_show (about);
}

#else

GtkWidget *about = 0;
GtkStyle *about_style;
GtkStyle *head_style;

int
about_close (void)
{
   gtk_style_unref (about_style);
   gtk_style_unref (head_style);
   about = 0;
   return 0;
}

void
menu_about (GtkWidget * wid, gpointer sess)
{
   char buf[512];
   struct utsname un;
   GtkWidget *vbox, *label, *hbox;

   if (about)
   {
      gtk_widget_destroy (about);
      return;
   }
   head_style = gtk_style_new ();
   head_style->font = gdk_font_load ("-adobe-new century schoolbook-bold-r-normal-*-*-240-*-*-p-*-iso8859-1");
   if (!head_style->font)
      head_style->font = gdk_font_load ("fixed");
   memcpy (head_style->fg, &colors[2], sizeof (GdkColor));

   about_style = gtk_style_new ();
   about_style->font = gdk_font_load ("fixed");

   about = gtk_window_new (GTK_WINDOW_DIALOG);
   gtk_window_position (GTK_WINDOW (about), GTK_WIN_POS_CENTER);
   gtk_window_set_title (GTK_WINDOW (about), "About X-Chat");
   gtk_container_set_border_width (GTK_CONTAINER (about), 6);
   gtk_signal_connect (GTK_OBJECT (about), "destroy",
    GTK_SIGNAL_FUNC (about_close), 0);
   gtk_widget_realize (about);

   vbox = gtk_vbox_new (0, 2);
   gtk_container_add (GTK_CONTAINER (about), vbox);
   gtk_widget_show (vbox);

   label = gtk_entry_new ();
   gtk_entry_set_editable (GTK_ENTRY (label), FALSE);
   gtk_entry_set_text (GTK_ENTRY (label), "X-Chat " VERSION);
   gtk_widget_set_style (label, head_style);
   gtk_container_add (GTK_CONTAINER (vbox), label);
   gtk_widget_show (label);

   label = gtk_label_new (
                            "\n"
                            "   X-Chat " VERSION " Copyright (c) 1998-99 By Peter Zelezny\n"
                            "    This software comes with NO WARRANTY at all, please\n"
                            "         refer to COPYING file for more details.\n"
      );
   gtk_container_add (GTK_CONTAINER (vbox), label);
   gtk_widget_show (label);

   uname (&un);
   snprintf (buf, sizeof buf,
             "  X-Chat Project Started ... : Jul 27 1998\n"
             "  This Binary Compiled ..... : " __DATE__ "\n"
             "  GTK+ Version ............. : %d.%d.%d\n"
             "  Currently Running on ..... : %s %s [%s]\n"
             "  Author EMail ............. : zed@linuxpower.org\n"
             "  X-Chat Web Page .......... : http://xchat.linuxpower.org\n",
             gtk_major_version, gtk_minor_version, gtk_micro_version,
   un.sysname, un.release, un.machine);

   label = gtk_label_new (buf);
   gtk_container_add (GTK_CONTAINER (vbox), label);
   gtk_widget_set_style (label, about_style);
   gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_LEFT);
   gtk_widget_show (label);

   wid = gtk_hseparator_new ();
   gtk_container_add (GTK_CONTAINER (vbox), wid);
   gtk_widget_show (wid);

   hbox = gtk_hbox_new (0, 2);
   gtk_container_add (GTK_CONTAINER (vbox), hbox);
   gtk_widget_show (hbox);

   wid = gtk_button_new_with_label ("  Continue  ");
   gtk_container_set_border_width (GTK_CONTAINER (hbox), 2);
   GTK_WIDGET_SET_FLAGS (wid, GTK_HAS_DEFAULT);
   GTK_WIDGET_SET_FLAGS (wid, GTK_HAS_FOCUS);
   gtk_box_pack_end (GTK_BOX (hbox), wid, 0, 0, 0);
   gtk_signal_connect (GTK_OBJECT (wid), "clicked",
                       GTK_SIGNAL_FUNC (gtkutil_destroy), about);
   gtk_widget_show (wid);

   gtk_widget_show (about);
}
#endif

#ifdef USE_PERL
void
menu_loadperl_callback (struct session *sess, void *data2, char *file)
{
   if (file)
   {
      char tbuf[256];
      sprintf (tbuf, "/LOAD %s", file);
      free (file);
      handle_command (tbuf, sess, FALSE, FALSE);
   }
}

void
menu_loadperl (void)
{
   gtkutil_file_req ("Select a script to load", menu_loadperl_callback,
                 menu_sess, 0, FALSE);
}

void
menu_unloadall (void)
{
   cmd_unloadall (menu_sess, 0, 0, 0);
}
#endif

void
menu_loadplugin_callback (struct session *sess, void *data2, char *file)
{
   if (file)
   {
      char tbuf[256];

      sprintf (tbuf, "/LOADDLL %s", file);
      free (file);
      handle_command (tbuf, sess, FALSE, FALSE);
   }
}

#ifdef USE_PLUGIN
void
menu_loadplugin (void)
{
   gtkutil_file_req ("Select a Plugin to load", menu_loadplugin_callback,
                 menu_sess, 0, FALSE);
}

void
menu_pluginlist (void)
{
   module_glist (menu_sess);
}

void
menu_unloadallplugins (void)
{
   module_unload (0, menu_sess);
}
#endif

#define usercommands_help "User Commands - Special codes:\n\n%1  =  word 1\n%2  =  word 2\n&1  = word 1 to the end of line\n&2  =  word 2 to the end of line\n\neg:\n/cmd john hello\n\n%2 would be \042john\042\n&2 would be \042john hello\042."

void
menu_usercommands (void)
{
   editlist_gui_open (command_list, "X-Chat: User Defined Commands", "commands", "commands.conf", usercommands_help);
}

void
menu_ulpopup (void)
{
   editlist_gui_open (popup_list, "X-Chat: Userlist Popup menu", "popup", "popup.conf", 0);
}

void
menu_rpopup (void)
{
   editlist_gui_open (replace_list, "X-Chat: Replace menu", "replace", "replace.conf", 0);
}

void
menu_evtpopup (void)
{
   pevent_dialog_show ();
}

void
menu_ulbuttons (void)
{
   editlist_gui_open (button_list, "X-Chat: Userlist buttons", "buttons", "buttons.conf", 0);
}

void
menu_ctcpguiopen (void)
{
   editlist_gui_open (ctcp_list, "X-Chat: CTCP Replies", "ctcpreply", "ctcpreply.conf", 0);
}

#ifdef USE_GNOME

#include "win.xpm"

GnomeUIInfo xchatmenu[] =
{
   {
      GNOME_APP_UI_ITEM,
      N_ ("Server List.."), 0,
      menu_open_server_list, 0, 0,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_BOOK_OPEN,
      0, 0, 0
   },
   GNOMEUIINFO_SEPARATOR,
   {
      GNOME_APP_UI_ITEM,
      N_ ("New Server Tab.."), 0,
      menu_newserver_tab, 0, 0,
      GNOME_APP_PIXMAP_DATA, win_xpm,
      0, 0, 0
   },
   {
      GNOME_APP_UI_ITEM,
      N_ ("New Server Window.."), 0,
      menu_newserver_window, 0, 0,
      GNOME_APP_PIXMAP_DATA, win_xpm,
      0, 0, 0
   },
   {
      GNOME_APP_UI_ITEM,
      N_ ("New Channel Tab.."), 0,
      menu_newchannel_tab, 0, 0,
      GNOME_APP_PIXMAP_DATA, win_xpm,
      0, 0, 0
   },
   {
      GNOME_APP_UI_ITEM,
      N_ ("New Channel Window.."), 0,
      menu_newchannel_window, 0, 0,
      GNOME_APP_PIXMAP_DATA, win_xpm,
      0, 0, 0
   },
   GNOMEUIINFO_SEPARATOR,
   {
      GNOME_APP_UI_ITEM,
      N_ ("Close"), 0,
      menu_close, 0, 0,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_CLOSE,
      0, 0, 0
   },
   GNOMEUIINFO_SEPARATOR,
   {
      GNOME_APP_UI_ITEM,
      N_ ("Quit"), 0,
      xchat_cleanup, 0, 0,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_QUIT,
      'Q', GDK_MOD1_MASK, 0
   },
   GNOMEUIINFO_END
};

GnomeUIInfo windowsmenu[] =
{
   {
      GNOME_APP_UI_ITEM,
      N_ ("Channel List Window.."), 0,
      menu_chanlist, 0, 0, GNOME_APP_PIXMAP_DATA, win_xpm,
   },
   {
      GNOME_APP_UI_ITEM,
      N_ ("DCC Send Window.."), 0,
      open_dcc_send_window, 0, 0, GNOME_APP_PIXMAP_DATA, win_xpm,
   },
   {
      GNOME_APP_UI_ITEM,
      N_ ("DCC Receive Window.."), 0,
      open_dcc_recv_window, 0, 0, GNOME_APP_PIXMAP_DATA, win_xpm,
   },
   {
      GNOME_APP_UI_ITEM,
      N_ ("DCC Chat Window.."), 0,
      open_dcc_chat_window, 0, 0, GNOME_APP_PIXMAP_DATA, win_xpm,
   },
   {
      GNOME_APP_UI_ITEM,
      N_ ("Raw Log Window.."), 0,
      menu_rawlog, 0, 0, GNOME_APP_PIXMAP_DATA, win_xpm,
   },
   {
      GNOME_APP_UI_ITEM,
      N_ ("URL Grabber Window.."), 0,
      url_opengui, 0, 0, GNOME_APP_PIXMAP_DATA, win_xpm,
   },
   {
      GNOME_APP_UI_ITEM,
      N_ ("Notify List Window.."), 0,
      (menucallback) notify_opengui, 0, 0, GNOME_APP_PIXMAP_DATA, win_xpm,
   },
   {
      GNOME_APP_UI_ITEM,
      N_ ("Lastlog Window.."), 0,
      menu_lastlog, 0, 0, GNOME_APP_PIXMAP_DATA, win_xpm,
   },
   GNOMEUIINFO_SEPARATOR,
   {
      GNOME_APP_UI_ITEM,
      N_ ("Flush Buffer"), 0,
      menu_flushbuffer, 0, 0,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_TRASH,
      'L', GDK_CONTROL_MASK, 0
   },
   {
      GNOME_APP_UI_ITEM,
      N_ ("Save Buffer.."), 0,
      menu_savebuffer, 0, 0, GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_SAVE, 0, 0, 0
   },
   GNOMEUIINFO_END
};

GnomeUIInfo usermodesmenu[] =
{
   {
      GNOME_APP_UI_TOGGLEITEM,
      N_ ("Invisible"), 0,
   menu_invisible, 0, 0, 0, 0, 0, 0, 0
   },
   {
      GNOME_APP_UI_TOGGLEITEM,
      N_ ("Receive Wallops"), 0,
      menu_wallops, 0, 0, GNOME_APP_PIXMAP_NONE, 0, 0, 0, 0
   },
   {
      GNOME_APP_UI_TOGGLEITEM,
      N_ ("Receive Server Notices"), 0,
menu_servernotice, 0, 0, 0, 0, 0, 0, 0
   },
   GNOMEUIINFO_SEPARATOR,
   {
      GNOME_APP_UI_TOGGLEITEM,
      N_ ("Marked Away"), 0,
      menu_away, 0, 0, GNOME_APP_PIXMAP_NONE, 0, 'A', GDK_MOD1_MASK, 0
   },
   GNOMEUIINFO_SEPARATOR,
   {
      GNOME_APP_UI_TOGGLEITEM,
      N_ ("Auto Rejoin on Kick"), 0,
  menu_autorejoin, 0, 0, 0, 0, 0, 0, 0
   },
   {
      GNOME_APP_UI_TOGGLEITEM,
   N_ ("Auto ReConnect to Server"), 0,
      menu_autoreconnect, 0, 0, GNOME_APP_PIXMAP_NONE, 0, 0, 0, 0
   },
   GNOMEUIINFO_SEPARATOR,
   {
      GNOME_APP_UI_TOGGLEITEM,
   N_ ("Auto Open Dialog Windows"), 0,
  menu_autodialog, 0, 0, 0, 0, 0, 0, 0
   },
   {
      GNOME_APP_UI_TOGGLEITEM,
      N_ ("Auto Accept DCC Chat"), 0,
 menu_autodccchat, 0, 0, 0, 0, 0, 0, 0
   },
   {
      GNOME_APP_UI_TOGGLEITEM,
      N_ ("Auto Accept DCC Send"), 0,
 menu_autodccsend, 0, 0, 0, 0, 0, 0, 0
   },
   GNOMEUIINFO_END
};

GnomeUIInfo settingsmenu[] =
{
   {
      GNOME_APP_UI_ITEM,
      N_ ("Setup.."), 0,
      menu_settings, 0, 0,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_PREF,
      'S', GDK_MOD1_MASK, 0
   },
   {
      GNOME_APP_UI_ITEM,
      N_ ("User Commands.."), 0,
      menu_usercommands, 0, 0,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_PREF,
      0, 0, 0
   },
   {
      GNOME_APP_UI_ITEM,
      N_ ("CTCP Replies.."), 0,
      menu_ctcpguiopen, 0, 0,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_PREF,
      0, 0, 0
   },
   {
      GNOME_APP_UI_ITEM,
      N_ ("Userlist Buttons.."), 0,
      menu_ulbuttons, 0, 0,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_PREF,
      0, 0, 0
   },
   {
      GNOME_APP_UI_ITEM,
      N_ ("Userlist Popup.."), 0,
      menu_ulpopup, 0, 0,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_PREF,
      0, 0, 0
   },
   {
      GNOME_APP_UI_ITEM,
      N_ ("Function Keys.."), 0,
      fkeys_edit, 0, 0,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_PREF,
      'F', GDK_MOD1_MASK, 0
   },
   {
      GNOME_APP_UI_ITEM,
      N_ ("Replace Popup.."), 0,
      menu_rpopup, 0, 0,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_PREF,
      0, 0, 0
   },
   {
      GNOME_APP_UI_ITEM,
      N_ ("Edit Event Texts.."), 0,
      menu_evtpopup, 0, 0,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_PREF,
      0, 0, 0
   },
   GNOMEUIINFO_SEPARATOR,
   {
      GNOME_APP_UI_ITEM,
      N_ ("Save Settings now"), 0,
      menu_savedefault, 0, 0,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_SAVE,
      0, 0, 0
   },
   {
      GNOME_APP_UI_TOGGLEITEM,
      N_ ("Save Settings on exit"), 0,
      menu_saveexit, 0, 0,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_SAVE,
      0, 0, 0
   },
   GNOMEUIINFO_END
};

GnomeUIInfo helpmenu[] =
{
   {
      GNOME_APP_UI_ITEM,
      N_ ("Help.."), 0,
      menu_help, 0, 0,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_HELP,
      0, 0, 0
   },
   GNOMEUIINFO_SEPARATOR,
   {
      GNOME_APP_UI_ITEM,
      N_ ("About X-Chat.."), 0,
      menu_about, 0, 0,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_ABOUT,
      0, 0, 0
   },
   GNOMEUIINFO_END
};

GnomeUIInfo scriptsmenu[] =
{
#ifdef USE_PERL
   {
      GNOME_APP_UI_ITEM,
      N_ ("Load Perl Script.."), 0,
      menu_loadperl, 0, 0,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_SAVE,
      0, 0, 0
   },
   {
      GNOME_APP_UI_ITEM,
      N_ ("Unload All Scripts"), 0,
      menu_unloadall, 0, 0,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_CLOSE,
      0, 0, 0
   },
#else
   {
      GNOME_APP_UI_ITEM,
      N_ ("Load Perl Script.."), 0,
      0, 0, 0,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_SAVE,
      0, 0, 0
   },
   {
      GNOME_APP_UI_ITEM,
      N_ ("Unload All Scripts"), 0,
      0, 0, 0,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_CLOSE,
      0, 0, 0
   },
#endif
   GNOMEUIINFO_SEPARATOR,
#ifdef USE_PLUGIN
   {
      GNOME_APP_UI_ITEM,
      N_ ("Load Plugin (*.so).."), 0,
      menu_loadplugin, 0, 0,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_SAVE,
      0, 0, 0
   },
   {
      GNOME_APP_UI_ITEM,
      N_ ("Unload All Plugins"), 0,
      menu_unloadallplugins, 0, 0,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_CLOSE,
      0, 0, 0
   },
   {
      GNOME_APP_UI_ITEM,
      N_ ("Plugin List"), 0,
      menu_pluginlist, 0, 0,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_BOOK_GREEN,
      0, 0, 0
   },
#else
   {
      GNOME_APP_UI_ITEM,
      N_ ("Load Plugin (*.so).."), 0,
      0, 0, 0,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_SAVE,
      0, 0, 0
   },
   {
      GNOME_APP_UI_ITEM,
      N_ ("Unload All Plugins"), 0,
      0, 0, 0,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_CLOSE,
      0, 0, 0
   },
   {
      GNOME_APP_UI_ITEM,
      N_ ("Plugin List"), 0,
      0, 0, 0,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_BOOK_GREEN,
      0, 0, 0
   },
#endif
   GNOMEUIINFO_END
};

GnomeUIInfo mainmenu[] =
{
   GNOMEUIINFO_SUBTREE (N_ ("_X-Chat"), xchatmenu),
   GNOMEUIINFO_SUBTREE (N_ ("_Windows"), windowsmenu),
   GNOMEUIINFO_SUBTREE (N_ ("User _Modes"), usermodesmenu),
   GNOMEUIINFO_SUBTREE (N_ ("_Settings"), settingsmenu),
   GNOMEUIINFO_SUBTREE (N_ ("S_cripts & Plugins"), scriptsmenu),
   GNOMEUIINFO_SUBTREE (N_ ("_Help"), helpmenu),
   GNOMEUIINFO_END
};

void
createmenus (void *app, struct session *sess)
{
   if (!menu_sess)
      menu_sess = sess;
   gnome_app_create_menus (GNOME_APP (app), mainmenu);
   gtk_menu_item_right_justify (GTK_MENU_ITEM (mainmenu[5].widget));

   /* is it legal to poke in the initial values?? */
   ((GtkCheckMenuItem *) (usermodesmenu[0].widget))->active = prefs.invisible;
   ((GtkCheckMenuItem *) (usermodesmenu[1].widget))->active = prefs.wallops;
   ((GtkCheckMenuItem *) (usermodesmenu[2].widget))->active = prefs.servernotice;
   ((GtkCheckMenuItem *) (usermodesmenu[4].widget))->active = prefs.away;
   ((GtkCheckMenuItem *) (usermodesmenu[6].widget))->active = prefs.autorejoin;
   ((GtkCheckMenuItem *) (usermodesmenu[7].widget))->active = prefs.autoreconnect;
   ((GtkCheckMenuItem *) (usermodesmenu[9].widget))->active = prefs.autodialog;
   ((GtkCheckMenuItem *) (usermodesmenu[10].widget))->active = prefs.autodccchat;
   ((GtkCheckMenuItem *) (usermodesmenu[11].widget))->active = prefs.autodccsend;
   ((GtkCheckMenuItem *) (settingsmenu[10].widget))->active = prefs.autosave;

#ifndef USE_PERL
   gtk_widget_set_sensitive (scriptsmenu[0].widget, FALSE);
   gtk_widget_set_sensitive (scriptsmenu[1].widget, FALSE);
#endif
#ifndef USE_PLUGIN
   gtk_widget_set_sensitive (scriptsmenu[3].widget, FALSE);
   gtk_widget_set_sensitive (scriptsmenu[4].widget, FALSE);
   gtk_widget_set_sensitive (scriptsmenu[5].widget, FALSE);
#endif
}

#else
static struct mymenu mymenu[] =
{
   {M_NEWMENU, "X-Chat", 0, 0, 1},
   {M_MENU, "Server List..", menu_open_server_list, 0, 1},
   {M_SEP, 0, 0, 0, 0},
   {M_MENU, "New Server Tab..", menu_newserver_tab, 0, 1},
   {M_MENU, "New Server Window..", menu_newserver_window, 0, 1},
   {M_MENU, "New Channel Tab..", menu_newchannel_tab, 0, 1},
   {M_MENU, "New Channel Window..", menu_newchannel_window, 0, 1},
   {M_SEP, 0, 0, 0, 0},
   {M_MENU, "Close", menu_close, 0, 1},
   {M_SEP, 0, 0, 0, 0},
{M_MENU, "Quit", xchat_cleanup, 0, 1},  /* 10 */

   {M_NEWMENU, "Windows", 0, 0, 1},
   {M_MENU, "Channel List Window..", menu_chanlist, 0, 1},
   {M_MENU, "DCC Send Window..", open_dcc_send_window, 0, 1},
   {M_MENU, "DCC Receive Window..", open_dcc_recv_window, 0, 1},
   {M_MENU, "DCC Chat Window..", open_dcc_chat_window, 0, 1},
   {M_MENU, "Raw Log Window..", menu_rawlog, 0, 1},
   {M_MENU, "URL Grabber Window..", url_opengui, 0, 1},
   {M_MENU, "Notify List Window..", (menucallback) notify_opengui, 0, 1},
   {M_MENU, "Lastlog Window..", menu_lastlog, 0, 1},
   {M_SEP, 0, 0, 0, 0},
   {M_MENU, "Flush Buffer", menu_flushbuffer, 0, 1},
   {M_MENU, "Save Buffer..", menu_savebuffer, 0, 1},  /* 22 */

   {M_NEWMENU, "User Modes", 0, 0, 1},
   {M_MENUTOG, "Invisible", menu_invisible, 1, 1},
   {M_MENUTOG, "Receive Wallops", menu_wallops, 1, 1},
   {M_MENUTOG, "Receive Server Notices", menu_servernotice, 1, 1},
   {M_SEP, 0, 0, 0, 0},
   {M_MENUTOG, "Marked Away", menu_away, 0, 1},
   {M_SEP, 0, 0, 0},
   {M_MENUTOG, "Auto ReJoin on Kick", menu_autorejoin, 0, 1},
   {M_MENUTOG, "Auto ReConnect to Server", menu_autoreconnect, 0, 1},
   {M_SEP, 0, 0, 0},
   {M_MENUTOG, "Auto Open Dialog Windows", menu_autodialog, 0, 1},  /* 33 */
   {M_MENUTOG, "Auto Accept DCC Chat", menu_autodccchat, 0, 1},
   {M_MENUTOG, "Auto Accept DCC Send", menu_autodccsend, 0, 1},

   {M_NEWMENU, "Settings", 0, 0, 1},  /* 36 */
   {M_MENU, "Setup..", menu_settings, 0, 1},
   {M_MENU, "User Commands..", (menucallback) menu_usercommands, 0, 1},
   {M_MENU, "CTCP Replies..", (menucallback) menu_ctcpguiopen, 0, 1},
   {M_MENU, "Userlist Buttons..", (menucallback) menu_ulbuttons, 0, 1},
   {M_MENU, "Userlist Popup..", (menucallback) menu_ulpopup, 0, 1},
   {M_MENU, "Function Keys..", (menucallback) fkeys_edit, 0, 1},
   {M_MENU, "Replace..", (menucallback) menu_rpopup, 0, 1},
   {M_MENU, "Edit Event Texts..", (menucallback) menu_evtpopup, 0, 1},
   {M_SEP, 0, 0, 0, 0},
   {M_MENU, "Save Settings now", menu_savedefault, 0, 1},
   {M_MENUTOG, "Save Settings on exit", menu_saveexit, 1, 1},

   {M_NEWMENU, "Scripts & Plugins", 0, 0, 1},
#ifdef USE_PERL
   {M_MENU, "Load Perl Script..", (menucallback) menu_loadperl, 0, 1},
   {M_MENU, "Unload All Scripts", (menucallback) menu_unloadall, 0, 1},
#else
   {M_MENU, "Load Perl Script..", 0, 0, 0},
   {M_MENU, "Unload All Scripts", 0, 0, 0},
#endif
   {M_SEP, 0, 0, 0, 0},
#ifdef USE_PLUGIN
   {M_MENU, "Load Plugin (*.so)..", (menucallback) menu_loadplugin, 0, 1},
   {M_MENU, "Unload All Plugins", (menucallback) menu_unloadallplugins, 0, 1},
   {M_MENU, "Plugin List", (menucallback) menu_pluginlist, 0, 1},
#else
   {M_MENU, "Load Plugin (*.so)..", 0, 0, 0},
   {M_MENU, "Unload All Plugins", 0, 0, 0},
   {M_MENU, "Plugin List", 0, 0, 0},
#endif

   {M_NEWMENURIGHT, "Help", 0, 0, 1},
   {M_MENU, "Help..", menu_help, 0, 1},
   {M_SEP, 0, 0, 0, 0},
   {M_MENU, "About X-Chat..", menu_about, 0, 1},

   {M_END, 0, 0, 0, 0},
};


GtkWidget *
createmenus (struct session *sess)
{
   int i = 0;
   GtkWidget *item;
   GtkWidget *menu = 0;
   GtkWidget *menu_item = 0;
   GtkWidget *menu_bar = gtk_menu_bar_new ();

   if (!menu_sess)
      menu_sess = sess;

   mymenu[24].state = prefs.invisible;
   mymenu[25].state = prefs.wallops;
   mymenu[26].state = prefs.servernotice;
   mymenu[30].state = prefs.autorejoin;
   mymenu[31].state = prefs.autoreconnect;
   mymenu[33].state = prefs.autodialog;
   mymenu[34].state = prefs.autodccchat;
   mymenu[35].state = prefs.autodccsend;
   mymenu[47].state = prefs.autosave;

   while (1)
   {
      switch (mymenu[i].type)
      {
      case M_NEWMENURIGHT:
      case M_NEWMENU:
         if (menu)
            gtk_menu_item_set_submenu (GTK_MENU_ITEM (menu_item), menu);
         menu = gtk_menu_new ();
         menu_item = gtk_menu_item_new_with_label (mymenu[i].text);
         if (mymenu[i].type == M_NEWMENURIGHT)
            gtk_menu_item_right_justify ((GtkMenuItem *) menu_item);
         gtk_menu_bar_append (GTK_MENU_BAR (menu_bar), menu_item);
         gtk_widget_show (menu_item);
         break;
      case M_MENU:
         item = gtk_menu_item_new_with_label (mymenu[i].text);
         if (mymenu[i].callback)
            gtk_signal_connect_object (GTK_OBJECT (item), "activate",
                                       GTK_SIGNAL_FUNC (mymenu[i].callback), (gpointer) sess);
         gtk_menu_append (GTK_MENU (menu), item);
         gtk_widget_show (item);
         gtk_widget_set_sensitive (item, mymenu[i].activate);
         break;
      case M_MENUTOG:
         item = gtk_check_menu_item_new_with_label (mymenu[i].text);
         gtk_check_menu_item_set_state (GTK_CHECK_MENU_ITEM (item), mymenu[i].state);
         if (mymenu[i].callback)
            gtk_signal_connect_object (GTK_OBJECT (item), "toggled",
                                       GTK_SIGNAL_FUNC (mymenu[i].callback), (gpointer) sess);
         gtk_menu_append (GTK_MENU (menu), item);
         gtk_widget_show (item);
         gtk_widget_set_sensitive (item, mymenu[i].activate);
         break;
      case M_SEP:
         item = gtk_menu_item_new ();
         gtk_widget_set_sensitive (item, FALSE);
         gtk_menu_append (GTK_MENU (menu), item);
         gtk_widget_show (item);
         break;
      case M_END:
         if (menu)
            gtk_menu_item_set_submenu (GTK_MENU_ITEM (menu_item), menu);
         return (menu_bar);
      }
      i++;
   }
}
#endif
