/* X-Chat
 * Copyright (C) 1998 Peter Zelezny.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#undef USE_GNOME
#include "xchat.h"
#include "draw.h"
#include "dcc.h"
#include "style.h"
#include "gtkutil.h"
#include <time.h>

extern int tcp_send_len(struct server *serv, char *buf, int len);
extern int tcp_send(struct server *serv, char *buf);
extern struct DCC *find_dcc(char *nick, char *file, int type);
extern struct DCC *new_dcc(void);
extern void PrintText(struct session *sess, char *text);

struct xchatpixel
{
   unsigned int x:16;
   unsigned int y:16;
};

static int	oldx = -1, oldy = -1;
static int	curx, cury;

#define min(a, b) ( (a)<(b) ? (a) : (b) )
#define max(a, b) ( (a)>(b) ? (a) : (b) )

void dcc_close_draw_window(struct DCC *dcc)
{
   free(dcc->dccdraw);
   dcc->dccdraw = 0;
}

static void kill_dccdraw(GtkWidget *win, struct DCC *dcc)
{
   dcc_close(dcc, 0, TRUE);
}

static gint configure_event (GtkWidget *widget, GdkEventConfigure *event, struct DCC *dcc)
{
   if(dcc->dccdraw->pixmap) gdk_pixmap_unref(dcc->dccdraw->pixmap);

   dcc->dccdraw->pixmap = gdk_pixmap_new(widget->window,
	       	                     widget->allocation.width,
	                             widget->allocation.height,
	                             -1);
   gdk_draw_rectangle (dcc->dccdraw->pixmap,
		       widget->style->white_gc,
                       TRUE,
		       0, 0,
		       widget->allocation.width,
		       widget->allocation.height);
   return TRUE;
}

         /* Redraw the screen from the backing pixmap */
static gint expose_event (GtkWidget *widget, GdkEventExpose *event, struct DCC *dcc)
{
   gdk_draw_pixmap(widget->window,
	           widget->style->fg_gc[GTK_WIDGET_STATE (widget)],
	           dcc->dccdraw->pixmap,
                   event->area.x, event->area.y,
                   event->area.x, event->area.y,
                   event->area.width, event->area.height);
   return FALSE;
}

static void dcc_send_pixel(struct DCC *dcc, int x, int y, char type)
{
   char buf[sizeof(struct xchatpixel)+2];
   struct xchatpixel pix;
   pix.x = htons(x);
   pix.y = htons(y);
   buf[0] = type;  /* packet type */
   buf[1] = 4;  /* packet size: 4 bytes */
   memcpy((char *)&buf[2], (char *)&pix, sizeof(struct xchatpixel));
   send(dcc->sok, buf, sizeof(struct xchatpixel) + 2, 0);
}

        /* Draw a rectangle on the screen */
static void draw_brush (GtkWidget *widget, gdouble x, gdouble y, struct DCC *dcc)
{
   GdkRectangle update_rect;
   update_rect.x = x - 1;
   update_rect.y = y - 1;
   update_rect.width = x + 1;
   update_rect.height = y + 1;
   gdk_draw_rectangle(dcc->dccdraw->pixmap,
	               widget->style->black_gc,
	               TRUE,
		       update_rect.x, update_rect.y,
	               update_rect.width, update_rect.height);
   gtk_widget_draw(widget, &update_rect);
}

static void draw_line (GtkWidget *widget, gdouble x, gdouble y, gdouble x2, gdouble y2, struct DCC *dcc)
{
	int X, Y, XX, YY;
	GdkRectangle update_rect;
	
	X = min(x, x2);
	Y = min(y, y2);
	XX = max(x, x2);
	YY = max(y, y2);
	
	update_rect.x = X - 1;
	update_rect.y = Y - 1;
	update_rect.width = XX + 1;
	update_rect.height = YY + 1;
	
	gdk_draw_line (dcc->dccdraw->pixmap,
		widget->style->black_gc,
		x, y, x2, y2);
	gtk_widget_draw(widget, &update_rect);
}

static gint button_press_event (GtkWidget *widget, GdkEventButton *event, struct DCC *dcc)
{
   if (event->button == 1 && dcc->dccdraw->pixmap != NULL) {
   	oldx = event->x;
	oldy = event->y;
   	dcc_send_pixel(dcc, event->x, event->y, 2);
   }
   
   return TRUE;
}

static gint button_release_event (GtkWidget *widget, GdkEventButton *event, struct DCC *dcc)
{
   /*fprintf(stderr, "release\n");*/
   if (event->button == 1 && dcc->dccdraw->pixmap != NULL) {
   	oldx = -1;
   }
   return TRUE;
}

static gint motion_notify_event (GtkWidget *widget, GdkEventMotion *event, struct DCC *dcc)
{
   int x, y;
   GdkModifierType state;

   if (oldx == -1)
       return TRUE;

   if (event->is_hint)
      gdk_window_get_pointer (event->window, &x, &y, &state);
   else {
      x = event->x;
      y = event->y;
      state = event->state;
   }

   if (state & GDK_BUTTON1_MASK && dcc->dccdraw->pixmap != NULL) {
      draw_line(dcc->dccdraw->drawing_area, oldx, oldy, x, y, dcc);
      oldx = x;
      oldy = y;
      dcc_send_pixel(dcc, x, y, 3);
   }

   return TRUE;
}

void dcc_open_draw_window(struct DCC *dcc)
{
   GtkWidget *vbox;
   char tbuf[128];

   dcc->dccdraw = malloc(sizeof(struct dccdraw));
   memset(dcc->dccdraw, 0, sizeof(struct dccdraw));

   sprintf(tbuf, "DCC-Draw: %s", dcc->nick);
   dcc->dccdraw->window = gtkutil_window_new(tbuf, "dcc-draw", dcc->pos+4, dcc->size+4,
					     kill_dccdraw, dcc);

   vbox = gtk_vbox_new (FALSE, 0);
   gtk_container_add (GTK_CONTAINER (dcc->dccdraw->window), vbox);
   gtk_widget_show (vbox);

   dcc->dccdraw->drawing_area = gtk_drawing_area_new ();
   gtk_drawing_area_size (GTK_DRAWING_AREA (dcc->dccdraw->drawing_area), dcc->pos, dcc->size);
   gtk_box_pack_start (GTK_BOX (vbox), dcc->dccdraw->drawing_area, TRUE, TRUE, 0); 
   gtk_widget_show (dcc->dccdraw->drawing_area);

   gtk_signal_connect(GTK_OBJECT(dcc->dccdraw->drawing_area), "expose_event",
		      (GtkSignalFunc) expose_event, dcc);
   gtk_signal_connect(GTK_OBJECT(dcc->dccdraw->drawing_area),"configure_event",
	              (GtkSignalFunc) configure_event, dcc);
   gtk_signal_connect(GTK_OBJECT(dcc->dccdraw->drawing_area), "motion_notify_event",
		      (GtkSignalFunc) motion_notify_event, dcc);
   gtk_signal_connect(GTK_OBJECT (dcc->dccdraw->drawing_area), "button_release_event",
                      GTK_SIGNAL_FUNC(button_release_event), dcc);
   gtk_signal_connect(GTK_OBJECT (dcc->dccdraw->drawing_area), "button_press_event",
                      (GtkSignalFunc) button_press_event, dcc);

   gtk_widget_set_events (dcc->dccdraw->drawing_area, GDK_EXPOSURE_MASK
			              | GDK_LEAVE_NOTIFY_MASK
		                      | GDK_BUTTON_PRESS_MASK
		                      | GDK_POINTER_MOTION_MASK
			              | GDK_POINTER_MOTION_HINT_MASK);
   gtk_widget_show(dcc->dccdraw->window);
}

void dcc_read_draw(struct DCC *dcc, gint sok)
{
   char buf[256];
   long len;
   unsigned char type[2];
   struct xchatpixel pix;

   len = recv(dcc->sok, type, 2, 0);
   if(len < 1)
   {
      gtk_widget_destroy(dcc->dccdraw->window);
      return;
   }
   switch(type[0])
   {
    case 1:
      len = recv(dcc->sok, (char *)&pix, sizeof(struct xchatpixel), 0);
      if(len < 1)
      {
         gtk_widget_destroy(dcc->dccdraw->window);
	 return;
      }
      pix.x = ntohs(pix.x);
      pix.y = ntohs(pix.y);
      draw_brush(dcc->dccdraw->drawing_area, pix.x, pix.y, dcc);
      break;
    case 2:
      len = recv(dcc->sok, (char *)&pix, sizeof(struct xchatpixel), 0);
      if (len < 1)
      {
         gtk_widget_destroy(dcc->dccdraw->window);
	 return;
      }
      curx = ntohs(pix.x);
      cury = ntohs(pix.y);
      break;
    case 3:
      len = recv(dcc->sok, (char *)&pix, sizeof(struct xchatpixel), 0);
      if (len < 1)
      {
         gtk_widget_destroy(dcc->dccdraw->window);
	 return;
      }
      pix.x = ntohs(pix.x);
      pix.y = ntohs(pix.y);
      draw_line(dcc->dccdraw->drawing_area, curx, cury, pix.x, pix.y, dcc);
      curx = pix.x;
      cury = pix.y;
      break;
    default:
      recv(dcc->sok, buf, type[1], 0);
      printf("Unsupported DRAW packet: %d\n", type[0]);
   }
}

void dcc_draw(struct session *sess, char *nick)
{
   char outbuf[256];
   struct DCC *dcc;

   dcc = find_dcc(nick, "", TYPE_DRAWSEND);
   if(dcc)
   {
      switch(dcc->stat)
      {
       case STAT_ACTIVE:
       case STAT_QUEUED:
       case STAT_CONNECTING:
	 sprintf(outbuf, STARTON" Already offering DCC DRAW to %s\n", nick);
	 PrintText(sess, outbuf);
	 return;
       case STAT_ABORTED:
       case STAT_FAILED:
	 dcc_close(dcc, 0, TRUE);
      }
   }

   dcc = find_dcc(nick, "", TYPE_DRAWRECV);
   if(dcc)
   {
      switch(dcc->stat)
      {
	 case STAT_QUEUED:
	   dcc_connect(0, dcc);
	   break;
	 case STAT_FAILED:
	 case STAT_ABORTED:
	   dcc_close(dcc, 0, TRUE);
      }
      return;
   }

   /* offer DCC DRAW */

   dcc = new_dcc();
   if(!dcc) return;
   dcc->pos = 400;
   dcc->size = 300;
   dcc->serv = sess->server;
   dcc->stat = STAT_QUEUED;
   dcc->type = TYPE_DRAWSEND;
   time(&dcc->starttime);
   strcpy(dcc->nick, nick);
   if(dcc_listen_init(dcc))
   {
      sprintf(outbuf, "PRIVMSG %s :\001DCC DRAW 400 300 %lu %ld\001\r\n",
		 nick, dcc->addr, dcc->port);
      tcp_send(dcc->serv, outbuf);
      sprintf(outbuf, STARTON" Offering DCC DRAW to %s\n", nick);
      PrintText(sess, outbuf);
   } else
      dcc_close(dcc, 0, TRUE); 
}
