/* valaccodewriter.vala
 *
 * Copyright (C) 2006-2009  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <glib.h>
#include <glib-object.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <glib/gstdio.h>
#include <gobject/gvaluecollector.h>


#define VALA_TYPE_CCODE_WRITER (vala_ccode_writer_get_type ())
#define VALA_CCODE_WRITER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CCODE_WRITER, ValaCCodeWriter))
#define VALA_CCODE_WRITER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CCODE_WRITER, ValaCCodeWriterClass))
#define VALA_IS_CCODE_WRITER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CCODE_WRITER))
#define VALA_IS_CCODE_WRITER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CCODE_WRITER))
#define VALA_CCODE_WRITER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CCODE_WRITER, ValaCCodeWriterClass))

typedef struct _ValaCCodeWriter ValaCCodeWriter;
typedef struct _ValaCCodeWriterClass ValaCCodeWriterClass;
typedef struct _ValaCCodeWriterPrivate ValaCCodeWriterPrivate;

#define VALA_TYPE_CCODE_NODE (vala_ccode_node_get_type ())
#define VALA_CCODE_NODE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CCODE_NODE, ValaCCodeNode))
#define VALA_CCODE_NODE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CCODE_NODE, ValaCCodeNodeClass))
#define VALA_IS_CCODE_NODE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CCODE_NODE))
#define VALA_IS_CCODE_NODE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CCODE_NODE))
#define VALA_CCODE_NODE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CCODE_NODE, ValaCCodeNodeClass))

typedef struct _ValaCCodeNode ValaCCodeNode;
typedef struct _ValaCCodeNodeClass ValaCCodeNodeClass;

#define VALA_TYPE_CCODE_LINE_DIRECTIVE (vala_ccode_line_directive_get_type ())
#define VALA_CCODE_LINE_DIRECTIVE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CCODE_LINE_DIRECTIVE, ValaCCodeLineDirective))
#define VALA_CCODE_LINE_DIRECTIVE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CCODE_LINE_DIRECTIVE, ValaCCodeLineDirectiveClass))
#define VALA_IS_CCODE_LINE_DIRECTIVE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CCODE_LINE_DIRECTIVE))
#define VALA_IS_CCODE_LINE_DIRECTIVE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CCODE_LINE_DIRECTIVE))
#define VALA_CCODE_LINE_DIRECTIVE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CCODE_LINE_DIRECTIVE, ValaCCodeLineDirectiveClass))

typedef struct _ValaCCodeLineDirective ValaCCodeLineDirective;
typedef struct _ValaCCodeLineDirectiveClass ValaCCodeLineDirectiveClass;
typedef struct _ValaParamSpecCCodeWriter ValaParamSpecCCodeWriter;

/**
 * Represents a writer to write C source files.
 */
struct _ValaCCodeWriter {
	GTypeInstance parent_instance;
	volatile int ref_count;
	ValaCCodeWriterPrivate * priv;
};

struct _ValaCCodeWriterClass {
	GTypeClass parent_class;
	void (*finalize) (ValaCCodeWriter *self);
};

struct _ValaCCodeWriterPrivate {
	char* _filename;
	gboolean _line_directives;
	char* temp_filename;
	gboolean file_exists;
	FILE* stream;
	gint indent;
	gint current_line_number;
	gboolean using_line_directive;
	gboolean _bol;
};

struct _ValaParamSpecCCodeWriter {
	GParamSpec parent_instance;
};


static gpointer vala_ccode_writer_parent_class = NULL;

gpointer vala_ccode_writer_ref (gpointer instance);
void vala_ccode_writer_unref (gpointer instance);
GParamSpec* vala_param_spec_ccode_writer (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_ccode_writer (GValue* value, gpointer v_object);
gpointer vala_value_get_ccode_writer (const GValue* value);
GType vala_ccode_writer_get_type (void);
#define VALA_CCODE_WRITER_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_CCODE_WRITER, ValaCCodeWriterPrivate))
enum  {
	VALA_CCODE_WRITER_DUMMY_PROPERTY
};
void vala_ccode_writer_set_filename (ValaCCodeWriter* self, const char* value);
ValaCCodeWriter* vala_ccode_writer_new (const char* filename);
ValaCCodeWriter* vala_ccode_writer_construct (GType object_type, const char* filename);
const char* vala_ccode_writer_get_filename (ValaCCodeWriter* self);
gboolean vala_ccode_writer_open (ValaCCodeWriter* self);
void vala_ccode_writer_close (ValaCCodeWriter* self);
gboolean vala_ccode_writer_get_line_directives (ValaCCodeWriter* self);
gpointer vala_ccode_node_ref (gpointer instance);
void vala_ccode_node_unref (gpointer instance);
GParamSpec* vala_param_spec_ccode_node (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_ccode_node (GValue* value, gpointer v_object);
gpointer vala_value_get_ccode_node (const GValue* value);
GType vala_ccode_node_get_type (void);
void vala_ccode_node_write (ValaCCodeNode* self, ValaCCodeWriter* writer);
void vala_ccode_writer_write_string (ValaCCodeWriter* self, const char* s);
void vala_ccode_writer_write_newline (ValaCCodeWriter* self);
gboolean vala_ccode_writer_get_bol (ValaCCodeWriter* self);
GType vala_ccode_line_directive_get_type (void);
void vala_ccode_writer_write_indent (ValaCCodeWriter* self, ValaCCodeLineDirective* line);
void vala_ccode_writer_write_begin_block (ValaCCodeWriter* self);
void vala_ccode_writer_write_end_block (ValaCCodeWriter* self);
void vala_ccode_writer_write_comment (ValaCCodeWriter* self, const char* text);
void vala_ccode_writer_set_line_directives (ValaCCodeWriter* self, gboolean value);
static void vala_ccode_writer_finalize (ValaCCodeWriter* obj);
static void _vala_array_destroy (gpointer array, gint array_length, GDestroyNotify destroy_func);
static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func);
static gint _vala_array_length (gpointer array);



ValaCCodeWriter* vala_ccode_writer_construct (GType object_type, const char* filename) {
	ValaCCodeWriter* self;
	g_return_val_if_fail (filename != NULL, NULL);
	self = (ValaCCodeWriter*) g_type_create_instance (object_type);
	vala_ccode_writer_set_filename (self, filename);
	return self;
}


ValaCCodeWriter* vala_ccode_writer_new (const char* filename) {
	return vala_ccode_writer_construct (VALA_TYPE_CCODE_WRITER, filename);
}


/**
 * Opens the file.
 *
 * @return true if the file has been opened successfully,
 *         false otherwise
 */
gboolean vala_ccode_writer_open (ValaCCodeWriter* self) {
	gboolean result;
	g_return_val_if_fail (self != NULL, FALSE);
	self->priv->file_exists = g_file_test (self->priv->_filename, G_FILE_TEST_EXISTS);
	if (self->priv->file_exists) {
		char* _tmp0_;
		FILE* _tmp1_;
		_tmp0_ = NULL;
		self->priv->temp_filename = (_tmp0_ = g_strdup_printf ("%s.valatmp", self->priv->_filename), self->priv->temp_filename = (g_free (self->priv->temp_filename), NULL), _tmp0_);
		_tmp1_ = NULL;
		self->priv->stream = (_tmp1_ = fopen (self->priv->temp_filename, "w"), (self->priv->stream == NULL) ? NULL : (self->priv->stream = (fclose (self->priv->stream), NULL)), _tmp1_);
	} else {
		FILE* _tmp2_;
		_tmp2_ = NULL;
		self->priv->stream = (_tmp2_ = fopen (self->priv->_filename, "w"), (self->priv->stream == NULL) ? NULL : (self->priv->stream = (fclose (self->priv->stream), NULL)), _tmp2_);
	}
	result = self->priv->stream != NULL;
	return result;
}


/**
 * Closes the file.
 */
void vala_ccode_writer_close (ValaCCodeWriter* self) {
	GError * _inner_error_;
	FILE* _tmp0_;
	g_return_if_fail (self != NULL);
	_inner_error_ = NULL;
	_tmp0_ = NULL;
	self->priv->stream = (_tmp0_ = NULL, (self->priv->stream == NULL) ? NULL : (self->priv->stream = (fclose (self->priv->stream), NULL)), _tmp0_);
	if (self->priv->file_exists) {
		gboolean changed;
		changed = TRUE;
		{
			GMappedFile* old_file;
			GMappedFile* new_file;
			glong len;
			GMappedFile* _tmp1_;
			GMappedFile* _tmp2_;
			old_file = g_mapped_file_new (self->priv->_filename, FALSE, &_inner_error_);
			if (_inner_error_ != NULL) {
				if (_inner_error_->domain == G_FILE_ERROR) {
					goto __catch0_g_file_error;
				}
				goto __finally0;
			}
			new_file = g_mapped_file_new (self->priv->temp_filename, FALSE, &_inner_error_);
			if (_inner_error_ != NULL) {
				(old_file == NULL) ? NULL : (old_file = (g_mapped_file_free (old_file), NULL));
				if (_inner_error_->domain == G_FILE_ERROR) {
					goto __catch0_g_file_error;
				}
				goto __finally0;
			}
			len = g_mapped_file_get_length (old_file);
			if (len == g_mapped_file_get_length (new_file)) {
				if (memcmp (g_mapped_file_get_contents (old_file), g_mapped_file_get_contents (new_file), (gsize) len) == 0) {
					changed = FALSE;
				}
			}
			_tmp1_ = NULL;
			old_file = (_tmp1_ = NULL, (old_file == NULL) ? NULL : (old_file = (g_mapped_file_free (old_file), NULL)), _tmp1_);
			_tmp2_ = NULL;
			new_file = (_tmp2_ = NULL, (new_file == NULL) ? NULL : (new_file = (g_mapped_file_free (new_file), NULL)), _tmp2_);
			(old_file == NULL) ? NULL : (old_file = (g_mapped_file_free (old_file), NULL));
			(new_file == NULL) ? NULL : (new_file = (g_mapped_file_free (new_file), NULL));
		}
		goto __finally0;
		__catch0_g_file_error:
		{
			GError * e;
			e = _inner_error_;
			_inner_error_ = NULL;
			{
				(e == NULL) ? NULL : (e = (g_error_free (e), NULL));
			}
		}
		__finally0:
		if (_inner_error_ != NULL) {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, _inner_error_->message);
			g_clear_error (&_inner_error_);
			return;
		}
		/* assume changed if mmap comparison doesn't work*/
		if (changed) {
			g_rename (self->priv->temp_filename, self->priv->_filename);
		} else {
			g_unlink (self->priv->temp_filename);
		}
	}
}


/**
 * Writes tabs according to the current indent level.
 */
void vala_ccode_writer_write_indent (ValaCCodeWriter* self, ValaCCodeLineDirective* line) {
	g_return_if_fail (self != NULL);
	if (self->priv->_line_directives) {
		if (line != NULL) {
			vala_ccode_node_write ((ValaCCodeNode*) line, self);
			self->priv->using_line_directive = TRUE;
		} else {
			if (self->priv->using_line_directive) {
				char* _tmp1_;
				char* _tmp0_;
				/* no corresponding Vala line, emit line directive for C line*/
				_tmp1_ = NULL;
				_tmp0_ = NULL;
				vala_ccode_writer_write_string (self, _tmp1_ = g_strdup_printf ("#line %d \"%s\"", self->priv->current_line_number + 1, _tmp0_ = g_path_get_basename (self->priv->_filename)));
				_tmp1_ = (g_free (_tmp1_), NULL);
				_tmp0_ = (g_free (_tmp0_), NULL);
				vala_ccode_writer_write_newline (self);
				self->priv->using_line_directive = FALSE;
			}
		}
	}
	if (!vala_ccode_writer_get_bol (self)) {
		vala_ccode_writer_write_newline (self);
	}
	{
		gint i;
		i = 0;
		{
			gboolean _tmp2_;
			_tmp2_ = TRUE;
			while (TRUE) {
				if (!_tmp2_) {
					i++;
				}
				_tmp2_ = FALSE;
				if (!(i < self->priv->indent)) {
					break;
				}
				fputc ('\t', self->priv->stream);
			}
		}
	}
	self->priv->_bol = FALSE;
}


/**
 * Writes the specified string.
 *
 * @param s a string
 */
void vala_ccode_writer_write_string (ValaCCodeWriter* self, const char* s) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (s != NULL);
	fprintf (self->priv->stream, "%s", s);
	self->priv->_bol = FALSE;
}


/**
 * Writes a newline.
 */
void vala_ccode_writer_write_newline (ValaCCodeWriter* self) {
	g_return_if_fail (self != NULL);
	fputc ('\n', self->priv->stream);
	self->priv->current_line_number++;
	self->priv->_bol = TRUE;
}


/**
 * Opens a new block, increasing the indent level.
 */
void vala_ccode_writer_write_begin_block (ValaCCodeWriter* self) {
	g_return_if_fail (self != NULL);
	if (!vala_ccode_writer_get_bol (self)) {
		fputc (' ', self->priv->stream);
	} else {
		vala_ccode_writer_write_indent (self, NULL);
	}
	fputc ('{', self->priv->stream);
	vala_ccode_writer_write_newline (self);
	self->priv->indent++;
}


/**
 * Closes the current block, decreasing the indent level.
 */
void vala_ccode_writer_write_end_block (ValaCCodeWriter* self) {
	g_return_if_fail (self != NULL);
	g_assert (self->priv->indent > 0);
	self->priv->indent--;
	vala_ccode_writer_write_indent (self, NULL);
	fprintf (self->priv->stream, "}");
}


/**
 * Writes the specified text as comment.
 *
 * @param text the comment text
 */
void vala_ccode_writer_write_comment (ValaCCodeWriter* self, const char* text) {
	gboolean first;
	char** _tmp1_;
	gint lines_size;
	gint lines_length1;
	char** _tmp0_;
	char** lines;
	g_return_if_fail (self != NULL);
	g_return_if_fail (text != NULL);
	vala_ccode_writer_write_indent (self, NULL);
	fprintf (self->priv->stream, "/*");
	first = TRUE;
	/* separate declaration due to missing memory management in foreach statements */
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	lines = (_tmp1_ = _tmp0_ = g_strsplit (text, "\n", 0), lines_length1 = _vala_array_length (_tmp0_), lines_size = lines_length1, _tmp1_);
	{
		char** line_collection;
		int line_collection_length1;
		int line_it;
		line_collection = lines;
		line_collection_length1 = lines_length1;
		for (line_it = 0; line_it < lines_length1; line_it = line_it + 1) {
			const char* _tmp2_;
			char* line;
			_tmp2_ = NULL;
			line = (_tmp2_ = line_collection[line_it], (_tmp2_ == NULL) ? NULL : g_strdup (_tmp2_));
			{
				if (!first) {
					vala_ccode_writer_write_indent (self, NULL);
				} else {
					first = FALSE;
				}
				fprintf (self->priv->stream, "%s", line);
				line = (g_free (line), NULL);
			}
		}
	}
	fprintf (self->priv->stream, "*/");
	vala_ccode_writer_write_newline (self);
	lines = (_vala_array_free (lines, lines_length1, (GDestroyNotify) g_free), NULL);
}


const char* vala_ccode_writer_get_filename (ValaCCodeWriter* self) {
	const char* result;
	g_return_val_if_fail (self != NULL, NULL);
	result = self->priv->_filename;
	return result;
}


void vala_ccode_writer_set_filename (ValaCCodeWriter* self, const char* value) {
	char* _tmp1_;
	const char* _tmp0_;
	g_return_if_fail (self != NULL);
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	self->priv->_filename = (_tmp1_ = (_tmp0_ = value, (_tmp0_ == NULL) ? NULL : g_strdup (_tmp0_)), self->priv->_filename = (g_free (self->priv->_filename), NULL), _tmp1_);
}


gboolean vala_ccode_writer_get_line_directives (ValaCCodeWriter* self) {
	gboolean result;
	g_return_val_if_fail (self != NULL, FALSE);
	result = self->priv->_line_directives;
	return result;
}


void vala_ccode_writer_set_line_directives (ValaCCodeWriter* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_line_directives = value;
}


gboolean vala_ccode_writer_get_bol (ValaCCodeWriter* self) {
	gboolean result;
	g_return_val_if_fail (self != NULL, FALSE);
	result = self->priv->_bol;
	return result;
}


static void vala_value_ccode_writer_init (GValue* value) {
	value->data[0].v_pointer = NULL;
}


static void vala_value_ccode_writer_free_value (GValue* value) {
	if (value->data[0].v_pointer) {
		vala_ccode_writer_unref (value->data[0].v_pointer);
	}
}


static void vala_value_ccode_writer_copy_value (const GValue* src_value, GValue* dest_value) {
	if (src_value->data[0].v_pointer) {
		dest_value->data[0].v_pointer = vala_ccode_writer_ref (src_value->data[0].v_pointer);
	} else {
		dest_value->data[0].v_pointer = NULL;
	}
}


static gpointer vala_value_ccode_writer_peek_pointer (const GValue* value) {
	return value->data[0].v_pointer;
}


static gchar* vala_value_ccode_writer_collect_value (GValue* value, guint n_collect_values, GTypeCValue* collect_values, guint collect_flags) {
	if (collect_values[0].v_pointer) {
		ValaCCodeWriter* object;
		object = collect_values[0].v_pointer;
		if (object->parent_instance.g_class == NULL) {
			return g_strconcat ("invalid unclassed object pointer for value type `", G_VALUE_TYPE_NAME (value), "'", NULL);
		} else if (!g_value_type_compatible (G_TYPE_FROM_INSTANCE (object), G_VALUE_TYPE (value))) {
			return g_strconcat ("invalid object type `", g_type_name (G_TYPE_FROM_INSTANCE (object)), "' for value type `", G_VALUE_TYPE_NAME (value), "'", NULL);
		}
		value->data[0].v_pointer = vala_ccode_writer_ref (object);
	} else {
		value->data[0].v_pointer = NULL;
	}
	return NULL;
}


static gchar* vala_value_ccode_writer_lcopy_value (const GValue* value, guint n_collect_values, GTypeCValue* collect_values, guint collect_flags) {
	ValaCCodeWriter** object_p;
	object_p = collect_values[0].v_pointer;
	if (!object_p) {
		return g_strdup_printf ("value location for `%s' passed as NULL", G_VALUE_TYPE_NAME (value));
	}
	if (!value->data[0].v_pointer) {
		*object_p = NULL;
	} else if (collect_flags && G_VALUE_NOCOPY_CONTENTS) {
		*object_p = value->data[0].v_pointer;
	} else {
		*object_p = vala_ccode_writer_ref (value->data[0].v_pointer);
	}
	return NULL;
}


GParamSpec* vala_param_spec_ccode_writer (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags) {
	ValaParamSpecCCodeWriter* spec;
	g_return_val_if_fail (g_type_is_a (object_type, VALA_TYPE_CCODE_WRITER), NULL);
	spec = g_param_spec_internal (G_TYPE_PARAM_OBJECT, name, nick, blurb, flags);
	G_PARAM_SPEC (spec)->value_type = object_type;
	return G_PARAM_SPEC (spec);
}


gpointer vala_value_get_ccode_writer (const GValue* value) {
	g_return_val_if_fail (G_TYPE_CHECK_VALUE_TYPE (value, VALA_TYPE_CCODE_WRITER), NULL);
	return value->data[0].v_pointer;
}


void vala_value_set_ccode_writer (GValue* value, gpointer v_object) {
	ValaCCodeWriter* old;
	g_return_if_fail (G_TYPE_CHECK_VALUE_TYPE (value, VALA_TYPE_CCODE_WRITER));
	old = value->data[0].v_pointer;
	if (v_object) {
		g_return_if_fail (G_TYPE_CHECK_INSTANCE_TYPE (v_object, VALA_TYPE_CCODE_WRITER));
		g_return_if_fail (g_value_type_compatible (G_TYPE_FROM_INSTANCE (v_object), G_VALUE_TYPE (value)));
		value->data[0].v_pointer = v_object;
		vala_ccode_writer_ref (value->data[0].v_pointer);
	} else {
		value->data[0].v_pointer = NULL;
	}
	if (old) {
		vala_ccode_writer_unref (old);
	}
}


static void vala_ccode_writer_class_init (ValaCCodeWriterClass * klass) {
	vala_ccode_writer_parent_class = g_type_class_peek_parent (klass);
	VALA_CCODE_WRITER_CLASS (klass)->finalize = vala_ccode_writer_finalize;
	g_type_class_add_private (klass, sizeof (ValaCCodeWriterPrivate));
}


static void vala_ccode_writer_instance_init (ValaCCodeWriter * self) {
	self->priv = VALA_CCODE_WRITER_GET_PRIVATE (self);
	self->priv->current_line_number = 1;
	self->priv->_bol = TRUE;
	self->ref_count = 1;
}


static void vala_ccode_writer_finalize (ValaCCodeWriter* obj) {
	ValaCCodeWriter * self;
	self = VALA_CCODE_WRITER (obj);
	self->priv->_filename = (g_free (self->priv->_filename), NULL);
	self->priv->temp_filename = (g_free (self->priv->temp_filename), NULL);
	(self->priv->stream == NULL) ? NULL : (self->priv->stream = (fclose (self->priv->stream), NULL));
}


GType vala_ccode_writer_get_type (void) {
	static GType vala_ccode_writer_type_id = 0;
	if (vala_ccode_writer_type_id == 0) {
		static const GTypeValueTable g_define_type_value_table = { vala_value_ccode_writer_init, vala_value_ccode_writer_free_value, vala_value_ccode_writer_copy_value, vala_value_ccode_writer_peek_pointer, "p", vala_value_ccode_writer_collect_value, "p", vala_value_ccode_writer_lcopy_value };
		static const GTypeInfo g_define_type_info = { sizeof (ValaCCodeWriterClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_ccode_writer_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaCCodeWriter), 0, (GInstanceInitFunc) vala_ccode_writer_instance_init, &g_define_type_value_table };
		static const GTypeFundamentalInfo g_define_type_fundamental_info = { (G_TYPE_FLAG_CLASSED | G_TYPE_FLAG_INSTANTIATABLE | G_TYPE_FLAG_DERIVABLE | G_TYPE_FLAG_DEEP_DERIVABLE) };
		vala_ccode_writer_type_id = g_type_register_fundamental (g_type_fundamental_next (), "ValaCCodeWriter", &g_define_type_info, &g_define_type_fundamental_info, 0);
	}
	return vala_ccode_writer_type_id;
}


gpointer vala_ccode_writer_ref (gpointer instance) {
	ValaCCodeWriter* self;
	self = instance;
	g_atomic_int_inc (&self->ref_count);
	return instance;
}


void vala_ccode_writer_unref (gpointer instance) {
	ValaCCodeWriter* self;
	self = instance;
	if (g_atomic_int_dec_and_test (&self->ref_count)) {
		VALA_CCODE_WRITER_GET_CLASS (self)->finalize (self);
		g_type_free_instance ((GTypeInstance *) self);
	}
}


static void _vala_array_destroy (gpointer array, gint array_length, GDestroyNotify destroy_func) {
	if ((array != NULL) && (destroy_func != NULL)) {
		int i;
		for (i = 0; i < array_length; i = i + 1) {
			if (((gpointer*) array)[i] != NULL) {
				destroy_func (((gpointer*) array)[i]);
			}
		}
	}
}


static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func) {
	_vala_array_destroy (array, array_length, destroy_func);
	g_free (array);
}


static gint _vala_array_length (gpointer array) {
	int length;
	length = 0;
	if (array) {
		while (((gpointer*) array)[length]) {
			length++;
		}
	}
	return length;
}




