/* valanamespace.vala
 *
 * Copyright (C) 2006-2009  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <glib.h>
#include <glib-object.h>
#include <stdlib.h>
#include <string.h>
#include <gee.h>


#define VALA_TYPE_CODE_NODE (vala_code_node_get_type ())
#define VALA_CODE_NODE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CODE_NODE, ValaCodeNode))
#define VALA_CODE_NODE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CODE_NODE, ValaCodeNodeClass))
#define VALA_IS_CODE_NODE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CODE_NODE))
#define VALA_IS_CODE_NODE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CODE_NODE))
#define VALA_CODE_NODE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CODE_NODE, ValaCodeNodeClass))

typedef struct _ValaCodeNode ValaCodeNode;
typedef struct _ValaCodeNodeClass ValaCodeNodeClass;
typedef struct _ValaCodeNodePrivate ValaCodeNodePrivate;

#define VALA_TYPE_CODE_VISITOR (vala_code_visitor_get_type ())
#define VALA_CODE_VISITOR(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CODE_VISITOR, ValaCodeVisitor))
#define VALA_CODE_VISITOR_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CODE_VISITOR, ValaCodeVisitorClass))
#define VALA_IS_CODE_VISITOR(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CODE_VISITOR))
#define VALA_IS_CODE_VISITOR_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CODE_VISITOR))
#define VALA_CODE_VISITOR_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CODE_VISITOR, ValaCodeVisitorClass))

typedef struct _ValaCodeVisitor ValaCodeVisitor;
typedef struct _ValaCodeVisitorClass ValaCodeVisitorClass;

#define VALA_TYPE_SEMANTIC_ANALYZER (vala_semantic_analyzer_get_type ())
#define VALA_SEMANTIC_ANALYZER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SEMANTIC_ANALYZER, ValaSemanticAnalyzer))
#define VALA_SEMANTIC_ANALYZER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SEMANTIC_ANALYZER, ValaSemanticAnalyzerClass))
#define VALA_IS_SEMANTIC_ANALYZER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SEMANTIC_ANALYZER))
#define VALA_IS_SEMANTIC_ANALYZER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SEMANTIC_ANALYZER))
#define VALA_SEMANTIC_ANALYZER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SEMANTIC_ANALYZER, ValaSemanticAnalyzerClass))

typedef struct _ValaSemanticAnalyzer ValaSemanticAnalyzer;
typedef struct _ValaSemanticAnalyzerClass ValaSemanticAnalyzerClass;

#define VALA_TYPE_DATA_TYPE (vala_data_type_get_type ())
#define VALA_DATA_TYPE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_DATA_TYPE, ValaDataType))
#define VALA_DATA_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_DATA_TYPE, ValaDataTypeClass))
#define VALA_IS_DATA_TYPE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_DATA_TYPE))
#define VALA_IS_DATA_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_DATA_TYPE))
#define VALA_DATA_TYPE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_DATA_TYPE, ValaDataTypeClass))

typedef struct _ValaDataType ValaDataType;
typedef struct _ValaDataTypeClass ValaDataTypeClass;

#define VALA_TYPE_EXPRESSION (vala_expression_get_type ())
#define VALA_EXPRESSION(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_EXPRESSION, ValaExpression))
#define VALA_EXPRESSION_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_EXPRESSION, ValaExpressionClass))
#define VALA_IS_EXPRESSION(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_EXPRESSION))
#define VALA_IS_EXPRESSION_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_EXPRESSION))
#define VALA_EXPRESSION_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_EXPRESSION, ValaExpressionClass))

typedef struct _ValaExpression ValaExpression;
typedef struct _ValaExpressionClass ValaExpressionClass;

#define VALA_TYPE_SYMBOL (vala_symbol_get_type ())
#define VALA_SYMBOL(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SYMBOL, ValaSymbol))
#define VALA_SYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SYMBOL, ValaSymbolClass))
#define VALA_IS_SYMBOL(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SYMBOL))
#define VALA_IS_SYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SYMBOL))
#define VALA_SYMBOL_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SYMBOL, ValaSymbolClass))

typedef struct _ValaSymbol ValaSymbol;
typedef struct _ValaSymbolClass ValaSymbolClass;

#define VALA_TYPE_LOCAL_VARIABLE (vala_local_variable_get_type ())
#define VALA_LOCAL_VARIABLE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_LOCAL_VARIABLE, ValaLocalVariable))
#define VALA_LOCAL_VARIABLE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_LOCAL_VARIABLE, ValaLocalVariableClass))
#define VALA_IS_LOCAL_VARIABLE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_LOCAL_VARIABLE))
#define VALA_IS_LOCAL_VARIABLE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_LOCAL_VARIABLE))
#define VALA_LOCAL_VARIABLE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_LOCAL_VARIABLE, ValaLocalVariableClass))

typedef struct _ValaLocalVariable ValaLocalVariable;
typedef struct _ValaLocalVariableClass ValaLocalVariableClass;

#define VALA_TYPE_ATTRIBUTE (vala_attribute_get_type ())
#define VALA_ATTRIBUTE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_ATTRIBUTE, ValaAttribute))
#define VALA_ATTRIBUTE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_ATTRIBUTE, ValaAttributeClass))
#define VALA_IS_ATTRIBUTE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_ATTRIBUTE))
#define VALA_IS_ATTRIBUTE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_ATTRIBUTE))
#define VALA_ATTRIBUTE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_ATTRIBUTE, ValaAttributeClass))

typedef struct _ValaAttribute ValaAttribute;
typedef struct _ValaAttributeClass ValaAttributeClass;
typedef struct _ValaSymbolPrivate ValaSymbolPrivate;

#define VALA_TYPE_NAMESPACE (vala_namespace_get_type ())
#define VALA_NAMESPACE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_NAMESPACE, ValaNamespace))
#define VALA_NAMESPACE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_NAMESPACE, ValaNamespaceClass))
#define VALA_IS_NAMESPACE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_NAMESPACE))
#define VALA_IS_NAMESPACE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_NAMESPACE))
#define VALA_NAMESPACE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_NAMESPACE, ValaNamespaceClass))

typedef struct _ValaNamespace ValaNamespace;
typedef struct _ValaNamespaceClass ValaNamespaceClass;
typedef struct _ValaNamespacePrivate ValaNamespacePrivate;

#define VALA_TYPE_TYPESYMBOL (vala_typesymbol_get_type ())
#define VALA_TYPESYMBOL(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_TYPESYMBOL, ValaTypeSymbol))
#define VALA_TYPESYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_TYPESYMBOL, ValaTypeSymbolClass))
#define VALA_IS_TYPESYMBOL(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_TYPESYMBOL))
#define VALA_IS_TYPESYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_TYPESYMBOL))
#define VALA_TYPESYMBOL_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_TYPESYMBOL, ValaTypeSymbolClass))

typedef struct _ValaTypeSymbol ValaTypeSymbol;
typedef struct _ValaTypeSymbolClass ValaTypeSymbolClass;

#define VALA_TYPE_OBJECT_TYPE_SYMBOL (vala_object_type_symbol_get_type ())
#define VALA_OBJECT_TYPE_SYMBOL(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_OBJECT_TYPE_SYMBOL, ValaObjectTypeSymbol))
#define VALA_OBJECT_TYPE_SYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_OBJECT_TYPE_SYMBOL, ValaObjectTypeSymbolClass))
#define VALA_IS_OBJECT_TYPE_SYMBOL(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_OBJECT_TYPE_SYMBOL))
#define VALA_IS_OBJECT_TYPE_SYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_OBJECT_TYPE_SYMBOL))
#define VALA_OBJECT_TYPE_SYMBOL_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_OBJECT_TYPE_SYMBOL, ValaObjectTypeSymbolClass))

typedef struct _ValaObjectTypeSymbol ValaObjectTypeSymbol;
typedef struct _ValaObjectTypeSymbolClass ValaObjectTypeSymbolClass;

#define VALA_TYPE_CLASS (vala_class_get_type ())
#define VALA_CLASS(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CLASS, ValaClass))
#define VALA_CLASS_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CLASS, ValaClassClass))
#define VALA_IS_CLASS(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CLASS))
#define VALA_IS_CLASS_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CLASS))
#define VALA_CLASS_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CLASS, ValaClassClass))

typedef struct _ValaClass ValaClass;
typedef struct _ValaClassClass ValaClassClass;

#define VALA_TYPE_INTERFACE (vala_interface_get_type ())
#define VALA_INTERFACE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_INTERFACE, ValaInterface))
#define VALA_INTERFACE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_INTERFACE, ValaInterfaceClass))
#define VALA_IS_INTERFACE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_INTERFACE))
#define VALA_IS_INTERFACE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_INTERFACE))
#define VALA_INTERFACE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_INTERFACE, ValaInterfaceClass))

typedef struct _ValaInterface ValaInterface;
typedef struct _ValaInterfaceClass ValaInterfaceClass;

#define VALA_TYPE_STRUCT (vala_struct_get_type ())
#define VALA_STRUCT(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_STRUCT, ValaStruct))
#define VALA_STRUCT_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_STRUCT, ValaStructClass))
#define VALA_IS_STRUCT(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_STRUCT))
#define VALA_IS_STRUCT_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_STRUCT))
#define VALA_STRUCT_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_STRUCT, ValaStructClass))

typedef struct _ValaStruct ValaStruct;
typedef struct _ValaStructClass ValaStructClass;

#define VALA_TYPE_ENUM (vala_enum_get_type ())
#define VALA_ENUM(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_ENUM, ValaEnum))
#define VALA_ENUM_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_ENUM, ValaEnumClass))
#define VALA_IS_ENUM(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_ENUM))
#define VALA_IS_ENUM_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_ENUM))
#define VALA_ENUM_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_ENUM, ValaEnumClass))

typedef struct _ValaEnum ValaEnum;
typedef struct _ValaEnumClass ValaEnumClass;

#define VALA_TYPE_ERROR_DOMAIN (vala_error_domain_get_type ())
#define VALA_ERROR_DOMAIN(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_ERROR_DOMAIN, ValaErrorDomain))
#define VALA_ERROR_DOMAIN_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_ERROR_DOMAIN, ValaErrorDomainClass))
#define VALA_IS_ERROR_DOMAIN(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_ERROR_DOMAIN))
#define VALA_IS_ERROR_DOMAIN_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_ERROR_DOMAIN))
#define VALA_ERROR_DOMAIN_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_ERROR_DOMAIN, ValaErrorDomainClass))

typedef struct _ValaErrorDomain ValaErrorDomain;
typedef struct _ValaErrorDomainClass ValaErrorDomainClass;

#define VALA_TYPE_DELEGATE (vala_delegate_get_type ())
#define VALA_DELEGATE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_DELEGATE, ValaDelegate))
#define VALA_DELEGATE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_DELEGATE, ValaDelegateClass))
#define VALA_IS_DELEGATE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_DELEGATE))
#define VALA_IS_DELEGATE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_DELEGATE))
#define VALA_DELEGATE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_DELEGATE, ValaDelegateClass))

typedef struct _ValaDelegate ValaDelegate;
typedef struct _ValaDelegateClass ValaDelegateClass;

#define VALA_TYPE_MEMBER (vala_member_get_type ())
#define VALA_MEMBER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_MEMBER, ValaMember))
#define VALA_MEMBER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_MEMBER, ValaMemberClass))
#define VALA_IS_MEMBER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_MEMBER))
#define VALA_IS_MEMBER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_MEMBER))
#define VALA_MEMBER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_MEMBER, ValaMemberClass))

typedef struct _ValaMember ValaMember;
typedef struct _ValaMemberClass ValaMemberClass;

#define VALA_TYPE_CONSTANT (vala_constant_get_type ())
#define VALA_CONSTANT(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CONSTANT, ValaConstant))
#define VALA_CONSTANT_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CONSTANT, ValaConstantClass))
#define VALA_IS_CONSTANT(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CONSTANT))
#define VALA_IS_CONSTANT_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CONSTANT))
#define VALA_CONSTANT_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CONSTANT, ValaConstantClass))

typedef struct _ValaConstant ValaConstant;
typedef struct _ValaConstantClass ValaConstantClass;

#define VALA_TYPE_FIELD (vala_field_get_type ())
#define VALA_FIELD(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_FIELD, ValaField))
#define VALA_FIELD_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_FIELD, ValaFieldClass))
#define VALA_IS_FIELD(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_FIELD))
#define VALA_IS_FIELD_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_FIELD))
#define VALA_FIELD_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_FIELD, ValaFieldClass))

typedef struct _ValaField ValaField;
typedef struct _ValaFieldClass ValaFieldClass;

#define VALA_TYPE_METHOD (vala_method_get_type ())
#define VALA_METHOD(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_METHOD, ValaMethod))
#define VALA_METHOD_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_METHOD, ValaMethodClass))
#define VALA_IS_METHOD(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_METHOD))
#define VALA_IS_METHOD_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_METHOD))
#define VALA_METHOD_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_METHOD, ValaMethodClass))

typedef struct _ValaMethod ValaMethod;
typedef struct _ValaMethodClass ValaMethodClass;

#define VALA_TYPE_SOURCE_REFERENCE (vala_source_reference_get_type ())
#define VALA_SOURCE_REFERENCE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SOURCE_REFERENCE, ValaSourceReference))
#define VALA_SOURCE_REFERENCE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SOURCE_REFERENCE, ValaSourceReferenceClass))
#define VALA_IS_SOURCE_REFERENCE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SOURCE_REFERENCE))
#define VALA_IS_SOURCE_REFERENCE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SOURCE_REFERENCE))
#define VALA_SOURCE_REFERENCE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SOURCE_REFERENCE, ValaSourceReferenceClass))

typedef struct _ValaSourceReference ValaSourceReference;
typedef struct _ValaSourceReferenceClass ValaSourceReferenceClass;

#define VALA_TYPE_SYMBOL_ACCESSIBILITY (vala_symbol_accessibility_get_type ())

#define VALA_TYPE_SCOPE (vala_scope_get_type ())
#define VALA_SCOPE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SCOPE, ValaScope))
#define VALA_SCOPE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SCOPE, ValaScopeClass))
#define VALA_IS_SCOPE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SCOPE))
#define VALA_IS_SCOPE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SCOPE))
#define VALA_SCOPE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SCOPE, ValaScopeClass))

typedef struct _ValaScope ValaScope;
typedef struct _ValaScopeClass ValaScopeClass;

#define TYPE_MEMBER_BINDING (member_binding_get_type ())

#define VALA_TYPE_CREATION_METHOD (vala_creation_method_get_type ())
#define VALA_CREATION_METHOD(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CREATION_METHOD, ValaCreationMethod))
#define VALA_CREATION_METHOD_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CREATION_METHOD, ValaCreationMethodClass))
#define VALA_IS_CREATION_METHOD(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CREATION_METHOD))
#define VALA_IS_CREATION_METHOD_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CREATION_METHOD))
#define VALA_CREATION_METHOD_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CREATION_METHOD, ValaCreationMethodClass))

typedef struct _ValaCreationMethod ValaCreationMethod;
typedef struct _ValaCreationMethodClass ValaCreationMethodClass;

/**
 * Represents a part of the parsed source code.
 *
 * Code nodes get created by the parser and are used throughout the whole
 * compilation process.
 */
struct _ValaCodeNode {
	GTypeInstance parent_instance;
	volatile int ref_count;
	ValaCodeNodePrivate * priv;
	GList* attributes;
};

struct _ValaCodeNodeClass {
	GTypeClass parent_class;
	void (*finalize) (ValaCodeNode *self);
	void (*accept) (ValaCodeNode* self, ValaCodeVisitor* visitor);
	void (*accept_children) (ValaCodeNode* self, ValaCodeVisitor* visitor);
	gboolean (*check) (ValaCodeNode* self, ValaSemanticAnalyzer* analyzer);
	void (*replace_type) (ValaCodeNode* self, ValaDataType* old_type, ValaDataType* new_type);
	void (*replace_expression) (ValaCodeNode* self, ValaExpression* old_node, ValaExpression* new_node);
	char* (*to_string) (ValaCodeNode* self);
	void (*get_defined_variables) (ValaCodeNode* self, GeeCollection* collection);
	void (*get_used_variables) (ValaCodeNode* self, GeeCollection* collection);
};

/**
 * Represents a node in the symbol tree.
 */
struct _ValaSymbol {
	ValaCodeNode parent_instance;
	ValaSymbolPrivate * priv;
};

struct _ValaSymbolClass {
	ValaCodeNodeClass parent_class;
	char* (*get_cprefix) (ValaSymbol* self);
	char* (*get_lower_case_cname) (ValaSymbol* self, const char* infix);
	char* (*get_lower_case_cprefix) (ValaSymbol* self);
	GeeList* (*get_cheader_filenames) (ValaSymbol* self);
};

/**
 * Represents a namespace declaration in the source code.
 */
struct _ValaNamespace {
	ValaSymbol parent_instance;
	ValaNamespacePrivate * priv;
};

struct _ValaNamespaceClass {
	ValaSymbolClass parent_class;
};

struct _ValaNamespacePrivate {
	GeeList* classes;
	GeeList* interfaces;
	GeeList* structs;
	GeeList* enums;
	GeeList* error_domains;
	GeeList* delegates;
	GeeList* constants;
	GeeList* fields;
	GeeList* methods;
	GeeList* cprefixes;
	char* lower_case_cprefix;
	GeeList* cheader_filenames;
	GeeList* namespaces;
};

typedef enum  {
	VALA_SYMBOL_ACCESSIBILITY_PRIVATE,
	VALA_SYMBOL_ACCESSIBILITY_INTERNAL,
	VALA_SYMBOL_ACCESSIBILITY_PROTECTED,
	VALA_SYMBOL_ACCESSIBILITY_PUBLIC
} ValaSymbolAccessibility;

typedef enum  {
	MEMBER_BINDING_INSTANCE,
	MEMBER_BINDING_CLASS,
	MEMBER_BINDING_STATIC
} MemberBinding;


static gpointer vala_namespace_parent_class = NULL;

gpointer vala_code_node_ref (gpointer instance);
void vala_code_node_unref (gpointer instance);
GParamSpec* vala_param_spec_code_node (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_code_node (GValue* value, gpointer v_object);
gpointer vala_value_get_code_node (const GValue* value);
GType vala_code_node_get_type (void);
gpointer vala_code_visitor_ref (gpointer instance);
void vala_code_visitor_unref (gpointer instance);
GParamSpec* vala_param_spec_code_visitor (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_code_visitor (GValue* value, gpointer v_object);
gpointer vala_value_get_code_visitor (const GValue* value);
GType vala_code_visitor_get_type (void);
GType vala_semantic_analyzer_get_type (void);
GType vala_data_type_get_type (void);
GType vala_expression_get_type (void);
GType vala_symbol_get_type (void);
GType vala_local_variable_get_type (void);
GType vala_attribute_get_type (void);
GType vala_namespace_get_type (void);
GType vala_typesymbol_get_type (void);
GType vala_object_type_symbol_get_type (void);
GType vala_class_get_type (void);
GType vala_interface_get_type (void);
GType vala_struct_get_type (void);
GType vala_enum_get_type (void);
GType vala_error_domain_get_type (void);
GType vala_delegate_get_type (void);
GType vala_member_get_type (void);
GType vala_constant_get_type (void);
GType vala_field_get_type (void);
GType vala_method_get_type (void);
#define VALA_NAMESPACE_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_NAMESPACE, ValaNamespacePrivate))
enum  {
	VALA_NAMESPACE_DUMMY_PROPERTY
};
gpointer vala_source_reference_ref (gpointer instance);
void vala_source_reference_unref (gpointer instance);
GParamSpec* vala_param_spec_source_reference (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_source_reference (GValue* value, gpointer v_object);
gpointer vala_value_get_source_reference (const GValue* value);
GType vala_source_reference_get_type (void);
ValaSymbol* vala_symbol_new (const char* name, ValaSourceReference* source_reference);
ValaSymbol* vala_symbol_construct (GType object_type, const char* name, ValaSourceReference* source_reference);
GType vala_symbol_accessibility_get_type (void);
void vala_symbol_set_access (ValaSymbol* self, ValaSymbolAccessibility value);
ValaNamespace* vala_namespace_new (const char* name, ValaSourceReference* source_reference);
ValaNamespace* vala_namespace_construct (GType object_type, const char* name, ValaSourceReference* source_reference);
ValaNamespace* vala_namespace_new (const char* name, ValaSourceReference* source_reference);
gpointer vala_scope_ref (gpointer instance);
void vala_scope_unref (gpointer instance);
GParamSpec* vala_param_spec_scope (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_scope (GValue* value, gpointer v_object);
gpointer vala_value_get_scope (const GValue* value);
GType vala_scope_get_type (void);
ValaScope* vala_symbol_get_scope (ValaSymbol* self);
ValaSymbol* vala_scope_lookup (ValaScope* self, const char* name);
const char* vala_symbol_get_name (ValaSymbol* self);
gboolean vala_symbol_get_external_package (ValaSymbol* self);
ValaSourceReference* vala_code_node_get_source_reference (ValaCodeNode* self);
void vala_code_node_set_source_reference (ValaCodeNode* self, ValaSourceReference* value);
GeeList* vala_namespace_get_namespaces (ValaNamespace* self);
void vala_namespace_add_namespace (ValaNamespace* self, ValaNamespace* ns);
GeeList* vala_namespace_get_classes (ValaNamespace* self);
void vala_namespace_add_class (ValaNamespace* self, ValaClass* cl);
GeeList* vala_namespace_get_structs (ValaNamespace* self);
void vala_namespace_add_struct (ValaNamespace* self, ValaStruct* st);
GeeList* vala_namespace_get_interfaces (ValaNamespace* self);
void vala_namespace_add_interface (ValaNamespace* self, ValaInterface* iface);
GeeList* vala_namespace_get_delegates (ValaNamespace* self);
void vala_namespace_add_delegate (ValaNamespace* self, ValaDelegate* d);
GeeList* vala_namespace_get_enums (ValaNamespace* self);
void vala_namespace_add_enum (ValaNamespace* self, ValaEnum* en);
GeeList* vala_namespace_get_error_domains (ValaNamespace* self);
void vala_namespace_add_error_domain (ValaNamespace* self, ValaErrorDomain* edomain);
GeeList* vala_namespace_get_constants (ValaNamespace* self);
void vala_namespace_add_constant (ValaNamespace* self, ValaConstant* constant);
GeeList* vala_namespace_get_fields (ValaNamespace* self);
void vala_namespace_add_field (ValaNamespace* self, ValaField* f);
GeeList* vala_namespace_get_methods (ValaNamespace* self);
void vala_namespace_add_method (ValaNamespace* self, ValaMethod* m);
void vala_scope_add (ValaScope* self, const char* name, ValaSymbol* sym);
ValaSymbolAccessibility vala_symbol_get_access (ValaSymbol* self);
void vala_scope_remove (ValaScope* self, const char* name);
void vala_namespace_remove_struct (ValaNamespace* self, ValaStruct* st);
GType member_binding_get_type (void);
MemberBinding vala_field_get_binding (ValaField* self);
void vala_report_error (ValaSourceReference* source, const char* message);
void vala_code_node_set_error (ValaCodeNode* self, gboolean value);
GType vala_creation_method_get_type (void);
MemberBinding vala_method_get_binding (ValaMethod* self);
void vala_code_visitor_visit_namespace (ValaCodeVisitor* self, ValaNamespace* ns);
static void vala_namespace_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
void vala_code_node_accept (ValaCodeNode* self, ValaCodeVisitor* visitor);
static void vala_namespace_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
ValaSymbol* vala_symbol_get_parent_symbol (ValaSymbol* self);
char* vala_symbol_get_cprefix (ValaSymbol* self);
static char* vala_namespace_real_get_cprefix (ValaSymbol* base);
GeeList* vala_namespace_get_cprefixes (ValaNamespace* self);
void vala_namespace_add_cprefix (ValaNamespace* self, const char* cprefix);
char* vala_symbol_get_lower_case_cprefix (ValaSymbol* self);
char* vala_symbol_camel_case_to_lower_case (const char* camel_case);
static char* vala_namespace_real_get_lower_case_cprefix (ValaSymbol* base);
void vala_namespace_set_lower_case_cprefix (ValaNamespace* self, const char* cprefix);
static GeeList* vala_namespace_real_get_cheader_filenames (ValaSymbol* base);
void vala_namespace_set_cheader_filename (ValaNamespace* self, const char* cheader_filename);
void vala_namespace_add_cheader_filename (ValaNamespace* self, const char* cheader_filename);
gboolean vala_attribute_has_argument (ValaAttribute* self, const char* name);
char* vala_attribute_get_string (ValaAttribute* self, const char* name);
static void vala_namespace_process_ccode_attribute (ValaNamespace* self, ValaAttribute* a);
const char* vala_attribute_get_name (ValaAttribute* self);
void vala_namespace_process_attributes (ValaNamespace* self);
gboolean vala_code_node_get_checked (ValaCodeNode* self);
gboolean vala_code_node_get_error (ValaCodeNode* self);
void vala_code_node_set_checked (ValaCodeNode* self, gboolean value);
gboolean vala_code_node_check (ValaCodeNode* self, ValaSemanticAnalyzer* analyzer);
static gboolean vala_namespace_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer);
static void vala_namespace_finalize (ValaCodeNode* obj);
static void _vala_array_destroy (gpointer array, gint array_length, GDestroyNotify destroy_func);
static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func);
static gint _vala_array_length (gpointer array);
static int _vala_strcmp0 (const char * str1, const char * str2);



/**
 * Creates a new namespace.
 *
 * @param name             namespace name
 * @param source_reference reference to source code
 * @return                 newly created namespace
 */
ValaNamespace* vala_namespace_construct (GType object_type, const char* name, ValaSourceReference* source_reference) {
	ValaNamespace* self;
	self = (ValaNamespace*) vala_symbol_construct (object_type, name, source_reference);
	vala_symbol_set_access ((ValaSymbol*) self, VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
	return self;
}


ValaNamespace* vala_namespace_new (const char* name, ValaSourceReference* source_reference) {
	return vala_namespace_construct (VALA_TYPE_NAMESPACE, name, source_reference);
}


/**
 * Adds the specified namespace to this source file.
 *
 * @param ns a namespace
 */
void vala_namespace_add_namespace (ValaNamespace* self, ValaNamespace* ns) {
	ValaSymbol* _tmp0_;
	gboolean _tmp1_;
	g_return_if_fail (self != NULL);
	g_return_if_fail (ns != NULL);
	_tmp0_ = NULL;
	if ((_tmp1_ = VALA_IS_NAMESPACE (_tmp0_ = vala_scope_lookup (vala_symbol_get_scope ((ValaSymbol*) self), vala_symbol_get_name ((ValaSymbol*) ns))), (_tmp0_ == NULL) ? NULL : (_tmp0_ = (vala_code_node_unref (_tmp0_), NULL)), _tmp1_)) {
		ValaNamespace* old_ns;
		gboolean _tmp2_;
		/* merge if namespace already exists*/
		old_ns = VALA_NAMESPACE (vala_scope_lookup (vala_symbol_get_scope ((ValaSymbol*) self), vala_symbol_get_name ((ValaSymbol*) ns)));
		_tmp2_ = FALSE;
		if (vala_symbol_get_external_package ((ValaSymbol*) old_ns)) {
			_tmp2_ = !vala_symbol_get_external_package ((ValaSymbol*) ns);
		} else {
			_tmp2_ = FALSE;
		}
		if (_tmp2_) {
			vala_code_node_set_source_reference ((ValaCodeNode*) old_ns, vala_code_node_get_source_reference ((ValaCodeNode*) ns));
		}
		{
			GeeList* _tmp3_;
			GeeIterator* _tmp4_;
			GeeIterator* _sub_ns_it;
			_tmp3_ = NULL;
			_tmp4_ = NULL;
			_sub_ns_it = (_tmp4_ = gee_iterable_iterator ((GeeIterable*) (_tmp3_ = vala_namespace_get_namespaces (ns))), (_tmp3_ == NULL) ? NULL : (_tmp3_ = (gee_collection_object_unref (_tmp3_), NULL)), _tmp4_);
			while (TRUE) {
				ValaNamespace* sub_ns;
				if (!gee_iterator_next (_sub_ns_it)) {
					break;
				}
				sub_ns = (ValaNamespace*) gee_iterator_get (_sub_ns_it);
				vala_namespace_add_namespace (old_ns, sub_ns);
				(sub_ns == NULL) ? NULL : (sub_ns = (vala_code_node_unref (sub_ns), NULL));
			}
			(_sub_ns_it == NULL) ? NULL : (_sub_ns_it = (gee_collection_object_unref (_sub_ns_it), NULL));
		}
		{
			GeeList* _tmp5_;
			GeeIterator* _tmp6_;
			GeeIterator* _cl_it;
			_tmp5_ = NULL;
			_tmp6_ = NULL;
			_cl_it = (_tmp6_ = gee_iterable_iterator ((GeeIterable*) (_tmp5_ = vala_namespace_get_classes (ns))), (_tmp5_ == NULL) ? NULL : (_tmp5_ = (gee_collection_object_unref (_tmp5_), NULL)), _tmp6_);
			while (TRUE) {
				ValaClass* cl;
				if (!gee_iterator_next (_cl_it)) {
					break;
				}
				cl = (ValaClass*) gee_iterator_get (_cl_it);
				vala_namespace_add_class (old_ns, cl);
				(cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL));
			}
			(_cl_it == NULL) ? NULL : (_cl_it = (gee_collection_object_unref (_cl_it), NULL));
		}
		{
			GeeList* _tmp7_;
			GeeIterator* _tmp8_;
			GeeIterator* _st_it;
			_tmp7_ = NULL;
			_tmp8_ = NULL;
			_st_it = (_tmp8_ = gee_iterable_iterator ((GeeIterable*) (_tmp7_ = vala_namespace_get_structs (ns))), (_tmp7_ == NULL) ? NULL : (_tmp7_ = (gee_collection_object_unref (_tmp7_), NULL)), _tmp8_);
			while (TRUE) {
				ValaStruct* st;
				if (!gee_iterator_next (_st_it)) {
					break;
				}
				st = (ValaStruct*) gee_iterator_get (_st_it);
				vala_namespace_add_struct (old_ns, st);
				(st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL));
			}
			(_st_it == NULL) ? NULL : (_st_it = (gee_collection_object_unref (_st_it), NULL));
		}
		{
			GeeList* _tmp9_;
			GeeIterator* _tmp10_;
			GeeIterator* _iface_it;
			_tmp9_ = NULL;
			_tmp10_ = NULL;
			_iface_it = (_tmp10_ = gee_iterable_iterator ((GeeIterable*) (_tmp9_ = vala_namespace_get_interfaces (ns))), (_tmp9_ == NULL) ? NULL : (_tmp9_ = (gee_collection_object_unref (_tmp9_), NULL)), _tmp10_);
			while (TRUE) {
				ValaInterface* iface;
				if (!gee_iterator_next (_iface_it)) {
					break;
				}
				iface = (ValaInterface*) gee_iterator_get (_iface_it);
				vala_namespace_add_interface (old_ns, iface);
				(iface == NULL) ? NULL : (iface = (vala_code_node_unref (iface), NULL));
			}
			(_iface_it == NULL) ? NULL : (_iface_it = (gee_collection_object_unref (_iface_it), NULL));
		}
		{
			GeeList* _tmp11_;
			GeeIterator* _tmp12_;
			GeeIterator* _d_it;
			_tmp11_ = NULL;
			_tmp12_ = NULL;
			_d_it = (_tmp12_ = gee_iterable_iterator ((GeeIterable*) (_tmp11_ = vala_namespace_get_delegates (ns))), (_tmp11_ == NULL) ? NULL : (_tmp11_ = (gee_collection_object_unref (_tmp11_), NULL)), _tmp12_);
			while (TRUE) {
				ValaDelegate* d;
				if (!gee_iterator_next (_d_it)) {
					break;
				}
				d = (ValaDelegate*) gee_iterator_get (_d_it);
				vala_namespace_add_delegate (old_ns, d);
				(d == NULL) ? NULL : (d = (vala_code_node_unref (d), NULL));
			}
			(_d_it == NULL) ? NULL : (_d_it = (gee_collection_object_unref (_d_it), NULL));
		}
		{
			GeeList* _tmp13_;
			GeeIterator* _tmp14_;
			GeeIterator* _en_it;
			_tmp13_ = NULL;
			_tmp14_ = NULL;
			_en_it = (_tmp14_ = gee_iterable_iterator ((GeeIterable*) (_tmp13_ = vala_namespace_get_enums (ns))), (_tmp13_ == NULL) ? NULL : (_tmp13_ = (gee_collection_object_unref (_tmp13_), NULL)), _tmp14_);
			while (TRUE) {
				ValaEnum* en;
				if (!gee_iterator_next (_en_it)) {
					break;
				}
				en = (ValaEnum*) gee_iterator_get (_en_it);
				vala_namespace_add_enum (old_ns, en);
				(en == NULL) ? NULL : (en = (vala_code_node_unref (en), NULL));
			}
			(_en_it == NULL) ? NULL : (_en_it = (gee_collection_object_unref (_en_it), NULL));
		}
		{
			GeeList* _tmp15_;
			GeeIterator* _tmp16_;
			GeeIterator* _ed_it;
			_tmp15_ = NULL;
			_tmp16_ = NULL;
			_ed_it = (_tmp16_ = gee_iterable_iterator ((GeeIterable*) (_tmp15_ = vala_namespace_get_error_domains (ns))), (_tmp15_ == NULL) ? NULL : (_tmp15_ = (gee_collection_object_unref (_tmp15_), NULL)), _tmp16_);
			while (TRUE) {
				ValaErrorDomain* ed;
				if (!gee_iterator_next (_ed_it)) {
					break;
				}
				ed = (ValaErrorDomain*) gee_iterator_get (_ed_it);
				vala_namespace_add_error_domain (old_ns, ed);
				(ed == NULL) ? NULL : (ed = (vala_code_node_unref (ed), NULL));
			}
			(_ed_it == NULL) ? NULL : (_ed_it = (gee_collection_object_unref (_ed_it), NULL));
		}
		{
			GeeList* _tmp17_;
			GeeIterator* _tmp18_;
			GeeIterator* _c_it;
			_tmp17_ = NULL;
			_tmp18_ = NULL;
			_c_it = (_tmp18_ = gee_iterable_iterator ((GeeIterable*) (_tmp17_ = vala_namespace_get_constants (ns))), (_tmp17_ == NULL) ? NULL : (_tmp17_ = (gee_collection_object_unref (_tmp17_), NULL)), _tmp18_);
			while (TRUE) {
				ValaConstant* c;
				if (!gee_iterator_next (_c_it)) {
					break;
				}
				c = (ValaConstant*) gee_iterator_get (_c_it);
				vala_namespace_add_constant (old_ns, c);
				(c == NULL) ? NULL : (c = (vala_code_node_unref (c), NULL));
			}
			(_c_it == NULL) ? NULL : (_c_it = (gee_collection_object_unref (_c_it), NULL));
		}
		{
			GeeList* _tmp19_;
			GeeIterator* _tmp20_;
			GeeIterator* _f_it;
			_tmp19_ = NULL;
			_tmp20_ = NULL;
			_f_it = (_tmp20_ = gee_iterable_iterator ((GeeIterable*) (_tmp19_ = vala_namespace_get_fields (ns))), (_tmp19_ == NULL) ? NULL : (_tmp19_ = (gee_collection_object_unref (_tmp19_), NULL)), _tmp20_);
			while (TRUE) {
				ValaField* f;
				if (!gee_iterator_next (_f_it)) {
					break;
				}
				f = (ValaField*) gee_iterator_get (_f_it);
				vala_namespace_add_field (old_ns, f);
				(f == NULL) ? NULL : (f = (vala_code_node_unref (f), NULL));
			}
			(_f_it == NULL) ? NULL : (_f_it = (gee_collection_object_unref (_f_it), NULL));
		}
		{
			GeeList* _tmp21_;
			GeeIterator* _tmp22_;
			GeeIterator* _m_it;
			_tmp21_ = NULL;
			_tmp22_ = NULL;
			_m_it = (_tmp22_ = gee_iterable_iterator ((GeeIterable*) (_tmp21_ = vala_namespace_get_methods (ns))), (_tmp21_ == NULL) ? NULL : (_tmp21_ = (gee_collection_object_unref (_tmp21_), NULL)), _tmp22_);
			while (TRUE) {
				ValaMethod* m;
				if (!gee_iterator_next (_m_it)) {
					break;
				}
				m = (ValaMethod*) gee_iterator_get (_m_it);
				vala_namespace_add_method (old_ns, m);
				(m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL));
			}
			(_m_it == NULL) ? NULL : (_m_it = (gee_collection_object_unref (_m_it), NULL));
		}
		(old_ns == NULL) ? NULL : (old_ns = (vala_code_node_unref (old_ns), NULL));
	} else {
		gee_collection_add ((GeeCollection*) self->priv->namespaces, ns);
		vala_scope_add (vala_symbol_get_scope ((ValaSymbol*) self), vala_symbol_get_name ((ValaSymbol*) ns), (ValaSymbol*) ns);
	}
}


/**
 * Returns a copy of the list of namespaces.
 *
 * @return namespace list
 */
GeeList* vala_namespace_get_namespaces (ValaNamespace* self) {
	GeeList* result;
	g_return_val_if_fail (self != NULL, NULL);
	result = (GeeList*) gee_read_only_list_new (VALA_TYPE_NAMESPACE, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, self->priv->namespaces);
	return result;
}


/**
 * Adds the specified class to this namespace.
 *
 * @param cl a class
 */
void vala_namespace_add_class (ValaNamespace* self, ValaClass* cl) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (cl != NULL);
	/* namespaces do not support private memebers*/
	if (vala_symbol_get_access ((ValaSymbol*) cl) == VALA_SYMBOL_ACCESSIBILITY_PRIVATE) {
		vala_symbol_set_access ((ValaSymbol*) cl, VALA_SYMBOL_ACCESSIBILITY_INTERNAL);
	}
	gee_collection_add ((GeeCollection*) self->priv->classes, cl);
	vala_scope_add (vala_symbol_get_scope ((ValaSymbol*) self), vala_symbol_get_name ((ValaSymbol*) cl), (ValaSymbol*) cl);
}


/**
 * Adds the specified interface to this namespace.
 *
 * @param iface an interface
 */
void vala_namespace_add_interface (ValaNamespace* self, ValaInterface* iface) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (iface != NULL);
	/* namespaces do not support private memebers*/
	if (vala_symbol_get_access ((ValaSymbol*) iface) == VALA_SYMBOL_ACCESSIBILITY_PRIVATE) {
		vala_symbol_set_access ((ValaSymbol*) iface, VALA_SYMBOL_ACCESSIBILITY_INTERNAL);
	}
	gee_collection_add ((GeeCollection*) self->priv->interfaces, iface);
	vala_scope_add (vala_symbol_get_scope ((ValaSymbol*) self), vala_symbol_get_name ((ValaSymbol*) iface), (ValaSymbol*) iface);
}


/**
 * Adds the specified struct to this namespace.
 *
 * @param st a struct
 */
void vala_namespace_add_struct (ValaNamespace* self, ValaStruct* st) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (st != NULL);
	/* namespaces do not support private memebers*/
	if (vala_symbol_get_access ((ValaSymbol*) st) == VALA_SYMBOL_ACCESSIBILITY_PRIVATE) {
		vala_symbol_set_access ((ValaSymbol*) st, VALA_SYMBOL_ACCESSIBILITY_INTERNAL);
	}
	gee_collection_add ((GeeCollection*) self->priv->structs, st);
	vala_scope_add (vala_symbol_get_scope ((ValaSymbol*) self), vala_symbol_get_name ((ValaSymbol*) st), (ValaSymbol*) st);
}


/**
 * Removes the specified struct from this namespace.
 *
 * @param st a struct
 */
void vala_namespace_remove_struct (ValaNamespace* self, ValaStruct* st) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (st != NULL);
	gee_collection_remove ((GeeCollection*) self->priv->structs, st);
	vala_scope_remove (vala_symbol_get_scope ((ValaSymbol*) self), vala_symbol_get_name ((ValaSymbol*) st));
}


/**
 * Adds the specified enum to this namespace.
 *
 * @param en an enum
 */
void vala_namespace_add_enum (ValaNamespace* self, ValaEnum* en) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (en != NULL);
	/* namespaces do not support private memebers*/
	if (vala_symbol_get_access ((ValaSymbol*) en) == VALA_SYMBOL_ACCESSIBILITY_PRIVATE) {
		vala_symbol_set_access ((ValaSymbol*) en, VALA_SYMBOL_ACCESSIBILITY_INTERNAL);
	}
	gee_collection_add ((GeeCollection*) self->priv->enums, en);
	vala_scope_add (vala_symbol_get_scope ((ValaSymbol*) self), vala_symbol_get_name ((ValaSymbol*) en), (ValaSymbol*) en);
}


/**
 * Adds the specified error domain to this namespace.
 *
 * @param edomain an error domain
 */
void vala_namespace_add_error_domain (ValaNamespace* self, ValaErrorDomain* edomain) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (edomain != NULL);
	/* namespaces do not support private memebers*/
	if (vala_symbol_get_access ((ValaSymbol*) edomain) == VALA_SYMBOL_ACCESSIBILITY_PRIVATE) {
		vala_symbol_set_access ((ValaSymbol*) edomain, VALA_SYMBOL_ACCESSIBILITY_INTERNAL);
	}
	gee_collection_add ((GeeCollection*) self->priv->error_domains, edomain);
	vala_scope_add (vala_symbol_get_scope ((ValaSymbol*) self), vala_symbol_get_name ((ValaSymbol*) edomain), (ValaSymbol*) edomain);
}


/**
 * Adds the specified delegate to this namespace.
 *
 * @param d a delegate
 */
void vala_namespace_add_delegate (ValaNamespace* self, ValaDelegate* d) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (d != NULL);
	/* namespaces do not support private memebers*/
	if (vala_symbol_get_access ((ValaSymbol*) d) == VALA_SYMBOL_ACCESSIBILITY_PRIVATE) {
		vala_symbol_set_access ((ValaSymbol*) d, VALA_SYMBOL_ACCESSIBILITY_INTERNAL);
	}
	gee_collection_add ((GeeCollection*) self->priv->delegates, d);
	vala_scope_add (vala_symbol_get_scope ((ValaSymbol*) self), vala_symbol_get_name ((ValaSymbol*) d), (ValaSymbol*) d);
}


/**
 * Returns a copy of the list of structs.
 *
 * @return struct list
 */
GeeList* vala_namespace_get_structs (ValaNamespace* self) {
	GeeList* result;
	g_return_val_if_fail (self != NULL, NULL);
	result = (GeeList*) gee_read_only_list_new (VALA_TYPE_STRUCT, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, self->priv->structs);
	return result;
}


/**
 * Returns a copy of the list of classes.
 *
 * @return class list
 */
GeeList* vala_namespace_get_classes (ValaNamespace* self) {
	GeeList* result;
	g_return_val_if_fail (self != NULL, NULL);
	result = (GeeList*) gee_read_only_list_new (VALA_TYPE_CLASS, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, self->priv->classes);
	return result;
}


/**
 * Returns a copy of the list of interfaces.
 *
 * @return interface list
 */
GeeList* vala_namespace_get_interfaces (ValaNamespace* self) {
	GeeList* result;
	g_return_val_if_fail (self != NULL, NULL);
	result = (GeeList*) gee_read_only_list_new (VALA_TYPE_INTERFACE, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, self->priv->interfaces);
	return result;
}


/**
 * Returns a copy of the list of enums.
 *
 * @return enum list
 */
GeeList* vala_namespace_get_enums (ValaNamespace* self) {
	GeeList* result;
	g_return_val_if_fail (self != NULL, NULL);
	result = (GeeList*) gee_read_only_list_new (VALA_TYPE_ENUM, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, self->priv->enums);
	return result;
}


/**
 * Returns a copy of the list of error domains.
 *
 * @return error domain list
 */
GeeList* vala_namespace_get_error_domains (ValaNamespace* self) {
	GeeList* result;
	g_return_val_if_fail (self != NULL, NULL);
	result = (GeeList*) gee_read_only_list_new (VALA_TYPE_ERROR_DOMAIN, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, self->priv->error_domains);
	return result;
}


/**
 * Returns a copy of the list of fields.
 *
 * @return field list
 */
GeeList* vala_namespace_get_fields (ValaNamespace* self) {
	GeeList* result;
	g_return_val_if_fail (self != NULL, NULL);
	result = (GeeList*) gee_read_only_list_new (VALA_TYPE_FIELD, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, self->priv->fields);
	return result;
}


/**
 * Returns a copy of the list of constants.
 *
 * @return constant list
 */
GeeList* vala_namespace_get_constants (ValaNamespace* self) {
	GeeList* result;
	g_return_val_if_fail (self != NULL, NULL);
	result = (GeeList*) gee_read_only_list_new (VALA_TYPE_CONSTANT, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, self->priv->constants);
	return result;
}


/**
 * Returns a copy of the list of delegates.
 *
 * @return delegate list
 */
GeeList* vala_namespace_get_delegates (ValaNamespace* self) {
	GeeList* result;
	g_return_val_if_fail (self != NULL, NULL);
	result = (GeeList*) gee_read_only_list_new (VALA_TYPE_DELEGATE, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, self->priv->delegates);
	return result;
}


/**
 * Returns a copy of the list of methods.
 *
 * @return method list
 */
GeeList* vala_namespace_get_methods (ValaNamespace* self) {
	GeeList* result;
	g_return_val_if_fail (self != NULL, NULL);
	result = (GeeList*) gee_read_only_list_new (VALA_TYPE_METHOD, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, self->priv->methods);
	return result;
}


/**
 * Adds the specified constant to this namespace.
 *
 * @param constant a constant
 */
void vala_namespace_add_constant (ValaNamespace* self, ValaConstant* constant) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (constant != NULL);
	/* namespaces do not support private memebers*/
	if (vala_symbol_get_access ((ValaSymbol*) constant) == VALA_SYMBOL_ACCESSIBILITY_PRIVATE) {
		vala_symbol_set_access ((ValaSymbol*) constant, VALA_SYMBOL_ACCESSIBILITY_INTERNAL);
	}
	gee_collection_add ((GeeCollection*) self->priv->constants, constant);
	vala_scope_add (vala_symbol_get_scope ((ValaSymbol*) self), vala_symbol_get_name ((ValaSymbol*) constant), (ValaSymbol*) constant);
}


/**
 * Adds the specified field to this namespace.
 *
 * @param f a field
 */
void vala_namespace_add_field (ValaNamespace* self, ValaField* f) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (f != NULL);
	/* namespaces do not support private memebers*/
	if (vala_symbol_get_access ((ValaSymbol*) f) == VALA_SYMBOL_ACCESSIBILITY_PRIVATE) {
		vala_symbol_set_access ((ValaSymbol*) f, VALA_SYMBOL_ACCESSIBILITY_INTERNAL);
	}
	if (vala_field_get_binding (f) == MEMBER_BINDING_INSTANCE) {
		vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) f), "instance members are not allowed outside of data types");
		vala_code_node_set_error ((ValaCodeNode*) f, TRUE);
		return;
	} else {
		if (vala_field_get_binding (f) == MEMBER_BINDING_CLASS) {
			vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) f), "class members are not allowed outside of classes");
			vala_code_node_set_error ((ValaCodeNode*) f, TRUE);
			return;
		}
	}
	gee_collection_add ((GeeCollection*) self->priv->fields, f);
	vala_scope_add (vala_symbol_get_scope ((ValaSymbol*) self), vala_symbol_get_name ((ValaSymbol*) f), (ValaSymbol*) f);
}


/**
 * Adds the specified method to this namespace.
 *
 * @param m a method
 */
void vala_namespace_add_method (ValaNamespace* self, ValaMethod* m) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (m != NULL);
	/* namespaces do not support private memebers*/
	if (vala_symbol_get_access ((ValaSymbol*) m) == VALA_SYMBOL_ACCESSIBILITY_PRIVATE) {
		vala_symbol_set_access ((ValaSymbol*) m, VALA_SYMBOL_ACCESSIBILITY_INTERNAL);
	}
	if (VALA_IS_CREATION_METHOD (m)) {
		vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) m), "construction methods may only be declared within classes and structs");
		vala_code_node_set_error ((ValaCodeNode*) m, TRUE);
		return;
	}
	if (vala_method_get_binding (m) == MEMBER_BINDING_INSTANCE) {
		vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) m), "instance members are not allowed outside of data types");
		vala_code_node_set_error ((ValaCodeNode*) m, TRUE);
		return;
	} else {
		if (vala_method_get_binding (m) == MEMBER_BINDING_CLASS) {
			vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) m), "class members are not allowed outside of classes");
			vala_code_node_set_error ((ValaCodeNode*) m, TRUE);
			return;
		}
	}
	gee_collection_add ((GeeCollection*) self->priv->methods, m);
	vala_scope_add (vala_symbol_get_scope ((ValaSymbol*) self), vala_symbol_get_name ((ValaSymbol*) m), (ValaSymbol*) m);
}


static void vala_namespace_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaNamespace * self;
	self = (ValaNamespace*) base;
	g_return_if_fail (visitor != NULL);
	vala_code_visitor_visit_namespace (visitor, self);
}


static void vala_namespace_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaNamespace * self;
	self = (ValaNamespace*) base;
	g_return_if_fail (visitor != NULL);
	{
		GeeIterator* _ns_it;
		_ns_it = gee_iterable_iterator ((GeeIterable*) self->priv->namespaces);
		while (TRUE) {
			ValaNamespace* ns;
			if (!gee_iterator_next (_ns_it)) {
				break;
			}
			ns = (ValaNamespace*) gee_iterator_get (_ns_it);
			vala_code_node_accept ((ValaCodeNode*) ns, visitor);
			(ns == NULL) ? NULL : (ns = (vala_code_node_unref (ns), NULL));
		}
		(_ns_it == NULL) ? NULL : (_ns_it = (gee_collection_object_unref (_ns_it), NULL));
	}
	/* process enums first to avoid order problems in C code */
	{
		GeeIterator* _en_it;
		/* process enums first to avoid order problems in C code */
		_en_it = gee_iterable_iterator ((GeeIterable*) self->priv->enums);
		/* process enums first to avoid order problems in C code */
		while (TRUE) {
			ValaEnum* en;
			/* process enums first to avoid order problems in C code */
			if (!gee_iterator_next (_en_it)) {
				/* process enums first to avoid order problems in C code */
				break;
			}
			/* process enums first to avoid order problems in C code */
			en = (ValaEnum*) gee_iterator_get (_en_it);
			vala_code_node_accept ((ValaCodeNode*) en, visitor);
			(en == NULL) ? NULL : (en = (vala_code_node_unref (en), NULL));
		}
		(_en_it == NULL) ? NULL : (_en_it = (gee_collection_object_unref (_en_it), NULL));
	}
	{
		GeeIterator* _edomain_it;
		_edomain_it = gee_iterable_iterator ((GeeIterable*) self->priv->error_domains);
		while (TRUE) {
			ValaErrorDomain* edomain;
			if (!gee_iterator_next (_edomain_it)) {
				break;
			}
			edomain = (ValaErrorDomain*) gee_iterator_get (_edomain_it);
			vala_code_node_accept ((ValaCodeNode*) edomain, visitor);
			(edomain == NULL) ? NULL : (edomain = (vala_code_node_unref (edomain), NULL));
		}
		(_edomain_it == NULL) ? NULL : (_edomain_it = (gee_collection_object_unref (_edomain_it), NULL));
	}
	{
		GeeIterator* _cl_it;
		_cl_it = gee_iterable_iterator ((GeeIterable*) self->priv->classes);
		while (TRUE) {
			ValaClass* cl;
			if (!gee_iterator_next (_cl_it)) {
				break;
			}
			cl = (ValaClass*) gee_iterator_get (_cl_it);
			vala_code_node_accept ((ValaCodeNode*) cl, visitor);
			(cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL));
		}
		(_cl_it == NULL) ? NULL : (_cl_it = (gee_collection_object_unref (_cl_it), NULL));
	}
	{
		GeeIterator* _iface_it;
		_iface_it = gee_iterable_iterator ((GeeIterable*) self->priv->interfaces);
		while (TRUE) {
			ValaInterface* iface;
			if (!gee_iterator_next (_iface_it)) {
				break;
			}
			iface = (ValaInterface*) gee_iterator_get (_iface_it);
			vala_code_node_accept ((ValaCodeNode*) iface, visitor);
			(iface == NULL) ? NULL : (iface = (vala_code_node_unref (iface), NULL));
		}
		(_iface_it == NULL) ? NULL : (_iface_it = (gee_collection_object_unref (_iface_it), NULL));
	}
	{
		GeeIterator* _st_it;
		_st_it = gee_iterable_iterator ((GeeIterable*) self->priv->structs);
		while (TRUE) {
			ValaStruct* st;
			if (!gee_iterator_next (_st_it)) {
				break;
			}
			st = (ValaStruct*) gee_iterator_get (_st_it);
			vala_code_node_accept ((ValaCodeNode*) st, visitor);
			(st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL));
		}
		(_st_it == NULL) ? NULL : (_st_it = (gee_collection_object_unref (_st_it), NULL));
	}
	{
		GeeIterator* _d_it;
		_d_it = gee_iterable_iterator ((GeeIterable*) self->priv->delegates);
		while (TRUE) {
			ValaDelegate* d;
			if (!gee_iterator_next (_d_it)) {
				break;
			}
			d = (ValaDelegate*) gee_iterator_get (_d_it);
			vala_code_node_accept ((ValaCodeNode*) d, visitor);
			(d == NULL) ? NULL : (d = (vala_code_node_unref (d), NULL));
		}
		(_d_it == NULL) ? NULL : (_d_it = (gee_collection_object_unref (_d_it), NULL));
	}
	{
		GeeIterator* _c_it;
		_c_it = gee_iterable_iterator ((GeeIterable*) self->priv->constants);
		while (TRUE) {
			ValaConstant* c;
			if (!gee_iterator_next (_c_it)) {
				break;
			}
			c = (ValaConstant*) gee_iterator_get (_c_it);
			vala_code_node_accept ((ValaCodeNode*) c, visitor);
			(c == NULL) ? NULL : (c = (vala_code_node_unref (c), NULL));
		}
		(_c_it == NULL) ? NULL : (_c_it = (gee_collection_object_unref (_c_it), NULL));
	}
	{
		GeeIterator* _f_it;
		_f_it = gee_iterable_iterator ((GeeIterable*) self->priv->fields);
		while (TRUE) {
			ValaField* f;
			if (!gee_iterator_next (_f_it)) {
				break;
			}
			f = (ValaField*) gee_iterator_get (_f_it);
			vala_code_node_accept ((ValaCodeNode*) f, visitor);
			(f == NULL) ? NULL : (f = (vala_code_node_unref (f), NULL));
		}
		(_f_it == NULL) ? NULL : (_f_it = (gee_collection_object_unref (_f_it), NULL));
	}
	{
		GeeIterator* _m_it;
		_m_it = gee_iterable_iterator ((GeeIterable*) self->priv->methods);
		while (TRUE) {
			ValaMethod* m;
			if (!gee_iterator_next (_m_it)) {
				break;
			}
			m = (ValaMethod*) gee_iterator_get (_m_it);
			vala_code_node_accept ((ValaCodeNode*) m, visitor);
			(m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL));
		}
		(_m_it == NULL) ? NULL : (_m_it = (gee_collection_object_unref (_m_it), NULL));
	}
}


static char* vala_namespace_real_get_cprefix (ValaSymbol* base) {
	ValaNamespace * self;
	char* result;
	self = (ValaNamespace*) base;
	if (gee_collection_get_size ((GeeCollection*) self->priv->cprefixes) > 0) {
		result = (char*) gee_list_get (self->priv->cprefixes, 0);
		return result;
	} else {
		if (NULL != vala_symbol_get_name ((ValaSymbol*) self)) {
			char* parent_prefix;
			parent_prefix = NULL;
			if (vala_symbol_get_parent_symbol ((ValaSymbol*) self) == NULL) {
				char* _tmp0_;
				_tmp0_ = NULL;
				parent_prefix = (_tmp0_ = g_strdup (""), parent_prefix = (g_free (parent_prefix), NULL), _tmp0_);
			} else {
				char* _tmp1_;
				_tmp1_ = NULL;
				parent_prefix = (_tmp1_ = vala_symbol_get_cprefix (vala_symbol_get_parent_symbol ((ValaSymbol*) self)), parent_prefix = (g_free (parent_prefix), NULL), _tmp1_);
			}
			result = g_strconcat (parent_prefix, vala_symbol_get_name ((ValaSymbol*) self), NULL);
			parent_prefix = (g_free (parent_prefix), NULL);
			return result;
		} else {
			result = g_strdup ("");
			return result;
		}
	}
}


GeeList* vala_namespace_get_cprefixes (ValaNamespace* self) {
	GeeList* result;
	gboolean _tmp0_;
	GeeList* _tmp1_;
	g_return_val_if_fail (self != NULL, NULL);
	_tmp0_ = FALSE;
	if (0 == gee_collection_get_size ((GeeCollection*) self->priv->cprefixes)) {
		_tmp0_ = NULL != vala_symbol_get_name ((ValaSymbol*) self);
	} else {
		_tmp0_ = FALSE;
	}
	if (_tmp0_) {
		gee_collection_add ((GeeCollection*) self->priv->cprefixes, vala_symbol_get_name ((ValaSymbol*) self));
	}
	_tmp1_ = NULL;
	result = (_tmp1_ = self->priv->cprefixes, (_tmp1_ == NULL) ? NULL : gee_collection_object_ref (_tmp1_));
	return result;
}


/**
 * Adds a camel case string to be prepended to the name of members of
 * this namespace when used in C code.
 *
 * @param cprefixes the camel case prefixes used in C code
 */
void vala_namespace_add_cprefix (ValaNamespace* self, const char* cprefix) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (cprefix != NULL);
	gee_collection_add ((GeeCollection*) self->priv->cprefixes, cprefix);
}


/**
 * Returns the lower case string to be prepended to the name of members
 * of this namespace when used in C code.
 *
 * @return the lower case prefix to be used in C code
 */
static char* vala_namespace_real_get_lower_case_cprefix (ValaSymbol* base) {
	ValaNamespace * self;
	char* result;
	const char* _tmp5_;
	self = (ValaNamespace*) base;
	if (self->priv->lower_case_cprefix == NULL) {
		if (vala_symbol_get_name ((ValaSymbol*) self) == NULL) {
			char* _tmp0_;
			_tmp0_ = NULL;
			self->priv->lower_case_cprefix = (_tmp0_ = g_strdup (""), self->priv->lower_case_cprefix = (g_free (self->priv->lower_case_cprefix), NULL), _tmp0_);
		} else {
			char* parent_prefix;
			char* _tmp4_;
			char* _tmp3_;
			parent_prefix = NULL;
			if (vala_symbol_get_parent_symbol ((ValaSymbol*) self) == NULL) {
				char* _tmp1_;
				_tmp1_ = NULL;
				parent_prefix = (_tmp1_ = g_strdup (""), parent_prefix = (g_free (parent_prefix), NULL), _tmp1_);
			} else {
				char* _tmp2_;
				_tmp2_ = NULL;
				parent_prefix = (_tmp2_ = vala_symbol_get_lower_case_cprefix (vala_symbol_get_parent_symbol ((ValaSymbol*) self)), parent_prefix = (g_free (parent_prefix), NULL), _tmp2_);
			}
			_tmp4_ = NULL;
			_tmp3_ = NULL;
			self->priv->lower_case_cprefix = (_tmp4_ = g_strdup_printf ("%s%s_", parent_prefix, _tmp3_ = vala_symbol_camel_case_to_lower_case (vala_symbol_get_name ((ValaSymbol*) self))), self->priv->lower_case_cprefix = (g_free (self->priv->lower_case_cprefix), NULL), _tmp4_);
			_tmp3_ = (g_free (_tmp3_), NULL);
			parent_prefix = (g_free (parent_prefix), NULL);
		}
	}
	_tmp5_ = NULL;
	result = (_tmp5_ = self->priv->lower_case_cprefix, (_tmp5_ == NULL) ? NULL : g_strdup (_tmp5_));
	return result;
}


/**
 * Sets the lower case string to be prepended to the name of members of
 * this namespace when used in C code.
 *
 * @param cprefix the lower case prefix to be used in C code
 */
void vala_namespace_set_lower_case_cprefix (ValaNamespace* self, const char* cprefix) {
	char* _tmp1_;
	const char* _tmp0_;
	g_return_if_fail (self != NULL);
	g_return_if_fail (cprefix != NULL);
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	self->priv->lower_case_cprefix = (_tmp1_ = (_tmp0_ = cprefix, (_tmp0_ == NULL) ? NULL : g_strdup (_tmp0_)), self->priv->lower_case_cprefix = (g_free (self->priv->lower_case_cprefix), NULL), _tmp1_);
}


static GeeList* vala_namespace_real_get_cheader_filenames (ValaSymbol* base) {
	ValaNamespace * self;
	GeeList* result;
	self = (ValaNamespace*) base;
	result = (GeeList*) gee_read_only_list_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, self->priv->cheader_filenames);
	return result;
}


/**
 * Sets the C header filename of this namespace to the specified
 * filename.
 *
 * @param cheader_filename header filename
 */
void vala_namespace_set_cheader_filename (ValaNamespace* self, const char* cheader_filename) {
	GeeList* _tmp0_;
	g_return_if_fail (self != NULL);
	g_return_if_fail (cheader_filename != NULL);
	_tmp0_ = NULL;
	self->priv->cheader_filenames = (_tmp0_ = (GeeList*) gee_array_list_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_direct_equal), (self->priv->cheader_filenames == NULL) ? NULL : (self->priv->cheader_filenames = (gee_collection_object_unref (self->priv->cheader_filenames), NULL)), _tmp0_);
	gee_collection_add ((GeeCollection*) self->priv->cheader_filenames, cheader_filename);
}


/**
 * Adds a C header filename of this namespace.
 *
 * @param cheader_filename header filename
 */
void vala_namespace_add_cheader_filename (ValaNamespace* self, const char* cheader_filename) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (cheader_filename != NULL);
	gee_collection_add ((GeeCollection*) self->priv->cheader_filenames, cheader_filename);
}


static void vala_namespace_process_ccode_attribute (ValaNamespace* self, ValaAttribute* a) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (a != NULL);
	if (vala_attribute_has_argument (a, "cprefix")) {
		char* value;
		value = vala_attribute_get_string (a, "cprefix");
		if (_vala_strcmp0 (value, "") == 0) {
			/* split of an empty string returns an empty array*/
			vala_namespace_add_cprefix (self, "");
		} else {
			{
				char** _tmp0_;
				char** name_collection;
				int name_collection_length1;
				int name_it;
				_tmp0_ = NULL;
				name_collection = _tmp0_ = g_strsplit (value, ",", 0);
				name_collection_length1 = _vala_array_length (_tmp0_);
				for (name_it = 0; name_it < _vala_array_length (_tmp0_); name_it = name_it + 1) {
					const char* _tmp1_;
					char* name;
					_tmp1_ = NULL;
					name = (_tmp1_ = name_collection[name_it], (_tmp1_ == NULL) ? NULL : g_strdup (_tmp1_));
					{
						vala_namespace_add_cprefix (self, name);
						name = (g_free (name), NULL);
					}
				}
				name_collection = (_vala_array_free (name_collection, name_collection_length1, (GDestroyNotify) g_free), NULL);
			}
		}
		value = (g_free (value), NULL);
	}
	if (vala_attribute_has_argument (a, "lower_case_cprefix")) {
		char* _tmp2_;
		_tmp2_ = NULL;
		vala_namespace_set_lower_case_cprefix (self, _tmp2_ = vala_attribute_get_string (a, "lower_case_cprefix"));
		_tmp2_ = (g_free (_tmp2_), NULL);
	}
	if (vala_attribute_has_argument (a, "cheader_filename")) {
		char* val;
		val = vala_attribute_get_string (a, "cheader_filename");
		{
			char** _tmp3_;
			char** filename_collection;
			int filename_collection_length1;
			int filename_it;
			_tmp3_ = NULL;
			filename_collection = _tmp3_ = g_strsplit (val, ",", 0);
			filename_collection_length1 = _vala_array_length (_tmp3_);
			for (filename_it = 0; filename_it < _vala_array_length (_tmp3_); filename_it = filename_it + 1) {
				const char* _tmp4_;
				char* filename;
				_tmp4_ = NULL;
				filename = (_tmp4_ = filename_collection[filename_it], (_tmp4_ == NULL) ? NULL : g_strdup (_tmp4_));
				{
					gee_collection_add ((GeeCollection*) self->priv->cheader_filenames, filename);
					filename = (g_free (filename), NULL);
				}
			}
			filename_collection = (_vala_array_free (filename_collection, filename_collection_length1, (GDestroyNotify) g_free), NULL);
		}
		val = (g_free (val), NULL);
	}
}


/**
 * Process all associated attributes.
 */
void vala_namespace_process_attributes (ValaNamespace* self) {
	g_return_if_fail (self != NULL);
	{
		GList* a_collection;
		GList* a_it;
		a_collection = ((ValaCodeNode*) self)->attributes;
		for (a_it = a_collection; a_it != NULL; a_it = a_it->next) {
			ValaAttribute* _tmp0_;
			ValaAttribute* a;
			_tmp0_ = NULL;
			a = (_tmp0_ = (ValaAttribute*) a_it->data, (_tmp0_ == NULL) ? NULL : vala_code_node_ref (_tmp0_));
			{
				if (_vala_strcmp0 (vala_attribute_get_name (a), "CCode") == 0) {
					vala_namespace_process_ccode_attribute (self, a);
				}
				(a == NULL) ? NULL : (a = (vala_code_node_unref (a), NULL));
			}
		}
	}
}


static gboolean vala_namespace_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer) {
	ValaNamespace * self;
	gboolean result;
	self = (ValaNamespace*) base;
	g_return_val_if_fail (analyzer != NULL, FALSE);
	if (vala_code_node_get_checked ((ValaCodeNode*) self)) {
		result = !vala_code_node_get_error ((ValaCodeNode*) self);
		return result;
	}
	vala_code_node_set_checked ((ValaCodeNode*) self, TRUE);
	vala_namespace_process_attributes (self);
	{
		GeeIterator* _ns_it;
		_ns_it = gee_iterable_iterator ((GeeIterable*) self->priv->namespaces);
		while (TRUE) {
			ValaNamespace* ns;
			if (!gee_iterator_next (_ns_it)) {
				break;
			}
			ns = (ValaNamespace*) gee_iterator_get (_ns_it);
			vala_code_node_check ((ValaCodeNode*) ns, analyzer);
			(ns == NULL) ? NULL : (ns = (vala_code_node_unref (ns), NULL));
		}
		(_ns_it == NULL) ? NULL : (_ns_it = (gee_collection_object_unref (_ns_it), NULL));
	}
	result = !vala_code_node_get_error ((ValaCodeNode*) self);
	return result;
}


static void vala_namespace_class_init (ValaNamespaceClass * klass) {
	vala_namespace_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_NODE_CLASS (klass)->finalize = vala_namespace_finalize;
	g_type_class_add_private (klass, sizeof (ValaNamespacePrivate));
	VALA_CODE_NODE_CLASS (klass)->accept = vala_namespace_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_namespace_real_accept_children;
	VALA_SYMBOL_CLASS (klass)->get_cprefix = vala_namespace_real_get_cprefix;
	VALA_SYMBOL_CLASS (klass)->get_lower_case_cprefix = vala_namespace_real_get_lower_case_cprefix;
	VALA_SYMBOL_CLASS (klass)->get_cheader_filenames = vala_namespace_real_get_cheader_filenames;
	VALA_CODE_NODE_CLASS (klass)->check = vala_namespace_real_check;
}


static void vala_namespace_instance_init (ValaNamespace * self) {
	self->priv = VALA_NAMESPACE_GET_PRIVATE (self);
	self->priv->classes = (GeeList*) gee_array_list_new (VALA_TYPE_CLASS, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	self->priv->interfaces = (GeeList*) gee_array_list_new (VALA_TYPE_INTERFACE, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	self->priv->structs = (GeeList*) gee_array_list_new (VALA_TYPE_STRUCT, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	self->priv->enums = (GeeList*) gee_array_list_new (VALA_TYPE_ENUM, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	self->priv->error_domains = (GeeList*) gee_array_list_new (VALA_TYPE_ERROR_DOMAIN, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	self->priv->delegates = (GeeList*) gee_array_list_new (VALA_TYPE_DELEGATE, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	self->priv->constants = (GeeList*) gee_array_list_new (VALA_TYPE_CONSTANT, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	self->priv->fields = (GeeList*) gee_array_list_new (VALA_TYPE_FIELD, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	self->priv->methods = (GeeList*) gee_array_list_new (VALA_TYPE_METHOD, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	self->priv->cprefixes = (GeeList*) gee_array_list_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_direct_equal);
	self->priv->cheader_filenames = (GeeList*) gee_array_list_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_direct_equal);
	self->priv->namespaces = (GeeList*) gee_array_list_new (VALA_TYPE_NAMESPACE, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
}


static void vala_namespace_finalize (ValaCodeNode* obj) {
	ValaNamespace * self;
	self = VALA_NAMESPACE (obj);
	(self->priv->classes == NULL) ? NULL : (self->priv->classes = (gee_collection_object_unref (self->priv->classes), NULL));
	(self->priv->interfaces == NULL) ? NULL : (self->priv->interfaces = (gee_collection_object_unref (self->priv->interfaces), NULL));
	(self->priv->structs == NULL) ? NULL : (self->priv->structs = (gee_collection_object_unref (self->priv->structs), NULL));
	(self->priv->enums == NULL) ? NULL : (self->priv->enums = (gee_collection_object_unref (self->priv->enums), NULL));
	(self->priv->error_domains == NULL) ? NULL : (self->priv->error_domains = (gee_collection_object_unref (self->priv->error_domains), NULL));
	(self->priv->delegates == NULL) ? NULL : (self->priv->delegates = (gee_collection_object_unref (self->priv->delegates), NULL));
	(self->priv->constants == NULL) ? NULL : (self->priv->constants = (gee_collection_object_unref (self->priv->constants), NULL));
	(self->priv->fields == NULL) ? NULL : (self->priv->fields = (gee_collection_object_unref (self->priv->fields), NULL));
	(self->priv->methods == NULL) ? NULL : (self->priv->methods = (gee_collection_object_unref (self->priv->methods), NULL));
	(self->priv->cprefixes == NULL) ? NULL : (self->priv->cprefixes = (gee_collection_object_unref (self->priv->cprefixes), NULL));
	self->priv->lower_case_cprefix = (g_free (self->priv->lower_case_cprefix), NULL);
	(self->priv->cheader_filenames == NULL) ? NULL : (self->priv->cheader_filenames = (gee_collection_object_unref (self->priv->cheader_filenames), NULL));
	(self->priv->namespaces == NULL) ? NULL : (self->priv->namespaces = (gee_collection_object_unref (self->priv->namespaces), NULL));
	VALA_CODE_NODE_CLASS (vala_namespace_parent_class)->finalize (obj);
}


GType vala_namespace_get_type (void) {
	static GType vala_namespace_type_id = 0;
	if (vala_namespace_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaNamespaceClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_namespace_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaNamespace), 0, (GInstanceInitFunc) vala_namespace_instance_init, NULL };
		vala_namespace_type_id = g_type_register_static (VALA_TYPE_SYMBOL, "ValaNamespace", &g_define_type_info, 0);
	}
	return vala_namespace_type_id;
}


static void _vala_array_destroy (gpointer array, gint array_length, GDestroyNotify destroy_func) {
	if ((array != NULL) && (destroy_func != NULL)) {
		int i;
		for (i = 0; i < array_length; i = i + 1) {
			if (((gpointer*) array)[i] != NULL) {
				destroy_func (((gpointer*) array)[i]);
			}
		}
	}
}


static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func) {
	_vala_array_destroy (array, array_length, destroy_func);
	g_free (array);
}


static gint _vala_array_length (gpointer array) {
	int length;
	length = 0;
	if (array) {
		while (((gpointer*) array)[length]) {
			length++;
		}
	}
	return length;
}


static int _vala_strcmp0 (const char * str1, const char * str2) {
	if (str1 == NULL) {
		return -(str1 != str2);
	}
	if (str2 == NULL) {
		return str1 != str2;
	}
	return strcmp (str1, str2);
}




